/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2021-08-25 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */


#include "espdr_preprocessing.h"

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief         Preprocessing for NIR instruments, used only for DARK
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_preprocess(const cpl_frameset *raw_frameset,
                                const cpl_imagelist *flat_mask_iml,
                                espdr_CCD_geometry *CCD_geom,
                                espdr_inst_config *inst_config,
                                cpl_frameset **preprocessed_set) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_frame *curr_frame = NULL;
    cpl_image *curr_image = NULL;
    cpl_image *real_image = NULL;
    const cpl_image *mask_image = NULL;
    cpl_image *preprocessed_image = NULL;
    cpl_propertylist *keywords = NULL;
    const char* raw_name = NULL;
    char tmp_name[FILENAME_LENGTH];
    double exposure_time = 0.0;
    double conad = 1.0;
    
    espdr_msg("Starting preprocesing");
    
    int nx = CCD_geom->exts[0].outputs[0][0].raw_nx;
    int ny = CCD_geom->exts[0].outputs[0][0].raw_ny;
    int pscan_nx = CCD_geom->exts[0].outputs[0][0].pscan_nx;
    int oscan_nx = CCD_geom->exts[0].outputs[0][0].oscan_nx;
    int ppscan_ny = CCD_geom->exts[0].outputs[0][0].ppscan_ny;
    int poscan_ny = CCD_geom->exts[0].outputs[0][0].poscan_ny;
    //espdr_msg("nx = %d\tny = %d\tpscan_nx = %d\toscan_nx = %d\tppscan_ny = %d\tposcan_ny = %d",
    //          nx, ny, pscan_nx, oscan_nx, ppscan_ny, poscan_ny);
    
    *preprocessed_set = cpl_frameset_new();
    
    int nset = cpl_frameset_get_size(raw_frameset);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(raw_frameset);
    curr_frame = cpl_frameset_iterator_get(iter);
    
    //espdr_msg("Size of the flat_image mask list: %lld", cpl_imagelist_get_size(flat_mask_iml));
    
    if (flat_mask_iml != NULL) {
        mask_image = cpl_imagelist_get_const(flat_mask_iml, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_imagelist_get failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        mask_image = NULL;
    }
    
    //espdr_msg("Flat mask image size: %lld x %lld",
    //          cpl_image_get_size_x(mask_image), cpl_image_get_size_y(mask_image));
    //my_error = cpl_image_save(mask_image, "NIRPS_flat_mask_image.fits",
    //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    int j = 0, k = 0, l = 0, index = 0;
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        raw_name = cpl_frame_get_filename(curr_frame);
        espdr_msg("Preprocessing raw frame: %s ...", raw_name);
        keywords = cpl_propertylist_load(raw_name, 0);
        exposure_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        char *keyword_to_get = espdr_add_output_index_to_keyword(inst_config->conad_kw_first_part,
                                                                 inst_config->conad_kw_last_part,
                                                                 1);
        conad = 1.0;
        if (cpl_propertylist_has(keywords, keyword_to_get)) {
            conad = cpl_propertylist_get_double(keywords, keyword_to_get);
        } else {
            espdr_msg_error("The conad KW: %s not found in the header, exiting",
                            keyword_to_get);
            return (CPL_ERROR_NULL_INPUT);
        }
        if (conad <= 0.0) conad = 1.0;
        //espdr_msg("Read conad: %f", conad);
        
        curr_image = cpl_image_load(raw_name, CPL_TYPE_DOUBLE, 0, 1);
        espdr_msg("Image median: %f", cpl_image_get_median(curr_image));
        
        for (int i = 0; i < 3; i++) {
            my_error = espdr_top_bottom_correction(curr_image, ppscan_ny, poscan_ny);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_top_bottom_correction failed: %s",
                         cpl_error_get_message_default(my_error));
            //espdr_msg("Top-bottom correction done");
            
            my_error = espdr_left_right_correction(curr_image, ppscan_ny, poscan_ny);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_ledt_right_correction failed: %s",
                         cpl_error_get_message_default(my_error));
            //espdr_msg("Left-right correction done");
        }
        
        //my_error = cpl_image_save(curr_image, "NIRPS_after_3_top_bottom_left_right.fits",
        //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        
        // xtalk
        my_error = espdr_xtalk(curr_image, mask_image, inst_config);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_xtalk failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Xtalk correction done");
        
        // amplifiers correction
        my_error = espdr_corr_amp(curr_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_corr_amp failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Amplifiers correction done");
        
        real_image = cpl_image_extract(curr_image,
                                       pscan_nx+1, ppscan_ny+1,
                                       nx - oscan_nx, ny - poscan_ny);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Loading image failed: %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = cpl_image_multiply_scalar(real_image, exposure_time);
        
        /* Correct the very negative pixels (< 1000.0) --> 0.0
         to reduce the number of bins in the science background calculation */
        int pxl_rjct, pxl_changed_nb = 0;
        for (cpl_size i = 1; i <= cpl_image_get_size_x(real_image); i++) {
            for (cpl_size j = 1; j <= cpl_image_get_size_y(real_image); j++) {
                if (cpl_image_get(real_image, i, j, &pxl_rjct) < NEGATIVE_PXL_LIMIT) {
                    cpl_image_set(real_image, i, j, 0.0);
                    pxl_changed_nb++;
                }
            }
        }
        espdr_msg("Number of very negative pixels (< -1000.0) set to 0.0: %d", pxl_changed_nb);
        
        my_error = cpl_image_multiply_scalar(real_image, conad);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Multiplying by Texp or multiplying by CONAD failed: %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = espdr_remove_nans(real_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Removing nans failed: %s",
                     cpl_error_get_message_default(my_error));
        
        preprocessed_image = cpl_image_duplicate(real_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_duplicate failed: %s",
                     cpl_error_get_message_default(my_error));
        
        /* AMO: here we should rather save on disk and add to a
         * frameset the filename of the saved image */
        sprintf(tmp_name,
                "preprocessed_img_ext_%d_port_x%d_y%d_index_%d.fits",
                j, k, l, index);
        
        //espdr_msg("Saving tmp product: %s", tmp_name);
        cpl_frame* frame_prep = cpl_frame_new();
        cpl_frame_set_filename(frame_prep, tmp_name);
        cpl_frame_set_tag(frame_prep, "DUMMY");

        my_error = cpl_frameset_insert(*preprocessed_set,
                                       cpl_frame_duplicate(frame_prep));
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_frameset_insert failed: %s",
                     cpl_error_get_message_default(my_error));
        

        my_error = cpl_image_save(preprocessed_image, tmp_name,
                                  CPL_TYPE_DOUBLE, keywords,
                                  CPL_IO_DEFAULT);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save %s failed: %s",
                     tmp_name, cpl_error_get_message_default(my_error));
        
        cpl_frame_delete(frame_prep);
        
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
        
        char *filename_pp2 = (char *)cpl_malloc(64*sizeof(char));
        strcpy(tmp_name, raw_name);
        char *filename_arc = strrchr(tmp_name, '/')+1;
        filename_arc[strlen(filename_arc) - 5] = '\0';
        //espdr_msg("Filename of raw image: %s", filename_arc);
        sprintf(filename_pp2, "%s_prep.fits", filename_arc);
        //espdr_msg("Filename of preprocessed image: %s", filename_pp2);
        my_error = cpl_image_save(preprocessed_image, filename_pp2,
                                  CPL_TYPE_DOUBLE, keywords,
                                  CPL_IO_DEFAULT);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save %s failed: %s",
                     filename_pp2, cpl_error_get_message_default(my_error));
        espdr_msg("Preprocessed image saved in %s", filename_pp2);

#endif
        
        cpl_image_delete(preprocessed_image);
        cpl_propertylist_delete(keywords);

        cpl_frameset_iterator_advance(iter, 1);
        curr_frame = cpl_frameset_iterator_get(iter);
        
        index++;
    }

    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief         Preprocessing for NIR instruments, used for contam, orders, science, wave
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_preprocess_iml(cpl_imagelist *raw_iml,
                                    const cpl_imagelist *flat_mask_iml,
                                    const espdr_CCD_geometry *CCD_geom,
                                    espdr_inst_config *inst_config,
                                    double exposure_time,
                                    double conad,
                                    cpl_imagelist **preprocessed_iml) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int raw_iml_size = cpl_imagelist_get_size(raw_iml);
    cpl_image *curr_image = NULL;
    cpl_image *real_image = NULL;
    const cpl_image *mask_image = NULL;
    
    int nx = CCD_geom->exts[0].outputs[0][0].raw_nx;
    int ny = CCD_geom->exts[0].outputs[0][0].raw_ny;
    int pscan_nx = CCD_geom->exts[0].outputs[0][0].pscan_nx;
    int oscan_nx = CCD_geom->exts[0].outputs[0][0].oscan_nx;
    int ppscan_ny = CCD_geom->exts[0].outputs[0][0].ppscan_ny;
    int poscan_ny = CCD_geom->exts[0].outputs[0][0].poscan_ny;
    //espdr_msg("nx = %d\tny = %d\tpscan_nx = %d\toscan_nx = %d\tppscan_ny = %d\tposcan_ny = %d",
    //          nx, ny, pscan_nx, oscan_nx, ppscan_ny, poscan_ny);
    
    for (int i = 0; i < raw_iml_size; i++) {
        
        curr_image = cpl_imagelist_get(raw_iml, i);
        //espdr_msg("Raw image size: %lld x %lld",
        //          cpl_image_get_size_x(curr_image),
        //          cpl_image_get_size_y(curr_image));
        
        for (int j = 0; j < 3; j++) {
            my_error = espdr_top_bottom_correction(curr_image, ppscan_ny, poscan_ny);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_top_bottom_correction failed: %s",
                         cpl_error_get_message_default(my_error));
            //espdr_msg("Top-bottom correction done");
            
            my_error = espdr_left_right_correction(curr_image, pscan_nx, oscan_nx);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_ledt_right_correction failed: %s",
                         cpl_error_get_message_default(my_error));
            //espdr_msg("Left-right correction done");
        }
        
        //my_error = cpl_image_save(curr_image, "NIRPS_after_3_top_bottom_left_right.fits",
        //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        
        //espdr_msg("curr_image size: %lld x %lld",
        //          cpl_image_get_size_x(curr_image),
        //          cpl_image_get_size_y(curr_image));
        
        // xtalk
        if (flat_mask_iml != NULL) {
            mask_image = cpl_imagelist_get_const(flat_mask_iml, i);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_imagelist_get failed: %s",
                         cpl_error_get_message_default(my_error));
            //espdr_msg("flat_mask_image size: %lld x %lld",
            //          cpl_image_get_size_x(mask_image),
            //          cpl_image_get_size_y(mask_image));
        } else {
            mask_image = NULL;
        }
        my_error = espdr_xtalk(curr_image, mask_image, inst_config);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_xtalk failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Xtalk correction done");
        
        // amplifiers correction
        my_error = espdr_corr_amp(curr_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_corr_amp failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Amplifiers correction done");
        
        real_image = cpl_image_extract(curr_image,
                                       pscan_nx+1, ppscan_ny+1,
                                       nx - oscan_nx, ny - poscan_ny);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Extracting image failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Image extracted");
        
        my_error = cpl_image_multiply_scalar(real_image, exposure_time);
        
        /* Correct the very negative pixels (< 1000.0) --> 0.0
         to reduce the number of bins in the science background calculation */
        int pxl_rjct, pxl_changed_nb = 0;
        for (cpl_size i = 1; i <= cpl_image_get_size_x(real_image); i++) {
            for (cpl_size j = 1; j <= cpl_image_get_size_y(real_image); j++) {
                if (cpl_image_get(real_image, i, j, &pxl_rjct) < NEGATIVE_PXL_LIMIT) {
                    cpl_image_set(real_image, i, j, 0.0);
                    pxl_changed_nb++;
                }
            }
        }
        espdr_msg("Number of very negative pixels (< -1000.0) set to 0.0: %d", pxl_changed_nb);
        
        my_error = cpl_image_multiply_scalar(real_image, conad);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Multiplying by Texp or multiplying by CONAD failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Real flux got");
        
        my_error = espdr_remove_nans(real_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Removing nans failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Nans & infs removed");
        
        cpl_imagelist_set(*preprocessed_iml, cpl_image_duplicate(real_image), i);
        cpl_image_delete(real_image);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief         Preprocessing for NIR instruments, used only for FLAT and LED_FF
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_preprocess_img(cpl_image *raw_img,
                                    const cpl_image *flat_mask_img,
                                    cpl_propertylist *keywords,
                                    const espdr_CCD_geometry *CCD_geom,
                                    espdr_inst_config *inst_config,
                                    double *ron_RE,
                                    double *conad_RE,
                                    cpl_image **preprocessed_img) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_image *curr_image = NULL;
    cpl_image *real_image = NULL;
    
    int nx = CCD_geom->exts[0].outputs[0][0].raw_nx;
    int ny = CCD_geom->exts[0].outputs[0][0].raw_ny;
    int pscan_nx = CCD_geom->exts[0].outputs[0][0].pscan_nx;
    int oscan_nx = CCD_geom->exts[0].outputs[0][0].oscan_nx;
    int ppscan_ny = CCD_geom->exts[0].outputs[0][0].ppscan_ny;
    int poscan_ny = CCD_geom->exts[0].outputs[0][0].poscan_ny;
    //espdr_msg("nx = %d\tny = %d\tpscan_nx = %d\toscan_nx = %d\tppscan_ny = %d\tposcan_ny = %d",
    //          nx, ny, pscan_nx, oscan_nx, ppscan_ny, poscan_ny);
    
    espdr_msg("Preprocessing");
    double exp_time = cpl_propertylist_get_double(keywords,
                                                  inst_config->Texp_kw);
    char *keyword_to_get = espdr_add_output_index_to_keyword(inst_config->conad_kw_first_part,
                                                             inst_config->conad_kw_last_part,
                                                             1);
    double conad = 1.0;
    if (cpl_propertylist_has(keywords, keyword_to_get)) {
        conad = cpl_propertylist_get_double(keywords, keyword_to_get);
    } else {
        espdr_msg_error("The conad KW: %s not found in the header, conad set to 1.0",
                        keyword_to_get);
        return (CPL_ERROR_NULL_INPUT);
    }
    if (conad <= 0.0) conad = 1.0;
    espdr_msg("CONAD = %f [e-/ADU]", conad);
    *conad_RE = conad;
    
    double ron = sqrt(400.0/(exp_time/5.57) + 36.0) / conad; // to get ADUs
    espdr_msg("RON = %f [ADU]", ron);
    *ron_RE = ron;
    
    curr_image = cpl_image_duplicate(raw_img);
    //espdr_msg("Raw image size: %lld x %lld",
    //          cpl_image_get_size_x(curr_image),
    //          cpl_image_get_size_y(curr_image));
    
    for (int i = 0; i < 3; i ++) {
        my_error = espdr_top_bottom_correction(curr_image, ppscan_ny, poscan_ny);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_top_bottom_correction failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Top-bottom correction done");
        
        my_error = espdr_left_right_correction(curr_image, pscan_nx, oscan_nx);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_ledt_right_correction failed: %s",
                     cpl_error_get_message_default(my_error));
        //espdr_msg("Left-right correction done");
    }
    
    //espdr_msg("curr_image size: %lld x %lld",
    //          cpl_image_get_size_x(curr_image),
    //          cpl_image_get_size_y(curr_image));
    //if (flat_mask_img != NULL){
        //espdr_msg("flat_mask_image size: %lld x %lld",
        //          cpl_image_get_size_x(flat_mask_img),
        //          cpl_image_get_size_y(flat_mask_img));
    //}
    
    // xtalk
    my_error = espdr_xtalk(curr_image, flat_mask_img, inst_config);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_xtalk failed: %s",
                 cpl_error_get_message_default(my_error));
    //espdr_msg("Xtalk correction done");
    
    // amplifiers correction
    my_error = espdr_corr_amp(curr_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_corr_amp failed: %s",
                 cpl_error_get_message_default(my_error));
    //espdr_msg("Amplifiers correction done");
    
    real_image = cpl_image_extract(curr_image,
                                   pscan_nx+1, ppscan_ny+1,
                                   nx - oscan_nx, ny - poscan_ny);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Extracting image failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_image_multiply_scalar(real_image, exp_time);
    
    /* Correct the very negative pixels (< 1000.0) --> 0.0
     to reduce the number of bins in the science background calculation */
    int pxl_rjct, pxl_changed_nb = 0;
    for (cpl_size i = 1; i <= cpl_image_get_size_x(real_image); i++) {
        for (cpl_size j = 1; j <= cpl_image_get_size_y(real_image); j++) {
            if (cpl_image_get(real_image, i, j, &pxl_rjct) < NEGATIVE_PXL_LIMIT) {
                cpl_image_set(real_image, i, j, 0.0);
                pxl_changed_nb++;
            }
        }
    }
    espdr_msg("Number of very negative pixels (< -1000.0) set to 0.0: %d", pxl_changed_nb);
    
    my_error = cpl_image_multiply_scalar(real_image, conad);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Multiplying by Texp or multiplying by CONAD failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_remove_nans(real_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Removing nans failed: %s",
                 cpl_error_get_message_default(my_error));
    
    *preprocessed_img = cpl_image_duplicate(real_image);
    
    cpl_image_delete(real_image);
    cpl_image_delete(curr_image);
    
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // This will save only one output, but for NIRPS it is equivalent to the whole image
    char *filename_pp2 = (char *)cpl_malloc(64*sizeof(char));
    char *filename_arc = (char *)cpl_malloc(64*sizeof(char));
    if (cpl_propertylist_has(keywords, "ARCFILE")) {
        filename_arc = cpl_propertylist_get_string(keywords, "ARCFILE");
        filename_arc[strlen(filename_arc) - 5] = '\0';
        espdr_msg("Filename of raw image: %s", filename_arc);
        sprintf(filename_pp2, "%s_prep.fits", filename_arc);
        espdr_msg("Filename of preprocessed image: %s", filename_pp2);
        my_error = cpl_image_save(*preprocessed_img,
                                  filename_pp2,
                                  CPL_TYPE_DOUBLE,
                                  keywords,
                                  CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Image saving %s failed: %s",
                     filename_pp2, cpl_error_get_message_default(my_error));
        espdr_msg("Preprocessed image %s saved", filename_pp2);
    } else {
        espdr_msg("Can't save the preprocessed image, because the original filename is not present in the header");
    }
    cpl_free(filename_pp2);
#endif
    espdr_msg("Preprocessing finished");
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_top_bottom_correction(cpl_image *img,
                                           int ppscan_ny,
                                           int poscan_ny) {
    
    //cpl_error_code my_error = CPL_ERROR_NONE;
    int size_x = cpl_image_get_size_x(img);
    int size_y = cpl_image_get_size_y(img);
    int box_nb = size_x / BOX_SIZE;
    
    //espdr_msg("size_x: %d, box size; %d, size_x mod box_size: %d, box_nb: %d, ppscan_ny: %d, poscan_ny: %d",
    //          size_x, BOX_SIZE, size_x % BOX_SIZE, box_nb, ppscan_ny, poscan_ny);
    if ((size_x % BOX_SIZE) != 0) {
        return (CPL_ERROR_INCOMPATIBLE_INPUT);
    }
    
    double med1[box_nb], med2[box_nb];
    cpl_image *curr_img = NULL;
    double *curr_data = NULL;
    
    for (int i = 0; i < box_nb; i++) {
        // top one
        curr_img = cpl_image_extract(img,
                                     i*BOX_SIZE+1, size_y-poscan_ny+1,
                                     (i+1)*BOX_SIZE, size_y);
        curr_data = cpl_image_get_data_double(curr_img);
        med2[i] = espdr_get_median_off_nans(curr_data, poscan_ny*BOX_SIZE);
        
        // bottom one
        curr_img = cpl_image_extract(img,
                                     i*BOX_SIZE+1, 1,
                                     (i+1)*BOX_SIZE, ppscan_ny);
        curr_data = cpl_image_get_data_double(curr_img);
        med1[i] = espdr_get_median_off_nans(curr_data, ppscan_ny*BOX_SIZE);
        
        //espdr_msg("med1[%d] = %.10f, med2[%d] = %.10f", i, med1[i], i, med2[i]);
    }
    
    double slope[size_y];
    for (int i = 0; i < size_y; i++) {
        slope[i] = (double)i / (double)size_y;
    }
    
    //espdr_msg("Creating the median image of the size: %d x %d", size_x, size_y);
    cpl_image *med_img = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
    // Create the image with medians to subtract from the whole image
    for (int i = 0; i < box_nb; i++) {
        for (int j = 0; j < BOX_SIZE; j++) {
            for (int k = 0; k < size_y; k++) {
                cpl_image_set(med_img, i*BOX_SIZE+j+1, k+1,
                              med1[i] + slope[k]*(med2[i] - med1[i]));
                //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                //             "cpl_image_set failed for x = %d, y = %d, value = %f: %s",
                //             i*BOX_SIZE+j+1, k+1, med1[i] + slope[k]*(med2[i] - med1[i]),
                //             cpl_error_get_message_default(my_error));
            }
        }
    }
    
    //cpl_image_save(img, "NIRPS_raw_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //cpl_image_save(med_img, "NIRPS_median_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    //espdr_msg("Subtracting the slope");
    cpl_image_subtract(img, med_img);
    
    //cpl_image_save(img, "NIRPS_slope_corrected_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    cpl_image_delete(med_img);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_left_right_correction(cpl_image *img,
                                           int pscan_nx,
                                           int oscan_nx) {
    
    int size_x = cpl_image_get_size_x(img);
    int size_y = cpl_image_get_size_y(img);
    int median_pixels_nb = pscan_nx+oscan_nx-2;
    cpl_image *curr_img = NULL;
    double *curr_data = NULL;
    double curr_median = 0.0;
    double median_profile[size_y][median_pixels_nb];
    double median_column[size_y];
    cpl_image *median_column_img = NULL;
    cpl_image *correlated_noise_img = NULL;
    
    // Take the left and right reference pixels, without the external ones
    // number of columns: ppscan_nx - 1 + poscan_nx -1
    // Then, take the median off nans of each column and subtract it from the column
    int index = 0;
    for (int i = 2; i < pscan_nx+1; i++) {
        curr_img = cpl_image_extract(img, i, 1, i, size_y);
        curr_data = cpl_image_get_data_double(curr_img);
        curr_median = espdr_get_median_off_nans(curr_data, size_y);
        for (int j = 0; j < size_y; j++) {
            median_profile[j][index] = curr_data[j] - curr_median;
        }
        index++;
    }
    for (int i = size_x-oscan_nx+1; i < size_x; i++) {
        curr_img = cpl_image_extract(img, i, 1, i, size_y);
        curr_data = cpl_image_get_data_double(curr_img);
        curr_median = espdr_get_median_off_nans(curr_data, size_y);
        for (int j = 0; j < size_y; j++) {
            median_profile[j][index] = curr_data[j] - curr_median;
        }
        index++;
    }
    //espdr_msg("Creating median column");
    // Take the median across the columns
    for (int j = 0; j < size_y; j++) {
        median_column[j] = espdr_get_median_off_nans(median_profile[j], median_pixels_nb);
    }
    median_column_img = cpl_image_wrap_double(1, size_y, median_column);
    
    correlated_noise_img = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
    for (int i = 0; i < size_x; i++) {
        cpl_image_copy(correlated_noise_img, median_column_img, i+1, 1);
    }
    //espdr_msg("Correlated noise created, size: %lld x %lld",
    //          cpl_image_get_size_x(correlated_noise_img),
    //          cpl_image_get_size_y(correlated_noise_img));
    
    cpl_image_subtract(img, correlated_noise_img);
    
    //cpl_image_save(img, "NIRPS_corr_noise_corrected_image.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    cpl_image_delete(correlated_noise_img);
    cpl_image_unwrap(median_column_img);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_xtalk(cpl_image *img,
                           const cpl_image *mask_img,
                           espdr_inst_config *inst_config) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_image *tmp_img = cpl_image_duplicate(img);
    cpl_image *lowf_img = NULL;
    cpl_image *lowf_subtracted_img = NULL;
    cpl_image *curr_image = NULL;
    double *curr_data = NULL;
    int size_x = cpl_image_get_size_x(img);
    int size_y = cpl_image_get_size_y(img);
    double median_row = 0.0;
    cpl_image *median_full = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);

    double *masked_img_data = (double *)cpl_calloc(size_x*size_y, sizeof(double));
    const int *mask_data = NULL;
    cpl_image *masked_img = NULL;
    double *tmp_img_data = cpl_image_get_data_double(tmp_img);
    int nan_pixels_nb = 0;
    if (mask_img != NULL) {
        espdr_msg("masking orders");
        mask_data = cpl_image_get_data_int_const(mask_img);
        for (int i = 0; i < size_x * size_y; i++) {
            if (mask_data[i] == 1) {
                masked_img_data[i] = NAN;
                nan_pixels_nb++;
            } else {
                masked_img_data[i] = tmp_img_data[i];
            }
        }
        masked_img = cpl_image_wrap_double(size_x, size_y, masked_img_data);
    } else {
        espdr_msg("Not masking orders, orders mask not provided");
        masked_img = cpl_image_duplicate(tmp_img);
    }
    
    espdr_msg("Masked %d pixels with NANs", nan_pixels_nb);
    
    my_error = espdr_measure_background_prep(masked_img,
                                             256,
                                             256,
                                             &lowf_img,
                                             &lowf_subtracted_img);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_measure_background_ff failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("Background measured");
    
    //cpl_image_save(lowf_img, "NIRPS_background_256-256-median.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    //cpl_image_save(lowf_subtracted_img, "NIRPS_background_subtracted.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    //double *lowf_subtracted_data = cpl_image_get_data_double(lowf_subtracted_img);
    
    
    //cpl_image_save(masked_img, "NIRPS_masked_image.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    for (int i = 0; i < size_y; i++) {
        curr_image = cpl_image_extract(lowf_subtracted_img,
                                       1, i+1, size_x, i+1);
        curr_data = cpl_image_get_data_double(curr_image);
        median_row = espdr_get_median_off_nans(curr_data, size_x);
        cpl_image_fill_window(median_full, 1, i+1, size_x, i+1,
                              median_row);
    }
    
    //cpl_image_save(median_full, "NIRPS_median_full.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    my_error = cpl_image_subtract(img, median_full);
    
    cpl_image_delete(median_full);
    cpl_image_delete(lowf_img);
    cpl_image_delete(lowf_subtracted_img);
    
    //cpl_image_save(img, "NIRPS_xtalk_corrected_image.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_corr_amp(cpl_image *img) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int size_x = cpl_image_get_size_x(img);
    int size_y = cpl_image_get_size_y(img);
    
    cpl_vector *amp_first = cpl_vector_new_from_image_column(img, 1);
    cpl_vector *amp_last = cpl_vector_new_from_image_column(img, size_x);
    cpl_vector *median_first_vector = cpl_vector_new(size_y);
    cpl_vector *median_last_vector = cpl_vector_new(size_y);
    
    my_error = espdr_get_running_median_off_nans(amp_first, MEDIAN_WINDOW,
                                                 median_first_vector);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_running_median_off_nans failed for amp_first: %s",
                 cpl_error_get_message_default(my_error));
    my_error = espdr_get_running_median_off_nans(amp_last, MEDIAN_WINDOW,
                                                 median_last_vector);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_running_median_off_nans failed for amp_last: %s",
                 cpl_error_get_message_default(my_error));
    my_error = cpl_vector_add(median_first_vector, median_last_vector);
    my_error = cpl_vector_divide_scalar(median_first_vector, 2.0);
    
    double *median_amp = cpl_vector_get_data(median_first_vector);
    double *img_data = cpl_image_get_data_double(img);
    
    for (int col = 0; col < size_x/BOX_SIZE/2; col++) { // col = 0..15
        int left_col = col*2*BOX_SIZE+1;
        int right_col = (col+1)*2*BOX_SIZE;
        for (int row = 0; row < size_y; row++) {
            img_data[row * size_x + left_col - 1] -= median_amp[row];
            img_data[row * size_x + right_col - 1] -= median_amp[row];
        }
    }
    
    cpl_vector_delete(amp_first);
    cpl_vector_delete(amp_last);
    cpl_vector_delete(median_first_vector);
    cpl_vector_delete(median_last_vector);

    //cpl_image_save(img, "NIRPS_amps_corrected_image.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    // Additional correction on amplifiers, needed mainly when detector is reset
    double amp_median = 0.0;
    for (int amp = 0; amp < size_x/BOX_SIZE/2; amp++) { // amp = 0..15
        int col = amp*BOX_SIZE*2;
        cpl_vector *amp_left_in = cpl_vector_new_from_image_column(img, col+1); // 1, 257, ..., 3841
        cpl_vector *amp_left_out = cpl_vector_new_from_image_column(img, col+2); // 2, 258, ..., 3842
        cpl_vector *amp_left_median = cpl_vector_duplicate(amp_left_in);
        my_error = cpl_vector_subtract(amp_left_median, amp_left_out);
        double *amp_left_median_data = cpl_vector_get_data(amp_left_median);
        amp_median = espdr_get_median_off_nans(amp_left_median_data, cpl_vector_get_size(amp_left_median));
        int pxl;
        for (int row = 0; row < size_y; row++) {
            img_data[row * size_y + col] -= amp_median;
        }
        cpl_vector_delete(amp_left_in);
        cpl_vector_delete(amp_left_out);
        cpl_vector_delete(amp_left_median);
        
        cpl_vector *amp_right_in = cpl_vector_new_from_image_column(img, col+BOX_SIZE*2); // 256, 512, ..., 4096
        cpl_vector *amp_right_out = cpl_vector_new_from_image_column(img, col+BOX_SIZE*2-1); // 255, 511, ..., 4095
        cpl_vector *amp_right_median = cpl_vector_duplicate(amp_right_in);
        my_error = cpl_vector_subtract(amp_right_median, amp_right_out);
        double *amp_right_median_data = cpl_vector_get_data(amp_right_median);
        amp_median = espdr_get_median_off_nans(amp_right_median_data, cpl_vector_get_size(amp_right_median));
        for (int row = 0; row < size_y; row++) {
            img_data[row * size_y + col + BOX_SIZE*2-1] -= amp_median;
        }
        cpl_vector_delete(amp_right_in);
        cpl_vector_delete(amp_right_out);
        cpl_vector_delete(amp_right_median);
    }
    
    //cpl_image_save(img, "NIRPS_2nd_amps_corrected_image.fits",
    //               CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_update_bad_pixel_mask(cpl_frameset *raw_frameset,
                                           espdr_CCD_geometry *CCD_geom,
                                           cpl_image *mask) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    double *img_data = NULL;
    int *mask_data = NULL;
    const char* raw_name = NULL;
    cpl_image *curr_image = NULL;
    cpl_image *real_image = NULL;

    int nx = CCD_geom->exts[0].outputs[0][0].raw_nx;
    int ny = CCD_geom->exts[0].outputs[0][0].raw_ny;
    int pscan_nx = CCD_geom->exts[0].outputs[0][0].pscan_nx;
    int oscan_nx = CCD_geom->exts[0].outputs[0][0].oscan_nx;
    int ppscan_ny = CCD_geom->exts[0].outputs[0][0].ppscan_ny;
    int poscan_ny = CCD_geom->exts[0].outputs[0][0].poscan_ny;
    //espdr_msg("nx = %d\tny = %d\tpscan_nx = %d\toscan_nx = %d\tppscan_ny = %d\tposcan_ny = %d",
    //          nx, ny, pscan_nx, oscan_nx, ppscan_ny, poscan_ny);
    
    mask_data = cpl_image_get_data_int(mask);
    int masked_pixels_nb_before = 0;
    for (int i = 0 ; i < cpl_image_get_size_x(mask)*cpl_image_get_size_y(mask); i++) {
        if (mask_data[i] != 0) {
            masked_pixels_nb_before++;
        }
    }
    //espdr_msg("BAD PIXELS NB before updating: %d", masked_pixels_nb_before);
    
    int nset = cpl_frameset_get_size(raw_frameset);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(raw_frameset);
    cpl_frame *curr_frame = cpl_frameset_iterator_get(iter);
    
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        raw_name = cpl_frame_get_filename(curr_frame);
        curr_image = cpl_image_load(raw_name, CPL_TYPE_DOUBLE, 0, 1);
        //espdr_msg("curr image size: %lld x %lld",
        //          cpl_image_get_size_x(curr_image), cpl_image_get_size_y(curr_image));
        real_image = cpl_image_extract(curr_image,
                                       pscan_nx+1, ppscan_ny+1,
                                       nx - oscan_nx, ny - poscan_ny);
        //espdr_msg("real image size: %lld x %lld",
        //          cpl_image_get_size_x(real_image), cpl_image_get_size_y(real_image));
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Loading image failed: %s",
                     cpl_error_get_message_default(my_error));
        
        //espdr_msg("mask size: %lld x %lld",
        //          cpl_image_get_size_x(mask), cpl_image_get_size_y(mask));
        img_data = cpl_image_get_data_double(real_image);
        
        for (int i = 0; i < cpl_image_get_size_x(real_image)*cpl_image_get_size_y(real_image); i++) {
            if (isnan(img_data[i]) || isinf(img_data[i])) {
                mask_data[i] = PERMANENT_CCD_DEFECT;
            }
        }
        
        cpl_image_delete(real_image);
        cpl_image_delete(curr_image);
        
        cpl_frameset_iterator_advance(iter, 1);
        curr_frame = cpl_frameset_iterator_get(iter);
    }
    
    int masked_pixels_nb_after = 0;
    for (int i = 0 ; i < cpl_image_get_size_x(mask)*cpl_image_get_size_y(mask); i++) {
        if (mask_data[i] != 0) {
            masked_pixels_nb_after++;
        }
    }
    //espdr_msg("BAD PIXELS NB after updating: %d", masked_pixels_nb_after);
    
    espdr_msg("Number of pixels set as bad from the raw frame: %d",
              masked_pixels_nb_after - masked_pixels_nb_before);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_nans(cpl_image *img) {
    
    
    double *img_data = cpl_image_get_data_double(img);
    
    for (int i = 0; i < cpl_image_get_size_x(img)*cpl_image_get_size_y(img); i++) {
        if (isnan(img_data[i]) || isinf(img_data[i])) {
            img_data[i] = -OTHER_BAD_PIXEL;
        }
    }
    
    for (int i = 0; i < cpl_image_get_size_x(img)*cpl_image_get_size_y(img); i++) {
        if (isnan(img_data[i]) || isinf(img_data[i])) {
            espdr_msg(ANSI_COLOR_RED"This pixel %d is still nan or inf"ANSI_COLOR_RESET, i);
        }
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

double espdr_get_median_off_nans(double *data, int data_size) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i = 0, index = 0;
    int good_pixels_nb = 0;
    double *good_data = NULL;
    cpl_vector *good_data_vector = NULL;
    double median_off_nans = 0.0;
    
    for (i = 0; i < data_size; i++) {
        //if (!isnan(data[i]) && !isinf(data[i])) {
        if (!isnan(data[i])) {
            good_pixels_nb++;
        }
    }
    
    if (good_pixels_nb < 1) {
        return(0.0);
    }
    
    good_data = (double *) cpl_calloc (good_pixels_nb, sizeof(double));
    for (i = 0; i < data_size; i++) {
        //if (!isnan(data[i]) && !isinf(data[i])) {
        if (!isnan(data[i])) {
            good_data[index] = data[i];
            index++;
        }
    }
    
    good_data_vector = cpl_vector_wrap(good_pixels_nb, good_data);
    median_off_nans = cpl_vector_get_median(good_data_vector);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_vector_get_median failed for good_data_vector: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_vector_unwrap(good_data_vector);
    cpl_free(good_data);
    
    return (median_off_nans);
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param         recipe_id
 @param         list
 @param[out]    p
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_running_median_off_nans(cpl_vector *input_vector,
                                                 int median_window,
                                                 cpl_vector *median_vector) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_vector *tmp_vector = NULL;
    cpl_size data_size = cpl_vector_get_size(input_vector);
    double *median = cpl_vector_get_data(median_vector);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_vector_get_data failed for median_vector: %s",
                 cpl_error_get_message_default(my_error));

    for (int i = 0; i < data_size; i++) {
        if (i < median_window) {
            if ((i + median_window) >= data_size) {
                tmp_vector = cpl_vector_extract(input_vector,
                                                0, data_size-1, 1);
            } else {
                tmp_vector = cpl_vector_extract(input_vector,
                                                0, 2*i, 1);
            }
        } else {
            if ((i + median_window) >= data_size) {
                tmp_vector = cpl_vector_extract(input_vector,
                                                2*i - data_size + 1,
                                                data_size-1, 1);
            } else {
                tmp_vector = cpl_vector_extract(input_vector,
                                                i-median_window,
                                                i+median_window, 1);
            }
        }
        
        median[i] = espdr_get_median_off_nans(cpl_vector_get_data(tmp_vector),
                                              cpl_vector_get_size(tmp_vector));
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_get_median_off_nans failed for median[%d] (size: %lld): %s",
                     i, cpl_vector_get_size(tmp_vector),
                     cpl_error_get_message_default(my_error));
        
        cpl_vector_delete(tmp_vector);
    }
    
    //espdr_msg("running median result vector size: %lld",
    //          cpl_vector_get_size(median_vector));
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief  computes the background ilumination for flat field frames by measuring
 grid sectors over the image
 @param  input_image image under treatment
 @param  bkgr_grid_size_x grid size in X
 @param  bkgr_grid_size_y grid size in Y
 @param  minlevel_factor min factor to be apply in order to chose the min level
 flux inside the grid.
 @param  image_background_measured_RE resultant background image
 @param  image_background_substracted_RE resultant background substracted image
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_background_prep(const cpl_image *input_image,
                                             const int bkgr_grid_size_x,
                                             const int bkgr_grid_size_y,
                                             cpl_image **image_background_measured_RE,
                                             cpl_image **image_background_substracted_RE) {
    
    int n1 = cpl_image_get_size_x(input_image);
    int n2 = cpl_image_get_size_y(input_image);
    
    int size_x = bkgr_grid_size_x;
    int size_y = bkgr_grid_size_y;
    
    int dim[2];
    dim[0] = n1/2/size_x;
    dim[1] = n2/2/size_y;
    int *xc = (int *) cpl_calloc(dim[0],sizeof(int));
    int *yc = (int *) cpl_calloc(dim[1],sizeof(int));
    double *xcd = (double *) cpl_calloc(dim[0],sizeof(double));
    double *ycd = (double *) cpl_calloc(dim[1],sizeof(double));
    
    for (int i = 0; i < dim[0]; i++) {
        xc[i] = size_x + 2*i*size_x;
        xcd[i] = size_x + 2*i*size_x;
    }
    for (int j = 0; j < dim[1]; j++) {
        yc[j] = size_y + 2*j*size_y;
        ycd[j] = size_y + 2*j*size_y;
    }
    
    double *aa_vector = (double *) cpl_calloc((n1*n2),sizeof(double));
    double **aa = (double **) cpl_calloc(n1,sizeof(double *));
    
    for (int i = 0; i < n1; i++) {
        aa[i] = (double *)cpl_calloc((n2),sizeof(double));
    }
    
    double **minlevel = (double **) cpl_calloc(dim[1],sizeof(double *));
    
    int xx = 0;
    int yy = 0;
    int llx, lly, urx, ury;
    double *grid_data;
    cpl_image *grid = NULL;
    for (int col = 0; col < dim[1]; col++) {
        
        minlevel[col] = (double *)cpl_calloc(dim[0],sizeof(double));
        lly = yy+1;
        ury = yy+(2*size_y);
        xx = 0;
        
        for (int lin = 0; lin < dim[0]; lin++) {
            
            llx = xx+1;
            urx = xx+(2*size_x);
            
            grid = cpl_image_extract(input_image, llx, lly, urx, ury);
            
            grid_data = cpl_image_get_data_double(grid);
            minlevel[col][lin] = espdr_get_median_off_nans(grid_data,
                                    cpl_image_get_size_x(grid)*cpl_image_get_size_y(grid));
            cpl_image_delete(grid);
            
            xx = urx;
        }
        yy = ury;
    }
    
    double *bkgr = (double *) cpl_calloc(dim[0],sizeof(double));
    double *y2 = (double *) cpl_calloc(n1,sizeof(double));
    double *x2 = (double *) cpl_calloc(n1,sizeof(double));
    
#pragma omp parallel for default(none) shared(n1,x2)
    for (int i = 0; i < n1; i++) {
        x2[i] = (double)i;
    }
    
    for (int j = 0; j < dim[1]; j++) {
        
        cpl_error_code my_error = CPL_ERROR_NONE;
        my_error = espdr_spline_bounded_interpolation(xcd,
                                                      minlevel[j],
                                                      dim[0],
                                                      x2, y2, n1, 0, n1);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_spline_bounded_interpolation failed: %s",
                     cpl_error_get_message_default(my_error));
        
#pragma omp parallel for default(none) shared(n1,y2,yc,aa,j)
        for (int i = 0; i < n1; i++) {
            aa[i][yc[j]] = y2[i];
        }
    }
    
    cpl_free(bkgr);
    cpl_free(y2);
    cpl_free(x2);
    
    bkgr = (double *) cpl_calloc(dim[1],sizeof(double));
    y2 = (double *) cpl_calloc(n2,sizeof(double));
    x2 = (double *) cpl_calloc(n2,sizeof(double));
    
#pragma omp parallel for default(none) shared(n2,x2)
    for (int j = 0; j < n2; j++) {
        x2[j] = (double)j;
    }
    
    for (int i = 0; i < n1; i++) {
        for (int j = 0; j < dim[1]; j++) {
            bkgr[j] = aa[i][yc[j]];
        }
        
        cpl_error_code my_error = CPL_ERROR_NONE;
        my_error = espdr_spline_bounded_interpolation(ycd, bkgr, dim[1],
                                                      x2, y2, n2, 0, n2);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_spline_bounded_interpolation failed: %s",
                     cpl_error_get_message_default(my_error));
        
#pragma omp parallel for default(none) shared(n2,y2,yc,aa,i)
        for (int j = 0; j < n2; j++) {
            aa[i][j] = y2[j];
        }
    }
    
    cpl_free(bkgr);
    cpl_free(y2);
    cpl_free(x2);
    
    int ind = 0;
    for (int j = 0; j < n2; j++) {
        for (int i = 0; i < n1; i++) {
            aa_vector[ind] = aa[i][j];
            ind++;
        }
    }
    cpl_image* ima_tmp = cpl_image_wrap_double(n1, n2, aa_vector);
    *image_background_measured_RE = cpl_image_duplicate(ima_tmp);
    cpl_image_unwrap(ima_tmp);
    
    *image_background_substracted_RE = cpl_image_duplicate(input_image);
    cpl_image_subtract(*image_background_substracted_RE,
                       *image_background_measured_RE);
    
    for (int j = 0; j < dim[1]; j++) {
        cpl_free(minlevel[j]);
    }
    cpl_free(minlevel);
    
    for (int i = 0; i < n1; i++) {
        cpl_free(aa[i]);
    }
    cpl_free(aa);
    cpl_free(xc);
    cpl_free(yc);
    cpl_free(xcd);
    cpl_free(ycd);
    cpl_free(aa_vector);
    
    return cpl_error_get_code();
}


