/*                                                                           a
 *   This file is part of the ESO X-shooter Pipeline                         *
 *   Copyright (C) 2006 European Southern Observatory                        *
 *                                                                           *
 *   This library is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the Free Software             *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA    *
 *                                                                           */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-05 17:38:50 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */
#ifndef ESPDR_UTILS_H
#define ESPDR_UTILS_H

/*----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/


#include <cpl.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <espdr_CCD.h>
#include <libgen.h>
#include <espdr_instrument.h>
#include <espdr_keywords.h>

#include <gsl/gsl_vector.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_cblas.h>
#include <gsl/gsl_multifit_nlin.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_min.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_multifit.h>

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/
#define MAXIMUM(x, y) ((x) > (y)) ? (x) : (y)
/* Caveat: multiple evalutation of arguments */

#ifndef M_PI
#define M_PI 3.1415926535897932384626433832795
#endif

#define LIGHT_SPEED 299792458.0 // [m/s]
#define GRAVITATIONAL_CONSTANT 6.6743e-11 // [m^3/kg/s^2]
#define PLANCK_CONSTANT 6.62607004e-34 // [Js]
#define M_SUN 1.9884e30 // [kg]
#define M_EARTH 5.9722e24 // [kg]
#define R_EARTH 6371000.0 // [m]

#define N_AIR_COEFF_T 15.0
#define N_AIR_COEFF_P 760.0

#define NB_SEC_IN_HOUR 3600.0
#define NB_MC_IN_CM 10000
#define DBL_REPLACING_NAN -999.0
#define RIGHT_ANGLE_DEG 90

#define KEYWORD_LENGTH 64
#define PREFIX_LENGTH 64
#define FILENAME_LENGTH 512
#define TAG_LENGTH 32
#define COMMENT_LENGTH 1024
#define PARAM_NAME_LENGTH 256

#define PRO_CATG "ESO PRO CATG"
#define ARCFILE "ARCFILE"
#define RECIPE "ESO PRO REC1 ID"
#define REF_ARC	"ARC REF"

#define SAVE_DEBUG_PRODUCT 1

#define ESPDR_MAX(A,B) A > B ? A : B
#define ESPDR_MIN(A,B) A > B ? B : A

#define espdr_ensure(CONDITION, ERROR_CODE, ...) \
		if (CONDITION) { \
			espdr_msg_error(__VA_ARGS__); \
			cpl_ensure(0, ERROR_CODE, ERROR_CODE); }

#define espdr_ensure_code(CONDITION, ERROR_CODE, TEXT) \
		if (CONDITION) { \
			espdr_msg_error(TEXT); \
			cpl_ensure_code(0, ERROR_CODE); }

#define ESPDR_MALLOC( POINTER, TYPE, SIZE) \
		assure(POINTER == NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
				"Try to allocate non NULL pointer");\
				POINTER = (TYPE*)(cpl_malloc(SIZE*sizeof(TYPE)));\
				assure (POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory allocation failed!")

#define ESPDR_CALLOC( POINTER, TYPE, SIZE) \
		assure(POINTER == NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
				"Try to allocate non NULL pointer");\
				POINTER = (TYPE*)(cpl_calloc(SIZE,sizeof(TYPE)));\
				assure (POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory allocation failed!")

#define ESPDR_REALLOC( POINTER, TYPE, SIZE ) \
		assure(POINTER != NULL, CPL_ERROR_ILLEGAL_INPUT,\
				"Try to re-allocate NULL pointer") ;\
				POINTER = (TYPE *)cpl_realloc(POINTER,SIZE*sizeof(TYPE)));\
				assure( POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory re-allocation failed!")

#define ESPDR_NEW_PROPERTYLIST( POINTER) \
		assure(POINTER == NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
				"Try to allocate non NULL pointer");\
				POINTER = cpl_propertylist_new();\
				assure (POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory allocation for propertylist failed!")

#define ESPDR_NEW_FRAME( POINTER) \
		assure(POINTER == NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
				"Try to allocate non NULL pointer");\
				POINTER = cpl_frame_new();\
				assure (POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory allocation for frame failed!")

#define ESPDR_NEW_FRAMESET( POINTER) \
		assure(POINTER == NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
				"Try to allocate non NULL pointer");\
				POINTER = cpl_frameset_new();\
				assure (POINTER != NULL, CPL_ERROR_ILLEGAL_OUTPUT,\
						"Memory allocation for frameset failed!")


#define ESPDR_FREE(POINTER)\
		if(POINTER!=NULL) cpl_free(POINTER);\
		POINTER = NULL

#define ESPDR_TABLE_NEW_COL(TABLE, NAME, UNIT, TYPE) \
		check( cpl_table_new_column(TABLE, NAME, TYPE));\
		check( cpl_table_set_column_unit( TABLE, NAME, UNIT))

#define BOOLEAN_TO_STRING( boolean) \
		boolean == 0 ? "false" : "true"

enum {
	ESPDR_DEBUG_LEVEL_NONE, ESPDR_DEBUG_LEVEL_LOW,
	ESPDR_DEBUG_LEVEL_MEDIUM, ESPDR_DEBUG_LEVEL_HIGH
} ;


/*----------------------------------------------------------------------------
 Prototypes
 ----------------------------------------------------------------------------*/

cpl_error_code espdr_check_error_code(const char* func_id);

cpl_error_code espdr_max_flux_QC(espdr_inst_config *inst_config,
                                 espdr_CCD_geometry *CCD_geom,
                                 espdr_qc_keywords *qc_kws,
                                 double *max_flux,
                                 char *global_QC_kw,
                                 cpl_propertylist *keywords_RE);

cpl_error_code espdr_add_qc_airm_info(cpl_propertylist** plist);
cpl_error_code espdr_add_qc_iwv_info(cpl_propertylist** plist);
int *espdr_get_ord_ref(cpl_propertylist *keywords, const char *recipe);

cpl_error_code
espdr_add_qc_key_stat_ext(cpl_propertylist* keywords, const int ext_nb,
		         const char* key_val,const char** key_suff,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum,
		         cpl_propertylist*** kext);
cpl_error_code
espdr_add_qc_key_stat_pri(cpl_propertylist** keywords, const int ext_nb,
		         const char* key_val,const char** key_suff,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum);
cpl_error_code
espdr_add_qc_key_stat_ord_pri(cpl_propertylist** keywords, const int ext_nb,
		         const char* key_val,const char* expr_reject,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum);

cpl_error_code
espdr_add_qc_key_stat_ord(cpl_propertylist* keywords, const int ext_nb,
		         const char* key_val,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum,
		         cpl_propertylist*** kext);

cpl_error_code
espdr_add_qc_key_orderdef_ord_ref(cpl_propertylist* keywords, const int ext_nb,
			 const int* ord_ref, const char* key_val,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum,
		         cpl_propertylist*** kext);
cpl_error_code
espdr_add_qc_key_flat_ord_ref(cpl_propertylist** keywords, const int ext_nb,
			 const int* ord_ref, const char* key_val,const char** key_suffix,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum);
cpl_error_code
espdr_add_qc_key_contam_ord_ref(cpl_propertylist** keywords, const int ext_nb,
			 const int* ord_ref, const char* key_val,const char** key_suffix,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum);

cpl_error_code
espdr_add_qc_key_effab_ord_ref(cpl_propertylist** keywords,const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum);
cpl_error_code
espdr_add_qc_key_wave_FP_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum);

cpl_error_code
espdr_add_qc_key_wave_THAR_ord_ref(cpl_propertylist** keywords,const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum);

cpl_error_code
espdr_add_qc_key_wave_LFC_ord_ref(cpl_propertylist** keywords,
				  const int* orders_fit, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum);


cpl_error_code
espdr_add_qc_key_flux_ord_ref(cpl_propertylist** keywords, const int ext_nb,
			 const int* ord_ref, const char* key_val,const char** key_suffix,
			   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
			   const cpl_boolean compute_min,const cpl_boolean compute_max,
			   const cpl_boolean compute_sum);
double
espdr_drift_vel(const double drift_mean, const double mjd_obs_delta_time_wave_a);

char * espdr_get_basename(const char *filename);
int espdr_poisson_random(const double expectedValue);

int espdr_print_rec_status(const int val);

const char *espdr_get_license(void);

cpl_error_code espdr_check_input_tags(cpl_frameset *set,
                                      const char** tags,
                                      int* is_required,
                                      const int ntags);

cpl_error_code espdr_check_input_inst_config(cpl_frameset *set);

cpl_frame* espdr_get_inst_config(cpl_frameset *set);

cpl_error_code espdr_check_input_flat(cpl_frameset *set,
                                      int blaze_flag);

cpl_error_code espdr_image_merge_2_dim(const cpl_imagelist *images_to_merge,
		const espdr_CCD_geometry *CCD_geom,
		const cpl_type type,
		cpl_imagelist **merged_image_RE);

cpl_error_code espdr_verify_fits_image(const char *FITS_filename);

cpl_error_code espdr_verify_imagelist(const cpl_imagelist *input_imagelist);

cpl_error_code espdr_create_mask(const cpl_image *input_image,
		const int total_size,
		const double reject_low,
		const double reject_high,
		const int mask_bit,
		cpl_image **mask_RE);

cpl_error_code espdr_RON_compute_old(cpl_imagelist *input_imagelist,
		double ksigma_low,
		double ksigma_high,
		const char *sigma_clipping_method,
		int iter_max,
		int reject_zeroMAD,
		double forced_MAD,
		double *RON_RE,
		double *mean_RE);

/* AMO added */
cpl_error_code espdr_RON_compute(const cpl_frameset *input_set,
		const espdr_CCD_geometry *CCD_geom,
		const double ksigma_low,
		const double ksigma_high,
		const char *sigma_clipping_method,
		const int iter_max,
		const int reject_zeroMAD,
		const double forced_MAD,
		double *RON_RE,
		double *mean_RE);
/************/

cpl_error_code espdr_compute_snr(cpl_image *s2d_flux,
                                 cpl_image *sd2_error,
                                 int snr_window,
                                 double *snr);

int espdr_get_output_index_for_pixel(const int ext_nr,
		const int pixel,
		const espdr_CCD_geometry *CCD_geom);

int espdr_get_ext_index_for_order(const int order,
		const int fibre_nr,
		const espdr_inst_config *inst_config,
		const espdr_CCD_geometry *CCD_geom);

int espdr_get_order_nr_in_ext(const int order,
		const int fibre_nr,
		const int ext_nr,
		const espdr_inst_config *inst_config,
		const espdr_CCD_geometry *CCD_geom);

int espdr_get_first_order(const int fibre_nr,
                          const int ext_nr,
                          const espdr_inst_config *inst_config,
                          const espdr_CCD_geometry *CCD_geom);

int espdr_get_last_order(const int fibre_nr,
                         const int ext_nr,
                         const espdr_inst_config *inst_config,
                         const espdr_CCD_geometry *CCD_geom);

cpl_error_code espdr_lambda_pixel_size(const double *wave,
		const int length,
		double *pixel_size_RE);

int getposition(char* c);

cpl_error_code espdr_get_min_flux_one_output(const cpl_image *input_image,
		const cpl_image *pixel_mask,
		const double cosmics_bounds,
		double *min_flux_RE);

double min_val(const double* array, const int size, int *index_RE);
double max_val(const double* array, const int size);
double find_val(const double* array, const int size, const double key, int *index_RE);
double convert_coord_to_dbl(const char *coord_read);
double convert_HHMMSS_to_HH(double ra);
double convert_DDMMSS_to_DD(double dec);	

const char *espdr_get_mask_name(const char *spec_type_id,
		const cpl_table *mask_lut);

char *espdr_get_base_file_name(const char *filename);

const char *espdr_get_default_spec_type(const espdr_inst_config *inst_config);

cpl_error_code espdr_split_by_slices(const cpl_image *input_s2d,
		const int slices_nb,
		cpl_imagelist **s2d_sliced_RE);

cpl_error_code espdr_group_slices(const cpl_imagelist *s2d_sliced,
		const int slices_nb,
		cpl_image **s2d_RE);

cpl_error_code espdr_match_orders(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE);

cpl_error_code espdr_match_orders_quality_map(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE);

cpl_error_code espdr_match_orders_drift_matrix(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE);

double espdr_compute_n_air(const double lambda, const double t, const double p);

cpl_error_code
espdr_copy_qc_ext_key_to_ext(cpl_propertylist* keywords,
		cpl_propertylist*** kext, const int ext_nb);

#endif
