/* $Id: espdr_bp_detect-test.c,v 1.2 2013-07-26 11:44:44 amodigli Exp $
 *
 * This file is part of the ESPDR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: amodigli $
 * $Date: 2013-07-26 11:44:44 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>
#include <math.h>
#include <espdr_sigma_clipping.h>
#include <espdr_pixels.h>
#include <espdr_utils.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_bp_detect_test  Unit test to verify bad pixel detection
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/

/* test bp detection on a bias image */
cpl_error_code
test_detect_trap(void)
{
    cpl_image* ima;
    cpl_image* map;

    cpl_size sx=100;
    cpl_size sy=100;
    double value=100.;
    double rms=sqrt(value);
    double kappa=10;
    int maxiter=100;
    int reject_zero_MAD=1;
    //cpl_error_code my_error = CPL_ERROR_NONE;
    const char* method="median";
    double mean;
    int hot_pixels;
    double sigma;
    //int* pmap=NULL;
    double reject_low;
    double reject_high;
    double stdev;
   
    double center;
    double* data;
    int i;
    cpl_vector* vec;
    ima=cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);
    map=cpl_image_new(sx,sy,CPL_TYPE_INT);

    cpl_image_fill_noise_uniform(ima, value-rms,value+rms);
    stdev=cpl_image_get_stdev(ima);
   
    espdr_msg_warning("value=%g rms=%g stdev=%g",value,rms,stdev);

    data=cpl_image_get_data(ima);

   for(i=0;i<sx;i++) {
      data[sy/2*sx+i]=140.;
   }

    //cpl_image_save(ima,"ima.fits",CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);

    /*
    pmap=cpl_image_get_data_int(map);
    espdr_sigma_clipping(cpl_image_get_data_double(ima), sx*sy, kappa,
                    method, maxiter, reject_zero_MAD,&mean, &hot_pixels, &sigma,
                    pmap);
    */
    data=cpl_image_get_data_double(ima);
    vec=cpl_vector_wrap(sx*sy,data);
    espdr_sig_clip(vec, kappa,kappa, method, maxiter, reject_zero_MAD,1,
                   &center, &sigma, &hot_pixels, &mean, &stdev, &reject_low,&reject_high);
    cpl_vector_unwrap(vec);
    espdr_create_mask(ima,sx*sy,reject_low,reject_high,HOT_PIXEL,&map);

    //cpl_image_save(map,"map.fits",CPL_TYPE_INT,NULL,CPL_IO_CREATE);

    cpl_image_delete(ima);
    cpl_image_delete(map);

    return cpl_error_get_code();
}


/* test bp detection on a bias image */
cpl_error_code
test_detect_hot_pix(void)
{
    cpl_image* ima;
    cpl_image* map;

    cpl_size sx=100;
    cpl_size sy=100;
    double value=100.;
    double rms=sqrt(value);
    double kappa=10;
    int maxiter=100;
    int reject_zero_MAD=1;
    //cpl_error_code my_error = CPL_ERROR_NONE;
    const char* method="median";
    double mean;
    int hot_pixels;
    double sigma;
    //int* pmap=NULL;
    double reject_low;
    double reject_high;
    double stdev;
    double fudge=0.8596;/* 0.8597 get it 0.8596 not */
    double outlier;
    double center;
    double* data;
    cpl_vector* vec;
    ima=cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);
    map=cpl_image_new(sx,sy,CPL_TYPE_INT);

    cpl_image_fill_noise_uniform(ima, value-rms,value+rms);
    stdev=cpl_image_get_stdev(ima);
    outlier=value+fudge*kappa*stdev;
    espdr_msg_warning("value=%g rms=%g stdev=%g",value,rms,stdev);

    cpl_image_set(ima,10,10,outlier);
    //cpl_image_save(ima,"ima.fits",CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);

    /*
    pmap=cpl_image_get_data_int(map);
    espdr_sigma_clipping(cpl_image_get_data_double(ima), sx*sy, kappa,
                    method, maxiter, reject_zero_MAD,&mean, &hot_pixels, &sigma,
                    pmap);
    */
    data=cpl_image_get_data_double(ima);
    vec=cpl_vector_wrap(sx*sy,data);
    espdr_sig_clip(vec, kappa,kappa, method, maxiter, reject_zero_MAD,1,
                   &center, &sigma, &hot_pixels, &mean, &stdev, &reject_low,&reject_high);
    cpl_vector_unwrap(vec);
    espdr_create_mask(ima,sx*sy,reject_low,reject_high,HOT_PIXEL,&map);

    //cpl_image_save(map,"map.fits",CPL_TYPE_INT,NULL,CPL_IO_CREATE);

    cpl_image_delete(ima);
    cpl_image_delete(map);

    return cpl_error_get_code();
}


static cpl_error_code espdr_process_ext(const char* iname,const int ext,const double kappa)
{

    cpl_image* ima;
    cpl_image* map;
    //int* pmap=NULL;

    double center;
    double stdev;
    int sx;
    int sy;
    int maxiter;
    const char* method="median";
    int reject_zero_MAD=1;
    char oname[80];
    double mean;
    int hot_pixels;
    double sigma;
    const char* fname;
    double* data=NULL;
    double reject_low;
    double reject_high;
    cpl_vector* vec=NULL;

    ima = cpl_image_load(iname, CPL_TYPE_DOUBLE, 0, ext);

    sx=cpl_image_get_size_x(ima);
    sy=cpl_image_get_size_y(ima);
    maxiter=sx*sy;

    map=cpl_image_new(sx,sy,CPL_TYPE_INT);


    espdr_msg("kappa=%g",kappa);
    /*
    pmap=cpl_image_get_data_int(map);
    espdr_sigma_clipping(cpl_image_get_data_double(ima), sx*sy, kappa,
                         method, maxiter, reject_zero_MAD,&mean, &hot_pixels, &sigma,
                         pmap);
    */
    data=cpl_image_get_data_double(ima);
    vec=cpl_vector_wrap(sx*sy,data);
    espdr_sig_clip(vec, kappa, kappa,
                      method, maxiter, reject_zero_MAD,1, &center, &sigma, &hot_pixels, &mean, &stdev,
                      &reject_low,&reject_high);
    cpl_vector_unwrap(vec);
    espdr_create_mask(ima,sx*sy,reject_low,reject_high,1,&map);

    fname=espdr_get_basename(iname);

    sprintf(oname,"ima_ext_%d_kappa_%3.1f_%s",ext,kappa,fname);
    espdr_msg("oname=%s",oname);
    //cpl_image_save(ima,oname,CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);
    sprintf(oname,"map_ext_%d_kappa_%3.1f_%s",ext,kappa,fname);
    //cpl_image_save(map,oname,CPL_TYPE_INT,NULL,CPL_IO_CREATE);


    cpl_image_delete(ima);
    cpl_image_delete(map);

    return cpl_error_get_code();
}


/* test bp detection on a bias image (external files) */
cpl_error_code
test_detect_bias(int argc, char* argv[])
{

    double kappa=10;
    int next=0;
    const char* iname=NULL;
    espdr_msg("argc=%d",argc);
    if((size_t)argc>1){
        kappa=atof(argv[1]);
    }
    espdr_msg("kappa=%g",kappa);
    for (size_t i = 2; i < (size_t)argc; i++) {
        cpl_frame * frm = cpl_frame_new();
        iname=argv[i];
        cpl_frame_set_filename(frm, iname);
        next=cpl_frame_get_nextensions(frm);
        cpl_frame_delete(frm);
        if(next>0) {
            for(int j=0;j<next;j++) {
                espdr_process_ext(iname,j+1,kappa);
            }
        } else {
            espdr_process_ext(iname,0,kappa);
        }
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of espdr_bp_detect module
 */
/*----------------------------------------------------------------------------*/

int main(int argc, char* argv[])
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    test_detect_hot_pix();
    test_detect_trap();
    test_detect_bias(argc,argv);

    return cpl_test_end(0);
}

/**@}*/
