/* $Id: espdr_dfs-test.c,v 1.2 2013-07-26 11:44:44 amodigli Exp $
 *
 * This file is part of the ESPDR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: amodigli $
 * $Date: 2013-07-26 11:44:44 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>

#include <espdr_sigma_clipping.h>
#include <espdr_pixels.h>
#include <espdr_dark.h>
#include <espdr_hdrl_func.h>
#include <espdr_msg.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_dfs_test  Unit test of espdr_dfs
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of espdr_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/
static void test_sig_clip(void)
{
	cpl_vector *test_data_vector = NULL;
	int data_length = 5;
	static double test_data[5] = {10.0, 11.0, 12.0, 13.0, 14.0};
	static double test_data_with_cosmics[5] = {10.0, 11.0, 12.0, 133.0, 14.0};
	double ksig_low = 4.0;
	double ksig_high = 4.0;
	const char *method = "mean";
	int maxiter = 5.0;
	int reject_zeroMAD = 1;
    double forced_MAD = 1.0;
	double CoD = 0.0;
	double sigma = 0.0;
	int cosmics = 0;
	double mean = 0.0;
	double stddev = 0.0;
	double reject_low = 0.0;
	double reject_high = 0.0;
	double accuracy=DBL_EPSILON;

	
    /* Test with invalid input */
	espdr_sig_clip(NULL, ksig_low, ksig_high,
                       method, maxiter, reject_zeroMAD, forced_MAD, 
					   &CoD, &sigma, &cosmics, &mean, &stddev, 
					   &reject_low, &reject_high);

	cpl_test_error(CPL_ERROR_NULL_INPUT);

	test_data_vector = cpl_vector_wrap(data_length, test_data);
	
	espdr_sig_clip(test_data_vector, ksig_low, ksig_high,
                       "abc", maxiter, reject_zeroMAD, forced_MAD,
					   &CoD, &sigma, &cosmics, &mean, &stddev, 
					   &reject_low, &reject_high);

	cpl_test_error(CPL_ERROR_TYPE_MISMATCH);
	cpl_vector_unwrap(test_data_vector);
    /* Test with valid input */
    {
	    test_data_vector = cpl_vector_wrap(data_length, test_data);
		espdr_sig_clip(test_data_vector, ksig_low, ksig_high,
                           "mean", maxiter, reject_zeroMAD, forced_MAD,
						   &CoD, &sigma, &cosmics, &mean, &stddev, 
						   &reject_low, &reject_high);
		cpl_test_error(CPL_ERROR_NONE);


		cpl_test_abs(CoD , 12.0, accuracy);
		cpl_test_abs(mean , 12.0, accuracy);
		cpl_test_abs(sigma, 1.5811, 4.e-5);
		cpl_test_abs(stddev, 1.5811, 4.e-5);

		espdr_sig_clip(test_data_vector, ksig_low, ksig_high,
                           "median", maxiter, reject_zeroMAD, forced_MAD,
						   &CoD, &sigma, &cosmics, &mean, &stddev, 
						   &reject_low, &reject_high);
		cpl_test_error(CPL_ERROR_NONE);

		
		cpl_test_abs(CoD , 12.0, accuracy);
		cpl_test_abs(mean , 12.0, accuracy);
		cpl_test_abs(sigma, CPL_MATH_STD_MAD, accuracy);
		cpl_test_abs(stddev, 1.5811, 4e-5);

		cpl_vector_unwrap(test_data_vector);
		
		test_data_vector = cpl_vector_wrap(data_length, test_data_with_cosmics);		
		
		espdr_sig_clip(test_data_vector, ksig_low, ksig_high,
                           "mean", maxiter, reject_zeroMAD, forced_MAD,
						   &CoD, &sigma, &cosmics, &mean, &stddev, 
						   &reject_low, &reject_high);

	    cpl_test_error(CPL_ERROR_NONE);
		

		//espdr_msg("CoD: %lf, sigma: %lf, mean: %lf, stddev: %lf, cosmics: %d", 
		//		  CoD, sigma, mean, stddev, cosmics);
		
		cpl_test_abs(CoD , 36.0, accuracy);
		cpl_test_abs(sigma, 54.2448, 4e-5);
		cpl_test_abs(mean , 36.0, accuracy);
		cpl_test_abs(stddev, 54.2448, 4e-5);
		cpl_test_eq(cosmics, 0);
		
		espdr_sig_clip(test_data_vector, ksig_low, ksig_high,
                           "median", maxiter, reject_zeroMAD, forced_MAD,
						   &CoD, &sigma, &cosmics, &mean, &stddev, 
						   &reject_low, &reject_high);

	    cpl_test_error(CPL_ERROR_NONE);

		//espdr_msg("CoD: %lf, sigma: %lf, mean: %lf, stddev: %lf, cosmics: %d", 
		//		  CoD, sigma, mean, stddev, cosmics);
		
		cpl_test_abs(CoD , 11.5, accuracy);
		cpl_test_abs(sigma, CPL_MATH_STD_MAD, accuracy);
		cpl_test_abs(mean , 11.75, accuracy);
		cpl_test_abs(stddev, 1.7078,3e-5);
		cpl_test_eq(cosmics, 1);
		
    }
    
	cpl_vector_unwrap(test_data_vector);
	
    return;
}

/* test bp detection on a bias image (external files) */
cpl_error_code
test_bp_detect_stack(int argc, char* argv[])
{

    double kappa=10,ron=1;
    int niter=3,next,cosmics,k=0;
    cpl_size nx=0,ny=0;
    const char* method="median";
    const char* iname=NULL;
    cpl_image* img=NULL;
    cpl_image* master=NULL;
    cpl_imagelist* ilist=NULL;
    cpl_frame* frm=NULL;

    espdr_msg_warning("argc=%d",argc);
    if((size_t)argc==1){
        return cpl_error_get_code();
    }
    if((size_t)argc>1){
        kappa=atof(argv[1]);
        niter=atoi(argv[2]);
        method=argv[3];
    }
    espdr_msg_warning("kappa=%g miter=%d method=%s",kappa,niter,method);

    /* load input data */
    ilist=cpl_imagelist_new();
    for (size_t i = 4; i < (size_t)argc; i++) {
        iname=argv[i];
        frm=cpl_frame_new();
        cpl_frame_set_filename(frm, iname);
        next=cpl_frame_get_nextensions(frm);
        cpl_frame_delete(frm);

        if(next>0) {
            img=cpl_image_load(iname,CPL_TYPE_DOUBLE,0,1);
        } else {
            img=cpl_image_load(iname,CPL_TYPE_DOUBLE,0,0);
        }

        nx=cpl_image_get_size_x(img);
        ny=cpl_image_get_size_y(img);
        cpl_imagelist_set(ilist,img,k);
        k++;
    }

    //cpl_imagelist_save(ilist,"pippo.fits", CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);

    /* master creation: ESPRESSO based computation */
    espdr_stack_sigma_one_output(ilist, nx, ny, kappa, method, niter, &master,
                                 &ron, &cosmics);
    espdr_msg_warning("cosmics=%d",cosmics);
    //cpl_image_save(master, "master.fits", CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);

    /* hot-pixel determination: ESPRESSO based */
    double forced_MAD=0.01;
    double CoD=0;
    double sigma=0;
    cpl_vector* vector;
    int hot_pixels=0;
    double mean=0;
    double stddev=0;
    double low_limit=0;
    double high_limit=0;
    cpl_image* hp_image=NULL;
    vector = cpl_vector_wrap(nx * ny,cpl_image_get_data_double(master));
    espdr_sig_clip(vector,kappa,kappa,method,niter, 1, forced_MAD,&CoD, &sigma,
                   &hot_pixels,&mean, &stddev,&low_limit, &high_limit);

    espdr_msg_warning("CoD=%g sigma=%g hot_pixels=%d mean=%g stdev=%g low_limit=%g high_limit=%g",
                      CoD,sigma,hot_pixels,mean,stddev,low_limit,high_limit);

    hp_image = cpl_image_new(nx, ny,CPL_TYPE_INT);
    espdr_create_mask(master,nx * ny,low_limit, high_limit, HOT_PIXEL,&hp_image);
    //cpl_image_save(hp_image, "mask.fits", CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);

    cpl_vector_unwrap(vector);
    cpl_image_delete(master);
    cpl_image_delete(hp_image);

    /* master abd bad pixel map creation: HDRL based computation */
    espdr_hdrl_master_port(ilist,kappa,kappa,niter,&master,&ron,&cosmics);
    espdr_msg_warning("cosmics=%d",cosmics);
    //cpl_image_save(master, "master_hdrl.fits", CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);


    cpl_imagelist_delete(ilist);
    cpl_image_delete(master);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of espdr_dfs module
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[])
{
	
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_sig_clip();
    test_bp_detect_stack(argc, argv);

    return cpl_test_end(0);
}

/**@}*/
