/*-----------------------------------------------------------------*/
/*! 
  \file calceph.h 
  \brief public API for calceph library
        access and interpolate INPOP and JPL Ephemeris data.

  \author  M. Gastineau, H. Manche
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris. 

   Copyright, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, CNRS
   email of the author : gastineau@imcce.fr, manche@imcce.fr

  History:                                                                
    \note M. Gastineau 12/10/10 : add custom error handler     
    \note M. Gastineau 14/06/11 : add calceph_open_array     
    \note M. Gastineau 22/08/11 : add calceph_compute_unit
    \note M. Gastineau 05/06/13 : replace <nul> character by space in 
    f90calceph_(s)getconstantindex 
    add the functions  f2003calceph_(s)getconstantindex
*/
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses 
 below to apply on this file.
 
    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )
   
 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)
  
 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )
 

This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under 
French law and abiding by the rules of distribution of free software.  
You can  use, modify and/ or redistribute the software under the terms 
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA  
at the following URL "http://www.cecill.info". 

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability. 

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or 
data to be ensured and,  more generally, to use and operate it in the 
same conditions as regards security. 

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/

#include "calcephconfig.h"
#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_ERRNO_H
#include <errno.h>
#endif
#define __CALCEPH_WITHIN_CALCEPH 1
#include "calceph.h"
#include "real.h"
#include "util.h"
#include "calcephinternal.h"

#if __STRICT_ANSI__
#define inline 
#endif

int FC_FUNC_(f90calceph_sopen, F90CALCEPH_SOPEN) (char *filename, int len);

void FC_FUNC_(f90calceph_sclose, F90CALCEPH_SCLOSE) (void);

int FC_FUNC_(f90calceph_scompute, F90CALCEPH_SCOMPUTE) (double *JD0, double *time, int *target, int *center,
                                                        double PV[6]);
int FC_FUNC_(f90calceph_sgetconstant, F90CALCEPH_SGETCONSTANT) (char *name, double *value, int len);

int FC_FUNC_(f90calceph_sgetconstantcount, F90CALCEPH_SGETCONSTANTCOUNT) (void);

int FC_FUNC_(f90calceph_sgetconstantindex, F90CALCEPH_SGETCONSTANTINDEX) (int *index,
                                                                          char name[CALCEPH_MAX_CONSTANTNAME],
                                                                          double *value);
int FC_FUNC_(f90calceph_open, F90CALCEPH_OPEN) (long long *eph, char *filename, int len);

int FC_FUNC_(f90calceph_open_array, F90CALCEPH_OPEN_ARRAY) (long long *eph, int *n, char *filename, int *len);

void FC_FUNC_(f90calceph_close, F90CALCEPH_CLOSE) (long long *eph);

int FC_FUNC_(f90calceph_prefetch, F90CALCEPH_PREFETCH) (long long *eph);

void FC_FUNC_(f90calceph_seterrorhandler, F90CALCEPH_SETERRORHANDLER) (int *type, void (*userfunc) (const char *));

int FC_FUNC_(f90calceph_getconstantindex, F90CALCEPH_GETCONSTANTINDEX) (long long *eph, int *index,
                                                                        char name[CALCEPH_MAX_CONSTANTNAME],
                                                                        double *value);
int FC_FUNC_(f90calceph_getconstant, F90CALCEPH_GETCONSTANT) (long long *eph, char *name, double *value, int len);

int FC_FUNC_(f90calceph_getconstantcount, F90CALCEPH_GETCONSTANTCOUNT) (long long *eph);

int FC_FUNC_(f90calceph_compute, F90CALCEPH_COMPUTE) (long long *eph, double *JD0, double *time, int *target,
                                                      int *center, double PV[6]);
int FC_FUNC_(f90calceph_compute_order, F90CALCEPH_COMPUTE_ORDER) (long long *eph, double *JD0, double *time,
                                                                int *target, int *center, int *unit, int *order, double *PVAJ);
int FC_FUNC_(f90calceph_orient_order, F90CALCEPH_ORIENT_ORDER) (long long *eph, double *JD0, double *time,
                                                              int *target, int *unit, int *order, double *PVAJ);
int FC_FUNC_(f90calceph_compute_unit, F90CALCEPH_COMPUTE_UNIT) (long long *eph, double *JD0, double *time,
                                                                int *target, int *center, int *unit, double PV[6]);
int FC_FUNC_(f90calceph_orient_unit, F90CALCEPH_ORIENT_UNIT) (long long *eph, double *JD0, double *time,
                                                              int *target, int *unit, double PV[6]);

void *f2003calceph_open_array(int n, char *filename, int len);

int f2003calceph_sgetconstantindex(int index, char name[CALCEPH_MAX_CONSTANTNAME], double *value);

int f2003calceph_getconstantindex(t_calcephbin * eph, int index, char name[CALCEPH_MAX_CONSTANTNAME], double *value);

void FC_FUNC_(f90calceph_getversion_str, F90CALCEPH_GETVERSION_STR) (char version[CALCEPH_MAX_CONSTANTNAME]);

void f2003calceph_getversion_str(char version[CALCEPH_MAX_CONSTANTNAME]);

int calceph_(double *JD0, double *time, int *target, int *center, double PV[6]);

int calcephinit_(char *filename, int len);

static inline t_calcephbin *f90calceph_getaddresseph(long long *eph);

/********************************************************/
/*! Open the specified ephemeris file 
  return 0 on error.
  return 1 on success.

 @param filename (in) nom du fichier 
*/
/********************************************************/
int FC_FUNC_(f90calceph_sopen, F90CALCEPH_SOPEN) (char *filename, int len)
{
    int res = 0;

    char *newname;

    newname = (char *) malloc(sizeof(char) * (len + 1));
    if (newname != NULL)
    {
        memcpy(newname, filename, len * sizeof(char));
        newname[len] = '\0';
        res = calceph_sopen(newname);
        free(newname);
    }
    else
    {
        fatalerror("Can't allocate memory for f90calceph_sopen\nSystem error : '%s'\n", strerror(errno));
    }
    return res;
}

/********************************************************/
/*! Close the  ephemeris file 
*/
/********************************************************/
void FC_FUNC_(f90calceph_sclose, F90CALCEPH_SCLOSE) (void)
{
    calceph_sclose();
}

/********************************************************/
/*! Return the position and the velocity of the object 
  "target" in the vector PV at the specified time 
  (time ellapsed from JD0).
  The center is specified in center.
  
  return 0 on error.
  return 1 on success.

   @param JD0 (in) reference time (could be 0)
   @param time (in) time ellapsed from JD0
   @param target (in) "target" object 
   @param center (in) "center" object 
   @param PV (out) position and the velocity of the "target" object
*/
/********************************************************/
int FC_FUNC_(f90calceph_scompute, F90CALCEPH_SCOMPUTE) (double *JD0, double *time, int *target, int *center,
                                                        double PV[6])
{
    return calceph_scompute(*JD0, *time, *target, *center, PV);
}

/********************************************************/
/*! return the value of the specified constant frothe ephemeris file 
  return 0 if constant is not found
  return 1 if constant is found and set value.
  
  @param name (in) constant name
  @param value (out) value of the constant
*/
/********************************************************/
int FC_FUNC_(f90calceph_sgetconstant, F90CALCEPH_SGETCONSTANT) (char *name, double *value, int len)
{
    int res = 0;

    char *newname;

    newname = (char *) malloc(sizeof(char) * (len + 1));
    if (newname != NULL)
    {
        memcpy(newname, name, len * sizeof(char));
        newname[len] = '\0';
        res = calceph_sgetconstant(newname, value);
        free(newname);
    }
    else
    {
        fatalerror("Can't allocate memory for f90calceph_sgetconstant\nSystem error : '%s'\n", strerror(errno));
    }
    return res;
}

/********************************************************/
/*! return the number of constants available in the ephemeris file 
  
*/
/********************************************************/
int FC_FUNC_(f90calceph_sgetconstantcount, F90CALCEPH_SGETCONSTANTCOUNT) (void)
{
    return calceph_sgetconstantcount();
}

/********************************************************/
/*! get the name and its value of the constant available 
    at some index from the ephemeris file 
   return 1 if index is valid
   return 0 if index isn't valid
  
  @param index (in) index of the constant (must be between 1 and calceph_getconstantcount() )
  @param name (out) name of the constant 
  @param value (out) value of the constant 
*/
/********************************************************/
int FC_FUNC_(f90calceph_sgetconstantindex, F90CALCEPH_SGETCONSTANTINDEX) (int *index,
                                                                          char name[CALCEPH_MAX_CONSTANTNAME],
                                                                          double *value)
{
    int k;

    int res = calceph_sgetconstantindex(*index, name, value);

    if (res == 1)
    {
        for (k = 0; k < CALCEPH_MAX_CONSTANTNAME - 1; k++)
        {
            if (name[k] == '\0')
                name[k] = ' ';
        }
    }
    return res;
}

/********************************************************/
/*! get the name and its value of the constant available 
    at some index from the ephemeris file 
   return 1 if index is valid
   return 0 if index isn't valid
  
  @param index (in) index of the constant (must be between 1 and calceph_sgetconstantcount() )
  @param name (out) name of the constant 
  @param value (out) value of the constant 
*/
/********************************************************/
int f2003calceph_sgetconstantindex(int index, char name[CALCEPH_MAX_CONSTANTNAME], double *value)
{
    int k;

    int res = calceph_sgetconstantindex(index, name, value);

    if (res == 1)
    {
        for (k = 0; k < CALCEPH_MAX_CONSTANTNAME - 1; k++)
        {
            if (name[k] == '\0')
                name[k] = ' ';
        }
    }
    return res;
}

/********************************************************/
/*! return eph as an address 
  @param eph (inout) parameter to cast
*/
/********************************************************/
static inline t_calcephbin *f90calceph_getaddresseph(long long *eph)
{
    return (t_calcephbin *) (*eph);
}

/********************************************************/
/*! Open the specified ephemeris file 
  return 0 on error.
  return 1 on success.

 @param eph (out) ephemeris descriptor (as integer for F77 compatibility)
 @param filename (in) file name 
*/
/********************************************************/
int FC_FUNC_(f90calceph_open, F90CALCEPH_OPEN) (long long *eph, char *filename, int len)
{
    int res = 0;

    char *newname;

    newname = (char *) malloc(sizeof(char) * (len + 1));
    if (newname != NULL)
    {
        memcpy(newname, filename, len * sizeof(char));
        newname[len] = '\0';
        *eph = (long long) calceph_open(newname);
        res = (*eph != 0);
        free(newname);
    }
    else
    {
        fatalerror("Can't allocate memory for f90calceph_open\nSystem error : '%s'\n", strerror(errno));
    }
    return res;
}

/********************************************************/
/*! Open the list of specified ephemeris file  - Fortran 2003 interface
  Use C binding interface.
  return the ephemeris descriptor.

 @param n (in)  number of file
 @param filename (in) array of file name 
 @param len (in) length of each file name
*/
/********************************************************/
void *f2003calceph_open_array(int n, char *filename, int len)
{
    int k, l;

    char **filear;

    char *newname;

    void *eph = NULL;

    filear = (char **) malloc(sizeof(char *) * (n));
    newname = (char *) malloc(sizeof(char) * (len + 1) * (n));
    if (newname != NULL || filear != NULL)
    {
        for (k = 0; k < n; k++)
        {
            filear[k] = newname + (len + 1) * k;
            memcpy(filear[k], filename + len * k, len * sizeof(char));
            filear[k][len] = '\0';
            l = len - 1;
            while (l > 0 && filear[k][l] == ' ')
            {
                filear[k][l] = '\0';
                l--;
            }
        }
        eph = calceph_open_array(n, (const char *const *) filear);
    }
    else
    {
        fatalerror("Can't allocate memory for f90calceph_open\nSystem error : '%s'\n", strerror(errno));
    }
    if (newname != NULL)
        free(newname);
    if (filear != NULL)
        free(filear);
    return eph;
}

/********************************************************/
/*! Open the list of specified ephemeris file 
  return 0 on error.
  return 1 on success.

 @param eph (out) ephemeris descriptor (as integer for F77 compatibility)
 @param n (in)  number of file
 @param filename (in) array of file name 
 @param len (in) length of each file name
*/
/********************************************************/
int FC_FUNC_(f90calceph_open_array, F90CALCEPH_OPEN_ARRAY) (long long *eph, int *n, char *filename, int *len)
{
    int res;

    *eph = (long long) f2003calceph_open_array(*n, filename, *len);
    res = (*eph != 0);
    return res;
}

/********************************************************/
/*! Close the  ephemeris file 
  @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
*/
/********************************************************/
void FC_FUNC_(f90calceph_close, F90CALCEPH_CLOSE) (long long *eph)
{
    calceph_close(f90calceph_getaddresseph(eph));
}

/********************************************************/
/*! Prefetch the  ephemeris file to the main memory
  @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
*/
/********************************************************/
int FC_FUNC_(f90calceph_prefetch, F90CALCEPH_PREFETCH) (long long *eph)
{
    return calceph_prefetch(f90calceph_getaddresseph(eph));
}


/********************************************************/
/*! Return the position and the velocity of the object 
  "target" in the vector PV at the specified time 
  (time ellapsed from JD0).
  The center is specified in center.
  The output is expressed in AU, AU/day, RAD/day and for TT-TDB in seconds.
  
  return 0 on error.
  return 1 on success.

   @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
   @param JD0 (in) reference time (could be 0)
   @param time (in) time ellapsed from JD0
   @param target (in) "target" object 
   @param center (in) "center" object 
   @param PV (out) position and the velocity of the "target" object
*/
/********************************************************/
int FC_FUNC_(f90calceph_compute, F90CALCEPH_COMPUTE) (long long *eph, double *JD0, double *time, int *target,
                                                      int *center, double PV[6])
{
    return calceph_compute(f90calceph_getaddresseph(eph), *JD0, *time, *target, *center, PV);
}

/********************************************************/
/*! Return the position and the velocity of the object
 "target" in the vector PV at the specified time
 (time ellapsed from JD0).
 The center is specified in center.
 The output is expressed according to unit.
 
 return 0 on error.
 return 1 on success.
 
 @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
 @param JD0 (in) reference time (could be 0)
 @param time (in) time ellapsed from JD0
 @param target (in) "target" object
 @param center (in) "center" object
 @param unit (in) units of PV ( combination of CALCEPH_UNIT_??? )  and/or CALCEPH_USE_NAIFID
 @param order (in) order of the computation
 =0 : Positions are  computed
 =1 : Position+Velocity  are computed
 =2 : Position+Velocity+Acceleration  are computed
 =3 : Position+Velocity+Acceleration+Jerk  are computed
 @param PVAJ (out) orientation (euler angles and their derivatives) of the "target" object
 This array must be able to contain at least 3*(order+1) floating-point numbers.
 On exit, PVAJ[0:3*(order+1)-1] contain valid floating-point numbers.
 */
/********************************************************/
int FC_FUNC_(f90calceph_compute_order, F90CALCEPH_COMPUTE_ORDER) (long long *eph, double *JD0, double *time,
                                                                int *target, int *center, int *unit,
                                                                int *order, double *PVAJ)
{
    return calceph_compute_order(f90calceph_getaddresseph(eph), *JD0, *time, *target, *center, *unit, *order, PVAJ);
}

/********************************************************/
/*! Return the orientation of the object
 "target" in the vector PV at the specified time
 (time ellapsed from JD0).
 The output is expressed according to unit.
 
 return 0 on error.
 return 1 on success.
 
 @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
 @param JD0 (in) reference time (could be 0)
 @param time (in) time ellapsed from JD0
 @param target (in) "target" object
 @param unit (in) units of PV ( combination of CALCEPH_UNIT_??? ) and/or CALCEPH_USE_NAIFID
 @param order (in) order of the computation
 =0 : Positions are  computed
 =1 : Position+Velocity  are computed
 =2 : Position+Velocity+Acceleration  are computed
 =3 : Position+Velocity+Acceleration+Jerk  are computed
 @param PVAJ (out) orientation (euler angles and their derivatives) of the "target" object
 This array must be able to contain at least 3*(order+1) floating-point numbers.
 On exit, PVAJ[0:3*(order+1)-1] contain valid floating-point numbers.
 */
/********************************************************/
int FC_FUNC_(f90calceph_orient_order, F90CALCEPH_ORIENT_ORDER) (long long *eph, double *JD0, double *time,
                                                              int *target, int *unit,
                                                              int *order, double *PVAJ)
{
    return calceph_orient_order(f90calceph_getaddresseph(eph), *JD0, *time, *target, *unit, *order, PVAJ);
}


/********************************************************/
/*! Return the position and the velocity of the object 
  "target" in the vector PV at the specified time 
  (time ellapsed from JD0).
  The center is specified in center.
  The output is expressed according to unit.
  
  return 0 on error.
  return 1 on success.

   @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
   @param JD0 (in) reference time (could be 0)
   @param time (in) time ellapsed from JD0
   @param target (in) "target" object 
   @param center (in) "center" object 
   @param unit (in) units of PV ( combination of CALCEPH_UNIT_??? )  and/or CALCEPH_USE_NAIFID
   @param PV (out) position and the velocity of the "target" object
*/
/********************************************************/
int FC_FUNC_(f90calceph_compute_unit, F90CALCEPH_COMPUTE_UNIT) (long long *eph, double *JD0, double *time,
                                                                int *target, int *center, int *unit, double PV[6])
{
    return calceph_compute_unit(f90calceph_getaddresseph(eph), *JD0, *time, *target, *center, *unit, PV);
}

/********************************************************/
/*! Return the orientation of the object 
  "target" in the vector PV at the specified time 
  (time ellapsed from JD0).
  The output is expressed according to unit.
  
  return 0 on error.
  return 1 on success.

   @param eph (in) ephemeris descriptor (as integer for F77 compatibility)
   @param JD0 (in) reference time (could be 0)
   @param time (in) time ellapsed from JD0
   @param target (in) "target" object 
   @param unit (in) units of PV (combination of CALCEPH_UNIT_??? ) and/or CALCEPH_USE_NAIFID
   @param PV (out) orientation (euler angles and their derivatives) of the "target" object
*/
/********************************************************/
int FC_FUNC_(f90calceph_orient_unit, F90CALCEPH_ORIENT_UNIT) (long long *eph, double *JD0, double *time,
                                                              int *target, int *unit, double PV[6])
{
    return calceph_orient_unit(f90calceph_getaddresseph(eph), *JD0, *time, *target, *unit, PV);
}

/********************************************************/
/*! return the number of constants available in the ephemeris file 
  
  @param eph (inout) ephemeris descriptor
*/
/********************************************************/
int FC_FUNC_(f90calceph_getconstantcount, F90CALCEPH_GETCONSTANTCOUNT) (long long *eph)
{
    return calceph_getconstantcount(f90calceph_getaddresseph(eph));
}

/********************************************************/
/*! return the name and the associated value of the constant available 
    at some index from the ephemeris file 
   return 1 if index is valid
   return 0 if index isn't valid
  
  @param eph (inout) ephemeris descriptor
  @param index (in) index of the constant (must be between 1 and calceph_getconstantcount() )
  @param name (out) name of the constant 
  @param value (out) value of the constant 
*/
/********************************************************/
int FC_FUNC_(f90calceph_getconstantindex, F90CALCEPH_GETCONSTANTINDEX) (long long *eph, int *index,
                                                                        char name[CALCEPH_MAX_CONSTANTNAME],
                                                                        double *value)
{
    int k;

    int res = calceph_getconstantindex(f90calceph_getaddresseph(eph), *index, name, value);

    if (res == 1)
    {
        for (k = 0; k < CALCEPH_MAX_CONSTANTNAME - 1; k++)
        {
            if (name[k] == '\0')
                name[k] = ' ';
        }
    }
    return res;
}

/********************************************************/
/*! return the name and the associated value of the constant available 
    at some index from the ephemeris file 
   return 1 if index is valid
   return 0 if index isn't valid
  
  @param eph (inout) ephemeris descriptor
  @param index (in) index of the constant (must be between 1 and calceph_getconstantcount() )
  @param name (out) name of the constant 
  @param value (out) value of the constant 
*/
/********************************************************/
int f2003calceph_getconstantindex(t_calcephbin * eph, int index, char name[CALCEPH_MAX_CONSTANTNAME], double *value)
{
    int k;

    int res = calceph_getconstantindex(eph, index, name, value);

    if (res == 1)
    {
        for (k = 0; k < CALCEPH_MAX_CONSTANTNAME - 1; k++)
        {
            if (name[k] == '\0')
                name[k] = ' ';
        }
    }
    return res;
}

/********************************************************/
/*! return the value of the specified constant frothe ephemeris file 
  return 0 if constant is not found
  return 1 if constant is found and set value.
  
  @param name (inout) constant name
  @param value (out) value of the constant
*/
/********************************************************/
int FC_FUNC_(f90calceph_getconstant, F90CALCEPH_GETCONSTANT) (long long *eph, char *name, double *value, int len)
{
    int res = 0;

    char *newname;

    newname = (char *) malloc(sizeof(char) * (len + 1));
    if (newname != NULL)
    {
        memcpy(newname, name, len * sizeof(char));
        newname[len] = '\0';
        res = calceph_getconstant(f90calceph_getaddresseph(eph), newname, value);
        free(newname);
    }
    else
    {
        fatalerror("Can't allocate memory for f90calceph_getconstant\nSystem error : '%s'\n", strerror(errno));
    }
    return res;
}

/********************************************************/
/*!  set the error handler for the calceph library
  
  @param type (in) possible value :
             1 : prints the message and the program continues (default)
             2 : prints the message and the program exits (call exit(1))  
             3 : call the function userfunc and the programm continues
  @param userfunc (in) pointer to a function
             useful only if type = 3
             This function is called on error with a message as the first argument
*/
/********************************************************/
void FC_FUNC_(f90calceph_seterrorhandler, F90CALCEPH_SETERRORHANDLER) (int *type, void (*userfunc) (const char *))
{
    calceph_seterrorhandler(*type, userfunc);
}

/********************************************************/
/*! return the version as a string.
  The trailing character are filled with a space character.
  
  @param version (out) name of the version 
*/
/********************************************************/
void f2003calceph_getversion_str(char version[CALCEPH_MAX_CONSTANTNAME])
{
    int k;
    int found = 0;

    calceph_getversion_str(version);
    for (k = 0; k < CALCEPH_MAX_CONSTANTNAME; k++)
    {
        if (version[k] == '\0') found = 1;
        if (found==1) version[k] = ' ';
    }
}

/********************************************************/
/*! return the version as a string.
  The trailing character are filled with a space character.
  
  @param version (out) name of the version 
*/
/********************************************************/
void FC_FUNC_(f90calceph_getversion_str, F90CALCEPH_GETVERSION_STR) (char version[CALCEPH_MAX_CONSTANTNAME])
{
    f2003calceph_getversion_str(version);
}

/********************************************************/
/*! Open the specified ephemeris file 
  return 0 on error.
  return 1 on success.

 @param filename (in) nom du fichier 
*/
/********************************************************/
int calcephinit_(char *filename, int len)
{
    return FC_FUNC_(f90calceph_sopen, F90CALCEPH_SOPEN) (filename, len);
}

/********************************************************/
/*! Return the position and the velocity of the object 
  "target" in the vector PV at the specified time 
  (time ellapsed from JD0).
  The center is specified in center.
  
  return 0 on error.
  return 1 on success.

   @param JD0 (in) reference time (could be 0)
   @param time (in) time ellapsed from JD0
   @param target (in) "target" object 
   @param center (in) "center" object 
   @param PV (out) position and the velocity of the "target" object
*/
/********************************************************/
int calceph_(double *JD0, double *time, int *target, int *center, double PV[6])
{
    return calceph_scompute(*JD0, *time, *target, *center, PV);
}
