/*-----------------------------------------------------------------*/
/*!
  \file timescales.c
  \brief transformation from a time scale to another ones

  \author  H. Manche
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris.

   Copyright 2014, 2015, 2016, CNRS
   email of the author : herve.manche@obspm.fr

  last revision: 05/06/2014

  History:
  \note H. Manche 05/06/2014 : creation
 */
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses
 below to apply on this file.

    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )

 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)

 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )


This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under
French law and abiding by the rules of distribution of free software.
You can  use, modify and/ or redistribute the software under the terms
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA
at the following URL "http://www.cecill.info".

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability.

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or
data to be ensured and,  more generally, to use and operate it in the
same conditions as regards security.

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/

#include <stdio.h>
#include <math.h>


#include "erfa.h"
#include "erfam.h"
#include "errors.h"
#include "calceph.h"
#include "timescales.h"
#include "vec_utils.h"

/*!
  Transform a UTC gregorian date (YYYY/MM/DD/hh:mm:ss format ) into a TT julian day (Terrestrial Time)

  @param year (in): year
  @param month (in): month
  @param day (in): day
  @param hour (in): hour
  @param minute (in): minute
  @param second (in): second
  @param JulianDay_TT (out): julian day in Terrestrial Time

 */
void GregorianDateUTC_to_JulianDayTT( const int year, const int month, const int day,
                                      const int hour, const int minute, const double second,
                                      double JulianDay_TT[2] )
{
  int status ;
  double JulianDay_TAI[2] ;
  double JulianDay_UTC[2] ;

  /* Converts the gregorian date into the julian day (might not be rigorous near leap of second )*/
  status=eraDtf2d("UTC", year, month, day, hour, minute, second, &JulianDay_UTC[0], &JulianDay_UTC[1]) ;
  if ( status<0 ) FatalError("Problem in the the sofa/erfa transformation of UTC->UTC",__FILE__,__LINE__) ;
  if ( status == 1 ) WarningError("'Dubious year': date too early or too late for UTC-TAI - update SOFA/ERFA dat.c routine ?",__FILE__,__LINE__) ;
  /* UTC -> TAI transformation (leap of second) */
  status=eraUtctai( JulianDay_UTC[0], JulianDay_UTC[1], &JulianDay_TAI[0], &JulianDay_TAI[1]);
  if ( status<0 ) FatalError("Problem in the the sofa/erfa transformation of UTC->TAI",__FILE__,__LINE__) ;
  /* TAI -> TT transformation (offset of 32.184 seconds ) */
  status=eraTaitt( JulianDay_TAI[0], JulianDay_TAI[1], &JulianDay_TT[0], &JulianDay_TT[1]);
  if ( status!=0 ) FatalError("Problem in the the sofa/erfa transformation of TAI->TT",__FILE__,__LINE__) ;

  /* At this point, JulianDay_TT[0] always ends with 0.5 */
  if ( JulianDay_TT[1] >=0.5 )
  {
    JulianDay_TT[0] +=0.5 ;
    JulianDay_TT[1] -=0.5 ;
  }
  else
  {
    JulianDay_TT[0] -=0.5 ;
    JulianDay_TT[1] +=0.5 ;
  }


}

/*!
  Transform the julian day from TT (Terrestrial Time) to TDB (Barycentric Dynamical Time)
  @param PTR_eph (in): pointer to an ephemeris
  @param JulianDay_TT (in): julian day Terrestrial Time
  @param JulianDay_TDB (out): julian day TDB
  @param JulianDay_UT1 (in): julian day UT1 (Earth's rotation)
  @param vecStationITRF (in): cartesian coordinates of the station (in km, in ITRF )
  @param iMethod (in): method to compute TT-TDB

 */
void TT_to_TDB(t_calcephbin *PTR_eph,
               const double JulianDay_TT[2], double JulianDay_TDB[2],
               const double JulianDay_UT1[2],
               const double vecStationITRF[3],
               const int iMethod)
{
  /* Local variables */
  double TDBmTT ; /* TDB-TT value (in s) */
  double longitude,DistSpinAxis,DistEqua ; /* longitude of the station (rad)
                                            * distance from Earth spin axis (km)
                                            * distance north of equatorial plane (km) */
  double FDUT1; /* Fraction of one day UT1 (begining at 0h) */


  DistEqua = vecStationITRF[2] ;
  longitude = atan2(vecStationITRF[1], vecStationITRF[0]);
  DistSpinAxis = sqrt( vecStationITRF[0]*vecStationITRF[0]
                      +vecStationITRF[1]*vecStationITRF[1] ) ;
  /* Fraction of day UT1 ; this is not the decimal part of the Julian Day UT1 */
  /* SOFA/ERFA routine dtdb needs "fraction of one day" that begins at 0h */
  /* The Julian Day UT1 ends with .0 at 12h00 UT1*/
  FDUT1=fmod(JulianDay_UT1[0],1.0) ; /* should be 0.0 or 0.5*/
  FDUT1-=0.5 ; /* should be -0.5 or 0.0 */
  FDUT1+=fmod(JulianDay_UT1[1],1.0) ; /* should be between -0.5 and 1.0 */

  switch (iMethod)
  {
    case 0: /* No difference*/
      TDBmTT=0.0 ;
      break;
    case 1: /* use of sofa/erfa subroutine (Fairhead & Bretagnon 1990 + corrections of DE405 + topocentric term) */
      TDBmTT=eraDtdb(JulianDay_TT[0], JulianDay_TT[1], FDUT1,
                     longitude,DistSpinAxis,DistEqua) ;
      break;
    case 2: /* use of 4-D ephemeris + topocentric term */
      TDBmTT=imcceDtdb(PTR_eph, JulianDay_TT, FDUT1,
                       longitude,DistSpinAxis,DistEqua);
      break;
    default:
      TDBmTT=-999.9 ; /* useless, only to avoid warning "not initialized" at compilation */
      FatalError("iMethod is not valid [0:2]",__FILE__,__LINE__) ;
    break;
  }

  (void)eraTttdb(JulianDay_TT[0], JulianDay_TT[1], TDBmTT,
                 &JulianDay_TDB[0], &JulianDay_TDB[1]);

}


/*!
  Computes TDB-TT from a 4D-ephemeris
  This function is based on SOFA/ERFA eraDtdb routine, with a copy of the topocentric terms

  @param PTR_eph (in): pointer to the ephemeris
  @param JulianDay_TDB (in): julian day TDB
  @param ut (in): universal time (UT1, fraction of one day)
  @param elong (in): longitude (east positive, radians)
  @param u (in): distance from Earth spin axis (km)
  @param v (in): distance north of equatorial plane (km)
 */
double imcceDtdb(t_calcephbin *PTR_eph, const double JulianDay_TDB[2],
                 const double ut, const double elong, const double u, const double v)
{
   double t, tsol, w, elsun, emsun, d, elj, els, wt ;
   int test ;
   double vec6[6] ;


   /* THIS PART IS TAKEN FROM SOFA iauDtdb FUNCTION */
   /* Time since J2000.0 in Julian millennia. */
   t = ((JulianDay_TDB[0] - ERFA_DJ00) + JulianDay_TDB[1]) / ERFA_DJM;

   /* ================= */
   /* Topocentric terms */
   /* ================= */

   /* Convert UT to local solar time in radians. */
   tsol = fmod(ut, 1.0) * ERFA_D2PI + elong;

   /* FUNDAMENTAL ARGUMENTS:  Simon et al. 1994. */

   /* Combine time argument (millennia) with deg/arcsec factor. */
   w = t / 3600.0;

   /* Sun Mean Longitude. */
   elsun = fmod(280.46645683 + 1296027711.03429 * w, 360.0) * ERFA_DD2R;

   /* Sun Mean Anomaly. */
   emsun = fmod(357.52910918 + 1295965810.481 * w, 360.0) * ERFA_DD2R;

   /* Mean Elongation of Moon from Sun. */
   d = fmod(297.85019547 + 16029616012.090 * w, 360.0) * ERFA_DD2R;

   /* Mean Longitude of Jupiter. */
   elj = fmod(34.35151874 + 109306899.89453 * w, 360.0) * ERFA_DD2R;

   /* Mean Longitude of Saturn. */
   els = fmod(50.07744430 + 44046398.47038 * w, 360.0) * ERFA_DD2R;

   /* TOPOCENTRIC TERMS:  Moyer 1981 and Murray 1983. */
   wt =   +  0.00029e-10 * u * sin(tsol + elsun - els)
          +  0.00100e-10 * u * sin(tsol - 2.0 * emsun)
          +  0.00133e-10 * u * sin(tsol - d)
          +  0.00133e-10 * u * sin(tsol + elsun - elj)
          -  0.00229e-10 * u * sin(tsol + 2.0 * elsun + emsun)
          -  0.02200e-10 * v * cos(elsun + emsun)
          +  0.05312e-10 * u * sin(tsol - emsun)
          -  0.13677e-10 * u * sin(tsol + 2.0 * elsun)
          -  1.31840e-10 * v * cos(elsun)
          +  3.17679e-10 * u * sin(tsol);
   /* END OF THE PART TAKEN FROM SOFA iauDtdb FUNCTION */

   /* Geocentric terms: from the ephemeris  vec6[0] constains TT-TDB at geocenter */
   test=calceph_compute(PTR_eph,JulianDay_TDB[0],JulianDay_TDB[1], 16, 0, vec6);
   if (!test) FatalError("Problem in the computation of TT-TDB",__FILE__,__LINE__);

/* ============ */
/* Final result */
/* ============ */

/* TDB-TT in seconds. */
   w = wt -vec6[0]  ;

   return w;

}

/*!
  Transform a vector from GCRF (geocetric frame, TT) to BCRF (barycentric, TDB)
  See IERS Conventions 2010, p164, eq. 11.19

  @param PTR_eph (in): pointer to the ephemeris
  @param GMs (in): array containing the GM of all bodies (Mercury to Pluto, Moon and Sun)
  @param JulianDay_TDB (in): julian day TDB
  @param vecTT (in): position vector in GCRF (TT, km)
  @param vecSSBEarth (in): position/velocity vector of the Earth in BCRF (km and km/s)
  @param vecTDB (out): position vector in BCRF (TDB, km)
 */
void vecTT_to_vecTDB(t_calcephbin *PTR_eph, const double *GMs,
                     const double JulianDay_TDB[2],
                     const double vecTT[3], const double vecSSBEarth[6], double vecTDB[3])
{
  double vecEarthBody[6] ; /* vector Earth-Body (position/velocity) */
  double U ; /* potential at geocenter */
  double fac_r, fac_v ;
  const double LC=(ERFA_ELB-ERFA_ELG)/(1.0-ERFA_ELG) ; /* ERFA_ELB and ELC are defined in erfam.h */
  int j ;
  int iBody;

  /* Lorrentz part (velocity) */
  fac_v = 0.5* DotProduct(3, &vecSSBEarth[3],vecTT )*1.0e6/(ERFA_CMPS*ERFA_CMPS); /*ERFA_CMPS (light velocity) is defined in erfam.h */

  /* potential part */
  U=0.0 ;
  for (iBody=1 ; iBody<12 ; iBody++)
  {
    if (iBody==3) continue ;
    (void)calceph_compute_unit(PTR_eph, JulianDay_TDB[0], JulianDay_TDB[1], iBody, 3,
                               CALCEPH_UNIT_KM + CALCEPH_UNIT_SEC + CALCEPH_UNIT_RAD, vecEarthBody);
    U+=GMs[iBody]/norm(3,&vecEarthBody[0]) ;
  };
  fac_r =(1.0-U*1.0e6/(ERFA_CMPS*ERFA_CMPS)-LC) ;

  for (j=0 ; j<3 ; j++)
  {
    vecTDB[j] = fac_r*vecTT[j] - fac_v*vecSSBEarth[3+j] ;
  }

}
