/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-10-24 15:16:17 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

/* Utility fonctions */
#include <espdr_utils.h>
#include <espdr_msg.h>

/* DFS functions */
#include <espdr_dfs.h>
#include <espdr_instrument.h>
#include <espdr_keywords.h>

/* DRL functions */
#include <espdr_bias.h>
#include <espdr_dark.h>
#include <espdr_parameters.h>
#include <espdr_CCD.h>

#include <espdr_hdrl_func.h>
/* Library */
#include <cpl.h>
#include <assert.h>

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

#define RECIPE_ID "espdr_mdark"
#define RECIPE_AUTHOR "P. Figueira, Ch. Lovis, D. Sosnowska"
#define RECIPE_CONTACT "https://support.eso.org"

/*-----------------------------------------------------------------------------
 Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list);

/*----------------------------------------------------------------------------
 Private functions prototypes
 ----------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

static int espdr_mdark_create(cpl_plugin *);
static int espdr_mdark_exec(cpl_plugin *);
static int espdr_mdark_destroy(cpl_plugin *);

/*----------------------------------------------------------------------------
 Static variables
 ----------------------------------------------------------------------------*/
static char espdr_mdark_description_short[] = "Creates the master dark & hot pixel mask";

static char espdr_mdark_description[] =
"This recipe creates a master dark frame by computing the median of all \n\
input dark frames and hot pixel mask through the sigma clipping on the \n\
one output image.\n\
Input Frames : \n\
- A set of n RAW frames (Format=RAW, n >=5, Tag = DARK)\n\
- A static table indicating the detector geometry (Tag = CCD_GEOM)\n\
- A static table to set instrument config. parameters (Tag = INST_CONFIG)\n\
- A master bias residuals frame (Format=PRE, Tag = MASTER_BIAS_RES)\n\
Products : \n\
- A hot pixel map frame (Format=PRE, PRO.CATG = HOT_PIXEL_MASK)\n\
- A master dark frame (Format=PRE, PRO.CATG = MASTER_DARK)\n";



/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief    Build the list of available plugins, for this module.
 @param    list    the plugin list
 @return   0 if everything is ok, -1 otherwise
 
 Create the recipe instance and make it available to the application using
 the interface. This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list)
{
    cpl_recipe *recipe = NULL;
    cpl_plugin *plugin = NULL;
	
    recipe = cpl_calloc(1, sizeof(*recipe));
    if (recipe == NULL) {
		return 1;
    }
	
    plugin = &recipe->interface;
	
    if (cpl_plugin_init(plugin, 
						CPL_PLUGIN_API,	/* Plugin API */
						ESPDR_BINARY_VERSION,	/* Plugin version */
						CPL_PLUGIN_TYPE_RECIPE,	/* Plugin type */
						RECIPE_ID,	/* Plugin name */
						espdr_mdark_description_short,	/* Short help */
						espdr_mdark_description,	/* Detailed help */
						RECIPE_AUTHOR,	/* Author name */
						RECIPE_CONTACT,	/* Contact address as in xsh, 
                                         PACKAGE_BUGREPORT in the tpl */
						espdr_get_license(),	/* Copyright */
						espdr_mdark_create,
						espdr_mdark_exec,
						espdr_mdark_destroy)) {
		cpl_msg_error(cpl_func, "Plugin initialiazation failed");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	if (cpl_pluginlist_append(list, plugin)) {
		cpl_msg_error(cpl_func, "Error adding plugin to list");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	
    /*return (cpl_error_get_code() != CPL_ERROR_NONE); as in xsh*/
	return 0;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Setup the recipe options
 @param    plugin  the plugin
 @return   0 if everything is ok
 
 Create the recipe instance and make it available to the application using
 the interface.
 
 */
/*--------------------------------------------------------------------------*/

static int espdr_mdark_create(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	cpl_error_code error_got = CPL_ERROR_NONE;
	const char* init_ovsc_method = "mean";
	double init_ovsc_ksigma = 4.0;
	int init_ovsc_max_iter = 10;

	const char* init_bias_correction_method = "overscan";


	const char* init_dark_method = "median"; // As static table, Was "mean";
	double init_dark_ksigma = 10.;//  As static table, Was 4.0;
	const char* init_hotpixels_method = "mean";
	double init_hotpixels_ksigma = 10.; // As static table, 4.0;
	int init_hotpixels_max_iter = 10000;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }
	
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
			   CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;
	
    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
	espdr_ensure(recipe->parameters == NULL, 
			   CPL_ERROR_ILLEGAL_OUTPUT, 
			   "Parameter list allocation failed");
	
	/* Set mdark parameters */
	espdr_OVSC_param OVSC_param = {init_ovsc_method, init_ovsc_ksigma,
				init_ovsc_max_iter, init_bias_correction_method};

    espdr_DARK_param DARK_param = {init_dark_method, init_dark_ksigma,
        init_hotpixels_method, init_hotpixels_ksigma, init_hotpixels_max_iter};
    
	error_got = espdr_parameters_OVSC_create(RECIPE_ID,
										   recipe->parameters, 
										   &OVSC_param);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got, 
			   "Error creating the OVSC parameters data structure");	
	
	error_got = espdr_parameters_DARK_create(RECIPE_ID, 
										   recipe->parameters, 
										   &DARK_param);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got, 
			   "Error creating the DARK parameters data structure");
    
	return (error_got);
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Execute the plugin instance given by the interface
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_mdark_exec(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();
	
    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }
	
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
			   CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    /* Verify parameter and frame lists */
	espdr_ensure(recipe->parameters == NULL, 
			   CPL_ERROR_NULL_INPUT, 
			   "Recipe invoked with NULL parameter list");
	
	espdr_ensure(recipe->frames == NULL, CPL_ERROR_NULL_INPUT, 
			   "Recipe invoked with NULL frame set");
	
    /* Set the time display for logs */
    cpl_msg_set_time_on();
    
    /* Invoke the recipe */
    recipe_status = espdr_mdark(recipe->parameters, recipe->frames, RECIPE_ID);
	
    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status) recipe_status = (int)cpl_error_get_code();
    }
	
    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
		 At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }
	
	/*printf("recipe status: %d\n", recipe_status);*/
    
    /* Cancel the time display for logs */
    cpl_msg_set_time_off();
    
	return recipe_status;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Destroy what has been created by the 'create' function
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static int espdr_mdark_destroy(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	
	cpl_error_reset();
	
	/* Verify the plugin existance */
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
			   CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    cpl_parameterlist_delete(recipe->parameters);
	
	return (cpl_error_get_code());
}
