/* 
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H 
#include <config.h>
#endif

#include <string.h>
#include <cpl.h>
#include <hdrl.h>


cpl_error_code
fors_barycorr_adjust_header(cpl_propertylist * header,
                             const char * keyword, double barycorr)
{
  /* Adjust IDP header keywords */
  if (cpl_propertylist_has(header, keyword)) {
      double value = cpl_propertylist_get_double(header, keyword);
      value *= (1 + barycorr / CPL_PHYS_C);
      cpl_propertylist_update_double(header, keyword, value);
  }
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   derives the barycentric correction of an observation
 *
 * Note that only one barycentric correction value is calculated,
 * no matter how many science frames are provided. Multiple frames are
 * allowed so that the MJD range can be determined - the barycentric
 * correction is calculated for the mid MJD of all supplied science frames.
 *
 * @param   frameset     input set of frames
 * @param   science_tag  the tag for the frames to process
 * @param   eop_frame    the EOP_PARAM frame
 * @param   wavelength   the reference wavelength in Angstroms
 * @param   barycorr     the returned barycentric correction
 *
 * @return  cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
int fors_calculate_barycorr(
    const cpl_frameset *frameset,
    const char *science_tag,
    const cpl_frame *eop_frame,
    double wavelength,
    double *barycorr)
{
  double  ra = 0., dec = 0., mjdobs = 0., exptime = 0., longitude = 0.;
  double  latitude = 0., elevation = 0., pressure = 0., temperature = 0.;
  double  humidity = 0.;

  double min_mjd = DBL_MAX, max_mjd = DBL_MIN;
  double press_sum = 0., temp_sum = 0., humid_sum = 0.;
  int n = 0;

  /* These are the header names for the other parameters */
  static const char hname_ra[] = "RA";
  static const char hname_dec[] = "DEC";
  static const char hname_mjdobs[] = "MJD-OBS";
  static const char hname_exptime[] = "EXPTIME";
  static const char hname_longitude[] = "ESO TEL GEOLON";
  static const char hname_latitude[] = "ESO TEL GEOLAT";
  static const char hname_elevation[] = "ESO TEL GEOELEV";
  static const char hname_pres_0[] = "ESO TEL AMBI PRES START";
  static const char hname_pres_1[] = "ESO TEL AMBI PRES END";
  static const char hname_temp[] = "ESO TEL AMBI TEMP";
  static const char hname_rhum[] = "ESO TEL AMBI RHUM";

  // Some checks on the input
  if (frameset == NULL) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "The frameset was not specified");
  }

  if (barycorr == NULL) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "barycorr is NULL");
  }

  /* Load EOP file */
  if (eop_frame == NULL) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "The EOP_PARAM table was not specified");
  }
  cpl_msg_info(cpl_func, "Reading table from EOP_PARAM frame %s",
               cpl_frame_get_filename(eop_frame) );
  cpl_table * eop_table = cpl_table_load(cpl_frame_get_filename(eop_frame), 1, 0);
  if (eop_table == NULL) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "The EOP_PARAM table could not be read");
  }

  /* Doing some baisc checks on the eop table */
  if (!cpl_table_has_column(eop_table, "MJD") ||
      !cpl_table_has_column(eop_table, "PMX") ||
      !cpl_table_has_column(eop_table, "PMY") ||
      !cpl_table_has_column(eop_table, "DUT")){
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "The EOP_PARAM table does not have all required "
                                   "columns, i.e. MJD, PMX, PMY, DUT");
  }

  // Count the science frames
  int nscience = cpl_frameset_count_tags(frameset, science_tag);
  if (nscience < 1) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                   "No frames found in the frameset");
  }

  // Find the first science frame
  const cpl_frame * science_frame =
      cpl_frameset_find_const(frameset, science_tag);

  // Iterate over all science frames
  while (science_frame != NULL) {
      const char * filename = cpl_frame_get_filename(science_frame);
      cpl_propertylist * sci_header = cpl_propertylist_load(filename, 0) ;

      if (sci_header == NULL) {
          return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                       "Could not load science header");
      }

      // Only need these headers once since they should be the same
      // for all frames.
      // Note that if the user supplies frames with very differen RA/Dec
      // values then they should not expect sensible results from this
      // function.
      if (n == 0) {
          ra        = cpl_propertylist_get_double(sci_header, hname_ra       );
          dec       = cpl_propertylist_get_double(sci_header, hname_dec      );
          longitude = cpl_propertylist_get_double(sci_header, hname_longitude);
          latitude  = cpl_propertylist_get_double(sci_header, hname_latitude );
          elevation = cpl_propertylist_get_double(sci_header, hname_elevation);
      }
      mjdobs = cpl_propertylist_get_double(sci_header, hname_mjdobs);
      exptime = cpl_propertylist_get_double(sci_header, hname_exptime);

      // Keep track of sums because we take an average of these over
      // all supplied science frames
      press_sum += cpl_propertylist_get_double(sci_header, hname_pres_0);
      press_sum += cpl_propertylist_get_double(sci_header, hname_pres_1);
      temp_sum += cpl_propertylist_get_double(sci_header, hname_temp);
      humid_sum += cpl_propertylist_get_double(sci_header, hname_rhum);

      if (cpl_error_get_code() != CPL_ERROR_NONE) {
          cpl_msg_warning(cpl_func, "Can not read one or more requested keywords"
              " from the science header.");
          return (int)cpl_error_get_code();
      }

      // Keep track of the min/max mjdobs
      if (mjdobs < min_mjd) {
        min_mjd = mjdobs;
      }
      if (mjdobs + exptime > max_mjd) {
        max_mjd = mjdobs + (exptime/3600./24.);
      }
      cpl_propertylist_delete(sci_header);

      // Grab the next frame
      ++n;
      science_frame = cpl_frameset_find_const(frameset, NULL);
  }

  pressure = press_sum / nscience / 2;
  temperature = temp_sum / nscience;
  humidity = humid_sum / nscience / 100;
  cpl_msg_debug(cpl_func, "Using pressure=%f hPa, temperature=%f C, humidity=%f "
      "for barycentric calculation", pressure, temperature, humidity);

  // Wavelength must be in microns, but it was supplied in Angstroms
  wavelength /= 10000;
  cpl_msg_debug(cpl_func, "Using wavelength=%f um for barycentric calculation", wavelength);

  if (nscience > 1) {
      cpl_msg_debug(cpl_func, "Beginning of first exposure at %f", min_mjd);
      cpl_msg_debug(cpl_func, "End of last exposure at %f", max_mjd);
  }
  cpl_msg_debug(cpl_func, "Using mjdobs=%f for barycentric calculation", (min_mjd + max_mjd) / 2.);

  // We set exptime to 0 because we already calculated the midpoint of the
  // exposure
  cpl_error_code err = CPL_ERROR_NONE;
  err = hdrl_barycorr_compute(ra, dec, eop_table, (min_mjd + max_mjd)/2., 0,
                               longitude,  latitude, elevation,
                               pressure, temperature, humidity, wavelength,
                               barycorr);

  if (err != CPL_ERROR_NONE) {
      cpl_msg_warning(cpl_func, "ERFA can not compute the barycentric "
                      "correction for this frame.");
      return (int)cpl_error_get_code();
  }

  /*** Cleanup ***/
  cpl_table_delete(eop_table);

  return (int)cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   applies the barycentric correction of an observation
 *
 * @param   frameset   input set of frames
 * @param   barycorr   the correction to apply
 *
 * @return  cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
int fors_apply_barycorr_image(
    cpl_propertylist *header,
    const double barycorr,
    const int apply_barycorr)
{
    cpl_propertylist_update_double(header,"ESO DRS BARYCORR", barycorr/1000);
    cpl_propertylist_set_comment(header,"ESO DRS BARYCORR", "Barycentric "
        "correction [km/s]");

    /* Check header for SPECSYS */
    char * specsys = NULL;
    if (cpl_propertylist_has(header, "SPECSYS")) {
        specsys = cpl_sprintf("%s",
                              cpl_propertylist_get_string(header, "SPECSYS"));
    } else {
        cpl_msg_warning(cpl_func, "No header keyword SPECSYS found. "
            "Assuming TOPOCENT");
        specsys = cpl_sprintf("%s",  "TOPOCENT");
    }

    if (!strcmp(specsys, "TOPOCENT") && apply_barycorr) {
        cpl_msg_info(cpl_func, "Updating SPECSYS keyword in the primary header");
        cpl_propertylist_update_string(header,"SPECSYS", "BARYCENT");
        cpl_propertylist_set_comment(header, "SPECSYS",
            "Reference frame for spectral coordinates");
        cpl_msg_info(cpl_func, "Updating WCS keywords in the primary header");
        fors_barycorr_adjust_header(header, "CRVAL1", barycorr);
        fors_barycorr_adjust_header(header, "CD1_1", barycorr);
    }

    cpl_free(specsys);

    return (int)cpl_error_get_code();
}
