/* $Id: fors_extract.c,v 1.45 2011-10-13 14:29:24 cgarcia Exp $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2011-10-13 14:29:24 $
 * $Revision: 1.45 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_extract.h>
#include <fors_star.h>

#include <fors_tools.h>
#include <fors_dfs.h>
#include <fors_pfits.h>
#include <fors_utils.h>

#include <irplib_utils.h>

#include <cpl.h>

#include <string.h>
#include <stdbool.h>
#include <math.h>

/**
 * @defgroup fors_extract  Image source extraction
 */

/**@{*/

#define RCORE_INIT 5.0
#define SATURATION_INIT 60000.

#define ARRAY_LEN(a) (sizeof((a))/sizeof((a)[0]))

/*-----------------------------------------------------------------------------
    (Proto)types
 -----------------------------------------------------------------------------*/

struct _extract_method
{
    enum {SEX, HDRL, TEST} method;
    const char *sex_exe;
    const char *sex_config;
    const char *sex_mag;
    const char *sex_magerr;
    int sex_radius;
    double conf_thresh;
    double min_class_star;
    const char *apcor_keyword;
    // HDRL catalogue parameters
    int obj_min_pixels;
    double obj_threshold;
    cpl_boolean obj_deblending;
    double obj_core_radius;
    cpl_boolean bkg_estimate;
    int bkg_mesh_size;
    hdrl_catalogue_options resulttype;
    double bkg_smooth_fwhm;
    double det_saturation;
};

static fors_star_list *
extract_sex(                                const fors_image *image,
                                            const fors_setting *setting,
                                            const char *sex_exe,
                                            const char *sex_config,
                                            const char *sex_mag,
                                            const char *sex_magerr,
                                            int radius,
                                            double magsyserr,
                                            fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources);

static fors_star_list *
extract_hdrl(                               const fors_image *image,
                                            const fors_setting *setting,
                                            const char *sex_mag,
                                            const char *sex_magerr,
                                            int radius,
                                            double magsyserr,
                                            const cpl_image *confidence_map,
                                            const cpl_wcs *wcs,
                                            const char *apcor_keyword,
                                            const int obj_min_pixels,
                                            const double obj_threshold,
                                            const cpl_boolean obj_deblending,
                                            const double obj_core_radius,
                                            const cpl_boolean bkg_estimate,
                                            const int bkg_mesh_size,
                                            const double bkg_smooth_fwhm,
                                            const double det_saturation,
                                            const double min_class_star,
                                            fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources,
                                            cpl_propertylist **extract_qc);

static fors_star_list *
extract_test(                               fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources);

static hdrl_parameter *
fors_catp_create(                           const int obj_min_pixels,
                                            const double obj_threshold,
                                            const cpl_boolean obj_deblending,
                                            const double obj_core_radius,
                                            const cpl_boolean bkg_estimate,
                                            const int bkg_mesh_size,
                                            const double bkg_smooth_fwhm,
                                            const double det_saturation,
                                            const double gain);

cpl_error_code
normalise_confidence(                       cpl_image * confidence);


/*-----------------------------------------------------------------------------
    Implementation
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  * @brief  create a new hdrl_parameter for the Catalogue module from a few
  *         dynamic values and a lot of hardcoded defaults
  *
  * @param  gain     1st dynamic value: a gain value
  * @param  opt      2nd dynamic value: an output specifier for the Cat module
  *
  * @return the new hdrl_parameter if everything is ok, NULL otherwise
  */
/*----------------------------------------------------------------------------*/
static hdrl_parameter *
fors_catp_create(                           const int obj_min_pixels,
                                            const double obj_threshold,
                                            const cpl_boolean obj_deblending,
                                            const double obj_core_radius,
                                            const cpl_boolean bkg_estimate,
                                            const int bkg_mesh_size,
                                            const double bkg_smooth_fwhm,
                                            const double det_saturation,
                                            const double gain)
{
    return hdrl_catalogue_parameter_create(
               obj_min_pixels, obj_threshold, obj_deblending,
               obj_core_radius, bkg_estimate, bkg_mesh_size,
               bkg_smooth_fwhm, gain, det_saturation,
               HDRL_CATALOGUE_ALL);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check SExtractor object feature: FLAG
 * @param   SExtractor object's binary contamination flags
 * @return  1 if successful, 0 on failure
 * 
 * All FLAGS 1, 2, 4, ..., 128 are severe enough that
 * we do not want to use the source for photometry
 * see SExtractor doc. for the meaning of each flag.
 */
/*----------------------------------------------------------------------------*/
bool
fors_extract_check_sex_flag(                unsigned int    sex_flag)
{
    return (sex_flag == 0x0);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check a sextracted star for validity
 * @param   star    (S)Extracted star
 * @param   ref_img (Optional) reference image (to check for image range)
 * @return  1 if successful, 0 on failure
 * 
 * The following SExtractor failures are caught:
 * - Check for allowed general parameters using fors_star_check_values()
 * - Invalid SExtractor magnitude (can be 99.0)
 * - (x, y) out of range (if @a ref_img != NULL)
 * 
 * @todo
 * - FIXME: FAP: verify the criterion for magnitude rejection
 */
/*----------------------------------------------------------------------------*/
bool
fors_extract_check_sex_star(                const fors_star *star,
                                            const cpl_image *ref_img)
{
    bool    success = 1;
    
    if (star == NULL)
        return 0;
    
    success &= fors_star_check_values(star);
    
    success &= (star->magnitude < 98);
    
    if (ref_img != NULL)
    {
        success &= star->pixel->x >= 1;
        success &= star->pixel->x <= cpl_image_get_size_x(ref_img);
        success &= star->pixel->y >= 1;
        success &= star->pixel->y <= cpl_image_get_size_y(ref_img);
    }
    
    return success;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 * @param    context        parameters context
 */
/*----------------------------------------------------------------------------*/
void 
fors_extract_define_parameters(             cpl_parameterlist *parameters, 
                                            const char *context)
{
    cpl_parameter *p;
    const char *full_name = NULL;
    const char *name;
    
    name = "extract_method";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_enum(full_name,
                               CPL_TYPE_STRING,
                               "Source extraction method",
                               context,
                               "hdrl", 3,
                               "sex", "hdrl", "test");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    name = "sex_exe";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "SExtractor executable",
                                context,
                                FORS_SEXTRACTOR_PATH "/sex");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);
    
    name = "sex_config";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "SExtractor configuration file",
                                context,
                                FORS_SEXTRACTOR_CONFIG "/fors.sex");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);
    
    
    name = "sex_mag";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "SExtractor magnitude",
                                context,
                                "Aper_flux_3");
                                // "MAG_APER");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);
    
    name = "sex_magerr";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "SExtractor magnitude error",
                                context,
                                "Aper_flux_3_err");
                                // "MAGERR_APER");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    name = "sex_radius";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_INT,
                                "Background error map median filter "
                                "radius (unbinned pixels)",
                                context,
                                64);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;
    
    name = "conf_thresh";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Threshold below which pixels in the "
                                "confidence map are set to zero "
                                "(from 0 to 100)",
                                context,
                                10.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;
    
    name = "apcor_keyword";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "Keyword containing HDRL aperture correction "
                                "value",
                                context,
                                "APCOR3");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "min_class_star";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Minimum value of CLASS_STAR below which an object is "
                                "excluded from the extracted sources list",
                                context,
                                0.01);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    /* Add all the hdrl catalogue parameters */
    int                    obj_min_pixels  = 4;
    double                 obj_threshold   = 2.5;
    cpl_boolean            obj_deblending  = CPL_FALSE;
    double                 obj_core_radius = RCORE_INIT;
    cpl_boolean            bkg_estimate    = CPL_TRUE;
    int                    bkg_mesh_size   = 64;
    double                 bkg_smooth_fwhm = 2.;
    double                 det_eff_gain    = 1.; // The real value is read from FITS headers later
    double                 det_saturation  = SATURATION_INIT;
    hdrl_catalogue_options resulttype      = HDRL_CATALOGUE_ALL;

    hdrl_parameter *c_def = hdrl_catalogue_parameter_create(
            obj_min_pixels, obj_threshold, obj_deblending, obj_core_radius,
            bkg_estimate, bkg_mesh_size, bkg_smooth_fwhm, det_eff_gain,
            det_saturation, resulttype);

    cpl_parameterlist *s_param = hdrl_catalogue_parameter_create_parlist(context, "", c_def);
    hdrl_parameter_delete(c_def) ;
    char* gain_name = cpl_sprintf("%s.det.effective-gain", context);
    for (cpl_parameter *p = cpl_parameterlist_get_first(s_param); p != NULL; p = cpl_parameterlist_get_next(s_param)) {
      const char* name = cpl_parameter_get_name(p);
      if (strcmp(name, gain_name) != 0) {
        // Don't add the gain parameter, it's set from header information
        cpl_parameter* newp = cpl_parameter_duplicate(p);
        cpl_parameter_disable(newp, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(parameters, newp);
      }
    }
    cpl_free((void *)gain_name); gain_name = NULL;
    cpl_parameterlist_delete(s_param);
}

/*----------------------------------------------------------------------------*/
#undef cleanup
#define cleanup \
do { \
    cpl_free((void *)name); \
    cpl_free((void *)method); \
} while (0)
/**
 * @brief    Get extraction method from parameter list
 * @param    parameters     recipe parameter list
 * @param    context        read extraction method from this context
 * @return   newly allocated extraction method
 *
 * The parameter list should have been previously created using
 * fors_extract_define_parameters()
 */
/*----------------------------------------------------------------------------*/
extract_method *
fors_extract_method_new(                    const cpl_parameterlist *parameters,
                                            const char *context)
{
    extract_method *em = (extract_method*)cpl_malloc(sizeof(*em));
    const char *name = NULL;
    const char *method = NULL;

    cpl_msg_info(cpl_func, "Extraction method:");

    cpl_msg_indent_more();
    //"sex" method is the default. The parameter won't appear when calling
    //the recipe from esorex, but it will in the unit tests fors_zeropoint-test,
    //fors_img_science-test.
    name = cpl_sprintf("%s.%s", context, "extract_method");
    if(cpl_parameterlist_find_const(parameters, name) == NULL)
        method = cpl_sprintf("%s", "sex");
    else
        method = cpl_sprintf("%s",dfs_get_parameter_string_const(parameters, name));
    cpl_free((void *)name); name = NULL;
    cpl_msg_indent_less();

    assure( !cpl_error_get_code(), {cpl_free(em); return NULL;}, NULL );
    assure( method != NULL, {cpl_free(em); return NULL;}, NULL );

    if (strcmp(method, "sex") == 0) {
        em->method = SEX;
        
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_exe");
        em->sex_exe = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_config");
        em->sex_config = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_mag");
        em->sex_mag = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_magerr");
        em->sex_magerr = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_radius");
        em->sex_radius = dfs_get_parameter_int_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "conf_thresh");
        em->conf_thresh = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "min_class_star");
        em->min_class_star = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
    }
    else if (strcmp(method, "hdrl") == 0) {
        em->method = HDRL;

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_mag");
        em->sex_mag = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_magerr");
        em->sex_magerr = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "sex_radius");
        em->sex_radius = dfs_get_parameter_int_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "conf_thresh");
        em->conf_thresh = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "min_class_star");
        em->min_class_star = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "apcor_keyword");
        em->apcor_keyword = dfs_get_parameter_string_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        /* Parse out the HDRL parameters */
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "obj.min-pixels");
        em->obj_min_pixels = dfs_get_parameter_int_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "obj.threshold");
        em->obj_threshold = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "obj.deblending");
        em->obj_deblending = dfs_get_parameter_bool_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "obj.core-radius");
        em->obj_core_radius = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "bkg.estimate");
        em->bkg_estimate = dfs_get_parameter_bool_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "bkg.mesh-size");
        em->bkg_mesh_size = dfs_get_parameter_int_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "bkg.smooth-gauss-fwhm");
        em->bkg_smooth_fwhm = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();

        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "det.saturation");
        em->det_saturation = dfs_get_parameter_double_const(parameters, name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
    }
    else if (strcmp(method, "test") == 0) {
        em->method = TEST;
    }
    else {
        assure( false, return NULL, "Unknown extraction method '%s'", method);
    }

    cleanup;
    return em;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Deallocate extraction method and set the pointer to NULL
 */
/*----------------------------------------------------------------------------*/
void
fors_extract_method_delete(                 extract_method **em)
{
    if (em && *em) {
        cpl_free(*em); *em = NULL;
    }
    return;
}

/*----------------------------------------------------------------------------*/
#undef cleanup
#define cleanup
/**
 * @brief   Extract sources
 * @param   image                  source image
 * @param   setting                instrument setting (gain)
 * @param   em                     extraction method to use
 * @param   sky_stats              (output) statistics on determined sky
 * @param   background             (output) inferred background image
 * @param   extracted_sources      (output) if non-NULL, table of extracted sources
 * @return  newly allocated list of extracted stars
 */
/*----------------------------------------------------------------------------*/
fors_star_list *
fors_extract(                               const fors_image *image, 
                                            const fors_setting *setting,
                                            const extract_method *em,
                                            double magsyserr,
                                            const cpl_image *confidence_map,
                                            const cpl_wcs *wcs,
                                            fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources,
                                            cpl_propertylist **extract_qc)
{
    assure( em != NULL, return NULL, NULL );

    cpl_msg_info(cpl_func, "Extracting sources");
    
    switch (em->method ) {
    case SEX: return extract_sex(image,
                                 setting,
                                 em->sex_exe,
                                 em->sex_config,
                                 em->sex_mag,
                                 em->sex_magerr,
                                 em->sex_radius,
                                 magsyserr,
                                 sky_stats,
                                 background,
                                 extracted_sources); break;
    case HDRL: return extract_hdrl(image,
                                   setting,
                                   em->sex_mag,
                                   em->sex_magerr,
                                   em->sex_radius,
                                   magsyserr,
                                   confidence_map,
                                   wcs,
                                   em->apcor_keyword,
                                   em->obj_min_pixels,
                                   em->obj_threshold,
                                   em->obj_deblending,
                                   em->obj_core_radius,
                                   em->bkg_estimate,
                                   em->bkg_mesh_size,
                                   em->bkg_smooth_fwhm,
                                   em->det_saturation,
                                   em->min_class_star,
                                   sky_stats,
                                   background,
                                   extracted_sources,
                                   extract_qc); break;
    case TEST: return extract_test(sky_stats,
                                   background,
                                   extracted_sources); break;
    default:
        assure( false, return NULL, "Unknown method %d", em->method );
        break;
    }
}

/*----------------------------------------------------------------------------*/
#undef cleanup
#define cleanup \
do { \
    cpl_table_delete(out); out = NULL; \
    cpl_free((void *)command); \
    cpl_image_delete(work_back); work_back = NULL; \
    cpl_image_delete(bmaxsigma); bmaxsigma = NULL; \
    cpl_image_delete(bsigma); bsigma = NULL; \
    fors_image_delete(&fbsigma); \
} while (0)
/**
 * @brief   Extract sources using SExtractor
 * @param   image           source image
 * @param   setting         instrument setting (gain)
 * @param   sex_exe         SExtractor executable
 * @param   sex_config      SExtractor configuration file
 * @param   sex_mag         SExtractor catalog magnitude
 * @param   sex_magerr      SExtractor catalog magnitude error
 * @param   radius          background error map median filter radius
 * @param   sky_stats       (output) statistics on determined sky
 * @param   background      (output) background image
 * @param   extracted_sources (output) if non-NULL, SExtractor output table
 * @return  newly allocated list of stars
 *
 * Note: The gain given in the setting must describe the
 * image. Therefore, if the provided value of the gain is just
 * the detector gain, the input image must not be stacked or 
 * normalized to e.g. 1s exposure time.
 *
 * A background error map is given to SExtractor. This error map is
 * obtained by applying a median filter to the input image error map 
 * (in order to remove sources).
 */
/*----------------------------------------------------------------------------*/
static fors_star_list *
extract_sex(                                const fors_image *image,
                                            const fors_setting *setting,
                                            const char *sex_exe,
                                            const char *sex_config,
                                            const char *sex_mag,
                                            const char *sex_magerr,
                                            int radius,
                                            double magsyserr,
                                            fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources)
{
    const char *const filename_data  = "sextract_data.fits";
    const char *const filename_sigma = "sextract_bkg_sigma.fits";
    const char *const filename_cat   = "sextract_cat.fits";
    const char *const filename_bkg   = "sextract_bkg.fits";
    const char *const filename_minus_obj   = "sextract_minus_obj.fits";
    cpl_table *out = NULL;
    const char *command = NULL;
    fors_star_list *stars = NULL;
    cpl_image *work_back = NULL;
    cpl_image *bmaxsigma = NULL;
    cpl_image *bsigma = NULL;
    fors_image *fbsigma = NULL;
    fors_image *cropped = NULL;
    int croplx, croply, cropux, cropuy;
    const char *const filename_data_full  = "sextract_data_full.fits";
    const char *const filename_sigma_full = "sextract_bkg_sigma_full.fits";
    int        vignetted = 1;

    assure( setting != NULL, return NULL, NULL );

    assure( image != NULL, return NULL, NULL );

    assure( sky_stats != NULL, return NULL, NULL );
    assure( background != NULL, return NULL, NULL );


    /*
     * In case of new chips, crop
     */

    if (strcmp(setting->chip_id, "CCID20-14-5-6") == 0) {
        croplx =  380 / setting->binx;
        croply =  626 / setting->biny;
        cropux = 3714 / setting->binx;
        cropuy = 2048 / setting->biny;
    }
    else if (strcmp(setting->chip_id, "CCID20-14-5-3") == 0) {
        croplx =  380 / setting->binx;
        croply =    2 / setting->biny;
        cropux = 3714 / setting->binx;
        cropuy = 1920 / setting->biny;
    }
    else if (strncmp(setting->chip_id, "Marl", 4) == 0) {
        croplx =  380 / setting->binx;
        croply =  694 / setting->biny;
        cropux = 3690 / setting->binx;
        cropuy = 2048 / setting->biny;
    }
    else if (strncmp(setting->chip_id, "Norm", 4) == 0) {
        croplx =  380 / setting->binx;
        croply =    2 / setting->biny;
        cropux = 3690 / setting->binx;
        cropuy = 1894 / setting->biny;
    }
    else {
        vignetted = 0;
    }

    cropped = (fors_image *)image;       /* As a default.... */

    if (vignetted) {
       fors_image_save_sex(image, NULL,
                            filename_data_full,
                            filename_sigma_full,
                            radius);

        assure( !cpl_error_get_code(), return NULL,
                "Could not save image to %s and %s",
                filename_data_full, filename_sigma_full);

        cropped = fors_image_duplicate(image);
        fors_image_crop(cropped, croplx, croply, cropux, cropuy);
    }

    /* provide data and error bars in separate files,
       pass to sextractor */

    fors_image_save_sex(cropped, NULL,
                        filename_data,
                        filename_sigma,
                        radius);
    assure( !cpl_error_get_code(), return NULL,
            "Could not save image to %s and %s",
            filename_data, filename_sigma);

    if (vignetted)
        fors_image_delete(&cropped);

    
    /*
     * A = load filename_sigma
     *
     * M = fors_image_max_filter(A)
     *
     * |A-M|
     */

    command = cpl_sprintf("%s %s,%s "
                          "-c %s "
                          /* Use SExtractor's double mode which is probably
                             more tested and more bugfree than the
                             single image mode */
                          "-GAIN %f "   /* Different terminology here:
                                           SExtractor-gain == ESO-conad,
                                           unit = e- / ADU
                                        */
                          "-PIXEL_SCALE %f "
                          "-CHECKIMAGE_TYPE BACKGROUND,-OBJECTS "
                          "-CHECKIMAGE_NAME %s,%s "
                          "-WEIGHT_TYPE MAP_RMS "
                          "-WEIGHT_IMAGE %s,%s "
                          "-CATALOG_TYPE FITS_1.0 "
                          "-CATALOG_NAME %s",
                          sex_exe,
                          filename_data, filename_data,
                          sex_config,
                          1.0/setting->average_gain,
                          setting->pixel_scale,
                          filename_bkg, filename_minus_obj,
                          filename_sigma, filename_sigma,
                          filename_cat);

    cpl_msg_info(cpl_func, "Running '%s'", command);

    assure( system(command) == 0, return stars, "'%s' failed", command);

    /* 
     * The background map is here used just to evaluate a QC parameter,
     * and is also returned to the caller.
     */
    {

        const int plane = 0;
        const int extension = 0;

        *background = cpl_image_load(filename_bkg,
                                     CPL_TYPE_FLOAT, plane, extension);

        cpl_image * minus_objs = cpl_image_load(filename_minus_obj,
                                                CPL_TYPE_FLOAT, plane,
                                                extension);

        assure( !cpl_error_get_code(), return NULL,
                "Could not load SExtractor background image %s",
                filename_bkg );

        if (vignetted) {
            work_back = cpl_image_new(fors_image_get_size_x(image), 
                                      fors_image_get_size_y(image),
                                      CPL_TYPE_FLOAT);
            cpl_image_copy(work_back, *background, croplx, croply);

            assure( !cpl_error_get_code(), return NULL,
                    "Could not insert background image %s",
                    filename_bkg );

            cpl_image_delete(*background);
            *background = work_back;
            work_back = NULL;
        }
        
        /* 
         * To avoid using the non-illuminated parts, use only the central 
         * 100x100 window.
         *
         * It does not make too much sense to trend these QC parameters
         * anyway because they mostly describe the individual science 
         * exposures.
         */

        int nx = cpl_image_get_size_x(*background);
        int ny = cpl_image_get_size_y(*background);
        int xlo = nx/2 - 50;
        int xhi = nx/2 + 50;
        int ylo = ny/2 - 50;
        int yhi = ny/2 + 50;
        
        if (xlo <   0) xlo = 0;       /* Just in case... */
        if (xhi >= nx) xhi = nx - 1;
        if (ylo <   0) ylo = 0;
        if (yhi >= ny) yhi = ny - 1;

        work_back = cpl_image_duplicate(*background);
        
        sky_stats->mean   = cpl_image_get_mean_window(work_back, 
                                                      xlo, ylo, xhi, yhi);
        sky_stats->median = cpl_image_get_median_window(work_back, 
                                                 xlo, ylo, xhi, yhi);
        cpl_image_subtract_scalar(work_back, sky_stats->median);
        cpl_image_abs(work_back);
        sky_stats->rms    = cpl_image_get_median_window(work_back, 
                                                        xlo, ylo, xhi, yhi) * STDEV_PR_MAD;;

        sky_stats->bkg_rms = cpl_image_get_stdev_window(minus_objs, xlo, ylo, xhi, yhi);

        cpl_image_delete(work_back); work_back = NULL;

        assure( !cpl_error_get_code(), return NULL,
                "Could not calculate sky statistics" );

        cpl_image_delete(minus_objs);

    }
    
    cpl_msg_info(cpl_func, "Background = %f +- %f ADU",
                 sky_stats->median, sky_stats->rms);

    /* 
     * SExtractors background estimation is not reliable near
     * non-illuminated areas. The underestimated background 
     * leads to false detections.
     * Therefore, reject sources which are too close to the
     * illumination edge. The edge is masked using a max filter 
     * on the smoothed variance image of the background map. 
     * The filter must have comparable size to the SExtractor 
     * BACK_SIZE parameter. The discrimination level is half-way
     * between the high-variance and low-variance regions.
     */

    float level;
    {
        int xradius = 64;
        int yradius = 64;
        int plane = 0;
        int extension = 0;

        if (vignetted) {
            bsigma = cpl_image_load(filename_sigma_full,
                                    CPL_TYPE_FLOAT, plane, extension);
        }
        else {
            bsigma = cpl_image_load(filename_sigma,
                                    CPL_TYPE_FLOAT, plane, extension);
        }

        assure( !cpl_error_get_code(), return NULL,
                "Could not load SExtractor background error image %s",
                filename_sigma );

        /*
         * Duplication is necessary for creating the fors_image
         * to be passed to fors_image_filter_max_create()
         */
        
        /* this wraps the fors_image around the cpl images,
         * so set them to NULL */

        fbsigma = fors_image_new(cpl_image_duplicate(bsigma), bsigma);
        bsigma = NULL;

        {
            bool use_variance = true;
            bmaxsigma = fors_image_filter_max_create(fbsigma, xradius, 
                                                     yradius, use_variance);

     /* cpl_image_save(bmaxsigma, "/tmp/test.fits", CPL_BPP_IEEE_FLOAT, NULL,
                              CPL_IO_DEFAULT); */
            
        }

        fors_image_delete(&fbsigma);

        /*
          This is not robust if there are no non-illuminated areas.

          maxima = cpl_image_get_max(bmaxsigma);
          minima = cpl_image_get_min(bmaxsigma);
          level = (maxima + minima) / 2; 
        */
        
        /* 5 sigma rejection */
        level = cpl_image_get_median(bmaxsigma) * 5;

        cpl_msg_debug(cpl_func, "Threshold level = %f",
                      level);

    }

    out = cpl_table_load(filename_cat, 1, 1);

    assure( !cpl_error_get_code(), return NULL,
            "Could not load SExtractor output table %s",
            filename_cat); 

    /* Validate sextractor output */
    assure( cpl_table_has_column(out, "FLAGS"), return NULL,
            "%s: Missing column: %s", filename_cat, "FLAGS");

    assure( cpl_table_has_column(out, "CLASS_STAR"), return NULL,
            "%s: Missing column: %s", filename_cat, "CLASS_STAR");

    assure( cpl_table_has_column(out, "BACKGROUND"), return NULL,
            "%s: Missing column: %s", filename_cat, "BACKGROUND");

    assure( cpl_table_has_column(out, "X_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "X_IMAGE");

    assure( cpl_table_has_column(out, "Y_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "Y_IMAGE");

    assure( cpl_table_has_column(out, "FWHM_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "FWHM_IMAGE");

    assure( cpl_table_has_column(out, "A_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "A_IMAGE");

    assure( cpl_table_has_column(out, "B_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "B_IMAGE");

    assure( cpl_table_has_column(out, "THETA_IMAGE"), return NULL,
            "%s: Missing column: %s", filename_cat, "THETA_IMAGE");

    assure( cpl_table_has_column(out, sex_mag), return NULL,
            "%s: Missing column: %s", filename_cat, sex_mag);

    assure( cpl_table_has_column(out, sex_magerr), return NULL,
            "%s: Missing column: %s", filename_cat, sex_magerr);


    /* cpl_table_dump_structure(out, stdout); */

    if (vignetted) {
        cpl_table_add_scalar(out, "X_IMAGE", croplx - 1);
        cpl_table_add_scalar(out, "Y_IMAGE", croply - 1);
    }

    stars = fors_star_list_new();

    {
        int i;
        int bkg_rejected = 0;
        int rejected = 0;
        int nx = cpl_image_get_size_x(bmaxsigma);
        int ny = cpl_image_get_size_y(bmaxsigma);

        for (i = 0; i < cpl_table_get_nrow(out); i++) {
            fors_star       *s = NULL;
            unsigned int    flags = 0x0;
            int             x, y, tmp;
            double          bg_err;
            
            s = fors_star_new_from_table(   out,
                                            i,
                                            "X_IMAGE",
                                            "Y_IMAGE",
                                            "FWHM_IMAGE",
                                            "A_IMAGE",
                                            "B_IMAGE",
                                            "THETA_IMAGE", 
                                            sex_mag,
                                            sex_magerr,
                                            "CLASS_STAR",
                                            "FLUX_APER",
                                            "FLUX_MAX");
            (*s).orientation *= M_PI/180;
            (*s).semi_major *= setting->binx;
            (*s).semi_minor *= setting->binx;
            (*s).fwhm *= setting->binx;
            (*s).dmagnitude = sqrt((*s).dmagnitude * (*s).dmagnitude + magsyserr * magsyserr);
            
            flags = cpl_table_get_int(      out, "FLAGS", i, NULL);
            
            x = (int)(s->pixel->x + 0.5);
            y = (int)(s->pixel->y + 0.5);
            if (x >= 1 && x <= nx && y >= 1 && y <= ny)
                bg_err = cpl_image_get(bmaxsigma, x, y, &tmp);
            else
                bg_err = -1.0;

            if (fors_extract_check_sex_flag(flags)
                && fors_extract_check_sex_star(s, bmaxsigma)
                && bg_err < level)
            {
                cpl_msg_debug(              cpl_func,
                                            "Source at (%f, %f): fwhm = %f px",
                                            s->pixel->x, s->pixel->y, s->fwhm);
                assure(                     !cpl_error_get_code(), return NULL,
                                            "Could not read SExtractor "
                                            "output table %s",
                                            filename_cat);
                fors_star_list_insert(stars, s);
            }
            else
            {
                cpl_msg_debug(              cpl_func,
                                            "Rejecting source at (%f, %f): "
                                            "flags = 0x%x; fwhm = %f pix; "
                                            "background error = %f; "
                                            "level = %f; background = %f; "
                                            "magnitude = %f", s->pixel->x,
                                            s->pixel->y, flags, s->fwhm, bg_err,
                                            level, cpl_table_get_float(out,
                                                "BACKGROUND", i, NULL),
                                            s->magnitude);
                fors_star_delete(&s);
                rejected++;
                if (bg_err >= level)
                    bkg_rejected++;
            }
        }
        
        cpl_msg_info(cpl_func, "%d sources sextracted, %d rejected", 
                     fors_star_list_size(stars) + rejected,
                     rejected);
    }
    
    if (extracted_sources != NULL) {
        *extracted_sources = cpl_table_duplicate(out);
    }

    /* Remove the temporary files needed by sextractor */
    int status = unlink(filename_data);
    status +=    unlink(filename_sigma);
    if (vignetted) 
    {
        status += unlink(filename_sigma_full);
        status += unlink(filename_data_full);
    }
    status += unlink(filename_cat);
    status += unlink(filename_bkg);
    status += unlink(filename_minus_obj);

    assure( status == 0, return NULL,
            "Problem removing temporary files");
    
    cleanup;
    return stars;
}

#undef cleanup
/*----------------------------------------------------------------------------*/
/**
  @brief    Normalise confidence array so that mean of good pixels is 100.
  @param    confidence  The confidence array
  @return   CPL_ERROR_NONE on success, otherwise a cpl error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
normalise_confidence(                       cpl_image * confidence)
{

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    int nreject = cpl_image_count_rejected(confidence);
    cpl_error_ensure(nreject == 0, CPL_ERROR_ILLEGAL_INPUT, goto cleanup,
                     "confidence array must have no bad pixels (%d)", nreject);

    const cpl_size nx = cpl_image_get_size_x(confidence);
    const cpl_size ny = cpl_image_get_size_x(confidence);
    double * data = cpl_image_get_data_double(confidence);

    double sum = 0.0;
    int nonzero = 0;

    for (cpl_size i = 0; i < nx*ny; i++) {
        if (data[i] < 0.0) data[i] = 0.0;
        sum += data[i];
        if (data[i] > 0.0) nonzero++;
    }

    if (sum > 0.0) {
        cpl_image_multiply_scalar(confidence, 100.0 * (double)nonzero / sum);
    }

 cleanup:

    return cpl_error_get_code();
}

#undef cleanup
#define cleanup \
do { \
    fors_image_delete(&crop); crop = NULL; \
    hdrl_catalogue_result_delete(cres); cres = NULL; \
    cpl_image_delete(bmaxsigma); bmaxsigma = NULL; \
} while (0)
/*----------------------------------------------------------------------------*/
/**
 * @brief   Extract sources using the HDRL Catalogue module
 * @param   image           source image
 * @param   setting         instrument setting (gain)
 * @param   sex_mag         HDRL catalog flux to be converted to magnitude
 * @param   sex_magerr      HDRL catalog flux error to be converted to mag error
 * @param   radius          background error map median filter radius
 * @param   sky_stats       (output) statistics on determined sky
 * @param   background      (output) background image
 * @param   extracted_sources (output) if non-NULL, HDRL catalog output table
 * @return  newly allocated list of stars
 *
 * Note: The gain given in the setting must describe the
 * image. Therefore, if the provided value of the gain is just
 * the detector gain, the input image must not be stacked or 
 * normalized to e.g. 1s exposure time.
 *
 * A background error map is given to SExtractor. This error map is
 * obtained by applying a median filter to the input image error map 
 * (in order to remove sources).
 */
/*----------------------------------------------------------------------------*/
static fors_star_list *
extract_hdrl(                               const fors_image * image,
                                            const fors_setting * setting,
                                            const char *sex_mag,
                                            const char *sex_magerr,
                                            int radius,
                                            double magsyserr,
                                            const cpl_image * confidence_map,
                                            const cpl_wcs *wcs,
                                            const char *apcor_keyword,
                                            const int obj_min_pixels,
                                            const double obj_threshold,
                                            const cpl_boolean obj_deblending,
                                            const double obj_core_radius,
                                            const cpl_boolean bkg_estimate,
                                            const int bkg_mesh_size,
                                            const double bkg_smooth_fwhm,
                                            const double det_saturation,
                                            const double min_class_star,
                                            fors_extract_sky_stats * sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources,
                                            cpl_propertylist **extract_qc)
{
    fors_image* crop = NULL;
    hdrl_catalogue_result* cres = NULL;
    cpl_image* bmaxsigma = NULL;
    fors_star_list* stars = NULL;

    if (!setting || !image || !sky_stats || !background) {
        cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
        cleanup;
        return NULL;
    }

    /* In case of new chips, crop */
    /* Cropping is disabled for HDRL as the use of the
     * confidence map means that unilluminated pixels are effectively
     * ignored anyway. The removal of cropping also solves PIPE-7731.
    int croplx, croply, cropux, cropuy, vignetted = 1;
    SET(crop, fors_image, w) = fors_image_duplicate(image);
    if (strcmp(setting->chip_id, "CCID20-14-5-6") == 0) {
        croplx =  380 / setting->binx;
        croply =  626 / setting->biny;
        cropux = 3714 / setting->binx;
        cropuy = 2048 / setting->biny;
    }
    else if (strcmp(setting->chip_id, "CCID20-14-5-3") == 0) {
        croplx =  380 / setting->binx;
        croply =    2 / setting->biny;
        cropux = 3714 / setting->binx;
        cropuy = 1920 / setting->biny;
    }
    else if (strncmp(setting->chip_id, "Marl", 4) == 0) {
        croplx =  380 / setting->binx;
        croply =  694 / setting->biny;
        cropux = 3690 / setting->binx;
        cropuy = 2048 / setting->biny;
    }
    else if (strncmp(setting->chip_id, "Norm", 4) == 0) {
        croplx =  380 / setting->binx;
        croply =    2 / setting->biny;
        cropux = 3690 / setting->binx;
        cropuy = 1894 / setting->biny;
    }
    else {
        vignetted = 0;
    }

    if (vignetted) fors_image_crop(crop, croplx, croply, cropux, cropuy);
    */
    crop = fors_image_duplicate(image);
    int vignetted = 0;

    hdrl_parameter* par = fors_catp_create(
        obj_min_pixels,
        obj_threshold,
        obj_deblending,
        obj_core_radius,
        bkg_estimate,
        bkg_mesh_size,
        bkg_smooth_fwhm,
        det_saturation,
        setting->average_gain);

    //SET(sigma, cpl_image) = cpl_image_power_create(crop->variance, 0.5);
    //CSET(hi, hdrl_image) = hdrl_image_wrap(
    //    crop->data, sigma, hdrl_image_unwrap, 1);
    //const cpl_image * prior = hdrl_image_get_image_const(hi);
    cres = hdrl_catalogue_compute(crop->data, confidence_map, wcs, par);

    //if (cpl_error_get_code()) return ERROR(
    //    0, "The BPM sense may need to be reversed");

    /* 
     * The background map is here used just to evaluate a QC parameter,
     * and is also returned to the caller.
     */
    {
        *background = cres->background;
        /* Cropping is disabled for HDRL as the use of the
         * confidence map means that unilluminated pixels are effectively
         * ignored anyway. The removal of cropping also solves PIPE-7731.
        if (vignetted) {
            const int xsz = fors_image_get_size_x(image);
            const int ysz = fors_image_get_size_y(image);
            SET(work, cpl_image) = cpl_image_new(xsz, ysz, CPL_TYPE_DOUBLE);
            cpl_image_copy(work, *background, croplx, croply);
            cpl_image_delete(*background);
            *background = YANK(work);
        }
         */
        cres->background = NULL;
        
        /* 
         * To avoid using the non-illuminated parts, use only the central 
         * 100x100 window.
         *
         * It does not make too much sense to trend these QC parameters
         * anyway because they mostly describe the individual science 
         * exposures.
         */

        if (! *background) {
            cpl_msg_warning(cpl_func, "HDRL catalogue routine did not create a background");
            cpl_error_reset();
        } else {
            cpl_msg_info(cpl_func, "Calculating sky statistics");
            const int nx = cpl_image_get_size_x(*background);
            const int ny = cpl_image_get_size_y(*background);
            int xlo = nx/2 - 50;
            int xhi = nx/2 + 50;
            int ylo = ny/2 - 50;
            int yhi = ny/2 + 50;
            
            if (xlo <   0) xlo = 0;       /* Just in case... */
            if (xhi >= nx) xhi = nx - 1;
            if (ylo <   0) ylo = 0;
            if (yhi >= ny) yhi = ny - 1;

            cpl_image* work = cpl_image_duplicate(*background);
            sky_stats->mean = cpl_image_get_mean_window(
                work, xlo, ylo, xhi, yhi);
            sky_stats->median = cpl_image_get_median_window(
                work, xlo, ylo, xhi, yhi);
            cpl_image_subtract_scalar(work, sky_stats->median);
            cpl_image_abs(work);
            sky_stats->rms = cpl_image_get_median_window(
                work, xlo, ylo, xhi, yhi);
            sky_stats->rms *= STDEV_PR_MAD;

            cpl_image* confbpm = create_bpm_from_confidence(confidence_map);
            double skymed, skysigma;
            backstats(crop->data, confbpm, *background, &skymed, &skysigma);
            cpl_image_delete(confbpm);
            // FIXME This is a good estimate but a new calculation is
            // required. See PIPE-9769
            sky_stats->bkg_rms = skysigma;

            cpl_image_delete(work);
        }
    }

    cpl_msg_info(cpl_func, "Background = %f +- %f ADU",
                 sky_stats->median, sky_stats->rms);

    /* 
     * SExtractors background estimation is not reliable near
     * non-illuminated areas. The underestimated background 
     * leads to false detections.
     * Therefore, reject sources which are too close to the
     * illumination edge. The edge is masked using a max filter 
     * on the smoothed variance image of the background map. 
     * The filter must have comparable size to the SExtractor 
     * BACK_SIZE parameter. The discrimination level is half-way
     * between the high-variance and low-variance regions.
     */

    double level;
    {
        cpl_msg_info(cpl_func, "Creating background error map");

        fors_image* img = vignetted ? (fors_image *)image : crop;
        const int step = radius / 2;  /* 25 points (5x5) sampling grid */
        const int xend = fors_image_get_size_x(img);
        const int yend = fors_image_get_size_y(img);
        cpl_image* bvar = fors_image_filter_median_create(
            img, radius, radius, 1, 1, xend, yend, step, step, false);

        cpl_image* bsigma = cpl_image_power_create(bvar, 0.5);
        cpl_image_delete(bvar);

        fors_image* fbsigma = fors_image_new(bsigma,
            cpl_image_duplicate(bsigma));

        bmaxsigma = fors_image_filter_max_create(
            fbsigma, 64, 64, true);
        fors_image_delete(&fbsigma);

        /*
          This is not robust if there are no non-illuminated areas.

          const float maxima = cpl_image_get_max(bmaxsigma);
          const float minima = cpl_image_get_min(bmaxsigma);
          level = (maxima + minima) / 2; 
        */
        
        /* 5 sigma rejection */
        const double median = cpl_image_get_median(bmaxsigma);
        level = median * 5;

        cpl_msg_debug(cpl_func, "Threshold level = %f", level);
    }

    /* Validate HDRL Catalogue output */
    cpl_table * out = cres->catalogue;
    bool missing = false;
    const char * const cols[] = {
        "Error_bit_flag", "X_coordinate", "Y_coordinate", "Position_angle",
        "Classification", "Ellipticity", "Sky_level", "FWHM",
        "Peak_height", sex_mag, sex_magerr };
    for (size_t i = 0; i < ARRAY_LEN(cols); ++i) {
        if (!cpl_table_has_column(out, cols[i])) {
          cpl_msg_info(cpl_func, "column %s is missing", cols[i]);
        }
        missing = missing || !cpl_table_has_column(out, cols[i]);
    }

    if (missing) {
        cpl_msg_error(cpl_func, "Catalogue output is missing some columns");
        cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
    }
    
    /* Cropping is disabled for HDRL as the use of the
     * confidence map means that unilluminated pixels are effectively
     * ignored anyway. The removal of cropping also solves PIPE-7731.
    if (vignetted) {
        cpl_table_add_scalar(out, "X_coordinate", croplx - 1);
        cpl_table_add_scalar(out, "Y_coordinate", croply - 1);
    }
     */

    // Copy the qclist so our caller can see it
    if (extract_qc != NULL) {
        *extract_qc = cpl_propertylist_duplicate(cres->qclist);
    }

    // Extract value of apcor_keyword from cres->qclist and use it to correct
    // the calculated magnitude.
    float apcor = 0.0;
    if (strlen(apcor_keyword) == 0) {
        cpl_msg_info(cpl_func, "Not applying magnitude correction, apcor_keyword not specified");
    } else if (cpl_propertylist_has(cres->qclist, apcor_keyword)) {
        apcor = cpl_propertylist_get_float(cres->qclist, apcor_keyword);
        cpl_msg_info(cpl_func, "Applying magnitude correction %s = %f", apcor_keyword, apcor);
    } else {
        cpl_msg_warning(cpl_func, "Not applying magnitude correction, %s keyword not found", apcor_keyword);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        cpl_error_reset();
    }

    stars = fors_star_list_new();
    {
        int bkg_rejected = 0;
        int rejected = 0;
        const int nx = cpl_image_get_size_x(bmaxsigma);
        const int ny = cpl_image_get_size_y(bmaxsigma);

        const int nrow = cpl_table_get_nrow(out);

        for (int i = 0; i < nrow; i++) {
            fors_star* s = fors_star_new_from_table(
                out, i,
                "X_coordinate"   /*"X_IMAGE"*/,
                "Y_coordinate"   /*"Y_IMAGE"*/,
                "FWHM"           /*"FWHM_IMAGE"*/,
                0                /*"A_IMAGE"*/,      // semi-major axis
                0                /*"B_IMAGE"*/,      // semi-minor axis
                "Position_angle" /*"THETA_IMAGE"*/, 
                sex_mag          /*"MAG_APER"*/,     // not actually a magnitude yet
                sex_magerr       /*"MAGERR_APER"*/,
                "Statistic"      /*"CLASS_STAR"*/,   // normally distributed but not normalised
                sex_mag          /*"FLUX_APER"*/,
                "Peak_height"    /*"FLUX_MAX"*/);
            const double ell = cpl_table_get(out, "Ellipticity", i, &(int){0});

            if (!s) {
                continue;
            }

            // This is the average of the confidence map pixels corresponding
            // to pixels within the object. If it's low then the confidence in
            // this object is also low.
            const double av_conf = cpl_table_get(out, "Av_conf", i, &(int){0});
            // Convert from flux to magnitude.
            // Do the error term first, as it uses the flux value
            // cpl_msg_debug(cpl_func, "before mag=%.2f dmag=%.2f", s->magnitude, s->dmagnitude);
            s->dmagnitude = 2.0 * (2.5 / CPL_MATH_LN10) * (s->dmagnitude / s->magnitude);
            s->magnitude = -2.5 * log10(s->magnitude) - apcor;  // flux -> mag
            // cpl_msg_debug(cpl_func, "after mag=%.2f dmag=%.2f", s->magnitude, s->dmagnitude);
            s->semi_major = setting->binx;
            s->semi_minor = s->semi_major - ell * s->semi_major;
            s->fwhm *= setting->binx;
            s->dmagnitude = sqrt(
                s->dmagnitude * s->dmagnitude + magsyserr * magsyserr);

            // Adjust the stellarity value. The set of values should be normally distributed.
            // The documentation suggests that this is "An equivalent N(0,1) measure of how
            // stellar like an image is".
            // We turn it into a value in the range [0, 1] 
            // (i.e. probability that this object is a star, based on this metric)
            // FIXME this is not exactly how the SExtractor CLASS_STAR value behaves
            // double zvalue = (s->stellarity_index - stat_mean) / stat_std;
            s->stellarity_index = exp(-0.5 * (s->stellarity_index * s->stellarity_index));

            // According to the manual, this column counts the number of bad pixels in the
            // object.
            const double flags = cpl_table_get_double(
                    out, "Error_bit_flag" /*"FLAGS"*/, i, NULL);

            // The HDRL Classification column is set to -1 for stars and -2 for compact
            // (possibly stellar) objects
            const double classification = cpl_table_get_double(
                    out, "Classification", i, NULL);

            const int x = (int)(s->pixel->x + 0.5);
            const int y = (int)(s->pixel->y + 0.5);
            double bg_err = -1.0;
            if (x >= 1 && x <= nx && y >= 1 && y <= ny) {
                bg_err = cpl_image_get(bmaxsigma, x, y, &(int){0});
            }
            
            if ((flags == 0) &&
                (bg_err < level) &&
                (av_conf > 99.) && 
                (classification < 0) &&
                (s->flux_max > 0) &&
                (s->flux_max < det_saturation) &&
                (s->stellarity_index >= min_class_star) &&
                fors_extract_check_sex_star(s, bmaxsigma))
            {
                cpl_msg_debug(cpl_func, "Source at (%f, %f): fwhm = %f px, stellarity = %f, classification = %f",
                              s->pixel->x, s->pixel->y, s->fwhm, s->stellarity_index, classification);
                fors_star_list_insert(stars, s);
            }
            else {
                const double bkg = cpl_table_get_double(out, "Sky_level", i, 0);
                cpl_msg_debug(cpl_func, "Rejecting source at (%f, %f): "
                              "flags = %f; fwhm = %f pix; "
                              "background error = %f; level = %f; "
                              "background = %f; magnitude = %f; "
                              "semi_major = %f; semi_minor = %f; "
                              "stellarity = %f; classification = %f; "
                              "avg confidence = %f; ellipticity = %f; "
                              "flux_max = %f; ",
                              s->pixel->x,
                              s->pixel->y, flags, s->fwhm, bg_err, level,
                              bkg, s->magnitude, s->semi_major, s->semi_minor,
                              s->stellarity_index, classification, av_conf, ell,
                              s->flux_max);
                fors_star_delete(&s);
                rejected++;
                if (bg_err >= level) bkg_rejected++;
            }
        }
        
        cpl_msg_info(cpl_func, "%d sources extracted, %d rejected", 
                     fors_star_list_size(stars) + rejected,
                     rejected);
    }
    
    if (extracted_sources) {
        *extracted_sources = out;
        cres->catalogue = NULL;
    }

    cleanup;

    return stars;
}

/*----------------------------------------------------------------------------*/
#undef cleanup
#define cleanup
/**
 * @brief   Extract sources
 * @param   sky_stats      (output) sky statistics
 * @param   background     (output) background
 * @param   extracted_sources (output) if non-NULL, table of sources
 * @return  newly allocated list of extracted stars
 *
 * The star positions are hardcoded!
 *
 * This method is used only for testing without relying on external
 * packages
 */
/*----------------------------------------------------------------------------*/
static fors_star_list *
extract_test(                               fors_extract_sky_stats *sky_stats,
                                            cpl_image **background,
                                            cpl_table **extracted_sources)
{
    assure( sky_stats != NULL, return NULL, NULL );
    assure( background != NULL, return NULL, NULL );
    
    sky_stats->mean = 1;
    sky_stats->median = 2;
    sky_stats->rms = 3;

    *background = cpl_image_new(10, 20, CPL_TYPE_FLOAT); /* Zero, wrong size */

    fors_star_list *stars = fors_star_list_new();

    struct {
        double x, y, magnitude, dmagnitude;
    } 
    data[] = {
        {100 ,  200, -10, 0.01},
        {1100,  200, -11, 0.01},
        {1   ,  5  , -10, 0.01},
        {100 , 1200, -12, 0.01},
        {1100, 1200, -13, 0.01}
    };
       
    int N = sizeof(data) / sizeof(*data);
    int i;
    double a = 2;
    double b = 1;
    double fwhm = 1.5;
    double orientation = 1.0; /* radians */

    for (i = 0; i < N; i++) {
        fors_star_list_insert(stars,
                              fors_star_new(data[i].x,
                                            data[i].y,
                                            fwhm,
                                            a, b, orientation,
                                            data[i].magnitude,
                                            data[i].dmagnitude,
                                            1.0));
    }

    if (extracted_sources != NULL) {
        *extracted_sources = fors_create_sources_table(stars);
        
        assure (!cpl_error_get_code(), return NULL, 
                "Could not create extracted sources table");
    }
    
    return stars;
}

cpl_image* create_initial_confidence_map(const cpl_image *flat_field, const cpl_mask *bpm)
{
    /* Create a confidence map from the master flat */
    cpl_image* confmap = cpl_image_duplicate(flat_field);

    /* Get the mean (which should be 1 if it's a flat but
     * better to check and be sure */
    // double mean = cpl_image_get_mean(confmap);
    double mean = 1.0;
    int nx = (int)cpl_image_get_size_x(confmap);
    int ny = (int)cpl_image_get_size_y(confmap);
    long npts = nx*ny;
    long bcount = 0;
    float *confdata = cpl_image_get_data_float(confmap);
    const cpl_binary *bpmdata = (bpm ? cpl_mask_get_data_const(bpm) : NULL);

    /* This calculation comes from the casu_mkconf() function */
    for (long i = 0; i < npts; i++) {
        if (bpm && bpmdata[i]) {
            confdata[i] = 0;
            ++bcount;
        } else {
            confdata[i] = fmax(0,fmin(110,(int)(100*confdata[i]/mean)));
        }
    }

    cpl_msg_info(cpl_func, "Set %ld / %ld (%.2f%%) bad pixels to zero in confidence map", bcount, npts, 100.0 * bcount / npts);
    return confmap;
}

void update_confidence_map(cpl_image *confmap, const extract_method* em)
{
    int nx = (int)cpl_image_get_size_x(confmap);
    int ny = (int)cpl_image_get_size_y(confmap);
    long npts = nx*ny;
    long zcount = 0;
    float *confdata = cpl_image_get_data_float(confmap);

    for (long i = 0; i < npts; i++) {
        if (confdata[i] < em->conf_thresh) {
          confdata[i] = 0;
          ++zcount;
        }
    }
    cpl_msg_info(cpl_func, "Set %ld / %ld (%.2f%%) low confidence pixels to zero in confidence map", zcount, npts, 100.0 * zcount / npts);
}

cpl_image* create_confidence_map(const cpl_image *flat_field, const cpl_mask *bpm, const extract_method* em)
{
    /* Create a confidence map from the master flat */
    cpl_image* confmap = cpl_image_duplicate(flat_field);

    /* Get the mean (which should be 1 if it's a flat but
     * better to check and be sure */
    double mean = cpl_image_get_mean(confmap);
    int nx = (int)cpl_image_get_size_x(confmap);
    int ny = (int)cpl_image_get_size_y(confmap);
    long npts = nx*ny;
    long bcount = 0, zcount = 0;
    float *confdata = cpl_image_get_data_float(confmap);
    const cpl_binary *bpmdata = (bpm ? cpl_mask_get_data_const(bpm) : NULL);

    /* This calculation comes from the casu_mkconf() function */
    for (long i = 0; i < npts; i++) {
        if (bpm && bpmdata[i]) {
            confdata[i] = 0;
            ++bcount;
        } else {
            confdata[i] = fmax(0,fmin(110,(int)(100*confdata[i]/mean)));
            if (confdata[i] < em->conf_thresh) {
              confdata[i] = 0;
              ++zcount;
            }
        }
    }
    cpl_msg_info(cpl_func, "Set %ld / %ld (%.2f%%) bad pixels to zero in confidence map", bcount, npts, 100.0 * bcount / npts);
    cpl_msg_info(cpl_func, "Set %ld / %ld (%.2f%%) low confidence pixels to zero in confidence map", zcount, npts, 100.0 * zcount / npts);
    return confmap;
}

cpl_image* create_bpm_from_confidence(const cpl_image* confmap)
{
    /* Create a bpm from the confidence map*/
    cpl_image* bpm = cpl_image_duplicate(confmap);
    int nx = (int)cpl_image_get_size_x(confmap);
    int ny = (int)cpl_image_get_size_y(confmap);
    long npts = nx*ny;
    const float *confdata = cpl_image_get_data_float_const(confmap);
    float *bpmdata = cpl_image_get_data_float(bpm);
    for (long i = 0; i < npts; i++) {
        if (confdata[i] <= 0) {
            bpmdata[i] = 1;
        } else {
            bpmdata[i] = 0;
        }
    }
    return bpm;
}

cpl_mask* create_mask_from_bpm(const cpl_image* bpm)
{
    /* Create a mask from the bpm image */
    cpl_size nx = cpl_image_get_size_x(bpm);
    cpl_size ny = cpl_image_get_size_y(bpm);
    cpl_mask* mask = cpl_mask_new(nx, ny);
    cpl_size npts = nx*ny;
    const float *bpmdata = cpl_image_get_data_float_const(bpm);
    cpl_binary *maskdata = cpl_mask_get_data(mask);
    for (long i = 0; i < npts; i++) {
        if (bpmdata[i] < 0.5) {
            maskdata[i] = CPL_FALSE;
        } else {
            maskdata[i] = CPL_TRUE;
        }
    }
    return mask;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief    Work out robust background estimate over a whole input image
 *
 * @param    in_img     Input image
 * @param    bpm        Bad pixel mask for the input image
 * @param    background Background image
 * @param    skymed     Output sky median
 * @param    skysig     Output sky noise
 *
 * @return   CPL_ERROR_NONE if all went well and CPL_ERROR_ILLEGAL_INPUT
 *           if there aren't enough good values to do the calculation.
 *
 * Description:
 *      The image is analysed to work out a robust estimate of the background
 *      median, and sigma.
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code backstats(const cpl_image* in_img, const cpl_image* bpm, const cpl_image* background, double *skymed, double *skysig)
{
    cpl_image *img = cpl_image_duplicate(in_img);
    cpl_mask* mask = create_mask_from_bpm(bpm);
    cpl_image_set_bpm(img, mask);

    if (background) {
        cpl_image_subtract(img, background);
    }

    const cpl_size nx = cpl_image_get_size_x(in_img);
    const cpl_size ny = cpl_image_get_size_y(in_img);
    cpl_mask *image_mask = cpl_image_get_bpm(img);

    /* do kappa-sigma clipping to thresh out basic outliers */
    cpl_size rej_new;

    cpl_size niter = 30;
    for (cpl_size i = 0; i < niter; i++ ) {

        double mad;
        double median     = cpl_image_get_mad(img, &mad);
        double stdev      = mad * CPL_MATH_STD_MAD;

        double kappa_low  = 2.5;
        double lo_cut     = median - kappa_low * stdev;

        double kappa_high = 2.5;
        double hi_cut     = median + kappa_high * stdev;

        cpl_size rej_orig = cpl_image_count_rejected(img);

        if (lo_cut < hi_cut) {
            cpl_mask_threshold_image(image_mask, img, lo_cut, hi_cut, CPL_BINARY_0);
        }
        rej_new = cpl_image_count_rejected(img);

        if (rej_orig == rej_new){
            break;
        }
    }

    /* Set the final answer. Here the normal mean and standard deviation is used
     * as all outliers should be masked and thus the mean is more accurate */
    cpl_error_code e;
    if (rej_new ==  nx * ny) {
        *skymed = 0.;
        *skysig = 0.;
        e       =  CPL_ERROR_ILLEGAL_INPUT;
    } else {
        *skymed = cpl_image_get_mean(img);
        *skysig = cpl_image_get_stdev(img);
        e       = CPL_ERROR_NONE;
    }

    cpl_image_delete(img);

    return e;
}
/**@}*/
