/* $Id: fors_img_science_impl.c,v 1.50 2013-09-10 19:16:03 cgarcia Exp $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-10 19:16:03 $
 * $Revision: 1.50 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_img_basic_science_impl.h>

#include <fors_extract.h>
#include <fors_tools.h>
#include <fors_setting.h>
#include <fors_data.h>
#include <fors_qc.h>
#include <fors_dfs.h>
#include <fors_utils.h>
#include "fiera_config.h"
#include "fors_overscan.h"
#include "fors_ccd_config.h"
#include "fors_detmodel.h"
#include <fors_subtract_bias.h>
#include "fors_bpm.h"
#include "fors_dfs_idp.h"
#include "fors_img_idp.h"

// CASU code is C only
extern "C" {
#include "casu_mods.h"
#include "casu_wcsutils.h"
#include "casu_utils.h"
};

#include <limits>
#include <cpl.h>
#include <math.h>
#include <stdbool.h>
#include <string.h>
#include <set>

/**
 * @addtogroup fors_img_basic_science
 */

/**@{*/

const char *const fors_img_basic_science_name = "fors_img_basic_science";
const char *const fors_img_basic_science_description_short = "Basic reduction of an imaging scientific exposure";
const char *const fors_img_basic_science_author = "ESO PPS Group";
const char *const fors_img_basic_science_email = PACKAGE_BUGREPORT;
const char *const fors_img_basic_science_description = 
"This recipe is used to reduce a direct imaging scientific exposure.\n"
"The master bias calibration is subtracted. The debiased signal is\n"
"then divided by the normalised sky flat field, and the overscan regions,\n"
"if present, are trimmed from the result. The product includes data,\n"
"error, bad pixel mask, and confidence map extensions. The confidence map\n"
"is generated from the normalised flat field, where the mean is scaled to\n"
"a value of 100 and any bad pixels are set to 0.\n"
"\n"
"Input files:\n"
"\n"
"  DO category:               Type:       Explanation:            Number:\n"
"  SCIENCE_IMG                 Raw         Science image                1\n"
"  MASTER_BIAS                 FITS image  Master bias                  1\n"
"  MASTER_SKY_FLAT_IMG         FITS image  Master sky flat field        1\n"
"\n"
"Output files:\n"
"\n"
"  DO category:               Data type:  Explanation:\n"
"  SCIENCE_REDUCED_IMG        FITS image  Reduced science image\n";

/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */

void fors_img_basic_science_define_parameters(cpl_parameterlist *parameters)
{
  // There are no parameters currently defined for this recipe.
  // Silence compiler warning:
  (void)parameters;
  /*
    cpl_parameter *p;
    char *context = cpl_sprintf("fors.%s", fors_img_basic_science_name);
    char *full_name = NULL;
    const char *name;

    cpl_free((void *)context);
  */

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(sci_frame); \
    cpl_frameset_delete(master_bias_frame); \
    cpl_frameset_delete(master_flat_frame); \
    fors_image_delete(&sci); \
    fors_image_delete(&master_bias); \
    fors_image_delete(&master_flat); \
    cpl_image_delete(confbpm); \
    cpl_image_delete(confmap); \
    fors_setting_delete(&setting); \
    cpl_propertylist_delete(qc); \
    cpl_propertylist_delete(ext_header); \
    cpl_propertylist_delete(wcs_header); \
    cpl_propertylist_delete(sci_header); \
} while (0)

/* %%% Removed from cleanup
    cpl_frameset_delete(phot_table); \
*/

/**
 * @brief    Do the processing
 *
 * @param    frames         set of frames
 * @param    parameters     parameters
 */
void fors_img_basic_science(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    /* Raw */
    cpl_frameset *sci_frame      = NULL;
    fors_image *sci              = NULL;

    /* Calibration */
    cpl_frameset *master_bias_frame = NULL;
    fors_image *master_bias   = NULL; 

    cpl_frameset *master_flat_frame = NULL;
    fors_image *master_flat         = NULL; 

    /* Static calibrations */
    
    /* Products */
    cpl_propertylist *qc = cpl_propertylist_new();
    cpl_propertylist *ext_header = cpl_propertylist_new();
    cpl_propertylist *wcs_header = cpl_propertylist_new();
    cpl_propertylist *sci_header = NULL;
    cpl_image *confbpm = NULL;
    cpl_image *confmap = NULL;

    /* Other */
    fors_setting *setting = NULL;
    double avg_airmass = 0.0;

    /* Get parameters */
    cpl_msg_indent_more();

    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );
    
    /* Find raw */
    sci_frame = fors_frameset_extract(frames, SCIENCE_IMG);
    assure( cpl_frameset_get_size(sci_frame) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found", 
            SCIENCE_IMG, cpl_frameset_get_size(sci_frame) );

    /* Find calibration */
    master_bias_frame = fors_frameset_extract(frames, MASTER_BIAS);
    assure( cpl_frameset_get_size(master_bias_frame) == 1, return, 
            "One %s required. %" CPL_SIZE_FORMAT" found", 
            MASTER_BIAS, cpl_frameset_get_size(master_bias_frame) );

    master_flat_frame = fors_frameset_extract(frames, MASTER_SKY_FLAT_IMG);
    assure( cpl_frameset_get_size(master_flat_frame) == 1, return, 
            "One %s required. %" CPL_SIZE_FORMAT" found", 
            MASTER_SKY_FLAT_IMG, cpl_frameset_get_size(master_flat_frame) );

    /* Find static calibrations */

    /* Done finding frames */

    /* Get instrument setting */
    setting = fors_setting_new(cpl_frameset_get_position(sci_frame, 0));
    sci_header = cpl_propertylist_load(cpl_frame_get_filename(
        cpl_frameset_get_position(sci_frame, 0)), 0);
    if (sci_header == NULL) {
        cpl_msg_error(cpl_func, "Failed to load raw header");
        cleanup;
        return;
    }
    fors::fiera_config ccd_config(sci_header);    
    assure( !cpl_error_get_code(), return, "Could not get instrument setting" );

    /* Update RON estimation from bias */
    cpl_propertylist * master_bias_header =
       cpl_propertylist_load(cpl_frame_get_filename(
               cpl_frameset_get_position(master_bias_frame, 0)), 0);
    fors::update_ccd_ron(ccd_config, master_bias_header);
    assure( !cpl_error_get_code(), return, "Could not get RON from master bias"
            " (missing QC DET OUT? RON keywords)");

    master_bias =
            fors_image_load(cpl_frameset_get_position(master_bias_frame, 0));
    assure( !cpl_error_get_code(), return, 
            "Could not load master bias");

    /* Load raw frames */
    fors_image * sci_raw = 
            fors_image_load(cpl_frameset_get_position(sci_frame, 0));
    assure( !cpl_error_get_code(), return, "Could not load science frames");

    /* Check that the overscan configuration is consistent */
    bool perform_preoverscan = !fors_is_preoverscan_empty(ccd_config);

    assure(perform_preoverscan == 
           fors_is_master_bias_preoverscan_corrected(master_bias_header),
           return, "Master bias overscan configuration doesn't match science");

    /* Copy in the RON keywords */
    cpl_propertylist_copy_property_regexp(qc, master_bias_header, "^ESO QC RON.*", false);
    cpl_propertylist_delete(master_bias_header);
    
    /* Create variance */
    std::vector<double> overscan_levels; 
    if(perform_preoverscan)
        overscan_levels = fors_get_bias_levels_from_overscan(sci_raw, ccd_config);
    else
        overscan_levels = fors_get_bias_levels_from_mbias(master_bias, ccd_config);

    double average_overscan_level = 0;
    fors_image_variance_from_detmodel(sci_raw, ccd_config, overscan_levels,
                                      &average_overscan_level);
    cpl_propertylist_update_double(qc, "ESO DRS AVG_OVERSCAN", average_overscan_level);
    
    /* Get the A/D saturated pixels */
    cpl_mask *bpm = cpl_image_get_bpm(sci_raw->data);
 
    cpl_mask *sci_sat_mask = cpl_mask_threshold_image_create(sci_raw->data,
                                            65534., std::numeric_limits<double>::max());
    cpl_mask *sci_sat_0_mask = cpl_mask_threshold_image_create(sci_raw->data,
                                             -std::numeric_limits<double>::max(),
                                             std::numeric_limits<double>::min());
    cpl_mask_or(sci_sat_mask, sci_sat_0_mask);
    cpl_mask_delete(sci_sat_0_mask);
    cpl_mask_or(bpm, sci_sat_mask);
    cpl_mask_delete(sci_sat_mask);

    /*Subtract overscan */
    if(perform_preoverscan)
        sci = fors_subtract_prescan(sci_raw, ccd_config);
    else 
    {
        sci = fors_image_duplicate(sci_raw); 
        //The rest of the recipe assumes that the images carry a bpm.
        fors_bpm_image_make_explicit(sci); 
    }
    assure( !cpl_error_get_code(), return, "Could not subtract overscan");

    /* Trim pre/overscan */
    if(perform_preoverscan)
        fors_trimm_preoverscan(sci, ccd_config);
    fors_image_delete(&sci_raw);
    assure( !cpl_error_get_code(), return, "Could not trimm overscan");

    /* Propagate WCS information properly */
    fors_dfs_copy_wcs(wcs_header, cpl_frameset_get_position(sci_frame, 0), 0);
    if(perform_preoverscan)
        fors_trimm_preoverscan_fix_wcs(wcs_header, ccd_config);

    /* Subtract bias */
    fors_subtract_bias(sci, master_bias);
    assure( !cpl_error_get_code(), return, "Could not subtract master bias");
    fors_image_delete(&master_bias);

    /* Load master flat */
    master_flat =
            fors_image_load(cpl_frameset_get_position(master_flat_frame, 0));

    assure( !cpl_error_get_code(), return, "Could not load master flat");
    
    /* Divide by normalized flat */
    fors_image_divide_scalar(master_flat,
                             fors_image_get_median(master_flat, NULL), -1.0);
    assure( !cpl_error_get_code(), return, "Could not normalise flat");

    fors_image_divide(sci, master_flat);
    assure( !cpl_error_get_code(), return, "Could not divide by master flat");
    
    /* Create a confidence map from the master flat */
    const cpl_mask *bpm_for_conf = cpl_image_get_bpm_const(sci->data);
    confmap = create_initial_confidence_map(master_flat->data, bpm_for_conf);
    confbpm = create_bpm_from_confidence(confmap);

    /* QC */
    fors_qc_start_group(qc, fors_qc_dic_version, setting->instrument);
    
    fors_qc_write_group_heading(cpl_frameset_get_position(sci_frame, 0),
                                SCIENCE_REDUCED_IMG,
                                setting->instrument);
    assure( !cpl_error_get_code(), return, "Could not write %s QC parameters", 
            SCIENCE_REDUCED_IMG);

    /* Write the relevant settings to the QC header */
    fors_setting_write_qc(setting, qc);

    // Get the ID of the night, from the point of view of the photometry calculations.
    // Despite the fact that the fors_photometry product puts this night_id into
    // a column called MJD_NIGHT, it's actually a JD.
    int this_night = fors_photometry_get_night_id(sci_header);
    cpl_msg_info(cpl_func, "this_night = %d", this_night);
    if (this_night == 0) {
        cpl_error_reset();
    } else {
        fors_qc_write_qc_double(qc,
                             this_night - 2400000.5 + 1.0,
                             "QC.MJDOBS",
                             NULL,
                             "MJD of night",
                             setting->instrument);
    }

    fors_qc_end_group();

    /* Save SCIENCE_REDUCED, PHOT_BACKGROUND_SCI_IMG */

/* %%% */

    avg_airmass = fors_get_airmass(sci_header);

    cpl_propertylist_update_double(qc, "AIRMASS", avg_airmass);

/* %%% */

    fors_dfs_add_exptime(qc, cpl_frameset_get_position(sci_frame, 0), 0.);

    /* Add unique name for two detectors and exposures */
    const char *det_exp = NULL;
    det_exp = fors_get_det_exp(sci_header);
    cpl_propertylist_append_string(qc, "ESO PRO DET_EXP", det_exp);
    /* Add TITLE keyword */
    const char *title = NULL;
    title = fors_get_title(sci_header);
    cpl_propertylist_append_string(qc, "TITLE", title);

    /* The WCS headers go into each extension rather than into the primary header */
    cpl_propertylist_copy_property_regexp(ext_header, wcs_header, ".*", false);

    const cpl_frame * inherit_frame = cpl_frameset_get_position_const(sci_frame, 0);

    const char *del_kwords =
            perform_preoverscan ? "^ESO DET OUT[0-9]+ (OV|PR)SC[XY]$" : NULL;
    fors_dfs_save_image_err_2(frames, sci, confbpm, confmap,
        SCIENCE_REDUCED_IMG,
        qc, ext_header, parameters, fors_img_basic_science_name,
        inherit_frame, del_kwords);
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            SCIENCE_REDUCED_IMG);

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when writing QC parameters: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    fors_image_delete(&sci);
    cpl_image_delete(confbpm); confbpm = NULL;
    cpl_image_delete(confmap); confmap = NULL;
    
    cleanup;
    return;
}
