/* $Id: $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-10 19:16:03 $
 * $Revision: 1.50 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_img_photom_science_impl.h>
#include <fors_zeropoint_utils.h>

#include <fors_extract.h>
#include <fors_identify.h>
#include <fors_tools.h>
#include <fors_setting.h>
#include <fors_data.h>
#include <fors_qc.h>
#include <fors_dfs.h>
#include <fors_utils.h>
#include "fiera_config.h"
#include "fors_overscan.h"
#include "fors_ccd_config.h"
#include "fors_detmodel.h"
#include "fors_trimm_illum.h"
#include <fors_subtract_bias.h>
#include "fors_bpm.h"
#include "fors_dfs_idp.h"
#include "fors_img_idp.h"
#include "fors_std_cat.h"
#include "fors_instrument.h"

#include <cpl.h>
#include <math.h>
#include <stdbool.h>
#include <string.h>
#include <set>

/**
 * @addtogroup fors_img_photom_science
 */

/**@{*/

const char *const fors_img_photom_science_name = "fors_img_photom_science";
const char *const fors_img_photom_science_description_short = "Photometric calibration of an imaging scientific exposure";
const char *const fors_img_photom_science_author = "ESO PPS Group";
const char *const fors_img_photom_science_email = PACKAGE_BUGREPORT;
const char *const fors_img_photom_science_description = 
"This recipe performs photometric calibration of the input science frame. If an\n"
"adequate number of suitable stars from the Gaia synthetic photometry catalog\n"
"are available, an in-situ calibration is performed by matching the sources in\n"
"OBJECT_TABLE_SCI_IMG against that catalog. Otherwise, data from either the\n"
"PHOT_COEFF_TABLE or STATIC_PHOT_COEFF_TABLE are used. As a last step, and if\n"
"the input files contain a DETECTOR_ILLUMINATED_REGION file, the recipe will\n"
"trim the image to the rectangular area of the detector that actually receives\n"
"light from the sky.  The output SCIENCE_REDUCED_IMG_WCS_PHOTOM image is in IDP\n"
"format, containing trimmed science, error, and BPM extensions taken from the\n"
"input, and a weight extension that is generated from the confidence map via the\n"
"weights threshold (--idp_weights_threshold). The confidence map itself is not\n"
"propagated into the final product.\n"
"\n"
"Input files:\n"
"\n"
"  DO category:               Type:       Explanation:                        Number:\n"
"  SCIENCE_REDUCED_IMG_WCS     FITS image  Reduced science image with accurate wcs  1\n"
"  OBJECT_TABLE_SCI_IMG        FITS table  Extracted sources properties             1\n"
"  SOURCES_SCI_IMG_WCS         FITS table  Unfiltered source list                   1\n"
"  PHOT_COEFF_TABLE            FITS table  Observed extinction coefficients         0+\n"
"  PHOT_TABLE                  FITS table  Filter ext. coeff, color                 1\n"
"  EXTINCTION_PER_NIGHT        FITS table  Extinction per night                     0+\n"
"  STATIC_PHOT_COEFF_TABLE     FITS table  Static filters photometry coefficients   0+\n"
"  DETECTOR_ILLUMINATED_REGION FITS table  Table with detector illuminated regions  0+\n"
"\n"
"Output files:\n"
"\n"
"  DO category:               Data type:  Explanation:\n"
"  SCIENCE_REDUCED_IMG_WCS_PHOTOM\n"
"                             FITS image  Reduced science image in IDP-compliant format\n"
"  PHOT_STD_PHOTOM            FITS table  List of Gaia sources with synthetic photometry in the field of view\n"
"  PHOT_STARS_PHOTOM          FITS table  List of FORS sources,\n"
"                                         including zeropoint values for sources matched to Gaia.\n";



static const
fors_img_idp_tags IDP_Products_Filenames(SCIENCE_REDUCED_IMG_WCS_PHOTOM, fors_img_photom_science_name);

/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */

void fors_img_photom_science_define_parameters(cpl_parameterlist *parameters)
{
    cpl_parameter *p;
    char *context = cpl_sprintf("fors.%s", fors_img_photom_science_name);
    char *full_name = NULL;
    const char *name;

    name = "magcutE";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Zeropoint absolute cutoff (magnitude)",
                                context,
                                1.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "magcutk";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Zeropoint kappa rejection parameter",
                                context,
                                5.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "magsyserr";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Systematic error in magnitude",
                                context,
                                0.01);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "cacheloc";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "Location for the standard star cache",
                                context,
                                ".");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "idp_weights_threshold";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Every pixel in the MASTER_SKY_FLAT_IMG below "
                                "the threshold will be put to 0 in the weight-map, "
                                "1 otherwise.",
                                context,
                                0.5);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(sci_frame); \
    cpl_frameset_delete(extinct_frame); \
    cpl_frameset_delete(illum_region_frames); \
    cpl_frameset_delete(phot_table); \
    fors_image_delete(&sci); \
    cpl_table_delete(extinct_table); \
    cpl_table_delete(phot); \
    cpl_table_delete(sources); \
    cpl_image_delete(confmap); \
    cpl_image_delete(confbpm); \
    fors_star_list_delete(&stars, fors_star_delete); \
    cpl_free((void *)context); \
    fors_setting_delete(&setting); \
    cpl_propertylist_delete(qc); \
    cpl_propertylist_delete(ext_header); \
    cpl_propertylist_delete(wcs_header); \
    cpl_propertylist_delete(sci_header); \
    cpl_propertylist_delete(img_header); \
} while (0)

/**
 * @brief    Do the processing
 *
 * @param    frames         set of frames
 * @param    parameters     parameters */
void fors_img_photom_science(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    /* Inputs */
    cpl_frameset *sci_frame      = NULL;
    cpl_frameset *phot_frame     = NULL;
    cpl_frameset *sources_frame  = NULL;
    fors_image *sci              = NULL;
    cpl_image *confmap           = NULL;

    /* Calibration */
    cpl_frameset *phot_table    = NULL;
    cpl_frameset *extinct_frame = NULL;
    cpl_table *extinct_table    = NULL;

    /* Static calibrations */
    cpl_frameset *illum_region_frames = NULL;
    cpl_table * illum_region = NULL;
    
    /* Products */
    cpl_propertylist *qc = cpl_propertylist_new();
    cpl_propertylist *ext_header = cpl_propertylist_new();
    cpl_propertylist *wcs_header = cpl_propertylist_new();
    cpl_propertylist *sci_header = NULL;
    cpl_propertylist *img_header = NULL;
    cpl_table *phot = NULL;
    cpl_image *confbpm = NULL;
    cpl_table *sources = NULL;

    /* Parameters */
    double           magsyserr;
    double           cutoffe, cutoffk;

    /* Other */
    char *context   = cpl_sprintf("fors.%s", fors_img_photom_science_name);
    fors_star_list *stars = NULL;
    fors_setting *setting = NULL;
    double avg_airmass = 0.0;
    double average_overscan_level = 0.0;
    char *name;

    /* QC summary parameters */
    std::set<int> night_id_set;

    /* Get parameters */
    cpl_msg_indent_more();
    name = cpl_sprintf("%s.%s", context, "magcutE");
    cutoffe = dfs_get_parameter_double_const(parameters, name);
    cpl_free((void *)name); name = NULL;
    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );
        
    cpl_msg_indent_more();
    name = cpl_sprintf("%s.%s", context, "magcutk");
    cutoffk = dfs_get_parameter_double_const(parameters, name);
    cpl_free((void *)name); name = NULL;
    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );
        
    cpl_msg_indent_more();
    name = cpl_sprintf("%s.%s", context, "magsyserr");
    magsyserr = dfs_get_parameter_double_const(parameters, name);
    cpl_free((void *)name); name = NULL;
    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );

    name = cpl_sprintf("%s.%s", context, "idp_weights_threshold");
    const double idp_weights_threshold = dfs_get_parameter_double_const(parameters, name);
    cpl_free((void *)name); name = NULL;

    name = cpl_sprintf("%s.%s", context, "cacheloc");
    const char* cacheloc = dfs_get_parameter_string_const(parameters, name);
    cpl_free((void *)name); name = NULL;

    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );
    assure( magsyserr >= 0, return, 
            "Input systematic error (magsyserr=%f) cannot be negative",
            magsyserr);
    
    /* Check the cache location is writable */
    assure( access(cacheloc, R_OK + W_OK + X_OK) == 0, return,
            "Cache location %s inacessible", cacheloc);

    /* Find raw */
    sci_frame = fors_frameset_extract(frames, SCIENCE_REDUCED_IMG_WCS);
    assure( cpl_frameset_get_size(sci_frame) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found", 
            SCIENCE_REDUCED_IMG_WCS, cpl_frameset_get_size(sci_frame) );

    phot_frame = fors_frameset_extract(frames, PHOTOMETRY_TABLE);
    assure( cpl_frameset_get_size(phot_frame) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found", 
            PHOTOMETRY_TABLE, cpl_frameset_get_size(phot_frame) );

    sources_frame = fors_frameset_extract(frames, SOURCES_SCI_IMG_WCS);
    assure( cpl_frameset_get_size(sources_frame) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found", 
            SOURCES_SCI_IMG_WCS, cpl_frameset_get_size(sources_frame) );

    /* Find calibration */
    phot_table = fors_frameset_extract(frames, PHOT_TABLE);
    assure( cpl_frameset_get_size(phot_table) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found",
            PHOT_TABLE, cpl_frameset_get_size(phot_table));

    extinct_frame = fors_frameset_extract(frames, EXTINCTION_PER_NIGHT);
    assure( cpl_frameset_get_size(extinct_frame) <= 1, return, 
            "Zero or one %s required. %" CPL_SIZE_FORMAT" found",
            EXTINCTION_PER_NIGHT, cpl_frameset_get_size(extinct_frame));

    /* Find static calibrations */
    illum_region_frames = fors_frameset_extract(frames, DETECTOR_ILLUMINATED_REGION);
    if(cpl_frameset_get_size(illum_region_frames) == 0)
    {
        cpl_msg_warning(cpl_func, "Input frames do not include "
                        DETECTOR_ILLUMINATED_REGION". Images won't be trimmed "
                        " to illuminated region.");
    }
    else if(cpl_frameset_get_size(illum_region_frames) != 1)
    {
        cpl_msg_error(cpl_func, "At most 1 frame of type "
                      DETECTOR_ILLUMINATED_REGION" is allowed");
        cleanup;
        return;
    } else {
        cpl_frame * illum_region_frame = cpl_frameset_get_position(
            illum_region_frames, 0);
        const char * fname = cpl_frame_get_filename(illum_region_frame);
        illum_region = cpl_table_load(fname, 1, 1);
    }

    assure( cpl_frameset_get_size(illum_region_frames), return, 
            "Input frames do not include " DETECTOR_ILLUMINATED_REGION". "
            "IDP generation cannot be performed");


    /* Done finding frames */

    /* Get instrument setting */
    setting = fors_setting_new(cpl_frameset_get_position(sci_frame, 0));
    sci_header = cpl_propertylist_load(cpl_frame_get_filename(
        cpl_frameset_get_position(sci_frame, 0)), 0);
    if (sci_header == NULL) {
        cpl_msg_error(cpl_func, "Failed to load science header");
        cleanup;
        return;
    }
    img_header = cpl_propertylist_load(cpl_frame_get_filename(
        cpl_frameset_get_position(sci_frame, 0)), 1);
    if (img_header == NULL) {
        cpl_msg_error(cpl_func, "Failed to load image data header");
        cleanup;
        return;
    }

    /* Load frames */
    fors_dfs_copy_wcs(wcs_header, cpl_frameset_get_position(sci_frame, 0), 1);

    /* FIXME this is temporary loading code */
    cpl_image *sci_data = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG_WCS, CPL_TYPE_FLOAT, "IMAGE.DAT", 0);
    cpl_image *sci_var = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG_WCS, CPL_TYPE_FLOAT, "IMAGE.ERR", 0);
    cpl_image_power(sci_var, 2);
    sci = fors_image_new(sci_data, sci_var);

    phot = dfs_load_table(frames, PHOTOMETRY_TABLE, 1);

    int nrows = (int)cpl_table_get_nrow(phot);
    cpl_msg_info(cpl_func, "Number of stars in object list: %d", nrows);

    sources = dfs_load_table(frames, SOURCES_SCI_IMG_WCS, 1);

    // sci = fors_image_load(cpl_frameset_get_position(sci_frame, 0));
    assure( !cpl_error_get_code(), return, "Could not load science frames");

    /* Coefficients loaded from PHOT_TABLE */
    double color_term = 0.0, dcolor_term = 0.0;
    double ext_coeff = 0.0, dext_coeff = 0.0;
    double expected_zeropoint = 0.0, dexpected_zeropoint = 0.0;
    cpl_errorstate  local_ers = cpl_errorstate_get();

    /* Load filter coefficients */
    fors_phot_table_load(cpl_frameset_get_position(phot_table, 0), setting,
                         &color_term, &dcolor_term,
                         &ext_coeff, &dext_coeff,
                         &expected_zeropoint, &dexpected_zeropoint);
    assure(                             cpl_errorstate_is_equal(local_ers),
                                        return,
                                        "Could not load photometry table" );

    cpl_msg_info(cpl_func, "Color term from %s: %f +- %f", PHOT_TABLE,
        color_term, dcolor_term);
    cpl_msg_info(cpl_func, "Expected zeropoint from %s: %f +- %f", PHOT_TABLE,
        expected_zeropoint, dexpected_zeropoint);

    char filter_band = fors_instrument_filterband_get_by_setting(setting);
    // The g_HIGH filter is reported as V, so we exclude it here
    if (!setting->filter_name || !strcmp(setting->filter_name, "g_HIGH")) {
        filter_band = '?';
    }
    cpl_error_reset();
    if (setting->filter_name) {
        cpl_msg_info(cpl_func, "Filter %s is %c", setting->filter_name, filter_band);
    } else {
        cpl_msg_info(cpl_func, "Unknown filter");
    }

    double dzp = 0.0;
    int nzp = 0;
    double zp = 0.0;

    /* Calculate the zeropoint */
    cpl_table *phot_stds_to_save = NULL;
    cpl_table *updated_stars_to_save = NULL;
    calculate_zp_from_gaia(filter_band, cutoffe, cutoffk, phot, &phot_stds_to_save,
                           &updated_stars_to_save, img_header,
                           cacheloc, color_term, dcolor_term, zp, dzp, nzp);

    /* Add unique name for two detectors and exposures */
    const char *det_exp = NULL;
    det_exp = fors_get_det_exp(sci_header);
    cpl_propertylist_append_string(wcs_header, "ESO PRO DET_EXP", det_exp);
    /* Add TITLE keyword */
    const char *title = NULL;
    title = fors_get_title(sci_header);
    cpl_propertylist_append_string(wcs_header, "TITLE", title);

    if (cpl_propertylist_has(wcs_header, "ESO PRO DET_EXP")) {
        cpl_propertylist_erase(wcs_header, "ESO PRO DET_EXP");
    }
    cpl_propertylist_append_string(qc, "ESO PRO DET_EXP", det_exp);
    if (cpl_propertylist_has(wcs_header, "TITLE")) {
        cpl_propertylist_erase(wcs_header, "TITLE");
    }
    cpl_propertylist_append_string(qc, "TITLE", title);


    double sky_mag = 0;
    //double sky_mag_rms = 0;
    double ellipticity = 0;
    double bkg_rms = 1;

    /* Load header values */
    if (cpl_propertylist_has(sci_header, "ESO DRS AVG_OVERSCAN")) {
        average_overscan_level = cpl_propertylist_get_double(sci_header, "ESO DRS AVG_OVERSCAN");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO DRS AVG_OVERSCAN found. "
            "Assuming zero.");
    }

    if (cpl_propertylist_has(sci_header, "ESO QC SKYMED")) {
        sky_mag = cpl_propertylist_get_double(sci_header, "ESO QC SKYMED");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO QC SKYMED found. "
            "Assuming zero.");
    }

    /*
    if (cpl_propertylist_has(sci_header, "ESO QC SKYRMS")) {
        sky_mag_rms = cpl_propertylist_get_double(sci_header, "ESO QC SKYRMS");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO QC SKYRMS found. "
            "Assuming zero.");
    }
    */

    if (cpl_propertylist_has(sci_header, "ESO QC IMGQUELL")) {
        ellipticity = cpl_propertylist_get_double(sci_header, "ESO QC IMGQUELL");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO QC IMGQUELL found. "
            "Assuming zero.");
    }

    if (cpl_propertylist_has(sci_header, "ESO QC SKY_NOISE")) {
        bkg_rms = cpl_propertylist_get_double(sci_header, "ESO QC SKY_NOISE");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO QC SKY_NOISE found. "
            "Assuming 1.");
    }

    /* Load the confidence map */
    confmap = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG_WCS, CPL_TYPE_FLOAT, "IMAGE.CNF", 0);
    confbpm = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG_WCS, CPL_TYPE_FLOAT, "IMAGE.BPM", 0);

    /* QC */
    fors_qc_start_group(qc, fors_qc_dic_version, setting->instrument);
    
    /* Write the relevant settings to the QC header */
    fors_setting_write_qc(setting, qc);

    // Get the ID of the night, from the point of view of the photometry calculations.
    // Despite the fact that the fors_photometry product puts this night_id into
    // a column called MJD_NIGHT, it's actually a JD.
    int this_night = fors_photometry_get_night_id(sci_header);
    cpl_msg_info(cpl_func, "this_night = %d", this_night);
    if (this_night == 0) {
        cpl_error_reset();
    } else {
        fors_qc_write_qc_double(qc,
                             this_night - 2400000.5 + 1.0,
                             "QC.MJDOBS",
                             NULL,
                             "MJD of night",
                             setting->instrument);

        if (cpl_frameset_get_size(extinct_frame) > 0) {
            extinct_table = cpl_table_load(
                                      cpl_frame_get_filename(
                                        cpl_frameset_get_position(extinct_frame, 0)),
                                      1, 0);

            bool found_ext = false;
            int num_ext = 0;
            for(cpl_size i = 0; i < cpl_table_get_nrow(extinct_table); i++) {
                int night_id = cpl_table_get_int(extinct_table, "MJD_NIGHT", i, NULL);
                cpl_msg_info(cpl_func, "night_id = %d", night_id);
                if (night_id == this_night) {
                    // Count the number of extinction frames for this night
                    ++num_ext;
                    if (!found_ext) {
                        // They will all have the same value for EXT and DEXT so we
                        // only need to record it once
                        found_ext = true;
                        double ext = cpl_table_get_double(extinct_table, "EXT", i, NULL);
                        fors_qc_write_qc_double(qc,
                                             ext,
                                             "QC.EXTINCTION",
                                             NULL,
                                             "Extinction for this night",
                                             setting->instrument);
                        double dext = cpl_table_get_double(extinct_table, "DEXT", i, NULL);
                        fors_qc_write_qc_double(qc,
                                             dext,
                                             "QC.EXTINCTION_ERR",
                                             NULL,
                                             "Extinction error for this night",
                                             setting->instrument);
                    }
                }
                night_id_set.insert(night_id);
            }

            fors_qc_write_qc_int(qc,
                                 num_ext,
                                 "QC.NUM_EXT",
                                 NULL,
                                 "Number of extinction measurements for this night",
                                 setting->instrument);

            fors_qc_write_qc_int(qc,
                                 night_id_set.size(),
                                 "QC.NUM_NIGHTS",
                                 NULL,
                                 "Number of nights in extinction table",
                                 setting->instrument);

            fors_qc_write_qc_int(qc,
                                 cpl_table_get_nrow(extinct_table),
                                 "QC.NUM_FIELDS",
                                 NULL,
                                 "Number of fields in extinction table",
                                 setting->instrument);
        }
    }

    fors_qc_end_group();

    avg_airmass = fors_get_airmass(sci_header);
    cpl_propertylist_update_double(qc, "AIRMASS", avg_airmass);

    /* Make sure the average overscan value propagates through */
    cpl_propertylist_update_double(qc, "ESO DRS AVG_OVERSCAN", average_overscan_level);

    /* Copy the QC headers through */
    cpl_propertylist_copy_property_regexp(qc, sci_header, "ESO QC.*", false);

    /* And the APCOR headers */
    cpl_propertylist_copy_property_regexp(qc, sci_header, "APCOR.*", false);

    fors_dfs_add_exptime(qc, cpl_frameset_get_position(sci_frame, 0), 0.);

    /* The WCS headers go into each extension rather than into the primary header */
    cpl_propertylist_copy_property_regexp(ext_header, wcs_header, ".*", false);

    double image_fwhm = 0; //in arcsec
    double image_fwhm_pix = 0; //in pixel
    stars = fors_create_star_list(phot);
    const bool fwhm_available = fors_img_idp_get_image_psf_fwhm(stars,
        setting, sci_header, &image_fwhm, &image_fwhm_pix);
    
    
    
    if (phot_stds_to_save != NULL) {
        cpl_table_erase_column(phot_stds_to_save, "Source");
        fors_dfs_save_table(frames, phot_stds_to_save, PHOT_STD_PHOTOM,
               qc, parameters, fors_img_photom_science_name,
            cpl_frameset_get_position(sci_frame, 0));
    }
    cpl_table_delete(phot_stds_to_save);
    if (updated_stars_to_save != NULL) {
        fors_dfs_save_table(frames, updated_stars_to_save, PHOT_STARS_PHOTOM,
               qc, parameters, fors_img_photom_science_name,
            cpl_frameset_get_position(sci_frame, 0));
    }
    cpl_table_delete(updated_stars_to_save);

    /* Trimm non-illuminated areas */
    if (fwhm_available) {
        if (!illum_region) {
            cpl_msg_warning(cpl_func, "Unable to read illuminated region table, aborting IDP generation");
            cpl_error_reset();
        } else {
            cpl_msg_info(cpl_func, "image_fwhm = %f, image_fwhm_pix = %f", image_fwhm, image_fwhm_pix);
            /* Trimming science image. 
             * Note that it also updates the header WCS in wcs_header */
            bool match = fors_trimm_non_illum(sci, wcs_header, setting,
                                              illum_region);

            /* Trim the confidence map */
            fors_trimm_non_illum(&confmap, NULL, setting, illum_region);

            /* And the bpm */
            fors_trimm_non_illum(&confbpm, NULL, setting, illum_region);

            if (match) {
                fors_img_idp_quality qty(image_fwhm, image_fwhm_pix, ellipticity,
                    bkg_rms, sky_mag, average_overscan_level, sources);

                const cpl_frame * inherit_frame = cpl_frameset_get_position_const(sci_frame, 0);

                /* Use sci_frame to grab ESO QC RON */
                fors_img_idp_save(frames, sci, confbpm, qc, wcs_header, parameters, inherit_frame,
                    cpl_frameset_get_position(sci_frame, 0), confmap,
                    idp_weights_threshold, IDP_Products_Filenames, qty, zp, dzp, nzp, filter_band);
            }
            else {
                cpl_msg_warning(cpl_func, "Detector & collimator not found in "
                    DETECTOR_ILLUMINATED_REGION " table, aborting IDP generation");
                cpl_error_reset();
            }
        }
    }
    else {
        cpl_msg_warning(cpl_func, "Unable to calculate PSF_FWHM, aborting IDP generation");
        cpl_error_reset();
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when generating IDPs: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    if (illum_region) {
        cpl_table_delete(illum_region);
    }

    fors_image_delete(&sci);
    cpl_image_delete(confmap); confmap = NULL;
    cpl_image_delete(confbpm); confbpm = NULL;
    
    cpl_error_reset();

    cleanup;
    return;
}
