/*
 * This file is part of the FORS pipeline
 * Copyright (C) 2014-2023 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_img_stack_impl.h>

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include "hdrl.h"
#include "fors_extract.h"
#include "fors_utils.h"
#include "fors_tools.h"
#include "fors_dfs.h"
#include "fors_data.h"
#include "fors_instrument.h"
#include "fors_zeropoint_utils.h"
#include "fors_img_idp.h"
#include <cpl.h>

#include <list>
#include <vector>
#include <map>
#include <set>
#include <string>
#include <numeric>
#include <algorithm>

/*----------------------------------------------------------------------------*/
/**
  @defgroup fors_img_stack Pixel resampling recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/
/**
   @brief    Check if an error has happened and returns error kind and location
   @param    val input value
   @return   0 if no error is detected,-1 else
 */


/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/

/*
static cpl_table*
fors_resample_get_table_from_frameset(const cpl_frame* data_frm,
                const cpl_frame* errs_frm,
                const cpl_frame* qual_frm,
                const cpl_size data_ext_id,
                const cpl_size errs_ext_id,
                const cpl_size qual_ext_id,
                const cpl_boolean is_variance,
                const cpl_boolean subtract_bkg,
                const int edge_trim);
static cpl_wcs *
fors_resample_get_wcs_from_frameset(cpl_frameset* frameset);
*/

/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/
/* resampling params default values */
#define RENKA_CRITICAL_RADIUS 1.25 //As in MUSE (see user manual)
#define LANCZOS_KERNEL_SIZE 2 //

#define DRIZZLE_DOWN_SCALING_FACTOR_X 0.8
#define DRIZZLE_DOWN_SCALING_FACTOR_Y 0.8
#define LOOP_DISTANCE 1
#define RECIPE_NAME "fors.fors_img_stack"
/* Default field margin (in percent), if the user does not specify any. 5
 * percent is also used in the software package swarp */
#define FIELDMARGIN 5.
#define EDGETRIM 0

static const char *input_cats[3] = {"SCIENCE_REDUCED_IMG", "SCIENCE_REDUCED_IMG_WCS", "SCIENCE_REDUCED_IMG_WCS_PHOTOM"};

#define OUTPUT_CAT "SCIENCE_STACKED_IMG"

/* extension names for the three images */
#define EXTNAME_DATA  "IMAGE.DAT"
#define EXTNAME_BPM   "IMAGE.BPM"
#define EXTNAME_ERROR "IMAGE.ERR"
#define EXTNAME_EXPMAP "EXPMAP"
#define EXTNAME_DATA_COMMENT  "This extension contains data values"
#define EXTNAME_BPM_COMMENT   "This extension contains bad pixel values"
#define EXTNAME_ERROR_COMMENT "This extension contains data errors"
#define EXTNAME_EXPMAP_COMMENT "This extension contains the exposure map"

/**
 * @addtogroup fors_img_stack */

/**@{*/

const char *const fors_img_stack_name = "fors_img_stack";
const char *const fors_img_stack_description_short = "Stacking of reduced imaging scientific exposures, with additional source detection and photometric calibration";
const char *const fors_img_stack_author = "ESO PPS Group";
const char *const fors_img_stack_email = PACKAGE_BUGREPORT;
const char *const fors_img_stack_description = 
"This recipe stacks or mosaics individual reduced images via pixel re-sampling.\n"
"Various interpolations methods are available and are detailed below.  After\n"
"stacking, the source finding routine is run on the combined image, and then the\n"
"photometric calibration is performed as for fors_img_photom_science.\n"
"\n"
"Input files:\n"
"\n"
"  DO category:                 Explanation:                                  Number:\n"
"  SCIENCE_REDUCED_IMG_WCS_PHOTOM\n"
"    or SCIENCE_REDUCED_IMG_WCS\n"
"    or SCIENCE_REDUCED_IMG     Science images(s)                                   1+\n"
"\n"
"  PHOT_TABLE                   FITS table  Filter ext. coeff, color                1\n"
"  STATIC_PHOT_COEFF_TABLE      FITS table  Static filters photometry coefficients  1\n"
"\n"
"Output files:\n"
"\n"
"  DO category:                 Explanation:\n"
"  SCIENCE_STACKED_IMG          Stacked multi extension science image containing\n"
"                               the data, the associated error, the bad pixel\n"
"                               mask, and the exposure map in different extensions.\n"
"  SOURCES_SCI_IMG_STACK        Unfiltered source list\n"
"  PHOT_STD_PHOTOM_STACK        List of Gaia sources with synthetic photometry in the field of view.\n"
"  PHOT_STARS_PHOTOM_STACK      List of FORS sources,\n"
"                               including zeropoint values for sources matched to Gaia.\n"
"\n"
"                                                                                \n"
"Usage of the recipe:                                                            \n"
"                                                                                \n"
"There are currently six interpolation methods:                                  \n"
"                                                                                \n"
"(--method=NEAREST): There is                                                    \n"
"                  no control parameter for this method                          \n"
"                                                                                \n"
"                  The algorithm does not use any weighting functions but simply \n"
"                  uses the value of the nearest neighbor inside a output voxel  \n"
"                  center as the final output value. If there is no nearest      \n"
"                  neighbor inside the voxel (but only outside), the voxel is    \n"
"                  marked as bad. This considerably speeds up the algorithm.     \n"
"                                                                                \n"
"(--method=LINEAR): Controlled by parameters:                                    \n"
"                  method.loop-distance; method.use-errorweights                 \n"
"                                                                                \n"
"                  The algorithm uses a linear inverse distance weighting        \n"
"                  function (1/r) for the interpolation. The method.loop-distance\n"
"                  controls the number of surrounding pixels that are taken into \n"
"                  account on the final grid, e.g. a method.loop-distance of 1   \n"
"                  uses 3 pixels (x - 1, x, x + 1) in both dimensions.\n"
"                  Moreover, if method.use-errorweights is set to TRUE, an\n"
"                  additional weight defined as 1/variance is taken into account.\n"
"                                                                                \n"
"(--method=QUADRATIC): Controlled by parameters:                                 \n"
"                  method.loop-distance; method.use-errorweights                 \n"
"                                                                                \n"
"                  The algorithm uses a quadratic inverse distance weighting     \n"
"                  function (1/r^2) for the interpolation. The                   \n"
"                  method.loop-distance controls the number of surrounding pixels\n"
"                  that are taken into account on the final grid, e.g. a         \n"
"                  method.loop-distance of 1 uses 3 pixels (x - 1, x, x + 1) in  \n"
"                  both dimensions. Moreover, if method.use-errorweights is set  \n"
"                  to TRUE, an additional weight defined as 1/variance is taken  \n"
"                  into account.\n"
"                                                                                \n"
"(--method=RENKA): Controlled by parameters:                                     \n"
"                  method.renka.critical-radius; method.loop-distance;           \n"
"                  method.use-errorweights                                       \n"
"                                                                                \n"
"                  The algorithm uses a modified Shepard-like distance weighting \n"
"                  function following Renka for the interpolation. The           \n"
"                  method.renka.critical-radius defines the distance beyond which\n"
"                  the weights are set to 0 and the pixels are therefore not     \n"
"                  taken into account.  The method.loop-distance controls the    \n"
"                  number of surrounding pixels that are taken into account on   \n"
"                  the final grid, e.g. a method.loop-distance of 1 uses 3 pixels\n"
"                  (x - 1, x, x + 1) in both dimensions. Moreover, \n"
"                  if method.use-errorweights is set to TRUE, an additional      \n"
"                  weight defined as 1/variance is taken into account.           \n"
"                                                                                \n"
"(--method=LANCZOS): Controlled by parameters:                                   \n"
"                  method.loop-distance; method.use-errorweights;                \n"
"                  method.lanczos.kernel-size                                    \n"
"                                                                                \n"
"                  The algorithm uses a restricted SINC distance weighting       \n"
"                  function (sin(r)/r) with the kernel size given by             \n"
"                  method.lanczos.kernel-size for the interpolation. The         \n"
"                  method.loop-distance controls the number of surrounding pixels\n"
"                  that are taken into account on the final grid, e.g. a         \n"
"                  method.loop-distance of 1 uses 3 pixels (x - 1, x, x + 1) in  \n"
"                  both dimensions. Moreover, if method.use-errorweights is set  \n"
"                  to TRUE, an additional weight defined as 1/variance is taken  \n"
"                  into account.\n"
"                                                                                \n"
"(--method=DRIZZLE): Controlled by parameters:                                   \n"
"                  method.drizzle.downscale-x; method.drizzle.downscale-y;       \n"
"                  method.loop-distance; method.use-errorweights                 \n"
"                                                                                \n"
"                  The algorithm uses a drizzle-like distance weighting function \n"
"                  for the interpolation. The down-scaling factors               \n"
"                  method.drizzle.downscale-x and method.drizzle.downscale-y     \n"
"                  for x and y direction control the percentage of flux of the   \n"
"                  original pixel/voxel that \"drizzles\" into the target        \n"
"                  pixel/voxel. The method.loop-distance controls the number of  \n"
"                  surrounding pixels that are taken into account on the final   \n"
"                  grid, e.g. a method.loop-distance of 1 uses 3 pixels          \n"
"                  (x - 1, x, x + 1) in both dimensions. Moreover, \n"
"                  if method.use-errorweights is set to TRUE, an additional      \n"
"                  weight defined as 1/variance is taken into account.           \n"
"                                                                                \n"
"Beside the interpolation method the user can also define the output grid with   \n"
"the following recipe parameters:                                                \n"
"                                                                                \n"
"(--outgrid.ra-min):     Minimum right ascension of the output image/cube [deg]  \n"
"(--outgrid.ra-max):     Maximum right ascension of the output image/cube [deg]  \n"
"(--outgrid.dec-min):    Minimum declination of the output image/cube [deg]      \n"
"(--outgrid.dec-max):    Maximum declination of the output image/cube [deg]      \n"
"                                                                                \n"
"If the user does not explicitly set these values (keeps the recipe default      \n"
"values), the recipe derives them from the input data.                           \n"
"                                                                                \n"
"(--outgrid.delta-ra):         step-size in right ascension [deg]                \n"
"(--outgrid.delta-dec):        step-size in declination [deg]                    \n"
"                                                                                \n"
"If the user does not explicitly set these values (keeps the recipe default      \n"
"values), the recipe derives them from the input world coordinate system of the  \n"
"input data (CD1_1, CD2_2).                                                      \n"
"                                                                                \n"
"The recipe parameter (--save-table) can be used to save the table derived from  \n"
"the input images/cubes before doing the interpolation for debugging.            \n"
"                                                                                \n"
"The recipe parameter (--subtract-background) can be used to subtract the median \n"
"of 2 dimensional images to align the background.                                \n"
"                                                                                \n"
"The recipe parameter (--fieldmargin) can be used to add a margin (in percent)   \n"
"to the resampled image like the resampling software package SWARP is doing.     \n"
"                                                                                \n"
"The recipe parameter (--x-edge-trim) can be used to trim the edges of each input\n"
"by a number of pixel equal to this parameter value.                             \n"
"                                                                                \n"
"Please note that most of the code is paralelised under Linux. You can control   \n"
"the number of processing cores with the OMP_NUM_THREADS environment variable:   \n"
"                                                                                \n"
"- OMP_NUM_THREADS variable is not set -> use all system cores (preferred choice)\n"
"                                                                                \n"
"- OMP_NUM_THREADS variable is set to a value -> use this value to determine the \n"
"  cores assigned to the process (e.g. export OMP_NUM_THREADS=4 -> 4 cores are   \n"
"  used, even if the system has more cores)                                      \n";

/* Not particularly efficient but the vectors are always small */
double get_median(std::vector<double> &v)
{
  if(v.empty()) {
      return 0.0;
  }
  std::sort(v.begin(), v.end());
  size_t n = v.size() / 2;
  double med = v[n];
  if(v.size() % 2 == 0) { //If the list size is even
      med = (v[n-1] + med) / 2.0;
  }
  return med;
}

/*
 * Given a list of cpl_propertylists and a header keyword,
 * returns a set containing the values of that keyword as
 * extracted from the propertylists */
std::set<std::string>
get_string_header_set(std::list<cpl_propertylist*> plists, std::string key)
{
    std::set<std::string> result;
    // iterate over the propertylists
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        // extract the keyword from this propertylist
        const char* str = cpl_propertylist_get_string(*i, key.c_str());
        // if it exists then store it
        if (str) {
          // cpl_msg_info(cpl_func, "%s = %s", key.c_str(), str);
          result.insert(std::string(str));
        }
        cpl_error_reset();
        i++;
    }
    return result;
}

/*
 * Given a list of cpl_propertylists and a header keyword,
 * returns a set containing the values of that keyword as
 * extracted from the propertylists */
std::set<long>
get_long_header_set(std::list<cpl_propertylist*> plists, std::string key)
{
    std::set<long> result;
    // iterate over the propertylists
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        // extract the keyword from this propertylist
        const long val = cpl_propertylist_get_long(*i, key.c_str());
        // if it exists then store it
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
          // cpl_msg_info(cpl_func, "%s = %ld", key.c_str(), val);
          result.insert(val);
        }
        cpl_error_reset();
        i++;
    }
    return result;
}

/*
 * Given a list of cpl_propertylists, estimates the
 * FORS FWHM from the DIMM FWHM and AIRMASS
 * using a formula from PIPE-11419 and returns
 * a list of results */
std::list<double>
handle_fwhm(std::list<cpl_propertylist*> plists, double wavecen)
{
    std::list<double> result;
    // iterate over the propertylists
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        // extract the keyword from this propertylist
        cpl_error_reset();
        const double f1 = cpl_propertylist_get_double(*i, "ESO TEL AMBI FWHM START");
        const double f2 = cpl_propertylist_get_double(*i, "ESO TEL AMBI FWHM END");
        // cpl_msg_info(cpl_func, "TEL AMBI FWHM %f to %f", f1, f2);
        const double z = cpl_propertylist_get_double(*i, "AIRMASS");
        // cpl_msg_info(cpl_func, "AIRMASS %f", z);
        // if it exists then store it
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            double lambda_nm = wavecen * 0.1;
            double lambda_um = lambda_nm * 1e-3;
            double lambda_m = lambda_nm * 1e-9;
            double fwhm_dimm = (f1 + f2) / 2.0;
            double k = pow(1 - 78.08 * (pow(lambda_m, 0.4) * pow(z, -0.2) / pow(fwhm_dimm, 1.0/3)), 0.5);
            double fwhm_fors = fwhm_dimm * pow(0.5 / lambda_um, 0.2) * pow(z, 0.6) * k;
            // cpl_msg_info(cpl_func, "z = %f, lambda = %f, %f, fwhm_dimm = %f, k = %f, fwhm_fors = %f",
                // z, lambda_nm, lambda_m, fwhm_dimm, k, fwhm_fors);
            result.push_back(fwhm_fors);
        }
        i++;
    }
    return result;
}

/*
 * Given a list of cpl_propertylists and a header keywords,
 * returns a list of values for that keyword as extracted
 * from each of the propertylists */
std::vector<double>
get_double_header_vector(std::list<cpl_propertylist*> plists, std::string key)
{
    std::vector<double> result;
    // iterate over the propertylists
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        // extract the keyword from this propertylist
        const double val = cpl_propertylist_get_double(*i, key.c_str());
        // if it exists then store it
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
          // cpl_msg_info(cpl_func, "%s = %f", key.c_str(), val);
          result.push_back(val);
        }
        cpl_error_reset();
        i++;
    }
    return result;
}

void handle_mjd(std::list<cpl_propertylist*> plists, cpl_propertylist* plist)
{
  // iterate over the propertylists
  std::list<cpl_propertylist*>::iterator i = plists.begin();

  double earliest_mjd = DBL_MAX;
  double latest_mjd = 0;
  while (i != plists.end()) {
      // extract the keyword from this propertylist
      const double mjdobs = cpl_propertylist_get_double(*i, "MJD-OBS");
      // cpl_msg_info(cpl_func, "mjdobs %f", mjdobs);
      if (cpl_error_get_code() == CPL_ERROR_NONE) {
          if (mjdobs < earliest_mjd) {
              earliest_mjd = mjdobs;
          }
      }
      cpl_error_reset();
      const double mjdend = cpl_propertylist_get_double(*i, "MJD-END");
      // cpl_msg_info(cpl_func, "mjdend %f", mjdend);
      if (cpl_error_get_code() == CPL_ERROR_NONE) {
          if (mjdend > latest_mjd) {
              latest_mjd = mjdend;
          }
      }
      cpl_error_reset();
      i++;
  }
  if (earliest_mjd < DBL_MAX) {
      cpl_msg_info(cpl_func, "Earliest MJD-OBS is %f", earliest_mjd);
      cpl_propertylist_update_double(plist, "MJD-OBS", earliest_mjd);
  } else {
      cpl_msg_warning(cpl_func, "Could not find earliest MJD-OBS");
  }
  if (latest_mjd > 0) {
      cpl_msg_info(cpl_func, "Latest MJD-END is %f", latest_mjd);
      cpl_propertylist_update_double(plist, "MJD-END", latest_mjd);
  } else {
      cpl_msg_warning(cpl_func, "Could not find latest MJD-OBS");
  }
  cpl_error_reset();
}

/*
 * Given a list of cpl_propertylists and a list of header keywords,
 * returns a map containing the list of values for each of those keywords
 * in each of the propertylists, keyed on the keyword */
std::map<std::string, std::list<std::string>>
get_string_headers(std::list<cpl_propertylist*> plists, std::list<std::string> keys)
{
    std::map<std::string, std::list<std::string>> result;
    // iterate over the propertylists
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        std::list<std::string>::iterator j = keys.begin();
        // iterate over the keywords
        while (j != keys.end()) {
            // extract the keyword from this propertylist
            const char* str = cpl_propertylist_get_string(*i, j->c_str());
            // if it exists then store it, otherwise store an empty string
            if (str) {
              // cpl_msg_info(cpl_func, "%s = %s", j->c_str(), str);
              result[*j].push_back(std::string(str));
            } else {
              result[*j].push_back(std::string(""));
            }
            cpl_error_reset();
            j++;
        }
        i++;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   save an cube (imagelist, product of pixel resampling) on a FITS file
 * with proper FITS header
 *
 * @param   aCube       cube to save
 * @param   procatg     FITS product category
 * @param   recipe      recipe name
 * @param   filename    product filename
 * @param   parlist     input recipe parameters
 * @param   frameset    product set of frames

 *
 * @return  cpl_error_code
 */
static cpl_error_code
fors_img_stack_save(hdrl_image *stack_img,
                    cpl_image *exp_img,
                    const cpl_propertylist *stack_header,
                    const char              *procatg,
                    const char              *recipe,
                    const char              *filename,
                    const char              *version,
                    const cpl_parameterlist *parlist,
                    const cpl_propertylist  *base_header,
                    const cpl_propertylist  *plist,
                    const cpl_propertylist  *elist,
                    const cpl_propertylist  *qclist,
                    cpl_frameset            *frameset)
{
    const char* mv_regex = 
      "PHOTZP|PHOTSYS|ABMAGLIM|ABMAGSAT|PSF_FWHM";

    cpl_ensure_code(base_header && stack_img && stack_header, CPL_ERROR_NULL_INPUT);

    /* Add the product category and save the primary header without data */

    cpl_error_code rc;
    cpl_propertylist *primary_header = cpl_propertylist_duplicate(base_header);
    cpl_propertylist *extension_header = cpl_propertylist_duplicate(stack_header);
    cpl_frame *frame = cpl_frame_new();

    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, procatg);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_FINAL);
    rc = cpl_error_get_code();
    if (rc) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        cpl_msg_error(cpl_func, "Cannot initialise the product frame");
        cpl_frame_delete(frame);
        cpl_propertylist_delete(primary_header);
        return rc;
    }

    rc = cpl_dfs_setup_product_header(primary_header, frame, frameset, parlist,
                                     recipe, version, "PRO-1.15", 
                                     NULL);
    if (rc) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        cpl_msg_error(cpl_func, "Problem with product %s FITS header definition",
                      procatg);
        cpl_propertylist_delete(primary_header);
        cpl_frame_delete(frame);
        return rc;
    }

    // Add in the fixed headers for the PHU
    cpl_propertylist_update_string(primary_header, CPL_DFS_PRO_CATG, procatg);
    cpl_propertylist_update_string(primary_header, "OBSTECH", "IMAGE");

    // Replace APCOR values with values from stacked image
    if (cpl_propertylist_has(primary_header, "APCOR1") && cpl_propertylist_has(qclist, "APCOR1"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR1");
    if (cpl_propertylist_has(primary_header, "APCOR2") && cpl_propertylist_has(qclist, "APCOR2"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR2");
    if (cpl_propertylist_has(primary_header, "APCOR3") && cpl_propertylist_has(qclist, "APCOR3"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR3");
    if (cpl_propertylist_has(primary_header, "APCOR4") && cpl_propertylist_has(qclist, "APCOR4"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR4");
    if (cpl_propertylist_has(primary_header, "APCOR5") && cpl_propertylist_has(qclist, "APCOR5"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR5");
    if (cpl_propertylist_has(primary_header, "APCOR6") && cpl_propertylist_has(qclist, "APCOR6"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR6");
    if (cpl_propertylist_has(primary_header, "APCOR7") && cpl_propertylist_has(qclist, "APCOR7"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCOR7");
    if (cpl_propertylist_has(primary_header, "APCORPK") && cpl_propertylist_has(qclist, "APCORPK"))
        cpl_propertylist_copy_property(primary_header, qclist, "APCORPK");

    if (cpl_propertylist_has(primary_header, "ESO PRO DET_EXP")) {
        cpl_propertylist_erase(primary_header, "ESO PRO DET_EXP");
    }
    if (cpl_propertylist_has(primary_header, "TITLE")) {
        cpl_propertylist_erase(primary_header, "TITLE");
    }

    // Add in the extra headers supplied
    if (plist) {
        cpl_propertylist_append(primary_header, plist);
    }

    /* Move some of the properties from the primary header to the extension */
    cpl_propertylist_copy_property_regexp(extension_header, primary_header, mv_regex, 0);
    cpl_propertylist_erase_regexp(primary_header, mv_regex, 0);

    // Save the PHU
    rc = cpl_propertylist_save(primary_header, filename, CPL_IO_DEFAULT);
    if (rc) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        cpl_msg_error(cpl_func, "Cannot save product %s to disk", filename);
        cpl_frame_delete(frame);
        cpl_propertylist_delete(primary_header);
        return rc;
    }

    // Make sure the frame goes in the frameset so it's
    // recognised as a product
    rc = cpl_frameset_insert(frameset, frame);

    cpl_propertylist_delete(primary_header); primary_header = NULL;

    rc = cpl_error_get_code();
    if (rc != CPL_ERROR_NONE) {
        return rc;
    }

    /* The extension header has already been created from the WCS headers
     * returned from the stacking routine */

    /* Add in the extra headers supplied */
    if (elist) {
        cpl_propertylist_append(extension_header, elist);
    }

    cpl_image * data = hdrl_image_get_image(stack_img);
    cpl_image * errors = hdrl_image_get_error(stack_img);
    cpl_image * bpm = cpl_image_new_from_mask(hdrl_image_get_mask(stack_img));

    cpl_propertylist_update_string(extension_header, "HDUCLASS", "ESO");
    cpl_propertylist_set_comment(extension_header, "HDUCLASS", "class name (ESO format)");
    cpl_propertylist_update_string(extension_header, "HDUDOC",  "SDP");
    cpl_propertylist_set_comment(extension_header, "HDUDOC", "ESO Science Data Products standard");
    cpl_propertylist_update_string(extension_header, "HDUVERS",  "SDP version 8");
    cpl_propertylist_set_comment(extension_header, "HDUVERS", "version number");

    /* Save the data itself in the first extension as an image */
    cpl_propertylist_update_string(extension_header, "EXTNAME", EXTNAME_DATA);
    cpl_propertylist_set_comment(extension_header, "EXTNAME", EXTNAME_DATA_COMMENT);
    cpl_propertylist_update_string(extension_header, "HDUCLAS1",  "IMAGE");
    cpl_propertylist_update_string(extension_header, "HDUCLAS2",  "DATA");
    cpl_propertylist_update_string(extension_header, "ERRDATA",  "IMAGE.ERR");
    cpl_propertylist_update_string(extension_header, "QUALDATA",  "IMAGE.BPM");
    cpl_propertylist_update_string(extension_header, "BUNIT",  "ADU");

    cpl_image_save(data,
                   filename, CPL_TYPE_FLOAT, extension_header,
                   CPL_IO_EXTEND);

    /* Remove the keywords that only go in the science header */
    cpl_propertylist_erase_regexp(extension_header, mv_regex, 0);

    /* Save the data error in the second extension as an image */
    cpl_propertylist_update_string(extension_header, "EXTNAME", EXTNAME_ERROR);
    cpl_propertylist_set_comment(extension_header, "EXTNAME", EXTNAME_ERROR_COMMENT);
    cpl_propertylist_update_string(extension_header, "HDUCLAS1",  "IMAGE");
    cpl_propertylist_update_string(extension_header, "HDUCLAS2",  "ERROR");
    cpl_propertylist_update_string(extension_header, "HDUCLAS3",  "RMSE");
    cpl_propertylist_update_string(extension_header, "SCIDATA",  "IMAGE.DAT");
    cpl_propertylist_update_string(extension_header, "QUALDATA",  "IMAGE.BPM");
    cpl_propertylist_update_string(extension_header, "BUNIT",  "ADU");
    cpl_propertylist_erase(extension_header, "ERRDATA");

    cpl_image_save(errors,
                   filename, CPL_TYPE_FLOAT, extension_header, CPL_IO_EXTEND);

    /* Save the bpm in the third extension as an image */
    cpl_propertylist_update_string(extension_header, "EXTNAME", EXTNAME_BPM);
    cpl_propertylist_set_comment(extension_header, "EXTNAME", EXTNAME_BPM_COMMENT);
    cpl_propertylist_update_string(extension_header, "HDUCLAS1",  "IMAGE");
    cpl_propertylist_update_string(extension_header, "HDUCLAS2",  "QUALITY");
    cpl_propertylist_update_string(extension_header, "HDUCLAS3",  "MASKZERO");
    cpl_propertylist_update_string(extension_header, "SCIDATA",  "IMAGE.DAT");
    cpl_propertylist_update_string(extension_header, "ERRDATA",  "IMAGE.ERR");
    cpl_propertylist_update_string(extension_header, "BUNIT",  "");
    cpl_propertylist_erase(extension_header, "QUALDATA");

    cpl_image_save(bpm, filename,
                   CPL_TYPE_INT, extension_header, CPL_IO_EXTEND);
    cpl_image_delete(bpm); /* As bpm was created with cpl_image_new_from_mask */


    cpl_propertylist_update_string(extension_header, "EXTNAME", EXTNAME_EXPMAP);
    cpl_propertylist_set_comment(extension_header, "EXTNAME", EXTNAME_EXPMAP_COMMENT);
    cpl_propertylist_update_string(extension_header, "HDUCLAS1",  "IMAGE");
    cpl_propertylist_update_string(extension_header, "HDUCLAS2",  "EXPOSURE");
    cpl_propertylist_erase(extension_header, "HDUCLAS3");
    cpl_propertylist_update_string(extension_header, "SCIDATA",  "IMAGE.DAT");
    cpl_propertylist_update_string(extension_header, "ERRDATA",  "IMAGE.ERR");
    cpl_propertylist_update_string(extension_header, "QUALDATA",  "IMAGE.BPM");
    cpl_propertylist_update_string(extension_header, "BUNIT",  "s");
    cpl_propertylist_set_comment(extension_header, "BUNIT",  "exposure time in seconds");

    cpl_image_save(exp_img,
                   filename, CPL_TYPE_FLOAT, extension_header, CPL_IO_EXTEND);
    /* cleanup */
    cpl_propertylist_delete(extension_header);

    return cpl_error_get_code();
}


/* Function needed by cpl_recipe_define to fill the input parameters */
/*----------------------------------------------------------------------------*/
/**
 @brief   define recipe parameters
 @param   self input parameters
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code fors_img_stack_define_parameters(cpl_parameterlist *self)
{
  cpl_parameter   *   par ;

  /* --fors_img_stack.ext-nb-raw */
  par = cpl_parameter_new_value(RECIPE_NAME".ext-nb-raw", CPL_TYPE_INT,
                                "FITS extension of the RAW", RECIPE_NAME, 1);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "ext-r");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.ext-nb-raw-err */
  par = cpl_parameter_new_value(RECIPE_NAME".ext-nb-raw-err", CPL_TYPE_INT,
                                "FITS extension of the ERROR/VARIANCE",
                                RECIPE_NAME, 2);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "ext-e");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.ext-nb-raw-bpm */
  par = cpl_parameter_new_value(RECIPE_NAME".ext-nb-raw-bpm", CPL_TYPE_INT,
                                "FITS extension of the BPM", RECIPE_NAME, 4);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "ext-b");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method */
  par = cpl_parameter_new_enum(RECIPE_NAME".method",CPL_TYPE_STRING,
                               "Resampling method", RECIPE_NAME, "DRIZZLE", 6,
                               "NEAREST", "LINEAR", "QUADRATIC", "RENKA",
                               "DRIZZLE","LANCZOS");
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "method");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.loop-distance */
  par = cpl_parameter_new_value(RECIPE_NAME".method.loop-distance", CPL_TYPE_INT,
                                "Loop distance used by all (but NEAREST) "
                                "methods to control the number of surrounding "
                                "voxels that are taken into account",
                                RECIPE_NAME, LOOP_DISTANCE);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "method.loop-distance");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.use-errorweights*/
  par = cpl_parameter_new_value(RECIPE_NAME".method.use-errorweights",
                                CPL_TYPE_BOOL,
                                "Use additional weights of 1/err^2", RECIPE_NAME,
                                CPL_FALSE);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI,
                          "method.use-errorweights");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.renka.critical-radius */
  par = cpl_parameter_new_value(RECIPE_NAME".method.renka.critical-radius",
                                CPL_TYPE_DOUBLE,"Critical radius of the Renka "
                                "method",
                                RECIPE_NAME, RENKA_CRITICAL_RADIUS);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI,
                          "method.renka.critical-radius");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.lanczos.kernel-size */
  par = cpl_parameter_new_value(RECIPE_NAME".method.lanczos.kernel-size",
                                CPL_TYPE_INT,"Kernel size of the Lanczos "
                                "method",
                                RECIPE_NAME, LANCZOS_KERNEL_SIZE);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI,
                          "method.lanczos.kernel-size");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.drizzle.downscale-x */
  par = cpl_parameter_new_value(RECIPE_NAME".method.drizzle.downscale-x",
                                CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
                                    "in x direction",
                                RECIPE_NAME, DRIZZLE_DOWN_SCALING_FACTOR_X);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI,
                          "method.drizzle.downscale-x");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.method.drizzle.downscale-y */
  par = cpl_parameter_new_value(RECIPE_NAME".method.drizzle.downscale-y",
                                CPL_TYPE_DOUBLE, "Drizzle down-scaling factor "
                                    "in y direction",
                                RECIPE_NAME, DRIZZLE_DOWN_SCALING_FACTOR_Y);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI,
                          "method.drizzle.downscale-y");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.ra-min */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.ra-min", CPL_TYPE_DOUBLE,
                                "Minimum right ascension of the output "
                                "image/cube in degree", RECIPE_NAME, -1.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.ra-min");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.ra-max */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.ra-max", CPL_TYPE_DOUBLE,
                                "Maximum right ascension of the output "
                                "image/cube in degree", RECIPE_NAME, -1.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.ra-max");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.dec-min */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.dec-min", CPL_TYPE_DOUBLE,
                                "Minimum declination of the output "
                                "image/cube in degree", RECIPE_NAME, -1.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.dec-min");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.dec-max */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.dec-max", CPL_TYPE_DOUBLE,
                                "Maximum declination of the output "
                                "image/cube in degree", RECIPE_NAME, -1.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.dec-max");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.delta-ra */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.delta-ra",CPL_TYPE_DOUBLE,
                                "Output step-size in right ascension in degree",
                                RECIPE_NAME, -1.);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.delta-ra");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.outgrid.delta-dec */
  par = cpl_parameter_new_value(RECIPE_NAME".outgrid.delta-dec",
                                CPL_TYPE_DOUBLE, "Output step-size in "
                                    "declination in degree",
                                RECIPE_NAME, -1.);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "outgrid.delta-dec");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.save-table */
  par = cpl_parameter_new_value(RECIPE_NAME".save-table", CPL_TYPE_BOOL,
                                "Save the table before resampling", RECIPE_NAME,
                                CPL_FALSE);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "save-table");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  
  /* --fors_img_stack.subtract-background */
  par = cpl_parameter_new_value(RECIPE_NAME".subtract-background", CPL_TYPE_BOOL,
                "Subtract median of the images in 2D only", RECIPE_NAME,
                CPL_TRUE);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "subtract-background");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);


  /* --fors_img_stack.fieldmargin */
  par = cpl_parameter_new_value(RECIPE_NAME".fieldmargin", CPL_TYPE_DOUBLE,
                                "Ad this margin/border (in percent) to the "
                                "resampled image/cube", RECIPE_NAME, FIELDMARGIN);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "fieldmargin");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  /* --fors_img_stack.edge-trim */
  par = cpl_parameter_new_value(RECIPE_NAME".edge-trim", CPL_TYPE_INT,
                  "Number or pixels to trim for each plane of the input frames. "
                  "It should be smaller than half image size", RECIPE_NAME, EDGETRIM);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "edge-trim");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  par = cpl_parameter_new_value(RECIPE_NAME".magcutE",
                                CPL_TYPE_DOUBLE,
                                "Zeropoint absolute cutoff (magnitude)",
                                RECIPE_NAME,
                                1.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "magcutE");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  par = cpl_parameter_new_value(RECIPE_NAME".magcutk",
                              CPL_TYPE_DOUBLE,
                              "Zeropoint kappa rejection parameter",
                              RECIPE_NAME,
                              5.0);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "magcutk");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  par = cpl_parameter_new_value(RECIPE_NAME".magsyserr",
                              CPL_TYPE_DOUBLE,
                              "Systematic error in magnitude",
                              RECIPE_NAME,
                              0.01);
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "magsyserr");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  par = cpl_parameter_new_value(RECIPE_NAME".cacheloc",
                              CPL_TYPE_STRING,
                              "Location for the standard star cache",
                              RECIPE_NAME,
                              ".");
  cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "cacheloc");
  cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append(self, par);

  fors_extract_define_parameters(self, RECIPE_NAME);

  return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 @brief   verify if a parameter value has been changed (from command line or
          or rc file by a user)
 @param    p    parameter
 @return   1 if parameter has changes, else 0
 */
/*----------------------------------------------------------------------------*/

static int
fors_img_stack_param_has_changed(const cpl_parameter* p)
{
  cpl_type type = cpl_parameter_get_type(p);
  int has_changed = 0;
  /* handle different data types */
  switch (type) {

    case CPL_TYPE_INT: {
      int val = cpl_parameter_get_int(p);
      int def = cpl_parameter_get_default_int(p);
      has_changed = (val != def)  ? 1 : 0;
    }
    break;

    case CPL_TYPE_FLOAT: {
      float val = cpl_parameter_get_double(p);
      float def = cpl_parameter_get_default_double(p);
      has_changed = (val != def)  ? 1 : 0;
    }
    break;

    case CPL_TYPE_DOUBLE: {
      double val = cpl_parameter_get_double(p);
      double def = cpl_parameter_get_default_double(p);
      has_changed = (val != def)  ? 1 : 0;
    }
    break;

    case CPL_TYPE_STRING:{
      const char* val = cpl_parameter_get_string(p);
      const char* def = cpl_parameter_get_default_string(p);
      if ( strcmp(val,def) != 0 ) {
          has_changed = 1;
      }
    }
    break;

    default:
      cpl_msg_error(cpl_func,"case not found! %d string type: %d",type,
                    CPL_TYPE_STRING);
      break;

  }
  return has_changed;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   get double parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   pvalue  recipe parameter value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

/* set propertylist (double) value */
static cpl_error_code
fors_img_stack_parameters_get_double(const cpl_parameterlist* parlist,
                                        const char* pname, double *pvalue)
{
  const cpl_parameter* p = cpl_parameterlist_find_const(parlist, pname);
  int p_has_changed = fors_img_stack_param_has_changed(p);

  if  ( cpl_parameter_get_default_flag(p) && p_has_changed != 0) {
      *pvalue = cpl_parameter_get_double(p);
  } else {
      *pvalue = cpl_parameter_get_default_double(p);
  }

  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   set double variable from parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   table_value  table value (value to be used if recipe parameter has
                       default value)
 @param   value  variable value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/


static cpl_error_code
set_double_from_parameters(const cpl_parameterlist* parlist,
                           const char* pname,
                           const double table_value,
                           double *value) {

  double temp_value;
  fors_img_stack_parameters_get_double (parlist, pname, &temp_value);
  if (temp_value != -1)
    {
      *value = temp_value;
    }
  else
    {
      *value = table_value;
    }
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief   get int parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   pvalue  recipe parameter value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

/* set propertylist (int) value */
static cpl_error_code
fors_img_stack_parameters_get_int(const cpl_parameterlist* parlist,
                                        const char* pname, int *pvalue)
{
  const cpl_parameter* p = cpl_parameterlist_find_const(parlist, pname);
  int p_has_changed = fors_img_stack_param_has_changed(p);

  if  ( cpl_parameter_get_default_flag(p) && p_has_changed != 0) {
      *pvalue = cpl_parameter_get_int(p);
  } else {
      *pvalue = cpl_parameter_get_default_int(p);
  }

  return cpl_error_get_code();
}





/*----------------------------------------------------------------------------*/
/**
 @brief   set int variable from parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   table_value  table value (value to be used if recipe parameter has
                       default value)
 @param   value  variable value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/


static cpl_error_code
set_int_from_parameters(const cpl_parameterlist* parlist,
                           const char* pname,
                           const int table_value,
                           int *value) {

  int temp_value;
  fors_img_stack_parameters_get_int (parlist, pname, &temp_value);
  if (temp_value != -1)
    {
      *value = temp_value;
    }
  else
    {
      *value = table_value;
    }
  return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief  print content of the wcs structure
 @param  wcs cpl_wcs to be printed
 @return cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
fors_img_stack_wcs_print(cpl_wcs *wcs)
{
  cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);

  const cpl_array *crval = cpl_wcs_get_crval(wcs);
  const cpl_array *crpix = cpl_wcs_get_crpix(wcs);
  const cpl_array *ctype = cpl_wcs_get_ctype(wcs);
  const cpl_array *cunit = cpl_wcs_get_cunit(wcs);

  const cpl_matrix *cd = cpl_wcs_get_cd(wcs);
  const cpl_array *dims = cpl_wcs_get_image_dims(wcs);
  cpl_size naxis = cpl_wcs_get_image_naxis(wcs);

  cpl_msg_info(cpl_func, "NAXIS:  %lld", naxis);
  int testerr = 0;

  cpl_msg_indent_more();
  /* Check NAXIS */
  for (cpl_size i = 0; i < naxis; i++) {
      cpl_msg_info(cpl_func, "NAXIS%lld: %d", i + 1,
                   cpl_array_get_int(dims, i, &testerr));
  }
  cpl_msg_indent_less();

  double cd11 = cpl_matrix_get(cd, 0, 0);
  double cd12 = cpl_matrix_get(cd, 0, 1);
  double cd21 = cpl_matrix_get(cd, 1, 0);
  double cd22 = cpl_matrix_get(cd, 1, 1);
  double crpix1 = cpl_array_get_double(crpix, 0, &testerr);
  double crpix2 = cpl_array_get_double(crpix, 1, &testerr);
  double crval1 = cpl_array_get_double(crval, 0, &testerr);
  double crval2 = cpl_array_get_double(crval, 1, &testerr);

  cpl_msg_info(cpl_func, "1st and 2nd dimension");
  cpl_msg_indent_more();
  cpl_msg_info(cpl_func, "CD1_1:  %g", cd11);
  cpl_msg_info(cpl_func, "CD1_2:  %g", cd12);
  cpl_msg_info(cpl_func, "CD2_1:  %g", cd21);
  cpl_msg_info(cpl_func, "CD2_2:  %g", cd22);
  
  cpl_msg_info(cpl_func, "CRPIX1: %g", crpix1);
  cpl_msg_info(cpl_func, "CRPIX2: %g", crpix2);
  cpl_msg_info(cpl_func, "CRVAL1: %g", crval1);
  cpl_msg_info(cpl_func, "CRVAL2: %g", crval2);
  if (ctype) {
      cpl_msg_info(cpl_func, "CTYPE1: %s", cpl_array_get_string(ctype, 0));
      cpl_msg_info(cpl_func, "CTYPE2: %s", cpl_array_get_string(ctype, 1));
  }

  if (cunit) {
      cpl_msg_info(cpl_func, "CUNIT1: %s", cpl_array_get_string(cunit, 0));
      cpl_msg_info(cpl_func, "CUNIT2: %s", cpl_array_get_string(cunit, 1));
  }  
  cpl_msg_indent_less();

  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   get WCS and put into output cdij, crpixj, crvalj, cdeltj (j=1,2,3)
 variables
 @param   wcs input wcs
 @param   cdelt1
 @param   cdelt2
 @param   cdelt3
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
fors_img_stack_get_cdelt12(cpl_wcs *wcs, double *cdelt1, double *cdelt2)
{
  cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(cdelt1, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(cdelt2, CPL_ERROR_NULL_INPUT);

  const cpl_matrix *cd = cpl_wcs_get_cd(wcs);
  *cdelt1 = fabs(cpl_matrix_get(cd, 0, 0)); /* CD1_1 */
  *cdelt2 = fabs(cpl_matrix_get(cd, 1, 1)); /* CD2_2 */

  /* Check keys (for debug) */
  cpl_msg_debug(cpl_func, "cdelt1: %g, cdelt2: %g", *cdelt1, *cdelt2);

  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   set values into hdrl_parameter* structure
 @param   parlist input recipe parameters
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static hdrl_parameter *
fors_img_stack_set_method (const cpl_parameterlist *parlist)
{
  hdrl_parameter      *res_method = NULL;
  const cpl_parameter *p = NULL;
  int p_has_changed = 0;
  int loop_distance = 0;
  double critical_radius_renka = 0.;
  int kernel_size_lanczos = 0;
  double pix_frac_drizzle_x = 0.;
  double pix_frac_drizzle_y= 0.;
  cpl_boolean use_errorweights = CPL_FALSE;

  const cpl_parameter *par = NULL;


  /* Apply additional weights based on the error if required */
  par = cpl_parameterlist_find_const (parlist,
                                      RECIPE_NAME".method.use-errorweights");
  if (cpl_parameter_get_bool(par)) {
      use_errorweights = CPL_TRUE;
  } else {
      use_errorweights = CPL_FALSE;
  }

  /** loop distance, used for all weighted resampling schemes */
  p = cpl_parameterlist_find_const (parlist, RECIPE_NAME".method.loop-distance");
  p_has_changed = fors_img_stack_param_has_changed (p);
  if (cpl_parameter_get_default_flag (p) && p_has_changed != 0)
    {
      loop_distance = cpl_parameter_get_int (p);
    }
  else
    {
      loop_distance = LOOP_DISTANCE;
    }

  /** critical radius of the Renka-weighted method */
  set_double_from_parameters(parlist,
                             RECIPE_NAME".method.renka.critical-radius",
                             RENKA_CRITICAL_RADIUS, &critical_radius_renka);

  /** Kernel size of the Lanczos-weighted method */
  set_int_from_parameters(parlist,
                             RECIPE_NAME".method.lanczos.kernel-size",
                             LANCZOS_KERNEL_SIZE, &kernel_size_lanczos);

  /** the pixfrac parameter of the drizzle method: down-scaling factor  *
   *  of input pixel size before computing drizzling weights; different *
   *  values for x-, y-, and lambda directions are possible
   *
   *              */
  set_double_from_parameters(parlist,
                             RECIPE_NAME".method.drizzle.downscale-x",
                             DRIZZLE_DOWN_SCALING_FACTOR_X, &pix_frac_drizzle_x);

  set_double_from_parameters(parlist,
                             RECIPE_NAME".method.drizzle.downscale-y",
                             DRIZZLE_DOWN_SCALING_FACTOR_Y, &pix_frac_drizzle_y);


  /* Create the right re-sampling parameter */
  par = cpl_parameterlist_find_const (parlist, RECIPE_NAME".method");
  const char *method = cpl_parameter_get_string (par);
  if (strcmp (method, "NEAREST") == 0)
    {
      res_method = hdrl_resample_parameter_create_nearest();
    }
  else if (strcmp (method, "RENKA") == 0)
    {
      res_method = hdrl_resample_parameter_create_renka(loop_distance,
                                                            use_errorweights,
                                                            critical_radius_renka);
    }
  else if (strcmp (method, "LINEAR") == 0)
    {
      res_method = hdrl_resample_parameter_create_linear(loop_distance,
                                                             use_errorweights);
    }
  else if (strcmp (method, "QUADRATIC") == 0)
    {
      res_method = hdrl_resample_parameter_create_quadratic(loop_distance,
                                                                use_errorweights);
    }
  else if (strcmp (method, "DRIZZLE") == 0)
    {
      res_method = hdrl_resample_parameter_create_drizzle(loop_distance,
                                                              use_errorweights,
                                                              pix_frac_drizzle_x,
                                                              pix_frac_drizzle_y,
                                                              1.0);
    }
  else if (strcmp (method, "LANCZOS") == 0)
    {
      res_method = hdrl_resample_parameter_create_lanczos(loop_distance,
                                                              use_errorweights,
                                                              kernel_size_lanczos);
    }
  else
    {
      res_method = hdrl_resample_parameter_create_lanczos(loop_distance,
                                                              use_errorweights,
                                                              kernel_size_lanczos);
      cpl_msg_warning (cpl_func,
                       "%s is an unsupported method! Default to LANCZOS",
                       method);
    }

  //fors_img_stack_print_method_params(res_method);

  return res_method;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   set values into hdrl_parameter* structure
 @param   parlist input recipe parameters
 @param   muse_table input table with x,y,lambda values
 @param   wcs input wcs
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static hdrl_parameter *
fors_img_stack_set_outputgrid (
    const cpl_parameterlist *parlist, cpl_table *muse_table, cpl_wcs *wcs)
{
  /* Should be done in a dedicated _new function */
  cpl_error_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT,
                   return NULL, "NULL Input Parameters");
  cpl_error_ensure(muse_table != NULL, CPL_ERROR_NULL_INPUT,
                   return NULL, "NULL Input table");
  cpl_error_ensure(wcs != NULL, CPL_ERROR_NULL_INPUT,
                   return NULL, "NULL Input wcs");

  hdrl_parameter      *res_outputgrid = NULL;
  double ra_min =  0.;
  double ra_max =  0.;
  double dec_min =  0.;
  double dec_max =  0.;

  double dx =  0.;
  double dy =  0.;

  /* init relevant parameters */
  double ra_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double ra_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_RA);
  double dec_min_tmp =
      cpl_table_get_column_min (muse_table, HDRL_RESAMPLE_TABLE_DEC);
  double dec_max_tmp =
      cpl_table_get_column_max (muse_table, HDRL_RESAMPLE_TABLE_DEC);

  /* We have the rare case that the image spans over ra = 0.*/
  if(ra_max_tmp - ra_min_tmp > 180){
      const double *ra = cpl_table_get_data_double_const(muse_table,
                                                         HDRL_RESAMPLE_TABLE_RA);
      /* Should we also take the bpm into account ?
  const int    *bpm = cpl_table_get_data_int_const(ResTable,
                                                   HDRL_RESAMPLE_TABLE_BPM);
       */

      /* set to extreme values for a start */
      ra_min_tmp = 0.;
      ra_max_tmp = 360.;
      cpl_size nrow = cpl_table_get_nrow(muse_table);

      for (cpl_size i = 0; i < nrow; i++) {
          if (ra[i] > ra_min_tmp && ra[i] <= 180.) ra_min_tmp = ra[i]; /* get the maximum */
          if (ra[i] < ra_max_tmp && ra[i] >  180.) ra_max_tmp = ra[i]; /* get the minimum */
      }
  }

  /* Check output (for debug) */
  cpl_msg_debug (cpl_func, "min x %10.7f", ra_min_tmp);
  cpl_msg_debug (cpl_func, "max x %10.7f", ra_max_tmp);
  cpl_msg_debug (cpl_func, "min y %10.7f", dec_min_tmp);
  cpl_msg_debug (cpl_func, "max y %10.7f", dec_max_tmp);

  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.ra-min",
                             ra_min_tmp, &ra_min);
  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.ra-max",
                             ra_max_tmp, &ra_max);

  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.dec-min",
                             dec_min_tmp, &dec_min);
  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.dec-max",
                             dec_max_tmp, &dec_max);

  /* Reset all delta values */
  double cdelt1 = 0., cdelt2 = 0.;
  fors_img_stack_get_cdelt12 (wcs, &cdelt1, &cdelt2);

  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.delta-ra",
                             cdelt1, &dx);
  set_double_from_parameters(parlist,
                             RECIPE_NAME".outgrid.delta-dec",
                             cdelt2, &dy);

  /* Assign the field margin */
  const cpl_parameter *par = NULL;
  par = cpl_parameterlist_find_const (parlist,
                                      RECIPE_NAME".fieldmargin");
  double fieldmargin = 0.;
  fieldmargin = cpl_parameter_get_double(par);

  /* create final outgrid parameter structure */
  int naxis = cpl_wcs_get_image_naxis(wcs);

  if(naxis == 2) {
      res_outputgrid =
          hdrl_resample_parameter_create_outgrid2D_userdef(dx, dy,
                                                           ra_min, ra_max,
                                                           dec_min, dec_max,
                                                           fieldmargin);
  }

  return res_outputgrid;
}


/**
 * @brief This function does the actual re-sampling.
 * @param append input table to append
 * @param storage output table
 * @return cpl_error_code
 */
static cpl_error_code
fors_img_stack_update_table(const cpl_table* append, cpl_table** storage) {

  cpl_ensure_code(append, CPL_ERROR_NULL_INPUT);

  if(*storage == NULL) {
      *storage = cpl_table_new(0);
      cpl_table_copy_structure(*storage, append);
      cpl_table_insert(*storage, append, 0);
  } else {
      cpl_size nrow = cpl_table_get_nrow(*storage);
      cpl_table_insert(*storage, append, nrow);
  }

  return cpl_error_get_code();

}

//------------------------------------------------------------------------------

/**
 * @brief This function determines WCS from input data
 *
 * @param frameset input set of frames
 * @return filled cpl_wcs object
 *
 * @doc
 * Generate a FITS header to hold WCS that define the output re-sampled image
 * This is INSTRUMENT mode dependent, because the information on the WCS may
 * come from the primary header or from a FITS extensions in case of different
 * instruments
 *
 */
static cpl_wcs *
fors_img_stack_get_wcs_from_frameset(cpl_frameset* frameset) {

  cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);

  /* Read wcs from firts raw image */
  cpl_frameset* in_set = cpl_frameset_new();

  for (int i = 0; i < 3; i++) {
      cpl_frameset* tmp_set = fors_frameset_extract(frameset, input_cats[i]);
      if (tmp_set == NULL) {
          cpl_msg_info(cpl_func, "No input %s frames found", input_cats[i]);
      } else {
          cpl_frameset_join(in_set, tmp_set);
          cpl_frameset_delete(tmp_set);
          tmp_set = NULL;
      }
  }

  if (in_set == NULL){
      cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND, "Missing RAW "
                            "files");
      return NULL;
  }

  cpl_frame* frame = cpl_frameset_get_position(in_set, 0);
  const char* fname = cpl_frame_get_filename(frame);


  cpl_wcs *wcs = NULL;
  cpl_propertylist* head = NULL;

  cpl_errorstate prestate = cpl_errorstate_get();
  head = cpl_propertylist_load(fname, 0);
  wcs = cpl_wcs_new_from_propertylist(head);
  if (wcs == NULL) {
      /* Not possible to read wcs - trying from extension */
      cpl_errorstate_set(prestate);
      cpl_propertylist_delete(head);
  } else {
      cpl_propertylist_delete(head);
      cpl_frameset_delete(in_set);
      return wcs;
  }

  prestate = cpl_errorstate_get();
  head = cpl_propertylist_load(fname, 1);
  wcs = cpl_wcs_new_from_propertylist(head);
  cpl_propertylist_delete(head);
  cpl_frameset_delete(in_set);

  return wcs ;

}

/**
 * @brief transform and insert the data (pixel flux, error, quality) into a
 * cpl_table. Parallelised code
 * @param data_frm input frame with data
 * @param data_ext_id input frame extension containing data
 * @param errs_frm input frame with errors
 * @param errs_ext_id input frame extension containing errors
 * @param qual_frm input frame with pixel quality
 * @param qual_ext_id input frame extension containing pixel quality
 * @param is_variance is input an error or a variance?
 * @param subtract_bkg should be the background be subtracted?
 * @param edge_trim how many pixels are trimmed from resampled plane(s) edges
 * @doc
 * @return cpl_table with filled information
 * */

static cpl_error_code
fors_img_stack_frameset_to_table(cpl_table** restbl,
                                 cpl_table** exptbl,
                                 cpl_table** bpmtbl,
                                 const cpl_frame *data_frm,
                                 const cpl_size data_ext_id,
                                 const cpl_frame *errs_frm,
                                 const cpl_size errs_ext_id,
                                 const cpl_frame *qual_frm,
                                 const cpl_size qual_ext_id,
                                 const cpl_boolean is_variance,
                                 const cpl_boolean subtract_bkg,
                                 const int edgetrim)
{

  cpl_ensure(data_frm, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

  const char *name = cpl_frame_get_filename(data_frm);
  cpl_imagelist *dlist = NULL;
  cpl_imagelist *elist = NULL;
  cpl_imagelist *qlist = NULL;
  cpl_imagelist *explist = NULL;
  cpl_imagelist *bpmlist = NULL;

  cpl_table *tab_res = NULL;
  cpl_table *tab_exp = NULL;
  cpl_table *tab_bpm = NULL;

  cpl_errorstate  prestate = cpl_errorstate_get();

  cpl_msg_info(cpl_func, "Loading data from %s %lld", name, data_ext_id);
  dlist = cpl_imagelist_load(name, CPL_TYPE_DOUBLE, data_ext_id);

  if (dlist == NULL) {
      /*It was not an image nor a imagelist - reset cpl error and return NULL*/
      if (!cpl_errorstate_is_equal(prestate)) {
          cpl_errorstate_set(prestate);
      }
      return CPL_ERROR_ILLEGAL_INPUT;
  }
  cpl_propertylist *header_data = cpl_propertylist_load(name, 0);
  double exptime = cpl_propertylist_get_double(header_data, "EXPTIME");
  cpl_propertylist_delete(header_data);

  cpl_propertylist *xheader_data = cpl_propertylist_load(name,
                                                         data_ext_id);
  cpl_wcs *wcs = cpl_wcs_new_from_propertylist(xheader_data);

  if (errs_frm != NULL) {
      name = cpl_frame_get_filename(errs_frm);
      cpl_msg_info(cpl_func, "Loading error from %s %lld", name, errs_ext_id);
      elist = cpl_imagelist_load(name, CPL_TYPE_DOUBLE, errs_ext_id);
      if(is_variance) {
          /* transform variance into error */
          cpl_imagelist_power(elist, 0.5);
      }
  }
  if (qual_frm != NULL) {
      name = cpl_frame_get_filename(qual_frm);
      cpl_msg_info(cpl_func, "Loading bpm from %s %lld", name, qual_ext_id);
      qlist = cpl_imagelist_load(name, CPL_TYPE_INT, qual_ext_id);
  }

  cpl_size size = cpl_imagelist_get_size(dlist);
  explist = cpl_imagelist_new();
  bpmlist = cpl_imagelist_new();

  /* Create the exposure map */
  for (cpl_size k = 0; k < size; k++) {
      cpl_image* data = cpl_imagelist_get(dlist, k);
      cpl_image* edata = cpl_image_duplicate(data);
      int nx = (int)cpl_image_get_size_x(edata);
      int ny = (int)cpl_image_get_size_y(edata);
      cpl_image_fill_window(edata, 1, 1, nx, ny, exptime);
      cpl_imagelist_set(explist, edata, k);
  }
    
   /* Create the bad pixel map */
   for (cpl_size k = 0; k < size; k++) {
       cpl_image* qual = cpl_imagelist_get(qlist, k);
       cpl_image* qdata = cpl_image_duplicate(qual);
       cpl_imagelist_set(bpmlist, qdata, k);
   }


  /* ingest pixel quality in data */
  if (qual_frm != NULL && size > 0){
          for(cpl_size k = 0; k < size; k++) {
             cpl_image* data = cpl_imagelist_get(dlist, k);
             cpl_image* qual = cpl_imagelist_get(qlist, k);

             /*we use INT_MAX instead of 1.1 as some pipeline
              * may use pixel codes as qualifier */
             cpl_mask* mask = cpl_mask_threshold_image_create(qual, 0, INT_MAX);

             cpl_image_reject_from_mask(data, mask);
             cpl_mask_delete(mask);
             cpl_imagelist_set(dlist, data, k);
          }
  }

  /* In case the error is passed as variance we take the square root. This could
   * cause on some data to add a bad pixel mask - thus we have to synchronize it
   * */
  if(elist != NULL && is_variance){
      for(cpl_size k = 0; k < size; k++) {
          cpl_image* data = cpl_imagelist_get(dlist, k);
          cpl_mask*  data_mask = cpl_image_get_bpm(data);

          cpl_image* error = cpl_imagelist_get(elist, k);
          cpl_mask*  error_mask = cpl_image_get_bpm(error);

          cpl_mask* merged = cpl_mask_duplicate(data_mask);

          /*Add original bad pixels to previous iteration*/
          cpl_mask_or(merged, error_mask);
          cpl_image_reject_from_mask(data, merged);
          cpl_image_reject_from_mask(error, merged);
          cpl_mask_delete(merged);

      }
  }

  hdrl_imagelist* hlist = NULL;
  hdrl_imagelist* ehlist = NULL;
  hdrl_imagelist* bhlist = NULL;
  if (size > 0) {
          hlist = hdrl_imagelist_create(dlist, elist);
          ehlist = hdrl_imagelist_create(explist, NULL);
          bhlist = hdrl_imagelist_create(bpmlist, NULL);
  }
  int edge_trim = edgetrim;
  if (edge_trim > 0) {
      cpl_msg_info(cpl_func, "Trim input image edges of %d pixels", edge_trim);
          /* trim each product plane edge of edgetrim pixels */
          cpl_size sx = hdrl_imagelist_get_size_x(hlist);
          cpl_size sy = hdrl_imagelist_get_size_y(hlist);
          cpl_size sz = hdrl_imagelist_get_size(hlist);
      if (edge_trim >= 0.5* sx) {
          edge_trim = 0;
          cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                          "size. Reset to 0");
      }
      if (edge_trim >= 0.5* sy) {
          edge_trim = 0;
          cpl_msg_warning(cpl_func, "edge-trim must be smaller than half image "
                          "size. Reset to 0");
      }
          for(cpl_size k = 0; k < sz; k++) {

                  hdrl_image* hima = hdrl_imagelist_get(hlist, k);
                  hdrl_image* ehima = hdrl_imagelist_get(ehlist, k);
                  hdrl_image* bhima = hdrl_imagelist_get(bhlist, k);
                  /* Note FITS convention to loop over pixels */
                  /* trim lower image border along X direction */
                  for(cpl_size j = 1; j <= edge_trim; j++) {
                          for(cpl_size i = 1; i <= sx; i++) {
                                  hdrl_image_reject(hima, i, j);
                                  hdrl_image_reject(ehima, i, j);
                                  hdrl_image_reject(bhima, i, j);
                          }
                  }
                  /* trim upper image border along X direction */
                  for(cpl_size j = sy - edge_trim + 1; j <= sy; j++) {
                          for(cpl_size i = 1; i <= sx; i++) {
                                  hdrl_image_reject(hima, i, j);
                                  hdrl_image_reject(ehima, i, j);
                                  hdrl_image_reject(bhima, i, j);
                          }
                  }
                  /* trim left image border along Y direction */
                  for(cpl_size j = 1; j <= sy; j++) {
                          for(cpl_size i = 1; i <= edge_trim; i++) {
                                  hdrl_image_reject(hima, i, j);
                                  hdrl_image_reject(ehima, i, j);
                                  hdrl_image_reject(bhima, i, j);
                          }
                  }
                  /* trim right image border along Y direction */
                  for(cpl_size j = 1; j <= sy; j++) {
                          for(cpl_size i = sx - edge_trim + 1; i <= sx; i++) {
                                  hdrl_image_reject(hima, i, j);
                                  hdrl_image_reject(ehima, i, j);
                                  hdrl_image_reject(bhima, i, j);
                          }
                  }
          }
  }

  /* single list are not anymore needed as data are copied to the hdrl_imagelist
   * therefore we free them */
  cpl_imagelist_delete(dlist);
  cpl_imagelist_delete(explist);
  cpl_imagelist_delete(bpmlist);
  if (qual_frm != NULL)
    cpl_imagelist_delete(qlist);
  if (errs_frm != NULL)
    cpl_imagelist_delete(elist);

  /* Do the calculation */
  if (size == 1){
      /* Subtract the background if we have an image and not a cube */
      cpl_msg_info(cpl_func, "Reading the image ...");
      hdrl_image* hima = hdrl_imagelist_get(hlist, 0);
      hdrl_image* ehima = hdrl_imagelist_get(ehlist, 0);
      hdrl_image* bhima = hdrl_imagelist_get(bhlist, 0);

      /* Subtract the background on request */
      if(subtract_bkg == CPL_TRUE) {
          hdrl_value m = hdrl_image_get_median(hima);
          cpl_msg_info(cpl_func, "Subtracting the median (%f, %f) as requested ...", m.data, m.error);
          hdrl_image_sub_scalar(hima, m);
      }

      /*!!!!!!!!! Interface with the hdrl library function !!!!!!!!!! */
      tab_res = hdrl_resample_image_to_table(hima, wcs);
      tab_exp = hdrl_resample_image_to_table(ehima, wcs);
      tab_bpm = hdrl_resample_image_to_table(bhima, wcs);
  } else {
      cpl_msg_info(cpl_func, "Converting imagelist to table with hdrl function");
      tab_res = hdrl_resample_imagelist_to_table(hlist, wcs);
      tab_exp = hdrl_resample_imagelist_to_table(ehlist, wcs);
      tab_bpm = hdrl_resample_imagelist_to_table(bhlist, wcs);
  }

  /*Cleanup the memory */
  if (hlist != NULL)
          hdrl_imagelist_delete(hlist);
  if (ehlist != NULL)
          hdrl_imagelist_delete(ehlist);
  if (bhlist != NULL)
          hdrl_imagelist_delete(bhlist);
  cpl_wcs_delete(wcs);
  cpl_propertylist_delete(xheader_data);

  *restbl = tab_res;
  *exptbl = tab_exp;
  *bpmtbl = tab_bpm;
  return CPL_ERROR_NONE;
}

/**
 * @brief transform and insert the data (pixel flux, error, quality) into a
 * cpl_table. Parallelised code
 * @param data_frm input frame with data
 * @param errs_frm input frame with errors
 * @param qual_frm input frame with pixel quality
 * @param data_ext_id input frame extension containing data
 * @param errs_ext_id input frame extension containing errors
 * @param qual_ext_id input frame extension containing pixel quality
 * @param is_variance  boolean to indicate if errs_frm contains variance or error
 * @param subtract_bkg boolean to indicate if background need to be corrected
 * @param edgetrim     indicates how many pixels need to be trimmed from each
 *                     resampled plane
 * @return cpl_table with filled information
 * @doc
 * */
static cpl_error_code
fors_img_stack_get_table_from_frameset(cpl_table** restbl,
                                       cpl_table** exptbl,
                                       cpl_table** bpmtbl,
                                       const cpl_frame* data_frm,
                                       const cpl_frame* errs_frm,
                                       const cpl_frame* qual_frm,
                                       const cpl_size data_ext_id,
                                       const cpl_size errs_ext_id,
                                       const cpl_size qual_ext_id,
                                       const cpl_boolean is_variance,
                                       const cpl_boolean subtract_bkg,
                                       const int edge_trim) {

  cpl_ensure(data_frm, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  /* Assumption - all have the same number of extensions */
  cpl_size next = cpl_frame_get_nextensions(data_frm);
  cpl_msg_info(cpl_func, "Analysing and processing file %s",
               cpl_frame_get_filename(data_frm));

  /* Assumption
   * if data_ext_id == -1 the loop can be done from 0 to next for data, error,
   * and quality; only the filename changes
   * if data_ext_id != -1 no loop is needed and is difficult to do in the loop
   *  */

  cpl_msg_indent_more();
  cpl_table* table = NULL;
  cpl_table* exp_table = NULL;
  cpl_table* bpm_table = NULL;

  if(data_ext_id != -1) {
      cpl_msg_info(cpl_func, "Extension: %02lld", data_ext_id);
      /* No Loop, only use the extension given by the user */
      fors_img_stack_frameset_to_table(&table, &exp_table, &bpm_table,
                                       data_frm, data_ext_id,
                                       errs_frm, errs_ext_id,
                                       qual_frm, qual_ext_id,
                                       is_variance, subtract_bkg, edge_trim);
  } else {
      /* Loop */
      for (cpl_size i = 0; i < next + 1; i++ ) {
          cpl_table * table_local = NULL;
          cpl_table * exp_table_local = NULL;
          cpl_table * bpm_table_local = NULL;
          cpl_msg_info(cpl_func,"Extension: %02lld", i);
          fors_img_stack_frameset_to_table(&table_local, &exp_table_local, &bpm_table_local,
                                           data_frm, i,
                                           errs_frm, i,
                                           qual_frm, i,
                                           is_variance, subtract_bkg, edge_trim);

          /* If table_local == NULL go to the next extension */
          if (table_local == NULL) {
              cpl_msg_info(cpl_func, "No suitable data found - continuing");
              continue;
          }

          /* Now table_local is always != NULL */
          if (table == NULL){
              table = table_local;
          } else {
              cpl_size nrow = cpl_table_get_nrow(table);
              cpl_table_insert(table, table_local, nrow);
              cpl_table_delete(table_local);
              table_local = NULL;
          }
          if (exp_table == NULL){
              exp_table = exp_table_local;
          } else {
              cpl_size nrow = cpl_table_get_nrow(exp_table);
              cpl_table_insert(exp_table, exp_table_local, nrow);
              cpl_table_delete(exp_table_local);
              exp_table_local = NULL;
          }
          if (bpm_table == NULL){
              bpm_table = bpm_table_local;
          } else {
              cpl_size nrow = cpl_table_get_nrow(bpm_table);
              cpl_table_insert(bpm_table, bpm_table_local, nrow);
              cpl_table_delete(bpm_table_local);
              bpm_table_local = NULL;
          }
      }
  }
  cpl_msg_indent_less();

  *restbl = table;
  *exptbl = exp_table;
  *bpmtbl = bpm_table;
  return CPL_ERROR_NONE;
}

/**
 * @brief save table with ra, dec,lambda information

 * @param par input parameters
 * @param parlist input parameters
 * @param frameset input frameset
 * @param tab_final input table
 * @return cpl_error_code
 * */
static cpl_error_code
fors_img_stack_tablesave (const cpl_parameter *par,
                             const cpl_parameterlist *parlist,
                             cpl_frameset *frameset, cpl_table *tab_final)
{
  /* Save the final table if required */
  par = cpl_parameterlist_find_const (parlist, RECIPE_NAME".save-table");
  const int save_table = cpl_parameter_get_bool (par);
  if (save_table)
    {
      cpl_msg_info (cpl_func, "Saving the table before resampling");
      cpl_propertylist *applist = cpl_propertylist_new ();
      cpl_propertylist_update_string (applist, CPL_DFS_PRO_CATG,
                                      "FORS_IMG_STACK_TABLE");
      cpl_dfs_save_table (frameset, NULL, parlist, frameset, NULL, tab_final,
                          NULL, fors_img_stack_name, applist, NULL,
                          PACKAGE "/" PACKAGE_VERSION,
                          "fors_img_stack_table.fits");
      cpl_propertylist_delete (applist);
    }
  return cpl_error_get_code();
}

static cpl_image *
fors_img_stack_create_confidence_map(hdrl_image* stack_img)
{
    cpl_image * data = hdrl_image_get_image(stack_img);
    cpl_image * errors = hdrl_image_get_error(stack_img);
    cpl_mask  * bpm = hdrl_image_get_mask(stack_img);

    // Now turn that into a confidence from 0 to 100
    int nx = (int)cpl_image_get_size_x(data);
    int ny = (int)cpl_image_get_size_y(data);
    cpl_image * confmap = cpl_image_new((cpl_size)nx, (cpl_size)ny, CPL_TYPE_FLOAT);
    long npts = nx*ny;

    double *imgdata = cpl_image_get_data_double(data);
    double *errdata = cpl_image_get_data_double(errors);
    float *confdata = cpl_image_get_data_float(confmap);
    const cpl_binary *bpmdata = (bpm ? cpl_mask_get_data_const(bpm) : NULL);

    if (cpl_error_get_code()) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        return NULL;
    }

    // Iterate over the values and calculate the confidence
    for (long i = 0; i < npts; i++) {
        if (bpm && bpmdata[i]) {
            confdata[i] = 0;
        } else if (imgdata[i] <= 0) {
            confdata[i] = 0;
        } else {
            confdata[i] = 100.0 / exp(errdata[i] / imgdata[i]);
            if (confdata[i] > 100) {
                confdata[i] = 100;
            } else if (confdata[i] < 0) {
                confdata[i] = 0;
            }
        }
    }
    return confmap;
}

/* Update bad pixel map with 'stacked' bad pixel map from single exposures.  */
void fors_img_stack_update_mask_bpm(cpl_mask *bpmmask, cpl_image *bpmmap)
{
    float *bpmdata = cpl_image_get_data_float(bpmmap);
    
    int nx = (int)cpl_image_get_size_x(bpmmap);
    int ny = (int)cpl_image_get_size_y(bpmmap);
    long npts = nx*ny;
    long zcount = 0;
    cpl_mask *newmask = cpl_mask_new(nx,ny);
    cpl_binary *newdata = cpl_mask_get_data(newmask);
    for (long i = 0; i < npts; i++) {
        if (bpmdata[i] > 0) {
            newdata[i] = CPL_TRUE;
            ++zcount;
        } else {
            newdata[i] = CPL_FALSE;
        }
    }
    
    cpl_mask_or(bpmmask, newmask);
    cpl_mask_delete(newmask);
    
    cpl_msg_info(cpl_func, "Bad pixel mask updated, second bpm map contained %ld / %ld (%.2f%%) pixels from single frames", zcount, npts, 100.0 * zcount / npts);
    
}

/* A helper function for get_unique_exposures() that is used to determine
 * whether two exposures are simultaneous in time */
static bool
exposures_are_simultaneous(const cpl_propertylist* plist1, const cpl_propertylist* plist2)
{
    // Exposures are simultaneous if they have the same MJD-OBS, MJD-END and EXPTIME.
    // Because we are only considering a single instrument, this is enough
    // to determine that these exposures will create a stack.
    const double mjdobs1 = cpl_propertylist_get_double(plist1, "MJD-OBS");
    const double mjdobs2 = cpl_propertylist_get_double(plist2, "MJD-OBS");
    const double mjdend1 = cpl_propertylist_get_double(plist1, "MJD-END");
    const double mjdend2 = cpl_propertylist_get_double(plist2, "MJD-END");
    const double exptime1 = cpl_propertylist_get_double(plist1, "EXPTIME");
    const double exptime2 = cpl_propertylist_get_double(plist2, "EXPTIME");
    if ((cpl_error_get_code() == CPL_ERROR_NONE) &&
        (fabs(mjdobs1 - mjdobs2) < DBL_EPSILON) &&
        (fabs(mjdend1 - mjdend2) < DBL_EPSILON) &&
        (fabs(exptime1 - exptime2) < DBL_EPSILON)) {
        return true;
    }
    // Exposures aren't equal if there was an error
    cpl_error_reset();
    return false;
}

/* From the given list of propertylists corresponding to all the input frames,
 * determine which are unique, and return a list of only unique inputs */
std::list<cpl_propertylist*>
get_unique_exposures(std::list<cpl_propertylist*> plists)
{
    std::list<cpl_propertylist*> retval;
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    // The first exposure is always in the return list
    retval.push_back(*i);
    i++;
    // Now loop over all the exposures we are given, comparing them
    // to the list of unique exposures we are building. If we haven't
    // already got this exposure in our list then add it.
    while (i != plists.end()) {
        bool are_simultaneous = false;
        std::list<cpl_propertylist*>::iterator j = retval.begin();
        while (!are_simultaneous && (j != retval.end())) {
            if (exposures_are_simultaneous(*i, *j)) {
                are_simultaneous = true;
            }
            j++;
        }
        if (!are_simultaneous) {
            retval.push_back(*i);
        }
        i++;
    }
    return retval;
}

/* Calculate the value of TEXPTIME, from the list of unique exposures */
double
get_texptime(std::list<cpl_propertylist*> plists)
{
    double retval = 0.0;
    std::list<cpl_propertylist*>::iterator i = plists.begin();
    while (i != plists.end()) {
        double exptime = cpl_propertylist_get_double(*i, "EXPTIME");
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            retval += exptime;
        }
        cpl_error_reset();
        i++;
    }
    return retval;
}

void calculate_abmag(cpl_table* sources, fors_idp_zp_data& zp_data, double zp, double dzp, int nzp, char filter_band, double abmag_minus_vega,
    double psf_fwhm_pix, double sky_mag, double pixscale, cpl_propertylist* plist)
{
    char* phot_calib = NULL;
    double zpdiff = 99.9;

    if ((filter_band == '?') || (nzp < 3)) {
        // Non-BVRI filters use the static table
        // as do images with too few Gaia stars
        phot_calib = cpl_sprintf("STATIC");
    } else {
        // BVRI filters with enough Gaia stars use the ZP calculated
        // from the Gaia synthetic photometry
        phot_calib = cpl_sprintf("GAIA");
    }

    // Report on the photometric calibration
    cpl_msg_info(cpl_func, "ESO QC PHOT_CALIB = %s", phot_calib);
    cpl_propertylist_update_string(plist, "ESO QC PHOT_CALIB", phot_calib);

    if (!strcmp(phot_calib, "GAIA")) {
        cpl_msg_info(cpl_func, "Using zeropoint calculated from %d Gaia syntphot stars: %f +- %f", nzp, zp, dzp);
        zpdiff = zp_data.zp - zp;
        zp_data.zp = zp;
        zp_data.zp_e = dzp;
        zp_data.extinction = 0;
    } else {
        cpl_msg_info(cpl_func, "Using extracted zeropoint: %f +- %f", zp_data.zp, zp_data.zp_e);
    }
    cpl_propertylist_update_double(plist, "ESO QC DIFFZP", zpdiff);

    cpl_propertylist_update_double(plist, "PHOTZP", zp_data.zp);
    cpl_propertylist_set_comment(plist, "PHOTZP", "Photometric zeropoint MAG=-2.5*log(data)+PHOTZP");
    cpl_propertylist_update_double(plist, "PHOTZPER", zp_data.zp_e);
    cpl_propertylist_set_comment(plist, "PHOTZPER", "Uncertainty on PHOTZP");
    if (!strcmp(phot_calib, "GAIA")) {
        cpl_propertylist_update_double(plist, "PHOTZPDI", zpdiff);
        cpl_propertylist_set_comment(plist, "PHOTZPDI", "Difference zeropoint STATIC-Gaia");
    }

    const double aperture_correction = 0.0; //No aperture correction is performed
    const double ab_mag_lim = calc_ab_mag_lim(zp_data.zp, abmag_minus_vega, sky_mag,
            psf_fwhm_pix, aperture_correction);

    cpl_propertylist_update_double(plist, "ABMAGLIM", ab_mag_lim);
    cpl_propertylist_set_comment(plist, "ABMAGLIM", "5-sigma limiting AB magnitude for point sources");
    cpl_msg_info(cpl_func, "AB mag lim = %f", ab_mag_lim);


    double ab_mag_sat = 0.0;
    double sky_med_level = 0.0;

    sky_med_level = pow(10.0, -0.4 * sky_mag) * pixscale * pixscale;
    double pre_overscan_level = 0;

    const bool success = calc_ab_mag_sat(sources, sky_med_level, pre_overscan_level,
            zp_data.zp, abmag_minus_vega, &ab_mag_sat);

    if(!success){
        cpl_msg_warning(cpl_func, "Calculation of ABMAGSAT failed.");
        cpl_free(phot_calib);
        return;
    }
    cpl_propertylist_update_double(plist, "ABMAGSAT", ab_mag_sat);
    cpl_propertylist_set_comment(plist, "ABMAGSAT", "Saturation limit for point sources (AB mags)");
    cpl_msg_info(cpl_func, "AB mag sat = %f", ab_mag_sat);

    cpl_free(phot_calib);
}

/**
 * @brief This function resample input data
 *
 * @param frameset input set of frames
 * @param parlist input recipe parameters
 * @return cpl_error_code
 * @doc
 *
 *  */
cpl_error_code fors_img_stack(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{

  cpl_frameset* in_set = NULL;
  cpl_frameset *phot_table    = NULL;
  cpl_frameset *static_table  = NULL;
  const cpl_parameter     *   par = NULL;
  cpl_propertylist *plist = NULL;
  cpl_propertylist *elist = NULL;
  fors_setting *setting = NULL;
  extract_method  *em = NULL;
  int                         extnum_raw = 0;
  int                         extnum_err = 0;
  int                         extnum_bpm = 0;
  int                         edge_trim = 0;
  double                      magcutE = 0;
  double                      magcutk = 0;
  double                      magsyserr = 0;
  const char                 *cacheloc = NULL;
  cpl_boolean                 is_variance = CPL_FALSE;
  cpl_boolean                 subtract_bkg = CPL_FALSE;
  fors_star_list *stars = NULL;
  fors_extract_sky_stats sky_stats;
  cpl_image *background = NULL;
  cpl_table *sources = NULL;
  cpl_propertylist *extract_qc = NULL;

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".ext-nb-raw");
  extnum_raw = cpl_parameter_get_int(par);
  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".ext-nb-raw-err");
  extnum_err = cpl_parameter_get_int(par);
  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".ext-nb-raw-bpm");
  extnum_bpm = cpl_parameter_get_int(par);

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".subtract-background");
  if (cpl_parameter_get_bool(par)) {
      subtract_bkg = CPL_TRUE;
  }

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".edge-trim");
  edge_trim = cpl_parameter_get_int(par);

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".magsyserr");
  magsyserr = cpl_parameter_get_double(par);

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".magcutE");
  magcutE = cpl_parameter_get_double(par);

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".magcutk");
  magcutk = cpl_parameter_get_double(par);

  par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".cacheloc");
  cacheloc = cpl_parameter_get_string(par);
        
  /* Check initial Entries */
  fors_dfs_set_groups(frameset);

  /* Find calibrations */
  phot_table = fors_frameset_extract(frameset, PHOT_TABLE);
  if (cpl_frameset_get_size(phot_table) != 1) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                                   "Exactly 1 " PHOT_TABLE " required.");
  }
  static_table = fors_frameset_extract(frameset, STATIC_PHOT_COEFF_TABLE);
  if (cpl_frameset_get_size(static_table) != 1) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                                   "Exactly 1 " STATIC_PHOT_COEFF_TABLE " required.");
  }

  cpl_msg_info(cpl_func, "Searching for files ...");
  cpl_msg_indent_more();

  in_set = cpl_frameset_new();

  for (int i = 0; i < 3; i++) {
      cpl_frameset* tmp_set = fors_frameset_extract(frameset, input_cats[i]);
      if (tmp_set == NULL) {
          cpl_msg_info(cpl_func, "No input %s frames found", input_cats[i]);
      } else {
          cpl_msg_info(cpl_func, "%02d file(s) of type %s",
                   (int)cpl_frameset_get_size(tmp_set), input_cats[i]);
          cpl_frameset_join(in_set, tmp_set);
          cpl_frameset_delete(tmp_set);
          tmp_set = NULL;
      }
  }

  if (in_set == NULL){
      return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                                   "Missing RAW files");
  } else {
      cpl_msg_info(cpl_func, "%02d file(s) in total",
                   (int)cpl_frameset_get_size(in_set));
  }

  cpl_msg_indent_less();

  /*Do some consistency checks*/

  cpl_ensure(in_set, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

  cpl_size nb_frames = cpl_frameset_get_size(in_set) ;

  /* Get instrument setting */
  setting = fors_setting_new(cpl_frameset_get_position(in_set, 0));

  /*convert the images in a table of predefined structure*/

  cpl_table* restable = NULL; /*Final table to resample*/
  cpl_table* restmp = NULL;

  std::list<cpl_table*> exptables;
  std::list<cpl_table*> bpmtables;
  cpl_table* exptmp = NULL;
  cpl_table* bpmtmp = NULL;

  cpl_frame   *   data_frame = NULL;
  cpl_frame   *   err_frame = NULL;
  cpl_frame   *   bpm_frame = NULL;

  const char* fname = NULL;
  cpl_propertylist* phulist = NULL;
  cpl_propertylist* datlist = NULL;
  std::list<cpl_propertylist*> plists;
  char * filter_name = NULL;
  double gain0 = 0;
  double exptime0 = 0;

  /*Loops over all frames and adds the images/imagelists to the final table */
  for (cpl_size i = 0; i < nb_frames ; i++) {

      data_frame = cpl_frameset_get_position(in_set, i);

      // Gather the gain and exposure time from first input frame
      fors_setting* tmp_setting = fors_setting_new(data_frame);
      if (i == 0) {
          gain0 = 1. / tmp_setting->average_gain; // [e-/ADU]
          exptime0 = tmp_setting->exposure_time;
      }
      fors_setting_delete(&tmp_setting);

      if (extnum_err >= 0) {
          err_frame = cpl_frame_duplicate(data_frame);
      }

      if (extnum_bpm >= 0) {
          bpm_frame = cpl_frame_duplicate(data_frame);
      }

      /* Load the PHU */
      fname = cpl_frame_get_filename(data_frame);
      phulist = cpl_propertylist_load(fname, 0);
      datlist = cpl_propertylist_load(fname, extnum_raw);
      /* PHOTSYS is in the data header, but we want to keep track of it so
       * just stuff it in the copy of the PHU for now */
      cpl_propertylist_copy_property(phulist, datlist, "PHOTSYS");
      cpl_propertylist_delete(datlist);
      plists.push_back(phulist);

      /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
      /* !!! Calls the hdrl_resample_image/imagelis_to_table functions !!! */
      /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
      fors_img_stack_get_table_from_frameset(&restmp, &exptmp, &bpmtmp,
                                             data_frame,
                                             err_frame,
                                             bpm_frame,
                                             extnum_raw, extnum_err, extnum_bpm,
                                             is_variance, subtract_bkg, edge_trim);

      if(nb_frames == 1) {
          restable = restmp;
      } else {
          fors_img_stack_update_table(restmp, &restable);
          cpl_table_delete(restmp);
      }
      exptables.push_back(exptmp);
      bpmtables.push_back(bpmtmp);

  }
  cpl_ensure_code(restable, CPL_ERROR_NULL_INPUT);

  /* Save the final table if required */
  fors_img_stack_tablesave (par, parlist, frameset, restable);

  /* Headers to add the PHU */
  plist = cpl_propertylist_new();
  /* Headers to add to each extension */
  elist = cpl_propertylist_new();

  /* A flag to clear if there are problems with the headers */
  bool headers_ok = true;

  /* Handle PROG_ID */
  std::set<std::string> prog_ids = get_string_header_set(plists, "PROG_ID");
  std::set<std::string>::iterator i = prog_ids.begin();
  if (prog_ids.size() == 0) {
      cpl_msg_warning(cpl_func, "No PROG_ID header found in input files");
  } else if (prog_ids.size() == 1) {
      cpl_propertylist_update_string(plist, "PROG_ID", i->c_str());
  } else {
      cpl_propertylist_update_string(plist, "PROG_ID", "MULTI");
      int j = 1;
      while (i != prog_ids.end()) {
          char* key = cpl_sprintf("PROGID%d", j);
          cpl_propertylist_update_string(plist, key, i->c_str());
          i++;
          j++;
          cpl_free(key);
      }
  }

  /* Handle FLUXCAL */
  std::set<std::string> fluxcals = get_string_header_set(plists, "FLUXCAL");
  i = fluxcals.begin();
  if (fluxcals.size() == 0) {
      cpl_msg_warning(cpl_func, "No FLUXCAL header found in input files");
  } else if (fluxcals.size() == 1) {
      cpl_propertylist_update_string(plist, "FLUXCAL", i->c_str());
  } else {
      cpl_msg_error(cpl_func, "Multiple FLUXCAL headers with different values found in input files");
      headers_ok = false;
  }

  /* Handle ESO OBS ID */
  std::set<long> obs_ids = get_long_header_set(plists, "ESO OBS ID");
  std::set<long>::iterator k = obs_ids.begin();
  if (obs_ids.size() == 0) {
      cpl_msg_warning(cpl_func, "No ESO OBS ID header found in input files");
  } else {
      int j = 1;
      while (k != obs_ids.end()) {
          char* key = cpl_sprintf("OBID%d", j);
          cpl_propertylist_update_long(plist, key, *k);
          k++;
          j++;
          cpl_free(key);
      }
  }

  /* Handle FILTER */
  std::set<std::string> filters = get_string_header_set(plists, "FILTER");
  i = filters.begin();
  if (filters.size() == 0) {
      cpl_msg_warning(cpl_func, "No FILTER header found in input files");
  } else if (filters.size() == 1) {
      filter_name = cpl_strdup(i->c_str());
      cpl_propertylist_update_string(plist, "FILTER", filter_name);
  } else {
      cpl_msg_error(cpl_func, "Multiple FILTER headers with different values found in input files");
      headers_ok = false;
  }

  /* Handle PHOTSYS */
  std::set<std::string> photsyss = get_string_header_set(plists, "PHOTSYS");
  i = photsyss.begin();
  if (photsyss.size() == 0) {
      cpl_msg_warning(cpl_func, "No PHOTSYS header found in input files");
  } else if (photsyss.size() == 1) {
      cpl_propertylist_update_string(plist, "PHOTSYS", i->c_str());
  } else {
      cpl_msg_error(cpl_func, "Multiple PHOTSYS headers with different values found in input files");
      headers_ok = false;
  }
  cpl_error_reset();

  /* Handle PROCSOFT */
  std::set<std::string> procsofts = get_string_header_set(plists, "PROCSOFT");
  i = procsofts.begin();
  if (procsofts.size() == 0) {
      cpl_msg_warning(cpl_func, "No PROCSOFT header found in input files");
  } else {
      cpl_propertylist_update_string(plist, "PROCSOFT", i->c_str());
      if (procsofts.size() > 1) {
          cpl_msg_warning(cpl_func, "Multiple PROCSOFT headers with different values found in input files");
      }
  }

  /* Handle REFERENC */
  std::set<std::string> references = get_string_header_set(plists, "REFERENC");
  i = references.begin();
  if (references.size() == 0) {
      cpl_msg_warning(cpl_func, "No REFERENC header found in input files");
  } else {
      cpl_propertylist_update_string(plist, "REFERENC", i->c_str());
      if (references.size() > 1) {
          cpl_msg_warning(cpl_func, "Multiple REFERENC headers with different values found in input files");
      }
  }

  /* Handle MJD-OBS and MJD-END */
  handle_mjd(plists, plist);

  /* If any headers were bad, then bail out here */
  cpl_ensure_code(headers_ok, CPL_ERROR_ILLEGAL_INPUT);

  /* Get the number of unique exposures */
  std::list<cpl_propertylist*> unique_plists = get_unique_exposures(plists);
  cpl_msg_info(cpl_func, "Number of unique exposures is %lu", unique_plists.size());
  double texptime = get_texptime(unique_plists);
  cpl_propertylist_update_double(plist, "TEXPTIME", texptime);

  /* Total number of combined frames */
  cpl_propertylist_update_int(plist, "NCOMBINE", nb_frames);

  /* Number of unique combined frames */
  cpl_propertylist_update_int(plist, "NSTACK", unique_plists.size());

  /* Getting the input wcs needed for determine the scaling of output cube */
  cpl_wcs *wcs = fors_img_stack_get_wcs_from_frameset(in_set);
  cpl_ensure_code(wcs, CPL_ERROR_NULL_INPUT);

  cpl_msg_info(cpl_func, "WCS used for the output grid definition and passed to "
      "hdrl_resample_compute():");
  cpl_msg_indent_more();
  fors_img_stack_wcs_print(wcs);
  cpl_msg_indent_less();

  hdrl_parameter *res_method = NULL;
  /* set re-sampling method */
  res_method = fors_img_stack_set_method(parlist);
  cpl_ensure_code(res_method, CPL_ERROR_NULL_INPUT);

  /* set re-sampling outputgrid */
  hdrl_parameter *res_outputgrid = NULL;
  res_outputgrid = fors_img_stack_set_outputgrid(parlist, restable, wcs);
  cpl_ensure_code(res_outputgrid, CPL_ERROR_NULL_INPUT);

  /* Get parameters */
  em = fors_extract_method_new(parlist, RECIPE_NAME);
  cpl_ensure_code(em, CPL_ERROR_NULL_INPUT);

  hdrl_resample_result *result = NULL;
  hdrl_resample_result *exp_result = NULL;
  hdrl_resample_result *bpm_result = NULL;

  /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
  /* !!! Calling the hdrl_resample_compute function doing the real work !!! */
  /* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

  result = hdrl_resample_compute(restable, res_method, res_outputgrid, wcs);

  /* Also resample each exposure table and sum into an exposure map */
  cpl_image* exp_img = NULL;
  std::list<cpl_table*>::iterator eiter = exptables.begin();
  while (eiter != exptables.end()) {
      exp_result = hdrl_resample_compute(*eiter, res_method, res_outputgrid, wcs);
      cpl_ensure_code(exp_result, CPL_ERROR_NULL_INPUT);
      cpl_size planes = hdrl_imagelist_get_size(exp_result->himlist);
      cpl_ensure_code(planes == 1, CPL_ERROR_ILLEGAL_OUTPUT);
      hdrl_image* tmphimg = hdrl_imagelist_get(exp_result->himlist, 0);
      cpl_image * tmpimg = cpl_image_cast(hdrl_image_get_image(tmphimg), CPL_TYPE_FLOAT);
      cpl_ensure_code(tmpimg, CPL_ERROR_ILLEGAL_OUTPUT);
      if (exp_img) {
          cpl_image_add(exp_img, tmpimg);
          cpl_image_delete(tmpimg);
      } else {
          exp_img = tmpimg;
      }
      eiter++;
  }
  /* Also resample each bad pixel table and sum into an bad pixel map */
  cpl_image* bpm_img = NULL;
  std::list<cpl_table*>::iterator biter = bpmtables.begin();
  while (biter != bpmtables.end()) {
        bpm_result = hdrl_resample_compute(*biter, res_method, res_outputgrid, wcs);
        cpl_ensure_code(bpm_result, CPL_ERROR_NULL_INPUT);
        cpl_size planes = hdrl_imagelist_get_size(bpm_result->himlist);
        cpl_ensure_code(planes == 1, CPL_ERROR_ILLEGAL_OUTPUT);
        hdrl_image* tmphimg = hdrl_imagelist_get(bpm_result->himlist, 0);
        cpl_image * tmpimg = cpl_image_cast(hdrl_image_get_image(tmphimg), CPL_TYPE_FLOAT);
        cpl_ensure_code(tmpimg, CPL_ERROR_ILLEGAL_OUTPUT);
        if (bpm_img) {
            cpl_image_add(bpm_img, tmpimg);
            cpl_image_delete(tmpimg);
        } else {
            bpm_img = tmpimg;
        }
        biter++;
  }

  // Check that the resample succeeded
  cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(exp_img, CPL_ERROR_ILLEGAL_OUTPUT);
  cpl_ensure_code(bpm_img, CPL_ERROR_ILLEGAL_OUTPUT);

  // Check that we got the expected shape of result
  cpl_size planes = hdrl_imagelist_get_size(result->himlist);
  cpl_ensure_code(planes == 1, CPL_ERROR_ILLEGAL_OUTPUT);

  // Grab the image
  hdrl_image* result_img = hdrl_imagelist_get(result->himlist, 0);
  cpl_ensure_code(result_img, CPL_ERROR_ILLEGAL_OUTPUT);

  // Improve mask with bad pixel maps from single exposures
  cpl_mask* bpm = hdrl_image_get_mask(result_img);
  fors_img_stack_update_mask_bpm(bpm, bpm_img);
  hdrl_image_reject_from_mask(result_img, bpm);

  /* Printing the wcs after resampling: */
  cpl_wcs_delete(wcs);
  wcs = cpl_wcs_new_from_propertylist(result->header);
  cpl_msg_info(cpl_func, "Final WCS after resampling: ");
  cpl_msg_indent_more();
  fors_img_stack_wcs_print(wcs);
  cpl_msg_indent_less();

  // Reject any pixels that are invalid in the exposure map
  cpl_image_accept_all(exp_img);
  cpl_image_reject_value(exp_img, (cpl_value)(CPL_VALUE_NAN | CPL_VALUE_ZERO | CPL_VALUE_NOTFINITE));
  cpl_image_reject_from_mask(exp_img, bpm);

  /* Create a confidence map */
  cpl_image *confmap = fors_img_stack_create_confidence_map(result_img);
  update_confidence_map(confmap, em);
  cpl_image_save(confmap, "confmap.fits", CPL_TYPE_FLOAT, NULL, CPL_IO_DEFAULT);

  /* Extract sources */
  cpl_image * sci_data = cpl_image_cast(hdrl_image_get_image(result_img), CPL_TYPE_FLOAT);
  cpl_image * sci_var = cpl_image_cast(hdrl_image_get_error(result_img), CPL_TYPE_FLOAT);
  cpl_image_power(sci_var, 2);
  fors_image* sci = fors_image_new(sci_data, sci_var);

  stars = fors_extract(sci, setting, em, magsyserr, confmap, wcs,
                       &sky_stats, &background, &sources, &extract_qc);
  cpl_ensure_code(stars, CPL_ERROR_ILLEGAL_OUTPUT);
  cpl_ensure_code(sources, CPL_ERROR_ILLEGAL_OUTPUT);

  /* Save the source table */
  fors_dfs_save_table(frameset, sources, SOURCES_SCI_IMG_STACK,
                      extract_qc, parlist, fors_img_stack_name,
                      cpl_frameset_get_position(frameset, 0));

  /* Now calculate PSF_FWHM */
  cpl_table* star_sources = cpl_table_duplicate(sources);
  cpl_table_select_all(star_sources);
  int nrows = (int)cpl_table_get_nrow(star_sources);
  cpl_msg_info(cpl_func, "Considering %d initial sources for PSF_FWHM", nrows);

  if (cpl_table_has_column(star_sources, "CLASS_STAR")) {
      cpl_table_and_selected_float(star_sources, "CLASS_STAR", CPL_GREATER_THAN, 0.0);
      if (cpl_error_get_code()) {
          cpl_msg_warning(cpl_func, "Error when selecting CLASS_STAR > 0.0: %s", cpl_error_get_message());
          cpl_error_reset();
      }
  } else if (cpl_table_has_column(star_sources, "Classification")) {
      cpl_table_and_selected_double(star_sources, "Classification", CPL_LESS_THAN, 0.0);
      if (cpl_error_get_code()) {
          cpl_msg_warning(cpl_func, "Error when selecting Classification < 0.0: %s", cpl_error_get_message());
          cpl_error_reset();
      }
  }

  /* Load the static photometry table */
  cpl_msg_info(cpl_func, "Filling from %s", STATIC_PHOT_COEFF_TABLE);
  const cpl_frame * frame_static_table = cpl_frameset_get_position_const(static_table, 0);

  double airmass = 1.0;
  fors_idp_zp_data zp_data = {0, 0, 0};
  double amag_vega = 0;
  double wavecen = 6000;
  double pixscale = 0;

  /* Calculate an average airmass */
  std::vector<double> airmasses = get_double_header_vector(plists, "AIRMASS");
  airmass = std::accumulate(airmasses.begin(), airmasses.end(), 0.0) / airmasses.size();
  cpl_msg_info(cpl_func, "Average airmass is %f", airmass);

  /* And gain from first frame */
  cpl_msg_info(cpl_func, "Gain is %f [e-/ADU]", gain0);

  /* And a median of the sky */
  std::vector<double> skybkgs = get_double_header_vector(plists, "ESO QC SKY_NOISE");
  double bkg_rms = get_median(skybkgs);
  cpl_msg_info(cpl_func, "Sky bkg_rms is %f", bkg_rms);

  /* Now read the data from the static table */
  bool res =  fill_from_static_table(frame_static_table,
          filter_name, exptime0, gain0, airmass,
          &zp_data, &amag_vega, &wavecen);
  if (!res) {
      cpl_msg_warning(cpl_func, "Could not fill information from %s", STATIC_PHOT_COEFF_TABLE);
  }
  cpl_msg_info(cpl_func, "Using central wavelength value of %f", wavecen);

  /* Calculate FWHM from the headers */
  std::list<double> psf_fwhm_guesses = handle_fwhm(plists, wavecen);
  double psf_fwhm_guess = std::accumulate(psf_fwhm_guesses.begin(), psf_fwhm_guesses.end(), 0.0) / psf_fwhm_guesses.size();
  cpl_msg_info(cpl_func, "PSF_FWHM estimate from headers is %f arcsec", psf_fwhm_guess);

  /* Figure out if we can calculate FWHM more accurately from the extracted sources */
  double psf_fwhm = -1;
  cpl_table* tmp_sources = cpl_table_extract_selected(star_sources);
  cpl_table_delete(star_sources);
  nrows = (int)cpl_table_get_nrow(tmp_sources);
  if (nrows > 0) {
      cpl_msg_info(cpl_func, "Using %d probable stellar sources to calculate PSF_FWHM", nrows);

      /* Calculate the pixel scale of the output WCS */
      const cpl_matrix * cd = cpl_wcs_get_cd(wcs);
      double cd11 = cpl_matrix_get(cd, 0, 0);
      double cd12 = cpl_matrix_get(cd, 0, 1);
      double cd21 = cpl_matrix_get(cd, 1, 0);
      double cd22 = cpl_matrix_get(cd, 1, 1);
      pixscale = 3600 * (sqrt(cd11*cd11 + cd12*cd12) + sqrt(cd21*cd21 + cd22*cd22)) / 2;
      cpl_msg_info(cpl_func, "Pixel scale of output WCS is %f arcsec/pixel", pixscale);

      if (cpl_table_has_column(tmp_sources, "FWHM_IMAGE")) {
          psf_fwhm =  cpl_table_get_column_median(tmp_sources, "FWHM_IMAGE") * pixscale;
      } else if (cpl_table_has_column(tmp_sources, "FWHM")) {
          psf_fwhm =  cpl_table_get_column_median(tmp_sources, "FWHM") * pixscale;
      } else {
          cpl_msg_warning(cpl_func, "Could not calculate PSF_FWHM from sources, using estimate");
      }
  }
  /* If the calculation of PSF_FWHM failed for any reason, then use the alternate strategy */
  if ((psf_fwhm < 0) && (psf_fwhm_guess > 0)) {
      psf_fwhm = psf_fwhm_guess;
  }
  cpl_msg_info(cpl_func, "Using PSF_FWHM value of %f arcsec", psf_fwhm);
  cpl_propertylist_update_double(plist, "PSF_FWHM", psf_fwhm);
  cpl_propertylist_set_comment(plist, "PSF_FWHM", "Spatial resolution (arcsec)");

  /* Gather the information we need to calculate the zeropoint */

  char filter_band = fors_instrument_filterband_get_by_setting(setting);
  // The g_HIGH filter is reported as V, so we exclude it here
  if (!setting->filter_name || !strcmp(setting->filter_name, "g_HIGH")) {
      filter_band = '?';
  }
  cpl_error_reset();

  double dzp = 0.0;
  int nzp = 0;
  double zp = 0.0;

  /* First decide if we can even calculate the zeropoint */
  if (!setting->filter_name || filter_band == '?') {
      cpl_msg_warning(cpl_func, "Gaia zeropoint computation is not supported for non-standard filters");
  } else {
      cpl_msg_info(cpl_func, "Filter %s is %c", setting->filter_name, filter_band);

      /* Coefficients loaded from PHOT_TABLE */
      double color_term = 0.0, dcolor_term = 0.0;
      double ext_coeff = 0.0, dext_coeff = 0.0;
      double expected_zeropoint = 0.0, dexpected_zeropoint = 0.0;

      /* Load filter coefficients */
      cpl_error_reset();
      fors_phot_table_load(cpl_frameset_get_position(phot_table, 0), setting,
                           &color_term, &dcolor_term,
                           &ext_coeff, &dext_coeff,
                           &expected_zeropoint, &dexpected_zeropoint);
      if (cpl_error_get_code() != CPL_ERROR_NONE) {
          cpl_msg_error(cpl_func, "Error found in %s: %s",
                        cpl_error_get_where(), cpl_error_get_message());
          cpl_msg_error(cpl_func, "Could not load photometry table");
          return cpl_error_get_code();
      }

      cpl_msg_info(cpl_func, "Color term from %s: %f +- %f", PHOT_TABLE,
          color_term, dcolor_term);
      cpl_msg_info(cpl_func, "Expected zeropoint from %s: %f +- %f", PHOT_TABLE,
          expected_zeropoint, dexpected_zeropoint);

      cpl_msg_info(cpl_func, "Creating list of stellar sources");
      cpl_table *phot = fors_create_sources_table(stars);
      cpl_table_select_all(phot);
      nrows = (int)cpl_table_get_nrow(phot);
      cpl_msg_info(cpl_func, "Using %d sources to match against catalog", nrows);
      if (cpl_error_get_code() != CPL_ERROR_NONE) {
          cpl_msg_error(cpl_func, "Error found in %s: %s",
                        cpl_error_get_where(), cpl_error_get_message());
          cpl_msg_error(cpl_func, "Failed to create extracted sources table");
          return cpl_error_get_code();
      }

      /* Calculate the zeropoint */
      cpl_table *phot_stds_to_save = NULL;
      cpl_table *updated_stars_to_save = NULL;
      calculate_zp_from_gaia(filter_band, magcutE, magcutk, phot, &phot_stds_to_save, &updated_stars_to_save, result->header,
                             cacheloc, color_term, dcolor_term, zp, dzp, nzp);
      if (phot_stds_to_save != NULL) {
          cpl_table_erase_column(phot_stds_to_save, "Source");
    	  fors_dfs_save_table(frameset, phot_stds_to_save, PHOT_STD_PHOTOM_STACK,
    			  extract_qc, parlist, fors_img_stack_name,
				  cpl_frameset_get_position(frameset, 0));
      }
      cpl_table_delete(phot_stds_to_save);
      if (updated_stars_to_save != NULL) {
          fors_dfs_save_table(frameset, updated_stars_to_save, PHOT_STARS_PHOTOM_STACK,
                  extract_qc, parlist, fors_img_stack_name,
                  cpl_frameset_get_position(frameset, 0));
      }
      cpl_table_delete(updated_stars_to_save);
  }

  /* This routine calculates ABMAGLIM and ABMAGSAT and writes them to the header
   * It also writes the zeropoint headers
   */
  calculate_abmag(sources, zp_data, zp, dzp, nzp, filter_band, amag_vega,
    psf_fwhm / pixscale, bkg_rms, pixscale, plist);

  // Save the resampled product, along with the exposure map
  fors_img_stack_save(result_img, exp_img, result->header, OUTPUT_CAT, fors_img_stack_name,
                          "fors_img_stack.fits", setting->version,
                          parlist, *(plists.begin()), plist, elist, extract_qc,
                          frameset);

  /* Cleanup */

  /* Delete the parameters */
  hdrl_parameter_delete(res_method);
  hdrl_parameter_delete(res_outputgrid);

  fors_star_list_delete(&stars, fors_star_delete); \
  cpl_image_delete(background);
  cpl_table_delete(sources);
  cpl_propertylist_delete(extract_qc);
  hdrl_resample_result_delete(result);
  cpl_table_delete(restable);
  cpl_wcs_delete(wcs);
  cpl_propertylist_delete(plist);
  cpl_propertylist_delete(elist);
  cpl_frameset_delete(in_set);

  return cpl_error_get_code();
}


/**@}*/

