/* $Id: $
 *
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-10 19:16:03 $
 * $Revision: 1.50 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_img_wcs_impl.h>

#include <fors_extract.h>
#include <fors_tools.h>
#include <fors_setting.h>
#include <fors_data.h>
#include <fors_qc.h>
#include <fors_dfs.h>
#include <fors_utils.h>
#include "fiera_config.h"
#include "fors_overscan.h"
#include "fors_ccd_config.h"
#include "fors_detmodel.h"
#include "fors_trimm_illum.h"
#include <fors_subtract_bias.h>
#include "fors_bpm.h"

// CASU code is C only
extern "C" {
#include "casu_mods.h"
#include "casu_wcsutils.h"
#include "casu_utils.h"
};

#include <cpl.h>
#include <math.h>
#include <stdbool.h>
#include <string.h>
#include <set>

/**
 * @addtogroup fors_img_wcs
 */

/**@{*/

const char *const fors_img_wcs_name = "fors_img_wcs";
const char *const fors_img_wcs_description_short = "Source detection and astrometric calibration of an imaging scientific exposure";
const char *const fors_img_wcs_author = "ESO PPS Group";
const char *const fors_img_wcs_email = PACKAGE_BUGREPORT;
const char *const fors_img_wcs_description = 
"This recipe is used to perform source detection and extraction, by default\n"
"using the HDRL catalog routine.  It will match the found sources against a\n"
"catalog (by default Gaia DR3) to calculate the WCS solution.  If the input files\n"
"contain a DETECTOR_ILLUMINATED_REGION file, the recipe will filter out objects\n"
"in the source list that appear in the area of the detector that does not\n"
"receives light from the sky.  The output SCIENCE_REDUCED_IMG_WCS is largely the\n"
"same as the input SCIENCE_REDUCED_IMG, but' with the updated WCS solution. In\n"
"addition, the confidence map extension is updated by setting values to 0 where\n"
"they are less than the threshold parameter (--conf_thresh).\n"
"\n"
"Input files:\n"
"\n"
"  DO category:               Type:       Explanation:                         Number:\n"
"  SCIENCE_REDUCED_IMG         FITS image  Reduced science image                     1\n"
"  EXTINCTION_PER_NIGHT        FITS table  Extinction per night                      0+\n"
"  DETECTOR_ILLUMINATED_REGION FITS table  Table with detector illuminated regions   0+\n"
"\n"
"Output files:\n"
"\n"
"  DO category:               Data type:  Explanation:\n"
"  PHOT_BACKGROUND_SCI_IMG    FITS image  Reduced science image background\n"
"  SOURCES_SCI_IMG_WCS        FITS table  Unfiltered source list\n"
"  OBJECT_TABLE_SCI_IMG       FITS table  Extracted sources properties\n"
"  SCIENCE_REDUCED_IMG_WCS    FITS image  Reduced science (background subtracted) image with updated WCS\n";

static double
get_image_quality(const fors_star_list *sources, double *image_quality_err,
                  double *stellarity,
                  double *ellipticity,
                  double *ellipticity_rms);

/*
 * This function is based on the equivalent vimos_wcsfit
 */
static void
fors_wcsfit(cpl_propertylist *hdr, cpl_table *cat,
            char *catname, char *catpath, int cdssearch,
            char *cacheloc, float search_radius, int keepms,
            cpl_table **mstds);

/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */

void fors_img_wcs_define_parameters(cpl_parameterlist *parameters)
{
    cpl_parameter *p;
    char *context = cpl_sprintf("fors.%s", fors_img_wcs_name);
    char *full_name = NULL;
    const char *name;

    name = "magsyserr";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Systematic error in magnitude",
                                context,
                                0.01);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    fors_extract_define_parameters(parameters, context);

    name = "cacheloc";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_STRING,
                                "Location for the standard star cache",
                                context,
                                ".");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "cdssearch_astrom";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_enum(full_name,
                               CPL_TYPE_STRING,
                               "CDS astrometric catalogue",
                               context,
                               "gaiadr3",6,"none","2mass","usnob","ppmxl","wise","gaiadr3");
                               // possible to add "gaiasyntphot" to this list, but it doesn't
                               // really make sense to use it as a separate astrometric catalog.
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "savemstd";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_BOOL,
                                "Save matched std catalogues?",
                                context,
                                CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;

    name = "search_radius";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Search radius for matching found sources against catalog",
                                context,
                                20.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name); full_name = NULL;
    cpl_free((void *)context);

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(sci_frame); \
    cpl_frameset_delete(master_flat_frame); \
    cpl_frameset_delete(extinct_frame); \
    cpl_frameset_delete(illum_region_frames); \
    fors_image_delete(&sci); \
    fors_image_delete(&master_flat); \
    cpl_table_delete(extinct_table); \
    cpl_table_delete(phot); \
    cpl_table_delete(sources); \
    cpl_image_delete(background); \
    cpl_image_delete(confmap); \
    cpl_image_delete(updated_confmap); \
    cpl_image_delete(confbpm); \
    fors_extract_method_delete(&em); \
    fors_star_list_delete(&stars, fors_star_delete); \
    cpl_free((void *)context); \
    fors_setting_delete(&setting); \
    cpl_propertylist_delete(qc); \
    cpl_propertylist_delete(extract_qc); \
    cpl_propertylist_delete(ext_header); \
    cpl_propertylist_delete(wcs_header); \
    cpl_propertylist_delete(product_header); \
    cpl_propertylist_delete(sci_header); \
    cpl_propertylist_delete(img_header); \
} while (0)

/* %%% Removed from cleanup
    cpl_frameset_delete(phot_table); \
*/

/**
 * @brief    Do the processing
 *
 * @param    frames         set of frames
 * @param    parameters     parameters
 */
void fors_img_wcs(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    int status = 0;

    /* Inputs */
    cpl_frameset *sci_frame      = NULL;
    fors_image *sci              = NULL;
    cpl_image *confmap           = NULL;
    cpl_image *updated_confmap   = NULL;

    /* Calibration */
    cpl_frameset *master_flat_frame = NULL;
    fors_image *master_flat         = NULL; 
    cpl_frameset *extinct_frame = NULL;
    cpl_table *extinct_table = NULL;

    /* Static calibrations */
    cpl_frameset *illum_region_frames = NULL;
    cpl_table * illum_region = NULL;
    
    /* Products */
    cpl_propertylist *qc = cpl_propertylist_new();
    cpl_propertylist *ext_header = cpl_propertylist_new();
    cpl_propertylist *wcs_header = cpl_propertylist_new();
    cpl_propertylist *product_header = cpl_propertylist_new();
    cpl_propertylist *sci_header = NULL;
    cpl_propertylist *img_header = NULL;
    cpl_table *phot = NULL;
    fors_extract_sky_stats sky_stats;
    cpl_image *background = NULL;
    cpl_image *confbpm = NULL;
    cpl_table *sources = NULL;
    cpl_propertylist *extract_qc = NULL;

    /* Parameters */
    extract_method  *em = NULL;
    double           magsyserr;

    /* Other */
    double x, y;
    char *context   = cpl_sprintf("fors.%s", fors_img_wcs_name);
    fors_star_list *stars = NULL;
    fors_setting *setting = NULL;
    double avg_airmass = 0.0;
    double average_overscan_level = 0.0;
    char *name;

    /* QC summary parameters */
    std::set<int> night_id_set;

    /* Get parameters */
    em = fors_extract_method_new(parameters, context);
    assure( !cpl_error_get_code(), return, 
            "Could not get extraction parameters" );

    cpl_msg_indent_more();
    name = cpl_sprintf("%s.%s", context, "magsyserr");
    magsyserr = dfs_get_parameter_double_const(parameters,
                                               name);
    cpl_free((void *)name); name = NULL;

    name = cpl_sprintf("%s.%s", context, "cacheloc");
    const char* cacheloc = dfs_get_parameter_string_const(parameters, name);
    cpl_free((void *)name); name = NULL;

    name = cpl_sprintf("%s.%s", context, "cdssearch_astrom");
    const char* cdssearch_astrom_str = dfs_get_parameter_string_const(parameters, name);
    cpl_free((void *)name); name = NULL;
    int cdssearch_astrom = casu_get_cdschoice(cdssearch_astrom_str);
    cpl_msg_info(cpl_func, "cdssearch_astrom=%d", cdssearch_astrom);

    name = cpl_sprintf("%s.%s", context, "savemstd");
    bool savemstd = dfs_get_parameter_bool_const(parameters, name);
    cpl_free((void *)name); name = NULL;

    name = cpl_sprintf("%s.%s", context, "search_radius");
    const double search_radius = dfs_get_parameter_double_const(parameters, name);
    cpl_free((void *)name); name = NULL;

    cpl_msg_indent_less();
    assure( !cpl_error_get_code(), return, NULL );
    assure( magsyserr >= 0, return, 
            "Input systematic error (magsyserr=%f) cannot be negative",
            magsyserr);
    
    /* Check the cache location is writable */
    assure( access(cacheloc, R_OK + W_OK + X_OK) == 0, return,
            "Cache location %s inacessible", cacheloc);

    /* Find raw */
    sci_frame = fors_frameset_extract(frames, SCIENCE_REDUCED_IMG);
    assure( cpl_frameset_get_size(sci_frame) == 1, return, 
            "Exactly 1 %s required. %" CPL_SIZE_FORMAT" found", 
            SCIENCE_REDUCED_IMG, cpl_frameset_get_size(sci_frame) );

    /* Find calibration */
    extinct_frame = fors_frameset_extract(frames, EXTINCTION_PER_NIGHT);
    assure( cpl_frameset_get_size(extinct_frame) <= 1, return, 
            "Zero or one %s required. %" CPL_SIZE_FORMAT" found",
            EXTINCTION_PER_NIGHT, cpl_frameset_get_size(extinct_frame));

    /* Find static calibrations */
    illum_region_frames = fors_frameset_extract(frames, DETECTOR_ILLUMINATED_REGION);
    if(cpl_frameset_get_size(illum_region_frames) == 0)
    {
        cpl_msg_warning(cpl_func, "Input frames do not include "
                        DETECTOR_ILLUMINATED_REGION". Images won't be trimmed "
                        " to illuminated region.");
    }
    else if(cpl_frameset_get_size(illum_region_frames) != 1)
    {
        cpl_msg_error(cpl_func, "At most 1 frame of type "
                      DETECTOR_ILLUMINATED_REGION" is allowed");
        cleanup;
        return;
    } else {
        cpl_frame * illum_region_frame = cpl_frameset_get_position(
            illum_region_frames, 0);
        const char * fname = cpl_frame_get_filename(illum_region_frame);
        illum_region = cpl_table_load(fname, 1, 1);
    }

    /* Done finding frames */

    /* Get instrument setting */
    setting = fors_setting_new(cpl_frameset_get_position(sci_frame, 0));
    sci_header = cpl_propertylist_load(cpl_frame_get_filename(
        cpl_frameset_get_position(sci_frame, 0)), 0);
    if (sci_header == NULL) {
        cpl_msg_error(cpl_func, "Failed to load science header");
        cleanup;
        return;
    }
    img_header = cpl_propertylist_load(cpl_frame_get_filename(
        cpl_frameset_get_position(sci_frame, 0)), 1);
    if (img_header == NULL) {
        cpl_msg_error(cpl_func, "Failed to load image data header");
        cleanup;
        return;
    }
    /* Load frames */
    fors_dfs_copy_wcs(wcs_header, cpl_frameset_get_position(sci_frame, 0), 1);

    /* FIXME this is temporary loading code */
    cpl_image *sci_data = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG, CPL_TYPE_FLOAT, "IMAGE.DAT", 0);
    cpl_image *sci_var = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG, CPL_TYPE_FLOAT, "IMAGE.ERR", 0);
    cpl_image_power(sci_var, 2);
    sci = fors_image_new(sci_data, sci_var);
    // sci = fors_image_load(cpl_frameset_get_position(sci_frame, 0));
    assure( !cpl_error_get_code(), return, "Could not load science frames");

    /* Load header values */
    if (cpl_propertylist_has(sci_header, "ESO DRS AVG_OVERSCAN")) {
        average_overscan_level = cpl_propertylist_get_double(sci_header, "ESO DRS AVG_OVERSCAN");
    } else {
        cpl_msg_warning(cpl_func, "No header keyword ESO DRS AVG_OVERSCAN found. "
            "Assuming zero.");
    }
    /* Load the confidence map */
    confmap = dfs_load_image_ext(frames, SCIENCE_REDUCED_IMG, CPL_TYPE_FLOAT, "IMAGE.CNF", 0);
    updated_confmap = cpl_image_duplicate(confmap);

    /* Update the confidence map now that we have the threshold */
    update_confidence_map(updated_confmap, em);
    confbpm = create_bpm_from_confidence(updated_confmap);

    /* Create the preliminary WCS */
    cpl_wcs *tmpwcs = cpl_wcs_new_from_propertylist(img_header);

    /* Extract sources */
    stars = fors_extract(sci, setting, em, magsyserr, updated_confmap, tmpwcs,
                         &sky_stats, &background, &sources, &extract_qc);
    cpl_wcs_delete(tmpwcs);
    assure( !cpl_error_get_code(), return, "Could not extract objects");  

    // The catalogue name can be anything. It's just a label.
    char catname[] = "fors_wcsfit";

    // Variables relating to saving the matched stars catalogue
    cpl_table *matchstds = NULL;
    int keepms = (savemstd ? 1 : 0);

    // To do the wcs fit, we use the complete source list. While it's tempting
    // to use the pre-filtered list of stars, they have been selected from the
    // point of view of photometry and so we may miss some useful stars for
    // astrometry (e.g. those with a saturated core).
    //
    // We do reject any non-stellar sources as a first step though.
 
    cpl_table* star_sources = cpl_table_duplicate(sources);
    cpl_table_select_all(star_sources);
    int nrows = (int)cpl_table_get_nrow(star_sources);
    cpl_msg_info(cpl_func, "Considering %d initial sources for WCS fit", nrows);

    if (cpl_table_has_column(star_sources, "CLASS_STAR")) {
        cpl_table_and_selected_float(star_sources, "CLASS_STAR", CPL_GREATER_THAN, 0.0);
        if (cpl_error_get_code()) {
            cpl_msg_warning(cpl_func, "Error when selecting CLASS_STAR > 0.0: %s", cpl_error_get_message());
            cpl_error_reset();
        }
    } else if (cpl_table_has_column(star_sources, "Classification")) {
        cpl_table_and_selected_double(star_sources, "Classification", CPL_LESS_THAN, 0.0);
        if (cpl_error_get_code()) {
            cpl_msg_warning(cpl_func, "Error when selecting Classification < 0.0: %s", cpl_error_get_message());
            cpl_error_reset();
        }
    }

    cpl_table* tmp_sources = cpl_table_extract_selected(star_sources);
    cpl_table_delete(star_sources);
    nrows = (int)cpl_table_get_nrow(tmp_sources);
    cpl_msg_info(cpl_func, "Using %d probable stellar sources for WCS fit", nrows);

    /* Adapt the columns to look like what CASU expects */
    if (cpl_table_has_column(tmp_sources, "X_coordinate") && cpl_table_has_column(tmp_sources, "Y_coordinate") &&
        ((cpl_table_get_column_type(tmp_sources, "X_coordinate") != CPL_TYPE_FLOAT) ||
        (cpl_table_get_column_type(tmp_sources, "Y_coordinate") != CPL_TYPE_FLOAT))) {
      cpl_table_duplicate_column(tmp_sources, "X_IMAGE", tmp_sources, "X_coordinate");
      cpl_table_duplicate_column(tmp_sources, "Y_IMAGE", tmp_sources, "Y_coordinate");
      cpl_table_erase_column(tmp_sources, "X_coordinate");
      cpl_table_erase_column(tmp_sources, "Y_coordinate");
    }
    cpl_table_new_column(tmp_sources, "X_coordinate", CPL_TYPE_FLOAT);
    cpl_table_new_column(tmp_sources, "Y_coordinate", CPL_TYPE_FLOAT);
    for (int i = 0; i < nrows; i++) {
        x = cpl_table_get(tmp_sources, "X_IMAGE", i, &status);
        y = cpl_table_get(tmp_sources, "Y_IMAGE", i, &status);
        cpl_table_set_float(tmp_sources, "X_coordinate", i, (float)x);
        cpl_table_set_float(tmp_sources, "Y_coordinate", i, (float)y);
    }
    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when adapting columns: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    if (illum_region) {
      // Exclude objects that are outside the illuminated region
      mosca::rect_region crop_region;
      fors_trimm_non_illum_get_region(NULL, setting, illum_region, crop_region);

      cpl_table_select_all(tmp_sources);
      (void)cpl_table_and_selected_float(tmp_sources,"X_coordinate",
                 CPL_GREATER_THAN,(float)crop_region.llx());
      (void)cpl_table_and_selected_float(tmp_sources,"X_coordinate",
                 CPL_LESS_THAN,(float)crop_region.urx());
      (void)cpl_table_and_selected_float(tmp_sources,"Y_coordinate",
                 CPL_GREATER_THAN,(float)crop_region.lly());
      (void)cpl_table_and_selected_float(tmp_sources,"Y_coordinate",
                 CPL_LESS_THAN,(float)crop_region.ury());
      star_sources = cpl_table_extract_selected(tmp_sources);
      cpl_table_delete(tmp_sources);
      tmp_sources = star_sources;
      nrows = (int)cpl_table_get_nrow(tmp_sources);
      cpl_msg_info(cpl_func, "After trimming to the illuminated region there are %d sources", nrows);
    }

    if (cdssearch_astrom > 0) {
        // Do the WCS fit
        fors_wcsfit(img_header, tmp_sources, catname, NULL, cdssearch_astrom, (char*)cacheloc, (float)search_radius, keepms, &matchstds);
        cpl_table_delete(tmp_sources);
        assure( !cpl_error_get_code(), return, "Could not do WCS fit");  

        // Update the WCS header with the new data
        cpl_propertylist_erase_regexp(wcs_header, CPL_WCS_REGEXP, 0);
        cpl_propertylist_copy_property_regexp(wcs_header, img_header, CPL_WCS_REGEXP, 0);
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when doing WCS fits: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    /* QC */
    fors_qc_start_group(qc, fors_qc_dic_version, setting->instrument);
    
    /* Write the relevant settings to the QC header */
    fors_setting_write_qc(setting, qc);

    /* Copy in the extra QC keywords from the extraction routine */
    if (extract_qc != NULL) {
        cpl_propertylist_append(qc, extract_qc);
    }

    double sky_mag;
    double sky_mag_rms;
    if (sky_stats.mean > 0) {
        sky_mag = -2.5*log(sky_stats.mean /
                           (setting->pixel_scale*setting->pixel_scale))/log(10);
    }
    else {
        cpl_msg_warning(cpl_func, 
                        "Average sky background is negative (%f ADU), "
                        "cannot compute magnitude, setting QC.SKYAVG to 99999.",
                        sky_stats.mean);
        sky_mag = 99999.;
    }
    fors_qc_write_qc_double(qc,
                            sky_mag,
                            "QC.SKYAVG",
                            "mag/arcsec^2",
                            "Mean of sky background",
                            setting->instrument);
    
    if (sky_stats.median > 0) {
        sky_mag = -2.5*log(sky_stats.median /
                           (setting->pixel_scale*setting->pixel_scale))/log(10);
        /* deltaM = -2.5*log10(e)*deltaF/F */
        sky_mag_rms = fabs(-2.5 * (1.0/log(10))*sky_stats.rms/sky_stats.median);
    }
    else {
        cpl_msg_warning(cpl_func, 
                        "Median sky background is negative (%f ADU), "
                        "cannot compute magnitude: setting both QC.SKYMED "
                        "and QC.SKYRMS to 99999.",
                        sky_mag);
        sky_mag = 99999.;
        sky_mag_rms = 99999.;
    }
    fors_qc_write_qc_double(qc,
                            sky_mag,
                            "QC.SKYMED",
                            "mag/arcsec^2",
                            "Median of sky background",
                            setting->instrument);

    fors_qc_write_qc_double(qc,
                            sky_mag_rms,
                            "QC.SKYRMS",
                            "mag/arcsec^2",
                            "Standard deviation of sky background",
                            setting->instrument);


    fors_qc_write_qc_double(qc,
                            sky_stats.bkg_rms,
                            "QC.SKY_NOISE",
                            "ADU",
                            "Pixel noise at sky level",
                            setting->instrument);

    double image_quality_error;
    double stellarity;
    double ellipticity, ellipticity_rms;
    double image_quality = get_image_quality(stars, 
                                             &image_quality_error,
                                             &stellarity,
                                             &ellipticity,
                                             &ellipticity_rms);

    if (image_quality > 0.) {
        image_quality *= TWOSQRT2LN2 * setting->pixel_scale;
        image_quality_error *= TWOSQRT2LN2 * setting->pixel_scale;
    }

    fors_qc_write_qc_double(qc,
                            image_quality,
                            "QC.IMGQU",
                            "arcsec",
                            "Image quality of scientific exposure",
                            setting->instrument);

    fors_qc_write_qc_double(qc,
                            image_quality_error,
                            "QC.IMGQUERR",
                            "arcsec",
                            "Uncertainty of image quality",
                            setting->instrument);

    fors_qc_write_qc_double(qc,
                            stellarity,
                            "QC.STELLAVG",
                            NULL,
                            "Mean stellarity index",
                            setting->instrument);

    fors_qc_write_qc_double(qc,
                            ellipticity,
                            "QC.IMGQUELL",
                            NULL,
                            "Mean star ellipticity",
                            setting->instrument);

    fors_qc_write_qc_double(qc,
                            ellipticity_rms,
                            "QC.IMGQUELLERR",
                            NULL,
                            "Standard deviation of star ellipticities",
                            setting->instrument);

    // Get the ID of the night, from the point of view of the photometry calculations.
    // Despite the fact that the fors_photometry product puts this night_id into
    // a column called MJD_NIGHT, it's actually a JD.
    int this_night = fors_photometry_get_night_id(sci_header);
    cpl_msg_info(cpl_func, "this_night = %d", this_night);
    if (this_night == 0) {
        cpl_error_reset();
    } else {
        fors_qc_write_qc_double(qc,
                             this_night - 2400000.5 + 1.0,
                             "QC.MJDOBS",
                             NULL,
                             "MJD of night",
                             setting->instrument);

        if (cpl_frameset_get_size(extinct_frame) > 0) {
            extinct_table = cpl_table_load(
                                      cpl_frame_get_filename(
                                        cpl_frameset_get_position(extinct_frame, 0)),
                                      1, 0);

            bool found_ext = false;
            int num_ext = 0;
            for(cpl_size i = 0; i < cpl_table_get_nrow(extinct_table); i++) {
                int night_id = cpl_table_get_int(extinct_table, "MJD_NIGHT", i, NULL);
                cpl_msg_info(cpl_func, "night_id = %d", night_id);
                if (night_id == this_night) {
                    // Count the number of extinction frames for this night
                    ++num_ext;
                    if (!found_ext) {
                        // They will all have the same value for EXT and DEXT so we
                        // only need to record it once
                        found_ext = true;
                        double ext = cpl_table_get_double(extinct_table, "EXT", i, NULL);
                        fors_qc_write_qc_double(qc,
                                             ext,
                                             "QC.EXTINCTION",
                                             NULL,
                                             "Extinction for this night",
                                             setting->instrument);
                        double dext = cpl_table_get_double(extinct_table, "DEXT", i, NULL);
                        fors_qc_write_qc_double(qc,
                                             dext,
                                             "QC.EXTINCTION_ERR",
                                             NULL,
                                             "Extinction error for this night",
                                             setting->instrument);
                    }
                }
                night_id_set.insert(night_id);
            }

            fors_qc_write_qc_int(qc,
                                 num_ext,
                                 "QC.NUM_EXT",
                                 NULL,
                                 "Number of extinction measurements for this night",
                                 setting->instrument);

            fors_qc_write_qc_int(qc,
                                 night_id_set.size(),
                                 "QC.NUM_NIGHTS",
                                 NULL,
                                 "Number of nights in extinction table",
                                 setting->instrument);

            fors_qc_write_qc_int(qc,
                                 cpl_table_get_nrow(extinct_table),
                                 "QC.NUM_FIELDS",
                                 NULL,
                                 "Number of fields in extinction table",
                                 setting->instrument);
        }
    }

    fors_qc_end_group();

    /* Subtract background from sci */
    cpl_image * dummy = cpl_image_new(cpl_image_get_size_x(background), cpl_image_get_size_y(background), CPL_TYPE_FLOAT);
    fors_image * bckg = fors_image_new(background, dummy);
    fors_image_subtract(sci, bckg);

    /* Save PHOT_BACKGROUND_SCI_IMG */

    avg_airmass = fors_get_airmass(sci_header);
    cpl_propertylist_update_double(qc, "AIRMASS", avg_airmass);
    cpl_propertylist_update_double(product_header, "AIRMASS", avg_airmass);

    /* Make sure the average overscan value propagates through */
    cpl_propertylist_update_double(qc, "ESO DRS AVG_OVERSCAN", average_overscan_level);
    cpl_propertylist_update_double(product_header, "ESO DRS AVG_OVERSCAN", average_overscan_level);

    fors_dfs_add_exptime(qc, cpl_frameset_get_position(sci_frame, 0), 0.);
    fors_dfs_add_exptime(product_header, cpl_frameset_get_position(sci_frame, 0), 0.);

    /* Add unique name for two detectors and exposures */
    const char *det_exp = NULL;
    det_exp = fors_get_det_exp(sci_header);
    cpl_propertylist_append_string(qc, "ESO PRO DET_EXP", det_exp);
    cpl_propertylist_append_string(product_header, "ESO PRO DET_EXP", det_exp);
    /* Add TITLE keyword */
    const char *title = NULL;
    title = fors_get_title(sci_header);
    cpl_propertylist_append_string(qc, "TITLE", title);
    cpl_propertylist_append_string(product_header, "TITLE", title);

    /* The WCS headers go into each extension rather than into the primary header */
    cpl_propertylist_copy_property_regexp(ext_header, wcs_header, ".*", false);

    /* The PHOT_BACKGROUND_SCI_IMG also contains the WCS headers */
    cpl_propertylist_copy_property_regexp(product_header, wcs_header, ".*", false);

    const cpl_frame * inherit_frame = cpl_frameset_get_position_const(sci_frame, 0);

    fors_dfs_save_image_err_2(frames, sci, confbpm, updated_confmap, SCIENCE_REDUCED_IMG_WCS, qc,
        ext_header, parameters, fors_img_wcs_name,
        inherit_frame, NULL);
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            SCIENCE_REDUCED_IMG_WCS);

    dfs_save_image(frames, background, PHOT_BACKGROUND_SCI_IMG,
                   product_header, parameters, fors_img_wcs_name, 
                   setting->version);
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            PHOT_BACKGROUND_SCI_IMG);

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when writing QC parameters: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    if (illum_region) {
        cpl_table_delete(illum_region);
    }

    fors_image_delete(&sci);
    cpl_image_delete(confmap); confmap = NULL;
    cpl_image_delete(updated_confmap); updated_confmap = NULL;
    cpl_image_delete(confbpm); confbpm = NULL;
    cpl_image_delete(background); background = NULL;
    
    /* Create, save FITS product */
    cpl_msg_info(cpl_func, "Creating list of stellar sources");
    phot = fors_create_sources_table(stars);
    cpl_table_select_all(phot);
    assure( !cpl_error_get_code(), return,
            "Failed to create extracted sources table");

    cpl_table_erase_column(phot, "INSTR_CMAG");
    cpl_table_erase_column(phot, "DINSTR_CMAG");
    cpl_table_erase_column(phot, "OBJECT");
    cpl_table_erase_column(phot, "MAG");
    cpl_table_erase_column(phot, "DMAG");
    cpl_table_erase_column(phot, "CAT_MAG");
    cpl_table_erase_column(phot, "DCAT_MAG");
    cpl_table_erase_column(phot, "COLOR");
    /* new columns since 4.4.10 */
    if (cpl_table_has_column(phot, "DCOLOR"))
        cpl_table_erase_column(phot, "DCOLOR");
    if (cpl_table_has_column(phot, "COV_CATM_COL"))
        cpl_table_erase_column(phot, "COV_CATM_COL");
    cpl_table_erase_column(phot, "USE_CAT");
    cpl_table_erase_column(phot, "SHIFT_X");
    cpl_table_erase_column(phot, "SHIFT_Y");
    cpl_table_erase_column(phot, "ZEROPOINT");
    cpl_table_erase_column(phot, "DZEROPOINT");
    cpl_table_erase_column(phot, "WEIGHT");

    /* Update the RA and DEC of the objects in the stars catalogue */

    cpl_wcs *wcs = cpl_wcs_new_from_propertylist(wcs_header);

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when creating source list: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    nrows = (int)cpl_table_get_nrow(phot);
    if (wcs == NULL) {
        cpl_msg_error(cpl_func,"Failed to fill RA and Dec in catalogue");
        return;
    }
    double r, d;
    for (int i = 0; i < nrows; i++) {
        x = cpl_table_get(phot, "X", i, &status);
        y = cpl_table_get(phot, "Y", i, &status);
        casu_xytoradec(wcs, x, y, &r, &d);
        cpl_table_set_double(phot, "RA", i, r);
        cpl_table_set_double(phot, "DEC", i, d);
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when updating source list: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    /* And now update RA and DEC in the full source catalogue */
    /* FIXME this could be done more neatly */
    nrows = (int)cpl_table_get_nrow(sources);
    for (int i = 0; i < nrows; i++) {
        int xstatus, ystatus;
        double stmpx = cpl_table_get(sources, "X_IMAGE", i, &xstatus); 
        double stmpy = cpl_table_get(sources, "Y_IMAGE", i, &ystatus);
        if (!xstatus && !ystatus) {
            // SExtractor
            // cpl_msg_debug(cpl_func, "Updating ALPHA_J2000 / DELTA_J2000 for source %d", i);
            casu_xytoradec(wcs, stmpx, stmpy, &r, &d);
            cpl_table_set_double(sources, "ALPHA_J2000", i, r);
            cpl_table_set_double(sources, "DELTA_J2000", i, d);
        }
        double htmpx = cpl_table_get(sources, "X_coordinate", i, &xstatus);
        double htmpy = cpl_table_get(sources, "Y_coordinate", i, &ystatus);
        if (!xstatus && !ystatus) {
            // HDRL
            // cpl_msg_debug(cpl_func, "Updating RA / DEC for source %d", i);
            casu_xytoradec(wcs, htmpx, htmpy, &r, &d);
            cpl_table_set_double(sources, "RA", i, r);
            cpl_table_set_double(sources, "DEC", i, d);
        }
    }

    cpl_error_reset();
    cpl_wcs_delete(wcs);

    // These tables are often unable to be opened by CDS Aladin.
    // Stripping out all the headers from the primary extension
    // allows them to be opened. Let the user do that if they
    // need to. Stripping headers out here is problematic.

    cpl_propertylist_append_string(extract_qc, "ESO PRO DET_EXP", det_exp);
    cpl_propertylist_append_string(extract_qc, "TITLE", title);


    fors_dfs_save_table(frames, sources, SOURCES_SCI_IMG_WCS,
                        extract_qc, parameters, fors_img_wcs_name,
                        cpl_frameset_get_position(sci_frame, 0));

    assure( !cpl_error_get_code(), return, "Saving %s failed",
            SOURCES_SCI_IMG_WCS);

    fors_dfs_save_table(frames, phot, PHOTOMETRY_TABLE,
                        extract_qc, parameters, fors_img_wcs_name, 
                        cpl_frameset_get_position(sci_frame, 0));
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            PHOTOMETRY_TABLE);

    /* Save the list of matched stars */
    if (keepms && (matchstds != NULL)) {
        fors_dfs_save_table(frames, matchstds, MATCHED_TABLE,
               NULL, parameters, fors_img_wcs_name, 
               cpl_frameset_get_position(sci_frame, 0));
        assure( !cpl_error_get_code(), return, "Saving %s failed",
                MATCHED_TABLE);
    }

    cleanup;
    return;
}

#undef cleanup
#define cleanup
/**
 * @brief    Determine if source is a star
 * @param    s             sources
 * @param    data          not used
 * @return   true iff the source is more star-like
 */
static bool
is_star(const fors_star *s, void *data)
{
    (void)data;
    assure( s != NULL, return false, NULL );

    return s->stellarity_index >= 0.7;
}


#undef cleanup
#define cleanup \
do { \
    fors_star_list_delete(&stars, fors_star_delete); \
} while(0)



/**
 * @brief    Compute image quality
 * @param    sources             extracted sources
 * @param    image_quality_err   (output) empirical scatter
 *                               (based on median absolute deviation)
 * @param    stellarity          (output) average stellarity of
 *                               sources above stellarity cutoff
 * @param    ellipticity         (output) average ellipticity of
 *                               sources above stellarity cutoff
 * @param    ellipticity_rms    (output) RMS ellipticity of
 *                               sources above stellarity cutoff
 * @param   median source extension (in the one sigma sense)
 */
static double
get_image_quality(const fors_star_list *sources, double *image_quality_err,
                  double *stellarity,
                  double *ellipticity,
                  double *ellipticity_rms)
{
    fors_star_list *stars = fors_star_list_extract(sources,
                                                   fors_star_duplicate,
                                                   is_star, NULL);

    double fwhm;
    if (fors_star_list_size(stars) >= 1) {
        *image_quality_err = fors_star_list_mad(stars, fors_star_extension, NULL) 
            * STDEV_PR_MAD;
        
        fwhm = fors_star_list_median(stars, fors_star_extension , NULL);
        
        *stellarity      = fors_star_list_mean(stars, fors_star_stellarity, NULL);
        *ellipticity     = fors_star_list_mean(stars, fors_star_ellipticity, NULL);
        *ellipticity_rms = fors_star_list_mad(stars, fors_star_ellipticity, NULL)
            * STDEV_PR_MAD;
    }
    else {
        cpl_msg_warning(cpl_func, "No stars found! Cannot compute image quality, "
                        "setting QC parameters to -1");

        /* -1 is not a valid value for any of these */
        *image_quality_err = -1;
        fwhm = -1;
        *stellarity = -1;
        *ellipticity = -1;
        *ellipticity_rms = -1;
    }

    cleanup;
    return fwhm;
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        fors_wcsfit
    \par Purpose:
        Do a WCS fit to an individual image
    \par Description:
        For each image in a list, create a source catalogue if one isn't 
        given in the argument list. Fit a WCS to the source positions.
    \par Language:
        C
    \param in
        The property list of the input image
    \param incat
        The input source catalogue
    \param catname
        The name of the standard catalogue
    \param catpath
        The full path to the standard index file
    \param cdssearch
        The CDS catalogue to be searched for standards. 0 if using local
        catalogues.
    \param cacheloc
        The location of the standard star cache
    \param search_radius
        The search radius for matching stars (in pixels).
    \param keepms
        If set, then the matched standards files will be kept
    \param mstds
        The matched standards catalogue
    \returns
        Nothing
    \author
        Jim Lewis, CASU
        modifications for FORS by Jon Nielsen, ANU
 */
/*---------------------------------------------------------------------------*/
    
void fors_wcsfit(cpl_propertylist *hdr, cpl_table *incat,
                  char *catname, char *catpath, int cdssearch,
                  char *cacheloc, float search_radius, int keepms,
                  cpl_table **mstds) {
    int status,nstd,ncat,nmatch,slevel,n,i;
    float *x,*y;
    double r,d,*ra_d,*dec_d;
    cpl_table *stdscat,*tmp,*tmp2,*matchstds;
    cpl_propertylist *p;
    cpl_wcs *wcs;
    const char *fctid = "fors_wcsfit";
    cpl_table *cat;
    const double PM_LIMIT = 10; // mas/year limit on proper motion

    status = CASU_OK;

    /* Get some standard stars */

    (void)casu_getstds(hdr,1,catpath,catname,cdssearch,
                       cacheloc,&stdscat,NULL,&status);
    if (status != CASU_OK) {
        freetable(stdscat);
        cpl_msg_error(fctid, "Failed to find any standards");
        return;
    }

    nstd = (int)cpl_table_get_nrow(stdscat);
    cpl_msg_info(cpl_func, "Number of catalog stars: %d", nstd);

    /* Check for proper motion columns, and reject stars with values that are too high */
    if (cpl_table_has_column(stdscat, "pmRA")) {
        cpl_table_select_all(stdscat);
        (void)cpl_table_and_selected_double(stdscat, "pmRA", CPL_LESS_THAN, PM_LIMIT);
        (void)cpl_table_and_selected_double(stdscat, "pmRA", CPL_GREATER_THAN, -PM_LIMIT);
        cpl_table *tmp = cpl_table_extract_selected(stdscat);
        cpl_table_delete(stdscat);
        stdscat = tmp;
    }
    if (cpl_table_has_column(stdscat, "pmDec")) {
        cpl_table_select_all(stdscat);
        (void)cpl_table_and_selected_double(stdscat, "pmDec", CPL_LESS_THAN, PM_LIMIT);
        (void)cpl_table_and_selected_double(stdscat, "pmDec", CPL_GREATER_THAN, -PM_LIMIT);
        cpl_table *tmp = cpl_table_extract_selected(stdscat);
        cpl_table_delete(stdscat);
        stdscat = tmp;
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when rejecting high proper motion stars: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    nstd = (int)cpl_table_get_nrow(stdscat);
    cpl_msg_info(cpl_func, "Number of catalog stars: %d, after rejecting stars with either proper motion > %f mas/yr", nstd, PM_LIMIT);

#if 0
    /* Apply proper motions */

    cpl_type coltype;
    double mjd = cpl_propertylist_get_double(plist, "MJD-OBS");
    // Number of years past 2000.0 to the observation date
    double years_past_epoch = (mjd - 51544.5) / 365.25;
    double sumpm = 0;
    int do_pm = 1;
    int st;
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_warning(cpl_func, "Could not retrieve MJD-OBS from header");
        do_pm = 0;
    }
    if (do_pm && cpl_table_has_column(*stds, "pmRA") && cpl_table_has_column(*stds, "pmDE")) {
        cpl_msg_info(cpl_func, "Applying proper motions from 2000.0 to %f", 2000 + years_past_epoch);
        int err = 0, success=0;
        for (i = 0; i < n; i++) {
            r = cpl_table_get(*stds, "RA", i, &st);
            if (st != 0) {
                err++;
                continue;
            }
            pmra = cpl_table_get(*stds, "pmRA", i, &st);
            if ((st != 0) || isnan(pmra)) {
                err++;
                continue;
            }
            sumpm += pmra;
            d = cpl_table_get(*stds, "Dec", i, &st);
            if (st != 0) {
                err++;
                continue;
            }
            pmde = cpl_table_get(*stds, "pmDE", i, &st);
            if ((st != 0) || isnan(pmde)) {
                err++;
                continue;
            }
            sumpm += pmde;
            cpl_msg_info(cpl_func, "RA=%f Dec=%f pmRA=%f pmDE=%f", r, d, pmra, pmde);
            // Convert from mas/yr to degrees at epoch
            r += years_past_epoch * pmra / (3600 * 1000);
            d += years_past_epoch * pmde / (3600 * 1000);
            cpl_msg_info(cpl_func, "RA:=%f Dec:=%f", r, d);
            coltype = cpl_table_get_column_type(*stds, "RA");
            if (coltype == CPL_TYPE_FLOAT) {
              cpl_table_set_float(*stds, "RA", i, (float)r);
            } else if (coltype == CPL_TYPE_DOUBLE) {
              cpl_table_set_double(*stds, "RA", i, r);
            }
            coltype = cpl_table_get_column_type(*stds,"Dec");
            if (coltype == CPL_TYPE_FLOAT) {
              cpl_table_set_float(*stds, "Dec", i, (float)d);
            } else if (coltype == CPL_TYPE_DOUBLE) {
              cpl_table_set_double(*stds, "Dec", i, d);
            }
            success++;
        }
        cpl_msg_info(cpl_func, "Proper motions applied to %d, failed for %d", success, err);
        cpl_msg_info(cpl_func, "Mean pm applied was %g mas/year, %g degrees", sumpm / (success * 2), years_past_epoch * sumpm / (3600 * 1000 * success * 2));
    } else {
        cpl_msg_info(cpl_func, "Not applying proper motion. Either observation is missing MJD-OBS or catalog is missing pmRA and/or pmDE columns.");
    }
#endif

    // Select all stars from the input catalog
    cat = cpl_table_duplicate(incat);
    // FIXME - should restrict by some stellarity measure here

    /* If there are too many objects in the catalogue then first restrict
       ourselves by ellipticity. Cut so that there are similar numbers of
       objects in the standards and the object catalogues by retaining the
       brighter objects */

    tmp = NULL;
    ncat = (int)cpl_table_get_nrow(cat);

    if ((ncat > 500) && (ncat > 2 * nstd)) {
        // Restrict by ellipticity if possible
        // This only works for the HDRL-generated table
        tmp = cpl_table_duplicate(cat);
        cpl_table_unselect_all(tmp);
        if (cpl_table_has_column(tmp, "Ellipticity")) {
          (void)cpl_table_or_selected_double(tmp,"Ellipticity",CPL_LESS_THAN,0.5);
        } else {
          cpl_table_select_all(tmp);
        }
        cpl_table_select_all(tmp);
        tmp2 = cpl_table_extract_selected(tmp);
        ncat = (int)cpl_table_get_nrow(tmp2);
        freetable(tmp);
        p = cpl_propertylist_new();
        // Sort by flux
        if (cpl_table_has_column(tmp2, "Aper_flux_3")) {
            cpl_propertylist_append_bool(p, "Aper_flux_3", TRUE);
        } else if (cpl_table_has_column(tmp2, "FLUX_APER")) {
            cpl_propertylist_append_bool(p, "FLUX_APER", TRUE);
        } else {
            cpl_msg_error(fctid, "Failed to find flux column Aper_flux_3 or FLUX_APER in source table");
            return;
        }
        cpl_table_sort(tmp2,(const cpl_propertylist *)p);
        cpl_propertylist_delete(p);
        slevel = std::min(ncat,std::max(1,std::min(5000,std::max(500,2*nstd))));
        tmp = cpl_table_extract(tmp2,1,(cpl_size)slevel);
        freetable(tmp2);
        ncat = (int)cpl_table_get_nrow(tmp);
        freetable(cat);
        cat = tmp;
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when restricting source selection: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    cpl_table_save(cat, NULL, NULL, "restricted_sources.fits", CPL_IO_CREATE);

    /* Now match this against the catalogue */

    ncat = (int)cpl_table_get_nrow(cat);
    cpl_msg_info(fctid, "Using %d sources to match against catalogue", ncat);

    casu_matchstds(cat, stdscat, search_radius, &matchstds, &status);

    nmatch = (int)cpl_table_get_nrow(matchstds);

    cpl_msg_info(fctid, "Matched %d sources", nmatch);

    freetable(cat);
    freetable(stdscat);
    if (status != CASU_OK) {
        freetable(matchstds);
        cpl_msg_error(fctid,"Failed to match standards to catalogue");
        return;
    }

    /* Fit the plate solution */

    (void)casu_platesol(hdr, NULL, matchstds, 6, 1, &status);

    if (keepms) {
        // Prepare new columns for the comparison between
        // found sources and catalogue sources.
        cpl_table_duplicate_column(matchstds, "RA_calc", matchstds, "RA");
        cpl_table_duplicate_column(matchstds, "diffRA", matchstds, "RA");
        cpl_table_duplicate_column(matchstds, "Dec_calc", matchstds, "Dec");
        cpl_table_duplicate_column(matchstds, "diffDec", matchstds, "Dec");
        *mstds = matchstds;
    } else {
        freetable(matchstds);
    }
    if (status != CASU_OK) {
        cpl_msg_error(fctid,"Failed to fit WCS");
        return;
    }

    if (cpl_error_get_code()) {
        cpl_msg_warning(cpl_func, "Error when fitting WCS: %s", cpl_error_get_message());
        cpl_error_reset();
    }

    /* If keeping the matchstds catalogue, then add some stuff to it */

    if (keepms) {

        wcs = cpl_wcs_new_from_propertylist(hdr);
        if (wcs == NULL) {
            cpl_msg_error(fctid,"Failed to fill RA and Dec in catalogue");
            return;
        }

        /* Now compute the equatorial coordinates and compare with the 
           standard values */
        n = (int)cpl_table_get_nrow(matchstds);
        cpl_msg_info(fctid, "Updating %d objects in matched catalog", n);
        x = cpl_table_get_data_float(matchstds,"X_coordinate");
        y = cpl_table_get_data_float(matchstds,"Y_coordinate");
        ra_d = cpl_table_get_data_double(matchstds,"RA");
        dec_d = cpl_table_get_data_double(matchstds,"Dec");
        for (i = 0; i < n; i++) {
            casu_xytoradec(wcs,(double)x[i],(double)y[i],&r,&d);
            cpl_table_set_double(matchstds, "RA_calc", i, r);
            cpl_table_set_double(matchstds, "Dec_calc", i, d);
            cpl_table_set_double(matchstds, "diffRA", i, r - ra_d[i]);
            cpl_table_set_double(matchstds, "diffDec", i, d - dec_d[i]);
        }

        cpl_wcs_delete(wcs);
    }
}
