/*
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * fors_response_utils.cc
 *
 *  Created on: 2014 4 2
 *      Author: cgarcia
 */

#include <stdexcept>
#include <vector>
#include <sstream>
#include <cstring>
#include <fors_response_utils.h>
#include "extinction.h"
#include "spec_std_star.h"
#include "response.h"
#include "xsh_star_index.h"

#define MAX_COLNAME      (80)


void fors_extract_brightest_spectrum
(cpl_image *spectra, cpl_image *spectra_err, cpl_table *objects, cpl_size& brightest_idx,
 cpl_size& obj_slit, cpl_image **spectrum, cpl_image **spectrum_err)
{
    *spectrum       = NULL; // Extracted standard star spectrum
    /*
     * Find brightest spectrum and duplicate it.
     */
    obj_slit = -1;
    {
        cpl_size        maxpos_x, maxpos_y;
        cpl_image *brights = cpl_image_collapse_create(spectra, 1);

        cpl_image_get_maxpos(brights, &maxpos_x, &maxpos_y);
        brightest_idx = maxpos_y;

        if (objects != NULL)
        {
            /* Identify the slit of this object which will be stored in obj_slit*/
            //TODO: Move it away from here. It is also repeated in the science photometric correction
            size_t nslits = cpl_table_get_nrow(objects);
            int   maxobjects = 1;
            char  name[MAX_COLNAME];
            snprintf(name, MAX_COLNAME, "object_%d", maxobjects);
            while (cpl_table_has_column(objects, name)) {
                maxobjects++;
                snprintf(name, MAX_COLNAME, "object_%d", maxobjects);
            }
            maxobjects--;
            for (cpl_size i_slit = 0; i_slit < (cpl_size)nslits; i_slit++) {
                for (int i_obj = 1; i_obj <= maxobjects; i_obj++) {
                    snprintf(name, MAX_COLNAME, "row_%d", i_obj);
                    if (cpl_table_is_valid(objects, name, i_slit))
                    {
                        int null;
                        int idx_obj = 
                            cpl_table_get_int(objects, name, i_slit, &null);;
                        if (maxpos_y == idx_obj+1)
                            obj_slit = i_slit;
                    }
                }
            }
        }
        
        cpl_image_delete(brights);
        cpl_size nx = cpl_image_get_size_x(spectra);
        *spectrum = cpl_image_extract(spectra, 1, maxpos_y, nx, maxpos_y);
        *spectrum_err = cpl_image_extract(spectra_err, 1, maxpos_y, nx, maxpos_y);
    }
}

void fors_extract_standard
(cpl_image *spectra, cpl_image *spectra_err,
 cpl_image * mapped_flat_sed,
 cpl_propertylist *flat_sed_header, cpl_table *objects,
 double& flat_sed_norm_factor,
 const fors::detected_slits& det_slits,
 cpl_image** spectrum, cpl_image** spectrum_sedcorr,
 cpl_image** spectrum_err, cpl_image** spectrum_sedcorr_err,
 cpl_size& brightest_idx)
{
    cpl_image *ffsed = NULL;

    if (spectra == NULL)
        throw std::invalid_argument("Empty spectra");

    int nx = cpl_image_get_size_x(spectra);

    /*
     * Find brightest spectrum and duplicate it.
     */
    brightest_idx = -1;
    cpl_size obj_slit = -1;
    fors_extract_brightest_spectrum(spectra, spectra_err, objects, brightest_idx, obj_slit, spectrum, spectrum_err);

    /* Divide the target spectrum by the corresponding slit profile */
    /* Also divide the error by the same profile (which is assumed to have no error of its own) */
    if(mapped_flat_sed != NULL)
    {
        ffsed = cpl_image_extract(mapped_flat_sed, 1, obj_slit+1, nx, obj_slit+1);
        std::ostringstream norm_key;
        norm_key<< "ESO QC FLAT SED_"<<det_slits[obj_slit].slit_id()<<" NORM";
        flat_sed_norm_factor = 
          cpl_propertylist_get_double(flat_sed_header, norm_key.str().c_str());
        if(cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
        {
            cpl_image_delete(*spectrum); *spectrum = NULL;
            cpl_image_delete(*spectrum_err); *spectrum_err = NULL;
            cpl_image_delete(ffsed); ffsed = NULL;
            std::string error_msg("Could not find keyword in flat sed: ");
            throw std::runtime_error(error_msg+norm_key.str());
        }

        *spectrum_sedcorr = cpl_image_duplicate(*spectrum);
        *spectrum_sedcorr_err = cpl_image_duplicate(*spectrum_err);
        for (int i_pix = 0; i_pix < nx; i_pix++)
        {
            int   null;
            const double profile_val  = cpl_image_get(ffsed, i_pix+1, 1, &null);
            const double val = cpl_image_get(*spectrum_sedcorr, i_pix+1, 1, &null);
            const double err = cpl_image_get(*spectrum_sedcorr_err, i_pix+1, 1, &null);
            if(profile_val != 0) {
                cpl_image_set(*spectrum_sedcorr, i_pix+1, 1, val / profile_val);
                // Assume profile has no error
                cpl_image_set(*spectrum_sedcorr_err, i_pix+1, 1, err / profile_val);
            } else {
                cpl_image_set(*spectrum_sedcorr, i_pix+1, 1, 0.0);
                cpl_image_set(*spectrum_sedcorr_err, i_pix+1, 1, 0.0);
            }
        }

        cpl_image_delete(ffsed);
    }
}

std::pair<mosca::spectrum, mosca::spectrum> fors_scale_standard
(cpl_image *spectrum, cpl_image *spectrum_sedcorr,
 double startwave, double dispersion, double gain,
 double exptime)
{
    if (spectrum == NULL)
        throw std::invalid_argument("Empty spectrum");

    if (gain < 0.1) 
        throw std::invalid_argument("Invalid gain factor (<0.1)");

    if (dispersion < 0.001) 
        throw std::invalid_argument("Invalid dispersion (<0.001)");

    if (exptime < 0.001) 
        throw std::invalid_argument("Invalid exposure time (<0.001)");

    /*
     * Convert standard star spectrum in electrons per second per Angstrom.
     */
    cpl_image* tmp = cpl_image_duplicate(spectrum);
    cpl_image_multiply_scalar(tmp, gain / exptime / dispersion);

    /* Cast to a mosca::spectrum */
    mosca::spectrum std_obs_spectrum = mosca::spectrum(tmp, startwave, dispersion);
    cpl_image_delete(tmp);

    /* Do the same for the SED correction */
    if(spectrum_sedcorr != NULL)
    {
        /* Convert standard star spectrum in electrons per second per Angstrom. */
        tmp = cpl_image_duplicate(spectrum_sedcorr);
        cpl_image_multiply_scalar(tmp, gain / exptime / dispersion);

        /* Cast to a mosca::spectrum */
        mosca::spectrum std_obs_spectrum_sedcorr = mosca::spectrum(tmp, startwave, dispersion);
        cpl_image_delete(tmp);

        /* Have a separate return here to handle the fact that mosca::spectrum does
         * not have a copy assignment operator which means we can't easily use those
         * objects as pass-by-reference parameters */
        return std::make_pair(std_obs_spectrum, std_obs_spectrum_sedcorr);
    }

    return std::make_pair(std_obs_spectrum, mosca::spectrum());
}

cpl_table * fors_compute_response
(mosca::spectrum std_obs_spectrum,
 mosca::spectrum std_obs_spectrum_sedcorr,
cpl_table *ext_table, double airmass, cpl_table *flux_table,
 const std::vector<double>& ignored_waves,
 const std::vector<std::pair<double, double> >& ignored_wave_ranges,
 int nknots, int degree, cpl_table *& response_obs_binning)
{
    cpl_table *response_table;

    if (flux_table == NULL) 
        throw std::invalid_argument("Empty flux_table");

    if (ext_table) {
        if (!cpl_table_has_column(ext_table, "WAVE")) 
            throw std::invalid_argument("Column WAVE in atmospheric extinction table");

        if (!cpl_table_has_column(ext_table, "EXTINCTION")) 
            throw std::invalid_argument("Column EXTINCTION in atmospheric extinction table");
    }

    if (!cpl_table_has_column(flux_table, "WAVE")) 
        throw std::invalid_argument("Column WAVE in standard star flux table");

    if (!cpl_table_has_column(flux_table, "FLUX")) 
        throw std::invalid_argument("Column FLUX in standard star flux table");

    if (nknots < 2 && degree < 0 ) 
        throw std::invalid_argument("Number of knots in spline fitting the " 
                                    "instrument response must be at least 2");

    if (degree < 1 && nknots < 0)
        throw std::invalid_argument("Order of the polynomial fitting the " 
                                    "instrument response must be at least 1");

    /* Prepare response computation */
    mosca::spec_std_star std_star(flux_table);
    mosca::response resp;
    mosca::response resp_sedcorr;

    /* Correct from atmospheric extinction */
    mosca::extinction atm_extinction = (ext_table ? mosca::extinction(ext_table) : mosca::extinction());
    mosca::spectrum std_extcorrect = atm_extinction.correct_spectrum(std_obs_spectrum, airmass);

    /* Compute the normal response */
    resp.compute_response(std_extcorrect, std_star);

    /* Do the same for the SED correction */
    bool use_sedcorr = !std_obs_spectrum_sedcorr.flux().empty();

    if(use_sedcorr)
    {
        /* Correct from atmospheric extinction */
        mosca::spectrum std_extcorrect_sedcorr = 
                atm_extinction.correct_spectrum(std_obs_spectrum_sedcorr, airmass);

        //Compute the response with the SED correction
        resp_sedcorr.compute_response(std_extcorrect_sedcorr, std_star);
    }

    /* Fit the response */
    try
    {
        if(nknots > 0 )
            resp.fit_response_spline(nknots, ignored_waves, ignored_wave_ranges);
        else if(degree > 0 )
            resp.fit_response_pol(degree, ignored_waves, ignored_wave_ranges);
        if(use_sedcorr)
        {
            if(nknots > 0 )
                resp_sedcorr.fit_response_spline(nknots, ignored_waves, ignored_wave_ranges);
            else if(degree > 0 )
                resp_sedcorr.fit_response_pol(degree, ignored_waves, ignored_wave_ranges);
        }
    }
    catch (std::length_error& ex)
    {
        cpl_msg_error(cpl_func, "Too few points in response fitting");
        return NULL;
    }
    
    if(nknots > 0 && resp.nknots_used_response() != (size_t)nknots)
        cpl_msg_warning(cpl_func, "Number of nknots in response fitting too high. "
                "Changed to maximum: %zd", resp.nknots_used_response());
    if(nknots > 0 && resp.nknots_used_efficiency() != (size_t)nknots)
        cpl_msg_warning(cpl_func, "Number of nknots in efficiency fitting too high. "
                "Changed to maximum: %zd", resp.nknots_used_efficiency());
    if(degree > 0 && resp.degree_used_response() != (size_t)degree)
        cpl_msg_warning(cpl_func, "Degree in response fitting too high. "
                "Changed to maximum: %zd", resp.degree_used_response());
    if(degree > 0 && resp.degree_used_efficiency() != (size_t)degree)
        cpl_msg_warning(cpl_func, "Degree in efficiency fitting too high. "
                "Changed to maximum: %zd", resp.degree_used_efficiency());

    /*
     * Assemble the product spectrophotometric response_table.
     */
    response_table = create_response_table(resp.wave_tab(), resp.flux_tab(),
        resp.flux_obs(), resp.efficiency_raw(), resp.efficiency_fit(),
        resp.response_raw());

    /*Add USED_FIT col*/
    {
        cpl_table_new_column(response_table, "USED_FIT", CPL_TYPE_INT);
        std::vector<int> used_fit(resp.wave_tab().size());
        for(size_t i_used = 0; i_used < resp.wave_tab().size(); ++i_used)
        {
            used_fit[i_used] = 1;
            for(size_t i_ignore = 0 ; i_ignore < resp.ignored_waves().size(); ++i_ignore)
            {
                if(resp.wave_tab()[i_used] == resp.ignored_waves()[i_ignore])
                    used_fit[i_used] = 0;
            }
        }
        cpl_table_copy_data_int(response_table, "USED_FIT", used_fit.data());
    }

    const bool inc_extrapolation = false;
    response_obs_binning = cpl_table_new(resp.wave_obs(inc_extrapolation).size());
    cpl_table_new_column(response_obs_binning, "WAVE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_obs_binning, "WAVE", "Angstrom");
    cpl_table_copy_data_double(response_obs_binning, "WAVE", &(resp.wave_obs(inc_extrapolation)[0]));

    cpl_table_new_column(response_obs_binning, "EFFICIENCY", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_obs_binning, "EFFICIENCY", "electron/photon");
    cpl_table_copy_data_double(response_obs_binning, "EFFICIENCY", &(resp.efficiency_fit_obs(inc_extrapolation)[0]));

    if(use_sedcorr)
    {
        update_response_table_ffsed(response_table,
            resp_sedcorr.flux_obs(), resp_sedcorr.response_raw(), resp_sedcorr.response_fit());
        cpl_table_new_column(response_obs_binning, "RESPONSE_FFSED", CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(response_obs_binning, 
                                  "RESPONSE_FFSED", "10^(-16) erg/(cm^2 electron)");
        cpl_table_copy_data_double(response_obs_binning, "RESPONSE_FFSED", &(resp_sedcorr.response_fit_obs(inc_extrapolation)[0]));
    }
    else
    {
        update_response_table(response_table, resp.response_fit());
        cpl_table_new_column(response_obs_binning, "RESPONSE", CPL_TYPE_DOUBLE);
        cpl_table_set_column_unit(response_obs_binning, 
                                  "RESPONSE", "10^(-16) erg/(cm^2 electron)");
        cpl_table_copy_data_double(response_obs_binning, "RESPONSE", &(resp.response_fit_obs(inc_extrapolation)[0]));
    }

    return response_table;
}

/**
 * @brief    Get the flat sed normalisation factor, taking into account the differences
 *           in gain, binning, and original normalisation between flat sed and specphot.
 *
 * @param    flat_sed_header  The flat sed header
 * @param    specphot_header  The specphot header
 * @param        slit_id                  The id of the slit to obtain the factor for
 *
 * @return   The flat sed normalisation factor for the given slit
 */
static double get_flat_sed_final_norm_factor(cpl_propertylist* flat_sed_header, cpl_propertylist* specphot_header, int slit_id)
{
    /* Get the proper normalisation factor. The flat sed has to be
     * normalised by the same normalisation factor applied in the
     * creation of specphot.
     * So the original factor applied to the flat sed has to be removed 
     * and then apply the one in the specphot.
     *
     * Note also that any differences in gain and binning must also be accounted for here.
     *
     * */

    std::ostringstream norm_key;
    norm_key << "ESO QC FLAT SED_" << slit_id << " NORM";
    double flat_sed_norm_orig =
        cpl_propertylist_get_double(flat_sed_header, norm_key.str().c_str());
    double specphot_flat_sed_norm = 
        cpl_propertylist_get_double(specphot_header, "ESO QC RESP FLAT_SED_NORM");

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        throw std::runtime_error("Missing keyword ESO QC FLAT_SED_* in flat sed or specphot header");
    }

    double flat_sed_gain = cpl_propertylist_get_double(flat_sed_header, "ESO DET OUT1 CONAD");
    double specphot_gain = cpl_propertylist_get_double(specphot_header, "ESO DET OUT1 CONAD");

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        throw std::runtime_error("Missing keyword ESO DET OUT1 CONAD in flat sed or specphot header");
    }

    int flat_sed_binx = cpl_propertylist_get_int(flat_sed_header, "ESO DET WIN1 BINX");
    int flat_sed_biny = cpl_propertylist_get_int(flat_sed_header, "ESO DET WIN1 BINY");
    int specphot_binx = cpl_propertylist_get_int(specphot_header, "ESO DET WIN1 BINX");
    int specphot_biny = cpl_propertylist_get_int(specphot_header, "ESO DET WIN1 BINY");

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        throw std::runtime_error("Missing keywords ESO DET WIN1 BINX/Y in flat sed or specphot header");
    }

    return (specphot_flat_sed_norm * specphot_gain * flat_sed_binx * flat_sed_biny) / 
        (flat_sed_norm_orig * flat_sed_gain * specphot_binx * specphot_biny);
}

void fors_science_correct_flat_sed
(cpl_image *spectra,  cpl_table *objects,
 cpl_image * mapped_flat_sed,
 cpl_propertylist * flat_sed_header,
 cpl_propertylist * specphot_header,
 const fors::detected_slits& det_slits)
{

    cpl_size nx = cpl_image_get_size_x(spectra);
    cpl_size nslits = cpl_table_get_nrow(objects);
    int   maxobjects = 1;
    char  name[MAX_COLNAME];
    snprintf(name, MAX_COLNAME, "object_%d", maxobjects);
    while (cpl_table_has_column(objects, name)) {
        maxobjects++;
        snprintf(name, MAX_COLNAME, "object_%d", maxobjects);
    }
    maxobjects--;
    //TODO: Move it away from here. It is also repeated in the science photometric correction
    for (cpl_size i_slit = 0; i_slit < nslits; i_slit++) {
        /* Get the normalisation factor */
        double flat_sed_final_norm_factor = 
            get_flat_sed_final_norm_factor(flat_sed_header, specphot_header,
                    det_slits[i_slit].slit_id());

        for (int i_obj = 1; i_obj <= maxobjects; i_obj++) {
            snprintf(name, MAX_COLNAME, "row_%d", i_obj);
            if (cpl_table_is_valid(objects, name, i_slit))
            {
               int null;
               int idx_obj = 
                   cpl_table_get_int(objects, name, i_slit, &null);
                /* Divide the target spectrum by the corresponding slit profile */
                for (cpl_size i_pix = 0; i_pix < nx; i_pix++)
                {
                    double profile_val  = cpl_image_get(mapped_flat_sed, i_pix+1, i_slit+1, &null);
                    if(profile_val != 0)
                        cpl_image_set(spectra, i_pix+1, idx_obj+1, 
                                cpl_image_get(spectra, i_pix+1, idx_obj+1, &null) / profile_val *
                                flat_sed_final_norm_factor);
                }
            }
        }
    }
}


void fors_science_correct_flat_sed_mapped
(cpl_image *mapped_image,  cpl_table *objects,
 cpl_image * mapped_flat_sed,
 cpl_propertylist * flat_sed_header,
 cpl_propertylist * specphot_header,
 const fors::detected_slits& det_slits)
{

    cpl_size nx = cpl_image_get_size_x(mapped_image);
    cpl_size nslits = cpl_table_get_nrow(objects);
    //TODO: Move it away from here. It is also repeated in the science photometric correction
    for (cpl_size i_slit = 0; i_slit < nslits; i_slit++) {
        /* Get the normalisation factor */
        double flat_sed_final_norm_factor = 
            get_flat_sed_final_norm_factor(flat_sed_header, specphot_header,
                    det_slits[i_slit].slit_id());

        int null;
        int slit_start = cpl_table_get_int(objects, "position", i_slit, &null);
        int slit_length = cpl_table_get_int(objects, "length", i_slit, &null);
        for (int j_pix = slit_start; j_pix < slit_start + slit_length; j_pix++) 
        {
            /* Divide the target spectrum by the corresponding slit profile */
            for (cpl_size i_pix = 0; i_pix < nx; i_pix++)
            {
                double profile_val  = cpl_image_get(mapped_flat_sed, i_pix+1, i_slit+1, &null);
                if(profile_val != 0)
                    cpl_image_set(mapped_image, i_pix+1, j_pix+1, 
                            cpl_image_get(mapped_image, i_pix+1, j_pix+1, &null) / profile_val *
                            flat_sed_final_norm_factor);
                else    
                    cpl_image_set(mapped_image, i_pix+1, j_pix+1, 0.);
            }
        }
    }
}

/*---------------------------------------------------------------------------*/
/**
  @brief    load fit point table

  @param    frames    input frames list
  @param    grism_filter grism and filter name
  @param    star_name name of star
  @param    pptable   pointer to new table
 */
/*---------------------------------------------------------------------------*/

void 
fors_load_fp_table(const cpl_frame* frm_ref, 
                    const char* grism_filter,
                    const char* star_name,
                    cpl_array** fit_points)
{
    const char* name = NULL;

    /* fits point catalog frame */
    if (frm_ref && grism_filter && star_name && fit_points) {
        name=cpl_frame_get_filename(frm_ref);
        cpl_msg_info(cpl_func, "Loading from catalog %s for %s and %s", name, grism_filter, star_name);
        if (name) {
            cpl_table* pindex = cpl_table_load(name, 1, 0);
            if (pindex) {
                cpl_size nrows = cpl_table_get_nrow(pindex);
                const char* curr_grism_filter = NULL;
                int inull = 0;
                for (int i = 0; i < nrows; i++) {
                    int ext_id = cpl_table_get_int(pindex, "ext_id", i, &inull);
                    curr_grism_filter = cpl_table_get_string(pindex, "grism_filter", i);
                    if(cpl_error_get_code() != CPL_ERROR_NONE) {
                        break;
                    }
                    if ((ext_id > 0) && !strcmp(curr_grism_filter, grism_filter)) {
                        // found
                        // retrieve the data
                        cpl_table* tmp_table = cpl_table_load(name, ext_id, 0);
                        int nfp = cpl_table_get_nrow(tmp_table);
                        cpl_table_unselect_all(tmp_table);
                        for (int j = 0; j < nfp; j++) {
                            int inuse = cpl_table_get_int(tmp_table, star_name, j, NULL);
                            if (!inuse) {
                                cpl_table_select_row(tmp_table, j);
                            }
                        }
                        cpl_table_erase_selected(tmp_table);
                        nfp = cpl_table_get_nrow(tmp_table);
                        double* fit_point_data = cpl_table_get_data_double(tmp_table, "LAMBDA");
                        cpl_array* tmp_fp = cpl_array_wrap_double(fit_point_data, nfp);
                        *fit_points = cpl_array_duplicate(tmp_fp);
                        cpl_array_unwrap(tmp_fp);
                        cpl_table_delete(tmp_table);
                        return;
                    }
              }
            }
        }
    }
    cpl_msg_error(cpl_func, "Could not load the catalog");
}

/*---------------------------------------------------------------------------*/
/**
  @brief    load reference table

  @param    frames    input frames list
  @param    dRA       Right Ascension
  @param    dDEC      Declination
  @param    EPSILON   tolerance to find ref spectra on catalog on (ra,dec)
  @param    pptable    pointer to new table
  @return   Interpolated data points
 */
/*---------------------------------------------------------------------------*/

void 
fors_load_ref_table(const cpl_frame* frm_ref, 
                    double dRA, 
                    double dDEC, 
                    double EPSILON, 
                    cpl_table** pptable,
                    const char** star_name)
{
    const char* name = NULL;
    const char* tmp_star_name = NULL;

    /* REF STD catalog frame */
    if (frm_ref) {
        name=cpl_frame_get_filename(frm_ref);
        cpl_msg_info(cpl_func, "Loading from catalog %s", name);
        if (name) {
            star_index* pstarindex = star_index_load(name);
            if (pstarindex) {
                cpl_msg_info(cpl_func, "Searching std RA[%f] DEC[%f] with tolerance[%f] in star catalog", dRA, dDEC, EPSILON);
                *pptable = star_index_get(pstarindex, dRA, dDEC, EPSILON, EPSILON, &tmp_star_name);
   
                if (*pptable && tmp_star_name) {
                    cpl_msg_info(cpl_func, "Found STD star: %s", tmp_star_name);
                    if (star_name) {
                        *star_name = cpl_strdup(tmp_star_name);
                    }
                } else {
                    cpl_msg_error(cpl_func, "REF star %s could not be found in the catalog", tmp_star_name);
                    if (star_name) {
                        *star_name = NULL;
                    }
                }
                star_index_delete(pstarindex);
            } else {
                cpl_msg_error(cpl_func, "Could not load the catalog");
            }
        }
    }
}

/*Corrects the spectrum s by the doppler offset offset.*/
hdrl_spectrum1D * correct_spectrum_for_doppler_shift(
    const hdrl_spectrum1D * s, const hdrl_data_t offset)
{
    if(offset == 0.0) {
        cpl_msg_warning(cpl_func, "Applied doppler shift is 0");
        return hdrl_spectrum1D_duplicate(s);
    }

    const hdrl_image * flux = hdrl_spectrum1D_get_flux(s);
    cpl_array * wavs = cpl_array_duplicate(hdrl_spectrum1D_get_wavelength(s).wavelength);

    for(cpl_size i = 0; i < cpl_array_get_size(wavs); ++i) {
        const double w = cpl_array_get(wavs, i, NULL);
        const double d = w * (1. + offset);
        cpl_array_set(wavs, i, d);
    }

    hdrl_spectrum1D * to_ret = hdrl_spectrum1D_create(
        hdrl_image_get_image_const(flux),
        hdrl_image_get_error_const(flux),
        wavs,
        hdrl_spectrum1D_get_scale(s));
    cpl_array_delete(wavs);

    return to_ret;
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Generates a 1D Gaussian Kernel
  @param   kernel_size   kernel size
  @param   sigma   the gaussian sigma

  @return  The corresponding CPL vector object or NULL in case of error
  The returned matrix must be deallocated using cpl_vector_delete().
  The user must provide kernel_size > 0, fwhm > 0
 */
/*----------------------------------------------------------------------------*/

cpl_vector *
create_gaussian_kernel(const cpl_size kernel_size, const double sigma)
{
  cpl_ensure(kernel_size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
  cpl_ensure(sigma > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

  cpl_vector * kernel = cpl_vector_new(kernel_size);

  double* pkernel = cpl_vector_get_data(kernel);

  double arg_exp;
  double x;
  double sum = 0;

  double factor = 2.0 * sigma * sigma;

  for(cpl_size i = 0; i < kernel_size; i++) {
      x = i - 0.5 * (kernel_size - 1);
      arg_exp = (x * x) / factor;
      pkernel[i] = exp( -arg_exp );
      sum += pkernel[i];
  }
  // Normalise
  cpl_vector_divide_scalar(kernel, sum);
  return kernel;
}

/* This code copied from molecfit_calctrans.c */
cpl_vector* convol(cpl_vector* v, cpl_vector* kernel)
{
    /* Code taken from cpl_vector_filter_lowpass_create */
    int nv = cpl_vector_get_size(v);
    int nk = cpl_vector_get_size(kernel);
    cpl_vector* filtered = cpl_vector_new(nv);
    int hw = floor(nk/2);
    double replace;
    cpl_size i,j;

    /* compute edge effects for the first hw elements */
    for (i = 0; i < hw; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            if (i + j < 0) {
                replace += cpl_vector_get(kernel,hw + j) * cpl_vector_get(v,0);
            }
            else {
                replace += cpl_vector_get(kernel,hw + j) * cpl_vector_get(v,i + j);
            }
        }
        cpl_vector_set(filtered,i,replace);
    }

    /* compute edge effects for the last hw elements */
    for (i = nv - hw; i < nv; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            if (i + j > nv - 1) {
                replace += cpl_vector_get(kernel,hw + j) * cpl_vector_get(v,nv - 1);
            }
            else {
                replace += cpl_vector_get(kernel,hw + j) * cpl_vector_get(v,i + j);
            }
        }
        cpl_vector_set(filtered,i,replace);
    }

    /* compute all other elements */
    for (i = hw; i < nv - hw; i++) {
        replace = 0.0;
        for (j = -hw; j <= hw; j++) {
            replace += cpl_vector_get(kernel,hw + j) * cpl_vector_get(v,i + j);
        }
        cpl_vector_set(filtered,i,replace);
    }
    //cpl_vector_delete(kernel);
    //cpl_tools_add_flops(4 * hw * nv);
    return filtered;
}   

/*
 * WARNING
 * This routine just convolves the error in the same way as the flux
 * This routine doesn't account for non-uniform wavelength sampling
 */
hdrl_spectrum1D * gaussian_convolve(hdrl_spectrum1D* s, cpl_size kernel_size, double sigma)
{
  const hdrl_image * flux = hdrl_spectrum1D_get_flux(s);
  const cpl_image * im = hdrl_image_get_image_const(flux);
  const cpl_image * err = hdrl_image_get_error_const(flux);
  cpl_vector * dummy = NULL;

  cpl_size nx = cpl_image_get_size_x(im);

  /* build convolution kernel */
  cpl_vector * kernel = create_gaussian_kernel(kernel_size, sigma);

  /* Convolve the image */
  dummy = cpl_vector_new_from_image_row(im, 1);
  cpl_vector * im_convol_v = convol(dummy, kernel);
  cpl_image * im_convol = cpl_image_wrap_double(nx, 1, cpl_vector_get_data(im_convol_v));
  cpl_vector_delete(dummy); dummy = NULL;

  dummy = cpl_vector_new_from_image_row(err, 1);
  cpl_vector * err_convol_v = convol(dummy, kernel);
  cpl_image * err_convol = cpl_image_wrap_double(nx, 1, cpl_vector_get_data(err_convol_v));
  cpl_vector_delete(dummy); dummy = NULL;

  /* Return the spectrum */
        hdrl_spectrum1D * to_ret = hdrl_spectrum1D_create(
                im_convol,
                err_convol,
                hdrl_spectrum1D_get_wavelength(s).wavelength,
                hdrl_spectrum1D_get_scale(s));

  cpl_image_unwrap(im_convol);
  cpl_vector_delete(im_convol_v);
  cpl_image_unwrap(err_convol);
  cpl_vector_delete(err_convol_v);
  cpl_vector_delete(kernel);

  return to_ret;
}

cpl_table* create_response_table(
    const std::vector<double>& wave_tab,
    const std::vector<double>& flux_tab,
    const std::vector<double>& flux_obs,
    const std::vector<double>& efficiency_raw,
    const std::vector<double>& efficiency_fit,
    const std::vector<double>& response_raw)
{
    /*
     * Assemble the product spectrophotometric response_table.
     */
    cpl_table* response_table = cpl_table_new(wave_tab.size());

    cpl_table_new_column(response_table, "WAVE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "WAVE", "Angstrom");
    cpl_table_copy_data_double(response_table, "WAVE", &wave_tab[0]);

    cpl_table_new_column(response_table, "STD_FLUX", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "STD_FLUX", 
                              "10^(-16) erg/(cm^2 s Angstrom)");
    cpl_table_copy_data_double(response_table, "STD_FLUX", &flux_tab[0]);

    cpl_table_new_column(response_table, "OBS_FLUX", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "OBS_FLUX", "electron/(s Angstrom)");
    cpl_table_copy_data_double(response_table, "OBS_FLUX", &flux_obs[0]);

    cpl_table_new_column(response_table, "RAW_EFFICIENCY", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "RAW_EFFICIENCY", "electron/photon");
    cpl_table_copy_data_double(response_table, "RAW_EFFICIENCY", &efficiency_raw[0]);

    cpl_table_new_column(response_table, "EFFICIENCY", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "EFFICIENCY", "electron/photon");
    cpl_table_copy_data_double(response_table, "EFFICIENCY", &efficiency_fit[0]);

    cpl_table_new_column(response_table, "RAW_RESPONSE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "RAW_RESPONSE", 
                              "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RAW_RESPONSE", &response_raw[0]);

    return response_table;
}

void update_response_table(
    cpl_table* response_table,
    const std::vector<double>& response_fit)
{
    cpl_table_new_column(response_table, "RESPONSE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, 
                              "RESPONSE", "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RESPONSE", &response_fit[0]);
}

void update_response_table_ffsed(
    cpl_table* response_table,
    const std::vector<double>& flux_obs,
    const std::vector<double>& response_raw,
    const std::vector<double>& response_fit)
{
    cpl_table_new_column(response_table, "OBS_FLUX_FFSED", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "OBS_FLUX_FFSED", "electron/(s Angstrom)");
    cpl_table_copy_data_double(response_table, "OBS_FLUX_FFSED", &flux_obs[0]);

    cpl_table_new_column(response_table, "RAW_RESPONSE_FFSED", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "RAW_RESPONSE_FFSED", 
                              "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RAW_RESPONSE_FFSED", &response_raw[0]);

    cpl_table_new_column(response_table, "RESPONSE_FFSED", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, 
                              "RESPONSE_FFSED", "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RESPONSE_FFSED", &response_fit[0]);
}

cpl_table * create_resp_eff_table(
    const std::vector<double>& wave_tab,
    const char* resp_col,
    const std::vector<double>& resp_tab,
    const char* eff_col,
    const std::vector<double>& eff_tab)
{
    /*
     * Assemble the tabe.
     */
    cpl_table* selected_table = cpl_table_new(wave_tab.size());

    cpl_table_new_column(selected_table, "WAVE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(selected_table, "WAVE", "Angstrom");
    cpl_table_copy_data_double(selected_table, "WAVE", &wave_tab[0]);

    cpl_table_new_column(selected_table, eff_col, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(selected_table, eff_col, "electron/photon");
    cpl_table_copy_data_double(selected_table, eff_col, &eff_tab[0]);

    cpl_table_new_column(selected_table, resp_col, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(selected_table, resp_col, 
                              "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(selected_table, resp_col, &resp_tab[0]);

    return selected_table;
}

/*Selects all the wavelengths between wmin and wmax*/
hdrl_spectrum1D * select_win(
    const hdrl_spectrum1D * s,
    const hdrl_data_t wmin,
    const hdrl_data_t wmax)
{
    cpl_bivector * bv = cpl_bivector_new(1);

    cpl_vector_set(cpl_bivector_get_x(bv), 0, wmin);
    cpl_vector_set(cpl_bivector_get_y(bv), 0, wmax);

    hdrl_spectrum1D * to_ret = hdrl_spectrum1D_select_wavelengths(s, bv, CPL_TRUE);

    cpl_bivector_delete(bv);

    return to_ret;
}

/* For each point p in fit_points generates a new spectrum whose wavelengths are
 * fit_points and whose flux are the medians taken in the range
 * [p - wrange, p + wrange]. */
hdrl_spectrum1D * get_median_on_fit_points(
    const hdrl_spectrum1D * s_input,
		const cpl_array * fit_points,
    const hdrl_data_t wrange)
{
	cpl_size final_fit_points_size = cpl_array_get_size(fit_points);

	cpl_array * wlens_fit = cpl_array_new(final_fit_points_size,
			HDRL_TYPE_DATA);
	hdrl_image * flux_fit = hdrl_image_new(final_fit_points_size, 1);

	for(cpl_size i = 0; i < final_fit_points_size; ++i){
		const double w_fit = cpl_array_get(fit_points, i, NULL);
		cpl_array_set(wlens_fit, i, w_fit);
		hdrl_spectrum1D * f_s =
				select_win(s_input, w_fit - wrange, w_fit + wrange);

		if(f_s == NULL){
			cpl_error_reset();
			hdrl_image_reject(flux_fit, i + 1, 1);
			continue;
		}

		const hdrl_value v = hdrl_image_get_median(hdrl_spectrum1D_get_flux(f_s));
		hdrl_image_set_pixel(flux_fit, i + 1, 1, v);
		hdrl_spectrum1D_delete(&f_s);
	}

	const hdrl_spectrum1D_wave_scale scale = hdrl_spectrum1D_get_scale(s_input);

	hdrl_spectrum1D * to_ret = hdrl_spectrum1D_create(
			hdrl_image_get_image(flux_fit),
			hdrl_image_get_error(flux_fit),
			wlens_fit,
			scale
			);
	cpl_array_delete(wlens_fit);
	hdrl_image_delete(flux_fit);
	return to_ret;
}
