/*
 * This file is part of the FORS Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fors_zeropoint_utils.h>
#include <fors_utils.h>
#include <fors_tools.h>
#include <fors_identify.h>
#include <fors_std_cat.h>
#include <math.h>
#include <cstring>

// CASU code is C only
extern "C" {
#include "casu_mods.h"
#include "casu_wcsutils.h"
#include "casu_utils.h"
};

/**
 * @brief  determine if zeropoint is inside cuts
 * @param  s      star
 * @param  data   hi and lo cuts
 * @return true iff the star's zeropoint is inside 
 *         the given intervals
 * 
 */
static bool
zeropoint_inside(const fors_star *s,
                 void *data) 
{
    struct zeropoint_inside_cuts{
        double hi, lo;   /* magnitude */
        double z, kappa; /* avg zeropoint, kappa */
    };
    zeropoint_inside_cuts * cuts = (zeropoint_inside_cuts *)data;
    
    double z  = fors_star_get_zeropoint(s, NULL);
    double dz = fors_star_get_zeropoint_err(s, NULL);

    return
        (cuts->lo                 <= z && z <= cuts->hi) ||
        (cuts->z - cuts->kappa*dz <= z && z <= cuts->z + cuts->kappa*dz);
}

#undef cleanup
#define cleanup \
do { \
    fors_star_list_delete(&subset, fors_star_delete); \
    fors_star_list_delete(&identified, fors_star_delete); \
} while(0)
/**
 * @brief   Compute zeropoint
 * @param   stars       list of stars, flags are set to 1 iff star is
 *                      used in final zeropoint computation
 * @param   cutoffE     rejection parameter (magnitude)
 * @param   cutoffk     rejection parameter (magnitude)
 * @param   dext_coeff  extinction coefficient error
 * @param   dcolor_term color coefficient error
 * @param   dzeropoint  (output) zeropoint stdev
 * @param   n           (output) number of stars used for zeropoint computation
 * @return  median zeropoint after rejection of negative outliers
 */
double
get_zeropoint(fors_star_list *stars, 
              double cutoffE,
              double cutoffk,
              double dext_coeff,
              double dcolor_term,
              double avg_airmass,
              double *dzeropoint,
              int *n)
{
    fors_star_list *subset = NULL;
    fors_star_list *identified = 
        fors_star_list_extract(stars, fors_star_duplicate,
                               fors_star_is_identified, NULL);

    assure( stars != NULL, return 0, NULL );
    assure( dzeropoint != NULL, return 0, NULL );
    assure( n != NULL, return 0, NULL );

    if ( fors_star_list_size(identified) == 0 ) {
        cpl_msg_warning(cpl_func, 
                        "No identified stars for zeropoint computation");
        *n = 0;
        *dzeropoint = 0;
        cleanup;
        return 0;
    }
    
    cpl_msg_info(cpl_func, "Computing zeropoint (assuming extinction)");
    cpl_msg_indent_more();

    double zeropoint;
    double red_chisq = -1.0;

    /* This method does not take into account that the error bars are
       correlated, and therefore computes an unrealistically low
       dzeropoint */
    zeropoint = fors_star_list_mean_optimal(identified,
                                            fors_star_get_zeropoint, NULL,
                                            fors_star_get_zeropoint_err, NULL,
                                            dzeropoint,
                                            fors_star_list_size(identified) >= 2 ? &red_chisq : NULL);

    cpl_msg_info(cpl_func, "Optimal zeropoint (no rejection, %d stars) = %f mag",
                 fors_star_list_size(identified), zeropoint);  
    fors_star_print_list(CPL_MSG_INFO, identified);
    
    /* Reject stars that are absolute (0.3 mag) outliers and
       kappa sigma outliers. For robustness (against error in the initial
       zeropoint estimates) apply the absolute cut in two steps
       and update the estimated zeropoint after the first step.
    */
    struct {
        double hi, lo;   /* magnitude */
        double z, kappa; /* avg zeropoint, kappa */
    } cuts;
    cuts.hi = zeropoint + 5*cutoffE;
    cuts.lo = zeropoint - 5*cutoffE;
    cuts.z = zeropoint;
    cuts.kappa = cutoffk;
    
    subset = fors_star_list_extract(identified, fors_star_duplicate,
                                    zeropoint_inside, &cuts);


    if ( fors_star_list_size(subset) == 0 ) {
        cpl_msg_warning(cpl_func, 
                        "All stars rejected (%f mag). Cannot "
                        "compute zeropoint", 5*cutoffE);
        *n = 0;
        *dzeropoint = 0;
        cleanup;
        return 0;
    }

    zeropoint = fors_star_list_mean_optimal(subset,
                                            fors_star_get_zeropoint, NULL,
                                            fors_star_get_zeropoint_err, NULL,
                                            dzeropoint,
                                            fors_star_list_size(subset) >= 2 ? &red_chisq : NULL);

    cpl_msg_debug(cpl_func, "Optimal zeropoint (%.2f mag, %.2f sigma rejection) = %f mag",
                  5*cutoffE, cutoffk, zeropoint);  
    
    cuts.hi = zeropoint + cutoffE;
    cuts.lo = zeropoint - cutoffE;
    cuts.z = zeropoint;
    cuts.kappa = cutoffk;
    
    {
        fors_star_list *tmp = fors_star_list_duplicate(subset, fors_star_duplicate);
        fors_star_list_delete(&subset, fors_star_delete);

        subset = fors_star_list_extract(tmp, fors_star_duplicate,
                                        zeropoint_inside, &cuts);

        if ( fors_star_list_size(subset) == 0 ) {
            cpl_msg_warning(cpl_func, 
                            "All stars rejected (%f mag, %f sigma). Cannot "
                            "compute zeropoint", cutoffE, cutoffk);
            *n = 0;
            *dzeropoint = 0;
            cleanup;
            return 0;
        }

        fors_star_list_delete(&tmp, fors_star_delete);
    }

    zeropoint = fors_star_list_mean_optimal(subset,
                                            fors_star_get_zeropoint, NULL,
                                            fors_star_get_zeropoint_err, NULL,
                                            dzeropoint,
                                            fors_star_list_size(subset) >= 2 ? &red_chisq : NULL);
    
    cpl_msg_info(cpl_func, "Optimal zeropoint (%.2f mag, %.2f sigma rejection) = %f mag",
                 cutoffE, cutoffk, zeropoint);  
    
    *n = fors_star_list_size(subset);
    {
        int outliers = 
            fors_star_list_size(identified) - fors_star_list_size(subset);
        cpl_msg_info(cpl_func, "%d outlier%s rejected, %d non-outliers",
                     outliers, outliers == 1 ? "" : "s",
                     *n);
    }

    if ( *n == 0 ) {
        cpl_msg_warning(cpl_func, 
                        "All stars were rejected during zeropoint computation" );
        *dzeropoint = 0;
        cleanup;
        return 0;
    }







    /*
      Build zeropoint covariance matrix.
      We have already the variances from fors_star_get_zeropoint_err().
      Non-diagonal terms are
         Cij = airmass^2 * Variance(ext.coeff) + color_i * color_j * Variance(color.coeff)

      It was considered and tried to subtract the term
               airmass^2 * Variance(ext.coeff)
      from every Cij. This has no effect on the relative weights, only the weight's overall
      normalization. Since we use the normalization for computing the zeropoint error this
      term is kept.

    */
    cpl_matrix *covariance = cpl_matrix_new(*n,
                                            *n);

    /* Duplicate the list to allow simultaneous iterations */
    fors_star_list *ident_dup = fors_star_list_duplicate(subset, fors_star_duplicate);
    {
      
      fors_star *s, *t;
      int i, j;
      for (s = fors_star_list_first(subset), i = 0;
           s != NULL;
           s = fors_star_list_next(subset), i++) {

        for (t = fors_star_list_first(ident_dup), j = 0;
             t != NULL;
             t = fors_star_list_next(ident_dup), j++) {
          
          double cij;

          if (fors_star_equal(s, t)) {
              cij = fors_star_get_zeropoint_err(s, NULL)*fors_star_get_zeropoint_err(s, NULL);
              /*  -avg_airmass*avg_airmass*dext_coeff*dext_coeff */
          }
          else {
              cij = s->id->color * t->id->color * dcolor_term*dcolor_term
                  + avg_airmass*avg_airmass*dext_coeff*dext_coeff;
          }
          
          cpl_matrix_set(covariance, i, j, cij);
        }
      }
    }
    /* cpl_matrix_dump(covariance, stdout); */
    /* cpl_matrix_dump(cpl_matrix_invert_create(covariance), stdout); */

    /*
      Compute optimal weights, w, as
      
      w = C^-1 * const

      C    : nxn covariance matrix
      const: nx1 constant vector with all elements equal to 1
     */

    cpl_matrix *covariance_inverse = cpl_matrix_invert_create(covariance);

    assure( !cpl_error_get_code(), return 0,
            "Could not invert zeropoints covariance matrix");

    /*  cpl_matrix_dump(cpl_matrix_product_create(covariance_inverse, covariance), stdout); */

    /* fprintf(stderr, "is_identity = %d\n", cpl_matrix_is_identity(cpl_matrix_product_create(covariance_inverse, covariance),1e-10)); */

    cpl_matrix_delete(covariance); covariance = NULL;

    cpl_matrix *const_vector = cpl_matrix_new(*n, 1);
    cpl_matrix_fill(const_vector, 1.0);
    
    cpl_matrix *weights = cpl_matrix_product_create(covariance_inverse, const_vector);

    cpl_matrix_delete(const_vector); const_vector = NULL;

    /* cpl_matrix_dump(weights, stdout); */
    
    
    {
        double wz = 0;
        double w = 0;
        
        int i;
        fors_star *s;
        for (i = 0, s = fors_star_list_first(subset);
             s != NULL;
             s = fors_star_list_next(subset), i++) {
            
            double weight = cpl_matrix_get(weights, i, 0);

            cpl_msg_debug(cpl_func, "Weight_%d = %f", i, weight);
            
            wz += weight * fors_star_get_zeropoint(s, NULL);
            w += weight;

            /* Loop through original list to record the weight of this star */
            {
                fors_star *t;
                
                for (t = fors_star_list_first(stars);
                     t != NULL;
                     t = fors_star_list_next(stars)) {
                    
                    if (fors_star_equal(s, t)) {
                        t->weight = weight;
                    }
                }
            }
        }

        cpl_matrix_delete(weights); weights = NULL;

        cpl_msg_debug(cpl_func, "Sum of weights = %f", w);

        /* 
           C is positive definite (because all eigenvalues are positive). Therefore
           C^-1 is also positive definite, a property of positive definite matrices.
           
           Positive definite matrices always have the property that
                 z* C z > 0     for any non-zero (complex) vector z

           The sum of the weights is just
                const.* w = const.* C^-1 const.
           where const. is our constant vector filled with 1. Therefore the sum of
           the weights should always be positive. But make the paranoia check anyway:
         */

        assure( w != 0, return 0, "Sum of optimal weights is zero!" );
        assure( sqrt(w) != 0, return 0, "Square root of sum of optimal weights is zero!" );
        
        zeropoint = wz / w;
        *dzeropoint = 1 / sqrt(w);
    }

    /* Previous code: weighted average, uncorrelated errors

    zeropoint = fors_star_list_mean_optimal(
        subset, 
        fors_star_get_zeropoint, NULL,
        fors_star_get_zeropoint_err, NULL,
        dzeropoint,
        *n >= 2 ? &red_chisq : NULL);
    */
    
    cpl_msg_info(cpl_func, "Optimal zeropoint = %f +- %f mag", 
                 zeropoint, *dzeropoint);

    if (*n >= 2) {
        fors_star *s, *t;
        int i, j;

        red_chisq = 0;

        for (s = fors_star_list_first(subset), i = 0;
             s != NULL;
             s = fors_star_list_next(subset), i++) {
            
            for (t = fors_star_list_first(ident_dup), j = 0;
                 t != NULL;
                 t = fors_star_list_next(ident_dup), j++) {
                
                red_chisq += 
                    (fors_star_get_zeropoint(s, NULL) - zeropoint) *
                    (fors_star_get_zeropoint(t, NULL) - zeropoint) *
                    cpl_matrix_get(covariance_inverse, i, j);
            }
        }
        red_chisq /= (*n - 1);
    }

    cpl_matrix_delete(covariance_inverse); covariance_inverse = NULL;

    fors_star_list_delete(&ident_dup, fors_star_delete);
    
    cpl_msg_info(cpl_func, "Reduced chi square = %f", red_chisq);
    
    cpl_msg_indent_less();

    cleanup;
    return zeropoint;
}

#undef cleanup
#define cleanup \
do { \
    fors_identify_method_delete(&im); \
    cpl_parameterlist_delete(tmp_param); \
    cpl_free(context); \
    cpl_wcs_delete(wcs); \
} while (0)
void
calculate_zp_from_gaia(char filter_band, double cutoffe, double cutoffk, cpl_table* phot,
             cpl_table** phot_stds_to_save, cpl_table** updated_stars_to_save,
             cpl_propertylist* img_header, const char* cacheloc, double color_term,
             double dcolor_term, double& zp, double& dzp, int& nzp)
{
    identify_method *im = NULL;
    cpl_wcs *wcs = NULL;
    cpl_parameterlist* tmp_param = cpl_parameterlist_new();
    char *context   = cpl_sprintf("fors");

    // We don't allow the user to change the identify parameters
    // because the only user-configurable value in the set is
    // not relevant here, since the WCS solution is already done.
    fors_identify_define_parameters(tmp_param, context);
    im = fors_identify_method_new(tmp_param, context);

    assure( !cpl_error_get_code(), return, 
            "Could not get identification parameters" );

    /* Get the WCS */
    wcs = cpl_wcs_new_from_propertylist(img_header);

    // Only look at the synthetic photometry if the filter band is appropriate
    if ((filter_band != 'U') && (filter_band != '?')) {
        /* Get the Gaia synthetic photometry catalog */
        char catname1[] = "gaiasyntphot";
        cpl_table *phot_stds = NULL;
        int cdssearch1 = casu_get_cdschoice(catname1);
        int status, n_phot_stds, n_main_stds;
        status = 0;
        casu_getstds(img_header, 1, NULL, catname1, cdssearch1, (char*)cacheloc, &phot_stds, NULL, &status);
        if (status != 0) {
            cpl_msg_warning(cpl_func, "Error in retrieving %s catalog: %d", catname1, status);
        }
        n_phot_stds = (int)cpl_table_get_nrow(phot_stds);
        cpl_msg_info(cpl_func, "Number of %s stars: %d", catname1, n_phot_stds);

        /* Get the main Gaia DR3 catalog */
        char catname2[] = "gaiadr3";
        cpl_table *main_stds = NULL;
        int cdssearch2 = casu_get_cdschoice(catname2);
        status = 0;
        casu_getstds(img_header, 1, NULL, catname2, cdssearch2, (char*)cacheloc, &main_stds, NULL, &status);
        if (status != 0) {
            cpl_msg_warning(cpl_func, "Error in retrieving %s catalog: %d", catname2, status);
        }
        n_main_stds = (int)cpl_table_get_nrow(main_stds);
        cpl_msg_info(cpl_func, "Number of %s stars: %d", catname2, n_main_stds);

        // cpl_table_dump(main_stds, 0, nstd, NULL);
        // cpl_table_save(main_stds, NULL, NULL, "main_stds.fits", CPL_IO_DEFAULT);

        /* Cross match the two catalogs.
         * This is not very efficient, but both tables will be small because of the
         * limited field of view of the instrument.  I doubt it's worth trying to optimise.
         *
         * It's not possible to use the standard vizier query in the casu code to join
         * the tables. It would be possible to do this via an ADQL query, but that 
         * would require additional code, whereas the simple Vizier query code is already
         * in use and easy to adapt for this routine.
         */
        if ((n_phot_stds <= 0) || (n_main_stds <= 0)) {
            cpl_msg_warning(cpl_func, "Unable to proceed with in-situ photometry "
                "(too few stars, or error in retrieving catalog)");
        } else {
            cpl_table_unselect_all(phot_stds);
            for (int i=0; i<n_phot_stds; i++) {
                // For every source in the gaiasyntphot table
                long long src = cpl_table_get_long_long(phot_stds, "Source", i, NULL);
                for (int j=0; j<n_main_stds; j++) {
                    // Find the matching source in the main gaiadr3 table
                    long long src2 = cpl_table_get_long_long(main_stds, "Source", j, NULL);
                    if (src == src2) {
                        // Found it
                        cpl_msg_debug(cpl_func, "Matched source %lld", src);
                        // Grab the relevant data
                        double ruwe = cpl_table_get(main_stds, "RUWE", j, NULL);
                        cpl_msg_debug(cpl_func, "Source %lld, ruwe = %f", src, ruwe);
                        if (ruwe >= 1.3) {
                            cpl_table_select_row(phot_stds, i);
                            break;
                        }
                        double c_star = cpl_table_get(phot_stds, "E_BP_RP_corr", i, NULL);
                        double phot_g_mean_mag = cpl_table_get(main_stds, "Gmag", j, NULL);
                        cpl_msg_debug(cpl_func, "Source %lld, c_star = %f", src, c_star);
                        cpl_msg_debug(cpl_func, "Source %lld, phot_g_mean_mag = %f", src, phot_g_mean_mag);
                        if (c_star >= 0.0059898 + 8.817481e-12 * pow(phot_g_mean_mag, 7.618399)) {
                            cpl_table_select_row(phot_stds, i);
                            break;
                        }
                        const char* phot_variable_flag = cpl_table_get_string(main_stds, "VarFlag", j);
                        cpl_msg_debug(cpl_func, "Source %lld, phot_variable_flag = %s", src, phot_variable_flag);
                        if (phot_variable_flag && !strcmp(phot_variable_flag, "VARIABLE")) {
                            cpl_table_select_row(phot_stds, i);
                            break;
                        }
                        int non_single_star = cpl_table_get_int(main_stds, "NSS", j, NULL);
                        cpl_msg_debug(cpl_func, "Source %lld, non_single_star = %d", src, non_single_star);
                        if (non_single_star != 0) {
                            cpl_table_select_row(phot_stds, i);
                            break;
                        }
                        int b_jkc_flag = cpl_table_get_int(phot_stds, "BFlag", i, NULL);
                        cpl_msg_debug(cpl_func, "Source %lld, b_jkc_flag = %d", src, b_jkc_flag);
                        if (b_jkc_flag <= 0) {
                            cpl_table_select_row(phot_stds, i);
                            break;
                        }
                        /*
                         * Fragment of ADQL used to construct this section of code
                           AND dr3.ruwe<1.3 \
                           AND \
                           ABS(gspc.c_star)<(0.0059898 + 8.817481e-12 * \
                           POWER(dr3.phot_g_mean_mag,7.618399)) \
                           AND \
                           dr3.phot_g_mean_mag<17.65 \
                           AND dr3.phot_variable_flag != 'VARIABLE' \
                           AND dr3.non_single_star = 000 \
                           AND gspc.b_jkc_flag > 0 "
                        */
                    }
                }
                // The above code selects the row if it is to be excluded
                if (cpl_table_is_selected(phot_stds, i)) {
                  cpl_msg_debug(cpl_func, "Rejecting source %lld", src);
                }
            }
            // Erase all the excluded rows
            cpl_table_erase_selected(phot_stds);

            n_phot_stds = (int)cpl_table_get_nrow(phot_stds);
            cpl_msg_info(cpl_func, "Number of %s stars after filtering: %d", catname1, n_phot_stds);
            // cpl_table_dump(phot_stds, 0, nstd, NULL);
            // cpl_table_save(phot_stds, NULL, NULL, "phot_stds.fits", CPL_IO_DEFAULT);

            /* Now work on the list of stellar sources found in the previous recipe */
            fors_star_list* tmpstars = fors_create_star_list(phot);
            fors_std_star_list *stdlist = fors_std_star_list_new();
            /* Load the Gaia syntphot catalog as a list of standard stars */
            fors_std_table_load(phot_stds, stdlist, filter_band, false, color_term, dcolor_term);
            if (cpl_error_get_code()) {
                cpl_msg_warning(cpl_func, "Error when loading standards: %s", cpl_error_get_message());
                cpl_error_reset();
            }

            /* Fill in the x and y coordinates for the standard star list */
            for (fors_std_star* s = fors_std_star_list_first(stdlist);
                s != NULL;
                s = fors_std_star_list_next(stdlist)) {
                
                casu_radectoxy(wcs, s->ra, s->dec, &s->pixel->x, &s->pixel->y);
            }

            /* Identify (stdlist stars are duplicated and linked to tmpstars)
             * Note that we give an explicit WCS offset of 0, 0 because the
             * WCS solution has already been done.
             */
            fors_identify_associate(tmpstars, stdlist, im, 0, 0);
            assure( !cpl_error_get_code(), return, "Failed to identify sources");

            /* Set the "corrected" magnitudes to the instrumental magnitudes.
             * We do this because the in-situ zeropoint is calculated without
             * regard to any other corrections
             */
            for (fors_star* s = fors_star_list_first(tmpstars);
                s != NULL;
                s = fors_star_list_next(tmpstars)) {
                if (s->id) {
                    s->magnitude_corr = s->magnitude;
                    s->dmagnitude_corr = s->dmagnitude;
                }
            }

            /* Attempt to calculate the zeropoint based on the Gaia syntphot catalog */
            zp = get_zeropoint(tmpstars, cutoffe, cutoffk, 0, dcolor_term, 1.0, &dzp, &nzp);
            *updated_stars_to_save = fors_create_sources_table(tmpstars);

            fors_star_list_delete(&tmpstars, fors_star_delete);
            fors_std_star_list_delete(&stdlist, fors_std_star_delete);
        }
        if (phot_stds != NULL) {
            *phot_stds_to_save = cpl_table_duplicate(phot_stds);
        }
  
        cpl_table_delete(main_stds);
        cpl_table_delete(phot_stds);
    }
    cleanup;
}
