/* $Id: mosca_pfits.c,v 1.12 2011-10-24 13:08:17 cgarcia Exp $
 *
 * This file is part of the MOSCA Library
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "mosca_pfits.h"

#include <cpl.h>
#include <string.h>

#ifndef MOSCA_MJD_OBS_THRESHOLD
#define MOSCA_MJD_OBS_THRESHOLD   59580.
#endif

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mosca_pfits  MOSCA Protected FITS functionality
 */
/*----------------------------------------------------------------------------*/

/**
 * @brief  Determine the header version (default, or the new FORS1 NGCII)
 * @param  self The header to process
 * @return 1 for new FORS1 (NGCII), zero otherwise
 * @note This function cannot fail, it returns its default on unexpected input
 * 
 * Currently, the header has to have
 * INSTRUME = 'FORS1' and
 * MJD-OBS >= MOSCA_MJD_OBS_THRESHOLD
 * - for NGCII, otherwise the default is assumed
 */
size_t mosca_pfits_version_get(const cpl_propertylist* self)
{
    const cpl_errorstate prestate = cpl_errorstate_get();
    /* Read relevant contents */
    const char*  instrume = cpl_propertylist_get_string(self, "INSTRUME");
    const double mjd_obs = cpl_propertylist_get_double(self, "MJD-OBS");


    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_warning(cpl_func, "Unexpected header, assuming default");
        cpl_errorstate_dump(prestate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(prestate);
        return MOSCA_PFITS_DEFAULT;
    }

    if (instrume == NULL) return MOSCA_PFITS_DEFAULT; /* Guard strcmp() */
    if (strcmp(instrume, "FORS1")) return MOSCA_PFITS_DEFAULT;

    if (mjd_obs < MOSCA_MJD_OBS_THRESHOLD) return MOSCA_PFITS_DEFAULT;

    return MOSCA_PFITS_FORS_NGC;
}

/**@}*/
