/* 
 * This file is part of the FORS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <time.h>
#include <string.h>
#include "fors_dfs.h"
#include <hdrl.h>
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int fors_eop(cpl_frameset *, const cpl_parameterlist *);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char fors_eop_description[] =
"This recipe downloads the latest version of the Earth Orientation Parameter and \n"
"DUT from IERS. It stores the parameter in a fitsfile that is used as input for  \n"
"the computation of the barycentric correction.                                  \n"
"                                                                                \n"
"Input files:                                                                    \n"
"                                                                                \n"
"   DO category:                 Explanation:                                    \n"
"   --------------------------   --------                                        \n"
"           (no input)           Will retrieve everything from the Internet      \n"
"                                                                                \n"
"Output files:                                                                   \n"
"                                                                                \n"
"   DO category:                 Explanation:                                    \n"
"   --------------------------   --------                                        \n"
"   EOP_PARAM                    The IERS Earth Orientation Parameter            \n";


/* Standard CPL recipe definition */
cpl_recipe_define( fors_eop,
		   FORS_BINARY_VERSION,
		   "HDRL/ESOTK Group",
		   PACKAGE_BUGREPORT,
		   "2022",
		   "Retrieves the IERS Earth Orientation Parameter",
		   fors_eop_description);



/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param   self         Output parameterlist
 *
 * @return  cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code fors_eop_fill_parameterlist(
    cpl_parameterlist *self)
{

  cpl_parameter *par;

  /* --eop_host */
  par = cpl_parameter_new_value ("fors.eop.eop_host",
				 CPL_TYPE_STRING,
				 "FTP Host to retrieve the EOP from",
				 "fors.eop",
				 "https://datacenter.iers.org");
  cpl_parameter_set_alias (par, CPL_PARAMETER_MODE_CLI, "eop_host");
  cpl_parameter_disable (par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append (self, par);

  /* --eop_urlpath */
  par = cpl_parameter_new_value ("fors.eop.eop_urlpath",
				 CPL_TYPE_STRING,
				 "URL path of the EOP file to retrieve",
				 "fors.eop",
				 "/products/eop/rapid/standard/finals2000A.data");
  cpl_parameter_set_alias (par, CPL_PARAMETER_MODE_CLI, "eop_urlpath");
  cpl_parameter_disable (par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append (self, par);

  /* parameter requested by SDG */
  par = cpl_parameter_new_value ("fors.eop.eop_usertag",
				 CPL_TYPE_STRING,
				 "TAG provided by the user",
				 "fors.eop",
				 "None");
  cpl_parameter_set_alias (par, CPL_PARAMETER_MODE_CLI, "eop_usertag");
  cpl_parameter_disable (par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append (self, par);

  /* setting the mandatory INSTRUME keyword for the ESO archive */
  par = cpl_parameter_new_value ("fors.eop.eop_instrument",
				 CPL_TYPE_STRING,
				 "Product header keyword value for INSTRUME",
				 "fors.eop",
				 "FORS2");
  cpl_parameter_set_alias (par, CPL_PARAMETER_MODE_CLI, "eop_instrument");
  cpl_parameter_disable (par, CPL_PARAMETER_MODE_ENV);
  cpl_parameterlist_append (self, par);


  return CPL_ERROR_NONE;
}

static cpl_error_code eop_compute_qc (cpl_table * eop_table,
				     cpl_propertylist* header)
{
  double mjd_start;
  double mjd_lastprediction = 0.;
  int null;
  double mjd_lastfinal = 0.;

  mjd_start = cpl_table_get_double (eop_table, "MJD", 0, &null);
  for(int i = 0 ; i < cpl_table_get_nrow(eop_table);  i++)
    {
      const char * flag = cpl_table_get_string(eop_table, "FLAG", i);
      if(!strncmp(flag, "I", 1))
	mjd_lastfinal = cpl_table_get_double(eop_table, "MJD", i, &null);
      if(!strncmp(flag, "P", 1))
	mjd_lastprediction = cpl_table_get_double(eop_table, "MJD", i, &null);
    }

  cpl_msg_info (cpl_func, "QC EOP MJD START = %.3f", mjd_start);
  cpl_msg_info (cpl_func, "QC EOP MJD LAST FINAL = %.3f", mjd_lastfinal);
  cpl_msg_info (cpl_func, "QC EOP MJD LAST PREDICTION = %.3f", mjd_lastprediction);

  cpl_propertylist_append_double (header, "ESO QC EOP MJD START", mjd_start);
  cpl_propertylist_append_double (header, "ESO QC EOP MJD LAST FINAL", mjd_lastfinal);
  cpl_propertylist_append_double (header, "ESO QC EOP MJD LAST PREDICTION", mjd_lastprediction);
  cpl_propertylist_append_double (header, "MJD-OBS", mjd_lastfinal);

  return CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Implement the recipe functionality
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int fors_eop(cpl_frameset            * frameset,
		     const cpl_parameterlist * parlist)
{
  cpl_propertylist    *   applist;
  char                *   filename;


  /* Use the errorstate to detect an error in a function that does not
       return an error code. */
  cpl_errorstate          prestate = cpl_errorstate_get();

  /*** Get input parameter from the cpl_paramterlist ***/
  const cpl_parameter *par = NULL;

  /* Retrieving eop_host */
  par = cpl_parameterlist_find_const(parlist, "fors.eop.eop_host");
  const char*  eop_host = cpl_parameter_get_string(par);

  /* Retrieving eop_urlpath */
  par = cpl_parameterlist_find_const(parlist, "fors.eop.eop_urlpath");
  const char*  eop_urlpath= cpl_parameter_get_string(par);

  /* Retrieving eop_usertag */
  par = cpl_parameterlist_find_const(parlist, "fors.eop.eop_usertag");
  const char*  eop_usertag= cpl_parameter_get_string(par);

  /* Retrieving eop_instrument */
  par = cpl_parameterlist_find_const(parlist, "fors.eop.eop_instrument");
  const char*  eop_instrument= cpl_parameter_get_string(par);

  if (!cpl_errorstate_is_equal(prestate)) {
      return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
					"Could not retrieve the input "
					"parameters");
  }


  /* Retrieve EOP file from the site. The data are stored in the buffer
     eop_data, which should be freed with free() function */
  char * eop_data;
  size_t  data_length;
  cpl_msg_info (cpl_func, "Retrieving EOP file ");
  //eop_data = esotk_eop_download_finals2000A (eop_host, eop_urlpath, &data_length);

  const char * url = cpl_sprintf("%s%s", eop_host, eop_urlpath);
  eop_data = hdrl_download_url_to_buffer (url, &data_length);
  cpl_free((char *)url);

  if (eop_data == NULL || !cpl_errorstate_is_equal(prestate)) {
      return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
					"Could not download data from server");
  }

  /* Convert to a CPL_TABLE */
  cpl_msg_info (cpl_func, "Convert EOP data to cpl_table");
  cpl_table * eop_table = hdrl_eop_data_totable (eop_data, (cpl_size)data_length);

  /* Check for a change in the CPL error state */
  cpl_ensure_code(cpl_errorstate_is_equal(prestate), cpl_error_get_code());

  applist = cpl_propertylist_new();

  /* Add the product category  */
  cpl_propertylist_append_string (applist, CPL_DFS_PRO_CATG, "EOP_PARAM");
  cpl_propertylist_append_string (applist, "ESO PRO TECH", "CATALOG");
  cpl_propertylist_append_string (applist, "ESO PRO TYPE", "IERS");

  /* Add a QC parameter  */
  eop_compute_qc (eop_table, applist);

  /* Add INSTRUME keyword requested by the ESO archive */
  cpl_propertylist_append_string (applist, "INSTRUME", eop_instrument);

  /* Add Usertag requested by SDG */
  cpl_propertylist_append_string (applist, "ESO DRS USERTAG", eop_usertag);

  /* Saving the product */
  filename = cpl_sprintf("%s", "fors_eop_param.fits");
  cpl_table_save (eop_table, applist, NULL, filename, CPL_IO_CREATE);

  cpl_msg_info (cpl_func,"Update the frameset");

  /* Updating the frameset */
  cpl_frame * product_frame = cpl_frame_new();
  cpl_frame_set_filename (product_frame, filename);
  cpl_frame_set_tag (product_frame, CPL_DFS_PRO_CATG);
  cpl_frame_set_type (product_frame, CPL_FRAME_TYPE_TABLE);
  cpl_frame_set_group (product_frame, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level (product_frame, CPL_FRAME_LEVEL_FINAL);
  cpl_frameset_insert (frameset, product_frame);
  cpl_ensure_code (cpl_errorstate_is_equal(prestate), cpl_error_get_code());

  /* Cleanup */
  cpl_propertylist_delete(applist);
  cpl_table_delete(eop_table);
  free(eop_data);
  cpl_free(filename);

  return (int)cpl_error_get_code();
}


