/*
 * This file is part of the ESO X-Shooter Pipeline
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/* xshooter headers */
//#include <fors_error.h>
//#include <fors_utils.h>

/* Molecfit Model */
#include "fors_molecfit_correct.h"
#include <mf_wrap_config.h>
#include <telluriccorr.h>
//#include <mf_spectrum.h>
//#include <mf_wrap.h>
/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
                             Recipe Defines
 ---------------------------------------------------------------------------*/

#define RECIPE_ID "fors_molecfit_correct"
#define RECIPE_AUTHOR "N. Fernando, B. Miszalski"
#define RECIPE_CONTACT "nuwanthika.fernando@partner.eso.org"

/*---------------------------------------------------------------------------
                            Functions prototypes
 ---------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

int fors_molecfit_correct_create(cpl_plugin *);
int fors_molecfit_correct_exec(cpl_plugin *);
int fors_molecfit_correct_destroy(cpl_plugin *);

/* The actual executor function */
//int fors_molecfit_correct(cpl_frameset *frameset, const cpl_parameterlist  *parlist);

/*----------------------------------------------------------------------------*/
/**
 *                 static variables
 */
/*----------------------------------------------------------------------------*/

char fors_molecfit_correct_description_short[] =
"Applies molecfit_correct";

char fors_molecfit_correct_description[] =
"Applies molecfit_correct";

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup molecfit_correct  It runs Molecfit on a generic input spectrum file to compute an atmospheric model.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, -1 otherwise

  Create the recipe instance and make it available to the application using
  the interface. This function is exported.
 */
/*--------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist *list) {
  cpl_recipe *recipe = NULL;
  cpl_plugin *plugin = NULL;

  recipe = cpl_calloc(1, sizeof(*recipe));
  if ( recipe == NULL ){
    return -1;
  }

  plugin = &recipe->interface ;

  cpl_plugin_init(plugin,
                  CPL_PLUGIN_API,                   /* Plugin API */
                  FORS_BINARY_VERSION,            /* Plugin version */
                  CPL_PLUGIN_TYPE_RECIPE,           /* Plugin type */
                  RECIPE_ID,                        /* Plugin name */
                  fors_molecfit_correct_description_short, /* Short help */
                  fors_molecfit_correct_description,   /* Detailed help */
                  RECIPE_AUTHOR,                    /* Author name */
                  RECIPE_CONTACT,                   /* Contact address */
                  "",//fors_get_license(),                /* Copyright */
                  fors_molecfit_correct_create,
                  fors_molecfit_correct_exec,
                  fors_molecfit_correct_destroy);

  cpl_pluginlist_append(list, plugin);

  return (cpl_error_get_code() != CPL_ERROR_NONE);
 }

/*--------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using
  the interface.

 */
/*--------------------------------------------------------------------------*/

int fors_molecfit_correct_create(cpl_plugin *plugin){
  cpl_recipe *recipe = NULL;
  /*fors_clipping_param detarc_clip_param =  {2.0, 0, 0.7, 0, 0.3};
  fors_detect_arclines_param detarc_param =
  {6, 3, 0, 5, 4, 1, 5, 5.0,
    XSH_GAUSSIAN_METHOD, FALSE};
  fors_dispersol_param dispsol_param = { 4, 5 } ; 
  char paramname[256];
  cpl_parameter* p=NULL;
  int ival=DECODE_BP_FLAG_DEF;
  */

  /* Reset library state */
  //fors_init();

  /* Check input */
  //assure( plugin != NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

  /* Get the recipe out of the plugin */
  //assure( cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE,
  //        CPL_ERROR_TYPE_MISMATCH,
  //        "Plugin is not a recipe");

  recipe = (cpl_recipe *)plugin;

  /* Create the parameter list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  //assure( recipe->parameters != NULL,
  //        CPL_ERROR_ILLEGAL_OUTPUT,
  //        "Memory allocation failed!");

  //fors_molecfit_correct parameters

  //column_wave
  fors_parameters_new_string(recipe->parameters,RECIPE_ID,
  "COLUMN_WAVE","WAVE",
  "In the case of fits binary science input: name of the column in the input that identifies the wavelength.");

  //column_flux
  fors_parameters_new_string(recipe->parameters,RECIPE_ID,
  "COLUMN_FLUX","FLUX",
  "In the case of fits binary science input: name of the column in the input that identifies the flux.");

  //column_dflux
  fors_parameters_new_string(recipe->parameters,RECIPE_ID,
  "COLUMN_DFLUX","ERR",
  "In the case of fits binary science input: name of the column in the input that identifies the flux errors.");

  //threshold 
  fors_parameters_new_double(recipe->parameters,RECIPE_ID,
  "THRESHOLD",0.01,
  "Use this value when the transmission function is lower than the specified threshold.");

  cleanup:
    if ( cpl_error_get_code() != CPL_ERROR_NONE ){
      //fors_error_dump(CPL_MSG_ERROR);
      return 1;
    }
    else {
      return 0;
    }
}


/*--------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*--------------------------------------------------------------------------*/

int fors_molecfit_correct_exec(cpl_plugin *plugin) {
  cpl_recipe *recipe = NULL;

  /* Check parameter */
  //assure( plugin != NULL, CPL_ERROR_NULL_INPUT, "Null plugin" );

  /* Get the recipe out of the plugin */
  //assure( cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE,
  //        CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

  recipe = (cpl_recipe *)plugin;
  /* Check recipe */
  fors_molecfit_correct( recipe->frames, recipe->parameters);

  cleanup:
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      //fors_error_dump(CPL_MSG_ERROR);
      cpl_error_reset();
      return 1;
    }
    else {
      return 0;
    }
}

/*--------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*--------------------------------------------------------------------------*/
int fors_molecfit_correct_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe = NULL;

    /* Check parameter */
    //assure( plugin != NULL, CPL_ERROR_NULL_INPUT, "Null plugin" );

    /* Get the recipe out of the plugin */
    //assure( cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE,
    //        CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

    recipe = (cpl_recipe *)plugin;

    //fors_free_parameterlist(&recipe->parameters);

  cleanup:
    if (cpl_error_get_code() != CPL_ERROR_NONE)
        {
            return 1;
        }
    else
        {
            return 0;
        }
}

int fors_molecfit_correct(cpl_frameset *frameset, const cpl_parameterlist  *parlist){

  cpl_frameset_dump(frameset,NULL);

  /* Get initial errorstate */
  cpl_error_ensure(frameset && parlist, CPL_ERROR_NULL_INPUT, return CPL_ERROR_NULL_INPUT, "NULL input : frameset and/or parlist");
  cpl_errorstate initial_errorstate = cpl_errorstate_get();
  cpl_error_code err = CPL_ERROR_NONE;//molecfit_check_and_set_groups(frameset);

  cpl_parameterlist* ilist = cpl_parameterlist_new();
  cpl_parameterlist* iframelist = cpl_parameterlist_new();
  // get instrument specific parameter defaults
  err = fors_molecfit_correct_config(frameset,parlist,ilist,iframelist);

  const char* input_name = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"INPUTNAME"));
  const char* is_sedcorr = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"SEDCORR"));
  const char* obstype = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"OBSTYPE"));
  const char* obsmode = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"OBSMODE"));
  //const char* is_idp = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"IDP"));
  //const char* fname = cpl_parameter_get_string(cpl_parameterlist_find(iframelist,"INPUTFILENAME"));


  /* Check frameset TAGS */
    //setup some necessary frameset tags 
    cpl_frame* f = cpl_frameset_find(frameset,input_name);
    if(f){
      cpl_frame_set_group(f,CPL_FRAME_GROUP_RAW);
    }
    const char* tcor_tag = "TELLURIC_CORR";//mf_wrap_tag_suffix("TELLURIC_CORR",arm,CPL_FALSE);
    f = cpl_frameset_find(frameset,tcor_tag);
    if(f){
      cpl_frame_set_group(f,CPL_FRAME_GROUP_CALIB);
    }
    //cpl_frameset_dump(frameset,NULL);

    if(strstr(input_name,"MAPPED")){
        cpl_msg_info(cpl_func,"Input is a 2D frame");
    }


    cpl_parameterlist* mergedlist = cpl_parameterlist_new();

    cpl_msg_info(cpl_func,"calling mf_wrap_merge_parameterlists");
    err = mf_wrap_merge_parameterlists(ilist, parlist,mergedlist);

    /* Recipe Parameters : Need scientific_header_primary */
    molecfit_correct_parameter *parameters = NULL;
    if (!err) {

        /* Get recipe parameters and update the molecfit default configuration */
        cpl_msg_info(cpl_func, "Load 'MOLECFIT_CORRECT' recipe parameters ...");
        parameters = mf_wrap_config_corr_init(frameset, mergedlist,NULL);

        if (!parameters) err = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Illegal INPUT PARAMETERS!");
        else             err = cpl_error_get_code();
    }


  /* Load TAG = MAPPING_CORRECT */
  cpl_table *mapping_correct = NULL;
  if (!err) {

      if (parameters->mapping_correct_table) {
          mapping_correct = cpl_table_duplicate(parameters->mapping_correct_table);
      } else {
          const char* tag_name = mf_wrap_tag_suffix(MOLECFIT_MAPPING_CORRECT,NULL,CPL_FALSE);
          cpl_msg_info (cpl_func, "Loading %s cpl_table", tag_name);
            mapping_correct = mf_wrap_load_unique_table(frameset, tag_name);
      }

      if (     !mapping_correct                                                           ) err = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,      "Illegal INPUT MAPPING_CORRECT -> Not input!");
      else if (cpl_table_get_column_min(mapping_correct, MOLECFIT_MAPPING_CORRECT_EXT) < 0) err = cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Illegal INPUT MAPPING_CORRECT -> Minimum column < 0!");
      else                                                                                  err = cpl_error_get_code();
  }

  /*** Correct SCIENCE_CALCTRANS/SCIENCE spectra ***/
  if (!err) {

      cpl_size n_frames = cpl_frameset_get_size(frameset);
      for (cpl_size frame = 0; frame < n_frames && !err; frame++) {

         /* Check all inputs in frameset to get only the SCIENCE_CALCTRANS/SCIENCE frames */
         cpl_frame  *frame_data = cpl_frameset_get_position(frameset, frame);
         const char *tag        = cpl_frame_get_tag(frame_data);
         if( strstr(tag,"MAPPED_FLUX_SCI_MOS") ||  
              strstr(tag,"MAPPED_FLUX_SCI_MXU") ||  
              strstr(tag,"MAPPED_SCI_MOS") ||  
              strstr(tag,"MAPPED_SCI_MXU")  ){
            cpl_msg_error(cpl_func,"MAPPED_(SCI|FLUX)_(MOS|MXU) input data are not yet supported\n");
            return CPL_ERROR_ILLEGAL_INPUT;
         }
         /* Handle longslit modes */
         if(strstr(tag,"MAPPED_FLUX_SCI_LSS") ||  
              strstr(tag,"MAPPED_SCI_LSS")
            ){
             /* Create a new independent cpl_frameset for this concrete input DATA frame */
             cpl_frameset *frameset_output = cpl_frameset_new();
             cpl_frameset_insert(frameset_output, cpl_frame_duplicate(frame_data));

             /* Load every frame in the input */
             const char    *filename = cpl_frame_get_filename(frame_data);
             mf_wrap_fits *data     = mf_wrap_fits_load(filename, CPL_FALSE);

             /* Check DATA */
             if (!data) err = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Illegal INPUT %s cpl_frame!", tag);
             else       err = cpl_error_get_code();
               
             int ext = 0;
             int null;
             data->v_ext[ext].spectrum_head = cpl_propertylist_duplicate(data->v_ext[ext].header);

             /* Get TELLURIC_CORR */
             cpl_size         index_telluric_corr_ext = cpl_table_get(mapping_correct, MOLECFIT_MAPPING_CORRECT_EXT, ext, &null);
             /* We use the same telluric_corr vector for both SCI (FLUX) and ERR (DFLUX) images */
             cpl_vector *telluric_corr          = parameters->telluric_corr->v_ext[index_telluric_corr_ext].vector;
             double threshold = cpl_parameter_get_double(cpl_parameterlist_find(mergedlist,"THRESHOLD"));
             //only apply the threshold if it is a sensible value
             if(threshold >= 0 && threshold <= 1.0) {
                cpl_size vs = cpl_vector_get_size(telluric_corr);
                //iterate over vector, applying the threshold
                for (cpl_size vdx=0;vdx < vs; vdx++){
                    if(cpl_vector_get(telluric_corr,vdx) < threshold){
                        cpl_vector_set(telluric_corr,vdx,threshold);
                    }
                }
             }
             /*Prepare the output */
             char *out_telluric_corr = NULL;
             const char *filename_base = input_name;
             if (parameters->suppress_extension) 
                 out_telluric_corr  = cpl_sprintf("%s_%lld.fits", MOLECFIT_SCIENCE_TELLURIC_CORR, frame);
             else                                
                 out_telluric_corr  = cpl_sprintf("%s_%s.fits",MOLECFIT_SCIENCE_TELLURIC_CORR, filename_base);

             if(data->n_ext > 0 && data->v_ext[0].image){
                cpl_propertylist* img_hdr = cpl_propertylist_duplicate(data->v_ext[0].header);
                cpl_image* img_corr = fors_correct_image(data->v_ext[0].image,telluric_corr,img_hdr,parameters);
                if(!img_corr){
                    cpl_msg_error(cpl_func,"Error creating corrected image: %s \n",cpl_error_get_message());
                    return CPL_ERROR_ILLEGAL_OUTPUT;
                }
                cpl_propertylist_update_string(img_hdr, CPL_DFS_PRO_CATG, MOLECFIT_SCIENCE_TELLURIC_CORR);
                
                err = cpl_dfs_save_image(frameset, NULL, parlist, frameset_output, NULL, img_corr, CPL_TYPE_DOUBLE, RECIPE_ID, img_hdr, NULL, PACKAGE "/" PACKAGE_VERSION, out_telluric_corr);
                cpl_msg_info(cpl_func, "Writing to disk [%s] the corrected 2D image ext: %d of %s (PRO.CATG=%s)", 
                    out_telluric_corr, 0, data->filename, MOLECFIT_SCIENCE_TELLURIC_CORR);

                if(img_corr) cpl_image_delete(img_corr);
                if(img_hdr) cpl_propertylist_delete(img_hdr);

             }
             if(data->n_ext > 1 && data->v_ext[1].image){
                cpl_propertylist* err_hdr = cpl_propertylist_duplicate(data->v_ext[1].header);
                cpl_image* err_corr = fors_correct_image(data->v_ext[1].image,telluric_corr,cpl_propertylist_duplicate(err_hdr),parameters);
                if(!err_corr){
                    cpl_msg_warning(cpl_func,"Could not create corrected image error ext 1\n"); 
                } else {
                    cpl_image_save(err_corr, out_telluric_corr, CPL_TYPE_DOUBLE, err_hdr, CPL_IO_EXTEND);
                    cpl_msg_info(cpl_func, "Appending to disk [%s] the corrected 2D image ext: %d of %s (PRO.CATG=%s)", 
                        out_telluric_corr, 1, data->filename, MOLECFIT_SCIENCE_TELLURIC_CORR);
                }
                if(err_corr) cpl_image_delete(err_corr);
                if(err_hdr) cpl_propertylist_delete(err_hdr);
             }
                /* Cleanup */
                if (frameset_output) cpl_frameset_delete(frameset_output);
                if (data) mf_wrap_fits_delete(data);
                cpl_free(out_telluric_corr);
         }
         //Handle IDP products as per usual approach
         if (!strncmp(tag, "REDUCED_IDP_", 12)) {

             /* Create a new independent cpl_frameset for this concrete input DATA frame */
             cpl_frameset *frameset_output = cpl_frameset_new();
             cpl_frameset_insert(frameset_output, cpl_frame_duplicate(frame_data));

             /* Load every frame in the input */
             const char    *filename = cpl_frame_get_filename(frame_data);
             mf_wrap_fits *data     = mf_wrap_fits_load(filename, CPL_FALSE);

             /* Check DATA */
             if (!data) err = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Illegal INPUT %s cpl_frame!", tag);
             else       err = cpl_error_get_code();

             /* Convert to table for execute molecfit */
             //if (!err) err = molecfit_data_convert_to_table( data,
             if (!err) err = mf_wrap_data_convert_to_table( data,
                                                             parameters->chip_extensions,
                                                             parameters->use_only_input_pri_ext,
                                                             parameters->dflux_extension_data,
                                                             parameters->mask_extension_data,
                                                             parameters->column_lambda,
                                                             parameters->column_flux,
                                                             parameters->column_dflux,
                                                             parameters->column_mask);

             /* Execution extensions */
             cpl_size n_ext;
             if (     parameters->use_only_input_pri_ext) n_ext = 1;
             else if (parameters->chip_extensions       ) n_ext = data->v_ext[0].spectrum_data ? 1 : 2;
             else n_ext = data->n_ext;

             /* Create output SPECTRUM SCIENCE FITS file */
             mf_wrap_fits *spectrum_corr_data = NULL;
             //molecfit_fits *spectrum_corr_data = NULL;
             if (!err) {

                 char *spectrum_filename = cpl_sprintf("SPECTRUM_%s.fits", input_name);//data->filename);
                 //char *spectrum_filename = cpl_sprintf("SPECTRUM_%s", data->filename);
                 //char *spectrum_filename = cpl_sprintf("SPECTRUM_%s", basename(data->filename));
                 //spectrum_corr_data = molecfit_fits_create(spectrum_filename, molecfit_fits_file_image_1D, molecfit_fits_vector, data->n_ext);
                 spectrum_corr_data = mf_wrap_fits_create(spectrum_filename, mf_wrap_fits_file_image_1D, mf_wrap_fits_vector, data->n_ext);
                 cpl_free(spectrum_filename);

                 if (spectrum_corr_data) err = cpl_error_get_code();
                 else                    err = cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT, "I cannot create the FITS output file %s!", MOLECFIT_SPECTRUM_TELLURIC_CORR);

                 /* Create dummy extensions */
                 for (cpl_size ext = 0; ext < data->n_ext; ext++) {
                     if (data->v_ext[ext].spectrum_data) {
                         cpl_size size = -1;
                         //if (     data->format == molecfit_fits_file_table   ) {
                         if (     data->format == mf_wrap_fits_file_table) {
                             if (   cpl_propertylist_get_int(data->v_ext[ext].header, MOLECFIT_FITS_KEYWORD_NAXIS1) >  1
                                 && cpl_propertylist_get_int(data->v_ext[ext].header, MOLECFIT_FITS_KEYWORD_NAXIS2) == 1) {
                                 size = cpl_propertylist_get_int(data->v_ext[ext].header, MOLECFIT_FITS_KEYWORD_NAXIS1);
                             } else {
                                 size = cpl_table_get_nrow(data->v_ext[ext].table);
                             }
                         }
                         //else if (data->format == molecfit_fits_file_image_1D) size = cpl_vector_get_size(   data->v_ext[ext].vector);
                         //else if (data->format == molecfit_fits_file_image_2D) size = cpl_image_get_size_x(  data->v_ext[ext].image );
                         //else if (data->format == molecfit_fits_file_image_3D) size = cpl_imagelist_get_size(data->v_ext[ext].cube  );
                         else if (data->format == mf_wrap_fits_file_image_1D) size = cpl_vector_get_size(   data->v_ext[ext].vector);
                         else if (data->format == mf_wrap_fits_file_image_2D) size = cpl_image_get_size_x(  data->v_ext[ext].image );
                         else if (data->format == mf_wrap_fits_file_image_3D) size = cpl_imagelist_get_size(data->v_ext[ext].cube  );

                         spectrum_corr_data->v_ext[ext].vector = cpl_vector_new(size);
                     }
                 }
             }

             /* Loop over the data extensions */
             int null;
             for (cpl_size ext = 0; ext < n_ext && !err; ext++) {

                 /* Create spectrum header output */
                 if (!err) spectrum_corr_data->v_ext[ext].header = cpl_propertylist_duplicate(data->v_ext[ext].spectrum_head);

                 /* Get data extension : Only manage extensions with data */
                 if (data->v_ext[ext].spectrum_data) {

                     /* Get TELLURIC_CORR */
                     cpl_size         index_telluric_corr_ext = cpl_table_get(mapping_correct, MOLECFIT_MAPPING_CORRECT_EXT, ext, &null);
                     cpl_vector *telluric_corr          = parameters->telluric_corr->v_ext[index_telluric_corr_ext].vector;
                     double threshold = cpl_parameter_get_double(cpl_parameterlist_find(mergedlist,"THRESHOLD"));
                     //only apply the threshold if it is a sensible value
                     if(threshold >= 0 && threshold <= 1.0) {
                        cpl_size vs = cpl_vector_get_size(telluric_corr);
                        //iterate over vector, applying the threshold
                        for (cpl_size vdx=0;vdx < vs; vdx++){
                            if(cpl_vector_get(telluric_corr,vdx) < threshold){
                                cpl_vector_set(telluric_corr,vdx,threshold);
                            }
                        }
                     }
                     //debug - shows that the threshold is applied
                     //const char* vfname = cpl_sprintf("vec%lld.fits",ext);
                     //cpl_vector_save(telluric_corr,vfname,CPL_TYPE_DOUBLE,NULL,CPL_IO_CREATE);


                     /* Check DATA with TELLURIC_CORR */
                     if (        !telluric_corr                                                                          ) err = cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "I cannot MATCH the TELLURIC_CORR extension with the %s extension for correct!", tag);
                     else if (   cpl_table_get_nrow(data->v_ext[ext].spectrum_data) <= 0
                              || cpl_table_get_nrow(data->v_ext[ext].spectrum_data) != cpl_vector_get_size(telluric_corr)) err = cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "I cannot MATCH the rows between the TELLURIC_CORR extension with the %s extension for correct!", tag);

                     if (!err) {

                         /* Create input molecfit data */
                         cpl_msg_info(cpl_func, "Create input molecfit cpl_table data : %s [%s], ext=%lld", tag, data->filename, ext);
                         cpl_table *molecfit_data = mf_spectrum_create_input_to_correct( data->v_ext[ext].spectrum_data,
                                                                                         parameters->column_lambda,
                                                                                         parameters->column_flux,
                                                                                         parameters->column_dflux,
                                                                                         parameters->column_mask,
                                                                                         telluric_corr);

                         if (molecfit_data) err = cpl_error_get_code();
                         else               err = cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "I cannot Create the input molecfit SPECTRUM_TABLE to correct!");


                         /* Apply correction to molecfit data */
                         if (!err) {
                             cpl_msg_info(cpl_func, "File : %s, ext=%lld : Apply correction", data->filename, ext);
                             err = mf_calctrans_correct_spectrum(molecfit_data, MF_PARAMETERS_TRANSMISSION_TRUE);
                         }

                         /* Create spectrum data output */
                         if (!err) {

                             /* Copy data */
                             err = mf_wrap_data_update_vector(molecfit_data, parameters->chip_extensions, spectrum_corr_data, ext);
                             //err = molecfit_data_update_vector(molecfit_data, parameters->chip_extensions, spectrum_corr_data, ext);

                             /* Copy additional if chip combined */
                             if (parameters->chip_extensions) {
                                 for (cpl_size i = ext + 1; i < data->n_ext; i++) {
                                     spectrum_corr_data->v_ext[i].header = cpl_propertylist_duplicate(data->v_ext[i].spectrum_head);
                                 }
                             }
                         }

                         /* Update Input data with telluric correction */
                         if (!err) {
                             if (     data->format == mf_wrap_fits_file_table   ) err = mf_wrap_data_update_table( molecfit_data, parameters->chip_extensions, data, ext, parameters->column_flux);
                             else if (data->format == mf_wrap_fits_file_image_1D) err = mf_wrap_data_update_vector(molecfit_data, parameters->chip_extensions, data, ext);
                             else if (data->format == mf_wrap_fits_file_image_2D) err = mf_wrap_data_update_image( molecfit_data, parameters->chip_extensions, data, ext);
                             else if (data->format == mf_wrap_fits_file_image_3D) err = mf_wrap_data_update_cube(  molecfit_data, parameters->chip_extensions, data, ext);
                             /*if (     data->format == molecfit_fits_file_table   ) err = molecfit_data_update_table( molecfit_data, parameters->chip_extensions, data, ext, parameters->column_flux);
                             else if (data->format == molecfit_fits_file_image_1D) err = molecfit_data_update_vector(molecfit_data, parameters->chip_extensions, data, ext);
                             else if (data->format == molecfit_fits_file_image_2D) err = molecfit_data_update_image( molecfit_data, parameters->chip_extensions, data, ext);
                             else if (data->format == molecfit_fits_file_image_3D) err = molecfit_data_update_cube(  molecfit_data, parameters->chip_extensions, data, ext);
                             */
                         }

                         /* Cleanup */
                         if (molecfit_data) cpl_table_delete(molecfit_data);
                     }
                 }
             }

             /* Save update OUT : SPECTRUM and DATA */
             if (!err) {

                 //const char *filename_base = basename(data->filename);
                 const char *filename_base = input_name;//data->filename;//basename(data->filename);
                 //const char* output_tag = mf_wrap_tag_suffix(MOLECFIT_SPECTRUM_TELLURIC_CORR,arm,CPL_FALSE);
                 //TODO: still to sort output filenames/tags etc.

                 char *spectrum_tag = cpl_sprintf("%s%s", MOLECFIT_SPECTRUM_TELLURIC_CORR, (!strcmp(is_sedcorr, "TRUE") ? "_SEDCORR" : ""));
                 /* Save input SPECTRUM extracted --> Corrected */
                 char *out_telluric_corr = NULL;
                 //const char* output_tag = mf_wrap_tag_suffix(MOLECFIT_SPECTRUM_TELLURIC_CORR,arm,CPL_FALSE);
                 if (parameters->suppress_extension) out_telluric_corr  = cpl_sprintf("%s_%lld.fits", spectrum_tag, frame        );
                 else                                out_telluric_corr  = cpl_sprintf("%s_%s.fits",   spectrum_tag, filename_base);

                 //cpl_msg_info(cpl_func, "Writing on disk [%s] %s input FITS corrected on SPECTRUM format --> %s (PRO.CATG=%s)", data->filename, tag, out_telluric_corr, spectrum_tag);
                 err = mf_wrap_fits_write(frameset, frameset_output, parlist, RECIPE_NAME, spectrum_tag, spectrum_corr_data, out_telluric_corr);

                 cpl_free(out_telluric_corr);
                 cpl_free(spectrum_tag);


                 /* Save input SCIENCE --> Corrected */
                 if (!err) {

                     char *science_tag = cpl_sprintf("%s_%s%s_TELLURIC_CORR", obstype, obsmode, (!strcmp(is_sedcorr, "TRUE") ? "_SEDCORR" : ""));
                     char *out_data = NULL;
                     if (parameters->suppress_extension) out_data = cpl_sprintf("%s_%lld.fits", science_tag, frame        );
                     else                                out_data = cpl_sprintf("%s_%s.fits",   science_tag, filename_base);

                     cpl_msg_info(cpl_func, "Writing on disk [%s] %s input FITS corrected on the same format --> %s (PRO.CATG=%s)", data->filename, tag, out_data, science_tag);
                     err = mf_wrap_fits_write(frameset, frameset_output, parlist, RECIPE_NAME, science_tag, data, out_data);
                     //err = molecfit_fits_write(frameset, frameset_output, parlist, RECIPE_NAME, science_tag, data, out_data);

                     cpl_free(out_data);
                     cpl_free(science_tag);
                 }
             }

             /* Cleanup */
             if (frameset_output   ) cpl_frameset_delete(frameset_output);
             if (data              ) mf_wrap_fits_delete(data);
             if (spectrum_corr_data) mf_wrap_fits_delete(spectrum_corr_data);
             //if (data              ) molecfit_fits_delete(data);
             //if (spectrum_corr_data) molecfit_fits_delete(spectrum_corr_data);
         }
      }
  }

  /* Cleanup */
  if (parameters     ) molecfit_correct_parameter_delete( parameters     );
  if (mapping_correct) cpl_table_delete(                        mapping_correct);


  /* Check Recipe status and end */
  if (!err && cpl_errorstate_is_equal(initial_errorstate)) {
      cpl_msg_info(cpl_func,"Recipe completed successfully!");
  } else {
      /* Dump the error history */
      cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
      cpl_msg_error(cpl_func,"Recipe failed!, error(%d)=%s", err, cpl_error_get_message());
  }

  return err;
}


cpl_error_code fors_molecfit_correct_config(cpl_frameset *frameset, const cpl_parameterlist  *parlist,
    		cpl_parameterlist* ilist, cpl_parameterlist* iframelist){

        cpl_parameterlist* iframe = cpl_parameterlist_new();
        cpl_error_code err= CPL_ERROR_NONE;

        // FIXME this code is duplicated between molecfit recipes
        const char *types[2] = {"SCI", "STD"};
        const char *modes[3] = {"LSS", "MOS", "MXU"};
        const char *suffixes[2] = {"", "_SEDCORR"};
        int i, j, k, found;
        found = 0;
        i = 0;
        while (!found && (i < 2)) {
            j = 0;
            while (!found && (j < 3)) {
                k = 0;
                while (!found && (k < 2)) {
                    char* tag = cpl_sprintf("REDUCED_IDP_%s_%s%s", types[i], modes[j], suffixes[k]);
                    cpl_msg_info(cpl_func, "Checking for tag %s", tag);
                    cpl_frame* input_frame = cpl_frameset_find(frameset, tag);
                    if(input_frame){
                        const char* filename =  cpl_frame_get_filename(input_frame);
                        cpl_msg_info(cpl_func,"Found FILENAME %s for tag %s",filename,tag);
                        cpl_parameterlist_append(iframe,cpl_parameter_new_value("INPUTNAME",CPL_TYPE_STRING,NULL,NULL,tag));
                        cpl_parameterlist_append(iframe,cpl_parameter_new_value("OBSTYPE",CPL_TYPE_STRING,NULL,NULL,types[i]));
                        cpl_parameterlist_append(iframe,cpl_parameter_new_value("OBSMODE",CPL_TYPE_STRING,NULL,NULL,modes[j]));
                        cpl_parameterlist_append(iframe,cpl_parameter_new_value("IDP",CPL_TYPE_STRING,NULL,NULL,"TRUE"));
                        if (strlen(suffixes[k]) > 0) {
                            cpl_parameterlist_append(iframe,cpl_parameter_new_value("SEDCORR",CPL_TYPE_STRING,NULL,NULL,"TRUE"));
                        } else {
                            cpl_parameterlist_append(iframe,cpl_parameter_new_value("SEDCORR",CPL_TYPE_STRING,NULL,NULL,"FALSE"));
                        }
                        cpl_parameterlist_append(iframe,cpl_parameter_new_value("INPUTFILENAME",CPL_TYPE_STRING,NULL,NULL,filename));
                        found = 1;
                    }
                    cpl_free(tag);
                    ++k;
                }
                ++j;
            }
            ++i;
        }

        /* Add support for 2D LSS input */
        const char* lss_tag[2] = {"MAPPED_SCI_LSS","MAPPED_FLUX_SCI_LSS"};
        for (int t=0;t<2;t++){
            cpl_frame* input_frame = cpl_frameset_find(frameset, lss_tag[t]);
            if(input_frame){
                const char* filename =  cpl_frame_get_filename(input_frame);
                cpl_msg_info(cpl_func,"Found FILENAME %s for tag %s",filename,lss_tag[t]);
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("INPUTNAME",CPL_TYPE_STRING,NULL,NULL,lss_tag[t]));
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("OBSTYPE",CPL_TYPE_STRING,NULL,NULL,"SCI"));
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("OBSMODE",CPL_TYPE_STRING,NULL,NULL,"LSS"));
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("IDP",CPL_TYPE_STRING,NULL,NULL,"FALSE"));
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("SEDCORR",CPL_TYPE_STRING,NULL,NULL,"FALSE"));
                cpl_parameterlist_append(iframe,cpl_parameter_new_value("INPUTFILENAME",CPL_TYPE_STRING,NULL,NULL,filename));
            }
        }

        const char* is_idp = cpl_parameter_get_string(cpl_parameterlist_find(iframe,"IDP"));

        //add iframe parameters (INPUTNAME,ARM,OBSMODE,IDP) to iframelist so that we can access them from fors_molecfit_correct
        //these are not added to ilist, as they are only meant to be instrument dependent molecfit parameters
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"INPUTNAME"));
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"OBSTYPE"));
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"OBSMODE"));
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"IDP"));
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"SEDCORR"));
        err = cpl_parameterlist_append(iframelist,cpl_parameterlist_find(iframe,"INPUTFILENAME"));

        //merge1d 
        const char* MAPPING_CORRECT;
        cpl_boolean USE_PRIMARY_DATA;
        int USE_DFLUX;
        //If IDP format
        if(!strcmp(is_idp,"TRUE")){
            MAPPING_CORRECT = "0,1";
            USE_DFLUX= 0;
            USE_PRIMARY_DATA = CPL_FALSE;
        } else {
            //if not IDP format
            MAPPING_CORRECT = "1";
            USE_PRIMARY_DATA= CPL_TRUE;
            USE_DFLUX = 1;
        }
        int USE_MASK = 0;
        cpl_boolean SUPPRESS_EXTENSION = CPL_FALSE;
        cpl_boolean CHIP_EXTENSIONS= CPL_FALSE;

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(MOLECFIT_PARAMETER_MAPPING_CORRECT,CPL_TYPE_STRING,NULL,NULL,MAPPING_CORRECT));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value("USE_ONLY_INPUT_PRIMARY_DATA",CPL_TYPE_BOOL,NULL,NULL,USE_PRIMARY_DATA));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value("USE_DATA_EXTENSION_AS_DFLUX",CPL_TYPE_INT,NULL,NULL,USE_DFLUX));

        cpl_parameterlist_append(ilist,cpl_parameter_new_value(MOLECFIT_PARAMETER_CHIP_EXTENSIONS,CPL_TYPE_BOOL,NULL,NULL,CHIP_EXTENSIONS));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value(MOLECFIT_PARAMETER_SUPPRESS_EXTENSION,CPL_TYPE_BOOL,NULL,NULL,SUPPRESS_EXTENSION));
        cpl_parameterlist_append(ilist,cpl_parameter_new_value("USE_DATA_EXTENSION_AS_MASK",CPL_TYPE_INT,NULL,NULL,USE_MASK));
        return err;
}

cpl_table* fors_extract_spectrum_from_cpl_vector(
    cpl_propertylist         *header,
    const cpl_vector         *flux_in,
    const cpl_vector         *dflux_in,
    const cpl_vector         *mask_in,
    const int                chip,
    const char               *column_lambda,
    const char               *column_flux,
    const char               *column_dflux,
    const char               *column_mask)
{
  /* Check inputs */
  cpl_error_ensure(header && flux_in, CPL_ERROR_NULL_INPUT,
                   return NULL, "Null inputs in parse data cpl_vector to spectrum cpl_table");
  cpl_error_ensure(   cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL1)
                   && (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD1_1) || cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CDELT1)), CPL_ERROR_NULL_INPUT,
                   return NULL, "Not correct KEYWORDS in the header [%s, %s or DEPRECATED %s]", MOLECFIT_FITS_KEYWORD_CRVAL1, MOLECFIT_FITS_KEYWORD_CD1_1,MOLECFIT_FITS_KEYWORD_CDELT1);

  /* Wavelength values: CRVAL1 (initial wavelength) and CD1_1 (step wavelength) */
  double wave_ini  = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRVAL1);
  double wave_step;
  if (cpl_propertylist_has(header,MOLECFIT_FITS_KEYWORD_CD1_1)) {
      wave_step = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CD1_1);
  } else {
      cpl_msg_warning(cpl_func,"Cannot find CD1_1 keyword. Will look for the DEPRECATED CDELT1 keyword instead!");
      wave_step = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CDELT1);
  } 
  /* No need to check for else here, as taken care of by cpl_error_ensure call above */

  double wave      = wave_ini;

  /* Create table structure for the input spectrum;  n_waves = number of wavelengths */
  double n_wave = cpl_vector_get_size(flux_in);

  /* Convert header */
  double crpix  = 1.;
  //mf_wrap_data_convert_header_Image2D_to_Spectrum is problematic for 2D images as it may not have CUNIT1
  cpl_error_code err = cpl_error_get_code();// = mf_wrap_data_convert_header_Image1D_to_Spectrum(header, n_wave, crpix);

  /* Create output table */
  cpl_table   *spectrum_table    = cpl_table_new(n_wave);
  const char  *column_lambda_str = strcmp(column_lambda, MF_PARAMETERS_COLUMN_LAMBDA_NULL) ? column_lambda : MF_PARAMETERS_COLUMN_LAMBDA_DEFAULT;
  const char  *column_flux_str   = strcmp(column_flux,   MF_PARAMETERS_COLUMN_FLUX_NULL  ) ? column_flux   : MF_PARAMETERS_COLUMN_FLUX_DEFAULT;
  const char  *column_dflux_str  = strcmp(column_dflux,  MF_PARAMETERS_COLUMN_DFLUX_NULL ) ? column_dflux  : MF_PARAMETERS_COLUMN_DFLUX_DEFAULT;
  const char  *column_mask_str   = strcmp(column_mask,   MF_PARAMETERS_COLUMN_MASK_NULL  ) ? column_mask   : MF_PARAMETERS_COLUMN_MASK_DEFAULT;

  //cpl_boolean exist_column_dflux = strcmp(column_dflux,  MF_PARAMETERS_COLUMN_DFLUX_NULL) || dflux_in;
  //cpl_boolean exist_column_mask  = strcmp(column_mask,   MF_PARAMETERS_COLUMN_MASK_NULL ) || mask_in;

  /* Create columns */
  cpl_table_new_column(                        spectrum_table, MF_COL_CHIP,       CPL_TYPE_INT   );
  cpl_table_new_column(                        spectrum_table, column_lambda_str, CPL_TYPE_DOUBLE);
  cpl_table_new_column(                        spectrum_table, column_flux_str,   CPL_TYPE_DOUBLE);
  //if (exist_column_dflux) cpl_table_new_column(spectrum_table, column_dflux_str,  CPL_TYPE_DOUBLE);
  //if (exist_column_mask ) cpl_table_new_column(spectrum_table, column_mask_str,   CPL_TYPE_INT   );

  /* Set chip */
  cpl_table_fill_column_window_int(spectrum_table, MF_COL_CHIP, 0, n_wave, chip);

  /* Set lambda array */
  for (cpl_size i = 0; i < n_wave; i++) {
      cpl_table_set_double(spectrum_table, column_lambda_str, i, wave);
      wave += wave_step;
  }

  /* Set flux array */
  cpl_table_copy_data_double(spectrum_table, column_flux_str, cpl_vector_get_data_const(flux_in));

  /* Check preState */
  if (cpl_error_get_code() != CPL_ERROR_NONE || err != CPL_ERROR_NONE) {
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                   "Parse 1D spectrum form IMAGE1D failed!");
      if (spectrum_table) cpl_table_delete(spectrum_table);
      return NULL;
  }

  return spectrum_table;
}

cpl_image* fors_correct_image(
    cpl_image* img, 
    cpl_vector* telluric_corr,
    cpl_propertylist* hdr,
    molecfit_correct_parameter *parameters
){
    cpl_size nx = cpl_image_get_size_x(img);
    cpl_size ny = cpl_image_get_size_y(img);
    cpl_error_code err = CPL_ERROR_NONE;
    int null;
    /* Check DATA with TELLURIC_CORR */
    if (!telluric_corr) {
        cpl_msg_error(cpl_func,"The telluric_corr vector is empty\n");
        return NULL;
    }
    else if (nx <= 0 || nx != cpl_vector_get_size(telluric_corr)) {
        cpl_msg_error(cpl_func,"Mismatch between image dimension and the telluric_corr vector\n");
        return NULL;
    }
    cpl_image* out_img = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);

    for (cpl_size j = 0; j < ny; j++) {
        //Get a vector of each image row
        cpl_vector* rv = cpl_vector_new_from_image_row(img,j+1);
        cpl_table* spec_tmp = cpl_table_new(nx);
        spec_tmp = fors_extract_spectrum_from_cpl_vector(hdr,
                        rv,NULL,NULL,1,parameters->column_lambda, parameters->column_flux, "NULL","NULL"); 
        if(!spec_tmp){
            cpl_msg_error(cpl_func,"Could not extract spectrum from row vector: %s\n",cpl_error_get_message());
            return NULL;
        }
        cpl_table* molecfit_data = mf_spectrum_create_input_to_correct(spec_tmp,parameters->column_lambda,parameters->column_flux,"NULL","NULL",telluric_corr);
        if(!molecfit_data){
            cpl_msg_error(cpl_func,"Could not create input spectrum table to correct: %s\n",cpl_error_get_message());
            return NULL;
        }
        err = mf_calctrans_correct_spectrum(molecfit_data, MF_PARAMETERS_TRANSMISSION_TRUE);
        if(err){
            cpl_msg_info(cpl_func,"mf_calctrans_correct_spectrum error: %s", cpl_error_get_message());
            return NULL;
        }
        //populate the new image row 
        for (cpl_size wave = 0; wave < nx; wave++) {
           double val = cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, wave, &null);
           cpl_image_set(out_img,wave+1,j+1,val);
        }
    }
    return out_img;
}

