/* $Id: fors_response.c,v 1.42 2013-08-14 15:04:57 cgarcia Exp $
 *
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H 
#include <config.h>
#endif

#define HDRL_USE_PRIVATE

#include <cmath>
#include <stdexcept>
#include <string>
#include <sstream>
#include <iostream>
#include <vector>
#include <functional>
#include <utility>
#include <limits>
#include <valarray>
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <moses.h>
#include <fors_dfs.h>
#include <fors_tools.h>
#include <fors_qc.h>
#include <fors_header.h>
#include <fors_utils.h>
#include "fors_ccd_config.h"
#include "fors_preprocess.h"
#include "fors_overscan.h"
#include "fors_subtract_bias.h"
#include "fors_response_utils.h"
#include "fors_detected_slits.h"
#include "fors_flat_normalise.h"
#include "fors_spec_idp.h"
#include "fors_bpm.h"
#include "global_distortion.h"
#include "slit_trace_distortion.h"

static int fors_response_create(cpl_plugin *);
static int fors_response_exec(cpl_plugin *);
static int fors_response_destroy(cpl_plugin *);
static int fors_response(cpl_parameterlist *, cpl_frameset *);

static bool fors_response_response_apply_flat_corr
(bool has_ffsed, std::string& resp_use_flat_sed, cpl_table * grism_table);

static cpl_array * get_wavpoints(const double start_w, const double end_w, const double step);
cpl_bivector * create_windows(const std::vector<double>& lines, double width, const std::vector<std::pair<double, double> >& range);
hdrl_spectrum1D * create_spectrum(
    const std::vector<double>& wave_vec,
    const std::vector<double>& flux_vec,
    const std::vector<double>& err_vec);
std::vector<double> extract_wavelength(const hdrl_spectrum1D* spectrum);
std::vector<double> extract_flux(const hdrl_spectrum1D* spectrum);
void load_from_table(cpl_table* table,
    const char* wavecol, const char* fluxcol, const char* errcol,
    std::vector<double>& wave_vec, std::vector<double>& flux_vec, std::vector<double>& err_vec);
void load_from_table_array(cpl_table* table,
    const char* wavecol, const char* fluxcol, const char* errcol,
    std::vector<double>& wave_vec, std::vector<double>& flux_vec, std::vector<double>& err_vec);

static const double STAR_MATCH_DEPSILON=0.0166667; /*60 arcsecs */

static char fors_response_description[] =
"This recipe is used to produce a response curve from an extracted\n"
"Phase3-compliant standard star spectrum. This recipe accepts both\n"
"FORS1 and FORS2 frames. A grism table (depending on the\n"
"grism/filter/detector used) may also be specified: this table contains\n"
"a default recipe parameter setting to control the way spectra are extracted\n"
"for a specific instrument mode, as it is used for automatic run of the\n"
"pipeline on Paranal and in Garching. If this table is specified, it will\n"
"modify the default recipe parameter setting, with the exception of those\n"
"parameters which have been explicitly modified on the command line.\n"
"If a grism table is not specified, the input recipe parameters values \n"
"will always be read from the command line, or from an esorex configuration\n"
"file if present, or from their generic default values (that are rarely\n"
"meaningful). In the table below the MXU acronym can be read alternatively as\n"
"MOS and LSS, depending on the instrument mode of the input data.\n"
"The atmospheric extinction table and a table with the physical fluxes of the\n"
"observed standard star must be specified in input, and a spectro-photometric\n"
"table (response curve) is created in output. This table can then be input to\n"
"the fors_science recipe.\n\n"
"For grisms which require a FLAT_SED correction, either the\n"
"FLAT_SED corrected standard spectrum should be supplied (in addition to\n"
"the non-corrected spectrum), or a FLAT_SED_MXU file should be supplied.\n\n"

"Input files:\n\n"
"  DO category:                   Type:       Explanation:         Required:\n"
"  REDUCED_IDP_STD_MXU            Calib       Extracted standard spectrum Y\n"
"  or, in case of molecfit-corrected data,\n"
"  STD_MXU_TELLURIC_CORR          Calib       Extracted standard spectrum Y\n"
"\n"
"  GRISM_TABLE                    Calib       Grism table                 .\n"
"\n"
"  DISP_COEFF_MXU                 Calib       Inverse dispersion          Y\n"
"  FLAT_SED_MXU                   Calib       Slits dispersion profile    .\n"
"  or, in case of LSS-like MOS/MXU data,\n"
"  DISP_COEFF_LONG_MXU            Calib       Inverse dispersion          Y\n"
"  FLAT_SED_LONG_MXU              Calib       Slits dispersion profile    .\n"
"\n"
"  EXTINCT_TABLE                  Calib       Atmospheric extinction      Y\n"
"  FIT_POINT_CATALOG              Calib       Fit points catalog          Y\n"
"  STD_FLUX_CATALOG               Calib       Standard star flux catalog  Y\n"
"\n"
"Output files:\n"
"\n"
"  DO category:                   Data type:  Explanation:\n"
"  SPECPHOT_TABLE                 FITS table  Efficiency and response curves\n";

#define fors_response_exit(message)            \
{                                             \
if ((const char *)message != NULL) cpl_msg_error(recipe, message);  \
cpl_table_delete(photcal);                    \
cpl_propertylist_delete(qclist);              \
cpl_propertylist_delete(header);              \
cpl_propertylist_delete(flat_sed_header);     \
cpl_image_delete(mapped_flat_sed);            \
cpl_table_delete(grism_table);                \
fors_setting_delete(&setting);                \
cpl_free(science_tag);                        \
cpl_free(sedcorr_tag);                        \
cpl_free(disp_coeff_tag);                     \
cpl_free(flat_sed_tag);                       \
cpl_msg_indent_less();                        \
return fors_get_error_for_exit();             \
}


#define fors_response_exit_memcheck(message)   \
{                                             \
if ((const char *)message != NULL) cpl_msg_info(recipe, message);   \
cpl_table_delete(photcal);                    \
cpl_propertylist_delete(qclist);              \
cpl_propertylist_delete(header);              \
cpl_propertylist_delete(flat_sed_header);     \
cpl_image_delete(mapped_flat_sed);            \
cpl_table_delete(grism_table);                \
fors_setting_delete(&setting);                \
cpl_free(science_tag);                        \
cpl_free(sedcorr_tag);                        \
cpl_free(disp_coeff_tag);                     \
cpl_free(flat_sed_tag);                       \
cpl_msg_indent_less();                        \
return 0;                                     \
}


/**
 * @brief    Build the list of available plugins, for this module. 
 *
 * @param    list    The plugin list
 *
 * @return   0 if everything is ok, -1 otherwise
 *
 * Create the recipe instance and make it available to the application 
 * using the interface. This function is exported.
 */

int cpl_plugin_get_info(cpl_pluginlist *plist)
{
    cpl_recipe *recipe = static_cast<cpl_recipe *>(cpl_calloc(1, sizeof *recipe ));
    cpl_plugin *plugin = &recipe->interface;

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    FORS_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "fors_response",
                    "Calculation of response curve",
                    fors_response_description,
                    "Jon Nielsen",
                    PACKAGE_BUGREPORT,
                    fors_get_license(),
                    fors_response_create,
                    fors_response_exec,
                    fors_response_destroy);

    cpl_pluginlist_append(plist, plugin);
    
    return 0;
}


/**
 * @brief    Setup the recipe options    
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 *
 * Defining the command-line/configuration parameters for the recipe.
 */

static int fors_response_create(cpl_plugin *plugin)
{
    cpl_recipe    *recipe;
    cpl_parameter *p;


    /* 
     * Check that the plugin is part of a valid recipe 
     */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    /* 
     * Create the parameters list in the cpl_recipe object 
     */

    recipe->parameters = cpl_parameterlist_new(); 

    p = cpl_parameter_new_value("fors.fors_response.resp_use_flat_sed",
                                CPL_TYPE_STRING,
                                "Use the flat SED to normalise the observed spectra. "
                                "Value are true, false, grism_table.",
                                "fors.fors_response",
                                "grism_table");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "resp_use_flat_sed");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);
    
    p = cpl_parameter_new_value("fors.fors_response.fit_window_pix",
                                CPL_TYPE_INT,
                                "Number of unbinned pixels to median combine around each fit point "
                                "when fitting the instrument response. Automatically adjusted for "
                                "the binning in use. (< 0: Read from grism table)",
                                "fors.fors_response",
                                -2);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fit_window_pix");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    p = cpl_parameter_new_value("fors.fors_response.velocity_wguess",
                                CPL_TYPE_DOUBLE,
                                "Wavelength (in Angstrom) of stellar feature to use "
                                "when calculating the doppler shift of the standard star."
                                "(0: No doppler shift calculated. < 0: Read from grism table)",
                                "fors.fors_response",
                                -2.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "velocity_wguess");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    return 0;
}


/**
 * @brief    Execute the plugin instance given by the interface
 *
 * @param    plugin  the plugin
 *
 * @return   0 if everything is ok
 */

static int fors_response_exec(cpl_plugin *plugin)
{
    cpl_recipe  *   recipe ;
    int             status = 1;

    /* Get the recipe out of the plugin */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin ;
    else return -1 ;

    /* Issue a banner */
    fors_print_banner();

    try
    {
        status = fors_response(recipe->parameters, recipe->frames);
    }
    catch(std::exception& ex)
    {
        cpl_msg_error(cpl_func, "Recipe error: %s", ex.what());
    }
    catch(...)
    {
        cpl_msg_error(cpl_func, "An uncaught error during recipe execution");
    }

    return status;
}


/**
 * @brief    Destroy what has been created by the 'create' function
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 */

static int fors_response_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    cpl_parameterlist_delete(recipe->parameters); 

    return 0;
}


/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    parlist     The parameters list
 * @param    frameset    The set-of-frames
 *
 * @return   0 if everything is ok
 */

static int fors_response(cpl_parameterlist *parlist, cpl_frameset *frameset)
{

    const char *recipe = "fors_response";


    /*
     * Input parameters
     */

    double      dispersion;
    double      dispersion_nm;
    const char *resp_ignore_mode;
    const char *resp_ignore_lines;
    double      startwavelength;
    double      endwavelength;
    int         fit_window_pix;
    double      velocity_wguess;
    /*
    int         resp_fit_nknots;
    int         resp_fit_degree;
    */

    /*
     * CPL objectsFit range is defined by the grism
     */

    cpl_table        *grism_table    = NULL;
    cpl_table        *photcal        = NULL;
    cpl_table        *response_obs_binning = NULL;

    cpl_propertylist *header         = NULL;
    cpl_propertylist *wcal_header    = NULL;
    cpl_propertylist *qclist         = NULL;
    const cpl_frame  *ref_sci_frame  = NULL;
    cpl_image        * mapped_flat_sed = NULL;
    cpl_propertylist * flat_sed_header = NULL;

    /*
     * Auxiliary variables
     */

    char        version[80];
    const char *instrume = NULL;
    const char *wheel4;
    char *science_tag = NULL;
    char *sedcorr_tag = NULL;
    char *disp_coeff_tag = NULL;
    char *flat_sed_tag = NULL;
    const char *specphot_tag;
    const char *telluric_contamination_tag = "TELLURIC_CONTAMINATION";
    const char *disp_coeff_tags[5] = {"DISP_COEFF_LSS", "DISP_COEFF_MOS", "DISP_COEFF_LONG_MOS", "DISP_COEFF_MXU", "DISP_COEFF_LONG_MXU"};
    int         mxu, mos, lss, mxu_sedcorr, mos_sedcorr, lss_sedcorr, mf, mf_sedcorr;
    int         nstd, nsedcorr;
    double      alltime;
    double      airmass = -1;
    double      iq = -1;
    double      ambi_req = -1;
    double      ra, dec;
    int         rebin;
    double      ref_wave;
    double      gain;
    int         standard;
    int         extinct;
    int         i;
    double      wstart;
    double      wstep;
    int         wcount;
    bool        have_sedcorr = false;
    fors_setting *setting = NULL;
    double      flat_sed_norm_factor = -1;
    bool        have_std_table = false; 
    bool        have_std_catalog = false;
    double      spatsigma, specsigma;
    int         slit_id = -1;

    snprintf(version, 80, "%s-%s", PACKAGE, PACKAGE_VERSION);

    cpl_msg_set_indentation(2);

    /* 
     * Get configuration parameters
     */

    cpl_msg_info(recipe, "Recipe %s configuration parameters:", recipe);
    cpl_msg_indent_more();

    if (cpl_frameset_count_tags(frameset, "GRISM_TABLE") > 1)
        fors_response_exit("Too many in input: GRISM_TABLE");

    grism_table = dfs_load_table(frameset, "GRISM_TABLE", 1);

    fit_window_pix = dfs_get_parameter_int(parlist, 
                                            "fors.fors_response.fit_window_pix",
                                            NULL);
    if(fit_window_pix < 0)   // means read from grism table
        fit_window_pix = dfs_get_parameter_int(parlist, 
                                            "fors.fors_response.fit_window_pix",
                                            grism_table);

    if (fit_window_pix < 0)
        fors_response_exit("Invalid instrument response fit_window_pix");

    velocity_wguess = dfs_get_parameter_double(parlist, 
                                            "fors.fors_response.velocity_wguess",
                                            NULL);
    if(velocity_wguess < 0)   // means read from grism table
        velocity_wguess = dfs_get_parameter_double(parlist, 
                                            "fors.fors_response.velocity_wguess",
                                            grism_table);

    std::string resp_use_flat_sed = dfs_get_parameter_string(parlist, 
                    "fors.fors_response.resp_use_flat_sed", NULL);
    std::transform(resp_use_flat_sed.begin(), resp_use_flat_sed.end(),
                   resp_use_flat_sed.begin(), ::tolower);

    if (cpl_error_get_code())
        fors_response_exit("Failure getting the configuration parameters");
    
    /* 
     * Check input set-of-frames
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Check input set-of-frames:");
    cpl_msg_indent_more();

    nstd = 0;
    nsedcorr = 0;
    const char *modes[3] = {"LSS", "MOS", "MXU"};
    const char *prefix[2] = {"REDUCED_IDP_", ""};
    const char *suffix[2] = {"", "_TELLURIC_CORR"};
    for (int i = 0; i < 3; i++) {
        for (int j = 0; j < 2; j++) {
            char* tag1 = cpl_sprintf("%sSTD_%s%s", prefix[j], modes[i], suffix[j]);
            char* tag2 = cpl_sprintf("%sSTD_%s_SEDCORR%s", prefix[j], modes[i], suffix[j]);
            int stdcount = cpl_frameset_count_tags(frameset, tag1);
            int sedcorrcount = cpl_frameset_count_tags(frameset, tag2);
            if (stdcount > 0) {
                if (science_tag != NULL) {
                    fors_response_exit("Only one reduced standard frame expected");
                }
                cpl_msg_info(cpl_func, "Found standard with tag %s", tag1);
                science_tag = tag1;
                tag1 = NULL;
                nstd += stdcount;
                if (sedcorrcount) {
                    if (sedcorr_tag != NULL) {
                        fors_response_exit("Only one reduced FLAT_SED corrected standard frame expected");
                    }
                    cpl_msg_info(cpl_func, "Found sed-corrected standard with tag %s", tag2);
                    sedcorr_tag = tag2;
                    tag2 = NULL;
                    have_sedcorr = true;
                    nsedcorr += sedcorrcount;
                }
                disp_coeff_tag = cpl_sprintf("DISP_COEFF_%s", modes[i]);
                flat_sed_tag = cpl_sprintf("FLAT_SED_%s", modes[i]);
                if (!cpl_frameset_count_tags(frameset, disp_coeff_tag) && strcmp(modes[i], "LSS")) {
                    cpl_free(disp_coeff_tag);
                    disp_coeff_tag = cpl_sprintf("DISP_COEFF_LONG_%s", modes[i]);
                    cpl_free(flat_sed_tag);
                    flat_sed_tag = cpl_sprintf("FLAT_SED_LONG_%s", modes[i]);
                }
                cpl_msg_info(cpl_func, "Using tags %s and %s", disp_coeff_tag, flat_sed_tag);
            } else if (sedcorrcount) {
                cpl_msg_error(recipe, "Found %s frame without corresponding %s frame", tag2, tag1);
                fors_response_exit(NULL);
            }
            cpl_free(tag1);
            cpl_free(tag2);
        }
    }

    if (nstd == 0)
        fors_response_exit("Missing input reduced standard frame");

    if (nstd > 1)
        fors_response_exit("Only one reduced standard frame expected");

    if (nsedcorr > 1)
        fors_response_exit("Only one reduced FLAT_SED corrected standard frame expected");

    specphot_tag             = "SPECPHOT_TABLE";

    if ((nsedcorr > 0) && !have_sedcorr)
        fors_response_exit("Mismatched standard and FLAT_SED corrected inputs");

    if (cpl_frameset_count_tags(frameset, telluric_contamination_tag) > 1)
        fors_response_exit("Too many in input: TELLURIC_CONTAMINATION");
    
    if ((cpl_frameset_count_tags(frameset, flat_sed_tag) == 0) && !have_sedcorr) {
        cpl_msg_info(recipe, "No %s or FLAT_SED corrected input spectrum provided; sed-corrected response will not be produced.", flat_sed_tag);
    }

    if ((cpl_frameset_count_tags(frameset, flat_sed_tag) > 0) && have_sedcorr) {
        cpl_msg_error(recipe, "Mismatched inputs. Both %s and %s provided.", flat_sed_tag, sedcorr_tag);
        fors_response_exit(NULL);
    }

    if (cpl_frameset_count_tags(frameset, flat_sed_tag) > 1) {
        cpl_msg_error(recipe, "Too many in input: %s", flat_sed_tag);
        fors_response_exit(NULL);
    }

    if (cpl_frameset_count_tags(frameset, disp_coeff_tag) == 0) {
        cpl_msg_error(recipe, "Missing required input: %s", disp_coeff_tag);
        fors_response_exit(NULL);
    }

    if (cpl_frameset_count_tags(frameset, disp_coeff_tag) > 1) {
        cpl_msg_error(recipe, "Too many in input: %s", disp_coeff_tag);
        fors_response_exit(NULL);
    }

    wcal_header = dfs_load_header(frameset, disp_coeff_tag, 0);
    
    dispersion = cpl_propertylist_get_double(wcal_header, "ESO PRO WLEN INC"); 
    
    ref_wave = cpl_propertylist_get_double(wcal_header, "ESO PRO WLEN CEN");
    
    startwavelength = cpl_propertylist_get_double(wcal_header, "ESO PRO WLEN START");
    
    endwavelength = cpl_propertylist_get_double(wcal_header, "ESO PRO WLEN END");

    cpl_propertylist_delete(wcal_header);

    if (cpl_error_get_code())
        fors_response_exit("Missing ESO PRO WLEN keywords in the DISP COEFF input. "
                          "Try re-running fors_calib recipe");

    if (cpl_frameset_count_tags(frameset, "EXTINCT_TABLE") > 0) {
        extinct = 1;
    } else {
        cpl_msg_info(recipe, "An EXTINCT_TABLE was not found in input.");
    }

    if (cpl_frameset_count_tags(frameset, "EXTINCT_TABLE") > 1)
        fors_response_exit("Too many in input: EXTINCT_TABLE");

    if (cpl_frameset_count_tags(frameset, "STD_FLUX_TABLE") == 0) {
        if (cpl_frameset_count_tags(frameset, "STD_FLUX_CATALOG") == 0) {
            fors_response_exit("Neither STD_FLUX_TABLE nor STD_FLUX_CATALOG "
                            "were found in the input: "
                            "instrument response curve will not be produced.");
        } else if (cpl_frameset_count_tags(frameset, "STD_FLUX_CATALOG") > 1) {
            fors_response_exit("Too many in input: STD_FLUX_CATALOG");
        } else {
            have_std_catalog = true;
        }
    } else if (cpl_frameset_count_tags(frameset, "STD_FLUX_TABLE") > 1) {
        fors_response_exit("Too many in input: STD_FLUX_TABLE");
    } else {
        have_std_table = true;
    }

    if (cpl_frameset_count_tags(frameset, "FIT_POINT_CATALOG") == 0) {
        fors_response_exit("A FIT_POINT_CATALOG was not found in input.");
    } else if (cpl_frameset_count_tags(frameset, "FIT_POINT_CATALOG") > 1) {
        fors_response_exit("Too many in input: FIT_POINT_CATALOG");
    }

    if (!dfs_equal_keyword(frameset, "ESO OBS TARG NAME")) {
        fors_response_exit("The target name of observation does not "
                        "match the standard star catalog: "
                        "instrument response curve will not be produced.");
    }

    if (!dfs_equal_keyword(frameset, "ESO INS GRIS1 ID"))
        cpl_msg_warning(cpl_func,"Input frames are not from the same grism");

    if (!dfs_equal_keyword(frameset, "ESO INS FILT1 ID"))
        cpl_msg_warning(cpl_func,"Input frames are not from the same filter");

    // Check that input frames are from the same chip
    if (!dfs_equal_keyword(frameset, "ESO DET CHIP1 ID"))
        cpl_msg_warning(cpl_func,"Input frames are not from the same chip");

    {
        cpl_frameset * frameset_detector = cpl_frameset_duplicate(frameset);
        cpl_frameset_erase(frameset_detector, "GRISM_TABLE");
        if (!dfs_equal_keyword(frameset_detector, "ESO DET CHIP1 NAME"))
            cpl_msg_warning(cpl_func,"Input frames are not from the same chip mosaic");
        cpl_frameset_delete(frameset_detector);
    }
 
    cpl_msg_indent_less();


    /*
     * Loading input data
     */

    /* Classify frames */
    fors_dfs_set_groups(frameset);
    
    //Get the reference frame to inherit science frames
    ref_sci_frame = cpl_frameset_find_const(frameset, science_tag);

    /* Get instrument setting */
    setting = fors_setting_new(ref_sci_frame);

    if (nstd == 1) {
        cpl_msg_info(recipe, "Load scientific exposure...");
        cpl_msg_indent_more();

        header = dfs_load_header(frameset, science_tag, 0);

        if (header == NULL)
            fors_response_exit("Cannot load scientific frame header");

        airmass = fors_get_airmass(header);
        if (airmass < 0.0) {
            cpl_msg_warning(cpl_func, "Missing airmass information in "
                            "scientific frame header. ");
            cpl_error_reset();
        }

        iq = fors_get_iq(header);
        ambi_req = fors_get_ambi_req(header);

        /*
         * Insert here a check on supported filters:
         */

        wheel4 = cpl_propertylist_get_string(header, 
                                                     "ESO INS OPTI9 TYPE");
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            fors_response_exit("Missing ESO INS OPTI9 TYPE in scientific "
                              "frame header");
        }

        if (strcmp("FILT", wheel4) == 0) {
            wheel4 = cpl_propertylist_get_string(header,
                                                         "ESO INS OPTI9 NAME");
            cpl_msg_error(recipe, "Unsupported filter: %s", wheel4);
            fors_response_exit(NULL);
        }

        alltime = cpl_propertylist_get_double(header, "EXPTIME");

        if (cpl_error_get_code() != CPL_ERROR_NONE)
            fors_response_exit("Missing keyword EXPTIME in scientific "
                              "frame header");

        ra = cpl_propertylist_get_double(header, "RA");
        dec = cpl_propertylist_get_double(header, "DEC");

        if (cpl_error_get_code() != CPL_ERROR_NONE)
            fors_response_exit("Missing keyword RA/DEC in scientific "
                              "frame header");

        cpl_propertylist_delete(header); header = NULL;

        cpl_msg_info(recipe, "Scientific frame exposure time: %.2f s", 
                     alltime);

    }

    cpl_msg_indent_less();

    /*
     * Get the reference wavelength and the rebin factor along the
     * dispersion direction from a scientific exposure
     */

    header = dfs_load_header(frameset, science_tag, 0);

    if (header == NULL)
        fors_response_exit("Cannot load scientific frame header");

    instrume = cpl_propertylist_get_string(header, "INSTRUME");
    if (instrume == NULL)
        fors_response_exit("Missing keyword INSTRUME in scientific header");

    if (instrume[4] == '1')
        snprintf(version, 80, "%s/%s", "fors1", VERSION);
    if (instrume[4] == '2')
        snprintf(version, 80, "%s/%s", "fors2", VERSION);

    cpl_msg_info(recipe, "The central wavelength is: %.2f A", ref_wave);

    rebin     = cpl_propertylist_get_int(header, "ESO DET WIN1 BINX");
    const int binning_y = cpl_propertylist_get_int(header, "ESO DET WIN1 BINY");

    if (cpl_error_get_code() != CPL_ERROR_NONE)
        fors_response_exit("Missing keywords ESO DET WIN1 BINX/Y in scientific "
                        "frame header");

    if (rebin != 1) {
        dispersion *= rebin;
        cpl_msg_warning(recipe, "The rebin factor is %d, and therefore the "
                        "resampling step used is %f A/pixel", rebin, 
                        dispersion);
    }
    // For convenience, have dispersion in nm as well
    dispersion_nm = dispersion / 10;

    gain = cpl_propertylist_get_double(header, "ESO DET OUT1 CONAD");

    if (cpl_error_get_code() != CPL_ERROR_NONE)
        fors_response_exit("Missing keyword ESO DET OUT1 CONAD in scientific "
                          "frame header");

    cpl_msg_info(recipe, "The gain factor is: %.2f e-/ADU", gain);

    std::string grism_name = 
            cpl_propertylist_get_string(header, "ESO INS GRIS1 NAME");

    std::string filter_name;
    const char* filt = cpl_propertylist_get_string(header, "ESO INS FILT1 NAME");
    cpl_error_reset();
    if (filt) {
        filter_name = filt;
    }

    if (cpl_frameset_count_tags(frameset, flat_sed_tag) > 0) {
        mapped_flat_sed = dfs_load_image(frameset, flat_sed_tag, CPL_TYPE_FLOAT, 0, 1);
        flat_sed_header = dfs_load_header(frameset, flat_sed_tag, 0);
    }

    if (have_sedcorr) {
        flat_sed_norm_factor = cpl_propertylist_get_double(header, "ESO QC RESP FLAT_SED_NORM");
        if (cpl_error_get_code() != CPL_ERROR_NONE)
            fors_response_exit("Missing keyword ESO QC RESP FLAT_SED_NORM in standard frame header");
        cpl_msg_info(cpl_func, "FLAT_SED norm factor is %f", flat_sed_norm_factor);
    } else if (flat_sed_header) {
        slit_id = cpl_propertylist_get_int(header, "ESO PRO SLIT_ID");
        char* norm_key = cpl_sprintf("ESO QC FLAT SED_%d NORM", slit_id);
        flat_sed_norm_factor = 
          cpl_propertylist_get_double(flat_sed_header, norm_key);
        cpl_free(norm_key);
        if (cpl_error_get_code() != CPL_ERROR_NONE)
            fors_response_exit("Missing keyword ESO PRO SLIT_ID in standard frame header, "
                "or missing keyword ESO QC FLAT SED_* NORM in FLAT_SED header");
        cpl_msg_info(cpl_func, "FLAT_SED norm factor is %f", flat_sed_norm_factor);
    }

    spatsigma = cpl_propertylist_get_double(header, "ESO DRS SPATSIGMA");
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        spatsigma = 4.0 / binning_y;
        cpl_msg_warning(recipe, "Missing keyword ESO DRS SPATSIGMA. Assuming a spatial sigma of %f", spatsigma);
        cpl_error_reset();
    }
    specsigma = spatsigma * binning_y / rebin;
    cpl_msg_info(recipe, "The spatial sigma is %f pixels and the estimated spectral sigma is %f pixels", spatsigma, specsigma);

    // Load the extracted spectra
    cpl_msg_info(cpl_func, "Loading standard spectrum");
    cpl_table* std_table = dfs_load_table(frameset, science_tag, 1);
    cpl_table* sedcorr_table = NULL;
    cpl_size stdtable_size = cpl_table_get_nrow(std_table);
    if ((stdtable_size != 1) ||
        !cpl_table_has_column(std_table, "WAVE") ||
        !cpl_table_has_column(std_table, "FLUX") ||
        !cpl_table_has_column(std_table, "ERR")) {
        fors_response_exit("Extracted standard spectrum does not have the required format");
    }
    if (have_sedcorr) {
        cpl_msg_info(cpl_func, "Loading FLAT_SED corrected standard spectrum");
        sedcorr_table = dfs_load_table(frameset, sedcorr_tag, 1);
        cpl_size sedcorrtable_size = cpl_table_get_nrow(sedcorr_table);
        if ((sedcorrtable_size != 1) ||
            !cpl_table_has_column(sedcorr_table, "WAVE") ||
            !cpl_table_has_column(sedcorr_table, "FLUX") ||
            !cpl_table_has_column(sedcorr_table, "ERR")) {
            fors_response_exit("Extracted FLAT_SED corrected standard spectrum does not have the required format");
        }
    } else if (flat_sed_header) {
        cpl_msg_info(cpl_func, "Applying FLAT_SED correction to standard spectrum");
        sedcorr_table = cpl_table_duplicate(std_table);
        cpl_size nx = cpl_image_get_size_x(mapped_flat_sed);
        cpl_array* flux_arr = cpl_array_duplicate(cpl_table_get_array(sedcorr_table, "FLUX", 0));
        cpl_array* err_arr = cpl_array_duplicate(cpl_table_get_array(sedcorr_table, "ERR", 0));
        cpl_size flux_size = cpl_array_get_size(flux_arr);
        cpl_size err_size = cpl_array_get_size(flux_arr);
        if ((nx != flux_size) || (nx != err_size)) {
            cpl_msg_error(recipe, "Extracted standard spectrum does not have the same size as the FLAT_SED profile (%lld != %lld)", flux_size, nx);
            fors_response_exit(NULL);
        }
        cpl_image* ffsed = cpl_image_extract(mapped_flat_sed, 1, slit_id+1, nx, slit_id+1);
        for (int i_pix = 0; i_pix < nx; i_pix++)
        {
            int   null;
            const double profile_val  = cpl_image_get(ffsed, i_pix+1, 1, &null);
            const double val = cpl_array_get(flux_arr, i_pix, &null);
            const double err = cpl_array_get(err_arr, i_pix, &null);
            if(profile_val != 0) {
                cpl_array_set(flux_arr, i_pix, val / profile_val);
                // Assume profile has no error
                cpl_array_set(err_arr, i_pix, err / profile_val);
            } else {
                cpl_array_set(flux_arr, i_pix, 0.0);
                cpl_array_set(err_arr, i_pix, 0.0);
            }
        }
        cpl_table_set_array(sedcorr_table, "FLUX", 0, flux_arr);
        cpl_table_set_array(sedcorr_table, "ERR", 0, err_arr);
        cpl_array_delete(flux_arr);
        cpl_array_delete(err_arr);
        // From now on, we have a sedcorr just as if it had been loaded from a separate extracted spectrum
        have_sedcorr = true;
        // cpl_table_save(sedcorr_table, NULL, NULL, "test_sedcorr_table.fits", CPL_IO_CREATE);
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            cpl_msg_error(cpl_func, "Error found in %s: %s",
                              cpl_error_get_where(), cpl_error_get_message());
            fors_response_exit("Error in applying the FLAT_SED correction");
        }
    }

    //Determine whether the response will contain the flat correction
    bool response_apply_flat_corr = 
        fors_response_response_apply_flat_corr(have_sedcorr, resp_use_flat_sed, grism_table);

    // Load and scale the spectra appropriately
    std::vector<double> wave_vec, flux_vec, err_vec, wave_vec_nm;
    load_from_table_array(std_table, "WAVE", "FLUX", "ERR", wave_vec, flux_vec, err_vec);
    cpl_table_delete(std_table);
    wave_vec_nm.resize(wave_vec.size());
    std::transform(wave_vec.begin(), wave_vec.end(), wave_vec_nm.begin(), std::bind(std::divides<double>(), std::placeholders::_1, 10));

    if (cpl_error_get_code()) {
        fors_response_exit("Could not load and scale the extracted standard spectrum");
    }

    // Now the sed-corrected spectrum
    std::vector<double> wave_vec_sedcorr, flux_vec_sedcorr, err_vec_sedcorr, wave_vec_sedcorr_nm;
    std::vector<double> wave_vec_sedcorr_A, flux_vec_sedcorr_old;
    if (response_apply_flat_corr) {
        load_from_table_array(sedcorr_table, "WAVE", "FLUX", "ERR", wave_vec_sedcorr, flux_vec_sedcorr, err_vec_sedcorr);
        cpl_table_delete(sedcorr_table);
        wave_vec_sedcorr_nm.resize(wave_vec_sedcorr.size());
        std::transform(wave_vec_sedcorr.begin(), wave_vec_sedcorr.end(), wave_vec_sedcorr_nm.begin(), std::bind(std::divides<double>(), std::placeholders::_1, 10));

        if (cpl_error_get_code()) {
            fors_response_exit("Could not load and scale the extracted SED-corrected standard spectrum");
        }
    }

    cpl_table *ext_table  = NULL;
    cpl_table *flux_table = NULL;
    cpl_table *fp_table = NULL;

    if (extinct) {
        ext_table = dfs_load_table(frameset, "EXTINCT_TABLE", 1);
    }

    const char* star_name = NULL;

    if (have_std_table) {
        flux_table = dfs_load_table(frameset, "STD_FLUX_TABLE", 1);
        star_name = cpl_propertylist_get_string(header, "OBJECT");
    } else if (have_std_catalog) {
        const cpl_frame* ref_table = cpl_frameset_find_const(frameset, "STD_FLUX_CATALOG");
        fors_load_ref_table(ref_table, ra, dec, STAR_MATCH_DEPSILON, &flux_table, &star_name);
    } else {
        fors_response_exit("No standard flux data found");
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                          cpl_error_get_where(), cpl_error_get_message());
        fors_response_exit("Could not load standard flux data");
    }
    cpl_propertylist_delete(header); header = NULL;

    std::string grism_filter = grism_name.substr(5) + "_" + (filter_name.empty() ? "free" : filter_name);
    const cpl_frame* fit_point_catalog = cpl_frameset_find_const(frameset, "FIT_POINT_CATALOG");
    cpl_array* fit_points = NULL;
    fors_load_fp_table(fit_point_catalog, grism_filter.c_str(), star_name, &fit_points);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                          cpl_error_get_where(), cpl_error_get_message());
        fors_response_exit("Could not load fit point data");
    }

    // Convert from A to nm
    cpl_array_divide_scalar(fit_points, 10);
    cpl_msg_info(cpl_func, "Read %lld fit points for %s and %s from %s", cpl_array_get_size(fit_points), grism_filter.c_str(), star_name, cpl_frame_get_filename(fit_point_catalog));

    /* Now the HDRL response */

    hdrl_value Ap = {0, 0};
    hdrl_value Am = {airmass, 0};
    hdrl_value invG = {1.0 / gain, 0};
    hdrl_value G = {gain, 0};
    // Set time to 1 because our observed standard is already in ADU/s
    hdrl_value Texp = {1.0, 0};
    hdrl_value Atel = {512000, 0}; // collecting area of VLT in cm2

    // If we want to ignore lines and regions like the original response code, then we can do so with
    // the following:
    // for (auto i = resp_ignore_lines_list.begin(); i != resp_ignore_lines_list.end(); i++) {
        // cpl_msg_info(cpl_func, "ignore line %f", *i);
    // }
    // cpl_bivector * high_abs_regions = create_windows(resp_ignore_lines_list, fit_wrange, resp_ignore_ranges_list);

    cpl_error_reset();

    // Load the flux standard. It may need to be scaled, depending on whether it's a FORS format table
    // or an XShooter format catalog.
    std::vector<double> tmp_wave_vec, std_wave_vec, std_flux_vec, std_err_vec;
    load_from_table(flux_table, "WAVE", "FLUX", NULL, tmp_wave_vec, std_flux_vec, std_err_vec);
    cpl_table_delete(flux_table);
    const double wavmult = ( have_std_table ? 0.1 : 1);
    std_wave_vec.resize(tmp_wave_vec.size());
    std::transform(tmp_wave_vec.begin(), tmp_wave_vec.end(), std_wave_vec.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, wavmult));

    // Load the extinction table
    // The wavelength column is in Angstrom
    std::vector<double> e_wave_vec, e_ext_vec, e_err_vec;
    tmp_wave_vec.clear();
    load_from_table(ext_table, "WAVE", "EXTINCTION", NULL, tmp_wave_vec, e_ext_vec, e_err_vec);
    cpl_table_delete(ext_table);
    e_wave_vec.resize(tmp_wave_vec.size());
    std::transform(tmp_wave_vec.begin(), tmp_wave_vec.end(), e_wave_vec.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, 0.1));

    /* Create an HDRL spectrum object of the observed standard */
    hdrl_spectrum1D* I_std = create_spectrum(wave_vec_nm, flux_vec, err_vec);
    if (I_std == NULL) {
        cpl_msg_warning(cpl_func, "I_std is null");
    }

    /* Create an HDRL spectrum object from the sed-corrected observed standard */
    hdrl_spectrum1D* I_std_sedcorr = NULL;
    if (response_apply_flat_corr) {
        I_std_sedcorr = create_spectrum(wave_vec_sedcorr_nm, flux_vec_sedcorr, err_vec_sedcorr);
        if (I_std_sedcorr == NULL) {
            cpl_msg_warning(cpl_func, "I_std_sedcorr is null");
        }
    }

    hdrl_spectrum1D_save(I_std, "std_obs.fits");
    hdrl_spectrum1D_save(I_std_sedcorr, "std_obs_sedcorr.fits");

    // Pick a size for the window that will encompass the likely velocity shift
    // and will also ensure that there are enough points to fit
    double velocity_fit_half_win = 2.5 * specsigma * dispersion_nm;

    // Convert to nm
    velocity_wguess /= 10;

    cpl_msg_info(cpl_func, "velocity_wguess = %f nm, velocity_fit_half_win = %f nm", velocity_wguess, velocity_fit_half_win);

    double velocity_range_wmin = velocity_wguess - 6 * velocity_fit_half_win;
    double velocity_range_wmax = velocity_wguess + 6 * velocity_fit_half_win;
    double velocity_fit_wmin = velocity_wguess - 3 * velocity_fit_half_win;
    double velocity_fit_wmax = velocity_wguess + 3 * velocity_fit_half_win;

    // Calculate the doppler shift
    hdrl_parameter * velocity_par = NULL;
    hdrl_data_t velocity_shift = 0.0;
    /* If velocity_wguess <= 0, then no doppler shift is calculated.
     * This occurs for the few grism/filter combinations where no appropriate
     * line is available. */
    if (velocity_wguess > 0) {

        // Oversample the input spectrum by 10 so that the routine that fits the doppler shift
        // can give us 1/10th pixel accuracy.
        cpl_array * tmpwav = get_wavpoints(velocity_range_wmin, velocity_range_wmax, dispersion_nm / 10);
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create(hdrl_spectrum1D_interp_linear);
        hdrl_spectrum1D* tmpspec = hdrl_spectrum1D_resample_on_array(I_std, tmpwav, params);
        cpl_array_delete(tmpwav);
        hdrl_parameter_delete(params);

        velocity_par = hdrl_spectrum1D_shift_fit_parameter_create(velocity_wguess,
            velocity_range_wmin, velocity_range_wmax,
            velocity_fit_wmin, velocity_fit_wmax, velocity_fit_half_win);

        velocity_shift =
            hdrl_spectrum1D_compute_shift_fit(tmpspec, velocity_par);

        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Doppler shift is %f", velocity_shift);
        } else {
            cpl_msg_warning(cpl_func, "Could not calculate doppler shift");
            cpl_msg_error(cpl_func, "Error found in %s: %s",
                              cpl_error_get_where(), cpl_error_get_message());
            hdrl_parameter_delete(velocity_par);
            velocity_par = NULL;
            cpl_error_reset();
        }
        hdrl_spectrum1D_delete(&tmpspec);
    }

    // Need to use inverse G here, as the hdrl_response code divides by the
    // observed spectrum but multiplies by the G parameter.
    hdrl_parameter * calc_par =
        hdrl_response_parameter_create(Ap, Am, invG, Texp);

    // No median smoothing of the spectrum before fitting.
    // fit_points is read from the FIT_POINT_CATALOG earlier

    // fit_wrange is defined by the grism table. It's given in unbinned pixels for
    // the full window size, then converted into a wavelength range for the half window.
    double fit_wrange = dispersion_nm * fit_window_pix / rebin / 2; // half window width in nm
    cpl_msg_info(cpl_func, "Final fit will median combine data +/- %f nm around each fit point", fit_wrange);
    hdrl_parameter * fit_par =
        hdrl_response_fit_parameter_create(1, fit_points, fit_wrange, NULL);
    // Use the following if we want to ignore certain lines and/or regions
    // hdrl_response_fit_parameter_create(1, fit_points, fit_wrange, high_abs_regions);

    /* Create the reference spectrum */
    hdrl_spectrum1D* I_std_ref = create_spectrum(std_wave_vec, std_flux_vec, std_err_vec);
    if (I_std_ref == NULL) {
        cpl_msg_warning(cpl_func, "I_std_ref is null");
    }
    std::vector<double> full_range_wave_tab_nm, full_range_wave_tab_A, full_range_eff_tab, full_range_response_tab;
    const hdrl_spectrum1D_wavelength full_range_wav = hdrl_spectrum1D_get_wavelength(I_std_ref);
    full_range_wave_tab_nm = extract_wavelength(I_std_ref);
    full_range_wave_tab_A.resize(full_range_wave_tab_nm.size());
    std::transform(full_range_wave_tab_nm.begin(), full_range_wave_tab_nm.end(), full_range_wave_tab_A.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, 10));
    // hdrl_spectrum1D_save(I_std_ref, "ref_original.fits");

    /* Shift the reference spectrum if required */
    hdrl_spectrum1D * ref_shifted = NULL;
    if (velocity_par != NULL) {
        ref_shifted =
            correct_spectrum_for_doppler_shift(I_std_ref, velocity_shift);
        cpl_msg_info(cpl_func, "Corrected reference spectrum for doppler shift");
        hdrl_parameter_delete(velocity_par);
    } else {
        cpl_msg_warning(cpl_func, "Reference spectrum is not corrected for doppler shift.");
        ref_shifted = hdrl_spectrum1D_duplicate(I_std_ref);
    }

    // hdrl_spectrum1D_save(ref_shifted, "ref_shifted.fits");

    const hdrl_spectrum1D_wavelength spec_wav = hdrl_spectrum1D_get_wavelength(I_std);
    std::vector<double> spec_wave_tab_nm, spec_wave_tab_A;
    spec_wave_tab_nm = extract_wavelength(I_std);
    spec_wave_tab_A.resize(spec_wave_tab_nm.size());
    std::transform(spec_wave_tab_nm.begin(), spec_wave_tab_nm.end(), spec_wave_tab_A.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, 10));

    /* Resample ref_shifted on I_std wavelengths */
    hdrl_spectrum1D * I_std_ref_resampled = NULL;
    {
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create
                (hdrl_spectrum1D_interp_linear);
        I_std_ref_resampled = hdrl_spectrum1D_resample(ref_shifted, &spec_wav, params);
        hdrl_parameter_delete(params);
    }

    if (I_std_ref_resampled == NULL) {
        cpl_msg_warning(cpl_func, "Reference spectrum could not be resampled.");
        I_std_ref_resampled = ref_shifted;
        cpl_error_reset();
    } else {
        cpl_msg_info(cpl_func, "Resampled reference spectrum to observed spectrum wavelength scale.");
        hdrl_spectrum1D_delete(&ref_shifted);
    }

    // hdrl_spectrum1D_save(I_std_ref_resampled, "ref_resampled.fits");

    /* Convolve the resampled reference spectrum with a 
     * gaussian of the appropriate size. This is based
     * on the spatial resolution as calculated when the
     * standard spectrum was extracted
     */
    hdrl_spectrum1D * I_std_ref_resampled_convolved = NULL;
    cpl_size kernel_size = round(specsigma * 6); // +/- 3 sigma is 99.7% of the area under a gaussian
    // Kernel size must be odd
    if (kernel_size % 2 == 0) {
      kernel_size++;
    }
    I_std_ref_resampled_convolved = gaussian_convolve(I_std_ref_resampled, kernel_size, specsigma);

    if (I_std_ref_resampled_convolved == NULL) {
        cpl_msg_warning(cpl_func, "Convolution of standard spectrum failed. Using unconvolved spectrum.");
        I_std_ref_resampled_convolved = I_std_ref_resampled;
        cpl_error_reset();
    } else {
        cpl_msg_info(cpl_func, "Convolved standard spectrum with gaussian kernel of size %lld pixels, "
                               "with sigma = %f", kernel_size, specsigma);
        hdrl_spectrum1D_delete(&I_std_ref_resampled);
    }

    // hdrl_spectrum1D_save(I_std_ref_resampled_convolved, "ref_convolved.fits");

    /* Create the extinction curve */
    hdrl_spectrum1D* E_x = create_spectrum(e_wave_vec, e_ext_vec, e_err_vec);
    if (E_x == NULL) {
        cpl_msg_warning(cpl_func, "E_x is null");
    }

    /* Resample E_x on I_std wavelengths */
    hdrl_spectrum1D * E_x_resampled = NULL;
    {
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create
                (hdrl_spectrum1D_interp_linear);
        E_x_resampled = hdrl_spectrum1D_resample(E_x, &spec_wav, params);
        hdrl_parameter_delete(params);
    }

    if (E_x_resampled == NULL) {
        cpl_msg_warning(cpl_func, "Extinction curve could not be resampled.");
        E_x_resampled = E_x;
        cpl_error_reset();
    } else {
        cpl_msg_info(cpl_func, "Resampled extinction curve to observed spectrum wavelength scale.");
        hdrl_spectrum1D_delete(&E_x);
    }

    // hdrl_spectrum1D_save(E_x_resampled, "extinct_resampled.fits");

    // Scale the observed standard so that our output matches the result
    // from the original FORS response calculation
    hdrl_spectrum1D* I_std_scaled = hdrl_spectrum1D_div_scalar_create(I_std, {dispersion, 0.0});
    hdrl_spectrum1D* I_std_sedcorr_scaled = NULL;
    if (response_apply_flat_corr) {
        I_std_sedcorr_scaled = hdrl_spectrum1D_div_scalar_create(I_std_sedcorr, {dispersion, 0.0});
        hdrl_spectrum1D_delete(&I_std_sedcorr);
    }

    cpl_msg_info(cpl_func, "Computing response curve");
    hdrl_response_result * response_res = hdrl_response_compute(
        I_std_scaled,
        I_std_ref_resampled_convolved,
        E_x_resampled,
        NULL, // no telluric correction - use molecfit
        NULL, // velocity correction already performed
        calc_par,
        fit_par);

    hdrl_response_result * response_res_sedcorr = NULL;
    if (response_apply_flat_corr) {
        cpl_msg_info(cpl_func, "Computing sed-corrected response curve");
        response_res_sedcorr = hdrl_response_compute(
            I_std_sedcorr_scaled,
            I_std_ref_resampled_convolved,
            E_x_resampled,
            NULL, // no telluric correction - use molecfit
            NULL, // velocity correction already performed
            calc_par,
            fit_par);
    }

    if (response_res == NULL) {
        fors_response_exit("Could not generate response curve");
    }
    if (response_apply_flat_corr && response_res_sedcorr == NULL) {
        fors_response_exit("Could not generate sed-corrected response curve");
    }
    hdrl_parameter_delete(calc_par);
    hdrl_parameter_delete(fit_par);

    cpl_msg_info(cpl_func, "Saving HDRL response");

    const hdrl_spectrum1D * final_response = 
        hdrl_response_result_get_final_response(response_res);
    hdrl_spectrum1D* final_response_scaled = NULL;
    if (final_response == NULL) {
        fors_response_exit("Could not extract final response");
    } else {
      final_response_scaled = hdrl_spectrum1D_mul_scalar_create(final_response, {1e16, 0.0});
      hdrl_spectrum1D_save(final_response_scaled, "final_response.fits");
    }

    const hdrl_spectrum1D * final_response_sedcorr = NULL;
    hdrl_spectrum1D * final_response_sedcorr_scaled = NULL;
    if (response_apply_flat_corr) {
        final_response_sedcorr = hdrl_response_result_get_final_response(response_res_sedcorr);
        if (final_response_sedcorr == NULL) {
            fors_response_exit("Could not extract final sed-corrected response");
        } else {
          final_response_sedcorr_scaled = hdrl_spectrum1D_mul_scalar_create(final_response_sedcorr, {1e16, 0.0});
          hdrl_spectrum1D_save(final_response_sedcorr_scaled, "final_response_sedcorr.fits");
        }
    }

    /* The selected response is
    * the raw response sampled in the fit points. This response is then
    * interpolated, creating the final response.
    */
    const hdrl_spectrum1D * selected_response = NULL;

    if (response_apply_flat_corr) {
        selected_response = hdrl_response_result_get_selected_response(response_res_sedcorr);
    } else {
        selected_response = hdrl_response_result_get_selected_response(response_res);
    }
    hdrl_spectrum1D* selected_response_scaled = NULL;
    if (selected_response == NULL) {
        cpl_msg_warning(cpl_func, "selected response is null");
    } else {
        selected_response_scaled = hdrl_spectrum1D_mul_scalar_create(selected_response, {1e16, 0.0});
        hdrl_spectrum1D_save(selected_response_scaled, "selected_response.fits");
    }
    const hdrl_spectrum1D_wavelength selected_wav = hdrl_spectrum1D_get_wavelength(selected_response_scaled);
    std::vector<double> selected_wave_tab_nm, selected_wave_tab_A, selected_eff_tab, selected_response_tab;
    selected_wave_tab_nm = extract_wavelength(selected_response_scaled);
    selected_wave_tab_A.resize(selected_wave_tab_nm.size());
    std::transform(selected_wave_tab_nm.begin(), selected_wave_tab_nm.end(), selected_wave_tab_A.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, 10));

    const hdrl_spectrum1D * raw_response =
        hdrl_response_result_get_raw_response(response_res);
    hdrl_spectrum1D* raw_response_scaled = NULL;
    if (raw_response == NULL) {
        cpl_msg_warning(cpl_func, "Could not extract raw response");
    } else {
      raw_response_scaled = hdrl_spectrum1D_mul_scalar_create(raw_response, {1e16, 0.0});
      hdrl_spectrum1D_save(raw_response_scaled, "raw_response.fits");
    }

    const hdrl_spectrum1D * raw_response_sedcorr = NULL;
    hdrl_spectrum1D* raw_response_sedcorr_scaled = NULL;
    if (response_apply_flat_corr) {
        raw_response_sedcorr = hdrl_response_result_get_raw_response(response_res_sedcorr);
        if (raw_response_sedcorr == NULL) {
            cpl_msg_warning(cpl_func, "Could not extract sed-corrected raw response");
        } else {
          raw_response_sedcorr_scaled = hdrl_spectrum1D_mul_scalar_create(raw_response_sedcorr, {1e16, 0.0});
          hdrl_spectrum1D_save(raw_response_sedcorr_scaled, "raw_response_sedcorr.fits");
        }
    }

    /*
    const hdrl_spectrum1D * corrected_obs =
        hdrl_response_result_get_corrected_obs_spectrum(response_res);
    if (corrected_obs == NULL) {
        cpl_msg_warning(cpl_func, "corrected obs is null");
    }
    hdrl_spectrum1D_save(corrected_obs, "corrected_obs.fits");
    */

    // Do the efficiency calculation
    cpl_msg_info(cpl_func, "Calculating efficiency");
    hdrl_parameter* efficiency_pars = hdrl_efficiency_parameter_create(
        Ap, Am, G, Texp, Atel);

    hdrl_spectrum1D * eff = hdrl_efficiency_compute(
        I_std_scaled,
        I_std_ref_resampled_convolved,
        E_x_resampled,
        efficiency_pars);
    hdrl_parameter_delete(efficiency_pars);

    /* Integrate eff on the selected fit points */
    cpl_msg_info(cpl_func, "Selecting efficiency fit points, and resampling");
    hdrl_spectrum1D * selected_eff = get_median_on_fit_points(
      eff, fit_points, fit_wrange);

    if (selected_eff == NULL) {
      fors_response_exit("Could not get selected efficiency");
    }

    hdrl_spectrum1D_save(eff, "efficiency.fits");

    /* Resample the selected efficiency points on the wavelength range of the observed standard */
    cpl_msg_info(cpl_func, "Resampling efficiency to wavelength scale of observation");
    hdrl_spectrum1D * final_efficiency = NULL;
    {
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create
                (hdrl_spectrum1D_interp_akima);
        final_efficiency = hdrl_spectrum1D_resample(selected_eff, &spec_wav, params);
        if (final_efficiency == NULL) {
          fors_response_exit("Could not resample efficiency");
        }
        hdrl_parameter_delete(params);
    }

    std::vector<double> wave_tab_nm, wave_tab_A, flux_tab, flux_obs, flux_obs_sedcorr, efficiency_raw, efficiency_fit, response_raw, response_raw_sedcorr, response_fit, response_fit_sedcorr;

    hdrl_spectrum1D* I_std_scaled_e = hdrl_spectrum1D_mul_scalar_create(I_std_scaled, G);
    hdrl_spectrum1D* I_std_sedcorr_scaled_e = NULL;
    if (response_apply_flat_corr) {
        I_std_sedcorr_scaled_e = hdrl_spectrum1D_mul_scalar_create(I_std_sedcorr_scaled, G);
    }

    wave_tab_nm = extract_wavelength(I_std);
    wave_tab_A.resize(wave_tab_nm.size());
    std::transform(wave_tab_nm.begin(), wave_tab_nm.end(), wave_tab_A.begin(), std::bind(std::multiplies<double>(), std::placeholders::_1, 10));

    // Rescale the flux standard to match the output of the old specphot_table
    hdrl_spectrum1D* I_std_ref_rescaled = hdrl_spectrum1D_mul_scalar_create(I_std_ref_resampled_convolved, {1e16, 0.0});
    flux_tab = extract_flux(I_std_ref_rescaled);
    hdrl_spectrum1D_delete(&I_std_ref_rescaled);

    // Now calculate the observed spectrum in e/s/AA, corrected for extinction.
    // This is what the old specphot_table contains.
    {
        hdrl_spectrum1D * exponential = hdrl_spectrum1D_duplicate(E_x_resampled);
        hdrl_spectrum1D * exponential2 = hdrl_spectrum1D_duplicate(exponential);

        /*0.4A_pE_x(f) */
        hdrl_spectrum1D_mul_scalar(exponential,	(hdrl_value){0.4, 0.0});
        hdrl_spectrum1D_mul_scalar(exponential, Ap);

        /*0.4A_mE_x(f) */
        hdrl_spectrum1D_mul_scalar(exponential2, (hdrl_value){0.4, 0.0});
        hdrl_spectrum1D_mul_scalar(exponential2, Am);

        /*0.4A_mE_x(f) - 0.4A_pE_x(f) */
        hdrl_spectrum1D_sub_spectrum(exponential2, exponential);

        hdrl_spectrum1D_delete(&exponential2);

        hdrl_spectrum1D_exp_scalar(exponential, (hdrl_value){10.0, 0.0});
        exponential2 = hdrl_spectrum1D_duplicate(exponential);

        hdrl_spectrum1D_mul_spectrum(exponential, I_std_scaled_e);
        flux_obs = extract_flux(exponential); // in e/(s.AA)?
        if (response_apply_flat_corr) {
            hdrl_spectrum1D_mul_spectrum(exponential2, I_std_sedcorr_scaled_e);
            flux_obs_sedcorr = extract_flux(exponential2);
        }

        hdrl_spectrum1D_delete(&exponential);
        hdrl_spectrum1D_delete(&exponential2);
    }

    efficiency_raw = extract_flux(eff);
    efficiency_fit = extract_flux(final_efficiency);
    response_raw = extract_flux(raw_response_scaled);
    response_fit = extract_flux(final_response_scaled);
    if (response_apply_flat_corr) {
        response_raw_sedcorr = extract_flux(raw_response_sedcorr_scaled);
        response_fit_sedcorr = extract_flux(final_response_sedcorr_scaled);
    }

    photcal = create_response_table(wave_tab_A, flux_tab,
        flux_obs, efficiency_raw, efficiency_fit, response_raw);

    if (response_apply_flat_corr) {
        update_response_table_ffsed(photcal,
                flux_obs_sedcorr, response_raw_sedcorr, response_fit_sedcorr);
    } else {
        update_response_table(photcal, response_fit);
    }

    // QC parameters
    float *data;
    char   keyname[30];

    qclist = dfs_load_header(frameset, science_tag, 0);
    if (qclist == NULL)
        fors_response_exit("Cannot reload scientific "
                "frame header");

    /*
     * QC1 parameters
     */

    wstart = 3700.;
    wstep  = 400.;
    wcount = 15;

    cpl_image* dummy = cpl_image_new(wcount, 1, CPL_TYPE_FLOAT);
    data = cpl_image_get_data_float(dummy);
    map_table(dummy, wstart, wstep, photcal,
              "WAVE", "EFFICIENCY");

    cpl_size nresp_obs_binning = cpl_table_get_nrow(photcal);
    std::valarray<double> wave_obs(cpl_table_get_data_double
            (photcal, "WAVE"), nresp_obs_binning);
    double min_valid_wave = wave_obs.min();
    double max_valid_wave = wave_obs.max();

    for (i = 0; i < wcount; i++) {
        sprintf(keyname, "QC.SPEC.EFFICIENCY%d.LAMBDA", 
                i + 1);
        if (fors_header_write_double(qclist, 
                wstart + wstep * i,
                keyname, "Angstrom",
                "Wavelength of "
                "efficiency evaluation")) {
            fors_response_exit("Cannot write qc param");
        }

        sprintf(keyname, "QC.SPEC.EFFICIENCY%d", i + 1);
        if (fors_header_write_double(qclist,
                data[i],
                keyname, "e-/photon",
                "Efficiency")) {
            fors_response_exit("Cannot write qc param");
        }
    }
    cpl_image_delete(dummy); dummy = NULL;

    std::string raw_resp_column;
    std::string fit_resp_column;
    if (response_apply_flat_corr) {
        if (fors_header_write_int(qclist,
                1,
                "QC.RESP.FLAT_SED_CORR", "",
                "Response corrected from flat dispersion profile")) {
            fors_response_exit("Cannot write qc param");
        }
        if (fors_header_write_double(qclist,
                flat_sed_norm_factor,
                "QC.RESP.FLAT_SED_NORM", "",
                "Normalisation factor applied to flat sed")) {
            fors_response_exit("Cannot write qc param");
        }
        raw_resp_column = "RAW_RESPONSE_FFSED";
        fit_resp_column = "RESPONSE_FFSED";
    } else {
        if (fors_header_write_int(qclist,
                0,
                "QC.RESP.FLAT_SED_CORR", "",
                "Response corrected from flat dispersion profile")) {
            fors_response_exit("Cannot write qc param");
        }
        raw_resp_column = "RAW_RESPONSE";
        fit_resp_column = "RESPONSE";
    }
        
    /*
     * We no longer have the USED_FIT column
     * so can't calculate the values for:
     *      QC.RESP_FIT_RATIO.MASK.MAX
     *      QC.RESP_FIT_RATIO.MASK.MIN
     *      QC.RESP_FIT_RATIO.USED.MAX
     *      QC.RESP_FIT_RATIO.USED.MIN
     */

    /* Write the relevant settings to the QC header */
    fors_setting_write_qc(setting, qclist);

    if (cpl_error_get_code()) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
    }

    cpl_error_reset();

    // Save the first extension in the output table
    fors_dfs_save_table(frameset, photcal, specphot_tag, qclist,
                       parlist, recipe, ref_sci_frame);

    // Create a table from the selected response points
    cpl_msg_info(cpl_func, "Creating response and efficiency table from selected points");
    selected_eff_tab = extract_flux(selected_eff);
    selected_response_tab = extract_flux(selected_response_scaled);
    const char* col1 = (response_apply_flat_corr ? "RAW_RESPONSE_FFSED" : "RAW_RESPONSE");
    cpl_table * selected_table = create_resp_eff_table(selected_wave_tab_A, col1, selected_response_tab, "RAW_EFFICIENCY", selected_eff_tab);

    /* Resample the selected efficiency points on the full wavelength range of the observed standard */
    cpl_msg_info(cpl_func, "Resampling efficiency to wavelength scale of observed standard");
    hdrl_spectrum1D * full_range_eff = NULL;
    {
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create
                (hdrl_spectrum1D_interp_akima);
        full_range_eff = hdrl_spectrum1D_resample(selected_eff, &spec_wav, params);
        if (full_range_eff == NULL) {
          fors_response_exit("Could not resample efficiency");
        }
        hdrl_parameter_delete(params);
    }

    /* Resample the selected response points on the full wavelength range of the input standard */
    cpl_msg_info(cpl_func, "Resampling response to wavelength scale of observed standard");
    hdrl_spectrum1D * full_range_response = NULL;
    {
        hdrl_parameter * params =
            hdrl_spectrum1D_resample_interpolate_parameter_create
                (hdrl_spectrum1D_interp_akima);
        full_range_response = hdrl_spectrum1D_resample(selected_response_scaled, &spec_wav, params);
        if (full_range_eff == NULL) {
          fors_response_exit("Could not resample response");
        }
        hdrl_parameter_delete(params);
    }

    cpl_msg_info(cpl_func, "Creating response & efficiency table");
    full_range_eff_tab = extract_flux(full_range_eff);
    full_range_response_tab = extract_flux(full_range_response);
    const char* col2 = (response_apply_flat_corr ? "RESPONSE_FFSED" : "RESPONSE");
    cpl_table * full_range_table = create_resp_eff_table(spec_wave_tab_A, col2, full_range_response_tab, "EFFICIENCY", full_range_eff_tab);

    /* Save the extra extensions to the SPECPHOT_TABLE */
    cpl_table_save(full_range_table, NULL, NULL,
        "specphot_table.fits", CPL_IO_EXTEND);
    cpl_table_save(selected_table, NULL, NULL,
        "specphot_table.fits", CPL_IO_EXTEND);

    cpl_table_delete(photcal);
    cpl_array_delete(fit_points);

    hdrl_spectrum1D_delete(&final_response_scaled);
    hdrl_spectrum1D_delete(&final_response_sedcorr_scaled);
    hdrl_spectrum1D_delete(&selected_response_scaled);
    hdrl_spectrum1D_delete(&raw_response_scaled);
    hdrl_spectrum1D_delete(&raw_response_sedcorr_scaled);

    hdrl_spectrum1D_delete(&I_std);
    hdrl_spectrum1D_delete(&I_std_sedcorr);
    hdrl_spectrum1D_delete(&I_std_scaled);
    hdrl_spectrum1D_delete(&I_std_sedcorr_scaled);
    hdrl_spectrum1D_delete(&I_std_scaled_e);
    hdrl_spectrum1D_delete(&I_std_ref);
    hdrl_spectrum1D_delete(&I_std_ref_resampled_convolved);
    hdrl_spectrum1D_delete(&E_x_resampled);
    hdrl_spectrum1D_delete(&eff);

    hdrl_response_result_delete(response_res);
    hdrl_response_result_delete(response_res_sedcorr);
    fors_setting_delete(&setting);
    cpl_free(science_tag);
    cpl_free(sedcorr_tag);
    cpl_free(disp_coeff_tag);
    cpl_table_delete(grism_table);

    if (cpl_error_get_code()) {
        cpl_msg_error(cpl_func, "Error found in %s: %s",
                      cpl_error_get_where(), cpl_error_get_message());
        fors_response_exit(NULL);
    }

    return 0;
}

/*
 * Determine whether or not to apply the flat sed correction
 */
static bool fors_response_response_apply_flat_corr
(bool has_ffsed, std::string& resp_use_flat_sed, cpl_table * grism_table)
{
    bool requested_in_grism_table = false;
    int null;
    if(cpl_table_get_int(grism_table, "RESP_USE_FLAT_SED", 0, &null))
        requested_in_grism_table = true;

    if(has_ffsed)
    {
        if(resp_use_flat_sed == "false" || (resp_use_flat_sed == "grism_table" &&
                !requested_in_grism_table))
        {
            cpl_msg_warning(cpl_func, "Flat SED is part of the input but "
                    "no correction has been requested");
            return false;
        }
        return true;
    }
    else
    {
        if(resp_use_flat_sed == "true" || (resp_use_flat_sed == "grism_table" &&
                requested_in_grism_table))
            throw std::invalid_argument("Flat SED correction requested "
                    "but FLAT_SED it is not part of input.");
        return false;
    }
    return false;
}


/*
 * Return an array of evenly spaced wavelength points
 */
static cpl_array * get_wavpoints(const double start_w, const double end_w, const double step)
{
    cpl_size num_samples = ceil((end_w - start_w) / step);
    cpl_array * to_ret = cpl_array_new(num_samples, CPL_TYPE_DOUBLE);
    for(cpl_size i = 0; i < num_samples; ++i){
        const double w = start_w + step * i;
        cpl_array_set(to_ret, i, w);
    }
    return to_ret;
}

/*
 * Create windows of excluded wavelengths
 */
cpl_bivector * create_windows(const std::vector<double>& lines, double width, const std::vector<std::pair<double, double> >& range)
{
	cpl_bivector * ret = cpl_bivector_new(lines.size() + range.size());
	cpl_vector * v1 = cpl_bivector_get_x(ret);
	cpl_vector * v2 = cpl_bivector_get_y(ret);

  cpl_size c = 0;
  for (auto i = lines.begin(); i != lines.end(); i++) {
    cpl_msg_info(cpl_func, "line %lld %f %f", c, *i - width, *i + width);
		cpl_vector_set(v1, c, *i - width);
		cpl_vector_set(v2, c, *i + width);
    ++c;
  }
  for (auto i = range.begin(); i != range.end(); i++) {
    cpl_msg_info(cpl_func, "window %lld %f %f", c, i->first, i->second);
		cpl_vector_set(v1, c, i->first);
		cpl_vector_set(v2, c, i->second);
    ++c;
	}

	return ret;
}

/*
 * Create a spectrum from three vectors
 */
hdrl_spectrum1D * create_spectrum(
    const std::vector<double>& wave_vec,
    const std::vector<double>& flux_vec,
    const std::vector<double>& err_vec)
{
    size_t sz = wave_vec.size();
    const double* flux = &flux_vec[0];
    const double* flux_e = &err_vec[0];
    const double* wavs = &wave_vec[0];
    cpl_image * flx = cpl_image_wrap_double(sz, 1, (double *)flux);
    cpl_image * flx_e = cpl_image_wrap_double(sz, 1, (double *)flux_e);
    cpl_array * wav = cpl_array_wrap_double((double *)wavs, sz);

    hdrl_spectrum1D * s = hdrl_spectrum1D_create(flx, flx_e, wav, hdrl_spectrum1D_wave_scale_linear);

    cpl_image_unwrap(flx);
    cpl_image_unwrap(flx_e);
    cpl_array_unwrap(wav);

    return s;
}

/*
 * Extract the wavelength from a spectrum and return it
 * as a vector
 */
std::vector<double> extract_wavelength(const hdrl_spectrum1D* spectrum)
{
    cpl_size sz = hdrl_spectrum1D_get_size(spectrum);
    double * lambdas = (double *)cpl_calloc(sz ,sizeof(double));
    for(cpl_size i = 0; i < sz; i++){
        lambdas[i] = hdrl_spectrum1D_get_wavelength_value(spectrum, i, NULL);
    }
    std::vector<double> w(lambdas, lambdas + sz);
    cpl_free(lambdas);
    return w;
}

/*
 * Extract the flux from a spectrum and return it
 * as a vector
 */
std::vector<double> extract_flux(const hdrl_spectrum1D* spectrum)
{
    cpl_size sz = hdrl_spectrum1D_get_size(spectrum);
    const cpl_image* flx = hdrl_image_get_image_const(hdrl_spectrum1D_get_flux(spectrum));
    const double* flxdata = cpl_image_get_data_double_const(flx);
    std::vector<double> f(flxdata, flxdata + sz);
    return f;
}

/*
 * Load vectors from a cpl_table
 */
void load_from_table(cpl_table* table,
    const char* wavecol, const char* fluxcol, const char* errcol,
    std::vector<double>& wave_vec, std::vector<double>& flux_vec, std::vector<double>& err_vec)
{
    double tmpwave, tmpflux, tmperr;
    int invalid = 0;
    cpl_size arr_size = cpl_table_get_nrow(table);
    cpl_msg_info(cpl_func, "Loading data from table of length %lld, using columns %s %s %s", arr_size, wavecol, fluxcol, errcol);
    for(cpl_size irow = 0; irow < arr_size; ++irow) {
        tmpwave = cpl_table_get(table, wavecol, irow, &invalid);
        tmpflux = cpl_table_get(table, fluxcol, irow, &invalid);
        if (errcol != NULL) {
            tmperr = cpl_table_get(table, errcol, irow, &invalid);
        } else {
            tmperr = 0;
        }
        if (!invalid) {
            wave_vec.push_back(tmpwave);
            flux_vec.push_back(tmpflux);
            err_vec.push_back(tmperr);
        }
    }
}

/*
 * Load vectors from a cpl_table, where
 * the data is stored as an array.
 */
void load_from_table_array(cpl_table* table,
    const char* wavecol, const char* fluxcol, const char* errcol,
    std::vector<double>& wave_vec, std::vector<double>& flux_vec, std::vector<double>& err_vec)
{
    const cpl_array* wave_arr = cpl_table_get_array(table, wavecol, 0);
    const cpl_array* flux_arr = cpl_table_get_array(table, fluxcol, 0);
    const cpl_array* err_arr = (errcol != NULL ? cpl_table_get_array(table, errcol, 0) : NULL);
    double tmpwave, tmpflux, tmperr;
    int invalid = 0;
    cpl_size arr_size = cpl_array_get_size(wave_arr);
    if (cpl_array_get_size(flux_arr) != arr_size) {
        cpl_msg_error(cpl_func, "Table does not have the required format");
        return;
    }
    if ((err_arr != NULL) && (cpl_array_get_size(err_arr) != arr_size)) {
        cpl_msg_error(cpl_func, "Table does not have the required format");
        return;
    }
    cpl_msg_info(cpl_func, "Loading data from array of length %lld, using columns %s %s %s", arr_size, wavecol, fluxcol, errcol);

    for(cpl_size irow = 0; irow < arr_size; ++irow) {
        int invalid = 0;
        tmpwave = cpl_array_get_double(wave_arr, irow, &invalid);
        tmpflux = cpl_array_get_double(flux_arr, irow, &invalid);
        if (err_arr != NULL) {
            tmperr = cpl_array_get_double(err_arr, irow, &invalid);
        } else {
            tmperr = 0;
        }
        if (!invalid) {
            wave_vec.push_back(tmpwave);
            flux_vec.push_back(tmpflux);
            err_vec.push_back(tmperr);
        }
    }
}
