# -*- coding: utf-8 -*-
# ========================================================================================================
# Jose A. Escartin
# 2019.06.03
#
# A python interactive window for use in the Molecfit Reflex workflow.
# This interactive window shows the results of the molecfit_calctrans recipe.   
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
# 1. Recipe parameters:
#
#    --USE_ONLY_INPUT_PRIMARY_DATA  In order to use only the primary extension of the input SCIENCE FITS file. [FALSE]
#    --USE_DATA_EXTENSION_AS_DFLUX       If you use only primary extension data, you can define other extension as DFLUX (error flux) [Default = 0, Not used].[0]
#    --USE_DATA_EXTENSION_AS_MASK        If you use only primary extension data, you can define other extension as MASK [Default = 0, Not used].[0]
#    --USE_INPUT_KERNEL                  In order to use kernel library if it is provide by the user. [TRUE]
#    --CALCTRANS_MAPPING_KERNEL          Mapping 'SCIENCE' - 'CALCTRANS_KERNEL_LIBRARY' [string with ext_number comma separated (int)] [NULL]
#    --MAPPING_ATMOSPHERIC               Mapping 'SCIENCE' - 'ATM_PARAMETERS' [string with ext_number comma separated (int)] [NULL]
#    --MAPPING_CONVOLVE                  Mapping 'LBLRTM_RESULT' - 'TELLURIC_CORR' [string with ext_number comma separated (int)] [NULL]
#
#
# Images to plot (TAG's): 
#
#    * SCIENCE                           The original input data
#    * LBLRTM_RESULTS                    For every input spectrum extension generate n-range extensions
#    * TELLURIC_DATA                     Telluric correction applied  for each extension 
#    * TELLURIC_CORR                     Telluric correction to apply for each extension
#
# ========================================================================================================

from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys

try:
    import numpy
    import os
    import re
    import reflex
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    from matplotlib.text import Text
    from pylab import *
    try:
        from astropy.io import fits as pyfits
    except ImportError:
        import pyfits
    import_success = True
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")

from molecfit_common import *

def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)


class DataPlotterManager(object):

    # static members
    recipe_name = "molecfit_calctrans"
    telluric_data_cat = [
        "TELLURIC_DATA",
        "TELLURIC_DATA_VIS",
        "TELLURIC_DATA_NIR",
    ]

    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        """
        This function specifies which are the parameters that should be presented
        in the window to be edited.  Note that the parameter has to also be in the
        in_sop port (otherwise it won't appear in the window). The descriptions are
        used to show a tooltip. They should match one to one with the parameter
        list.
        """
        self.orig_init_sop=init_sop_dict(interactive_app.inputs.in_sop)
        all_RecipeParameters=[

            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_ONLY_INPUT_PRIMARY_DATA",
                                   group="Recipe", description="In order to use only the primary extension of the input SCIENCE FITS file."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_DATA_EXTENSION_AS_DFLUX",
                                   group="Recipe", description="If you use only primary extension data, you can define other extension as DFLUX (error flux) [Default = 0, Not used]."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_DATA_EXTENSION_AS_MASK",
                                   group="Recipe", description="If you use only primary extension data, you can define other extension as MASK [Default = 0, Not used]."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_INPUT_KERNEL",
                                   group="Recipe", description="In order to use kernel library if it is provide by the user."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="CALCTRANS_MAPPING_KERNEL",
                                   group="Recipe", description="Mapping 'SCIENCE' - 'CALCTRANS_KERNEL_LIBRARY' [string with ext_number comma separated (int)]."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="MAPPING_ATMOSPHERIC",
                                   group="Recipe", description="Mapping 'SCIENCE' - 'ATM_PARAMETERS' [string with ext_number comma separated (int)]."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="MAPPING_CONVOLVE",
                                   group="Recipe", description="Mapping 'LBLRTM_RESULT' - 'TELLURIC_CORR' [string with ext_number comma separated (int)]."),
        ]
        recipe_RecipeParameters=[]
        for p in all_RecipeParameters :
            if p.displayName in self.orig_init_sop['by_name'].keys() :
                recipe_RecipeParameters+=[p,]
        return recipe_RecipeParameters

    def readFitsData(self, fitsFiles):
        """
        This function should be used to read and organize the raw fits files
        produced by the recipes.
        It receives as input a list of reflex.FitsFiles
        """

        self.frames = dict()
        self.input_errors=1; 

        # Loop on all FITS files_input 
        for f in fitsFiles:

            # For each sci_reconstructed image
            if f.category in self.telluric_data_cat :

                # Initialize
                self.data_extnames = []
                self.data_lambda = dict()
                self.data_flux = dict()
                self.data_cflux = dict()
                self.data_mtrans = dict()
                self.data_units = dict()

                # Get the expected files
                try:
                    telluric_data = PipelineProduct(f)
                    self.input_errors=0
                except: 
                    break 

                # Loop on extensions
                ext_num = -1
                for telluric_data_ext in telluric_data.all_hdu:

                    # Take extension in the output file
                    ext_num = ext_num + 1

                    naxis = telluric_data_ext.header['NAXIS']
                    if (naxis == 2):

                        if ('EXTNAME' in telluric_data_ext.header) :
                            extname = telluric_data_ext.header['EXTNAME']
                        else :
                            extname = 'EXT.' + str(ext_num)
                        self.data_extnames.append(extname)

                        # Get infos from telluric_data using the extname
                        self.data_lambda[extname] = telluric_data_ext.data.field("lambda")
                        self.data_flux[extname] = telluric_data_ext.data.field("flux")
                        self.data_cflux[extname] = telluric_data_ext.data.field("cflux")
                        self.data_mtrans[extname] = telluric_data_ext.data.field("mtrans")
                        self.data_units[extname] = "$\mu$m"

                # Set the plotting functions
        self._add_subplots = self._add_subplots
        self._plot = self._data_plot


    def addSubplots(self, figure):
        """
        This function should be used to setup the subplots of the gui.  The the
        matplotlib documentation for a description of subplots.
        """
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        """
        This function should be used to plot the data onto the subplots.
        """
        self._plot()

    def plotWidgets(self) :
        widgets = list()
        # Radio button
        if self.input_errors ==0:
            if len(self.data_extnames) > 1:
             self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(self.axradiobutton, self.setRadioCallback, 
                                                                            self.data_extnames, 0, title='EXT selection')
             widgets.append(self.radiobutton)
        return widgets

    def setRadioCallback(self, label) :

        # Plot Spectrum1
        specdisp1 = pipeline_display.SpectrumDisplay()
        self.spec1_plot.clear()
        specdisp1.setLabels(r"$\lambda$["+r"$\mu$m]"+r" (blue: input spectrum, red: corrected spectrum)", self._process_label(self.data_units[label]) )
        specdisp1.display(self.spec1_plot, "Input spectrum", self._data_plot1_get_tooltip(label), 
                self.data_lambda[label], self.data_flux[label])
        specdisp1.overplot(self.spec1_plot, self.data_lambda[label], self.data_cflux[label], 'red')

        # Plot Spectrum2
        specdisp2 = pipeline_display.SpectrumDisplay()
        self.spec2_plot.clear()
        specdisp2.setLabels(r"$\lambda$["+r"$\mu$m]", "Atmospheric transmission" )
        specdisp2.display(self.spec2_plot, "Telluric correction", self._data_plot2_get_tooltip(label), 
                self.data_lambda[label], self.data_mtrans[label])


    def _add_subplots(self, figure):
        try:
            lll = len(self.data_extnames)
        except:
            lll=0
        if lll > 1:
            gs = gridspec.GridSpec(2, 4)
            self.axradiobutton = figure.add_subplot(gs[0:2, 0  ])
            self.spec1_plot    = figure.add_subplot(gs[0,   1:4])
            self.spec2_plot    = figure.add_subplot(gs[1,   1:4])
        else:
            gs = gridspec.GridSpec(2, 1)
            self.spec1_plot    = figure.add_subplot(gs[0,   0])
            self.spec2_plot    = figure.add_subplot(gs[1,   0])

    def _data_plot1_get_tooltip(self, extname):
        # Create the tooltip
        tooltip = " \
            EXT       : %s \n \
            blue      : Input spectrum \n \
            red       : Best fit molecfit model" % (extname)
        return tooltip


    def _data_plot2_get_tooltip(self, extname):
        # Create the tooltip
        tooltip = " \
            EXT       : %s" % (extname)
        return tooltip

    def _data_plot(self):

        # Plot Spectrum1
        specdisp1 = pipeline_display.SpectrumDisplay()

        # Plot Spectrum2
        specdisp2 = pipeline_display.SpectrumDisplay()

        if self.input_errors ==0:

            extname = self.data_extnames[0]
            specdisp1.setLabels(r"$\lambda$["+r"$\mu$m]"+r" (blue: Input spectrum, red: corrected spectrum)", self._process_label(self.data_units[extname]) )
            specdisp1.display(self.spec1_plot, "Input spectrum", self._data_plot1_get_tooltip(extname), 
                         self.data_lambda[extname], self.data_flux[extname])
            specdisp1.overplot(self.spec1_plot, self.data_lambda[extname], self.data_cflux[extname], 'red')

            specdisp2.setLabels(r"$\lambda$["+r"$\mu$m]",  "Atmospheric transmission")
            specdisp2.display(self.spec2_plot, "Telluric correction", self._data_plot2_get_tooltip(extname), 
                         self.data_lambda[extname], self.data_mtrans[extname])
        else: 
            self.spec1_plot.set_xlabel('INPUT ERROR')
            self.spec1_plot.set_ylabel('INPUT ERROR')
            self.spec1_plot.set_title('INPUT ERROR: missing data files '+self.telluric_data_cat)

            self.spec2_plot.set_xlabel('INPUT ERROR')
            self.spec2_plot.set_ylabel('INPUT ERROR')
            self.spec2_plot.set_title('INPUT ERROR: missing data files '+self.telluric_data_cat)



    def _process_label(self, in_label):
        # If known, 'pretty print' the label
        if (in_label == "erg.s**(-1).cm**(-2).angstrom**(-1)"):
            return "Flux [erg sec" + r"$^{-1}$"+"cm" + r"$^{-2}$" + r"$\AA^{-1}$]"
        else:
            return in_label

#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    #Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        #Get the specific functions for this window
        dataPlotManager = DataPlotterManager()
        dataPlotManager.recipe_name=interactive_app.inputs.in_sop[0].recipe

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    # Append the BEST_FIT_PARAMETERS to the out_sof
    # so that we will have it for molecfit_correct and
    # we can append it (as an extension) to the product file(s)
    for file in interactive_app.inputs.in_sof_rec_orig.files :
        if file.category in ['BEST_FIT_PARAMETERS',] :
            interactive_app.inputs.in_sof.files+=[file,]

    #Print outputs. This is parsed by the Reflex python actor to
    #get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
