# -*- coding: utf-8 -*-
# ========================================================================================================
# Jose A. Escartin
# 2019.06.03
#
# A python interactive window for use in the Molecfit Reflex workflow.
# This interactive window shows the results of the molecfit_model recipe.
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
# 1. Recipe parameters:
#
#    --USE_ONLY_INPUT_PRIMARY_DATA  In order to use only the primary extension of the input STD_MODEL/SCIENCE FITS file. [FALSE]
#    --USE_DATA_EXTENSION_AS_DFLUX       If you use only primary extension data, you can define other extension as DFLUX (error flux) [Default = 0, Not used]. [0]
#    --USE_DATA_EXTENSION_AS_MASK        If you use only primary extension data, you can define other extension as MASK [Default = 0, Not used]. [0]
#    --USE_INPUT_KERNEL                  In order to use kernel library if it is provide by the user. [TRUE]
#    --MODEL_MAPPING_KERNEL              Mapping 'STD_MODEL/SCIENCE' - 'MODEL_KERNEL_LIBRARY' [string with ext_number comma separated (int)]. [NULL]
#    --LIST_MOLEC                        List of molecules to be included in the model (N_val: nmolec) [string with molecules comma separated]. [NULL]
#    --FIT_MOLEC                         Fit flags for molecules -- 1 = yes; 0 = no (N_val: nmolec) [string wit ints comma separated]. [NULL]
#    --REL_COL                           Values of molecular columns, expressed relatively to the input ATM profile columns (N_val: nmolec) [string with doubles comma separated]. [NULL]
#    --WAVE_INCLUDE                      Wavelength ranges included [string with couples wv_number comma separated (double)]. [NULL]
#    --WAVE_EXCLUDE                      Wavelength ranges excluded [string with couples wv_number comma separated (double)]. [NULL]
#    --PIXEL_EXCLUDE                     Pixel ranges excluded [string with couples pixel_number comma separated (int)]. [NULL]
#
# 2. Molecfit parameters
#
#    --TELLURICCORR_PATH                 Installation directory. [TELLURICCORR_PARAMETER_DEFAULT]
#    --TELLURICCORR_DATA_PATH            Data directory. [TELLURICCORR_PARAMETER_DEFAULT]
#    --TMP_PATH                          Temporary directory. [TELLURICCORR_PARAMETER_DEFAULT]
#    --OUTPUT_PATH                       Directory for output files. [TELLURICCORR_PARAMETER_DEFAULT]
#    --OUTPUT_NAME                       Root name of output files. [out]

#    --OPENMP_THREADS                    Number of OPENMP threads in the executions. [1]
#    --SILENT_EXTERNAL_BINS              Silent the output of the external binaries. [TRUE]
#    --TRANSMISSION                      Type of input spectrum : FALSE=Emission(radiance); TRUE=Transmission. [TRUE]
#    --COLUMN_LAMBDA                     Wavelength column ('NULL' can be used if the file is an image and that the data are in the primary (data are given by header keywords). [lambda]
#    --COLUMN_FLUX                       Flux column. [flux]
#    --COLUMN_DFLUX                      Flux error column (Avoided by writing 'NULL') : 1-sigma error on the flux. [NULL]
#    --COLUMN_MASK                       Mask column (Avoided by writing 'NULL') : Indicates if a pixel is invalid. [NULL]
#    --MASK_BINARY                       Flag: true, if the mask is a binary (0/1); false, if the mask is an integer (0 to 255). [TRUE]
#    --MASK_THRESHOLD                    Only used if mask_binary is set to false, in which case, it indicates the maximum value of the mask to be used by the fit. [0]
#    --DEFAULT_ERROR                     Default error relative to mean for the case that the error column is not provided. [0.01]
#    --WLG_TO_MICRON                     Multiplicative factor applied to the wavelength to express is in micron (E.g.: if wavelength is given in nm, the value should be 0.001). [1.]
#    --WAVELENGTH_FRAME                 Wavelengths in vacuum = Wavelength in vacuum = 'VAC'; Wavelength in air with the observatory reference frame = 'AIR'; Wavelength in vacuum with another reference frame = 'VAC_RV'. [VAC]
#    --OBS_ERF_RV_KEY                    The radial velocity of the observatory in km/s relative to the external reference frame. [NONE]
#    --OBS_ERF_RV_VALUE                  If OBS_ERF_RV_KEYWORD=='NONE' take this value. [0.]
#    --CLEAN_MODEL_FLUX                  Set model flux to 0 for non-fitted pixels. [FALSE]

#    --FTOL                              Relative chi-square convergence criterion. [1.e-10]
#    --XTOL                              Relative parameter convergence criterion. [1.e-10]
#    --FLUX_UNIT                         Conversion of fluxes from phot/(s*m2*mum*as2) (emission spectrum only) to flux unit of observed spectrum. [0]
#    --FIT_TELESCOPE_BACKGROUND          Fit of telescope background -- 1 = yes; 0 = no (emission spectrum only). [TRUE]
#    --TELESCOPE_BACKGROUND_CONST        Initial value for telescope background fit. [0.1]
#    --FIT_CONTINUUM                     Flag to enable/disable the polynomial fit of the continuum. [TRUE]
#    --CONTINUUM_N                       Degree of polynomial continuum fit. [0]
#    --CONTINUUM_CONST                   Initial constant term for continuum fit (valid for all fit ranges). [1.]
#    --CONST_BARY_RV                       Only used if a continuum spectrum is provided. [0.]
#    --MAP_REGIONS_TO_CHIP : Comma deliminated string of chip indices that each range is associated with.
#                          If set to NULL, check if the TAG[WAVE_INCLUDE] points to a FITS BINTABLE with column MAPPED_TO_CHIP provided. [1]
#    --FIT_WLC                           Flag to enable/disable the refinement of the wavelength solution. [TRUE]
#    --WLC_N                             Polynomial degree of the refined wavelength solution. [1]
#    --WLC_CONST                         Initial constant term for wavelength adjustment (shift relative to half wavelength range). [0.]
#    --WLC_REF                           Indicates that the reference for the wavelength calibration : If it is set to 'DATA' is the input  data; If it is set to 'MODEL', is the output model. [DATA]
#    --FIT_RES_BOX                       Fit resolution by Boxcar LSF. [TRUE]
#    --RES_BOX                           Initial value for FWHM of Boxcar rel. to slit width at the centre of the spectrum. [1.]
#    --FIT_RES_GAUSS                     Fit resolution by Gaussian. [TRUE]
#    --RES_GAUSS                         Initial value for FWHM of the Gaussian in pixels at the centre of the spectrum. [1.]
#    --FIT_RES_LORENTZ                   Fit resolution by Lorentzian. [TRUE]
#    --RES_LORENTZ                       Initial value for FWHM of the Lorentz in pixels at the centre of the spectrum. [1.]
#    --KERNMODE                          Voigtian profile approximation instead of independent Gaussian and Lorentzian?. [FALSE]
#    --KERNFAC                           Size of Voigtian/Gaussian/Lorentzian kernel in FWHM. [3.]
#    --VARKERN                           Does the kernel size increase linearly with wavelength?. [FALSE]

#    --OBSERVING_DATE_KEYWORD            Observing date in years or MJD in days. [MJD-OBS]
#    --OBSERVING_DATE_VALUE              If OBSERVING_DATE_KEYWORD=='NONE' take this value. [-1.]
#    --UTC_KEYWORD                       UTC in s. [UTC]
#    --UTC_VALUE                         If UTC_KEYWORD=='NONE' take this value. [-1.]
#    --TELESCOPE_ANGLE_KEYWORD           Telescope altitude angle in deg. [ESO TEL ALT]
#    --TELESCOPE_ANGLE_VALUE             If TELESCOPE_ANGLE_KEYWORD=='NONE' take this value. [90.]
#    --RELATIVE_HUMIDITY_KEYWORD         Relative humidity in %. [ESO TEL AMBI RHUM]
#    --RELATIVE_HUMIDITY_VALUE           If RELATIVE_HUMIDITY_KEYWORD=='NONE' take this value. [15.]
#    --PRESSURE_KEYWORD                  Pressure in hPa. [ESO TEL AMBI PRES START]
#    --PRESSURE_VALUE                    If PRESSURE_KEYWORD=='NONE' take this value. [750.]
#    --TEMPERATURE_KEYWORD               Ambient temperature in deg C. [ESO TEL AMBI TEMP]
#    --TEMPERATURE_VALUE                 If TEMPERATURE_KEYWORD=='NONE' take this value. [15.]
#    --MIRROR_TEMPERATURE_KEYWORD        Mirror temperature in deg C. [ESO TEL TH M1 TEMP]
#    --MIRROR_TEMPERATURE_VALUE          If MIRROR_TEMPERATURE_KEYWORD=='NONE' take this value. [15.]
#    --ELEVATION_KEYWORD                 Elevation above sea level in m (default is Paranal: 2635. m). [ESO TEL GEOELEV]
#    --ELEVATION_VALUE                   If ELEVATION_KEYWORD=='NONE' take this value. [2635.]
#    --LONGITUDE_KEYWORD                 Longitude (default is Paranal: -70.4051 deg). [ESO TEL GEOLON]
#    --LONGITUDE_VALUE                   If LONGITUDE_KEYWORD=='NONE' take this value. [-70.4051]
#    --LATITUDE_KEYWORD                  Latitude (default is Paranal: -24.6276 deg). [ESO TEL GEOLAT
#    --LATITUDE_VALUE                    If LATITUDE_KEYWORD=='NONE' take this value. [-24.6276]

#    --SLIT_WIDTH_KEYWORD                Slit width in arcsec (taken from FITS header if present). [ESO INS SLIT1 WID
#    --SLIT_WIDTH_VALUE                  If SLIT_WIDTH_KEYWORD=='NONE' take this value. [0.4]
#    --PIX_SCALE_KEYWORD                 Pixel scale in arcsec (taken from this file only). [NONE]
#    --PIX_SCALE_VALUE                   If PIX_SCALE_KEYWORD=='NONE' take this value. [0.086]

#    --REFERENCE_ATMOSPHERIC             Reference atmospheric profile.  [equ.fits]
#    --GDAS_PROFILE                      Specific GDAS-like input profile (P[hPa] HGT[m] T[K] RELHUM[%]). [auto]
#    --LAYERS                            Grid of layer heights for merging ref_atm and GDAS profile. [TRUE]
#    --EMIX                              Upper mixing height in km for considering data of a local meteo station. [5.]
#    --PWV                               PWV value in mm for the input water vapor profile. [-1.]
#
# 3. LNFL parameters
#
#    --LNFL_LINE_DB                      File name of the line list (must be stored in the directory : ({TELLURICCORR_DATA_PATH}/hitran/). [aer_v_3.6]
#    --LNFL_LINE_DB_FORMAT               Format of the line file: gives the length in terms of characters per line. [100.]
#
# 4. LBLRTM parameters
#
#    --LBLRTM_ICNTNM                     Continua and Rayleigh extinction (0,1,2,3,4,5). [5]
#    --LBLRTM_IAERSL                     Aerosols [0,1]. [0]
#    --LBLRTM_MPTS                       Number of optical depth values. [5]
#    --LBLRTM_NPTS                       Number of values for each panel. [5]
#    --LBLRTM_V1                         Beginning wavenumber value for the calculation. [1.9]
#    --LBLRTM_V2                         Ending wavenumber value for the calculation. [2.4]
#    --LBLRTM_SAMPLE                     Number of sample points per mean halfwidth [between 1 to 4]. [4]
#    --LBLRTM_ALFAL0                     Average collision broadened halfwidth [cm-1/atm]. [0.]
#    --LBLRTM_AVMASS                     Average molecular mass [amu] for Doppler halfwidth. [0.]
#    --LBLRTM_DPTMIN                     Minimum molecular optical depth below which lines will be rejected. [0.0002]
#    --LBLRTM_DPTFAC                     Factor multiplying molecular continuum optical depth. [0.001]
#    --LBLRTM_TBOUND                     Temperature of boundary [K]. [0.]
#    --LBLRTM_SREMIS1                    Emissivity coefficient 1. [0.]
#    --LBLRTM_SREMIS2                    Emissivity coefficient 2. [0.]
#    --LBLRTM_SREMIS3                    Emissivity coefficient 3. [0.]
#    --LBLRTM_SRREFL1                    Reflectivity coefficient 1. [0.]
#    --LBLRTM_SRREFL2                    Reflectivity coefficient 2. [0.]
#    --LBLRTM_SRREFL3                    Reflectivity coefficient 3. [0.]
#    --LBLRTM_MODEL                      Atmospheric profile (0,1,2,3,4,5,6). [0]
#    --LBLRTM_ITYPE                      Type of path (1,2,3). [3]
#    --LBLRTM_NOZERO                     Zeroing of small amounts of absorbers (0,1). [0]
#    --LBLRTM_NOPRNT                     Do not print output? (0,1). [0]
#    --LBLRTM_IPUNCH                     Write out layer data to TAPE7 [0,1]. [0]
#    --LBLRTM_RE                         Radius of earth [km]. [0.]
#    --LBLRTM_HSPACE                     Altitude definition for space [km]. [120.]
#    --LBLRTM_H1                         Observer altitude [km]. [2.64]
#    --LBLRTM_H2                         Upper height limit [km]. [0.]
#    --LBLRTM_RANGE                      Length of a straight path from H1 to H2 [km]. [0.]
#    --LBLRTM_BETA                       Earth centered angle from H1 to H2 [degrees]. [0.]
#    --LBLRTM_LEN                        Path length (0,1). [0]
#    --LBLRTM_HOBS                       Height of observer. [0.]
#    --LBLRTM_AVTRAT                     Maximum Voigt width ratio across a layer. [2.]
#    --LBLRTM_TDIFF1                     Maximum layer temperature difference at ALTD1 [K]. [5.]
#    --LBLRTM_TDIFF2                     Maximum layer temperature difference at ALTD2 [K]. [8.]
#    --LBLRTM_ALTD1                      Altitude of TDIFF1 [km]. [0.]
#    --LBLRTM_ALTD2                      Altitude of TDIFF2 [km]. [0.]
#    --LBLRTM_DELV                       Number of wavenumbers [cm-1] per major division. [1.]
#
#
# Images to plot (TAG's):
#
#    * STD_MODEL/SCIENCE                 The original input data
#    * ATM_PARAMETERS                    Atmospheric parameters
#    * BEST_FIT_PARAMETERS               Best fit parameters
#    * BEST_FIT_MODEL                    Best fit model
#
# ========================================================================================================

from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
try :
    # JPR: 2022-03-07: don't know what these are for
    # And they break this module on CentOS-7 & FC-28 (for example)
    # so put them in a try/except to have them when available and
    # ignore (which seems to work fine) when not...
    from asyncio.proactor_events import _ProactorBaseWritePipeTransport
    from statistics import fmean
except :
    pass
import collections
import sys
import importlib
import copy as sys_copy
import json
import warnings

import matplotlib
#from matplotlib.axes import Axes
#from matplotlib import _version as mp_version

from matplotlib.figure import Figure
from matplotlib.backends.backend_wxagg import FigureCanvasWxAgg

try:
    import numpy
    from numpy.polynomial import Polynomial
    import os
    import re
    import reflex
    from reflex_interactive_gui import FloatValidator,_all_reflex_apps
    from reflex_interactive_gui import ReflexInteractiveWxApp as reflex_ReflexInteractiveWxApp
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    from matplotlib.text import Text
    from matplotlib.widgets import SpanSelector
    import wx
    import wx.lib.scrolledpanel
    from pylab import *
    try:
        from astropy.io import fits as pyfits
        from astropy.table import Table
        from astropy.time import Time
        _have_Table=True
    except ImportError:
        import pyfits
    try :
        am = importlib.import_module('auto_molecule')
        iu = importlib.import_module('inst_utils')
    except ImportError:
        sys.path.insert(0, '%s/instruments' %(os.path.dirname(__file__)))
        am = importlib.import_module('auto_molecule')
        iu = importlib.import_module('inst_utils')
    except Exception as e:
        raise(e)
    from compare_versions import compare_versions
    try :
        from astropy.convolution import Gaussian1DKernel, Box1DKernel, Model1DKernel
        from astropy.modeling.functional_models import Lorentz1D
        from astropy.convolution import convolve, convolve_fft
        have_convolution = True
    except :
        have_convolution = False
    try :
        from astropy.modeling.polynomial import Chebyshev1D
        have_chebyshev=True
    except :
        have_chebyshev = False
    try :
        from specutils.spectra.spectrum1d import Spectrum1D
        from specutils.fitting.continuum import fit_continuum
        import astropy.units as u
        have_specutils=True
    except :
        have_specutils=False

    import_success = True
except ImportError as e:
    import_success = False
    _have_Table=False
    have_convolution = False
    print(e)
    print("Error importing modules pyfits, wx, matplotlib, numpy")

from molecfit_common import *

enable_gcf=os.environ.get('MF_ENABLE_GCF','false').lower() == 'true'

'''
try:
    from packaging import version
    version.parse(matplotlib.__version__)
except :
    from compare_versions import compare_versions
'''
# ---------------------------------------------------------------------------------------
# MouseButton is available from mpl-3.??? something on.
# Create a simple class incase it is on (e.g. on CentOS-7+py27)
try :
    if MouseButton.LEFT is not None :
        pass
    pass
except :
    class MouseButton :
        LEFT=1
        MIDDLE=2
        RIGHT=3
# ---------------------------------------------------------------------------------------

# Full list of molecules...
# [ INFO  ] molecfit_model: N2,O2,CO2,O3,H2O,CH4,N2O,HNO3,CO,NO2,N2O5,CLO,HOCL,CLONO2,NO,HNO4,HCN,NH3,F11,F12,F14,F22,CCL4,COF2,H2O2,C2H2,C2H6,OCS,SO2,SF6
m_specs=[
    'N2',
    'H2O',
    'CO2',
    'O3',
    'N2O',
    'CO',
    'CH4',
    'O2',
]
trace_m_specs=[
    'HNO3',
    'NO2',
    #'N2O5', #[ ERROR ] cpl_errorstate_dump_one_level:   [1/1] 'Illegal input: Invalid object value(s): cpl_table params->molectab (molecule N2O5 cannot be handled)' (14) at mf_atm_combined_gdas:mf_atm_combined_gdas.c:307
    'CLO',
    'HOCL',
    'CLONO2',
    'NO',
    'HNO4',
    'HCN',
    'NH3',
    'F11',
    'F12',
    'F14',
    'F22',
    'CCL4',
    'COF2',
    'H2O2',
    'C2H2',
    'C2H6',
    'OCS',
    'SO2',
    'SF6',
]
in_sop_by_name_defaults={
    'xsh_molecfit_model': {
        'COLUMN_LAMBDA': 'WAVE',
        'COLUMN_FLUX': 'FLUX',
        'USE_ONLY_INPUT_PRIMARY_DATA': False,
        'WLG_TO_MICRON': 0.001,
        'WAVELENGTH_FRAME': 'AIR',
        'OBS_ERF_RV_KEY': 'NULL',
        'MAP_REGIONS_TO_CHIP': "1,1",
    },
    'fors_molecfit_model': {
        'COLUMN_LAMBDA': 'WAVE',
        'COLUMN_FLUX': 'FLUX',
        'USE_ONLY_INPUT_PRIMARY_DATA': False,
        'WLG_TO_MICRON': 0.0001,
        'WAVELENGTH_FRAME': 'AIR',
        'OBS_ERF_RV_KEY': 'NULL',
        'MAP_REGIONS_TO_CHIP': "1,1",
    },
}
# ---------------------------------------------------------------------------------------
def get_attrs(klass):
   return [
      k for k in klass.__dict__.keys()
         if not k.startswith('__')
         and not k.endswith('__')
   ]
# ---------------------------------------------------------------------------------------
def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)

# ---------------------------------------------------------------------------------------
def toggle_UseBestFitParameters(event):
    print("Use Best Fit Parameters as initial values is set to %s" %(event.EventObject.GetValue()))
# ---------------------------------------------------------------------------------------
def fib_disp_choice(event):
    print("Selected fibre is %s" %(event.EventObject.GetString(event.EventObject.GetSelection())))
# ---------------------------------------------------------------------------------------
def wlR(x,wl,r) :
    return x[numpy.where(numpy.logical_and(wl>=r[0],wl<=r[-1]))]
# ---------------------------------------------------------------------------------------
'''
def init_sop_dict( init_sop, existing_dict=None ) :
    if init_sop is None :
        return existing_dict
    d=existing_dict or {}
    d['init_sop']=init_sop
    d['by_name']={
        p.name: i for i,p in enumerate(init_sop)
    }
    return d
'''
# ---------------------------------------------------------------------------------------
def init_sop_dict_param( d, p ) :
    if p not in d['by_name'] : return None
    return d['init_sop'][d['by_name'][p]]
# ---------------------------------------------------------------------------------------
def hdulist_strip_checksums( hdulist, keys=['CHECKSUM', 'DATASUM'] ) :
    for hdu in hdulist :
        for k in keys :
            if k in hdu.header :
                del hdu.header[k]
# ---------------------------------------------------------------------------------------
class ReflexInteractiveWxApp(reflex_ReflexInteractiveWxApp):

    def __init__(self, interactive_app, dataPlotManager):
        # The fits_viewer_children list stores Popen objects created in the
        # onItemActivated handler method. These will be terminated in the
        # OnExit handler.
        self.fits_viewer_children = []

        self.inter_app = interactive_app
        self.dataPlotManager = dataPlotManager

        # Initialize wx GUI application
        wx.App.__init__(self, 0)

        # Add this to the set of all objects to be handled in the _on_terminate
        # signal handler.
        _all_reflex_apps.add(self)

    '''
    def onResize(self, event):
        self.figureCanvas.SetSize(self.plotPanel.GetSize())
        if sys.platform == "darwin":
            ebn=self.inter_app.wxApp.dataPlotManager.wxApp_elements_by_name
            cp_size=self.ctrlPanel.GetSize()

            ebn['parameterPanel'][0].SetMaxSize(wx.Size(cp_size.width-40,-1))
            ebn['parameterPanel'][0].SetSize(ebn['parameterPanel'][0].GetMaxSize())

            self.parameterNotebook.SetMaxSize(
                wx.Size(ebn['parameterPanel'][0].GetSize().width-20,-1)
            )
            self.parameterNotebook.SetSize(self.parameterNotebook.GetMaxSize())
            for w in self.parameterNotebook.Children :
                w.SetMaxSize(wx.Size(self.parameterNotebook.GetMaxSize().x-10,-1))
                w.SetSize(w.GetMaxSize())

            self.filesList.SetMaxSize(
                wx.Size(ebn['parameterPanel'][0].GetSize().width,-1)
            )
            self.filesList.SetSize(self.filesList.GetMaxSize())
        else :
            self.filesList.SetMaxSize(self.parameterNotebook.GetSize())
        """
        # sort out CtrlPanel geometry a bit...
        if sys.platform == "darwin":
            window_width=1280
            window_height=1024
            sash_pos=window_width-410
            pp_width=410
            b_width=20
            # This fails for py2.7, but that's OK since it's really only needed fo macOS...
            # and older wx versions handle the geometry better anyway...
            try :
                dataPlotManager.wxApp_elements_by_name['ReflexInteractiveWxApp'][0].SetSize(50,50,window_width,window_height)
                dataPlotManager.wxApp_elements_by_name['ctrlandpltSplitter'][0].SetSashPosition(window_width-(pp_width+4*b_width))


                #widget=dataPlotManager.wxApp_elements_by_name['filesInfo'][0]
                #widget.Destroy()

                cs=dataPlotManager.wxApp_elements_by_name['filesInfo'][0].Size
                dataPlotManager.wxApp_elements_by_name['filesInfo'][0].SetSize(pp_width+2*b_width,cs.height)
                ns=dataPlotManager.wxApp_elements_by_name['filesInfo'][0].Size
                dataPlotManager.wxApp_elements_by_name['filesInfo'][0].SetMaxSize(ns)

                dataPlotManager.wxApp_elements_by_name['m_staticline2'][0].SetSize(pp_width,2)

                dataPlotManager.wxApp_elements_by_name['parameterNotebook'][0].SetSize(pp_width,-1)
                dataPlotManager.wxApp_elements_by_name['parameterPanel'][0].SetSize(pp_width,-1)
                dataPlotManager.wxApp_elements_by_name['-1'][0].SetSize(pp_width+2*b_width,-1)
                #cs=dataPlotManager.wxApp_elements_by_name['ctrlPanel'][0].Size
                #dataPlotManager.wxApp_elements_by_name['ctrlPanel'][0].SetSize(cs)
                cs=dataPlotManager.wxApp_elements_by_name['scrolledpanel'][0].Size
                for sp in dataPlotManager.wxApp_elements_by_name['scrolledpanel'] :
                    sp.SetSize(cs.width+10,cs.height)
            except :
                pass
        """
    '''
# ---------------------------------------------------------------------------------------
class DataPlotterManager(object):

    # static members
    recipe_name = "molecfit_model"
    best_fit_model_cat = "BEST_FIT_MODEL"
    wave_include = "WAVE_INCLUDE"
    wave_exclude = "WAVE_EXCLUDE"
    pixel_exclude = "PIXEL_EXCLUDE"
    telluric_corr = "TELLURIC_DATA"
    have_telluric_corr = False
    y_scalefactor = 1
    #is_first_pass = True
    sugcf_apply=False
    sugcf_mode_degree=3
# ---------------------------------------------------------------------------------------
    def get_regions(self, d, p ) :
        if init_sop_dict_param( d, p ).value != 'NULL' :
            _wi=init_sop_dict_param( d, p ).value.split(',')
            if p in [
                'WAVE_INCLUDE',
                'WAVE_EXCLUDE',
            ] :
                if _wi[0].lower() not in [
                    'null',
                ] :
                    '''
                    _wi=self.wl_to_topo_vac(
                        np.array(_wi, dtype=float)
                    )
                    _wi=",".join([iu.wrfmt(x) for x in _wi]).split(',')
                    '''
                    pass
            if p not in ['WAVE_INCLUDE',] :
                return {
                    p: _wi,
                }
            _n=len(_wi)/2
            _pv={p : _wi}
            for param in [
                'MAP_REGIONS_TO_CHIP',
                'FIT_CONTINUUM',
                'CONTINUUM_N',
                #'FIT_WLC', # FIT_WLC is per CIP, not per region...
            ] :
                if init_sop_dict_param( d, param ) is not None :
                    _pv[param]=init_sop_dict_param( d, param ).value.split(',')
                    if len(_pv) == 1 :
                        _pv[param]=list(np.repeat(_pv[param],_n))
            '''
            _cff=init_sop_dict_param( d, 'FIT_CONTINUUM').value.split(',')
            if len(_cff) == 1 :
                _cff=list(np.repeat(_cff,_n))
            _cpo=init_sop_dict_param( d, 'CONTINUUM_N').value.split(',')
            if len(_cpo) == 1 :
                _cpo=list(np.repeat(_cpo,_n))
            _wlf=init_sop_dict_param( d, 'FIT_WLC').value.split(',')
            if len(_wlf) == 1 :
                _wlf=list(np.repeat(_wlf,_n))
            '''
            # WLC_N is an integer...
            #_wln=init_sop_dict_param( d, 'WLC_N').value.split(',')
            #if len(_wln) == 1 :
            #    _wln=list(np.repeat(_wln,_n))
            return _pv
        return None
# ---------------------------------------------------------------------------------------
    def get_object_Name(self, o):
        w_list=[]
        if o is not None :
            if hasattr(o, 'Name' ) :
                if o.Name is not None :
                    w_list+=[[o.Name, o],]
        return w_list
# ---------------------------------------------------------------------------------------
    def get_window_list(self, o):
        w_list=[]
        if isinstance(o, wx.WindowList) :
            for w in o :
                w_list+=self.get_window_list(w)
        for otype in ['TopWindow','Window','Sizer','Children'] :
            if hasattr(o, otype ) :
                o_otype=getattr(o,otype)
                if o_otype is not None :
                    w_list+=self.get_window_list(o_otype)
        w_list+=self.get_object_Name(o)
        return w_list
# ---------------------------------------------------------------------------------------
    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        """
        This function specifies which are the parameters that should be presented
        in the window to be edited.  Note that the parameter has to also be in the
        in_sop port (otherwise it won't appear in the window). The descriptions are
        used to show a tooltip. They should match one to one with the parameter
        list.
        """
        group_parameters={
            "Experienced": [
                "FIT_TELESCOPE_BACKGROUND",
                "FIT_TELESCOPE_BACKGROUND",
                "TELESCOPE_BACKGROUND_CONST",
                "FIT_CONTINUUM",
                "CONTINUUM_N",
                "CONTINUUM_CONST",
                "CONST_BARY_RV",
                "FIT_WLC",
                "WLC_N",
                "WLC_CONST",
                "WLC_REF",
                "FIT_RES_BOX",
                "RES_BOX",
                "FIT_RES_GAUSS",
                "RES_GAUSS",
                "FIT_RES_LORENTZ",
                "RES_LORENTZ",
                "KERNMODE",
                "KERNFAC",
                "VARKERN",
            ],
            "Expert": [
                "LIST_MOLEC",
                "FIT_MOLEC",
                "REL_COL",
                "WAVE_INCLUDE",
                "MAP_REGIONS_TO_CHIP",
                "WAVE_EXCLUDE",
                "PIXEL_EXCLUDE",
                "USE_ONLY_INPUT_PRIMARY_DATA",
                "USE_DATA_EXTENSION_AS_DFLUX",
                "USE_DATA_EXTENSION_AS_MASK",
                "USE_INPUT_KERNEL",
                "MODEL_MAPPING_KERNEL",
                "TELLURICCORR_PATH",
                "TELLURICCORR_DATA_PATH",
                "TMP_PATH",
                "OUTPUT_PATH",
                "OUTPUT_NAME",
                "OPENMP_THREADS",
                "SILENT_EXTERNAL_BINS",
                "TRANSMISSION",
                "COLUMN_LAMBDA",
                "COLUMN_FLUX",
                "COLUMN_DFLUX",
                "COLUMN_MASK",
                "MASK_BINARY",
                "MASK_THRESHOLD",
                "DEFAULT_ERROR",
                "WLG_TO_MICRON",
                "WAVELENGTH_FRAME",
                "OBS_ERF_RV_KEY",
                "OBS_ERF_RV_VALUE",
                "CLEAN_MODEL_FLUX",
                "FTOL",
                "XTOL",
                "FLUX_UNIT",
                "MAP_REGIONS_TO_CHIP",
                "OBSERVING_DATE_KEYWORD",
                "OBSERVING_DATE_VALUE",
                "UTC_KEYWORD",
                "UTC_VALUE",
                "TELESCOPE_ANGLE_KEYWORD",
                "TELESCOPE_ANGLE_VALUE",
                "RELATIVE_HUMIDITY_KEYWORD",
                "RELATIVE_HUMIDITY_VALUE",
                "PRESSURE_KEYWORD",
                "PRESSURE_VALUE",
                "TEMPERATURE_KEYWORD",
                "TEMPERATURE_VALUE",
                "MIRROR_TEMPERATURE_KEYWORD",
                "MIRROR_TEMPERATURE_VALUE",
                "ELEVATION_KEYWORD",
                "ELEVATION_VALUE",
                "LONGITUDE_KEYWORD",
                "LONGITUDE_VALUE",
                "LATITUDE_KEYWORD",
                "LATITUDE_VALUE",
                "SLIT_WIDTH_KEYWORD",
                "SLIT_WIDTH_VALUE",
                "PIX_SCALE_KEYWORD",
                "PIX_SCALE_VALUE",
                "REFERENCE_ATMOSPHERIC",
                "GDAS_PROFILE",
                "LAYERS",
                "EMIX",
                "PWV",
                "LNFL_LINE_DB",
                "LNFL_LINE_DB_FORMAT",
                "LBLRTM_ICNTNM",
                "LBLRTM_IAERSL",
                "LBLRTM_MPTS",
                "LBLRTM_NPTS",
                "LBLRTM_V1",
                "LBLRTM_V2",
                "LBLRTM_SAMPLE",
                "LBLRTM_ALFAL0",
                "LBLRTM_AVMASS",
                "LBLRTM_DPTMIN",
                "LBLRTM_DPTFAC",
                "LBLRTM_TBOUND",
                "LBLRTM_SREMIS1",
                "LBLRTM_SREMIS2",
                "LBLRTM_SREMIS3",
                "LBLRTM_SRREFL1",
                "LBLRTM_SRREFL2",
                "LBLRTM_SRREFL3",
                "LBLRTM_MODEL",
                "LBLRTM_ITYPE",
                "LBLRTM_NOZERO",
                "LBLRTM_NOPRNT",
                "LBLRTM_IPUNCH",
                "LBLRTM_RE",
                "LBLRTM_HSPACE",
                "LBLRTM_H1",
                "LBLRTM_H2",
                "LBLRTM_RANGE",
                "LBLRTM_BETA",
                "LBLRTM_LEN",
                "LBLRTM_HOBS",
                "LBLRTM_AVTRAT",
                "LBLRTM_TDIFF1",
                "LBLRTM_TDIFF2",
                "LBLRTM_ALTD1",
                "LBLRTM_ALTD2",
                "LBLRTM_DELV",
            ],
        }

        _all_parameters=[]
        recipe_RecipeParameters=[]
        for group_name in group_parameters.keys() :
            for _dn in group_parameters[group_name] :
                if _dn in self.orig_init_sop['by_name'] :
                    _dn_i=self.orig_init_sop['by_name'][_dn]
                    _description=self.orig_init_sop['init_sop'][_dn_i].description
                    recipe_RecipeParameters+=[
                        reflex.RecipeParameter(
                            recipe=self.recipe_name,
                            displayName=_dn,
                            group=group_name,
                            description=_description,
                        ),
                    ]
                    _all_parameters+=[_dn,]
        ## Check all recipe parametersin self.orig_init_sop have been included...
        ignore_parameters=[
            "EXPERT_MODE",
            "CHIP_EXTENSIONS",
        ]
        for rp in self.orig_init_sop['init_sop'] :
            if rp.displayName not in _all_parameters and rp.displayName not in ignore_parameters :
                # TODO Python3
                # raise Exception(f"Parameter {rp.displayName} not known.")
                raise Exception("Parameter rp.displayName not known.")

        return recipe_RecipeParameters


    # ---------------------------------------------------------------------------------------
    def wl_to_topo_vac(self, wl, WL_FRAME=None):
        if WL_FRAME is None :
           WL_FRAME=self.get_pcc_Value(
               'WAVELENGTH_FRAME',
            )
        if WL_FRAME is None :
            WL_FRAME=self.in_sop_by_name.get('WAVELENGTH_FRAME')
            if WL_FRAME is not None :
                WL_FRAME=WL_FRAME.value
        # transform wavelengths to topocentric vacuum if necessary...
        if WL_FRAME in ['VAC',] :
            return wl

        elif WL_FRAME in ['VAC_RV',] :
            obs_erf_rv=self.f_product_header.get(
                self.get_pcc_Value('OBS_ERF_RV_KEY',),
                self.get_pcc_Value('OBS_ERF_RV_VALUE',),
            )
            return iu.bary_to_topo(wl, obs_erf_rv,)

        elif WL_FRAME in ['AIR',] :
            return iu.air_to_vac(wl)
        '''
        ## Test case... not currently supported by molecfit/telluric_corr
        elif WL_FRAME in ['AIR_RV',] :
            ## Undo the BERV correction then transform to VAC
            obs_erf_rv=self.f_product_header.header.get(
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_KEY',),
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_VALUE',),
            )
                 return iu.air_to_vac(
                    iu.bary_to_topo(
                        wl,
                        obs_erf_rv,
                    )
                )
        else :
            raise Exception('Unsupported value of WAVELENGTH_FRAME')
        '''
        return(wl)

    # ---------------------------------------------------------------------------------------
    def wl_back_from_topo_vac(self, wl, WL_FRAME=None):
        if WL_FRAME is None :
           WL_FRAME=self.get_pcc_Value('WAVELENGTH_FRAME',)
        if WL_FRAME is None :
            WL_FRAME=self.in_sop_by_name.get('WAVELENGTH_FRAME')
            if WL_FRAME is not None :
                WL_FRAME=WL_FRAME.value
        # transform wavelengths to topocentric vacuum if necessary...
        if WL_FRAME in ['VAC',] :
            return wl

        elif WL_FRAME in ['VAC_RV',] :
            obs_erf_rv=self.f_product_header.get(
                self.get_pcc_Value('OBS_ERF_RV_KEY',),
                self.get_pcc_Value('OBS_ERF_RV_VALUE',),
            )
            return iu.topo_to_bary(wl, obs_erf_rv,)

        elif WL_FRAME in ['AIR',] :
            return iu.vac_to_air(wl)
        '''
        ## Test case... not currently supported by molecfit/telluric_corr
        elif self.get_sop_param_Value(self.in_sop_by_name,'WAVELENGTH_FRAME',) in ['AIR_RV',] :
            ## Undo the BERV correction then transform to VAC
            obs_erf_rv=self.f_product_header.header.get(
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_KEY',),
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_VALUE',),
            )
            return iu.topo_to_bary(
                iu.vac_to_air(wl,)
                obs_erf_rv,
            )
        else :
            raise Exception('Unsupported value of WAVELENGTH_FRAME')
        '''
        return wl

    # ---------------------------------------------------------------------------------------
    def wl_back_from_topo_vac_from_uep(self, wl ):

        uep=init_sop_dict(interactive_app.user_edited_param)
        # transform wavelengths to topocentric vacuum if necessary...
        if self.get_sop_param_Value( uep, 'WAVELENGTH_FRAME',) in ['VAC',] :
            return wl

        elif self.get_sop_param_Value( uep, 'WAVELENGTH_FRAME',) in ['VAC_RV',] :
            obs_erf_rv=self.f_product_header.get(
                self.get_sop_param_Value( uep, 'OBS_ERF_RV_KEY',),
                self.get_sop_param_Value( uep, 'OBS_ERF_RV_VALUE',),
            )
            return iu.topo_to_bary(wl, obs_erf_rv,)

        elif self.get_sop_param_Value( uep, 'WAVELENGTH_FRAME',) in ['AIR',] :
            return iu.vac_to_air(wl)
        '''
        ## Test case... not currently supported by molecfit/telluric_corr
        elif self.get_sop_param_Value(self.in_sop_by_name,'WAVELENGTH_FRAME',) in ['AIR_RV',] :
            ## Undo the BERV correction then transform to VAC
            obs_erf_rv=self.f_product_header.header.get(
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_KEY',),
                self.get_sop_param_Value(self.in_sop_by_name,'OBS_ERF_RV_VALUE',),
            )
            return iu.topo_to_bary(
                iu.vac_to_air(wl,)
                obs_erf_rv,
            )
        else :
            raise Exception('Unsupported value of WAVELENGTH_FRAME')
        '''
        return wl

    # ---------------------------------------------------------------------------------------
    def readFitsData(self, fitsFiles):
        """
        This function should be used to read and organize the raw fits files
        produced by the recipes.
        It receives as input a list of reflex.FitsFiles
        """

        self.files_output = dict()
        self.wave_incl_lower = []
        self.wave_incl_upper = []
        self.wave_excl_lower = []
        self.wave_excl_upper = []
        self.pixel_excl_lower = []
        self.pixel_excl_upper = []
        self.input_errors=1
        self.disable_continue_wkf = False
        self.instrument=None
        self.inst_setup=None
        self.objectname=None
        self.write_sis=False
        self.write_sis_OS=False
        self.f_product_header=None

        self.orig_data_extnames = []
        self.orig_data_extnames_index = {}
        self.orig_data_lambda = dict()
        self.orig_data_lambda_ranges = dict()
        self.orig_data_flux = dict()
        self.orig_data_flux_mask = dict()
        self.orig_su_spectrum = dict()
        self.orig_su_spectrum['gcf']=dict()
        self.in_sop_by_name={}

        self.required_columns={}
        self.optional_columns={}

        if self.recipe_name in in_sop_by_name_defaults.keys() :
            for p in in_sop_by_name_defaults[self.recipe_name].keys() :
                self.in_sop_by_name[p]=reflex.RecipeParameter(
                        recipe=self.recipe_name,
                        name=p,
                        displayName=p,
                        value=in_sop_by_name_defaults[self.recipe_name][p],
                    )
        for p in interactive_app.inputs.in_sop :
            self.in_sop_by_name[p.name]=p
        WL_FRAME=self.in_sop_by_name.get('WAVELENGTH_FRAME')
        if WL_FRAME is not None :
            WL_FRAME=WL_FRAME.value

        # Get the data of the input spectrum...
        f_product=None
        for f in fitsFiles:
            if f.category in [
                'SCIENCE','SCIENCE_CALCTRANS','STD_MODEL',          # molecfit
                'REDUCED_IDP_SCI_LSS',                              # fors
                'REDUCED_IDP_SCI_MOS',                              # fors
                'REDUCED_IDP_SCI_MXU',                              # fors
                'REDUCED_IDP_STD_MOS',                              # fors
                'REDUCED_IDP_STD_MOS_SEDCORR',                      # fors
                'SCI_SLIT_FLUX_IDP_VIS','SCI_SLIT_FLUX_IDP_NIR',    # xshooter
                'TELL_SLIT_FLUX_IDP_VIS','TELL_SLIT_FLUX_IDP_NIR',  # xshooter
            ] :
                f_product = PipelineProduct(f)
                self.f_product_header=f_product.all_hdu[0].header
                break
        if f_product is not None :
            #self.pre_recipe_exec=True
            hdulist=f_product.all_hdu
            self.instrument=hdulist[0].header.get('INSTRUME')
            if self.instrument is not None and self.inst_setup is None:
                self.objectname=hdulist[0].header.get('HIERARCH ESO OBS TARG NAME',hdulist[0].header.get('OBJECT'))
                try:
                    inst_mod=iu.import_inst(self.instrument)
                    if hasattr(inst_mod,'get_object') :
                        self.objectname=inst_mod.get_object(hdulist[0].header)
                    self.inst_setup=inst_mod.set_inst_setup(hdulist[0].header)
                except:
                    pass
            #if not self.get_sop_param_Value(self.in_sop_by_name,'CHIP_EXTENSIONS',) :
            #    hdulist=[f_product.all_hdu[0],]
            #if self.get_sop_param_Value(self.in_sop_by_name,'USE_ONLY_INPUT_PRIMARY_DATA',) :
            WLG_TO_MICRON=(float(
                self.get_pcc_Value('WLG_TO_MICRON')
                or
                self.get_sop_param_Value(self.in_sop_by_name,'WLG_TO_MICRON',)
            ))
            if (
                self.get_pcc_Value('USE_ONLY_INPUT_PRIMARY_DATA')
                or
                self.get_sop_param_Value(self.in_sop_by_name, 'USE_ONLY_INPUT_PRIMARY_DATA', False)
            ) :
                # If this is true, I think it must be a 1-d spec' image...
                # so get the wl from WCS and the data from the image...
                hdulist=[f_product.all_hdu[0],]
                # here we are assuming the WL is is always the "last"...
                # Better to check for CTYPE
                AWAV_AX=hdulist[0].header.get('NAXIS',0)
                for ax in range(AWAV_AX) :
                    if hdulist[0].header.get('CTYPE%d' %(ax+1)) in ['AWAV',] :
                        AWAV_AX=ax+1
                        break
                if AWAV_AX > 0 :
                    extname=hdulist[0].header.get('EXTNAME','EXT.1')
                    self.orig_data_extnames+=[extname,]
                    self.orig_data_extnames_index[extname]=0
                    #
                    self.orig_data_flux_mask[extname] = ~np.isnan(hdulist[0].data)
                    self.orig_data_flux[extname] = hdulist[0].data[self.orig_data_flux_mask[extname]]
                    #
                    self.orig_data_lambda[extname] = (
                        (np.arange(1, hdulist[0].header['NAXIS%d' %(AWAV_AX)]+1, 1)-hdulist[0].header['CRPIX%d' %(AWAV_AX)])
                        *
                        hdulist[0].header['CDELT%d' %(AWAV_AX)] + hdulist[0].header['CRVAL%d' %(AWAV_AX)]
                    )[self.orig_data_flux_mask[extname]]*WLG_TO_MICRON
                    self.orig_data_lambda_ranges[extname] = {
                        'min': np.min(self.orig_data_lambda[extname]),
                        'max': np.max(self.orig_data_lambda[extname]),
                    }
                    self.orig_data_lambda_ranges['__span__']={extname: self.orig_data_lambda_ranges[extname]}
                    # use the following to find the minimum value of positive values...
                    # Needed for CR2RES IDP products that have overlapping WL ranges...
                    # Based on https://numpy.org/doc/stable/reference/generated/numpy.where.html
                    wl_diffs=(
                        self.orig_data_lambda[extname][1:]
                        -
                        self.orig_data_lambda[extname][0:-1]
                    )
                    self.orig_data_lambda_ranges['__wl_resolution__']={
                        extname: wl_diffs[np.where(wl_diffs > 0., wl_diffs, np.inf).argmin()]
                    }
            else :
                # Still can be image or bin table in the extensions...
                #self.orig_data_lambda['merged'] = []
                #self.orig_data_flux['merged'] = []
                orig_data_lambda = []
                orig_data_flux = []
                orig_data_extname = None
                for ext_num,hdu in enumerate(hdulist) :
                    if isinstance(hdu,pyfits.BinTableHDU) :
                        if (hdu.header.get('NAXIS',0) == 2):
                            COLUMN_LAMBDA=self.get_pcc_Value('COLUMN_LAMBDA') or self.get_sop_param_Value(self.in_sop_by_name,'COLUMN_LAMBDA',)
                            COLUMN_FLUX=self.get_pcc_Value('COLUMN_FLUX') or self.get_sop_param_Value(self.in_sop_by_name,'COLUMN_FLUX',)
                            COLUMN_DFLUX=self.get_pcc_Value('COLUMN_DFLUX') or self.get_sop_param_Value(self.in_sop_by_name,'COLUMN_DFLUX',)
                            COLUMN_MASK=self.get_pcc_Value('COLUMN_MASK') or self.get_sop_param_Value(self.in_sop_by_name,'COLUMN_MASK',)
                            self.required_columns={
                                'COLUMN_LAMBDA': {
                                    'value': COLUMN_LAMBDA,
                                    'exists': True,
                                },
                                'COLUMN_FLUX': {
                                    'value': COLUMN_FLUX,
                                    'exists': True,
                                },
                            }
                            self.optional_columns={
                                'COLUMN_DFLUX': {
                                    'value': COLUMN_DFLUX,
                                    'exists': True,
                                },
                                'COLUMN_MASK': {
                                    'value': COLUMN_MASK,
                                    'exists': True,
                                },
                            }
                            have_all_required_columns=True
                            for c in self.required_columns.keys() :
                                if self.required_columns[c]['value'] not in hdu.data.columns.names :
                                    print(
                                        "ERROR :: required column %s not found in extension %d"
                                        %(self.required_columns[c]['value'], ext_num)
                                    )
                                    self.required_columns[c]['exists']=False
                                    have_all_required_columns=False
                            for c in self.optional_columns.keys() :
                                if (
                                    self.optional_columns[c]['value'] not in ['NULL',None]
                                    and
                                    self.optional_columns[c]['value'] not in hdu.data.columns.names
                                ) :
                                    print(
                                        "WARNING :: optional column %s not found in extension %d"
                                        %(self.optional_columns[c]['value'], ext_num)
                                    )
                                    self.optional_columns[c]['exists']=False
                            if have_all_required_columns :
                                extname = hdu.header.get('EXTNAME','EXT.%d' %(ext_num))
                                self.orig_data_extnames+=[extname,]
                                self.orig_data_extnames_index[extname]=ext_num
                                # Get infos from star_spec using the extname
                                flux=np.ravel(hdu.data.field(COLUMN_FLUX))
                                self.orig_data_flux_mask[extname] = ~np.isnan(flux)
                                self.orig_data_flux[extname] = np.ravel(hdu.data.field(COLUMN_FLUX))[self.orig_data_flux_mask[extname]]
                                self.orig_data_lambda[extname] = np.ravel(hdu.data.field(COLUMN_LAMBDA))[self.orig_data_flux_mask[extname]]*WLG_TO_MICRON
                                self.orig_data_lambda_ranges[extname] = {
                                    'min': np.min(self.orig_data_lambda[extname][~np.isnan(self.orig_data_lambda[extname])]),
                                    'max': np.max(self.orig_data_lambda[extname][~np.isnan(self.orig_data_lambda[extname])]),
                                }

                                orig_data_lambda += list(
                                    np.ravel(hdu.data.field(COLUMN_LAMBDA))[self.orig_data_flux_mask[extname]]
                                )
                                orig_data_flux += list(
                                    np.ravel(hdu.data.field(COLUMN_FLUX))[self.orig_data_flux_mask[extname]]
                                )
                if len(self.orig_data_lambda) > 1 :
                    extname='merged'
                    self.orig_data_lambda[extname] = np.array(orig_data_lambda)[np.argsort(orig_data_lambda)]
                    self.orig_data_flux[extname] = np.array(orig_data_flux)[np.argsort(orig_data_lambda)]

                self.orig_data_lambda_ranges['__span__']={}
                self.orig_data_lambda_ranges['__wl_resolution__']={}
                for _e in self.orig_data_lambda.keys() :
                    if _e not in [
                        '__span__',
                        '__wl_resolution__',
                    ] :
                        self.orig_data_lambda[_e] = np.array(self.orig_data_lambda[_e])
                        if _e in ['merged'] :
                            self.orig_data_lambda[_e] *= WLG_TO_MICRON
                        self.orig_data_lambda[_e] = np.array(self.orig_data_lambda[_e])
                        self.orig_data_flux[_e] = np.array(self.orig_data_flux[_e])
                        self.orig_data_lambda_ranges['__span__'][_e]={'min': 0.,'max': 0.}
                        self.orig_data_lambda_ranges['__wl_resolution__'][_e]=0.
                        if len(self.orig_data_lambda[_e]) > 0 :
                            self.orig_data_lambda_ranges['__span__'][_e]={
                                'min': np.min(self.orig_data_lambda[_e][~np.isnan(self.orig_data_lambda[_e])]),
                                'max': np.max(self.orig_data_lambda[_e][~np.isnan(self.orig_data_lambda[_e])]),
                            }
                            # use the following to find the minimum value of positive values...
                            # Needed for CR2RES IDP products that have overlapping WL ranges...
                            # Based on https://numpy.org/doc/stable/reference/generated/numpy.where.html
                            wl_diffs=(
                                self.orig_data_lambda[_e][~np.isnan(self.orig_data_lambda[_e])][1:]
                                -
                                self.orig_data_lambda[_e][~np.isnan(self.orig_data_lambda[_e])][0:-1]
                            )
                            self.orig_data_lambda_ranges['__wl_resolution__'][_e]=(
                                wl_diffs[np.where(wl_diffs > 0., wl_diffs, np.inf).argmin()]
                            )
                #else :
                #    self.input_errors=1

            # transform wavelengths to topocentric vacuum if necessary...
            '''
            for e in self.orig_data_lambda.keys() :
                self.orig_data_lambda[e]=self.wl_to_topo_vac(
                    self.orig_data_lambda[e],
                    WL_FRAME=WL_FRAME,
                )
            '''

        if not self.pre_recipe_exec :

            '''
            in_sof should now include the output from molecfit_calctrans
            which should be displayed so that the user can examine the
            fit over the full spectrum, not just the fit regions...
            I'm thinking two tabs in the graphic panel, one for
            molecfit_model and one for molecfit_calctrans...
            '''

            # Get the data of the molecfit_model products...
            # Loop on all FITS files_input
            for i,f in enumerate(fitsFiles):

                if f.category[0:len(self.telluric_corr)] == self.telluric_corr :
                    self.have_telluric_corr = True
                    pp=PipelineProduct(f)
                    self.telluric_corr={
                        'lambda': [],
                        'mlambda': [],
                        'flux': [],
                        'mtrans': [],
                        'cflux': [],
                    }
                    _i=0
                    for extname in self.orig_data_flux_mask :
                        _n=len(self.orig_data_flux_mask[extname])
                        self.telluric_corr['lambda']=np.append(self.telluric_corr['lambda'],pp.all_hdu[1].data.field('lambda')[_i:_i+_n][self.orig_data_flux_mask[extname]])
                        self.telluric_corr['mlambda']=np.append(self.telluric_corr['mlambda'],pp.all_hdu[1].data.field('mlambda')[_i:_i+_n][self.orig_data_flux_mask[extname]])
                        self.telluric_corr['flux']=np.append(self.telluric_corr['flux'],pp.all_hdu[1].data.field('flux')[_i:_i+_n][self.orig_data_flux_mask[extname]])
                        self.telluric_corr['mtrans']=np.append(self.telluric_corr['mtrans'],pp.all_hdu[1].data.field('mtrans')[_i:_i+_n][self.orig_data_flux_mask[extname]])
                        self.telluric_corr['cflux']=np.append(self.telluric_corr['cflux'],pp.all_hdu[1].data.field('cflux')[_i:_i+_n][self.orig_data_flux_mask[extname]])
                        _i+=_n
                    self.instrument=pp.all_hdu[0].header.get('INSTRUME')
                    if self.instrument is not None and self.inst_setup is None:
                        try:
                            inst_mod=iu.import_inst(self.instrument)
                            self.inst_setup=inst_mod.set_inst_setup(pp.all_hdu[0].header)
                        except:
                            pass
                        self.objectname=pp.all_hdu[0].header.get('OBJECT')
                        self.objectname=pp.all_hdu[0].header.get('HIERARCH ESO OBS TARG NAME',hdulist[0].header.get('OBJECT'))

                if f.category[0:len(self.wave_include)] == self.wave_include :

                    wave_include = PipelineProduct(f)

                    # Loop on extensions
                    ext_num = -1
                    for wave_include_ext in wave_include.all_hdu:

                        # Take extension in the output file
                        ext_num = ext_num + 1
                        if (ext_num == 1):

                            self.wave_incl_lower = wave_include_ext.data.field("LOWER_LIMIT")
                            self.wave_incl_upper = wave_include_ext.data.field("UPPER_LIMIT")


                if f.category[0:len(self.wave_exclude)] == self.wave_exclude :

                    wave_exclude = PipelineProduct(f)

                    # Loop on extensions
                    ext_num = -1
                    for wave_exclude_ext in wave_exclude.all_hdu:

                        # Take extension in the output file
                        ext_num = ext_num + 1
                        if (ext_num == 1):

                            self.wave_excl_lower = wave_exclude_ext.data.field("LOWER_LIMIT")
                            self.wave_excl_upper = wave_exclude_ext.data.field("UPPER_LIMIT")


                if f.category[0:len(self.pixel_exclude)] == self.pixel_exclude :

                    pixel_exclude = PipelineProduct(f)

                    # Loop on extensions
                    ext_num = -1
                    for pixel_exclude_ext in pixel_exclude.all_hdu:

                        # Take extension in the output file
                        ext_num = ext_num + 1
                        if (ext_num == 1):

                            self.pixel_excl_lower = pixel_exclude_ext.data.field("LOWER_LIMIT")
                            self.pixel_excl_upper = pixel_exclude_ext.data.field("UPPER_LIMIT")


                # For each sci_reconstructed image
                if f.category[0:len(self.best_fit_model_cat)] == self.best_fit_model_cat :

                    # Initialize
                    self.data_extnames = []
                    self.data_lambda = dict()
                    self.data_mlambda = dict()
                    self.data_flux = dict()
                    self.data_mflux = dict()

                    # Get the expected files
                    try:
                        best_fit_model = PipelineProduct(f)

                    except:
                        break

                    # Loop on extensions
                    ext_num = -1
                    for best_fit_model_ext in best_fit_model.all_hdu :

                        # Take extension in the output file
                        ext_num = ext_num + 1

                        naxis = best_fit_model_ext.header['NAXIS']
                        if (naxis == 2):

                            # TELLURIC_DATA comes as a single HDU Binary-Table
                            # irrespective of how many HDUs are in the input spectrum...
                            self.data_lambda['merged'] = []
                            self.data_flux['merged'] = []
                            self.data_mflux['merged'] = []
                            self.orig_su_spectrum['gcf']['merged'] = []
                            _i=0
                            for extname in self.orig_data_flux_mask :
                                self.data_extnames.append(extname)
                                _n=len(self.orig_data_flux_mask[extname])
                                # Get infos from star_spec using the extname
                                self.data_lambda[extname] = best_fit_model_ext.data.field("lambda")[_i:_i+_n][self.orig_data_flux_mask[extname]]
                                self.data_mlambda[extname] = best_fit_model_ext.data.field("mlambda")[_i:_i+_n][self.orig_data_flux_mask[extname]]
                                self.data_flux[extname] = best_fit_model_ext.data.field("flux")[_i:_i+_n][self.orig_data_flux_mask[extname]]
                                self.data_mflux[extname] = best_fit_model_ext.data.field("mflux")[_i:_i+_n][self.orig_data_flux_mask[extname]]
                                self.orig_su_spectrum['gcf'][extname]=np.ones(len(self.orig_data_flux[extname]))
                                self.data_lambda['merged']=np.append(self.data_lambda['merged'],self.data_lambda[extname])
                                self.data_flux['merged']=np.append(self.data_flux['merged'],self.data_flux[extname])
                                self.data_mflux['merged']=np.append(self.data_mflux['merged'],self.data_mflux[extname])
                                self.orig_su_spectrum['gcf']['merged']=np.append(self.orig_su_spectrum['gcf']['merged'],self.orig_su_spectrum['gcf'][extname])
                                _i+=_n

                            # molecfit_model (4.2) can write essentially empty products
                            # so we only set no errors if we actually find an naxis==2 extension
                            # not just if we find the best_fit_model file...
                            self.input_errors = 0
            if self.input_errors == 1 :
                self.pre_recipe_exec = True
                self.input_errors = 0
                self.disable_continue_wkf = True
                self.data_extnames = self.orig_data_extnames
                self.data_flux = sys_copy.deepcopy(self.orig_data_flux)
                self.data_lambda = sys_copy.deepcopy(self.orig_data_lambda)
                for k in self.orig_data_extnames :
                    self.data_lambda[k]=self.wl_to_topo_vac(self.data_lambda[k])
                    self.orig_su_spectrum['gcf'][k]=np.ones(len(self.orig_data_flux[k]))
                if 'merged' in self.orig_data_flux :
                    self.orig_su_spectrum['gcf']['merged'] = np.ones(len(self.orig_data_flux['merged']))
                

        else :
            self.data_extnames = self.orig_data_extnames
            self.data_lambda = sys_copy.deepcopy(self.orig_data_lambda)
            self.data_flux = sys_copy.deepcopy(self.orig_data_flux)
            for k in self.orig_data_extnames :
                self.data_lambda[k]=self.wl_to_topo_vac(self.data_lambda[k])
                self.orig_su_spectrum['gcf'][k]=np.ones(len(self.orig_data_flux[k]))
            if 'merged' in self.orig_data_flux :
                self.orig_su_spectrum['gcf']['merged'] = np.ones(len(self.orig_data_flux['merged']))
            self.input_errors = 0

        #if 'merged' in self.orig_data_lambda.keys() :
        #    self.orig_su_spectrum['gcf']['merged']=np.ones(len(self.orig_data_flux[extname]))

        if have_specutils :
            self.orig_su_spectrum['data']={}
            for k in self.orig_data_extnames :
                self.orig_su_spectrum['data'][k] = Spectrum1D(
                    # ToDo: need to detect units
                    flux=self.orig_data_flux[k] * u.Jy,
                    # ToDo: need to check if this is always micron here...
                    spectral_axis=self.orig_data_lambda[k] * u.micron,
                )


        for p in ['WAVE_INCLUDE','WAVE_EXCLUDE',] :
            if p in self.in_sop_by_name.keys() :
                if self.in_sop_by_name[p].value not in ['NULL'] :
                    attr_name=p.replace('CLUDE','CL').lower()
                    setattr(
                        self,
                        '%s_lower' %(attr_name),
                        np.array(self.in_sop_by_name[p].value.split(',')[::2], dtype=float),
                    )
                    setattr(
                        self,
                        '%s_upper' %(attr_name),
                        np.array(self.in_sop_by_name[p].value.split(',')[1::2], dtype=float),
                    )
        for p in ['PIXEL_EXCLUDE',] :
            if p in self.in_sop_by_name.keys() :
                if self.in_sop_by_name[p].value not in ['NULL'] :
                    attr_name=p.replace('CLUDE','CL').lower()
                    setattr(
                        self,
                        '%s_lower' %(attr_name),
                        np.array(self.in_sop_by_name[p].value.split(',')[::2], dtype=int),
                    )
                    setattr(
                        self,
                        '%s_upper' %(attr_name),
                        np.array(self.in_sop_by_name[p].value.split(',')[1::2], dtype=int),
                    )

        '''
        if len(self.wave_incl_lower) > 0 :
            self.wave_incl_lower = self.wl_to_topo_vac(
                self.wave_incl_lower,
                WL_FRAME,
            )
            self.wave_incl_upper = self.wl_to_topo_vac(
                self.wave_incl_upper,
                WL_FRAME,
            )
        if len(self.wave_excl_lower) > 0 :
            self.wave_excl_lower = self.wl_to_topo_vac(
                self.wave_excl_lower,
                WL_FRAME,
            )
            self.wave_excl_upper = self.wl_to_topo_vac(
                self.wave_excl_upper,
                WL_FRAME,
            )
        '''

        # Set the plotting functions
        self._add_subplots = self._add_subplots
        self._plot = self._data_plot


    # ---------------------------------------------------------------------------------------
    def addSubplots(self, figure):
        """
        This function should be used to setup the subplots of the gui.  The the
        matplotlib documentation for a description of subplots.
        """
        self._add_subplots(figure)

    # ---------------------------------------------------------------------------------------
    def plotProductsGraphics(self):
        """
        This function should be used to plot the data onto the subplots.
        """
        self._plot()

    # ---------------------------------------------------------------------------------------
    def plotWidgets(self) :
        widgets = list()
        # Radio button
        if self.input_errors == 0 and 'merged' not in self.data_lambda.keys() :
            if len(self.data_extnames) > 1:
                self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(
                    self.axradiobutton, self.setRadioCallback,
                    self.data_extnames, 0, title='Extension selection'
                )
                widgets.append(self.radiobutton)
        return widgets

    # ---------------------------------------------------------------------------------------
    def setRadioCallback(self, label) :

        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        self.spec_plot.clear()
        specdisp.setLabels(
            (
                r"$\lambda$["
                +r"$\mu$m]"
                +r" (blue: input spectrum, red: best fit molecfit model)"
            ),
            "Flux (ADU)   [x" + str(self.y_scalefactor)+ "]"
        )

        specdisp.display(
            self.spec_plot,
            "Input spectrum",
            self._data_plot_get_tooltip(label),
            self.data_lambda[label],
            self.data_flux[label]/self.orig_su_spectrum['gcf'][label]/self.y_scalefactor
        )

        if not self.pre_recipe_exec :
            self.spec_plot.fill(
                self.wl_back_from_topo_vac(self.data_lambda[label]),
                #self.orig_data_lambda[label],
                self.data_mflux[label]/self.orig_su_spectrum['gcf'][label]/self.y_scalefactor,
                'green', alpha=0.3
            )
            specdisp.overplot(
                self.spec_plot,
                self.wl_back_from_topo_vac(self.data_lambda[label]),
                #self.orig_data_lambda[label],
                self.data_mflux[label]/self.orig_su_spectrum['gcf'][label]/self.y_scalefactor,
                'red'
            )

    # ---------------------------------------------------------------------------------------
    def unselect_region(self):
        self.selected_region['pname'] = None
        self.selected_region['index'] = None
        self.span.extents = (0.,0.)
        self.span.active = False
        self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
        self.select_region_cid=None
        self.draw_all_fill_regions(self.extname)
    # ---------------------------------------------------------------------------------------
    def onclick_IR_add(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.IR_add_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None :
            if self.span.extents != (0.,0.) :
                self.add_buttons["WAVE_INCLUDE"].Enable(True)
                if self.span.ver < 2 :
                    self.AddIncludeRegion(None)
                else :
                    self.spec_plot.set_title('LMB=Adjust, Click "Add" button when done, RMB=Cancel')
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    def onclick_IR_delete(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.IR_del_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None:
            x, y = event.xdata, event.ydata
            # find the region containing the clicked point...
            wave_range=self.get_param_ranges('WAVE_INCLUDE')
            selected_range=None
            for i, wr in enumerate(wave_range) :
                if ( x > wr[0] and x < wr[1] ) :
                    selected_range=wr
                    self.selected_region['pname']='WAVE_INCLUDE'
                    self.selected_region['index']=i
            if selected_range is not None :
                # Highlight the selected region somehow...
                self.draw_all_fill_regions(self.extname)
                #self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                #self.select_region_cid=None
                self.spec_plot.set_title('Click "Delete" button to confirm, RMB=Cancel')
                self.delete_buttons["WAVE_INCLUDE"].Enable(True)
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    def onclick_IR_modify(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.IR_mod_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None :
            x, y = event.xdata, event.ydata
            # find the region containing the clicked point...
            # set span to limits of selected region...
            wave_range=self.get_param_ranges('WAVE_INCLUDE')
            selected_range=None
            for i, wr in enumerate(wave_range) :
                if ( x > wr[0] and x < wr[1] ) :
                    selected_range=wr
                    self.selected_region['pname']='WAVE_INCLUDE'
                    self.selected_region['index']=i
            if selected_range is not None :
                self.span.extents=(selected_range[0],selected_range[1])
                self.span.active = True
                self.modify_buttons["WAVE_INCLUDE"].Enable(True)
                #self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                #self.select_region_cid=None
                self.spec_plot.set_title('LMB=Adjust, Click "Modify" button to confirm, RMB=Cancel')
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    def onclick_ER_add(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.ER_add_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None :
            if self.span.extents != (0.,0.) :
                self.add_buttons["WAVE_EXCLUDE"].Enable(True)
                if self.span.ver < 2 :
                    self.AddExcludeRegion(None)
                else :
                    self.spec_plot.set_title('LMB=Adjust, Click "Add" button when done, RMB=Cancel')
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    def onclick_ER_delete(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.ER_del_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None:
            x, y = event.xdata, event.ydata
            # find the region containing the clicked point...
            wave_range=self.get_param_ranges('WAVE_EXCLUDE')
            selected_range=None
            for i, wr in enumerate(wave_range) :
                if ( x > wr[0] and x < wr[1] ) :
                    selected_range=wr
                    self.selected_region['pname']='WAVE_EXCLUDE'
                    self.selected_region['index']=i
            if selected_range is not None :
                # Highlight the selected region somehow...
                self.draw_all_fill_regions(self.extname)
                #self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                #self.select_region_cid=None
                self.spec_plot.set_title('Click "Delete" button to confirm, RMB=Cancel')
                self.delete_buttons["WAVE_EXCLUDE"].Enable(True)
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    def onclick_ER_modify(self, event):
        if event.button == MouseButton.RIGHT :
            self.unselect_region()
            self.ER_mod_active=False
            self.enable_region_buttons(True)
            self.spec_plot.set_title(self.title_text)
        elif event.button == MouseButton.LEFT and self.selected_region['index'] is None :
            x, y = event.xdata, event.ydata
            # find the region containing the clicked point...
            # set span to limits of selected region...
            wave_range=self.get_param_ranges('WAVE_EXCLUDE')
            selected_range=None
            for i, wr in enumerate(wave_range) :
                if ( x > wr[0] and x < wr[1] ) :
                    selected_range=wr
                    self.selected_region['pname']='WAVE_EXCLUDE'
                    self.selected_region['index']=i
            if selected_range is not None :
                self.span.extents=(selected_range[0],selected_range[1])
                self.span.active = True
                self.modify_buttons["WAVE_EXCLUDE"].Enable(True)
                #self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                #self.select_region_cid=None
                self.spec_plot.set_title('LMB=Adjust, Click "Modify" button to confirm, RMB=Cancel')
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    region_properties={
        'WAVE_INCLUDE': {'color': 'green', 'alpha': 0.4},
        'WAVE_EXCLUDE': {'color': 'red', 'alpha': 0.4},
        'PIXEL_EXCLUDE': {'color': 'magenta', 'alpha': 0.4, 'type': int},
    }

    # ---------------------------------------------------------------------------------------
    def get_sop_param(self, sop, param):
        if param in sop.keys() :
            return sop[param]
        elif 'by_name' in sop.keys() and 'init_sop' in sop.keys() :
            return sop['init_sop'][sop['by_name'][param]]
        return None
    # ---------------------------------------------------------------------------------------
    def get_sop_param_Value(self, sop, param, default_value=None):
        sop_param=self.get_sop_param(sop, param)
        if sop_param is None : return default_value
        return sop_param.value
    # ---------------------------------------------------------------------------------------
    def get_pcc(self, param):
        if hasattr( self, 'recipe_name') and hasattr( self, 'params_by_name') :
            if (self.recipe_name,param) in self.params_by_name.keys() :
                return self.params_by_name[(self.recipe_name,param)].paramChangeCtrl
        return None
    # ---------------------------------------------------------------------------------------
    def get_pcc_Value(self, param, default_value=None):
        pcc=self.get_pcc(param)
        if pcc is None : return default_value
        valtypes={
            'double': float,
            'int': int,
            'bool': bool,
        }
        if hasattr( self, 'params_by_name') :
            return valtypes.get(self.params_by_name[(self.recipe_name,param)].parameter.valtype,str)(pcc.GetValue())
        else :
            return pcc.GetValue()
    # ---------------------------------------------------------------------------------------
    def set_pcc_Value(self, param, v):
        if self.get_pcc(param) is None : return
        self=self.params_by_name[(self.recipe_name,param)]
        param=reflex.RecipeParameter
        param.value=v
        if param == 'PIXEL_EXCLUDE' :
            pass
        # Following taken from reflex_interactive_gui.ReflexInteractiveWxApp.SetupParameters
        if self.parameter.valtype == 'double' :
            self.paramChangeCtrl.SetValue(str(param.value))
        elif self.parameter.valtype == 'int' :
            if self.parameter.partype != 'enum':
                self.paramChangeCtrl.SetValue(param.value)
            else :
                self.paramChangeCtrl.SetValue(str(param.value))
        elif self.parameter.valtype == 'bool' :
            self.paramChangeCtrl.SetValue(param.value)
        else:
            self.paramChangeCtrl.SetValue(param.value)
    # ---------------------------------------------------------------------------------------
    def parse_regions(self, ranges ):
        region_map=[]
        parsed_regions=[]
        if len(self.orig_data_extnames) == 1 :
            # All ranges in the single extension...
            for i,r in enumerate(ranges) :
                region_map+=[[1,i],]
            return (list(np.array(ranges).flatten()), region_map)
        else :
            for i,r in enumerate(ranges) :
                for extname in self.orig_data_extnames :
                    ext_num=self.orig_data_extnames_index[extname]
                    er=self.orig_data_lambda_ranges[extname]
                    if (
                        r[0] >= er['min'] and r[0] <= er['max']
                        or
                        r[1] >= er['min'] and r[1] <= er['max']
                        or
                        r[0] <= er['min'] and r[1] >= er['max']
                    ) :
                        if r[0] >= er['min'] and r[1] <= er['max'] :
                            # wl-range entirely within extn-wl-range...
                            region_map+=[[ext_num,i],]
                            parsed_regions+=list(r)
                        elif r[0] < er['min'] and r[1] <= er['max'] :
                            region_map+=[[ext_num,i],]
                            parsed_regions+=[er['min'],r[1],]
                        elif r[0] >= er['min'] and r[1] > er['max'] :
                            region_map+=[[ext_num,i],]
                            parsed_regions+=[r[0],er['max'],]
                        elif r[0] <= er['min'] and r[1] >= er['max'] :
                            region_map+=[[ext_num,i],]
                            parsed_regions+=[er['min'],er['max'],]
            return (parsed_regions,region_map)
    # ---------------------------------------------------------------------------------------
    def get_param_ranges(self, param):
        _ranges=self.get_pcc_Value(param)
        if _ranges != 'NULL' :
            _ranges=_ranges.split(',')
            ranges={
                param: np.array(
                    _ranges,
                    dtype=self.region_properties.get(param,{}).get('type',float)
                ).reshape([len(_ranges)//2,2])
            }
            '''
            if param in ['WAVE_INCLUDE',] :
                if self.is_first_pass :
                    # Don't touch regions and MAP_REGIONS_TO_CHIP params the first time through
                    # otherwise Lazy Mode is NEVER possible...
                    # In this case, as long as the
                    self.is_first_pass = False
                else :
                    # Sort out regions and MAP_REGIONS_TO_CHIP params...
                    parsed_regions,region_map = self.parse_regions( ranges[param] )
                    self.set_pcc_Value('WAVE_INCLUDE',','.join(str(x) for x in np.array(parsed_regions).flatten())
                    self.set_pcc_Value('MAP_REGIONS_TO_CHIP', ','.join(str(x) for x in region_map))
            '''
        else :
            ranges={
                'WAVE_INCLUDE': np.array([self.wave_incl_lower,self.wave_incl_upper]).transpose(),
                'WAVE_EXCLUDE': np.array([self.wave_excl_lower,self.wave_excl_upper]).transpose(),
                'PIXEL_EXCLUDE': np.array([self.pixel_excl_lower,self.pixel_excl_upper]).transpose(),
            }
        return ranges[param]
    # ---------------------------------------------------------------------------------------
    def draw_regions(self, param, extname):
        current_regions=''
        if hasattr( self, 'regions' ) :
            current_regions=self.regions.get(param,{}).get(param,'')
        if current_regions == '' :
            current_regions=[]
        else :
            current_regions=current_regions.split(',')
        try :
            '''
            current_regions=self.wl_to_topo_vac(np.array(
                current_regions,
                dtype=self.region_properties.get(param,{}).get('type',float)
            )).reshape([len(current_regions)//2,2])
            '''
            current_regions=np.array(
                current_regions,
                dtype=self.region_properties.get(param,{}).get('type',float)
            ).reshape([len(current_regions)//2,2])
        except :
            pass

        regions=self.get_param_ranges(param)
        region_names=[]
        for i,region in enumerate(regions) :
            region_name=','.join([ iu.wrfmt(x) for x in np.sort(region,axis=0).flatten()])
            region_names+=[region_name,]
            edgecolor='none'
            if self.selected_region['index'] is not None :
                if i == self.selected_region['index'] and param == self.selected_region['pname'] :
                    edgecolor='black'
            if param in ['PIXEL_EXCLUDE',] :
                self.spec_plot_axes['fills'][param][region_name]=self.spec_plot.fill_between(
                    self.wl_back_from_topo_vac(self.data_lambda[extname][region[0]:region[1]+1]),
                    self.data_flux[extname][region[0]:region[1]+1]/self.orig_su_spectrum['gcf'][extname][region[0]:region[1]+1]/self.y_scalefactor,
                    facecolor=self.region_properties.get(param,{}).get('color','green'),
                    edgecolor=edgecolor, linewidth=4,
                    alpha=self.region_properties.get(param,{}).get('alpha',0.6),
                    step='mid',
                )
            else :
                self.spec_plot_axes['fills'][param][region_name]=self.spec_plot.fill_between(
                    wlR(
                        self.wl_back_from_topo_vac(self.data_lambda[extname]),
                        self.wl_back_from_topo_vac(self.data_lambda[extname]),
                        region
                    ),
                    wlR(
                        self.data_flux[extname]/self.orig_su_spectrum['gcf'][extname]/self.y_scalefactor,
                        self.wl_back_from_topo_vac(self.data_lambda[extname]),
                        region
                    ),
                    facecolor=self.region_properties.get(param,{}).get('color','green'),
                    edgecolor=edgecolor, linewidth=4,
                    alpha=self.region_properties.get(param,{}).get('alpha',0.6),
                    step='mid',
                )

        regions_to_delete=[]
        for region_name in self.spec_plot_axes['fills'][param].keys() :
            if region_name not in region_names :
                regions_to_delete+=[region_name,]
        if ~np.all(regions == current_regions) or len(regions_to_delete) > 0 :
            self.set_regions(
                pnames=[param,],
                regions=[regions,],
                delete_regions=[regions_to_delete,],
            )

        for region_name in regions_to_delete :
            del self.spec_plot_axes['fills'][param][region_name]
        #if hasattr( self, 'regions' ) :
        #    print(f"self.regions['{param}']={self.regions[param]}")
        pass
    # ---------------------------------------------------------------------------------------
    def WAVE_INCLUDE_event_enter(self, event):
        """
        This event handler is called whenever <enter> is clicked in
        the textCtrl of WAVE_INCLUDE
        """
        pname="WAVE_INCLUDE"
        uep=init_sop_dict(interactive_app.user_edited_param)
        if self.get_pcc_Value(pname).lower() in [
            'null'
        ] :
            _wl='NULL'
        else :
            if self.get_pcc_Value(pname).lower() in [
                '',
            ] :
                _wl=[]
            else :
                _wl=np.array(self.get_pcc_Value(pname).split(','), dtype=float)
                #_wl=self.wl_back_from_topo_vac(_wl)
            _wl=",".join([iu.wrfmt(x) for x in _wl])
        interactive_app.user_edited_param[
            uep['by_name'][pname]
        ].value=_wl
        delete_regions=[]
        for k in self.regions.keys() :
            delete_regions+=[sys_copy.deepcopy(self.regions[k]['index']),]
        self.set_regions(delete_regions=delete_regions)
        self.region_text_event_enter(event)
    # ---------------------------------------------------------------------------------------
    def WAVE_EXCLUDE_event_enter(self, event):
        """
        This event handler is called whenever <enter> is clicked in
        the textCtrl of WAVE_EXCLUDE
        """
        pname="WAVE_EXCLUDE"
        uep=init_sop_dict(interactive_app.user_edited_param)
        if self.get_pcc_Value(pname).lower() in [
            'null'
        ] :
            _wl='NULL'
        else :
            if self.get_pcc_Value(pname).lower() in [
                '',
            ] :
                _wl=[]
            else :
                _wl=np.array(self.get_pcc_Value(pname).split(','), dtype=float)
                #_wl=self.wl_back_from_topo_vac(_wl)
            _wl=",".join([iu.wrfmt(x) for x in _wl])
        interactive_app.user_edited_param[
            uep['by_name'][pname]
        ].value=_wl
        self.region_text_event_enter(event)
    # ---------------------------------------------------------------------------------------
    def PIXEL_EXCLUDE_event_enter(self, event):
        """
        This event handler is called whenever <enter> is clicked in
        the textCtrl of PIXEL_EXCLUDE
        """
        self.region_text_event_enter(event)
    # ---------------------------------------------------------------------------------------
    def region_text_event_enter(self, event):
        """
        This event handler is called whenever <enter> is clicked in
        the textCtrl of one of WAVE_INCLUDE, WAVE_EXCLUDE, PIXEL_EXCLUDE
        """
        self.draw_all_fill_regions(self.extname)
    # ---------------------------------------------------------------------------------------
    def draw_all_fill_regions(self, extname):
        # Use a list to control the order in which they are drawn...
        #if version.parse(matplotlib.__version__) < version.parse('3.5.9') :
        if compare_versions(matplotlib.__version__,'3.5.9') == 'lower' :
            try :
                self.spec_plot_axes['axes_axes'].collections.clear()
            except:
                # looks like we are in mpl < 2.??
                self.spec_plot_axes['axes_axes'].collections=[]
        else :
            for c in self.spec_plot_axes['axes_axes'].collections :
                c.remove()
        for pname in [
            'WAVE_INCLUDE',
            'WAVE_EXCLUDE',
            'PIXEL_EXCLUDE',
        ] :
            pc=self.get_pcc(pname)
            if pc is not None :
                try:
                    self.draw_regions(pname, extname)
                    pc.SetBackgroundColour(wx.Colour(255, 255, 255))
                    #if pname in ['WAVE_INCLUDE',] :
                    #    self.set_map_regions_to_chips(pname)
                except :
                    #print(f'Error drawing {pname} regions')
                    pc.SetBackgroundColour(wx.Colour(255, 0, 0))
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()

    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    # https://pythonspot.com/wxpython-tabs/
    # # Define the tab content as classes:
    class plot_tab(wx.Panel):
        def __init__(self, parent, figure=None, figureCanvas=None):
            wx.Panel.__init__(self, parent)
            self.figure = figure or Figure()
            self.figureCanvas = figureCanvas or FigureCanvasWxAgg(
                self,
                wx.ID_ANY,
                figure,
            )
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    def add_molecule(self, w, molecule, _s, paramGroup) :
            s='all'
            if self.molecules is None :
                self.molecules={}
            if 'add_order' not in self.molecules.keys() :
                self.molecules['add_order']=[]
            self.molecules['add_order']+=[molecule,]
            have_atlas=True
            if molecule not in self.molecules.keys() :
                have_atlas=False
                self.molecules[molecule]={}
            p='name'
            self.molecules[molecule][p]=wx.StaticText(
                w,
                label=molecule,
                style=wx.ALIGN_CENTRE,
            )
            if _s in [
                'selected',
                'selected_but_not',
                'other',
            ] :
                mu={
                    'selected': ['<b>','</b>'],
                    'selected_but_not': ['<b><i>','</i></b>'],
                    'other': ['<i>','</i>'],
                }
                try:
                    self.molecules[molecule][p].SetLabelMarkup(
                        "%s%s%s" %(mu[_s][0],molecule,mu[_s][1])
                    )
                except:
                    pass
            self.wxApp.parameterGrid[paramGroup].Add(
                self.molecules[molecule][p],
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )

            p='include'
            self.molecules[molecule][p]=wx.CheckBox(w,)
            self.molecules[molecule][p].molecule=molecule
            self.wxApp.Bind(
                wx.EVT_CHECKBOX,
                self.toggle_molecule_include,
                self.molecules[molecule][p],
            )
            self.wxApp.parameterGrid[paramGroup].Add(
                self.molecules[molecule][p],
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
            self.LIST_MOLEC_changed( mol_list=[molecule] )

            p='fit'
            self.molecules[molecule][p]=wx.CheckBox(w,)
            self.molecules[molecule][p].molecule=molecule
            self.wxApp.Bind(
                wx.EVT_CHECKBOX,
                self.toggle_molecule_fit,
                self.molecules[molecule][p],
            )
            self.wxApp.parameterGrid[paramGroup].Add(
                self.molecules[molecule][p],
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
            self.FIT_MOLEC_changed( mol_list=[molecule] )

            p='relcol'
            # Create once...
            self.molecules[molecule][p]=wx.TextCtrl(
                w, -1, validator=FloatValidator(), style=wx.TE_PROCESS_ENTER,
            )
            # get the default sizes...
            self_size=self.molecules[molecule][p].Size
            font_size=self.molecules[molecule][p].Font.PixelSize
            self.molecules[molecule][p].SetMaxSize(wx.Size(font_size.width*12, self_size.height))
            self.molecules[molecule][p].SetSize(self.molecules[molecule][p].GetMaxSize())
            self.molecules[molecule][p].SetInitialSize(self.molecules[molecule][p].GetMaxSize())
            self.molecules[molecule][p].molecule=molecule
            self.wxApp.Bind(
                wx.EVT_TEXT,
                self.relcol_value_changed,
                self.molecules[molecule][p],
            )
            self.wxApp.Bind(
                wx.EVT_TEXT_ENTER,
                self.relcol_value_saved,
                self.molecules[molecule][p],
            )
            self.wxApp.parameterGrid[paramGroup].Add(
                self.molecules[molecule][p],
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
            self.REL_COL_changed( mol_list=[molecule] )

            p='display'
            if have_atlas :
                self.molecules[molecule][p]=wx.CheckBox(w)
                self.wxApp.Bind(
                    wx.EVT_CHECKBOX,
                    self.toggle_molecule_plot,
                    self.molecules[molecule][p],
                )
            else :
                self.molecules[molecule][p]=wx.StaticText(
                    w,
                    label="No atlas",
                    style=wx.ALIGN_CENTRE,
                )
            self.molecules[molecule][p].molecule=molecule
            self.wxApp.parameterGrid[paramGroup].Add(
                self.molecules[molecule][p],
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
            #self.wxApp.parameterGrid[paramGroup].Layout()
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    def _add_subplots(self, figure):

        def onselect(xmin, xmax):
            #print('Span Region : (%f,%f)' %(xmin, xmax))
            if self.span.is_old : self.span.extents=(xmin,xmax)

        '''
        # sort out CtrlPanel geometry a bit...
        self.wxApp_elements_by_name={}
        w_list=self.get_window_list(self.wxApp)
        w_list.reverse()
        for e in w_list :
            if e[0] not in self.wxApp_elements_by_name.keys() :
                self.wxApp_elements_by_name[e[0]]=[]
            self.wxApp_elements_by_name[e[0]]+=[e[1],]
        '''
        if False and self.have_telluric_corr :
            # https://pythonspot.com/wxpython-tabs/
            p=figure.canvas.Parent
            nb = wx.Notebook(p)

            # Destroy the existing figure canvas...
            for (i,o) in enumerate(p.Children) :
                if isinstance(o, FigureCanvasWxAgg) :
                    #fc=o
                    o.Destroy()

            tab1 = self.plot_tab(nb, figure, fc)
            #tab1 = self.plot_tab(nb)
            tab2 = self.plot_tab(nb)

            # Add the windows to tabs and name them.
            nb.AddPage(tab1, "Tab 1")
            nb.AddPage(tab2, "Tab 2")

            # Set noteboook in a sizer to create the layout
            sizer = wx.BoxSizer()
            sizer.Add(nb, 1, wx.EXPAND)
            sizer.SetMinSize(100,20)
            p.SetSizer(sizer)


        try:
            lll = len(self.data_extnames)
        except:
            lll=0
        if lll > 1 and 'merged' not in self.data_lambda.keys() :
            gs = gridspec.GridSpec(1, 4)
            self.axradiobutton =    figure.add_subplot(gs[0:1, 0  ])
            self.spec_plot =        figure.add_subplot(gs[0:1, 1:4])
        else:
            gs = gridspec.GridSpec(1, 1)
            self.spec_plot =        figure.add_subplot(gs[0:1, 0])

        try :
            # matplotlib > 3...
            self.spec_plot_axes={
                "axes":      self.spec_plot.axes,
                "axes_axes": self.spec_plot.axes._axes,
            }
            self.span = SpanSelector(
                self.spec_plot_axes['axes'],
                onselect,
                "horizontal",
                useblit=True,
                props=dict(alpha=0.5, facecolor="tab:blue"),
                interactive=True,
                drag_from_anywhere=True
            )
            self.span.is_old=False
            self.span.ver=3
        except :
            try :
                # 2 < matplotlib < 3...
                self.spec_plot_axes={
                    "axes":      self.spec_plot.axes,
                    "axes_axes": self.spec_plot.axes._axes,
                }
                self.span = SpanSelector(
                    self.spec_plot_axes['axes'],
                    onselect,
                    "horizontal",
                    useblit=True,
                    rectprops=dict(alpha=0.5, facecolor="tab:blue"),
                )
                self.span.is_old=True
                self.span.ver=2
                self.span.extents=(0.,0.)
            except :
                # matplotlib-1...
                self.spec_plot_axes={
                    "axes":      self.spec_plot.axes,
                    "axes_axes": self.spec_plot.axes.axes,
                }
                self.span = SpanSelector(
                    self.spec_plot_axes['axes'],
                    onselect,
                    "horizontal",
                    useblit=True,
                    rectprops=dict(alpha=0.5, facecolor="blue"),
                )
                self.span.is_old=True
                self.span.ver=1
                self.span.extents=(0.,0.)

        self.span.active = False
        self.IR_add_active = False
        self.IR_del_active = False
        self.IR_mod_active = False
        self.ER_add_active = False
        self.ER_del_active = False
        self.ER_mod_active = False
        self.selected_region={
            'pname': None,
            'index': None,
        }
        self.modified_region=None
        self.select_region_cid = None
        self.add_buttons = {}
        self.delete_buttons = {}
        self.modify_buttons = {}

    def _data_plot_get_tooltip(self, extname):
        # Create the tooltip
        if self.pre_recipe_exec :
            tooltip = " \
EXT       : %s \n \
blue         : Input spectrum \n \
yellow fill  : original Wavelength include regions \n \
green fill   : current Wavelength include regions \n \
cyan fill    : Wavelength exclude regions \n \
magenta fill : Pixel exclude regions" % (extname)
        else :
            tooltip = " \
EXT          : %s \n \
blue         : Input spectrum \n \
red line     : Best fit molecfit model \n \
green line   : Residuals (spectrum - best fit model) \n \
yellow fill  : original Wavelength include regions \n \
green fill   : current Wavelength include regions \n \
cyan fill    : Wavelength exclude regions \n \
magenta fill : Pixel exclude regions" % (extname)
        return tooltip

    def _data_plot(self):

        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        if 'input_spectrum' in self.spec_plot_axes :
            _spec_plot=sys_copy.deepcopy(self.spec_plot)
        else :
            _spec_plot=self.spec_plot
        _spec_plot.clear()

        specdisp.setLabels(
            (
                r"$\lambda$["
                +r"$\mu$m]"
                +r" (blue: input spectrum, red: best fit molecfit model, green: residuals)"
            ),
            "Flux (ADU)   [x" + str(self.y_scalefactor)+"]"
        )

        if self.input_errors == 0 :
            try:
                extname = self.orig_data_extnames[0]
            except:
                extname='EXT.1'
            if hasattr(self,'orig_data_lambda') :
                if 'merged' in self.orig_data_lambda.keys() :
                    extname='merged'
            self.extname=extname
            data_set=False
            if extname in self.orig_data_lambda :
                if len(self.orig_data_lambda[extname]) > 0 :
                    specdisp.display(
                        _spec_plot,
                        "Input spectrum",
                        self._data_plot_get_tooltip(extname),
                        self.orig_data_lambda[extname],
                        self.orig_data_flux[extname]/self.orig_su_spectrum['gcf'][extname]/self.y_scalefactor,
                        autolimits=True,
                    )
                    data_set=True
                if 'input_spectrum' in self.spec_plot_axes :
                    if data_set :
                        # If we are here then the display already exists...
                        # so we update it with the temporary one...
                        self.spec_plot_axes['input_spectrum'].set_data(
                            self.orig_data_lambda[extname],
                            self.data_flux[extname]/self.orig_su_spectrum['gcf'][extname]/self.y_scalefactor
                        )
                        tt=self._data_plot_get_tooltip(extname)
                        self.spec_plot_axes['axes'].set_xlim(specdisp.wave_lim)
                        self.spec_plot_axes['axes'].set_ylim(specdisp.flux_lim)

            if not data_set :
                specdisp.wave_lim=[0.3,10.]
                specdisp.flux_lim=[0.,1.]
                specdisp.display(
                    _spec_plot,
                    "Input spectrum",
                    self._data_plot_get_tooltip(None),
                    [],
                    []
                )
            self.residuals_min_max=list(self.spec_plot_axes['axes'].get_ylim())

            if 'input_spectrum' not in self.spec_plot_axes :
                self.spec_plot_axes['input_spectrum']=self.spec_plot_axes['axes_axes'].lines[-1]
                self.spec_plot_axes['input_spectrum']._drawstyle='steps-mid'
                self.spec_plot_axes['input_spectrum']._linewidth=1.0

            '''
            try :
                # matplotlib > ...
                self.spec_plot_axes={
                    "axes": self.spec_plot_axes['axes'],
                    "axes_axes": self.spec_plot_axes['axes']._axes,
                }
            except :
                # matplotlib-2
                self.spec_plot_axes={
                    "axes": self.spec_plot.axes,
                    "axes_axes": self.spec_plot.axes.axes,
                }
            '''

            '''
            self.spec_plot_axes['fill']=self.spec_plot.fill(
                self.data_lambda[extname],
                self.data_mflux[extname]/self.y_scalefactor,
                'green',
                alpha=0.3
            )
            '''
            self.spec_plot_axes['fills']={
                'WAVE_INCLUDE': {},
                'WAVE_EXCLUDE': {},
                'PIXEL_EXCLUDE': {},
            }

            if not self.pre_recipe_exec :
                self.residuals={}
                for extname in self.data_extnames :
                    try:
                        self.residuals = (
                            self.data_flux[extname]-self.data_mflux[extname]
                        )
                        idx=np.where(self.data_mflux[extname] == 0)[0]
                        self.residuals[idx]=0
                        specdisp.overplot(
                            self.spec_plot,
                            self.wl_back_from_topo_vac(self.data_lambda[extname]),
                            self.residuals/self.orig_su_spectrum['gcf'][extname]/self.y_scalefactor,
                            'green'
                        )
                        self.spec_plot_axes['residuals']=self.spec_plot_axes['axes_axes'].lines[-1]
                        self.spec_plot_axes['residuals']._drawstyle='steps-mid'
                        self.spec_plot_axes['residuals']._linewidth=1.0
                        self.residuals_min_max=[
                            np.nanmin([np.nanmin(self.residuals),specdisp.flux_lim[0]]),
                            np.nanmax([np.nanmax(self.residuals),specdisp.flux_lim[1]]),
                        ]
                    except:
                        notdone=1
                    specdisp.overplot(
                        self.spec_plot,
                        self.wl_back_from_topo_vac(self.data_lambda[extname]),
                        self.data_mflux[extname]/self.orig_su_spectrum['gcf'][extname]/self.y_scalefactor,
                        'red'
                    )
                    self.spec_plot_axes['model']=self.spec_plot_axes['axes_axes'].lines[-1]
                    self.spec_plot_axes['model']._drawstyle='steps-mid'
                    self.spec_plot_axes['model']._linewidth=1.0

                    if hasattr(self,'best_fit_parameters') :
                        if 'continuum_fit_coeffs' in self.best_fit_parameters.keys() :
                            self.continuum_fits={
                                'wl': [],
                                'continuum_fit': [],
                            }
                            specdisp.overplot(self.spec_plot, [], [], 'black' )
                            self.continuum_fits['mline']=self.spec_plot_axes['axes_axes'].lines[-1]
                            self.continuum_fits['mline']._drawstyle='steps-mid'
                            self.continuum_fits['mline']._linewidth=1.0
                            '''
                            chips=list(self.best_fit_parameters['continuum_fit_coeffs'].keys())
                            chips.sort()
                            i=0
                            for chip in chips :
                            '''
                            '''
                            for i,extn in enumerate(self.data_extnames) :
                                chip=i+1
                                if chip in self.best_fit_parameters['continuum_fit_coeffs'] :
                                    ordered_wil=np.argsort(self.wave_incl_lower)
                                    wir=[]
                                    for wili in ordered_wil :
                                        if (
                                            (self.wave_incl_lower[wili] < np.max(self.data_lambda[extn]))
                                            and
                                            (self.wave_incl_upper[wili] > np.min(self.data_lambda[extn]))
                                        ) :
                                            wir+=[[self.wave_incl_lower[wili],self.wave_incl_upper[wili]],]
                                    if len(wir) > 0 :
                                        _ranges=list(self.best_fit_parameters['continuum_fit_coeffs'][chip].keys())
                                        _ranges.sort()
                                        i=0
                                        for _range in _ranges :
                                            _range_wl=wlR(
                                                self.data_lambda[extn],
                                                self.data_lambda[extn],
                                                wir[i]
                                            )
                                            _p_coeffs=[x[0] for x in self.best_fit_parameters['continuum_fit_coeffs'][chip][_range]]
                                            _p=Polynomial(coef=_p_coeffs,)
                                            self.continuum_fits['wl']+=[_range_wl[0],]+list(_range_wl)+[_range_wl[-1],]
                                            _wmean=(np.nanmin(_range_wl)+np.nanmax(_range_wl))/2.
                                            _range_cf_wl=(_range_wl-_wmean)
                                            _range_cf=_p(_range_cf_wl)/self.orig_su_spectrum['gcf'][extname]
                                            self.continuum_fits['continuum_fit']+=[0.,]+list(_range_cf)+[0.,]
                                            i+=1
                            '''
                            self.compute_contimuum_fits()

            # Store & set ylim
            self.residuals_min_max[0]+=-0.01*(self.residuals_min_max[1]-self.residuals_min_max[0])
            self.spec_plot.set_ylim(self.residuals_min_max)

            # Add in the reading of the wave/pixel exclude regions to the plot
            self.spec_plot_axes['pixel_exclude_spans']=[]
            self.spec_plot_axes['wave_include_spans']=[]
            self.spec_plot_axes['wave_exclude_spans']=[]

            if (len(self.pixel_excl_lower) != 0) :
                #print("pixel exclude numbers "+str(len(self.pixel_excl_lower)))
                self.spec_plot_axes['pixel_exclude_spans']=[]
                i=0
                while i < len(self.pixel_excl_lower):
                    if self.pixel_excl_lower[i] == self.pixel_excl_upper[i]:
                        self.pixel_excl_upper[i] = self.pixel_excl_upper[i] + 1
                    self.spec_plot_axes['pixel_exclude_spans']+=[
                        self.spec_plot.axvspan(
                            self.data_lambda[extname][self.pixel_excl_lower[i]],
                            self.data_lambda[extname][self.pixel_excl_upper[i]],
                            facecolor='m', alpha=0.6, zorder=-100
                        )
                    ]
                    i=i+1

            if (len(self.wave_excl_lower) != 0) :
                i=0
                while i < len(self.wave_excl_lower):
                    if self.wave_excl_lower[i] == self.wave_excl_upper[i]:
                        self.wave_excl_upper[i] = self.wave_excl_upper[i] + 0.00001
                    self.spec_plot_axes['wave_exclude_spans']+=[
                        self.spec_plot.axvspan(
                            self.wave_excl_lower[i],
                            self.wave_excl_upper[i],
                            facecolor='c', alpha=0.2, zorder=-100
                        )
                    ]
                    i=i+1

            if (len(self.wave_incl_lower) != 0) :
                i=0
                while i < len(self.wave_incl_lower) :
                    if self.wave_incl_lower[i] == self.wave_incl_upper[i]:
                        self.wave_incl_upper[i] = self.wave_incl_upper[i] + 0.00001
                    self.spec_plot_axes['wave_include_spans']+=[
                        self.spec_plot.axvspan(
                            self.wave_incl_lower[i],
                            self.wave_incl_upper[i],
                            facecolor='yellow', alpha=0.2, zorder=-100
                        )
                    ]
                    i=i+1

            if hasattr(self, 'molecules') :
                if self.molecules is not None :
                    for _molecule in self.molecules['all_names'] :
                        if 'line' not in self.molecules[_molecule] :
                            specdisp.overplot(self.spec_plot, [], [], 'orange' )
                            self.molecules[_molecule]['line']=self.spec_plot_axes['axes_axes'].lines[-1]
                    if "ca_rayl" in self.molecules.keys() :
                        if 'line' not in self.molecules["ca_rayl"] :
                            specdisp.overplot(self.spec_plot, [], [], 'orange' )
                            self.molecules["ca_rayl"]['line']=self.spec_plot_axes['axes_axes'].lines[-1]

            if self.have_telluric_corr :
                specdisp.overplot(self.spec_plot, [], [], 'grey' )
                self.telluric_corr['line']=self.spec_plot_axes['axes_axes'].lines[-1]
                self.telluric_corr['line']._drawstyle='steps-mid'
                self.telluric_corr['line']._linewidth=1.0
                specdisp.overplot(self.spec_plot, [], [], 'black' )
                self.telluric_corr['mline']=self.spec_plot_axes['axes_axes'].lines[-1]
                self.telluric_corr['mline']._drawstyle='steps-mid'
                self.telluric_corr['mline']._linewidth=1.0
                specdisp.overplot(self.spec_plot, [], [], 'magenta' )
                self.telluric_corr['mmline']=self.spec_plot_axes['axes_axes'].lines[-1]
                self.telluric_corr['mmline']._drawstyle='steps-mid'
                self.telluric_corr['mmline']._linewidth=1.0

            if have_specutils :
                specdisp.overplot(self.spec_plot, [], [], 'black' )
                self.orig_su_spectrum['line']=self.spec_plot_axes['axes_axes'].lines[-1]
                #self.telluric_corr['line']._drawstyle='steps-mid'
                self.orig_su_spectrum['line']._linewidth=2.0

        else:
            self.spec_plot.set_xlabel('INPUT ERROR')
            self.spec_plot.set_ylabel('INPUT ERROR')
            self.spec_plot.set_title('INPUT ERROR: missing data files '+self.best_fit_model_cat)


    def _process_label(self, in_label):
        # If known, 'pretty print' the label
        if (in_label == "erg.s**(-1).cm**(-2).angstrom**(-1)"):
            return "Flux [erg sec" + r"$^{-1}$"+"cm" + r"$^{-2}$" + r"$\AA^{-1}$] (x" + str(self.y_scalefactor)+ ")"
        else:
            return in_label

    # ---------------------------------------------------------------------------------------
    def read_best_fit_parameters(self, files) :
        if _have_Table :
            _files={}
            for _f in files :
                _files[_f.category[0:len('BEST_FIT_PARAMETERS')]]=_f.name
            if 'BEST_FIT_PARAMETERS' in _files.keys() :
                # Create a BFP init_sop...
                self.bfp_init_sop=sys_copy.deepcopy(self.orig_init_sop)
                # Read in the input parameters from the header:
                # NO! we do not need to do this, because the BFPs were
                # generated using self.orig_init_sop as input, so those are
                # the input parameters used to generate the BFPs,
                # so they do not need to be 'updated'...
                '''
                _bfp_fits=pyfits.open(_files['BEST_FIT_PARAMETERS'])
                for i,p in enumerate(self.bfp_init_sop['init_sop']) :
                    pname=p.name
                    self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name'][pname]].value=(
                        _bfp_fits[0].header.get(
                            'HIERARCH ESO DRS MF PARAM %s' %(pname),
                            self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name'][pname]].value
                        )
                    )
                # And now read the PIXEL/WAVE_INCLUDE/EXCLUDE params from the files
                # because the parameter versions can be truncated...

                ## ATTN, LIST_MOLEC, FIT_MOLEC and REL_COL could also be truncated
                ## if for example one included ALL possible 43 lblrtm molecules...
                # get the list of molecules from ATM_PARAMETERS.fits
                t='ATM_PARAMETERS'
                if t in _files.keys() :
                    t_fits=pyfits.open(_files[t])
                    t_data=t_fits[1].data
                    LIST_MOLEC=[]
                    FIT_MOLEC=[]
                    REL_COL=[]
                    for c in t_fits[1].data.columns :
                        if c.name not in [
                            'HGT',
                            'PRE',
                            'TEM',
                        ] :
                            LIST_MOLEC+=[c.name,]
                            # Set FIT_MOLEC/REL_COL to 0/1. for each molecule for now and then
                            # set them to 1/<actual values> for the actually fitted molecules below...
                            FIT_MOLEC+=['0',]
                            REL_COL+=['1.',]
                    self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['LIST_MOLEC']].value=(
                        ",".join(LIST_MOLEC)
                    )
                    self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['FIT_MOLEC']].value=(
                        ",".join(FIT_MOLEC)
                    )
                    self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['REL_COL']].value=(
                        ",".join(REL_COL)
                    )
                    t_fits.close()
                ts={
                    'WAVE_INCLUDE': {
                        'WAVE_INCLUDE': ['LOWER_LIMIT','UPPER_LIMIT'],
                        'MAP_REGIONS_TO_CHIP': 'MAPPED_TO_CHIP',
                        'FIT_CONTINUUM': 'CONT_FIT_FLAG',
                        'CONTINUUM_N': 'CONT_POLY_ORDER',
                        'FIT_WLC': 'WLC_FIT_FLAG',
                    },
                    'WAVE_EXCLUDE': {
                        'WAVE_EXCLUDE': ['LOWER_LIMIT','UPPER_LIMIT'],
                    },
                    'PIXEL_EXCLUDE': {
                        'PIXEL_EXCLUDE': ['LOWER_LIMIT','UPPER_LIMIT'],
                    },
                }
                for t in ts.keys() :
                    if t in _files.keys() :
                        t_fits=pyfits.open(_files[t])
                        t_data=t_fits[1].data
                        for pname in ts[t].keys() :
                            p=ts[t][pname]
                            if isinstance(p, list) :
                                p_values=[]
                                for f in p :
                                    p_values+=[t_data.field(f),]
                            else :
                                p_values=t_data.field(p)
                            self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name'][pname]].value=(
                                ",".join([str(v) for v in np.array(p_values).transpose().ravel()])
                            )
                        t_fits.close()
                '''
                # set the values of the paramters from _BFP...
                _hdulist=pyfits.open(_files['BEST_FIT_PARAMETERS'])
                _BFP=Table.read(_files['BEST_FIT_PARAMETERS'], hdu=1, format='fits')
                best_fit_params_names={
                    #'wl_fit': 'WLC_CONST',  ## Should be a list per chip, but reflex.py can't handle this
                    #'continuum_fit': 'CONTINUUM_CONST',  ## Should be a list per range,chip, but reflex.py can't handle this
                    'rel_mol_col': 'REL_COL',
                    'boxfwhm': 'RES_BOX',
                    'gaussfwhm': 'RES_GAUSS',
                    'lorentzfwhm': 'RES_LORENTZ',
                    #'telback': 'TEL_BACK',
                }
                best_fit_params={
                    'wl_fit': [],
                    'continuum_fit': [],
                    'rel_mol_col': [],
                }
                fit_statistics={
                    'index': [
                        'status',
                        'initial_chi2',
                        'best_chi2',
                        'reduced_chi2',
                        'rms_rel_to_err',
                        'rms_rel_to_mean',
                        'fit_params',
                        'data_points',
                        'positive_weights',
                        'valid_pix_frac',
                        'iterations',
                        'func_eval',
                        'lblrtm_calls',
                    ],
                    'status': None,
                    'fit_params': None,
                    'data_points': None,
                    'positive_weights': None,
                    'valid_pix_frac': None,
                    'iterations': None,
                    'func_eval': None,
                    'lblrtm_calls': None,
                    'initial_chi2': None,
                    'best_chi2': None,
                    'reduced_chi2': None,
                    'rms_rel_to_err': None,
                    'rms_rel_to_mean': None,
                }
                fit_statistics_types={
                    'status': int,
                    'fit_params': int,
                    'data_points': int,
                    'positive_weights': int,
                    'valid_pix_frac': int,
                    'iterations': int,
                    'func_eval': int,
                    'lblrtm_calls': int,
                    'initial_chi2': float,
                    'best_chi2': float,
                    'reduced_chi2': float,
                    'rms_rel_to_err': float,
                    'rms_rel_to_mean': float,
                }
                rel_col_mols=[]
                fitted_rel_col_mols=[]
                best_fit_params['wl_fit_coeffs']={}
                best_fit_params['continuum_fit_coeffs']={}
                coeffs_order={
                    'wl_fit_coeffs': {},
                    'continuum_fit_coeffs': {},
                }
                for r in _BFP :
                    #print('Parameter %s has value %f, uncertainty %f' %(r['parameter'],r['value'],r['uncertainty']))
                    _m_wl_fit=re.match(r'^chip ([0-9][0-9]*), coef ([0-9][0-9]*)',r['parameter'])
                    _m_continuum_fit=re.match(r'^Range ([0-9][0-9]*), chip ([0-9][0-9]*), coef ([0-9][0-9]*)',r['parameter'])
                    _m_rel_mol_col=re.match(r'^rel_mol_col_(.*)',r['parameter'])
                    if r['parameter'].strip() in fit_statistics.keys() :
                        fit_statistics[r['parameter'].strip()]=fit_statistics_types[r['parameter'].strip()](r['value'])
                    elif r['uncertainty'] >= 0. :
                        ## ToDo: Need to flag params with r['uncertainty'] == 0.
                        ## ToDo: should tabulate the BFPs
                        if _m_wl_fit is not None :
                            chip,coef = (
                                int(_m_wl_fit.group(1)),
                                int(_m_wl_fit.group(2)),
                            )
                            if coef == 0 :
                                best_fit_params['wl_fit']+=[r['value'],]
                            if chip not in best_fit_params['wl_fit_coeffs'].keys() :
                                best_fit_params['wl_fit_coeffs'][chip]=[]
                                coeffs_order['wl_fit_coeffs'][chip]=[]
                            coeffs_order['wl_fit_coeffs'][chip]+=[coef]
                            best_fit_params['wl_fit_coeffs'][chip]+=[[r['value'],r['uncertainty']],]

                        elif _m_continuum_fit is not None :
                            _range,chip,coef = (
                                int(_m_continuum_fit.group(1)),
                                int(_m_continuum_fit.group(2)),
                                int(_m_continuum_fit.group(3)),
                            )
                            if coef == 0 :
                                best_fit_params['continuum_fit']+=[r['value'],]
                            if chip not in best_fit_params['continuum_fit_coeffs'].keys() :
                                best_fit_params['continuum_fit_coeffs'][chip]={}
                                coeffs_order['continuum_fit_coeffs'][chip]={}
                            if _range not in best_fit_params['continuum_fit_coeffs'][chip].keys() :
                                best_fit_params['continuum_fit_coeffs'][chip][_range]=[]
                                coeffs_order['continuum_fit_coeffs'][chip][_range]=[]
                            coeffs_order['continuum_fit_coeffs'][chip][_range]+=[coef]
                            best_fit_params['continuum_fit_coeffs'][chip][_range]+=[[r['value'],r['uncertainty']],]
                        else :
                            best_fit_params[r['parameter'].strip()]=str(r['value'])
                    if _m_rel_mol_col is not None :
                            best_fit_params['rel_mol_col']+=[r['value'],]
                            rel_col_mols+=[_m_rel_mol_col.group(1).strip(),]
                            if r['uncertainty'] >= 0. :
                                fitted_rel_col_mols+=[_m_rel_mol_col.group(1).strip(),]
                self.fit_statistics=fit_statistics
                for p in ['wl_fit', 'continuum_fit',] :
                    if len(best_fit_params[p]) == 0 :
                        del best_fit_params[p]
                    else :
                        best_fit_params[p]=",".join(["%8.6f" %(x) for x in best_fit_params[p]])
                for p in ['wl_fit_coeffs', ] :
                    if best_fit_params[p] == {} :
                        del best_fit_params[p]
                    else :
                        for chip in best_fit_params[p].keys() :
                            best_fit_params[p][chip]=list(
                                np.array(best_fit_params[p][chip])[coeffs_order[p][chip]]
                            )
                for p in ['continuum_fit_coeffs' ] :
                    if best_fit_params[p] == {} :
                        del best_fit_params[p]
                    else :
                        for chip in best_fit_params[p].keys() :
                            for _range in best_fit_params[p][chip].keys() :
                                best_fit_params[p][chip][_range]=list(
                                    np.array(best_fit_params[p][chip][_range])[coeffs_order[p][chip][_range]]
                                )
                for p in ['rel_mol_col',] :
                    fitted_REL_COL=best_fit_params[p]
                    LIST_MOLEC=init_sop_dict_param(self.bfp_init_sop, 'LIST_MOLEC').value.split(',')
                    REL_COL=init_sop_dict_param(self.bfp_init_sop, 'REL_COL').value.split(',')
                    FIT_MOLEC=init_sop_dict_param(self.bfp_init_sop, 'FIT_MOLEC').value.split(',')
                    #REL_COL=self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['REL_COL']].value.split(',')
                    #FIT_MOLEC=self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['FIT_MOLEC']].value.split(',')
                    for (i,_lm) in enumerate(LIST_MOLEC) :
                        if _lm in rel_col_mols :
                            j=rel_col_mols.index(_lm)
                            REL_COL[i]=str(fitted_REL_COL[j])
                            if _lm in fitted_rel_col_mols :
                                FIT_MOLEC[i]='1'
                    best_fit_params[p]=','.join(REL_COL)
                    self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name']['FIT_MOLEC']].value=(
                        ",".join(FIT_MOLEC)
                    )
                self.best_fit_parameters=best_fit_params
                # Probably redundnt now to have self.best_fit_parameters AND self.bfp_init_sop
                # but to avoid breaking anything...
                for p in best_fit_params_names.keys() :
                    pname=best_fit_params_names[p]
                    if p in best_fit_params :
                        self.bfp_init_sop['init_sop'][self.bfp_init_sop['by_name'][pname]].value=best_fit_params[p]
    # ---------------------------------------------------------------------------------------
    def read_molecules(self, files) :
        self.molecules=None
        #return
        AM_file=None
        M_file=None
        M_atlas=None
        molecule_list=[]
        for f in files :
            if f.category in ['AUTO_MOLECULES',] :
                AM_file=f
            if f.category in ['MOLECULES',] :
                M_file=f
            if f.category in ['MOLECULE_ATLAS',] :
                M_atlas=f
        if M_atlas is not None :
            self.molecules={
                "names": [],
                "all_names": [],
                "ca_names": [],
            }
            if M_file is not None :
                M_hdulist=pyfits.open(M_file.name)
                self.molecules["LIST_MOLEC"]=list(M_hdulist[1].data['LIST_MOLEC'])
            if AM_file is not None :
                AM_hdulist=pyfits.open(AM_file.name)
                for AM_hdu in AM_hdulist[1:] :
                    if AM_hdu.header.get(
                        'HIERARCH ESO PRO CATG',
                        AM_hdu.header.get('EXTNAME','UNKNOWN')
                    ) in ["AM_SPECTRUM","AUTO_MOLECULES"] :
                        if 'molecule' in AM_hdu.data.names :
                            for _molecule in AM_hdu.data['molecule'] :
                                if _molecule not in molecule_list :
                                    molecule_list+=[_molecule,]
            M_atlas_hdulist=pyfits.open(M_atlas.name)
            for M_atlas_hdu in M_atlas_hdulist[1:] :
                if M_atlas_hdu.header.get('EXTNAME') in ["MOLECULE_ATLAS",] :
                    #for _molecule in M_atlas_hdu.data.names[1:] :
                    for _molecule in m_specs+trace_m_specs :
                        if _molecule in M_atlas_hdu.data.names :
                            if _molecule in molecule_list+self.molecules.get("LIST_MOLEC",[]):
                                self.molecules['names']+=[_molecule,]
                            self.molecules['all_names']+=[_molecule,]
                            self.molecules[_molecule] = {
                                'mlambda': M_atlas_hdu.data['mlambda'],
                                'mtrans': M_atlas_hdu.data[_molecule],
                            }
                            if "%s_wca" %(_molecule) in M_atlas_hdu.columns.names :
                                self.molecules['ca_names']+=[_molecule,]
                                self.molecules[_molecule]['mtrans_wca'] = M_atlas_hdu.data["%s_wca" %(_molecule)]
                    if "ca_rayl" in M_atlas_hdu.columns.names :
                        self.molecules["ca_rayl"] = {
                            'mlambda': M_atlas_hdu.data['mlambda'],
                            'mtrans': M_atlas_hdu.data["ca_rayl"],
                        }

    # ---------------------------------------------------------------------------------------
    def toggle_sis(self, event):
        '''
        '''
        '''
        # I think we want to be able to write both... so comment this block out...
        if event.EventObject.Name == 'setInitSopCheck' :
            self.wxApp_elements_by_name['setInitSopCheck_OBJECT_and_SETTING'][0].SetValue(False)
        elif event.EventObject.Name == 'setInitSopCheck_OBJECT_and_SETTING' :
            self.wxApp_elements_by_name['setInitSopCheck'][0].SetValue(False)
        '''
        self.write_sis=self.wxApp_elements_by_name['setInitSopCheck'][0].GetValue()
        if 'setInitSopCheck_OBJECT_and_SETTING' in self.wxApp_elements_by_name.keys() :
            self.write_sis_OS=self.wxApp_elements_by_name['setInitSopCheck_OBJECT_and_SETTING'][0].GetValue()
    # ---------------------------------------------------------------------------------------
    def clicked_ApplyBestFitParameters_button(self, event):
        self.WLC_CONST_has_been_changed=False
        self.RES_BOX_has_been_changed=False
        self.RES_GAUSS_has_been_changed=False
        self.RES_LORENTZ_has_been_changed=False
        mean_wlc_const_value=None
        if 'wl_fit_coeffs' in self.best_fit_parameters.keys() :
            wlc_const_vals=[]
            for chip in self.best_fit_parameters['wl_fit_coeffs'].keys() :
                wlc_const_vals+=[
                    self.best_fit_parameters['wl_fit_coeffs'][chip][0][0]
                ]
            mean_wlc_const_value=np.mean(wlc_const_vals)
        self.clicked_ApplyParameters_button(self.bfp_init_sop, wlc_const_val=mean_wlc_const_value)
    # ---------------------------------------------------------------------------------------
    def clicked_ApplyOriginalParameters_button(self, event):
        self.WLC_CONST_has_been_changed=True
        self.RES_BOX_has_been_changed=True
        self.RES_GAUSS_has_been_changed=True
        self.RES_LORENTZ_has_been_changed=True
        self.clicked_ApplyParameters_button(self.orig_init_sop)
    # ---------------------------------------------------------------------------------------
    def clicked_apply_saved_dfd_params(self, event):
        pass
    # ---------------------------------------------------------------------------------------
    def clicked_ApplySavedParameters_button(self, event):
        self.WLC_CONST_has_been_changed=True
        self.RES_BOX_has_been_changed=True
        self.RES_GAUSS_has_been_changed=True
        self.RES_LORENTZ_has_been_changed=True
        set_dfd_params=not self.wxApp.apply_saved_dfd_params.GetValue()
        self.clicked_ApplyParameters_button(self.saved_init_sop, set_dfd_params=set_dfd_params)
    # ---------------------------------------------------------------------------------------
    def clicked_ApplyPSEParameters_button(self, event):
        self.WLC_CONST_has_been_changed=True
        self.RES_BOX_has_been_changed=True
        self.RES_GAUSS_has_been_changed=True
        self.RES_LORENTZ_has_been_changed=True
        self.clicked_ApplyParameters_button(self.pse_init_sop)
    # ---------------------------------------------------------------------------------------
    def clicked_ApplyParameters_button(self, sop_dict, set_dfd_params=False, wlc_const_val=None):
        readFitsFiles_params={}
        for pname in [
            'WLG_TO_MICRON',
            'COLUMN_LAMBDA',
            'COLUMN_FLUX',
            'COLUMN_DFLUX',
            'USE_ONLY_INPUT_PRIMARY_DATA',
            'USE_DATA_EXTENSION_AS_DFLUX',
            'USE_DATA_EXTENSION_AS_MASK',
        ] :
            if self.get_pcc_Value(pname) is not None :
                readFitsFiles_params[pname]=self.get_pcc_Value(pname)
        for p in self.wxApp.shownParamWidgets :
            pname=p[1].parameter.name
            if pname == 'WLC_CONST' and wlc_const_val is not None :
                if self.get_pcc_Value(pname) != wlc_const_val :
                    self.set_pcc_Value(pname, wlc_const_val)
            elif pname in [
                'WAVE_INCLUDE',
                'WAVE_EXCLUDE',
            ] :
                _wl=init_sop_dict_param(sop_dict,pname).value
                '''
                if _wl.lower() not in [
                    'null'
                ] :
                    _wl=self.wl_to_topo_vac(
                        np.array(init_sop_dict_param(sop_dict,pname).value.split(','), dtype=float)
                    )
                    _wl=",".join([iu.wrfmt(x) for x in _wl])
                    pass
                '''
                if self.get_pcc_Value(pname) != _wl :
                    self.set_pcc_Value(pname,_wl,)
            elif pname in [
                'PIXEL_EXCLUDE',
            ] :
                _wl=init_sop_dict_param(sop_dict,pname).value
                if _wl.lower() not in [
                    'null'
                ] :
                    _wl=np.array(np.array(init_sop_dict_param(sop_dict,pname).value.split(','), dtype=float), dtype=int)
                    _wl=",".join([iu.wrfmt(x) for x in _wl])
                if self.get_pcc_Value(pname) != _wl :
                    self.set_pcc_Value(pname,_wl,)                    
            else :
                if self.get_pcc_Value(pname) != init_sop_dict_param(sop_dict,pname).value :
                    self.set_pcc_Value(pname,init_sop_dict_param(sop_dict,pname).value)

        if set_dfd_params :
            # Use the following parameters from inputs.in_sop=self.orig_init_sop
            # since these are data-format dependent:
            for pname in [
                'WLG_TO_MICRON',
                'COLUMN_LAMBDA',
                'COLUMN_FLUX',
                'COLUMN_DFLUX',
                'USE_ONLY_INPUT_PRIMARY_DATA',
                'USE_DATA_EXTENSION_AS_DFLUX',
                'USE_DATA_EXTENSION_AS_MASK',
            ] :
                if self.get_pcc_Value(pname) is not None :
                    if self.get_pcc_Value(pname) != init_sop_dict_param(self.orig_init_sop,pname).value :
                        self.set_pcc_Value(pname,init_sop_dict_param(self.orig_init_sop,pname).value)

        readFitsFiles_params_changed=False
        for pname in readFitsFiles_params.keys() :
            if self.get_pcc_Value(pname) is not None :
                if readFitsFiles_params[pname] != self.get_pcc_Value(pname) :
                    readFitsFiles_params_changed=True
        if readFitsFiles_params_changed :
            self.readFitsParam_changed()

        if not hasattr( interactive_app, 'user_edited_param' ):
            interactive_app.user_edited_param=sop_dict['init_sop']
        if hasattr( self, 'molecules' ) :
            if self.molecules is not None :
                self.molecules['cont_abs'].SetValue( self.get_pcc_Value('LBLRTM_ICNTNM') == 5 )
        self.set_regions(sop_dict=sop_dict, init=True)
        if hasattr(self, 'extname') :
            self.draw_all_fill_regions(self.extname)
        self.LIST_MOLEC_changed()
        self.check_required_columns()
        self.check_header_keywords()
        self.replot_molecules()
        return
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    def check_required_columns(self, params=None) :
        req_opt_cols=dict(list(self.required_columns.items())+list(self.optional_columns.items()))
        for p in params or req_opt_cols.keys() :
            if p in req_opt_cols.keys() :
                if self.get_pcc(p) is not None :
                    if req_opt_cols.get(p,{}).get('exists',False) :
                        self.get_pcc(p).SetBackgroundColour(wx.Colour(255, 255, 255))
                    else :
                        self.get_pcc(p).SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def check_header_keywords(self, params=None) :
        for p in self.wxApp.shownParamWidgets :
            pname=p[1].parameter.name
            if '_KEYWORD' in pname :
                if (
                    self.get_pcc_Value(pname) in ['NONE',]
                    or
                    self.get_pcc_Value(pname) in self.f_product_header
                ) :
                    self.get_pcc(pname).SetBackgroundColour(wx.Colour(255, 255, 255))
                else :
                    print(
                        "WARNING :: keyword %s=%s not found in primary HDU header"
                        %(pname,self.get_pcc_Value(pname))
                    )
                    self.get_pcc(pname).SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def set_regions(
        self,
        pnames=None,
        sop=None, sop_dict=None,
        regions=None, delete_regions=None,
        init=False,
    ):
        '''
        Optional parameters:
        pnames: either a list of pnames, or a comma separated str-list... one or more of
                WAVE_INCLUDE,WAVE_EXCLUDE,PIXEL_EXCLUDE
        regions: a list of lists, number of lists in the list must be equal to len(pnames)
        delete_regions: a list of lists, number of lists in the list must be equal to len(pnames)
        '''
        if not hasattr(self, 'regions' ) :
            self.regions={
                'WAVE_INCLUDE': {'index': []},
                'WAVE_EXCLUDE': {'index': []},
                'PIXEL_EXCLUDE': {'index': []},
            }
            init=True

        if pnames is None :
            pnames=list(self.regions.keys())
        if isinstance(pnames,str) :
            pnames=pnames.split(',')
        elif isinstance(pnames,list) :
            pass
        else :
            raise ValueError('pname type not supported')

        if regions is not None :
            if len(regions) != len(pnames) :
                raise Exception('Number of pnames and regions lists does not match')

        if delete_regions is not None :
            if len(delete_regions) != len(pnames) :
                raise Exception('Number of pnames and delete_regions lists does not match')

        current_regions=[]
        for ii,pname in enumerate(pnames) :
            current_region=self.get_regions(
                sop_dict or init_sop_dict( sop or interactive_app.user_edited_param ),
                pname,
            ) or {}
            current_regions+=[current_region,]

        if regions == None :
            regions=[]
            for ii,pname in enumerate(pnames) :
                _cr=current_regions[ii].get(pname, [])
                if _cr in [['',],] :
                    _cr=[]
                regions+=[
                    np.array(
                        np.array(_cr,dtype=float),
                        dtype=self.region_properties.get(pname,{}).get('type',float)
                    ).reshape([len(current_regions[ii].get(pname, []))//2,2]),
                ]
        if len(regions) == 0 :
            return

        for ii,pname in enumerate(pnames) :
            if init :
                self.regions[pname]={'index': []}
            else :
                if delete_regions is not None :
                    _d_regions=delete_regions[ii]
                    for wr in _d_regions :
                        if isinstance(wr, str) :
                            wr_label=wr
                        else :
                            wr_label=','.join([ iu.wrfmt(x) for x in wr ])
                        if wr_label in self.regions[pname]['index'] :
                            del self.regions[pname]['index'][self.regions[pname]['index'].index(wr_label)]
                        if wr_label in self.regions[pname].keys() :
                            del self.regions[pname][wr_label]

            #regions[ii].sort()
            parsed_regions, regions_map = self.parse_regions( regions[ii] )

            _regions=np.array(
                parsed_regions,
                dtype=self.region_properties.get(pname,{}).get('type',float),
            ).reshape([len(parsed_regions)//2,2])

            for i,wr in enumerate(_regions) :
                wr_label=','.join([ iu.wrfmt(x) for x in wr ])
                if not wr_label in self.regions[pname].keys() :
                    if init :
                        self.regions[pname]['index']+=[wr_label,]
                        self.regions[pname][wr_label]={
                            'region': wr,
                        }
                        if pname in ['WAVE_INCLUDE',] :
                            for _pname in [
                                'MAP_REGIONS_TO_CHIP',
                            ] :
                                self.regions[pname][wr_label][_pname]=str(regions_map[i][0])
                            for _pname in [
                                'FIT_CONTINUUM',
                                'CONTINUUM_N',
                                #'FIT_WLC',
                                #'WLC_N',
                            ] :
                                if len(current_regions[ii][_pname]) > 1 :
                                    self.regions[pname][wr_label][_pname]=current_regions[ii][_pname][regions_map[i][1]]
                                else :
                                    self.regions[pname][wr_label][_pname]=current_regions[ii][_pname][0]
                    else :
                        # When region not currently in self.regions[pname] add it...
                        # need to insert this at the right place...
                        self.regions[pname]['index']+=[wr_label,]
                        self.regions[pname][wr_label]={
                            'region': wr,
                        }
                        if pname in ['WAVE_INCLUDE',] :
                            # set MAP_REGIONS_TO_CHIP to the first chip containing this region...
                            # this will get updated later by self.get_param_ranges()
                            for _pname in [
                                'MAP_REGIONS_TO_CHIP',
                            ] :
                                self.regions[pname][wr_label][_pname]=regions_map[i][0]

                            # Set the additional parameters appropriately based
                            for _pname in [
                                'FIT_CONTINUUM',
                                'CONTINUUM_N',
                                #'FIT_WLC',
                                #'WLC_N',
                            ] :
                                #selected_region['index']
                                if self.modified_region is None :
                                    self.regions[pname][wr_label][_pname]=0
                                    self.regions[pname][wr_label][_pname]=np.max(
                                            np.array(
                                                np.array(
                                                    self.orig_init_sop['init_sop'][self.orig_init_sop['by_name'][_pname]].value.split(','),
                                                    dtype=float
                                                )
                                                , dtype=int
                                            )
                                        )
                                    if _pname in current_regions[ii].keys() :
                                        self.regions[pname][wr_label][_pname]=np.max(
                                            np.array(
                                                np.array(
                                                    current_regions[ii][_pname],
                                                    dtype=float
                                                )
                                                , dtype=int
                                            )
                                        )
                                else :
                                    self.regions[pname][wr_label][_pname]=self.modified_region[_pname]
            wrs=[]
            if len(self.regions[pname]['index']) == 0 :
                if not init :
                    self.set_pcc_Value(
                        pname,
                        'NULL',
                    )
                    if pname in ['WAVE_INCLUDE',] :
                        for _pname in [
                            'MAP_REGIONS_TO_CHIP',
                            'FIT_CONTINUUM',
                            'CONTINUUM_N',
                            #'FIT_WLC',
                            #'WLC_N',
                        ] :
                            self.set_pcc_Value(
                                _pname,
                                'NULL',
                            )
                    self.regions[pname]['index']=[]
                    self.regions[pname][pname]=[]
            else :

                for wr_label in self.regions[pname]['index'] :
                    wrs+=[self.regions[pname][wr_label]['region'],]
                wrs=np.sort(wrs,axis=0)
                self.regions[pname][pname]=','.join(iu.wrfmt(x) for x in wrs.flatten())
                if not init :
                    self.set_pcc_Value(
                        pname,
                        self.regions[pname][pname],
                    )
                #wi=self.get_pcc_Value(pname)
                self.regions[pname]['index']=[]
                for i,wr in enumerate(wrs) :
                    wr_label=','.join([ iu.wrfmt(x) for x in wr ])
                    self.regions[pname]['index']+=[wr_label,]
                if pname in ['WAVE_INCLUDE',] :
                    for _pname in [
                        'MAP_REGIONS_TO_CHIP',
                        'FIT_CONTINUUM',
                        'CONTINUUM_N',
                        #'FIT_WLC',
                        #'WLC_N',
                    ] :
                        self.regions[pname][_pname]=[]
                        for i,wr_label in enumerate(self.regions[pname]['index']) :
                            if wr_label in self.regions[pname].keys() :
                                self.regions[pname][_pname]+=[self.regions[pname][wr_label][_pname],]
                        '''
                        val=','.join(str(x) for x in self.regions[pname][_pname])
                        uval=np.uninque(self.regions[pname][_pname])
                        if len(uval) == 1 :
                            val=str(uval[0])
                        '''
                        if not init :
                            self.set_pcc_Value(
                                _pname,
                                ','.join(str(x) for x in self.regions[pname][_pname]),
                            )

    # ---------------------------------------------------------------------------------------
    def addRegion(self, pname, wl_range, redraw=True):
        '''
        pname: one of WAVE_INCLUDE,WAVE_EXCLUDE,PIXEL_EXCLUDE
        '''

        wave_region=self.get_pcc_Value(pname).split(',')
        if wave_region[0].lower() in ["null",""] :
            wave_region=[]
        wave_region=np.array(wave_region, dtype=float).reshape([len(wave_region)//2,2])
        wrti=[]
        for _rr in wave_region :
            if wl_range[0] <= _rr[1] and wl_range[1] >= _rr[0] :
                wl_range=[
                    np.min([wl_range[0],_rr[0]]),
                    np.max([wl_range[1],_rr[1]]),
                ]
            else :
                wrti.append(list(_rr))
        wrti.append(wl_range)
        new_ranges=[]
        deleted_ranges=[]
        wrti=np.array(wrti)
        for _rr in wrti :
            if list(_rr) not in [ list(item) for item in wave_region ] :
                new_ranges+=[_rr,]
        for _rr in wave_region :
            if list(_rr) not in [ list(item) for item in wrti ] :
                deleted_ranges+=[_rr,]

        self.set_regions(
            pnames=pname,
            regions=[new_ranges],
            delete_regions=[deleted_ranges],
        )

        self.setEnabledDeleteButton(pname)
        if redraw : self.draw_all_fill_regions(self.extname)
    # ---------------------------------------------------------------------------------------
    def deleteRegion(self, pname, index, redraw=True ):

        if index is not None :
            wave_region=self.get_pcc_Value(pname).split(',')
            if wave_region[0] in ["NULL",""] :
                wave_region=[]
            #wave_region=np.delete(
            #    np.array(wave_region, dtype=float).reshape([len(wave_region)//2,2]),
            #    [index,], axis=0
            #)
            deleted_ranges=list(
                np.array(
                    wave_region, dtype=float
                ).reshape([len(wave_region)//2,2])[index,]
            )
            self.set_regions(
                pnames=pname,
                regions=[[],],
                delete_regions=[[deleted_ranges,],],
            )
        self.setEnabledDeleteButton(pname)
        if redraw : self.draw_all_fill_regions(self.extname)
    # ---------------------------------------------------------------------------------------
    def setEnabledDeleteButton(self, pname ):

        if pname in self.delete_buttons.keys() :
            if self.params_by_name[(self.recipe_name, pname)].paramChangeCtrl.GetValue() in ["NULL",] :
                # Set button Enabled=False
                self.delete_buttons[pname].Enable(False)
            else :
                # Set button Enabled=True
                self.delete_buttons[pname].Enable(True)
    # ---------------------------------------------------------------------------------------
    def any_is_active(self, exclude=[]):
        buttons_active={
            'IR_add': self.IR_add_active,
            'IR_del': self.IR_del_active,
            'IR_mod': self.IR_mod_active,
            'ER_add': self.ER_add_active,
            'ER_del': self.ER_del_active,
            'ER_mod': self.ER_mod_active,
        }
        _is_active=False
        for b in buttons_active.keys() :
            if b not in exclude :
                _is_active=_is_active or buttons_active[b]
        return _is_active
    # ---------------------------------------------------------------------------------------
    def enable_region_buttons(self, enable, exclude=[]):
        buttons_active={
            'IR_add': self.add_buttons.get("WAVE_INCLUDE"),
            'IR_del': self.delete_buttons.get("WAVE_INCLUDE"),
            'IR_mod': self.modify_buttons.get("WAVE_INCLUDE"),
            'ER_add': self.add_buttons.get("WAVE_EXCLUDE"),
            'ER_del': self.delete_buttons.get("WAVE_EXCLUDE"),
            'ER_mod': self.modify_buttons.get("WAVE_EXCLUDE"),
        }
        for b in buttons_active.keys() :
            if b not in exclude and buttons_active[b] is not None :
                buttons_active[b].Enable(enable)
        if enable :
            self.setEnabledDeleteButton('WAVE_INCLUDE')
            self.setEnabledDeleteButton('WAVE_EXCLUDE')
    # ---------------------------------------------------------------------------------------
    def AddIncludeRegion(self, event):
        if not self.any_is_active(['IR_add',]) :
            if self.IR_add_active :
                if self.span.extents != (0.,0.) :
                    # Set the region...
                    wl_range_to_include = list(self.span.extents)
                    #wl_range_to_include = self.wl_back_from_topo_vac(wl_range_to_include)
                    self.addRegion(
                        'WAVE_INCLUDE', wl_range_to_include,
                    )
                    self.span.extents=(0.,0.)
                    self.span.active = False
                    self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                    self.IR_add_active = not self.IR_add_active
                    self.spec_plot.set_title(self.title_text)
                    self.enable_region_buttons(True)
            else :
                self.enable_region_buttons(False)
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                self.span.extents=(0.,0.)
                self.span.active = True
                self.IR_add_active = not self.IR_add_active
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_IR_add
                )
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def DeleteIncludeRegion(self, event):
        if not self.any_is_active(['IR_del',]) :
            # Implement methods to enable/disable delete buttons if
            # WAVE_IN/EXCLUDE lists are empty="NULL"
            if self.IR_del_active :
                ind=self.selected_region['index']
                self.selected_region['pname'] = None
                self.selected_region['index'] = None
                # Delete the region...
                self.deleteRegion(
                    'WAVE_INCLUDE', ind,
                )
                self.span.extents=(0.,0.)
                self.span.active = False
                self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                self.IR_del_active = not self.IR_del_active
                self.spec_plot.set_title(self.title_text)
                self.enable_region_buttons(True)
            else :
                self.enable_region_buttons(False)
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                # Select region via mouse click
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_IR_delete
                )
                self.IR_del_active = not self.IR_del_active
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def ModifyIncludeRegion(self, event):
        if not self.any_is_active(exclude=['IR_mod',]) :
            if self.IR_mod_active :
                # Set the region...
                wl_range_to_include = list(self.span.extents)
                #wl_range_to_include = self.wl_back_from_topo_vac(wl_range_to_include)
                ind=self.selected_region['index']
                pname='WAVE_INCLUDE'
                selected_wr_label=self.regions[pname]['index'][self.selected_region['index']]
                self.modified_region=self.regions[pname][selected_wr_label]
                self.selected_region['pname'] = None
                self.selected_region['index'] = None
                self.deleteRegion(
                    'WAVE_INCLUDE', ind, redraw=False,
                )
                self.addRegion(
                    'WAVE_INCLUDE', wl_range_to_include,
                )
                self.modified_region=None
                self.span.extents=(0.,0.)
                self.span.prev=(0.,0.)
                self.span.active = False
                self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                self.IR_mod_active = not self.IR_mod_active
                self.enable_region_buttons(True)
                self.spec_plot.set_title(self.title_text)
            else :
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                # Select region via mouse click
                self.enable_region_buttons(False)
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_IR_modify
                )
                self.IR_mod_active = not self.IR_mod_active
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    def AddExcludeRegion(self, event):
        if not self.any_is_active(['ER_add',]) :
            if self.ER_add_active :
                if self.span.extents != (0.,0.) :
                    # Set the region...
                    wl_range_to_include = list(self.span.extents)
                    #wl_range_to_include = self.wl_back_from_topo_vac(wl_range_to_include)
                    self.addRegion(
                        'WAVE_EXCLUDE', wl_range_to_include,
                    )
                    self.span.extents=(0.,0.)
                    self.span.active = False
                    self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                    self.ER_add_active = not self.ER_add_active
                    self.spec_plot.set_title(self.title_text)
                    self.enable_region_buttons(True)
            else :
                self.enable_region_buttons(False)
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                self.span.extents=(0.,0.)
                self.span.active = True
                self.ER_add_active = not self.ER_add_active
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_ER_add
                )
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def DeleteExcludeRegion(self, event):
        if not self.any_is_active(['ER_del',]) :
            # Implement methods to enable/disable delete buttons if
            # WAVE_IN/EXCLUDE lists are empty="NULL"
            if self.ER_del_active :
                ind=self.selected_region['index']
                self.selected_region['pname'] = None
                self.selected_region['index'] = None
                # Delete the region...
                self.deleteRegion(
                    'WAVE_EXCLUDE', ind,
                )
                self.span.extents=(0.,0.)
                self.span.active = False
                self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                self.ER_del_active = not self.ER_del_active
                self.spec_plot.set_title(self.title_text)
                self.enable_region_buttons(True)
            else :
                self.enable_region_buttons(False)
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                # Select region via mouse click
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_ER_delete
                )
                self.ER_del_active = not self.ER_del_active
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def ModifyExcludeRegion(self, event):
        if not self.any_is_active(exclude=['ER_mod',]) :
            if self.ER_mod_active :
                ind=self.selected_region['index']
                pname='WAVE_EXCLUDE'
                selected_wr_label=self.regions[pname]['index'][self.selected_region['index']]
                self.modified_region=self.regions[pname][selected_wr_label]
                self.selected_region['pname'] = None
                self.selected_region['index'] = None
                # Set the region...
                wl_range_to_include = list(self.span.extents)
                #wl_range_to_include = self.wl_back_from_topo_vac(wl_range_to_include)
                self.deleteRegion(
                    'WAVE_EXCLUDE', ind, redraw=False,
                )
                self.addRegion(
                    'WAVE_EXCLUDE', wl_range_to_include,
                )
                self.modified_region=None
                self.span.extents=(0.,0.)
                self.span.active = False
                self.spec_plot_axes['axes'].figure.canvas.mpl_disconnect(self.select_region_cid)
                self.ER_mod_active = not self.ER_mod_active
                self.enable_region_buttons(True)
                self.spec_plot.set_title(self.title_text)
            else :
                self.spec_plot.set_title('LMB=Select region ; RMB=Cancel')
                # Select region via mouse click
                self.enable_region_buttons(False)
                self.select_region_cid = self.spec_plot_axes['axes'].figure.canvas.mpl_connect(
                    'button_release_event', self.onclick_ER_modify
                )
                self.ER_mod_active = not self.ER_mod_active
            self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
     # ---------------------------------------------------------------------------------------
    def read_input_spec(self, files) :
        """
        Currently only handles IDP format data...
        Not ready for real-world usage...
        """
        sfile=None
        for file in files :
            if file.category in ['STD_MODEL','SCIENCE_CALCTRANS','SCIENCE',] :
                sfile=file
                break
        if sfile is None :
            return

        # Get the expected files
        try:
            orig_data = PipelineProduct(sfile)
        except:
            return

        # Initialize
        self.orig_data_extnames = []
        self.orig_data_lambda = dict()
        self.orig_data_flux = dict()

        # Loop on extensions
        ext_num = -1
        for orig_data_ext in orig_data.all_hdu:

            # Take extension in the output file
            ext_num = ext_num + 1

            naxis = orig_data_ext.header['NAXIS']
            if (naxis == 2):

                if ('EXTNAME' in orig_data_ext.header) :
                    extname = orig_data_ext.header['EXTNAME']
                else :
                    extname = 'EXT.' + str(ext_num)
                self.orig_data_extnames.append(extname)

                # Get infos from star_spec using the extname
                self.orig_data_lambda[extname] = orig_data_ext.data.field("WAVE")
                self.orig_data_flux[extname] = orig_data_ext.data.field("FLUX")
    # ---------------------------------------------------------------------------------------
    def LIST_MOLEC_changed_event( self, event ):
        self.LIST_MOLEC_changed()
        self.REL_COL_changed()
        self.FIT_MOLEC_changed()
    # ---------------------------------------------------------------------------------------
    def LIST_MOLEC_changed( self, mol_list=None ):
        lm=self.params_by_name[(self.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')
        for m in mol_list or self.molecules['add_order'] :
            if self.molecules[m].get('include') is not None :
                self.molecules[m]['include'].SetValue( m in lm )
    # ---------------------------------------------------------------------------------------
    def REL_COL_changed_event( self, event ):
        self.REL_COL_changed()
    # ---------------------------------------------------------------------------------------
    def REL_COL_changed( self, mol_list=None):
        lm=self.params_by_name[(self.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')
        rc=self.params_by_name[(self.recipe_name,'REL_COL')].paramChangeCtrl.GetValue().split(',')
        for m in mol_list or self.molecules['add_order'] :
            if 'relcol_value' not in self.molecules[m].keys() :
                self.molecules[m]['relcol_value']=1.0
            if m in lm :
                try :
                    self.molecules[m]['relcol_value'] = float(rc[lm.index(m)])
                except :
                    pass
            if self.molecules[m].get('relcol') is not None :
                    self.molecules[m]['relcol'].ChangeValue( str(self.molecules[m]['relcol_value']) )
        self.replot_molecules()
    # ---------------------------------------------------------------------------------------
    def FIT_MOLEC_changed_event( self, event ):
        self.FIT_MOLEC_changed()
    # ---------------------------------------------------------------------------------------
    def FIT_MOLEC_changed( self, mol_list=None ):
        lm=self.params_by_name[(self.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')
        fm=self.params_by_name[(self.recipe_name,'FIT_MOLEC')].paramChangeCtrl.GetValue().split(',')
        for m in mol_list or self.molecules['add_order'] :
            self.molecules[m]['fit_value']=False
            if m in lm :
                try:
                    self.molecules[m]['fit_value'] = fm[lm.index(m)] == '1'
                except :
                    pass
            if self.molecules[m].get('fit') is not None :
                    self.molecules[m]['fit'].SetValue( self.molecules[m]['fit_value'] )
    # ---------------------------------------------------------------------------------------
    def region_param_changed( self, param='FIT_CONTINUUM' ):
        pname='WAVE_INCLUDE'
        if hasattr( self, 'regions') :
            self.regions[pname][param]=self.get_pcc_Value(param)
            if self.regions[pname][param] not in ['NULL',] :
                vals=np.array(np.array(self.get_pcc_Value(param).split(','), dtype=float), dtype=int)
                _n=len(self.get_pcc_Value(pname).split(','))//2
                if len(vals) == 1 :
                    vals=list(np.repeat(vals,_n))
                for i,wl_label in enumerate(self.regions[pname]['index']) :
                    self.regions[pname][wl_label][param]=vals[i]
    # ---------------------------------------------------------------------------------------
    def FIT_CONTINUUM_changed_event( self, event ):
        self.region_param_changed(param='FIT_CONTINUUM')
    # ---------------------------------------------------------------------------------------
    def CONTINUUM_N_changed_event( self, event ):
        self.region_param_changed(param='CONTINUUM_N')
    # ---------------------------------------------------------------------------------------
    def FIT_WLC_changed_event( self, event ):
        self.region_param_changed(param='FIT_WLC')
    # ---------------------------------------------------------------------------------------
    def WLC_CONST_changed_event( self, event ):
        self.replot_molecules()
        self.WLC_CONST_has_been_changed=True
        #self.region_param_changed(param='WLC_CONST')
    # ---------------------------------------------------------------------------------------
    def FIT_RES_BOX_changed_event( self, event ):
        self.replot_molecules()
        self.RES_BOX_has_been_changed=True
    # ---------------------------------------------------------------------------------------
    def RES_BOX_changed_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_BOX')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def RES_BOX_saved_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_BOX')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 255, 255))
        if True or self.params_by_name[(self.recipe_name,'FIT_RES_BOX')].paramChangeCtrl.GetValue() :
            self.FIT_RES_BOX_changed_event(None)
    # ---------------------------------------------------------------------------------------
    def FIT_RES_GAUSS_changed_event( self, event ):
        self.replot_molecules()
        self.RES_GAUSS_has_been_changed=True
    # ---------------------------------------------------------------------------------------
    def RES_GAUSS_changed_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_GAUSS')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def RES_GAUSS_saved_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_GAUSS')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 255, 255))
        if True or self.params_by_name[(self.recipe_name,'FIT_RES_GAUSS')].paramChangeCtrl.GetValue() :
            self.FIT_RES_GAUSS_changed_event(None)
    # ---------------------------------------------------------------------------------------
    def FIT_RES_LORENTZ_changed_event( self, event ):
        self.replot_molecules()
        self.RES_LORENTZ_has_been_changed=True
    # ---------------------------------------------------------------------------------------
    def RES_LORENTZ_changed_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_LORENTZ')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def RES_LORENTZ_saved_event( self, event ):
        self.params_by_name[(self.recipe_name,'RES_LORENTZ')].paramChangeCtrl.SetBackgroundColour(wx.Colour(255, 255, 255))
        if True or self.params_by_name[(self.recipe_name,'FIT_RES_LORENTZ')].paramChangeCtrl.GetValue() :
            self.FIT_RES_LORENTZ_changed_event(None)
    # ---------------------------------------------------------------------------------------
    def MAP_REGIONS_TO_CHIP_changed_event( self, event ):
        self.region_param_changed(param='MAP_REGIONS_TO_CHIP')
    # ---------------------------------------------------------------------------------------
    def relcol_value_changed(self, event):
        _molecule=event.EventObject.molecule
        self.molecules[_molecule]['relcol'].SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def relcol_value_saved(self, event):
        _molecule=event.EventObject.molecule
        try :
            self.molecules[_molecule]['relcol_value'] = (
                float(self.molecules[_molecule]['relcol'].GetValue())
            )
            lm=self.params_by_name[(self.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')

            self.params_by_name[(self.recipe_name,'REL_COL')].paramChangeCtrl.SetValue(
                ','.join([str(self.molecules[m]['relcol_value']) for m in lm])
            )
            self.params_by_name[(self.recipe_name,'FIT_MOLEC')].paramChangeCtrl.SetValue(
                ','.join([str(int(self.molecules[m]['fit_value'])) for m in lm])
            )
            self.molecules[_molecule]['relcol'].SetBackgroundColour(wx.Colour(255, 255, 255))
        except :
            pass
    # ---------------------------------------------------------------------------------------
    def COLUMN_LAMBDA_changed_event( self, event ):
        self.readFitsParam_changed()
    # ---------------------------------------------------------------------------------------
    def COLUMN_FLUX_changed_event( self, event ):
        self.readFitsParam_changed()
    # ---------------------------------------------------------------------------------------
    def COLUMN_DFLUX_changed_event( self, event ):
        # Not currently loaded...
        # Should implement a way to check it is loaded...
        self.readFitsParam_changed( update_plot=False)
        #self.readFitsData(self.wxApp.in_sof_fitsFiles)
        pass
    # ---------------------------------------------------------------------------------------
    def WLG_TO_MICRON_changed_event( self, event ):
        self.readFitsParam_changed()
    # ---------------------------------------------------------------------------------------
    def WAVELENGTH_FRAME_changed_event( self, event ):
        self.readFitsParam_changed()
        delete_regions=[]
        for k in self.regions.keys() :
            delete_regions+=[sys_copy.deepcopy(self.regions[k]['index']),]
        self.set_regions(delete_regions=delete_regions)
    # ---------------------------------------------------------------------------------------
    def readFitsParam_changed( self, update_plot=True ):
        self.readFitsData(self.wxApp.in_sof_fitsFiles)
        self.check_required_columns()
        if update_plot :
            self._plot()
            pass
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def KEYWORD_changed_event( self, event ):
        self.check_header_keywords()
    # ---------------------------------------------------------------------------------------
    def toggle_continuum_absorption(self, event):
        self.set_pcc_Value('LBLRTM_ICNTNM', 5 if self.molecules['cont_abs'].GetValue() else 0 )
        if 'ca_names' in self.molecules.keys() :
            self.replot_molecules(m_list=self.molecules['ca_names'])
    # ---------------------------------------------------------------------------------------
    def toggle_molecule_include(self, event):
        lm=[]
        for m in self.molecules['add_order'] :
            if self.molecules[m]['include'].GetValue() :
                lm+=[m]
        self.params_by_name[(self.recipe_name,'LIST_MOLEC')].paramChangeCtrl.SetValue(','.join(lm))

        self.params_by_name[(self.recipe_name,'REL_COL')].paramChangeCtrl.SetValue(
            ','.join([str(self.molecules[m]['relcol_value']) for m in lm])
        )
        self.params_by_name[(self.recipe_name,'FIT_MOLEC')].paramChangeCtrl.SetValue(
            ','.join([str(int(self.molecules[m]['fit_value'])) for m in lm])
        )

    # ---------------------------------------------------------------------------------------
    def toggle_molecule_fit(self, event):
        _molecule=event.EventObject.molecule
        self.molecules[_molecule]['fit_value'] = (
            float(self.molecules[_molecule]['fit'].GetValue())
        )
        lm=[]
        for m in self.molecules['add_order'] :
            if self.molecules[m]['include'].GetValue() :
                lm+=[m]

        self.params_by_name[(self.recipe_name,'REL_COL')].paramChangeCtrl.SetValue(
            ','.join([str(self.molecules[m]['relcol_value']) for m in lm])
        )
        self.params_by_name[(self.recipe_name,'FIT_MOLEC')].paramChangeCtrl.SetValue(
            ','.join([str(int(self.molecules[m]['fit_value'])) for m in lm])
        )
    # ---------------------------------------------------------------------------------------
    def toggle_molecule_plot(self, event):
        _molecule=event.EventObject.molecule
        if event.EventObject.IsChecked() :
            '''
            try :
                # matplotlib > ...
                ymin,ymax = self.spec_plot_axes['axes'].get_ylim()
            except :
                # matplotlib-2...
                ymin,ymax = self.spec_plot.axes.get_ylim()
            '''
            self.plot_molecule(_molecule)
        else :
            self.molecules[_molecule]['line'].set_data([], [])
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def mf_convolution_mod_wave_grid(self, w, chip, use_bfp=True ) :
        # For WL scaling, see:
        # molecfit-kit-debug/telluriccorr-4.3.1/src/mf_convolution.c::mf_convolution_mod_wave_grid()
        _w=sys_copy.deepcopy(w)
        if have_chebyshev :
            _used_bfp=False
            if use_bfp :
                if 'wl_fit_coeffs' in self.best_fit_parameters.keys() :
                    if chip in self.best_fit_parameters['wl_fit_coeffs'] :
                        _used_bfp=True
                        _T=Chebyshev1D(
                            degree=len(self.best_fit_parameters['wl_fit_coeffs'][chip]),
                        )
                        _T_coeffs=[x[0] for x in self.best_fit_parameters['wl_fit_coeffs'][chip]]
            if not _used_bfp :
                _T=Chebyshev1D(
                    degree=2,
                )
                _T_coeffs=[
                    float(
                        self.params_by_name[(self.recipe_name,'WLC_CONST')].paramChangeCtrl.GetValue()
                    ),
                    1.,
                ]
            _dlam=np.nanmax(_w)-np.nanmin(_w)
            _wmean=(np.nanmin(_w)+np.nanmax(_w))/2.
            _w=(_w-_wmean)/(_dlam/2.)
            _w=_T.clenshaw(_w,_T_coeffs)
            _w=(_w*(_dlam/2.))+_wmean
        return _w
    # ---------------------------------------------------------------------------------------
    def mf_convolution_mod_continuum(self, w, chip, _range, wir ) :
        '''
        _range_wl=wlR(
            self.mf_convolution_mod_wave_grid(self.data_lambda[extn], chip),
            self.data_lambda[extn],
            wir[i]
        )
        _p_coeffs=[x[0] for x in self.best_fit_parameters['continuum_fit_coeffs'][chip][_range]]
        _p=Polynomial(coef=_p_coeffs,)
        self.continuum_fits['wl']+=[_range_wl[0],]+list(_range_wl)+[_range_wl[-1],]
        _wmean=(np.nanmin(_range_wl)+np.nanmax(_range_wl))/2.
        _range_cf_wl=(_range_wl-_wmean)
        _range_cf=_p(_range_cf_wl)
        self.continuum_fits['continuum_fit']+=[0.,]+list(_range_cf)+[0.,]
        '''
        pass
    # ---------------------------------------------------------------------------------------
    def plot_molecule(self, _molecule):
        ymin,ymax = self.spec_plot_axes['axes'].get_ylim()
        mtrans_col='mtrans'
        if (
            self.molecules['cont_abs'].GetValue()
            and
            'mtrans_wca' in self.molecules[_molecule].keys()
        ):
            mtrans_col='mtrans_wca'
        if ymin < 0. : ymin=0.
        _use_bfp=False
        if have_convolution :
            from astropy.convolution import Gaussian1DKernel, Box1DKernel, Model1DKernel
            from astropy.modeling.functional_models import Lorentz1D
            from astropy.convolution import convolve, convolve_fft
        if have_chebyshev :
            from astropy.modeling.polynomial import Chebyshev1D
        atlas_wls=[]
        atlas_trans=[]
        for i,extn in enumerate(self.orig_data_extnames) :
            _er=np.nanmax(self.orig_data_lambda[extn])-np.nanmin(self.orig_data_lambda[extn])
            _atlas_wlR=[
                np.nanmin(self.orig_data_lambda[extn])-0.01*_er,
                np.nanmax(self.orig_data_lambda[extn])+0.01*_er,
            ]
            # the wavelengths of the atlas over the wavelength range of the extension...
            chip=i+1
            _atlas_wls=wlR(
                self.wl_back_from_topo_vac(self.molecules[_molecule]['mlambda']),
                self.wl_back_from_topo_vac(self.molecules[_molecule]['mlambda']),
                _atlas_wlR,
            )
            if (
                not getattr(self,'WLC_CONST_has_been_changed',False)
                and
                not getattr(self,'RES_BOX_has_been_changed',False)
                and
                not getattr(self,'RES_GAUSS_has_been_changed',False)
                and
                not getattr(self,'RES_LORENTZ_has_been_changed',False)
                and
                hasattr(self,'best_fit_parameters')
            ) :
                _use_bfp=True
            data_resolution=self.orig_data_lambda_ranges['__wl_resolution__'][extn]
            _atlas_resolution=np.min(_atlas_wls[1:]-_atlas_wls[0:-1])
            _atlas_wls=self.mf_convolution_mod_wave_grid(
                _atlas_wls, chip, use_bfp=_use_bfp,
            )
            atlas_wls+=list(_atlas_wls)

            # the transmission over the wavelength range of the extension...
            _atlas_trans=wlR(
                self.molecules[_molecule][mtrans_col],
                self.wl_back_from_topo_vac(self.molecules[_molecule]['mlambda']),
                _atlas_wlR,
            )
            if have_convolution :
                # Scale and convolve the transmission...
                # rough scaling by rel_col...
                _atlas_trans=1.-self.molecules[_molecule]['relcol_value']*(1.-_atlas_trans)

                # convolve with response functions...
                ## Box car...
                if True or self.params_by_name[(self.recipe_name,'FIT_RES_BOX')].paramChangeCtrl.GetValue() :
                    bp=self.params_by_name[(self.recipe_name,'RES_BOX')].paramChangeCtrl.GetValue()
                    if bp != '' :
                        bp=float(bp)
                        if bp > 0. :
                            box_kernel = Box1DKernel(
                                width=bp*data_resolution/_atlas_resolution
                            )
                            _atlas_trans=convolve(_atlas_trans, box_kernel,)
                ## Gaussian...
                if True or self.params_by_name[(self.recipe_name,'FIT_RES_GAUSS')].paramChangeCtrl.GetValue() :
                    gp=self.params_by_name[(self.recipe_name,'RES_GAUSS')].paramChangeCtrl.GetValue()
                    if gp != '' :
                        gp=float(gp)
                        if gp > 0. :
                            gauss_kernel = Gaussian1DKernel(
                                stddev=gp/2.*data_resolution/_atlas_resolution
                            )
                            _atlas_trans=convolve(_atlas_trans, gauss_kernel,)
                ## Lorentz...
                if True or self.params_by_name[(self.recipe_name,'FIT_RES_LORENTZ')].paramChangeCtrl.GetValue() :
                    lp=self.params_by_name[(self.recipe_name,'RES_LORENTZ')].paramChangeCtrl.GetValue()
                    if lp != '' :
                        lp=float(lp)
                        if lp > 0. :
                            lorentz_fwhm=lp*data_resolution/_atlas_resolution
                            lorentz=Lorentz1D(
                                amplitude=1.0,
                                x_0=0.,
                                fwhm=lorentz_fwhm,
                            )
                            lorentz_kernel = Model1DKernel(
                                model=lorentz,
                                x_size=int(lorentz_fwhm)*2+1,
                            )
                            _atlas_trans=convolve(_atlas_trans, lorentz_kernel,)            
            atlas_trans+=list(_atlas_trans*(ymax-ymin)*.95+ymin)

        self.molecules[_molecule]['line'].set_data(
            atlas_wls, atlas_trans,
        )
    # ---------------------------------------------------------------------------------------
    def replot_molecules( self, m_list=None ):
        for _molecule in (m_list or self.molecules.get('all_names',[])) :
            if 'display' in self.molecules[_molecule].keys() :
                if self.molecules[_molecule]['display'].GetValue():
                    self.plot_molecule(_molecule)
                    self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def toggle_tell_corr_plot(self, event):
        if event.EventObject.IsChecked() :
            self.telluric_corr['line'].set_data(
                self.wl_back_from_topo_vac(self.telluric_corr['lambda']),
                self.telluric_corr['cflux']/self.orig_su_spectrum['gcf'][self.extname],
            )
        else :
            self.telluric_corr['line'].set_data([], [])
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def toggle_tell_trans_plot_lambda(self, event):
        '''
        Toggle on/off the display of the WL corrected spectrum
        '''
        if event.EventObject.IsChecked() :
            ymin,ymax = self.spec_plot_axes['axes'].get_ylim()
            if ymin < 0. : ymin=0.
            self.telluric_corr['mline'].set_data(
                self.wl_back_from_topo_vac(self.telluric_corr['lambda']),
                self.telluric_corr['mtrans']*(ymax-ymin)*.95+ymin,
            )
        else :
            self.telluric_corr['mline'].set_data([], [])
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def toggle_tell_trans_plot_mlambda(self, event):
        '''
        Toggle on/off the display of the NON-WL corrected spectrum
        '''
        if event.EventObject.IsChecked() :
            ymin,ymax = self.spec_plot_axes['axes'].get_ylim()
            if ymin < 0. : ymin=0.
            self.telluric_corr['mmline'].set_data(
                self.wl_back_from_topo_vac(self.telluric_corr['mlambda']),
                self.telluric_corr['mtrans']*(ymax-ymin)*.95+ymin,
            )
        else :
            self.telluric_corr['mmline'].set_data([], [])
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    def toggle_continuum_fit_plot(self, event):
        if event.EventObject.IsChecked() :
            self.continuum_fits['mline'].set_data(
                self.continuum_fits['wl'],
                self.continuum_fits['continuum_fit'],
            )
        else :
            self.continuum_fits['mline'].set_data([], [])
        self.spec_plot_axes['axes_axes'].figure.canvas.draw_idle()
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    # ---------------------------------------------------------------------------------------
    def sugcf_toggle(self, event):
        if event.EventObject.IsChecked() :
            self.wxApp.sugcf_apply_toggle.Enable(True)
            self.sugcf_model_fit()
        else :
            self.orig_su_spectrum['line'].set_data([], [])
            self.wxApp.sugcf_apply_toggle.Enable(False)
            self.sugcf_apply=False
            self.wxApp.sugcf_apply_toggle.SetValue(False)
            self.sugcf_apply_fit()
    # ---------------------------------------------------------------------------------------
    def sugcf_model_fit(self):
        self.orig_su_spectrum['trial_gcf']={}
        self.orig_su_spectrum['trial_gcf']['merged']=[]
        #for k in self.orig_data_extnames :
        for k in self.orig_data_extnames :
            with warnings.catch_warnings():  # Ignore warnings
                warnings.simplefilter('ignore')
                #regions = [
                #    (0.54 * u.um, 0.7516 * u.um),
                #    (0.7749 * u.um, 1.0167 * u.um)
                #]
                fitted_continuum = fit_continuum(
                    self.orig_su_spectrum['data'][k],
                    model=Chebyshev1D(self.sugcf_mode_degree),
                    #window=regions,
                )
            self.orig_su_spectrum['trial_gcf'][k]=list(
                fitted_continuum(self.orig_su_spectrum['data'][k].spectral_axis).value
            )
            self.orig_su_spectrum['trial_gcf']['merged']+=self.orig_su_spectrum['trial_gcf'][k]
            self.orig_su_spectrum['trial_gcf'][k]=np.array(self.orig_su_spectrum['trial_gcf'][k])
        self.orig_su_spectrum['trial_gcf']['merged']=np.array(self.orig_su_spectrum['trial_gcf']['merged'])
        self.sugcf_apply_fit()
    # ---------------------------------------------------------------------------------------
    def sugcf_model_degree_changed(self, event):
        self.wxApp.sugcf_model_degree.SetBackgroundColour(wx.Colour(255, 0, 0))
    # ---------------------------------------------------------------------------------------
    def sugcf_model_degree_saved(self, event):
        try :
            self.sugcf_mode_degree = (
                int(self.wxApp.sugcf_model_degree.GetValue())
            )
            self.wxApp.sugcf_model_degree.SetBackgroundColour(wx.Colour(255, 255, 255))
            if self.wxApp.sugcf_toggle.IsChecked() :
                self.sugcf_model_fit()
        except :
            pass
    # ---------------------------------------------------------------------------------------
    def sugcf_apply_toggle(self, event):
        self.sugcf_apply=event.EventObject.IsChecked()
        self.sugcf_apply_fit()
    # ---------------------------------------------------------------------------------------
    def sugcf_apply_fit(self):
        if self.sugcf_apply and self.wxApp.sugcf_toggle.IsChecked() :
            self.orig_su_spectrum['gcf'][self.extname]=self.orig_su_spectrum['trial_gcf'][self.extname]
            self.spec_plot.set_ylim([self.residuals_min_max[0],1.4])
            self.orig_su_spectrum['line'].set_data([], [])
        else :
            self.orig_su_spectrum['gcf'][self.extname]=np.ones(len(self.orig_data_flux[self.extname]))
            self.spec_plot.set_ylim(getattr(self,'residuals_min_max',))
            if self.wxApp.sugcf_toggle.IsChecked() :
                self.orig_su_spectrum['line'].set_data(
                    self.orig_data_lambda[self.extname],
                    self.orig_su_spectrum['trial_gcf'][self.extname],
                )
        self.spec_plot_axes['input_spectrum'].set_data(
            self.orig_data_lambda[self.extname],
            self.orig_data_flux[self.extname]/self.orig_su_spectrum['gcf'][self.extname]/self.y_scalefactor,
        )
        if 'residuals' in self.spec_plot_axes :
            self.spec_plot_axes['residuals'].set_data(
                self.wl_back_from_topo_vac(self.data_lambda[self.extname]),
                self.residuals/self.orig_su_spectrum['gcf'][self.extname]/self.y_scalefactor,
            )
            self.spec_plot_axes['model'].set_data(
                self.wl_back_from_topo_vac(self.data_lambda[self.extname]),
                self.data_mflux[self.extname]/self.orig_su_spectrum['gcf'][self.extname]/self.y_scalefactor,
            )
        if hasattr(self,'continuum_fits') :
            self.compute_contimuum_fits()
        self.draw_all_fill_regions(self.extname)
# ---------------------------------------------------------------------------------------
    def compute_contimuum_fits(self) :
        self.continuum_fits['wl']=[]
        self.continuum_fits['continuum_fit']=[]
        #
        for i,extname in enumerate(self.orig_data_extnames) :
            chip=i+1
            if chip in self.best_fit_parameters['continuum_fit_coeffs'] :
                ordered_wil=np.argsort(self.wave_incl_lower)
                wir=[]
                for wili in ordered_wil :
                    if (
                        (self.wave_incl_lower[wili] < np.max(self.orig_data_lambda[extname]))
                        and
                        (self.wave_incl_upper[wili] > np.min(self.orig_data_lambda[extname]))
                    ) :
                        wir+=[[self.wave_incl_lower[wili],self.wave_incl_upper[wili]],]
                wir=np.array([self.wave_incl_lower,self.wave_incl_upper]).transpose()
                if len(wir) > 0 :
                    _ranges=list(self.best_fit_parameters['continuum_fit_coeffs'][chip].keys())
                    _ranges.sort()
                    i=0
                    for _range in _ranges :
                        _range_wl=wlR(
                            self.orig_data_lambda[extname],
                            self.orig_data_lambda[extname],
                            wir[_range-1]
                        )
                        _p_coeffs=[x[0] for x in self.best_fit_parameters['continuum_fit_coeffs'][chip][_range]]
                        _p=Polynomial(coef=_p_coeffs,)
                        self.continuum_fits['wl']+=[_range_wl[0],]+list(_range_wl)+[_range_wl[-1],]
                        _wmean=(np.nanmin(_range_wl)+np.nanmax(_range_wl))/2.
                        _range_cf_wl=(_range_wl-_wmean)
                        _e=extname
                        if 'merged' in self.orig_data_lambda.keys() :
                            _e='merged'
                        _range_cf=_p(_range_cf_wl)/wlR(
                            self.orig_su_spectrum['gcf'][_e],
                            self.orig_data_lambda[_e],
                            wir[_range-1]
                        )
                        self.continuum_fits['continuum_fit']+=[0.,]+list(_range_cf)+[0.,]
                        i+=1

# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------
def exec_main(interactive_app) :

    # get inputs from the command line
    #interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)


    #Open the interactive window if enabled
    interactive_app.pre_recipe_exec=False
    #Get the specific functions for this window
    dataPlotManager = DataPlotterManager()
    dataPlotManager.recipe_name=interactive_app.inputs.in_sop[0].recipe

    # Check if this is being run as pre-MolecfitModel GUI...
    dataPlotManager.pse_is_set=False
    dataPlotManager.orig_init_sop=init_sop_dict(interactive_app.inputs.in_sop)
    #if interactive_app.inputs.in_sof_rec_orig == interactive_app.inputs.in_sof :
    if not isinstance(interactive_app.inputs.in_sof_rec_orig, reflex.SetOfFiles) :
        # This is the case the a SOP was passed as in_sof_rec_orig
        # And is used by the pre_recipe_exec version to identify this
        # as pre_recipe_exec
        # In this case:
        # -- in_sop comes from RecipeLooper and is possibly the values from the
        #    last succesful execution
        # -- in_sof_rec_orig = in_sop_check_IDP
        interactive_app.pre_recipe_exec=True
        interactive_app.inputs.enable='true'
        dataPlotManager.pse_is_set=interactive_app.inputs.in_sof_rec_orig != interactive_app.inputs.in_sop
        dataPlotManager.pse_init_sop=init_sop_dict(interactive_app.inputs.in_sof_rec_orig)
        interactive_app.inputs.in_sof_rec_orig=interactive_app.inputs.in_sof

    dataPlotManager.pre_recipe_exec=interactive_app.pre_recipe_exec
    interactive_app.setPlotManager(dataPlotManager)
    dataPlotManager.read_molecules(interactive_app.inputs.in_sof_rec_orig.files)
    dataPlotManager.read_best_fit_parameters(interactive_app.inputs.in_sof.files)
    interactive_app.book_keeping_dir=os.path.dirname(
        os.path.dirname(
            os.path.dirname(
                interactive_app.values.input_parameters_file
            )
        )
    )
    ## --------------------------------------------------------------------
    ## copied from reflex_interactive_gui.shwGUI()
    try:
        #from reflex_interactive_gui import ReflexInteractiveWxApp
        interactive_app.wxApp = ReflexInteractiveWxApp(
            interactive_app,
            dataPlotManager,
        )

        dataPlotManager.wxApp=interactive_app.wxApp
        # get the 'recipe' parameters into a usable format...
        dataPlotManager.params_by_name = dict(
            [
                ((x[1].parameter.recipe, x[1].parameter.name), x[1])
                for x in interactive_app.wxApp.shownParamWidgets
            ]
        )
    except (ImportError, NameError) as e:
        print("Error importing modules: ", str(e))
        raise

    ## --------------------------------------------------------------------
    # Adjust the WAVE_INCLUDE/EXCLUDE values...
    '''
    if hasattr(interactive_app,'inputs') :
        for sop_name in [
            'in_sop',
        ] :

            if hasattr(interactive_app.inputs,sop_name) :
                sop = getattr(interactive_app.inputs,sop_name)
                sop_params_by_name = dict(
                    [
                        ((x.recipe, x.name), x)
                        for x in sop
                    ]
                )
                for p_name in [
                    'WAVE_INCLUDE',
                    'WAVE_EXCLUDE',
                ] :
                    if sop_params_by_name[('molecfit_model',p_name)].value.lower() not in [
                        'null',
                    ] :
                        _wl=dataPlotManager.wl_to_topo_vac(
                            np.array(sop_params_by_name[('molecfit_model',p_name)].value.split(','), dtype=float)
                        )
                        sop_params_by_name[('molecfit_model',p_name)].value=(
                            ",".join([iu.wrfmt(x) for x in _wl])
                        )
                        """
                        dataPlotManager.set_pcc_Value(
                            p_name,sop_params_by_name[('molecfit_model',p_name)].value,
                        )
                        """
                        pass
    '''
    # Check for saved parameters:
    inst_setup=dataPlotManager.inst_setup
    objectname=dataPlotManager.objectname
    saved_init_sop=None
    rp_list=None
    if inst_setup is not None :
        if objectname is not None :
            p='%s/save_settings/by_target/%s/%s.json' %(interactive_app.book_keeping_dir,objectname,inst_setup)
            if os.path.exists(p) :
                with open(p, 'r') as f:
                    saved_init_sop=json.load(f)
                saved_init_sop['type']='object'
        p='%s/save_settings/by_setup/%s.json' %(interactive_app.book_keeping_dir,inst_setup)
        if saved_init_sop is None and os.path.exists(p) :
            with open(p, 'r') as f:
                saved_init_sop=json.load(f)
            saved_init_sop['type']='setup'
    if saved_init_sop is not None :
        rp_list=[]
        for i,p in enumerate(saved_init_sop.get('init_sop',[])) :
            rp_list+=[
                reflex.RecipeParameter(
                    recipe=interactive_app.inputs.in_sop[0].recipe,
                    displayName=p.get('displayName'),
                    name=p['name'],
                    group=p['group'],
                    description=p['description'],
                    value=p['value'],
                    valtype=p.get('valtype'),
                    partype=p.get('partype'),
                    valmin=p.get('valmin'),
                    valmax=p.get('valmax'),
                    valenum=p.get('valenum'),
                )
            ]
    dataPlotManager.saved_init_sop=init_sop_dict(rp_list, existing_dict=saved_init_sop)

    if interactive_app.isGUIEnabled() :
        # patched in esoreflex... but want bigger boxes that the default 12 chars...
        for p in dataPlotManager.params_by_name.keys() :
            w=dataPlotManager.params_by_name[p].paramChangeCtrl
            if (
                w.ClassName
                in
                ['wxTextCtrl',]
            ) :
                self_size=w.Size
                font_size=w.Font.PixelSize
                w.SetMaxSize(wx.Size(font_size.x*20, self_size.y))
                w.SetSize(w.GetMaxSize())
                w.SetInitialSize(w.GetMaxSize())
            pass

        #if dataPlotManager.input_errors == 0 :
        #    dataPlotManager.draw_all_fill_regions(dataPlotManager.extname)
        dataPlotManager.title_text=dataPlotManager.spec_plot.title._text

        dataPlotManager.wxApp_elements_by_name={}
        w_list=dataPlotManager.get_window_list(interactive_app.wxApp)
        w_list.reverse()
        for e in w_list :
            if e[0] not in dataPlotManager.wxApp_elements_by_name.keys() :
                dataPlotManager.wxApp_elements_by_name[e[0]]=[]
            dataPlotManager.wxApp_elements_by_name[e[0]]+=[e[1],]

        # We disable the Continue WKF button in the case that molecfit_model failed...
        if dataPlotManager.disable_continue_wkf :
            dataPlotManager.wxApp_elements_by_name['contBtn'][0].Enable(False)

        ## modify the text of the Use the parameters.. button...
        sis_SETTING_cb=dataPlotManager.wxApp_elements_by_name['setInitSopCheck'][0]
        sis_SETTING_cb.Label="""\
Use the parameters above
as initial values in subsequent
executions of this recipe
for this SETTING = '%s'\
""" %(inst_setup)
        interactive_app.wxApp.Bind(
            wx.EVT_CHECKBOX,
            dataPlotManager.toggle_sis,
            sis_SETTING_cb,
        )

        if dataPlotManager.objectname is not None :
            ## Use these parameters for this OBJECT and SETTING...
            interactive_app.wxApp.sis_OBJECT_and_SETTING_cb = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label="""\
Use the parameters above
as initial values in subsequent
executions of this recipe
for this OBJECT = '%s'
and SETTING = '%s'\
""" %(objectname, inst_setup),
                style=wx.TAB_TRAVERSAL,
                name="setInitSopCheck_OBJECT_and_SETTING",
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(interactive_app.wxApp.sis_OBJECT_and_SETTING_cb, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
            interactive_app.wxApp.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.toggle_sis,
                interactive_app.wxApp.sis_OBJECT_and_SETTING_cb,
            )

        default_or_input="Default"
        if not interactive_app.pre_recipe_exec :
            ## BEST FIT PARAMETERS...
            vertical_container_BFP = wx.BoxSizer(wx.VERTICAL)
            # Apply the Best Fit Parameters...
            interactive_app.wxApp.ApplyBestFitParameters_button = wx.Button(
                interactive_app.wxApp.ctrlPanel,
                label="Apply Best Fit Parameters",
                style=wx.TAB_TRAVERSAL,
            )
            horizontal_sizer_BFP_button = wx.BoxSizer(wx.HORIZONTAL)
            horizontal_sizer_BFP_button.Add(interactive_app.wxApp.ApplyBestFitParameters_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
            vertical_container_BFP.Add(horizontal_sizer_BFP_button, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
            interactive_app.wxApp.Bind(
                wx.EVT_BUTTON,
                dataPlotManager.clicked_ApplyBestFitParameters_button,
                interactive_app.wxApp.ApplyBestFitParameters_button,
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_BFP, 0, wx.EXPAND | wx.CENTER, 10)
            interactive_app.wxApp.ctrlPanel.FitInside()
            default_or_input="Input"

        # Apply the Previous Successful Execution [PSE] Parameters button...
        if dataPlotManager.pse_is_set :
            vertical_container_PSEP = wx.BoxSizer(wx.VERTICAL)
            interactive_app.wxApp.ApplyPSEParameters_button = wx.Button(
                interactive_app.wxApp.ctrlPanel,
                label="Apply prev. successful exec. Parameters",
                style=wx.TAB_TRAVERSAL,
            )
            horizontal_sizer_PSEP_button = wx.BoxSizer(wx.HORIZONTAL)
            horizontal_sizer_PSEP_button.Add(interactive_app.wxApp.ApplyPSEParameters_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
            vertical_container_PSEP.Add(horizontal_sizer_PSEP_button, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
            interactive_app.wxApp.Bind(
                wx.EVT_BUTTON,
                dataPlotManager.clicked_ApplyPSEParameters_button,
                interactive_app.wxApp.ApplyPSEParameters_button,
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_PSEP, 0, wx.EXPAND | wx.CENTER, 10)
            interactive_app.wxApp.ctrlPanel.FitInside()

        # Apply the Original = Default|Input Parameters button...
        vertical_container_OP = wx.BoxSizer(wx.VERTICAL)
        interactive_app.wxApp.ApplyOriginalParameters_button = wx.Button(
            interactive_app.wxApp.ctrlPanel,
            label="Apply %s Parameters" %(default_or_input),
            style=wx.TAB_TRAVERSAL,
        )
        horizontal_sizer_OP_button = wx.BoxSizer(wx.HORIZONTAL)
        horizontal_sizer_OP_button.Add(interactive_app.wxApp.ApplyOriginalParameters_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        vertical_container_OP.Add(horizontal_sizer_OP_button, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
        interactive_app.wxApp.Bind(
            wx.EVT_BUTTON,
            dataPlotManager.clicked_ApplyOriginalParameters_button,
            interactive_app.wxApp.ApplyOriginalParameters_button,
        )
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_OP, 0, wx.EXPAND | wx.CENTER, 10)
        interactive_app.wxApp.ctrlPanel.FitInside()

        if dataPlotManager.saved_init_sop is not None :
            # Apply the Saved Parameters button...
            sp_str=dataPlotManager.saved_init_sop.get(
                'inst_setup',
                'UNKNOWN'
            )
            if dataPlotManager.saved_init_sop.get('object') is not None :
                sp_str="%s/%s" %(
                    dataPlotManager.saved_init_sop.get(
                        'object',
                        'UNKNOWN'
                    ),
                    dataPlotManager.saved_init_sop.get(
                        'inst_setup',
                        'UNKNOWN'
                    ),
                )
            vertical_container_SP = wx.BoxSizer(wx.VERTICAL)
            interactive_app.wxApp.ApplySavedParameters_button = wx.Button(
                interactive_app.wxApp.ctrlPanel,
                label="Apply saved parameters for %s" %(sp_str),
                style=wx.TAB_TRAVERSAL,
            )
            sis_type=dataPlotManager.saved_init_sop.get('type')
            if sis_type == 'object' :
                sis_tooltip="""\
Saved parameters for
OBJECT = '%s'
SETUP = %s
from dataset %s @ %s""" %(
                    dataPlotManager.saved_init_sop.get('object','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('inst_setup','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('data_set','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('datetime','UNKNOWN'),
                )
            elif sis_type == 'setup' :
                sis_tooltip="""\
Saved parameters for
SETUP = %s
from dataset %s @ %s""" %(
                    dataPlotManager.saved_init_sop.get('inst_setup','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('data_set','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('datetime','UNKNOWN'),
                )
            else :
                sis_tooltip="""Saved parameters from dataset %s @ %s""" %(
                    dataPlotManager.saved_init_sop.get('data_set','UNKNOWN'),
                    dataPlotManager.saved_init_sop.get('datetime','UNKNOWN'),
                )
            try:
                interactive_app.wxApp.ApplySavedParameters_button.SetToolTip(sis_tooltip)
            except:
                pass

            horizontal_sizer_SP_button = wx.BoxSizer(wx.HORIZONTAL)
            horizontal_sizer_SP_button.Add(interactive_app.wxApp.ApplySavedParameters_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
            vertical_container_SP.Add(horizontal_sizer_SP_button, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
            interactive_app.wxApp.Bind(
                wx.EVT_BUTTON,
                dataPlotManager.clicked_ApplySavedParameters_button,
                interactive_app.wxApp.ApplySavedParameters_button,
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_SP, 0, wx.EXPAND | wx.CENTER, 10)
            interactive_app.wxApp.ctrlPanel.FitInside()

            # master switch
            interactive_app.wxApp.apply_saved_dfd_params = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Apply saved Data-Format-Dependent parameters',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.apply_saved_dfd_params.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.clicked_apply_saved_dfd_params,
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(
                interactive_app.wxApp.apply_saved_dfd_params, 0, wx.EXPAND | wx.CENTRE, 10
            )

        # Add/Delete/Modify Include region...
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(
            wx.StaticText(
                interactive_app.wxApp.ctrlPanel,
                label="Include regions"
            ),
            0, wx.EXPAND | wx.CENTER, 10
        )
        vertical_container_IncludeRegions = wx.BoxSizer(wx.VERTICAL)
        horizontal_sizer_IncReg_buttons = wx.BoxSizer(wx.HORIZONTAL)
        # Add
        interactive_app.wxApp.AddIncludeRegion_button = wx.Button(
            interactive_app.wxApp.ctrlPanel,
            label="Add",
            style=wx.TAB_TRAVERSAL,
        )
        dataPlotManager.add_buttons["WAVE_INCLUDE"]=interactive_app.wxApp.AddIncludeRegion_button
        interactive_app.wxApp.Bind(
            wx.EVT_BUTTON,
            dataPlotManager.AddIncludeRegion,
            interactive_app.wxApp.AddIncludeRegion_button,
        )
        horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.AddIncludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        # Delete
        interactive_app.wxApp.DeleteIncludeRegion_button = wx.Button(
            interactive_app.wxApp.ctrlPanel,
            label="Delete",
            style=wx.TAB_TRAVERSAL,
        )
        dataPlotManager.delete_buttons["WAVE_INCLUDE"]=interactive_app.wxApp.DeleteIncludeRegion_button
        dataPlotManager.setEnabledDeleteButton("WAVE_INCLUDE")
        interactive_app.wxApp.Bind(
            wx.EVT_BUTTON,
            dataPlotManager.DeleteIncludeRegion,
            interactive_app.wxApp.DeleteIncludeRegion_button,
        )
        horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.DeleteIncludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        # Modify
        if not dataPlotManager.span.is_old :
            interactive_app.wxApp.ModifyIncludeRegion_button = wx.Button(
                interactive_app.wxApp.ctrlPanel,
                label="Modify",
                style=wx.TAB_TRAVERSAL,
            )
            dataPlotManager.modify_buttons["WAVE_INCLUDE"]=interactive_app.wxApp.ModifyIncludeRegion_button
            interactive_app.wxApp.Bind(
                wx.EVT_BUTTON,
                dataPlotManager.ModifyIncludeRegion,
                interactive_app.wxApp.ModifyIncludeRegion_button,
            )
            horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.ModifyIncludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)

        vertical_container_IncludeRegions.Add(horizontal_sizer_IncReg_buttons, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_IncludeRegions, 0, wx.EXPAND | wx.CENTER, 10)
        interactive_app.wxApp.ctrlPanel.FitInside()

        # Add/Delete/Modify Exclude region...
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(
            wx.StaticText(
                interactive_app.wxApp.ctrlPanel,
                label="Exclude regions"
            ),
            0, wx.EXPAND | wx.CENTER, 10
        )
        vertical_container_ExcludeRegions = wx.BoxSizer(wx.VERTICAL)
        horizontal_sizer_IncReg_buttons = wx.BoxSizer(wx.HORIZONTAL)
        # Add
        interactive_app.wxApp.AddExcludeRegion_button = wx.Button(
            interactive_app.wxApp.ctrlPanel,
            label="Add",
            style=wx.TAB_TRAVERSAL,
        )
        dataPlotManager.add_buttons["WAVE_EXCLUDE"]=interactive_app.wxApp.AddExcludeRegion_button
        interactive_app.wxApp.Bind(
            wx.EVT_BUTTON,
            dataPlotManager.AddExcludeRegion,
            interactive_app.wxApp.AddExcludeRegion_button,
        )
        horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.AddExcludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        # Delete
        interactive_app.wxApp.DeleteExcludeRegion_button = wx.Button(
            interactive_app.wxApp.ctrlPanel,
            label="Delete",
            style=wx.TAB_TRAVERSAL,
        )
        dataPlotManager.delete_buttons["WAVE_EXCLUDE"]=interactive_app.wxApp.DeleteExcludeRegion_button
        dataPlotManager.setEnabledDeleteButton("WAVE_EXCLUDE")
        interactive_app.wxApp.Bind(
            wx.EVT_BUTTON,
            dataPlotManager.DeleteExcludeRegion,
            interactive_app.wxApp.DeleteExcludeRegion_button,
        )
        horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.DeleteExcludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        # Modify
        if not dataPlotManager.span.is_old :
            interactive_app.wxApp.ModifyExcludeRegion_button = wx.Button(
                interactive_app.wxApp.ctrlPanel,
                label="Modify",
                style=wx.TAB_TRAVERSAL,
            )
            dataPlotManager.modify_buttons["WAVE_EXCLUDE"]=interactive_app.wxApp.ModifyExcludeRegion_button
            interactive_app.wxApp.Bind(
                wx.EVT_BUTTON,
                dataPlotManager.ModifyExcludeRegion,
                interactive_app.wxApp.ModifyExcludeRegion_button,
            )
            horizontal_sizer_IncReg_buttons.Add(interactive_app.wxApp.ModifyExcludeRegion_button, 0, wx.ALL | wx.ALIGN_CENTER, 10)

        vertical_container_ExcludeRegions.Add(horizontal_sizer_IncReg_buttons, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_ExcludeRegions, 0, wx.EXPAND | wx.CENTER, 10)
        interactive_app.wxApp.ctrlPanel.FitInside()

        '''
        ## Fibre display (eventually for moeclfit_correct...)
        vertical_container_fib_disp = wx.BoxSizer(wx.VERTICAL)
        #vertical_container_fib_disp.Add(self.title)

        interactive_app.wxApp.fib_disp_choice = wx.Choice(
            interactive_app.wxApp.ctrlPanel,
            choices=['fib_1','fib_2'],
        )
        horizontal_sizer_fib_disp = wx.BoxSizer(wx.HORIZONTAL)
        horizontal_sizer_fib_disp.Add(interactive_app.wxApp.fib_disp_choice, 0, wx.ALL | wx.ALIGN_CENTER, 10)
        #horizontal_sizer_fib_disp.Add(interactive_app.wxApp.help_msg, 1, wx.EXPAND)
        vertical_container_fib_disp.Add(horizontal_sizer_fib_disp, 0, wx.ALL | wx.ALIGN_CENTER_HORIZONTAL )
        interactive_app.wxApp.Bind(
            wx.EVT_CHOICE,
            fib_disp_choice,
            interactive_app.wxApp.fib_disp_choice,
        )
        interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(vertical_container_fib_disp, 0, wx.EXPAND | wx.CENTER, 10)
        interactive_app.wxApp.ctrlPanel.FitInside()
        '''
        # make a human readable list of windows...
        dataPlotManager.wxApp_elements_by_name={}
        w_list=dataPlotManager.get_window_list(interactive_app.wxApp)
        w_list.reverse()
        for e in w_list :
            if e[0] not in dataPlotManager.wxApp_elements_by_name.keys() :
                dataPlotManager.wxApp_elements_by_name[e[0]]=[]
            dataPlotManager.wxApp_elements_by_name[e[0]]+=[e[1],]

        if dataPlotManager.have_telluric_corr :
            vertical_container_tel_corr = wx.BoxSizer(wx.VERTICAL)
            # calctrans label
            interactive_app.wxApp.telluric_corr_label = wx.StaticText(
                interactive_app.wxApp.ctrlPanel,
                label="Calctrans Models"
            )
            vertical_container_tel_corr.Add(
                interactive_app.wxApp.telluric_corr_label, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # calctrans corrected spectrum
            interactive_app.wxApp.telluric_corr_plot = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Overplot calctrans corrected spectrum',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.telluric_corr_plot.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.toggle_tell_corr_plot,
            )
            vertical_container_tel_corr.Add(
                interactive_app.wxApp.telluric_corr_plot, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # calctrans transmission [lambda]
            interactive_app.wxApp.telluric_corr_plot = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Overplot calctrans transmission [lambda] (includes WLC)',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.telluric_corr_plot.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.toggle_tell_trans_plot_lambda,
            )
            vertical_container_tel_corr.Add(
                interactive_app.wxApp.telluric_corr_plot, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # calctrans transmission [mlambda]
            interactive_app.wxApp.telluric_corr_plot = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Overplot calctrans transmission [mlambda] (no WLC)',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.telluric_corr_plot.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.toggle_tell_trans_plot_mlambda,
            )
            vertical_container_tel_corr.Add(
                interactive_app.wxApp.telluric_corr_plot, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # continuum fits
            interactive_app.wxApp.continuum_fit_plot = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Overplot continuum fits',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.continuum_fit_plot.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.toggle_continuum_fit_plot,
            )
            vertical_container_tel_corr.Add(
                interactive_app.wxApp.continuum_fit_plot, 0, wx.EXPAND | wx.CENTRE, 10
            )
            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(
                vertical_container_tel_corr, 0, wx.EXPAND | wx.CENTER, 10
            )
            interactive_app.wxApp.ctrlPanel.FitInside()

        #if have_specutils :
        if have_specutils and enable_gcf :
            # Use specutils to do a global continuum fit...
            vertical_container_sugcf = wx.BoxSizer(wx.VERTICAL)
            # specutils label
            interactive_app.wxApp.telluric_corr_label = wx.StaticText(
                interactive_app.wxApp.ctrlPanel,
                label="Global continuum fit [specutils]"
            )
            vertical_container_sugcf.Add(
                interactive_app.wxApp.telluric_corr_label, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # master switch
            interactive_app.wxApp.sugcf_toggle = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Enable specutils Global Continuum fit',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.sugcf_toggle.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.sugcf_toggle,
            )
            vertical_container_sugcf.Add(
                interactive_app.wxApp.sugcf_toggle, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # model degree
            horizontal_container_sugcf_degree = wx.BoxSizer(wx.HORIZONTAL)
            # specutils label
            interactive_app.wxApp.sugcf_degree_label = wx.StaticText(
                interactive_app.wxApp.ctrlPanel,
                label="Model degree"
            )
            horizontal_container_sugcf_degree.Add(
                interactive_app.wxApp.sugcf_degree_label, 0, wx.EXPAND | wx.CENTRE, 10
            )
            interactive_app.wxApp.sugcf_model_degree = wx.TextCtrl(
                interactive_app.wxApp.ctrlPanel,
                -1,
                #validator=IntValidator(),
                style=wx.TE_PROCESS_ENTER,
            )
            interactive_app.wxApp.sugcf_model_degree.ChangeValue( str(dataPlotManager.sugcf_mode_degree) )
            # get the default sizes...
            self_size=interactive_app.wxApp.sugcf_model_degree.Size
            font_size=interactive_app.wxApp.sugcf_model_degree.Font.PixelSize
            interactive_app.wxApp.sugcf_model_degree.SetMaxSize(wx.Size(font_size.width*12, self_size.height))
            interactive_app.wxApp.sugcf_model_degree.SetSize(interactive_app.wxApp.sugcf_model_degree.GetMaxSize())
            interactive_app.wxApp.sugcf_model_degree.SetInitialSize(interactive_app.wxApp.sugcf_model_degree.GetMaxSize())

            interactive_app.wxApp.sugcf_model_degree.Bind(
                wx.EVT_TEXT,
                dataPlotManager.sugcf_model_degree_changed,
                interactive_app.wxApp.sugcf_model_degree,
            )
            interactive_app.wxApp.sugcf_model_degree.Bind(
                wx.EVT_TEXT_ENTER,
                dataPlotManager.sugcf_model_degree_saved,
                interactive_app.wxApp.sugcf_model_degree,
            )
            horizontal_container_sugcf_degree.Add(
                interactive_app.wxApp.sugcf_model_degree, 0, wx.EXPAND | wx.CENTRE, 10
            )
            vertical_container_sugcf.Add(
                horizontal_container_sugcf_degree, 0, wx.EXPAND | wx.CENTRE, 10
            )
            # apply gcf
            interactive_app.wxApp.sugcf_apply_toggle = wx.CheckBox(
                interactive_app.wxApp.ctrlPanel,
                label='Apply the specutils Global Continuum fit',
                style=wx.TAB_TRAVERSAL,
            )
            interactive_app.wxApp.sugcf_apply_toggle.Bind(
                wx.EVT_CHECKBOX,
                dataPlotManager.sugcf_apply_toggle,
            )
            interactive_app.wxApp.sugcf_apply_toggle.Enable(False)
            vertical_container_sugcf.Add(
                interactive_app.wxApp.sugcf_apply_toggle, 0, wx.EXPAND | wx.CENTRE, 10
            )

            interactive_app.wxApp.setDisableCheck.ContainingSizer.Add(
                vertical_container_sugcf, 0, wx.EXPAND | wx.CENTER, 10
            )
            interactive_app.wxApp.ctrlPanel.FitInside()

        '''
        nb=dataPlotManager.wxApp_elements_by_name['parameterNotebook'][0]
        m_tab = dataPlotManager.molecules_tab(nb)
        nb.InsertPage(0,m_tab,"Molecules",select=True)
        '''
        # Bind change events...
        bind_changed_events={
            'LIST_MOLEC': dataPlotManager.LIST_MOLEC_changed_event,
            'REL_COL' : dataPlotManager.REL_COL_changed_event,
            'FIT_MOLEC': dataPlotManager.FIT_MOLEC_changed_event,
            'FIT_CONTINUUM': dataPlotManager.FIT_CONTINUUM_changed_event,
            'CONTINUUM_N': dataPlotManager.CONTINUUM_N_changed_event,
            'FIT_WLC': dataPlotManager.FIT_WLC_changed_event,
            'WLC_CONST': dataPlotManager.WLC_CONST_changed_event,
            'FIT_RES_BOX': dataPlotManager.FIT_RES_BOX_changed_event,
            'RES_BOX': dataPlotManager.RES_BOX_changed_event,
            'FIT_RES_GAUSS': dataPlotManager.FIT_RES_GAUSS_changed_event,
            'RES_GAUSS': dataPlotManager.RES_GAUSS_changed_event,
            'FIT_RES_LORENTZ': dataPlotManager.FIT_RES_LORENTZ_changed_event,
            'RES_LORENTZ': dataPlotManager.RES_LORENTZ_changed_event,
            'MAP_REGIONS_TO_CHIP': dataPlotManager.MAP_REGIONS_TO_CHIP_changed_event,
        }
        for k in bind_changed_events.keys() :
            if k in dataPlotManager.orig_init_sop['by_name'].keys() :
                if isinstance(
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl,
                    wx.TextCtrl,
                ) :
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl.Bind(
                        wx.EVT_TEXT,
                        bind_changed_events[k],
                    )
                elif isinstance(
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl,
                    wx.CheckBox,
                ) :
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl.Bind(
                        wx.EVT_CHECKBOX,
                        bind_changed_events[k],
                    )
                else :
                    pass

        # Bind save events...
        bind_saved_events={
            'RES_BOX': dataPlotManager.RES_BOX_saved_event,
            'RES_GAUSS': dataPlotManager.RES_GAUSS_saved_event,
            'RES_LORENTZ': dataPlotManager.RES_LORENTZ_saved_event,
        }
        for k in bind_saved_events.keys() :
            if k in dataPlotManager.orig_init_sop['by_name'].keys() :
                if isinstance(
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl,
                    wx.TextCtrl,
                ) :
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl.SetWindowStyle(
                        wx.TE_PROCESS_ENTER,
                    )
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl.Bind(
                        wx.EVT_TEXT_ENTER,
                        bind_saved_events[k],
                    )
                    '''
                    '''

        # Create a Molecules paramGroup
        paramGroup="Molecules"
        interactive_app.wxApp.parameterGrid[paramGroup] = wx.FlexGridSizer(
            cols=5, vgap=1, hgap=4
        )
        interactive_app.wxApp.parameterGrid[paramGroup].AddGrowableCol(4)
        interactive_app.wxApp.parameterTabs[paramGroup] = (
            wx.lib.scrolledpanel.ScrolledPanel(
                interactive_app.wxApp.parameterNotebook
            )
        )
        interactive_app.wxApp.parameterTabs[paramGroup].SetupScrolling()
        interactive_app.wxApp.parameterTabs[paramGroup].SetSizer(
            interactive_app.wxApp.parameterGrid[paramGroup],
        )
        interactive_app.wxApp.parameterNotebook.InsertPage(
            0, interactive_app.wxApp.parameterTabs[paramGroup], paramGroup, select=True
        )
        interactive_app.wxApp.parameterNotebook.Layout()
        interactive_app.wxApp.parameterTabs[paramGroup].Layout()
        interactive_app.wxApp.parameterGrid[paramGroup].Layout()
        # A special line for Continuum Absorption...
        have_molecules=True
        if dataPlotManager.molecules is None :
            dataPlotManager.molecules={}
            have_molecules=False
        p="Cont.Abs."
        interactive_app.wxApp.parameterGrid[paramGroup].Add(
            wx.StaticText(
                interactive_app.wxApp.parameterTabs[paramGroup],
                label=p,
            ),
            flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
        )
        dataPlotManager.molecules['cont_abs']=wx.CheckBox(interactive_app.wxApp.parameterTabs[paramGroup],)
        interactive_app.wxApp.Bind(
            wx.EVT_CHECKBOX,
            dataPlotManager.toggle_continuum_absorption,
            dataPlotManager.molecules['cont_abs'],
        )
        interactive_app.wxApp.parameterGrid[paramGroup].Add(
            dataPlotManager.molecules['cont_abs'],
            flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
        )
        for p in [
            '',
            '',
            '',
            #'',
        ] :
            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                wx.StaticText(
                    interactive_app.wxApp.parameterTabs[paramGroup],
                    label=p,
                ),
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
        # Headings for the molecules...
        for p in [
            'name',
            'include',
            'fit',
            'relcol',
            'display',
            #'',
        ] :
            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                wx.StaticText(
                    interactive_app.wxApp.parameterTabs[paramGroup],
                    label=p,
                ),
                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
            )
        ## Display molecule spectra
        if have_molecules :
            selected_molecules=dataPlotManager.molecules.get(
                "LIST_MOLEC",
                dataPlotManager.params_by_name[(dataPlotManager.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')
            )
            interactive_app.wxApp.molecules={}

            s='selected'
            for _molecule in selected_molecules :
                if _molecule.lower() not in ['null',] :
                    if _molecule in dataPlotManager.molecules["names"] :
                        dataPlotManager.add_molecule(
                            interactive_app.wxApp.parameterTabs[paramGroup],_molecule, s, paramGroup,
                        )
                    else :
                        # This shouldn't happen, but just in case...
                        dataPlotManager.add_molecule(
                            interactive_app.wxApp.parameterTabs[paramGroup],_molecule, 'selected_but_not', paramGroup,
                        )
            s='nonselected'
            for _molecule in dataPlotManager.molecules["names"] :
                if _molecule not in selected_molecules :
                    dataPlotManager.add_molecule(
                        interactive_app.wxApp.parameterTabs[paramGroup], _molecule, s, paramGroup,
                    )
            s='other'
            for _molecule in dataPlotManager.molecules["all_names"] :
                if _molecule not in selected_molecules+dataPlotManager.molecules["names"] :
                    dataPlotManager.add_molecule(
                        interactive_app.wxApp.parameterTabs[paramGroup], _molecule, s, paramGroup,
                    )
        else :
            s='other'
            lm=dataPlotManager.params_by_name[(dataPlotManager.recipe_name,'LIST_MOLEC')].paramChangeCtrl.GetValue().split(',')
            for _molecule in m_specs+trace_m_specs :
                if _molecule in lm :
                    dataPlotManager.add_molecule(interactive_app.wxApp.parameterTabs[paramGroup], _molecule, s, paramGroup, )
            for _molecule in m_specs+trace_m_specs :
                if _molecule not in lm :
                    dataPlotManager.add_molecule(interactive_app.wxApp.parameterTabs[paramGroup], _molecule, s, paramGroup, )

        # Tabulate WL Fitting parameters
        if hasattr(dataPlotManager,'best_fit_parameters') :
            if 'wl_fit_coeffs' in dataPlotManager.best_fit_parameters.keys() :
                if len(dataPlotManager.best_fit_parameters['wl_fit_coeffs']) > 0 :
                    paramGroup="WL Fitting"
                    interactive_app.wxApp.parameterGrid[paramGroup] = wx.FlexGridSizer(
                        cols=4, vgap=1, hgap=4
                    )
                    interactive_app.wxApp.parameterGrid[paramGroup].AddGrowableCol(1)
                    interactive_app.wxApp.parameterTabs[paramGroup] = wx.lib.scrolledpanel.ScrolledPanel(interactive_app.wxApp.parameterNotebook)
                    interactive_app.wxApp.parameterTabs[paramGroup].SetupScrolling()
                    interactive_app.wxApp.parameterTabs[paramGroup].SetSizer(
                        interactive_app.wxApp.parameterGrid[paramGroup], 1
                    )
                    interactive_app.wxApp.parameterNotebook.InsertPage(
                        0, interactive_app.wxApp.parameterTabs[paramGroup], paramGroup, select=False
                    )
                    chips=list(dataPlotManager.best_fit_parameters['wl_fit_coeffs'].keys())
                    chips.sort()
                    for chip in chips :
                        interactive_app.wxApp.parameterGrid[paramGroup].Add(
                            wx.StaticText(
                                interactive_app.wxApp.parameterTabs[paramGroup],
                                label="chip#%d" %(chip),
                            ),
                            flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                        )
                        for p in [
                            '',
                            '',
                            '',
                        ] :
                            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                wx.StaticText(
                                    interactive_app.wxApp.parameterTabs[paramGroup],
                                    label=p,
                                ),
                                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                            )
                        for i,coeff in enumerate(dataPlotManager.best_fit_parameters['wl_fit_coeffs'][chip]) :
                            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                wx.StaticText(
                                    interactive_app.wxApp.parameterTabs[paramGroup],
                                    label=str(i),
                                ),
                                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                            )
                            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                wx.StaticText(
                                    interactive_app.wxApp.parameterTabs[paramGroup],
                                    label=str(coeff[0]),
                                ),
                                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                            )
                            try :
                                interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                    wx.StaticText(
                                        interactive_app.wxApp.parameterTabs[paramGroup],
                                        label="±",
                                    ),
                                    flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                                )
                            except :
                                interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                    wx.StaticText(
                                        interactive_app.wxApp.parameterTabs[paramGroup],
                                        label="±",
                                    ),
                                    flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                                )
                            interactive_app.wxApp.parameterGrid[paramGroup].Add(
                                wx.StaticText(
                                    interactive_app.wxApp.parameterTabs[paramGroup],
                                    label=str(coeff[1]),
                                ),
                                flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                            )

        # Tabulate Fitting Statistics
        if hasattr( dataPlotManager, 'fit_statistics' ) :
            if dataPlotManager.fit_statistics.get('status') is not None :
                paramGroup="Fit statistics"
                interactive_app.wxApp.parameterGrid[paramGroup] = wx.FlexGridSizer(
                    cols=2, vgap=1, hgap=4
                )
                interactive_app.wxApp.parameterGrid[paramGroup].AddGrowableCol(1)
                interactive_app.wxApp.parameterTabs[paramGroup] = wx.lib.scrolledpanel.ScrolledPanel(interactive_app.wxApp.parameterNotebook)
                interactive_app.wxApp.parameterTabs[paramGroup].SetupScrolling()
                interactive_app.wxApp.parameterTabs[paramGroup].SetSizer(
                    interactive_app.wxApp.parameterGrid[paramGroup], 1
                )
                interactive_app.wxApp.parameterNotebook.InsertPage(
                    0, interactive_app.wxApp.parameterTabs[paramGroup], paramGroup, select=True
                )
                for p in dataPlotManager.fit_statistics['index'] :
                    interactive_app.wxApp.parameterGrid[paramGroup].Add(
                        wx.StaticText(
                            interactive_app.wxApp.parameterTabs[paramGroup],
                            label=p,
                        ),
                        flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                    )
                    interactive_app.wxApp.parameterGrid[paramGroup].Add(
                        wx.StaticText(
                            interactive_app.wxApp.parameterTabs[paramGroup],
                            label=str(dataPlotManager.fit_statistics[p]),
                        ),
                        flag=wx.ALL | wx.ALIGN_CENTER_VERTICAL, border=5
                    )
            """
The meaning of the fit-statistic status values...
[14:58] Alain Smette
Hi John, here it is: https://jira.eso.org/browse/PIPE-9487 : 

[14:58] Alain Smette
If I am not mistaken, the 'status' value reports the 'info' from cpl_mpfit.
In cpl_mpfit.c I find:
info is an integer output variable. if the user has
terminated execution, info is set to the (negative)
value of iflag. see description of fcn. otherwise,
info is set as follows.info = 0 improper input parameters.info = 1 both actual and predicted relative reductions
in the sum of squares are at most ftol.info = 2 relative error between two consecutive iterates
is at most xtol.info = 3 conditions for info = 1 and info = 2 both hold.info = 4 the cosine of the angle between fvec and any
column of the jacobian is at most gtol in
absolute value.info = 5 number of calls to fcn has reached or
exceeded maxfev.info = 6 ftol is too small. no further reduction in
the sum of squares is possible.info = 7 xtol is too small. no further improvement in
the approximate solution x is possible.info = 8 gtol is too small. fvec is orthogonal to the
columns of the jacobian to machine precision.
            """

        # bind text box change events to redraw of include/exclude regions
        # Bind some Parameter change events to specific functions...
        params_to_bind=[
            [
                (dataPlotManager.recipe_name,'WAVE_INCLUDE'),
                wx.EVT_TEXT_ENTER,
                dataPlotManager.WAVE_INCLUDE_event_enter
            ],
            [
                (dataPlotManager.recipe_name,'WAVE_EXCLUDE'),
                wx.EVT_TEXT_ENTER,
                dataPlotManager.WAVE_EXCLUDE_event_enter
            ],
            [
                (dataPlotManager.recipe_name,'PIXEL_EXCLUDE'),
                wx.EVT_TEXT_ENTER,
                dataPlotManager.PIXEL_EXCLUDE_event_enter
            ],
        ]
        for p in params_to_bind :
            if p[0] in dataPlotManager.params_by_name :
                if p[1] == wx.EVT_TEXT_ENTER :
                    #dataPlotManager.params_by_name[p[0]].paramChangeCtrl
                    pass
                dataPlotManager.params_by_name[p[0]].paramChangeCtrl.Bind(
                    p[1],
                    p[2],
                )

        # sort out CtrlPanel geometry a bit...
        if sys.platform == "darwin":
            window_width=1280
            window_height=1024
            pp_width=410
            b_width=20
            # This fails for py2.7, but that's OK since it's really only needed fo macOS...
            # and older wx versions handle the geometry better anyway...
            try :
                dataPlotManager.wxApp_elements_by_name['ReflexInteractiveWxApp'][0].SetSize(50,50,window_width,window_height)
                dataPlotManager.wxApp_elements_by_name['ctrlandpltSplitter'][0].SetSashPosition(window_width-(pp_width+4*b_width))
            except :
                pass

        if interactive_app.pre_recipe_exec :
            # For now leave both off by default...
            dataPlotManager.wxApp_elements_by_name['setInitSopCheck'][0].SetValue(False)
            if 'setInitSopCheck_OBJECT_and_SETTING' in dataPlotManager.wxApp_elements_by_name.keys() :
                dataPlotManager.wxApp_elements_by_name['setInitSopCheck_OBJECT_and_SETTING'][0].SetValue(False)
            dataPlotManager.toggle_sis(None)
            # Set Param initial values...
            if dataPlotManager.pse_is_set :
                dataPlotManager.clicked_ApplyPSEParameters_button(None)
            elif dataPlotManager.saved_init_sop is not None :
                dataPlotManager.clicked_ApplySavedParameters_button(None)
            else :
                dataPlotManager.clicked_ApplyOriginalParameters_button(None)

            dataPlotManager.wxApp_elements_by_name['setDisableCheck'][0].Enable(False)
            dataPlotManager.wxApp_elements_by_name['repeatBtn'][0].Enable(False)
        else :
            dataPlotManager.clicked_ApplyOriginalParameters_button(None)

        bind_changed_events={
            'COLUMN_LAMBDA': dataPlotManager.COLUMN_LAMBDA_changed_event,
            'COLUMN_FLUX': dataPlotManager.COLUMN_FLUX_changed_event,
            'COLUMN_DFLUX': dataPlotManager.COLUMN_DFLUX_changed_event,
            'WLG_TO_MICRON': dataPlotManager.WLG_TO_MICRON_changed_event,
            'WAVELENGTH_FRAME': dataPlotManager.WAVELENGTH_FRAME_changed_event,
        }
        for r,p in dataPlotManager.params_by_name :
            if '_KEYWORD' in p :
                bind_changed_events[p]=dataPlotManager.KEYWORD_changed_event
        for k in bind_changed_events.keys() :
            if k in dataPlotManager.orig_init_sop['by_name'].keys() :
                interactive_app.wxApp.Bind(
                    wx.EVT_TEXT,
                    bind_changed_events[k],
                    dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl,
                )
        # MJD-OBS is special and MUST be present in the data otherwise esoreflex fails...
        for k in [
            'OBSERVING_DATE_KEYWORD',
        ] :
            if dataPlotManager.recipe_name in [ 'molecfit_model', ] :
                dataPlotManager.params_by_name[(dataPlotManager.recipe_name,k)].paramChangeCtrl.Enable(False)

        # Following "works" but is faulty not properly filling the screen...
        # top of window is hidden under macOS menu bar...
        # dataPlotManager.wxApp_elements_by_name['ReflexInteractiveWxApp'][0].ShowFullScreen(True)

        ## --------------------------------------------------------------------
        ## copied from reflex_interactive_gui.shwGUI()
            interactive_app.wxApp.MainLoop()
        ## --------------------------------------------------------------------
    else:
        interactive_app.set_continue_mode()
        # Check if this is being run as pre-MolecfitModel GUI...
        if interactive_app.pre_recipe_exec :
            interactive_app.user_edited_param=(
                dataPlotManager.saved_init_sop
                or
                dataPlotManager.pse_init_sop['init_sop']
            )

    # Make MOLECULES.fits, WAVE_INCLUDE.fits and WAVE_EXCLUDE.fits files
    # suitable for use as inputs (so the user can copy them to the RAW_DATA_DIR if desired)
    #
    try:
        _files={'index': []}
        '''
        # Working out what info is where...
        print(get_attrs(interactive_app))
        print(type(interactive_app.values))
        print(interactive_app.values.products_dir)
        for _a in get_attrs(interactive_app) :
            print(_a)
            print(getattr(interactive_app,_a))
        '''
        for _f in interactive_app.inputs.in_sof.files :
            _files[_f.category]=_f.name
            _files['index']+=[_f.category,]

        '''
        # MOLECULES.fits
        ## get MODEL_MOLECULES.fits from files and "massage"
        # Update the REL_COLs from the best fit...
        _t=Table.read(_files['MODEL_MOLECULES'], format='fits')
        _BFP=Table.read(_files['BEST_FIT_PARAMETERS'], format='fits')
        for r in _t :
            colname=np.array(["rel_mol_col_%-30s" %(r['LIST_MOLEC']),], dtype=_BFP['parameter'].dtype)[0]
            r['REL_COL']=_BFP[_BFP['parameter']==colname]['value']
        _t_fn="%s/%s_%s" %(
            interactive_app.values.products_dir,
            interactive_app.inputs.in_sof.datasetName
            ,'MOLECULES.fits'
        )
        _t.meta={}
        _t.write(_t_fn, overwrite=True )
        _t=pyfits.open(_t_fn)
        # reopen the original so that we get the FULL header
        _hduList=pyfits.open(_files['MODEL_MOLECULES'])
        _hduList[0].header['HIERARCH ESO PRO CATG']="MOLECULES"
        _hduList[1].header['HIERARCH ESO PRO CATG']="MOLECULES"
        _hduList[1].data=_t[1].data
        _t.close()
        _hduList.writeto(_t_fn, overwrite=True, checksum=True, output_verify='silentfix')
        _hduList.close()
        '''

        col_formats={
            'WAVE_INCLUDE': 'D',
            'WAVE_EXCLUDE': 'D',
            'PIXEL_EXCLUDE': 'I',
            'MAP_REGIONS_TO_CHIP': 'I',
            'FIT_CONTINUUM': 'I',
            'CONTINUUM_N': 'I',
            'FIT_WLC': 'I',
        }

        if hasattr(interactive_app,'user_edited_param') :
            uep=init_sop_dict(interactive_app.user_edited_param)
        else :
            uep=init_sop_dict( interactive_app.inputs.in_sop )

        # WAVE_INCLUDE.fits
        p='WAVE_INCLUDE'
        if (
            init_sop_dict_param( uep, p ).value != 'NULL'
            and
            p in _files['index']
        ) :
            _t_fn="%s/%s_%s.fits" %(
                interactive_app.values.products_dir,
                interactive_app.inputs.in_sof.datasetName,
                p,
            )
            _hduList=pyfits.open(_files[p])
            _wi=init_sop_dict_param( uep, p ).value.split(',')
            #_wi=dataPlotManager.wl_back_from_topo_vac(_wi)
            _n=len(_wi)/2
            _mtc=init_sop_dict_param( uep, 'MAP_REGIONS_TO_CHIP').value.split(',')
            if len(_mtc) == 1 :
                _mtc=list(np.repeat(_mtc,_n))
            _cff=init_sop_dict_param( uep, 'FIT_CONTINUUM').value.split(',')
            if len(_cff) == 1 :
                _cff=list(np.repeat(_cff,_n))
            _cpo=init_sop_dict_param( uep, 'CONTINUUM_N').value.split(',')
            if len(_cpo) == 1 :
                _cpo=list(np.repeat(_cpo,_n))
            _wlf=init_sop_dict_param( uep, 'FIT_WLC').value.split(',')
            if len(_wlf) == 1 :
                _wlf=list(np.repeat(_wlf,_n))
            _t=pyfits.HDUList([
                pyfits.PrimaryHDU(data=None, header=_hduList[0].header),
                pyfits.BinTableHDU.from_columns([
                    pyfits.Column(name='LOWER_LIMIT', format=col_formats['WAVE_INCLUDE'], array=_wi[::2]),
                    pyfits.Column(name='UPPER_LIMIT', format=col_formats['WAVE_INCLUDE'], array=_wi[1::2]),
                    pyfits.Column(name='MAPPED_TO_CHIP', format=col_formats['MAP_REGIONS_TO_CHIP'], array=_mtc),
                    pyfits.Column(name='CONT_FIT_FLAG', format=col_formats['FIT_CONTINUUM'], array=_cff),
                    pyfits.Column(name='CONT_POLY_ORDER', format=col_formats['CONTINUUM_N'], array=_cpo),
                    pyfits.Column(name='WLC_FIT_FLAG', format=col_formats['FIT_WLC'], array=_wlf),
                ])
            ])
            _hduList.close()
            #hdulist_strip_checksums(_t)
            # Do not write checksums other wise LazyMode is broken...
            _t.writeto(_t_fn, overwrite=True, checksum=True, output_verify='silentfix')
            '''
            if p in _files :
                interactive_app.inputs.in_sof.files[_files['index'].index(p)].name=_t_fn
            else :
                interactive_app.inputs.in_sof.files.append(
                    reflex.FitsFile(_t_fn, p, None, ['UNIVERSAL'])
                )
            '''

        # WAVE_EXCLUDE.fits & PIXEL_EXCLUDE.fits
        for p in ['WAVE_EXCLUDE','PIXEL_EXCLUDE'] :
            if (
                init_sop_dict_param( uep, p ).value != 'NULL'
                and
                p in _files['index']
            ) :
                _t_fn="%s/%s_%s.fits" %(
                    interactive_app.values.products_dir,
                    interactive_app.inputs.in_sof.datasetName,
                    p,
                )
                _hduList=pyfits.open(_files[p])
                _wi=init_sop_dict_param( uep, p ).value.split(',')
                #if p in ['WAVE_EXCLUDE',] :
                #    _wi=dataPlotManager.wl_back_from_topo_vac(_wi)
                _t=pyfits.HDUList([
                    pyfits.PrimaryHDU(data=None, header=_hduList[0].header),
                    pyfits.BinTableHDU.from_columns([
                        pyfits.Column(name='LOWER_LIMIT', format=col_formats[p], array=_wi[::2]),
                        pyfits.Column(name='UPPER_LIMIT', format=col_formats[p], array=_wi[1::2]),
                    ])
                ])
                _hduList.close()
                #hdulist_strip_checksums(_t)
                _t.writeto(_t_fn, overwrite=True, checksum=True, output_verify='silentfix')
                '''
                if p in _files :
                    interactive_app.inputs.in_sof.files[_files['index'].index(p)].name=_t_fn
                else :
                    interactive_app.inputs.in_sof.files.append(
                        reflex.FitsFile(_t_fn, p, None, ['UNIVERSAL'])
                    )
                '''
    except:
        notables=1

    if hasattr(interactive_app, 'dataPlotManager'):
        # First adjust the WL_INCLUDE|EXCLUDE regions so that when
            # molecfit 'corrects' to TOPO_VAC, it actually ends up with the
        # regions specified here...
        if hasattr(interactive_app,'user_edited_param') :
            for sop_name in [
                'user_edited_param',
            ] :
                sop = getattr(interactive_app,sop_name)
                sop_params_by_name = dict(
                    [
                        ((x.recipe, x.name), x)
                        for x in sop
                    ]
                )
        '''
                for p_name in [
                    'WAVE_INCLUDE',
                    'WAVE_EXCLUDE',
                ] :
                    if ('molecfit_model',p_name) in sop_params_by_name.keys() :
                        if sop_params_by_name.get(('molecfit_model',p_name),'null').value.lower() not in [
                            'null',
                        ] :
                            _wl=np.array(
                                sop_params_by_name[('molecfit_model',p_name)].value.split(','),
                                dtype=float,
                            )
                            _wl=dataPlotManager.wl_back_from_topo_vac(
                                _wl,
                                WL_FRAME=dataPlotManager.get_sop_param_Value( uep, 'WAVELENGTH_FRAME',)
                            )
                            sop_params_by_name[('molecfit_model',p_name)].value=(
                                ",".join([iu.wrfmt(x) for x in _wl])
                            )
        '''
        # Write the init SOP if this was run prior to running the recipe
        if hasattr(interactive_app,'book_keeping_dir') and hasattr(interactive_app,'user_edited_param') :
            inst_setup=dataPlotManager.inst_setup
            objectname=dataPlotManager.objectname
            if inst_setup is not None :
                params = {
                    'init_sop': [],
                    'data_set': interactive_app.inputs.in_sof.datasetName,
                    'datetime': str(Time.now()),
                    'inst_setup': inst_setup,
                }
                for rp in interactive_app.user_edited_param :
                    params['init_sop']+=[rp.__dict__,]

                if dataPlotManager.write_sis :
                    _path='%s/save_settings/by_setup/' %(interactive_app.book_keeping_dir)
                    if not os.path.exists(_path) :
                        os.makedirs(_path)
                    with open('%s/save_settings/by_setup/%s.json' %(interactive_app.book_keeping_dir,inst_setup), 'w') as f:
                        json.dump(params, f)
                if dataPlotManager.write_sis_OS :
                    params['object']=objectname
                    _path='%s/save_settings/by_target/%s' %(
                        interactive_app.book_keeping_dir, objectname
                    )
                    if not os.path.exists(_path) :
                        os.makedirs(_path)
                    with open(
                        '%s/save_settings/by_target/%s/%s.json' %(
                            interactive_app.book_keeping_dir,objectname,inst_setup
                        ), 'w'
                    ) as f:
                        json.dump(params, f)
    if interactive_app.pre_recipe_exec :
        interactive_app.enable_init_sop=True
        interactive_app.set_init_sop=True
        '''
        in_sop_by_name={}
        for (i,p) in enumerate(interactive_app.inputs.in_sop) :
            in_sop_by_name[p.name]=i
        user_edited_param=[]
        if hasattr(interactive_app,'user_edited_param') :
            for p in interactive_app.user_edited_param :
                if p.name in in_sop_by_name :
                    if p.value != interactive_app.inputs.in_sop[in_sop_by_name[p.name]].value :
                        user_edited_param+=[p,]
                else :
                        user_edited_param+=[p,]
            interactive_app.user_edited_param=user_edited_param
        '''
    else :
        # when not the pre-recipe-exec, never write to the canvas...
        interactive_app.set_init_sop=False
    #
    # -------------------------------------------------------------------------------------------
    # -------------------------------------------------------------------------------------------
    # -------------------------------------------------------------------------------------------
    # Print outputs. This is parsed by the Reflex python actor to get the results. Do not remove        
    interactive_app.print_outputs()
    sys.exit()

#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    exec_main( interactive_app )
