from .fors_utils import ForsSetupInfo
from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.plots.text import TextPlot

import os

from . import ForsReportMixin


class ForsFlatImgReport(
    ForsReportMixin, MasterRawCutsReport
):  # , MasterSkyFlatReport):
    files_needed = {}  # Defined in parse_sof to allow for screen vs. sky
    category = ""
    category_label = ""
    # suitable for unnormalised Screen flats, modified in parse_sof for Sky flats
    histmin = -5000.0
    histmax = 70000.0

    def __init__(self):
        super().__init__("fors_flat_img")
        self.hist_bins_max = 40
        self.center_size = 200

    def parse_sof(self):
        """Returns a list of files selected from a set of frames (sof).

        If more than one file fulfils the criteria, the first file
        in the array will be selected.

        SOF should contain either "MASTER_SKY_FLAT_IMG" or "MASTER_SCREEN_FLAT_IMG"
        """

        for path, cat in self.inputs:
            if (
                cat == "MASTER_SKY_FLAT_IMG"
                and "master_im" not in self.files_needed.keys()
            ):
                self.files_needed["master_im"] = path
                self.category = cat
                self.category_label = "Sky"
                self.histmin = 0.8
                self.histmax = 1.2
            elif (
                cat == "MASTER_SCREEN_FLAT_IMG"
                and "master_im" not in self.files_needed.keys()
            ):
                self.files_needed["master_im"] = path
                self.category = cat
                self.category_label = "Screen"
        return [self.files_needed]

    def generate_panels(self, **kwargs):
        """Create both single and multiple extension panels."""

        self.metadata = ForsSetupInfo.flat_img(list(self.hdus[0].values())[0])

        panels = super().generate_panels(
            master_im_ext="PRIMARY",
            master_title="Master {} Flat".format(self.category_label),
            master_center_title=f"{self.category}\ncentral {self.center_size}px",
            master_im_clipping="mad",
            master_im_n_clipping=4,
            master_im_zoom_clipping="mad",
            master_im_zoom_n_clipping=3,
            cut_clipping="mad",
            cut_n_clipping=7,
            cut_cent_clipping="mad",
            cut_cent_n_clipping=7,
            hist_clipping="val",
            hist_n_clipping=[self.histmin, self.histmax],
            collapse_title="Collapse",
        )
        for i, (panel, panel_descr) in enumerate(panels.items()):
            panel_descr["report_description"] = (
                f"FORS {self.category_label} Flat panel - "
                f"{os.path.basename(panel_descr['master_im'])}, "
                f"{panel_descr['master_im_ext']}"
            )
            master_im = self.hdus[i]["master_im"]

            # Text Plot
            px = 0
            py = 0
            # which hdul and ext to use
            vspace = 0.3
            fname = os.path.basename(str(master_im.filename()))
            t1 = TextPlot(columns=1, v_space=vspace)
            col1 = (
                str(master_im["PRIMARY"].header.get("INSTRUME", "Missing INSTRUME")),
                "EXTNAME: " + str(master_im["PRIMARY"].header.get("EXTNAME", "N/A")),
                "PRO CATG: "
                + str(
                    master_im["PRIMARY"].header.get(
                        "HIERARCH ESO PRO CATG", "Missing PRO CATG"
                    )
                ),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    master_im["PRIMARY"].header.get(
                        "HIERARCH ESO PRO REC1 RAW1 NAME", "Missing RAW1 NAME"
                    )
                ),
            )
            t1.add_data(col1)
            panel.assign_plot(t1, px, py, xext=2)

            px = px + 2
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = self.metadata
            t2.add_data(col2)
            panel.assign_plot(t2, px, py, xext=1)

        return panels


rep = ForsFlatImgReport()
