from .fors_utils import ForsSetupInfo
from adari_core.data_libs.master_spec_flat import MasterSpecFlatReport
from adari_core.plots.text import TextPlot
from adari_core.utils.utils import fetch_kw_or_default
import os
import logging

logger = logging.getLogger(__name__)


class ForsFlatSpecLssReport(MasterSpecFlatReport):
    files_needed = {
        "master_im": None,
        "raw_im": None,
    }  # Further elaborated in parse_sof
    category_label = ""
    center_size = 200

    def __init__(self):
        super().__init__("fors_flat_spec_lss")

    def parse_sof(self):
        """Returns a list of files selected from a set of frames (sof).

        If more than one file fulfils the criteria, the first file
        in the array will be selected.

        SOF should contain one of: MASTER_NORM_FLAT_{MXU/MOS/LONG_MOS/PMOS}
        """

        files_path, files_category = (
            [elem[0] for elem in self.inputs],
            [elem[1] for elem in self.inputs],
        )

        if "MASTER_NORM_FLAT_LSS" in files_category:
            self.category_label = "MLSS"
        if "FLUX_LAMP_LSS" in files_category:
            self.category_label = "MFLUX"
        if "SCREEN_FLAT_LSS" in files_category:
            self.category_label = "RLSS"
        if "FLUX_ARC_LSS" in files_category:
            self.category_label = "RFLUX"
        if "FLUX_FLAT_LSS" in files_category:
            self.category_label = "RFLAT"

        categories_needed = {}
        if "FLUX_ARC_LSS" in files_category:
            categories_needed["raw_im"] = "FLUX_ARC_LSS"
            categories_needed["master_im"] = "FLUX_LAMP_LSS"
        elif "FLUX_FLAT_LSS" in files_category:
            categories_needed["raw_im"] = "FLUX_FLAT_LSS"
            categories_needed["master_im"] = "FLUX_LAMP_LSS"
        else:
            categories_needed["raw_im"] = "SCREEN_FLAT_LSS"
            categories_needed["master_im"] = "MASTER_NORM_FLAT_LSS"

        for category_label, sof_procatg in categories_needed.items():
            # Check that category matches the requirement
            if sof_procatg in files_category:
                self.files_needed[category_label] = files_path[
                    files_category.index(sof_procatg)
                ]
            else:
                raise IOError("{} file not found".format(sof_procatg))

        return [self.files_needed]

    def generate_panels(self, **kwargs):
        vspace = 0.3

        logger.info("Working on category {} report".format(self.category_label))
        panels = {}
        new_panels = super().generate_panels(
                raw_hist_scale=True,
                raw_hist_scale_val=(-5000,70000),
        )

        for i, (panel, panel_descr) in enumerate(new_panels.items()):
        
            # Retrieve appropriate metadata
            self.metadata = ForsSetupInfo.flat(list(self.hdus[i].values())[0])
            self.metadata_2 = ForsSetupInfo.flat_2(list(self.hdus[i].values())[0])

            master_im = self.hdus[i]["master_im"]
            instru = fetch_kw_or_default(
                 master_im["PRIMARY"], "INSTRUME", "Missing INSTRUME"
            )
            master_procatg = fetch_kw_or_default(
                 master_im["PRIMARY"], "HIERARCH ESO PRO CATG", "Missing PRO CATG"
            )
            fname = os.path.basename(str(master_im.filename()))
            t1 = TextPlot(columns=1, v_space=vspace)
            col1 = (
                instru,
                "EXTNAME: "
                + str(fetch_kw_or_default(master_im["PRIMARY"], "EXTNAME", "N/A")),
                "PRO CATG: " + str(master_procatg),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                  fetch_kw_or_default(
                    master_im["PRIMARY"],
                    "HIERARCH ESO PRO REC1 RAW1 NAME",
                    "Missing RAW1 NAME",
                  )
                ),
            )
            t1.add_data(col1)
            panel.assign_plot(t1, 0, 0, xext=1)
            t2 = TextPlot(columns=1, v_space=vspace)
            col2 = self.metadata
            t2.add_data(col2)
            panel.assign_plot(t2, 1, 0, xext=1)
            t3 = TextPlot(columns=1, v_space=vspace, xext=1)
            col3 = self.metadata_2
            t3.add_data(col3)
            panel.assign_plot(t3, 2, 0, xext=1)

        panels = {**panels, **new_panels}

        return panels


rep = ForsFlatSpecLssReport()
