from astropy.io import fits
from edps import File, FitsFile
from edps import List, ClassifiedFitsFile, JobParameters, get_parameter, Job
from edps import RecipeInvocationArguments, RecipeInvocationResult, InvokerProvider, RecipeInputs, ProductRenamer

from . import fors_keywords as kwd


# --- Functions to determine the detector and disable data processing if det.chip1.id = CCID20-14-5-6
# def which_detector(files: List[ClassifiedFitsFile]):
#    return files[0].get_keyword_value(kwd.det_chip1_id, None)
#
# def disable_recipe(job:Job):
#    chip_id = job.parameters.workflow_parameters.get("which_detector",None)
#    if chip_id == "CCID20-14-5-6":
#        job.command = empty
#    return True
#
# def is_chip1(params: JobParameters) -> bool:
#    return params.get_workflow_param('which_detector') == 'CCID20-14-5-3'


# --- Function to set the recipe parameters endwavelength depending on the grism used -------------
def set_endwavelength(job: Job):
    grism_name = job.input_files[0].get_keyword_value(kwd.ins_gris1_name, None)
    if grism_name == "GRIS_300V":
        job.parameters.recipe_parameters["fors.fors_calib.endwavelength"] = 9300.


# --- Functions to determine which observation type the data are ----------------------------------
def which_observation_type(files: List[ClassifiedFitsFile]):
    # Note: files are only the main input files, not the associated files
    return files[0].get_keyword_value(kwd.ins_mode, None)


def which_bin_clock(files: List[ClassifiedFitsFile]):
    # Note: files are only the main input files, not the associated files
    bin = files[0].get_keyword_value(kwd.det_binx, None)
    clock = files[0].get_keyword_value(kwd.det_read_clock, None)
    if bin == 1 or clock != "100Kps/2ports/high_gain":
        return 'other'
    else:
        return 'regular'


def is_lss(params: JobParameters) -> bool:
    return params.get_workflow_param('ins_mode') == 'LSS'


def is_mos(params: JobParameters) -> bool:
    return params.get_workflow_param('ins_mode') == 'MOS'


def is_mxu(params: JobParameters) -> bool:
    return params.get_workflow_param('ins_mode') == 'MXU'


def is_regular(params: JobParameters) -> bool:
    return params.get_workflow_param('bin_clock') == 'regular'


def is_other(params: JobParameters) -> bool:
    return params.get_workflow_param('bin_clock') == 'other'


def is_lss_regular(params: JobParameters) -> bool:
    return is_lss(params) and is_regular(params)


def is_mos_regular(params: JobParameters) -> bool:
    return is_mos(params) and is_regular(params)


def is_mxu_regular(params: JobParameters) -> bool:
    return is_mxu(params) and is_regular(params)


def is_lss_other(params: JobParameters) -> bool:
    return is_lss(params) and is_other(params)


def is_mos_other(params: JobParameters) -> bool:
    return is_mos(params) and is_other(params)


def is_mxu_other(params: JobParameters) -> bool:
    return is_mxu(params) and is_other(params)


def is_pmos(params: JobParameters) -> bool:
    return params.get_workflow_param('ins_mode') == 'PMOS'


def is_mos_or_lss(params: JobParameters) -> bool:
    return is_mos(params) or is_lss(params)


def use_standard_stars(params: JobParameters) -> bool:
    return get_parameter(params, "use_standard_star") == "TRUE"


def use_standard_pstars(params: JobParameters) -> bool:
    return get_parameter(params, "use_standard_pstar") == "TRUE"


def use_static_table(params: JobParameters) -> bool:
    return get_parameter(params, "use_standard_star") != "TRUE"


# -------------------------------------------------------------------------------------------------

# --- FUNCTIONS TO PERFORM THE TELLURIC CORRECTION ------------------------------------------------

# --- This function checks the parameter telluric_correction in the fors_parameters.yaml file
# and determines whether the telluric correction has to be carried on or not.
def perform_telluric_correction(params: JobParameters) -> bool:
    return get_parameter(params, "telluric_correction") == "TRUE"


# --- This function finds the spectrum with higher signal-to-noise
def find_reference_file(science_files: List[File]):
    max_snr = 0
    reference_file = None
    for file in science_files:
        with fits.open(file.file_path) as hdus:
            snr = hdus[0].header['SNR']
            wavelen_max = hdus[0].header['WAVELMAX']
            if snr > max_snr and wavelen_max > 680.:
                max_snr = snr
                reference_file = file
    if reference_file:
        return File(reference_file.file_path, reference_file.category, ""), max_snr
    else:
        return None, None


# --- Generic function to run a recipe
def run_recipe(input_file, associated_files, parameters, recipe_name, args, invoker, renamer) -> (
        RecipeInvocationResult, List):
    # input_file: main input and category. Format: File(string_with_full_path, string_with_category, "")
    # associated_files: calibrations. Format List[file], where files have the format: File(string_with_full_path, string_with_category, "")
    # parameters: non default recipe parameters. Format {'parameter_name1': value1, 'parameter_name2': value2}
    # recipe_name: recipe name  Format: string
    # args, invoker: extra stuff provided by the task that calls the function calling run_recipe()

    inputs = RecipeInputs(main_upstream_inputs=[input_file], associated_upstream_inputs=associated_files)
    arguments = RecipeInvocationArguments(inputs=inputs, parameters=parameters,
                                          job_dir=args.job_dir, input_map={},
                                          logging_prefix=args.logging_prefix)

    results = invoker.invoke(recipe_name, arguments, renamer, create_subdir=True)
    output_files = [File(f.name, f.category, "") for f in results.output_files]
    return results, output_files


# --- This function coordinates the telluric correction process.
#  - It finds the spectrum with higher S/N
#  - It runs fors_molecfit_model on the higher S/N spectrum and determines the properties of the atmosphere.
#  - For each spectrum to correct, it
#      - computes the full telluric correction spectrum (fors_molecfit_calctrans)
#      - applies the correction to the science spectrum.
#
def telluric_correction(args: RecipeInvocationArguments, invoker_provider: InvokerProvider,
                        renamer: ProductRenamer) -> RecipeInvocationResult:
    invoker = invoker_provider.recipe_invoker
    results = []
    ret_codes = []
    calibration_categories = ['MOLECULES', 'WAVE_INCLUDE', 'WAVE_EXCLUDE', 'PIX_EXCLUDE', 'ATM_PROFILE_STANDARD',
                              'KERNEL_LIBRARY', 'GDAS']
    science_categories = ['REDUCED_IDP_SCI_LSS']  # , 'REDUCED_IDP_SCI_MOS', 'REDUCED_IDP_SCI_MXU']

    calibration_files = [File(f.name, f.category, "") for f in args.inputs.combined if
                         f.category in calibration_categories]
    science_files = [File(f.name, f.category, "") for f in args.inputs.combined if
                     f.category in science_categories]

    # find the file with higher signal-to-noise
    reference_file, max_snr = find_reference_file(science_files)

    # run fors_molecfit_model on reference_file if available
    if reference_file:
        molefit_parameters = {'WAVE_INCLUDE': '0.61,0.64,0.68,0.71,0.711,0.740,0.75,0.78,0.81,0.84,0.91,0.95,0.96,0.98',
                              'LIST_MOLEC': 'H2O, O2', "FIT_MOLEC": 1, "REL_COL": 1}

        result_reference, reference_files = run_recipe(reference_file, calibration_files, molefit_parameters,
                                                       'fors_molecfit_model', args, invoker, renamer)

        ret_codes.append(result_reference.return_code)
        reference_files = [File(f.name, f.category, "") for f in result_reference.output_files]

        # LOOPING fors_molecfit_calctrans and fors_molecfit_correct, using results from the reference file for all the inputs
        for science_file in science_files:
            # run fors_molecfit_calctrans and for_molecfit_correct
            results_calctrans, caltrans_output_files = run_recipe(science_file, reference_files, {},
                                                                  'fors_molecfit_calctrans', args, invoker, renamer)
            result_correct, correct_output_files = run_recipe(science_file, caltrans_output_files, {},
                                                              'fors_molecfit_correct', args, invoker, renamer)

            ret_codes.extend([results_calctrans.return_code, result_correct.return_code])
            results.append(result_correct)

        # Construct final RecipeInvocationResult
        ret_code = min(ret_codes)
        output_files = [FitsFile(name=f[0].name, category=f[0].category) for f in ([r.output_files for r in results])]
        corrected_files = RecipeInvocationResult(return_code=ret_code, output_files=output_files)
    else:
        output_files = [FitsFile(name=f.name, category=f.category) for f in args.inputs.combined]
        corrected_files = RecipeInvocationResult(return_code=0, output_files=output_files)
    return corrected_files


# -------------------------------------------------------------------------------------------------

# --- Function to set detmon recipe parameters depending on the input file ------------------------

def detmon_setting(job: Job):
    # ---  Recipe parameters and defaults values:--------------
    # Note: these correspond to the rc file detmon_opt_lg_fors2.rc
    # Method to be used when computing GAIN. Methods appliable: <PTC | MED>. By
    # default PTC method will be applied.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.method"] = "PTC"

    # Polynomial order for the fit (Linearity).
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.order"] = 2

    # Kappa value for the kappa-sigma clipping (Gain).
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.kappa"] = 4.0

    # Number of iterations to compute rms (Gain).
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.niter"] = 5

    # x coordinate of the lower-left point of the region of interest. If not
    # modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 1

    # y coordinate of the lower-left point of the region of interest. If not
    # modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 6

    # x coordinate of the upper-right point of the region of interest. If not
    # modified, default value will be X dimension of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 2048

    # y coordinate of the upper-right point of the region of interest. If not
    # modified, default value will be Y dimension of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 1029

    # User reference level.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ref_level"] = 40000

    # De-/Activate intermediate products.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.intermediate"] = "FALSE"

    # De-/Activate the autocorr option.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.autocorr"] = "FALSE"

    # De-/Activate the collapse option.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.collapse"] = "TRUE"

    # De-/Activate the image rescale option.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.rescale"] = "TRUE"

    # De-/Activate the computation with pixel to pixel accuracy.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.pix2pix"] = "TRUE"

    # De-/Activate the binary bpm option.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.bpmbin"] = "FALSE"

    # Maximum x-shift for the autocorr.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.m"] = 26

    # Upper limit of Median flux to be filtered.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.filter"] = -1

    # Maximum y-shift for the autocorr.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.n"] = 26

    # Tolerance for pair discrimination.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.tolerance"] = 0.1

    # Specific name for PAF file.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.pafname"] = "detmon_opt_lg"

    # Activate the multi-exts option.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.exts"] = 0

    # Method for computing Fixed Pattern Noise (SMOOTH or HISTOGRAM).
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.fpn_method"] = "HISTOGRAM"

    # template size in pixels for smoothing during FPN computation (only for SMOOTH
    # method).
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.fpn_smooth"] = 13

    # all frames with mean saturation above the limit would not be used in
    # calculation.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 65535.0

    # x coord of the lower-left point of the first field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = -1

    # y coord of the lower-left point of the first field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = -1

    # x coord of the upper-right point of the first field used for contamination
    # measurement. If not modified, default value will be X dimension of the
    # input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = -1

    # y coord of the upper-right point of the first field used for contamination
    # measurement. If not modified, default value will be Y dimension of the
    # input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = -1

    # x coord of the lower-left point of the second field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = -1

    # y coord of the lower-left point of the second field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = -1

    # x coord of the upper-right point of the second field used for contamination
    # measurement. If not modified, default value will be half of the X dimension
    # of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = -1

    # y coord of the upper-right point of the second field used for contamination
    # measurement. If not modified, default value will be half of the Y dimension
    # of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = -1

    # x coord of the lower-left point of the third field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = -1

    # y coord of the lower-left point of the third field used for contamination
    # measurement. If not modified, default value will be half of the Y dimension
    # of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = -1

    # x coord of the upper-right point of the third field used for contamination
    # measurement. If not modified, default value will be half of X dimension of
    # the image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = -1

    # y coord of the upper-right point of the third field used for contamination
    # measurement. If not modified, default value will be Y dimension of the
    # image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = -1

    # x coord of the lower-left point of the fourth field used for contamination
    # measurement. If not modified, default value will be half of X dimension of
    # the image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = -1

    # y coord of the lower-left point of the fourth field used for contamination
    # measurement. If not modified, default value will be half of the Y dimension
    # of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = -1

    # x coord of the upper-right point of the fourth field used for contamination
    # measurement. If not modified, default value will be X dimension of the
    # image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = -1

    # y coord of the upper-right point of the fourth field used for contamination
    # measurement. If not modified, default value will be Y dimension of the
    # input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = -1

    # x coord of the lower-left point of the fifth field used for contamination
    # measurement. If not modified, default value will be half of the X dimension
    # of the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = -1

    # y coord of the lower-left point of the fifth field used for contamination
    # measurement. If not modified, default value will be 1.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = -1

    # x coord of the upper-right point of the fifth field used for contamination
    # measurement. If not modified, default value will be X dimension of the
    # image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = -1

    # y coord of the upper-right point of the fifth field used for contamination
    # measurement. If not modified, default value will be half of Y dimension of
    # the input image.
    job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = -1
    # -----

    # ---- Change parameter setting depending on the input data type ----------
    # (if a parameter is not changed, default value is used)
    f = job.input_files
    det_chip1_id = f[0].get_keyword_value(kwd.det_chip1_id, None)
    det_binx = f[0].get_keyword_value(kwd.det_binx, None)

    if det_chip1_id == "CCID20-14-5-3" and det_binx == 1:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_C1_1.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 1924
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 62000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 1842
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 730
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 2242
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 1130
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 776
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 415
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 1524
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 776
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 1924
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 3313
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 415
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 3313
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 1524
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 1924

    elif det_chip1_id == "CCID20-14-5-6" and det_binx == 1:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_C2_1.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 2054
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 60000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 1856
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 1150
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 2256
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 1550
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 804
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 1046
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 804
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 2054
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 3308
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 1046
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 3308
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 2054

    elif det_chip1_id == "CCID20-14-5-3" and det_binx == 2:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_C1_2.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 963
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 58000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 921
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 365
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 1121
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 565
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 381
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 207
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 763
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 381
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 963
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 1661
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 207
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 1661
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 763
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 963

    elif det_chip1_id == "CCID20-14-5-6" and det_binx == 2:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_C2_2.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 1027
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 58000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 928
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 575
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 1128
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 775
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 402
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 523
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 827
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 402
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 1027
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 523
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 827
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 1027

    elif det_chip1_id == "Norma III" and det_binx == 1:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_B1_1.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 1924
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ref_level"] = 40000
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 62000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 1842
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 730
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 2242
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 1130
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 776
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 415
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 376
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 1524
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 776
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 1924
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 3313
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 15
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 415
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 3313
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 1524
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 3713
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 1924

    elif det_chip1_id == "Marlene" and det_binx == 1:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_B2_1.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 2054
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 60000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 1856
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 1150
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 2256
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 1550
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 804
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 1046
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 404
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 804
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 2054
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 3308
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 646
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 1046
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 3308
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 3708
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 2054

    elif det_chip1_id == "Norma III" and det_binx == 2:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_B1_2.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 963
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 62000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 921
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 365
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 1121
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 565
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 381
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 207
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 181
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 763
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 381
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 963
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 1661
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 7
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 207
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 1661
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 763
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 1861
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 963

    elif det_chip1_id == "Marlene" and det_binx == 2:
        # Note: these correspond to the rc file detmon_opt_lg_fors2_B2_2.rc
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury"] = 1027
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.saturation_limit"] = 60000.0
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx1"] = 928
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly1"] = 575
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx1"] = 1128
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury1"] = 775
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx2"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly2"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx2"] = 402
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury2"] = 523
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx3"] = 202
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly3"] = 827
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx3"] = 402
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury3"] = 1027
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx4"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly4"] = 323
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx4"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury4"] = 523
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.llx5"] = 1654
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.lly5"] = 827
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.urx5"] = 1854
        job.parameters.recipe_parameters["detmon.detmon_opt_lg.ury5"] = 1027


# --- Function to set sky subtraction parameters depending on the instrument mode ------
def set_skysub(job: Job):
    if is_lss(job.parameters):
        job.parameters.recipe_parameters["fors.fors_science.skyglobal"] = "TRUE"
        job.parameters.recipe_parameters["fors.fors_science.skylocal"] = "FALSE"


# --- Function to set fors_calib parameters depending on the input data ------

def set_fors_calib_mxu(job: Job):
    first_file = job.input_files[0]
    ins_mask_name = first_file.get_keyword_value(kwd.ins_mask_name, None)

    if ins_mask_name == "M012Distorti":
        job.parameters.recipe_parameters["fors.fors_calib.used_linesets"] = "standard,extended"
