from edps import science, calchecker, alternative_associated_inputs, FilterMode

from .giraffe_combine_spectra import combine_spectra
from .giraffe_flats import flats
from .giraffe_monitoring import *
from .giraffe_response import process_standard_star
from .giraffe_task_functions import *

__title__ = "GIRAFFE workflow"

idp = "idp"

# ---------------- PROCESSING TASKS ---------------------------------------------------------------

# - Task to process raw bias exposures
bias = (task('bias')
        .with_recipe('gimasterbias')
        .with_report('giraffe_rawdisp', ReportInput.RECIPE_INPUTS)
        .with_report('giraffe_master_bias', ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_main_input(raw_bias)
        .with_meta_targets([qc1calib])
        .build())

# - Task to process raw dark exposures
dark = (task('dark')
        .with_recipe('gimasterdark')
        .with_report('giraffe_rawdisp', ReportInput.RECIPE_INPUTS)
        .with_report('giraffe_master_dark', ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_main_input(raw_dark)
        .with_associated_input(bias, [master_bias_class, bad_pixel_map_class])
        .with_meta_targets([qc1calib])
        .build())

# - Subworkflow to process raw flat exposures
fibre_flat, nasmyth_flat, sky_flat = flats(bias)

slit_geometry = (alternative_associated_inputs()
                 .with_associated_input(slit_geometry_setup)
                 .with_associated_input(slit_geometry_master))

# - Task to process raw arc exposures
wavecal = (task('wavelength_calibration')
           .with_recipe('giwavecalibration')
           .with_report('giraffe_rawdisp', ReportInput.RECIPE_INPUTS)
           .with_report('giraffe_wavelength', ReportInput.RECIPE_INPUTS_OUTPUTS)
           .with_main_input(raw_arc_spectrum)
           .with_associated_input(bias, [master_bias_class])
           .with_associated_input(fibre_flat, [ff_loccentroid_class, ff_locwidth_class])
           .with_associated_input(line_catalog)
           .with_associated_input(grating_properties)
           .with_alternative_associated_inputs(slit_geometry)
           .with_associated_input(ref_dispersion_solution, min_ret=0)
           .with_input_filter(master_bias_class, master_dark_class, ff_loccentroid_class, ff_locwidth_class,
                              ff_psfcentroid_class, fiber_profile_class)
           .with_input_map({ref_dispersion_solution_class: dispersion_solution_class})
           .with_meta_targets([qc1calib])
           .build())

# --Subworkflow that runs the tasks used for quality control and instrument monitoring
detmon, grating_position, acquisition = quality_control(bias, fibre_flat, wavecal)

# -- Subworkflow to generate the tasks that create the response curves.
response_argus, response_ifu = process_standard_star(raw_std_argus, raw_std_ifu, bias, dark, fibre_flat, wavecal)

# - Task to process the scientific exposures.
# The raw scientific exposures can be grouped processed individually, by template, and by night, depending
# on the value of the workflow parameter $combine_science
science_task = (task("science")
                .with_recipe('giscience')
                .with_report('giraffe_science', ReportInput.RECIPE_INPUTS_OUTPUTS)
                .with_main_input(raw_science)
                .with_dynamic_parameter("instrument_mode", which_instrument_mode)
                .with_condition(process_data)
                .with_associated_input(bias, [master_bias_class])
                .with_associated_input(dark, [master_dark_class], min_ret=0, condition=associate_darks)
                .with_associated_input(wavecal, [dispersion_solution_class])
                .with_associated_input(grating_properties)
                .with_alternative_associated_inputs(slit_geometry)
                .with_associated_input(line_mask, min_ret=0)
                .with_associated_input(raw_acquisition, min_ret=0,
                                       max_ret=5)  # Not used by the recipe, but delivered by CalSelector
                .with_associated_input(fibre_flat, [ff_loccentroid_class, ff_locwidth_class, ff_extspectra_class,
                                                    ff_exterrors_class])
                .with_associated_input(nasmyth_flat, [nf_loccentroid_class, nf_locwidth_class, nf_extspectra_class],
                                       condition=is_argus_or_ifu)
                .with_associated_input(atmospheric_extinction, min_ret=0, condition=is_argus)
                .with_associated_input(flux_standards, min_ret=0, condition=is_argus)
                .with_associated_input(response_ifu, [INSTRUMENT_RESPONSE_ifu, STD_RBNSPECTRA_ifu, STD_RCSPECTRA_ifu],
                                       min_ret=1, condition=is_ifu)
                .with_associated_input(response_argus, [STD_RBNSPECTRA_argus, STD_RCSPECTRA_argus], min_ret=1,
                                       condition=is_argus)
                .with_job_processing(set_science_parameters)
                .with_grouping_keywords([kwd.tpl_start])
                .with_min_group_size(2)
                .with_input_filter(bad_pixel_map_class, mode=FilterMode.REJECT)
                .with_meta_targets([qc0, science, calchecker, idp])
                .build())

# Tasks to combine the science 1D spectra in IDP format using the idp_parameter set.
# For general science reduction, it is recommended to combine the spectra using the "science" task, and set the
# $combine_science workflow parameter according to the needs
combine_science = (task("combine_science")
                   .with_function(combine_spectra)
                   .with_report('giraffe_science', ReportInput.RECIPE_INPUTS_OUTPUTS)
                   .with_condition(process_data)
                   .with_dynamic_parameter("instrument_mode", which_instrument_mode)
                   .with_main_input(science_task)
                   .with_input_map({SCIENCE_RBNSPEC_IDP: SPECTRUM_1D})
                   .with_meta_targets([idp])
                   .build())
