import os
import uuid
from collections import defaultdict
from itertools import chain
from typing import List

from astropy.io import fits
from edps import RecipeInvocationArguments, InvokerProvider, ProductRenamer, RecipeInvocationResult, File, \
    RecipeInputs, FitsFile


def append_to_file(source_file: str, target_file: str):
    """
    Appends the content of source_file to target_file.
    If target_file does not exist, it will be created.
    """
    if os.path.exists(source_file):
        with open(source_file, 'rb') as source, open(target_file, 'ab') as target:
            target.write(source.read())


# --- Generic function to run a recipe
def run_recipe(input_file, associated_files, parameters, recipe_name, args, invoker, renamer) -> RecipeInvocationResult:
    # input_file: main input and category. Format: List[files], where files have the format
    #             File(string_with_full_path, string_with_category, "")
    # associated_files: calibrations. Format List[files], where files have the format
    #             File(string_with_full_path, string_with_category, "")
    # parameters: non default recipe parameters. Format {'parameter_name1': value1, 'parameter_name2': value2}
    # recipe_name: recipe name  Format: string
    # args, invoker: extra stuff provided by the task that calls the function calling run_recipe()

    inputs = RecipeInputs(main_upstream_inputs=input_file, associated_upstream_inputs=associated_files)
    job_dir = os.path.join(args.job_dir, str(uuid.uuid4()))
    arguments = RecipeInvocationArguments(inputs=inputs, parameters=parameters,
                                          job_dir=job_dir, input_map={},
                                          logging_prefix=args.logging_prefix)

    result = invoker.invoke(recipe_name, arguments, renamer, create_subdir=False)
    append_to_file(os.path.join(job_dir, 'esorex.log'), os.path.join(args.job_dir, 'esorex.log'))
    append_to_file(os.path.join(job_dir, 'esorex.stdout'), os.path.join(args.job_dir, 'esorex.stdout'))
    append_to_file(os.path.join(job_dir, 'esorex.stderr'), os.path.join(args.job_dir, 'esorex.stderr'))
    return result


def set_recipe_parameter(recipe_parameters, parameter_name, parameter_value):
    if parameter_name not in recipe_parameters:
        recipe_parameters[parameter_name] = parameter_value
    return recipe_parameters


# --- FUNCTIONS for Spectra combination.--------------------------------------------------------------------------------
def combine_function(input_files: List[File], args: RecipeInvocationArguments,
                     invoker_provider: InvokerProvider, renamer: ProductRenamer) -> RecipeInvocationResult:
    # This function sets the recipe parameters for esotk_spectrum1d_combine recipe and runs it.

    parameters = args.parameters.copy()
    nfiles = len(input_files)
    if nfiles == 1:
        # Single file input is returned in the RecipeInvocationResult format
        f = input_files[0]
        input_file = [FitsFile(name=f.file_path, category=f.category)]
        return RecipeInvocationResult(return_code=0, output_files=input_file)
    elif nfiles == 2:
        # Two files are combined by the esotk_spectrum_1d_combine recipe using the following default parameters
        # Any user input would override the defaults.
        parameters = set_recipe_parameter(parameters, "esotk_spectrum1d_combine.bpm.enable", "TRUE")
        parameters = set_recipe_parameter(parameters, "esotk_spectrum1d_combine.bpm.kappa-low", "1e6")
        parameters = set_recipe_parameter(parameters, "esotk_spectrum1d_combine.bpm.kappa-high", "3.")
        parameters = set_recipe_parameter(parameters, "esotk_spectrum1d_combine.bpm.method", "relative")
        return run_recipe(input_files, [], parameters, 'esotk_spectrum1d_combine', args,
                          invoker_provider.recipe_invoker, renamer)
    else:
        # More than two files are combined by the esotk_spectrum_1d_combine recipe using the following default parameters
        # Any user input would override the defaults.
        parameters = set_recipe_parameter(parameters, "esotk_spectrum1d_combine.bpm.enable", "FALSE")
        return run_recipe(input_files, [], parameters, 'esotk_spectrum1d_combine', args,
                          invoker_provider.recipe_invoker, renamer)


def combine_spectra(args: RecipeInvocationArguments, invoker_provider: InvokerProvider,
                    renamer: ProductRenamer) -> RecipeInvocationResult:
    # This function groups all the input 1D spectra by the FPS keyword and send each group
    # to the function that combines it.

    # List with all input science files with category SCIENCE_RBNSPEC_IDP and change their category
    # to "SPECTRUM_1D".
    science_files = [File(f.name, "SPECTRUM_1D", "") for f in args.inputs.combined if
                     f.category == "SCIENCE_RBNSPEC_IDP"]

    files_same_fps = defaultdict(list)
    for f in science_files:
        with fits.open(f.file_path) as hdul:
            fps = hdul[0].header.get('FPS', 'UNDEFINED')
            files_same_fps[fps].append(f)

    results = []
    for files_to_process in files_same_fps.values():
        results.append(combine_function(files_to_process, args, invoker_provider, renamer))

    # Returns the combined spectra and return codes for all the calls of "combine_function".
    error_codes = [res.return_code for res in results if res.return_code != 0]
    ret_code = 1 if len(error_codes) > 0 else 0
    output_files = chain.from_iterable([res.output_files for res in results])
    return RecipeInvocationResult(return_code=ret_code, output_files=list(output_files))
