from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.utils.esodet import strip_prescan_overscan
from adari_core.plots.text import TextPlot
from .giraffe_utils import GiraffeSetupInfo

import os

from .giraffe_utils import GiraffeReportMixin

center_size = 200


class GiraffeMasterBiasReport(GiraffeReportMixin, MasterRawCutsReport):
    def __init__(self):
        super().__init__("giraffe_master_bias")
        self.center_size = center_size

    def remove_raw_scan(self, im_hdu, **kwargs):
        return strip_prescan_overscan(im_hdu, debug=True, **kwargs)

    def parse_sof(self):
        raw_bias = None
        master_bias = None

        for filename, catg in self.inputs:
            if catg == "BIAS" and raw_bias is None:
                raw_bias = filename
            elif catg == "MASTER_BIAS" and master_bias is None:
                master_bias = filename

        # Build and return the file name list
        file_lists = []
        if raw_bias is not None:
            if master_bias is not None:
                file_lists.append(
                    {
                        "raw_im": raw_bias,
                        "master_im": master_bias,
                    }
                )
        return file_lists

    def generate_panels(self, **kwargs):
        exts = "PRIMARY"
        hist_bins_max = 50

        new_panels = super().generate_panels(
            master_im_ext=exts,
            raw_im_ext=exts,
            raw_title="First raw",
            master_title="Master bias",
        )
        for panel, panel_descr in new_panels.items():
            panel_descr["report_description"] = (
                f"GIRAFFE bias panel - "
                f"{os.path.basename(panel_descr['master_im'])}, "
                f"{panel_descr['master_im_ext']}"
            )

            master_im = self.hdus[0]["master_im"]
            raw_im = self.hdus[0]["raw_im"]
            raw_im_ext = panel_descr["raw_im_ext"]
            # Text Plot
            px = 0
            py = 0
            # which hdul and ext to use
            vspace = 0.3
            fname = os.path.basename(str(master_im.filename()))
            t1 = TextPlot(columns=1, v_space=vspace)

            col1 = (
                str(master_im["PRIMARY"].header.get("INSTRUME")),
                "EXTNAME: " + str(raw_im[raw_im_ext].header.get("EXTNAME", "N/A")),
                "PRO CATG: "
                + str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    master_im["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")
                ),
            )
            t1.add_data(col1)

            panel.assign_plot(t1, px, py, xext=2)

            px = px + 2
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            self.metadata = GiraffeSetupInfo.bias(master_im)
            col2 = self.metadata
            t2.add_data(col2)
            panel.assign_plot(t2, px, py, xext=1)

        histogram = panel.retrieve(4, 1)
        bin_per_val = hist_bins_max + 1  # Dummy val to start while
        vals_per_bin = 1
        rmin, rmax = histogram.get_vlim()
        while bin_per_val > hist_bins_max:
            bin_per_val = (rmax - rmin + 1) // vals_per_bin
            vals_per_bin += 1
        histogram.set_int_bins(bin_size=((rmax - rmin + 1) // bin_per_val))
        histogram.bins = bin_per_val

        return new_panels


rep = GiraffeMasterBiasReport()
