from typing import List

from edps import Job, ClassifiedFitsFile, JobParameters, get_parameter

from . import giraffe_keywords as kwd


# --- Functions to set the value of the recipe parameters fiber-splist and sloc-noise -------------
#     depending on the instrument setup, observation date, and flat type

# - Functions to compute splist
def set_splist(slit_name, exp_mode, night, default):  # Applies to all flat types
    splist = default
    if slit_name == "Medusa1":
        if exp_mode in ["H627.3", "H665.0", "L773.4", "L881.7"]:
            splist = "1-26,28-134"
        elif exp_mode == "H679.7":
            if night <= 53856.:
                splist = "1-26,28-133"
            else:
                splist = "1-26,28-135"
        elif exp_mode in ["H769.1", "H875.7", "L614.2"]:
            if night <= 53856.:
                splist = "1-26,28-134"
            else:
                splist = "1-26,28-133"
        elif exp_mode in ["H805.3A", "H836.6A", "H920.5A"]:
            if night <= 53856.:
                splist = "1-26,28-134"
            elif night <= 53948.:  # (mjd>53856.) & (mjd <= 53948.):
                splist = "1-26,28-48,50-133"
            else:
                splist = "1-26,28-134"

    elif slit_name == "Medusa2":
        if (exp_mode == "H627.3") & (night == 60115):
            splist = "1-83,85-108,110-135"
        if exp_mode == "H875.7":
            if night == 60065:
                splist = "2-108,110-135"
            else:
                splist = "1-108,110-135"
        elif exp_mode in ["H805.3A", "H920.5A"]:
            splist = "1-108,110-134"

    elif slit_name == "Argus":
        if exp_mode in ["H447.1B", "H465.6", "H484.5A", "H484.5B", "H504.8", "H548.8", "H572.8", "L427.2", "L479.7"]:
            splist = "1-317"
        elif exp_mode == "H447.1A":
            splist = "1-316"
        elif exp_mode in ["L773.4", "H920.5A", "H920.5B", "L682.2"]:
            splist = "1-315"
        elif exp_mode == "H679.7":
            splist = "1-314"
        elif exp_mode == "H429.7":
            splist = "1-318"

    elif slit_name == "IFU2":
        if exp_mode in ["L427.2", "L479.7"]:
            splist = "1-191,193-302,304-317"
        elif exp_mode in ["L385.7", "L614.2", "L682.2"]:
            splist = "1-191,193-302,304-316"
        elif exp_mode in ["L543.1", "L773.4", "L881.7"]:
            splist = "1-191,193-302,304-315"
        elif exp_mode in ["H379.0", "H484.5A", "H504.8", "H525.8A", "H525.8B", "H548.8", "H572.8", "H599.3", "H627.3",
                          "H651.5B"]:
            splist = "1-191,193-302,304-316"
        elif exp_mode in ["H679.7", "H710.5", "H737.0B", "H636.6B", "H920.5B", "H665.0"]:
            splist = "1-191,193-302,304-316"
        elif exp_mode in ["H651.5A", "H737.0A", "H769.1", "H805.3A", "H805.3B", "H836.6A", "H836.6B", "H875.7",
                          "H920.5A"]:
            splist = "1-191,193-302,304-315"

    elif slit_name == "IFU1":
        if exp_mode in ["H525.8A", "H599.3", "H627.3", "H651.5A", "H665.0", "H679.7", "H710.5", "H737.0A",
                        "H737.0B", "H769.1", "H805.3B", "H836.6B", "H920.5B", "L427.2", "L543.1", "L614.2", "L682.2",
                        "L773.4", "H805.3A", "H836.6A", "H875.7", "H920.5A", "L881.7"]:
            if (night > 54503) & (night <= 56999):  # 2008-02-07 <night<  2014-12-08
                splist = "1-191,193-228,230-277,279-317"
            elif night > 56999:  # >2014-12-08
                splist = "1-228,230-277,279-316"
            # for night <  2008-02-07 and other exp_mode, use default

    return splist


# - Functions to compute noise
def set_noise(slit_name, exp_mode, night, flat_type, default):
    noise = default
    if slit_name == "IFU2":
        if exp_mode == "H447.1A":
            if night <= 55569:  # <=2011-05
                noise = 1.5
            else:
                noise = 2.5
        elif exp_mode in ["H447.1A", "H665.0", "H599.3"]:
            noise = 2.5
        elif exp_mode in ["H395.8", "L385.7"]:
            noise = 5
        elif exp_mode == "H548.8" and flat_type == 'LAMP,FLAT,NASMYTH':
            noise = 2.5

    elif slit_name == "IFU1":
        # applies to all flats
        if exp_mode in ["H379.0", "H447.1A", "H447.1B", "H572.8", "H651.5B", "L385.7", "L543.1"]:
            noise = 2.5
        if exp_mode == "H599.3" and night >= 56614:  # 2013-11-18 (before: default values)
            noise = 2.5
        if exp_mode in ["L385.7", "L543.1"] and night >= 56614:  # 2013-11-18 (before: default values)
            noise = 5
        # applies only to nasmith flats
        if flat_type == 'LAMP,FLAT,NASMYTH':
            if exp_mode == "H525.8A":
                noise = 4
            elif exp_mode == "H548.8":
                noise = 2.5

    elif slit_name == "Argus" and night >= 54587:
        # Applies to all flats:
        if exp_mode in ["H412.4", "H447.1B"]:
            noise = 10.
        elif exp_mode == "H465.6":
            noise = 2
        elif exp_mode == "H447.1A":
            noise = 4
        # Applies to Fibre flats only
        if flat_type == 'LAMP,FLAT':
            if exp_mode == "H395.8":
                noise = 10.
            if exp_mode == "H379.0":
                noise = 2.5
            if exp_mode == "L427.2":
                noise = 3
            if exp_mode == "L543.1":
                noise = 6
        # Applies to Nasmith flats only
        if flat_type == 'LAMP,FLAT,NASMYTH':
            if exp_mode == "H395.8":
                noise = 5
            if exp_mode == "L881.7":
                noise = 12
            if exp_mode == "L385.7":
                noise = 4

    elif slit_name == "Medusa2" and night >= 54587:
        if exp_mode in ["L385.7", "L427.2", "H447.1A", "H447.1B", "H665.0"]:
            noise = 10

    elif slit_name == "Medusa1":
        if exp_mode in ["L427.2", "H805.3A"]:
            noise = 10
    return noise


# - Get properties of the input file and trigger the functions that computes the recipe paramters
def set_flat_recipe_parameters(job: Job):
    slit_name = job.input_files[0].get_keyword_value(kwd.ins_slit_name, None)
    exp_mode = job.input_files[0].get_keyword_value(kwd.ins_exp_mode, None)
    night = int(job.input_files[0].get_keyword_value(kwd.mjd_obs, None) - 0.5)
    # flat_type can be 'LAMP,FLAT', 'LAMP,FLAT,NASMYTH', or SKY,OzPoz'
    flat_type = job.input_files[0].get_keyword_value(kwd.dpr_type, None)

    splist = "giraffe.fibers.spectra"
    job.parameters.recipe_parameters[splist] = set_splist(slit_name, exp_mode, night, "setup")
    # "setup": recipe default used in reflex

    noise = "giraffe.localization.noise"
    job.parameters.recipe_parameters[noise] = set_noise(slit_name, exp_mode, night, flat_type, 7)
    # 7: recipe default


def set_science_parameters(job: Job):
    dpr_type = job.input_files[0].get_keyword_value(kwd.dpr_type, None)
    if dpr_type == "OBJECT,SimCal":
        job.parameters.recipe_parameters["giraffe.sgcalibration.zmax"] = -1.
    else:
        job.parameters.recipe_parameters["giraffe.sgcalibration.zmax"] = 10 ** 4.


# -------------------------------------------------------------------------------------------------

# -- Function to return the value of ins.mode of the main input (ins.mode = MED, ARG, IFU)
# The return value of the function is assigned to the dynamic parameter "instrument_mode"
def which_instrument_mode(files: List[ClassifiedFitsFile]):
    return files[0].get_keyword_value(kwd.ins_mode, None)


# -- Functions that determines conditions (true/false) depending on the value of the dynamic parameter
# "instrument_mode", which is computed by the function which_instrument_mode()

def is_argus(params: JobParameters) -> bool:
    return get_parameter(params, "instrument_mode") == "ARG"


def is_ifu(params: JobParameters) -> bool:
    return get_parameter(params, "instrument_mode") == "IFU"


def is_medusa(params: JobParameters) -> bool:
    return get_parameter(params, "instrument_mode") == "MED"


def is_argus_or_ifu(params: JobParameters) -> bool:
    return is_argus(params) or is_ifu(params)


def associate_darks(params: JobParameters) -> bool:
    return get_parameter(params, "associate_darks") == "TRUE"


# If IDP stream is active, the workflow parameter process_ifu_data is "FALSE". In this case, only MEDUSA files are processed.
def process_data(params: JobParameters) -> bool:
    # This condition determines whether the science and combine_science tasks are executed or not.
    # There are 2 types of data, IFU and MEDUSA.
    # If the parameter process_ifu_data is TRUE, then both IFU and MEDUSA data are reduced.
    # If the parameter process_ifu_data is FALSE, then only MEDUSA data are processed.
    return get_parameter(params, "process_ifu_data") == "TRUE" or is_medusa(params)
# ----------------------------------------------------------------------------------------------------
