/*                                                                            *
 *   This file is part of the ESO X-shooter Pipeline                          *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-05 17:38:45 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/
#include <assert.h>
#include <stdarg.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include <fcntl.h> /* used for fileutils_copy/move */
#include <sys/stat.h>/* used for fileutils_copy/move */

#include <espdr_dfs.h>
#include <espdr_utils.h>
#include <espdr_msg.h>
#include <espdr_parameters.h>
#include <espdr_keywords.h>
#include <espdr_sigma_clipping.h>

#include <cpl.h>
#include <ctype.h>
#include <stdbool.h>


/*---------------------------------------------------------------------------*/
/**
 * @defgroup espdr_utils     Tools: Miscellaneous Utilities
 *
 * This module is for the stuff that don't fit anywhere else. It should be
 * kept as small as possible.
 *
 */
/*---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/**@{*/

/**
  @brief    handle CPL errors

  @param    func_id    input function name
  @return   cpl_error_code

  @doc Example function to handle errors. In case of error this function
       displays the function caller name, the error message, the line of
       code at which the error occurs
 */
cpl_error_code
espdr_check_error_code(const char* func_id) {

   if(cpl_error_get_code() != CPL_ERROR_NONE) {
      cpl_msg_info(cpl_func,"Function %s has an error",func_id);
      cpl_msg_info(cpl_func, "%s",(const char*) cpl_error_get_message());
      cpl_msg_info(cpl_func, "%s",(const char*) cpl_error_get_where());
   }
   return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the max flux QC KWs
 @param         inst_config         instrument config
 @param         CCD_geom            CCD geometry
 @param         qc_kws              QC KWs names
 @param         max_flux            maximum flux values vector
 @param         global_QC_kw        global QC KW name
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_max_flux_QC(espdr_inst_config *inst_config,
                                 espdr_CCD_geometry *CCD_geom,
                                 espdr_qc_keywords *qc_kws,
                                 double *max_flux,
                                 char *global_QC_kw,
                                 cpl_propertylist *keywords_RE) {
    
    int i, j, k, index;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int saturation_QC = 1;
    int global_QC = 1;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_max_flux_kw,
                                                                    inst_config->prefix,
                                                                    i, j, k);
                //espdr_msg("Adding max flux KW for ext %d output [%d, %d]: %s",
                //          i, j, k, new_keyword);
                my_error = espdr_keyword_add_double(new_keyword, max_flux[index],
                                                    "Max flux [ADU], raw image",
                                                    &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_MAX_FLUX_KW to the propertylist failed: %s",
                             cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                if (max_flux[index] >= inst_config->satur_limit_adu) {
                    saturation_QC = 0;
                    global_QC = 0;
                }
                
                index++;
            }
        }
    }
    
    /* QC saturation CHECK KWs */
    
    my_error = espdr_keyword_add_int(qc_kws->qc_saturation_check_kw,
                                     saturation_QC,
                                     "Saturation [ADU] QC",
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_SATUR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "Overall WAVE QC");
    my_error = espdr_keyword_add_int(global_QC_kw,
                                     global_QC,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    return(cpl_error_get_code());
}


/* ------------------------------------------------------------------------ */
/* Andrea stuff linked to the extra qc KWs */

static cpl_error_code
espdr_add_qc_key_suff(cpl_vector* data, const char* key_val, const char* key_com,
		      const char* pkey, const char* key_suff,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum, cpl_propertylist* keywords)
{

  cpl_ensure(data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(key_val, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(pkey, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(key_suff, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(keywords, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

  char nkey_val[40];
  char comment[80];
  if (compute_mean) {
	  sprintf(nkey_val,"ESO QC %s AVG%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_vector_get_mean(data));
	  sprintf(comment,"Avg of %s%s%s",pkey,key_com,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_rms) {
	  sprintf(nkey_val,"ESO QC %s RMS%s",key_val, key_suff);
	  if(cpl_vector_get_size(data) > 1) {
	      cpl_propertylist_append_double(keywords,nkey_val,cpl_vector_get_stdev(data));
	  } else {
		  cpl_propertylist_append_double(keywords,nkey_val,0);
	  }
	  sprintf(comment,"RMS of %s%s%s",pkey,key_com, key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_sum) {
	  sprintf(nkey_val,"ESO QC %s TOT%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_vector_get_sum(data));
	  sprintf(comment,"Sum of %s%s%s",pkey,key_com,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_min) {
	  sprintf(nkey_val,"ESO QC %s MIN%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_vector_get_min(data));
	  sprintf(comment,"Min of %s%s%s",pkey,key_com,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_max) {
	  sprintf(nkey_val,"ESO QC %s MAX%s",key_val, key_suff);
	  //cpl_msg_info(cpl_func,"Max flux %s %g",nkey_val,cpl_vector_get_max(data));
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_vector_get_max(data));
	  sprintf(comment,"Max of %s%s%s",pkey,key_com,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }
    espdr_check_error_code("espdr_add_qc_key_suff");
    return cpl_error_get_code();
}
static cpl_error_code
espdr_add_qc_key_suff_select(cpl_vector* data, const int* orders_fit,
		   const char* key_val,const char* pkey, const char* key_suff,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum, cpl_propertylist* keywords)
{

  cpl_ensure(data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(key_val, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(pkey, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(key_suff, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
  cpl_ensure(keywords, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

  cpl_size size = cpl_vector_get_size(data);

  double* pdata = cpl_vector_get_data(data);
  cpl_array * array = cpl_array_wrap_double(pdata,size);

  for(cpl_size i = 0; i < size; i++) {
      if(orders_fit[i] == 0) {
	  cpl_array_set_invalid(array,i);
      }
  }
  cpl_size n_valid = size - cpl_array_count_invalid(array);
  char nkey_val[40];
  char comment[80];
  if (compute_mean) {
	  sprintf(nkey_val,"ESO QC %s AVG%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_array_get_mean(array));
	  sprintf(comment,"Avg of %s.ROX<n>.ROY<m>.%s",pkey,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_rms) {
	  sprintf(nkey_val,"ESO QC %s RMS%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_array_get_stdev(array));
	  sprintf(comment,"RMS of %s.ROX<n>.ROY<m>.%s",pkey,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }
    sprintf(nkey_val,"ESO QC LFC NORDER");
    cpl_propertylist_append_double(keywords,nkey_val, n_valid);
    sprintf(comment,"Total number of orders of with QC.WAVE==1");
    cpl_propertylist_set_comment(keywords,nkey_val,comment);
    if (compute_sum) {
	  sprintf(nkey_val,"ESO QC %s TOT%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,
					 cpl_array_get_mean(array) * n_valid);
	  sprintf(comment,"Sum of %s.ROX<n>.ROY<m>.%s",pkey,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_min) {
	  sprintf(nkey_val,"ESO QC %s MIN%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_array_get_min(array));
	  sprintf(comment,"Min of %s.ROX<n>.ROY<m>.%s",pkey,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }

    if (compute_max) {
	  sprintf(nkey_val,"ESO QC %s MAX%s",key_val, key_suff);
	  cpl_propertylist_append_double(keywords,nkey_val,cpl_array_get_max(array));
	  sprintf(comment,"Max of %s.ROX<n>.ROY<m>.%s",pkey,key_val);
	  cpl_propertylist_set_comment(keywords,nkey_val,comment);
    }
    cpl_array_delete(array);
    espdr_check_error_code("espdr_add_qc_key_suff_select");
    return cpl_error_get_code();
}
static cpl_vector*
espdr_get_vecdata_from_plist(cpl_propertylist* list)
{
  cpl_ensure(list, CPL_ERROR_NULL_INPUT, NULL);

  cpl_size size = cpl_propertylist_get_size(list);
  cpl_vector* data = cpl_vector_new(size);
  double* pdata = cpl_vector_get_data(data);
  cpl_property* p;

  cpl_type type;

  for(cpl_size k = 0; k < size; k++){

	  p = cpl_propertylist_get(list,k);
	  type = cpl_property_get_type(p);
	  if (type == CPL_TYPE_DOUBLE) {
	     pdata[k]=cpl_property_get_double(p);
	  } else if (type == CPL_TYPE_INT) {
	     pdata[k]=(double)cpl_property_get_int(p);
	  }  else {
	      cpl_msg_error(cpl_func,"error for %s",cpl_property_get_name(p));
	  }

  }
  espdr_check_error_code("espdr_get_vecdata_from_plist");
  return data;
}
static cpl_error_code
espdr_add_qc_key_stat_common_loop(const cpl_propertylist* keywords_in,
		   const char* key_val,const char** key_suff,
		   const cpl_size i, const char* regexp,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum, cpl_propertylist* keywords_ou)
{

  cpl_propertylist* xlist = NULL;
  char pkey[8];
  char prefix[13];
  xlist = cpl_propertylist_duplicate(keywords_in);
  sprintf(pkey,"QC.EXT%lld",i);
  sprintf(prefix,"^ESO %s",pkey);

  cpl_propertylist_erase_regexp(xlist, prefix, 1);

  //cpl_msg_info(cpl_func,"termination %s",regexp);
  cpl_propertylist_erase_regexp(xlist, regexp,1);

  cpl_vector* data = espdr_get_vecdata_from_plist(xlist);

  const char* key_com = ".ROX<n>.ROY<m>.";
  espdr_add_qc_key_suff(data, key_val, key_com, pkey, key_suff[i],
			compute_mean, compute_rms, compute_min,
			compute_max, compute_sum, keywords_ou);

  cpl_vector_delete(data);
  cpl_propertylist_delete(xlist);
  espdr_check_error_code("espdr_add_qc_key_stat_common_loop");

  return cpl_error_get_code();
}
static cpl_error_code
espdr_add_qc_key_stat_common(const cpl_propertylist* keywords_in,
			     const int ext_nb,
		   const char* key_val,const char** key_suff,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum, cpl_propertylist* keywords_ou)
{
  cpl_propertylist* xlist = NULL;
  char prefix[13];
  char pkey[8];
  char termination[40];
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  const char* key_com = ".ROX<n>.ROY<m>.";

  for(cpl_size i = 0; i < ext_nb; i++ ) {
      xlist = cpl_propertylist_duplicate(keywords_in);
      sprintf(pkey,"QC.EXT%lld",i);
      sprintf(prefix,"^ESO %s",pkey);

      cpl_propertylist_erase_regexp(xlist, prefix, 1);

      //cpl_msg_info(cpl_func,"termination %s",termination);
      cpl_propertylist_erase_regexp(xlist, termination,1);

      cpl_vector* data = espdr_get_vecdata_from_plist(xlist);

      espdr_add_qc_key_suff(data, key_val, key_com, "", key_suff[i],
				compute_mean, compute_rms, compute_min,
				compute_max, compute_sum, keywords_ou);

      cpl_vector_delete(data);
      cpl_propertylist_delete(xlist);

  }
  espdr_check_error_code("espdr_add_qc_key_stat_common");
  return cpl_error_get_code();
}


cpl_error_code
espdr_add_qc_airm_info(cpl_propertylist** plist)
{

    char key_name[40];
    double airm_start;
    double airm_end;
    double airm_avg = 0;
    double ia_fwhm_avg=0;
    double counter_airm = 0;
    double counter_fwhm = 0;
    cpl_boolean has_air_start = CPL_FALSE;
    cpl_boolean has_air_end = CPL_FALSE;

    for(int i = 1; i <= 4 ; i++) {
        has_air_start = CPL_FALSE;
        has_air_end = CPL_FALSE;
        sprintf(key_name,"ESO TEL%d AIRM START",i);
        if( cpl_propertylist_has(*plist,key_name) > 0 ) {
            has_air_start = CPL_TRUE;
            espdr_msg("found key %d", i);
            sprintf(key_name,"ESO TEL%d AIRM START",i);
            airm_start = cpl_propertylist_get_double(*plist,key_name);
        }
        sprintf(key_name,"ESO TEL%d AIRM END",i);
        if( cpl_propertylist_has(*plist,key_name) > 0 ) {
            has_air_end = CPL_TRUE;
            airm_end = cpl_propertylist_get_double(*plist,key_name);
        }
        if(has_air_start && has_air_end) {
            airm_avg += 0.5 * (airm_start + airm_end);
            counter_airm ++;
        } else if (has_air_start && (has_air_end == CPL_FALSE)) {
            airm_avg += airm_start;
            counter_airm ++;
        } else if ((has_air_start == CPL_FALSE) && has_air_end) {
            airm_avg +=  airm_end;
            counter_airm ++;
        }
        
        sprintf(key_name,"ESO TEL%d IA FWHM",i);
        if( cpl_propertylist_has(*plist,key_name) > 0 ) {
            ia_fwhm_avg += cpl_propertylist_get_double(*plist,key_name);
            counter_fwhm++;
        }
    }
    
    if((counter_airm > 0) && (counter_fwhm > 0)) {
        airm_avg /= counter_airm;
        ia_fwhm_avg /= counter_fwhm;
        
        cpl_propertylist_append_double(*plist, "ESO QC AIRM AVG",airm_avg);
        cpl_propertylist_set_comment(*plist,"ESO QC AIRM AVG", "average airmass");
        cpl_propertylist_append_double(*plist, "ESO QC IA FWHM AVG",ia_fwhm_avg);
        cpl_propertylist_set_comment(*plist,"ESO QC IA FWHM AVG", "average IA.FWHM");
    }
    
    espdr_check_error_code("espdr_add_qc_airm_info");
    return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_iwv_info(cpl_propertylist** plist)
{

  char key_name[40];
  double iwv_start;
  double iwv_end;
  double iwv_avg = 0;
  double counter = 0;

  for(int i = 1; i <= 4 ; i++) {
      sprintf(key_name,"ESO TEL%d AMBI IWV START",i);
      if( cpl_propertylist_has(*plist,key_name) > 0 ) {
	  espdr_msg("found key %d", i);
	  sprintf(key_name,"ESO TEL%d AMBI IWV START",i);
	  iwv_start = cpl_propertylist_get_double(*plist,key_name);
	  sprintf(key_name,"ESO TEL%d AMBI IWV END",i);
	  iwv_end = cpl_propertylist_get_double(*plist,key_name);
	  iwv_avg += 0.5 * (iwv_start + iwv_end);

	  counter++;
      }
  }

  if(counter > 0) {
      iwv_avg /= counter;

      cpl_propertylist_append_double(*plist, "ESO QC IWV AVG",iwv_avg);
      cpl_propertylist_set_comment(*plist,"ESO QC IWV AVG", "average AMBI IWV START,END");

  }
  espdr_check_error_code("espdr_add_qc_iwv_info");
  return cpl_error_get_code();
}
int*
espdr_get_ord_ref(cpl_propertylist* keywords, const char* recipe)
{
  const char* ins_mode = cpl_propertylist_get_string(keywords, "ESO INS MODE");
  int* order_ref = cpl_calloc(2,sizeof(int));
  if(strcmp(recipe,"espdr_orderdef") == 0 ) {
      if (strstr(ins_mode,"MULTIMR")) {
	      order_ref[0] = 28;
	      order_ref[1] = 24;
      } else if (strstr(ins_mode,"SINGLEHR") ||
                 strstr(ins_mode,"SINGLEUHR")) {
          order_ref[0] = 55;
	      order_ref[1] = 47;
      } else if (strstr(ins_mode,"HARPS") ||
                 strstr(ins_mode,"EGGS") ||
                 strstr(ins_mode,"HA") ||
                 strstr(ins_mode,"HE")) {
          order_ref[0] = 10;
	      order_ref[1] = 20;
      }
  } else if(strcmp(recipe,"espdr_wave_LFC") == 0 ) {
      if (strstr(ins_mode,"MULTIMR")) {
          order_ref[0] = 42;
	      order_ref[1] = 69;
      } else if (strstr(ins_mode,"SINGLEHR") ||
                 strstr(ins_mode,"SINGLEUHR")) {
	      order_ref[0] = 85;
	      order_ref[1] = 137;
      } else if (strstr(ins_mode,"HARPS") ||
                 strstr(ins_mode,"EGGS") ||
                 strstr(ins_mode,"HA") ||
                 strstr(ins_mode,"HE") ) {
          order_ref[0] = 10;
 	      order_ref[1] = 20;
      }
  } else if (strcmp(recipe,"espdr_mflat") == 0 ||
             strcmp(recipe,"espdr_cal_contam") == 0 ||
             strcmp(recipe,"espdr_wave_FP") == 0 ||
             strcmp(recipe,"espdr_wave_THAR") == 0 ||
             strcmp(recipe,"espdr_wave_LFC") == 0 ||
             strcmp(recipe,"espdr_cal_eff_ab") == 0 ||
             strcmp(recipe,"espdr_cal_flux") == 0) {
      if (strstr(ins_mode,"MULTIMR")) {
	      order_ref[0] = 28;
	      order_ref[1] = 69;
      } else if (strstr(ins_mode,"SINGLEHR") ||
                 strstr(ins_mode,"SINGLEUHR")) {
	      order_ref[0] = 55;
	      order_ref[1] = 137;
      } else if (strstr(ins_mode,"HARPS") ||
                 strstr(ins_mode,"EGGS") ||
    		     strstr(ins_mode,"HA") ||
			     strstr(ins_mode,"HE")) {
    	  order_ref[0] = 10;
    	  order_ref[1] = 20;
      }
  }
  espdr_check_error_code("espdr_get_ord_ref");
  return order_ref;
}
cpl_error_code
espdr_add_qc_key_stat_pri(cpl_propertylist** keywords,const int ext_nb,
		   const char* key_val,const char** key_suff,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  char termination[40];
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);

  for(cpl_size i = 0; i < ext_nb; i++ ) {

      espdr_add_qc_key_stat_common_loop(*keywords, key_val, key_suff, i,
					termination, compute_mean, compute_rms,
      		                        compute_min, compute_max, compute_sum,
					*keywords);
  }
  espdr_check_error_code("espdr_add_qc_key_stat_pri");
  return cpl_error_get_code();
}


static char *sliceString(char *str, int start, int end)
{

    int i;
    int size = (end - start) + 2;
    char *output = (char *)malloc(size * sizeof(char));

    for (i = 0; start <= end; start++, i++)
    {
        output[i] = str[start];
    }

    //output[size] = '\0';

    return output;
}

static cpl_error_code
espdr_rename_qc_keys(cpl_propertylist* keys_in,
		   cpl_propertylist* keys_out)
{
    cpl_size sz = cpl_propertylist_get_size(keys_in);
    cpl_property* p_old;
    //cpl_property* p_new;
    char* p_name = NULL;
    char* p_name_new = NULL;
    char* p_name_tmp = NULL;
    cpl_type p_type;
    double d_value;
    float f_value;
    int i_value;
    cpl_boolean b_value;
    const char* s_value;
    const char* comment;
    
    for(cpl_size i = 0; i < sz; i++) {

    	p_old = cpl_propertylist_get(keys_in, i);
    	p_type = cpl_property_get_type(p_old);
    	p_name = (char*) cpl_property_get_name(p_old);
    	comment = cpl_property_get_comment(p_old);

    	int slen = strlen(p_name);
    	p_name_tmp = sliceString(p_name, 12, slen);
    	p_name_new = cpl_sprintf("ESO QC %s", p_name_tmp);
    	cpl_free(p_name_tmp);
    	//cpl_msg_info(cpl_func,"p_name_new: %s", p_name_new);

    	switch(p_type) {
    	case CPL_TYPE_DOUBLE: {
    		//p_new = cpl_property_new(p_name_new, CPL_TYPE_DOUBLE);
    		//cpl_property_set_name(p_new, p_name_new);
    		d_value = cpl_property_get_double(p_old);
    		cpl_propertylist_append_double(keys_out, p_name_new, d_value);
    		cpl_propertylist_set_comment(keys_out,p_name_new, comment);
    		break;
    	}
    	case CPL_TYPE_INT: {
    		//p_new = cpl_property_new(p_name_new, CPL_TYPE_INT);
    		//cpl_property_set_name(p_new, p_name_new);
    		i_value = cpl_property_get_int(p_old);
    		//cpl_property_set_int(p_new, i_value);
    		cpl_propertylist_append_int(keys_out, p_name_new, i_value);
    		cpl_propertylist_set_comment(keys_out,p_name_new, comment);
    		break;
    	}
    	case CPL_TYPE_STRING: {
    		//p_new = cpl_property_new(p_name_new, CPL_TYPE_STRING);
    		//cpl_property_set_name(p_new, p_name_new);
    		s_value = cpl_property_get_string(p_old);
    		//cpl_property_set_string(p_new, s_value);
    		cpl_propertylist_append_string(keys_out, p_name_new, s_value);
    		cpl_propertylist_set_comment(keys_out, p_name_new, comment);
    		break;
    	}
    	case CPL_TYPE_BOOL: {
    		//p_new = cpl_property_new(p_name_new, CPL_TYPE_BOOL);
    		//cpl_property_set_name(p_new, p_name_new);
    		b_value = cpl_property_get_bool(p_old);
    		//cpl_property_set_bool(p_new, b_value);
    		cpl_propertylist_append_bool(keys_out, p_name_new, b_value);
    		cpl_propertylist_set_comment(keys_out, p_name_new, comment);
    		break;
    	}
    	case CPL_TYPE_FLOAT: {
    		//p_new = cpl_property_new(p_name_new, CPL_TYPE_FLOAT);
    		//cpl_property_set_name(p_new, p_name_new);
    		f_value = cpl_property_get_float(p_old);
    		//cpl_property_set_float(p_new, f_value);
    		cpl_propertylist_append_float(keys_out, p_name_new, f_value);
    		cpl_propertylist_set_comment(keys_out, p_name_new, comment);
    		break;
    	}
    	default: {
    		//added to reduce compiler warnings
    		break;
    	}
    	}
    	//cpl_property_set_comment(p_new, comment);
    	//cpl_propertylist_append_property(keys_out, p_new);
    	//cpl_property_delete(p_new);
    	cpl_free(p_name_new);
    
    }
	espdr_check_error_code("espdr_rename_qc_key");
	return cpl_error_get_code();
}

cpl_error_code
espdr_copy_qc_ext_key_to_ext(cpl_propertylist* keywords,
		cpl_propertylist*** kext, const int ext_nb)
{
	/* copy QC keywords to extension */
	char* regexp = NULL;
	cpl_propertylist* plist = NULL;
	for (int i = 0; i < ext_nb; i++) {

		regexp = cpl_sprintf("QC.EXT%d", i);
		plist = cpl_propertylist_new();

		cpl_propertylist_copy_property_regexp(plist, keywords, regexp, 0);

		//cpl_propertylist_erase_regexp(keywords,regexp, 0);

		espdr_rename_qc_keys(plist,(*kext)[i]);

		cpl_propertylist_delete(plist);
		cpl_free(regexp);

	}
	espdr_check_error_code("espdr_rename_qc_key");
	return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_key_stat_ext(cpl_propertylist* keywords, const int ext_nb,
		   const char* key_val,const char** key_suff,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum,
		   cpl_propertylist*** kext)
{

  char termination[40];
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);

  for(cpl_size i = 0; i < ext_nb; i++ ) {

      espdr_add_qc_key_stat_common_loop(keywords, key_val, key_suff, i, termination,
        		               compute_mean, compute_rms, compute_min,
      			       compute_max, compute_sum, (*kext)[i]);

  }
  espdr_check_error_code("espdr_add_qc_key_stat_ext");
  return cpl_error_get_code();
}


cpl_error_code
espdr_add_qc_key_stat_ord_pri(cpl_propertylist** keywords,const int ext_nb,
		   const char* key_val,const char* expr_reject,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{
  cpl_propertylist* xlist = NULL;
  char termination[40];
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);

  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);

  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);

  cpl_propertylist_erase_regexp(olist, expr_reject, 0);
  const char* pkey = "";
  const char* key_suf = "";
  const char* key_com = "QC.ORDER<o>.";

      xlist = cpl_propertylist_duplicate(olist);
      sprintf(termination,"%s$",key_val);
      //cpl_msg_info(cpl_func,"termination: %s",termination);
      cpl_propertylist_erase_regexp(xlist, termination,1);
      //cpl_propertylist_dump(xlist,stdout);
      cpl_vector* data = espdr_get_vecdata_from_plist(xlist);
      espdr_add_qc_key_suff(data, key_val, key_com, pkey, key_suf,
     				compute_mean, compute_rms, compute_min,
     				compute_max, compute_sum, *keywords);

      cpl_vector_delete(data);
      cpl_propertylist_delete(xlist);


  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_stat_ord_pri");
  return cpl_error_get_code();
}
cpl_error_code
espdr_add_qc_key_stat_ord(cpl_propertylist* keywords, const int ext_nb,
		   const char* key_val,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum,
		   cpl_propertylist*** kext)
{
  cpl_propertylist* xlist = NULL;
  const char* pkey= "";
  char* termination;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  termination = cpl_sprintf("%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  //cpl_propertylist_dump(olist,stdout);
  const char* key_com = "QC.ORDER<o>.";
  cpl_free(termination);
  cpl_size size = 0;
  for(cpl_size i = 0; i < ext_nb; i++ ) {
      xlist = cpl_propertylist_duplicate(olist);
      termination = cpl_sprintf("EXT%lld %s$",i,key_val);
      cpl_propertylist_erase_regexp(xlist, termination,1);
      size = cpl_propertylist_get_size(xlist);
      if(size > 0) {
          cpl_vector* data = espdr_get_vecdata_from_plist(xlist);
          espdr_add_qc_key_suff(data, key_val, key_com, pkey, "",
        				compute_mean, compute_rms, compute_min,
        				compute_max, compute_sum, (*kext)[i]);

          cpl_vector_delete(data);
      }
      cpl_propertylist_delete(xlist);
      cpl_free(termination);

  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_stat_ord");
  return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_key_orderdef_ord_ref(cpl_propertylist* keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum,
		   cpl_propertylist*** kext)
{

  cpl_propertylist* r1list = NULL;
  cpl_propertylist* r2list = NULL;

  cpl_propertylist* r3list = NULL;
  cpl_propertylist* r4list = NULL;
  cpl_propertylist* r5list = NULL;
  char prefix[40];
  char termination[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);

  cpl_size sz1, sz2, sz3, sz4, sz5;
  for(cpl_size i = 0; i < ext_nb; i++ ) {

      r1list = cpl_propertylist_duplicate(olist);
      r2list = cpl_propertylist_duplicate(olist);

      r3list = cpl_propertylist_duplicate(olist);
      r4list = cpl_propertylist_duplicate(olist);
      r5list = cpl_propertylist_duplicate(olist);

      sprintf(prefix,"QC.ORDER%d.EXT%lld.NB$",ord_ref[i],i);
      cpl_propertylist_erase_regexp(r1list, prefix,1);
      sz1 = cpl_propertylist_get_size(r1list);

      sprintf(prefix,"QC.ORDER%d.EXT%lld.POS$",ord_ref[i],i);
      cpl_propertylist_erase_regexp(r2list, prefix,1);
      sz2 = cpl_propertylist_get_size(r2list);

      sprintf(prefix,"QC.ORDER%d.EXT%lld.RES.STDEV$",ord_ref[i],i);
      cpl_propertylist_erase_regexp(r3list, prefix,1);
      sz3 = cpl_propertylist_get_size(r3list);

      sprintf(prefix,"QC.ORDER%d.EXT%lld.RES.MIN$",ord_ref[i],i);
      cpl_propertylist_erase_regexp(r4list, prefix,1);
      sz4 = cpl_propertylist_get_size(r4list);

      sprintf(prefix,"QC.ORDER%d.EXT%lld.RES.MAX$",ord_ref[i],i);
      cpl_propertylist_erase_regexp(r5list, prefix,1);
      sz5 = cpl_propertylist_get_size(r5list);

      if(sz1>0) {
         cpl_propertylist_append_int((*kext)[i],"ESO QC CENTORD",ord_ref[i]);
         sprintf(comment,"Reference order");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD",comment);

         p = cpl_propertylist_get(r1list,0);
         //espdr_msg("prop name: %s",cpl_property_get_name(p));
         cpl_propertylist_append_int((*kext)[i],"ESO QC CENTORD NB",
				     cpl_property_get_int(p));
         sprintf(comment,"Order number of ref order");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD NB",comment);
      }

      if(sz2>0) {
         p = cpl_propertylist_get(r2list,0);
         cpl_propertylist_append_double((*kext)[i],"ESO QC CENTORD POS",
 				     cpl_property_get_double(p));
         sprintf(comment,"Value of ref order pos");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD POS",comment);
      }

      if(sz3>0) {
         p = cpl_propertylist_get(r3list,0);
         cpl_propertylist_append_double((*kext)[i],"ESO QC CENTORD RES STDEV",
       				     cpl_property_get_double(p));
         sprintf(comment,"Value of ref order res stdev");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD RES STDEV",comment);
      }

      if(sz4>0) {
         p = cpl_propertylist_get(r4list,0);
         cpl_propertylist_append_double((*kext)[i],"ESO QC CENTORD RES MIN",
             	                     cpl_property_get_double(p));
         sprintf(comment,"Value of ref order res min");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD RES MIN",comment);
      }

      if(sz5>0) {
         p = cpl_propertylist_get(r5list,0);
         cpl_propertylist_append_double((*kext)[i],"ESO QC CENTORD RES MAX",
                   		     cpl_property_get_double(p));
         sprintf(comment,"Value of ref order res max");
         cpl_propertylist_set_comment((*kext)[i],"ESO QC CENTORD RES MAX",comment);
      }

      cpl_propertylist_delete(r1list);
      cpl_propertylist_delete(r2list);
      cpl_propertylist_delete(r3list);
      cpl_propertylist_delete(r4list);
      cpl_propertylist_delete(r5list);

  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_orderdef_ord_ref");
  return cpl_error_get_code();
}


cpl_error_code
espdr_add_qc_key_flat_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;
  cpl_propertylist* r2list = NULL;

  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  //cpl_propertylist_dump(olist,stdout);
  const char* key_com = "QC.ORDER<o>.";
  for(cpl_size i = 0; i < ext_nb; i++ ) {
      r1list = cpl_propertylist_duplicate(olist);
      r2list = cpl_propertylist_duplicate(olist);

      sprintf(prefix,"QC.ORDER%d.SNR$",ord_ref[i]);

      cpl_propertylist_erase_regexp(r1list, prefix,1);
      //cpl_propertylist_erase_regexp(r2list, "FLAT",1);
      sprintf(prefix,"QC.ORDER%d.FLAT.RMS$",ord_ref[i]);
      cpl_propertylist_erase_regexp(r2list, prefix,1);

      //cpl_propertylist_dump(r1list,stdout);
      //cpl_propertylist_dump(r2list,stdout);
      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);

      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      p = cpl_propertylist_get(r1list,0);
      sprintf(nkey_val,"ESO QC CENTORD SNR%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Value of %sSNR",key_com);
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r2list,0);
      sprintf(nkey_val,"ESO QC CENTORD RMS%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"%sFLAT.RMS value",key_com);
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      //cpl_propertylist_dump(kext[0],stdout);

      cpl_propertylist_delete(r1list);
      cpl_propertylist_delete(r2list);

  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_flat_ord_ref");
  return cpl_error_get_code();
}



cpl_error_code
espdr_add_qc_key_contam_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;


  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  cpl_propertylist_erase_regexp(olist, "MAX FLUX$", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);
  //cpl_propertylist_dump(olist,stdout);
  for(cpl_size i = 0; i < ext_nb; i++ ) {
      r1list = cpl_propertylist_duplicate(olist);


      sprintf(prefix,"QC.ORDER%d.MAX.FLUX$",ord_ref[i]);
      cpl_propertylist_erase_regexp(r1list, prefix,1);

      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r1list,0);
      sprintf(nkey_val,"ESO QC CENTORD MAX FLUX%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      cpl_propertylist_delete(r1list);


  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_contam_ord_ref");
  return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_key_effab_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{


  cpl_propertylist* r2list = NULL;

  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  //cpl_propertylist_erase_regexp(olist, "MAX FLUX$", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);
  //cpl_propertylist_dump(olist,stdout);

  for(cpl_size i = 0; i < ext_nb; i++ ) {

      r2list = cpl_propertylist_duplicate(olist);
      sprintf(prefix,"QC.ORDER%d.SNR$",ord_ref[i]);

      cpl_propertylist_erase_regexp(r2list, prefix,1);

      p = cpl_propertylist_get(r2list,0);

      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);

      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      sprintf(nkey_val,"ESO QC CENTORD SNR%s",key_suffix[i]);
      //cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      cpl_propertylist_delete(r2list);

  }

  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_effab_ord_ref");
  return cpl_error_get_code();
}



cpl_error_code
espdr_add_qc_key_wave_FP_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;


  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key >%s<",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);
  //cpl_propertylist_dump(olist,stdout);
  for(cpl_size i = 0; i < ext_nb; i++ ) {
      r1list = cpl_propertylist_duplicate(olist);


      sprintf(prefix,"QC ORDER%d %s$",ord_ref[i],key_val);

      cpl_msg_info(cpl_func,"prefix %s",prefix);
      cpl_propertylist_erase_regexp(r1list, prefix,1);

      cpl_msg_info(cpl_func,"key_suffix >%s<",key_suffix[i]);
      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val1 >%s<",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);

      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r1list,0);
      sprintf(nkey_val,"ESO QC CENTORD %s%s",key_val, key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val2 %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,cpl_property_get_int(p));
      sprintf(comment,"Order number of ref order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      cpl_propertylist_delete(r1list);


  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_wave_FP_ord_ref");
  return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_key_wave_THAR_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;
  cpl_propertylist* r2list = NULL;
  cpl_propertylist* r3list = NULL;


  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);
  //cpl_propertylist_dump(olist,stdout);
  for(cpl_size i = 0; i <  ext_nb; i++ ) {
      r1list = cpl_propertylist_duplicate(olist);
      r2list = cpl_propertylist_duplicate(olist);
      r3list = cpl_propertylist_duplicate(olist);


      sprintf(prefix,"QC ORDER%d %s$",ord_ref[i],key_val);
      cpl_msg_info(cpl_func,"prefix %s",prefix);
      cpl_propertylist_erase_regexp(r1list, prefix,1);

      sprintf(prefix,"QC.ORDER%d.CHI2$",ord_ref[i]);
      cpl_msg_info(cpl_func,"prefix %s",prefix);
      cpl_propertylist_erase_regexp(r2list, prefix,1);
      sprintf(prefix,"QC.ORDER%d.RMS$",ord_ref[i]);
      cpl_msg_info(cpl_func,"prefix %s",prefix);
      cpl_propertylist_erase_regexp(r3list, prefix,1);
      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);

      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r1list,0);
      sprintf(nkey_val,"ESO QC CENTORD %s%s",key_val, key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,cpl_property_get_int(p));
      sprintf(comment,"Order number of ref order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r2list,0);
      sprintf(nkey_val,"ESO QC CENTORD CHI2%s", key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r3list,0);
      sprintf(nkey_val,"ESO QC CENTORD RMS%s", key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);



      cpl_propertylist_delete(r1list);
      cpl_propertylist_delete(r2list);
      cpl_propertylist_delete(r3list);


  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_wave_THAR_ord_ref");
  return cpl_error_get_code();
}


cpl_error_code
espdr_add_qc_key_wave_LFC_ord_ref(cpl_propertylist** keywords,
				  const int* orders_fit, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;
  cpl_propertylist* r2list = NULL;
  cpl_propertylist* r3list = NULL;

  char prefix[40];
  char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);

  for(cpl_size i = 0; i < ext_nb; i++ ) {
      //espdr_msg("process ext: %lld",i);
      r1list = cpl_propertylist_duplicate(olist);
      r2list = cpl_propertylist_duplicate(olist);
      r3list = cpl_propertylist_duplicate(olist);

      sprintf(prefix,"QC ORDER%d %s$",ord_ref[i],key_val);

      cpl_propertylist_erase_regexp(r1list, prefix,1);

      sprintf(prefix,"QC.ORDER%d.CHI2$",ord_ref[i]);
      cpl_propertylist_erase_regexp(r2list, prefix,1);
      sprintf(prefix,"QC.ORDER%d.RMS$",ord_ref[i]);
      cpl_propertylist_erase_regexp(r3list, prefix,1);


      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);

      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);

      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r2list,0);
      sprintf(nkey_val,"ESO QC CENTORD CHI2%s", key_suffix[i]);

      if(orders_fit[ord_ref[i]] == 1) {
         cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      } else {
	  cpl_propertylist_append_double(*keywords,nkey_val,-1);
      }
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r3list,0);
      sprintf(nkey_val,"ESO QC CENTORD RMS%s", key_suffix[i]);
      //cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      if(orders_fit[ord_ref[i]] == 1) {
          cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      } else {
	  cpl_propertylist_append_double(*keywords,nkey_val,-1);
      }
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      cpl_propertylist_delete(r1list);
      cpl_propertylist_delete(r2list);
      cpl_propertylist_delete(r3list);

  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_wave_LFC_ord_ref");
  return cpl_error_get_code();
}

cpl_error_code
espdr_add_qc_key_flux_ord_ref(cpl_propertylist** keywords, const int ext_nb,
		   const int* ord_ref, const char* key_val,const char** key_suffix,
		   const cpl_boolean compute_mean, const cpl_boolean compute_rms,
		   const cpl_boolean compute_min,const cpl_boolean compute_max,
		   const cpl_boolean compute_sum)
{

  cpl_propertylist* r1list = NULL;

  char prefix[40];
  //char termination[40];
  char nkey_val[40];
  char comment[80];
  cpl_property* p;
  cpl_msg_info(cpl_func,"Add QC for key %s",key_val);
  //sprintf(termination,"%s$",key_val);
  cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
  cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
  cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);

  for(cpl_size i = 0; i < ext_nb; i++ ) {
      r1list = cpl_propertylist_duplicate(olist);


      sprintf(prefix,"QC.ORDER%d.%s$",ord_ref[i],key_val);

      //cpl_msg_info(cpl_func,"prefix %s",prefix);
      cpl_propertylist_erase_regexp(r1list, prefix,1);

      sprintf(nkey_val,"ESO QC CENTORD%s",key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_int(*keywords,nkey_val,ord_ref[i]);

      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);

      p = cpl_propertylist_get(r1list,0);
      sprintf(nkey_val,"ESO QC CENTORD %s%s",key_val,key_suffix[i]);
      cpl_msg_info(cpl_func,"nkey_val %s",nkey_val);
      cpl_propertylist_append_double(*keywords,nkey_val,cpl_property_get_double(p));
      sprintf(comment,"Reference order");
      cpl_propertylist_set_comment(*keywords,nkey_val,comment);


      cpl_propertylist_delete(r1list);

  }
  cpl_propertylist_delete(olist);
  espdr_check_error_code("espdr_add_qc_key_flux_ord_ref");
  return cpl_error_get_code();
}

double espdr_drift_vel(const double drift_mean,
                       const double mjd_obs_delta_time_wave_a) {

  return 100.0 / 0.002 * drift_mean / fabs(mjd_obs_delta_time_wave_a) / 24.;
}

/* End of the Andrea stuff linked to the extra qc KWs */
/* ------------------------------------------------------------------------ */

/*----------------------------------------------------------------------------*/
/**
 * AMO added (useful in unit tests)
  @brief    generates random data with Poisson distribution
  @param    expectedValue random generator goal value
  @return   random value
 */
/*----------------------------------------------------------------------------*/
int espdr_poisson_random(const double expectedValue) {
	int n = 0; //counter of iteration
	double limit;
	double x;  //pseudo random number
	limit = exp(-expectedValue);
	x = (double)rand() / (double)INT_MAX;
	while (x > limit) {
		n++;
		x *= (double)rand() / (double)INT_MAX;
	}
	return n;
}


/*----------------------------------------------------------------------------*/
/**
 * AMO added (useful in unit tests)
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
char *espdr_get_basename(const char *filename) {
	char *p ;
	p = strrchr (filename, '/');
	return p ? p + 1 : (char *) filename;
}

/*---------------------------------------------------------------------------*/
/* AMO added (useful to debug)
 @brief     Print recipe status
 @param     val reference point
 @return    status
 */
/*---------------------------------------------------------------------------*/
int espdr_print_rec_status(const int val) {
	if(cpl_error_get_code() != CPL_ERROR_NONE) {
		espdr_msg_error("Recipe status at %d",val);
		espdr_msg_error("%s",(const char*) cpl_error_get_message());
		espdr_msg_error("%s",(const char*) cpl_error_get_where());
		return -1;
	}
	return 0;
}

/*---------------------------------------------------------------------------*/
/**
 @brief	Get the pipeline copyright and license
 @return   The copyright and license string

 The function returns a pointer to the statically allocated license string.
 This string should not be modified using the returned pointer.
 */
/*---------------------------------------------------------------------------*/
const char *espdr_get_license (void) {
	const char *espdr_license =
			"This file is part of the ESPRESSO Instrument Pipeline\n"
			"Copyright (C) 2006 European Southern Observatory\n"
			"\n"
			"This program is free software; you can redistribute it and/or modify\n"
			"it under the terms of the GNU General Public License as published by\n"
			"the Free Software Foundation; either version 2 of the License, or\n"
			"(at your option) any later version.\n"
			"\n"
			"This program is distributed in the hope that it will be useful,\n"
			"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
			"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
			"GNU General Public License for more details.\n"
			"\n"
			"You should have received a copy of the GNU General Public License\n"
			"along with this program; if not, write to the Free Software\n"
			"Foundation, Inc., 59 Temple Place, Suite 330, Boston, \n"
			"MA  02111-1307  USA";
	return espdr_license;
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Remove the non-mandatory tags from the list
 @param     set         frameset to be checked
 @param     tags        reference TAGS list
 @param     ntags       number of tags in the list
 @param     is_required flag indicating the tag's importance
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_remove_frametag_from_set(cpl_frameset *set,
                                                     const char** tags,
                                                     const int ntags,
                                                     int* is_required)
{
    cpl_frame* f;
    
    for (int i = 0; i < ntags; i++){
        f = cpl_frameset_find(set, tags[i]);
        //espdr_msg("tag: %s is_required: %d found: %p",tags[i],is_required[i],f);
        if (f == NULL && is_required[i] == 1) {
            espdr_msg_error("frame with tag %s is not found! exit!", tags[i]);
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "Missing required input frame");
            cpl_error_set(cpl_func,CPL_ERROR_ILLEGAL_INPUT);
        } else {
            cpl_frameset_erase(set, tags[i]);
        }
    }
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief   Check the TAGS between mandatory ones and the SOF ones
 @param   set frameset to be checked
 @param   tags reference TAGS list
 @param   ntags number of tags in the list
 @return  CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_check_input_tags(cpl_frameset *set,
                                      const char** tags,
                                      int* is_required,
                                      const int ntags) {

	cpl_frameset* lset = cpl_frameset_duplicate(set);
	cpl_frame* f;
    const char* optional_tags[16] = {ESPDR_INST_CONFIG, ESPDR_MASTER_INST_CONFIG,
        ESPDR_PRO_CATG_ORDER_PROFILE_A, ESPDR_PRO_CATG_ORDER_PROFILE_B,
        ESPDR_PRO_CATG_BLAZE_A, ESPDR_PRO_CATG_BLAZE_B,
        ESPDR_PRO_CATG_FLAT_A, ESPDR_PRO_CATG_FLAT_B,
        ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_A, ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_B,
        ESPDR_PRO_CATG_MASTER_BLAZE_A, ESPDR_PRO_CATG_MASTER_BLAZE_B,
        ESPDR_PRO_CATG_MASTER_FLAT_A, ESPDR_PRO_CATG_MASTER_FLAT_B,
        ESPDR_PRO_CATG_PIXEL_GEOM_A, ESPDR_PRO_CATG_PIXEL_GEOM_B
    };
    int optional_is_required[16] = {0, 0,
        0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0,
        0, 0
    };

    espdr_remove_frametag_from_set(lset, tags, ntags, is_required);
    espdr_remove_frametag_from_set(lset, optional_tags, 16, optional_is_required);
    
    int nleft = cpl_frameset_get_size(lset);
    cpl_frameset_iterator* it = cpl_frameset_iterator_new(lset);
    for (int i = 0; i < nleft; i++){
        f = cpl_frameset_iterator_get(it);
        espdr_msg_warning("Unexpected frame tag %s", cpl_frame_get_tag(f));
        cpl_frameset_iterator_advance(it, 1);
    }
    cpl_frameset_iterator_delete(it);
    
    cpl_frameset_delete(lset);
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief   Check the inst config tab is provided in the SOF
 @param   set frameset to be checked
 @return  CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_check_input_inst_config(cpl_frameset *set) {

	cpl_frame* m_inst_cfg;
	cpl_frame* inst_cfg;

	m_inst_cfg = cpl_frameset_find(set, ESPDR_MASTER_INST_CONFIG);
	inst_cfg = cpl_frameset_find(set, ESPDR_INST_CONFIG);
	if ((NULL == m_inst_cfg) && (NULL == inst_cfg)) {
		espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
				ESPDR_MASTER_INST_CONFIG, ESPDR_INST_CONFIG );
		cpl_error_set_message(cpl_func, cpl_error_get_code(),
				"Missing required input frame");
		cpl_error_set(cpl_func,CPL_ERROR_ILLEGAL_INPUT);
	}

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief   Check the inst config tab is provided in the SOF
 @param   set frameset to be checked
 @return  CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_frame* espdr_get_inst_config(cpl_frameset *set) {
    
	cpl_frame* inst_cfg;
    
	inst_cfg = cpl_frameset_find(set, ESPDR_MASTER_INST_CONFIG);
	if (NULL == inst_cfg) {
		inst_cfg = cpl_frameset_find(set, ESPDR_INST_CONFIG);
	}
    
	return inst_cfg;
}

/*---------------------------------------------------------------------------*/
/**
 @brief   Check the mflat products (sijmple or masters) are provided in the SOF
 @param   set frameset to be checked
 @return  CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_check_input_flat(cpl_frameset *set,
                                      int blaze_flag) {
    
    cpl_frame *m_profile, *m_flat, *m_blaze;
    cpl_frame *profile, *flat, *blaze;
    
    m_profile = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_A);
    profile = cpl_frameset_find(set, ESPDR_PRO_CATG_ORDER_PROFILE_A);
    if ((NULL == m_profile) && (NULL == profile)) {
        espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                        ESPDR_PRO_CATG_ORDER_PROFILE_A,
                        ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_A);
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "Missing required input frame");
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }
    
    m_flat = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_FLAT_A);
    flat = cpl_frameset_find(set, ESPDR_PRO_CATG_FLAT_A);
    if ((NULL == m_flat) && (NULL == flat)) {
        espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                        ESPDR_PRO_CATG_FLAT_A,
                        ESPDR_PRO_CATG_MASTER_FLAT_A);
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "Missing required input frame");
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }
    
    if (blaze_flag) {
        m_blaze = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_BLAZE_A);
        blaze = cpl_frameset_find(set, ESPDR_PRO_CATG_BLAZE_A);
        if ((NULL == m_blaze) && (NULL == blaze)) {
            espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                            ESPDR_PRO_CATG_BLAZE_A,
                            ESPDR_PRO_CATG_MASTER_BLAZE_A);
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "Missing required input frame");
            cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
        }
    }
    
    m_profile = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_B);
    profile = cpl_frameset_find(set, ESPDR_PRO_CATG_ORDER_PROFILE_B);
    if ((NULL == m_profile) && (NULL == profile)) {
        espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                        ESPDR_PRO_CATG_ORDER_PROFILE_B,
                        ESPDR_PRO_CATG_MASTER_ORDER_PROFILE_B);
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "Missing required input frame");
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }
    
    m_flat = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_FLAT_B);
    flat = cpl_frameset_find(set, ESPDR_PRO_CATG_FLAT_B);
    if ((NULL == m_flat) && (NULL == flat)) {
        espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                        ESPDR_PRO_CATG_FLAT_B,
                        ESPDR_PRO_CATG_MASTER_FLAT_B);
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                              "Missing required input frame");
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }
    
    if (blaze_flag) {
        m_blaze = cpl_frameset_find(set, ESPDR_PRO_CATG_MASTER_BLAZE_B);
        blaze = cpl_frameset_find(set, ESPDR_PRO_CATG_BLAZE_B);
        if ((NULL == m_blaze) && (NULL == blaze)) {
            espdr_msg_error("No frame with tag %s nor with tag %s is found! exit!",
                            ESPDR_PRO_CATG_BLAZE_B,
                            ESPDR_PRO_CATG_MASTER_BLAZE_B);
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "Missing required input frame");
            cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
        }
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Merge a list of images into one
 @param         images_to_merge list of images to merge
 @param         CCD_geom        CCD geometry
 @param         type            images type
 @param[out]    merged_image_RE merged image
 @return	CPL_ERROR_NONE iff OK

 The function merges all the images form the list and saves it into one.
 The order of frames is first column (from bottom to top), then row
 (from left to right).

 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_image_merge_2_dim(const cpl_imagelist *images_to_merge,
		const espdr_CCD_geometry *CCD_geom,
		const cpl_type type,
		cpl_imagelist **merged_image_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;
	int i = 0, j = 0, k = 0;
	int index_from = 0, index_to = 0, index_from_bis = 0;
	int pos_x = 1, pos_y = 1;
	int image_nx = 0, image_ny = 0;
	cpl_image *ext_image;
	cpl_image *curr_image;

	espdr_ensure(images_to_merge == NULL, CPL_ERROR_NULL_INPUT,
			"Input images list is NULL");

	espdr_ensure(cpl_imagelist_get_size(images_to_merge) !=
			CCD_geom->total_output_nb,
			CPL_ERROR_ILLEGAL_INPUT, "Wrong number of input images");

	espdr_msg_debug("input images list size: %lld",
			cpl_imagelist_get_size(images_to_merge));

	index_from = 0;
	index_from_bis = 0;
	index_to = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		if (CCD_geom->exts[i].out_nb_x*CCD_geom->exts[i].out_nb_y == 1) {
			my_error = cpl_imagelist_set(*merged_image_RE,
					cpl_image_duplicate(cpl_imagelist_get_const
							(images_to_merge,
									index_from)),
									index_to);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_imagelist_set failed: %s",
					cpl_error_get_message_default(my_error));
			index_from++;
			index_to++;
			index_from_bis++;
		} else {
			image_nx = 0;
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				image_ny = 0;
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					curr_image = (cpl_image*) cpl_imagelist_get_const(images_to_merge,
							index_from_bis);
					//espdr_msg("curr_image %d[%d,%d,%d] size_x: %lld, size_y: %lld",
					//                index_from_bis, i, j, k,
					//                cpl_image_get_size_x(curr_image),
					//                cpl_image_get_size_y(curr_image));
					index_from_bis++;
					image_ny = image_ny + cpl_image_get_size_y(curr_image);
				}
				image_nx = image_nx + cpl_image_get_size_x(curr_image);
			}

			//espdr_msg("Merged image size x: %d, size y: %d",
			//                image_nx, image_ny);
			ext_image = cpl_image_new(image_nx, image_ny, type);
			my_error = cpl_error_get_code();
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_new failed: %s",
					cpl_error_get_message_default(my_error));

			pos_x = 1;
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				pos_y = 1;
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					curr_image = (cpl_image*) cpl_imagelist_get_const(images_to_merge, index_from);
					index_from++;
					my_error = cpl_image_copy(ext_image,
							curr_image,
							pos_x, pos_y);
					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_image_copy failed: %s",
							cpl_error_get_message_default(my_error));
					pos_y = pos_y + cpl_image_get_size_y(curr_image);
				}
				pos_x = pos_x + cpl_image_get_size_x(curr_image);
			}

			my_error = cpl_imagelist_set(*merged_image_RE,
					cpl_image_duplicate(ext_image),
					index_to);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_imagelist_set failed: %s",
					cpl_error_get_message_default(my_error));
			index_to++;
			cpl_image_delete(ext_image);
			my_error = cpl_error_get_code();
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_delete failed: %s",
					cpl_error_get_message_default(my_error));
		}
	}

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief		Verify FITS frame
 @param     FITS_filename   FITS frame filename
 @return	CPL_ERROR_NONE iff OK

 Verify if the image is not NULL, has the right number of extensions, is of the right size and computes the mean. */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_verify_fits_image(const char *FITS_filename) {

	int i;
	int extension_nb, size_x, size_y;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(FITS_filename == NULL, CPL_ERROR_NULL_INPUT,
			"FITS filename is NULL");

	espdr_msg("File checked: %s", FITS_filename);

	extension_nb = cpl_fits_count_extensions(FITS_filename);
	espdr_msg("\tnb of extensions = %d", extension_nb);
	cpl_image *images_checked[extension_nb];

	for (i = 0; i < extension_nb; i++) {
		images_checked[i] = cpl_image_load(FITS_filename, CPL_TYPE_FLOAT, 0, i+1);
		if (images_checked[i] == NULL) {
			espdr_msg("The images[%d] is NULL", i);
		}
		size_x = cpl_image_get_size_x(images_checked[i]);
		size_y = cpl_image_get_size_y(images_checked[i]);
		espdr_msg("\tsize X = %d, size Y = %d", size_x, size_y);

		//image_mean = cpl_image_get_mean(images_checked[i]);
		//espdr_msg("\timage mean: %lf", image_mean);

		//image_min = cpl_image_get_min(images_checked[i]);
		//espdr_msg("\timage min: %lf", image_min);

		//image_max = cpl_image_get_max(images_checked[i]);
		//espdr_msg("\timage max: %lf", image_max);

		cpl_image_delete(images_checked[i]);
	}

	my_error = cpl_error_get_code();
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"espdr_verify_fits_image failed: %s",
			cpl_error_get_message_default(my_error));

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Verify imagelist
 @param     input_imagelist     input image, outputs in the imagelist
 @return	CPL_ERROR_NONE iff OK

 Verify if the image is not NULL, has the right number of extensions, is of the right size and computes the mean. */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_verify_imagelist(const cpl_imagelist *input_imagelist) {

	int i, size_x, size_y;
	const cpl_image *curr_image = NULL;
	double image_mean, image_min, image_max;

	for (i = 0; i < cpl_imagelist_get_size(input_imagelist); i++) {
		curr_image = cpl_imagelist_get_const(input_imagelist, i);
		espdr_msg("Image nr %d:", i);

		size_x = cpl_image_get_size_x(curr_image);
		size_y = cpl_image_get_size_y(curr_image);
		espdr_msg("\tsize X = %d, size Y = %d", size_x, size_y);

		image_mean = cpl_image_get_mean(curr_image);
		espdr_msg("\timage mean: %lf", image_mean);

		image_min = cpl_image_get_min(curr_image);
		espdr_msg("\timage min: %lf", image_min);

		image_max = cpl_image_get_max(curr_image);
		espdr_msg("\timage max: %lf", image_max);
	}

	return cpl_error_get_code();
}




/*---------------------------------------------------------------------------*/
/**
 @brief		Create a mask from image and bounds
 @param     input_image input image
 @param     total_size  image size
 @param     relect_low  lower limit
 @param     reject_high upper limit
 @param     mask_bit    value of the pixel mask
 @param     mask_RE     created mask (returned)
 @return	CPL_ERROR_NONE iff OK

 The function merges all the images form the list and saves it into one.
 The order of frames is first column, then row.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_create_mask(const cpl_image *input_image,
		const int total_size,
		const double reject_low,
		const double reject_high,
		const int mask_bit,
		cpl_image **mask_RE) {

	int i;
	const double *input_image_data = cpl_image_get_data_double_const(input_image);
	int *mask_data = cpl_image_get_data_int(*mask_RE);

	for (i = 0; i < total_size; i++) {
		if ((input_image_data[i] > reject_high) || (input_image_data[i] < reject_low)) {
			mask_data[i] = mask_bit;
		} else {
			mask_data[i] = 0;
		}
	}

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Compute RON and mean of an image
 @param     input_set       input frameset
 @param     CCD_geom        CCD geometry parameters structure
 @param     ksigma_low      ksigma low rejection
 @param     ksigma_high     ksigma high rejection
 @param     sigma_clipping_method   mean or median
 @param     iter_max        max number of sigma clipping iterations
 @param     reject_zeroMAD  flag to reject MAD = 0
 @param     forced_MAD      value for MAD if it is equal to 0.0
 @param[out]    RON_RE      computed RON
 @param[out]    mean_RE     computed mean
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_RON_compute(const cpl_frameset *input_set,
		const espdr_CCD_geometry *CCD_geom,
		const double ksigma_low,
		const double ksigma_high,
		const char *sigma_clipping_method,
		const int iter_max,
		const int reject_zeroMAD,
		const double forced_MAD,
		double *RON_RE,
		double *mean_RE) {

	int i,j,k,l;
	int ix,iy;
	int llx, lly, urx, ury;
	int real_llx, real_lly, real_urx, real_ury;
	int index;
	int size, data_size;
	double *image_data;
	double CoD, sigma, mean, stddev, reject_low, reject_high;
	int cosmics;
	cpl_error_code my_error = CPL_ERROR_NONE;
	const char* fname=NULL;
	const cpl_frame* frame=NULL;
	cpl_image* image=NULL;
	cpl_vector *data_vector_wrapped=NULL;
	espdr_ensure(input_set == NULL, CPL_ERROR_NULL_INPUT,
			"Input frameset is NULL");

	int total_output_nb = CCD_geom->total_output_nb;

	size = cpl_frameset_get_size(input_set);
	index = 0;
	for (l = 0; l < size; l++) {
		frame = cpl_frameset_get_position_const(input_set, l);
		fname = cpl_frame_get_filename(frame);


		for (i = 0; i < CCD_geom->ext_nb; i++) {

			ix = 0;
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				llx = ix+1;
				urx = ix+CCD_geom->exts[i].outputs[j][0].raw_nx;
				iy=0;
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					espdr_msg_debug("ext: %d, out_x: %d, out_y:%d", i, j, k);
					lly = iy+1;
					ury = iy+CCD_geom->exts[i].outputs[j][k].raw_ny;

					espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
							llx, lly, urx, ury);

					real_llx = llx + CCD_geom->exts[i].outputs[j][k].pscan_llx;
					real_lly = lly + CCD_geom->exts[i].outputs[j][k].ppscan_lly;
					real_urx = urx - CCD_geom->exts[i].outputs[j][k].pscan_urx;
					real_ury = ury - CCD_geom->exts[i].outputs[j][k].ppscan_ury;

					espdr_msg("real_llx = %d, real_lly = %d, real_urx = %d, real_ury = %d",
							real_llx, real_lly, real_urx, real_ury);

					image=cpl_image_load_window(fname, CPL_TYPE_DOUBLE,
							0, i+1, llx, lly, urx, ury);
					//image=cpl_image_load_window(fname, CPL_TYPE_DOUBLE,
					//                            0, i+1, real_llx, real_lly,
					//                            real_urx, real_ury);
					espdr_msg("fname=%s", fname);

					image_data = cpl_image_get_data_double(image);

					data_size = cpl_image_get_size_x(image) *
							cpl_image_get_size_y(image);

					data_vector_wrapped = cpl_vector_wrap(data_size, image_data);

					my_error = espdr_sig_clip(data_vector_wrapped,
							ksigma_low, ksigma_high,
							sigma_clipping_method,
							iter_max, reject_zeroMAD, forced_MAD,
							&CoD, &sigma, &cosmics,
							&mean, &stddev,
							&reject_low, &reject_high);

					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"espdr_sig_clip failed");
					cpl_vector_unwrap(data_vector_wrapped);
					RON_RE[index] = stddev;
					mean_RE[index] = mean;
					cpl_image_delete(image);
					index++;
					iy = ury;
				}
				ix=urx;
			}
		}
	}

	for (i = 0; i< total_output_nb; i++) {
		espdr_msg("RON_RE[%d] = %lf", i, RON_RE[i]);
		espdr_msg("mean_RE[%d] = %lf", i, mean_RE[i]);
	}

	//cpl_free(image_data);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_snr(cpl_image *s2d_flux,
                                 cpl_image *sd2_error,
                                 int snr_window,
                                 double *snr) {
    
    cpl_image *extr_spec;
    cpl_image *error_spec;
    double *extr_spec_data;
    double *error_spec_data;
    cpl_size size_x = cpl_image_get_size_x(s2d_flux);
    int orders_nr = cpl_image_get_size_y(s2d_flux);
    int central_point = size_x / 2;
    int range_ini =  -1*(int)snr_window/2;
    int range_end = (int)snr_window/2;
    double order_snr_sum = 0.0;
    
    for (int i = 0; i < orders_nr; i++) {
        
        extr_spec = cpl_image_extract(s2d_flux, 1, i+1, size_x, i+1);
        extr_spec_data = cpl_image_get_data_double(extr_spec);
        error_spec = cpl_image_extract(sd2_error, 1, i+1, size_x, i+1);
        error_spec_data = cpl_image_get_data_double(error_spec);
        
        order_snr_sum = 0.0;
        for (int j = range_ini; j <= range_end; j++){
            if (error_spec_data[central_point+j] != 0.) {
                order_snr_sum += extr_spec_data[central_point+j]/error_spec_data[central_point+j];
            }
        }
        snr[i] = order_snr_sum / snr_window;
        
        cpl_image_delete(extr_spec);
        cpl_image_delete(error_spec);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Get the output index for the given pixel
 @param     ext_nr      extension number
 @param     pixel       pixel value
 @param     CCD_geom    CCD geometry structure
 @return	output number in the sequence
 */
/*---------------------------------------------------------------------------*/

int espdr_get_output_index_for_pixel(const int ext_nr,
		const int pixel,
		const espdr_CCD_geometry *CCD_geom) {

	int x, y;
	int CCD_width = 0, CCD_height = 0;
	int output_width = 0, output_height = 0;
	int output_nb_x = 0, output_nb_y = 0;
	int out_x_index, out_x_limit, out_y_index, out_y_limit;
	int result;

	espdr_msg_debug("CCD_nx = %d, CCD_ny = %d",
			CCD_geom->exts[ext_nr].CCD_nx,
			CCD_geom->exts[ext_nr].CCD_ny);

	if ((CCD_geom->exts[ext_nr].rot_angle != 0) &&
			(CCD_geom->exts[ext_nr].rot_angle != 180)) {
		CCD_width = CCD_geom->exts[ext_nr].CCD_ny;
		CCD_height = CCD_geom->exts[ext_nr].CCD_nx;
	} else {
		CCD_width = CCD_geom->exts[ext_nr].CCD_nx;
		CCD_height = CCD_geom->exts[ext_nr].CCD_ny;
	}
	x = (pixel % CCD_width) + 1;
	y = (int)(pixel/CCD_width) + 1;
	//espdr_msg("pixel %d = [%d, %d]", pixel, x, y);

	espdr_ensure(x > CCD_width, CPL_ERROR_ILLEGAL_OUTPUT,
			"Pixel x coordinate: %d is out of CCD limits: %d",
			x, CCD_width);
	espdr_ensure(y > CCD_height, CPL_ERROR_ILLEGAL_OUTPUT,
			"Pixel y coordinate: %d is out of CCD limits: %d",
			y, CCD_height);

	out_x_limit = 0;
	out_x_index = 0;
	if ((CCD_geom->exts[ext_nr].rot_angle != 0) &&
			(CCD_geom->exts[ext_nr].rot_angle != 180)) {
		output_width = CCD_geom->exts[ext_nr].outputs[out_x_index][0].real_ny;
		output_height = CCD_geom->exts[ext_nr].outputs[out_x_index][0].real_nx;
		output_nb_x = CCD_geom->exts[ext_nr].out_nb_y;
		output_nb_y = CCD_geom->exts[ext_nr].out_nb_x;
	} else {
		output_width = CCD_geom->exts[ext_nr].outputs[out_x_index][0].real_nx;
		output_height = CCD_geom->exts[ext_nr].outputs[out_x_index][0].real_ny;
		output_nb_x = CCD_geom->exts[ext_nr].out_nb_x;
		output_nb_y = CCD_geom->exts[ext_nr].out_nb_y;
	}
	while (out_x_limit < x) {
		out_x_limit = out_x_limit + output_width;
		out_x_index++;
	}
	out_x_index--;
	out_y_limit = 0;
	out_y_index = 0;
	while (out_y_limit < y) {
		out_y_limit = out_y_limit + output_height;
		out_y_index++;
	}
	out_y_index--;

	//espdr_msg("out_x_index = %d, out_y_index = %d", out_x_index, out_y_index);

	//result = ext_nr * output_nb_x * output_nb_y +
	//        out_x_index * output_nb_y + out_y_index;
	result = ext_nr * output_nb_x * output_nb_y +
			out_y_index * output_nb_x + out_x_index;

	//espdr_msg("output index: %d", result);

	return (result);
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Get the extension number for general order number
 @param     order       order number
 @param     fibre_nr    fibre number
 @param     inst_config instrument configuration structure
 @param     CCD_geom    CCD geometry structure
 @return	extention number for the given order
 */
/*---------------------------------------------------------------------------*/

int espdr_get_ext_index_for_order(const int order,
		const int fibre_nr,
		const espdr_inst_config *inst_config,
		const espdr_CCD_geometry *CCD_geom) {

	int ext, total_orders_nb;

	ext = 0;
	total_orders_nb = inst_config->orders_nb[fibre_nr*CCD_geom->ext_nb];
	while ((order > total_orders_nb) && (ext < CCD_geom->ext_nb)) {
		ext++;
		total_orders_nb += inst_config->orders_nb[fibre_nr*CCD_geom->ext_nb+ext];

	}

	//espdr_msg("------->>>> total for fibre %d order %d: %d, ext = %d",
	//          fibre_nr, order, total_orders_nb, ext);

	return(ext);
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Get the number of order within the extension from the general number
 @param     order       order number
 @param     fibre_nr    fibre number
 @param     ext_nr      extension number to which the order belongs
 @param     inst_config instrument configuration structure
 @param     CCD_geom    CCD geometry structure
 @return	extention number for the given order
 */
/*---------------------------------------------------------------------------*/

int espdr_get_order_nr_in_ext(const int order,
		const int fibre_nr,
		const int ext_nr,
		const espdr_inst_config *inst_config,
		const espdr_CCD_geometry *CCD_geom) {

	int order_in_ext, total_orders_nb, ext;

	total_orders_nb = 0;
	for (ext = 0; ext < ext_nr; ext++) {
		total_orders_nb += inst_config->orders_nb[fibre_nr*CCD_geom->ext_nb+ext];
	}

	order_in_ext = order - total_orders_nb;

	return(order_in_ext);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Get the number of the first order in the extension
 @param     fibre_nr    fibre number
 @param     ext_nr      extension number to which the order belongs
 @param     inst_config instrument configuration structure
 @param     CCD_geom    CCD geometry structure
 @return    number for the first order
 */
/*---------------------------------------------------------------------------*/

int espdr_get_first_order(const int fibre_nr,
                          const int ext_nr,
                          const espdr_inst_config *inst_config,
                          const espdr_CCD_geometry *CCD_geom) {

    int first_order = 0, orders_nb = 0, i;
    
    for (i = 0; i < ext_nr; i++) {
        orders_nb += inst_config->orders_nb[fibre_nr*CCD_geom->ext_nb+i];
    }
    first_order = orders_nb + 1;
    
    return(first_order);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Get the number of the last order in the extension
 @param     fibre_nr    fibre number
 @param     ext_nr      extension number to which the order belongs
 @param     inst_config instrument configuration structure
 @param     CCD_geom    CCD geometry structure
 @return    number for the first order
 */
/*---------------------------------------------------------------------------*/

int espdr_get_last_order(const int fibre_nr,
                         const int ext_nr,
                         const espdr_inst_config *inst_config,
                         const espdr_CCD_geometry *CCD_geom) {

    int last_order = 0, orders_nb = 0, i;
    
    for (i = 0; i <= ext_nr; i++) {
        orders_nb += inst_config->orders_nb[fibre_nr*CCD_geom->ext_nb+i];
    }
    last_order = orders_nb;
    
    return(last_order);
}


/*---------------------------------------------------------------------------*/
/**
 *  @brief    Given a wavelenght order, it calculates the pixel's size in
 *            wavelength units.
 *  @param    wave             wave values array.
 *  @param    length           wave length.
 *  @param    pixel_size_RE    array with pixel size in wave units.
 *  @return   CPL_ERROR_NONE iff OK
 *      */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_lambda_pixel_size(const double *wave,
		const int length,
		double *pixel_size_RE) {
	int i;

	for (i = 1; i < length; i++) {
		pixel_size_RE[i] = wave[i] - wave[i-1];
	}

	pixel_size_RE[0] = pixel_size_RE[1];
	return (cpl_error_get_code());
}

/*---------------------------------------------------------------------------*/
/**
 *  @brief    its gives back the fibre index.
 *  @param    c char to be looked for.
 *  @return   element index, -1 if it is not found. 
 *      */
/*---------------------------------------------------------------------------*/
int getposition(char* c) {
	if(strcmp(c,"A") == 0) return 0;
	else if (strcmp(c,"B") == 0) return 1;
	else if (strcmp(c,"C") == 0) return 2;
	else if (strcmp(c,"D") == 0) return 3;
	else if (strcmp(c,"E") == 0) return 4;
	else return -1;
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Get min flux
 @param         input_image     input CCD corrected raw images
 @param         pixel_mask      hot & bad pixels mask
 @param         cosmics_bounds  cosmics limits
 @param[out]    min_flux_RE     min flux vector
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_min_flux_one_output(const cpl_image *input_image,
		const cpl_image *pixel_mask,
		const double cosmics_bounds,
		double *min_flux_RE) {

	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
			"Input image is NULL");
	espdr_ensure(pixel_mask == NULL, CPL_ERROR_NULL_INPUT,
			"Pixel mask is NULL");

	int j;
	int size_x, size_y;
	const cpl_image *curr_image = NULL;
	const cpl_image *curr_pixels = NULL;
	const double *image_data = NULL;
	const int *mask_data = NULL;
	double min_val = 0.0;

	curr_image = input_image;
	size_x = cpl_image_get_size_x(curr_image);
	size_y = cpl_image_get_size_y(curr_image);
	curr_pixels = pixel_mask;
	image_data = cpl_image_get_data_double_const(curr_image);
	mask_data = cpl_image_get_data_int_const(curr_pixels);

	min_val = image_data[0];
	for (j = 0; j < size_x*size_y; j++) {
		if (mask_data[j] == 0) {
			if (image_data[j] < cosmics_bounds) {
				if (image_data[j] < min_val) {
					min_val = image_data[j];
				}
			}
		}
	}

	*min_flux_RE = min_val;

	//cpl_image_delete(curr_image);
	//cpl_image_delete(curr_pixels);

	return (cpl_error_get_code());
}

/* HEADER MISSING !!!!! */

double min_val(const double* array, const int size, int *index_RE){
	// returns the minimum value of array
	double val = array[0];
	for (int i = 1; i < size; ++i) {
		//val = val <= array[i] ? val : array[i];
		if(array[i] < val) {
			val = array[i];
			*index_RE = i;
		}
	}
	return val;
}

/* HEADER MISSING !!!!! */

double max_val(const double* array, const int size){
	// returns the minimum value of array
	double val = array[0];
	for (int i = 1; i < size; ++i) {
		val = val >= array[i] ? val : array[i];
	}
	return val;
}

/* HEADER MISSING !!!!! */

double find_val(const double* array, const int size, const double key, int *index_RE) {

	double diff = fabs( key - array[0]);
	double val = 0;

	for (int a = 0; a < size; a++) {
		if (diff > fabs( key - array[a] )) {
			diff = fabs( key - array[a]);
			val = array[a];
			*index_RE = a;
		}
	}
	return val;
}



double convert_coord_to_dbl(const char *coord_read) {
    
    char *coord_trans = (char *)cpl_malloc(16 * sizeof(char));
    double deg, min, sec, coord_dbl = 0.0, coord_converted = 0.0;
    int hours = 0;
    
    //espdr_msg("Coord read: %s", coord_read);
    
    if (strstr(coord_read, "h") != NULL) {
        hours = 1;
    }
    
    strcpy(coord_trans, coord_read);
    if ((strstr(coord_trans, "h") != NULL) ||
        (strstr(coord_trans, "d") != NULL) ||
        (strstr(coord_trans, "m") != NULL) ||
        (strstr(coord_trans, "s") != NULL) ||
        (strstr(coord_trans, ":") != NULL)) {
        //espdr_msg("Coordinates contain h, d, m, s or : --> extraction of hours/degrees, minutes and seconds");
        deg = atof(strtok(coord_trans, "hd:"));
        min = atof(strtok(NULL, "m:"));
        sec = atof(strtok(NULL, "s:"));
        //espdr_msg("deg: %f, min: %f, sec: %f",
        //          deg, min, sec);
        if (deg < 0.0) {
            deg = 0.0 - deg;
        }
        coord_converted = deg + min/60.0 + sec/3600.0;
        if (coord_read[0] == '-') {
            coord_converted = 0.0 - coord_converted;
        }
    } else {
        //espdr_msg("Coordinates in double format: --> converting string to a double");
        coord_dbl = strtod(coord_read, NULL);
        if (hours) {
            coord_converted = convert_HHMMSS_to_HH(coord_dbl);
        } else {
            coord_converted = convert_DDMMSS_to_DD(coord_dbl);
        }
    }
    
    //espdr_msg("Uniformed coordinate: %f", coord_converted);
    
    cpl_free(coord_trans);
    
    return (coord_converted);
}


/* HEADER MISSING !!!!! */

double convert_HHMMSS_to_HH(double ra){

	double h, m, s, res;
	h = trunc(ra/10000.);
	m = trunc(ra/100.-h*100.);
	s = ra-m*100.-h*10000.;
	res = h+m/60.+s/3600.;

	return res;
}

/* HEADER MISSING !!!!! */

double convert_DDMMSS_to_DD(double dec){

	double d, m, s, res;
	d = trunc(dec/10000.);
	m = trunc(dec/100.-d*100.);
	s = dec-m*100.-d*10000.;
	res = d+m/60.+s/3600.;

	return res;
}

/* HEADER MISSING !!!!! */

const char *espdr_get_mask_name(const char *spec_type_id,
		const cpl_table *mask_lut) {

	int mask_lut_length = cpl_table_get_nrow(mask_lut);

	for(int i = 0 ; i < mask_lut_length ; i++) {

		if (strcmp(cpl_table_get_string(mask_lut, "SPEC_TYPE", i),
				spec_type_id) == 0) {
			return (cpl_table_get_string(mask_lut, "MASK_TABLE", i));
		}

	}
	return "NOT_FOUND";
}

/* HEADER MISSING !!!!! */

const char *espdr_get_default_spec_type(const espdr_inst_config *inst_config) {

	return inst_config->default_spec_type;

}


char *espdr_get_base_filename(const char *filename) {

	char *base_file_name;

	base_file_name = basename((char *)filename);

	return  base_file_name;
}


cpl_error_code espdr_split_by_slices(const cpl_image *input_s2d,
		const int slices_nb,
		cpl_imagelist **s2d_sliced_RE) {

	int pixels = cpl_image_get_size_x(input_s2d);
	int orders = cpl_image_get_size_y(input_s2d);
	int pir, pix, order, slice, new_order;

	cpl_image *slice_current;

	for(order=1; order <= orders; order++){

		slice = (order-1)%slices_nb;	
		slice_current = cpl_imagelist_get(*s2d_sliced_RE,slice);
		new_order = (order-1)/slices_nb+1;

		for(pix=1; pix <= pixels; pix++){

			cpl_image_set(slice_current,pix,new_order,
					cpl_image_get(input_s2d,pix,order,&pir));

		}
	}

	return (cpl_error_get_code());
}

cpl_error_code espdr_group_slices(const cpl_imagelist *s2d_sliced,
		const int slices_nb,
		cpl_image **s2d_RE) {

	int pixels = cpl_image_get_size_x(*s2d_RE);
	int orders = cpl_image_get_size_y(*s2d_RE);
	int pir, pix, order, slice, new_order;

	const cpl_image *slice_current;

	for(order=1; order <= orders; order++){

		slice = (order-1)%slices_nb;
		slice_current = cpl_imagelist_get_const(s2d_sliced,slice);
		new_order = (order-1)/slices_nb+1;

		for(pix=1; pix <= pixels; pix++){
			cpl_image_set(*s2d_RE,pix,order,
					cpl_image_get(slice_current,pix,new_order,&pir));
		}
	}

	return (cpl_error_get_code());
}

cpl_error_code espdr_match_orders(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE) {

	//cpl_error_code my_error = CPL_ERROR_NONE;

	int order_a = 0;
	int order_b = 0;
	int found_nb = 0;
	int pir;
	bool found = false;
	int sz_order_x;
	double *match_orders_data_A;
	double *match_orders_data_B;
	int pix;
	int orders_nb;

	sz_order_x = cpl_image_get_size_x(input_s2d[0]);
	orders_nb = cpl_image_get_size_y(input_s2d[0]);

	match_orders_data_A = cpl_image_get_data_double(s2d_match_RE[0]);
	match_orders_data_B = cpl_image_get_data_double(s2d_match_RE[1]);


	for (int fibre = 0; fibre < inst_config->fibres_nb-1; fibre++) {
		for (order_a = 0; order_a < orders_nb; order_a++) {
			for (order_b = 0; order_b < orders_nb; order_b++) {
				if (physical_orders_id[fibre][order_a] ==
						physical_orders_id[fibre+1][order_b]) {

					/* order in A & B fibres ==> calculates ratio */

					for (int i = 0; i < sz_order_x; i++ ) {
						pix = sz_order_x*order_a+i;

						//match_orders_data[pix]=cpl_image_get(
						//input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_A[pix]=cpl_image_get(
								input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_B[pix]=cpl_image_get(
								input_s2d[fibre+1],i+1,order_b+1,&pir);
					}

					found = true;
					found_nb++;

				} else if (found_nb == 0){
					found = false;

				}
			}
			if (!found) { /* order in A, but not in B fibre ==> fibre B = 0.0 */

				//espdr_msg("ASE DEBUG : order is in A, NOT in B : phys_id[%d][%d]==%d [%d][%d]==%d",
				//fibre,order_a,physical_orders_id[fibre][order_a],fibre+1,order_b,
				//physical_orders_id[fibre+1][order_b]);


				for (int i = 0; i < sz_order_x; i++ ) {
					pix = sz_order_x*order_a+i;
					//match_orders_data_A[pix]=0.0;
					match_orders_data_B[pix]=0.0;
				}

			}
			found_nb=0;
		}
	}

	return (cpl_error_get_code());
}

cpl_error_code espdr_match_orders_quality_map(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE) {

	//cpl_error_code my_error = CPL_ERROR_NONE;

	int order_a = 0;
	int order_b = 0;
	int found_nb = 0;
	int pir;
	bool found = false;
	int sz_order_x;
	int *match_orders_data_A;
	int *match_orders_data_B;
	//int pix_a, pix_b;
	int pix;
	int orders_nb;

	sz_order_x = cpl_image_get_size_x(input_s2d[0]);
	orders_nb = cpl_image_get_size_y(input_s2d[0]);

	match_orders_data_A = cpl_image_get_data_int(s2d_match_RE[0]);
	match_orders_data_B = cpl_image_get_data_int(s2d_match_RE[1]);


	for (int fibre = 0; fibre < inst_config->fibres_nb-1; fibre++) {
		for (order_a = 0; order_a < orders_nb; order_a++) {
			for (order_b = 0; order_b < orders_nb; order_b++) {
				if (physical_orders_id[fibre][order_a] ==
						physical_orders_id[fibre+1][order_b]) {

					/* order in A & B fibres ==> calculates ratio */

					for (int i = 0; i < sz_order_x; i++ ) {
						pix = sz_order_x*order_a+i;

						//match_orders_data[pix]=cpl_image_get(
						//input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_A[pix]=cpl_image_get(
								input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_B[pix]=cpl_image_get(
								input_s2d[fibre+1],i+1,order_b+1,&pir);
					}

					found = true;
					found_nb++;

				} else if (found_nb == 0){
					found = false;

				}
			}
			if(!found){ /* order in A, but not in B fibre ==> fibre B = 0.0 */

				//espdr_msg("ASE DEBUG : order is in A, NOT in B : phys_id[%d][%d]==%d [%d][%d]==%d",
				//fibre,order_a,physical_orders_id[fibre][order_a],fibre+1,order_b,
				//physical_orders_id[fibre+1][order_b]);


				for (int i = 0; i < sz_order_x; i++ ) {
					pix = sz_order_x*order_a+i;
					//match_orders_data_A[pix]=0.0;
					match_orders_data_B[pix]=0.0;
				}

			}
			found_nb=0;
		}
	}


	return (cpl_error_get_code());
}

cpl_error_code espdr_match_orders_drift_matrix(const int **physical_orders_id,
		const espdr_inst_config *inst_config,
		cpl_image **input_s2d,
		cpl_image **s2d_match_RE) {

	//cpl_error_code my_error = CPL_ERROR_NONE;

	int order_a = 0;
	int order_b = 0;
	int found_nb = 0;
	int pir;
	bool found = false;
	int sz_order_x;
	double *match_orders_data_A;
	double *match_orders_data_B;
	//double pix_a, pix_b;
	int pix;
	int orders_nb;
	//double prev,next;

	sz_order_x = cpl_image_get_size_x(input_s2d[0]);
	orders_nb = cpl_image_get_size_y(input_s2d[0]);

	match_orders_data_A = cpl_image_get_data_double(s2d_match_RE[0]);
	match_orders_data_B = cpl_image_get_data_double(s2d_match_RE[1]);

	for (int fibre = 0; fibre < inst_config->fibres_nb-1; fibre++) {
		for (order_a = 0; order_a < orders_nb; order_a++) {
			for (order_b = 0; order_b < orders_nb; order_b++) {
				if(physical_orders_id[fibre][order_a] ==
						physical_orders_id[fibre+1][order_b]) {

					/* order in A & B fibres ==> calculates ratio */

					for (int i = 0; i < sz_order_x; i++ ) {
						pix = sz_order_x*order_a+i;

						//match_orders_data[pix]=cpl_image_get(
						//input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_A[pix]=cpl_image_get(
								input_s2d[fibre],i+1,order_a+1,&pir);

						match_orders_data_B[pix]=cpl_image_get(
								input_s2d[fibre+1],i+1,order_b+1,&pir);
					}

					found = true;
					found_nb++;

				} else if (found_nb == 0){
					found = false;

				}
			}
			if (!found) { /* order in A, but not in B fibre ==> fibre B = 0.0 */

				// ASE: Uses the previous and next orders on B, if both exists,
				// then average them, if not, uses the existing one.

				//double prev,next;
				/*for (int i = 0; i < sz_order_x; i++ ) {
                                        pix = sz_order_x*order_a+i;

					//if(!isnan(match_orders_data_B[pix-1]))
					//prev = match_orders_data_B[pix-1];
					//else prev = match_orders_data_B[pix];
			if((i>0) && !isnan(cpl_image_get(input_s2d[fibre+1],i,order_b+1,&pir))) 
			prev = cpl_image_get(input_s2d[fibre+1],i,order_b+1,&pir);	
			else prev = cpl_image_get(input_s2d[fibre+1],i+1,order_b+1,&pir);

					//if(!isnan(match_orders_data_B[pix+1]))
					//next = match_orders_data_B[pix+1];
					//else next = match_orders_data_B[pix];
			if((i<sz_order_x) && !isnan(cpl_image_get(input_s2d[fibre+1],i+2,order_b+1,&pir))) 
			next = cpl_image_get(input_s2d[fibre+1],i+2,order_b+1,&pir);	
			else next = cpl_image_get(input_s2d[fibre+1],i+1,order_b+1,&pir);

					if(!isnan(prev) && !isnan(next)) {
                                        match_orders_data_A[pix]=(prev+next)/2.;
					} else {
					match_orders_data_A[pix]=cpl_image_get(
                                	input_s2d[fibre+1],i+1,order_b+1,&pir);	
					}

			}*/
				for (int i = 0; i < sz_order_x; i++ ) {
					pix = sz_order_x*order_a+i;
					//match_orders_data_A[pix]=0.0;
					//match_orders_data_B[pix]=0.0;
					match_orders_data_B[pix]=cpl_image_get(
							input_s2d[fibre],i+1,order_a+1,&pir);
				}
			}
			found_nb = 0;
		}
	}

	return (cpl_error_get_code());
}

double espdr_compute_n_air(const double lambda, const double t, const double p) {

	double n, n1, n2, n3, n31, n32, n33;

	n1 = 1e-6 * p * (1.0 + (1.049-0.0157*t) * 1e-6 * p) / 720.883;
	n2 = 1.0 + 0.003661 * t;
	n31 = 64.328;
	n32 = 29498.1/(146.0 - (10000.0/lambda) * (10000.0/lambda));
	n33 = 255.4/(41.0 - (10000.0/lambda) * (10000.0/lambda));
	n3 = n31 + n32 + n33;
	n = (n1 / n2) * n3 + 1.0;

	//espdr_msg("n_air = %lf", n);

	return (n);
}


