/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2020-07-01 14:47:28 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#include <espdr_cte.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the CTE for the whole overscan subtracted raw image
 @param     input_iml       input image, outputs in the imagelist
 @param     RON             RON computed on the overscan
 @param     CCD_geom        CCD_geometry structure
 @param     inst_config     instrument config
 @param[out] cte_iml_RE     computed CTE image
 @return   CPL_ERROR_NONE iff OK

 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_cte_compute(const cpl_imagelist *input_iml,
                                 double *RON,
                                 const espdr_CCD_geometry *CCD_geom,
                                 espdr_inst_config *inst_config,
                                 cpl_imagelist *cte_iml_RE) {

    int i, j, k, index, nx, ny;
    cpl_image *cte_image = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;

    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                
                espdr_msg("Computing CTE for ext %d out [%d, %d]", i, j, k);
                
                nx = CCD_geom->exts[i].outputs[j][k].real_nx;
                ny = CCD_geom->exts[i].outputs[j][k].real_ny;
                
                cte_image = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
                
                my_error = espdr_cte_one_output(cpl_imagelist_get_const(input_iml, index),
                                                RON[index],
                                                i, j, k,
                                                CCD_geom, inst_config,
                                                cte_image);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                               "cte_one_output failed: %s",
                               cpl_error_get_message_default(my_error));
                
                /* Save the result in an imagelist */
                my_error = cpl_imagelist_set(cte_iml_RE,
                                             cpl_image_duplicate(cte_image),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed: %s",
                             cpl_error_get_message_default(my_error));

                cpl_image_delete(cte_image);

                index++;
            }
        }
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the CTE for the whole overscan subtracted raw image
 @param     input_iml       input image, outputs in the imagelist
 @param     RON             RON computed on the overscan
 @param     CCD_geom        CCD_geometry structure
 @param     inst_config     instrument config
 @param[out] cte_iml_RE     computed CTE image
 @return   CPL_ERROR_NONE iff OK
 
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_cte_one_output(const cpl_image *input_image,
                                    double RON,
                                    int ext_no,
                                    int out_x,
                                    int out_y,
                                    const espdr_CCD_geometry *CCD_geom,
                                    espdr_inst_config *inst_config,
                                    cpl_image *cte_image_RE) {
    
    cpl_image *input_row_img = NULL;
    cpl_image *corr_row_img = NULL;
    double *row_data = NULL;
    double *adj_data = NULL;
    double *corr_data = NULL;
    double *delta_data = NULL;
    
    int collumn_orientation = CCD_geom->exts[ext_no].coll_or;
    int correction_type = CCD_geom->exts[ext_no].corr_type;
    int nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
    int ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
    
    double threshold = RON/2.0;
    double alpha = inst_config->cte_alpha[ext_no];
    double beta = inst_config->cte_beta[ext_no];

    /* the direction is set by collumn_orientation and correction_type */
    if (((collumn_orientation == 0) && (correction_type == 1)) ||
        ((collumn_orientation == 1) && (correction_type == 0))) {
        // parallel reading is vertical, serial is horizontal
        
        for (int i = 0; i < nx; i++) {
            input_row_img = cpl_image_extract(input_image, i+1, 1, i+1, ny);
            row_data = cpl_image_get_data_double(input_row_img);
            delta_data = (double *)cpl_calloc(ny, sizeof(double));
            corr_data = (double *)cpl_calloc(ny, sizeof(double));
            adj_data = (double *)cpl_calloc(ny, sizeof(double));
            for (int j = 0; j < ny; j++) {
                if (fabs(row_data[j]) < 0.1) {
                    adj_data[j] = threshold;
                } else {
                    adj_data[j] = row_data[j] / (1.0 - exp(-row_data[j]/threshold));
                }
                adj_data[j] = adj_data[j] + pow(alpha, 1.0/beta);
                delta_data[j] = row_data[j] - adj_data[j];
            }
            
            corr_data[0] = delta_data[0];
            for (int j = 1; j < ny; j++) {
                corr_data[j] = adj_data[j] +
                    adj_data[j] * (1.0 - pow(1.0 - alpha * pow(adj_data[j], -beta), j)) -
                    adj_data[j-1] * (1.0 - pow(1.0 - alpha * pow(adj_data[j-1], -beta), j-1));
                corr_data[j] = corr_data[j] + delta_data[j];
            }
            
            corr_row_img = cpl_image_wrap_double(1, ny, corr_data);
            cpl_image_copy(cte_image_RE, corr_row_img, i+1, 1);
            cpl_image_unwrap(corr_row_img);
            cpl_free(adj_data);
            cpl_free(corr_data);
            cpl_free(delta_data);
            cpl_image_delete(input_row_img);
        }
        
    } else {
        if (((collumn_orientation == 1) && (correction_type == 1)) ||
            ((collumn_orientation == 0) && (correction_type == 0))) {
            // parallel reading is horizontal, serial is vertical

            for (int i = 0; i < ny; i++) {
                input_row_img = cpl_image_extract(input_image, 1, i+1, nx, i+1);
                row_data = cpl_image_get_data_double(input_row_img);
                delta_data = (double *)cpl_calloc(nx, sizeof(double));
                corr_data = (double *)cpl_calloc(nx, sizeof(double));
                adj_data = (double *)cpl_calloc(nx, sizeof(double));
                for (int j = 0; j < nx; j++) {
                    if (fabs(row_data[j]) < 0.1) {
                        adj_data[j] = threshold;
                    } else {
                        adj_data[j] = row_data[j] / (1.0 - exp(-row_data[j]/threshold));
                    }
                    adj_data[j] = adj_data[j] + pow(alpha, 1.0/beta);
                    delta_data[j] = row_data[j] - adj_data[j];
                }
                
                corr_data[0] = delta_data[0];
                for (int j = 1; j < nx; j++) {
                    corr_data[j] = adj_data[j] +
                        adj_data[j] * (1.0 - pow(1.0 - alpha * pow(adj_data[j], -beta), j)) -
                        adj_data[j-1] * (1.0 - pow(1.0 - alpha * pow(adj_data[j-1], -beta), j-1));
                    corr_data[j] = corr_data[j] + delta_data[j];
                }
                
                corr_row_img = cpl_image_wrap_double(nx, 1, corr_data);
                cpl_image_copy(cte_image_RE, corr_row_img, 1, i+1);
                cpl_image_unwrap(corr_row_img);
                cpl_free(adj_data);
                cpl_free(corr_data);
                cpl_free(delta_data);
                cpl_image_delete(input_row_img);
            }
            
        } else {
            espdr_ensure(0, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Wrong combination of collumn_orientation and correction_type, exiting");
        }
    }

    return cpl_error_get_code();
}


