/*
 * espdr_sigma_clipping.c
 *
 *  Created on: Mar 29, 2012
 *      Author: danuta
 */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

/*---------------------------------------------------------------------------*/
/**
 * Function to calculate the sigma clipping
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <hdrl.h>

/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/


/* Utility fonctions */
#include <espdr_sigma_clipping.h>


/*----------------------------------------------------------------------------
                              Functions code
 ----------------------------------------------------------------------------*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief get first index that compares greater than value
 * @param vec vector to check
 * @param val upper bound to check
 * @return upper bound
 */
/* ---------------------------------------------------------------------------*/
static long get_upper_bound(cpl_vector * vec, double val)
{
    double * d = cpl_vector_get_data(vec);
    long count = cpl_vector_get_size(vec);
    long first = 0, it;
    while (count>0)
    {
        long step = count / 2;
        it = first + step;
        if (!(val < d[it])) {
            first = ++it;
            count -= step + 1;
        }
        else
            count = step;
    }
    return first;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief get index that compares does not compare less than value
 * @param vec vector to check
 * @param val lower bound to check
 * @return lower bound
 */
/* ---------------------------------------------------------------------------*/
static long get_lower_bound(cpl_vector * vec, double val)
{
    double * d = cpl_vector_get_data(vec);
    long count = cpl_vector_get_size(vec);
    long first = 0, it;
    while (count>0)
    {
        long step = count / 2;
        it = first + step;
        if (d[it] < val) {
            first = ++it;
            count -= step + 1;
        }
        else
            count = step;
    }
    return first;
}


/*----------------------------------------------------------------------------*/
/**
 @brief	Take off the outsiders using kappa-sigma clipping.
 @param	data_vector		The vector to take off the outsiders from.
 @param	ksig_low		Number of sigma for lower threshold.
 @param	ksig_high		Number of sigmas for upper threshold.
 @param	method			Method used: "mean" or "median".
 @param	maxiter			Max number of iterations of clipping.
 @param	reject_zeroMAD	Flag to reject sigma=0 for MAD.
 @param forced_MAD      value to force the MAD if it is equal to 0.0
 @param[out]	CoD_RE			The center of clipped data (mean or median).
 @param[out]	sigma_RE		The sigma of clipped data (stddev or mad).
 @param[out]	cosmics_RE		Number of rejected values.
 @param[out]    mean_RE         mean value
 @param[out]    stdev_RE        stdev vale
 @param[out]	reject_low_RE	Values lower than this have been rejected.
 @param[out]	reject_high_RE	Values higher than this have been rejected.
 @return    CPL_ERROR_NONE or the appropriate error code.
 *
 *TODO: remove commented out code
 *----------------------------------------------------------------------------*/

cpl_error_code espdr_sig_clip (cpl_vector *data_vector,
                               double ksig_low,
                               double ksig_high,
                               const char *method,
                               int maxiter,
                               int reject_zeroMAD,
                               double forced_MAD,
                               double *CoD_RE,
                               double *sigma_RE,
                               int *cosmics_RE,
                               double *mean_RE,
                               double *stddev_RE,
                               double *reject_low_RE,
                               double *reject_high_RE) {

	if(reject_zeroMAD == 0) {
		/* Do nothing Just to document this parameter is not used*/
	}

	/* loop indices */
	int i;
	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* mean or MAD value */
	double center_of_data;
	/* sigma of the distribution */
	double sigma;
	/* vectors on which outliers are computed */
	cpl_vector *working_data_vector = NULL;
	cpl_vector *swap_data_vector = NULL;
	/* vector clipping bounds */
	double lower_bound = 0.0;
	double upper_bound = 0.0;
	cpl_size lower_index;
	cpl_size upper_index;
	cpl_size vec_size;

	//espdr_msg("ksig_low = %lf, ksig_high = %lf", ksig_low, ksig_high);
	/* Check input vector */
	espdr_ensure(data_vector == NULL, CPL_ERROR_NULL_INPUT,
				 "Input vector is NULL");
	
	/* Check the method */
	espdr_ensure((strcmp(method, "mean") != 0) && (strcmp(method, "median") != 0),
                 CPL_ERROR_TYPE_MISMATCH, "No such method: %s, exiting.", method);
	working_data_vector = cpl_vector_duplicate(data_vector);
	
	my_error = cpl_vector_sort(working_data_vector, CPL_SORT_ASCENDING);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Vector sort failed");
	
    
	/* Do the sigma clipping (at most maxiter times) */
	for (i = 0; i < maxiter; i++) {
        /* Nothing to do if only one data point */
        vec_size = cpl_vector_get_size(working_data_vector);
        if(vec_size == 1) {
            break;
        }
		
		if (strcmp(method, "mean") == 0) {
			center_of_data = cpl_vector_get_mean(working_data_vector);
			sigma = cpl_vector_get_stdev(working_data_vector);
		}
		if (strcmp(method, "median") == 0) {
			center_of_data = cpl_vector_get_median(working_data_vector);
            /* cpl_image_get_mad() takes the mean of two middle values 
               for vectors of even length */
			cpl_image *madimage =
            cpl_image_wrap_double(1,
                                  cpl_vector_get_size(working_data_vector),
                                  cpl_vector_get_data((cpl_vector*)working_data_vector));
			center_of_data = cpl_image_get_mad(madimage, &sigma);
			cpl_image_unwrap(madimage);
            
            /* New version of MAD control, working for all kind of vectors */
            if (sigma == 0.0) {
                sigma = forced_MAD;
            }
            
            /* Make MAD a real sigma */
			sigma = sigma * CPL_MATH_STD_MAD;
		}
		
		lower_bound = center_of_data - ksig_low * sigma;
		upper_bound = center_of_data + ksig_high * sigma;
		
		lower_index = get_lower_bound(working_data_vector, lower_bound);
		upper_index = get_upper_bound(working_data_vector, upper_bound);
		upper_index = CX_MAX(upper_index - 1, 0);
		
		
		/* Stop if no outliers were found */
		if ((lower_index == 0) && (upper_index == vec_size - 1)) {
			break;
		}
		

		swap_data_vector = cpl_vector_extract(working_data_vector,
											  lower_index, upper_index, 1);
		
		cpl_vector_set_size(working_data_vector,
							cpl_vector_get_size(swap_data_vector));
		
		cpl_vector_copy(working_data_vector, swap_data_vector);
		cpl_vector_delete(swap_data_vector);
        
	}

	if (strcmp(method, "mean") == 0) {
		center_of_data = cpl_vector_get_mean(working_data_vector);
		sigma = cpl_vector_get_stdev(working_data_vector);
	}
	if (strcmp(method, "median") == 0) {
		center_of_data = cpl_vector_get_median(working_data_vector);
		cpl_image *madimage =
		cpl_image_wrap_double(1,
							  cpl_vector_get_size(working_data_vector),
							  cpl_vector_get_data((cpl_vector*)working_data_vector));
		center_of_data = cpl_image_get_mad(madimage, &sigma);
		cpl_image_unwrap(madimage);
        
        /* New version of MAD control, working for all kind of vectors */
        if (sigma == 0.0) {
            sigma = forced_MAD;
        }
        
        /* Make MAD a real sigma */
        sigma = sigma * CPL_MATH_STD_MAD;
	}
    
	*mean_RE = cpl_vector_get_mean(working_data_vector);
	*stddev_RE = cpl_vector_get_stdev(working_data_vector);
	
	*CoD_RE = center_of_data;
	*sigma_RE = sigma;
	*cosmics_RE = cpl_vector_get_size(data_vector) - cpl_vector_get_size(working_data_vector);
	*reject_low_RE = center_of_data - ksig_low * sigma;
	*reject_high_RE = center_of_data + ksig_high * sigma;
	
	/* Free space */
	cpl_vector_delete(working_data_vector);
	
    
	return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief	Take off the outsiders using kappa-sigma clipping.
 @param	data_vector		The vector to take off the outsiders from.
 @param	ksig_low		Number of sigma for lower threshold.
 @param	ksig_high		Number of sigmas for upper threshold.
 @param	maxiter			Max number of iterations of clipping.
 @param	reject_zeroMAD	Flag to reject sigma=0 for MAD.
 @param forced_MAD      value to force the MAD if it is equal to 0.0
 @param[out]	CoD_RE			The center of clipped data (mean or median).
 @param[out]	sigma_RE		The sigma of clipped data (stddev or mad).
 @param[out]	cosmics_RE		Number of rejected values.
 @param[out]    mean_RE         mean value
 @param[out]    stdev_RE        stdev vale
 @param[out]	reject_low_RE	Values lower than this have been rejected.
 @param[out]	reject_high_RE	Values higher than this have been rejected.
 @return    CPL_ERROR_NONE or the appropriate error code.
 *
 *TODO: remove commented out code
 *----------------------------------------------------------------------------*/

cpl_error_code espdr_sig_clip_method_median (cpl_vector *data_vector,
                               double ksig_low,
                               double ksig_high,
                               int maxiter,
                               int reject_zeroMAD,
                               double forced_MAD,
                               double *CoD_RE,
                               double *sigma_RE,
                               int *cosmics_RE,
                               double *mean_RE,
                               double *stddev_RE,
                               double *reject_low_RE,
                               double *reject_high_RE) {

	if(reject_zeroMAD == 0) {
		/* Do nothing Just to document this parameter is not used*/
	}

	/* loop indices */
	int i;
	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* mean or MAD value */
	double center_of_data;
	/* sigma of the distribution */
	double sigma;
	/* vectors on which outliers are computed */
	cpl_vector *working_data_vector = NULL;
	cpl_vector *swap_data_vector = NULL;
	/* vector clipping bounds */
	double lower_bound = 0.0;
	double upper_bound = 0.0;
	cpl_size lower_index;
	cpl_size upper_index;
	cpl_size vec_size;

	//espdr_msg("ksig_low = %lf, ksig_high = %lf", ksig_low, ksig_high);
	/* Check input vector */
	espdr_ensure(data_vector == NULL, CPL_ERROR_NULL_INPUT,
				 "Input vector is NULL");

	working_data_vector = cpl_vector_duplicate(data_vector);

	my_error = cpl_vector_sort(working_data_vector, CPL_SORT_ASCENDING);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Vector sort failed");


	/* Do the sigma clipping (at most maxiter times) */
	for (i = 0; i < maxiter; i++) {
        /* Nothing to do if only one data point */
        vec_size = cpl_vector_get_size(working_data_vector);
        if(vec_size == 1) {
            break;
        }


		{ /* method median */
			center_of_data = cpl_vector_get_median(working_data_vector);
            /* cpl_image_get_mad() takes the mean of two middle values
               for vectors of even length */
			cpl_image *madimage =
            cpl_image_wrap_double(1,
                                  cpl_vector_get_size(working_data_vector),
                                  cpl_vector_get_data((cpl_vector*)working_data_vector));
			center_of_data = cpl_image_get_mad(madimage, &sigma);
			cpl_image_unwrap(madimage);

            /* New version of MAD control, working for all kind of vectors */
            if (sigma == 0.0) {
                sigma = forced_MAD;
            }

            /* Make MAD a real sigma */
			sigma = sigma * CPL_MATH_STD_MAD;
		}

		lower_bound = center_of_data - ksig_low * sigma;
		upper_bound = center_of_data + ksig_high * sigma;

		lower_index = get_lower_bound(working_data_vector, lower_bound);
		upper_index = get_upper_bound(working_data_vector, upper_bound);
		upper_index = CX_MAX(upper_index - 1, 0);


		/* Stop if no outliers were found */
		if ((lower_index == 0) && (upper_index == vec_size - 1)) {
			break;
		}


		swap_data_vector = cpl_vector_extract(working_data_vector,
											  lower_index, upper_index, 1);

		cpl_vector_set_size(working_data_vector,
							cpl_vector_get_size(swap_data_vector));

		cpl_vector_copy(working_data_vector, swap_data_vector);
		cpl_vector_delete(swap_data_vector);

	}


	{ /* method median */
		center_of_data = cpl_vector_get_median(working_data_vector);
		cpl_image *madimage =
		cpl_image_wrap_double(1,
							  cpl_vector_get_size(working_data_vector),
							  cpl_vector_get_data((cpl_vector*)working_data_vector));
		center_of_data = cpl_image_get_mad(madimage, &sigma);
		cpl_image_unwrap(madimage);

        /* New version of MAD control, working for all kind of vectors */
        if (sigma == 0.0) {
            sigma = forced_MAD;
        }

        /* Make MAD a real sigma */
        sigma = sigma * CPL_MATH_STD_MAD;
	}

	*mean_RE = cpl_vector_get_mean(working_data_vector);
	*stddev_RE = cpl_vector_get_stdev(working_data_vector);

	*CoD_RE = center_of_data;
	*sigma_RE = sigma;
	*cosmics_RE = cpl_vector_get_size(data_vector) - cpl_vector_get_size(working_data_vector);
	*reject_low_RE = center_of_data - ksig_low * sigma;
	*reject_high_RE = center_of_data + ksig_high * sigma;

	/* Free space */
	cpl_vector_delete(working_data_vector);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief	Take off the outsiders using kappa-sigma clipping.
 @param	data_vector		The vector to take off the outsiders from.
 @param	ksig_low		Number of sigma for lower threshold.
 @param	ksig_high		Number of sigmas for upper threshold.
 @param	maxiter			Max number of iterations of clipping.
 @param	reject_zeroMAD	Flag to reject sigma=0 for MAD.
 @param forced_MAD      value to force the MAD if it is equal to 0.0
 @param[out]	CoD_RE			The center of clipped data (mean or median).
 @param[out]	sigma_RE		The sigma of clipped data (stddev or mad).
 @param[out]	cosmics_RE		Number of rejected values.
 @param[out]    mean_RE         mean value
 @param[out]    stdev_RE        stdev vale
 @param[out]	reject_low_RE	Values lower than this have been rejected.
 @param[out]	reject_high_RE	Values higher than this have been rejected.
 @return    CPL_ERROR_NONE or the appropriate error code.
 *
 *TODO: remove commented out code
 *----------------------------------------------------------------------------*/

cpl_error_code espdr_sig_clip_method_mean (cpl_vector *data_vector,
                               double ksig_low,
                               double ksig_high,
                               int maxiter,
                               int reject_zeroMAD,
                               double forced_MAD,
                               double *CoD_RE,
                               double *sigma_RE,
                               int *cosmics_RE,
                               double *mean_RE,
                               double *stddev_RE,
                               double *reject_low_RE,
                               double *reject_high_RE) {

	if(reject_zeroMAD == 0) {
		/* Do nothing Just to document this parameter is not used*/
	}

	/* loop indices */
	int i;
	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* mean or MAD value */
	double center_of_data;
	/* sigma of the distribution */
	double sigma;
	/* vectors on which outliers are computed */
	cpl_vector *working_data_vector = NULL;
	cpl_vector *swap_data_vector = NULL;
	/* vector clipping bounds */
	double lower_bound = 0.0;
	double upper_bound = 0.0;
	cpl_size lower_index;
	cpl_size upper_index;
	cpl_size vec_size;

	//espdr_msg("ksig_low = %lf, ksig_high = %lf", ksig_low, ksig_high);
	/* Check input vector */
	espdr_ensure(data_vector == NULL, CPL_ERROR_NULL_INPUT,
				 "Input vector is NULL");

	working_data_vector = cpl_vector_duplicate(data_vector);

	my_error = cpl_vector_sort(working_data_vector, CPL_SORT_ASCENDING);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Vector sort failed");


	/* Do the sigma clipping (at most maxiter times) */
	for (i = 0; i < maxiter; i++) {
        /* Nothing to do if only one data point */
        vec_size = cpl_vector_get_size(working_data_vector);
        if(vec_size == 1) {
            break;
        }

		{ /* method mean */
			center_of_data = cpl_vector_get_mean(working_data_vector);
			sigma = cpl_vector_get_stdev(working_data_vector);
		}

		lower_bound = center_of_data - ksig_low * sigma;
		upper_bound = center_of_data + ksig_high * sigma;

		lower_index = get_lower_bound(working_data_vector, lower_bound);
		upper_index = get_upper_bound(working_data_vector, upper_bound);
		upper_index = CX_MAX(upper_index - 1, 0);


		/* Stop if no outliers were found */
		if ((lower_index == 0) && (upper_index == vec_size - 1)) {
			break;
		}


		swap_data_vector = cpl_vector_extract(working_data_vector,
											  lower_index, upper_index, 1);

		cpl_vector_set_size(working_data_vector,
							cpl_vector_get_size(swap_data_vector));

		cpl_vector_copy(working_data_vector, swap_data_vector);
		cpl_vector_delete(swap_data_vector);

	}

	{ /* method mean */
		center_of_data = cpl_vector_get_mean(working_data_vector);
		sigma = cpl_vector_get_stdev(working_data_vector);
	}

	*mean_RE = cpl_vector_get_mean(working_data_vector);
	*stddev_RE = cpl_vector_get_stdev(working_data_vector);

	*CoD_RE = center_of_data;
	*sigma_RE = sigma;
	*cosmics_RE = cpl_vector_get_size(data_vector) - cpl_vector_get_size(working_data_vector);
	*reject_low_RE = center_of_data - ksig_low * sigma;
	*reject_high_RE = center_of_data + ksig_high * sigma;

	/* Free space */
	cpl_vector_delete(working_data_vector);

	return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief	Take off the outsiders using kappa-sigma clipping.
 @param	data_vector		The vector to take off the outsiders from.
 @param	ksig_low		Number of sigma for lower threshold.
 @param	ksig_high		Number of sigmas for upper threshold.
 @param	maxiter			Max number of iterations of clipping.
 @param	reject_zeroMAD	Flag to reject sigma=0 for MAD.
 @param forced_MAD      value to force the MAD if it is equal to 0.0
 @param[out]	CoD_RE			The center of clipped data (mean or median).
 @param[out]	sigma_RE		The sigma of clipped data (stddev or mad).
 @param[out]	cosmics_RE		Number of rejected values.
 @param[out]    mean_RE         mean value
 @param[out]    stdev_RE        stdev vale
 @param[out]	reject_low_RE	Values lower than this have been rejected.
 @param[out]	reject_high_RE	Values higher than this have been rejected.
 @return    CPL_ERROR_NONE or the appropriate error code.
 *
 *TODO: remove commented out code
 *----------------------------------------------------------------------------*/

cpl_error_code espdr_sig_clip_fast_method_median (cpl_vector *data_vector,
		double ksig_low,
		double ksig_high,
		int maxiter,
		int reject_zeroMAD,
		double forced_MAD,
		double *CoD_RE,
		double *sigma_RE,
		int *cosmics_RE,
		double *mean_RE,
		double *stddev_RE,
		double *reject_low_RE,
		double *reject_high_RE) {

	if(reject_zeroMAD == 0) {
		/* Do nothing Just to document this parameter is not used*/
	}

	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* mean or MAD value */
	double center_of_data;
	/* sigma of the distribution */
	double sigma;
	/* vectors on which outliers are computed */
	cpl_vector *working_data_vector = NULL;
	cpl_vector *swap_data_vector = NULL;
	/* vector clipping bounds */
	double lower_bound = 0.0;
	double upper_bound = 0.0;
	cpl_size lower_index;
	cpl_size upper_index;
	cpl_size vec_size;

	/* Check input vector */
	espdr_ensure(data_vector == NULL, CPL_ERROR_NULL_INPUT,
			"Input vector is NULL");

	working_data_vector = cpl_vector_duplicate(data_vector);

	my_error = cpl_vector_sort(working_data_vector, CPL_SORT_ASCENDING);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Vector sort failed");

	vec_size = cpl_vector_get_size(working_data_vector);
	if(vec_size == 1) {
		/* Nothing to do if only one data point */
	} else {
		/* Do the sigma clipping (at most maxiter times) */
		for (int i = 0; i < maxiter; i++) {

			{ /* case method median */
				center_of_data = cpl_vector_get_median(working_data_vector);
				/* cpl_image_get_mad() takes the mean of two middle values
               for vectors of even length */
				cpl_image *madimage =
						cpl_image_wrap_double(1,
								cpl_vector_get_size(working_data_vector),
								cpl_vector_get_data(working_data_vector));
				center_of_data = cpl_image_get_mad(madimage, &sigma);
				cpl_image_unwrap(madimage);

				/* New version of MAD control, working for all kind of vectors */
				if (sigma == 0.0) {
					sigma = forced_MAD;
				}

				/* Make MAD a real sigma */
				sigma = sigma * CPL_MATH_STD_MAD;
			}

			lower_bound = center_of_data - ksig_low * sigma;
			upper_bound = center_of_data + ksig_high * sigma;

			lower_index = get_lower_bound(working_data_vector, lower_bound);
			upper_index = get_upper_bound(working_data_vector, upper_bound);
			upper_index = CX_MAX(upper_index - 1, 0);

			/* Stop if no outliers were found */
			if ((lower_index == 0) && (upper_index == vec_size - 1)) {
				break;
			}

			swap_data_vector = cpl_vector_extract(working_data_vector,
					lower_index, upper_index, 1);

			cpl_vector_set_size(working_data_vector,
					cpl_vector_get_size(swap_data_vector));

			cpl_vector_copy(working_data_vector, swap_data_vector);
			cpl_vector_delete(swap_data_vector);

		} /* end for loop */
	}

	{ /* case method median */
		center_of_data = cpl_vector_get_median(working_data_vector);
		cpl_image *madimage =
				cpl_image_wrap_double(1,
						cpl_vector_get_size(working_data_vector),
						cpl_vector_get_data((cpl_vector*)working_data_vector));
		center_of_data = cpl_image_get_mad(madimage, &sigma);
		cpl_image_unwrap(madimage);

		/* New version of MAD control, working for all kind of vectors */
		if (sigma == 0.0) {
			sigma = forced_MAD;
		}

		/* Make MAD a real sigma */
		sigma = sigma * CPL_MATH_STD_MAD;
	}

	*mean_RE = cpl_vector_get_mean(working_data_vector);
	*stddev_RE = cpl_vector_get_stdev(working_data_vector);

	*CoD_RE = center_of_data;
	*sigma_RE = sigma;
	*cosmics_RE = cpl_vector_get_size(data_vector) -
			cpl_vector_get_size(working_data_vector);

	*reject_low_RE = center_of_data - ksig_low * sigma;
	*reject_high_RE = center_of_data + ksig_high * sigma;

	/* Free space */
	cpl_vector_delete(working_data_vector);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief	Take off the outsiders using kappa-sigma clipping.
 @param	data_vector		The vector to take off the outsiders from.
 @param	ksig_low		Number of sigma for lower threshold.
 @param	ksig_high		Number of sigmas for upper threshold.
 @param	maxiter			Max number of iterations of clipping.
 @param	reject_zeroMAD	Flag to reject sigma=0 for MAD.
 @param forced_MAD      value to force the MAD if it is equal to 0.0
 @param[out]	CoD_RE			The center of clipped data (mean or median).
 @param[out]	sigma_RE		The sigma of clipped data (stddev or mad).
 @param[out]	cosmics_RE		Number of rejected values.
 @param[out]    mean_RE         mean value
 @param[out]    stdev_RE        stdev vale
 @param[out]	reject_low_RE	Values lower than this have been rejected.
 @param[out]	reject_high_RE	Values higher than this have been rejected.
 @return    CPL_ERROR_NONE or the appropriate error code.
 *
 *TODO: remove commented out code
 *----------------------------------------------------------------------------*/

cpl_error_code espdr_sig_clip_fast_method_mean (cpl_vector *data_vector,
		double ksig_low,
		double ksig_high,
		int maxiter,
		int reject_zeroMAD,
		double forced_MAD,
		double *CoD_RE,
		double *sigma_RE,
		int *cosmics_RE,
		double *mean_RE,
		double *stddev_RE,
		double *reject_low_RE,
		double *reject_high_RE) {

	if(reject_zeroMAD == 0) {
		/* Do nothing Just to document this parameter is not used*/
	}


	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* mean or MAD value */
	double center_of_data;
	/* sigma of the distribution */
	double sigma;
	/* vectors on which outliers are computed */
	cpl_vector *working_data_vector = NULL;
	cpl_vector *swap_data_vector = NULL;
	/* vector clipping bounds */
	double lower_bound = 0.0;
	double upper_bound = 0.0;
	cpl_size lower_index;
	cpl_size upper_index;
	cpl_size vec_size;

	/* Check input vector */
	espdr_ensure(data_vector == NULL, CPL_ERROR_NULL_INPUT,
			"Input vector is NULL");

	working_data_vector = cpl_vector_duplicate(data_vector);

	my_error = cpl_vector_sort(working_data_vector, CPL_SORT_ASCENDING);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Vector sort failed");

	vec_size = cpl_vector_get_size(working_data_vector);
	if(vec_size == 1) {
		/* Nothing to do if only one data point */
	} else {
		/* Do the sigma clipping (at most maxiter times) */
		for (int i = 0; i < maxiter; i++) {

			{ /* case method mean */
				center_of_data = cpl_vector_get_mean(working_data_vector);
				sigma = cpl_vector_get_stdev(working_data_vector);
			}

			lower_bound = center_of_data - ksig_low * sigma;
			upper_bound = center_of_data + ksig_high * sigma;

			lower_index = get_lower_bound(working_data_vector, lower_bound);
			upper_index = get_upper_bound(working_data_vector, upper_bound);
			upper_index = CX_MAX(upper_index - 1, 0);

			/* Stop if no outliers were found */
			if ((lower_index == 0) && (upper_index == vec_size - 1)) {
				break;
			}

			swap_data_vector = cpl_vector_extract(working_data_vector,
					lower_index, upper_index, 1);

			cpl_vector_set_size(working_data_vector,
					cpl_vector_get_size(swap_data_vector));

			cpl_vector_copy(working_data_vector, swap_data_vector);
			cpl_vector_delete(swap_data_vector);

		} /* end for loop */
	}

	{ /* case method mean */
		center_of_data = cpl_vector_get_mean(working_data_vector);
		sigma = cpl_vector_get_stdev(working_data_vector);
	}

	*mean_RE = cpl_vector_get_mean(working_data_vector);
	*stddev_RE = cpl_vector_get_stdev(working_data_vector);

	*CoD_RE = center_of_data;
	*sigma_RE = sigma;
	*cosmics_RE = cpl_vector_get_size(data_vector) -
			cpl_vector_get_size(working_data_vector);

	*reject_low_RE = center_of_data - ksig_low * sigma;
	*reject_high_RE = center_of_data + ksig_high * sigma;

	/* Free space */
	cpl_vector_delete(working_data_vector);

	return cpl_error_get_code();
}

cpl_error_code
espdr_get_master_and_cosmics_parallel(double ** images_table,
		const int frames_nb,
		const int nx, const int ny,
		const double ksig_low,
		const double ksig_high,
		const char* method,
		cpl_image **master_image_RE,int *totalCosmics_RE)
{
	double *data_vector;
	int point = 0;
	int i = 0;
	int image_size=nx*ny;
	int reject_zeroMAD = 1;
	double forced_MAD = 1.0;
	double mean = 0.0;
	double stddev = 0.0;
	double sigma=0;
	double CoD = 0.0;

	double reject_low = -1.0;
	double reject_high = -1.0;
	int cosmics = 0;
	int total_cosmics = 0;
	double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
	double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
	cpl_error_code my_error;

    HDRL_OMP(omp parallel private(data_vector,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
	{
		// vector of the same pixel in all the frames
		data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
		cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);

        HDRL_OMP(omp for)
		for (point = 0; point < image_size; point++) {
			// vector of the same pixel in all the frames
			//data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
			for (i = 0; i < frames_nb; i++) {
				data_vector[i] = images_table[i][point];
			}
			//cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
			my_error = espdr_sig_clip(data_vector_wrapped, ksig_low, ksig_high,
					method, frames_nb-1, reject_zeroMAD,
					forced_MAD, &CoD, &sigma, &cosmics,
					&mean, &stddev, &reject_low, &reject_high);


			//cpl_vector_unwrap(data_vector_wrapped);

			master_vector[point] = mean;
			RON_image[point] = stddev;
        HDRL_OMP(omp critical)
			{
			//HDRL_OMP(omp atomic)
			total_cosmics = total_cosmics + cosmics;
			}
		}
		cpl_vector_unwrap(data_vector_wrapped);
		cpl_free(data_vector);
	}
	// Not sure if the image is copied into the list or
	// only the pointer is inserted: image is copied
	*master_image_RE = cpl_image_wrap_double(nx, ny, master_vector);

	*totalCosmics_RE = total_cosmics;
	espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);


	cpl_free(RON_image);
	return cpl_error_get_code();
}


cpl_error_code
espdr_get_master_and_cosmics_parallel_mean(const cpl_imagelist* images,
		const int frames_nb,
		const int nx, const int ny,
		const double ksig_low,
		const double ksig_high,
		cpl_image **master_image_RE,int *totalCosmics_RE)
{
	double *data_vector;
	int point = 0;
	int i = 0;
	int image_size=nx*ny;
	int reject_zeroMAD = 1;
	double forced_MAD = 1.0;
	double mean = 0.0;
	double stddev = 0.0;
	double sigma=0;
	double CoD = 0.0;

	double reject_low = -1.0;
	double reject_high = -1.0;
	int cosmics = 0;
	int total_cosmics = 0;
	double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
	double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
	cpl_error_code my_error;
    const cpl_image* image;
    const double* data;
    
    HDRL_OMP(omp parallel private(image,data,data_vector,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
	{
		// vector of the same pixel in all the frames
		data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
		cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);

        HDRL_OMP(omp for)
		for (point = 0; point < image_size; point++) {
			// vector of the same pixel in all the frames
			//data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
			for (i = 0; i < frames_nb; i++) {
                image = cpl_imagelist_get_const(images,i);
                data = cpl_image_get_data_double_const(image);
				data_vector[i] = data[point];
			}
			//cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
			my_error = espdr_sig_clip_method_mean(data_vector_wrapped, ksig_low, ksig_high,
					frames_nb-1, reject_zeroMAD,
					forced_MAD, &CoD, &sigma, &cosmics,
					&mean, &stddev, &reject_low, &reject_high);


			//cpl_vector_unwrap(data_vector_wrapped);

			master_vector[point] = mean;
			RON_image[point] = stddev;
        HDRL_OMP(omp critical)
			{
			//HDRL_OMP(omp atomic)
			total_cosmics = total_cosmics + cosmics;
			}
		}
		cpl_vector_unwrap(data_vector_wrapped);
		cpl_free(data_vector);
	}
	// Not sure if the image is copied into the list or
	// only the pointer is inserted: image is copied
	*master_image_RE = cpl_image_wrap_double(nx, ny, master_vector);

	*totalCosmics_RE = total_cosmics;
	espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);


	cpl_free(RON_image);
	return cpl_error_get_code();
}




cpl_error_code
espdr_get_master_and_cosmics_parallel_median(const cpl_imagelist* images,
		const int frames_nb,
		const int nx, const int ny,
		const double ksig_low,
		const double ksig_high,
		cpl_image **master_image_RE,int *totalCosmics_RE)
{
	double *data_vector;
	int point = 0;
	int i = 0;
	int image_size=nx*ny;
	int reject_zeroMAD = 1;
	double forced_MAD = 1.0;
	double mean = 0.0;
	double stddev = 0.0;
	double sigma=0;
	double CoD = 0.0;

	double reject_low = -1.0;
	double reject_high = -1.0;
	int cosmics = 0;
	int total_cosmics = 0;
	int total_cosmics_tmp = 0;
	double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
	double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
	cpl_error_code my_error;
    const cpl_image* image;
    const double* data;
    
    HDRL_OMP(omp parallel private(image,data,data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
	{
		// vector of the same pixel in all the frames
		data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
		cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
		total_cosmics_tmp=0;
        HDRL_OMP(omp for)
		for (point = 0; point < image_size; point++) {
			// vector of the same pixel in all the frames
			//data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
			for (i = 0; i < frames_nb; i++) {
                image = cpl_imagelist_get_const(images,i);
                data = cpl_image_get_data_double_const(image);
				data_vector[i] = data[point];
			}
			//cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
			my_error = espdr_sig_clip_method_median(data_vector_wrapped, ksig_low, ksig_high,
					frames_nb-1, reject_zeroMAD,
					forced_MAD, &CoD, &sigma, &cosmics,
					&mean, &stddev, &reject_low, &reject_high);


			//cpl_vector_unwrap(data_vector_wrapped);

			master_vector[point] = mean;
			RON_image[point] = stddev;
			total_cosmics_tmp = total_cosmics_tmp+cosmics;

		}
		cpl_vector_unwrap(data_vector_wrapped);
		cpl_free(data_vector);
        HDRL_OMP(omp critical)
		{
			total_cosmics = total_cosmics + total_cosmics_tmp;
		}
	}
	// Not sure if the image is copied into the list or
	// only the pointer is inserted: image is copied
	*master_image_RE = cpl_image_wrap_double(nx, ny, master_vector);

	*totalCosmics_RE = total_cosmics;
	espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);


	cpl_free(RON_image);
	return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the master bias for one output
 @param         raw_bias_imagelist  imagelist of one output of raw bias images
 @param         nx              x size of the output
 @param         ny              y size of the output
 @param         BIAS_param      BIAS parameters structure
 @param         stats_param     stat parameter structure
 @param[out]    master_bias_image_RE    computed one output master bias (returned)
 @param[out]    RON_RE          computed one output RON (returned)
 @param[out]    mean_RE         computed one output mean (returned)
 @param[out]    totalCosmics_RE computed cosmics number (returned)
 @return    CPL_ERROR_NONE iff OK

 TODO: AMO: This function computes the RON on each raw frame and on the master
 On the other side the computation of the RON on each frame is
 not used. Either we should add that information as QC or drop
 the storing of that information on the corresponding RON_image
 vector.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_master_frame_method_mean(cpl_imagelist *raw_bias_imagelist,
                                            espdr_CCD_geometry *CCD_geom,
                                            espdr_inst_config *inst_config,
                                            int nx,
                                            int ny,
                                            int ext_no,
                                            int out_x,
                                            int out_y,
                                            cpl_image **master_bias_image_RE,
                                            double *RON_RE,
                                            double *mean_RE,
                                            int *totalCosmics_RE) {

    /* Loop indices */
    int i = 0;
    int point = 0;

    /* Total size of the image */
    int image_size = nx * ny;

    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* MAD forced in case MAD == 0.0 */
    double forced_MAD = 1.0;

    /* error code */
    cpl_error_code my_error;


    espdr_ensure(raw_bias_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "Input bias list is NULL");

    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(raw_bias_imagelist);

    espdr_msg_debug("On %d frames", frames_nb);

    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get(raw_bias_imagelist, i) == NULL,
                     CPL_ERROR_NULL_INPUT,
                     "Input bias image %d in the raw bias imagelist is NULL", i);
    }

    /* data tables extracted from the input images list */
    double *images_table[frames_nb];

    /* vector of the same pixel in all the frames */
    //double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

    /* master bias data table */
    double *master_bias_vector = (double *)cpl_calloc(nx*ny, sizeof(double));

    /* center of data: mean or MAD */
    double CoD = 0.0;

    /* number of cosmics rejected */
    int cosmics = 0;
    int total_cosmics = 0;
    int total_cosmics_tmp = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;

    /* mean of the vector */
    double mean = 0.0;

    /* standard deviation of the vector */
    double stddev = 0.0;

    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;

    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));


    /* get the data from images list */
    for (i = 0; i < frames_nb; i++) {
        images_table[i] =
        cpl_image_get_data_double(cpl_imagelist_get(raw_bias_imagelist,
                                                    i));
    }

    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */
    double *data_vector;

    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
    {
        // vector of the same pixel in all the frames
        data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
    for (point = 0; point < image_size; point++) {
        // vector of the same pixel in all the frames
        //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        for (i = 0; i < frames_nb; i++) {
            data_vector[i] = images_table[i][point];
        }
        //cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        my_error = espdr_sig_clip_method_mean(data_vector_wrapped,
                                  inst_config->bias_ksigma,
                                  inst_config->bias_ksigma,
                                  frames_nb-1,
                                  reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);

        //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
        //             "espdr_sig_clip failed");

        //cpl_vector_unwrap(data_vector_wrapped);

        master_bias_vector[point] = mean;
        RON_image[point] = stddev;
        total_cosmics_tmp = total_cosmics_tmp + cosmics;

    }
    cpl_vector_unwrap(data_vector_wrapped);
    cpl_free(data_vector);
    HDRL_OMP(omp critical)
    {
            //HDRL_OMP(omp atomic)

        total_cosmics = total_cosmics + total_cosmics_tmp;
    }
    }
    // Not sure if the image is copied into the list or
    // only the pointer is inserted: image is copied
    *master_bias_image_RE = cpl_image_wrap_double(nx, ny, master_bias_vector);

    *totalCosmics_RE = total_cosmics;
    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

    //cpl_vector *master_bias_vector_wrapped = cpl_vector_wrap(image_size,
    //                                                         master_bias_vector);



    // RON is computed only on the real part of the image

    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;

    cpl_image *real_master_bias = cpl_image_extract(*master_bias_image_RE,
                                                    real_llx, real_lly,
                                                    real_urx, real_ury);

    double *real_master_bias_data = cpl_image_get_data_double(real_master_bias);

    int real_master_bias_size = (real_urx - real_llx + 1) * (real_ury - real_lly + 1);
    cpl_vector *master_bias_vector_wrapped = cpl_vector_wrap(real_master_bias_size,
                                                             real_master_bias_data);

    my_error = espdr_sig_clip_method_mean(master_bias_vector_wrapped,
                              inst_config->stat_ksigma,
                              inst_config->stat_ksigma,
                              inst_config->stat_max_iter,
                              reject_zeroMAD, forced_MAD,
                              &CoD, &sigma, &cosmics,
                              &mean, &stddev,
                              &reject_low, &reject_high);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_sig_clip failed");
    cpl_vector_unwrap(master_bias_vector_wrapped);

    *RON_RE = stddev * sqrt(frames_nb);
    *mean_RE = mean;
    espdr_msg_debug("RON for 1 output = %lf, mean for 1 output = %lf, nb eliminated: %d",
                    *RON_RE, *mean_RE, cosmics);

    /*espdr_msg("total cosmics: %d", total_cosmics);*/


    //cpl_free(real_master_bias_data);
    cpl_image_delete(real_master_bias);

    cpl_free(RON_image);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the master bias for one output
 @param         raw_bias_imagelist  imagelist of one output of raw bias images
 @param         nx              x size of the output
 @param         ny              y size of the output
 @param         BIAS_param      BIAS parameters structure
 @param         stats_param     stat parameter structure
 @param[out]    master_bias_image_RE    computed one output master bias (returned)
 @param[out]    RON_RE          computed one output RON (returned)
 @param[out]    mean_RE         computed one output mean (returned)
 @param[out]    totalCosmics_RE computed cosmics number (returned)
 @return    CPL_ERROR_NONE iff OK

 TODO: AMO: This function computes the RON on each raw frame and on the master
 On the other side the computation of the RON on each frame is
 not used. Either we should add that information as QC or drop
 the storing of that information on the corresponding RON_image
 vector.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_master_frame_method_median(cpl_imagelist *raw_bias_imagelist,
                                            espdr_CCD_geometry *CCD_geom,
                                            espdr_inst_config *inst_config,
                                            int nx,
                                            int ny,
                                            int ext_no,
                                            int out_x,
                                            int out_y,
                                            cpl_image **master_bias_image_RE,
                                            double *RON_RE,
                                            double *mean_RE,
                                            int *totalCosmics_RE) {

    /* Loop indices */
    int i = 0;
    int point = 0;

    /* Total size of the image */
    int image_size = nx * ny;

    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* MAD forced in case MAD == 0.0 */
    double forced_MAD = 1.0;

    /* error code */
    cpl_error_code my_error;


    espdr_ensure(raw_bias_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "Input bias list is NULL");

    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(raw_bias_imagelist);

    espdr_msg_debug("On %d frames", frames_nb);

    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get(raw_bias_imagelist, i) == NULL,
                     CPL_ERROR_NULL_INPUT,
                     "Input bias image %d in the raw bias imagelist is NULL", i);
    }

    /* data tables extracted from the input images list */
    double *images_table[frames_nb];

    /* vector of the same pixel in all the frames */
    //double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

    /* master bias data table */
    double *master_bias_vector = (double *)cpl_calloc(nx*ny, sizeof(double));

    /* center of data: mean or MAD */
    double CoD = 0.0;

    /* number of cosmics rejected */
    int cosmics = 0;
    int total_cosmics = 0;
    int total_cosmics_tmp = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;

    /* mean of the vector */
    double mean = 0.0;

    /* standard deviation of the vector */
    double stddev = 0.0;

    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;

    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));


    /* get the data from images list */
    for (i = 0; i < frames_nb; i++) {
        images_table[i] =
        cpl_image_get_data_double(cpl_imagelist_get(raw_bias_imagelist,
                                                    i));
    }

    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */
    double *data_vector;


    HDRL_OMP(omp parallel private(data_vector,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
    {
        // vector of the same pixel in all the frames
        data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);

    HDRL_OMP(omp for)
    for (point = 0; point < image_size; point++) {
        // vector of the same pixel in all the frames
        //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        for (i = 0; i < frames_nb; i++) {
            data_vector[i] = images_table[i][point];
        }
        //cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        my_error = espdr_sig_clip_method_median(data_vector_wrapped,
                                  inst_config->bias_ksigma,
                                  inst_config->bias_ksigma,
                                  frames_nb-1,
                                  reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);

        //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
        //             "espdr_sig_clip failed");

        //cpl_vector_unwrap(data_vector_wrapped);

        master_bias_vector[point] = mean;
        RON_image[point] = stddev;
        HDRL_OMP(omp critical)
        {
            //HDRL_OMP(omp atomic)
            total_cosmics = total_cosmics + cosmics;
        }
    }
    cpl_vector_unwrap(data_vector_wrapped);
    cpl_free(data_vector);
    }
    // Not sure if the image is copied into the list or
    // only the pointer is inserted: image is copied
    *master_bias_image_RE = cpl_image_wrap_double(nx, ny, master_bias_vector);

    *totalCosmics_RE = total_cosmics;
    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

    //cpl_vector *master_bias_vector_wrapped = cpl_vector_wrap(image_size,
    //                                                         master_bias_vector);



    // RON is computed only on the real part of the image

    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;

    cpl_image *real_master_bias = cpl_image_extract(*master_bias_image_RE,
                                                    real_llx, real_lly,
                                                    real_urx, real_ury);

    double *real_master_bias_data = cpl_image_get_data_double(real_master_bias);

    int real_master_bias_size = (real_urx - real_llx + 1) * (real_ury - real_lly + 1);
    cpl_vector *master_bias_vector_wrapped = cpl_vector_wrap(real_master_bias_size,
                                                             real_master_bias_data);


    my_error = espdr_sig_clip_method_median(master_bias_vector_wrapped,
                              inst_config->stat_ksigma,
                              inst_config->stat_ksigma,
                              inst_config->stat_max_iter,
                              reject_zeroMAD, forced_MAD,
                              &CoD, &sigma, &cosmics,
                              &mean, &stddev,
                              &reject_low, &reject_high);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_sig_clip failed");
    cpl_vector_unwrap(master_bias_vector_wrapped);

    *RON_RE = stddev * sqrt(frames_nb);
    *mean_RE = mean;
    espdr_msg_debug("RON for 1 output = %lf, mean for 1 output = %lf, nb eliminated: %d",
                    *RON_RE, *mean_RE, cosmics);

    /*espdr_msg("total cosmics: %d", total_cosmics);*/


    //cpl_free(real_master_bias_data);
    cpl_image_delete(real_master_bias);

    cpl_free(RON_image);

    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Compare two double variables
 @param     a   first variable
 @param     b   second variable
 @return    comparison result
 */
/*----------------------------------------------------------------------------*/

int compare_doubles (const void *a, const void *b) {
	const double *da = (const double *) a;
	const double *db = (const double *) b;

	return (*da > *db) - (*da < *db);
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Sort an array via Quick Sort
 @param     array       array to be sorted
 @param     array_size  array size
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_quicksort(double *array, int array_size) {

	espdr_ensure(array == NULL, CPL_ERROR_NULL_INPUT, 
				 "Input data vector is NULL");
		
	qsort (array, array_size, sizeof(double), compare_doubles);
	
	return cpl_error_get_code();
}

