/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2014-12-16 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_wave_cal.h>


/*----------------------------------------------------------------------------
 Functions code, functions common to all wave recipes
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_fibre_source(const char *WAVE_tag,
                                      espdr_src_type *fibre_source_RE,
                                      char **fibre_source_str_RE) {
    
    int position = 0;
    int WAVE_len = strlen(WAVE_tag);
    
    for (int i = 0; i < WAVE_len; i++) {
        if (WAVE_tag[i] == '_') {
            position = i;
            break;
        }
    }
    
    strncpy(fibre_source_str_RE[0], WAVE_tag, position);
    fibre_source_str_RE[0][position]='\0';
    strncpy(fibre_source_str_RE[1], WAVE_tag+position+1, WAVE_len-1);
    espdr_msg("TAGs: fibre A: %s, fibre B: %s", fibre_source_str_RE[0], fibre_source_str_RE[1]);
    
    for (int i = 0; i < 2; i++) {
        if (strcmp(fibre_source_str_RE[i], ESPDR_WAVE_THAR_RAW) == 0) {
            fibre_source_RE[i] = THAR;
        } else {
            if (strcmp(fibre_source_str_RE[i], ESPDR_WAVE_FP_RAW) == 0) {
                fibre_source_RE[i] = FP;
            } else {
                if (strcmp(fibre_source_str_RE[i], ESPDR_WAVE_LFC_RAW) == 0) {
                    fibre_source_RE[i] = LFC;
                } else {
                    if (strcmp(fibre_source_str_RE[i], ESPDR_WAVE_DARK_RAW) == 0) {
                        fibre_source_RE[i] = DARK;
                    } else {
                        espdr_msg_error("Unknown WAVE source: %s, should be THAR, FP, LFC or DARK, exiting",
                                        fibre_source_str_RE[i]);
                        return (CPL_ERROR_INCOMPATIBLE_INPUT);
                    }
                }
            }
        }
    }
    
    return (cpl_error_get_code());
}




/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_process_wave_till_extraction(cpl_frameset *frameset,
                                                  cpl_parameterlist *parameters,
                                                  const char *RECIPE_ID,
                                                  cpl_frameset *wave_frames_to_reduce,
                                                  cpl_frame *wave_frame,
                                                  cpl_frameset *used_frames,
                                                  espdr_inst_config *inst_config,
                                                  espdr_CCD_geometry *CCD_geom,
                                                  espdr_qc_keywords *qc_kws,
                                                  int *orders_nb_per_fibre,
                                                  espdr_src_type *fibre_source,
                                                  char **fibre_source_str,
                                                  cpl_propertylist **keywords_fibre,
                                                  cpl_imagelist *CCD_corrected_wave,
                                                  cpl_table ****orders_coeffs,
                                                  double *RON,
                                                  cpl_image **flat_corr_spectrum,
                                                  cpl_image **flat_corr_error,
                                                  cpl_image **flat_corr_qual,
                                                  cpl_image **flat_blaze_corr_spectrum,
                                                  cpl_image **flat_blaze_corr_error,
                                                  cpl_image **flat_blaze_corr_qual,
                                                  cpl_image **pixel_geom_image,
                                                  cpl_image **pixel_size_image) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int *pixel_geom_nb_check = NULL;
    int *pixel_size_nb_check = NULL;
    
    espdr_msg("Start processing wave");
    
    const char *WAVE_tag = cpl_frame_get_tag(wave_frame);
    
    /* Extract hot pixels mask frame */
    cpl_frame *hot_pixels_frame = espdr_get_hpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract master dark frame */
    cpl_frame *master_dark_frame = espdr_get_mdark_from_set(frameset,
                                                            used_frames);
    
    /* Extract bad pixels mask frame */
    cpl_frame *bad_pixels_frame = espdr_get_bpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract orders mask frame */
    cpl_frame *orders_mask_frame = espdr_get_orders_mask_from_set(frameset,
                                                                  used_frames);
    
    /* Extract orders maps frame */
    espdr_get_orders_coeff_from_set(frameset, used_frames,
                                    inst_config->fibres_nb,
                                    CCD_geom->ext_nb,
                                    orders_coeffs);
    
    /* Extract order profile frame per fibre */
    cpl_frame **order_profile_fibre_frame = NULL;
    espdr_get_order_profile_from_set(frameset, used_frames,
                                     inst_config->fibres_nb,
                                     CCD_geom->ext_nb,
                                     &order_profile_fibre_frame);
    
    /* Extract flat frame per fibre */
    cpl_frame **flat_fibre_frame = NULL;
    espdr_get_flat_fibre_from_set(frameset, used_frames,
                                  inst_config->fibres_nb,
                                  CCD_geom->ext_nb,
                                  &flat_fibre_frame);
    
    /* Extract BLAZE frame per fibre */
    cpl_frame **blaze_fibre_frame = NULL;
    espdr_get_blaze_fibre_from_set(frameset, used_frames,
                                   inst_config->fibres_nb,
                                   CCD_geom->ext_nb,
                                   &blaze_fibre_frame);
    
    /* Extract pixel geometry frame per fibre */
    cpl_frame** pixel_geom_fibre_frame = NULL;
    espdr_get_pixel_geom_fibre_from_set(frameset, used_frames,
                                        inst_config->fibres_nb,
                                        CCD_geom->ext_nb,
                                        &pixel_geom_nb_check,
                                        &pixel_geom_fibre_frame);
    
    /* Extract pixel size frame per fibre */
    cpl_frame** pixel_size_fibre_frame = NULL;
    espdr_get_pixel_size_fibre_from_set(frameset, used_frames,
                                        inst_config->fibres_nb,
                                        CCD_geom->ext_nb,
                                        &pixel_size_nb_check,
                                        &pixel_size_fibre_frame);
    
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom,0,
                                          &hot_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    cpl_imagelist *master_dark_pxl_nb_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame,
                                              CPL_TYPE_DOUBLE,
                                              CCD_geom, 0,
                                              &master_dark_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
        master_dark_pxl_nb_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame,
                                              CPL_TYPE_INT,
                                              CCD_geom, CCD_geom->ext_nb,
                                              &master_dark_pxl_nb_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0,
                                          &bad_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Extract all the outputs from the orders mask frame if provided */
    cpl_imagelist *orders_mask_list = NULL;
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot & bad pixels mask");
    
    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        my_error = espdr_update_bad_pixel_mask(wave_frames_to_reduce, CCD_geom,
                                               cpl_imagelist_get(bad_pixels_list, 0));
    }
    
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list,
                                                bad_pixels_list,
                                                &pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_hot_bad_pixels_mask failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(merged_pixels_mask);
    
    espdr_msg("Extracting order profile");
    /* Extract extensions from order profile frames */
    cpl_imagelist **order_profile_fibre_imagelist = (cpl_imagelist **)cpl_malloc
                                (inst_config->fibres_nb * sizeof(cpl_imagelist *));
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        order_profile_fibre_imagelist[i] = cpl_imagelist_new();
        my_error = espdr_extract_extensions(order_profile_fibre_frame[i],
                                            CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb,
                                            &order_profile_fibre_imagelist[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    espdr_msg("Extracting flat");
    /* Extract extensions from flat frames */
    cpl_image **flat_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_error_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_quality_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    const char *flat_filename = NULL;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        flat_filename = cpl_frame_get_filename(flat_fibre_frame[i]);
        flat_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 1);
        flat_error_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 2);
        flat_quality_table[i] = cpl_image_load(flat_filename, CPL_TYPE_INT, 0, 3);
    }
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Loading FLAT image failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("Extracting blaze");
    /* Extract image from blaze frame */
    cpl_image **blaze_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    const char *blaze_filename = NULL;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        blaze_filename = cpl_frame_get_filename(blaze_fibre_frame[i]);
        blaze_table[i] = cpl_image_load(blaze_filename, CPL_TYPE_DOUBLE, 0, 1);
    }
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Loading BLAZE image failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("Extracting wave");
    /* Extract extensions from the wave frame */
    cpl_imagelist *raw_wave_imagelist = cpl_imagelist_new();
    cpl_imagelist *real_wave_imagelist = cpl_imagelist_new();
    if (inst_config->inst_type == NIR) {
        my_error = espdr_extract_extensions(wave_frame, CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb,
                                            &raw_wave_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_extract_raw_outputs(wave_frames_to_reduce, CPL_TYPE_DOUBLE,
                                             CCD_geom,
                                             &raw_wave_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_raw_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    my_error = espdr_extract_real_outputs_from_raw(wave_frames_to_reduce,
                                                   CPL_TYPE_DOUBLE,
                                                   CCD_geom,
                                                   &real_wave_imagelist);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_extensions failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("Reading headers");
    /* keywords contains all the keywords read from
     the first input frame header */
    cpl_propertylist *keywords = NULL;
    cpl_propertylist *keywords_HP = NULL;
    cpl_propertylist *keywords_BP = NULL;
    cpl_propertylist **keywords_ext = (cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb *
                                                                     sizeof(cpl_propertylist*));
    
    /* Load primary keywords from the WAVE image (primary header) */
    const char *input_filename = cpl_frame_get_filename(wave_frame);
    espdr_msg("KEYWORDS for WAVE input filename: %s", input_filename);
    keywords = cpl_propertylist_load(input_filename, 0);
    
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
    if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
        char *arc_filename = strrchr(input_filename, '/')+1;
        espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
        my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Updating the ARCFILE KW (%s) failed: %s",
                     arc_filename, cpl_error_get_message_default(my_error));
    }
#endif
    
    /* Load extension keywords from the WAVE image (extension headers) */
    for (int j = 0; j < CCD_geom->ext_nb; j++) {
        keywords_ext[j] = cpl_propertylist_load(input_filename, j+1);
    }
    
    /* Load primary header keywords from the hot pixels image */
    const char *input_filename_HP = NULL;
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s", input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s", input_filename_HP);
    }
    keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    const char *input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s", input_filename_BP);
    keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    /* Checking for saturation, creating saturation mask */
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_wave_imagelist),
                                            sizeof(double));
    int print_max_flux = 0;
    if (strstr(WAVE_tag, "THAR") == NULL) {
        print_max_flux = 1;
    }
    // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
    if (inst_config->inst_type == NIR) {
        double texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_wave_imagelist, texp);
    }
    my_error = espdr_check_saturation(real_wave_imagelist,
                                      CCD_geom,
                                      inst_config,
                                      pixels_mask,
                                      inst_config->image_cosmics_part,
                                      max_flux,
                                      print_max_flux,
                                      &geometry_corrected_saturation_mask);
    
    cpl_imagelist_delete(real_wave_imagelist);
    
    /* Add QC KWs to the header */
    
    if (strstr(WAVE_tag, "THAR") == NULL) {
        char qc_check_kw[KEYWORD_LENGTH];
        if (strstr(WAVE_tag, "LFC") != NULL) {
            strcpy(qc_check_kw, qc_kws->qc_lfc_check_kw);
        } else {
            strcpy(qc_check_kw, qc_kws->qc_wave_check_kw);
        }
        my_error = espdr_max_flux_QC(inst_config, CCD_geom, qc_kws,
                                     max_flux, qc_check_kw, keywords);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_max_flux_QC failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    espdr_msg("CCD signature correction on WAVE frame (with %d ext)",
              CCD_geom->ext_nb);
    
    /* Correcting the CCD signature: bias, dark, geometry */
    espdr_msg("Removing the CCD signature");
    double *CONAD = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    double *DARK_current = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    double *DARK_RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    my_error = espdr_remove_det_signature(raw_wave_imagelist, orders_mask_list,
                                          NULL, master_dark_list,
                                          keywords, keywords_HP, keywords_BP,
                                          qc_kws, inst_config, 0,
                                          CCD_geom, CPL_TYPE_DOUBLE,
                                          RON, DARK_current, DARK_RON, CONAD,
                                          CCD_corrected_wave);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_detector_signature failed: %s",
                 cpl_error_get_message_default(my_error));
    
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg("Raw WAVE mean: %lf, median: %lf",
                  cpl_image_get_mean(cpl_imagelist_get(raw_wave_imagelist, i)),
                  cpl_image_get_median(cpl_imagelist_get(raw_wave_imagelist, i)));
    }
    
    cpl_imagelist_delete(raw_wave_imagelist);
    cpl_imagelist_delete(hot_pixels_list);
    cpl_imagelist_delete(bad_pixels_list);
    cpl_imagelist_delete(orders_mask_list);
    
#if SAVE_DEBUG_CCD_CLEANED_WAVE
    
    espdr_msg("Saving CCD cleaned wave frame");
    
    char ccd_cleaned_filename[FILENAME_LENGTH];
    sprintf(ccd_cleaned_filename,
            "%s_CCD_cleaned_%s.fits",
            inst_config->instrument, WAVE_tag);
    
    /* Save the cleaned WAVE fits frame */
    my_error = espdr_save_ccd_cleaned_wave_frame(frameset,
                                                 used_frames,
                                                 parameters,
                                                 RECIPE_ID,
                                                 ccd_cleaned_filename,
                                                 CCD_geom,
                                                 keywords,
                                                 keywords_ext,
                                                 CCD_corrected_wave);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_ccd_cleaned_wave_frame failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 ccd_cleaned_filename);
#endif
    
    /* Extraction */
    cpl_image **extracted_spectrum_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **spectrum_error_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **spectrum_quality_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    
    double exp_time_hour = cpl_propertylist_get_double(keywords,inst_config->Texp_kw) / NB_SEC_IN_HOUR;
    double exp_time_hour_mdark = cpl_propertylist_get_double(keywords_HP, inst_config->Texp_kw) / NB_SEC_IN_HOUR;
    
    int **cosmics_nb = (int **)cpl_malloc(inst_config->fibres_nb * sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cosmics_nb[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
            cosmics_nb[i][j] = 0;
        }
    }
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Starting extraction for fibre %c", fibre_name[i]);
        double ksigma;
        double tolerance = inst_config->tolerance_rejection;
        int window_size = inst_config->extraction_window_size[i];
        if (fibre_source[i] == THAR) {
            ksigma = inst_config->extraction_ksigma_thar;
        } else {
            if (fibre_source[i] == FP) {
                ksigma = inst_config->extraction_ksigma_fp;
            } else {
                if (fibre_source[i] == LFC) {
                    ksigma = inst_config->extraction_ksigma_lfc;
                } else {
                    if (fibre_source[i] == DARK) {
                        ksigma = inst_config->extraction_ksigma_dark;
                    } else {
                        espdr_msg_error("Unknown WAVE source: %s, should be THAR, FP or DARK, exiting",
                                        fibre_source_str[i]);
                        return (CPL_ERROR_INCOMPATIBLE_INPUT);
                    }
                }
            }
        }
        
        if (strcmp(inst_config->extraction_method, "horne") == 0) {
            
            my_error = espdr_horne_extraction_one_fibre(CCD_corrected_wave,
                                                        geometry_corrected_pixels_mask,
                                                        geometry_corrected_saturation_mask,
                                                        NULL, // no CRH mask
                                                        NULL, // no background map
                                                        order_profile_fibre_imagelist[i],
                                                        NULL, // no contamination frame
                                                        master_dark_list, master_dark_pxl_nb_list,
                                                        (*orders_coeffs)[i], i,
                                                        inst_config, CCD_geom,
                                                        RON, DARK_current, DARK_RON, CONAD,
                                                        exp_time_hour, exp_time_hour_mdark,
                                                        window_size, tolerance, ksigma,
                                                        cosmics_nb[i],
                                                        &extracted_spectrum_table[i],
                                                        &spectrum_error_table[i],
                                                        &spectrum_quality_table[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_horne_extraction failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("Horne's extraction done");
            
        } else if (strcmp(inst_config->extraction_method, "simple") == 0) {
            
            espdr_msg_warning(ANSI_COLOR_RED
                              "Warning : Simple extraction method ..."
                              ANSI_COLOR_RESET);
            
            my_error = espdr_simple_extraction_one_fibre(CCD_corrected_wave,
                                                         geometry_corrected_pixels_mask,
                                                         geometry_corrected_saturation_mask,
                                                         NULL, // no CRH mask
                                                         NULL, // no background map
                                                         master_dark_list, master_dark_pxl_nb_list,
                                                         (*orders_coeffs)[i], i,
                                                         inst_config, CCD_geom,
                                                         RON, DARK_current, DARK_RON, CONAD,
                                                         exp_time_hour, exp_time_hour_mdark,
                                                         window_size,
                                                         &extracted_spectrum_table[i],
                                                         &spectrum_error_table[i],
                                                         &spectrum_quality_table[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_simple_extraction failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("Simple extraction done");
            
        } else {
            
            espdr_msg("Error: %s is an unknown extraction method. Allowed only 'horne' and 'simple'",
                      inst_config->extraction_method);
            exit(EXIT_FAILURE);

        }
    }
    
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    if (master_dark_frame != NULL) {
        cpl_imagelist_delete(master_dark_list);
        cpl_imagelist_delete(master_dark_pxl_nb_list);
    }
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_imagelist_delete(order_profile_fibre_imagelist[i]);
    }
    cpl_free(order_profile_fibre_imagelist);
    cpl_free(order_profile_fibre_frame);
    cpl_free(flat_fibre_frame);
    cpl_free(blaze_fibre_frame);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        int total_cosmics = 0;
        if (((fibre_source[i] == FP) && (inst_config->extraction_ksigma_fp > 0.0)) ||
            ((fibre_source[i] == LFC) && (inst_config->extraction_ksigma_lfc > 0.0))) {
                for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                    //espdr_msg("Cosmics nb fibre %c, order %d: %d",
                    //          fibre_name[i], j+1, cosmics_nb[i][j]);
                    total_cosmics += cosmics_nb[i][j];
                }
        }
        espdr_msg("Total number of cosmics detected: %d", total_cosmics);
    }
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Extracted spectrum of fibre %c has size %lldx%lld",
                  fibre_name[i],
                  cpl_image_get_size_x(extracted_spectrum_table[i]),
                  cpl_image_get_size_y(extracted_spectrum_table[i]));
    }
    
#if SAVE_DEBUG_PRODUCT_WAVE
    
    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        char extracted_filename[64];
        sprintf(extracted_filename, "%s_spectrum_extracted_%s_%c.fits",
                inst_config->instrument, WAVE_tag, fibre_name[fibre]);
        espdr_msg("Saving extracted spectrum %c in %s",
                  fibre_name[fibre], extracted_filename);
        
        my_error = espdr_dfs_save_S2D(extracted_spectrum_table[fibre],
                                      spectrum_error_table[fibre],
                                      spectrum_quality_table[fibre],
                                      frameset, used_frames, parameters,
                                      keywords, extracted_filename,
                                      ESPDR_PRO_CATG_TEST_PRODUCT,
                                      RECIPE_ID);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_save_S2D failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     extracted_filename);
    }
    
#endif
    
    /* Flat & blaze correction */
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_correct_flat(extracted_spectrum_table[i],
                                      spectrum_error_table[i],
                                      spectrum_quality_table[i],
                                      flat_table[i],
                                      flat_error_table[i],
                                      flat_quality_table[i],
                                      &flat_corr_spectrum[i],
                                      &flat_corr_error[i],
                                      &flat_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_flat failed for fiber %c: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
        
    }
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_correct_blaze(flat_corr_spectrum[i],
                                       flat_corr_error[i],
                                       flat_corr_qual[i],
                                       blaze_table[i],
                                       &flat_blaze_corr_spectrum[i],
                                       &flat_blaze_corr_error[i],
                                       &flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_blaze failed for fiber %c: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
    }
    
    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        
        keywords_fibre[fibre] = cpl_propertylist_duplicate(keywords);
        
        if (((fibre_source[fibre] == FP) || (fibre_source[fibre] == LFC)) &&
            (strcmp(RECIPE_ID, "espdr_wave_TH_drift") != 0)) {
            my_error = espdr_wave_cosmics_QC(cosmics_nb[fibre],
                                             orders_nb_per_fibre[fibre],
                                             qc_kws, keywords_fibre[fibre]);
        }
        if (strstr(WAVE_tag, "THAR") != NULL) {
            my_error = cpl_propertylist_update_int(keywords_fibre[fibre],
                                                   qc_kws->qc_wave_check_kw, 1);
        }
    }
    
    int recipes[10] = {1, 1, 1, 1, 1, 0, 0, 0, 0, 0};
    char recipe_name[32];
    if (strcmp(RECIPE_ID, "espdr_wave_FP") == 0) {
        strcpy(recipe_name, "WAVE_FP");
    } else {
        if (strcmp(RECIPE_ID, "espdr_wave_THAR") == 0) {
            strcpy(recipe_name, "WAVE_THAR");
            recipes[5] = 1;
        } else {
            strcpy(recipe_name, "WAVE");
        }
    }
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords_fibre[i],
                                                        NULL, NULL, NULL, recipe_name, recipes);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_calibrations_intervals failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    espdr_msg("Extracting pixel geometry");
    /* Read the pixel geometry image or prepare one if not provided */
    const char *pixel_geom_filename = NULL;
    cpl_image *pixel_geom_fibre_image = NULL;
    const char *pixel_size_filename = NULL;
    cpl_image *pixel_size_fibre_image = NULL;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Pixel geom for fibre %c, nb check: %d", fibre_name[i], pixel_geom_nb_check[i]);
        int size_x = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
        int size_y = cpl_image_get_size_y(flat_blaze_corr_spectrum[i]);
        double *pixel_geom_fibre_data = (double *)cpl_calloc(size_x * size_y, sizeof(double));
        if (pixel_geom_nb_check[i] > 0) {
            pixel_geom_filename = cpl_frame_get_filename(pixel_geom_fibre_frame[i]);
            espdr_msg("Reading pixel geom calib: %s", pixel_geom_filename);
            pixel_geom_fibre_image = cpl_image_load(pixel_geom_filename,
                                                    CPL_TYPE_DOUBLE, 0, 1);
        } else {
            espdr_msg("No pixel geom calib given, generating product");
            int index_pxl_geom = 0;
            for (int j = 0; j < size_y; j++) {
                for (int k = 0; k < size_x; k++) {
                    pixel_geom_fibre_data[index_pxl_geom] = (double)k + 1.0;
                    index_pxl_geom++;
                }
            }
            pixel_geom_fibre_image = cpl_image_wrap_double(size_x, size_y,
                                                           pixel_geom_fibre_data);
        }
        pixel_geom_image[i] = cpl_image_duplicate(pixel_geom_fibre_image);
        if (pixel_geom_nb_check[i] > 0) {
            cpl_image_delete(pixel_geom_fibre_image);
        } else {
            cpl_image_unwrap(pixel_geom_fibre_image);
        }
        cpl_free(pixel_geom_fibre_data);
        
        double *pixel_size_fibre_data = (double *)cpl_calloc(size_x * size_y, sizeof(double));
        if (pixel_size_nb_check[i] > 0) {
            pixel_size_filename = cpl_frame_get_filename(pixel_size_fibre_frame[i]);
            espdr_msg("Reading pixel size calib: %s", pixel_size_filename);
            pixel_size_fibre_image = cpl_image_load(pixel_size_filename,
                                                    CPL_TYPE_DOUBLE, 0, 1);
        } else {
            espdr_msg("No pixel size calib given, generating product");
            int index_pxl_size = 0;
            for (int j = 0; j < size_y; j++) {
                for (int k = 0; k < size_x; k++) {
                    pixel_size_fibre_data[index_pxl_size] = 1.0;
                    index_pxl_size++;
                }
            }
            pixel_size_fibre_image = cpl_image_wrap_double(size_x, size_y,
                                                           pixel_size_fibre_data);
        }
        pixel_size_image[i] = cpl_image_duplicate(pixel_size_fibre_image);
        if (pixel_size_nb_check[i] > 0) {
            cpl_image_delete(pixel_size_fibre_image);
        } else {
            cpl_image_unwrap(pixel_size_fibre_image);
        }
        cpl_free(pixel_size_fibre_data);
    }
    
    
    espdr_msg("Cleaning the memory");
    
    espdr_msg("Cleaning keywords lists");
    cpl_propertylist_delete(keywords);
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);
    cpl_propertylist_delete(keywords_HP);
    cpl_propertylist_delete(keywords_BP);
    
    espdr_msg("Cleaning products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(extracted_spectrum_table[i]);
        cpl_image_delete(spectrum_error_table[i]);
        cpl_image_delete(spectrum_quality_table[i]);
    }
    
    cpl_free(extracted_spectrum_table);
    cpl_free(spectrum_error_table);
    cpl_free(spectrum_quality_table);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_table[i]);
        cpl_image_delete(flat_error_table[i]);
        cpl_image_delete(flat_quality_table[i]);
        cpl_image_delete(blaze_table[i]);
    }
    
    cpl_free(flat_table);
    cpl_free(flat_error_table);
    cpl_free(flat_quality_table);
    cpl_free(blaze_table);
    
    
    espdr_msg("Cleaning global variables");
    cpl_free(CONAD);
    cpl_free(DARK_current);
    cpl_free(max_flux);
    
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_extract_lines_tables(cpl_frameset *frameset,
                                          const char *RECIPE_ID,
                                          cpl_frameset *used_frames,
                                          espdr_inst_config *inst_config,
                                          espdr_src_type *fibre_source,
                                          char **fibre_source_str,
                                          int compute_drift,
                                          char *first_TAG,
                                          char *second_TAG,
                                          cpl_frame **REF_line_table_frame,
                                          cpl_frame **FP_line_table_frame,
                                          cpl_image **static_wave_matrix_img,
                                          cpl_image **static_dll_matrix_img,
                                          cpl_frame **s2d_blaze_FP_frame) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int index = -1;
    
    espdr_msg("Extracting lines tables");
    char pro_catg_tag[TAG_LENGTH];
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            espdr_msg("THAR is on fibre %c", fibre_name[i]);
            sprintf(pro_catg_tag, "%s_%c", first_TAG, fibre_name[i]);
            REF_line_table_frame[i] = espdr_frame_find(frameset, pro_catg_tag);
            espdr_ensure(REF_line_table_frame[i] == NULL, CPL_ERROR_NULL_INPUT,
                         "%s frame for fibre %c not found", first_TAG, fibre_name[i]);
            my_error = cpl_frameset_insert(used_frames,
                                           cpl_frame_duplicate(REF_line_table_frame[i]));
            
            sprintf(pro_catg_tag, "%s_%c", second_TAG, fibre_name[i]);
            FP_line_table_frame[i] = espdr_frame_find(frameset, pro_catg_tag);
            espdr_ensure(FP_line_table_frame[i] == NULL, CPL_ERROR_NULL_INPUT,
                         "%s frame for fibre %c not found", second_TAG, fibre_name[i]);
            my_error = cpl_frameset_insert(used_frames,
                                           cpl_frame_duplicate(FP_line_table_frame[i]));
            
            if ((strcmp(RECIPE_ID, "espdr_wave_TH_drift") == 0) ||
                ((strcmp(RECIPE_ID, "espdr_wave_THAR") == 0) && (compute_drift == 1))) {
                if (strcmp(RECIPE_ID, "espdr_wave_THAR") == 0) {
                    espdr_msg("Extracting drift needed products");
                    index = (i+1)%2; // for the drift we need the products of the other fibre
                } else {
                    espdr_msg("Extracting reference matrixes");
                    index = i;
                }
                
                // Extract the DRIFT needed products: STATIC_WAVE, STATIC_DLL & S2D_BLAZE
                sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_STATIC_WAVE_MATRIX, fibre_name[index]);
                cpl_frame *static_wave_matrix_frame = espdr_frame_find(frameset, pro_catg_tag);
                espdr_ensure(static_wave_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
                             "Static wave matrix frame extraction failed");
                my_error = cpl_frameset_insert(used_frames,
                                               cpl_frame_duplicate(static_wave_matrix_frame));
                
                sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_STATIC_DLL_MATRIX, fibre_name[index]);
                cpl_frame *static_dll_matrix_frame = espdr_frame_find(frameset, pro_catg_tag);
                espdr_ensure(static_dll_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
                             "Static dll matrix frame extraction failed");
                my_error = cpl_frameset_insert(used_frames,
                                               cpl_frame_duplicate(static_dll_matrix_frame));
                
                static_wave_matrix_img[index] = cpl_image_load(cpl_frame_get_filename(static_wave_matrix_frame),
                                                               CPL_TYPE_DOUBLE, 0, 1);
                static_dll_matrix_img[index] = cpl_image_load(cpl_frame_get_filename(static_dll_matrix_frame),
                                                              CPL_TYPE_DOUBLE, 0, 1);

                //cpl_frame_delete(static_wave_matrix_frame);
                //cpl_frame_delete(static_dll_matrix_frame);
            }
            
            if ((strcmp(RECIPE_ID, "espdr_wave_THAR") == 0) && (compute_drift == 1)) {
                sprintf(pro_catg_tag, "%s_FP_FP_%c",
                        ESPDR_PRO_CATG_S2D_BLAZE, fibre_name[(i+1)%2]);
                *s2d_blaze_FP_frame = espdr_frame_find(frameset,
                                                       pro_catg_tag);
                espdr_ensure(*s2d_blaze_FP_frame == NULL, CPL_ERROR_NULL_INPUT,
                             "S2D BLAZE FP FP frame extraction failed");
                my_error = cpl_frameset_insert(used_frames,
                                               cpl_frame_duplicate(*s2d_blaze_FP_frame));
                
            }
        }
    }
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs on cosmics nb
 @param         cosmics             number of cosmics for each order
 @param         orders_nb           number of orders
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_cosmics_QC(int *cosmics,
                                     int orders_nb,
                                     espdr_qc_keywords *qc_kws,
                                     cpl_propertylist *keywords_RE) {
    
    int i, total_cosmics_nb = 0;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    /* QC KWs per order */
    for (i = 0; i < orders_nb; i++) {
        new_keyword =
        espdr_add_index_to_keyword(qc_kws->qc_wave_order_cosmic_nb_kw_first,
                                   qc_kws->qc_wave_order_cosmic_nb_kw_last,
                                   i+1);
        sprintf(comment, "Number of cosmics detected in order %d", i+1);
        my_error = espdr_keyword_add_int(new_keyword,
                                         cosmics[i],
                                         comment,
                                         &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER%d_COSMIC_NB_KW to the propertylist failed: %s", i+1,
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        total_cosmics_nb += cosmics[i];
    }
    
    /* Global QC KWs */
    
    sprintf(comment, "Total number of detected cosmics");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_cosmic_nb_kw,
                                     total_cosmics_nb,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_COSMIC_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_check_delta_FP_peaks(espdr_line_param *lines_table,
                                          int lines_nb,
                                          double delta_fp_tolerance) {
    
    int qc_delta_FP_peaks = 1;
    for (int j = 1; j < lines_nb-1; j++) {
        if ((lines_table[j].order == -1) || (lines_table[j].order == -1)) {
            espdr_msg("pxl: %lf %lf --> diff: %lf",
                      lines_table[j].peak_pxl, lines_table[j+1].peak_pxl,
                      lines_table[j+1].peak_pxl - lines_table[j].peak_pxl);
        }
        double delta_x, delta_x_prev;
        if ((lines_table[j].order == lines_table[j+1].order) &&
            (lines_table[j].order == lines_table[j-1].order)) {
            delta_x = lines_table[j+1].peak_pxl - lines_table[j].peak_pxl;
            delta_x_prev = lines_table[j].peak_pxl - lines_table[j-1].peak_pxl;
            if (fabs(delta_x - delta_x_prev) > delta_fp_tolerance) {
                espdr_msg_warning("Delta x bigger for order %d pxl %lf, delta = %lf, %lf, %lf",
                                  lines_table[j].order, lines_table[j].peak_pxl,
                                  delta_x_prev, delta_x, delta_x - delta_x_prev);
                espdr_msg_warning("Peak_pxls: %f, %f, %f",
                                  lines_table[j-1].peak_pxl,
                                  lines_table[j].peak_pxl,
                                  lines_table[j+1].peak_pxl);
                qc_delta_FP_peaks = 0;
            }
        }
    }
    
    if (qc_delta_FP_peaks == 0) {
        espdr_msg_warning(ANSI_COLOR_RED
                          "Missing peaks in the FP image"
                          ANSI_COLOR_RESET);
        return (CPL_ERROR_INCOMPATIBLE_INPUT);
    }
    
    return (cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief Compute dispersion
 @param[out]    lines_table     detected lines peaks
 @param         lines_nb        number of lines in the table
 @param         inst_config     instrument config
 @param         coeffs          coeffs of the orders fit in ll
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_dlldx(espdr_line_param *lines_table,
                                   int lines_nb,
                                   int orders_nb,
                                   espdr_inst_config *inst_config,
                                   double **coeffs) {
    int i, j, poly_deg, order, pxl;
    double ll, dlldx;
    
    poly_deg = inst_config->wave_sol_poly_deg;
    //espdr_msg("Poly deg = %d, lines_nb = %d", poly_deg, lines_nb);
    
    /*
     for (i = 0; i < 71; i++) {
     espdr_msg("COMPUTE DLLDX: order %d coeffs: %lf %lf %lf %lf",
     i+1, coeffs[i][0], coeffs[i][1],
     coeffs[i][2], coeffs[i][3]);
     }
     */
    
    for (i = 0; i < lines_nb; i++) {
        if ((lines_table[i].qc_flag == 1) & (lines_table[i].order <= orders_nb)) {
            order = lines_table[i].order;
            pxl = lines_table[i].peak_pxl;
            
            //if (order == 10) {
            //    espdr_msg("order %d COEFFS: %lf, %lf, %lf, %lf", order,
            //              coeffs[order-1][0], coeffs[order-1][1],
            //              coeffs[order-1][2], coeffs[order-1][3]);
            //}
            
            ll = lines_table[i].wavelength;
            
            dlldx = 0.0;
            for (j = poly_deg - 1; j > 1; j--) {
                //dlldx = (dlldx + j * coeffs[order-1][j]) * ll;
                dlldx = (dlldx + j * coeffs[order-1][j]) * pxl;
            }
            dlldx = dlldx + coeffs[order-1][1];
            
            lines_table[i].dispersion = dlldx;
            
            if (order == -1) {
                espdr_msg("order %d ll = %lf, dlldx = %lf", order,
                          lines_table[i].wavelength,
                          lines_table[i].dispersion);
            }
            
            //lines_table[i].wavelength = 5500.0;
            //lines_table[i].dispersion = 0.01;
        }
    }
    
    return(cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code sort_lines_pxl(espdr_line_param* lines_table,
                              int lines_nb) {
    
    int i, j;
    espdr_line_param swap_line;
    
    for (i = 0; i < lines_nb-1; i++) {
        for (j = i+1; j < lines_nb; j++) {
            if (lines_table[i].peak_pxl > lines_table[j].peak_pxl) {
                swap_line = lines_table[i];
                lines_table[i] = lines_table[j];
                lines_table[j] = swap_line;
            }
        }
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Find spectral lines for FP or LC
 @param         s2d_data        input S2D image for 1 fiber
 @param         s2d_error       input S2D image errors for 1 fiber
 @param         s2d_quality     input S2D image quality flags for 1 fiber
 @param         wave_matrix     static matrix with lines wavelengths
 (optional, not used yet)
 @param         flux_threshold  threshold on the flux to detect the spectral pic
 @param         noise_threshold threshold on the noise to detect the spectral pic
 @param[out]    lines_table     computed detected lines table
 @param[out]    lines_nb_RE     computer number of detected lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_search_lines(cpl_image *s2d_data,
                                  cpl_image *s2d_error,
                                  cpl_image *s2d_quality,
                                  espdr_inst_config *inst_config,
                                  espdr_src_type fibre_source,
                                  espdr_line_param *lines_table,
                                  int *lines_nb_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int CCD_border_cut;
    int search_window_size;
    double flux_threshold;
    double rms_threshold;
    double noise_threshold;
    double delta_fp_peak_limit;
    int ngauss_flag;
    
    int i, order, pxl, pxl_min, pxl_max;
    int index, index_p_data;
    int orders_nb, order_length;
    int pxl_start, pxl_end;
    cpl_image *order_image;
    double *order_data = NULL;
    cpl_image *error_image;
    double *error_data = NULL;
    cpl_image *quality_image;
    int *quality_data = NULL;
    double mean, rms;
    espdr_poly_data *p_data = NULL;
    int p_data_size;
    double *fit = NULL;
    double *coeffs = NULL;
    double *coeffs_err = NULL;
    double max_flux;
    int max_flux_pxl;
    double max_val = 0.0;
    double fitted_pic_dist;
    double chisq = 0.0;
    cpl_size px, py;
    int total_fake_peaks_nb = 0;
    int real_peak = 1;
    
    espdr_ensure(s2d_data == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D image is NULL");
    espdr_ensure(s2d_error == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D error image is NULL");
    espdr_ensure(s2d_quality == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D quality image is NULL");
    
    //int size_x = cpl_image_get_size_x(s2d_quality);
    //int size_y = cpl_image_get_size_y(s2d_quality);
    //espdr_msg("s2d_q is of size %d x %d", size_x, size_y);
    
    orders_nb = cpl_image_get_size_y(s2d_data);
    order_length = cpl_image_get_size_x(s2d_data);
    
    if (fibre_source == FP) {
        CCD_border_cut = inst_config->fp_ccd_border_cut;
    } else {
        CCD_border_cut = inst_config->thar_ccd_border_cut;
    }
    search_window_size = inst_config->fp_search_window_size;
    flux_threshold = inst_config->fp_flux_threshold;
    rms_threshold = inst_config->fp_peak_rms_threshold;
    noise_threshold = inst_config->fp_noise_threshold;
    delta_fp_peak_limit = inst_config->delta_fp_peak_limit;
    ngauss_flag = inst_config->fp_fit_ngauss_flag;
    
    if (ngauss_flag == 1) {
        p_data_size = 5;
        pxl_start = 2;
        pxl_end = 3;
    } else {
        p_data_size = 3;
        pxl_start = 1;
        pxl_end = 2;
    }
    //espdr_msg("p_data_size: %d, pxl_start: %d, pxl_end: %d",
    //          p_data_size, pxl_start, pxl_end);
    
    p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    
    p_data->x = (double *) cpl_calloc (p_data_size, sizeof (double));
    p_data->y = (double *) cpl_calloc (p_data_size, sizeof (double));
    p_data->err = (double *) cpl_calloc (p_data_size, sizeof (double));
    p_data->n = p_data_size;
    
    fit = (double *) cpl_calloc (p_data_size, sizeof(double));
    coeffs = (double *) cpl_calloc (3, sizeof(double));
    coeffs_err = (double *) cpl_calloc (3, sizeof(double));
    
    espdr_msg("Start espdrp_search_FP_lines, flux threshold: %lf, noise threshold: %lf, search window size: %d",
              flux_threshold, noise_threshold, search_window_size);
    
    index = 0;
    for (order = 0; order < orders_nb; order++) {
        espdr_msg_debug("Searching lines for order %d", order+1);
        order_image = cpl_image_extract(s2d_data, 1, order+1,
                                        order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_data");
        order_data = cpl_image_get_data_double(order_image);
        error_image = cpl_image_extract(s2d_error, 1, order+1,
                                        order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_error");
        error_data = cpl_image_get_data_double(error_image);
        quality_image = cpl_image_extract(s2d_quality, 1, order+1,
                                          order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_quality");
        quality_data = cpl_image_get_data_int(quality_image);
        
        pxl_min = 0;
        if (quality_data == NULL) espdr_msg("Q_data is NULL");
        while ((quality_data[pxl_min] == OTHER_BAD_PIXEL) &&
               (pxl_min < order_length)) {
            pxl_min++;
        }
        espdr_ensure(pxl_min == order_length, CPL_ERROR_INCOMPATIBLE_INPUT,
                     "The whole order is not valid, exiting");
        
        pxl_max = order_length-1;
        while ((quality_data[pxl_max] == OTHER_BAD_PIXEL) &&
               (pxl_max >= 0)) {
            pxl_max--;
        }
        espdr_ensure(pxl_max < 0, CPL_ERROR_INCOMPATIBLE_INPUT,
                     "The whole order is not valid, exiting");
        
        int half_window = search_window_size;
        //espdr_msg("Searching loop for order %d", order+1);
        //pxl = order_length/2.0; // SINGLEHR, SINGLEUHR, MULTIMR42
        pxl = (int)order_length/2; // MULTIMR84
        int peak_found = 1;
        int index_start = index;
        
        if (order == -1) {
            espdr_msg("Searching lines first half, order %d", order+1);
            espdr_msg("ERROR: %s", cpl_error_get_message_default(cpl_error_get_code()));
        }
        
        while ((pxl > pxl_min+CCD_border_cut) && (peak_found)) {
            rms = cpl_image_get_stdev_window(order_image, pxl-half_window+1, 1, pxl+half_window+1, 1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_stdev_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            mean = cpl_image_get_median_window(error_image, pxl-half_window+1, 1, pxl+half_window+1, 1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_median_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            my_error = cpl_image_get_maxpos_window(order_image, pxl-half_window+1, 1,
                                                   pxl+half_window+1, 1, &px, &py);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_maxpos_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            max_val = cpl_image_get_max_window(order_image, pxl-half_window+1, 1,
                                               pxl+half_window+1, 1);
            my_error = cpl_error_get_code();
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_max_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            if ((order == -1) && (pxl > 100) && (pxl < 200)) {
                espdr_msg("order %d pxl: %d, px: %lld, rms: %f, mean: %f",
                          order+1, pxl+1, px, rms, mean);
            }
            
            real_peak = 1;
            if (inst_config->fp_add_fake_peaks == 1) {
                if ((index > 1) && (max_val < 0.5*lines_table[index-1].peak_flux) &&
                    (max_val < 0.5*lines_table[index-2].peak_flux) && (lines_table[index-1].order == order+1)) {
                    real_peak = 0;
                }
            }
            
            if ((rms > (rms_threshold * mean)) && (px == pxl+1) && (real_peak == 1)) {
                if (order == -1) {
                    espdr_msg("Checking pxl: %d, rms: %f, mean: %f, threshold*mean: %f",
                              pxl, rms, mean, rms_threshold * mean);
                }
                // delta check
                if ((index - index_start) > 1) {
                    double delta_peak = pxl - 2 * lines_table[index-1].peak_pxl + lines_table[index-2].peak_pxl;
                    
                    // Real peak not found even if some missed and added artificially
                    if (fabs(delta_peak) > delta_fp_peak_limit) {
                        peak_found = 0;
                    }
                    
                    if (inst_config->fp_add_fake_peaks == 1) {
                        if ((peak_found == 0) && (index - index_start > 4)) {
                            double curr_interval = fabs(pxl - lines_table[index-1].peak_pxl);
                            double last_interval = fabs(lines_table[index-1].peak_pxl - lines_table[index-2].peak_pxl);
                            last_interval = (fabs(lines_table[index-1].peak_pxl - lines_table[index-2].peak_pxl) +
                                             fabs(lines_table[index-2].peak_pxl - lines_table[index-3].peak_pxl) +
                                             fabs(lines_table[index-3].peak_pxl - lines_table[index-4].peak_pxl))/3.0;
                            if (order == -1) {
                                espdr_msg("pxl: %d, pxl-1: %f, pxl-2: %f",
                                          pxl, lines_table[index-1].peak_pxl, lines_table[index-2].peak_pxl);
                                espdr_msg("index: %d --> current interval: %f, last interval: %f",
                                          index, curr_interval, last_interval);
                            }
                            if ((curr_interval > 1.5 * last_interval)) {
                                
                                // Add fake peaks if the difference is N * interval
                                int N = round(curr_interval / last_interval);
                                if ((fabs((curr_interval / last_interval) - N) < 0.3) &&
                                    (pxl > pxl_min+3*CCD_border_cut)) {
                                    espdr_msg("curr_interval: %f, last_interval: %f, N: %d, fabs: %f",
                                              curr_interval, last_interval, N, fabs((curr_interval / last_interval) - N));
                                    espdr_msg("Adding %d fake pixels in order %d (int difference: %f)",
                                              N-1, order+1, fabs((curr_interval / last_interval) - N));
                                    total_fake_peaks_nb += N-1;
                                    for (int i = 0; i < N-1 ; i++) {
                                        lines_table[index].order = lines_table[index-1].order;
                                        lines_table[index].peak_pxl = lines_table[index-1].peak_pxl - last_interval;
                                        lines_table[index].peak_pxl_err = last_interval / 2.0;
                                        lines_table[index].peak_flux = lines_table[index-1].peak_flux;
                                        lines_table[index].peak_flux_err = lines_table[index-1].peak_flux_err;
                                        lines_table[index].qc_flag = 0;
                                        espdr_msg("At pxl: %f", lines_table[index].peak_pxl);
                                        index++;
                                    }
                                    peak_found = 1;
                                }
                            }
                        }
                        
                        if ((order == -1) && (pxl > 240) && (pxl < 285)) {
                            espdr_msg("delta_peak: %f, delta_limit: %f, peak_found: %d",
                                      delta_peak, delta_fp_peak_limit, peak_found);
                        }
                    } // if fp_add_fake_peaks
                }
                
                if (peak_found) {
                    max_flux = 0.0;
                    max_flux_pxl = pxl;
                    for (i = pxl; i < pxl+2; i++) {
                        if (order_data[i-1] < order_data[i]) {
                            max_flux = order_data[i];
                            max_flux_pxl = i;
                        }
                    }
                    /* fit poly */
                    if ((order == -1) && ((pxl > 100) && (pxl < 200))) {
                        espdr_msg("Fit for pxl %d", pxl);
                    }
                    index_p_data = 0;
                    for (i = pxl-pxl_start; i < pxl+pxl_end; i++) {
                        p_data->x[index_p_data] = i;
                        p_data->y[index_p_data] = order_data[i];
                        p_data->err[index_p_data] = error_data[i];
                        index_p_data++;
                    }
                    
                    my_error = espdr_fit_poly(p_data, 3,
                                              fit, coeffs, coeffs_err, &chisq, 0);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_fit_poly failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    fitted_pic_dist = -coeffs[1] / (2.0 * coeffs[2]) - max_flux_pxl;
                    if ((fitted_pic_dist < 1.5) && (fitted_pic_dist > -1.5)) {
                        lines_table[index].peak_pxl = -coeffs[1] / (2.0 * coeffs[2]) +1;
                    } else {
                        lines_table[index].peak_pxl = max_flux_pxl+1;
                    }
                    lines_table[index].peak_flux = order_data[max_flux_pxl];
                    lines_table[index].order = order+1;
                    
                    if ((order == -1) && (pxl > 1230) && (pxl < 1270)) {
                        espdr_msg("lines_table[%d].peak_pxl = %lf",
                                  index, lines_table[index].peak_pxl);
                    }
                    
                    index++;
                }
            }
            pxl--;
        }
        
        // sort lines_table for first half of the order
        if (order == -1) {
            espdr_msg("Sorting first half lines for order %d", order+1);
            espdr_msg("ERROR: %s", cpl_error_get_message_default(cpl_error_get_code()));
        }
        int peaks_nb = index - index_start;
        espdr_line_param found_peaks_table[index];
        for (i = 0; i < peaks_nb; i++) {
            found_peaks_table[i].peak_pxl = lines_table[i+index_start].peak_pxl;
            found_peaks_table[i].peak_pxl_err = lines_table[i+index_start].peak_pxl_err;
            found_peaks_table[i].peak_flux = lines_table[i+index_start].peak_flux;
            found_peaks_table[i].peak_flux_err = lines_table[i+index_start].peak_flux_err;
            found_peaks_table[i].order = lines_table[i+index_start].order;
            found_peaks_table[i].qc_flag = lines_table[i+index_start].qc_flag;
        }
        sort_lines_pxl(found_peaks_table, peaks_nb);
        for (i = 0; i < peaks_nb; i++) {
            lines_table[i+index_start].peak_pxl = found_peaks_table[i].peak_pxl;
            lines_table[i+index_start].peak_pxl_err = found_peaks_table[i].peak_pxl_err;
            lines_table[i+index_start].peak_flux = found_peaks_table[i].peak_flux;
            lines_table[i+index_start].peak_flux_err = found_peaks_table[i].peak_flux_err;
            lines_table[i+index_start].order = found_peaks_table[i].order;
            lines_table[i+index_start].qc_flag = found_peaks_table[i].qc_flag;
            found_peaks_table[i].peak_pxl = 0.0;
            found_peaks_table[i].peak_pxl_err = 0.0;
            found_peaks_table[i].peak_flux = 0.0;
            found_peaks_table[i].peak_flux_err = 0.0;
            found_peaks_table[i].order = 0;
            found_peaks_table[i].qc_flag = 0;
            if ((lines_table[i+index_start].order == -1) &&
                (lines_table[i+index_start].peak_pxl > -1) &&
                (lines_table[i+index_start].peak_pxl < 2100000000)) {
                espdr_msg("lines_table[%d].peak_pxl = %lf",
                          i+index_start, lines_table[i+index_start].peak_pxl);
            }
        }
        
        if (order == -1) {
            espdr_msg("Searching lines second half, order %d", order+1);
            espdr_msg("ERROR: %s", cpl_error_get_message_default(cpl_error_get_code()));
        }
        
        pxl = (int)order_length/2 + 1;
        peak_found = 1;
        while ((pxl < pxl_max-CCD_border_cut) && (peak_found)) {
            rms = cpl_image_get_stdev_window(order_image, pxl-half_window+1, 1,
                                             pxl+half_window+1, 1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_stdev_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            mean = cpl_image_get_median_window(error_image, pxl-half_window+1, 1,
                                               pxl+half_window+1, 1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_median_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            my_error = cpl_image_get_maxpos_window(order_image, pxl-half_window+1, 1,
                                                   pxl+half_window+1, 1, &px, &py);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_maxpos_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            max_val = cpl_image_get_max_window(order_image, pxl-half_window+1, 1,
                                                      pxl+half_window+1, 1);
            my_error = cpl_error_get_code();
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_get_max_window failed: %s",
                         cpl_error_get_message_default(my_error));
            
            if (order == -1 && pxl > 2400 && pxl < 2415) {
                espdr_msg("Testing pxl %d flux: %lf, rms: %lf, mean: %lf",
                          pxl, order_data[pxl], rms, mean);
                espdr_msg("ERROR: %s",
                          cpl_error_get_message_default(cpl_error_get_code()));
            }
            if (order == -1 && pxl > 2400 && pxl < 2415) {
                if ((rms > (3.0 * mean)) && (px == pxl+1)) {
                    espdr_msg("Peak found at %d", pxl);
                } else {
                    espdr_msg("Peak not found: rms = %lf, mean = %lf",
                              rms, mean);
                }
            }
            
            real_peak = 1;
            if (inst_config->fp_add_fake_peaks == 1) {
                if ((index > 1) && (max_val < 0.5*lines_table[index-1].peak_flux) &&
                    (max_val < 0.5*lines_table[index-2].peak_flux) && (lines_table[index-1].order == order+1)) {
                    real_peak = 0;
                }
            }
            
            if ((order == -1) && (pxl > 2485) & (pxl < 2540)) {
                espdr_msg("pxl: %d, rms = %f, mean = %f, threshold*mean = %f, real_peak = %d, max_val = %f, prev max = %f",
                          pxl, rms, mean, rms_threshold * mean, real_peak, max_val, lines_table[index-1].peak_flux);
            }
            
            if ((rms > (rms_threshold * mean)) && (px == pxl+1) && (real_peak == 1)) {
                
                // delta check
                if ((index - index_start) > 1) {
                    double delta_peak = pxl - 2 * lines_table[index-1].peak_pxl + lines_table[index-2].peak_pxl;
                    if (order == -1 && pxl > 2400 && pxl < 2415) {
                        espdr_msg("Testing delta %lf, peak[%d] = %lf, peak[%d] = %lf",
                                  delta_peak, index-2, lines_table[index-2].peak_pxl,
                                  index-1, lines_table[index-1].peak_pxl);
                    }
                    if (fabs(delta_peak) > delta_fp_peak_limit) {
                        peak_found = 0;
                    }
                    
                    if (inst_config->fp_add_fake_peaks == 1) {
                        if ((peak_found == 0) && (index - index_start > 4)) {
                            double curr_interval = fabs(pxl - lines_table[index-1].peak_pxl);
                            double last_interval = fabs(lines_table[index-1].peak_pxl - lines_table[index-2].peak_pxl);
                            last_interval = (fabs(lines_table[index-1].peak_pxl - lines_table[index-2].peak_pxl) +
                                             fabs(lines_table[index-2].peak_pxl - lines_table[index-3].peak_pxl) +
                                             fabs(lines_table[index-3].peak_pxl - lines_table[index-4].peak_pxl))/3.0;
                            if (order == -1) {
                                espdr_msg("pxl: %d, pxl-1: %f, pxl-2: %f",
                                          pxl, lines_table[index-1].peak_pxl, lines_table[index-2].peak_pxl);
                                espdr_msg("index: %d --> current interval: %f, last interval: %f", index, curr_interval, last_interval);
                            }
                            if ((curr_interval > 1.5 * last_interval)) {
                                
                                // Add fake peaks if the difference is N * interval
                                int N = round(curr_interval / last_interval);
                                if ((fabs((curr_interval / last_interval) - N) < 0.3) &&
                                    (pxl < pxl_max-3*CCD_border_cut)) {
                                    espdr_msg("Adding %d fake pixels in order %d", N-1, order+1);
                                    total_fake_peaks_nb += N-1;
                                    for (int i = 0; i < N-1 ; i++) {
                                        lines_table[index].order = lines_table[index-1].order;
                                        lines_table[index].peak_pxl = lines_table[index-1].peak_pxl + last_interval;
                                        lines_table[index].peak_pxl_err = last_interval / 2.0;
                                        lines_table[index].peak_flux = lines_table[index-1].peak_flux;
                                        lines_table[index].peak_flux_err = lines_table[index-1].peak_flux_err;
                                        lines_table[index].qc_flag = 0;
                                        espdr_msg("At pxl: %f", lines_table[index].peak_pxl);
                                        index++;
                                    }
                                    peak_found = 1;
                                }
                            }
                        }
                    } // if fp_add_fake_peaks
                }
                
                if (order == -1) {
                    espdr_msg("Peak found (%d) at pxl %d", peak_found, pxl);
                }
                
                if (peak_found) {
                    max_flux = 0.0;
                    max_flux_pxl = pxl;
                    for (i = pxl; i < pxl+2; i++) {
                        if (order_data[i-1] < order_data[i]) {
                            max_flux = order_data[i];
                            max_flux_pxl = i;
                        }
                    }
                    /* fit poly */
                    if (order == -1 && pxl > 2400 && pxl < 2415) {
                        espdr_msg("Fit for pxl %d", pxl);
                    }
                    index_p_data = 0;
                    for (i = pxl-pxl_start; i < pxl+pxl_end; i++) {
                        p_data->x[index_p_data] = i;
                        p_data->y[index_p_data] = order_data[i];
                        p_data->err[index_p_data] = error_data[i];
                        index_p_data++;
                    }
                    
                    my_error = espdr_fit_poly(p_data, 3,
                                              fit, coeffs, coeffs_err, &chisq, 0);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_fit_poly failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    
                    fitted_pic_dist = -coeffs[1] / (2.0 * coeffs[2]) - max_flux_pxl;
                    if ((fitted_pic_dist < 1.5) && (fitted_pic_dist > -1.5)) {
                        lines_table[index].peak_pxl = -coeffs[1] / (2.0 * coeffs[2]) +1;
                    } else {
                        lines_table[index].peak_pxl = max_flux_pxl+1;
                    }
                    lines_table[index].peak_flux = order_data[max_flux_pxl];
                    lines_table[index].order = order+1;
                    
                    if (order == -1 && pxl > 2400 && pxl < 2415) {
                        espdr_msg("lines_table[%d].peak_pxl = %lf",
                                  index, lines_table[index].peak_pxl);
                    }
                    
                    index++;
                }
            }
            pxl++;
        }
        
        cpl_image_delete(order_image);
        cpl_image_delete(error_image);
        cpl_image_delete(quality_image);
    }
    
    espdr_msg("Total number of fake peaks added: %d", total_fake_peaks_nb);
    
    cpl_free(p_data->x);
    cpl_free(p_data->y);
    cpl_free(p_data->err);
    cpl_free(p_data);
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    
    // define the fit window as distance between two peaks, put in N
    lines_table[0].N = (int)((lines_table[1].peak_pxl - lines_table[0].peak_pxl + 1.0)/2.0);
    if ((lines_table[0].N < 3) && (ngauss_flag == 1)) { // For SINGLEHR and SINGLEUHR and MULTIMR42
        lines_table[0].qc_flag = 0;
    }
    if ((lines_table[0].N < 2) && (ngauss_flag == 0)) { // For MULTIMR84
        lines_table[0].qc_flag = 0;
    }
    
    for (i = 1; i < index; i++) {
        if (lines_table[i].order == lines_table[i-1].order) {
            lines_table[i].N = (int)((lines_table[i].peak_pxl - lines_table[i-1].peak_pxl + 1.0)/2.0);
        } else {
            lines_table[i].N = (int)((lines_table[i+1].peak_pxl - lines_table[i].peak_pxl + 1.0)/2.0);
        }
        if ((lines_table[i].order == lines_table[i-1].order) &&
            abs(lines_table[i].N - lines_table[i-1].N) > 1) {
            //espdr_msg_warning("FP peak lost for order %d at pxl %lf, delta = %d, peak_pxl = %lf, %lf, %lf",
            //                  lines_table[i].order, lines_table[i].peak_pxl,
            //                  abs(lines_table[i].N - lines_table[i-1].N),
            //                  lines_table[i-2].peak_pxl, lines_table[i-1].peak_pxl, lines_table[i].peak_pxl);
        }
        
        if ((lines_table[i].N < 3) && (ngauss_flag == 1)) {
            lines_table[i].qc_flag = 0;
        }
        if ((lines_table[i].N < 2) && (ngauss_flag == 0)) {
            lines_table[i].qc_flag = 0;
        }
    }
    
    
    *lines_nb_RE = index;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Fit reference ThAr calibration lines
 @param[out]    lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         s2d                     input data
 @param         s2d_err                 input data errors
 @param         s2d_qual                input data quality flags
 @param         pixel_geom              pixel geometry image
 @param         flux_threshold          flux treshold for line detection
 @param         min_x0_err              limit for the peak pixel error
 @param         window_size             lines fitting window size
 @param[out]    lines_nb_per_order_RE   computed number of lines for each order
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_lines(espdr_line_param *lines_table,
                               int lines_nb,
                               cpl_image *s2d,
                               cpl_image *s2d_err,
                               cpl_image *s2d_qual,
                               cpl_image *pixel_geom,
                               double flux_threshold,
                               //double min_x0_err,
                               int window_size,
                               int tolerance_window,
                               int NGauss_flag,
                               int *lines_nb_per_order_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int line_index, fit_index;
    int order, fit_start, fit_end, fit_vector_length, pxl;
    int left_margin, right_margin, high_pxl;
    double max_flux, min_flux1, min_flux2;
    cpl_image *s2d_order = NULL;
    cpl_image *s2d_line = NULL;
    cpl_image *s2d_err_order = NULL;
    cpl_image *pxl_geom_order = NULL;
    double *s2d_order_data = NULL;
    //double *s2d_line_data = NULL;
    double *s2d_err_order_data = NULL;
    double *pxl_geom_order_data = NULL;
    int s2d_size_x = 0;
    int s2d_size_y = 0;
    espdr_ngauss_data *g_data = NULL;
    espdr_ngauss_result *g_res = NULL;
    double *x0 = NULL;
    cpl_size px, py;
    
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Lines input table is NULL");
    espdr_ensure(s2d == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image is NULL");
    espdr_ensure(s2d_err == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image error is NULL");
    espdr_ensure(s2d_qual == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image quality is NULL");
    espdr_ensure(pixel_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "pixel geometry image is NULL");
    
    g_data = (espdr_ngauss_data *)cpl_malloc(sizeof(espdr_ngauss_data));
    g_res = (espdr_ngauss_result *)cpl_malloc(sizeof(espdr_ngauss_result));
    
    s2d_size_x = cpl_image_get_size_x(s2d);
    s2d_size_y = cpl_image_get_size_y(s2d);
    espdr_msg("S2D size: %d x %d", s2d_size_x, s2d_size_y);
    
    line_index = 0;
    //tolerance_window = 2; // MULTIMR84
    while (line_index < lines_nb) {
        fit_index = 0;
        order = lines_table[line_index].order;
        window_size = lines_table[line_index].N;
        if ((lines_table[line_index].qc_flag == 1) && (order <= s2d_size_y) &&
            (window_size > 1)) {
            left_margin = (int)(lines_table[line_index].peak_pxl + 0.5) - window_size;
            if (left_margin <= 0) {
                lines_table[line_index].qc_flag = 2;
            } else {
                right_margin = (int)(lines_table[line_index].peak_pxl + 0.5) + window_size;
                if (right_margin >= s2d_size_x) {
                    lines_table[line_index].qc_flag = 2;
                } else {
                    high_pxl = (int)(lines_table[line_index].peak_pxl + 0.5);
                    my_error = cpl_image_get_maxpos_window(s2d,
                                                           high_pxl-tolerance_window, order,
                                                           high_pxl+tolerance_window, order,
                                                           &px, &py);
                    
                    if (llabs(high_pxl - px) < tolerance_window) {
                        //espdr_msg("Peak position: %lld, %lld, high pxl: %d", px, py, high_pxl);
                        
                        high_pxl = px;
                        s2d_line = cpl_image_extract(s2d,
                                                     high_pxl-window_size, order,
                                                     high_pxl+window_size, order);
                        //espdr_msg("s2d_line size x: %lld, window_size: %d, high pxl: %d",
                        //          cpl_image_get_size_x(s2d_line), window_size, high_pxl);
                        my_error = cpl_error_get_code();
                        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                     "Extract S2D line: order = %d, high_pxl = %d: %s",
                                     order, high_pxl,
                                     cpl_error_get_message_default(my_error));
                        
                        max_flux = cpl_image_get_max(s2d_line);
                        min_flux1 = cpl_image_get_min_window(s2d_line,
                                                             1, 1,
                                                             window_size+1, 1);
                        min_flux2 = cpl_image_get_min_window(s2d_line,
                                                             window_size+1, 1,
                                                             window_size*2+1, 1);
                        
                        if (order == -1) {
                            espdr_msg("order: %d, pxl: %lf, max_flux: %lf, min_flux: %lf",
                                      order, lines_table[line_index].peak_pxl,
                                      max_flux, (min_flux1+min_flux2)/2.0);
                        }
                        
                        if ((max_flux - ((min_flux1+min_flux2)/2.0)) > flux_threshold) {
                            // Do the fit
                            //espdr_msg("Preparing the fit of line %d at pxl %d in order %d",
                            //          line_index, high_pxl, order);
                            fit_index = 1;
                            fit_start = high_pxl - window_size;
                            fit_end = high_pxl + window_size;
                            espdr_ensure(fit_end > s2d_size_x, CPL_ERROR_ILLEGAL_OUTPUT,
                                         "fit window end outside of the CCD");
                            
                            s2d_order = cpl_image_extract(s2d, fit_start, order, fit_end, order);
                            my_error = cpl_error_get_code();
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "Extract S2D order %d: %s", order,
                                         cpl_error_get_message_default(my_error));
                            
                            s2d_err_order = cpl_image_extract(s2d_err, fit_start,
                                                              order, fit_end, order);
                            
                            pxl_geom_order = cpl_image_extract(pixel_geom,
                                                               fit_start, order, fit_end, order);
                            s2d_order_data = cpl_image_get_data_double(s2d_order);
                            s2d_err_order_data = cpl_image_get_data_double(s2d_err_order);
                            pxl_geom_order_data = cpl_image_get_data_double(pxl_geom_order);
                            
                            
                            fit_vector_length = fit_end - fit_start + 1;
                            g_data->n = fit_vector_length;
                            g_data->m = fit_index;
                            g_data->x = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                            g_data->y = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                            g_data->err = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                            g_data->fit = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                            
                            g_res->k = (double *)cpl_calloc(fit_index, sizeof (double));
                            g_res->sig_k = (double *)cpl_calloc(fit_index, sizeof (double));
                            g_res->x0 = (double *)cpl_calloc(fit_index, sizeof (double));
                            g_res->sig_x0 = (double *)cpl_calloc(fit_index, sizeof (double));
                            g_res->fwhm = (double *)cpl_calloc(fit_index, sizeof (double));
                            g_res->sig_fwhm = (double *)cpl_calloc(fit_index, sizeof (double));
                            
                            x0 = (double *)cpl_calloc(fit_index, sizeof(double));
                            x0[0] = high_pxl - fit_start;
                            
                            if (line_index == -1) {
                                espdr_msg("window size: %d, fit start: %d, fit end: %d",
                                          window_size, fit_start, fit_end);
                            }
                            for (pxl = 0; pxl <= fit_end - fit_start; pxl++) {
                                g_data->x[pxl] = pxl_geom_order_data[pxl] - fit_start;
                                g_data->y[pxl] = s2d_order_data[pxl];
                                g_data->err[pxl] = s2d_err_order_data[pxl];
                                if (line_index == -1) {
                                    espdr_msg("BEFORE NGAUSS: pxl: %d, x: %lf, y: %lf, err: %lf",
                                              pxl, g_data->x[pxl], g_data->y[pxl],
                                              g_data->err[pxl]);
                                }
                            }
                            
                            //espdr_msg("Fitting line at pxl %d in order %d", high_pxl, order);
                            
                            // MULTIMR84
                            if (NGauss_flag) {
                                my_error = espdr_fit_Ngauss(g_data, x0, 3.0, g_res, 1, 0);
                                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                             "espdr_fit_Ngauss failed: %s",
                                             cpl_error_get_message_default(my_error));
                                
                                // If the fit moves the peak_pxl too far away, no assignement and qc_flag = 0
                                if (fabs(lines_table[line_index].peak_pxl - (g_res->x0[0] + fit_start)) > 2.0) {
                                    lines_table[line_index].qc_flag = 0;
                                } else {
                                    lines_table[line_index].peak_pxl = g_res->x0[0] + fit_start;
                                    lines_table[line_index].peak_pxl_err = g_res->sig_x0[0];
                                    lines_table[line_index].fwhm = g_res->fwhm[0];
                                    lines_table[line_index].fwhm_err = g_res->sig_fwhm[0];
                                    lines_table[line_index].peak_flux = g_res->k[0];
                                    lines_table[line_index].peak_flux_err = g_res->sig_k[0];
                                }
                            } else { // set bigger error if no Ngauss fit, only the 2nd degree polynomial
                                lines_table[line_index].peak_pxl_err = 0.05;
                            }
                            
                            //if (lines_table[line_index].peak_pxl_err < min_x0_err) {
                            //    lines_table[line_index].peak_pxl_err = min_x0_err;
                            //}
                            
                            if (line_index == -1) {
                                espdr_msg("AFTER NGAUSS: peak_pxl: %lf, fwhm: %lf, peak_flux: %lf, fit_start: %d",
                                          lines_table[line_index].peak_pxl,
                                          lines_table[line_index].fwhm,
                                          lines_table[line_index].peak_flux,
                                          fit_start);
                            }
                            
                            //espdr_msg("Cleaning fit data structures memory");
                            cpl_free(x0);
                            
                            cpl_free(g_data->x);
                            cpl_free(g_data->y);
                            cpl_free(g_data->fit);
                            cpl_free(g_data->err);
                            
                            cpl_free(g_res->k);
                            cpl_free(g_res->sig_k);
                            cpl_free(g_res->x0);
                            cpl_free(g_res->sig_x0);
                            cpl_free(g_res->fwhm);
                            cpl_free(g_res->sig_fwhm);
                            
                            cpl_image_delete(s2d_order);
                            cpl_image_delete(s2d_err_order);
                            cpl_image_delete(pxl_geom_order);
                            
                            lines_nb_per_order_RE[order-1]++;
                            
                        } else {
                            espdr_msg_warning("Peak height too small at pixel %d (%d) in order %d: %lf",
                                              high_pxl, high_pxl + ORDER_SHIFT, order,
                                              (max_flux - ((min_flux1+min_flux2)/2.0)));
                            lines_table[line_index].qc_flag = 0;
                            //if (order == 10) {
                            //    espdr_msg("order: %d, pxl: %lf rejected, max_flux: %lf, min_flux: %lf",
                            //              order,
                            //              lines_table[line_index].peak_pxl,
                            //              max_flux, (min_flux1+min_flux2)/2.0);
                            //}
                        } // if ((max_flux - ((min_flux1+min_flux2)/2.0)) > flux_threshold)
                        cpl_image_delete(s2d_line);
                    } else {
                        espdr_msg_warning("FP line not found at pixel %d (%d) in order %d",
                                          high_pxl, high_pxl + ORDER_SHIFT, order);
                        lines_table[line_index].qc_flag = 0;
                    } // if (fabs(high_pxl - px) < tolerance_window)
                    
                } // if (right_margin >= s2d_size_x)
            } // (left_margin <= 0)
        } // if ((lines_table[line_index].qc_flag == 1) && (order <= s2d_size_y))
        line_index++;
    } // while (line_index < lines_nb)
    
    cpl_free(g_data);
    cpl_free(g_res);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Compute the weighted RMS
 @param x   vector of values
 @param w   vector of weights
 @param len vector length
 @return    computed weighted RMS
 */
/*----------------------------------------------------------------------------*/

double espdr_wrms(double *x, double *w, int len) {
    
    int i, OK = 0;
    double result = 0.0, sum_up = 0.0, sum_down = 0.0;
    double sum_wx = 0.0, sum_w = 0.0, sum_w2 = 0.0;
    
    /* Checks */
    if (len <= 1) return 0.0;
    for (i = 0; i < len; i++) {
        if (w[i] != 0.0) OK++;
    }
    if (OK <= 1) return 0.0;
    
    /* Computation */
    
    for (i = 0; i < len; i++) {
        sum_wx += w[i] * x[i];
        sum_w += w[i];
        sum_w2 += w[i] * w[i];
    }
    
    for (i = 0; i < len; i++) {
        sum_up += w[i] * (x[i] - sum_wx/sum_w) * (x[i] - sum_wx/sum_w);
    }
    sum_down = sum_w - sum_w2 / sum_w;
    
    result = sqrt(sum_up/sum_down);
    
    return (result);
}



/*----------------------------------------------------------------------------*/
/**
 @brief Fit wavelength solution
 @param         lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         inst_config             instrument configuration
 @param         pixel_geom_fiber        pixel geometry image
 @param         source                  calibration source type
 @param[out]    lines_nb_per_order_RE   number of lines for each order
 @param[out]    rms_per_order_RE        RMS for each order
 @param[out]    chisq_per_order_RE      CHI2 for each order
 @param[out]    coeffs_RE               polynomial fit coefficients
 @param[out]    wave_matrix_RE          matrix of wavelangths for all s2d pixels
 @param[out]    dll_RE                  pixels width in ll
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_ll_sol(espdr_line_param *lines_table,
                                int lines_nb,
                                espdr_inst_config *inst_config,
                                cpl_image *pixel_geom_fiber,
                                cpl_image *pixel_size_fiber,
                                espdr_src_type source,
                                int fibre_nr,
                                int *lines_nb_per_order_RE,
                                double *rms_per_order_RE,
                                double *chisq_per_order_RE,
                                int *order_fit,
                                double **coeffs_RE,
                                double *wave_matrix_RE,
                                double *dll_RE,
                                double *air_wave_matrix_RE,
                                double *air_dll_RE) {
    
    int i, j, k, l, order, last_order = 0;
    int oln, oln2, index, pxl;
    int wave_matrix_index = 0;
    int dll_index = 0;
    int poly_deg, gsize_x, gsize_y, size_x, size_y;
    double ll = 0.0, dxdll = 0.0, dlldx = 0.0;
    espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    espdr_poly_data *p2_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    double *fit = NULL;
    double *fit2 = NULL;
    double *coeffs = NULL;
    double *coeffs_err = NULL;
    double chisq = 0.0;
    
    double *x_vector = NULL;
    double *weights = NULL;
    
    cpl_image *pxl_geom_order = NULL;
    double *pxl_geom_data = NULL;
    cpl_image *pxl_size_order = NULL;
    double *pxl_size_data = NULL;
    
    int min_nb_lines_per_order = 0;
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(pixel_geom_fiber == NULL, CPL_ERROR_NULL_INPUT,
                 "Pixel geometry image is NULL");
    
    if (source == THAR) {
        min_nb_lines_per_order = inst_config->wave_sol_min_lines_per_order;
    } else {
        min_nb_lines_per_order = inst_config->lfc_sol_min_lines_per_order[fibre_nr];
    }
    
    size_x = cpl_image_get_size_x(pixel_size_fiber);
    size_y = cpl_image_get_size_y(pixel_size_fiber);
    //espdr_msg("pixel size size: %d x %d", size_x, size_y);
    
    gsize_x = cpl_image_get_size_x(pixel_geom_fiber);
    gsize_y = cpl_image_get_size_y(pixel_geom_fiber);
    //espdr_msg("pixel geom size: %d x %d", gsize_x, gsize_y);
    
    espdr_ensure(((size_x != gsize_x) || (size_y != gsize_y)),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "The sizes of pixel geom (%d x %d) and pixel size (%d x %d) maps differ",
                 gsize_x, gsize_y, size_x, size_y);
    
    poly_deg = inst_config->wave_sol_poly_deg;
    espdr_msg("Fitting ll solution with polynomial of degree %d", poly_deg);
    
    i = 0;
    while ((i < lines_nb) && (lines_table[i].order <= size_y)) {
        order = lines_table[i].order;
        lines_nb_per_order_RE[order-1] = 0;
        j = i;
        while ((j < lines_nb) && (lines_table[j].order == order)) {
            //if (espdr_is_line_valid(lines_table[j], source)) {
            if (lines_table[j].qc_flag == 1) {
                lines_nb_per_order_RE[order-1]++;
            }
            j++;
        }
        
        // Added to fill up wave_matrix & dll_matrix with zeros for the orders non existing in the table
        while ((order - last_order) > 1) {
            wave_matrix_index += size_x;
            dll_index += size_x;
            last_order++;
            espdr_msg("FIT_LL: NOT fitting lines for order: %d --> no lines",
                      last_order);
            //order_fit[last_order-1] = 0;
        }
        last_order++;
        
        if (lines_nb_per_order_RE[order-1] < min_nb_lines_per_order) {
            
            espdr_msg_warning("FIT_LL: Not enough lines (%d) in order %d to compute the wavelength solution. Should be at least %d",
                              lines_nb_per_order_RE[order-1], order,
                              min_nb_lines_per_order);
            // Add the order length to the wave_matrix_index,
            // in order to compute the ll for the right order
            // When too few lines, the ll sol will be 0
            wave_matrix_index += size_x;
            dll_index += size_x;
            //order_fit[order-1] = 0;
            
        } else {
            
            espdr_msg("FIT_LL: Fitting lines for order: %d with %d lines",
                      order, lines_nb_per_order_RE[order - 1]);
            
            oln = lines_nb_per_order_RE[order-1];
            //espdr_msg("Enough lines (%d), order %d will be fitted",
            //          oln, order);
            
            //espdr_msg("Lines nb per order: %d", oln);
            
            /* Fit the x0(lambda) */
            
            p_data->x = (double *) cpl_calloc (oln, sizeof(double));
            p_data->y = (double *) cpl_calloc (oln, sizeof(double));
            p_data->err = (double *) cpl_calloc (oln, sizeof(double));
            p_data->n = oln;
            
            /*
             for (k = i; k < j; k++) {
             if (order == 35) {
             espdr_msg("lines_table[%d]: ll = %lf, pxl = %lf, err = %lf, qc = %d", k,
             lines_table[k].wavelength,
             lines_table[k].peak_pxl,
             lines_table[k].peak_pxl_err,
             lines_table[k].qc_flag);
             }
             }
             */
            
            index = 0;
            for (k = i; k < j; k++) {
                //if (espdr_is_line_valid(lines_table[k], source)) {
                if (lines_table[k].qc_flag == 1) {
                    ll = lines_table[k].wavelength;
                    // Temporary calculation set in vacuum
                    //p_data->x[index] = ll / espdr_compute_n_air(ll,
                    //                                            N_AIR_COEFF_T,
                    //                                            N_AIR_COEFF_P);
                    p_data->x[index] = ll;
                    p_data->y[index] = lines_table[k].peak_pxl;
                    p_data->err[index] = sqrt(lines_table[k].peak_pxl_err *
                                              lines_table[k].peak_pxl_err +
                                              inst_config->line_min_sig_x0 *
                                              inst_config->line_min_sig_x0);
                    // If the error on peak position is 0, it makes the fit nan, so we put it to a big value
                    if (p_data->err[index] == 0) p_data->err[index] = 10.0;
                    index++;
                }
            }
            
            if (order == -1) {
                for (k = 0; k < oln; k++) {
                    espdr_msg("FIRST FIT [%d] x = %lf, y = %lf, err = %lf",
                              k, p_data->x[k], p_data->y[k], p_data->err[k]);
                }
            }
            
            //espdr_msg("Middle(%d) for order %d ll_air: %lf, ll: %lf",
            //          (int)index/2, order, p_data->x[(int)index/2],
            //          p_data->x[(int)index/2] *
            //          espdr_compute_n_air(p_data->x[(int)index/2], 15.0, 760.0));
            
            fit = (double *) cpl_calloc (oln, sizeof(double));
            coeffs = (double *) cpl_calloc (poly_deg, sizeof(double));
            coeffs_err = (double *) cpl_calloc (poly_deg, sizeof(double));
            
            my_error = espdr_fit_poly(p_data, poly_deg,
                                      fit, coeffs, coeffs_err, &chisq, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            if (order == -1) {
                espdr_msg("Order %d fitted, with CHI2 = %lf, COEFFS: %lf, %lf, %lf, %lf",
                          order, chisq,
                          coeffs[0], coeffs[1], coeffs[2], coeffs[3]);
                espdr_msg("FIRST FIT");
                for (k = 0; k < oln; k++) {
                    espdr_msg("fit[%d] = %lf", k, fit[k]);
                }
                
                double chi_self = 0.0;
                for (k = 0; k < oln; k++) {
                    chi_self += (fit[k] - p_data->y[k]) *
                    (fit[k] - p_data->y[k]);
                }
                chi_self = chi_self / (oln - poly_deg);
                espdr_msg("CHI2 self: %lf", chi_self);
            }
            
            
            /* Get the lambda errors from the fit */
            index = 0;
            for (k = i; k < j; k++) {
                //if (espdr_is_line_valid(lines_table[k], source)) {
                if (lines_table[k].qc_flag == 1) {
                    dxdll = 0.0;
                    ll = lines_table[k].wavelength;
                    for (l = poly_deg - 1; l > 1; l--) {
                        dxdll = (dxdll + l * coeffs[l]) * ll;
                    }
                    dxdll = dxdll + coeffs[1];
                    p_data->err[index] = sqrt(lines_table[k].peak_pxl_err *
                                              lines_table[k].peak_pxl_err +
                                              inst_config->line_min_sig_x0 *
                                              inst_config->line_min_sig_x0)/dxdll;
                    // If the error on peak position is 0, it makes the fit nan, so we put it to a big value
                    if (p_data->err[index] == 0) p_data->err[index] = 10.0;
                    //p_data->y[index] = p_data->x[index];
                    // Temporary calculation set in vacuum
                    //p_data->y[index] = ll / espdr_compute_n_air(ll,
                    //                                            N_AIR_COEFF_T,
                    //                                            N_AIR_COEFF_P);
                    p_data->y[index] = ll;
                    p_data->x[index] = lines_table[k].peak_pxl;
                    index++;
                }
            }
            
            if ((order == -1) || (order == -2)) {
                for (k = 0; k < oln; k++) {
                    espdr_msg("[%d] x = %lf, y = %lf, err = %lf",
                              k, p_data->x[k], p_data->y[k], p_data->err[k]);
                }
            }
            
            //espdr_msg("Errors on x0 computed");
            
            /* Fit the lambda(x0) */
            
            my_error = espdr_fit_poly(p_data, poly_deg,
                                      fit, coeffs, coeffs_err, &chisq, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("Second fit for order %d done", order);
            
            // Remove outliers & do the second fit
            
            double mean_mic = 0.0;
            double std_mic = 0.0;
            for (k = 0; k < oln; k++) {
                mean_mic += (fit[k] - p_data->y[k]);
            }
            for (k = 0; k < oln; k++) {
                std_mic += pow(((fit[k] - p_data->y[k]) - mean_mic/oln),2.0);
            }
            std_mic = sqrt(std_mic/(oln-1));
            //espdr_msg("STD : %f, MEAN2 : %f",std_mic, mean_mic/oln);
            
            oln2 = 0;
            for (k = 0; k < oln; k++) {
                if (fabs(fit[k] - p_data->y[k]) <
                    (inst_config->wave_sol_FP_reject_alpha * p_data->err[k] +
                     inst_config->wave_sol_FP_reject_beta * fabs(std_mic))) {
                    oln2++;
                } else {
                    if (order == -1) {
                        espdr_msg("FP line in order %d at pxl %lf (in table %lf) rejected, distance: %f (%f - %f) (threshold: %f)",
                                  order, p_data->x[k], lines_table[i+k].peak_pxl,
                                  fabs(fit[k] - p_data->y[k]),
                                  fit[k], p_data->y[k],
                                  inst_config->wave_sol_FP_reject_alpha * p_data->err[k] +
                                  inst_config->wave_sol_FP_reject_beta * fabs(std_mic));
                    }
                    
                    lines_table[i+k].qc_flag = 0;
                }
            }
            
            espdr_msg("Nb of lines rejected for order %d: %d of %d",
                      order, oln-oln2, oln);
            
            p2_data->x = (double *) cpl_calloc (oln2, sizeof(double));
            p2_data->y = (double *) cpl_calloc (oln2, sizeof(double));
            p2_data->err = (double *) cpl_calloc (oln2, sizeof(double));
            p2_data->n = oln2;
            
            index = 0;
            for (k = 0; k < oln; k++) {
                if (fabs(fit[k] - p_data->y[k]) <
                    (inst_config->wave_sol_FP_reject_alpha * p_data->err[k] +
                     inst_config->wave_sol_FP_reject_beta * fabs(std_mic))) {
                    p2_data->x[index] = p_data->x[k];
                    p2_data->y[index] = p_data->y[k];
                    p2_data->err[index] = p_data->err[k];
                    index++;
                }
            }
            
            fit2 = (double *) cpl_calloc (oln2, sizeof(double));
            
            my_error = espdr_fit_poly(p2_data, poly_deg,
                                      fit2, coeffs, coeffs_err, &chisq, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            if (order == -1) {
                espdr_msg("Order %d fitted, with CHI2 = %lf, COEFFS: %lf, %lf, %lf, %lf",
                          order, chisq,
                          coeffs[0], coeffs[1], coeffs[2], coeffs[3]);
                espdr_msg("SECOND FIT");
                for (k = 0; k < oln2; k++) {
                    espdr_msg("fit[%d] = %lf", k, fit2[k]);
                }
                
                double chi_self = 0.0;
                for (k = 0; k < oln2; k++) {
                    chi_self += (fit2[k] - p2_data->y[k]) *
                    (fit2[k] - p2_data->y[k]);
                }
                chi_self = chi_self / (oln2 - poly_deg);
                espdr_msg("CHI2 self: %lf", chi_self);
            }
            
            for (k = 0; k < poly_deg; k++) {
                coeffs_RE[order-1][k] = coeffs[k];
            }
            
            pxl_geom_order = cpl_image_extract(pixel_geom_fiber,
                                               1, order, size_x, order);
            pxl_geom_data = cpl_image_get_data_double(pxl_geom_order);
            pxl_size_order = cpl_image_extract(pixel_size_fiber,
                                               1, order, size_x, order);
            pxl_size_data = cpl_image_get_data_double(pxl_size_order);

            double wm_beg = 0.0, wm_mid = 0.0, wm_end = 0.0;
            for (pxl = 0; pxl < size_x; pxl++) {
                ll = 0.0;
                for (k = poly_deg-1; k > 0; k--) {
                    ll = (ll + coeffs[k]) * pxl_geom_data[pxl];
                }
                ll = ll + coeffs[0];
                wave_matrix_RE[wave_matrix_index] = ll;
                air_wave_matrix_RE[wave_matrix_index] = ll / espdr_compute_n_air(ll,
                                                                                 N_AIR_COEFF_T,
                                                                                 N_AIR_COEFF_P);
                
                if (isnan(wave_matrix_RE[wave_matrix_index])) {
                    espdr_msg("!!!!! WAVE_MATRIX for order %d, pxl %d is nan",
                              order, pxl);
                }
                if (isinf(wave_matrix_RE[wave_matrix_index])) {
                    espdr_msg("!!!!! WAVE_MATRIX for order %d, pxl %d is inf",
                              order, pxl);
                }
                
                if (pxl == 0) {
                    wm_beg = wave_matrix_RE[wave_matrix_index];
                }
                if (pxl == (int)size_x/2) {
                    wm_mid = wave_matrix_RE[wave_matrix_index];
                }
                if (pxl == size_x-1) {
                    wm_end = wave_matrix_RE[wave_matrix_index];
                }
                
                wave_matrix_index++;
            }
            
            //espdr_msg("WAVE_MATRIX: beginning: %f, middle: %f, end: %f",
            //          wm_beg, wm_mid, wm_end);
            
            for (pxl = 0; pxl < size_x; pxl++) {
                dlldx = 0.0;
                for (k = poly_deg-1; k > 1; k--) {
                    dlldx = (dlldx + coeffs[k]*k) * pxl_geom_data[pxl];
                }
                dlldx = dlldx + coeffs[1];
                dll_RE[dll_index] = dlldx * pxl_size_data[pxl];
                air_dll_RE[dll_index] = dlldx / espdr_compute_n_air(ll,
                                                                    N_AIR_COEFF_T,
                                                                    N_AIR_COEFF_P);
                dll_index++;
            }
            
            /* Compute the RMS per order */
            x_vector = (double *) cpl_calloc (oln2, sizeof(double));
            weights = (double *) cpl_calloc (oln2, sizeof(double));
            
            for (k = 0; k < oln2; k++) {
                x_vector[k] = (p2_data->y[k] - fit2[k])/p2_data->y[k] * LIGHT_SPEED;
                weights[k] = 1.0 /
                ((p2_data->err[k]/p2_data->y[k] * LIGHT_SPEED) *
                 (p2_data->err[k]/p2_data->y[k] * LIGHT_SPEED));
            }
            rms_per_order_RE[order-1] = espdr_wrms(x_vector, weights, oln2);
            chisq_per_order_RE[order-1] = chisq;
            
            if ((source == LFC) &&
                (rms_per_order_RE[order-1] <= inst_config->lfc_sol_max_rms) &&
                (chisq_per_order_RE[order-1] <= inst_config->lfc_sol_max_chi2)) {
                order_fit[order-1] = 1;
            }
            
            //espdr_msg("FIT_LL: COEFFS for order %d: %lf, %lf, %E, %E\tCHI2: %lf, RMS: %lf",
            //          order, coeffs[0], coeffs[1], coeffs[2], coeffs[3],
            //          chisq_per_order_RE[order-1], rms_per_order_RE[order-1]);
            
            if (source == THAR) {
                espdr_msg("FIT_LL: RMS: %lf & CHI2: %lf for order %d",
                          rms_per_order_RE[order-1], chisq, order);
            } else {
                espdr_msg("FIT_LL: RMS: %lf & CHI2: %lf LFC fit: %d for order %d",
                          rms_per_order_RE[order-1], chisq, order_fit[order-1], order);
            }
            
            cpl_free(x_vector);
            cpl_free(weights);
            
            cpl_free(fit);
            cpl_free(fit2);
            cpl_free(coeffs);
            cpl_free(coeffs_err);
            
            cpl_free(p_data->x);
            cpl_free(p_data->y);
            cpl_free(p_data->err);
            cpl_free(p2_data->x);
            cpl_free(p2_data->y);
            cpl_free(p2_data->err);
            
            cpl_image_delete(pxl_geom_order);
        }
        
        i = j;
    }
    
    cpl_free(p_data);
    cpl_free(p2_data);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Create the structure for the FITS table
 @param cpl_table   FITS table to be created
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_lines_table_create(cpl_table* lines_table_FITS) {
    
    cpl_table_new_column(lines_table_FITS, COL_NAME_ORDER, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ORDER, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_QC_FLAG, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_QC_FLAG, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, "[angstrom]");
    //my_error = cpl_table_new_column(lines_table_FITS, COL_NAME_N, CPL_TYPE_INT);
    cpl_table_new_column(lines_table_FITS, COL_NAME_DISPERSION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_DISPERSION, "[angstrom/pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_GROUPING, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_GROUPING, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_ELEMENT_NAME, CPL_TYPE_STRING);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ELEMENT_NAME, "");
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fill the FITS table stricture
 @param cpl_table   FITS table to be filled
 @param lines_table Data for the filling
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_lines_table_fill(cpl_table* lines_table_FITS,
                                           espdr_line_param *lines_table,
                                           const int lines_nb) {
    
    for (int i = 0; i < lines_nb; i++) {
        cpl_table_set_int(lines_table_FITS, COL_NAME_ORDER, i,
                          lines_table[i].order);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL, i,
                             lines_table[i].peak_pxl);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, i,
                             lines_table[i].peak_pxl_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM, i,
                             lines_table[i].fwhm);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM_ERR, i,
                             lines_table[i].fwhm_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX, i,
                             lines_table[i].peak_flux);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, i,
                             lines_table[i].peak_flux_err);
        cpl_table_set_int(lines_table_FITS, COL_NAME_QC_FLAG, i,
                          lines_table[i].qc_flag);
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH, i,
                             lines_table[i].wavelength);
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH_ERR, i,
                             lines_table[i].wavelength_err);
        //cpl_table_set_int(lines_table_FITS, COL_NAME_N, i,
        //                  lines_table[i].N);
        cpl_table_set_double(lines_table_FITS,COL_NAME_DISPERSION, i,
                             lines_table[i].dispersion);
        cpl_table_set_int(lines_table_FITS, COL_NAME_GROUPING, i,
                          lines_table[i].grouping);
        cpl_table_set_string(lines_table_FITS,COL_NAME_ELEMENT_NAME, i,
                             lines_table[i].element_name);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Create the structure for the FITS table
 @param cpl_table   FITS table to be created
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_lines_table_resol_create(cpl_table* lines_table_FITS) {
    
    cpl_table_new_column(lines_table_FITS, COL_NAME_ORDER, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ORDER, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_QC_FLAG, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_QC_FLAG, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_DISPERSION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_DISPERSION, "[angstrom/pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_GROUPING, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_GROUPING, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_ELEMENT_NAME, CPL_TYPE_STRING);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ELEMENT_NAME, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_RESOLUTION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_RESOLUTION, "");
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fill the FITS table stricture
 @param cpl_table   FITS table to be filled
 @param lines_table Data for the filling
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_lines_table_resol_fill(cpl_table* lines_table_FITS,
                                                 espdr_line_param *lines_table,
                                                 double *resolution,
                                                 const int lines_nb,
                                                 int save_table) {
    
    for (int i = 0; i < lines_nb; i++) {
        cpl_table_set_int(lines_table_FITS, COL_NAME_ORDER, i,
                          lines_table[i].order);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL, i,
                             lines_table[i].peak_pxl);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, i,
                             lines_table[i].peak_pxl_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM, i,
                             lines_table[i].fwhm);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM_ERR, i,
                             lines_table[i].fwhm_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX, i,
                             lines_table[i].peak_flux);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, i,
                             lines_table[i].peak_flux_err);
        if (save_table) {
            // Output to generate the input; all lines should be OK
            cpl_table_set_int(lines_table_FITS, COL_NAME_QC_FLAG, i, 1);
        } else {
            // Normal output
            cpl_table_set_int(lines_table_FITS, COL_NAME_QC_FLAG, i,
                              lines_table[i].qc_flag);
        }
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH, i,
                             lines_table[i].wavelength);
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH_ERR, i,
                             lines_table[i].wavelength_err);
        cpl_table_set_double(lines_table_FITS,COL_NAME_DISPERSION, i,
                             lines_table[i].dispersion);
        cpl_table_set_int(lines_table_FITS, COL_NAME_GROUPING, i,
                          lines_table[i].grouping);
        cpl_table_set_string(lines_table_FITS,COL_NAME_ELEMENT_NAME, i,
                             lines_table[i].element_name);
        cpl_table_set_double(lines_table_FITS,COL_NAME_RESOLUTION, i,
                             resolution[i]);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Create the structure for the FITS table
 @param cpl_table   FITS table to be created
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_ll_sol_table_create(cpl_table* lines_table_FITS) {
    
    cpl_table_new_column(lines_table_FITS, COL_NAME_ORDER, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ORDER, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_FWHM_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_FWHM_ERR, "[pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, "[e-]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_QC_FLAG, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_QC_FLAG, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_WAVELENGTH_ERR, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_DISPERSION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_DISPERSION, "[angstrom/pix]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_GROUPING, CPL_TYPE_INT);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_GROUPING, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_ELEMENT_NAME, CPL_TYPE_STRING);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_ELEMENT_NAME, "");
    cpl_table_new_column(lines_table_FITS, COL_NAME_D, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_D, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_D_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_D_ERR, "[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_D_ERR_NO_MIN, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_D_ERR_NO_MIN,"[angstrom]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_RESIDUALS, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_RESIDUALS, "[m/s]");
    cpl_table_new_column(lines_table_FITS, COL_NAME_RESOLUTION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(lines_table_FITS, COL_NAME_RESOLUTION,"");

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fill the FITS table stricture
 @param cpl_table   FITS table to be filled
 @param lines_table Data for the filling
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_ll_sol_table_fill(cpl_table* lines_table_FITS,
                                            espdr_line_param *lines_table,
                                            double *d_computed,
                                            double *d_computed_err,
                                            double *d_computed_err_no_min,
                                            double *residuals,
                                            double *resolution,
                                            const int lines_nb) {
    
    for (int i = 0; i < lines_nb; i++) {
        cpl_table_set_int(lines_table_FITS, COL_NAME_ORDER, i,
                          lines_table[i].order);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL, i,
                             lines_table[i].peak_pxl);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_PXL_ERR, i,
                             lines_table[i].peak_pxl_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM, i,
                             lines_table[i].fwhm);
        cpl_table_set_double(lines_table_FITS, COL_NAME_FWHM_ERR, i,
                             lines_table[i].fwhm_err);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX, i,
                             lines_table[i].peak_flux);
        cpl_table_set_double(lines_table_FITS, COL_NAME_PEAK_FLUX_ERR, i,
                             lines_table[i].peak_flux_err);
        // Normal output
        cpl_table_set_int(lines_table_FITS, COL_NAME_QC_FLAG, i,
                          lines_table[i].qc_flag);
#if SAVE_TH_REF_TABLES
        // Output to generate the input; all lines should be OK
        cpl_table_set_int(lines_table_FITS, COL_NAME_QC_FLAG, i, 1);
#endif
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH, i,
                             lines_table[i].wavelength);
        cpl_table_set_double(lines_table_FITS,COL_NAME_WAVELENGTH_ERR, i,
                             lines_table[i].wavelength_err);
        cpl_table_set_double(lines_table_FITS,COL_NAME_DISPERSION, i,
                             lines_table[i].dispersion);
        cpl_table_set_int(lines_table_FITS, COL_NAME_GROUPING, i,
                          lines_table[i].grouping);
        cpl_table_set_string(lines_table_FITS,COL_NAME_ELEMENT_NAME, i,
                             lines_table[i].element_name);
        cpl_table_set_double(lines_table_FITS, COL_NAME_D, i,
                             d_computed[i]);
        cpl_table_set_double(lines_table_FITS, COL_NAME_D_ERR, i,
                             d_computed_err[i]);
        cpl_table_set_double(lines_table_FITS, COL_NAME_D_ERR_NO_MIN, i,
                             d_computed_err_no_min[i]);
        cpl_table_set_double(lines_table_FITS, COL_NAME_RESIDUALS, i,
                             residuals[i]);
        cpl_table_set_double(lines_table_FITS, COL_NAME_RESOLUTION, i,
                             resolution[i]);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Create the structure for the FITS table
 @param cpl_table   FITS table to be created
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_raw_lines_table_create(cpl_table* raw_lines_table_FITS) {
    
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_ORDER, CPL_TYPE_INT);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_ORDER, "");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_PEAK_X, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_PEAK_X, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_PEAK_X_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_PEAK_X_ERR, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FWHM_X, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FWHM_X, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FWHM_X_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FWHM_X_ERR, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FLUX_X, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FLUX_X, "[e-]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FLUX_X_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FLUX_X_ERR, "[e-]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_PEAK_Y, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_PEAK_Y, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_PEAK_Y_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_PEAK_Y_ERR, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FWHM_Y, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FWHM_Y, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FWHM_Y_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FWHM_Y_ERR, "[pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FLUX_Y, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FLUX_Y, "[e-]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_FLUX_Y_ERR, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_FLUX_Y_ERR, "[e-]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_QC_FLAG, CPL_TYPE_INT);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_QC_FLAG, "");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_WAVELENGTH, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_WAVELENGTH, "[angstrom]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_DISPERSION, CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_DISPERSION, "[angstrom/pix]");
    cpl_table_new_column(raw_lines_table_FITS, COL_NAME_ELEMENT_NAME, CPL_TYPE_STRING);
    cpl_table_set_column_unit(raw_lines_table_FITS, COL_NAME_ELEMENT_NAME, "");
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fill the FITS table structure
 @param cpl_table   FITS table to be filled
 @param lines_table Data for the filling
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FITS_raw_lines_table_fill(cpl_table* raw_lines_table_FITS,
                                               espdr_raw_line_param *raw_lines_table,
                                               const int lines_nb) {
    
    for (int i = 0; i < lines_nb; i++) {
        cpl_table_set_int(raw_lines_table_FITS, COL_NAME_ORDER, i,
                          raw_lines_table[i].order);
        
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_PEAK_X, i,
                             raw_lines_table[i].peak_x);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_PEAK_X_ERR, i,
                             raw_lines_table[i].peak_x_err);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FWHM_X, i,
                             raw_lines_table[i].fwhm_x);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FWHM_X_ERR, i,
                             raw_lines_table[i].fwhm_x_err);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FLUX_X, i,
                             raw_lines_table[i].flux_x);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FLUX_X_ERR, i,
                             raw_lines_table[i].flux_x_err);
        
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_PEAK_Y, i,
                             raw_lines_table[i].peak_y);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_PEAK_Y_ERR, i,
                             raw_lines_table[i].peak_y_err);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FWHM_Y, i,
                             raw_lines_table[i].fwhm_y);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FWHM_Y_ERR, i,
                             raw_lines_table[i].fwhm_y_err);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FLUX_Y, i,
                             raw_lines_table[i].flux_y);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_FLUX_Y_ERR, i,
                             raw_lines_table[i].flux_y_err);
        
        cpl_table_set_int(raw_lines_table_FITS, COL_NAME_QC_FLAG, i,
                          raw_lines_table[i].qc_flag);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_WAVELENGTH, i,
                             raw_lines_table[i].wavelength);
        cpl_table_set_double(raw_lines_table_FITS, COL_NAME_DISPERSION, i,
                             raw_lines_table[i].dispersion);
        
        cpl_table_set_string(raw_lines_table_FITS, COL_NAME_ELEMENT_NAME, i,
                             raw_lines_table[i].element_name);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Init the lines_table structure
 @param lines_table table to be initialized
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_lines_table_init(espdr_line_param *lines_table,
                                      int lines_nb) {
    
    for (int i = 0; i < lines_nb; i++) {
        lines_table[i].order = 0;
        lines_table[i].peak_pxl = 0.0;
        lines_table[i].peak_pxl_err = 0.0;
        lines_table[i].fwhm = 0.0;
        lines_table[i].fwhm_err = 0.0;
        lines_table[i].peak_flux = 0.0;
        lines_table[i].peak_flux_err = 0.0;
        lines_table[i].qc_flag = 1;
        lines_table[i].wavelength = 0.0;
        lines_table[i].wavelength_err = 0.0;
        lines_table[i].lambda_peak = 0.0;
        lines_table[i].Nreal = 0.0;
        lines_table[i].N = 0;
        lines_table[i].dispersion = 0.0;
        lines_table[i].grouping = i+1;
        strcpy(lines_table[i].element_name, "NONE");
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Init the lines_table structure
 @param lines_table table to be initialized
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_raw_lines_table_init(espdr_raw_line_param *lines_table,
                                          int lines_nb) {
    
    for (int i = 0; i < lines_nb; i++) {
        lines_table[i].order = 0;
        lines_table[i].peak_x = 0.0;
        lines_table[i].peak_x_err = 0.0;
        lines_table[i].fwhm_x = 0.0;
        lines_table[i].fwhm_x_err = 0.0;
        lines_table[i].flux_x = 0.0;
        lines_table[i].flux_x_err = 0.0;
        lines_table[i].peak_y = 0.0;
        lines_table[i].peak_y_err = 0.0;
        lines_table[i].fwhm_y = 0.0;
        lines_table[i].fwhm_y_err = 0.0;
        lines_table[i].flux_y = 0.0;
        lines_table[i].flux_y_err = 0.0;
        lines_table[i].qc_flag = 1;
        lines_table[i].wavelength = 0.0;
        lines_table[i].wavelength_err = 0.0;
        lines_table[i].dispersion = 0.0;
        strcpy(lines_table[i].element_name, "NONE");
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_lines_table_RDB(espdr_line_param *lines_table,
                                          char *filename,
                                          int lines_nb,
                                          int withN,
                                          int withNreal) {

    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    if (withNreal) {
        fprintf(ltaptr, "order\tx0\tsig_x0\tfwhm\tsig_fwhm\tk\tsig_k\tqc\tll\tsig_ll\tdlldx\tgrouping\tN\tNreal\tN diff\telem\n");
        fprintf(ltaptr, "-----\t--\t------\t----\t--------\t-\t-----\t--\t--\t------\t-----\t--------\t-\t-----\t------\t----\n");
        for (int i = 0; i < lines_nb; i++) {
            fprintf(ltaptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%d\t%f\t%f\t%s\n",
                    lines_table[i].order,
                    lines_table[i].peak_pxl,
                    lines_table[i].peak_pxl_err,
                    lines_table[i].fwhm,
                    lines_table[i].fwhm_err,
                    lines_table[i].peak_flux,
                    lines_table[i].peak_flux_err,
                    lines_table[i].qc_flag,
                    lines_table[i].wavelength,
                    lines_table[i].wavelength_err,
                    lines_table[i].dispersion,
                    lines_table[i].grouping,
                    lines_table[i].N,
                    lines_table[i].Nreal,
                    fabs(lines_table[i].Nreal - lines_table[i].N),
                    lines_table[i].element_name);
        }
    } else {
        if (withN) {
            fprintf(ltaptr, "order\tx0\tsig_x0\tfwhm\tsig_fwhm\tk\tsig_k\tqc\tll\tsig_ll\tdlldx\tgrouping\twindow_fit/N\telem\n");
            fprintf(ltaptr, "-----\t--\t------\t----\t--------\t-\t-----\t--\t--\t------\t-----\t--------\t------------\t----\n");
            for (int i = 0; i < lines_nb; i++) {
                fprintf(ltaptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%d\t%s\n",
                        lines_table[i].order,
                        lines_table[i].peak_pxl,
                        lines_table[i].peak_pxl_err,
                        lines_table[i].fwhm,
                        lines_table[i].fwhm_err,
                        lines_table[i].peak_flux,
                        lines_table[i].peak_flux_err,
                        lines_table[i].qc_flag,
                        lines_table[i].wavelength,
                        lines_table[i].wavelength_err,
                        lines_table[i].dispersion,
                        lines_table[i].grouping,
                        lines_table[i].N,
                        lines_table[i].element_name);
            }
        } else {
            fprintf(ltaptr, "order\tx0\tsig_x0\tfwhm\tsig_fwhm\tk\tsig_k\tqc\tll\tsig_ll\tdlldx\tgrouping\telem\n");
            fprintf(ltaptr, "-----\t--\t------\t----\t--------\t-\t-----\t--\t--\t------\t-----\t--------\t----\n");
            for (int i = 0; i < lines_nb; i++) {
                fprintf(ltaptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%s\n",
                        lines_table[i].order,
                        lines_table[i].peak_pxl,
                        lines_table[i].peak_pxl_err,
                        lines_table[i].fwhm,
                        lines_table[i].fwhm_err,
                        lines_table[i].peak_flux,
                        lines_table[i].peak_flux_err,
                        lines_table[i].qc_flag,
                        lines_table[i].wavelength,
                        lines_table[i].wavelength_err,
                        lines_table[i].dispersion,
                        lines_table[i].grouping,
                        lines_table[i].element_name);
            }
        }
    }
    
    fclose(ltaptr);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_lines_table_RDB_short(espdr_line_param *lines_table,
                                                char *filename,
                                                int lines_nb) {
    
    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    fprintf(ltaptr, "order\tx0\tll\terr\tintensity\tgrouping\tqc\n");
    fprintf(ltaptr, "-----\t--\t--\t---\t---------\t--------\t--\n");
    for (int i = 0; i < lines_nb; i++) {
        fprintf(ltaptr, "%d\t%.4lf\t%lf\t%lf\t%.0lf\t%d\t%d\n",
                lines_table[i].order,
                lines_table[i].peak_pxl,
                lines_table[i].wavelength,
                lines_table[i].wavelength_err,
                lines_table[i].peak_flux,
                lines_table[i].grouping,
                lines_table[i].qc_flag);
    }
    
    fclose(ltaptr);
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_lines_table_RDB_long(espdr_line_param *lines_table,
                                               double *d_static,
                                               double *d_static_err,
                                               double *d_static_err_no_min,
                                               char *filename,
                                               int lines_nb) {
    
    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    fprintf(ltaptr, "order\tx0\tsig_x0\tfwhm\tsig_fwhm\tk\tsig_k\tqc\tll\tsig_ll\tdlldx\tgrouping\telem\tD\tsig_D\tsig_D_no_min\n");
    fprintf(ltaptr, "-----\t--\t------\t----\t--------\t-\t-----\t--\t--\t------\t-----\t--------\t----\t-\t-----\t------------\n");
    for (int i = 0; i < lines_nb; i++) {
        fprintf(ltaptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%s\t%.10f\t%.10f\t%.10f\n",
                lines_table[i].order,
                lines_table[i].peak_pxl,
                lines_table[i].peak_pxl_err,
                lines_table[i].fwhm,
                lines_table[i].fwhm_err,
                lines_table[i].peak_flux,
                lines_table[i].peak_flux_err,
                lines_table[i].qc_flag,
                lines_table[i].wavelength,
                lines_table[i].wavelength_err,
                lines_table[i].dispersion,
                lines_table[i].grouping,
                lines_table[i].element_name,
                d_static[i],
                d_static_err[i],
                d_static_err_no_min[i]);
    }
    fclose(ltaptr);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_lines_table_RDB_ll_sol(espdr_line_param *lines_table,
                                                 double *d_static,
                                                 double *d_static_err,
                                                 double *d_static_err_no_min,
                                                 double *residuals,
                                                 double *resolution,
                                                 char *filename,
                                                 int lines_nb) {
    
    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    fprintf(ltaptr, "order\tx0\tsig_x0\tfwhm\tsig_fwhm\tk\tsig_k\tqc\tll\tsig_ll\tdlldx\tgrouping\telem\tD\tsig_D\tsig_D_no_min\tresiduals\tresolution\n");
    fprintf(ltaptr, "-----\t--\t------\t----\t--------\t-\t-----\t--\t--\t------\t-----\t--------\t----\t-\t-----\t------------\t---------\t----------\n");
    for (int i = 0; i < lines_nb; i++) {
        fprintf(ltaptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%s\t%.10f\t%.10f\t%.10f\t%f\t%f\n",
                lines_table[i].order,
                lines_table[i].peak_pxl,
                lines_table[i].peak_pxl_err,
                lines_table[i].fwhm,
                lines_table[i].fwhm_err,
                lines_table[i].peak_flux,
                lines_table[i].peak_flux_err,
                lines_table[i].qc_flag,
                lines_table[i].wavelength,
                lines_table[i].wavelength_err,
                lines_table[i].dispersion,
                lines_table[i].grouping,
                lines_table[i].element_name,
                d_static[i],
                d_static_err[i],
                d_static_err_no_min[i],
                residuals[i],
                resolution[i]);
    }
    fclose(ltaptr);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_raw_lines_table_RDB(espdr_raw_line_param *raw_lines_table,
                                              char *filename,
                                              int raw_lines_nb) {
    
    FILE *ltarawptr = fopen(filename, "w");
    espdr_ensure(ltarawptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    fprintf(ltarawptr, "order\tpeak_x\tpeak_x_err\tfwhm_x\tfwhm_x_err\tflux_x\tflux_x_err\tpeak_y\tpeak_y_err\tfwhm_y\tfwhm_y_err\tflux_y\tflux_y_err\tqc\tll\tsig_ll\tdlldx\telem\n");
    fprintf(ltarawptr, "-----\t------\t----------\t------\t----------\t------\t----------\t------\t----------\t------\t----------\t------\t----------\t--\t--\t------\t-----\t----\n");
    for (int j = 0; j < raw_lines_nb; j++) {
        fprintf(ltarawptr, "%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%s\n",
                raw_lines_table[j].order,
                raw_lines_table[j].peak_x,
                raw_lines_table[j].peak_x_err,
                raw_lines_table[j].fwhm_x,
                raw_lines_table[j].fwhm_x_err,
                raw_lines_table[j].flux_x,
                raw_lines_table[j].flux_x_err,
                raw_lines_table[j].peak_y,
                raw_lines_table[j].peak_y_err,
                raw_lines_table[j].fwhm_y,
                raw_lines_table[j].fwhm_y_err,
                raw_lines_table[j].flux_y,
                raw_lines_table[j].flux_y_err,
                raw_lines_table[j].qc_flag,
                raw_lines_table[j].wavelength,
                raw_lines_table[j].wavelength_err,
                raw_lines_table[j].dispersion,
                raw_lines_table[j].element_name);
    }
    
    fclose(ltarawptr);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the lines table in RDB format
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_wave_table_RDB(double **coeffs_per_order,
                                         char *filename,
                                         int orders_nb,
                                         int poly_deg) {
    
    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    
    fprintf(ltaptr, "order");
    for (int i = 0; i < poly_deg; i++) {
        fprintf(ltaptr, "\tcoeff%d", i);
    }
    fprintf(ltaptr, "\n");
    fprintf(ltaptr, "-----");
    for (int i = 0; i < poly_deg; i++) {
        fprintf(ltaptr, "\t-------");
    }
    fprintf(ltaptr, "\n");
    
    for (int i = 0; i < orders_nb; i++) {
        fprintf(ltaptr, "%d", i+1);
        for (int j = 0; j < poly_deg; j++) {
            fprintf(ltaptr, "\t%.10E", coeffs_per_order[i][j]);
        }
        fprintf(ltaptr, "\n");
    }
    
    fclose(ltaptr);
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     save the input wave frame cleaned fm rthe CCD eefects
 @param     frameset    input frameset
 @param     used_frames used_frames saved
 @param     parameters  recipe parameters
 @param     recipe      recipe ID
 @param     filename    filename to save the images into
 @param     CCD_geom    CCD geometry config
 @param     keywords    header keywords of the input frame
 @param     keywords_ext extensions keywords
 @param     CCD_corrected_wave  images to be saved
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_ccd_cleaned_wave_frame(cpl_frameset* frameset,
                                                 cpl_frameset* used_frames,
                                                 cpl_parameterlist* parameters,
                                                 const char *recipe,
                                                 char *filename,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 cpl_propertylist* keywords,
                                                 cpl_propertylist** keywords_ext,
                                                 cpl_imagelist *CCD_corrected_wave) {

    espdr_msg("Saving CCD cleaned image in %s", filename);
    
    /* Save the PRO.CATG */
    cpl_error_code my_error;
    
    my_error = cpl_propertylist_update_string(keywords,PRO_CATG_KW,
                                              ESPDR_PRO_CATG_TEST_PRODUCT);
    
    if (my_error != CPL_ERROR_NONE) {
        espdr_msg_error("Error adding product category is %s",
                        cpl_error_get_message());
    }
    
    /* Save the cleaned WAVE fits frame */
    my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
                                    recipe, keywords, keywords_ext,
                                    filename,
                                    CCD_corrected_wave,
                                    CPL_TYPE_FLOAT, CCD_geom);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 filename);
    
    return cpl_error_get_code();
}




/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_table_RDB(espdr_line_param *lines_table,
                                          int lines_nb,
                                          const char *name,
                                          int fibre_nr,
                                          espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    FILE *ltaPtr = NULL;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving line table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_lines_table_RDB(lines_table,
                                          line_table_filename,
                                          lines_nb, 1, 0);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_lines_table_RDB failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_table_RDB_short(espdr_line_param *lines_table,
                                                int lines_nb,
                                                const char *name,
                                                int fibre_nr,
                                                espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    FILE *ltaPtr = NULL;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving line table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_lines_table_RDB_short(lines_table,
                                                line_table_filename,
                                                lines_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_lines_table_RDB_short failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_table_RDB_long(espdr_line_param *lines_table,
                                               double *d_static,
                                               double *d_static_err,
                                               double *d_static_err_no_min,
                                               int lines_nb,
                                               const char *name,
                                               int fibre_nr,
                                               espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving line table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_lines_table_RDB_long(lines_table,
                                               d_static,
                                               d_static_err,
                                               d_static_err_no_min,
                                               line_table_filename,
                                               lines_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_lines_table_RDB_long failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_table_RDB_ll_sol(espdr_line_param *lines_table,
                                                 double *d_static,
                                                 double *d_static_err,
                                                 double *d_static_err_no_min,
                                                 double *residuals,
                                                 double *resolution,
                                                 int lines_nb,
                                                 const char *name,
                                                 int fibre_nr,
                                                 espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving line table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_lines_table_RDB_ll_sol(lines_table,
                                                 d_static,
                                                 d_static_err,
                                                 d_static_err_no_min,
                                                 residuals,
                                                 resolution,
                                                 line_table_filename,
                                                 lines_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_lines_table_RDB_ll_sol failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_wave_table_RDB(double **wave_table,
                                               int orders_nb,
                                               const char *name,
                                               int fibre_nr,
                                               espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving wave table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_wave_table_RDB(wave_table,
                                         line_table_filename,
                                         orders_nb,
                                         inst_config->wave_sol_poly_deg);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_wave_table_RDB failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save lines table in RDB format
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_debug_raw_table_RDB(espdr_raw_line_param *lines_table,
                                              int lines_nb,
                                              const char *name,
                                              int fibre_nr,
                                              espdr_inst_config *inst_config) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char line_table_filename[FILENAME_LENGTH];
    sprintf(line_table_filename, "%s_%s_%c.rdb",
            inst_config->instrument, name, fibre_name[fibre_nr]);
    espdr_msg("Saving line table in %s, no PRO.CATG", line_table_filename);
    my_error = espdr_save_raw_lines_table_RDB(lines_table,
                                              line_table_filename,
                                              lines_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_raw_lines_table_RDB failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save static wave and dll matrixes
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_static_matrixes(cpl_image *wave_matrix_sol,
                                          cpl_image *dll_matrix_sol,
                                          cpl_frameset *frameset,
                                          cpl_frameset *used_frames,
                                          cpl_parameterlist *parameters,
                                          espdr_inst_config *inst_config,
                                          cpl_propertylist *keywords,
                                          const int fibre,
                                          double MJD,
                                          const char *date) {
    
    /* Save the STATIC WAVE & DLL MATRICES FITS */
    cpl_propertylist *static_matrix_KWs = cpl_propertylist_new();
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    my_error = cpl_propertylist_append_double(static_matrix_KWs,
                                              "MJD-OBS", MJD);
    
    my_error = cpl_propertylist_append_string(static_matrix_KWs,
                                              "INSTRUME",
                                              inst_config->instrument);
    
    const char *ins_mode = cpl_propertylist_get_string(keywords,
                                                       "ESO INS MODE");
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        ins_mode = cpl_propertylist_get_string(keywords,
                                               "INSTRUME");
        cpl_error_reset();
    }
    my_error = cpl_propertylist_append_string(static_matrix_KWs,
                                              "ESO INS MODE",
                                              ins_mode);
    
    int det_binx = cpl_propertylist_get_int(keywords,
                                            inst_config->det_binx_kw);
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        det_binx = 1;
        cpl_error_reset();
    }
    my_error = cpl_propertylist_append_int(static_matrix_KWs,
                                           inst_config->det_binx_kw,
                                           det_binx);
    
    int det_biny = cpl_propertylist_get_int(keywords,
                                            inst_config->det_biny_kw);
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        det_biny = 1;
        cpl_error_reset();
    }
    my_error = cpl_propertylist_append_int(static_matrix_KWs,
                                           inst_config->det_biny_kw,
                                           det_biny);
    
    my_error = cpl_propertylist_append_string(static_matrix_KWs,
                                              "ESO DET READ SPEED",
                                              "FAST");
    
    /* Save the PRO.CATG */
    char new_keyword[80];
    sprintf(new_keyword, "STATIC_WAVE_MATRIX_%c", fibre_name[fibre]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_append_string(static_matrix_KWs,
                                              PRO_CATG_KW,
                                              new_keyword);
    
    char static_matrix_filename[64];
    char static_matrix_valid_date[32];
    char static_matrix_ins_mode_bin[32];
    sprintf(static_matrix_valid_date, "%s", date);
    if (strcmp(inst_config->instrument, "ESPRESSO") == 0) {
        if (strcmp(ins_mode, "SINGLEUHR") == 0) {
            sprintf(static_matrix_ins_mode_bin, "%s_%s",
                    inst_config->instrument, ins_mode);
        } else {
            sprintf(static_matrix_ins_mode_bin, "%s_%s_%dx%d",
                    inst_config->instrument, ins_mode, det_binx, det_biny);
        }
    } else {
        if (strcmp(inst_config->instrument, "NIRPS") == 0) {
            sprintf(static_matrix_ins_mode_bin, "%s_%s",
                    inst_config->instrument, ins_mode);
        } else {
            sprintf(static_matrix_ins_mode_bin, "%s", ins_mode);
        }
    }
    
    //espdr_msg("INS.MODE = %s, INS.MODE with binning = %s",
    //          ins_mode, ins_mode_bin);
    
    sprintf(static_matrix_filename,
            "%s_STATIC_WAVE_MATRIX_%c_%s.fits",
            static_matrix_ins_mode_bin, fibre_name[fibre],
            static_matrix_valid_date);
    espdr_msg("Saving STATIC_WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre], static_matrix_filename, new_keyword);
    
    //my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
    //                                        used_frames, recipe,
    //                                        static_matrix_KWs,
    //                                        static_matrix_filename,
    //                                        wave_matrix_sol,
    //                                        CPL_TYPE_DOUBLE);
    
    my_error = cpl_image_save(NULL,
                              static_matrix_filename,
                              CPL_TYPE_DOUBLE,
                              static_matrix_KWs,
                              CPL_IO_CREATE);
    
    my_error = cpl_image_save(wave_matrix_sol,
                              static_matrix_filename,
                              CPL_TYPE_DOUBLE,
                              NULL,
                              CPL_IO_EXTEND);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 static_matrix_filename);
    
    espdr_msg("STATIC_WAVE_MATRIX saved under: %s", static_matrix_filename);
    
    sprintf(static_matrix_filename,
            "%s_STATIC_DLL_MATRIX_%c_%s.fits",
            static_matrix_ins_mode_bin, fibre_name[fibre],
            static_matrix_valid_date);
    espdr_msg("Saving STATIC_DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre], static_matrix_filename, new_keyword);
    
    /* Save the PRO.CATG */
    sprintf(new_keyword, "STATIC_DLL_MATRIX_%c", fibre_name[fibre]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(static_matrix_KWs,
                                              PRO_CATG_KW,
                                              new_keyword);
    
    //my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
    //                                        used_frames, "wave",
    //                                        static_matrix_KWs,
    //                                        static_matrix_filename,
    //                                        dll_matrix_sol,
    //                                        CPL_TYPE_DOUBLE);
    
    my_error = cpl_image_save(NULL,
                              static_matrix_filename,
                              CPL_TYPE_DOUBLE,
                              static_matrix_KWs,
                              CPL_IO_CREATE);
    
    my_error = cpl_image_save(dll_matrix_sol,
                              static_matrix_filename,
                              CPL_TYPE_DOUBLE,
                              NULL,
                              CPL_IO_EXTEND);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 static_matrix_filename);
    
    espdr_msg("STATIC_DLL_MATRIX saved under: %s", static_matrix_filename);
    
    cpl_propertylist_delete(static_matrix_KWs);
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief Save TH REF table
 @param inst_config instrument configuration
 @param lines_table table to be saved
 @param filename    filename to save dthe table into
 @param lines_nb    number of lines in the lines_table structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_TH_REF_TABLE(cpl_table *THAR_lines_table_FITS,
                                       cpl_frameset *frameset,
                                       cpl_frameset *used_frames,
                                       cpl_parameterlist *parameters,
                                       espdr_inst_config *inst_config,
                                       cpl_propertylist *keywords,
                                       const int fibre,
                                       const char *lines_type,
                                       double MJD,
                                       const char *date,
                                       char *ins_mode_bin_RE) {
    
    /* Save the TH_REF_TABLE FITS */
    cpl_propertylist *th_ref_KWs = cpl_propertylist_new();
    
    /* Save the PRO.CATG */
    char new_keyword[80];
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Saving new AR table for date: %s, MJD: %f", date, MJD);
    
    sprintf(new_keyword, "%s_LINE_TABLE_%c", lines_type, fibre_name[fibre]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_append_string(th_ref_KWs,
                                              PRO_CATG_KW,
                                              new_keyword);
    
    my_error = cpl_propertylist_append_double(th_ref_KWs,
                                              "MJD-OBS",
                                              MJD);
    
    my_error = cpl_propertylist_append_string(th_ref_KWs,
                                              "INSTRUME",
                                              inst_config->instrument);
    
    const char *ins_mode = cpl_propertylist_get_string(keywords,
                                                       "ESO INS MODE");
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        ins_mode = cpl_propertylist_get_string(keywords,
                                               "INSTRUME");
        cpl_error_reset();
    }
    
    my_error = cpl_propertylist_append_string(th_ref_KWs,
                                              "ESO INS MODE",
                                              ins_mode);
    
    int det_binx = cpl_propertylist_get_int(keywords,
                                            inst_config->det_binx_kw);
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        det_binx = 1;
        cpl_error_reset();
    }
    my_error = cpl_propertylist_append_int(th_ref_KWs,
                                           inst_config->det_binx_kw,
                                           det_binx);
    
    int det_biny = cpl_propertylist_get_int(keywords,
                                            inst_config->det_biny_kw);
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        det_biny = 1;
        cpl_error_reset();
    }
    my_error = cpl_propertylist_append_int(th_ref_KWs,
                                           inst_config->det_biny_kw,
                                           det_biny);
    
    my_error = cpl_propertylist_append_string(th_ref_KWs,
                                              "ESO DET READ SPEED",
                                              "SLOW");
    
    char th_ref_line_table_filename[64];
    char th_ref_table_valid_date[32];
    char ins_mode_bin[32];
    sprintf(th_ref_table_valid_date, "%s", date);
    if (strcmp(inst_config->instrument, "ESPRESSO") == 0) {
        if (strcmp(ins_mode, "SINGLEUHR") == 0) {
            sprintf(ins_mode_bin, "%s_%s",
                    inst_config->instrument, ins_mode);
        } else {
            sprintf(ins_mode_bin, "%s_%s_%dx%d",
                    inst_config->instrument, ins_mode, det_binx, det_biny);
        }
    } else {
        if (strcmp(inst_config->instrument, "NIRPS") == 0) {
            sprintf(ins_mode_bin, "%s_%s",
                    inst_config->instrument, ins_mode);
        } else {
            sprintf(ins_mode_bin, "%s", ins_mode);
        }
    }
    
    strcpy(ins_mode_bin_RE, ins_mode_bin);
    
    //espdr_msg("INS.MODE = %s, INS.MODE with binning = %s",
    //          ins_mode, ins_mode_bin);
    
    if (strcmp(lines_type, "REF") == 0) {
        sprintf(th_ref_line_table_filename,
                "%s_TH_REF_LINE_TABLE_%c_%s.fits",
                ins_mode_bin, fibre_name[fibre], th_ref_table_valid_date);
    } else {
        sprintf(th_ref_line_table_filename,
                "%s_%s_LINE_TABLE_%c_%s.fits",
                ins_mode_bin, lines_type, fibre_name[fibre], th_ref_table_valid_date);
    }
    espdr_msg("Saving TH_REF_LINE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre], th_ref_line_table_filename, new_keyword);
    
    my_error = cpl_table_save(THAR_lines_table_FITS,
                              th_ref_KWs, NULL,
                              th_ref_line_table_filename,
                              CPL_IO_CREATE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 th_ref_line_table_filename);
    
    cpl_propertylist_delete(th_ref_KWs);
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save wave_THAR products
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_wave_S2D_products(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            const char *recipe_id,
                                            cpl_frameset *used_frames,
                                            const char *WAVE_tag,
                                            int fibre_nr,
                                            cpl_propertylist *keywords,
                                            espdr_inst_config *inst_config,
                                            cpl_image *flat_corr_spectrum,
                                            cpl_image *flat_corr_error,
                                            cpl_image *flat_corr_qual,
                                            cpl_image *flat_blaze_corr_spectrum,
                                            cpl_image *flat_blaze_corr_error,
                                            cpl_image *flat_blaze_corr_qual) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    char filename[FILENAME_LENGTH];
    char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));
    
    /* Save the ed2s extracted spectra immediately after creation */
    espdr_msg("Saving the S2D extracted spectra");
    cpl_image **images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
    
    espdr_msg("Saving flat corrected spectra");
    sprintf(new_keyword, "%s_BLAZE_%s_%c",
            ESPDR_PRO_CATG_S2D, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    sprintf(filename, "%s_BLAZE_%s_%c.fits",
            inst_config->extr_spectr_filename_no_fits, WAVE_tag, fibre_name[fibre_nr]);
    
    images_to_save[0] = cpl_image_duplicate(flat_corr_spectrum);
    images_to_save[1] = cpl_image_duplicate(flat_corr_error);
    images_to_save[2] = cpl_image_duplicate(flat_corr_qual);
    
    my_error = espdr_dfs_save_data_err_qual(frameset, parameters, used_frames, recipe_id,
                                            keywords, filename, images_to_save);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    
    for (int i = 0; i < 3; i++) {
        cpl_image_delete(images_to_save[i]);
    }
    
    espdr_msg("Saving flat & blaze corrected spectra");
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_S2D, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    sprintf(filename, "%s_%s_%c.fits",
            inst_config->extr_spectr_filename_no_fits, WAVE_tag, fibre_name[fibre_nr]);
    
    images_to_save[0] = cpl_image_duplicate(flat_blaze_corr_spectrum);
    images_to_save[1] = cpl_image_duplicate(flat_blaze_corr_error);
    images_to_save[2] = cpl_image_duplicate(flat_blaze_corr_qual);
    
    my_error = espdr_dfs_save_data_err_qual(frameset, parameters, used_frames, recipe_id,
                                            keywords, filename, images_to_save);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    
    for (int i = 0; i < 3; i++) {
        cpl_image_delete(images_to_save[i]);
    }
    cpl_free(images_to_save);
    
    return (cpl_error_get_code());
}


