/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ESPDR_WAVE_CAL_H
#define ESPDR_WAVE_CAL_H


/* DRL functions*/
#include <espdr_instrument.h>
#include <espdr_msg.h>
#include <espdr_CCD.h>
#include <espdr_dfs.h>
#include <espdr_pixels.h>
#include <espdr_fit.h>
#include <espdr_detector_signature.h>
#include <espdr_blaze.h>
#include <espdr_led_flat.h>
#include <espdr_extraction.h>
#include <espdr_utils.h>

/* Library */
#include <cpl.h>
#include <assert.h>


/*----------------------------------------------------------------------------
 Constants
 ----------------------------------------------------------------------------*/
#define MAX_NB_SPECTRAL_LINES 100000.0

// order_start pixel with respect to the old start at 1210
// For day 192 (11th July 2017)
//#define ORDER_SHIFT 0
// For day 221 (9th August 2017) for Christophe's rdb input
//#define ORDER_SHIFT 25
// For the FITS input
#define ORDER_SHIFT 0

#define SAVE_TH_REF_TABLES 0
#define SAVE_DEBUG_PRODUCT_WAVE 0
#define SAVE_DEBUG_CCD_CLEANED_WAVE 0

/* Columns of the LINES_TABLE */
static const char COL_NAME_D[] = "D";
static const char COL_NAME_D_ERR[] = "SIG_D";
static const char COL_NAME_D_ERR_NO_MIN[] = "SIG_D_NO_MIN_ERR";
static const char COL_NAME_RESIDUALS[] = "RESIDUALS";
static const char COL_NAME_RESOLUTION[] = "RESOLUTION";

/* Columns of the LINES_TABLE */
static const char COL_NAME_ORDER[] = "ORDER";
static const char COL_NAME_PEAK_PXL[] = "X0";
static const char COL_NAME_PEAK_PXL_ERR[] = "SIG_X0";
static const char COL_NAME_FWHM[] = "FWHM";
static const char COL_NAME_FWHM_ERR[] = "SIG_FWHM";
static const char COL_NAME_PEAK_FLUX[] = "K";
static const char COL_NAME_PEAK_FLUX_ERR[] = "SIG_K";
static const char COL_NAME_QC_FLAG[] = "QC";
static const char COL_NAME_WAVELENGTH[] = "LL";
static const char COL_NAME_WAVELENGTH_ERR[] = "SIG_LL";
static const char COL_NAME_NREAL[] = "NREAL";
static const char COL_NAME_N[] = "N";
static const char COL_NAME_DISPERSION[] = "DLLDX";
static const char COL_NAME_GROUPING[] = "GROUPING";
static const char COL_NAME_ELEMENT_NAME[] = "ELEM";

static const char COL_NAME_PEAK_X[] = "X0";
static const char COL_NAME_PEAK_X_ERR[] = "SIG_X0";
static const char COL_NAME_PEAK_Y[] = "Y0";
static const char COL_NAME_PEAK_Y_ERR[] = "SIG_Y0";
static const char COL_NAME_FLUX_X[] = "FLUX_X";
static const char COL_NAME_FLUX_X_ERR[] = "SIG_FLUX_X";
static const char COL_NAME_FLUX_Y[] = "FLUX_Y";
static const char COL_NAME_FLUX_Y_ERR[] = "SIG_FLUX_Y";
static const char COL_NAME_FWHM_X[] = "FWHM_X";
static const char COL_NAME_FWHM_X_ERR[] = "SIG_FWHM_X";
static const char COL_NAME_FWHM_Y[] = "FWHM_Y";
static const char COL_NAME_FWHM_Y_ERR[] = "SIG_FWHM_Y";

/* Columns of the D_FIT_TABLE */
static const char COL_NAME_D_FIT_COEFF[] = "D_FIT_COEFF";
static const char COL_NAME_D_FIT_COEFF_ERR[] = "D_FIT_COEFF_ERR";


/*----------------------------------------------------------------------------
 Data structures
 ----------------------------------------------------------------------------*/

/* Data structure for LINES TABLE */
typedef struct {
    int order;
    double peak_pxl;
    double peak_pxl_err;
    double fwhm;
    double fwhm_err;
    double peak_flux;
    double peak_flux_err;
    int qc_flag;
    double wavelength;
    double wavelength_err;
    double lambda_peak;
    double Nreal;
    int N;
    double dispersion;
    int grouping;
    char element_name[10];
} espdr_line_param;

/* Data structure for RAW LINES TABLE */
typedef struct {
    int order;
    double peak_x;
    double peak_x_err;
    double peak_y;
    double peak_y_err;
    double flux_x;
    double flux_x_err;
    double flux_y;
    double flux_y_err;
    double fwhm_x;
    double fwhm_x_err;
    double fwhm_y;
    double fwhm_y_err;
    int qc_flag;
    double wavelength;
    double wavelength_err;
    double dispersion;
    char element_name[10];
} espdr_raw_line_param;

typedef enum { THAR, FP, LFC, DARK } espdr_src_type;

/*----------------------------------------------------------------------------
 Functions prototypes
 ----------------------------------------------------------------------------*/

cpl_error_code espdr_get_fibre_source(const char *WAVE_tag,
                                      espdr_src_type *fibre_source_RE,
                                      char **fibre_source_str_RE);

cpl_error_code espdr_process_wave_till_extraction(cpl_frameset *frameset,
                                                  cpl_parameterlist *parameters,
                                                  const char *RECIPE_ID,
                                                  cpl_frameset *wave_frames_to_reduce,
                                                  cpl_frame *wave_frame,
                                                  cpl_frameset *used_frames,
                                                  espdr_inst_config *inst_config,
                                                  espdr_CCD_geometry *CCD_geom,
                                                  espdr_qc_keywords *qc_kws,
                                                  int *orders_nb_per_fibre,
                                                  espdr_src_type *fibre_source,
                                                  char **fibre_source_str,
                                                  cpl_propertylist **keywords_fibre,
                                                  cpl_imagelist *CCD_corrected_wave,
                                                  cpl_table ****orders_coeffs,
                                                  double *RON,
                                                  cpl_image **flat_corr_spectrum,
                                                  cpl_image **flat_corr_error,
                                                  cpl_image **flat_corr_qual,
                                                  cpl_image **flat_blaze_corr_spectrum,
                                                  cpl_image **flat_blaze_corr_error,
                                                  cpl_image **flat_blaze_corr_qual,
                                                  cpl_image **pixel_geom_image,
                                                  cpl_image **pixel_size_image);

cpl_error_code espdr_extract_lines_tables(cpl_frameset *frameset,
                                          const char *RECIPE_ID,
                                          cpl_frameset *used_frames,
                                          espdr_inst_config *inst_config,
                                          espdr_src_type *fibre_source,
                                          char **fibre_source_str,
                                          int compute_drift,
                                          char *first_TAG,
                                          char *second_TAG,
                                          cpl_frame **REF_line_table_frame,
                                          cpl_frame **FP_line_table_frame,
                                          cpl_image **static_wave_matrix_img,
                                          cpl_image **static_dll_matrix_img,
                                          cpl_frame **s2d_blaze_FP_frame);

cpl_error_code espdr_wave_cosmics_QC(int *cosmics,
                                     int orders_nb,
                                     espdr_qc_keywords *qc_kws,
                                     cpl_propertylist *keywords_RE);

cpl_error_code espdr_check_delta_FP_peaks(espdr_line_param *lines_table,
                                          int lines_nb,
                                          double delta_fp_tolerance);

cpl_error_code espdr_compute_dlldx(espdr_line_param *lines_table,
                                   int lines_nb,
                                   int orders_nb,
                                   espdr_inst_config *inst_config,
                                   double **coeffs);

cpl_error_code espdr_search_lines(cpl_image *s2d_data,
                                  cpl_image *s2d_error,
                                  cpl_image *s2d_quality,
                                  espdr_inst_config *inst_config,
                                  espdr_src_type fibre_source,
                                  espdr_line_param *lines_table,
                                  int *lines_nb_RE);

cpl_error_code espdr_fit_lines(espdr_line_param *lines_table,
                               int lines_nb,
                               cpl_image *s2d,
                               cpl_image *s2d_err,
                               cpl_image *s2d_qual,
                               cpl_image *pixel_geom,
                               double flux_threshold,
                               //double min_x0_err,
                               int window_size,
                               int tolerance_window,
                               int NGauss_flag,
                               int *lines_nb_per_order_RE);

double espdr_wrms(double *x, double *w, int len);

cpl_error_code espdr_fit_ll_sol(espdr_line_param *lines_table,
                                int lines_nb,
                                espdr_inst_config *inst_config,
                                cpl_image *pixel_geom_fiber,
                                cpl_image *pixel_size_fiber,
                                espdr_src_type source,
                                int fibre_nr,
                                int *lines_nb_per_order_RE,
                                double *rms_per_order_RE,
                                double *chisq_per_order_RE,
                                int *order_fit,
                                double **coeffs_RE,
                                double *wave_matrix_RE,
                                double *dll_RE,
                                double *air_wave_matrix_RE,
                                double *air_dll_RE);

cpl_error_code espdr_FITS_lines_table_create(cpl_table* lines_table_FITS);

cpl_error_code espdr_FITS_lines_table_fill(cpl_table* lines_table_FITS,
                                           espdr_line_param *lines_table,
                                          const int lines_nb);

cpl_error_code espdr_FITS_lines_table_resol_create(cpl_table* lines_table_FITS);

cpl_error_code espdr_FITS_lines_table_resol_fill(cpl_table* lines_table_FITS,
                                                 espdr_line_param *lines_table,
                                                 double *resolution,
                                                 const int lines_nb,
                                                 int save_table);

cpl_error_code espdr_FITS_ll_sol_table_create(cpl_table* lines_table_FITS);

cpl_error_code espdr_FITS_ll_sol_table_fill(cpl_table* lines_table_FITS,
                                            espdr_line_param *lines_table,
                                            double *d_computed,
                                            double *d_computed_err,
                                            double *d_static_err_no_min,
                                            double *residuals,
                                            double *resolution,
                                            const int lines_nb);

cpl_error_code espdr_FITS_raw_lines_table_create(cpl_table* raw_lines_table_FITS);

cpl_error_code espdr_FITS_raw_lines_table_fill(cpl_table* raw_lines_table_FITS,
                                               espdr_raw_line_param *raw_lines_table,
                                               const int lines_nb);

cpl_error_code espdr_lines_table_init(espdr_line_param *lines_table,
                                      int lines_nb);

cpl_error_code espdr_raw_lines_table_init(espdr_raw_line_param *lines_table,
                                          int lines_nb);

cpl_error_code espdr_save_lines_table_RDB(espdr_line_param *lines_table,
                                          char *filename,
                                          int lines_nb,
                                          int withN,
                                          int withNreal);

cpl_error_code espdr_save_lines_table_RDB_short(espdr_line_param *lines_table,
                                                char *filename,
                                                int lines_nb);

cpl_error_code espdr_save_lines_table_RDB_long(espdr_line_param *lines_table,
                                               double *d_static,
                                               double *d_static_err,
                                               double *d_static_err_no_min,
                                               char *filename,
                                               int lines_nb);

cpl_error_code espdr_save_lines_table_RDB_ll_sol(espdr_line_param *lines_table,
                                                 double *d_static,
                                                 double *d_static_err,
                                                 double *d_static_err_no_min,
                                                 double *residuals,
                                                 double *resolution,
                                                 char *filename,
                                                 int lines_nb);

cpl_error_code espdr_save_raw_lines_table_RDB(espdr_raw_line_param *raw_lines_table,
                                              char *filename,
                                              int raw_lines_nb);

cpl_error_code espdr_save_wave_table_RDB(double **coeffs_per_order,
                                         char *filename,
                                         int orders_nb,
                                         int poly_deg);

cpl_error_code espdr_save_ccd_cleaned_wave_frame(cpl_frameset* frameset,
                                                 cpl_frameset* used_frames,
                                                 cpl_parameterlist* parameters,
                                                 const char *recipe,
                                                 char *filename,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 cpl_propertylist* keywords,
                                                 cpl_propertylist** keywords_ext,
                                                 cpl_imagelist *CCD_corrected_wave);

cpl_error_code espdr_save_debug_table_RDB(espdr_line_param *lines_table,
                                          int lines_nb,
                                          const char *name,
                                          int fibre_nr,
                                          espdr_inst_config *inst_config);

cpl_error_code espdr_save_debug_table_RDB_short(espdr_line_param *lines_table,
                                                int lines_nb,
                                                const char *name,
                                                int fibre_nr,
                                                espdr_inst_config *inst_config);

cpl_error_code espdr_save_debug_table_RDB_long(espdr_line_param *lines_table,
                                               double *d_static,
                                               double *d_static_err,
                                               double *d_static_err_no_min,
                                               int lines_nb,
                                               const char *name,
                                               int fibre_nr,
                                               espdr_inst_config *inst_config);

cpl_error_code espdr_save_debug_table_RDB_ll_sol(espdr_line_param *lines_table,
                                                 double *d_static,
                                                 double *d_static_err,
                                                 double *d_static_err_no_min,
                                                 double *residuals,
                                                 double *resolution,
                                                 int lines_nb,
                                                 const char *name,
                                                 int fibre_nr,
                                                 espdr_inst_config *inst_config);

cpl_error_code espdr_save_debug_wave_table_RDB(double **wave_table,
                                               int orders_nb,
                                               const char *name,
                                               int fibre_nr,
                                               espdr_inst_config *inst_config);

cpl_error_code espdr_save_debug_raw_table_RDB(espdr_raw_line_param *lines_table,
                                              int lines_nb,
                                              const char *name,
                                              int fibre_nr,
                                              espdr_inst_config *inst_config);

cpl_error_code espdr_save_static_matrixes(cpl_image *wave_matrix_sol,
                                          cpl_image *dll_matrix_sol,
                                          cpl_frameset *frameset,
                                          cpl_frameset *used_frames,
                                          cpl_parameterlist *parameters,
                                          espdr_inst_config *inst_config,
                                          cpl_propertylist *keywords,
                                          const int fibre,
                                          double MJD,
                                          const char *date);

cpl_error_code espdr_save_TH_REF_TABLE(cpl_table *THAR_lines_table_FITS,
                                       cpl_frameset *frameset,
                                       cpl_frameset *used_frames,
                                       cpl_parameterlist *parameters,
                                       espdr_inst_config *inst_config,
                                       cpl_propertylist *keywords,
                                       const int fibre,
                                       const char *lines_type,
                                       double MJD,
                                       const char *date,
                                       char *ins_mode_bin_RE);

cpl_error_code espdr_save_wave_S2D_products(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            const char *recipe_id,
                                            cpl_frameset *used_frames,
                                            const char *WAVE_tag,
                                            int fibre_nr,
                                            cpl_propertylist *keywords,
                                            espdr_inst_config *inst_config,
                                            cpl_image *flat_corr_spectrum,
                                            cpl_image *flat_corr_error,
                                            cpl_image *flat_corr_qual,
                                            cpl_image *flat_blaze_corr_spectrum,
                                            cpl_image *flat_blaze_corr_error,
                                            cpl_image *flat_blaze_corr_qual);

#endif /* ESPDR_WAVE_CAL_H */
