/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_science.h>
#include <espdr_overscan.h>
#include <espdr_instrument.h>
#include <espdr_background.h>
#include <espdr_blaze.h>
#include <espdr_wave_cal.h>
#include <espdr_orders.h>
#include <espdr_pixels.h>
#include <espdr_ccf.h>
#include <stdlib.h>
#include <gsl/gsl_sort.h>
//#include <omp.h>
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_sci_red(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

	cpl_frameset *used_frames = NULL;
    cpl_frame *science_frame;
	cpl_image *rel_eff = NULL;

	const char *extinction_filename = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 47;
    const char* rec_tags[47] = {
        ESPDR_CCD_GEOM, ESPDR_PRO_CATG_MBIAS_RES,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK, ESPDR_PRO_CATG_BAD_PIXELS,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_MASK, ESPDR_MASK_LUT_TABLE,
        ESPDR_FLUX_TEMPLATE, ESPDR_EXTINCTION_TABLE, ESPDR_STD_STARS_TABLE,
        ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_WAVE_MATRIX_LFC_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_FP_LFC_B,
        ESPDR_PRO_CATG_DLL_MATRIX_LFC_FP_A, ESPDR_PRO_CATG_DLL_MATRIX_FP_LFC_B,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_A,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_B, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_FP_A,
        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_A, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_B,
        ESPDR_PRO_CATG_CONTAM_FP, ESPDR_PRO_CATG_CONTAM_LFC, ESPDR_PRO_CATG_CONTAM_THAR,
        ESPDR_PRO_CATG_REL_EFF, ESPDR_PRO_CATG_ABS_EFF,
        ESPDR_PRO_CATG_S2D_BLAZE_THAR_FP_B, ESPDR_PRO_CATG_S2D_BLAZE_THAR_THAR_B,
        ESPDR_SCIENCE_RAW_DARK, ESPDR_SCIENCE_RAW_SKY,
        ESPDR_SCIENCE_RAW_FP, ESPDR_SCIENCE_RAW_THAR, ESPDR_SCIENCE_RAW_LFC,
        ESPDR_CRH_MAP, ESPDR_ORDERS_MASK,
        ESPDR_HITRAN_LINES_QT, ESPDR_HITRAN_LINES_STRONGEST, ESPDR_HITRAN_CCF_MASK, ESPDR_RES_MAP_A,
        ESPDR_OH_SKY_MASTER_A, ESPDR_OH_SKY_MASTER_B, ESPDR_OH_LINES_REGIONS
    };
    int is_required[47] = {1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // WAVE/DLL matrixes
        0, 0, 0, // CONTAMs
        0, 0, // Efficiencies
        0, 0, // S2D_BLAZE for drift calculation
        0, 0, 0, 0, 0, // raw frames
        0, // CRH map
        0, // orders mask
        0, 0, 0, 0, // telluric correction
        0, 0, 0}; // OH correction

	cpl_msg_set_level(CPL_MSG_INFO);

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 1) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

	espdr_CCD_geometry *CCD_geom = NULL;
	espdr_inst_config *inst_config = NULL;
	espdr_qc_keywords *qc_kws = NULL;
	espdr_OVSC_param *OVSC_param = NULL;
	espdr_SCI_RED_param *SCI_RED_param = NULL;
	cpl_frame* CCD_geom_frame = NULL;
	cpl_frame* inst_config_frame = NULL;

	espdr_msg("Starting sci_red");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw science image with FP or SKY, tagged as OBJ_FP or OBJ_SKY\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "table of stabdards stars fluxes, tagged as STD_TABLE\n"
                 "extinction table, tagged as EXT_TABLE\n"
                 "one or more mask tables, tagged as MASK_TABLE\n"
                 "mask look-up table, tagged as MASK_LUT\n"
                 "flux_template, tagged as FLUX_TEMPLATE\n"
                 "master bias residuals image (optional), tagged as MASTER_BIAS_RES\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n"
                 "wave matrix for fibres A and B, tagged as WAVE_MATRIX_THAR_FP_A or WAVE_MATRIX_FP_THAR_B\n"
                 "dll matrix for fibres A and B, tagged as DLL_MATRIX_THAR_FP_A or DLL_MATRIX_FP_THAR_B\n"
                 "relative efficiency image for fibre B, tagged as REL_EFF_B\n"
                 "absolute image for fibre A, tagged as ABS_EFF_A\n"
                 "s2d blaze of THAR_FP, if OBJ_FP, tagged as S2D_BLAZE_THAR_FP_B\n"
                 "contamination image, tagged as CAL_CONTAM_FP\n");

	/* AMO Changed way to read CCD geom frame and to init relevant recipe
	* internal parameters structures to simplify this calling function
	*/
	OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
	CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
	espdr_ensure(CCD_geom_frame == NULL, cpl_error_get_code(),
                 "CCD geometry frame not found!");
	CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
	inst_config_frame = espdr_get_inst_config(frameset);
	espdr_ensure(inst_config_frame == NULL, cpl_error_get_code(),
                 "Instrument config frame not found!");
	inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
	SCI_RED_param     = espdr_SCI_RED_param_init(recipe_id, parameters);

	used_frames = cpl_frameset_new();
	my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
	my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_DARK);
    if (science_frame == NULL) {
    	science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_CP);
    	if (science_frame == NULL) {
    		science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_SKY);
    		if (science_frame == NULL) {
    			science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_FP);
    			if (science_frame == NULL) {
    				science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_THAR);
    				if (science_frame == NULL) {
    					science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_LFC);
    					if (science_frame == NULL) {
    						espdr_msg_error("No raw SCIENCE frame, exiting");
    						return(CPL_ERROR_NULL_INPUT);
    					}
    				}
    			}
    		}
    	}
        science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_SKY);
        if (science_frame == NULL) {
            science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_FP);
            if (science_frame == NULL) {
                science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_THAR);
                if (science_frame == NULL) {
                    science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_LFC);
                    if (science_frame == NULL) {
                        science_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_RAW_CP);
                        if (science_frame == NULL) {
                            espdr_msg_error("No raw SCIENCE frame, exiting");
                            return(CPL_ERROR_NULL_INPUT);
                        }
                    }
                }
            }
        }
    }
    const char *SCIENCE_tag = cpl_frame_get_tag(science_frame);

    const char *input_filename = cpl_frame_get_filename(science_frame);
    if (espdr_frameset_has_frame(frameset, ESPDR_SCIENCE_HEADER_RAW)) {
        cpl_frame *header_frame = espdr_frame_find(frameset, ESPDR_SCIENCE_HEADER_RAW);
        my_error = cpl_frameset_insert(used_frames,
                                       cpl_frame_duplicate(header_frame));
        input_filename = cpl_frame_get_filename(header_frame);
    }
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(science_frame));

    /* Load science keywords from the first image (primary header) */
    espdr_msg("KEYWORDS for SCI_RED input filename: %s", input_filename);
    cpl_propertylist *keywords = cpl_propertylist_load(input_filename, 0);

    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "keywords are NULL");

#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
    if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
        char *arc_filename = strrchr(input_filename, '/')+1;
        espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
        my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Updating the ARCFILE KW (%s) failed: %s",
                     arc_filename, cpl_error_get_message_default(my_error));
    }
#endif

    espdr_msg("SCI_RED processing started ...");

    char *spec_type = (char *)cpl_malloc(32*sizeof(char));
    my_error = espdr_get_spectral_type(inst_config, keywords, spec_type);

    /* ASE: for mask_table_id, make sure to take the 2 first chars */
    char *mask_table_id = (char *)cpl_malloc(4 * sizeof(char));
    strncpy(mask_table_id, spec_type, 2);
    mask_table_id[2] = '\0';

	/* Getting from DPR.TYPE what we have on each fibre ...*/
    char *fibre_a = (char *)cpl_malloc(32 * sizeof(char));
    char *fibre_b = (char *)cpl_malloc(32 * sizeof(char));

    if (!cpl_propertylist_has(keywords, inst_config->dpr_type_kw)) {
        espdr_msg_error("No %s KW found, exiting", inst_config->dpr_type_kw);
    }
    const char *dpr_type = cpl_propertylist_get_string(keywords,
                                                       inst_config->dpr_type_kw);
    espdr_msg( "%s DPR TYPE: %s", inst_config->instrument, dpr_type);

    char *comma;
    comma = strtok((char*)dpr_type, ",");
    strcpy(fibre_a, comma);
    comma = strtok(NULL, ",");
    strcpy(fibre_b, comma);
    
    if (strcmp(fibre_b, "NONE") == 0) {
        strcpy(fibre_b, "DARK");
    }
    
    // To be able to reduce the HARPS polarimetry data
    if (strstr(fibre_b, "POL") != NULL) {
        strcpy(fibre_b, "SKY");
    }
    
    // To be able to reduce POET OBJECT,OBJECT exposures as OBJECT,SKY
    if (strcmp(fibre_b, "OBJECT") == 0) {
        strcpy(fibre_b, "SKY");
    }
    
    // To be able to reduce the RV standards
    if ((strcmp(fibre_a, "STD") == 0) && (strcmp(fibre_b, "RV") == 0)) {
        strcpy(fibre_a, "OBJECT");
        strcpy(fibre_b, "FP");
    }
    
    // Beware, the wavecal_src_kw is not used as it should be.
    // Since it was not used for calibration reference - wave matrix (THAR or LFC),
    // it is used as calibration source on fibre B (mostly THAR or FP) for HARPS & HARPS-N
    if (strcmp(fibre_b, "WAVE") == 0) {
        strcpy(fibre_b, cpl_propertylist_get_string(keywords,
                                                    inst_config->wavecal_src_kw));
        if (strstr(fibre_b, "THAR") != NULL) {
            fibre_b[4] = '\0';
        }
    }
    espdr_msg( "FIBRE A = %s", fibre_a);
    espdr_msg( "FIBRE B = %s", fibre_b);

	int fibres_processed;

	if(strcmp(fibre_b,"DARK")==0) {
		fibres_processed = inst_config->fibres_nb - 1;
	} else {
		fibres_processed = inst_config->fibres_nb;
	}

	espdr_msg( "FIBRES TO BE PROCESSED = %d",fibres_processed);

    char sim_ref[10];
    char *token;
    token = strtok((char *)SCIENCE_tag,"OBJ_");
    strcpy(sim_ref, token);

	/* Parameters overloading ...*/

	double log_lambda_ini = inst_config->log_lambda_ini;
	double log_lambda_end = inst_config->log_lambda_end;
	double delta_log_lambda = inst_config->delta_log_lambda;
	double mask_width = inst_config->mask_width;
    int mask_changed = 0;
	double rv_center;
	double rv_range = inst_config->rv_range;
	double rv_step = inst_config->rv_step;
	double ksigma_cosmic = inst_config->extraction_ksigma_sky;
    int remove_bias_res_flag = 1;
	char *wave_cal_source = (char *)cpl_malloc(16 * sizeof(char));
	char *extraction_method = (char *)cpl_malloc(16 * sizeof(char));
	char *background_sw = (char *)cpl_malloc(16 * sizeof(char));
    char *bias_res_removal_sw = (char *)cpl_malloc(16 * sizeof(char));
    char *flux_correction_type = (char *)cpl_malloc(16 * sizeof(char));
	char *drift_correction_sw = (char *)cpl_malloc(16 * sizeof(char));
    int drift_set = 0;
    char *sky_correction_method = (char *)cpl_malloc(32 * sizeof(char));
    int sky_subtraction_slidebox = inst_config->sci_sky_sub_sliding_box_size;
    double slit_loss = -1.0;
    int lacosmic_flag = 0;
    int tell_corr_flag = 0;
    
    my_error = espdr_get_science_params(inst_config, keywords, parameters,
                                        SCI_RED_param, fibre_b,
                                        &log_lambda_ini, &log_lambda_end,
                                        &delta_log_lambda,
                                        &mask_width, mask_table_id, &mask_changed,
                                        &rv_center, &rv_range, &rv_step,
                                        &ksigma_cosmic,
                                        &remove_bias_res_flag,
                                        wave_cal_source,
                                        extraction_method,
                                        background_sw,
                                        bias_res_removal_sw,
                                        flux_correction_type,
                                        drift_correction_sw,
                                        &drift_set,
                                        sky_correction_method,
                                        &sky_subtraction_slidebox,
                                        &slit_loss, &lacosmic_flag,
                                        &tell_corr_flag);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_science_params failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (strcmp(fibre_b, "DARK") == 0) {
        strcpy(sky_correction_method, "none");
    }
    
    cpl_table *extinction_table = NULL;
    cpl_table *abs_eff_table = NULL;
    cpl_table *flux_template_table = NULL;
    cpl_table *mask_table = NULL;
    const char *mask_id = NULL;
    my_error = espdr_science_get_static_tables(frameset, used_frames,
                                               keywords, inst_config,
                                               &rel_eff, &abs_eff_table,
                                               &extinction_table,
                                               mask_changed, mask_table_id,
                                               &mask_id, &mask_table,
                                               flux_correction_type, spec_type,
                                               &flux_template_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_science_get_static_tables failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if ((inst_config->inst_type == VIS) && (strcmp(token, "SKY") == 0) &&
        (strcmp(sky_correction_method, "none") != 0) && (rel_eff == NULL)) {
        espdr_msg(ANSI_COLOR_RED"\t\tWarning : NO relative efficiency table provided. No sky subtraction."ANSI_COLOR_RESET);
        strcpy(sky_correction_method, "none");
    }
    
    espdr_msg("mask_id: %s, mask_table_id: %s, flux_correction_type: %s, spec_type: %s",
              mask_id, mask_table_id, flux_correction_type, spec_type);

    double exp_time_hour, ra_ft, dec_ft, pm_ra, pm_dec;
    int year, month, day, hour, minutes;
    double seconds, time_to_mid_exposure = 0.5, airmass;

    my_error = espdr_get_berv_params(inst_config, qc_kws, keywords, science_frame,
                                     &exp_time_hour, &ra_ft, &dec_ft, &pm_ra, &pm_dec,
                                     &year, &month, &day, &hour, &minutes, &seconds,
                                     &time_to_mid_exposure, &airmass);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_berv_params failed: %s",
                 cpl_error_get_message_default(my_error));

    /* BERV calculation */
    double BERV;    /* Barycentric Earth radial velocity (km/s) */
    double VTOT;    /* Total velocity of observer (km/s) */
    double BJD;     /* Barycentric time of light arrival (UTC Julian date) */
    double BERVMAX; /* Yearly maximum value of BERV (km/s) */
    double SED;     /* Sun-Earth distance (in km) */
    double BERV_factor;

    my_error = espdr_compute_BERV(ra_ft, dec_ft, pm_ra, pm_dec,
                                  year, month, day, hour, minutes, seconds,
                                  time_to_mid_exposure,
                                  inst_config->tel_geolon,
                                  inst_config->tel_geolat,
                                  inst_config->tel_geoelev,
                                  inst_config->berv_type_init,
                                  &BERV, &VTOT, &BJD, &BERVMAX, &SED);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_BERV failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_compute_BERV_factor(BERV, SED, VTOT, &BERV_factor);
    
    
    int *orders_nb_per_fibre = NULL;
    orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    /* Getting the physical orders id */
    int **physical_orders_id = (int **)cpl_calloc(inst_config->fibres_nb, sizeof(int *));
    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        physical_orders_id[fibre] = (int *)cpl_calloc(orders_nb_per_fibre[fibre], sizeof(int));
        espdr_get_order_phys_numbers(CCD_geom, inst_config, fibre, physical_orders_id[fibre]);
    }

    int **cosmics_nb = (int **) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cosmics_nb[i] = (int *) cpl_calloc(orders_nb_per_fibre[i],sizeof(int));
    }

    double **snr_per_fibre = (double **) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        snr_per_fibre[i] = (double *) cpl_calloc(orders_nb_per_fibre[i],sizeof(double));
    }

    double lamp_offset_ar[inst_config->fibres_nb];
    double lamp_offset_ar1[inst_config->fibres_nb];
    double lamp_offset_ar2[inst_config->fibres_nb];
    double mjd_obs_delta_time_wave[inst_config->fibres_nb];
    double *ron = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    cpl_image **blaze_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **wave_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **wave_matrix_air = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **dll_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **dll_matrix_air = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_flux = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_err = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **s2d_ref_image = (cpl_image **)cpl_malloc(3 * sizeof(cpl_image *));
    my_error = espdr_process_raw_till_extraction(frameset, parameters, recipe_id,
                                                 science_frame,
                                                 used_frames,
                                                 inst_config, CCD_geom, qc_kws,
                                                 keywords,
                                                 remove_bias_res_flag,
                                                 wave_cal_source, fibre_b, lacosmic_flag,
                                                 background_sw, extraction_method,
                                                 lamp_offset_ar, lamp_offset_ar1, lamp_offset_ar2,
                                                 ron, cosmics_nb, snr_per_fibre,
                                                 mjd_obs_delta_time_wave,
                                                 blaze_table,
                                                 wave_matrix, wave_matrix_air,
                                                 dll_matrix, dll_matrix_air,
                                                 s2d_ref_image,
                                                 flat_corr_flux,
                                                 flat_corr_err,
                                                 flat_corr_qual);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_raw_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Adding CDELT 1 to QC keywords */
    my_error = espdr_keyword_add_double(qc_kws->cdelt1,
                                        rv_step,
                                        "Coordinate increment per pixel",
                                        &keywords);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword CDELT1 to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Simultaneous reference: -%s-",sim_ref);

    cpl_image *flux_sky_corr;
    cpl_image *err_sky_corr;
    cpl_image *qual_sky_corr;
    cpl_image *flux_sky_sub_s2d_blaze;
    cpl_image *error_sky_sub_s2d_blaze;
    cpl_image *qual_sky_sub_s2d_blaze;
    cpl_image *flux_sky_sub_s2d;
    cpl_image *error_sky_sub_s2d;
    cpl_image *qual_sky_sub_s2d;
    
    cpl_propertylist *OH_corr_QC_keywords = NULL;

    int *nx = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    int *ny = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));

    for (int fibre = 0; fibre < fibres_processed; fibre++) {
        nx[fibre] = cpl_image_get_size_x(flat_corr_flux[fibre]);
        ny[fibre] = cpl_image_get_size_y(flat_corr_flux[fibre]);
    }
    
    if (strcmp(token, "SKY") == 0) {
        
        if (inst_config->inst_type == NIR) {
            espdr_msg("sky correction method: %s", sky_correction_method);
            if (strcmp(sky_correction_method, "oh-corr") == 0) {
                espdr_msg("OH correction...");
                espdr_msg("Using the S2D_BLAZE_A");
                double Texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
                double airmass_mean = (cpl_propertylist_get_double(keywords, inst_config->airm_start_kw) +
                                       cpl_propertylist_get_double(keywords, inst_config->airm_end_kw)) / 2.0;
                OH_corr_QC_keywords = cpl_propertylist_new();
                my_error = espdr_correct_OH_lines(frameset, used_frames, inst_config, qc_kws, Texp, airmass_mean,
                                                  flat_corr_flux[0], flat_corr_err[0], flat_corr_qual[0],
                                                  flat_corr_flux[1], flat_corr_err[1], flat_corr_qual[1],
                                                  wave_matrix[0], wave_matrix[1],
                                                  &flux_sky_sub_s2d_blaze, &error_sky_sub_s2d_blaze,
                                                  &qual_sky_sub_s2d_blaze, &OH_corr_QC_keywords);
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_correct_OH_lines failed: %s",
                             cpl_error_get_message_default(my_error));
                
                espdr_msg("OH corrected");
            } else {
                espdr_msg("No OH lines correction");
            }
        } else { // inst_type == VIS

            if (strcmp(sky_correction_method, "none") != 0) {
                flux_sky_sub_s2d_blaze = cpl_image_new(nx[0], ny[0], CPL_TYPE_DOUBLE);
                error_sky_sub_s2d_blaze = cpl_image_new(nx[0], ny[0], CPL_TYPE_DOUBLE);
                qual_sky_sub_s2d_blaze = cpl_image_new(nx[0], ny[0], CPL_TYPE_INT);
                
                my_error = espdr_process_sky(
                            flat_corr_flux[1],
                            flat_corr_err[1],
                            flat_corr_qual[1],
                            wave_matrix_air,
                            dll_matrix_air,
                            rel_eff,
                            inst_config,
                            CCD_geom,
                            physical_orders_id,
                            &flux_sky_corr,
                            &err_sky_corr,
                            &qual_sky_corr);

                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_process_sky failed: %s",
                             cpl_error_get_message_default(my_error));

                /* substraction only if fibre related to SKY */
                my_error = espdr_subtract_sky(inst_config,
                                              sky_correction_method,
                                              sky_subtraction_slidebox,
                                              flat_corr_flux[0],
                                              flat_corr_err[0],
                                              flat_corr_qual[0],
                                              flux_sky_corr,
                                              err_sky_corr,
                                              qual_sky_corr,
                                              &flux_sky_sub_s2d_blaze,
                                              &error_sky_sub_s2d_blaze,
                                              &qual_sky_sub_s2d_blaze);

                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_subtract_sky failed: %s",
                             cpl_error_get_message_default(my_error));
            } else {
                espdr_msg("No sky subtraction");
            }
        }
        
        if (strcmp(sky_correction_method, "none") != 0) {
            espdr_msg("Correct blaze skysub ...");
            my_error = espdr_correct_blaze(flux_sky_sub_s2d_blaze,
                                           error_sky_sub_s2d_blaze,
                                           qual_sky_sub_s2d_blaze,
                                           blaze_table[0],
                                           &flux_sky_sub_s2d,
                                           &error_sky_sub_s2d,
                                           &qual_sky_sub_s2d);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_correct_blaze failed for skysub: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("Correct blaze skysubfinished !");
        }
	}

    // Creating the wavelength grids
    cpl_image *wavelenght_grid_pow_10 = NULL;
    cpl_image *wavelenght_air_grid_pow_10 = NULL;
    my_error = espdr_create_wavelength_grid(log_lambda_ini,
                                            log_lambda_end,
                                            delta_log_lambda,
                                            &wavelenght_grid_pow_10,
                                            &wavelenght_air_grid_pow_10);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_wavelength_grid failed: %s",
                 cpl_error_get_message_default(my_error));

	cpl_image **flux_s2d;
	cpl_image **error_s2d;
	cpl_image **qual_s2d;

    cpl_image *flux_tell_corr_s2d_blaze;
    cpl_image *error_tell_corr_s2d_blaze;
    cpl_image *qual_tell_corr_s2d_blaze;
    cpl_image *flux_tell_corr_s2d;
    cpl_image *error_tell_corr_s2d;
    cpl_image *qual_tell_corr_s2d;

	cpl_table *s1d_energy = NULL;
	cpl_table *s1d_fluxcal = NULL;
	cpl_table *s1d_fluxcal_skysub = NULL;
    cpl_table *s1d_fluxcal_tell_corr = NULL;
    
    flux_s2d = (cpl_image **) cpl_malloc(fibres_processed*sizeof(cpl_image *));
    error_s2d = (cpl_image **) cpl_malloc(fibres_processed*sizeof(cpl_image *));
    qual_s2d = (cpl_image **) cpl_malloc(fibres_processed*sizeof(cpl_image *));

    cpl_table **s1d_table_merged = (cpl_table **) cpl_malloc(fibres_processed*sizeof(cpl_table *));
    cpl_table *s1d_table_merged_skysub = NULL;
    cpl_table *s1d_table_merged_tell_corr = NULL;

	cpl_image **wave_shifted = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **wave_air_shifted = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **dll_shifted = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **dll_air_shifted = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **wave_shifted_drift_corr = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **wave_air_shifted_drift_corr = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **dll_shifted_drift_corr = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **dll_air_shifted_drift_corr = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));

    /* Creating table drift_results */
    cpl_table **drift_results = NULL;
    drift_results = (cpl_table **) cpl_malloc(fibres_processed*sizeof(cpl_table *));
    for (int fibre = 0; fibre < fibres_processed; fibre++) {
        drift_results[fibre] = cpl_table_new(CCD_geom->ext_nb);
        int first_order_second_ext;
        my_error = espdr_drift_table_init(drift_results[fibre], CCD_geom, inst_config,
                                          fibre, &first_order_second_ext);
    } // fibres

	cpl_image **drift_matrix = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	for (int fibre = 0; fibre < fibres_processed; fibre++) {
		drift_matrix[fibre] = cpl_image_new(nx[fibre], ny[fibre], CPL_TYPE_DOUBLE);
	}
	cpl_image **wave_matrix_drift_corrected = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
	cpl_image **wave_matrix_air_drift_corrected = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));

    /* Calculate and correct drift ...*/
    char *drift_method = (char *)cpl_malloc(64*sizeof(char));
    if (strcmp(token, "THAR") == 0) { strcpy(drift_method, inst_config->drift_method_thar); }
    if (strcmp(token, "FP") == 0) { strcpy(drift_method, inst_config->drift_method_fp); }
    if (strcmp(token, "LFC") == 0) { strcpy(drift_method, inst_config->drift_method_lfc); }

	if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
        cpl_image *drift_matrix_test = cpl_image_new(nx[1], ny[1], CPL_TYPE_DOUBLE);
        /* Measure drift for fibre B !*/
        int correct_drift = 1;
        int drift_QC = 1;
        my_error = espdr_measure_drift(flat_corr_flux[1],
                                       flat_corr_err[1],
                                       flat_corr_qual[1],
                                       s2d_ref_image[0],
                                       s2d_ref_image[1],
                                       s2d_ref_image[2],
                                       wave_matrix_air[1],
                                       dll_matrix_air[1],
                                       drift_method,
                                       inst_config->drift_space,
                                       inst_config->drift_ksigma,
                                       inst_config->drift_max_flux_threshold,
                                       inst_config->drift_min_flux_threshold,
                                       inst_config->image_cosmics_part,
                                       &drift_results[1],
                                       &drift_matrix_test);
        
        if (strcmp(token, "FP") == 0) {
            int ndet = cpl_table_get_nrow(drift_results[1]);
            int null;
            double drift_mean_comp;
            espdr_msg("DRIFT FP factor: %f", inst_config->drift_fp_factor);
            for (int det = 0; det < ndet; det++) {
                drift_mean_comp = cpl_table_get(drift_results[1],"drift_mean",det,&null);
                cpl_table_set(drift_results[1],"drift_mean",det,drift_mean_comp*inst_config->drift_fp_factor);
                espdr_msg("DET%d: drift computed: %f, drift corrected: %f",
                          det, drift_mean_comp, drift_mean_comp*inst_config->drift_fp_factor);
            }
            double final_drift;
            for (int det = 0; det < ndet; det++) {
                final_drift = cpl_table_get(drift_results[1],"drift_mean",det,&null);
                espdr_msg("DET%d: final drift: %f", det, final_drift);
            }
        }
        
        if (my_error == CPL_ERROR_ILLEGAL_INPUT) {
            correct_drift = 0;
            drift_QC = 0;
            cpl_error_reset();
        } else {
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_measure_drift failed for fibre %c: %s",
                         fibre_name[1], cpl_error_get_message_default(my_error));
        }
        cpl_image_delete(drift_matrix_test);

        if (correct_drift == 1) {
            my_error = espdr_create_drift_matrix(inst_config->drift_space,
                                                 nx, ny, physical_orders_id,
                                                 inst_config, CCD_geom,
                                                 wave_matrix_air,
                                                 dll_matrix_air,
                                                 drift_results,
                                                 drift_matrix);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_create_drift_matrix failed: %s",
                         cpl_error_get_message_default(my_error));

            my_error = espdr_check_drift_QC(drift_results[1], inst_config, CCD_geom->ext_nb, &drift_QC);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_check_drift_QC failed: %s",
                         cpl_error_get_message_default(my_error));
        } else {
            espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied, because the drift measure failed"ANSI_COLOR_RESET);
            drift_QC = 0;
        }

        // Correct drift on wave matrix if the QC is 1 or the user specified it
        //espdr_msg("Drift QC: %d", drift_QC);
        if (drift_set == 1) {
            if (strcmp(drift_correction_sw, "on") != 0) { correct_drift = 0; }
        } else {
            if (drift_QC == 0) { correct_drift = 0; }
        }

        for (int i = 0; i < fibres_processed; i++) {
            if (correct_drift) {
                espdr_msg("Drift correction applied for fibre %c", fibre_name[i]);
                my_error = espdr_correct_drift(wave_matrix[i], drift_matrix[i], //fibre B
                                               &wave_matrix_drift_corrected[i]);
                my_error = espdr_correct_drift(wave_matrix_air[i], drift_matrix[i], //fibre B
                                               &wave_matrix_air_drift_corrected[i]);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "drift correction failed for fibre %c: %s",
                             fibre_name[i], cpl_error_get_message_default(my_error));
            } else {
                if ((i == 0) && (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0)) {
                    espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied."ANSI_COLOR_RESET);
                }
                wave_matrix_drift_corrected[i] = cpl_image_duplicate(wave_matrix[i]);
                wave_matrix_air_drift_corrected[i] = cpl_image_duplicate(wave_matrix_air[i]);
            }

            my_error = espdr_shift_wavelength_solution(BERV_factor,
                                                       wave_matrix_drift_corrected[i],
                                                       dll_matrix[i],
                                                       &wave_shifted_drift_corr[i],
                                                       &dll_shifted_drift_corr[i]);
            my_error = espdr_shift_wavelength_solution(BERV_factor,
                                                       wave_matrix_air_drift_corrected[i],
                                                       dll_matrix_air[i],
                                                       &wave_air_shifted_drift_corr[i],
                                                       &dll_air_shifted_drift_corr[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "wavelength shift failed for fibre %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
        }
    } // if WAVE_SOURCE is processed.

    double seeing = 0.0;
    int seeing_kw_qc = 1;
    my_error = espdr_get_seeing(inst_config, keywords, &seeing, &seeing_kw_qc);
    cpl_propertylist *telluric_keywords;
    cpl_image *telluric_spectrum = NULL;
    
    // process by fibre ...
    for (int i = 0; i < fibres_processed; i++) {
        my_error = espdr_shift_wavelength_solution(BERV_factor,
                                                   wave_matrix[i],
                                                   dll_matrix[i],
                                                   &wave_shifted[i],
                                                   &dll_shifted[i]);
        my_error = espdr_shift_wavelength_solution(BERV_factor,
                                                   wave_matrix_air[i],
                                                   dll_matrix_air[i],
                                                   &wave_air_shifted[i],
                                                   &dll_air_shifted[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "wavelength shift failed for fibre %c: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));

        if ((strcmp(token, "SKY") == 0) && (i == 0) && (strcmp(sky_correction_method, "none") != 0)) {
            espdr_msg("rebin s1d skysub process fibre %c ...", fibre_name[i]);
            // rebin s1d process ...
            my_error = espdr_rebin_and_merge_orders(
                                        wavelenght_grid_pow_10,
                                        wavelenght_air_grid_pow_10,
                                        wave_shifted[i], /* wave_matrix in vacuum */
                                        dll_shifted[i],
                                        flux_sky_sub_s2d,
                                        error_sky_sub_s2d,
                                        qual_sky_sub_s2d,
                                        &s1d_table_merged_skysub);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_rebin_and_merge_orders failed for skysub fiber %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));

            if ((extinction_table != NULL) && (abs_eff_table != NULL)) {
                s1d_energy = cpl_table_duplicate(s1d_table_merged_skysub);
                my_error = espdr_convert_photons_to_energy(inst_config->tel_surface,
                                                           exp_time_hour,
                                                           &s1d_energy);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_convert_photons_to_energy failed for fiber %c: %s",
                             fibre_name[i], cpl_error_get_message_default(my_error));

                my_error = espdr_calibrate_flux(airmass,
                                                seeing,
                                                slit_loss,
                                                1, // slit_loss correction flag
                                                inst_config,
                                                s1d_energy,
                                                extinction_table,
                                                abs_eff_table,
                                                &s1d_fluxcal_skysub);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_calibrate_flux failed for fiber %c: %s",
                             fibre_name[i], cpl_error_get_message_default(my_error));

                cpl_table_delete(s1d_energy);
            }
            espdr_msg("merging orders skysub fibre %c finished !",fibre_name[i]);
        } // SKY processing, fibre A

        espdr_msg("rebin s1d process fibre %c ...",fibre_name[i]);
        espdr_msg("Correct blaze fibre %c ...",fibre_name[i]);
        my_error = espdr_correct_blaze(flat_corr_flux[i],
                                       flat_corr_err[i],
                                       flat_corr_qual[i],
                                       blaze_table[i],
                                       &flux_s2d[i],
                                       &error_s2d[i],
                                       &qual_s2d[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_blaze failed for fibre %c: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
        espdr_msg("Correct blaze fibre %c finished !",fibre_name[i]);
        
        // rebin s1d process ...
        if (strcmp(token,"THAR") == 0 ||
            strcmp(token,"FP") == 0 ||
            strcmp(token,"LFC") == 0) {
            my_error = espdr_rebin_and_merge_orders(
                                    wavelenght_grid_pow_10,
                                    wavelenght_air_grid_pow_10,
                                    wave_shifted_drift_corr[i], /* wave_matrix in vacuum */
                                    dll_shifted_drift_corr[i],
                                    flux_s2d[i],
                                    error_s2d[i],
                                    qual_s2d[i],
                                    &s1d_table_merged[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_rebin_and_merge_orders failed for fiber %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
        } else {
            my_error = espdr_rebin_and_merge_orders(
                                    wavelenght_grid_pow_10,
                                    wavelenght_air_grid_pow_10,
                                    wave_shifted[i], /* wave_matrix in vacuum */
                                    dll_shifted[i],
                                    flux_s2d[i],
                                    error_s2d[i],
                                    qual_s2d[i],
                                    &s1d_table_merged[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_rebin_and_merge_orders failed for fiber %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
        }

        int slit_loss_flag = 1;
        if ((extinction_table != NULL) && (abs_eff_table != NULL) & (i == 0)) {
            if ((strcmp(token, "SKY") == 0) && (i == 1)) {
                slit_loss_flag = 0;
            }

            s1d_energy = cpl_table_duplicate(s1d_table_merged[i]);
            my_error = espdr_convert_photons_to_energy(inst_config->tel_surface,
                                                       exp_time_hour,
                                                       &s1d_energy);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_convert_photons_to_energy failed for fiber %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));

            my_error = espdr_calibrate_flux(airmass,
                                            seeing,
                                            slit_loss,
                                            slit_loss_flag,
                                            inst_config,
                                            s1d_energy,
                                            extinction_table,
                                            abs_eff_table,
                                            &s1d_fluxcal);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_calibrate_flux failed for fiber %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));

            cpl_table_delete(s1d_energy);
        }
        espdr_msg("merging orders fibre %c finished !",fibre_name[i]);
        
        if (tell_corr_flag == 1) {
            // Tellurics correction start
            
            if (i == 0) { // process only the star spectrum, not the calibration lamp or sky one
                espdr_msg("Tellurics correction...");
                telluric_keywords = cpl_propertylist_new();
            
                if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
                        //espdr_msg("wave matrix size: %lld x %lld",
                        //          cpl_image_get_size_x(wave_matrix_drift_corrected[i]),
                        //          cpl_image_get_size_y(wave_matrix_drift_corrected[i]));
                        
                        my_error = espdr_correct_tellurics(frameset, used_frames, inst_config,
                                                           flat_corr_flux[i], flat_corr_err[i],
                                                           flat_corr_qual[i], blaze_table[i], keywords,
                                                           wave_matrix_drift_corrected[i],
                                                           dll_matrix[i], rv_step, BERV, BERVMAX,
                                                           &telluric_keywords, &telluric_spectrum,
                                                           &flux_tell_corr_s2d_blaze,
                                                           &error_tell_corr_s2d_blaze,
                                                           &qual_tell_corr_s2d_blaze);
                } else {
                        //espdr_msg("wave matrix size: %lld x %lld",
                        //          cpl_image_get_size_x(wave_matrix[i]),
                        //          cpl_image_get_size_y(wave_matrix[i]));
                    int OH_corr_QC = 0;
                    if ((OH_corr_QC_keywords != NULL) &&
                        cpl_propertylist_has(OH_corr_QC_keywords, qc_kws->qc_sci_red_oh_corr_check_kw)) {
                        OH_corr_QC = cpl_propertylist_get_int(OH_corr_QC_keywords, qc_kws->qc_sci_red_oh_corr_check_kw);
                    }
                    if (((strcmp(sky_correction_method, "oh-corr") == 0) && (OH_corr_QC == 1)) ||
                        (strcmp(sky_correction_method, "pixel-by-pixel") == 0) ||
                        (strcmp(sky_correction_method, "smoothed") == 0)) {
                        espdr_msg("Telluric correction on S2D_BLAZE_SKYSUB_A");
                        my_error = espdr_correct_tellurics(frameset, used_frames, inst_config,
                                                           flux_sky_sub_s2d_blaze, error_sky_sub_s2d_blaze,
                                                           qual_sky_sub_s2d_blaze, blaze_table[i], keywords,
                                                           wave_matrix[i], dll_matrix[i],
                                                           rv_step, BERV, BERVMAX,
                                                           &telluric_keywords, &telluric_spectrum,
                                                           &flux_tell_corr_s2d_blaze,
                                                           &error_tell_corr_s2d_blaze,
                                                           &qual_tell_corr_s2d_blaze);
                    } else {
                        espdr_msg("Telluric correction on S2D_BLAZE_A");
                        my_error = espdr_correct_tellurics(frameset, used_frames, inst_config,
                                                           flat_corr_flux[i], flat_corr_err[i],
                                                           flat_corr_qual[i], blaze_table[i], keywords,
                                                           wave_matrix[i], dll_matrix[i],
                                                           rv_step, BERV, BERVMAX,
                                                           &telluric_keywords, &telluric_spectrum,
                                                           &flux_tell_corr_s2d_blaze,
                                                           &error_tell_corr_s2d_blaze,
                                                           &qual_tell_corr_s2d_blaze);
                    }
                }
                
                if (my_error != CPL_ERROR_NONE) {
                    espdr_msg_warning("Tellurics correction failed: %s",
                                      cpl_error_get_message_default(my_error));
                    return (my_error);
                }
                
                espdr_msg("Tellurics corrected");
                
                // Tellurics correction end
                
                espdr_msg("rebin s1d tell_corr process fibre %c ...", fibre_name[i]);
                espdr_msg("Correct blaze tell_corr fibre %c ...", fibre_name[i]);
                my_error = espdr_correct_blaze(flux_tell_corr_s2d_blaze,
                                               error_tell_corr_s2d_blaze,
                                               qual_tell_corr_s2d_blaze,
                                               blaze_table[i],
                                               &flux_tell_corr_s2d,
                                               &error_tell_corr_s2d,
                                               &qual_tell_corr_s2d);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_correct_blaze failed for tell_corr fibre %c: %s",
                             fibre_name[i], cpl_error_get_message_default(my_error));
                espdr_msg("Correct blaze tell_corr fibre %c finished !", fibre_name[i]);
                
#if 0
                cpl_propertylist *keywords_test = cpl_propertylist_duplicate(keywords);
                cpl_image **images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
                images_to_save[0] = cpl_image_duplicate(flux_tell_corr_s2d);
                images_to_save[1] = cpl_image_duplicate(error_tell_corr_s2d);
                images_to_save[2] = cpl_image_duplicate(qual_tell_corr_s2d);
                my_error = cpl_propertylist_update_string(keywords_test, PRO_CATG_KW,
                                                          "S2D_TELL_CORR_A");
                my_error = cpl_propertylist_update_string(keywords_test,
                                                          PRODCATG_KW, "ANCILLARY.2DECHELLE");
                my_error = espdr_dfs_save_data_err_qual(frameset,
                                                        parameters,
                                                        used_frames,
                                                        "sci_red",
                                                        keywords_test,
                                                        "NIRPS_S2D_TELL_CORR_A.fits",
                                                        images_to_save);
                my_error = cpl_propertylist_erase(keywords_test, PRODCATG_KW);
                
                for (int im = 0; im < 3; im++) {
                    cpl_image_delete(images_to_save[im]);
                }
                cpl_free(images_to_save);
#endif

                // rebin s1d process ...
                if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
                    my_error = espdr_rebin_and_merge_orders(
                                                wavelenght_grid_pow_10,
                                                wavelenght_air_grid_pow_10,
                                                wave_shifted_drift_corr[i], /* wave_matrix in vacuum */
                                                dll_shifted_drift_corr[i],
                                                flux_tell_corr_s2d,
                                                error_tell_corr_s2d,
                                                qual_tell_corr_s2d,
                                                &s1d_table_merged_tell_corr);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_rebin_and_merge_orders failed for skysub fiber %c: %s",
                                 fibre_name[i], cpl_error_get_message_default(my_error));
                } else {
                    my_error = espdr_rebin_and_merge_orders(
                                                wavelenght_grid_pow_10,
                                                wavelenght_air_grid_pow_10,
                                                wave_shifted[i], /* wave_matrix in vacuum */
                                                dll_shifted[i],
                                                flux_tell_corr_s2d,
                                                error_tell_corr_s2d,
                                                qual_tell_corr_s2d,
                                                &s1d_table_merged_tell_corr);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_rebin_and_merge_orders failed for skysub fiber %c: %s",
                                 fibre_name[i], cpl_error_get_message_default(my_error));
                }

                if ((extinction_table != NULL) && (abs_eff_table != NULL)) {
                    s1d_energy = cpl_table_duplicate(s1d_table_merged_tell_corr);
                    my_error = espdr_convert_photons_to_energy(inst_config->tel_surface,
                                                               exp_time_hour,
                                                               &s1d_energy);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_convert_photons_to_energy failed for fiber %c: %s",
                                 fibre_name[i], cpl_error_get_message_default(my_error));
                    
                    my_error = espdr_calibrate_flux(airmass,
                                                    seeing,
                                                    slit_loss,
                                                    1, // slit_loss correction flag
                                                    inst_config,
                                                    s1d_energy,
                                                    extinction_table,
                                                    abs_eff_table,
                                                    &s1d_fluxcal_tell_corr);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_calibrate_flux failed for fiber %c: %s",
                                 fibre_name[i], cpl_error_get_message_default(my_error));

                    cpl_table_delete(s1d_energy);
                }
                espdr_msg("merging orders telluric corrected fibre %c finished !",fibre_name[i]);
            }
        } else {
            espdr_msg("No tellurics correction");
        }
        
    } // end process by fibre.

    /* CCF calculation by fibre  */
    cpl_array *RV_table;
    my_error = espdr_compute_rv_table(rv_range, rv_step, rv_center, &RV_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_rv_table failed");

    /* Adding CRVAL1 keywords */
    my_error = espdr_keyword_add_double(qc_kws->crval1,
                                        cpl_array_get_double(RV_table,0,NULL),
                                        "Coordinate at reference pixel",
                                        &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword CRVAL1 to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    int ny_s2d;
    int nx_ccf = cpl_array_get_size(RV_table);

    cpl_image *CCF_flux_sky_sub = NULL;
    cpl_image *CCF_error_sky_sub = NULL;
    cpl_image *CCF_quality_sky_sub = NULL;

    cpl_image *CCF_flux_tell_corr = NULL;
    cpl_image *CCF_error_tell_corr = NULL;
    cpl_image *CCF_quality_tell_corr = NULL;

    cpl_image **CCF_flux = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **CCF_error = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **CCF_quality = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **flux_corr_s2d = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **flux_no_cosmics_s2d = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));

    for (int i = 0; i < fibres_processed; i++) {
        CCF_flux[i] = NULL;
        CCF_error[i] = NULL;
        CCF_quality[i] = NULL;
        flux_corr_s2d[i] = cpl_image_new(nx[i], ny[i], CPL_TYPE_DOUBLE);
        flux_no_cosmics_s2d[i] = cpl_image_new(nx[i], ny[i], CPL_TYPE_DOUBLE);
    }

	double *flux_correction = (double *)cpl_calloc(orders_nb_per_fibre[0], sizeof(double));
    double *flux_correction_sky_sub = (double *)cpl_calloc(orders_nb_per_fibre[0], sizeof(double));
    double *flux_correction_tell_corr = (double *)cpl_calloc(orders_nb_per_fibre[0], sizeof(double));
    int **cosmics_nb_2 = (int **)cpl_malloc(fibres_processed*sizeof(int *));
    for (int i = 0; i < fibres_processed; i++) {
        cosmics_nb_2[i] = (int *)cpl_calloc(ny[i], sizeof(int));
    }
    int *cosmics_nb_2_skysub = (int *)cpl_calloc(ny[0], sizeof(int));
    int *cosmics_nb_2_tell_corr = (int *)cpl_calloc(ny[0], sizeof(int));

	if (strcmp(token,"SKY") == 0) {
        if (strcmp(sky_correction_method, "none") != 0) {
            espdr_msg("Cosmics correction on sky-subtracted s2d ...");
            my_error = espdr_remove_cosmics_via_histogram(flux_sky_sub_s2d_blaze,
                                                          error_sky_sub_s2d_blaze,
                                                          flux_no_cosmics_s2d[0],
                                                          cosmics_nb_2_skysub);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_remove_cosmics_via_histogram failed for sky-subtracted s2d: %s",
                         cpl_error_get_message_default(my_error));

            espdr_msg("Starting flux correction on sky-subtracted s2d ...");
            my_error = espdr_correct_flux(flux_no_cosmics_s2d[0],
                                          wave_matrix_air[0],
                                          flux_template_table,
                                          mask_table,
                                          flux_correction_type, // changed from flux_corr_type
                                          flux_corr_s2d[0],
                                          flux_correction_sky_sub);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_correct_flux failed for sky-subtracted s2d: %s",
                         cpl_error_get_message_default(my_error));

            ny_s2d = cpl_image_get_size_y(flux_sky_sub_s2d);

            espdr_msg("Computing CCF for fibre %c sky_sub for size_y = %d ...",
                      fibre_name[0], ny_s2d);

            CCF_flux_sky_sub    = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
            CCF_error_sky_sub   = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
            CCF_quality_sky_sub = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);

            my_error = espdr_compute_CCF(wave_air_shifted[0],
                                         dll_air_shifted[0],
                                         flux_corr_s2d[0],
                                         error_sky_sub_s2d_blaze,
                                         blaze_table[0],
                                         qual_sky_sub_s2d,
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux_sky_sub,
                                         &CCF_error_sky_sub,
                                         &CCF_quality_sky_sub);

            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_CCF failed for sky_sub: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("Compute CCF for fibre %c sky_sub finished!", fibre_name[0]);
        }
        for (int i = 0; i < fibres_processed; i++) {
            espdr_msg("Cosmics correction on s2d ...");
            my_error = espdr_remove_cosmics_via_histogram(flat_corr_flux[i],
                                                          flat_corr_err[i],
                                                          flux_no_cosmics_s2d[i],
                                                          cosmics_nb_2[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_remove_cosmics_via_histogram failed for sky-subtracted s2d: %s",
                         cpl_error_get_message_default(my_error));

            if (i == 0) {
                espdr_msg("Starting flux correction for fibre A ...");
                my_error = espdr_correct_flux(flux_no_cosmics_s2d[i],
                                              wave_matrix_air[i],
                                              flux_template_table,
                                              mask_table,
                                              flux_correction_type, // changed from flux_corr_type
                                              flux_corr_s2d[i],
                                              flux_correction);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_correct_flux failed: %s",
                             cpl_error_get_message_default(my_error));
            } else {
                // No flux correction on the SKY
                flux_corr_s2d[i] = cpl_image_duplicate(flux_no_cosmics_s2d[i]);
            }

            ny_s2d = cpl_image_get_size_y(flux_s2d[i]);

            espdr_msg("Computing CCF for fibre %c for size_y = %d ...",fibre_name[i], ny_s2d);
            CCF_flux[i]    = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
            CCF_error[i]   = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
            CCF_quality[i] = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);
            my_error = espdr_compute_CCF(wave_air_shifted[i],
                                         dll_air_shifted[i],
                                         flux_corr_s2d[i],
                                         flat_corr_err[i],
                                         blaze_table[i],
                                         qual_s2d[i],
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux[i],
                                         &CCF_error[i],
                                         &CCF_quality[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_CCF failed for fibre %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
            espdr_msg("Compute CCF for fibre %c finished !", fibre_name[i]);
        }
	} else {
		/* Processing CCF for wave branch, i.e., just fibre A */
        espdr_msg("Cosmics correction on s2d ...");
        my_error = espdr_remove_cosmics_via_histogram(flat_corr_flux[0],
                                                      flat_corr_err[0],
                                                      flux_no_cosmics_s2d[0],
                                                      cosmics_nb_2[0]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_remove_cosmics_via_histogram failed for fibre A: %s",
                     cpl_error_get_message_default(my_error));

		espdr_msg("Starting flux correction ...");
		my_error = espdr_correct_flux(flux_no_cosmics_s2d[0],
									  wave_matrix_air[0],
									  flux_template_table,
                                      mask_table,
                                      flux_correction_type, // changed from flux_corr_type
									  flux_corr_s2d[0],
									  flux_correction);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					 "espdr_correct_flux failed: %s",
					 cpl_error_get_message_default(my_error));

		ny_s2d = cpl_image_get_size_y(flux_s2d[0]);

        espdr_msg("Computing CCF for fibre %c for size_y = %d ...",
                  fibre_name[0], ny_s2d);
        CCF_flux[0]    = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_error[0]   = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_quality[0] = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);

        if (strcmp(token,"DARK") == 0) {
            my_error = espdr_compute_CCF(wave_air_shifted[0],
                                         dll_air_shifted[0],
                                         flux_corr_s2d[0],
                                         flat_corr_err[0],
                                         blaze_table[0],
                                         qual_s2d[0],
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux[0],
                                         &CCF_error[0],
                                         &CCF_quality[0]);

        } else {
            my_error = espdr_compute_CCF(wave_air_shifted_drift_corr[0],
                                         dll_air_shifted_drift_corr[0],
                                         flux_corr_s2d[0],
                                         flat_corr_err[0],
                                         blaze_table[0],
                                         qual_s2d[0],
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux[0],
                                         &CCF_error[0],
                                         &CCF_quality[0]);
        }
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_CCF failed for fibre %c: %s",
                     fibre_name[0], cpl_error_get_message_default(my_error));
        espdr_msg("Compute CCF for fibre %c finished !", fibre_name[0]);

        for (int j = 1; j < fibres_processed; j++) {
            CCF_flux[j] = NULL;
            CCF_error[j] = NULL;
            CCF_quality[j] = NULL;
        }
	}

    /* Processing tellurics corrected spectrum, if applicable */
    if (tell_corr_flag == 1) {
        espdr_msg("Cosmics correction on s2d corrected from tellurics...");
        my_error = espdr_remove_cosmics_via_histogram(flux_tell_corr_s2d_blaze,
                                                      error_tell_corr_s2d_blaze,
                                                      flux_no_cosmics_s2d[0],
                                                      cosmics_nb_2_tell_corr);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_remove_cosmics_via_histogram failed for fibre A: %s",
                     cpl_error_get_message_default(my_error));

        espdr_msg("Starting flux correction ...");
        my_error = espdr_correct_flux(flux_no_cosmics_s2d[0],
                                      wave_matrix_air[0],
                                      flux_template_table,
                                      mask_table,
                                      flux_correction_type, // changed from flux_corr_type
                                      flux_corr_s2d[0],
                                      flux_correction_tell_corr);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_flux failed: %s",
                     cpl_error_get_message_default(my_error));

        ny_s2d = cpl_image_get_size_y(flux_s2d[0]);

        espdr_msg("Computing CCF for fibre %c for size_y = %d ...",
                  fibre_name[0], ny_s2d);
        CCF_flux_tell_corr    = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_error_tell_corr   = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_quality_tell_corr = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);

        if ((strcmp(token,"SKY") == 0) || (strcmp(token,"DARK") == 0)) {
            my_error = espdr_compute_CCF(wave_air_shifted[0],
                                         dll_air_shifted[0],
                                         flux_corr_s2d[0],
                                         error_tell_corr_s2d_blaze,
                                         blaze_table[0],
                                         qual_s2d[0],
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux_tell_corr,
                                         &CCF_error_tell_corr,
                                         &CCF_quality_tell_corr);

        } else {
            my_error = espdr_compute_CCF(wave_air_shifted_drift_corr[0],
                                         dll_air_shifted_drift_corr[0],
                                         flux_corr_s2d[0],
                                         error_tell_corr_s2d_blaze,
                                         blaze_table[0],
                                         qual_s2d[0],
                                         RV_table,
                                         mask_table,
                                         mask_width,
                                         BERV,
                                         BERVMAX,
                                         &CCF_flux_tell_corr,
                                         &CCF_error_tell_corr,
                                         &CCF_quality_tell_corr);
        }
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_CCF failed for fibre %c: %s",
                     fibre_name[0], cpl_error_get_message_default(my_error));
        espdr_msg("Compute CCF for fibre %c finished !", fibre_name[0]);
    }
    
    
    /* CCF Gauss fitting and residuals calculation fibres A, B */
    cpl_table **residuals = NULL;
    residuals = (cpl_table **)cpl_malloc(fibres_processed*sizeof(cpl_table *));
    espdr_ngauss_result **g_res = NULL;
    g_res = (espdr_ngauss_result **)cpl_malloc(fibres_processed*sizeof(espdr_ngauss_result *));
    double *asym = (double *)cpl_calloc (fibres_processed , sizeof(double));
    double *asym_error = (double *)cpl_calloc (fibres_processed , sizeof(double));
    double *bispan = (double *)cpl_calloc (fibres_processed , sizeof(double));
    double *bispan_error = (double *)cpl_calloc (fibres_processed , sizeof(double));

    for (int fibre = 0; fibre < fibres_processed; fibre++) {
        g_res[fibre] = NULL;
        residuals[fibre] = NULL;
        g_res[fibre] = (espdr_ngauss_result *)cpl_malloc (sizeof(espdr_ngauss_result));
        espdr_msg("Fitting CCF for fibre %c ...", fibre_name[fibre]);
        my_error = espdr_fit_ccf(RV_table,
                                 fibres_processed,
                                 CCF_flux[fibre],
                                 CCF_error[fibre],
                                 CCF_quality[fibre],
                                 &asym[fibre],
                                 &asym_error[fibre],
                                 &bispan[fibre],
                                 &bispan_error[fibre],
                                 &residuals[fibre],
                                 g_res[fibre]);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_ccf failed for fibre %c: %s",
                     fibre_name[fibre], cpl_error_get_message_default(my_error));
    }

	/* CCF Gauss fitting and residuals calculation fibres A sky subtracted */
    cpl_table *residuals_skysub = NULL;
    espdr_ngauss_result *g_res_skysub = (espdr_ngauss_result *)cpl_malloc(sizeof(espdr_ngauss_result));
    double asym_skysub;
    double asym_error_skysub;
    double bispan_skysub;
    double bispan_error_skysub;

	if ((strcmp(token,"SKY") == 0) && (strcmp(sky_correction_method, "none") != 0)) {
        espdr_msg("Fitting CCF for fibre %c sky_sub ...", fibre_name[0]);
        my_error = espdr_fit_ccf(RV_table,
                                 fibres_processed,
                                 CCF_flux_sky_sub,
                                 CCF_error_sky_sub,
                                 CCF_quality_sky_sub,
                                 &asym_skysub,
                                 &asym_error_skysub,
                                 &bispan_skysub,
                                 &bispan_error_skysub,
                                 &residuals_skysub,
                                 g_res_skysub);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_ccf failed for skysub: %s",
                     cpl_error_get_message_default(my_error));
	}

    /* CCF Gauss fitting and residuals calculation fibre A tellluric corrected */
    cpl_table *residuals_tell_corr = NULL;
    espdr_ngauss_result *g_res_tell_corr = (espdr_ngauss_result *)cpl_malloc(sizeof(espdr_ngauss_result));
    double asym_tell_corr;
    double asym_error_tell_corr;
    double bispan_tell_corr;
    double bispan_error_tell_corr;

    if (tell_corr_flag == 1) {
        espdr_msg("Fitting CCF for fibre %c tell_corr ...", fibre_name[0]);
        my_error = espdr_fit_ccf(RV_table,
                                 fibres_processed, // not used, should be removed form teh function
                                 CCF_flux_tell_corr,
                                 CCF_error_tell_corr,
                                 CCF_quality_tell_corr,
                                 &asym_tell_corr,
                                 &asym_error_tell_corr,
                                 &bispan_tell_corr,
                                 &bispan_error_tell_corr,
                                 &residuals_tell_corr,
                                 g_res_tell_corr);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_ccf failed for skysub: %s",
                     cpl_error_get_message_default(my_error));
    }

	/* QC function */

	cpl_propertylist *keywords_fibre[fibres_processed];
	cpl_propertylist *keywords_skysub = NULL;
    cpl_propertylist *keywords_tell_corr = NULL;
    
    char *fibre_src[2];
    fibre_src[0] = (char *)cpl_malloc(10*sizeof(char));
    fibre_src[1] = (char *)cpl_malloc(10*sizeof(char));
    strcpy(fibre_src[0], fibre_a);
    strcpy(fibre_src[1], fibre_b);

	for (int i = 0; i < fibres_processed; i++) {
		keywords_fibre[i] = cpl_propertylist_duplicate(keywords);
        espdr_msg("Quality check for fibre %c ...", fibre_name[i]);

        my_error = espdr_sci_red_QC(CCD_geom,
                                    inst_config,
                                    qc_kws,
                                    fibre_src[i],
                                    fibre_src[1],
                                    BERV,
                                    BJD,
                                    BERVMAX,
                                    SED,
                                    VTOT,
                                    BERV_factor,
                                    lamp_offset_ar[i],
                                    lamp_offset_ar1[i],
                                    lamp_offset_ar2[i],
                                    cosmics_nb[i],
                                    cosmics_nb_2[i],
                                    snr_per_fibre[i],
                                    orders_nb_per_fibre[i],
                                    g_res[i],
                                    mask_id,
                                    drift_results[1],
                                    token,
                                    asym[i],
                                    asym_error[i],
                                    bispan[i],
                                    bispan_error[i],
                                    ron,
                                    flux_correction,
                                    &keywords_fibre[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sci_red_QC failed: %s",
                     cpl_error_get_message_default(my_error));
        espdr_msg("Quality check for fibre %c finished!", fibre_name[i]);

        if ((strcmp(token, "SKY") == 0) && (i == 0) && strcmp(sky_correction_method, "none") != 0) {
            keywords_skysub = cpl_propertylist_duplicate(keywords);
            espdr_msg("Quality check for fibre %c skysub ...", fibre_name[0]);

            // Compute the snr for sky-sub spectrum
            my_error = espdr_compute_snr(flux_sky_sub_s2d,
                                         error_sky_sub_s2d,
                                         inst_config->snr_averaging_window,
                                         snr_per_fibre[0]);

            my_error = espdr_sci_red_QC(CCD_geom,
                                        inst_config,
                                        qc_kws,
                                        fibre_src[0],
                                        fibre_src[1],
                                        BERV,
                                        BJD,
                                        BERVMAX,
                                        SED,
                                        VTOT,
                                        BERV_factor,
                                        lamp_offset_ar[i],
                                        lamp_offset_ar1[i],
                                        lamp_offset_ar2[i],
                                        cosmics_nb[0],
                                        cosmics_nb_2_skysub,
                                        snr_per_fibre[0],
                                        orders_nb_per_fibre[0],
                                        g_res_skysub,
                                        mask_id,
                                        drift_results[0],
                                        token,
                                        asym_skysub,
                                        asym_error_skysub,
                                        bispan_skysub,
                                        bispan_error_skysub,
                                        ron,
                                        flux_correction_sky_sub,
                                        &keywords_skysub);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_sci_red_QC failed: %s",
                         cpl_error_get_message_default(my_error));
            
            if (strcmp(sky_correction_method, "oh-corr") == 0) {
                my_error = espdr_propagate_OH_QC(qc_kws, OH_corr_QC_keywords, &keywords_skysub);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_propagate_OH_QC failed: %s",
                             cpl_error_get_message_default(my_error));
            }

            espdr_msg("Quality check for fibre %c skysub finished!", fibre_name[0]);
        }
	}

    if (tell_corr_flag == 1) {
        keywords_tell_corr = cpl_propertylist_duplicate(keywords);
        espdr_msg("Quality check for fibre A tell_corr ...");

        my_error = espdr_sci_red_QC(CCD_geom,
                                    inst_config,
                                    qc_kws,
                                    fibre_src[0],
                                    fibre_src[1],
                                    BERV,
                                    BJD,
                                    BERVMAX,
                                    SED,
                                    VTOT,
                                    BERV_factor,
                                    lamp_offset_ar[0],
                                    lamp_offset_ar1[0],
                                    lamp_offset_ar2[0],
                                    cosmics_nb[0],
                                    cosmics_nb_2_tell_corr,
                                    snr_per_fibre[0],
                                    orders_nb_per_fibre[0],
                                    g_res_tell_corr,
                                    mask_id,
                                    drift_results[1],
                                    token,
                                    asym_tell_corr,
                                    asym_error_tell_corr,
                                    bispan_tell_corr,
                                    bispan_error_tell_corr,
                                    ron,
                                    flux_correction_tell_corr,
                                    &keywords_tell_corr);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sci_red_QC failed: %s",
                     cpl_error_get_message_default(my_error));
        
        if ((strcmp(token, "SKY") == 0) && (strcmp(sky_correction_method, "oh-corr") == 0)) {
            my_error = cpl_propertylist_append(keywords_tell_corr, OH_corr_QC_keywords);
            my_error = espdr_propagate_OH_QC(qc_kws, OH_corr_QC_keywords, &keywords_tell_corr);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_propagate_OH_QC failed: %s",
                         cpl_error_get_message_default(my_error));
        }
        
        espdr_msg("Quality check for fibre %c tell_corr finished!", fibre_name[0]);
    }
    
    my_error = espdr_save_science_products(frameset, parameters, recipe_id, used_frames,
                                           keywords_fibre, keywords_skysub,
                                           keywords_tell_corr, telluric_keywords, OH_corr_QC_keywords,
                                           mjd_obs_delta_time_wave,
                                           inst_config, CCD_geom->ext_nb, fibres_processed,
                                           token, SCIENCE_tag,
                                           sky_correction_method, tell_corr_flag,
                                           residuals[0], drift_matrix[1],
                                           flat_corr_flux, flat_corr_err, flat_corr_qual,
                                           flux_s2d, error_s2d, qual_s2d,
                                           flux_sky_sub_s2d_blaze, error_sky_sub_s2d_blaze, qual_sky_sub_s2d_blaze,
                                           flux_sky_sub_s2d, error_sky_sub_s2d, qual_sky_sub_s2d,
                                           wave_shifted, wave_air_shifted, dll_shifted, dll_air_shifted,
                                           wave_shifted_drift_corr, wave_air_shifted_drift_corr,
                                           dll_shifted_drift_corr, dll_air_shifted_drift_corr,
                                           wave_matrix, dll_matrix, wave_matrix_air, dll_matrix_air,
                                           wave_matrix_drift_corrected, wave_matrix_air_drift_corrected,
                                           s1d_table_merged, s1d_table_merged_skysub,
                                           s1d_fluxcal_skysub, s1d_fluxcal,
                                           s1d_table_merged_tell_corr, s1d_fluxcal_tell_corr,
                                           CCF_flux, CCF_error, CCF_quality,
                                           CCF_flux_sky_sub, CCF_error_sky_sub, CCF_quality_sky_sub,
                                           CCF_flux_tell_corr, CCF_error_tell_corr, CCF_quality_tell_corr,
                                           telluric_spectrum, flux_tell_corr_s2d_blaze,
                                           error_tell_corr_s2d_blaze, qual_tell_corr_s2d_blaze,
                                           flux_tell_corr_s2d, error_tell_corr_s2d,
                                           qual_tell_corr_s2d);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_science_products failed: %s",
                 cpl_error_get_message_default(my_error));

    /* cleaning memory */
    espdr_msg("Cleaning memory");
    espdr_msg("Cleaning wave input products");
	for (int i = 0; i < inst_config->fibres_nb; i++) {
		cpl_image_delete(wave_matrix[i]);
		cpl_image_delete(wave_matrix_air[i]);
		cpl_image_delete(dll_matrix[i]);
		cpl_image_delete(dll_matrix_air[i]);
		cpl_image_delete(blaze_table[i]);
		cpl_free(physical_orders_id[i]);
	}
    cpl_free(wave_matrix);
    cpl_free(wave_matrix_air);
    cpl_free(dll_matrix);
    cpl_free(dll_matrix_air);
    cpl_free(blaze_table);
    cpl_free(physical_orders_id);
    
    espdr_msg("Cleaning the input parameters");
    cpl_free(fibre_src[0]);
    cpl_free(fibre_src[1]);
    cpl_free(mask_table_id);
    cpl_free(wave_cal_source);
    cpl_free(extraction_method);
    cpl_free(background_sw);
    cpl_free(bias_res_removal_sw);
    cpl_free(flux_correction_type);
    cpl_free(drift_correction_sw);
    cpl_free(nx);
    cpl_free(ny);

    espdr_msg("Cleaning transformed wave products & S2D & CCF spectra");
    for (int i = 0; i < fibres_processed; i++) {
		cpl_image_delete(dll_shifted[i]);
		cpl_image_delete(dll_air_shifted[i]);
		cpl_image_delete(wave_shifted[i]);
		cpl_image_delete(wave_air_shifted[i]);
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            cpl_image_delete(wave_matrix_drift_corrected[i]);
            cpl_image_delete(wave_shifted_drift_corr[i]);
            cpl_image_delete(wave_air_shifted_drift_corr[i]);
            cpl_image_delete(dll_shifted_drift_corr[i]);
            cpl_image_delete(dll_air_shifted_drift_corr[i]);
        }
		cpl_image_delete(flux_s2d[i]);
		cpl_image_delete(error_s2d[i]);
		cpl_image_delete(qual_s2d[i]);
		cpl_table_delete(s1d_table_merged[i]);
		cpl_table_delete(drift_results[i]);
		cpl_image_delete(flat_corr_flux[i]);
		cpl_image_delete(flat_corr_err[i]);
		cpl_image_delete(flat_corr_qual[i]);

        if (residuals[i] != NULL) {
            cpl_table_delete(residuals[i]);
        }
		if (CCF_flux[i] != NULL) {
            cpl_image_delete(CCF_flux[i]);
            cpl_image_delete(CCF_error[i]);
            cpl_image_delete(CCF_quality[i]);
		}
        if (g_res[i] != NULL) {
            cpl_free(g_res[i]->k);
            cpl_free(g_res[i]->sig_k);
            cpl_free(g_res[i]->x0);
            cpl_free(g_res[i]->sig_x0);
            cpl_free(g_res[i]->fwhm);
            cpl_free(g_res[i]->sig_fwhm);
            cpl_free(g_res[i]);
        }
	}

    cpl_free(wave_shifted);
    cpl_free(wave_shifted_drift_corr);
    cpl_free(wave_air_shifted_drift_corr);
    cpl_free(wave_matrix_drift_corrected);
    cpl_free(dll_shifted);
    cpl_free(dll_shifted_drift_corr);
    cpl_free(dll_air_shifted_drift_corr);

    cpl_free(flat_corr_flux);
    cpl_free(flat_corr_err);
    cpl_free(flat_corr_qual);
    cpl_free(flux_s2d);
    cpl_free(error_s2d);
    cpl_free(qual_s2d);

    cpl_free(CCF_flux);
    cpl_free(CCF_error);
    cpl_free(CCF_quality);
    if (CCF_flux_sky_sub != NULL) {
        cpl_image_delete(CCF_flux_sky_sub);
        cpl_image_delete(CCF_error_sky_sub);
        cpl_image_delete(CCF_quality_sky_sub);
    }
    cpl_free(residuals);
    cpl_table_delete(residuals_skysub);

    for (int i = 0; i < fibres_processed; i++) {
        cpl_image_delete(drift_matrix[i]);
        cpl_image_delete(flux_corr_s2d[i]);
        cpl_image_delete(flux_no_cosmics_s2d[i]);
    }
    cpl_free(drift_matrix);
    cpl_free(flux_corr_s2d);
    cpl_free(flux_no_cosmics_s2d);
	cpl_free(flux_correction);
    cpl_free(flux_correction_sky_sub);
    for (int i = 0; i < fibres_processed; i++) {
        cpl_free(cosmics_nb[i]);
        cpl_free(cosmics_nb_2[i]);
    }
    cpl_free(cosmics_nb);
    cpl_free(cosmics_nb_2);
    cpl_free(cosmics_nb_2_skysub);
    
    if (tell_corr_flag == 1) {
        cpl_image_delete(telluric_spectrum);
        cpl_image_delete(flux_tell_corr_s2d_blaze);
        cpl_image_delete(error_tell_corr_s2d_blaze);
        cpl_image_delete(qual_tell_corr_s2d_blaze);
        cpl_image_delete(flux_tell_corr_s2d);
        cpl_image_delete(error_tell_corr_s2d);
        cpl_image_delete(qual_tell_corr_s2d);
        cpl_propertylist_delete(telluric_keywords);
    }

    espdr_msg("Cleaning bissector data");
	cpl_free(asym);
	cpl_free(asym_error);
	cpl_free(bispan);
	cpl_free(bispan_error);
	cpl_image_delete(rel_eff);

    espdr_msg("Cleaning S1D products");
	cpl_free(s1d_table_merged);
	cpl_free(drift_results);
    if (s1d_table_merged_skysub != NULL) {
            cpl_table_delete(s1d_table_merged_skysub);
    }
    if (s1d_table_merged_tell_corr != NULL) {
            cpl_table_delete(s1d_table_merged_tell_corr);
    }
    if (s1d_fluxcal != NULL) { cpl_table_delete(s1d_fluxcal); }
    if (s1d_fluxcal_skysub != NULL) { cpl_table_delete(s1d_fluxcal_skysub); }
    if (s1d_fluxcal_tell_corr != NULL) { cpl_table_delete(s1d_fluxcal_tell_corr); }

    if (g_res != NULL) { cpl_free(g_res); }
    if (g_res_skysub != NULL) { cpl_free(g_res_skysub); }
    if (g_res_tell_corr != NULL) { cpl_free(g_res_tell_corr); }
    
	cpl_table_delete(mask_table);
    if (extinction_table != NULL) { cpl_table_delete(extinction_table); }
    if (abs_eff_table != NULL) { cpl_table_delete(abs_eff_table); }
    cpl_table_delete(flux_template_table);

    cpl_propertylist_delete(keywords);
    cpl_propertylist_delete(keywords_skysub);
    cpl_propertylist_delete(OH_corr_QC_keywords);

	for (int i = 0; i < fibres_processed; i++) { cpl_propertylist_delete(keywords_fibre[i]); }

    cpl_free(qc_kws);
	cpl_frameset_delete(used_frames);

	cpl_image_delete(wavelenght_grid_pow_10);
	cpl_image_delete(wavelenght_air_grid_pow_10);

	cpl_free(orders_nb_per_fibre);
	cpl_free(ron);

	if ((strcmp(token,"SKY") == 0) && (strcmp(sky_correction_method, "none") != 0)) {
        if (inst_config->inst_type == VIS) {
            cpl_image_delete(flux_sky_corr);
            cpl_image_delete(err_sky_corr);
            cpl_image_delete(qual_sky_corr);
        }
		cpl_image_delete(flux_sky_sub_s2d_blaze);
		cpl_image_delete(error_sky_sub_s2d_blaze);
		cpl_image_delete(qual_sky_sub_s2d_blaze);
        cpl_image_delete(flux_sky_sub_s2d);
        cpl_image_delete(error_sky_sub_s2d);
        cpl_image_delete(qual_sky_sub_s2d);
	}

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_SCI_RED_delete(SCI_RED_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    cpl_array_delete(RV_table);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for SCI_RED input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input SCI_RED parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_SCI_RED_create(const char* recipe_id,
                                               cpl_parameterlist *list,
                                               espdr_SCI_RED_param *p) {
	cpl_error_code error_got;

	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, 
                 "The parameters list NULL");
    char comment[COMMENT_LENGTH];
	int bkgr_x_min = BKGR_X_MIN;
    int bkgr_x_max = BKGR_X_MAX;
	int bkgr_y_min = BKGR_Y_MIN;
	int bkgr_y_max = BKGR_Y_MAX;
    
	/* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "background_sw",
                                            p->background_sw,
                                            "Background subtraction activation (on/on_using_B/off)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter background_sw to the list");
    
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "sci_bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");
    
    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "sci_bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "wave_cal_source", p->wave_cal_source,
                    "Wavelength calibration source to be used on science fiber \
                                            ('THAR' or 'LFC')");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter wave_cal_source to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "mask_table_id", p->mask_table_id,
                                            "Mask table to be used, defined by it's spectral type Id " \
                                            "('F9', 'G2', 'G8', 'G9', 'K2', 'K6', "
											"'M0', 'M2', 'M3', 'M4' or 'M5'. " \
                                            "In case of default ('XX') the pipeline will read the value "\
                                            "of the FITS keyword OCS OBJ SP TYPE and uses a correspondence "\
                                            "table to assign the closest available mask to the spectral "\
                                            "type given by the FITS keyword or the user");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter mask_table_id to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "rv_center", p->rv_center, -1e4, 1e4,
                                                  "Approximate RV. In case of default (-9999) "\
                                                  "the pipeline will use the value of the FITS keyword OCS OBJ RV");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter rv_center to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "rv_range", p->rv_range, 0.0, 300.0,
                                                  "Range for the RV table.");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter rv_range to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "rv_step", p->rv_step, 0.0, 5.0,
                                                  "Range's step for the RV table.");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter rv_step to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "ksigma_cosmic", p->ksigma_cosmic, -1.0, 20.0,
                                                  "ksigma for removing cosmics on fiber A or SKY, -1.0 - no cosmics removal");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksigma_cosmic to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "bias_res_removal_sw",
                                            p->bias_res_removal_sw,
                                            "Flag indicating to remove or not MB residuals");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bias_res_removal_sw to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "flux_correction_type",
                                            p->flux_correction_type,
                                            "Flux correction: NONE, AUTO or <spectral_type>");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter flux_correction_type to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "drift_correction_sw",
                                            p->drift_correction_sw,
                                            "Drift correction activation (on/off)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter drift_correction_sw to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "drift_method_fp",
                                            p->drift_method,
                                            "Method adopted to compute drift");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter drift_method to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "drift_space",
                                            p->drift_space,
                                            "Space to compute drift (pixel/velocity)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter drift_space to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "drift_ksigma", p->drift_ksigma, -1.0, 100.0,
                                                  "ksigma for computing drift, -1.0 - no ksigma clip");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksigma_drift to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "sky_corr_method",
                                            p->sky_corr_method,
                                            "Method used to correct the sky (pixel-by-pixel/smoothed/oh-corr/none)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter sky_correction_method to the list");
    
    error_got = espdr_parameters_new_int(recipe_id, list,
                                        "sky_sub_sliding_box_size",
                                         p->sky_sub_sliding_box_size,
                                         "Sliding box size in smoothed sky subtraction");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter sky_subtraction_slidebox to the list");
    
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "slit_loss", p->slit_loss, -1.0, 100.0,
                                                  "Slit loss in flux calibration");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter slit_loss to the list");
    
    /* Cosmic Detection Parameters */
    
    /* --espdr.espdr_sci_red.crh_detection_sw */
    error_got = espdr_parameters_new_int(recipe_id, list,
                                         "cosmic_detection_sw",
                                         p->crh_detection_sw,
                                         "LA Cosmic detection activation switch");
    
    /* --espdr.espdr_sci_red.lacosmic.post-filter-x */
    error_got = espdr_parameters_new_int(recipe_id, list,
                                         "lacosmic.post-filter-x",
                                         p->post_filter_x,
                                         "X Size of the post filtering kernel");
    
    /* --espdr.espdr_sci_red.lacosmic.post-filter-y */
    error_got = espdr_parameters_new_int(recipe_id, list,
                                         "lacosmic.post-filter-y",
                                         p->post_filter_y,
                                         "Y Size of the post filtering kernel");
    
    /* --espdr.espdr_sci_red.lacosmic.post-filter-mode */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "lacosmic.post-filter-mode",
                                            p->post_filter_mode,
                                            "Post filtering mode");
    
    /* --espdr.espdr_sci_red.lacosmic.sigma_lim */
    error_got = espdr_parameters_new_double(recipe_id, list,
                                            "lacosmic.sigma_lim",
                                            p->lacosmic_sigma_lim,
                                            "LA Cosmic Poisson fluctuation threshold");
    
    /* --espdr.espdr_sci_red.lacosmic.f_lim */
    error_got = espdr_parameters_new_double(recipe_id, list,
                                            "lacosmic.f_lim",
                                            p->lacosmic_f_lim,
                                            "LA Cosmic minimum contrast");
    
    /* --espdr.espdr_sci_red.lacosmic.max_iter */
    error_got = espdr_parameters_new_int(recipe_id, list,
                                         "lacosmic.max_iter",
                                         p->lacosmic_max_iter,
                                         "LA Cosmic max number of iterations");
    
    /* --espdr.espdr_sci_red.extra_products_sw */
    error_got = espdr_parameters_new_bool(recipe_id, list,
                                          "extra_products_sw",
                                          p->extra_products_sw,
                                          "Set to TRUE to create extra products to inspect LA Cosmic results");
    
    /* Telluric correction parameters */
    
    /* --espdr.espdr_sci_red.telluric_corr_sw */
    error_got = espdr_parameters_new_int(recipe_id, list,
                                         "telluric_corr_sw",
                                         p->telluric_corr_sw,
                                         "Telluric correction activation switch");
    
	return(CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for SCI_RED input parameters
 @param    p	input SCI_RED parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_SCI_RED_delete(espdr_SCI_RED_param* p) {
	
    cpl_free((void *)p->wave_cal_source);
    cpl_free((void *)p->mask_table_id);
    cpl_free((void *)p->extraction_method);
    cpl_free((void *)p->background_sw);
    cpl_free((void *)p->bias_res_removal_sw);
    cpl_free((void *)p->flux_correction_type);
    cpl_free((void *)p->drift_correction_sw);
    cpl_free((void *)p->sky_corr_method);
    cpl_free((void *)p->post_filter_mode);
    cpl_free(p);
    p = NULL;
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the sci_red recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     SCI_RED_param  SCI_RED parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_SCI_RED_get(const char* recipe_id,
                                            cpl_parameterlist* param_list,
                                            espdr_SCI_RED_param *SCI_RED_param) {

	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
	SCI_RED_param->wave_cal_source =
                    espdr_parameters_get_string(recipe_id, param_list,
                                                "wave_cal_source");
	
	SCI_RED_param->ksigma_cosmic =
                    espdr_parameters_get_double(recipe_id, param_list,
                                                "ksigma_cosmic");

	SCI_RED_param->rv_center =
                    espdr_parameters_get_double(recipe_id, param_list,
                                                "rv_center");

	SCI_RED_param->rv_range = 
                    espdr_parameters_get_double(recipe_id, param_list,
                                                "rv_range");

	SCI_RED_param->rv_step = 
                    espdr_parameters_get_double(recipe_id, param_list,
                                                "rv_step");

    SCI_RED_param->slit_loss =
                    espdr_parameters_get_double(recipe_id, param_list,
                                                "slit_loss");
    
	SCI_RED_param->mask_table_id =
                    espdr_parameters_get_string(recipe_id, param_list,
                                                "mask_table_id");

    SCI_RED_param->extraction_method =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "extraction_method");

    SCI_RED_param->background_sw =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "background_sw");

    SCI_RED_param->flux_correction_type =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "flux_correction_type");
    
    SCI_RED_param->drift_correction_sw =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "drift_correction_sw");

    SCI_RED_param->bias_res_removal_sw =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "bias_res_removal_sw");
    
    SCI_RED_param->sky_corr_method =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "sky_corr_method");
    
    SCI_RED_param->sky_sub_sliding_box_size =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "sky_sub_sliding_box_size");
    
    SCI_RED_param->crh_detection_sw =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "cosmic_detection_sw");
    
    SCI_RED_param->post_filter_x =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "lacosmic.post-filter-x");
    
    SCI_RED_param->post_filter_y =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "lacosmic.post-filter-y");
    
    SCI_RED_param->post_filter_mode =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "lacosmic.post-filter-mode");
    
    SCI_RED_param->lacosmic_sigma_lim =
                    espdr_parameters_get_double(recipe_id,
                                                param_list,
                                                "lacosmic.sigma_lim");
    
    SCI_RED_param->lacosmic_f_lim =
                    espdr_parameters_get_double(recipe_id,
                                                param_list,
                                                "lacosmic.f_lim");
    
    SCI_RED_param->lacosmic_max_iter =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "lacosmic.max_iter");
    
    SCI_RED_param->extra_products_sw =
                    espdr_parameters_get_bool(recipe_id,
                                              param_list,
                                              "extra_products_sw");
    
    SCI_RED_param->telluric_corr_sw =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "telluric_corr_sw");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the SCI_RED parameters
 @param     SCI_RED_param  SCI_RED parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_SCI_RED_print(
				espdr_SCI_RED_param *SCI_RED_param) {
	
	espdr_msg("\tSCI_RED parameters (defaults or user set):");

	espdr_msg("\t\tSCI_RED wave_cal_source = %s",
              SCI_RED_param->wave_cal_source);

	espdr_msg("\t\tSCI_RED ksigma_cosmic = %.2f",
              SCI_RED_param->ksigma_cosmic);

	espdr_msg("\t\tSCI_RED rv_center = %.4f",
              SCI_RED_param->rv_center);

	espdr_msg("\t\tSCI_RED rv_range = %.2f",
              SCI_RED_param->rv_range);

	espdr_msg("\t\tSCI_RED rv_step = %.4f",
              SCI_RED_param->rv_step);

	espdr_msg("\t\tSCI_RED mask_table_id = %s",
              SCI_RED_param->mask_table_id);

    espdr_msg("\t\tSCI_RED extraction method = %s",
              SCI_RED_param->extraction_method);

    espdr_msg("\t\tSCI_RED background subtraction = %s",
              SCI_RED_param->background_sw);

    espdr_msg("\t\tSCI_RED flux correction type = %s",
              SCI_RED_param->flux_correction_type);
    
    espdr_msg("\t\tSCI_RED drift correction activation = %s",
              SCI_RED_param->drift_correction_sw);

    espdr_msg("\t\tSCI_RED bias_res_removal_sw = %s",
              SCI_RED_param->bias_res_removal_sw);
    
    espdr_msg("\t\tSCI_RED sky correction method = %s",
              SCI_RED_param->sky_corr_method);
    
    espdr_msg("\t\tSCI_RED sky subtraction slide box size = %d",
              SCI_RED_param->sky_sub_sliding_box_size);
    
    espdr_msg("\t\tSCI_RED slit loss in flux calibration = %.2f",
              SCI_RED_param->slit_loss);
    
    espdr_msg("\t\tSCI_RED LaCosmic activation switch (1 = ON, 0 = OFF) = %d",
              SCI_RED_param->crh_detection_sw);
    
    espdr_msg("\t\tSCI_RED LaCosmic x size of the post filtering kernel = %d",
              SCI_RED_param->post_filter_x);
    
    espdr_msg("\t\tSCI_RED LaCosmic y size of the post filtering kernel = %d",
              SCI_RED_param->post_filter_y);
    
    espdr_msg("\t\tSCI_RED LaCosmic post filtering mode = %s (closing or dilation)",
              SCI_RED_param->post_filter_mode);
    
    espdr_msg("\t\tSCI_RED LaCosmic Poisson fluctuation threshold = %.2f",
              SCI_RED_param->lacosmic_sigma_lim);
    
    espdr_msg("\t\tSCI_RED LaCosmic minimum contrast = %.2f",
              SCI_RED_param->lacosmic_f_lim);
    
    espdr_msg("\t\tSCI_RED LaCosmic maximum number of iterations = %d",
              SCI_RED_param->lacosmic_max_iter);
    
    espdr_msg("\t\tSCI_RED Extra products switch (1 = save, 0 = don't save) = %d",
              SCI_RED_param->extra_products_sw);
    
    espdr_msg("\t\tSCI_RED telluric correction switch (1 = ON, 0 = OFF) = %d",
              SCI_RED_param->telluric_corr_sw);
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init SCI_RED_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_SCI_RED_param * espdr_SCI_RED_param_init(const char *recipe_id, 
					 cpl_parameterlist *parameters) {
    
    espdr_SCI_RED_param *SCI_RED_param =
    (espdr_SCI_RED_param *)cpl_malloc(sizeof(espdr_SCI_RED_param));
    
    /* Read the sci_red parameters */
	espdr_parameters_SCI_RED_get(recipe_id, parameters, SCI_RED_param);
	espdr_parameters_SCI_RED_print(SCI_RED_param);
    if(cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return SCI_RED_param;
    }

}


/*----------------------------------------------------------------------------*/
/**
 @brief    Read telescope KWs
 
 @param
 @param
 
 @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_spectral_type(espdr_inst_config *inst_config,
                                       cpl_propertylist *keywords,
                                       char *spec_type) {
    
    const char *spt_kw;
    
    if (!cpl_propertylist_has(keywords, inst_config->spec_type_kw)) {
        espdr_msg_error("No %s KW found, exiting", inst_config->spec_type_kw);
        return (CPL_ERROR_NULL_INPUT);
    }
    
    /* Parsing in case of HARPS, the spectral type from spec_type_kw */
    /* Getting the keyword containing the spectral type */
    /* In case of HARPS, the spectral type comes with DPR.TYPE keyword*/
    
    spt_kw = cpl_propertylist_get_string(keywords,inst_config->spec_type_kw);
    if (strcmp(inst_config->instrument, "HARPS") == 0) {
        char *separator;
        char to_split[64];
        strcpy(to_split, spt_kw);
        separator = strtok(to_split,",");
        while (separator != NULL) {
            strcpy(spec_type, separator);
            separator = strtok(NULL,",");
        }
    } else {
        strcpy(spec_type, spt_kw);
    }
    
    /* Defining the default spectral type in case of NULL or empty */
    if(strcmp(spec_type,"NULL") == 0 || /* spec_type NULL */
       strcmp(spec_type,"")     == 0 /* spec_type blank */
       ) {
        
        espdr_msg(ANSI_COLOR_RED"Warning: spectral type: %s, not valid."ANSI_COLOR_RESET,
                  spec_type);
        strcpy(spec_type, inst_config->default_spec_type);
        espdr_msg(ANSI_COLOR_RED"Setting default spectral type"ANSI_COLOR_RESET);
        espdr_msg(ANSI_COLOR_RED"SPECTRAL TYPE = %s"ANSI_COLOR_RESET,spec_type);
    } else {
        espdr_msg("SPECTRAL TYPE = %s", spec_type);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    get recipe user parameters
 
 @param
 @param
 
 @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_science_params(espdr_inst_config *inst_config,
                                        cpl_propertylist *keywords,
                                        cpl_parameterlist *parameters,
                                        espdr_SCI_RED_param *SCI_RED_param,
                                        char *fibre_b,
                                        double *log_lambda_ini,
                                        double *log_lambda_end,
                                        double *delta_log_lambda,
                                        double *mask_width,
                                        char *mask_table_id,
                                        int *mask_changed,
                                        double *rv_center,
                                        double *rv_range,
                                        double *rv_step,
                                        double *ksigma_cosmic,
                                        int *remove_bias_res_flag,
                                        char *wave_cal_source,
                                        char *extraction_method,
                                        char *background_sw,
                                        char *bias_res_removal_sw,
                                        char *flux_correction_type,
                                        char *drift_correction_sw,
                                        int *drift_set,
                                        char *sky_correction_method,
                                        int *sky_subtraction_slidebox,
                                        double *slit_loss,
                                        int *lacosmic_flag,
                                        int *tell_corr_flag) {
    
    if (!cpl_propertylist_has(keywords, inst_config->rv_center_kw)) {
        espdr_msg_error("No %s kw found, exiting", inst_config->rv_center_kw);
        return (CPL_ERROR_NULL_INPUT);
    }
    *rv_center = cpl_propertylist_get_double(keywords, inst_config->rv_center_kw);
    espdr_msg("rv_center = %g", *rv_center);
    
    sprintf(wave_cal_source,"%s","THAR");
    sprintf(extraction_method,"%s","horne");
    sprintf(background_sw,"%s","on");
    sprintf(bias_res_removal_sw,"%s","on");
    sprintf(flux_correction_type,"%s","AUTO");
    sprintf(drift_correction_sw,"%s","on");
    sprintf(sky_correction_method,"%s",inst_config->sci_sky_corr_method);
    
    cpl_parameter *par_wave_cal_source =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.wave_cal_source");
    int wave_cal_source_set = cpl_parameter_get_default_flag(par_wave_cal_source);
    if (wave_cal_source_set == 1) {
        sprintf(wave_cal_source, "%s", SCI_RED_param->wave_cal_source);
        cpl_parameter_set_string(par_wave_cal_source, SCI_RED_param->wave_cal_source);
    }
    
    cpl_parameter *par_rv_center =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.rv_center");
    int rv_center_set = cpl_parameter_get_default_flag(par_rv_center);
    int rv_center_has_changed = espdr_param_has_changed(par_rv_center);
    espdr_msg("rv_center_has_changed = %d", rv_center_has_changed);
    if ((rv_center_set == 1) && (rv_center_has_changed == 1))    {
        *rv_center = SCI_RED_param->rv_center;
    }
    cpl_parameter_set_double(par_rv_center, *rv_center);
    
    cpl_parameter *par_rv_range =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.rv_range");
    int rv_range_set = cpl_parameter_get_default_flag(par_rv_range);
    if (rv_range_set == 1) {
        *rv_range = SCI_RED_param->rv_range;
    }
    cpl_parameter_set_double(par_rv_range, *rv_range);
    
    cpl_parameter *par_mask_table_id =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.mask_table_id");
    int mask_table_id_set = cpl_parameter_get_default_flag(par_mask_table_id);
    int mask_table_id_has_changed = espdr_param_has_changed(par_mask_table_id);
    if ((mask_table_id_set == 1) && (mask_table_id_has_changed == 1)) {
        sprintf(mask_table_id, "%s", SCI_RED_param->mask_table_id);
        *mask_changed = 1;
    }
    cpl_parameter_set_string(par_mask_table_id, mask_table_id);
    //espdr_msg("User given mask_table_id = %s", SCI_RED_param->mask_table_id);
    
    cpl_parameter *par_extraction_method =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.extraction_method");
    int extraction_method_set = cpl_parameter_get_default_flag(par_extraction_method);
    if (extraction_method_set == 1) {
        sprintf(extraction_method, "%s", SCI_RED_param->extraction_method);
    }
    cpl_parameter_set_string(par_extraction_method, extraction_method);
    
    cpl_parameter *par_background_sw =
                    cpl_parameterlist_find(parameters,"espdr.espdr_sci_red.background_sw");
    int background_sw_set = cpl_parameter_get_default_flag(par_background_sw);
    int background_sw_has_changed = espdr_param_has_changed(par_background_sw);
    if ((background_sw_set == 1) && (background_sw_has_changed == 1)) {
        sprintf(background_sw, "%s", SCI_RED_param->background_sw);
    }
    /* By default ON for any case, just OFF in case of SKY, unless overrided
     * by command line by -background_sw=on */
    if ((strcmp(fibre_b, "SKY") == 0) && (background_sw_set == 0)) {
        sprintf(background_sw, "%s", "off");
    }
    cpl_parameter_set_string(par_background_sw, background_sw);
    
    cpl_parameter *par_bkgr_grid_size_x =
                    cpl_parameterlist_find(parameters,"espdr.espdr_sci_red.sci_bkgr_grid_size_x");
    cpl_parameter_set_int(par_bkgr_grid_size_x, inst_config->sci_bkgr_grid_size_x);
    cpl_parameter *par_bkgr_grid_size_y =
                    cpl_parameterlist_find(parameters,"espdr.espdr_sci_red.sci_bkgr_grid_size_y");
    cpl_parameter_set_int(par_bkgr_grid_size_y, inst_config->sci_bkgr_grid_size_y);
    
    cpl_parameter *par_bias_res_removal_sw =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.bias_res_removal_sw");
    int bias_res_removal_sw_set = cpl_parameter_get_default_flag(par_bias_res_removal_sw);
    if (bias_res_removal_sw_set == 1) {
        sprintf(bias_res_removal_sw, "%s", SCI_RED_param->bias_res_removal_sw);
        if (strcmp(bias_res_removal_sw, "off") == 0) {
            remove_bias_res_flag = 0;
        }
    }
    cpl_parameter_set_string(par_bias_res_removal_sw, bias_res_removal_sw);
    if (inst_config->raw_bias_limit_nb == 0) {
        *remove_bias_res_flag = 0;
    }
    if (*remove_bias_res_flag == 1) {
        cpl_parameter_set_string(par_bias_res_removal_sw, "on");
    } else {
        cpl_parameter_set_string(par_bias_res_removal_sw, "off");
    }
    
    cpl_parameter *par_flux_correction_type =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.flux_correction_type");
    int flux_correction_type_set = cpl_parameter_get_default_flag(par_flux_correction_type);
    if (flux_correction_type_set == 1) {
        sprintf(flux_correction_type, "%s", SCI_RED_param->flux_correction_type);
    }
    cpl_parameter_set_string(par_flux_correction_type, flux_correction_type);
    
    cpl_parameter *par_drift_correction_sw =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.drift_correction_sw");
    int drift_correction_sw_set = cpl_parameter_get_default_flag(par_drift_correction_sw);
    if (drift_correction_sw_set == 1) {
        sprintf(drift_correction_sw, "%s", SCI_RED_param->drift_correction_sw);
        *drift_set = 1;
    }
    cpl_parameter_set_string(par_drift_correction_sw, drift_correction_sw);
    
    cpl_parameter *par_slit_loss =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.slit_loss");
    int slit_loss_set = cpl_parameter_get_default_flag(par_slit_loss);
    int slit_loss_has_changed = espdr_param_has_changed(par_slit_loss);
    if ((slit_loss_set == 1) && (slit_loss_has_changed == 1)) {
        *slit_loss = SCI_RED_param->slit_loss;
    }
    cpl_parameter_set_double(par_slit_loss, *slit_loss);
    
    cpl_parameter *par_sky_corr_method =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.sky_corr_method");
    int sky_corr_method_set = cpl_parameter_get_default_flag(par_sky_corr_method);
    if (sky_corr_method_set == 1) {
        sprintf(sky_correction_method, "%s", SCI_RED_param->sky_corr_method);
    }
    cpl_parameter_set_string(par_sky_corr_method, sky_correction_method);
    
    cpl_parameter *par_crh_detection_sw =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.cosmic_detection_sw");
    int crh_detection_sw_set = cpl_parameter_get_default_flag(par_crh_detection_sw);
    int crh_detection_sw_has_changed = espdr_param_has_changed(par_crh_detection_sw);
    if ((crh_detection_sw_set == 1) && (crh_detection_sw_has_changed == 1)) {
        *lacosmic_flag = SCI_RED_param->crh_detection_sw;
    } else {
        if ((strcmp(fibre_b, "SKY") == 0) || (strcmp(fibre_b, "DARK") == 0)) {
            *lacosmic_flag = inst_config->lacosmic_sky_sw;
        } else {
            *lacosmic_flag = inst_config->lacosmic_calib_sw;
        }
    }
    cpl_parameter_set_int(par_crh_detection_sw, *lacosmic_flag);
    
    cpl_parameter *par_telluric_corr_sw =
                    cpl_parameterlist_find(parameters, "espdr.espdr_sci_red.telluric_corr_sw");
    int telluric_corr_sw_set = cpl_parameter_get_default_flag(par_telluric_corr_sw);
    int telluric_corr_sw_has_changed = espdr_param_has_changed(par_telluric_corr_sw);
    if ((telluric_corr_sw_set == 1) && (telluric_corr_sw_has_changed == 1)) {
        *tell_corr_flag = SCI_RED_param->telluric_corr_sw;
    }
    cpl_parameter_set_int(par_telluric_corr_sw, *tell_corr_flag);
    
    espdr_msg(ANSI_COLOR_GREEN"rv_step = %.2f"ANSI_COLOR_RESET, *rv_step);
    espdr_msg(ANSI_COLOR_GREEN"rv_center = %.2f"ANSI_COLOR_RESET, *rv_center);
    espdr_msg(ANSI_COLOR_GREEN"rv_range = %.2f"ANSI_COLOR_RESET, *rv_range);
    espdr_msg(ANSI_COLOR_GREEN"wave_cal_source = %s"ANSI_COLOR_RESET,
              wave_cal_source);
    espdr_msg(ANSI_COLOR_GREEN"ksigma_cosmic = %.2f"ANSI_COLOR_RESET,
              *ksigma_cosmic);
    espdr_msg(ANSI_COLOR_GREEN"mask_table_id = %s"ANSI_COLOR_RESET,
              mask_table_id);
    espdr_msg(ANSI_COLOR_GREEN"extraction_method = %s"ANSI_COLOR_RESET,
              extraction_method);
    espdr_msg(ANSI_COLOR_GREEN"background_sw = %s"ANSI_COLOR_RESET,
              background_sw);
    espdr_msg(ANSI_COLOR_GREEN"flux_correction_type = %s"ANSI_COLOR_RESET,
              flux_correction_type);
    espdr_msg(ANSI_COLOR_GREEN"drift_correction_sw = %s"ANSI_COLOR_RESET,
              drift_correction_sw);
    espdr_msg(ANSI_COLOR_GREEN"bias_res_removal_sw = %s"ANSI_COLOR_RESET,
              bias_res_removal_sw);
    espdr_msg(ANSI_COLOR_GREEN"sky_correction_method = %s"ANSI_COLOR_RESET,
              sky_correction_method);
    espdr_msg(ANSI_COLOR_GREEN"sky_subtraction_slidebox = %d"ANSI_COLOR_RESET,
              *sky_subtraction_slidebox);
    espdr_msg(ANSI_COLOR_GREEN"slit_loss = %.2f"ANSI_COLOR_RESET, *slit_loss);
    espdr_msg(ANSI_COLOR_GREEN"cosmic_detection_sw = %d"ANSI_COLOR_RESET,
              *lacosmic_flag);
    espdr_msg(ANSI_COLOR_GREEN"telluric_corr_sw = %d"ANSI_COLOR_RESET,
              *tell_corr_flag);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_science_get_static_tables(cpl_frameset *frameset,
                                               cpl_frameset *used_frames,
                                               cpl_propertylist *keywords,
                                               espdr_inst_config *inst_config,
                                               cpl_image **rel_eff,
                                               cpl_table **abs_eff_table,
                                               cpl_table **extinction_table,
                                               int mask_changed,
                                               const char *mask_table_id,
                                               const char **mask_id,
                                               cpl_table **mask_table,
                                               char *flux_correction_type,
                                               char *spec_type,
                                               cpl_table **flux_template_table) {
    
    const char *table_filename = NULL;
    
    /* Extract rel_eff frame */
    cpl_frame *rel_eff_frame = espdr_frame_find(frameset, ESPDR_PRO_CATG_REL_EFF);
    if (rel_eff_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(rel_eff_frame));
        *rel_eff = cpl_image_load(cpl_frame_get_filename(rel_eff_frame), CPL_TYPE_DOUBLE, 0, 1);
    } else {
        espdr_msg("No relative efficiency frame");
    }
    
    /* Extract absolute efficiency table file frame */
    cpl_frame *abs_eff_frame = espdr_frame_find(frameset, ESPDR_PRO_CATG_ABS_EFF);
    if (abs_eff_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(abs_eff_frame));
    }
    if (abs_eff_frame != NULL) {
        table_filename = cpl_frame_get_filename(abs_eff_frame);
        espdr_msg("Absolute efficiency table: %s", table_filename);
        *abs_eff_table = cpl_table_load(table_filename, 1, 0);
        espdr_msg("Absolute efficiency table rows number = %lld", cpl_table_get_nrow(*abs_eff_table));
        espdr_msg("Absolute efficiency table cols number = %lld", cpl_table_get_ncol(*abs_eff_table));
    } else {
        espdr_msg(ANSI_COLOR_RED"\t\tWarning : NO Absolute efficiency table provided."ANSI_COLOR_RESET);
        *abs_eff_table = NULL;
    }
    
    /* Extract extinction table file frame */
    cpl_frame *ext_table_frame = espdr_frame_find(frameset, ESPDR_EXTINCTION_TABLE);
    if (ext_table_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(ext_table_frame));
        table_filename = cpl_frame_get_filename(ext_table_frame);
        espdr_msg("Extinction table: %s", table_filename);
        *extinction_table = cpl_table_load(table_filename, 1, 0);
        espdr_msg("Extinction table rows number = %lld", cpl_table_get_nrow(*extinction_table));
        espdr_msg("Extinction table cols number = %lld", cpl_table_get_ncol(*extinction_table));
    } else {
        espdr_msg_error("No Extinction table provided, exiting");
        return (CPL_ERROR_NULL_INPUT);
    }
    
    /* Extract mask lut file frame */
    cpl_frame *lut_frame = espdr_frame_find(frameset, ESPDR_MASK_LUT_TABLE);
    if (lut_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(lut_frame));
    }
    cpl_table *mask_lut_table = NULL;
    if (lut_frame != NULL) {
        table_filename = cpl_frame_get_filename(lut_frame);
        espdr_msg("Mask LUT : %s", table_filename);
        mask_lut_table = cpl_table_load(table_filename, 1, 0);
        espdr_msg("Mask LUT rows number = %lld", cpl_table_get_nrow(mask_lut_table));
        espdr_msg("Mask LUT cols number = %lld", cpl_table_get_ncol(mask_lut_table));
    }
    
    /* Get the right mask ID */
    if (mask_changed == 0) {
        *mask_id = espdr_get_mask_name(mask_table_id, mask_lut_table);
        if (strcmp(*mask_id, "NOT_FOUND") == 0) {
            espdr_msg("Spectral type not found. Default mask is %s", inst_config->default_spec_type);
            *mask_id = espdr_get_default_spec_type(inst_config);
        }
    } else {
        *mask_id = mask_table_id;
    }
    espdr_msg( "Mask ID to be selected: %s", *mask_id);
    
    /* Extract mask file frame */
    cpl_frameset *mask_frames = cpl_frameset_new();
    espdr_frame_extract_by_tag(frameset, ESPDR_PRO_CATG_MASK, mask_frames);
    int mask_table_nb = cpl_frameset_get_size(mask_frames);
    
    int imask = -1, idefault = 0;
    const char *default_mask_id = espdr_get_default_spec_type(inst_config);
    const char *mask_filename;
    for (int i = 0 ; i < mask_table_nb ; i++) {
        cpl_frame *mask_frame = cpl_frameset_get_position(mask_frames,i);
        mask_filename = cpl_frame_get_filename(mask_frame);
        if (strstr(mask_filename, *mask_id) != NULL) { imask = i; }
        if (strstr(mask_filename, default_mask_id) != NULL) { idefault = i; }
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(mask_frame));
    }
    
    if (imask == -1) {
        espdr_msg("Mask table %s not found, using default mask", *mask_id);
        imask = idefault;
        *mask_id = default_mask_id;
    }
    
    mask_filename = cpl_frame_get_filename(cpl_frameset_get_position(mask_frames, imask));
    espdr_msg("Mask table selected : %s", mask_filename);
    
    *mask_table = cpl_table_load(mask_filename, 1, 0);
    espdr_msg("Mask table rows number = %lld", cpl_table_get_nrow(*mask_table));
    espdr_msg("Mask table cols number = %lld", cpl_table_get_ncol(*mask_table));
    
    cpl_frameset_delete(mask_frames);
    
    /* Extract flux template file frame */
    //char flux_corr_type[10];
    //sprintf(flux_corr_type, "NONE");
    if ((strcmp(flux_correction_type, "NONE") != 0) && (strcmp(flux_correction_type, "none") != 0)) {
        cpl_frame *flux_template_frame = espdr_frame_find(frameset, ESPDR_FLUX_TEMPLATE);
        espdr_ensure(flux_template_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "No input FLUX TEMPLATE frame");
        
        table_filename = cpl_frame_get_filename(flux_template_frame);
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(flux_template_frame));
        espdr_msg("Flux template : %s", table_filename);
        *flux_template_table = cpl_table_load(table_filename, 1, 0);
        espdr_msg("Flux template rows number =%lld", cpl_table_get_nrow(*flux_template_table));
        espdr_msg("Flux template cols number =%lld", cpl_table_get_ncol(*flux_template_table));
        
        if ((strcmp(flux_correction_type, "AUTO") == 0) ||
            (strcmp(flux_correction_type, "auto") == 0)) {
            strncpy(flux_correction_type, spec_type, 2);
            flux_correction_type[2] = '\0';
        }
        
        //if ((strcmp(flux_correction_type, "AUTO") == 0) ||
        //    (strcmp(flux_correction_type, "auto") == 0)) {
        //    memset(flux_corr_type, '\0', sizeof(flux_corr_type));
        //    strncpy(flux_corr_type, spec_type, 2);
        //} else {
        //    sprintf(flux_corr_type, "%s", flux_correction_type);
        //}
        
    } else {
        espdr_msg("User set no flux correction (flux_correction_type parameter is NONE)");
        *flux_template_table = NULL;
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Get BERV parameters
 
 @param
 @param
 
 @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_berv_params(espdr_inst_config *inst_config,
                                     espdr_qc_keywords *qc_kws,
                                     cpl_propertylist *keywords,
                                     cpl_frame *raw_frame,
                                     double *exp_time,
                                     double *ra_ft,
                                     double *dec_ft,
                                     double *pm_ra,
                                     double *pm_dec,
                                     int *year,
                                     int *month,
                                     int *day,
                                     int *hour,
                                     int *minutes,
                                     double *seconds,
                                     double *time_to_mid_exposure,
                                     double *airmass) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    char* tel_alpha_kw;
    char* tel_delta_kw;
    char* tel_pma_kw;
    char* tel_pmd_kw;
    char* airm_start_kw;
    char* airm_end_kw;
    
    const char* telescope;
    char tel_sel[5];
    
    /* Setting keywords related according to the telescope selected. */
    
    if (strcmp(inst_config->instrument, "ESPRESSO") == 0) {
        
        if (cpl_propertylist_has(keywords, "TELESCOP")) {
            
            telescope = cpl_propertylist_get_string(keywords,"TELESCOP");
            sprintf(tel_sel,"TEL%c",telescope[9]);
            
            tel_alpha_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->tel_alpha_kw);
            tel_delta_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->tel_delta_kw);
            tel_pma_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->tel_pma_kw);
            tel_pmd_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->tel_pmd_kw);
            airm_start_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->airm_start_kw);
            airm_end_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->airm_end_kw);
            
        } else {
            espdr_msg_error("No TELESCOPE KW found, exiting");
            return (CPL_ERROR_NULL_INPUT);
        }
        
    } else {
        if (strlen(inst_config->tel_alpha_kw) < 4) { // alpha = 'RA'
            tel_alpha_kw = cpl_sprintf("%s", inst_config->tel_alpha_kw);
        } else {
            tel_alpha_kw = cpl_sprintf("%s %s", inst_config->prefix,
                                       inst_config->tel_alpha_kw);
        }
        if (strlen(inst_config->tel_delta_kw) < 5) { // delta = 'DEC'
            tel_delta_kw = cpl_sprintf("%s", inst_config->tel_delta_kw);
        } else {
            tel_delta_kw = cpl_sprintf("%s %s", inst_config->prefix,
                                       inst_config->tel_delta_kw);
        }
    	tel_pma_kw = cpl_sprintf("%s %s", inst_config->prefix,
                inst_config->tel_pma_kw);
    	tel_pmd_kw = cpl_sprintf("%s %s", inst_config->prefix,
                inst_config->tel_pmd_kw);
    	airm_start_kw = cpl_sprintf("%s", inst_config->airm_start_kw);
    	airm_end_kw = cpl_sprintf("%s", inst_config->airm_end_kw);
    }
    
    espdr_msg("tel selected = -%s-",tel_sel);
    
    espdr_msg("-%s-", tel_alpha_kw);
    espdr_msg("-%s-", tel_delta_kw);
    espdr_msg("-%s-", tel_pma_kw);
    espdr_msg("-%s-", tel_pmd_kw);
    espdr_msg("-%s-", airm_start_kw);
    espdr_msg("-%s-", airm_end_kw);
    
    double airm_end = cpl_propertylist_get_double(keywords,airm_end_kw);
    double airm_start = cpl_propertylist_get_double(keywords,airm_start_kw);
    *airmass = (airm_end + airm_start)/2.0;
    espdr_msg("airmass = %f", *airmass);

    *exp_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
    *pm_ra = cpl_propertylist_get_double(keywords, tel_pma_kw);
    *pm_dec = cpl_propertylist_get_double(keywords, tel_pmd_kw);
    const char *date_obs = cpl_propertylist_get_string(keywords, inst_config->tel_date_obs_kw);
    
    double tmmean = 0.5;
    /* Extract the exposure meter data from the raw frame, for now only for ESPRESSO */
    if (strcmp(inst_config->instrument, "ESPRESSO") == 0) {
        
        cpl_table *exp_meter = cpl_table_load(cpl_frame_get_filename(raw_frame), 3, 0);
        if (exp_meter == NULL) {
            espdr_msg_warning("No exposure meter extension, returning tmmean = 0.5");
            cpl_error_reset();
            tmmean = 0.5;
        } else {
            double *flux_1 = cpl_table_get_data_double(exp_meter, "objCh1");
            double *flux_2 = cpl_table_get_data_double(exp_meter, "objCh2");
            double *flux_3 = cpl_table_get_data_double(exp_meter, "objCh3");
            double *time_step = cpl_table_get_data_double(exp_meter, "Time");
            if ((flux_1 == NULL) || (flux_2 == NULL) || (flux_3 == NULL) || (time_step == NULL)) {
                espdr_msg_warning("No exposure meter data, returning tmmean = 0.5");
                tmmean = 0.5;
            } else {
                int rows_nb = cpl_table_get_nrow(exp_meter);
                int negatif_times_nb = 0;
                double upper_sum = 0.0, lower_sum = 0.0;
                for (int i = 0; i < rows_nb; i++) {
                    if (time_step[i] > 0.0) {
                        double sum_flux = flux_1[i] + flux_2[i] + flux_3[i];
                        upper_sum += sum_flux * time_step[i];
                        lower_sum += sum_flux;
                    } else {
                        negatif_times_nb++;
                    }
                }
                if ((negatif_times_nb > 1) || (upper_sum == 0.0) || (lower_sum == 0.0)) {
                    tmmean = 0.5;
                    espdr_msg_warning("Too many negatif time stamps in the exposure meter table, setting the TMMEAN to 0.5");
                } else {
                    double texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
                    tmmean = upper_sum / lower_sum / texp;
                    espdr_msg("TMMEAN = %f (computed from the exposure meter table)", tmmean);
                }
            }
        }
        cpl_table_delete(exp_meter);
        
    } else { // All other instruments, except NIRPS, get tmmean from the FITS header KW
        
        if (strcmp(inst_config->instrument, "NIRPS") != 0) {
            
            if (cpl_propertylist_has(keywords, inst_config->tmmean_kw)) {
                tmmean = cpl_propertylist_get_double(keywords, inst_config->tmmean_kw);
            }
            /* tmmean protection against unphysical values */
            if ( (tmmean < 0.1) || (tmmean > 0.9) ) {
                espdr_msg(ANSI_COLOR_RED"\t\tWarning : tmmean corrected to 0.5"ANSI_COLOR_RESET);
                tmmean = 0.5;
            }
            espdr_msg("TMMEAN = %f (taken from the FITS header, KW: %s)", tmmean, inst_config->tmmean_kw);
        } else { // NIRPS tmmean = 0.5
            espdr_msg("TMMEAN is fixed: 0.5");
        }
    }
    
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_tmmean_used, tmmean,
                                        "TMMEAN used in BERV", &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Adding TMMEAN failed: %s",
                 cpl_error_get_message_default(my_error));
    
    const char *ra_str, *dec_str;
    double ra_dbl, dec_dbl;
    
    // Reading ALPHA KW from double or string depending on the instrument
    cpl_type kw_type = cpl_propertylist_get_type(keywords, tel_alpha_kw);
    if (kw_type == CPL_TYPE_DOUBLE) {
        ra_dbl = cpl_propertylist_get_double(keywords,tel_alpha_kw);
        if (strcmp(inst_config->instrument, "CORALIE") == 0) {
            *ra_ft = ra_dbl/15.0;
        } else {
            *ra_ft = convert_HHMMSS_to_HH(ra_dbl);
        }
    } else {
        if (kw_type == CPL_TYPE_STRING) {
            ra_str = cpl_propertylist_get_string(keywords,tel_alpha_kw);
            *ra_ft = convert_coord_to_dbl(ra_str);
        } else {
            espdr_msg_warning("Unknown type of the telescope alpha coordinate: %s",
                              cpl_type_get_name(kw_type));
            return(CPL_ERROR_TYPE_MISMATCH);
        }
    }
    //espdr_msg("Converted RA: %f, orig dbl RA: %f, orig str RA: %s", *ra_ft, ra_dbl, ra_str);
    
    // Reading DELTA KW from double or string depending on the instrument
    kw_type = cpl_propertylist_get_type(keywords, tel_delta_kw);
    if (kw_type == CPL_TYPE_DOUBLE) {
        dec_dbl = cpl_propertylist_get_double(keywords,tel_delta_kw);
        if (strcmp(inst_config->instrument, "CORALIE") == 0) {
            *dec_ft = dec_dbl;
        } else {
            *dec_ft = convert_DDMMSS_to_DD(dec_dbl);
        }
    } else {
        if (kw_type == CPL_TYPE_STRING) {
            dec_str = cpl_propertylist_get_string(keywords,tel_delta_kw);
            *dec_ft = convert_coord_to_dbl(dec_str);
        } else {
            espdr_msg_warning("Unknown type of the telescope delta coordinate: %s",
                              cpl_type_get_name(kw_type));
            return(CPL_ERROR_TYPE_MISMATCH);
        }
    }
    //espdr_msg("Converted DEC: %f, orig dbl DEC: %f, orig str DEC: %s", *dec_ft, dec_dbl, dec_str);
    
    *time_to_mid_exposure = tmmean * (*exp_time);
    
    *year    = atoi(strtok ((char *)date_obs, "-:T"));
    *month   = atoi(strtok (NULL, "-:T"));
    *day     = atoi(strtok (NULL, "-:T"));
    *hour    = atoi(strtok (NULL, "-:T"));
    *minutes = atoi(strtok (NULL, "-:T"));
    *seconds = atof(strtok (NULL, "-:T"));
    
    cpl_free(tel_alpha_kw);
    cpl_free(tel_delta_kw);
    cpl_free(tel_pma_kw);
    cpl_free(tel_pmd_kw);
    cpl_free(airm_start_kw);
    cpl_free(airm_end_kw);
    //cpl_free(tel_sel);

    return cpl_error_get_code();
}
                                        

/*----------------------------------------------------------------------------*/
/**
 @brief    Get BERV parameters
 
 @param
 @param
 
 @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_seeing(espdr_inst_config *inst_config,
                                cpl_propertylist *keywords,
                                double *seeing,
                                int *seeing_kw_qc) {
    
    char *seeing_kw = (char *)cpl_malloc(KEYWORD_LENGTH * sizeof(char));
    sprintf(seeing_kw, "%s %s",
            inst_config->seeing_kw_first_part,
            inst_config->seeing_kw_last_part);
    //espdr_msg("Seeing KW (no TEL): %s", seeing_kw);
    if (cpl_propertylist_has(keywords, seeing_kw)) {
        *seeing = cpl_propertylist_get_double(keywords, seeing_kw);
    } else {
        sprintf(seeing_kw, "%s1 %s",
                inst_config->seeing_kw_first_part,
                inst_config->seeing_kw_last_part);
        //espdr_msg("Seeing KW (TEL1): %s", seeing_kw);
        if (cpl_propertylist_has(keywords, seeing_kw)) {
            *seeing = cpl_propertylist_get_double(keywords, seeing_kw);
        } else  {
            sprintf(seeing_kw, "%s2 %s",
                    inst_config->seeing_kw_first_part,
                    inst_config->seeing_kw_last_part);
            //espdr_msg("Seeing KW (TEL2): %s", seeing_kw);
            if (cpl_propertylist_has(keywords, seeing_kw)) {
                *seeing = cpl_propertylist_get_double(keywords, seeing_kw);
            } else {
                sprintf(seeing_kw, "%s3 %s",
                        inst_config->seeing_kw_first_part,
                        inst_config->seeing_kw_last_part);
                //espdr_msg("Seeing KW (TEL3): %s", seeing_kw);
                if (cpl_propertylist_has(keywords, seeing_kw)) {
                    *seeing = cpl_propertylist_get_double(keywords, seeing_kw);
                } else {
                    sprintf(seeing_kw, "%s4 %s",
                            inst_config->seeing_kw_first_part,
                            inst_config->seeing_kw_last_part);
                    //espdr_msg("Seeing KW (TEL4): %s", seeing_kw);
                    if (cpl_propertylist_has(keywords, seeing_kw)) {
                        *seeing = cpl_propertylist_get_double(keywords, seeing_kw);
                    } else {
                        espdr_msg_warning("The seeing KW (%s) not found in the FITS header, taking default: 1.0",
                                          seeing_kw);
                        *seeing_kw_qc = 0;
                        *seeing = 1.0;
                    }
                }
            }
        }
    }
    
    if (*seeing < inst_config->vltsh_iq) {
        espdr_msg_warning("Seeing measure too low (%f), setting to the VLT IQ (%f)",
                          *seeing, inst_config->vltsh_iq);
        *seeing_kw_qc = 0;
        *seeing = inst_config->vltsh_iq;
    } else {
        if (*seeing > 3.0) {
            espdr_msg_warning("Seeing measure not reliable (%f), default value used: 1.0 arcsec",
                              *seeing);
            *seeing_kw_qc = 0;
            *seeing = 1.0;
        } else {
            espdr_msg("Seeing from the header: %f", *seeing);
        }
    }
    
    cpl_free(seeing_kw);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   compute photon count error in [ADU]
 @param   ima_data in [ADU]
 @param   conad detector's conversion factor in [e- / ADU]
 @param   ron  detector's read out noise in [ADU]
 @param   ima_errs output error image in [ADU]
 @return  cpl_error_code
 @note ima_errs need to be deallocated
 ima_data must contain the photon counts with no offsets
 this usually means the image must be overscan and bias corrected
 Then the shot noise can be calculated from the poissonian distribution
 as sqrt(electron-counts). To this (transformed back into ADUs) the
 readout noise is added in quadrature.
 @doc
 error is computed with standard formula
 
 \f$ err_{ADU} = \sqrt{ \frac{ counts }{ conad } + ron^{ 2 } } \f$
 
 If an image value is negative the associated error is set to RON
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_detector_shotnoise_model(const cpl_image* ima_data,
                                                     const double conad,
                                                     const double ron,
                                                     cpl_image ** ima_errs) {
    
    cpl_ensure_code(ima_data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ima_errs, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(conad > 0., CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ron > 0., CPL_ERROR_ILLEGAL_INPUT);
    
    *ima_errs = cpl_image_duplicate(ima_data);
    /* set negative values (= zero measurable electrons) to read out noise */
    cpl_image_threshold(*ima_errs, 0., INFINITY, ron, ron);
    
    /* err_ADU = sqrt(counts*conad + ron * ron)*/
    
    cpl_image_multiply_scalar(*ima_errs, conad);
    cpl_image_add_scalar(*ima_errs, ron * ron);
    cpl_image_power(*ima_errs, 0.5);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Detect cosmics (van-Dokkum method implemented in HDRL)
 @param   science input imagelist [e-]
 @param   ron  detector's read out noise in [ADU]
 @param   conad detector's conversion factor in [e- / ADU] (1 if image is in e- units)
 @param   plist FITS header
 @param   inst_config instrument configuration
 @param   crh_map output array of cpl_masks cosmic map
 @param   crh_map output array of cpl_masks post-filtered cosmic map
 @return  cpl_error_code
 
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_detect_cosmics(const cpl_imagelist* science,
                                           const double ron,
                                           const double conad,
                                           cpl_parameterlist* plist,
                                           espdr_inst_config *inst_config,
                                           cpl_mask** crh_map,
                                           cpl_mask** fcrh_map) {
    
    const char* RECIPE_NAME = "espdr_sci_red";
    double slim = 4.0;
    double flim = 3.0;
    int max_iter = 5;
    int pfx = 0;
    int pfy = 0;
    const char* pfm = "dilation";
    cpl_filter_mode             filter_mode = CPL_FILTER_DILATION ;
    cpl_parameter     *   par = NULL;
    hdrl_parameter          *   lacosmic_params = NULL;
    hdrl_image              *   hima ;
    int size = cpl_imagelist_get_size(science);
    const cpl_image* ima = NULL;
    cpl_image* err = NULL;
    
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.post-filter-x");
    pfx = inst_config->lacosmic_post_filter_x;
    cpl_parameter_set_int(par,pfx);
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.post-filter-y");
    pfy = inst_config->lacosmic_post_filter_y;
    cpl_parameter_set_int(par,pfy);
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.post-filter-mode");
    pfm = inst_config->lacosmic_post_filter_mode;
    cpl_parameter_set_string(par,pfm);
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.sigma_lim");
    slim = inst_config->lacosmic_sigma_lim;
    cpl_parameter_set_double(par,slim);
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.f_lim");
    flim = inst_config->lacosmic_f_lim;
    cpl_parameter_set_double(par,flim);
    par = cpl_parameterlist_find(plist, "espdr.espdr_sci_red.lacosmic.max_iter");
    max_iter = inst_config->lacosmic_max_iter;
    cpl_parameter_set_int(par,max_iter);
    
    if(!strcmp(pfm, "closing")) {
        filter_mode = CPL_FILTER_CLOSING ;
    } else if(!strcmp(pfm, "dilation")) {
        filter_mode = CPL_FILTER_DILATION ;
    } else {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Filter mode can only be \"closing\" or \"dilation\" (not %s)",
                                     pfm);
    }
    
    /* Parse the LaCosmic Parameters */
    lacosmic_params = hdrl_lacosmic_parameter_parse_parlist(plist,
                                                            "espdr.espdr_sci_red.lacosmic");
    
    if (pfx > 0 && pfy > 0) {
        espdr_ensure((pfx % 2) == 0, CPL_ERROR_ILLEGAL_INPUT,
                     "post-filter-x must be an odd number and now is %d", pfx);
        espdr_ensure((pfy % 2) == 0, CPL_ERROR_ILLEGAL_INPUT,
                     "post-filter-y must be an odd number and now is %d", pfy);
    }
    
    espdr_msg(ANSI_COLOR_CYAN"LA cosmic parameters used:"ANSI_COLOR_RESET);
    espdr_msg(ANSI_COLOR_CYAN"\tPoisson fluctuation threshold: %.2f"ANSI_COLOR_RESET, slim);
    espdr_msg(ANSI_COLOR_CYAN"\tMinimum contrast: %.2f"ANSI_COLOR_RESET, flim);
    espdr_msg(ANSI_COLOR_CYAN"\tMaximum iterations number: %d"ANSI_COLOR_RESET, max_iter);
    espdr_msg(ANSI_COLOR_CYAN"\tpost_filtering x: %d"ANSI_COLOR_RESET, pfx);
    espdr_msg(ANSI_COLOR_CYAN"\tpost_filtering y: %d"ANSI_COLOR_RESET, pfy);
    espdr_msg(ANSI_COLOR_CYAN"\tpost_filtering mode: %s"ANSI_COLOR_RESET, pfm);
    
    
    /* Lacosmic Computation */
    espdr_msg("Starting cosmic pixel detection");
    espdr_msg("Shot-noise model: conad: %g RON: %g", conad, ron);
    for (int i = 0; i < size; i++) {
        ima = cpl_imagelist_get_const(science, i);
        espdr_detector_shotnoise_model(ima, conad, ron, &err);
        hima = hdrl_image_create(ima,err);
        crh_map[i] = hdrl_lacosmic_edgedetect(hima, lacosmic_params);
        
        /* Post Filtering */
        if (pfx > 0 && pfy > 0) {
            fcrh_map[i]=hdrl_bpm_filter(crh_map[i], pfx, pfy, filter_mode);
        }
        
    }
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static long int espdr_count_sat_pix(cpl_imagelist* raw_science_imagelist,
                                    double thresh_max){
    
    long int nsat = 0;
    cpl_image* ima;
    cpl_size sz = cpl_imagelist_get_size(raw_science_imagelist);
    double* pima;
    cpl_size sx, sy, ipix, jsx;
    for(cpl_size k = 0; k < sz; k++) {
        ima = cpl_imagelist_get(raw_science_imagelist, k);
        pima = cpl_image_get_data_double(ima);
        sx = cpl_image_get_size_x(ima);
        sy = cpl_image_get_size_y(ima);
        for(cpl_size j = 0; j < sy; j++) {
            jsx = j * sx;
            for(cpl_size i = 0; i < sx; i++) {
                ipix = i + jsx;
                if( pima[ipix] >= thresh_max) {
                    nsat++;
                }
            }
        }
    }
    return nsat;
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_process_cosmics(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            cpl_frameset *used_frames,
                                            cpl_imagelist *CCD_corrected_science,
                                            cpl_propertylist *keywords,
                                            cpl_propertylist **keywords_ext,
                                            int lacosmic_flag,
                                            espdr_CCD_geometry *CCD_geom,
                                            espdr_inst_config *inst_config,
                                            double *ron,
                                            double *conad,
                                            cpl_mask ***used_crh_map_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_frame* van_dok_mask = cpl_frameset_find(frameset, ESPDR_CRH_MAP);
    cpl_mask** crh_map = NULL;
    cpl_mask** fcrh_map = NULL;
    int size = cpl_imagelist_get_size(CCD_corrected_science);
    
    if (van_dok_mask == NULL) {
        if (lacosmic_flag == 0) {
            espdr_msg(ANSI_COLOR_CYAN"LA Cosmic NOT activated"ANSI_COLOR_RESET);
        } else {
            espdr_msg("Cosmics detection via LA Cosmic ...");
            /* cosmic detection */
            crh_map = (cpl_mask**) cpl_calloc(size, sizeof(cpl_mask*));
            fcrh_map = (cpl_mask**) cpl_calloc(size, sizeof(cpl_mask*));
            cpl_vector* v_conad = cpl_vector_wrap(CCD_geom->total_output_nb, conad);
            cpl_vector* v_ron = cpl_vector_wrap(CCD_geom->total_output_nb, ron);
            double lc_conad = cpl_vector_get_mean(v_conad);
            lc_conad = 1.0; // the CCD cleaned frame is already in e-, not ADUs
            double lc_ron  = cpl_vector_get_mean(v_ron);
            cpl_vector_unwrap(v_conad);
            cpl_vector_unwrap(v_ron);
            espdr_msg("lc_ron = %f, lc_conad = %f", lc_ron, lc_conad);
            my_error = espdr_detect_cosmics(CCD_corrected_science, lc_ron, lc_conad,
                                            parameters, inst_config, crh_map, fcrh_map);
            
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_detect_cosmics failed: %s",
                         cpl_error_get_message_default(my_error));
            
            /* we use a pointer (no need to de/allocate memory) to point to the
             * appropriate crh_map, so that if the user wants to do post-smoothing
             * it is used the smoothed mask
             */
            if(inst_config->lacosmic_post_filter_x > 0 &&
               inst_config->lacosmic_post_filter_y > 0) {
                *used_crh_map_RE = fcrh_map;
            } else {
                *used_crh_map_RE = crh_map;
            }
            
            cpl_parameter *par_extra_products_sw = cpl_parameterlist_find(parameters,
                                                                          "espdr.espdr_sci_red.extra_products_sw");
            cpl_boolean extra_products_sw = cpl_parameter_get_bool(par_extra_products_sw);
            
            if (extra_products_sw == CPL_TRUE) {
                espdr_msg("Saving extra products to inspect LA-cosmic results");
                /* create extra products to allow check quality of CRH detection */
                
                char crh_map_filename[64];
                sprintf(crh_map_filename, "%s_crh_map.fits", inst_config->instrument);
                cpl_propertylist* plist = cpl_propertylist_new();
                my_error = cpl_propertylist_update_string(plist, PRO_CATG_KW, ESPDR_CRH_MAP);
                
                my_error = espdr_mask_save(frameset, parameters,
                                           used_frames, "espdr_sci_red",
                                           plist, keywords_ext,
                                           crh_map_filename,
                                           crh_map, CCD_geom);
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "crh map saving has failed: %s",
                             cpl_error_get_message_default(my_error));
                
                if (inst_config->lacosmic_post_filter_x > 0 &&
                    inst_config->lacosmic_post_filter_y > 0) {
                    sprintf(crh_map_filename, "%s_fcrh_map.fits",
                            inst_config->instrument);
                    
                    my_error = cpl_propertylist_update_string(plist,
                                                              PRO_CATG_KW,
                                                              ESPDR_FCRH_MAP);
                    my_error = espdr_mask_save(frameset, parameters,
                                               used_frames, "espdr_sci_red",
                                               plist, keywords_ext,
                                               crh_map_filename,
                                               fcrh_map, CCD_geom);
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "fcrh map saving has failed: %s",
                                 cpl_error_get_message_default(my_error));
                }
                
                sprintf(crh_map_filename, "%s_CCD_corrected_science.fits",
                        inst_config->instrument);
                
                my_error = cpl_propertylist_update_string(plist,
                                                          PRO_CATG_KW,
                                                          "CCD_CORR_SCIENCE");
                
                my_error = espdr_dfs_image_save(frameset, parameters,
                                                used_frames, "espdr_sci_red",
                                                plist, keywords_ext,
                                                crh_map_filename, CCD_corrected_science,
                                                CPL_TYPE_DOUBLE, CCD_geom);
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "CCD corrected science saving has failed: %s",
                             cpl_error_get_message_default(my_error));
                cpl_propertylist_delete(plist);
                
            }
            espdr_msg("Cosmics detection finished");
        }
    } else {
        espdr_msg("Checking user provided CRH map ...");
        /* Checking & taking user provided CRH map*/
        cpl_boolean check = CPL_TRUE;
        
        const char* fname = cpl_frame_get_filename(van_dok_mask);
        const cpl_size next = cpl_frame_get_nextensions(van_dok_mask);
        espdr_msg("map filename: %s", fname);
        
        /* Consistency checks on optional input CRH_MAP */
        cpl_propertylist* crh_map_head = cpl_propertylist_load(fname,0);
        const char* arcfile = cpl_propertylist_get_string(crh_map_head,"ARCFILE");
        const int binx = cpl_propertylist_get_int(crh_map_head,"ESO DET BINX");
        const int biny = cpl_propertylist_get_int(crh_map_head,"ESO DET BINY");
        espdr_msg("map arcfile: %s, binning: %d x %d, nb of extensions: %lld",
                  arcfile, binx, biny, next);
        
        const char* sarcfile = cpl_propertylist_get_string(keywords,"ARCFILE");
        const int sbinx = cpl_propertylist_get_int(keywords,"ESO DET BINX");
        const int sbiny = cpl_propertylist_get_int(keywords,"ESO DET BINY");
        
        if ((strcmp(arcfile, sarcfile) != 0 || binx != sbinx || biny != sbiny || next != size)) {
            espdr_msg(ANSI_COLOR_RED"Wrong optional input CRH_MAP, check that "\
                      "ARCFILE, BINX, BINY and number of extensions are the same as "\
                      "in the input science frame. Assumed CRH_MAP is not provided."ANSI_COLOR_RESET);
            check = CPL_FALSE;
        } else {
            espdr_msg(ANSI_COLOR_CYAN"Cosmics removal with user provided map"ANSI_COLOR_RESET);
            crh_map = (cpl_mask**) cpl_calloc(size, sizeof(cpl_mask*));
            cpl_image* sci_ima = NULL;
            
            for (cpl_size i = 0; i < size; i++) {
                crh_map[i] = cpl_mask_load(fname, 0, i+1);
                sci_ima = cpl_imagelist_get(CCD_corrected_science, i);
                if (cpl_mask_get_size_x(crh_map[i]) != cpl_image_get_size_x(sci_ima) ||
                    cpl_mask_get_size_y(crh_map[i]) != cpl_image_get_size_y(sci_ima)) {
                    espdr_msg(ANSI_COLOR_RED"CRH MAP EXT[%lld] X, Y sizes differ from CCD corrected science X, Y sizes. Assumed as not provided."ANSI_COLOR_RESET, i);
                    check = CPL_FALSE;
                }
            }
            
            if (check) {
                *used_crh_map_RE = crh_map;
            } else {
                *used_crh_map_RE = NULL;
            }
        }
        cpl_propertylist_delete(crh_map_head);
    }
    
    /*
     // strange error when trying to clean the memory, even when assigning the masks duplicates
    if (van_dok_mask == NULL) {
        if (lacosmic_flag == 1) {
            for (int i = 0; i < size; i++) {
                cpl_mask_delete(crh_map[i]);
                cpl_mask_delete(fcrh_map[i]);
            }
            cpl_free(crh_map);
            cpl_free(fcrh_map);
        }
    } else {
        if (crh_map != NULL){
            for (int i = 0; i < size; i++) {
                cpl_mask_delete(crh_map[i]);
            }
            cpl_free(crh_map);
        }
    }
    */
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_process_raw_till_extraction(cpl_frameset *frameset,
                                                 cpl_parameterlist *parameters,
                                                 const char *RECIPE_ID,
                                                 cpl_frame *raw_frame,
                                                 cpl_frameset *used_frames,
                                                 espdr_inst_config *inst_config,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 espdr_qc_keywords *qc_kws,
                                                 cpl_propertylist *keywords,
                                                 int remove_bias_res_flag,
                                                 char *wave_cal_source,
                                                 char *fibre_b,
                                                 int lacosmic_flag,
                                                 char *background_sw,
                                                 char *extraction_method,
                                                 double *lamp_offset_ar,
                                                 double *lamp_offset_ar1,
                                                 double *lamp_offset_ar2,
                                                 double *ron,
                                                 int **cosmics_nb,
                                                 double **snr_per_fibre,
                                                 double *mjd_obs_delta_time_wave,
                                                 cpl_image **blaze_table,
                                                 cpl_image **wave_matrix,
                                                 cpl_image **wave_matrix_air,
                                                 cpl_image **dll_matrix,
                                                 cpl_image **dll_matrix_air,
                                                 cpl_image **s2d_ref_image,
                                                 cpl_image **flat_corr_flux_RE,
                                                 cpl_image **flat_corr_err_RE,
                                                 cpl_image **flat_corr_qual_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Start processing %s", RECIPE_ID);
    
    cpl_frame* master_bias_frame;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        master_bias_frame = espdr_get_mbias_from_set(frameset, used_frames);
    }

    
    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame = espdr_get_hpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_get_mdark_from_set(frameset,
                                                            used_frames);
    
    /* Extract bad pixels mask frame */
    cpl_frame* bad_pixels_frame = espdr_get_bpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract orders mask frame */
    cpl_frame* orders_mask_frame = espdr_get_orders_mask_from_set(frameset,
                                                                  used_frames);
    
    /* Extract orders maps frame */
    cpl_table ***orders_coeffs = NULL;
    espdr_get_orders_coeff_from_set(frameset, used_frames,
                                    inst_config->fibres_nb,
                                    CCD_geom->ext_nb,
                                    &orders_coeffs);
    
    /* Extract order profile frame per fibre */
    cpl_frame ** order_profile_fibre_frame = NULL;
    espdr_get_order_profile_from_set(frameset, used_frames,
                                     inst_config->fibres_nb,
                                     CCD_geom->ext_nb,
                                     &order_profile_fibre_frame);
    
    /* Extract flat frame per fibre */
    cpl_frame** flat_fibre_frame = NULL;
    espdr_get_flat_fibre_from_set(frameset, used_frames,
                                  inst_config->fibres_nb,
                                  CCD_geom->ext_nb,
                                  &flat_fibre_frame);
    
    /* Extract BLAZE frame per fibre */
    cpl_frame** blaze_fibre_frame = NULL;
    espdr_get_blaze_fibre_from_set(frameset, used_frames,
                                   inst_config->fibres_nb,
                                   CCD_geom->ext_nb,
                                   &blaze_fibre_frame);
    
    /* Extract wave & dll frames */
    char suffix1[2][64], suffix2[2][64];
    if ((strcmp(fibre_b, "SKY") == 0) || (strcmp(fibre_b, "DARK") == 0)) {
        sprintf(suffix1[0],"%s_FP_A", wave_cal_source);
        sprintf(suffix2[0],"%s_THAR_A", wave_cal_source);
    } else {
        sprintf(suffix1[0],"%s_%s_A", wave_cal_source, fibre_b);
        sprintf(suffix2[0],"%s_%s_A", wave_cal_source, fibre_b);
    }
    sprintf(suffix1[1],"FP_%s_B", wave_cal_source);
    sprintf(suffix2[1],"THAR_%s_B", wave_cal_source);
    
    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        cpl_frame *wave_matrix_frame = espdr_get_wave_matrix_frame_from_set(frameset, used_frames,
                                                                            suffix1[fibre], suffix2[fibre], fibre);
        espdr_ensure(wave_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "espdr_get_wave_matrix_frame_from_set failed for fibre %c", fibre_name[fibre]);
        
        const char *wave_name = cpl_frame_get_filename(wave_matrix_frame);
        espdr_msg("WAVE MATRIX for fibre %c: %s", fibre_name[fibre], wave_name);
        wave_matrix[fibre] = cpl_image_load(wave_name, CPL_TYPE_DOUBLE, 0, 1);
        wave_matrix_air[fibre] = cpl_image_load(wave_name, CPL_TYPE_DOUBLE, 0, 1);
        // Converting vacuum wave matrix to air
        int pir;
        for (int order = 1; order <= cpl_image_get_size_y(wave_matrix[fibre]); order++){
            for (int pix = 1; pix <= cpl_image_get_size_x(wave_matrix[fibre]); pix++){
                double ll = cpl_image_get(wave_matrix[fibre], pix, order, &pir);
                double n_air = espdr_compute_n_air(ll, N_AIR_COEFF_T, N_AIR_COEFF_P);
                cpl_image_set(wave_matrix_air[fibre], pix, order, ll/n_air);
            }
        }
        
        // Get the LAMP_OFFSET
        cpl_propertylist *keywords_wave = cpl_propertylist_load(wave_name, 0);
        if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
            if (cpl_propertylist_has(keywords_wave, qc_kws->qc_wave_thar_lamp_offset_ar_kw)) {
                lamp_offset_ar[fibre] = cpl_propertylist_get_double(keywords_wave,
                                                                    qc_kws->qc_wave_thar_lamp_offset_ar_kw);
            } else {
                lamp_offset_ar[fibre] = -9999.9;
            }
            espdr_msg("Lamp offset AR for fiber A: %f", lamp_offset_ar[fibre]);
            
            if (cpl_propertylist_has(keywords_wave, qc_kws->qc_wave_thar_lamp_offset_ar1_kw)) {
                lamp_offset_ar1[fibre] = cpl_propertylist_get_double(keywords_wave,
                                                                     qc_kws->qc_wave_thar_lamp_offset_ar1_kw);
            } else {
                lamp_offset_ar1[fibre] = -9999.9;
            }
            espdr_msg("Lamp offset AR1 for fiber A: %f", lamp_offset_ar1[fibre]);
            
            if (cpl_propertylist_has(keywords_wave, qc_kws->qc_wave_thar_lamp_offset_ar2_kw)) {
                lamp_offset_ar2[fibre] = cpl_propertylist_get_double(keywords_wave,
                                                                     qc_kws->qc_wave_thar_lamp_offset_ar2_kw);
            } else {
                lamp_offset_ar2[fibre] = -9999.9;
            }
            espdr_msg("Lamp offset AR2 for fiber A: %f", lamp_offset_ar2[fibre]);
        }
        
        double mjd_obs_science = 0.0, mjd_obs_wave_matrix = 0.0;
        if (cpl_propertylist_has(keywords, "MJD-OBS") && cpl_propertylist_has(keywords_wave, "MJD-OBS")) {
            mjd_obs_science = espdr_pfits_get_mjdobs(keywords);
            mjd_obs_wave_matrix = espdr_pfits_get_mjdobs(keywords_wave);
        }
        mjd_obs_delta_time_wave[fibre] = mjd_obs_wave_matrix - mjd_obs_science;
        
        cpl_propertylist_delete(keywords_wave);

        cpl_frame *dll_matrix_frame = espdr_get_dll_matrix_frame_from_set(frameset, used_frames,
                                                                          suffix1[fibre], suffix2[fibre], fibre);
        espdr_ensure(dll_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "espdr_get_dll_matrix_frame_from_set failed for fibre %c", fibre_name[fibre]);
        
        const char *dll_name = cpl_frame_get_filename(dll_matrix_frame);
        espdr_msg("DLL MATRIX for fibre %c: %s", fibre_name[fibre], dll_name);
        dll_matrix[fibre] = cpl_image_load(dll_name, CPL_TYPE_DOUBLE, 0, 1);
        dll_matrix_air[fibre] = cpl_image_load(dll_name, CPL_TYPE_DOUBLE, 0, 1);
        // Converting vacuum wave matrix to air
        for (int order = 1; order <= cpl_image_get_size_y(dll_matrix[fibre]); order++){
            for (int pix = 1; pix <= cpl_image_get_size_x(dll_matrix[fibre]); pix++){
                double ll = cpl_image_get(dll_matrix[fibre], pix, order, &pir);
                double n_air = espdr_compute_n_air(ll, N_AIR_COEFF_T, N_AIR_COEFF_P);
                cpl_image_set(dll_matrix_air[fibre], pix, order, ll/n_air);
            }
        }
        
    }
    
    cpl_frame *contam_frame = NULL;
    cpl_imagelist *contam_imagelist = NULL;
    if (strcmp(fibre_b, "THAR") == 0 || strcmp(fibre_b, "FP") == 0 || strcmp(fibre_b, "LFC") == 0) {
        char contam_catg_tag[32];
        sprintf(contam_catg_tag, "%s_%s", ESPDR_PRO_CATG_CONTAM_SRC, fibre_b);
        contam_frame = espdr_frame_find(frameset, contam_catg_tag);
        if (contam_frame == NULL) {
            espdr_msg(ANSI_COLOR_RED"\t\tWarning : NO contamination frame provided."ANSI_COLOR_RESET);
        } else {
            my_error = cpl_frameset_insert(used_frames,
                                           cpl_frame_duplicate(contam_frame));
            contam_imagelist = cpl_imagelist_new();
            my_error = espdr_extract_extensions(contam_frame, CPL_TYPE_DOUBLE,
                                                CCD_geom->ext_nb, &contam_imagelist);
        }
        
        char s2d_tag[32];
        sprintf(s2d_tag, "S2D_BLAZE_%s_%s_%c", wave_cal_source, fibre_b, fibre_name[1]);
        cpl_frame *s2d_ref_frame = espdr_frame_find(frameset, s2d_tag);
        espdr_ensure(s2d_ref_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "No input S2D reference frame");
        my_error = cpl_frameset_insert(used_frames,
                                       cpl_frame_duplicate(s2d_ref_frame));
        
        const char *s2d_ref_filename = cpl_frame_get_filename(s2d_ref_frame);
        s2d_ref_image[0] = cpl_image_load(s2d_ref_filename, CPL_TYPE_DOUBLE, 0, 1);
        s2d_ref_image[1] = cpl_image_load(s2d_ref_filename, CPL_TYPE_DOUBLE, 0, 2);
        s2d_ref_image[2] = cpl_image_load(s2d_ref_filename, CPL_TYPE_INT, 0, 3);
        
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_frame_get_filename or cpl_image_load for %s failed: %s",
                     s2d_ref_filename, cpl_error_get_message_default(my_error));
    }
    
    /* Adding KW with time interval to calibrations */
    int recipes[10] = {1, 1, 1, 1, 1, 0, 1, 1, 1, 1};
    char recipe_name[32];
    if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
        strcpy(recipe_name, "SCIENCE");
    } else {
        strcpy(recipe_name, "CAL_FLUX");
        recipes[9] = 0;
    }
    my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords,
                                                    wave_cal_source, fibre_b,
                                                    mjd_obs_delta_time_wave, recipe_name, recipes);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_calibrations_intervals failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *master_bias_list = NULL;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        /* Extract all the outputs from the master bias frame */
        master_bias_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_bias_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_bias_list);
    }
    
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &hot_pixels_list);
    
    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    cpl_imagelist *master_dark_pxl_nb_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_dark_list);
        master_dark_pxl_nb_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_INT,
                                              CCD_geom, CCD_geom->ext_nb,
                                              &master_dark_pxl_nb_list);
    }
    
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &bad_pixels_list);
    
    cpl_imagelist *orders_mask_list = NULL;
    /* Extract all the outputs from the orders mask frame if provided */
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_list);
    }
    
    /* Extract extensions from order profile frames */
    cpl_imagelist **order_profile_fibre_imagelist = (cpl_imagelist **)cpl_malloc
                                    (inst_config->fibres_nb * sizeof(cpl_imagelist *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        order_profile_fibre_imagelist[i] = cpl_imagelist_new();
        my_error = espdr_extract_extensions(order_profile_fibre_frame[i],
                                            CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb,
                                            &order_profile_fibre_imagelist[i]);
    }
    
    /* Extract extensions from flat frames */
    cpl_image **flat_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_error_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_quality_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        const char *flat_filename = cpl_frame_get_filename(flat_fibre_frame[i]);
        flat_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 1);
        flat_error_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 2);
        flat_quality_table[i] = cpl_image_load(flat_filename, CPL_TYPE_INT, 0, 3);
    }
    
    /* Extract image from blaze frame */
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        const char *blaze_filename = cpl_frame_get_filename(blaze_fibre_frame[i]);
        blaze_table[i] = cpl_image_load(blaze_filename, CPL_TYPE_DOUBLE, 0, 1);
    }
    
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Extraction of image from frame failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Extract raw and real images from the raw frame */
    cpl_imagelist *raw_imagelist = cpl_imagelist_new();
    cpl_imagelist *real_imagelist = cpl_imagelist_new();
    if (inst_config->inst_type == NIR) {
        my_error = espdr_extract_extensions(raw_frame, CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_extract_raw_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                       CCD_geom, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_raw_outputs_one_frame failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    my_error = espdr_extract_real_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                    CCD_geom, &real_imagelist);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs_one_frame failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot & bad pixels mask");
    
    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        cpl_frameset *raw_frames_to_reduce = cpl_frameset_new();
        cpl_frameset_insert(raw_frames_to_reduce, cpl_frame_duplicate(raw_frame));
        my_error = espdr_update_bad_pixel_mask(raw_frames_to_reduce, CCD_geom,
                                               cpl_imagelist_get(bad_pixels_list, 0));
        cpl_frameset_delete(raw_frames_to_reduce);
    }
    
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list,
                                                bad_pixels_list,
                                                &pixels_mask);
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(merged_pixels_mask);
    
    /* Checking for saturation, creating saturation mask */
    
    /* Checking the saturation outside hot and bad pixels and cosmics */
    double cosmics_part = inst_config->image_cosmics_part;
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_imagelist),
                                            sizeof(double));
    int print_max_flux = 1;
    if (strcmp(fibre_b, "THAR") == 0) {
        cosmics_part = inst_config->image_thar_part;
        print_max_flux = 0;
    }
    
    // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
    if (inst_config->inst_type == NIR) {
        double texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_imagelist, texp);
    }
    my_error = espdr_check_saturation(real_imagelist,
                                      CCD_geom, inst_config,
                                      pixels_mask, cosmics_part,
                                      max_flux, print_max_flux,
                                      &geometry_corrected_saturation_mask);
    cpl_imagelist_delete(real_imagelist);
    
    /* Add QC KWs to the header */
    if (strcmp(fibre_b, "THAR") != 0) {
        char *check_kw = NULL;
        if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
            check_kw = qc_kws->qc_sci_red_check_kw;
        } else {
            check_kw = qc_kws->qc_cal_flux_check_kw;
        }
        my_error = espdr_max_flux_QC(inst_config, CCD_geom, qc_kws,
                                     max_flux, check_kw, keywords);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_max_flux_QC failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    double *dark = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    double *dark_ron = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    double *conad = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    cpl_imagelist *CCD_corrected_raw = cpl_imagelist_new();
    
    espdr_msg("Loading headers");
    /* Load primary header keywords from the hot pixels image */
    const char *input_filename_HP = NULL;
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s", input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s", input_filename_HP);
    }
    cpl_propertylist *keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    const char *input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s", input_filename_BP);
    cpl_propertylist *keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    
    // Detector signature removal for the science frame process ...
    espdr_msg("Removing the CCD signature ...");
    if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
        double sat_threshold = inst_config->satur_limit_adu;
        long int nsat = espdr_count_sat_pix(raw_imagelist, sat_threshold);
        cpl_propertylist_append_long(keywords, "ESO QC SAT LEVEL", sat_threshold);
        cpl_propertylist_set_comment(keywords, "ESO QC SAT LEVEL",
                                     "Saturation level for pixels in input raw frames");
        cpl_propertylist_append_long(keywords, "ESO QC SAT NB", nsat);
        cpl_propertylist_set_comment(keywords, "ESO QC SAT NB",
            "Number of pixels in input raw file with pixel value >= QC.SAT.LEVEL (RED and BLUE)");
    }
    
    my_error = espdr_remove_det_signature(raw_imagelist,
                                          orders_mask_list,
                                          master_bias_list,
                                          master_dark_list,
                                          keywords, keywords_HP, keywords_BP,
                                          qc_kws, inst_config, remove_bias_res_flag,
                                          CCD_geom, CPL_TYPE_DOUBLE,
                                          ron, dark, dark_ron, conad,
                                          CCD_corrected_raw);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_detector_signature failed: %s",
                 cpl_error_get_message_default(my_error));

    cpl_imagelist_delete(raw_imagelist);
    
    cpl_propertylist **keywords_ext = (cpl_propertylist**)cpl_malloc(
                                            CCD_geom->ext_nb * sizeof(cpl_propertylist*));
    
    const char *input_filename_ext = cpl_frame_get_filename(raw_frame);
    for (int j = 0; j < CCD_geom->ext_nb; j++) {
        keywords_ext[j] = cpl_propertylist_load(input_filename_ext,j+1);
    }
    
    cpl_mask **used_crh_map = NULL;
    if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
        my_error = espdr_process_cosmics(frameset, parameters, used_frames,
                                         CCD_corrected_raw, keywords, keywords_ext,
                                         lacosmic_flag,
                                         CCD_geom, inst_config,
                                         ron, conad,
                                         &used_crh_map);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_process_cosmics failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
#if SAVE_DEBUG_CCD_CLEANED_SCIENCE
    espdr_msg("Saving cleaned science frame");
    /* Save the PRO.CATG */
    char pro_catg[16];
    if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
        strcpy(pro_catg, ESPDR_SCIENCE_RAW);
    } else {
        strcpy(pro_catg, ESPDR_FLUX_RAW);
    }
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, pro_catg);
    char ccd_cleaned_filename[FILENAME_LENGTH];
    sprintf(ccd_cleaned_filename, "%s_CCD_cleaned_%s_%s.fits",
            inst_config->instrument, pro_catg, fibre_b);
    
    /* Save the cleaned WAVE fits frame */
    my_error = espdr_save_ccd_cleaned_wave_frame(frameset,
                                                 used_frames,
                                                 parameters,
                                                 RECIPE_ID,
                                                 ccd_cleaned_filename,
                                                 CCD_geom,
                                                 keywords,
                                                 keywords_ext,
                                                 CCD_corrected_raw);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_ccd_cleaned_wave_frame failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 ccd_cleaned_filename);
#endif
    
    cpl_imagelist *raw_bkgr_measured = cpl_imagelist_new();
    cpl_imagelist *raw_bkgr_subs = cpl_imagelist_new();
    
    /* Background calculation process ... */
    espdr_msg("background switch: <%s>", background_sw);
    if (strstr(background_sw, "on") != NULL) {
        espdr_msg("Background measurement process ...");
        cpl_imagelist *order_profile_fibre_iml_both_fibres = cpl_imagelist_duplicate(order_profile_fibre_imagelist[0]);
        if (strcmp(background_sw, "on") == 0) {
            espdr_msg("Removing fiber B from background pixels");
            cpl_imagelist_add(order_profile_fibre_iml_both_fibres,
                              cpl_imagelist_duplicate(order_profile_fibre_imagelist[1]));
        } else {
            espdr_msg("Using also pixels from fibre B for background calculation");
        }
        cpl_imagelist_add(order_profile_fibre_imagelist[0], order_profile_fibre_imagelist[1]);
        espdr_msg("Background box size: %d x %d",
                  inst_config->sci_bkgr_grid_size_x, inst_config->sci_bkgr_grid_size_y);
        my_error = espdr_measure_background_science_by_ext(
                                            CCD_corrected_raw,
                                            order_profile_fibre_iml_both_fibres,
                                            conad,
                                            ron,
                                            inst_config->sci_bkgr_grid_size_x,
                                            inst_config->sci_bkgr_grid_size_y,
                                            CCD_geom,
                                            raw_bkgr_measured,
                                            raw_bkgr_subs);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_measure_inter_order_background failed");
        cpl_imagelist_delete(order_profile_fibre_iml_both_fibres);
        espdr_msg("Background measurement process finished !");
    } else {
        espdr_msg(ANSI_COLOR_RED"Warning : Background measurement NOT performed !"ANSI_COLOR_RESET);
        raw_bkgr_measured = NULL;
    }
    
    //espdr_msg("Saving BKGR KWs");
    char *new_keyword = NULL;
    double bkgr_mean = -9999.9, bkgr_max = -9999.9, bkgr_min = -9999.9;
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        if (raw_bkgr_measured != NULL) {
            bkgr_mean = cpl_image_get_mean(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_min = cpl_image_get_min(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_max = cpl_image_get_max(cpl_imagelist_get(raw_bkgr_measured,i));
        }
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_mean_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_mean_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_mean, "Bkgr mean [e-] in the extension", &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_min_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_min_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_min, "Bkgr min [e-] in the extension", &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_max_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_max_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_max, "Bkgr max [e-] in the extension", &keywords);
        cpl_free(new_keyword);
    }
    

    
#if SAVE_DEBUG_PRODUCT_SCIENCE
    //-----------------------------------------------------------
    
    if (strcmp(background_sw, "on") == 0) {
        
        char bkgr_map_filename[64];
        char bkgr_subs_filename[64];
        
        sprintf(bkgr_map_filename, "%s_background_map.fits", inst_config->instrument);
        my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_TEST_PRODUCT);
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        used_frames, "sci_red",
                                        keywords, keywords_ext,
                                        bkgr_map_filename,
                                        raw_bkgr_measured,
                                        CPL_TYPE_FLOAT, CCD_geom);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "background_map image saving has failed: %s",
                     cpl_error_get_message_default(my_error));
        
        sprintf(bkgr_subs_filename, "%s_science_bkgr_subs.fits", inst_config->instrument);
        
        //SCIENCE (backgroud substracted) image saved
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        used_frames, "sci_red",
                                        keywords, keywords_ext,
                                        bkgr_subs_filename,
                                        raw_bkgr_subs,
                                        CPL_TYPE_FLOAT, CCD_geom);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "science_bkgr_subs image saving has failed: %s",
                     cpl_error_get_message_default(my_error));
        
    }
    
#endif
    
    
    double exp_time_hour = cpl_propertylist_get_double(keywords, inst_config->Texp_kw) / 3600.0;
    double exp_time_hour_mdark = cpl_propertylist_get_double(keywords_HP, inst_config->Texp_kw) / 3600.0;
    
    int fibres_processed;
    if (strcmp(fibre_b,"DARK") == 0) {
        fibres_processed = inst_config->fibres_nb - 1;
    } else {
        fibres_processed = inst_config->fibres_nb;
    }
    
    cpl_image **s2d_flux = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **s2d_error = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    cpl_image **s2d_quality = (cpl_image **)cpl_malloc(fibres_processed*sizeof(cpl_image *));
    
    for (int t = 0; t < fibres_processed; t++) {
        /* extraction process */
        espdr_msg("Horne extraction process fibre %c ...",fibre_name[t]);
        double extraction_ksigma[2];
        extraction_ksigma[0] = inst_config->extraction_ksigma_object;
        if (strcmp(fibre_b, "FP") == 0) {
            extraction_ksigma[1] = inst_config->extraction_ksigma_fp;
        } else {
            if (strcmp(fibre_b, "THAR") == 0) {
                extraction_ksigma[1] = inst_config->extraction_ksigma_thar;
            } else {
                if (strcmp(fibre_b, "LFC") == 0) {
                    extraction_ksigma[1] = inst_config->extraction_ksigma_lfc;
                } else {
                    if (strcmp(fibre_b, "SKY") == 0) {
                        extraction_ksigma[1] = inst_config->extraction_ksigma_sky;
                    } else { // DARK on fibre B
                        extraction_ksigma[1] = inst_config->extraction_ksigma_dark;
                    }
                }
            }
        }
        espdr_msg("KSIGMA used for extraction: %f", extraction_ksigma[t]);
        
        /* ASE: No CONTAM considered on extraction on fibre B */
        if (t == 1) contam_imagelist = NULL;
        
        cpl_imagelist *input_iml = NULL;
        if (strcmp(background_sw, "off") == 0) {
            input_iml = CCD_corrected_raw;
        } else {
            input_iml = raw_bkgr_subs;
        }
        
        if (strcmp(extraction_method, "horne") == 0) {
            my_error = espdr_horne_extraction_one_fibre(input_iml,
                                                        geometry_corrected_pixels_mask,
                                                        geometry_corrected_saturation_mask,
                                                        used_crh_map,
                                                        raw_bkgr_measured,
                                                        order_profile_fibre_imagelist[t],
                                                        contam_imagelist,
                                                        master_dark_list, master_dark_pxl_nb_list,
                                                        orders_coeffs[t], t,
                                                        inst_config, CCD_geom,
                                                        ron, dark, dark_ron, conad,
                                                        exp_time_hour, exp_time_hour_mdark,
                                                        inst_config->extraction_window_size[t],
                                                        inst_config->tolerance_rejection,
                                                        extraction_ksigma[t],
                                                        cosmics_nb[t],
                                                        &s2d_flux[t],
                                                        &s2d_error[t],
                                                        &s2d_quality[t]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_horne_extraction failed for fibre %c: %s",
                         fibre_name[t], cpl_error_get_message_default(my_error));
            espdr_msg("Horne extraction process fibre %c finished !", fibre_name[t]);
            
        } else if (strcmp(extraction_method, "simple") == 0) {
            espdr_msg(ANSI_COLOR_RED"Warning : Simple extraction method ..."ANSI_COLOR_RESET);
            my_error = espdr_simple_extraction_one_fibre(input_iml,
                                                         geometry_corrected_pixels_mask,
                                                         geometry_corrected_saturation_mask,
                                                         used_crh_map,
                                                         raw_bkgr_measured,
                                                         master_dark_list, master_dark_pxl_nb_list,
                                                         orders_coeffs[t], t,
                                                         inst_config, CCD_geom,
                                                         ron, dark, dark_ron, conad,
                                                         exp_time_hour, exp_time_hour_mdark,
                                                         inst_config->extraction_window_size[t],
                                                         &s2d_flux[t],
                                                         &s2d_error[t],
                                                         &s2d_quality[t]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_simple_extraction failed for fibre %c: %s",
                         fibre_name[t], cpl_error_get_message_default(my_error));
            espdr_msg("Simple extraction done");
        } else {
            espdr_msg("Error: %s is an unknown extraction method. Allowed only 'horne' and 'simple'",
                      inst_config->extraction_method);
                return (CPL_ERROR_ILLEGAL_INPUT);
        }
        
#if SAVE_DEBUG_PRODUCT_SCIENCE
        
        char pro_catg_extr[16];
        char extracted_filename[64];
        if (strcmp(RECIPE_ID, "espdr_sci_red") == 0) {
            strcpy(pro_catg_extr, ESPDR_SCIENCE_RAW);
        } else {
            strcpy(pro_catg_extr, ESPDR_FLUX_RAW);
        }
        sprintf(extracted_filename, "%s_spectrum_extracted_%s_%c.fits",
                inst_config->instrument, pro_catg_extr, fibre_name[t]);
        espdr_msg("Saving extracted spectrum %c in %s",
                  fibre_name[t], extracted_filename);
        
        my_error = espdr_dfs_save_S2D(s2d_flux[t], s2d_error[t], s2d_quality[t],
                                      frameset, used_frames, parameters,
                                      keywords, extracted_filename,
                                      ESPDR_PRO_CATG_TEST_PRODUCT,
                                      "sci_red");
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_save_S2D failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     extracted_filename);
#endif
        
        /* Flat correction */
        espdr_msg("Flat correction process fibre %c ...",fibre_name[t]);
        my_error = espdr_correct_flat(s2d_flux[t],
                                      s2d_error[t],
                                      s2d_quality[t],
                                      flat_table[t],
                                      flat_error_table[t],
                                      flat_quality_table[t],
                                      &flat_corr_flux_RE[t],
                                      &flat_corr_err_RE[t],
                                      &flat_corr_qual_RE[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_flat failed for fiber %c: %s",
                     fibre_name[t], cpl_error_get_message_default(my_error));
        espdr_msg("---->Flat correction process fibre %c finished !", fibre_name[t]);
        
        if ((t == 0) || (strcmp(fibre_b, "SKY") == 0)) { // fibre A or SKY
            espdr_msg("Compute SNR for fiber %c on extracted spectrum", fibre_name[t]);
            my_error = espdr_compute_snr(s2d_flux[t], s2d_error[t],
                                         inst_config->snr_averaging_window,
                                         snr_per_fibre[t]);
        }

    } // end loop fibre.
    

    /* Cleaning some memory */
    cpl_imagelist_delete(CCD_corrected_raw);
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    cpl_imagelist_delete(raw_bkgr_subs);
    cpl_imagelist_delete(raw_bkgr_measured);

    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);
    cpl_propertylist_delete(keywords_HP);
    cpl_propertylist_delete(keywords_BP);
    
    cpl_free(conad);
    cpl_free(dark);
    cpl_free(dark_ron);
    cpl_free(max_flux);
    
    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);
    
    for (int i = 0; i < fibres_processed; i++) {
        cpl_image_delete(s2d_flux[i]);
        cpl_image_delete(s2d_error[i]);
        cpl_image_delete(s2d_quality[i]);
    }
    cpl_free(s2d_flux);
    cpl_free(s2d_error);
    cpl_free(s2d_quality);
    
    if (contam_frame != NULL) {
        cpl_imagelist_delete(contam_imagelist);
    }
    
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        cpl_imagelist_delete(master_bias_list);
    }
    cpl_imagelist_delete(hot_pixels_list);
    if (master_dark_frame != NULL) {
        cpl_imagelist_delete(master_dark_list);
        cpl_imagelist_delete(master_dark_pxl_nb_list);
    }
    cpl_imagelist_delete(bad_pixels_list);
    if (orders_mask_frame != NULL) {
        cpl_imagelist_delete(orders_mask_list);
    }
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_imagelist_delete(order_profile_fibre_imagelist[i]);
        cpl_image_delete(flat_table[i]);
        cpl_image_delete(flat_error_table[i]);
        cpl_image_delete(flat_quality_table[i]);
    }
    cpl_free(order_profile_fibre_imagelist);
    cpl_free(flat_table);
    cpl_free(flat_error_table);
    cpl_free(flat_quality_table);
    
    cpl_imagelist_delete(pixels_mask);
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
    
    //cpl_mask_delete(used_crh_map);
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_create_wavelength_grid(double log_lambda_ini,
                                            double log_lambda_end,
                                            double delta_log_lambda,
                                            cpl_image **wavelenght_grid_pow_10,
                                            cpl_image **wavelenght_air_grid_pow_10) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    // s1d wavelength matrix process ...
    int grid_length = (int)((log_lambda_end - log_lambda_ini)/delta_log_lambda);
    cpl_image *wavelenght_grid = cpl_image_new(grid_length, 1, CPL_TYPE_DOUBLE);
    my_error = espdr_s1d_wavelenght_grid(log_lambda_ini, delta_log_lambda, grid_length,
                                         &wavelenght_grid);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_s1d_wavelenght_grid failed : %s",
                 cpl_error_get_message_default(my_error));
    
    *wavelenght_grid_pow_10 = cpl_image_new(grid_length, 1, CPL_TYPE_DOUBLE);
    double *wavelenght_grid_pow_10_data = cpl_image_get_data_double(*wavelenght_grid_pow_10);
    double *s1d_wavelenght_grid_data = cpl_image_get_data_double(wavelenght_grid);
    for (int pxl = 0; pxl < grid_length; pxl++) {
        wavelenght_grid_pow_10_data[pxl] = pow(10, s1d_wavelenght_grid_data[pxl]);
    }
    
    // s1d wavelength matrix air process ...
    *wavelenght_air_grid_pow_10 = cpl_image_new(grid_length, 1, CPL_TYPE_DOUBLE);
    int pixels = cpl_image_get_size_x(*wavelenght_air_grid_pow_10);
    int orders = cpl_image_get_size_y(*wavelenght_air_grid_pow_10);
    for (int order = 1; order <= orders; order++){
        for (int pix = 1; pix <= pixels; pix++){
            int pir;
            double ll = cpl_image_get(*wavelenght_grid_pow_10, pix, order, &pir);
            double n_air = espdr_compute_n_air(ll, N_AIR_COEFF_T, N_AIR_COEFF_P);
            cpl_image_set(*wavelenght_air_grid_pow_10, pix, order, ll/n_air);
        }
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Process the sky S2D to prepare for the sky subtraction
 @param     flat_corr_flux
 @param     flat_corr_err
 @param     flat_corr_qual
 @param     wave_matrix
 @param     rel_eff
 @param     inst_config
 @param     CCD_geom
 @param     physical_orders_id
 @param     flux_RE
 @param     err_RE
 @param     qual_RE
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_process_sky(cpl_image *flat_corr_flux,
                                 cpl_image *flat_corr_err,
                                 cpl_image *flat_corr_qual,
                                 cpl_image **wave_matrix,
                                 cpl_image **dll_matrix,
                                 cpl_image *rel_eff,
                                 espdr_inst_config *inst_config,
                                 espdr_CCD_geometry *CCD_geom,
                                 int **physical_orders_id,
                                 cpl_image **flux_RE,
                                 cpl_image **err_RE,
                                 cpl_image **qual_RE) {

	cpl_error_code my_error;

	cpl_image *scaled_flux;
	cpl_image *scaled_error;
	cpl_image *scaled_quality;

	espdr_msg("scale process ...");

	my_error = espdr_scale_sky(
			flat_corr_flux,
			flat_corr_err,
			flat_corr_qual,
			rel_eff,
			&scaled_flux,
			&scaled_error,
			&scaled_quality);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
	"espdr_scale_sky failed");
    
    //my_error = cpl_image_save(scaled_flux, "ESPRESSO_SKY_SCALED.fits",
    //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
    //             "espdr_smooth_flux failed to save image ESPRESSO_SKY_SCALED.fits: %s",
    //             cpl_error_get_message_default(my_error));
    //espdr_msg("Image ESPRESSO_SKY_SCALED.fits saved");
    
	espdr_msg("scale process finished ...");

	espdr_msg("select and rebin orders and create rebin image process ... ");

	espdr_select_orders_and_create_rebin_image(
				scaled_flux,
				scaled_error,
				scaled_quality,
				wave_matrix,
                dll_matrix,
                physical_orders_id,
                inst_config,
                CCD_geom,
				flux_RE,
				err_RE,
				qual_RE);
    
    //my_error = cpl_image_save(*flux_RE, "ESPRESSO_SKY_REBINNED.fits",
    //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
    //             "espdr_smooth_flux failed to save image ESPRESSO_SKY_REBINNED.fits: %s",
    //             cpl_error_get_message_default(my_error));
    //espdr_msg("Image ESPRESSO_SKY_REBINNED.fits saved");
    
	espdr_msg("select and rebin orders and create rebin image finished ...");

	cpl_image_delete(scaled_flux);
	cpl_image_delete(scaled_error);
	cpl_image_delete(scaled_quality);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Scale the fulx of the sky fibre according to the science fibre
 @param     s2d_b_input_image_flux
 @param     s2d_b_input_image_error
 @param     s2d_b_input_image_quality_map
 @param     rel_eff
 @param     s2d_a_image_flux_RE (returned)
 @param     s2d_a_image_error_RE (returned)
 @param     s2d_a_image_quality_map_RE (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_scale_sky(cpl_image *s2d_b_input_image_flux,
                               cpl_image *s2d_b_input_image_error,
                               cpl_image *s2d_b_input_image_quality_map,
                               cpl_image *rel_eff,
                               cpl_image **scaled_image_flux_RE,
                               cpl_image **scaled_image_error_RE,
                               cpl_image **scaled_image_quality_map_RE) {
    
#if 0
    double *flux_b_data = cpl_image_get_data_double(s2d_b_input_image_flux);
    double *rel_eff_data = cpl_image_get_data_double(rel_eff);
    int width = cpl_image_get_size_x(s2d_b_input_image_flux);
    int orders_nb = cpl_image_get_size_y(s2d_b_input_image_flux);
    
    int index = 0;
    for (int order = 1; order <= orders_nb; order++) {
        for (int pxl = 1; pxl <= width; pxl++) {
            if (((order == 131) || (order == 132)) && (pxl > 3044) & (pxl < 3049)) {
                espdr_msg("---------> DSO: [%d, %d] flux: %f, rel_eff: %f, scaled: %f",
                          order, pxl, flux_b_data[index], rel_eff_data[index],
                          flux_b_data[index] / rel_eff_data[index]);
            }
            index++;
        }
    }
#endif
    
    if (s2d_b_input_image_flux == NULL) {
        espdr_msg("s2d_b_input_image_flux is NULL");
    } else {
        espdr_msg("s2d_b_input_image_flux is NOT NULL");
    }
    if (s2d_b_input_image_error == NULL) {
        espdr_msg("s2d_b_input_image_error is NULL");
    } else {
        espdr_msg("s2d_b_input_image_error is NOT NULL");
    }
    if (s2d_b_input_image_quality_map == NULL) {
        espdr_msg("s2d_b_input_image_qual is NULL");
    } else {
        espdr_msg("s2d_b_input_image_qual is NOT NULL");
    }
    if (rel_eff == NULL) {
        espdr_msg("rel_eff is NULL");
    } else {
        espdr_msg("rel_eff is NOT NULL");
    }

    *scaled_image_flux_RE = cpl_image_divide_create(s2d_b_input_image_flux,
                                                    rel_eff);
    
    *scaled_image_error_RE = cpl_image_divide_create(s2d_b_input_image_error,
                                                     rel_eff);
    
    *scaled_image_quality_map_RE = cpl_image_duplicate(s2d_b_input_image_quality_map);
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief    Select the corresponding orders/slices between the two fibres for rebinning
 @param     scaled_flux
 @param     scaled_error
 @param     scaled_quality
 @param     wave_matrix
 @param     physical_orders_id
 @param     inst_config
 @param     CCD_geom
 @param     rebined_image_RE
 @param     error_image_RE
 @param     qual_image_RE
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_select_orders_and_create_rebin_image(
				cpl_image *scaled_flux,
				cpl_image *scaled_error,
				cpl_image *scaled_quality,
				cpl_image **wave_matrix,
                cpl_image **dll_matrix,
                int **physical_orders_id,
                espdr_inst_config *inst_config,
                espdr_CCD_geometry *CCD_geom,
				cpl_image **rebined_image_RE,
				cpl_image **error_image_RE,
				cpl_image **qual_image_RE) {

	cpl_error_code my_error;

	int order_a = 0;
	int order_b = 0;
    int last_order_b_matched = -1;
    int ext_a;
    int ext_b;
	bool found = false;
    int fibre_A = 0;
    int fibre_B = 1;
	cpl_image *order_rebined;
	cpl_image *order_error;
	cpl_image *order_quality;

	cpl_image *flux_b, *error_b, *quality_b, *wave_a, *wave_b, *dll_a, *dll_b;
	cpl_image *rebin_image;
	cpl_image *error_image;
	cpl_image *qual_image;

	cpl_image *flux_b_0;
	cpl_image *error_b_0;
	cpl_image *quality_b_0;

	int sz_order_x = cpl_image_get_size_x(scaled_flux);
    int orders_nb_a = cpl_image_get_size_y(wave_matrix[fibre_A]);
    int orders_nb_b = cpl_image_get_size_y(wave_matrix[fibre_B]);

	rebin_image = cpl_image_new(sz_order_x, orders_nb_a, CPL_TYPE_DOUBLE);
	error_image = cpl_image_new(sz_order_x, orders_nb_a, CPL_TYPE_DOUBLE);
	qual_image = cpl_image_new(sz_order_x, orders_nb_a, CPL_TYPE_INT);

    for (order_a = 0; order_a < orders_nb_a; order_a++) {
        ext_a = espdr_get_ext_index_for_order(order_a+1, fibre_A,
                                              inst_config, CCD_geom);
        found = false;
        
        for (order_b = 0; order_b < orders_nb_b; order_b++) {
            ext_b = espdr_get_ext_index_for_order(order_b+1, fibre_B,
                                                  inst_config, CCD_geom);
            
            if ((physical_orders_id[fibre_A][order_a] == physical_orders_id[fibre_B][order_b]) &&
                (ext_a == ext_b) && (!found) && (order_b > last_order_b_matched)) {
                
                last_order_b_matched = order_b; // in order not to much twice the same order from fibre B
                found = true;                   // once order found, we don't look for another one
                
                /* order in A & B fibres ==> rebin */
                
                //espdr_msg("ASE DEBUG : order is in A and B : phys_id[%d][%d]==%d [%d][%d]==%d"
                //" wave_%d size x=%d y=%d wave_%d size x=%d y=%d",
                //fibre,order_a,physical_orders_id[fibre][order_a],fibre+1,order_b,
                //physical_orders_id[fibre+1][order_b],
                //fibre,
                //cpl_image_get_size_x(wave_matrix[fibre]),
                //cpl_image_get_size_y(wave_matrix[fibre]),
                //fibre+1,
                //cpl_image_get_size_x(wave_matrix[fibre+1]),
                //cpl_image_get_size_y(wave_matrix[fibre+1]));
                
                //espdr_msg("Rebinning order A %d with order B %d",
                //          order_a, order_b);
                
                flux_b = cpl_image_extract(scaled_flux, 1, order_b+1, sz_order_x, order_b+1);
                error_b = cpl_image_extract(scaled_error, 1, order_b+1, sz_order_x, order_b+1);
                quality_b = cpl_image_extract(scaled_quality, 1, order_b+1, sz_order_x, order_b+1);
                wave_a = cpl_image_extract(wave_matrix[fibre_A], 1, order_a+1, sz_order_x, order_a+1);
                wave_b = cpl_image_extract(wave_matrix[fibre_B], 1, order_b+1, sz_order_x, order_b+1);
                dll_a = cpl_image_extract(dll_matrix[fibre_A], 1, order_a+1, sz_order_x, order_a+1);
                dll_b = cpl_image_extract(dll_matrix[fibre_B], 1, order_b+1, sz_order_x, order_b+1);
                
                my_error = espdr_rebin(flux_b,
                                       error_b,
                                       quality_b,
                                       wave_b,         // wavelength matrix B
                                       dll_b,
                                       wave_a,         // wavelength matrix A
                                       dll_a,
                                       &order_rebined,
                                       &order_error,
                                       &order_quality);
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_rebin failed");
                
                my_error = cpl_image_copy(rebin_image, order_rebined, 1, order_a+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_copy with rebin_image failed: %s",
                             cpl_error_get_message_default(my_error));
                
                my_error = cpl_image_copy(error_image, order_error, 1, order_a+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_copy with error_image failed: %s",
                             cpl_error_get_message_default(my_error));
                
                my_error = cpl_image_copy(qual_image, order_quality, 1, order_a+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_copy with qual_image failed: %s",
                             cpl_error_get_message_default(my_error));
                
                cpl_image_delete(flux_b);
                cpl_image_delete(error_b);
                cpl_image_delete(quality_b);
                cpl_image_delete(wave_a);
                cpl_image_delete(wave_b);
                
                cpl_image_delete(order_rebined);
                cpl_image_delete(order_error);
                cpl_image_delete(order_quality);
                
            }
        }
        
        if (!found) { /* order in A, but not in B fibre ==> unchanged */
            
            //espdr_msg("ASE DEBUG : order is in A, NOT in B : phys_id[%d][%d]==%d [%d][%d]==%d",
            //fibre,order_a,physical_orders_id[fibre][order_a],fibre+1,order_b,
            //physical_orders_id[fibre+1][order_b]);
            
            flux_b_0 = cpl_image_new(sz_order_x, 1, CPL_TYPE_DOUBLE);
            error_b_0 = cpl_image_new(sz_order_x, 1, CPL_TYPE_DOUBLE);
            quality_b_0 = cpl_image_new(sz_order_x, 1, CPL_TYPE_INT);
            
            my_error = cpl_image_copy(rebin_image, flux_b_0, 1, order_a+1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_copy with prescan failed: %s",
                         cpl_error_get_message_default(my_error));
            
            my_error = cpl_image_copy(error_image, error_b_0, 1, order_a+1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_copy with prescan failed: %s",
                         cpl_error_get_message_default(my_error));
            
            my_error = cpl_image_copy(qual_image, quality_b_0, 1, order_a+1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_copy with prescan failed: %s",
                         cpl_error_get_message_default(my_error));
            
            cpl_image_delete(flux_b_0);
            cpl_image_delete(error_b_0);
            cpl_image_delete(quality_b_0);
        }
    }
    
    *rebined_image_RE = cpl_image_duplicate(rebin_image);
    *error_image_RE = cpl_image_duplicate(error_image);
    *qual_image_RE = cpl_image_duplicate(qual_image);
    
    cpl_image_delete(rebin_image);
    cpl_image_delete(error_image);
    cpl_image_delete(qual_image);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Subtract the sky
 @param     inst_config
 @param     sky_subtraction_method
 @param     flux_obj
 @param     error_obj
 @param     quality_obj
 @param     flux_sky
 @param     error_sky
 @param     quality_sky
 @param     flux_RE
 @param     error_RE
 @param     quality_RE
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_subtract_sky(espdr_inst_config *inst_config,
                                  char *sky_subtraction_method,
                                  int slidingbox_size,
                                  cpl_image *flux_obj,
                                  cpl_image *error_obj,
                                  cpl_image *quality_obj,
                                  cpl_image *flux_sky,
                                  cpl_image *error_sky,
                                  cpl_image *quality_sky,
                                  cpl_image **flux_RE,
                                  cpl_image **error_RE,
                                  cpl_image **quality_RE /*science fibre*/) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int pixels, orders, pix;
    cpl_image *smoothed_sky = NULL;
    
    espdr_msg("The sky subtraction method is "ANSI_COLOR_GREEN"%s"ANSI_COLOR_RESET,
              sky_subtraction_method);
    if (strcmp(sky_subtraction_method, "smoothed") == 0) {
        espdr_msg("With the sliding box half size = %d", slidingbox_size);
    }
    
    pixels = cpl_image_get_size_x(flux_obj);
    orders = cpl_image_get_size_y(flux_obj);
    
    if (strcmp(sky_subtraction_method, "smoothed") == 0) {
        smoothed_sky = cpl_image_new(pixels, orders, CPL_TYPE_DOUBLE);
        my_error = espdr_smooth_flux(flux_sky, slidingbox_size, smoothed_sky);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_smooth_flux failed : %s",
                     cpl_error_get_message_default(my_error));
        espdr_msg("Smoothing done");
        
        *flux_RE = cpl_image_subtract_create(flux_obj, smoothed_sky);
        *error_RE = cpl_image_duplicate(error_obj);
        *quality_RE = cpl_image_duplicate(quality_obj);
        
        cpl_image_delete(smoothed_sky);
        
    } else {
        *flux_RE = cpl_image_subtract_create(flux_obj, flux_sky);
        
        cpl_image *error_obj2 = cpl_image_power_create(error_obj, 2.0);
        cpl_image *error_sky2 = cpl_image_power_create(error_sky, 2.0);
        *error_RE = cpl_image_add_create(error_obj2, error_sky2);
        
        cpl_image_delete(error_obj2);
        cpl_image_delete(error_sky2);
        
        my_error = cpl_image_power(*error_RE, 0.5);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_power failed : %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = cpl_image_or(*quality_RE, quality_obj, quality_sky);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_or failed : %s",
                     cpl_error_get_message_default(my_error));
    }
    
#if 0
    double *err_data = cpl_image_get_data_double(*error_RE);
    for (order = 0; order < orders; order++){
        for (pix = 0; pix < pixels; pix++){
            k = pixels * order + pix;
            
            err_data[k] = sqrt(pow(cpl_image_get(error_obj,pix+1,order+1,&pir),2.0)+
                               pow(cpl_image_get(error_sky,pix+1,order+1,&pir),2.0));
            
        }
    }
#endif
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Smooth the flux via sliding median
 @param    flux     flux to be smoothed
 @param    flux_RE  smoothed flux (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_smooth_flux(cpl_image *flux,
                                 int slidingbox_size,
                                 cpl_image *flux_RE) {
    
    int nx, ny, order, i, index, imin, imax;
    cpl_image *image_box = NULL;
    cpl_image *flux_order = NULL;
    double *flux_order_data = NULL;
    double *flux_RE_data = cpl_image_get_data_double(flux_RE);
    
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    //espdr_msg("Image sixe %dx%d", nx, ny);
    
    // Smoothing via the running mediane
    index = 0;
    for (order = 1; order <= ny; order++) {
        //espdr_msg("Smoothing order %d", order);
        flux_order = cpl_image_extract(flux, 1, order, nx, order);
        flux_order_data = cpl_image_get_data_double(flux_order);
        imin = 0;
        imax = nx - 1;
        while (flux_order_data[imin] == 0.0) imin++;
        while (flux_order_data[imax] == 0.0) imax--;
        
        //espdr_msg("imin: %d imax: %d", imin, imax);
        
        for (i = 0; i < imin; i++) {
            flux_RE_data[index] = 0.0;
            index++;
        }
        
        for (i = imin; i < imin + slidingbox_size; i++) {
            image_box = cpl_image_extract(flux_order, imin+1, 1, 2*i - imin+1, 1);
            flux_RE_data[index] = cpl_image_get_median(image_box);
            cpl_image_delete(image_box);
            index++;
        }
        
        for (i = imin + slidingbox_size; i <= imax - slidingbox_size; i++) {
            image_box = cpl_image_extract(flux_order, i - slidingbox_size+1, 1,
                                          i + slidingbox_size+1, 1);
            flux_RE_data[index] = cpl_image_get_median(image_box);
            cpl_image_delete(image_box);
            index++;
        }
        
        for (i = imax - slidingbox_size + 1; i <= imax; i++) {
            image_box = cpl_image_extract(flux_order, 2*i - imax+1, 1, imax+1, 1);
            flux_RE_data[index] = cpl_image_get_median(image_box);
            cpl_image_delete(image_box);
            index++;
        }
        
        for (i = imax + 1; i < nx; i++) {
            flux_RE_data[index] = 0.0;
            index++;
        }
        cpl_image_delete(flux_order);
    }
    
#if 0
    cpl_error_code my_error = CPL_ERROR_NONE;
    my_error = cpl_image_save(flux_RE, "ESPRESSO_FLUX_SMOOTHED.fits",
                              CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_smooth_flux failed to save image ESPRESSO_FLUX_SMOOTHED.fits: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("Image ESPRESSO_FLUX_SMOOTHED.fits saved");
#endif
    
    return cpl_error_get_code();
}




/*----------------------------------------------------------------------------*/
/**
 @brief    Make S1D wavelength grid
 @param    log_lambda_0     wavelength scale
 @param    delta_log_lambda delat wavelength
 @param    grid_length      langth of the grid
 @param    s1d_wavelength_RE    returned wavelengt grid
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_s1d_wavelenght_grid(double log_lambda_0,
                                         double delta_log_lambda,
                                         int grid_length,
                                         cpl_image **s1d_wavelenght_RE) {
    
    int step_count;
    double *s1d_wavelenght_RE_data = cpl_image_get_data_double(*s1d_wavelenght_RE);
    
    for (int i = 0; i < grid_length; i++) s1d_wavelenght_RE_data[i] = 0.0;
    
    for (step_count=0; step_count<grid_length; step_count++){
        s1d_wavelenght_RE_data[step_count] = log_lambda_0 +
                                    (double )step_count*(delta_log_lambda);
    }
    
    return cpl_error_get_code();
}




/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_shift_wavelength_solution(double berv_factor,
                                               cpl_image *wave_matrix,
                                               cpl_image *dll_matrix,
                                               cpl_image **wave_matrix_shifted_RE,
                                               cpl_image **dll_matrix_shifted_RE) {
    
#if 0
    double LIGHT_SPEED_SQ = LIGHT_SPEED * LIGHT_SPEED;
    
    //espdr_msg("BERV: %e", berv);
    //double berv_factor_simple = (1. + 1.55e-8) * (1. + berv / 299792.458);
    //espdr_msg("berv_factor (simple): %.15f", berv_factor_simple);
    //espdr_msg("berv_factor (simple) [m/s]: %f", (berv_factor_simple - 1.0) * LIGHT_SPEED);

    sed = sed * 1000.0; // km -> m
    vtot = vtot * 1000.0; // km -> m
    double phi = GRAVITATIONAL_CONSTANT * (M_SUN/sed + M_EARTH/R_EARTH);
    double berv_factor = (1.0/(1.0 - phi/LIGHT_SPEED_SQ - vtot*vtot/2.0/LIGHT_SPEED_SQ))*
                        (1.0 + berv/(LIGHT_SPEED/1000.0));
    //espdr_msg("berv_factor (complete): %.15f", berv_factor);
    //espdr_msg("berv_factor (complete) [m/s]: %f", (berv_factor - 1.0) * LIGHT_SPEED);
#endif
    
    if (wave_matrix != NULL) {
        *wave_matrix_shifted_RE = cpl_image_multiply_scalar_create(wave_matrix,
                                                                   berv_factor);
    }
    
    if (dll_matrix != NULL) {
        *dll_matrix_shifted_RE = cpl_image_multiply_scalar_create(dll_matrix,
                                                                  berv_factor);
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_rebin_and_merge_orders(cpl_image *s1d_grid,
                                            cpl_image *s1d_grid_air,
                                            cpl_image *wave_s2d,
                                            cpl_image *dll_s2d,
                                            cpl_image *flux_s2d,
                                            cpl_image *error_s2d,
                                            cpl_image *qual_s2d,
                                            cpl_table **s1d_RE) {
    
    cpl_error_code my_error;
    
    int nx = cpl_image_get_size_x(flux_s2d);
    
    int s1d_length = cpl_image_get_size_x(s1d_grid_air);
    
    cpl_image *flux_corr, *error_corr;
    cpl_image *quality_corr;
    cpl_image *s1d_rebined;
    cpl_image *s1d_error;
    cpl_image *s1d_quality;
    cpl_image *wave_s2d_order;
    cpl_image *dll_s2d_order;
    cpl_image *dll_s1d_grid;

    cpl_image *intermediate_rebin_flux;
    cpl_image *intermediate_rebin_error;
    cpl_image *intermediate_rebin_quality;
    
    int orders_nr = cpl_image_get_size_y(flux_s2d);
    
    intermediate_rebin_flux = cpl_image_new(s1d_length, orders_nr,
                                            cpl_image_get_type(flux_s2d));
    intermediate_rebin_error = cpl_image_new(s1d_length, orders_nr,
                                             cpl_image_get_type(error_s2d));
    intermediate_rebin_quality = cpl_image_new(s1d_length, orders_nr,
                                               cpl_image_get_type(qual_s2d));
    
    dll_s1d_grid = cpl_image_new(s1d_length, 1, CPL_TYPE_DOUBLE);
    double *s1d_grid_air_data = cpl_image_get_data_double(s1d_grid_air);
    double *s1d_grid_data = cpl_image_get_data_double(s1d_grid);
    double *dll_s1d_grid_data = cpl_image_get_data_double(dll_s1d_grid);
    
    dll_s1d_grid_data[0] = s1d_grid_data[1] - s1d_grid_data[0];
    for (int i = 1; i < s1d_length; i++) {
        dll_s1d_grid_data[i] = s1d_grid_data[i] - s1d_grid_data[i-1];
    }
    
    espdr_msg("rebin s1d process ...");
    
    for (int order = 0; order < orders_nr; order++) {
        wave_s2d_order = cpl_image_extract(wave_s2d, 1, order+1, nx, order+1);
        dll_s2d_order = cpl_image_extract(dll_s2d, 1, order+1, nx, order+1);
        flux_corr = cpl_image_extract(flux_s2d, 1, order+1, nx, order+1);
        error_corr = cpl_image_extract(error_s2d, 1, order+1, nx, order+1);
        quality_corr = cpl_image_extract(qual_s2d, 1, order+1, nx, order+1);
        
        my_error = espdr_rebin(flux_corr,
                               error_corr,
                               quality_corr,
                               wave_s2d_order,
                               dll_s2d_order,
                               s1d_grid, /*S1D grid in vacuum*/
                               dll_s1d_grid,
                               &s1d_rebined,
                               &s1d_error,
                               &s1d_quality);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_rebin failed");
        
        my_error = cpl_image_copy(intermediate_rebin_flux, s1d_rebined, 1, order+1);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_copy failed: %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = cpl_image_copy(intermediate_rebin_error, s1d_error, 1, order+1);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_copy failed: %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = cpl_image_copy(intermediate_rebin_quality, s1d_quality, 1, order+1);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_copy failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_image_delete(wave_s2d_order);
        cpl_image_delete(dll_s2d_order);
        cpl_image_delete(flux_corr);
        cpl_image_delete(error_corr);
        cpl_image_delete(quality_corr);
        
        cpl_image_delete(s1d_rebined);
        cpl_image_delete(s1d_error);
        cpl_image_delete(s1d_quality);
    }
    
    espdr_msg("rebin s1d process finished !");
    
    *s1d_RE = cpl_table_new(s1d_length);
    cpl_table_new_column(*s1d_RE,"wavelength",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_RE,"wavelength_air",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_RE,"flux",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_RE,"error",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_RE,"quality",CPL_TYPE_INT);
    my_error = cpl_table_set_column_unit(*s1d_RE,
                                         "wavelength",
                                         "angstrom");
    my_error = cpl_table_set_column_unit(*s1d_RE,
                                         "wavelength_air",
                                         "angstrom");
    my_error = cpl_table_set_column_unit(*s1d_RE,
                                         "flux",
                                         "e-");
                                         // spectrum_flux_unit
    my_error = cpl_table_set_column_unit(*s1d_RE,
                                         "error",
                                         "e-");
                                         // spectrum_flux_unit
    my_error = cpl_table_set_column_unit(*s1d_RE,
                                         "quality",
                                         "");
    
    espdr_msg("merging orders fibre ...");
    
    for (int pxl = 0; pxl < s1d_length; pxl++) {
        cpl_table_set_double(*s1d_RE,
                             "wavelength_air", pxl,
                             s1d_grid_air_data[pxl]);
        cpl_table_set_double(*s1d_RE,
                             "wavelength", pxl,
                             s1d_grid_data[pxl]);
    }
    
    my_error = espdr_merge_orders(intermediate_rebin_flux,
                                  intermediate_rebin_error,
                                  intermediate_rebin_quality,
                                  s1d_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_merge_orders failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("merging orders finished !");
    
    /* cleaning some memory */
    cpl_image_delete(intermediate_rebin_flux);
    cpl_image_delete(intermediate_rebin_error);
    cpl_image_delete(intermediate_rebin_quality);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Rebin the given image with respect to the two wave matrices
 @param     flux
 @param     error
 @param     quality
 @param     wave
 @param     wave_rebinned
 @param     flux_RE
 @param     error_RE
 @param     quality_RE
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_rebin(cpl_image *flux,
                           cpl_image *error,
                           cpl_image *quality,
                           cpl_image *wave,
                           cpl_image *dll,
                           cpl_image *wave_rebinned,
                           cpl_image *dll_rebinned,
                           cpl_image **flux_RE,
                           cpl_image **error_RE,
                           cpl_image **quality_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int nx = cpl_image_get_size_x(flux);
    
    int nx_rebinned = cpl_image_get_size_x(wave_rebinned);
    
    double *wave_data;
    double *dll_data;
    double *wave_data_rebinned;
    double *dll_data_rebinned;
    double *flux_data;
    
    double *error_data = cpl_image_get_data_double(error);
    int *quality_data = cpl_image_get_data_int(quality);
    
    double *flux_RE_data;
    double *error_RE_data;
    int *quality_RE_data;
    
    flux_RE_data = (double *)cpl_calloc(nx_rebinned,sizeof(double));
    error_RE_data = (double *)cpl_calloc(nx_rebinned,sizeof(double));
    quality_RE_data = (int *)cpl_calloc(nx_rebinned,sizeof(int));
    
    double *xx = (double *)cpl_calloc(nx_rebinned,sizeof(double));
    double *ll = (double *)cpl_calloc(nx,sizeof(double));
    
    for (int j=0; j < nx_rebinned; j++) {
        quality_RE_data[j] = OTHER_BAD_PIXEL;
    }
    
    wave_data = cpl_image_get_data_double(wave);
    dll_data = cpl_image_get_data_double(dll);
    
    /*
    ouble dy = 0.0;
    for (int j = 0; j < nx-1; j++) {
        //dy = wave_data[j+1] - wave_data[j];
        //ll[j] = wave_data[j] - (dy/2.0);
    }
    ll[nx-1] = wave_data[nx-1] - ((wave_data[nx-1] - wave_data[nx-2])/2.0);
    */
    
    for (int j = 0; j < nx; j++) {
        ll[j] = wave_data[j] - dll_data[j]/2.0;
    }
    
    wave_data_rebinned = cpl_image_get_data_double(wave_rebinned);
    dll_data_rebinned = cpl_image_get_data_double(dll_rebinned);

    /*
    double dx = 0.0;
    for (int j=0; j < nx_rebinned-1; j++) {
        dx = wave_data_rebinned[j+1] - wave_data_rebinned[j];
        xx[j] = wave_data_rebinned[j] - (dx/2.0);
    }
    
    xx[nx_rebinned-1] = wave_data_rebinned[nx_rebinned-1] -
    ((wave_data_rebinned[nx_rebinned-1] -
      wave_data_rebinned[nx_rebinned-2])/2.0);
    */
    
    for (int j = 0; j < nx_rebinned; j++) {
        xx[j] = wave_data_rebinned[j] - dll_data_rebinned[j]/2.0;
    }
    
    int chunk_length = 0;
    int index_start = 0;
    int index_end = 0;
    int istart,iend;
    
    istart = 0;
    iend = nx-1;
    
    while ((istart < (nx-1)) && (quality_data[istart]==OTHER_BAD_PIXEL)) istart++;
    while ((iend > 0) && (quality_data[iend]==OTHER_BAD_PIXEL)) iend--;
    
    index_start = 0;
    index_end = nx_rebinned-1;
    
    while ((index_start < (nx_rebinned-1)) && (xx[index_start] < ll[istart])) index_start++;
    while ((index_end > 0) && (xx[index_end] > ll[iend])) index_end--;
    
    chunk_length = index_end - index_start;
    
    if(chunk_length<=0) {
        espdr_msg("ERROR: wavelength out of bounds. Exit program.");
        exit(EXIT_FAILURE);
    }
    
    double *yy = (double *)cpl_calloc(iend-istart+1,sizeof(double));
    double *subll = (double *)cpl_calloc(iend-istart+1,sizeof(double));
    
    for (int j=0; j < iend-istart+1; j++) {
        subll[j] = ll[istart+j];
        if ((j > 0) && (subll[j] < subll[j-1])) {
            espdr_msg_warning("Wavelengths going down!!!: %f -> %f",
                              subll[j-1], subll[j]);
        }
    }
    
    double *subxx = (double *)cpl_calloc(chunk_length,sizeof(double));
    
    int ind = 0;
    for (int i = index_start; i < index_end; i++) {
        subxx[ind] = xx[i];
        ind++;
    }
    
    flux_data = cpl_image_get_data_double(flux);
    
    yy[0] = 0.;
    for (int j=1; j < iend-istart+1; j++) {
        yy[j] = yy[j-1]+(flux_data[istart+j-1]);
    }
    
    double *y2 = (double *) cpl_calloc(chunk_length,sizeof(double));
    
    my_error = espdr_spline_interpolation(subll,
                                          yy,
                                          iend-istart+1,
                                          subxx,
                                          y2,
                                          chunk_length);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_spline_interpolation failed: %s",
                 cpl_error_get_message_default(my_error));
    
    double *yi = (double *)cpl_calloc(chunk_length-1,sizeof(double));
    for (int j=0; j < chunk_length-1; j++) yi[j] = y2[j+1] - y2[j];
    
    int ind_x = 0;
    int i = istart;
    int n = 0;
    for (int j = index_start; j < index_end-1; j++) {
        flux_RE_data[j] = yi[ind_x];
        ind_x++;
        error_RE_data[j] = 0.;
        quality_RE_data[j] = 0;
        n = 0;
        do {
            error_RE_data[j] += error_data[i];
            quality_RE_data[j] = quality_RE_data[j] | quality_data[i];
            n++;
            i++;
        } while ((i <= iend) && (ll[i] < xx[j+1]));
        i--;
        error_RE_data[j] = error_RE_data[j] / n;
    }
    
    cpl_image *flux_reb;
    cpl_image *error_reb;
    cpl_image *qual_reb;
    
    flux_reb = cpl_image_wrap_double(nx_rebinned,1,flux_RE_data);
    error_reb = cpl_image_wrap_double(nx_rebinned,1,error_RE_data);
    qual_reb = cpl_image_wrap_int(nx_rebinned,1,quality_RE_data);
    
    *error_RE = cpl_image_duplicate(error_reb);
    cpl_image_unwrap(error_reb);
    
    *quality_RE = cpl_image_duplicate(qual_reb);
    cpl_image_unwrap(qual_reb);
    
    *flux_RE = cpl_image_duplicate(flux_reb);
    cpl_image_unwrap(flux_reb);
    
    cpl_free(xx);
    cpl_free(error_RE_data);
    cpl_free(quality_RE_data);
    cpl_free(yy);
    cpl_free(yi);
    cpl_free(flux_RE_data);
    cpl_free(ll);
    cpl_free(subll);
    cpl_free(subxx);
    cpl_free(y2);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Merge rebinned orders
 @param    flux to be merged
 @param    error to be merged
 @param    quality to be merged
 @param    resulting S1D
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_merge_orders(cpl_image *intermediate_rebin_flux,
                                  cpl_image *intermediate_rebin_error,
                                  cpl_image *intermediate_rebin_quality,
                                  cpl_table **s1d_table_merged_RE) {
    
    int nx = cpl_image_get_size_x(intermediate_rebin_flux);
    int ny = cpl_image_get_size_y(intermediate_rebin_flux);
    
    cpl_image *intermediate_flux_order;
    cpl_image *intermediate_error_order;
    cpl_image *intermediate_quality_order;
    
    double *int_flux_data;
    double *int_error_data;
    int *int_quality_data;
    
    double *s1d_table_flux_RE_data;
    double *s1d_table_error_RE_data;
    int *s1d_table_quality_RE_data;
    
    s1d_table_flux_RE_data =
    cpl_table_get_data_double(*s1d_table_merged_RE,"flux");
    
    s1d_table_error_RE_data =
    cpl_table_get_data_double(*s1d_table_merged_RE,"error");
    
    s1d_table_quality_RE_data =
    cpl_table_get_data_int(*s1d_table_merged_RE,"quality");
    
    for (int pixels = 0; pixels < nx; pixels++) {
        
        s1d_table_flux_RE_data[pixels] = 0.0;
        s1d_table_error_RE_data[pixels] = 0.0;
        s1d_table_quality_RE_data[pixels] = OTHER_BAD_PIXEL;
        
        intermediate_flux_order = cpl_image_extract(intermediate_rebin_flux,
                                                    pixels+1, 1, pixels+1, ny);
        
        int_flux_data = cpl_image_get_data_double(intermediate_flux_order);
        
        intermediate_error_order = cpl_image_extract(intermediate_rebin_error,
                                                     pixels+1, 1, pixels+1, ny);
        
        int_error_data = cpl_image_get_data_double(intermediate_error_order);
        
        intermediate_quality_order = cpl_image_extract(intermediate_rebin_quality,
                                                       pixels+1, 1, pixels+1, ny);
        
        int_quality_data = cpl_image_get_data_int(intermediate_quality_order);
        
        bool qual_zero_present = FALSE;
        bool qual_all_other_bad_pixel = TRUE;
        for (int order = 0; order < ny; order++) {
            if (int_quality_data[order] == 0) {
                qual_zero_present = TRUE;
            }
            if (int_quality_data[order] != OTHER_BAD_PIXEL) {
                qual_all_other_bad_pixel = FALSE;
            }
        }
        
        if (qual_all_other_bad_pixel) {
            // All the pixels have flag OTHER_BAD_PIXEL and flux = 0.0
            s1d_table_flux_RE_data[pixels] = 0.0;
            s1d_table_error_RE_data[pixels] = 0.0;
            s1d_table_quality_RE_data[pixels] = OTHER_BAD_PIXEL;
        } else {
            if (!qual_zero_present) {
                // All the pixels are bad or hot or saturated or other bad etc
                
                int flag_result = 0;
                double max_error = 0.0;
                int flux_nb = 0;
                
                for (int order = 0; order < ny; order++) {
                    
                    if (int_flux_data[order] != 0.0) {
                        s1d_table_flux_RE_data[pixels] += int_flux_data[order];
                        flux_nb++;
                    }
                    
                    if (int_error_data[order] > max_error) {
                        max_error = int_error_data[order];
                    }
                    
                    if (int_quality_data[order] != OTHER_BAD_PIXEL) {
                        flag_result = flag_result | int_quality_data[order];
                    }
                }
            
                if (flux_nb == 0) {
                    s1d_table_flux_RE_data[pixels] = 0.0;
                } else {
                    s1d_table_flux_RE_data[pixels] = s1d_table_flux_RE_data[pixels]/flux_nb;
                }
                s1d_table_error_RE_data[pixels] = max_error;
                s1d_table_quality_RE_data[pixels] = flag_result;
                
            } else {
                // There is at least one valid pixel with qual = 0
                double sum_ele = 0.0;
                double sum_weight = 0.0;
                for (int order = 0; order < ny; order++) {
                    if ((int_quality_data[order] == 0) &&
                        (int_error_data[order] != 0.0)) {
                        sum_weight = sum_weight + 1.0
                        / int_error_data[order] / int_error_data[order];
                        sum_ele = sum_ele + int_flux_data[order]
                        / int_error_data[order] / int_error_data[order];
                    }
                }
                if (sum_weight != 0.) {
                    s1d_table_flux_RE_data[pixels] = sum_ele/sum_weight;
                    s1d_table_error_RE_data[pixels] = 1.0/sqrt(sum_weight);
                }
                s1d_table_quality_RE_data[pixels] = 0;
            }
        }
        
        cpl_image_delete(intermediate_flux_order);
        cpl_image_delete(intermediate_error_order);
        cpl_image_delete(intermediate_quality_order);
        
        cpl_table_set_double(*s1d_table_merged_RE,
                             "flux", pixels,
                             s1d_table_flux_RE_data[pixels]);
        cpl_table_set_double(*s1d_table_merged_RE,
                             "error", pixels,
                             s1d_table_error_RE_data[pixels]);
        cpl_table_set_int(*s1d_table_merged_RE,
                          "quality", pixels,
                          s1d_table_quality_RE_data[pixels]);
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_calibrate_flux(double airmass,
                                    double seeing,
                                    double slit_loss,
                                    int slit_loss_flag,
                                    espdr_inst_config *inst_config,
                                    cpl_table *s1d_table,
                                    cpl_table *ext_table,
                                    cpl_table *abs_eff,
                                    cpl_table **s1d_fluxcal_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Starting calibrate_flux with airmass: %.2f, seeing: %.2f, slit_loss: %.2f (slit_loss_flag: %d)",
              airmass, seeing, slit_loss, slit_loss_flag);
    
    int s1d_length = cpl_table_get_nrow(s1d_table);
    double magnitude, ratio, fluxcal, errorcal;
    double wavelength, iq, slit;
    double fibre_size, vltsh_ll, vltsh_iq, inst_iq;

    *s1d_fluxcal_RE = cpl_table_new(s1d_length);
    cpl_table_new_column(*s1d_fluxcal_RE,
                         "wavelength", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_fluxcal_RE,
                         "flux_cal", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_fluxcal_RE,
                         "error_cal", CPL_TYPE_DOUBLE);
    my_error = cpl_table_set_column_unit(*s1d_fluxcal_RE,
                                         "wavelength",
                                         "angstrom");
    my_error = cpl_table_set_column_unit(*s1d_fluxcal_RE,
                                         "flux_cal", spectrum_flux_unit);
                                        
    my_error = cpl_table_set_column_unit(*s1d_fluxcal_RE, 
                                         "error_cal", spectrum_flux_unit);
                                         

    cpl_table *ext_interp;
    
    my_error = espdr_pre_extinction_table_interpolation(ext_table,
                                                        s1d_table,
                                                        &ext_interp);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_pre_extinction_table_interpolation failed : %s",
                 cpl_error_get_message_default(my_error));
    
    
    fibre_size = inst_config->fibre_size;
    vltsh_ll = inst_config->vltsh_wavelength;
    vltsh_iq = inst_config->vltsh_iq;
    inst_iq = inst_config->inst_iq;
    
    for (int j = 0; j < s1d_length; j++) {
        
        magnitude = cpl_table_get(ext_interp,"extinction_interp",j,NULL)*
                    (airmass); /* Airmass removed */
        
        ratio = pow(10, magnitude/2.5);
        
        wavelength = cpl_table_get(s1d_table,"wavelength", j, NULL);
        cpl_table_set_double(*s1d_fluxcal_RE,
                             "wavelength", j, wavelength);
        
        fluxcal = cpl_table_get(s1d_table, "flux", j, NULL) /
                    cpl_table_get(abs_eff, "efficiency_interpolated",
                                  j, NULL) * ratio;
        
        if (slit_loss <= 0.0) { // parameter slit_loss not set or set to a value <= 0.0 --> do not apply
            if (strcmp(inst_config->instrument, "NIRPS") == 0) {
                if (inst_config->fibre_size == 0.4) {
                    slit = 0.6;
                } else {
                    slit = 0.75;
                }
            } else {
                iq = sqrt(seeing*seeing - vltsh_iq*vltsh_iq) * pow((wavelength/vltsh_ll), -0.2);
                slit = 1.0 - exp(-0.53 * ((fibre_size * fibre_size)/(iq * iq + inst_iq * inst_iq)));
            }
        } else { // parameter slit_loss set to a value > 0.0 --> apply instead of computed slit
            slit = slit_loss;
        }
        
        if (slit_loss_flag == 1) {
            fluxcal = fluxcal/slit;
        }
        
        errorcal = cpl_table_get(s1d_table, "error", j,NULL) /
                    cpl_table_get(abs_eff, "efficiency_interpolated",
                                  j, NULL) * ratio;
        
        errorcal = errorcal/slit;
        
        cpl_table_set_double(*s1d_fluxcal_RE,
                             "flux_cal", j, fluxcal);
        
        cpl_table_set_double(*s1d_fluxcal_RE,
                             "error_cal", j, errorcal);
        
    }
    
    cpl_table_delete(ext_interp);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     compute the Radial Velocity (RV) table
 @param     RV_range        RV range
 @param     RV_step         RV step
 @param     RV_center       RV center
 @param     RV_table_RE     RV table calculated (returned).
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_rv_table(double RV_range,
                                      double RV_step,
                                      double RV_center,
                                      cpl_array **RV_table_RE) {
    
    int RV_end = round(RV_range/RV_step);
    int RV_ini = - RV_end;
    double RV = 0.0;
    
    cpl_size length = RV_end - RV_ini +1;
    cpl_size array_index = 0;
    
    *RV_table_RE = cpl_array_new(length,CPL_TYPE_DOUBLE);
    
    for (int i = RV_ini; i <= RV_end; i++) {
        RV = RV_center + i*RV_step;
        cpl_array_set_double(*RV_table_RE,array_index,RV);
        array_index++;
    }
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_flux(cpl_image *s2d_blaze,
                                  cpl_image *wave_matrix,
                                  cpl_table *flux_template,
                                  cpl_table *mask_table,
                                  char *spec_type,
                                  cpl_image *s2d_blaze_flux_corr_RE,
                                  double *flux_corr_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;

    int i, j, k, imin, fit_nb;
    double flux_sum, corr_min, corr_max, corr_model;
    int order_nb = cpl_image_get_size_y(s2d_blaze);
    int order_len = cpl_image_get_size_x(s2d_blaze);
    int flux_len = 0;
    int poly_deg = 8;
    
    double *flux_ref = NULL;
    double *s2d_blaze_data = cpl_image_get_data_double(s2d_blaze);
    double *wave_data = cpl_image_get_data_double(wave_matrix);
    double *s2d_blaze_corr_data = cpl_image_get_data_double(s2d_blaze_flux_corr_RE);
    double *flux = (double *) cpl_calloc(order_nb, sizeof(double));
    double *llc = (double *) cpl_calloc(order_nb, sizeof(double));
    double *corr = (double *) cpl_calloc(order_nb, sizeof(double));
    double chisq = 0.0;
    
    for (i = 0; i < order_nb; i++) {
        for (j = 0; j < order_len; j++) {
            s2d_blaze_corr_data[i*order_len+j] = s2d_blaze_data[i*order_len+j];
        }
        flux_corr_RE[i] = 0.;
    }
        
    if ((strcmp(spec_type, "NONE") == 0) ||
        (strcmp(spec_type, "none") == 0)) {
        
        espdr_msg("Flux correction not performed: switched off by the user");
        
    } else {
        
        espdr_msg("Performing flux correction for spectral type: %s", spec_type);
        
        flux_len = cpl_table_get_nrow(flux_template);
        espdr_ensure(order_nb != flux_len, CPL_ERROR_INCOMPATIBLE_INPUT,
                     "Flux template length (%d) doesn't correspond to the number of orders in s2d - %d",
                     flux_len, order_nb);
        
        if (cpl_table_has_column(flux_template, spec_type)) {
            
            for (i = 0; i < order_nb; i++) {
                llc[i] = wave_data[i*order_len+order_len/2];
		flux_corr_RE[i] = 1.0;
            }
	    
	    imin = 0;
	    while ((imin < order_nb-3) && 
	           (llc[imin+1] < cpl_table_get(mask_table,"lambda",0,NULL)) && 
		   (llc[imin+2] < cpl_table_get(mask_table,"lambda",0,NULL))) imin++;
	    fit_nb = order_nb - imin;
	    poly_deg = (int) round( (double) poly_deg * (double) fit_nb / (double) order_nb);
	    espdr_msg("Discarding %i orders and using polynomial of degree %i", imin, poly_deg);
            
            flux_ref = cpl_table_get_data_double(flux_template, spec_type);
            flux_sum = 0.;
            corr_min = 1000.;
            corr_max = -1000.;
            
	    for (i = imin; i < order_nb; i++) {
                flux[i] = 0.;
                for (j = (order_len/4); j < (3*order_len/4); j++) {
                    flux[i] += s2d_blaze_data[i*order_len+j];
                }
                flux[i] = flux[i]/flux_ref[i];
                flux_sum += flux[i];
            }

            espdr_poly_data *p_data = (espdr_poly_data *) cpl_malloc(sizeof(espdr_poly_data));
            p_data->x = (double *) cpl_calloc(fit_nb, sizeof(double));
            p_data->y = (double *) cpl_calloc(fit_nb, sizeof(double));
            p_data->err = (double *) cpl_calloc(fit_nb, sizeof(double));
            p_data->n = fit_nb;
            double *fit = (double *) cpl_calloc(fit_nb, sizeof(double));
            double *coeffs = (double *) cpl_calloc(poly_deg, sizeof(double));
            double *coeffs_err = (double *) cpl_calloc(poly_deg, sizeof(double));

            for (i = imin; i < order_nb; i++) {
                corr[i] = flux[i]/flux_sum*fit_nb;
                p_data->x[i-imin] = llc[i];
                p_data->y[i-imin] = corr[i];
                p_data->err[i-imin] = 1.0;
            }
            
            my_error = espdr_fit_poly(p_data, poly_deg, fit, coeffs, coeffs_err, &chisq, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            for (i = imin; i < order_nb; i++) {
                if (fit[i-imin] < corr_min) corr_min = fit[i-imin];
                if (fit[i-imin] > corr_max) corr_max = fit[i-imin];
                flux_corr_RE[i] = fit[i-imin];
            }
            
            if ((corr_min > 0.25) && (corr_max < 3.0)) {
                    		
                for (i = imin; i < order_nb; i++) {
                    for (j = 0; j < order_len; j++) {
                        corr_model = 0.;
                        for (k = 0; k < poly_deg; k++) {
                            corr_model += coeffs[k]*pow(wave_data[i*order_len+j],k);
                        }
                        s2d_blaze_corr_data[i*order_len+j] = s2d_blaze_data[i*order_len+j]/corr_model;
                    }
                }
                espdr_msg("Flux correction performed with min/max values: %f/%f", corr_min, corr_max);
                
            } else {
                
                espdr_msg("Flux correction not performed: min/max values out of range (%f/%f)",
                          corr_min, corr_max);
            }
	    
	    cpl_free(p_data->x);
            cpl_free(p_data->y);
            cpl_free(p_data->err);
            cpl_free(p_data);
            cpl_free(fit);
            cpl_free(coeffs);
            cpl_free(coeffs_err);

        } else {
            
	    espdr_msg_warning(ANSI_COLOR_RED"Flux correction not performed: no flux template available for spectral type %s"ANSI_COLOR_RESET, spec_type);
	    
        }
    }
        
	cpl_free(flux);
	cpl_free(llc);
	cpl_free(corr);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Remove cosmics using the method of 4 bins histogram of flux
 @param     flux        flux to be cleaned
 @param     err         error on the flux
 @param     flux_RE     cleaned flux (returned)
 @param     cosmics_RE  nb of removed cosmics per order (returned)
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_cosmics_via_histogram(cpl_image *flux,
                                                  cpl_image *err,
                                                  cpl_image *flux_RE,
                                                  int *cosmics_RE) {
    
    int order, pixel, nx, ny, index;
    cpl_image *flux_order = NULL;
    cpl_image *flux_order_corr = NULL;
    cpl_image *err_order = NULL;
    double *flux_order_data = NULL;
    double *flux_corr_data = cpl_image_get_data_double(flux_RE);
    double *flux_order_corr_data = NULL;
    double *err_order_data = NULL;
    double min_flux, max_flux, bin, detect_limit, order_median;
    int histogram[4];
    int *low_cosmics_nb, *high_cosmics_nb;
    int total_cosmics_nb = 0;
    
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    //espdr_msg("Image size: %dx%d", nx, ny);
    low_cosmics_nb = (int *)cpl_calloc(ny, sizeof(int));
    high_cosmics_nb = (int *)cpl_calloc(ny, sizeof(int));
    
    index = 0;
    for (order = 1; order <= ny; order++) {
        //espdr_msg("Cosmics correction for order %d", order);
        flux_order = cpl_image_extract(flux, 1, order, nx, order);
        err_order = cpl_image_extract(err, 1, order, nx, order);
        flux_order_data = cpl_image_get_data_double(flux_order);
        err_order_data = cpl_image_get_data_double(err_order);
        flux_order_corr_data = (double *)cpl_calloc(nx, sizeof(double));
        
        // construct histogram
        //espdr_msg("Constructing histogram for low cosmics order %d", order);
        for (int i = 0; i < 4; i++) {
            histogram[i] = 0;
        }
        min_flux = cpl_image_get_min(flux_order);
        max_flux = cpl_image_get_max(flux_order);
        bin = (max_flux - min_flux) / 4;
        for (pixel = 0; pixel < nx; pixel++) {
            if (flux_order_data[pixel] < (min_flux + bin)) {
                histogram[0]++;
            } else {
                if (flux_order_data[pixel] < (min_flux + 2*bin)) {
                    histogram[1]++;
                } else {
                    if (flux_order_data[pixel] < (min_flux + 3*bin)) {
                        histogram[2]++;
                    } else {
                        histogram[3]++;
                    }
                }
            }
        }
        //espdr_msg("Low cosmics histogram: %d %d %d %d",
        //          histogram[0], histogram[1], histogram[2], histogram[3]);
        //espdr_msg("Low cosmics histogram limits: min = %f %f %f %f %f max = %f",
        //          min_flux, min_flux+bin, min_flux+2*bin,
        //          min_flux+3*bin, min_flux+4*bin, max_flux);
        for (pixel = 0; pixel < nx; pixel++) {
            if (order == -1) {
                if (flux_order_data[pixel] < (min_flux+bin)) {
                    espdr_msg("data[%d] = %f", pixel, flux_order_data[pixel]);
                } else {
                    if (flux_order_data[pixel] < (min_flux+2*bin)) {
                        espdr_msg("data[%d] = %f", pixel, flux_order_data[pixel]);
                    }
                }
            }
        }
        // Histogram constructed, eliminate low cosmics
        //espdr_msg("Eliminating low cosmics");
        detect_limit = min_flux;
        if ((histogram[1] == 0) && (histogram[0] > 0) &&
            ((histogram[2] + histogram[3]) > histogram[0])) {
            low_cosmics_nb[order-1] += histogram[0];
            detect_limit = min_flux + 2*bin;
        }
        for (pixel = 0; pixel < nx; pixel++) {
            if (flux_order_data[pixel] < detect_limit) {
                //flux_order_corr_data[pixel] = min_flux + 3*bin;
                flux_corr_data[index] = cpl_image_get_median(flux_order);
                //espdr_msg("Low cosmic correction order %d, pixel %d", order, pixel);
            } else {
                flux_order_corr_data[pixel] = flux_order_data[pixel];
            }
        }
        flux_order_corr = cpl_image_wrap_double(nx, 1, flux_order_corr_data);
        
        // Eliminate the negative cosmics, which were not removed yet
        order_median = cpl_image_get_median(flux_order_corr);
        cpl_image_unwrap(flux_order_corr);
        for (pixel = 0; pixel < nx; pixel++) {
            if (flux_order_corr_data[pixel] < -10.0 * err_order_data[pixel]) {
                flux_order_corr_data[pixel] = order_median;
            }
        }
        flux_order_corr = cpl_image_wrap_double(nx, 1, flux_order_corr_data);
        
        // reconstruct the histogram
        //espdr_msg("Constructing histogram for high cosmics order %d", order);
        for (int i = 0; i < 4; i++) {
            histogram[i] = 0;
        }
        min_flux = cpl_image_get_min(flux_order_corr);
        max_flux = cpl_image_get_max(flux_order_corr);
        bin = (max_flux - min_flux) / 4;
        for (pixel = 0; pixel < nx; pixel++) {
            if (flux_order_corr_data[pixel] < (min_flux + bin)) {
                histogram[0]++;
            } else {
                if (flux_order_corr_data[pixel] < (min_flux + 2*bin)) {
                    histogram[1]++;
                } else {
                    if (flux_order_corr_data[pixel] < (min_flux + 3*bin)) {
                        histogram[2]++;
                    } else {
                        histogram[3]++;
                    }
                }
            }
        }
        //espdr_msg("High cosmics histogram: %d %d %d %d",
        //          histogram[0], histogram[1], histogram[2], histogram[3]);
        //espdr_msg("High cosmics histogram limits: min = %f %f %f %f %f max = %f",
        //          min_flux, min_flux+bin, min_flux+2*bin,
        //          min_flux+3*bin, min_flux+4*bin, max_flux);
        // Histogram constructed, eliminate high cosmics
        //espdr_msg("Eliminating high cosmics");
        for (pixel = 0; pixel < nx; pixel++) {
            if (order == -1) {
                if (flux_order_data[pixel] > (min_flux+3*bin)) {
                    espdr_msg("data[%d] = %f", pixel, flux_order_data[pixel]);
                } else {
                    if (flux_order_data[pixel] > (min_flux+2*bin)) {
                        espdr_msg("data[%d] = %f", pixel, flux_order_data[pixel]);
                    }
                }
            }
        }
        detect_limit = max_flux;
        if ((histogram[2] == 0) && (histogram[3] > 0) &&
            ((histogram[0] + histogram[1]) > histogram[3])) {
            high_cosmics_nb[order-1] += histogram[3];
            detect_limit = min_flux + 2*bin;
        }
        for (pixel = 0; pixel < nx; pixel++) {
            if (flux_order_corr_data[pixel] > detect_limit) {
                //flux_corr_data[index] = min_flux + 0.5 * bin;
                flux_corr_data[index] = cpl_image_get_median(flux_order_corr);
                //espdr_msg("High cosmic correction order %d, pixel %d: %f",
                //          order, pixel, flux_corr_data[index]);
            } else {
                flux_corr_data[index] = flux_order_corr_data[pixel];
            }
            index++;
        }
        
        cpl_image_unwrap(flux_order_corr);
        cpl_free(flux_order_corr_data);
        cpl_image_delete(flux_order);
        cpl_image_delete(err_order);
    }
    
    total_cosmics_nb = 0;
    for (order = 0; order < ny; order++) {
        if ((low_cosmics_nb[order] != 0) || (high_cosmics_nb[order] != 0)) {
            //espdr_msg("Removed cosmics for order %d: low = %d, high = %d",
            //          order+1, low_cosmics_nb[order], high_cosmics_nb[order]);
            total_cosmics_nb += low_cosmics_nb[order] + high_cosmics_nb[order];
            cosmics_RE[order] = low_cosmics_nb[order] + high_cosmics_nb[order];
        }
    }
    espdr_msg("Total nb of cosmics removed: %d", total_cosmics_nb);
    cpl_free(low_cosmics_nb);
    cpl_free(high_cosmics_nb);
    
#if 0
    cpl_error_code my_error = CPL_ERROR_NONE;
    my_error = cpl_image_save(flux_RE, "ESPRESSO_FLUX_COSMICS_REMOVED.fits",
                              CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_ccf_remove_cosmics failed to save image ESPRESSO_FLUX_COSMICS_REMOVED.fits: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("Image ESPRESSO_FLUX_COSMICS_REMOVED.fits saved");
#endif
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Check the sci_red QC
 @param    CCD_geom         CCD geometry
 @param    exposure_time    exposure time
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_sci_red_QC(espdr_CCD_geometry *CCD_geom,
                                espdr_inst_config *inst_config,
                                espdr_qc_keywords *qc_kws,
                                char *fibre_src,
                                char *fibre_B_src,
                                double berv,
                                double bjd,
                                double bervmax,
                                double sed,
                                double vtot,
                                double berv_factor,
                                double lamp_offset_ar,
                                double lamp_offset_ar1,
                                double lamp_offset_ar2,
                                int *cosmics_nb,
                                int *cosmics_nb_2,
                                double *snr,
                                int orders_nr,
                                espdr_ngauss_result *g_res,
                                const char *ccf_mask_id,
                                cpl_table *drift_results,
                                char *fibre_B_tag,
                                double asym,
                                double asym_err,
                                double bispan,
                                double bispan_err,
                                double *ron_RE,
                                double *flux_correction,
                                cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int index;
    char *new_keyword = NULL;
    int saturation_QC = 1;
    int flux_corr_QC = 1, global_QC = 1;
    char comment[COMMENT_LENGTH];
    
    //espdr_msg("Saving QC KWs");
    //espdr_msg("Saving RON KWs");
    index = 0;
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        for (int j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (int k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_kw_part,
                                                                    inst_config->prefix, i, j, k);
                sprintf(comment, "RON[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword, ron_RE[index], comment, keywords_RE);
                cpl_free(new_keyword);
                index++;
            }
        }
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add RON keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (cpl_propertylist_has(*keywords_RE, qc_kws->qc_saturation_check_kw)) {
        saturation_QC = cpl_propertylist_get_int(*keywords_RE, qc_kws->qc_saturation_check_kw);
        if ((saturation_QC == 0) && (strcmp(fibre_B_src, "THAR") != 0)) {
            global_QC = 0;
        }
    }
    
    
    //espdr_msg("Saving cosmics KWs");
    for (int i = 0; i < orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_order_cosmic_nb_kw_first,
                                                        qc_kws->qc_sci_red_order_cosmic_nb_kw_last,
                                                        i+1);
        my_error = espdr_keyword_add_int(new_keyword, cosmics_nb[i], "Cosmics number in the order",
                                         keywords_RE);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add QC_ORDER_COSMIC_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    //espdr_msg("Saving cosmics 2 KWs");
    for (int i = 1; i <= orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_cosmic_nb2_kw_first,
                                                        qc_kws->qc_sci_red_cosmic_nb2_kw_last, i);
        my_error = espdr_keyword_add_int(new_keyword, cosmics_nb_2[i-1],
                                         "Number of cosmics found in each order (2nd filtering)",
                                         keywords_RE);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add QC_ORDER_COSMICS_NB2_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    // espdr_msg("Saving SNR KWs");
    if ((strcmp(fibre_src, "STAR") == 0) || (strcmp(fibre_src, "OBJECT") == 0) ||
        (strcmp(fibre_src, "SKY") == 0) || (strcmp(fibre_src, "SUN") == 0)) {
        for (int i = 1; i <= orders_nr; i++) {
            
            new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_order_snr_kw_first,
                                                            qc_kws->qc_sci_red_order_snr_kw_last,
                                                            i);
            my_error = espdr_keyword_add_double(new_keyword, snr[i-1], "SNR in the order", keywords_RE);
            cpl_free(new_keyword);
        }
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add QC_ORDER_SNR_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    /* BERV KWs */
    //espdr_msg("Saving BERV KWs");
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_berv, berv,
                                        "Barycentric correction [km/s]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_bjd, bjd,
                                        "Barycentric Julian date (TDB) [JD]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_bervmax, bervmax,
                                        "Barycentric max [km/s]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_sed, sed,
                                        "Sun-Earth distance [km]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_vtot, vtot,
                                        "Barycentric observer's velocity [km/s]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_berv_factor, berv_factor,
                                        "BERV factor",
                                        keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add BERV keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Lamp offset */
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar_kw,
                                        lamp_offset_ar, "TH lamp offset AR [m/s]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar1_kw,
                                        lamp_offset_ar1, "TH lamp offset AR1 [m/s]",
                                        keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar2_kw,
                                        lamp_offset_ar2, "TH lamp offset AR2 [m/s]",
                                        keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add ThAr lamp offest KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* DRIFT QC KWs */
    /* The drift KWs are not saved, when fibre B is "SKY" or "DARK" */
    if ((strcmp(fibre_B_tag, "SKY") != 0) && (strcmp(fibre_B_tag, "DARK") != 0)) {
        
        int drift_chi2_check = 1;
        int drift_flux_ratio_check = 1;
        int drift_mean_check = 1;
        int drift_mean_err_check = 1;
        
        espdr_full_drift_QC(drift_results, inst_config, CCD_geom, qc_kws,
                            &drift_chi2_check, &drift_flux_ratio_check,
                            &drift_mean_check, &drift_mean_err_check,
                            *keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_full_drift_QC failed: %s",
                     cpl_error_get_message_default(my_error));
        
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_drift_chi2_check_kw,
                                         drift_chi2_check, "SCIRED drift CHI2 QC",
                                         keywords_RE);
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_drift_flux_ratio_check_kw,
                                         drift_flux_ratio_check, "SCIRED drift flux ratio QC",
                                         keywords_RE);
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_drift_mean_check_kw,
                                         drift_mean_check, "SCIRED drift mean QC",
                                         keywords_RE);
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_drift_mean_err_check_kw,
                                         drift_mean_err_check, "SCIRED drift mean error QC",
                                         keywords_RE);
        int drift_QC = 1;
        if ((drift_chi2_check == 0) || (drift_flux_ratio_check == 0) ||
            (drift_mean_check == 0) || (drift_mean_err_check == 0)) {
            drift_QC = 0;
        }
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_drift_check_kw,
                                         drift_QC, "SCIRED drift QC",
                                         keywords_RE);
        
        if (strcmp(inst_config->instrument, "CORALIE") == 0) {
            if (drift_QC == 0) {
                global_QC = 0;
            }
        }
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add drift QC keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Flux correction KWs */
    if ((strcmp(fibre_src, "STAR") == 0) || (strcmp(fibre_src, "OBJECT") == 0) ||
        (strcmp(fibre_src, "SKY") == 0) || (strcmp(fibre_src, "SUN") == 0)) {
        //espdr_msg("Saving flux correction KWs");
        for (int i = 0; i < orders_nr; i++) {
            new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_order_flux_corr_kw_first,
                                                            qc_kws->qc_sci_red_order_flux_corr_kw_last,
                                                            i+1);
            my_error = espdr_keyword_add_double(new_keyword, flux_correction[i],
                                                "Flux corr for the order", keywords_RE);
            cpl_free(new_keyword);
        }
        
        cpl_vector *flux_corr_vector = cpl_vector_wrap(orders_nr, flux_correction);
        double flux_correction_min = cpl_vector_get_min(flux_corr_vector);
        double flux_correction_max = cpl_vector_get_max(flux_corr_vector);
        cpl_vector_unwrap(flux_corr_vector);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_flux_corr_min_kw, flux_correction_min,
                                            "Min of flux correction", keywords_RE);
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_flux_corr_max_kw, flux_correction_max,
                                            "Max of flux correction", keywords_RE);
        
        if ((flux_correction_min <= 0.25) || (flux_correction_max >= 3.0)) {
            flux_corr_QC = 0;
            /* CLO: global_QC not set to zero when flux correction fails or is not applied */
        }
        
        my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_flux_corr_check_kw, flux_corr_QC,
                                         "Flux correction QC", keywords_RE);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add flux correction QC keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if ((g_res != NULL) &&
        ((strcmp(fibre_src, "STAR") == 0) || (strcmp(fibre_src, "OBJECT") == 0) ||
         (strcmp(fibre_src, "SKY") == 0) || (strcmp(fibre_src, "SUN") == 0) ||
         (strcmp(fibre_src, "TELLURIC") == 0))) {
        
        espdr_msg("Saving CCF KWs");
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_rv, g_res->x0[0],
                                            "Radial velocity [km/s]", keywords_RE);
        
        if ( g_res->sig_x0[0] != 0. && !isnan(g_res->sig_x0[0]) && !isinf(g_res->sig_x0[0])) {
            my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_rv_error, g_res->sig_x0[0],
                                                "Uncertainty on radial velocity [km/s]", keywords_RE);
        } else {
            my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_rv_error, 0.0,
                                                "Uncertainty on radial velocity [km/s]", keywords_RE);
        }
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_fwhm, fabs(g_res->fwhm[0]),
                                            "CCF FWHM [km/s]", keywords_RE);
        if ( g_res->sig_fwhm[0] != 0. && !isnan(g_res->sig_fwhm[0]) && !isinf(g_res->sig_fwhm[0])) {
            my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_fwhm_error, g_res->sig_fwhm[0],
                                                "Uncertainty on CCF FWHM [km/s]", keywords_RE);
        } else {
            my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_fwhm_error, 0.0,
                                                "Uncertainty on CCF FWHM [km/s]", keywords_RE);
        }
        
        double contrast;
        if (g_res->c != 0. && !isnan(g_res->c)) {
            contrast = (fabs(g_res->k[0]/g_res->c))*100.0;
        } else {
            espdr_msg("WARNING : division by zero. g->res =%f",g_res->c);
            contrast = 0.0;
        }
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_contrast, contrast,
                                            "CCF contrast %", keywords_RE);
        
        double contrast_error;
        if (g_res->c != 0. && !isnan(g_res->c)) {
            contrast_error = (fabs(g_res->sig_k[0]/g_res->c))*100.0;
        } else {
            espdr_msg("WARNING : division by zero. g->res =%f",g_res->c);
            contrast_error = 0.0;
        }
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_contrast_error, contrast_error,
                                            "CCF contrast error %", keywords_RE);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_ccf_continuum, g_res->c,
                                            "CCF continuum level [e-]", keywords_RE);
        
        my_error = espdr_keyword_add_string(qc_kws->qc_sci_ccf_mask, ccf_mask_id,
                                            "CCF mask used", keywords_RE);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_asym, asym,
                                            "CCF asymmetry (km/s)", keywords_RE);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_asym_error, asym_err,
                                            "CCF asymmetry error (km/s)", keywords_RE);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_bis_span, bispan,
                                            "CCF bisector span (km/s)", keywords_RE);
        
        my_error = espdr_keyword_add_double(qc_kws->qc_sci_bis_span_error, bispan_err,
                                            "CCF bisector span error (km/s)", keywords_RE);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add CCF QC keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

	my_error = espdr_keyword_add_int(qc_kws->qc_sci_red_check_kw, global_QC,
                                     "SCIRED global QC", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_SCIRED_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     mjd_obs_delta_time_wave   mysterious Andrea's parameter
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_frameset* frameset,
                                         const char* tag,
                                         int ext_nb,
                                         double *mjd_obs_delta_time_wave) {

    espdr_print_rec_status(0);
    //cpl_propertylist_dump(*keywords,stdout);
    const cpl_frame* frame = cpl_frameset_find_const(frameset,tag);
    const char* fname = cpl_frame_get_filename(frame);
    espdr_msg("add extra QC to: %s %s",tag,fname);
    cpl_propertylist* plist = cpl_propertylist_load(fname, 0);
    cpl_propertylist* xlist = cpl_propertylist_load(fname, 1);
    cpl_table* tab = cpl_table_load(fname, 1, 0);
    double spec_avg = 0;
    espdr_print_rec_status(1);
    //cpl_table_dump_structure(tab,stdout);
    cpl_propertylist_append_int(plist,"ESO QC PROC VERSION", 2);
    cpl_size raw_0 = 0;
    cpl_array* snr_array = (cpl_array*) cpl_table_get_array(tab, "SNR", raw_0);
    
    /* handle empty string units */
    int tfields = cpl_propertylist_get_int(xlist,"TFIELDS");
    const char* ttype = NULL;
    const char* cunit = NULL;
    int clen = 0;
    char cname[8];
    for(int i = 1; i <= tfields; i++) {
        sprintf(cname,"TTYPE%d",i);
        ttype = cpl_propertylist_get_string(xlist,cname);
        cunit = cpl_table_get_column_unit(tab,ttype);
        clen = strlen(cunit);
        //cpl_msg_warning(cpl_func,"unit col %s: >%s< %d", ttype,cunit, strlen(cunit));
        if (clen == 0 && cunit != NULL && *cunit == '\0') {
            cunit = " ";
            cpl_table_set_column_unit(tab, ttype, cunit);
        }
        
    }
    
    double* psnr = cpl_array_get_data_double(snr_array);
    cpl_size size = cpl_array_get_size(snr_array);
    cpl_size n_invalid = 0;
    for(cpl_size i = 0; i < size; i++) {
        if(psnr[i] <= 0 || psnr[i] == CPL_VALUE_NOTFINITE) {
            cpl_array_set_invalid(snr_array,i);
            n_invalid++;
        }
    }

    double snr_avg = 0;
    int status = 0;
    if (size > 1  && n_invalid < (size-1)) {
    	snr_avg = cpl_array_get_mean(snr_array);
    } else if (size == 1 || n_invalid == (size-1)) {
    	snr_avg = cpl_array_get(snr_array, 0, &status);
    }
   
    cpl_array* flux_array = NULL;
    n_invalid = 0;
    if(cpl_table_has_column(tab, "FLUX_EL_SKYSUB")) {
        flux_array = (cpl_array*) cpl_table_get_array(tab,"FLUX_EL_SKYSUB",0);
        double* pflux = cpl_array_get_data_double(flux_array);
        cpl_size size = cpl_array_get_size(flux_array);
        for(cpl_size i = 0; i < size; i++) {
            if(psnr[i] <= 0 || psnr[i] == CPL_VALUE_NOTFINITE) {
                cpl_array_set_invalid(flux_array,i);
                n_invalid++;
            }
        }
        if (size > 1  && n_invalid < (size-1)) {
            spec_avg = cpl_array_get_mean(flux_array);
        } else if (size == 1 || n_invalid == (size-1)) {
        	spec_avg = cpl_array_get(flux_array, 0, &status);
        }
        if((spec_avg == CPL_VALUE_NOTFINITE) || isnan(spec_avg) || isinf(spec_avg)) {
            spec_avg=999;
        }
        cpl_propertylist_append_double(plist, "ESO QC SPEC AVG",spec_avg);
        cpl_propertylist_set_comment(plist,"ESO QC SPEC AVG", "average FLUX_EL_SKYSUB");
    } else {
        flux_array = (cpl_array*) cpl_table_get_array(tab,"FLUX_EL",0);
        
        double* pflux = cpl_array_get_data_double(flux_array);
        
        cpl_size size = cpl_array_get_size(flux_array);
        for(cpl_size i = 0; i < size; i++) {
            
            if(psnr[i] <= 0 || psnr[i] == CPL_VALUE_NOTFINITE) {
                cpl_array_set_invalid(flux_array,i);
                n_invalid++;
            }
        }
        if (size > 1  && n_invalid < (size-1)) {
            spec_avg = cpl_array_get_mean(flux_array);
        } else if (size == 1 || n_invalid == (size-1)) {
        	spec_avg = cpl_array_get(flux_array, 0, &status);
        }
        if((spec_avg == CPL_VALUE_NOTFINITE) || isnan(spec_avg) || isinf(spec_avg)) {
            spec_avg=999;
        }
        cpl_propertylist_append_double(plist, "ESO QC SPEC AVG",spec_avg);
        cpl_propertylist_set_comment(plist,"ESO QC SPEC AVG", "average FLUX_EL");
    }
    espdr_print_rec_status(3);
    //espdr_msg("snr_avg: %g",snr_avg);
    //espdr_msg("spec_avg: %g",spec_avg);
    snr_avg = (snr_avg >= 0) ? snr_avg : 0;
    spec_avg = (spec_avg >= 0) ? spec_avg : 0;
    cpl_propertylist_append_double(plist, "ESO QC SNR AVG",snr_avg);
    cpl_propertylist_set_comment(plist,"ESO QC SNR AVG", "average SNR");
    
    cpl_propertylist_append_double(plist, "ESO QC SPEC AVG",spec_avg);
    cpl_propertylist_set_comment(plist,"ESO QC SPEC AVG", "average SPEC");
    
    espdr_add_qc_key_stat_ord_pri(&plist, ext_nb,"COSMIC NB", "DUMMY", CPL_FALSE,
                                  CPL_FALSE, CPL_FALSE, CPL_FALSE, CPL_TRUE);
    
    espdr_print_rec_status(4);
    
    espdr_add_qc_airm_info(&plist);
    espdr_add_qc_iwv_info(&plist);
    
    cpl_propertylist_append_string(plist, "ESO QC PROC SCHEME","ESPRE_R");
    
    /* Not used anymore
     int sat_nb = cpl_propertylist_get_int(plist,"ESO QC SAT NB");
     
     const char* sp_type = cpl_propertylist_get_string(plist,"ESO OCS OBJ SP TYPE");
     
     const double rv = cpl_propertylist_get_double(plist,"ESO OCS OBJ RV");
     */
    
    espdr_print_rec_status(5);
    //Here the DRIFT keys probably refer to WAVE_MATRIX solution
    char key_val[40];
    cpl_property* p;
    
    
    sprintf(key_val,"ESO QC DELTA TIME WAVE A");
    cpl_msg_info(cpl_func,"key_val %s",key_val);
    cpl_propertylist_append_double(plist, key_val, mjd_obs_delta_time_wave[0]);
    cpl_propertylist_set_comment(plist, key_val,
                                 "MJD_OBS difference of recipe inputs files WAVE_MATRIX_THAR_FP_A and OBJ_FP");
    
    sprintf(key_val,"ESO QC DELTA TIME WAVE B");
    cpl_msg_info(cpl_func,"key_val %s",key_val);
    cpl_propertylist_append_double(plist, key_val, mjd_obs_delta_time_wave[1]);
    cpl_propertylist_set_comment(plist, key_val,
                                 "MJD_OBS difference of recipe inputs files WAVE_MATRIX_THAR_FP_B and OBJ_FP");
    
    sprintf(key_val,"ESO QC DRIFT DET0 MEAN");
    if(cpl_propertylist_has(plist,key_val)) {
        cpl_propertylist_append_string(plist, "ESO QC FIB B","FP");
        
        
        
        espdr_msg("found DRIFT");
        for (cpl_size i = 0; i < ext_nb; i++) {
            
            sprintf(key_val,"ESO QC DRIFT DET%lld MEAN",i);
            cpl_msg_info(cpl_func,"key_val %s",key_val);
            //cpl_propertylist_dump(*keywords,stdout);
            p = cpl_propertylist_get_property(plist,key_val);
            
            double drift_mean = cpl_property_get_double(p);
            double qc_drift_vel =
            espdr_drift_vel(drift_mean, mjd_obs_delta_time_wave[0]);
            
            sprintf(key_val,"ESO QC DRIFT DET%lld VEL",i);
            cpl_msg_info(cpl_func,"key_val %s",key_val);
            cpl_propertylist_append_double(plist, key_val, qc_drift_vel);
            cpl_propertylist_set_comment(plist, key_val,
                                         "[cm/s/h] Instrumental drift of wavelength calibration");
            
        }
        
    } else {
        cpl_propertylist_append_string(plist, "ESO QC FIB B","SKY");
    }
    espdr_print_rec_status(6);
    
    //cpl_propertylist_save(plist,fname,CPL_IO_DEFAULT);
    cpl_table_save(tab, plist, xlist, fname, CPL_IO_CREATE);
    cpl_table_delete(tab);
    
    cpl_propertylist_delete(plist);
    cpl_propertylist_delete(xlist);
    
    espdr_check_error_code(cpl_func);
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Save science products
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_science_products(cpl_frameset *frameset,
                                           cpl_parameterlist *parameters,
                                           const char *RECIPE_ID,
                                           cpl_frameset *used_frames,
                                           cpl_propertylist **keywords_fibre,
                                           cpl_propertylist *keywords_skysub,
                                           cpl_propertylist *keywords_tell_corr,
                                           cpl_propertylist *telluric_keywords,
                                           cpl_propertylist *keywords_oh_corr,
                                           double *mjd_obs_delta_time_wave,
                                           espdr_inst_config *inst_config,
                                           int ext_nb,
                                           int fibres_processed,
                                           char *token,
                                           const char *tag,
                                           char *sky_correction_method,
                                           int tell_corr_flag,
                                           cpl_table *residuals,
                                           cpl_image *drift_matrix,
                                           cpl_image **flat_corr_flux,
                                           cpl_image **flat_corr_err,
                                           cpl_image **flat_corr_qual,
                                           cpl_image **flux_s2d,
                                           cpl_image **error_s2d,
                                           cpl_image **qual_s2d,
                                           cpl_image *flux_sky_sub_s2d_blaze,
                                           cpl_image *error_sky_sub_s2d_blaze,
                                           cpl_image *qual_sky_sub_s2d_blaze,
                                           cpl_image *flux_sky_sub_s2d,
                                           cpl_image *error_sky_sub_s2d,
                                           cpl_image *qual_sky_sub_s2d,
                                           cpl_image **wave_shifted,
                                           cpl_image **wave_air_shifted,
                                           cpl_image **dll_shifted,
                                           cpl_image **dll_air_shifted,
                                           cpl_image **wave_shifted_drift_corr,
                                           cpl_image **wave_air_shifted_drift_corr,
                                           cpl_image **dll_shifted_drift_corr,
                                           cpl_image **dll_air_shifted_drift_corr,
                                           cpl_image **wave_matrix,
                                           cpl_image **dll_matrix,
                                           cpl_image **wave_matrix_air,
                                           cpl_image **dll_matrix_air,
                                           cpl_image **wave_matrix_drift_corr,
                                           cpl_image **wave_matrix_air_drift_corr,
                                           cpl_table **s1d_table_merged,
                                           cpl_table *s1d_table_merged_skysub,
                                           cpl_table *s1d_fluxcal_skysub,
                                           cpl_table *s1d_fluxcal,
                                           cpl_table *s1d_table_merged_tell_corr,
                                           cpl_table *s1d_fluxcal_tell_corr,
                                           cpl_image **CCF_flux,
                                           cpl_image **CCF_error,
                                           cpl_image **CCF_quality,
                                           cpl_image *CCF_flux_sky_sub,
                                           cpl_image *CCF_error_sky_sub,
                                           cpl_image *CCF_quality_sky_sub,
                                           cpl_image *CCF_flux_tell_corr,
                                           cpl_image *CCF_error_tell_corr,
                                           cpl_image *CCF_quality_tell_corr,
                                           cpl_image *tell_spectrum,
                                           cpl_image *flux_tell_corr_s2d_blaze,
                                           cpl_image *error_tell_corr_s2d_blaze,
                                           cpl_image *qual_tell_corr_s2d_blaze,
                                           cpl_image *flux_tell_corr_s2d,
                                           cpl_image *error_tell_corr_s2d,
                                           cpl_image *qual_tell_corr_s2d) {
    
    //cpl_error_code my_error = CPL_ERROR_NONE;
    char product_filename[64];
    char pro_catg_string[64];
    cpl_frame* product_frame;

    /* CCF residuals */
    cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);
    if (espdr_frameset_has_frame(used_frames, ESPDR_SCIENCE_HEADER_RAW)) {
        cpl_frame *header_frame = espdr_frame_find(used_frames, ESPDR_SCIENCE_HEADER_RAW);
        cpl_frame *science_frame = espdr_frame_find(used_frames, tag);
        cpl_frame_set_tag(header_frame, tag);
        cpl_frame_set_tag(science_frame, ESPDR_SCIENCE_HEADER_RAW);
    }
    
    sprintf(product_filename, "%s_CCF_RESIDUALS_A.fits", inst_config->instrument);
    sprintf(pro_catg_string, "%s_RESIDUALS_A", ESPDR_PRO_CATG_SCIENCE_CCF);
    cpl_propertylist_update_string(keywords_fibre[0], PRO_CATG_KW,
                                              pro_catg_string);
	cpl_propertylist_erase(keywords_fibre[0], "RADECSYS");
    cpl_dfs_save_table(frameset_orig, NULL,
                       parameters, used_frames,
                       NULL, residuals,
                       NULL, RECIPE_ID,keywords_fibre[0],
                       "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                       product_filename);
    espdr_frame_new(&product_frame,
                    product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_TABLE,
                    pro_catg_string);
    cpl_frameset_insert(frameset, product_frame);
    
    /* drift matrix B */
    if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
        sprintf(product_filename, "%s_DRIFT_MATRIX_B.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_B", ESPDR_PRO_CATG_DRIFT_MATRIX);
        cpl_propertylist_update_string(keywords_fibre[1], PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_append_string(keywords_fibre[1], PRODCATG_KW, ESPDR_ANCILLARY_DRIFT);
        espdr_dfs_image_save_one_ext(frameset_orig, parameters,
                                     used_frames, RECIPE_ID,
                                     keywords_fibre[1],
                                     product_filename,
                                     drift_matrix,
                                     CPL_TYPE_FLOAT);
        cpl_propertylist_erase(keywords_fibre[1], PRODCATG_KW);
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
    } // if THAR, FP or LFC
    
    /* S2D products saving */
    cpl_image **images_to_save = (cpl_image**)cpl_malloc(7*sizeof(cpl_image*));
    for (int i = 0; i < fibres_processed; i++) {
        images_to_save[0] = cpl_image_duplicate(flat_corr_flux[i]);
        images_to_save[1] = cpl_image_duplicate(flat_corr_err[i]);
        images_to_save[2] = cpl_image_duplicate(flat_corr_qual[i]);
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            images_to_save[3] = cpl_image_duplicate(wave_shifted_drift_corr[i]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted_drift_corr[i]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted_drift_corr[i]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted_drift_corr[i]);
        } else {
            images_to_save[3] = cpl_image_duplicate(wave_shifted[i]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted[i]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted[i]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted[i]);
        }
        
        sprintf(product_filename, "%s_S2D_BLAZE_%c.fits", inst_config->instrument, fibre_name[i]);
        sprintf(pro_catg_string, "%s_BLAZE_%c", ESPDR_PRO_CATG_SCIENCE_S2D, fibre_name[i]);
        cpl_propertylist_update_string(keywords_fibre[i], PRO_CATG_KW, pro_catg_string);
        if (i == 0) {
            cpl_propertylist_append_string(keywords_fibre[i], PRODCATG_KW,
                                           "ANCILLARY.2DECHELLE.NOBLAZE.NOSKYSUB");
        }
        cpl_propertylist_erase(keywords_fibre[i], "RADECSYS");
        espdr_dfs_save_data_err_qual_wave(frameset_orig,
                                          parameters,
                                          used_frames,
                                          RECIPE_ID,
                                          keywords_fibre[i],
                                          product_filename,
                                          images_to_save);
        if (i == 0) {
            cpl_propertylist_erase(keywords_fibre[i], PRODCATG_KW);
        }
        
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }

        images_to_save[0] = cpl_image_duplicate(flux_s2d[i]);
        images_to_save[1] = cpl_image_duplicate(error_s2d[i]);
        images_to_save[2] = cpl_image_duplicate(qual_s2d[i]);
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            images_to_save[3] = cpl_image_duplicate(wave_shifted_drift_corr[i]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted_drift_corr[i]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted_drift_corr[i]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted_drift_corr[i]);
        } else {
            images_to_save[3] = cpl_image_duplicate(wave_shifted[i]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted[i]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted[i]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted[i]);
        }
        
        sprintf(product_filename, "%s_S2D_%c.fits", inst_config->instrument, fibre_name[i]);
        sprintf(pro_catg_string, "%s_%c", ESPDR_PRO_CATG_SCIENCE_S2D, fibre_name[i]);
        cpl_propertylist_update_string(keywords_fibre[i], PRO_CATG_KW, pro_catg_string);
        if (i == 0) {
            cpl_propertylist_append_string(keywords_fibre[i], PRODCATG_KW,
                                           "ANCILLARY.2DECHELLE.BLAZE.NOSKYSUB");
        }
        espdr_dfs_save_data_err_qual_wave(frameset_orig,
                                          parameters,
                                          used_frames,
                                          RECIPE_ID,
                                          keywords_fibre[i],
                                          product_filename,
                                          images_to_save);
        if (i == 0) {
            cpl_propertylist_erase(keywords_fibre[i], PRODCATG_KW);
        }
        
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    
    if ((strcmp(token, "SKY") == 0) && (strcmp(sky_correction_method, "none") != 0)) {
        images_to_save[0] = cpl_image_duplicate(flux_sky_sub_s2d_blaze);
        images_to_save[1] = cpl_image_duplicate(error_sky_sub_s2d_blaze);
        images_to_save[2] = cpl_image_duplicate(qual_sky_sub_s2d_blaze);
        images_to_save[3] = cpl_image_duplicate(wave_shifted[0]);
        images_to_save[4] = cpl_image_duplicate(wave_air_shifted[0]);
        images_to_save[5] = cpl_image_duplicate(dll_shifted[0]);
        images_to_save[6] = cpl_image_duplicate(dll_air_shifted[0]);
        
        sprintf(product_filename, "%s_S2D_BLAZE_SKYSUB_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S2D_BLAZE_SKYSUB);
        cpl_propertylist_update_string(keywords_skysub, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_append_string(keywords_skysub, PRODCATG_KW,
                                       "ANCILLARY.2DECHELLE.NOBLAZE.SKYSUB");
        cpl_propertylist_erase(keywords_skysub, "RADECSYS");
        if (strcmp(sky_correction_method, "oh-corr") == 0) {
            cpl_propertylist_append(keywords_skysub,
                                    cpl_propertylist_duplicate(keywords_oh_corr));
        }
        espdr_dfs_save_data_err_qual_wave(frameset_orig,
                                          parameters,
                                          used_frames,
                                          RECIPE_ID,
                                          keywords_skysub,
                                          product_filename,
                                          images_to_save);
        cpl_propertylist_erase(keywords_skysub, PRODCATG_KW);
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
        
        images_to_save[0] = cpl_image_duplicate(flux_sky_sub_s2d);
        images_to_save[1] = cpl_image_duplicate(error_sky_sub_s2d);
        images_to_save[2] = cpl_image_duplicate(qual_sky_sub_s2d);
        images_to_save[3] = cpl_image_duplicate(wave_shifted[0]);
        images_to_save[4] = cpl_image_duplicate(wave_air_shifted[0]);
        images_to_save[5] = cpl_image_duplicate(dll_shifted[0]);
        images_to_save[6] = cpl_image_duplicate(dll_air_shifted[0]);
        
        sprintf(product_filename, "%s_S2D_SKYSUB_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S2D_SKYSUB);
        cpl_propertylist_update_string(keywords_skysub, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_append_string(keywords_skysub, PRODCATG_KW,
                                       "ANCILLARY.2DECHELLE.BLAZE.SKYSUB");
        cpl_propertylist_erase(keywords_skysub, "RADECSYS");
        espdr_dfs_save_data_err_qual_wave(frameset_orig,
                                          parameters,
                                          used_frames,
                                          RECIPE_ID,
                                          keywords_skysub,
                                          product_filename,
                                          images_to_save);
        cpl_propertylist_erase(keywords_skysub, PRODCATG_KW);
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    
    if (tell_corr_flag == 1) {
        images_to_save[0] = cpl_image_duplicate(tell_spectrum);
        images_to_save[1] = NULL;
        images_to_save[2] = NULL;
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            images_to_save[3] = cpl_image_duplicate(wave_matrix_drift_corr[0]);
            images_to_save[4] = cpl_image_duplicate(wave_matrix_air_drift_corr[0]);
        } else {
            images_to_save[3] = cpl_image_duplicate(wave_matrix[0]);
            images_to_save[4] = cpl_image_duplicate(wave_matrix_air[0]);
        }
        images_to_save[5] = cpl_image_duplicate(dll_matrix[0]);
        images_to_save[6] = cpl_image_duplicate(dll_matrix_air[0]);
        
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        sprintf(product_filename, "%s_S2D_TELL_SPECTRUM_A.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_A", ESPDR_PRO_CATG_SCIENCE_TELL_SPECTRUM);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW,
                                       "ANCILLARY.2DECHELLE.TELLURIC");

        espdr_dfs_save_data_err_qual_wave(frameset_orig, parameters,
                                          used_frames, RECIPE_ID,
                                          tell_keywords_full,
                                          product_filename,
                                          images_to_save);
        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        cpl_propertylist_delete(tell_keywords_full);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    
    if (tell_corr_flag == 1) {
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        
        images_to_save[0] = cpl_image_duplicate(flux_tell_corr_s2d_blaze);
        images_to_save[1] = cpl_image_duplicate(error_tell_corr_s2d_blaze);
        images_to_save[2] = cpl_image_duplicate(qual_tell_corr_s2d_blaze);
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            images_to_save[3] = cpl_image_duplicate(wave_shifted_drift_corr[0]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted_drift_corr[0]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted_drift_corr[0]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted_drift_corr[0]);
        } else {
            images_to_save[3] = cpl_image_duplicate(wave_shifted[0]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted[0]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted[0]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted[0]);
        }
        
        sprintf(product_filename, "%s_S2D_BLAZE_TELL_CORR_A.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_A", ESPDR_PRO_CATG_SCIENCE_S2D_BLAZE_TELL_CORR);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW, "ANCILLARY.2DECHELLE.BLAZE");
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        espdr_dfs_save_data_err_qual_wave(frameset_orig, parameters,
                                          used_frames, RECIPE_ID,
                                          tell_keywords_full,
                                          product_filename,
                                          images_to_save);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        cpl_propertylist_delete(tell_keywords_full);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    
    if (tell_corr_flag == 1) {
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        
        images_to_save[0] = cpl_image_duplicate(flux_tell_corr_s2d);
        images_to_save[1] = cpl_image_duplicate(error_tell_corr_s2d);
        images_to_save[2] = cpl_image_duplicate(qual_tell_corr_s2d);
        if (strcmp(token, "THAR") == 0 || strcmp(token, "FP") == 0 || strcmp(token, "LFC") == 0) {
            images_to_save[3] = cpl_image_duplicate(wave_shifted_drift_corr[0]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted_drift_corr[0]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted_drift_corr[0]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted_drift_corr[0]);
        } else {
            images_to_save[3] = cpl_image_duplicate(wave_shifted[0]);
            images_to_save[4] = cpl_image_duplicate(wave_air_shifted[0]);
            images_to_save[5] = cpl_image_duplicate(dll_shifted[0]);
            images_to_save[6] = cpl_image_duplicate(dll_air_shifted[0]);
        }
        
        sprintf(product_filename, "%s_S2D_TELL_CORR_A.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_A", ESPDR_PRO_CATG_SCIENCE_S2D_TELL_CORR);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW, "ANCILLARY.2DECHELLE");
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        espdr_dfs_save_data_err_qual_wave(frameset_orig, parameters,
                                          used_frames, RECIPE_ID,
                                          tell_keywords_full,
                                          product_filename,
                                          images_to_save);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        cpl_propertylist_delete(tell_keywords_full);
        
        for (int im = 0; im < 7; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    cpl_free(images_to_save);
    
    /* S1D products saving */
    cpl_boolean has_ccf = CPL_FALSE;
    
    if ((strcmp(token, "SKY") == 0) && (strcmp(sky_correction_method, "none") != 0)) {
        sprintf(product_filename, "%s_S1D_SKYSUB_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S1D_SKYSUB);
        cpl_propertylist_update_string(keywords_skysub, PRO_CATG_KW, pro_catg_string);
        cpl_dfs_save_table(frameset_orig, NULL,
                           parameters, used_frames,
                           NULL, s1d_table_merged_skysub,
                           NULL, RECIPE_ID, keywords_skysub,
                           "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                           product_filename);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_TABLE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        
        if (s1d_fluxcal_skysub != NULL) {
            sprintf(product_filename, "%s_S1D_SKYSUB_FLUXCAL_A.fits", inst_config->instrument);
            sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S1D_SKYSUB_FLUXCAL);
            cpl_propertylist_update_string(keywords_skysub, PRO_CATG_KW, pro_catg_string);
            cpl_dfs_save_table(frameset_orig, NULL,
                               parameters, used_frames,
                               NULL, s1d_fluxcal_skysub,
                               NULL, RECIPE_ID, keywords_skysub,
                               "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                               product_filename);

            espdr_frame_new(&product_frame,
                            product_filename,
                            CPL_FRAME_GROUP_PRODUCT,
                            CPL_FRAME_LEVEL_FINAL,
                            CPL_FRAME_TYPE_TABLE,
                            pro_catg_string);
            cpl_frameset_insert(frameset, product_frame);
        }
    }

    if (s1d_fluxcal != NULL) {
        sprintf(product_filename, "%s_S1D_FLUXCAL_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S1D_FLUXCAL);
        cpl_propertylist_update_string(keywords_fibre[0], PRO_CATG_KW, pro_catg_string);
        cpl_dfs_save_table(frameset_orig, NULL,
                           parameters, used_frames,
                           NULL, s1d_fluxcal,
                           NULL, RECIPE_ID, keywords_fibre[0],
                           "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                           product_filename);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_TABLE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
    }
    
    //SDP creation was here
    
    if (tell_corr_flag == 1) {
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        sprintf(product_filename, "%s_S1D_TELL_CORR_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S1D_TELLCORR);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW, "ANCILLARY.SPECTRUM");
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        cpl_dfs_save_table(frameset_orig, NULL,
                           parameters, used_frames,
                           NULL, s1d_table_merged_tell_corr,
                           NULL, RECIPE_ID, tell_keywords_full,
                           "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                           product_filename);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_TABLE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
    }
    
    if (tell_corr_flag == 1 && s1d_fluxcal != NULL) {
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        sprintf(product_filename, "%s_S1D_TELL_CORR_FLUXCAL_A.fits", inst_config->instrument);
        sprintf(pro_catg_string,"%s_A", ESPDR_PRO_CATG_SCIENCE_S1D_TELLCORR_FLUXCAL);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW, "ANCILLARY.SPECTRUM");
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        cpl_dfs_save_table(frameset_orig, NULL,
                           parameters, used_frames,
                           NULL, s1d_fluxcal_tell_corr,
                           NULL, RECIPE_ID, tell_keywords_full,
                           "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                           product_filename);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_TABLE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
    }
    
    /* CCF products saving */
    images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
    for (int i = 0; i < fibres_processed; i++) {
        if (CCF_flux[i] != NULL) {
            images_to_save[0] = cpl_image_duplicate(CCF_flux[i]);
            images_to_save[1] = cpl_image_duplicate(CCF_error[i]);
            images_to_save[2] = cpl_image_duplicate(CCF_quality[i]);
            sprintf(product_filename, "%s_CCF_%c.fits", inst_config->instrument, fibre_name[i]);
            sprintf(pro_catg_string,"%s_%c", ESPDR_PRO_CATG_SCIENCE_CCF, fibre_name[i]);
            cpl_propertylist_update_string(keywords_fibre[i], PRO_CATG_KW,
                                                      pro_catg_string);
            if (i == 0) {
                cpl_propertylist_append_string(keywords_fibre[i], PRODCATG_KW,
                                                          ESPDR_ANCILLARY_CCF);
            }
            espdr_dfs_save_data_err_qual(frameset_orig,
                                         parameters,
                                         used_frames,
                                         RECIPE_ID,
                                         keywords_fibre[i],
                                         product_filename,
                                         images_to_save);
            if (i == 0) {
                cpl_propertylist_erase(keywords_fibre[i], PRODCATG_KW);
            }
            
            espdr_frame_new(&product_frame,
                            product_filename,
                            CPL_FRAME_GROUP_PRODUCT,
                            CPL_FRAME_LEVEL_FINAL,
                            CPL_FRAME_TYPE_IMAGE,
                            pro_catg_string);
            cpl_frameset_insert(frameset, product_frame);
            
            for (int im = 0; im < 3; im++) {
                cpl_image_delete(images_to_save[im]);
            }
        } else {
            espdr_msg(ANSI_COLOR_RED"No CCF saved for fibre %c"ANSI_COLOR_RESET, fibre_name[i]);
        }
    }
    
    if ((strcmp(token, "SKY") == 0) && (strcmp(sky_correction_method, "none") != 0)) {
        images_to_save[0] = cpl_image_duplicate(CCF_flux_sky_sub);
        images_to_save[1] = cpl_image_duplicate(CCF_error_sky_sub);
        images_to_save[2] = cpl_image_duplicate(CCF_quality_sky_sub);
        sprintf(product_filename, "%s_CCF_SKYSUB_A.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_SKYSUB_A", ESPDR_PRO_CATG_SCIENCE_CCF);
        cpl_propertylist_update_string(keywords_skysub, PRO_CATG_KW, pro_catg_string);
        espdr_dfs_save_data_err_qual(frameset_orig,
                                     parameters,
                                     used_frames,
                                     RECIPE_ID,
                                     keywords_skysub,
                                     product_filename,
                                     images_to_save);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        for (int im = 0; im < 3; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }
    
    if (tell_corr_flag == 1) {
        cpl_propertylist *tell_keywords_full = cpl_propertylist_duplicate(keywords_tell_corr);
        cpl_propertylist_append(tell_keywords_full,
                                cpl_propertylist_duplicate(telluric_keywords));
        images_to_save[0] = cpl_image_duplicate(CCF_flux_tell_corr);
        images_to_save[1] = cpl_image_duplicate(CCF_error_tell_corr);
        images_to_save[2] = cpl_image_duplicate(CCF_quality_tell_corr);
        sprintf(product_filename, "%s_CCF_TELL_CORR_A.fits", inst_config->instrument);
        sprintf(pro_catg_string, "%s_TELL_CORR_A", ESPDR_PRO_CATG_SCIENCE_CCF);
        cpl_propertylist_update_string(tell_keywords_full, PRO_CATG_KW, pro_catg_string);
        cpl_propertylist_update_string(tell_keywords_full, PRODCATG_KW, "ANCILLARY.CCF");
        cpl_propertylist_erase(tell_keywords_full, "RADECSYS");
        espdr_dfs_save_data_err_qual(frameset_orig,
                                     parameters,
                                     used_frames,
                                     RECIPE_ID,
                                     tell_keywords_full,
                                     product_filename,
                                     images_to_save);

        espdr_frame_new(&product_frame,
                        product_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        for (int im = 0; im < 3; im++) {
            cpl_image_delete(images_to_save[im]);
        }
    }

    //SDP creation moved here
    for (int i = 0; i < fibres_processed; i++) {
        sprintf(product_filename, "%s_S1D_%c.fits", inst_config->instrument, fibre_name[i]);
        sprintf(pro_catg_string,"%s_%c", ESPDR_PRO_CATG_SCIENCE_S1D, fibre_name[i]);
        cpl_propertylist_update_string(keywords_fibre[i], PRO_CATG_KW, pro_catg_string);
        cpl_dfs_save_table(frameset_orig, NULL,
                parameters, used_frames,
                NULL, s1d_table_merged[i],
                NULL, RECIPE_ID, keywords_fibre[i],
                "RADECSYS", PACKAGE "/" PACKAGE_VERSION,
                product_filename);
        espdr_frame_new(&product_frame,
                product_filename,
                CPL_FRAME_GROUP_PRODUCT,
                CPL_FRAME_LEVEL_FINAL,
                CPL_FRAME_TYPE_TABLE,
                pro_catg_string);
        cpl_frameset_insert(frameset, product_frame);
        
        //SDP frames
        if (i == 0 && CCF_flux[i] != NULL) {
            has_ccf = CPL_TRUE;
        }
        if ((strcmp(inst_config->instrument, "HARPN") != 0)) { //&&
            //(strcmp(inst_config->instrument, "CORALIE") != 0)) {
            /* For OBJ,SKY we create SDP format for FIB_A */
            if (i == 0) {
                if (strcmp(token,"SKY") == 0 ) {
                    cpl_frame* idp_frame = cpl_frame_duplicate(product_frame);
                    cpl_frameset_insert(used_frames, idp_frame);
                    espdr_msg("create S1D_FINAL_A");
                    espdr_add_sdp_product_merged_spectrum2(idp_frame,
                                                           frameset,
                                                           used_frames,
                                                           parameters,
                                                           RECIPE_ID,
                                                           s1d_fluxcal,
                                                           s1d_fluxcal_skysub,
                                                           s1d_table_merged_skysub,
                                                           has_ccf);
                    /* addition of extra QC keywords */
                    espdr_add_extra_qc(frameset, "S1D_FINAL_A", ext_nb,
                                                  mjd_obs_delta_time_wave);
                    cpl_frameset_erase_frame(used_frames, idp_frame);
                } else {
                    espdr_msg("Writing IDP for fibre A");
                    cpl_frame* idp_frame = cpl_frame_duplicate(product_frame);
                    cpl_frameset_insert(used_frames, idp_frame);
                    espdr_msg("create S1D_FINAL_A");
                    espdr_add_sdp_product_merged_spectrum2(idp_frame,
                                                           frameset,
                                                           used_frames,
                                                           parameters,
                                                           RECIPE_ID,
                                                           s1d_fluxcal,
                                                           NULL,
                                                           NULL,
                                                           has_ccf);
                    /* addition of extra QC keywords */
                    espdr_add_extra_qc(frameset, "S1D_FINAL_A", ext_nb,
                                                  mjd_obs_delta_time_wave);
                    cpl_frameset_erase_frame(used_frames, idp_frame);
                }
            } else {
                cpl_frame* idp_frame = cpl_frame_duplicate(product_frame);
                cpl_frameset_insert(used_frames, idp_frame);
                espdr_msg("create S1D_FINAL_B");
                espdr_add_sdp_product_spectrum2(idp_frame,
                                                frameset,
                                                used_frames,
                                                parameters,
                                                RECIPE_ID,
                                                CPL_FALSE);
                espdr_add_extra_qc(frameset, "S1D_FINAL_B", ext_nb,
                                              mjd_obs_delta_time_wave);
                cpl_frameset_erase_frame(used_frames, idp_frame);
            }
        } /* end SDP format addition */
    } // end for fibres
    
    cpl_free(images_to_save);
    cpl_frameset_delete(frameset_orig);
    
    return cpl_error_get_code();
}
