/*-----------------------------------------------------------------*/
/*!
  \file bepop.c
  \brief computation of various quantities usefull to reduce extra-solar
         planets observations

  \author  H. Manche
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris.

   Copyright 2014, 2015, 2016, CNRS
   email of the author : herve.manche@obspm.fr

  last revision: 05/06/2014

  History:
  \note H. Manche 05/06/2014 : creation
 */
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses
 below to apply on this file.

    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )

 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)

 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )


This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under
French law and abiding by the rules of distribution of free software.
You can  use, modify and/ or redistribute the software under the terms
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA
at the following URL "http://www.cecill.info".

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability.

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or
data to be ensured and,  more generally, to use and operate it in the
same conditions as regards security.

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/


#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "bepopinternal.h"
#include "bepop.h"
#include "calceph.h"
#include "earth_orientation_parameters.h"
#include "errors.h"
#include "erfa.h"
#include "erfam.h"
#include "erfa_version.h"
#include "itrf_icrf.h"
#include "polynomial_interpolation.h"
#include "timescales.h"
#include "vec_utils.h"



/*!
  Allocate memory for a typeParamBEPOP object
  @param ptrBEPOP (inout): pointer to structure of parameters
 */
typeParamBEPOP* BEPOP_initAllocate(typeParamBEPOP *ptrBEPOP)
{
  if (ptrBEPOP!=NULL)
  {
    FatalError("Pointer ptrBEPOP should be NULL in BEPOP_initAllocate",__FILE__,__LINE__) ;
  }

  /* Allocate memory */
  ptrBEPOP=(typeParamBEPOP*)malloc(sizeof(typeParamBEPOP));

  /* initialize all pointers to NULL */
    /* pointer to the ephemeris */
    ptrBEPOP->PTR_eph=NULL ;
    /* pointers related to EOPs */
    ptrBEPOP->ArraysAllEOPs.MJD_UTC=NULL ;
    ptrBEPOP->ArraysAllEOPs.xp=NULL ;
    ptrBEPOP->ArraysAllEOPs.yp=NULL ;
    ptrBEPOP->ArraysAllEOPs.UT1mUTC=NULL ;
    ptrBEPOP->ArraysAllEOPs.dX=NULL ;
    ptrBEPOP->ArraysAllEOPs.dY=NULL ;
    ptrBEPOP->ArraysAllEOPs.UT1mTT=NULL ;
    ptrBEPOP->ArraysAllEOPs.JulianDay_TT1=NULL ;
    ptrBEPOP->ArraysAllEOPs.JulianDay_TT2=NULL ;

  return(ptrBEPOP) ;
};


/*!
  Initialize a parameter structure to default values
  @param typeInit (in): model type (1, 2 or 3, see below)
  @param ptrBEPOP (inout): pointer to structure of parameters
 */
typeParamBEPOP* BEPOP_initDefault(const int typeInit,typeParamBEPOP *ptrBEPOP)
{
/*
 * typeInit = 1 : the most precise model (EOPs taken into account, complete CIP nutations)
 * typeInit = 2 : faster (simplified nutations), same precision, but not "up to date"
 * typeInit = 3 : less precise (simplified nutations, no EOPs taken into account)
*/
  /* Local variables */
  const char * dirEphem = "%s/ephemerides/%s" ;
  const char * fileEOPs = "%s/eop/eopc04_IAU2000.62-now" ;
  char *ephemerisFileFullName ;
  char *EOPsFileFullName ;


  /* Location of the default ephemeris file */
  ephemerisFileFullName = (char*)malloc(sizeof(char)*( strlen(_DATAINSTALLDIRECTORY)
                                                      +strlen(dirEphem)
                                                      +strlen(_DEFAULT_EPHEMERIS_FILE_NAME)
                                                      +1 ));
  sprintf(ephemerisFileFullName,dirEphem, _DATAINSTALLDIRECTORY, _DEFAULT_EPHEMERIS_FILE_NAME);


  /* Location of the default EOPs file */
  EOPsFileFullName = (char*)malloc(sizeof(char)*( strlen(_DATAINSTALLDIRECTORY)
                                                 +strlen(fileEOPs)
                                                 +1 ));
  sprintf(EOPsFileFullName,fileEOPs, _DATAINSTALLDIRECTORY);

  if (ptrBEPOP!=NULL)
  {
    FatalError("Pointer ptrBEPOP should be NULL in BEPOP_initDefault",__FILE__,__LINE__) ;
  }

  ptrBEPOP=BEPOP_initDefault2(ephemerisFileFullName,EOPsFileFullName,typeInit,ptrBEPOP) ;

  free(ephemerisFileFullName) ;
  free(EOPsFileFullName) ;

  return(ptrBEPOP) ;
};



/*!
  Initialize a parameter structure to default values (with full names for the ephemeris and EOPs files
  @param EphemerisFileName (in): name of the ephemeris file, with full path
  @param EOPsFileName (in): name of the EOPs file, with full path
  @param typeInit (in): model type (1, 2 or 3, see below)
  @param ptrBEPOP (inout): pointer to structure of parameters
 */
struct strucParamBEPOPprivate * BEPOP_initDefault2(const char *EphemerisFileName,
                                         const char *EOPsFileName, const int typeInit,
                                         struct strucParamBEPOPprivate *ptrBEPOP)
{
/*
 * typeInit = 1 : the most precise model (EOPs taken into account, complete CIP nutations)
 * typeInit = 2 : faster (simplified nutations), same precision, but not "up to date"
 * typeInit = 3 : less precise (simplified nutations, no EOPs taken into account)
*/

  if (ptrBEPOP!=NULL)
  {
    FatalError("Pointer ptrBEPOP should be NULL in BEPOP_initDefault2",__FILE__,__LINE__) ;
  }
  ptrBEPOP = BEPOP_initAllocate(ptrBEPOP) ;

  /* initialize ephemeris to INPOP13c and use TT-TDB from INPOP13c */
  BEPOP_initEphemeris(EphemerisFileName,2,ptrBEPOP) ;

  /* Configure the model to compute the Earth's orientation */
  switch (typeInit)
  {
    case 1:
      /* reads EOP from file, degree 7 for interpolation */
      BEPOP_initEOP(EOPsFileName,7,7,ptrBEPOP);
      /* Use of IAU2006/2006A model for CIP */
      init_CIPmodel("CIP/2006/2006A",&ptrBEPOP->ITCRF_Model) ;
    break;
    case 2:
      /* reads EOP from file, degree 7 for interpolation */
      BEPOP_initEOP(EOPsFileName,7,7,ptrBEPOP);
      /* Use of IAU2000/2000B model for CIP */
      init_CIPmodel("CIP/2000/2000B",&ptrBEPOP->ITCRF_Model) ;
    break;
    case 3:
      /* No EOPs are taken into account */
      BEPOP_initEOP("",7,0,ptrBEPOP);
      /* Use of IAU2000/2000B model for CIP */
      init_CIPmodel("CIP/2000/2000B",&ptrBEPOP->ITCRF_Model) ;
    break;

    default:
      FatalError("typeInit is unknown in BEPOP_initDefault",__FILE__,__LINE__);
    break;
  }

  /* sets the ellipsoid */
  BEPOP_initEllipsoid(2, ptrBEPOP) ; /* GRS80, standard of IERS Conventions 2010 */


  return(ptrBEPOP) ;
};



/*!
  Initialize the model for the Celestial Intermediate Pole (CIP)
  @param ModelName (in): name of the model
  @param ptrBEPOP (out): structure of parameters
 */
void BEPOP_initCIPmodel(const char *ModelName, typeParamBEPOP *ptrBEPOP)
{
  /* ModelName can be: "CIP/2000/2000A" or "CIP/2000/2000B" or "CIP/2006/2006A" */
  init_CIPmodel(ModelName,&ptrBEPOP->ITCRF_Model) ;
}




/*!
  Initialize the ephemeris fields in the structure of parameters
  @param FileName (in): file name of the ephemeris
  @param iMethod_TTmTDB (in): method to compute TT-TDB
  @param ptrBEPOP (out): structure of parameters
 */
void BEPOP_initEphemeris(const char *FileName,const int iMethod_TTmTDB,
                               typeParamBEPOP *ptrBEPOP)
{

  /*double ERFA_DJ00 is defined in erfam.h (J2000.0) */

  if ( (int)strlen(FileName) <=0 )
  {
    FatalError("Name of ephemeris file missing in BEPOP_initEphemeris",__FILE__,__LINE__) ;
  }

  /* if necessary, free memory if the pointer was already allocated */
  if (ptrBEPOP->PTR_eph!=NULL)
  {
    BEPOP_closeEphemeris(ptrBEPOP) ; ptrBEPOP->PTR_eph=NULL;
  }

  /* initialize the name of the file*/
  strcpy(ptrBEPOP->EphemFile,FileName);

  /* initialize the pointer to the ephemeris */
  ptrBEPOP->PTR_eph = calceph_open(ptrBEPOP->EphemFile);
  if (ptrBEPOP->PTR_eph==NULL) FatalError("Unable to affect pointer to ephemeris",
                                        __FILE__,__LINE__) ;

  /* initialize the method to compute TT-TDB */
  BEPOP_initMethodTTTDB(iMethod_TTmTDB, ptrBEPOP) ;

  /* initialize the GMs of bodies */
  BEPOP_initGMs(ptrBEPOP) ;
}


/*!
  Initialize the method to compute TT-TDB
  @param iMethod_TTmTDB (in): id of the method
  @param ptrBEPOP (inout): structure of parameters
 */
void BEPOP_initMethodTTTDB(const int iMethod_TTmTDB, typeParamBEPOP *ptrBEPOP)
{

  /* Ephemeris must be initialized before */
  if (ptrBEPOP->PTR_eph==NULL)
  {
    FatalError("Ephemeris must be initialized before the method to compute TT-TDB",__FILE__,
                __LINE__) ;
  }

  /* initialize the method to transform TT into TDB */
  switch (iMethod_TTmTDB)
  {
    case 0: /* TT=TDB */
    case 1: /* use of ERFA eraDtdb function*/
      ptrBEPOP->iMethod_TTmTDB = iMethod_TTmTDB ;
      break ;
    case 2:
    {
      /* One should test if the TT-TDB is present in ephemeris (4D-ephmeris ?) */
      double vec6[6] ;
      int test ;
      test=calceph_compute(ptrBEPOP->PTR_eph, ERFA_DJ00, 0.0, 16, 0, vec6);
      if (!test)
      {
        WarningError("BEPOP warning error: use of ERFA function eraDtdb instead",
                      __FILE__,__LINE__);
        ptrBEPOP->iMethod_TTmTDB = 1 ;
      }
      else
      {
        ptrBEPOP->iMethod_TTmTDB = iMethod_TTmTDB ;
      };
    }
    break;
    default:
      FatalError("iMethod_TTmTDB should be in [0:2] in BEPOP_initEphemeris",__FILE__,
                  __LINE__) ;
    break;
  }

}




/*!
  Initialize the GMs of bodies with values stored in an ephemeris file
  @param ptrBEPOP (inout): structure of parameters
 */
void BEPOP_initGMs(struct strucParamBEPOPprivate *ptrBEPOP)
{
  double value ; /* value of a parameter stored in an ephemeris file */
  int iBody ; /* indice for a body (from Mercury to Pluto, Sun and Earth-Moon barycenter */
  double EMRAT,GM_EMB ; /* Earth-Moon mass ratio and sum of the Moon's and Earth masses */
  double AU ; /* value of astronomical unit */

  /* Ephemeris must be initialized before */
  if (ptrBEPOP->PTR_eph==NULL)
  {
    FatalError("Ephemeris must be initialized before the method to compute TT-TDB",__FILE__,
                __LINE__) ;
  }

  /* initialize the array of GMs of bodies */
  for (iBody=0 ; iBody<12 ; iBody++) ptrBEPOP->tabGMs[iBody] = -999.99 ;
  EMRAT = -999.99 ; /* useless, but avoids "warning uninitialized" at compilation*/
  GM_EMB = -999.99 ; /* useless, but avoids "warning uninitialized" at compilation*/



  if (calceph_getconstant(ptrBEPOP->PTR_eph,"EMRAT",&value)) EMRAT = value ; /* Earth/Moon mass ratio */
  if (calceph_getconstant(ptrBEPOP->PTR_eph,"AU",&value)) AU = value ; /* value of astronomical unit in km */

  if ( calceph_getconstant(ptrBEPOP->PTR_eph,"KSIZER",&value) ) /* detects if inpop file */
  {
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Mer",&value)) ptrBEPOP->tabGMs[1] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Ven",&value)) ptrBEPOP->tabGMs[2] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_EMB",&value)) GM_EMB = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Mar",&value)) ptrBEPOP->tabGMs[4] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Jup",&value)) ptrBEPOP->tabGMs[5] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Sat",&value)) ptrBEPOP->tabGMs[6] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Ura",&value)) ptrBEPOP->tabGMs[7] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Nep",&value)) ptrBEPOP->tabGMs[8] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Plu",&value)) ptrBEPOP->tabGMs[9] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM_Sun",&value)) ptrBEPOP->tabGMs[11] = value ;
  }
  else
  {
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM1",&value)) ptrBEPOP->tabGMs[1] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM2",&value)) ptrBEPOP->tabGMs[2] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GMB",&value)) GM_EMB = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM4",&value)) ptrBEPOP->tabGMs[4] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM5",&value)) ptrBEPOP->tabGMs[5] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM6",&value)) ptrBEPOP->tabGMs[6] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM7",&value)) ptrBEPOP->tabGMs[7] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM8",&value)) ptrBEPOP->tabGMs[8] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GM9",&value)) ptrBEPOP->tabGMs[9] = value ;
    if (calceph_getconstant(ptrBEPOP->PTR_eph,"GMS",&value)) ptrBEPOP->tabGMs[11] = value ;
  }

  /* deduce GMs of Earth and Moon from EMRAT and GM_EMB */
  if ( (EMRAT>0.0) && (GM_EMB > 0.0) )
  {
    ptrBEPOP->tabGMs[3] =EMRAT*GM_EMB/(1.0+EMRAT) ;
    ptrBEPOP->tabGMs[10]=GM_EMB/(1.0+EMRAT) ;
  }
  else
  {
    FatalError("negative value for EMRAT or GM_EMB",__FILE__,__LINE__) ;
  }

  /* check if all GMs are positive and converts to km^3/s^2 */
  for (iBody=1 ; iBody<12 ; iBody++)
  {
    if ( ptrBEPOP->tabGMs[iBody]>0.0 )
    {
      ptrBEPOP->tabGMs[iBody]*=(AU*AU*AU)/(86400.0*86400.0 ); /* convert from AU^3/day^2 to km^3/s^2 */
    }
    else
    {
      FatalError("negative value for GM",__FILE__,__LINE__) ;
    }
  }



}


/*!
  Returns the value of Sun's GM
  @param ptrBEPOP (out): structure of parameters
 */
double BEPOP_getGMSun(const typeParamBEPOP *ptrBEPOP)
{
  /* Ephemeris must be initialized before */
  if (ptrBEPOP->PTR_eph==NULL)
  {
    FatalError("Ephemeris must be initialized before the method to compute TT-TDB",__FILE__,
                __LINE__) ;
  }
  if (ptrBEPOP->tabGMs[11]<=0.0) FatalError("Earth's GM is negative",__FILE__,__LINE__);

  return(ptrBEPOP->tabGMs[11]) ;
}







/*!
  Returns the value of Earth's GM
  @param ptrBEPOP (out): structure of parameters
 */
double BEPOP_getGMEarth(const typeParamBEPOP *ptrBEPOP)
{
  /* Ephemeris must be initialized before */
  if (ptrBEPOP->PTR_eph==NULL)
  {
    FatalError("Ephemeris must be initialized before the method to compute TT-TDB",__FILE__,
                __LINE__) ;
  }

  if (ptrBEPOP->tabGMs[3]<=0.0) FatalError("Earth's GM is negative",__FILE__,__LINE__);
  return(ptrBEPOP->tabGMs[3]) ;
}





/*!
  Initialize the EOPs in the structure of parameters
  @param FileName (in): file name where EOPs are stored (IERS C04 serie)
  @param InterpolationDegree (in): degree of the Lagrangian interpolation for EOPs
  @param ActiveEOPs (in): define which EOPs are taken into account
  @param ptrBEPOP (out): structure of parameters
  @todo desallouer les tableaux avant de reallouer
 */
void BEPOP_initEOP(const char *FileName, const int InterpolationDegree,
                 const int ActiveEOPs,
                 typeParamBEPOP *ptrBEPOP)
{
  /* ActiveEOPs writes in binay abc */
  /*  a=1 => takes into account corrections to CIP coordinates */
  /*  b=1 => takes into account UT1-UTC */
  /*  c=1 => takes into account polar motion */

  /* free memory and pointers if needed */
  BEPOP_closeEOPs(ptrBEPOP) ;

  if ( ( (int)strlen(FileName) <=0) && (ActiveEOPs > 0) )
  {
    FatalError("Name of EOP file missing in BEPOP_initEOP whereas ActiveEOPs > 0",__FILE__,__LINE__) ;
  }
  if ( (ActiveEOPs > 7)|| (ActiveEOPs < 0) )  FatalError("ActiveEOPs must be inside [0:7]",__FILE__,__LINE__) ;

  ptrBEPOP->ActiveEOPs = ActiveEOPs ;
  ptrBEPOP->ITCRF_Model.active_CorrectionCIP = (ActiveEOPs & 0x04)/4 ;
  ptrBEPOP->ITCRF_Model.active_UT1mUTC = (ActiveEOPs & 0x02)/2 ;
  ptrBEPOP->ITCRF_Model.active_PolarMotion = ActiveEOPs & 0x01 ;

  strcpy(ptrBEPOP->EOPfile,FileName); /* initialize the name of the file */

  /* read the Earth Orientation Parameters from the file */
  if ( (int)strlen(FileName) >0 )
  {
    extract_EOP_FromFile(ptrBEPOP->EOPfile, &ptrBEPOP->ArraysAllEOPs );
    if (ptrBEPOP->ArraysAllEOPs.N_EOP<=0) FatalError("No line read in EOP file",__FILE__,__LINE__) ;
  }

  /* Interpolation degree */
  ptrBEPOP->DegreeInterpolationEOP= InterpolationDegree ;

}






/*!
  Initialize the Reference Ellipsoid
  @param EllipsoidId (in): id of the reference ellipsoid
  @param ptrBEPOP (out): structure of parameters
 */
void BEPOP_initEllipsoid(const int EllipsoidId, typeParamBEPOP *ptrBEPOP)
{
   switch (EllipsoidId)
   {
     case 1: /* WGS84 */
     case 2: /* GRS80, standard of IERS Conventions 2010 */
     case 3: /* WGS72 */
       ptrBEPOP->ReferenceEllipsoid=EllipsoidId ;
       break;
     default:
       FatalError("Reference ellipsoid not valid [1:3]",__FILE__,__LINE__) ;
     break;
   }
}


/*!
  Check if all fields of the structure of parameters are correcty initialized
  @param ptrBEPOP (in): structure of parameters
 */
void BEPOP_checkInit( const typeParamBEPOP *ptrBEPOP)
{
  int test;
  double vec6[6] ;
  int j ;
  /*double ERFA_DJ00 is defined in erfam.h (J2000.0) */

  /* is the ephemeris file correctly opened */
  if (ptrBEPOP->PTR_eph==NULL) FatalError("Ephemeris not initilized",__FILE__,__LINE__) ;
  test=calceph_compute(ptrBEPOP->PTR_eph, ERFA_DJ00, 0.0, 3, 12, vec6);
  if (!test) FatalError("Ephemeris file does not contains the BSS-Earth vector at J2000.0",
                        __FILE__,__LINE__) ;
  if (ptrBEPOP->iMethod_TTmTDB==2) /* if use of TT-TDB of the ephemeris */
  {
    test=calceph_compute(ptrBEPOP->PTR_eph, ERFA_DJ00, 0.0, 16, 0, vec6);
    if (!test) FatalError("Ephemeris file does not contains TT-TDB",__FILE__,__LINE__);
  };
  /* Are all GMs positive ? */
  for (j=1;j<12;j++)
  {
    if (ptrBEPOP->tabGMs[j]<=0.0) FatalError("One of the GMs is negative or zero",__FILE__,__LINE__);
  };

  /* are EOPs downloaded */
  if (ptrBEPOP->ActiveEOPs>0) /* at least one of the EOPs must be taken into account */
  {
    if (ptrBEPOP->DegreeInterpolationEOP<=0) FatalError("Interpolation degree of EOPs must be positive",__FILE__,__LINE__);
    if (ptrBEPOP->ArraysAllEOPs.N_EOP<=0) FatalError("Tables contains no EOPs",__FILE__,__LINE__);
  }

  /* is ellipsoid correct ? */
   switch (ptrBEPOP->ReferenceEllipsoid)
   {
     case 1: /* WGS84 */
     case 2: /* GRS80, standard of IERS Conventions 2010 */
     case 3: /* WGS72 */
       /* Nothing to do */
       break;
     default:
       FatalError("Reference ellipsoid not valid [1:3]",__FILE__,__LINE__) ;
     break;
   }

  /* are pointer functions to compute CIP initialized */
  if ( (ptrBEPOP->ITCRF_Model.ptrF_XYS==NULL) || (ptrBEPOP->ITCRF_Model.ptrF_S==NULL) )  FatalError("Pointers to functions not initialized",__FILE__,__LINE__);

};





/*!
  Close the ephemeris and free memory
  @param ptrBEPOP (in): structure of parameters
 */
void BEPOP_closeEphemeris(typeParamBEPOP *ptrBEPOP)
{
  int j ; /* indice for do loop */


  /* close the ephemeris file */
  if (ptrBEPOP->PTR_eph) calceph_close(ptrBEPOP->PTR_eph);
  ptrBEPOP->PTR_eph=NULL ;
  strcpy(ptrBEPOP->EphemFile,"");

  /* reset the method between TT and TDB */
  ptrBEPOP->iMethod_TTmTDB = -1 ;

  /* reset the GMs of bodies */
  for (j=0 ; j<=11 ; j++)
  {
    ptrBEPOP->tabGMs[j] = -999.99 ;
  }

}

/*!
  Close the EOPs and free memory
  @param ptrBEPOP (in): structure of parameters
 */
void BEPOP_closeEOPs(typeParamBEPOP *ptrBEPOP)
{
  /* free all pointer to arrays */
  if (ptrBEPOP->ArraysAllEOPs.MJD_UTC!=NULL)        {  free(ptrBEPOP->ArraysAllEOPs.MJD_UTC) ;       ptrBEPOP->ArraysAllEOPs.MJD_UTC = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.xp!=NULL)             {  free(ptrBEPOP->ArraysAllEOPs.xp) ;            ptrBEPOP->ArraysAllEOPs.xp = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.yp!=NULL)             {  free(ptrBEPOP->ArraysAllEOPs.yp) ;            ptrBEPOP->ArraysAllEOPs.yp = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.UT1mUTC!=NULL)        {  free(ptrBEPOP->ArraysAllEOPs.UT1mUTC) ;       ptrBEPOP->ArraysAllEOPs.UT1mUTC = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.dX!=NULL)             {  free(ptrBEPOP->ArraysAllEOPs.dX) ;            ptrBEPOP->ArraysAllEOPs.dX = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.dY!=NULL)             {  free(ptrBEPOP->ArraysAllEOPs.dY) ;            ptrBEPOP->ArraysAllEOPs.dY = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.UT1mTT!=NULL)         {  free(ptrBEPOP->ArraysAllEOPs.UT1mTT) ;        ptrBEPOP->ArraysAllEOPs.UT1mTT = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.JulianDay_TT1!=NULL)  {  free(ptrBEPOP->ArraysAllEOPs.JulianDay_TT1) ; ptrBEPOP->ArraysAllEOPs.JulianDay_TT1 = NULL ;  };
  if (ptrBEPOP->ArraysAllEOPs.JulianDay_TT2!=NULL)  {  free(ptrBEPOP->ArraysAllEOPs.JulianDay_TT2) ; ptrBEPOP->ArraysAllEOPs.JulianDay_TT2 = NULL ;  };
  ptrBEPOP->DegreeInterpolationEOP=-1 ;
  ptrBEPOP->ActiveEOPs=-1 ;
  ptrBEPOP->ArraysAllEOPs.N_EOP=0 ;

  strcpy(ptrBEPOP->EOPfile,"");

}




/*!
  Free memory and all pointers
  @param ptrBEPOP (in): structure of parameters
  @todo verifier les autres champs que ceux associes a l'ephemeride
 */
typeParamBEPOP* BEPOP_close(typeParamBEPOP *ptrBEPOP)
{

  /* close the ephemeris file */
  BEPOP_closeEphemeris(ptrBEPOP) ;
  /* close the EOPs */
  BEPOP_closeEOPs(ptrBEPOP);

  /* free memory */
  free(ptrBEPOP) ; ptrBEPOP=NULL;
  return(ptrBEPOP);

}






/*!
  Computes the Barycentric Earth Radial Velocity, the Barycentric Time of Light Arrival, ...

  @param alpha_J2000 (in): right ascension of target at J2000, in ICRF (in hour fraction)
  @param delta_J2000 (in): declinaison of target at J2000 in ICRF (in degree)
  @param mua(in): proper motion in right ascension (arcsec/year)
  @param mud(in): proper motion in declinaison (arcsec/year)
  @param year (in): year of the observation
  @param month (in): month of the observation
  @param day (in): day of the observation
  @param hour (in): hour of the observation
  @param minute (in): minute of the observation
  @param second (in): second of the observation (fraction)
  @param TimeToMidExposure (in): time to be added (accout of the exposure time, in second TAI)
  @param StationLongitude (in): longitude of the station, positive East (in degree)
  @param StationLatitude (in): geodetic latitude of the station (in degree)
  @param StationAltitude (in): altitude of the station (in m)
  @param ptrBEPOP (in): pointer to structure containing all constants and parameters
  @param outBEPOP (out): structure containing the results
 */
void BEPOP_compute( const double alpha_2000, const double delta_2000,
                    const double mua, const double mud,
                    const int year, const int month, const int day,
                    const int hour, const int minute, const double second,
                    const double TimeToMidExposure,
                    const double StationLongitude, const double StationLatitude,
                    const double StationAltitude,
                    const typeParamBEPOP *ptrBEPOP,
                    typeOutBEPOP *outBEPOP)
{

  /* local variables */
  double JulianDay_TT[2] ; /* julian day (TT) corresponding to the gregorian date (UTC) */

  /* checks if ephemeris file is correctly open */
  if (!ptrBEPOP->PTR_eph) FatalError("No ephemeris file has been initialised",__FILE__,__LINE__) ;

  /* checks if EOPs are present */
  if ( (ptrBEPOP->ArraysAllEOPs.N_EOP<=0) && (ptrBEPOP->ActiveEOPs>0)   )
  {
     FatalError("No EOP read",__FILE__,__LINE__) ;
  }

  /* Conversion of the UTC Gregorian date into Julian date TT (Terrestrial Time) */
  GregorianDateUTC_to_JulianDayTT(year, month, day, hour, minute, second, JulianDay_TT ) ;

  /* Adds the time to mid exposure (expressed in TAI second, equivalent to TT second */
  if ( JulianDay_TT[1]+TimeToMidExposure/86400.0<1.0 )
  {
    JulianDay_TT[1] += TimeToMidExposure/86400.0 ;
  }
  else
  {
    JulianDay_TT[0] += 1.0 ;
    JulianDay_TT[1] += TimeToMidExposure/86400.0-1.0 ;
  }

  /* Computes parameters for the relevant time in TT */
  BEPOP_compute_viaJD( alpha_2000, delta_2000, mua, mud,
                       JulianDay_TT,
                       StationLongitude, StationLatitude, StationAltitude,
                       ptrBEPOP, outBEPOP);
}




/*!
  Computes the Barycentric Earth Radial Velocity, the Barycentric Time of Light Arrival, ...

  @param alpha_J2000 (in): right ascension of target at J2000, in ICRF (in hour fraction)
  @param delta_J2000 (in): declinaison of target at J2000 in ICRF (in degree)
  @param mua(in): proper motion in right ascension (arcsec/year)
  @param mud(in): proper motion in declinaison (arcsec/year)
  @param JulianDay_TT[2] (in);  julian day (Terrestrial Time), in two parts
  @param StationLongitude (in): longitude of the station, positive East (in degree)
  @param StationLongitude (in): longitude of the station, positive East (in degree)
  @param StationLatitude (in): geodetic latitude of the station (in degree)
  @param StationAltitude (in): altitude of the station (in m)
  @param ptrBEPOP (in): pointer to structure containing all constants and parameters
  @param outBEPOP (out): structure containing the results

 */
void BEPOP_compute_viaJD( const double alpha_2000, const double delta_2000,
                          const double mua, const double mud,
                          const double JulianDay_TT[2],
                          const double StationLongitude, const double StationLatitude,
                          const double StationAltitude,
                          const typeParamBEPOP *ptrBEPOP,
                          typeOutBEPOP *outBEPOP)
{

  /* local variables */
  double alpha, delta ; /* right ascension and declinaison of target at the observation time (radian) */
  double JulianDay_TDB[2] ;/* julian day (TDB) corresponding to the julian day (TT) */
  double JulianDay_UT1[2] ;/* julian day (UT1) corresponding to the julian day (UT1) */
  double JulianDay_TAI[2] ;/* julian day (TAI) corresponding to the julian day (TAI) */
  EarthOrientationParameters_t EOPvalues ; /* values of Earth Orientation Parameters at julian day (TT) */
  double matrix[3][3] ; /* Transformation matrix from ITRF to GCRF */
  double dmatrix[3][3] ; /* time derivative of the transformation matrix from ITRF to ICRF */
  double posStationITRF[3] ; /* station's ITRF cartesian coordinates (position only, in km) */
  double vecStationGCRF[6] ; /* station's GCRF cartesian coordinates (position/velocity, in km and km/s) */
  double vecStationBCRF[6] ; /*station's BCRF cartesian coordinates (position/velocity, in km and km/s) */
  double vecSSB_Earth[6] ; /* Earth's BCRF cartesian coordinates (position/velocity relativ to SSB, in km and km/s) */
  double vecBSS_Sun[6] ; /* Sun's BCRF cartesian coordinates (position/velocity relativ to SSB, in km and km/s) */
  double vecSun_Earth[3] ; /* heliocentric position of the Earth (in km) */
  int status ; /* output status variable for SOFA/ERFA subroutines */
  double vecTarget_ICRF[3]; /* unit vector from Solar System Barycenter to the target, in ICRF */
  double vecTarget_EC2000[3]; /* unit vector from Solar System Barycenter to the target, in mean ecliptic/mean equinox J2000 */
  int j ; /* indice for do loops */
  /* Variables defined in calceph.h :
    CALCEPH_UNIT_KM = 2
    CALCEPH_UNIT_SEC = 8
    CALCEPH_UNIT_RAD = 16
  */
  const int CALCEPH_OUTPUTUNITS=CALCEPH_UNIT_KM + CALCEPH_UNIT_SEC + CALCEPH_UNIT_RAD ;

  /* Variables defined in erfam.h :
       ERFA_DD2R = 1.745329251994329576923691e-2 (from degree to radian)
       ERFA_DAS2R = 4.848136811095359935899141e-6  (from arcsecond to radian)
       ERFA_DJ00 = 2451545.0  ( julian day for J2000.0 )
       ERFA_DJY = 365.25  ( days in a year )
       ERFA_CMPS = 299792458.0  (light velocity, in m/s )
       ERFA_DPI = 3.141592653589793238462643 (value of pi )
       ERFA_DAYSEC = 86400.0 (seconds in a day )
  */

  /* checks if ephemeris file is correctly open */
  if (!ptrBEPOP->PTR_eph) FatalError("No ephemeris file has been initialised",__FILE__,__LINE__) ;

  /* checks if EOPs are present */
  if ( (ptrBEPOP->ArraysAllEOPs.N_EOP<=0) && (ptrBEPOP->ActiveEOPs>0)   )
  {
     FatalError("No EOP read",__FILE__,__LINE__) ;
  }

  /* position of target in ICRF (radians) */
  alpha = ERFA_DPI*alpha_2000/12.0 ; /* converts from decimal hour to radian */
  delta = ERFA_DD2R*delta_2000 ; /* converts from degree to radian */
  if ( (fabs(cos(delta))<1.0e-10) && (fabs(mua)>0.0) )
  {
    printf("Very high declinaison of the target: %23.16e\n",delta_2000);
    WarningError("Possible problem with proper motion (division by 0)",__FILE__, __LINE__) ;
  }
  alpha += ERFA_DAS2R*mua*( (JulianDay_TT[0]-ERFA_DJ00)+JulianDay_TT[1] )/ERFA_DJY/cos(delta) ;
  delta += ERFA_DAS2R*mud*( (JulianDay_TT[0]-ERFA_DJ00)+JulianDay_TT[1] )/ERFA_DJY ;

  /* unit vector from Solar System Barycenter to target, expressed in ICRF */
  /* vecTarget[0:2]=[ cos(a)*cos(d), sin(a)*cos(d), sin(d)  ]*/
  eraS2c(alpha, delta, vecTarget_ICRF); /* spherical to cartesian coordinates */

  /* Interpolation of EOPs at JulianDay_TT*/
  if ( ptrBEPOP->ActiveEOPs>0 )
  {
    interpol_EOP(JulianDay_TT, ptrBEPOP->DegreeInterpolationEOP, ptrBEPOP->ActiveEOPs,ptrBEPOP->ArraysAllEOPs, &EOPvalues) ;
  }

  /* julian day UT1 */
  if (ptrBEPOP->ITCRF_Model.active_UT1mUTC==1) /* if UT1-UTC is taken into account */
  {
    status=eraTtut1(JulianDay_TT[0], JulianDay_TT[1], -EOPvalues.UT1mTT,
                    &JulianDay_UT1[0], &JulianDay_UT1[1]);
    if ( status<0 ) FatalError("Problem in the the sofa/erfa transformation TT->UT1",__FILE__,__LINE__) ;
  }
  else /* UT1-UTC is neglicted, JD_UT1 = "JD_UTC" (not clean) */
  {
    status = eraTttai(JulianDay_TT[0], JulianDay_TT[1], &JulianDay_TAI[0], &JulianDay_TAI[1]) ;
    if ( status<0 ) FatalError("Problem in the the sofa/erfa transformation of TT->TAI",__FILE__,__LINE__) ;
    status = eraTaiutc(JulianDay_TAI[0], JulianDay_TAI[1], &JulianDay_UT1[0], &JulianDay_UT1[1]) ;
    if ( status<0 ) FatalError("Problem in the the sofa/erfa transformation of TAI->UTC=UT1",__FILE__,__LINE__) ;
  }

  /* transformation matrix from ITRF to GCRF, and its time derivative (approximation) */
  itrf_to_icrf_matrix(JulianDay_TT, EOPvalues, matrix, dmatrix, ptrBEPOP->ITCRF_Model );

  /* cartesian coordinates of the station in ITRF, in meters then conversion to km */
  status = eraGd2gc ( ptrBEPOP->ReferenceEllipsoid, StationLongitude*ERFA_DD2R,
                      StationLatitude*ERFA_DD2R, StationAltitude, posStationITRF ) ;
  if (status!=0) FatalError("Problem in return status of eraGd2gc",__FILE__,__LINE__) ;
  for (j=0 ; j<3 ; j++) posStationITRF[j]/=1.0e3 ; /* conversion to km */

  /* cartesian coordinates (position and velocity) of the station in GCRF */
  eraRxp(matrix, posStationITRF, &vecStationGCRF[0]) ; /* position */
  eraRxp(dmatrix, posStationITRF, &vecStationGCRF[3]) ; /* velocity */

  /* cartesian coordinates of the SSB-Earth vector (pos/vel) in ICRF, in km and km/s  */
  TT_to_TDB(ptrBEPOP->PTR_eph, JulianDay_TT, JulianDay_TDB,JulianDay_UT1, posStationITRF, ptrBEPOP->iMethod_TTmTDB); /* TT to TDB */
  calceph_compute_unit(ptrBEPOP->PTR_eph, JulianDay_TDB[0], JulianDay_TDB[1], 3, 12,
                       CALCEPH_OUTPUTUNITS, vecSSB_Earth);

  /* transform GCRF vector to BCRF vector (Lorrentz + potential) */
  vecTT_to_vecTDB(ptrBEPOP->PTR_eph, ptrBEPOP->tabGMs, JulianDay_TDB, &vecStationGCRF[0],
                  vecSSB_Earth, &vecStationBCRF[0]) ;
  for (j=0 ; j<3 ; j++) vecStationBCRF[j+3]=vecStationGCRF[j+3] ; /* velocities are unchanged*/

  /* Adds the position/velocity of the Earth's center of mass */
  for (j=0 ; j<6 ; j++) vecStationBCRF[j]+=vecSSB_Earth[j]   ;

  /* Distance between the Sun and the Earth, in km  */
  calceph_compute_unit(ptrBEPOP->PTR_eph, JulianDay_TDB[0], JulianDay_TDB[1], 11, 12,
                       CALCEPH_OUTPUTUNITS, vecBSS_Sun);
  for (j=0 ; j<3 ; j++) vecSun_Earth[j]=vecSSB_Earth[j]-vecBSS_Sun[j] ; /* velocities useless */
  outBEPOP->SED=norm(3, vecSun_Earth) ; /* in km */


  /* Barycentric Earth Radial Velocity, more rigorously: projection of the barycentric observer's velocity along the target's direction */
    outBEPOP->BERV=DotProduct(3, &vecStationBCRF[3], vecTarget_ICRF); /* in km/s */

  /* Barycentric Observer's Velocity */
    outBEPOP->BOV=norm(3, &vecStationBCRF[3]) ; /* in km/s */

  /* Barycentric Time of Light Arrival (Julian day TDB)  */
    outBEPOP->BTLA= DotProduct(3, &vecStationBCRF[0], vecTarget_ICRF)*1.0e3/(ERFA_CMPS*ERFA_DAYSEC);
    outBEPOP->BTLA+= JulianDay_TDB[1];
    outBEPOP->BTLA+=JulianDay_TDB[0] ;

  /* Approximation of the maximum value of BERV (depends only on the target position) */
    /* Unit vector from SSB to target, expressed in mean ecliptic - mean equinox J2000 */
    /* transformation from ICRF to mean equator /mean equinox J2000 is neglicted */
    eq2000_to_ecl2000(vecTarget_ICRF, vecTarget_EC2000);
    /* Assuming the observer describes a circle whose axis is the pole of the ecliptic, at a speed of 32 km/s */
    outBEPOP->BERVMX=32.0*sqrt(vecTarget_EC2000[0]*vecTarget_EC2000[0]+vecTarget_EC2000[1]*vecTarget_EC2000[1]  ) ;

}


/*!
  Prints the versions of bepop, calceph and erfa
 */
void BEPOP_printVersions(void)
{
  printf("Libraries versions:\n");
  printf("  BEPOP: %d.%d.%d\n",BEPOP_VERSION_MAJOR,BEPOP_VERSION_MINOR,BEPOP_VERSION_PATCH);

  /* For calceph, several methods are available:
   *   (1) via CALCEPH_VERSION_MAJOR, CALCEPH_VERSION_MINOR and CALCEPH_VERSION_PATCH pre-processor variables
   *   (2) via CALCEPH_VERSION_STRING preprocessor variable (if >=2.3.0)
   *   (3) via calceph_getversion_str function (if >=2.3.0)
   * Methods using preprocessor variables gives the CALCEPH version when the BEPOP library is compiled
   * Method using calceph_getversion_str function gives the CALCEPH version used at runtime
   * They can give different results if the executable unsing BEPOP library is created
   * using a libcalceph.la inconsistent with calceph.h
   * In this case, a WARNING is generated.
  */
#ifdef CALCEPH_VERSION_STRING
  /* For calceph-2.3.0 and above */
  char sVersionCalceph[CALCEPH_MAX_CONSTANTNAME] ;
  calceph_getversion_str(sVersionCalceph);
  if (strcmp (sVersionCalceph, CALCEPH_VERSION_STRING)!=0)
  {
    WarningError("  Header and library do not match for CALCEPH",__FILE__,__LINE__);
    printf("    CALCEPH (from CALCEPH_VERSION_STRING) : %s\n",CALCEPH_VERSION_STRING);
    printf("    CALCEPH (from calceph_getversion_str) : %s\n",sVersionCalceph) ;
  }
  else
  {
    printf("  CALCEPH: %s\n",CALCEPH_VERSION_STRING);
  }
#else
  printf("  CALCEPH: %d.%d.%d\n",CALCEPH_VERSION_MAJOR,CALCEPH_VERSION_MINOR,CALCEPH_VERSION_PATCH);
#endif

#ifdef ERFA_VERSION_STRING
  printf("  ERFA: %s\n",ERFA_VERSION_STRING);
#else
  printf("  ERFA: unknown\n");
#endif

};
