/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#include <espdr_led_flat.h>
#include <espdr_detector_signature.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     kext        property list of the extensions (?)
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist* keywords,
                                         const int ext_nb,
                                         cpl_propertylist*** kext) {

    const char* ksuff[2] = {"", ""};
    cpl_propertylist_append_int(keywords,"ESO QC PROC VERSION", 2);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "CONAD", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "BADPIX NB", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_TRUE, kext);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "MASTER MIN FLUX", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_TRUE, CPL_FALSE, CPL_FALSE, kext);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "MASTER MAX FLUX", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_TRUE, CPL_TRUE, CPL_FALSE, kext);
    espdr_copy_qc_ext_key_to_ext(keywords, kext, ext_nb);
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Interpret the command line options and execute the data processing
 @param     parameters  the parameters list
 @param     frameset    the frames list

 In case of failure the cpl_error_code is set.
 TODO:
 1) as a general rule. All pointer variables should be initialized to NULL.
 Moreover any memory allocation of pointer variables should start after the last
 assert/cpl_ensure statement. The reason is that in case an assure/cpl_ensure
 statement fail, if one allocate memory before, one has a memory leak (relevant
 for example for Gasgano data reduction where differently from command-line or
 reflex data reduction, the memory allocated by a failing recipe is not
 automatically cleaned up.
 In conclusion: pls initialise pointer variables to NULL and create the relevant
 corresponding cpl objects after the assert statements just before they are
 needed (and clean the corresponding memory allocation of all imagelist cpl
 objects just after the last function that need them, and not at the very end
 of the function, in order to reduce RAM consumption as soon as possible).

 2) To have a cleaner function I would move the allocation of
 the led_ff_split_by_exp  structure and its initialization within the function
 espdr_frames_split_by_texp.  This is true for any other object that has also
 a function that creates/initializes it (for example in case of parameter
 structures CCD_gom_param, OVSC_param, LED_FF_param),
 and for led_ff_clean_input
 3) I would initialise/create each param structure (CCD_gom_param, OVSC_param,
 LED_FF_param) by a dedicated function, not a global one. This simplify code
 maintainability
 4) The part of printing the value of parameter structure, for debugging purpose,
 should be moved to the corresponding initialization function(s). This would
 simplify a lot the understanding of this function.
 5) Have a function to create and initialize the gain windows structure.

 */
/*---------------------------------------------------------------------------*/
int espdr_led_ff(cpl_parameterlist *parameters,
                        cpl_frameset *frameset, const char* recipe_id){

    const int rec_ntags = 5;
    const char* rec_tags[5] = {ESPDR_LED_FF_RAW, ESPDR_CCD_GEOM,
        ESPDR_LED_FF_GAIN_WINDOWS, ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK};
    int is_required[5] = {1, 1, 1, 1, 0};

	//const char instrument[10] = INSTRUMENT;

	cpl_error_code my_error = CPL_ERROR_NONE;

	/* AMO added */
	cpl_msg_set_level(CPL_MSG_INFO);
	espdr_msg("Starting led_ff");
    espdr_ensure(espdr_check_input_tags(frameset, rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                cpl_error_get_code(), "Wrong input tag!");
	/* Identify the RAW and CALIB frames in the input frameset */
	espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw led_ff images, tagged as LED_FF\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "table specifying windows to compute gain, "
                 "tagged as LED_FF_GAIN_WINDOWS\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as INST_CONFIG");

	/* Extract the raw led_ff frames and put them into led_ff_frames frameset */
	cpl_frameset *led_ff_frames = cpl_frameset_new();
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_LED_FF_RAW,
                                          led_ff_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "LED_FF frames extraction failed.");

    /* AMO Changed way to read CCD geom frame and to init relevant recipe
     * internal parameters structures to simplify this calling function
     */

	cpl_frame* CCD_geom_frame = NULL;
    CCD_geom_frame = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_CCD_geometry *CCD_geom = NULL;
    CCD_geom = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    cpl_frame* inst_config_frame = NULL;
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_inst_config *inst_config = NULL;
    inst_config = espdr_inst_config_init(parameters,
                                         CCD_geom->ext_nb, inst_config_frame);
    espdr_OVSC_param *OVSC_param = NULL;
    OVSC_param = espdr_parameters_OVSC_init(recipe_id, parameters);
    espdr_LED_FF_param *LED_FF_param = NULL;
    LED_FF_param = espdr_parameters_LED_FF_init(recipe_id, parameters);

    int led_ff_frameset_size = 0;
	led_ff_frameset_size = cpl_frameset_get_size(led_ff_frames);
	espdr_Texp_frames *led_ff_clean_input = NULL;
	int nb_of_different_texp = 0;
    my_error = espdr_led_ff_sort_by_texp(led_ff_frames, &led_ff_clean_input,
                                         inst_config,
                                         &nb_of_different_texp);
    if (my_error == CPL_ERROR_ILLEGAL_INPUT) {
        espdr_msg_warning("Exiting");
        espdr_parameters_OVSC_delete(OVSC_param);
        espdr_parameters_LED_FF_delete(LED_FF_param);
        espdr_parameters_CCD_geometry_delete(CCD_geom);
        espdr_parameters_inst_config_delete(inst_config);

        return(CPL_ERROR_ILLEGAL_INPUT);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_led_ff_sort_by_texp failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Add led_ff, CCD_geom & inst_config frames to used_frames */
    cpl_frameset *used_frames = cpl_frameset_duplicate(led_ff_frames);

    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Extract and read the LED_FF gain windows FITS table */
    cpl_frameset *LED_FF_gain_windows_frames = cpl_frameset_new();

    if (CPL_ERROR_NONE != espdr_insert_frame(frameset,
                                             ESPDR_LED_FF_GAIN_WINDOWS,
                                             LED_FF_gain_windows_frames,
                                             used_frames)) {
        cpl_frameset_delete(LED_FF_gain_windows_frames);
        return CPL_ERROR_NULL_INPUT;
    }
    cpl_frameset_delete(LED_FF_gain_windows_frames);

    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame =
        			espdr_get_hpixmap_from_set(frameset, used_frames);

    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_get_mdark_from_set(frameset,
                                                            used_frames);

    /* Extract orders mask frame */
    cpl_frame* orders_mask_frame = espdr_get_orders_mask_from_set(frameset,
                                                                  used_frames);

    //use_hdrl = espdr_parameters_get_int(RECIPE_ID, parameters, "use_hdrl");
    espdr_window_coord *gain_windows = NULL;
    gain_windows = espdr_gain_window_struct_init(frameset, CCD_geom);

    /* Extract all the outputs from the hot pixels frame */
    espdr_msg("Extracting outputs from HOT_PIXELS");
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &hot_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_raw_outputs failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame,
                                              CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, 
                                              &master_dark_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }

    /* Extract all the outputs from the orders mask frame if provided */
    cpl_imagelist *orders_mask_list = NULL;
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }

    // Should we update teh hot pixel mask with the NANs from the image?
    // Normally they go into bad pixle mask

    /* Saturation check on first frame */

    /* Extracting real outputs from the first longest frame */

    espdr_msg("Computing the max flux on raw images");
    cpl_frame *first_longest_led_frame =
        cpl_frameset_get_position(led_ff_clean_input[1].Texp_frameset, 0);
    //const char *first_longest_led_filename = cpl_frame_get_filename(first_longest_led_frame);
    //espdr_msg("First longest frame filename: %s", first_longest_led_filename);
    cpl_frameset *first_longest_led_frameset = cpl_frameset_new();
    my_error = cpl_frameset_insert(first_longest_led_frameset,
                                   cpl_frame_duplicate(first_longest_led_frame));
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_frameset_insert failed: %s",
                 cpl_error_get_message_default(my_error));

    cpl_imagelist *real_first_longest_led_iml = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs_from_raw(first_longest_led_frameset,
                                                   CPL_TYPE_DOUBLE,
                                                   CCD_geom,
                                                   &real_first_longest_led_iml);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs_from_raw failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Computing max flux on raw images */

    double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_first_longest_led_iml),
                                            sizeof(double));
    // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
    if (inst_config->inst_type == NIR) {
        const char *input_filename_led = cpl_frame_get_filename(first_longest_led_frame);
        //espdr_msg("First led frame input filename: %s", input_filename_led);
        cpl_propertylist *keywords_led = cpl_propertylist_load(input_filename_led, 0);
        double texp = cpl_propertylist_get_double(keywords_led, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_first_longest_led_iml, texp);
    }
    my_error = espdr_get_max_flux_ignoring_cosmics(real_first_longest_led_iml,
                                                   hot_pixels_list,
                                                   inst_config->image_cosmics_part,
                                                   max_flux);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Getting max flux failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("MAX FLUX:");
    int i = 0, j = 0, k = 0, index = 0;
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                espdr_msg("Flux of extension %d output [%d,%d] is %lf",
                          i, j, k, max_flux[index]);
                if (max_flux[index] >= inst_config->satur_limit_adu) {
                    espdr_msg_warning(ANSI_COLOR_RED
                                      "Extension %d ouptput [%d, %d] is saturated, max flux = %lf"
                                      ANSI_COLOR_RESET, i, j, k, max_flux[index]);
                }
                index++;
            }
        }
    }

    cpl_imagelist_delete(real_first_longest_led_iml);
    cpl_frameset_delete(first_longest_led_frameset);


    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = NULL;
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

	/* keywords contains all the keywords read from
	 the first input frame header */
	cpl_propertylist *keywords = NULL;

	/* keywords_ext contains keywords of all the extension headers
	 of the first input frame */
	cpl_propertylist **keywords_ext =
            (cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb * sizeof(cpl_propertylist*));
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        keywords_ext[i] = NULL;
    }

    /* Take the header of the first frames with the shortest exposure time */
    const char *input_filename = NULL;
	input_filename =
		cpl_frame_get_filename(cpl_frameset_get_position(led_ff_clean_input[0].Texp_frameset, 0));
	/* Load keywords from the first image
	 (primary header and extensions headers) */
    espdr_msg_debug("KEYWORDS input filename: %s", input_filename);
	keywords = cpl_propertylist_load(input_filename, 0);
    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "keywords are NULL");

	for (i = 0; i < CCD_geom->ext_nb; i++) {
		keywords_ext[i] = cpl_propertylist_load(input_filename, i+1);
	}

	espdr_msg("Detector clean on %d LED_FF input frames (%d extensions each)",
              led_ff_frameset_size, CCD_geom->ext_nb);
    
    int recipes[10] = {1, 1, 0, 0, 0, 0, 0, 0, 0, 0};
    my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords,
                                                    NULL, NULL, NULL, "LED_FF", recipes);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_calibrations_intervals failed: %s",
                 cpl_error_get_message_default(my_error));

    cpl_frameset** set_gain_array = NULL;
    set_gain_array = (cpl_frameset **)cpl_malloc(nb_of_different_texp *
                                                 sizeof(cpl_frameset *));
    cpl_imagelist** iml_mst_array = NULL;
    iml_mst_array = (cpl_imagelist **)cpl_malloc(nb_of_different_texp *
                                                 sizeof(cpl_imagelist *));

    my_error = espdr_oscan_correct_and_stack_frames(CCD_geom, inst_config,
                                                    orders_mask_list,
                                                    led_ff_clean_input,
                                                    nb_of_different_texp,
                                                    &iml_mst_array,
                                                    &set_gain_array);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_oscan_correct_and_stack_frames failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("All the masters created");

    /* Creating bad pixels mask */
    espdr_msg("Creating BAD PIXELS mask");
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_bad_pixels_prepare(iml_mst_array, CCD_geom,
                                        inst_config, qc_kws,
                                        &bad_pixels_list, &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_bad_pixels_prepare failed: %s",
                 cpl_error_get_message_default(my_error));

    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot&bad pixels mask");
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list,
                                                bad_pixels_list,
                                                &pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_hot_bad_pixels_mask failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Compute gain, evaluate flux and save QC KWs */
    espdr_msg("Gain computation ");
    my_error = espdr_prepare_gain(set_gain_array, nb_of_different_texp,
                                  pixels_mask, CCD_geom, qc_kws, inst_config,
                                  gain_windows, keywords_ext, &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_prepare_gain failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Merge all bad pixels output images into one extension,
	 all the merged extensions are in the final_bad_pixels */
	espdr_msg("Merging bad pixels image");
	cpl_imagelist *final_bad_pixels = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(bad_pixels_list,
                                       CCD_geom,
                                       CPL_TYPE_INT,
                                       &final_bad_pixels);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_prepare failed: %s",
                 cpl_error_get_message_default(my_error));
	espdr_msg("Bad pixels image merged");

    my_error = espdr_led_ff_QC(CCD_geom, qc_kws, inst_config, max_flux, &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_led_ff_QC failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Save the PRO.CATG */

    espdr_msg("Adding PRO.CATG KW");
	my_error = cpl_propertylist_append_string(keywords, PRO_CATG_KW,
											  ESPDR_PRO_CATG_BAD_PIXELS);
	if (my_error != CPL_ERROR_NONE) {
		espdr_msg_error("Error adding product category is %s",
						cpl_error_get_message());
	}

    /* ASE : Since led ff raw files has not INSTRUME keyword info
       let's put on keywords list, relying on the
       inst_config->instrument value */

    my_error = cpl_propertylist_append_string(keywords,
                                              "INSTRUME",
                                              inst_config->instrument);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_propertylist_append_string failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Saving the bad pixels FITS frame");

	/* Save the BAD PIXELS fits frame */

    if (strcmp(inst_config->instrument, "HARPN") != 0 ) {
        my_error = espdr_add_extra_qc(keywords, CCD_geom->ext_nb,
                                      &keywords_ext);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
    }

    my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
                                    recipe_id, keywords, keywords_ext,
                                    inst_config->bad_pixels_filename,
                                    final_bad_pixels,
                                    CPL_TYPE_USHORT, CCD_geom);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 inst_config->bad_pixels_filename);

#if SAVE_STATIC_PRODUCT_LED_FF

    my_error = cpl_propertylist_update_double(keywords, "MJD-OBS", 59945.0);

    char *ins_mode = NULL;
    if (cpl_propertylist_has(keywords, "ESO INS MODE")) {
        ins_mode = cpl_propertylist_get_string(keywords, "ESO INS MODE");
    } else {
        ins_mode = (char *)cpl_malloc(sizeof(char));
        ins_mode[1] = '\0';
    }
    char static_filename[64];
    sprintf(static_filename, "%s_%s_bad_pixels_2023-01-01.fits",
            inst_config->instrument, ins_mode);

    //my_error = cpl_propertylist_save(keywords, static_filename, CPL_IO_CREATE);
    //for (int i = 0; i < CCD_geom->ext_nb; i++) {
    //    my_error = cpl_image_save(cpl_imagelist_get(final_bad_pixels, i), static_filename,
    //                              CPL_TYPE_USHORT, keywords_ext[i], CPL_IO_EXTEND);
    //}
    my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
                                    recipe_id, keywords, keywords_ext,
                                    static_filename, final_bad_pixels,
                                    CPL_TYPE_USHORT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), static_filename);

#endif

    espdr_msg("Freeing the memory");

    /* Free the frameset structure */
    my_error = espdr_led_ff_cleanup(set_gain_array, nb_of_different_texp);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_led_ff_cleanup failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Free the propertylists */
	cpl_propertylist_delete(keywords);
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		cpl_propertylist_delete(keywords_ext[i]);
	}
	cpl_free(keywords_ext);
    /* Free the parameters structures */
    cpl_free(gain_windows);
    cpl_free(qc_kws);

    /* Free the images and frames */
    for (i = 0; i < nb_of_different_texp; i++) {
        /* AMO added */
        cpl_imagelist_delete(iml_mst_array[i]);
        cpl_frameset_delete(set_gain_array[i]);
    }


    /* AMO added */
    cpl_free(iml_mst_array);
    cpl_free(set_gain_array);
	cpl_imagelist_delete(bad_pixels_list);
	cpl_imagelist_delete(hot_pixels_list);
	cpl_imagelist_delete(final_bad_pixels);
    cpl_imagelist_delete(pixels_mask);
    /* Could not free it before, their propertylist are used */
    for (i = 0; i < nb_of_different_texp; i++) {
        cpl_frameset_delete(led_ff_clean_input[i].Texp_frameset);
    }
    cpl_free(led_ff_clean_input);
	cpl_frameset_delete(led_ff_frames);
    cpl_frameset_delete(used_frames);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_LED_FF_delete(LED_FF_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

	return cpl_error_get_code();
}




/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for LED_FF input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input LED_FF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_LED_FF_create(const char* recipe_id,
                                              cpl_parameterlist *list,
                                              espdr_LED_FF_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	
	/* Fill the parameter list */
	cpl_error_code error_got;
	char comment[COMMENT_LENGTH];
	double ksigma_min = KSIGMA_MIN;
	double ksigma_max = KSIGMA_MAX;
	int sig_clip_max_iter_min = SIGMA_CLIP_MAX_ITER_MIN;
	int sig_clip_max_iter_max = SIGMA_CLIP_MAX_ITER_MAX;

	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");

	/* Fill the parameter list */
	error_got = espdr_parameters_new_string(recipe_id, list,
			"led_ff_sig_clip_method",
			p->led_ff_sigma_clipping_method,
			"method for sigma clipping in master LED_FF, can be: mean or median");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter led_ff_sigma_clipping_method to the list");

	sprintf(comment,
			"ksigma for sigma clipping in master LED_FF, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);

	error_got = espdr_parameters_new_range_double(recipe_id, list,
			"led_ff_ksigma", p->led_ff_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter led_ff_ksigma to the list");
    
	error_got = espdr_parameters_new_string(recipe_id, list,
			"bad_pix_sig_clip_method",
			p->badpixels_sigma_clipping_method,
			"method for sigma clipping in BAD PIXELS, can be mean or median");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter badpixels_sigma_clipping_method to the list");

	sprintf(comment,
			"ksigma for sigma clipping in BAD PIXELS, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);

	error_got = espdr_parameters_new_range_double(recipe_id, list,
			"bad_pix_ksigma",
			p->badpixels_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter badpixels_ksigma to the list");

	sprintf(comment,
			"maximal number of iterations in BAD PIXELS, must be between: %d and %d",
			sig_clip_max_iter_min, sig_clip_max_iter_max);
	error_got = espdr_parameters_new_range_int(recipe_id, list,
			"bad_pix_max_iter",
			p->badpixels_max_iter,
			SIGMA_CLIP_MAX_ITER_MIN,
			SIGMA_CLIP_MAX_ITER_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter badpixels_max_iter to the list");
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 * AMO added
 @brief     Init the led_ff parameter structure using the recipe parameters
 @param     recipe_id       recipe identifier
 @param     param_list      parameters list
 @return    LED_FF_param iff OK else NULL
 
 TODO: same comment as for previous function regarding the function name.
 */
/*----------------------------------------------------------------------------*/
espdr_LED_FF_param* espdr_parameters_LED_FF_init(const char* recipe_id,
                                                 cpl_parameterlist* param_list) {
    
    espdr_LED_FF_param* LED_FF_param = NULL;
    
    cpl_ensure(recipe_id != NULL, CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(param_list != NULL, CPL_ERROR_NULL_INPUT,NULL);
    
    LED_FF_param = (espdr_LED_FF_param *)cpl_malloc(sizeof(espdr_LED_FF_param));
    espdr_parameters_LED_FF_get(recipe_id, param_list, LED_FF_param);
    
    /* TODO: AMO: this print should be optional or removed after debug phase */
    espdr_parameters_LED_FF_print(LED_FF_param);
    
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        espdr_msg_error("Error during LED_FF parameters initialisation");
        espdr_parameters_LED_FF_delete(LED_FF_param);
    }
    
    return LED_FF_param;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for LED_FF input parameters
 @param    p	input LED_FF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_LED_FF_delete(espdr_LED_FF_param* p) {
	
    cpl_free((void *)p->led_ff_sigma_clipping_method);
    cpl_free((void *)p->badpixels_sigma_clipping_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the led_ff recipe parameters
 @param         recipe_id       recipe identifier
 @param         param_list      parameters list
 @param[out]    LED_FF_param    LED_FF parameters structure
 @return    CPL_ERROR_NONE iff OK
 
 TODO: AMO does not understand why the function has been named
 espdr_parameters_LED_FF_get. It just look that it initializes the
 structure. Better choice would have been: espdr_parameters_LED_FF_init
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_LED_FF_get(const char* recipe_id,
                                           cpl_parameterlist* param_list,
                                           espdr_LED_FF_param *LED_FF_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    LED_FF_param->led_ff_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                            param_list,
                                                            "led_ff_sig_clip_method");
    
    LED_FF_param->led_ff_ksigma = espdr_parameters_get_double(recipe_id,
                                                              param_list,
                                                              "led_ff_ksigma");
    
    LED_FF_param->badpixels_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                                param_list,
                                                            "bad_pix_sig_clip_method");
    
    LED_FF_param->badpixels_ksigma = espdr_parameters_get_double(recipe_id,
                                                                 param_list,
                                                                 "bad_pix_ksigma");
    
    LED_FF_param->badpixels_max_iter = espdr_parameters_get_int(recipe_id,
                                                                param_list,
                                                                "bad_pix_max_iter");
    
	return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief print the LED_FF parameters
 @param LED_FF_param    LED_FF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_LED_FF_print(espdr_LED_FF_param *LED_FF_param) {
	
	espdr_msg("\tLED_FF parameters:");
    espdr_msg("\t\tLED FF stacking sigma clipping method = %s",
              LED_FF_param->led_ff_sigma_clipping_method);
    espdr_msg("\t\tLED FF stacking ksigma = %.2f",
              LED_FF_param->led_ff_ksigma);
    espdr_msg("\t\tLED FF badpixels sigma clipping method = %s",
              LED_FF_param->badpixels_sigma_clipping_method);
    espdr_msg("\t\tLED FF badpixels ksigma = %.2f",
              LED_FF_param->badpixels_ksigma);
    espdr_msg("\t\tLED FF badpixels max iter = %d",
              LED_FF_param->badpixels_max_iter);
    
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * AMO added
 @brief     Init the led_ff parameters structure using the input static table
 @param     set         input recipe frameset
 @param     CCD_geom    CCD geometry of the instrument
 @param     instrument  instrument name
 @return    allocated and filled espdr_window_coord structure iff OK else NULL
 */
/*----------------------------------------------------------------------------*/

espdr_window_coord* espdr_gain_window_struct_init(cpl_frameset* set,
                                                  espdr_CCD_geometry *CCD_geom) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    espdr_window_coord *gain_windows = NULL;
    int i = 0;
    int total_outputs_nb = 0;
    
    if (set == NULL) {
        cpl_error_set(__func__, CPL_ERROR_NULL_INPUT);
        espdr_msg_error("Input frameset is NULL");
        return (NULL);
    }
    
    if (CCD_geom == NULL) {
        cpl_error_set(__func__, CPL_ERROR_NULL_INPUT);
        espdr_msg_error("CCD_geom structure is NULL");
        return (NULL);
    }
    
    total_outputs_nb = CCD_geom->total_output_nb;
    
    gain_windows = (espdr_window_coord *)cpl_malloc(total_outputs_nb *
                                                    sizeof(espdr_window_coord));
    
    my_error = espdr_read_gain_windows_coords(set, CCD_geom, gain_windows);
    if ( my_error != CPL_ERROR_NONE ) {
        cpl_free(gain_windows);
        espdr_msg_error("Error during CCD_geom parameters initialisation");
        return NULL;
    }
    
    espdr_msg("\tGAIN windows coordinates per output:");
    for (i = 0; i < total_outputs_nb; i++) {
        espdr_msg("\t\t[%d]: [%d, %d] -> [%d, %d]", i,
                  gain_windows[i].llx, gain_windows[i].lly,
                  gain_windows[i].urx, gain_windows[i].ury);
    }
    
    return gain_windows;
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Read the LED_FF gain windows parameters from a FITS table
 @param         frameset        input recipe frameset
 @param         instrument      instrument name
 @param         CCD_geom        CCD geometry of the instrument
 @param[out]    gain_windows    gain windows structure
 @return   CPL_ERROR_NONE iff OK
 */
/*--------------------------------------------------------------------------*/
cpl_error_code espdr_read_gain_windows_coords(cpl_frameset *frameset,
                                              espdr_CCD_geometry *CCD_geom,
                                              espdr_window_coord *gain_windows) {
	
    int i, index, outputs_row_nb;
    int ext_no, out_x, out_y;
    int real_nx, real_ny;
    int llx, lly, urx, ury;
    cpl_frameset *LED_FF_gain_windows_frames = cpl_frameset_new();
    cpl_frame *LED_FF_gain_windows_frame = NULL;
	const char *LED_FF_gain_windows_filename = NULL;
	cpl_table *LED_FF_gain_windows_read_coords = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
	
	/* Extract and read the LED_FF gain windows FITS table */
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_LED_FF_GAIN_WINDOWS,
                                          LED_FF_gain_windows_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "LED_FF gain windows coordinates frame extraction failed");
	
	espdr_ensure(LED_FF_gain_windows_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "No LED_FF gain windows coordinates table %s, exiting",
                 cpl_error_get_message());
	
    LED_FF_gain_windows_frame =
                    cpl_frameset_get_position(LED_FF_gain_windows_frames, 0);
    LED_FF_gain_windows_filename =
                    cpl_frame_get_filename(LED_FF_gain_windows_frame);
	
	espdr_ensure(LED_FF_gain_windows_filename == NULL, CPL_ERROR_NULL_INPUT,
                 "The input filename is NULL");
	
	espdr_msg_debug("LED_FF gain windows table filename: %s",
                    LED_FF_gain_windows_filename);
	   
	LED_FF_gain_windows_read_coords = cpl_table_load(LED_FF_gain_windows_filename,
                                                     1, 0);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "Error loading FITS table with LED_FF gain windows coords");
    
    cpl_frameset_delete(LED_FF_gain_windows_frames);
    
    index = 0;
	outputs_row_nb = cpl_table_get_nrow(LED_FF_gain_windows_read_coords);
    //espdr_msg("outputs row nb = %d", outputs_row_nb);
    for (i = 0; i < outputs_row_nb; i++) {
        ext_no = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                   COL_NAME_EXT_NO, i, NULL);
        
        out_x = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                  COL_NAME_OUT_X, i, NULL);
        
        out_y = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                  COL_NAME_OUT_Y, i, NULL);
        
        real_nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
        real_ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
        
        /* Gain windows coordinates: llx, lly, urx, ury */
        llx = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                COL_NAME_GAIN_WINDOW_LLX, i, NULL);
        espdr_ensure((llx >= real_nx) || (llx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real llx coordinate should be within 1 and %d (is %d)",
                     real_nx, llx);
        gain_windows[index].llx = llx;
        
        lly = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                COL_NAME_GAIN_WINDOW_LLY, i, NULL);
        espdr_ensure((lly >= real_ny) || (lly < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real lly coordinate should be within 1 and %d (is %d)",
                     real_ny, lly);
        gain_windows[index].lly = lly;
        
        urx = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                COL_NAME_GAIN_WINDOW_URX, i, NULL);
        espdr_ensure((urx > real_nx) || (urx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real urx coordinate should be within 1 and %d (is %d)",
                     real_nx, urx);
        gain_windows[index].urx = urx;
        
        ury = cpl_table_get_int(LED_FF_gain_windows_read_coords,
                                COL_NAME_GAIN_WINDOW_URY, i, NULL);
        espdr_ensure((ury > real_ny) || (ury < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real urx coordinate should be within 1 and %d (is %d)",
                     real_ny, ury);
        gain_windows[index].ury = ury;
        
        index++;
    }
    
    /*
    for (i = 0; i < CCD_geom->total_output_nb; i++) {
        espdr_msg("window[%d] in espdr_read_gain_windows_coords", i);
        espdr_msg("\tllx: %d", gain_windows[i].llx);
        espdr_msg("\tlly: %d", gain_windows[i].lly);
        espdr_msg("\turx: %d", gain_windows[i].urx);
        espdr_msg("\tury: %d", gain_windows[i].ury);
    }
	*/
    cpl_table_delete(LED_FF_gain_windows_read_coords);
    
	return (cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 * AMO: added
 @brief    Sort frameset by EXPTIME
 @param[in]     led_ff_frames       input set of frames
 @param[out]    led_ff_clean_input  list of frames sorted by texp
 @param         inst_config         instrument config
 @param[out]    nb                  number of different text
 @return   CPL_ERROR_NONE iff OK
 @note this function allocate memory for led_ff_clean_input output
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_led_ff_sort_by_texp(cpl_frameset* led_ff_frames,
                                         espdr_Texp_frames **led_ff_clean_input,
                                         espdr_inst_config *inst_config,
                                         int *nb) {
    
    int size = 0;
    int i = 0;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_Texp_frames* led_ff_split_by_texp=NULL;
    
    cpl_ensure(led_ff_frames != NULL, CPL_ERROR_NULL_INPUT,CPL_ERROR_NULL_INPUT);
    
    size = cpl_frameset_get_size(led_ff_frames);
    espdr_msg("The input frameset contains %d LED_FF frames", size);
    
    led_ff_split_by_texp = (espdr_Texp_frames *)cpl_malloc
                                            (size * sizeof(espdr_Texp_frames));
    for (i = 0; i < size; i++) {
        led_ff_split_by_texp[i].Texp = 0.0;
        led_ff_split_by_texp[i].Texp_frameset = cpl_frameset_new();
    }
    
    my_error = espdr_frames_split_by_texp(led_ff_frames,
                                          &led_ff_split_by_texp,
                                          inst_config, nb);
    if (my_error == CPL_ERROR_ILLEGAL_INPUT) {
        return(CPL_ERROR_ILLEGAL_INPUT);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_frames_split_by_texp failed: %s",
                 cpl_error_get_message_default(my_error));
    
    //for (i = 0; i < size; i++) {
    //    espdr_msg("Texp[%d] = %f, size = %lld",
    //              i, led_ff_split_by_texp[i].Texp,
    //              cpl_frameset_get_size(led_ff_split_by_texp[i].Texp_frameset));
    //}
    
    
    /* Check the input frames exposure time and number */
    /* If there's not enough led_ff frames exiting */
    
    if (*nb < inst_config->raw_led_ff_different_texp) {
        espdr_msg_warning("There should be at least %d valid sets of frames with the same exposure time and minimum %d frames in each set, and there is %d",
                          inst_config->raw_led_ff_different_texp,
                          inst_config->raw_led_ff_per_texp_limit_nb, *nb);
        return(CPL_ERROR_ILLEGAL_INPUT);
    }
    
    // If problem with generating the clean led_ff set, uncomment this code
    //espdr_ensure(*nb < inst_config->raw_led_ff_different_texp,
    //             CPL_ERROR_ILLEGAL_INPUT,
    //             "There should be at least %d sets of frames with the same exposure time, %d frames at least in each set",
    //             inst_config->raw_led_ff_different_texp,
    //             inst_config->raw_led_ff_per_texp_limit_nb);
    
    *led_ff_clean_input = (espdr_Texp_frames *)
                                cpl_malloc(size * sizeof(espdr_Texp_frames));
    for (i = 0; i < size; i++) {
        (*led_ff_clean_input)[i].Texp = 0.0;
        (*led_ff_clean_input)[i].Texp_frameset = cpl_frameset_new();
    }

    my_error = espdr_led_ff_clean_input_set(led_ff_split_by_texp,
                                            inst_config,
                                            *nb,
                                            led_ff_clean_input);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_led_ff_clean_input_set failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("Input raw frames sorted by Texp:");
    for (i = 0; i < *nb; i++) {
        espdr_msg("Frameset no %d: size = %lld, texp = %lf", i,
                        cpl_frameset_get_size((*led_ff_clean_input)[i].Texp_frameset),
                        (*led_ff_clean_input)[i].Texp);
    }
    
    
    /* Free the memory, the led_ff_split_by_texp is not needed anymore */
    for (i = 0; i < size; i++) {
        cpl_frameset_delete(led_ff_split_by_texp[i].Texp_frameset);
    }
    cpl_free(led_ff_split_by_texp);
    
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Verify if input frames contains at least a minimal number of images
            with the same Texp and at least two different Texps and split them
            into separate imagelists
 @param         input_frameset  input frameset
 @param[out]    Texps_RE        array of structure consisting of Texp value and
                                frameset
 @param         inst_config     instrument config
 @param[out]    nb_of_valid_different_texp_RE   number of valid sets with 
                                                different exposure times
 @return   CPL_ERROR_NONE iff OK
 
 TODO: not clear why it is necessary to separate the imagelist in sub-imagelists
 with different TEXPs. Is not sufficient to sort it?
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_frames_split_by_texp(cpl_frameset *input_frameset,
                                          espdr_Texp_frames **Texps_RE,
                                          espdr_inst_config *inst_config,
                                          int *nb_of_valid_different_texp_RE) {
	
	double exposure_time = 0.0;
    cpl_size input_frameset_size = cpl_frameset_get_size(input_frameset);
    int Texp_nbs[input_frameset_size];
    int nb_of_valid_sets = 0;
	cpl_frame *curr_frame = NULL;
	cpl_propertylist *keywords = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i;
	
	//espdr_msg("Starting espdr_frames_split_by_texp()");
	espdr_msg("led_ff frameset size: %lld", input_frameset_size);
	
	espdr_ensure(input_frameset == NULL, CPL_ERROR_NULL_INPUT,
				 "Input frameset is NULL");
    
    for (i = 0; i < input_frameset_size; i++) {
        Texp_nbs[i] = 0;
    }
    
    int nset = cpl_frameset_get_size(input_frameset);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(input_frameset);
    curr_frame = cpl_frameset_iterator_get(iter);
    
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        
        keywords = cpl_propertylist_load(cpl_frame_get_filename(curr_frame), 0);
        exposure_time = cpl_propertylist_get_double(keywords,
                                                    inst_config->Texp_kw);
        exposure_time = round(exposure_time*100.0)/100.0;
        espdr_msg("%s: %lf", cpl_frame_get_filename(curr_frame),
                  exposure_time);
        
        if (exposure_time > 0.0) {
            i = 0;
            while (((*Texps_RE)[i].Texp != exposure_time) &&
                   ((*Texps_RE)[i].Texp != 0.0) && (i < input_frameset_size)) {
                i++;
            }
            
            if (i < input_frameset_size) {
                (*Texps_RE)[i].Texp = exposure_time;
                my_error = cpl_frameset_insert((*Texps_RE)[i].Texp_frameset,
                                                cpl_frame_duplicate(curr_frame));
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_framset_insert failed");
                Texp_nbs[i]++;
            } else {
                espdr_msg_error("Loop out of range");
                return(CPL_ERROR_ACCESS_OUT_OF_RANGE);
            }
        }
        
        cpl_propertylist_delete(keywords);
        cpl_frameset_iterator_advance(iter, 1);
        curr_frame = cpl_frameset_iterator_get(iter);
    }
    cpl_frameset_iterator_delete(iter);
    
    i = 0;
    espdr_msg("Number of frames per Texp (!= 0.0):");
    while ((Texp_nbs[i] > 0) && (i < input_frameset_size)) {
        espdr_msg("Texp = %lf: %d frames", (*Texps_RE)[i].Texp, Texp_nbs[i]);
        //if (Texp_nbs[i] < inst_config->raw_led_ff_per_texp_limit_nb) {
        //    espdr_msg_warning("There should be at least %d frames in each set and there are only %d for Texp = %f",
        //                      inst_config->raw_led_ff_different_texp,
        //                      Texp_nbs[i], (*Texps_RE)[i].Texp);
        //    return(CPL_ERROR_ILLEGAL_INPUT);
        //}
        i++;
    }
    
    nb_of_valid_sets = 0;
    for (i = 0; i < input_frameset_size; i++) {
        if (Texp_nbs[i] >= inst_config->raw_led_ff_per_texp_limit_nb) {
            nb_of_valid_sets++;
        }
    }
    
    espdr_msg("Nb of valid sets: %d", nb_of_valid_sets);
    *nb_of_valid_different_texp_RE = nb_of_valid_sets;
    
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_frames_split_by_texp failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Sort the input framesets by Texp
 @param         all_frames      all input frames, they are already checked for
 number & Texp
 @param         inst_config     instrument config
 @param         valid_sets_nb   number of framesets with Texp > 0.0 and at least
 minimal number of frames
 @param[out]    taken_frames_RE sorted framesets (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_led_ff_clean_input_set(espdr_Texp_frames *all_frames,
                                            espdr_inst_config *inst_config,
                                            int valid_sets_nb,
                                            espdr_Texp_frames **taken_frames_RE) {
    
    int frameset_size = 0;
    int i, j, Texps_index, all_frames_size = 0;
    double Texps[valid_sets_nb];
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    i = 0;
    Texps_index = 0;
    frameset_size = cpl_frameset_get_size(all_frames[i].Texp_frameset);
    while (frameset_size > 0) {
        if (frameset_size >= inst_config->raw_led_ff_per_texp_limit_nb) {
            Texps[Texps_index] = all_frames[i].Texp;
            Texps_index++;
        }
        i++;
        frameset_size = cpl_frameset_get_size(all_frames[i].Texp_frameset);
    }
    all_frames_size = i;
    
    /* sort all the exposure times */
    my_error = espdr_quicksort(Texps, valid_sets_nb);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_quicksort failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* initialize the taken_frames_RE table */
    for (i = 0; i < valid_sets_nb; i++) {
        (*taken_frames_RE)[i].Texp = 0.0;
    }
    
    for (i = 0; i < valid_sets_nb; i++) {
        j = 0;
        while ((all_frames[j].Texp != Texps[i]) && (j < all_frames_size)) {
            j++;
        }
        //espdr_ensure(j >= all_frames_size, CPL_ERROR_ACCESS_OUT_OF_RANGE,
        //             "Out of range of the exposure times table");
        
        (*taken_frames_RE)[i].Texp = Texps[i];
        (*taken_frames_RE)[i].Texp_frameset =
                        cpl_frameset_duplicate(all_frames[j].Texp_frameset);
        //espdr_msg("taken_frames[%d].Texp = %f, size = %lld",
        //          i, (*taken_frames_RE)[i].Texp,
        //          cpl_frameset_get_size((*taken_frames_RE)[i].Texp_frameset));
    }
    
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Remove the pre-overscan and stack
 @param     CCD_geom                CCD geometry
 @param     inst_config             instrument config
 @param     led_ff_clean_input      LED frames organised by Texp
 @param     nb_of_different_texp    nb od diff exposure times
 @param[out] iml_mst_array          returned imagelist of masters
 @param[out] set_gain_array         returned frameset of masters
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/


cpl_error_code espdr_oscan_correct_and_stack_frames(espdr_CCD_geometry* CCD_geom,
                                                    espdr_inst_config *inst_config,
                                                    cpl_imagelist *orders_mask_iml,
                                                    espdr_Texp_frames *led_ff_clean_input,
                                                    const int nb_of_different_texp,
                                                    cpl_imagelist*** iml_mst_array,
                                                    cpl_frameset*** set_gain_array) {

    /* Now we need to do the real job
     *
     * For each frame extension,
     *    For each detector port read-out
     *
     *       1) subtract the corresponding master bias (read-out region)
     *       2) remove cosmics, computing number of CRHs for QC
     *       3) compute extra QC: mean, RON
     *       4) generate bad pixel map
     *       5) compute conversion factor
     *
     *    end loop on detector port read-outs
     *
     *    build a master bad pixel map
     *    build a master hot pixel map
     *    merge QC computed on each detector extension
     *    save result on proper extension
     *
     * end loop over extensions
     *
     */
    

    int use_hdrl = 0;
    cpl_error_code my_error;
    cpl_propertylist* plist;
    cpl_type frame_type = CPL_TYPE_DOUBLE;
    char sub_name[80];
    int index = 0;
    for (int it = 0; it < nb_of_different_texp; it++) {
        espdr_msg("Creating master no %d", it);
        cpl_frameset* set = led_ff_clean_input[it].Texp_frameset;
        cpl_frame* frame = NULL;
        cpl_image* raw_port = NULL;
        //cpl_image* mst_port = NULL;
        cpl_image* cor_port = NULL;
        cpl_image *orders_mask_image = NULL;
        
        int ix, iy;
        int llx,lly,urx,ury;
        //cpl_frameset_dump(set, stdout);
        
        int imagelist_place = 0;
        
        double RON_LED_FF[CCD_geom->total_output_nb];
        double RON_fake = 0.0;
        double CONAD_fake = 1.0;

        int cosmics_tot = 0;
        int cosmics_port = 0;
        int cosmics_ext[CCD_geom->ext_nb];
        
        (*iml_mst_array)[it] = cpl_imagelist_new();
        int counter = 0;
        (*set_gain_array)[it] = cpl_frameset_new();
        const char* fname;
        
        /* process each frame extension */
        for (int i = 0; i < CCD_geom->ext_nb; i++) {
            
            cosmics_ext[i] = 0;
            
            /* not needed to load the whole image: it is sufficient to
             * load the part we are processing
             */
            
            /* process each read-out port */
            ix = 0;
            for (int j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
                
                llx = ix+1;
                urx = ix+CCD_geom->exts[i].outputs[j][0].raw_nx;
                iy = 0;
                
                for (int k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                    
                    lly = iy+1+CCD_geom->exts[i].outputs[j][k].ppscan_ny;
                    ury = iy+CCD_geom->exts[i].outputs[j][k].raw_ny
                    -CCD_geom->exts[i].outputs[j][k].poscan_ny;
                    
                    cpl_imagelist* iml_cor = NULL;
                    iml_cor = cpl_imagelist_new();
                    
                    /* correct for master bias each portion from each frame
                     * of a given exposure time
                     */
                    /* apply bias correction: use master bias or oscan region as
                     * set by the user
                     */
                    
                    int nset= cpl_frameset_get_size(set);
                    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(set);
                    frame = cpl_frameset_iterator_get(iter);
                    
                    for ( index = 0; index < nset; index++ ) {
                        /* load portion of image corresponding to a given
                         read-out-port */
                        
                        cpl_error_reset();
                        fname = cpl_frame_get_filename(frame);
                        raw_port = cpl_image_load_window(fname, frame_type, 0, i+1,
                                                         llx, lly, urx, ury);
                        my_error = cpl_error_get_code();
                        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                     "Error when reading output [%d, %d] in extension %d of the frame: %s: %s",
                                     j, k, i, fname,
                                     cpl_error_get_message_default(my_error));
                        
                        plist = cpl_propertylist_load(fname, 0);
                        
                        if (inst_config->inst_type == NIR) {
                            
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
                            if (cpl_propertylist_has(plist, "ARCFILE") == 0) {
                                char *arc_filename = strrchr(fname, '/')+1;
                                espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
                                my_error = cpl_propertylist_update_string(plist, "ARCFILE", arc_filename);
                                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                             "Updating the ARCFILE KW (%s) failed: %s",
                                             arc_filename, cpl_error_get_message_default(my_error));
                            }
#endif
                            
                            if (orders_mask_iml != NULL) {
                                orders_mask_image = cpl_imagelist_get(orders_mask_iml, 0);
                            }
                            raw_port = cpl_image_load(fname, frame_type, 0, i+1);
                            my_error = espdr_preprocess_img(raw_port,
                                                            orders_mask_image,
                                                            plist,
                                                            CCD_geom, inst_config,
                                                            &RON_fake, &CONAD_fake,
                                                            &cor_port);
                            
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "preprocesing on imagelist failed: %s",
                                         cpl_error_get_message_default(my_error));
                        } else {
                            /* get oscan correction */
                            cpl_image* oscan_port = NULL;
                            double RON_OV;
                            int cosmics_nb;
                            
                            if(use_hdrl) {
                                espdr_hdrl_overscan(raw_port,i, j, k, CCD_geom, inst_config,
                                                    &oscan_port, &RON_OV, &cosmics_nb);
                            } else {
                                my_error = espdr_overscan(raw_port,i, j, k, CCD_geom, inst_config,
                                                          &oscan_port, &RON_OV, &cosmics_nb);
                            }
                            //espdr_msg("oscan ron=%lf crh=%g",RON_OV,cosmics_nb);
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "Overscan failed: %s",
                                         cpl_error_get_message_default(my_error));
                            
                            espdr_remove_overscan(raw_port,i, j, k, CCD_geom,
                                                  oscan_port, &cor_port);
                            
                            cpl_image_delete(oscan_port);
                        }
                        
                        /* We save each corrected image on disk. Later this will
                         * be loaded again */
                        sprintf(sub_name,
                                "led_ff2_cor_exptime_%d_ext_%d_port_x%d_y%d_index_%d.fits",
                                it, i, j, k, counter);
                        cpl_frame* frame_cor = NULL;
                        frame_cor = cpl_frame_new();
                        
                        cpl_frame_set_filename(frame_cor, sub_name);
                        cpl_frame_set_tag(frame_cor, "DUMMY");
                        
                        cpl_frameset_insert((*set_gain_array)[it],
                                            cpl_frame_duplicate(frame_cor));
                        
                        cpl_image_save(cor_port, sub_name, CPL_BPP_IEEE_FLOAT,
                                       plist, CPL_IO_DEFAULT);
                        cpl_propertylist_delete(plist);
                        
                        /* store result on an imagelist (of small size) to later
                         * make the image stacking */
                        cpl_imagelist_set(iml_cor, cpl_image_duplicate(cor_port),
                                          index);
                        
                        /* cleanup memory */
                        cpl_image_delete(raw_port);
                        cpl_image_delete(cor_port);
                        cpl_frame_delete(frame_cor);
                        
                        counter++;
                        cpl_frameset_iterator_advance(iter, 1);
                        frame = cpl_frameset_iterator_get(iter);
                        
                    } /* end loop over frames of same texp */
                    cpl_frameset_iterator_delete(iter);
                    
                    cpl_image* mst_port = NULL;
                    
                    /* remove CRH via sigma-clipping and create a master frame
                     * on each image port. Compute corresponded RON and number
                     * of detected CRHs
                     */
                    
                    int real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                    int real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                    int real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                    int real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                    int used_real_nx = real_urx - real_llx + 1;
                    int used_real_ny = real_ury - real_lly + 1;
                    
                    if (strcmp(inst_config->led_ff_sigma_clipping_method, "median") == 0) {
                        espdr_stack_sigma_one_output_median(iml_cor,
                                                            used_real_nx,
                                                            used_real_ny,
                                                            inst_config->led_ff_ksigma,
                                                            nset - 1,
                                                            &mst_port,
                                                            &RON_LED_FF[imagelist_place],
                                                            &cosmics_port);
                    } else {
                        espdr_stack_sigma_one_output_mean(iml_cor,
                                                          used_real_nx,
                                                          used_real_ny,
                                                          inst_config->led_ff_ksigma,
                                                          nset - 1,
                                                          &mst_port,
                                                          &RON_LED_FF[imagelist_place],
                                                          &cosmics_port);
                    }
                    /*
                     espdr_stack_sigma_one_output(iml_cor,
                     used_real_nx, used_real_ny,
                     inst_config->led_ff_ksigma,
                     inst_config->led_ff_sigma_clipping_method,
                     nset - 1,
                     &mst_port,
                     &RON_LED_FF[imagelist_place],
                     &cosmics_port);
                     */
                    
                    double master_mean = cpl_image_get_mean(mst_port);
                    /* AMO: Check intermediate results */
                    espdr_msg("master mean %d[%d, %d] is %lf", i, j, k, master_mean);
                    cosmics_ext[i] += cosmics_port;
#if SAVE_DEBUG_PRODUCT_LED_FF
                    char filename_mst_img[FILENAME_LENGTH];
                    sprintf(filename_mst_img,
                            "%s_LED_mst_img_ext_%d_output_%d-%d_%.0f.fits",
                            inst_config->instrument, i, j, k,
                            led_ff_clean_input[it].Texp);
                    my_error = cpl_image_save(mst_port,
                                              filename_mst_img,
                                              CPL_TYPE_FLOAT,
                                              NULL, CPL_IO_CREATE);
                    espdr_msg("Master image saved for ext %d output [%d, %d] in %s",
                              i, j, k, filename_mst_img);
#endif
                    cpl_imagelist_set((*iml_mst_array)[it],
                                      cpl_image_duplicate(mst_port),
                                      imagelist_place);
                    cpl_image_delete(mst_port);
                    cpl_imagelist_delete(iml_cor);
                    imagelist_place++;
                    iy = ury + CCD_geom->exts[i].outputs[j][k].poscan_ny;
                } /* end loop over y direction read-out ports */
                
                ix = urx;
            } /* end loop over x direction read-out ports */
            
            espdr_msg("total cosmics on extension %d is %d",i,cosmics_ext[i]);
            cosmics_tot += cosmics_ext[i];
        } /* end loop over extensions */
        
        espdr_msg("total cosmics is %d",cosmics_tot);
    } /* end loop over it (nb_of_fifferent_texp) */
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Prepare the bad pixels computation
 @param         master_led_ff_list  list of master led_ff imagelists with different
                                    Texps, each imagelist contains one output images
 @param         CCD_geom            CCD geometry parameters structure
 @param         inst_config         instrument config
 @param         qc_kws              KWs names
 @param         LED_FF param        LED_FF parameters structure
 @param[out]    bad_pixel_mask_list_RE  imagelist with bad pixel mask (per output)
                                    (returned)
 @param[out]    keywords_RE         primary header keywords list (returned)
 @return   CPL_ERROR_NONE iff OK
 
 TODO:
 1) remove commented-out code
 2) There are two loops over the same indexes. Why not merge them in a single loop?
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_bad_pixels_prepare(cpl_imagelist **master_led_ff_list,
                                        espdr_CCD_geometry *CCD_geom,
                                        espdr_inst_config *inst_config,
                                        espdr_qc_keywords *qc_kws,
                                        cpl_imagelist **bad_pixel_mask_list_RE,
                                        cpl_propertylist **keywords_RE) {
    
    int total_output_nb = CCD_geom->total_output_nb;
    int real_llx, real_lly, real_urx, real_ury, used_real_nx, used_real_ny;
	double *mean = NULL;
	int *bad_pixels = NULL;
	int total_bad_pixels;
	double sigma[total_output_nb];
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i, j, k, l, index;
	char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int bad_pixels_QC = 1;
    cpl_imagelist *relative_images = cpl_imagelist_new();
    cpl_imagelist *dead_pixel_imagelist = cpl_imagelist_new();
    cpl_image *rel_image = NULL;
    cpl_image *master1 = NULL, *master2 = NULL;
    cpl_image *dead_pixel_image = NULL;
    double *data_master1 = NULL;
    double *data_master2 = NULL;
    int *dead_pixel_mask = NULL;
    double *rel_data = NULL;
    double rel_mean = 0.0;

    
    espdr_ensure(master_led_ff_list[0] == NULL, CPL_ERROR_NULL_INPUT,
                 "First input master imagelist is NULL");
    
    espdr_ensure(master_led_ff_list[1] == NULL, CPL_ERROR_NULL_INPUT,
                 "Second input master imagelist is NULL");
    
    espdr_msg_debug("First imagelist size: %lld",
                    cpl_imagelist_get_size(master_led_ff_list[0]));
    
    espdr_msg_debug("Second imagelist size: %lld",
                    cpl_imagelist_get_size(master_led_ff_list[1]));
    
    mean = (double *)cpl_calloc(total_output_nb, sizeof(double));
    bad_pixels = (int *)cpl_calloc(total_output_nb, sizeof(int));
    
    
    index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
        
                master1 = cpl_imagelist_get(master_led_ff_list[0], index);
                data_master1 = cpl_image_get_data_double(master1);
                master2 = cpl_imagelist_get(master_led_ff_list[1], index);
                data_master2 = cpl_image_get_data_double(master2);
                rel_mean = cpl_image_get_mean(master2)/cpl_image_get_mean(master1);
                
                real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;

                rel_data = (double *)cpl_calloc(used_real_nx * used_real_ny,
                                                sizeof(double));
                dead_pixel_mask = (int *)cpl_calloc(used_real_nx * used_real_ny,
                                                    sizeof(int));
        
                for (l = 0; l < used_real_nx * used_real_ny; l++) {
                    // Avoid the zero division
                    if (data_master1[l] == 0.0) {
                        rel_data[l] = rel_mean;
                    } else {
                        rel_data[l] = data_master2[l]/data_master1[l];
                    }
                    if ((data_master1[l] < inst_config->dead_pixel_flux_limit) ||
                        (data_master2[l] < inst_config->dead_pixel_flux_limit)) {
                        dead_pixel_mask[l] = 1;
                    } else {
                        dead_pixel_mask[l] = 0;
                    }
                }
        
                rel_image = cpl_image_wrap_double(used_real_nx, used_real_ny,
                                                  rel_data);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_wrap_double failed for rel_image: %s",
                             cpl_error_get_message_default(my_error));
                
#if SAVE_DEBUG_PRODUCT_LED_FF
                char filename_rel_img[FILENAME_LENGTH];
                sprintf(filename_rel_img, "%s_LED_rel_img_ext_%d_output_%d-%d.fits",
                        inst_config->instrument, i, j, k);
                cpl_image *rel_img = cpl_image_wrap_double(used_real_nx,
                                                           used_real_ny, rel_data);
                my_error = cpl_image_save(rel_img,
                                          filename_rel_img,
                                          CPL_TYPE_FLOAT,
                                          NULL, CPL_IO_CREATE);
                espdr_msg("Relative image saved for ext %d output [%d, %d] in %s",
                          i, j, k, filename_rel_img);
#endif
                
                dead_pixel_image = cpl_image_wrap_int(used_real_nx, used_real_ny,
                                                      dead_pixel_mask);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_wrap_double failed for dead_pixels_image: %s",
                             cpl_error_get_message_default(my_error));
        
                my_error = cpl_imagelist_set(relative_images,
                                             cpl_image_duplicate(rel_image),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed for relative_images: %s",
                             cpl_error_get_message_default(my_error));
                
                my_error = cpl_imagelist_set(dead_pixel_imagelist,
                                             cpl_image_duplicate(dead_pixel_image),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed for dead_pixel_imagelist: %s",
                             cpl_error_get_message_default(my_error));
        
                cpl_image_unwrap(rel_image);
                cpl_free(rel_data);
                cpl_image_unwrap(dead_pixel_image);
                cpl_free(dead_pixel_mask);

                
                index++;
            }
        }
    }
    
    
    my_error = espdr_bad_pixels(relative_images, dead_pixel_imagelist,
                                CCD_geom, inst_config,
                                mean, bad_pixels, sigma,
                                bad_pixel_mask_list_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_bad_pixels failed: %s",
                 cpl_error_get_message_default(my_error));
    
	total_bad_pixels = 0;
	for (i = 0; i < total_output_nb; i++) {
		espdr_msg_debug("mean[%d] = %lf", i, mean[i]);
		espdr_msg_debug("bad_pixels[%d] = %d", i, bad_pixels[i]);
		espdr_msg_debug("sigma[%d] = %lf", i, sigma[i]);
        espdr_msg_debug("image min = %lf",
                  cpl_image_get_min(cpl_imagelist_get(relative_images, i)));
        espdr_msg_debug("image max = %lf",
                  cpl_image_get_max(cpl_imagelist_get(relative_images, i)));
        espdr_msg_debug("image mean = %lf",
                  cpl_image_get_mean(cpl_imagelist_get(relative_images, i)));
		total_bad_pixels = total_bad_pixels + bad_pixels[i];
	}
	espdr_msg("Total number of bad pixels: %d", total_bad_pixels);
	
    index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
				new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_badpix_nb_kw_part,
                                                  inst_config->prefix,
                                                          i, j, k);
                sprintf(comment, "Badpix nb for ext %d, out %d, %d", i, j, k);
                
				my_error = espdr_keyword_add_int(new_keyword,
                                                 bad_pixels[index],
                                                 comment,
                                                 keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to th propertylist failed: %s",
                             new_keyword,
                             cpl_error_get_message_default(my_error));
				cpl_free(new_keyword);
                
				index++;
			}
		}
	}
    
    if (total_bad_pixels > inst_config->bad_pixels_limit) {
        bad_pixels_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_ledff_badpix_check_kw,
                                     bad_pixels_QC,
                                     "Bad pixels QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Add keyword QC_LEDFF_BADPIX_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(relative_images);
    cpl_imagelist_delete(dead_pixel_imagelist);
    cpl_free(mean);
    cpl_free(bad_pixels);
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Compute bad pixels mask via sigma_clipping
 @param         master_led_ff_list  imagelist of one image outputs
 @param         dead_pixel_imagelist  imagelist with dead pixels location
 @param         CCD_geom            CCD geometry parameters structure
 @param         LED_FF_param        LED_FF parameters struture
 @param[out]    mean_RE             mean of the image via sigma_clipping
 @param[out]    bad_pixels_RE       number of bad pixels
 @param[out]    sigma_RE            sigma of bad pixels via sigma_clipping
 @param[out]    bad_pixel_mask_list_RE  bad pixels mask computed via sigma_clipping
 @return   CPL_ERROR_NONE iff OK
 
 TODO:
 1) there is no need for two different input imagelist. The agreed bad pixel
 notation imply different codes for different bad pixel kind.  Thus a single
 bad pixel imagelist (actually a single bad pixel map) should be sufficient.
 This would also release a lot the RAM consumption.
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_bad_pixels(cpl_imagelist *master_led_ff_list,
                                cpl_imagelist *dead_pixel_imagelist,
                                espdr_CCD_geometry *CCD_geom,
                                espdr_inst_config *inst_config,
                                double *mean_RE,
                                int *bad_pixels_RE,
                                double *sigma_RE,
                                cpl_imagelist **bad_pixel_mask_list_RE) {
	
	int i, j, k, l, index;
    int real_llx, real_lly, real_urx, real_ury, used_real_nx, used_real_ny;
	cpl_image *curr_image = NULL;
    cpl_image *bad_pixel_image = NULL, *curr_dead_pixel_mask = NULL;
    cpl_image *final_bad_pixel_image = NULL;
	cpl_vector *curr_image_vector = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	double CoD = 0.0, sigma = 0.0, mean = 0.0, stddev = 0.0;
	double low_limit = 0.0, high_limit = 0.0;
	int bad_pixels;
    /* Value for the MAD, in case it is equal to 0.0 */
    double forced_MAD = 0.01;
    int *bad_pixel_data = NULL;
    int *dead_pixel_data = NULL;
    
    espdr_msg("Starting bad pixels calculation with ksigma = %f",
              inst_config->badpixels_ksigma);
    
    espdr_ensure(master_led_ff_list == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL, exiting");
    
    espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "CCD_geom structure is NULL, exiting");
    
    espdr_ensure(inst_config == NULL, CPL_ERROR_NULL_INPUT,
                 "inst_config structure is NULL, exiting");
    
    int total_output_nb = CCD_geom->total_output_nb;

    espdr_ensure((cpl_imagelist_get_size(master_led_ff_list) != total_output_nb),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "input imagelist has different nb of images (%lld) then the total number of outputs (%d)",
                 cpl_imagelist_get_size(master_led_ff_list), total_output_nb);
    
    espdr_ensure((cpl_imagelist_get_size(master_led_ff_list) != total_output_nb),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "dead pixels imagelist has different nb of images then the total number of outputs");
    
    index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                
                curr_image = cpl_imagelist_get(master_led_ff_list, index);
                curr_dead_pixel_mask = cpl_imagelist_get(dead_pixel_imagelist,
                                                         index);
		
                real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;
                
                curr_image_vector = cpl_vector_wrap(used_real_nx * used_real_ny,
											cpl_image_get_data_double(curr_image));
                my_error = espdr_sig_clip(curr_image_vector,
                                          inst_config->badpixels_ksigma,
                                          inst_config->badpixels_ksigma,
                                          inst_config->badpixels_sigma_clipping_method,
                                          inst_config->badpixels_max_iter, 1, forced_MAD,
                                          &CoD, &sigma, &bad_pixels,
                                          &mean, &stddev,
                                          &low_limit, &high_limit);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_sig_clip() failed");
                cpl_vector_unwrap(curr_image_vector);
		
                espdr_msg_debug("BAD PIXELS: CoD = %.2lf, MAD = %.2lf, mean = %.2lf, stddev = %.2lf",
                                CoD, sigma, mean, stddev);
                espdr_msg_debug("BAD PIXELS: lower: %lf, upper: %lf, bad_nb: %d",
                                low_limit, high_limit, bad_pixels);
        
                bad_pixel_image = cpl_image_new(used_real_nx, used_real_ny,
                                                CPL_TYPE_INT);
		
                my_error = espdr_create_mask(curr_image,
                                             used_real_nx * used_real_ny,
                                             low_limit, high_limit,
                                             PERMANENT_CCD_DEFECT,
                                             &bad_pixel_image);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_create_mask() failed");
		
        
                bad_pixel_data = cpl_image_get_data_int(bad_pixel_image);
                dead_pixel_data = cpl_image_get_data_int(curr_dead_pixel_mask);
                
                int total_bad_pixels = 0;
                for (l = 0; l < used_real_nx * used_real_ny; l++) {
                    if ((bad_pixel_data[l] == 0) && (dead_pixel_data[l] == 0)) {
                        bad_pixel_data[l] = 0;
                    } else {
                        bad_pixel_data[l] = PERMANENT_CCD_DEFECT;
                        total_bad_pixels++;
                    }
                }
                espdr_msg_debug("Bad pixels number for output %d: %d",
                                index, total_bad_pixels);
                
        
                final_bad_pixel_image = cpl_image_wrap_int(used_real_nx,
                                                           used_real_ny,
                                                           bad_pixel_data);
                
                mean_RE[index] = mean;
                bad_pixels_RE[index] = total_bad_pixels;
                sigma_RE[index] = stddev;
                cpl_imagelist_set(*bad_pixel_mask_list_RE,
                                  cpl_image_duplicate(final_bad_pixel_image),
                                  index);
                cpl_image_unwrap(final_bad_pixel_image);
                cpl_image_delete(bad_pixel_image);
                
                index++;
            }
        }
	}
	
	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Compute hot&bad pixels mask
 @param         hot_pixels_list imagelist of hot pixels
 @param         bad_pixels_list imagelist of bad pixels
 @param[out]    pixel_mask_RE   computed bad AND hot pixels mask (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_create_hot_bad_pixels_mask(cpl_imagelist *hot_pixels_list,
                                                cpl_imagelist *bad_pixels_list,
                                                cpl_imagelist **pixel_mask_RE) {
    
    int i;
    cpl_image *curr_image = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(hot_pixels_list == NULL, CPL_ERROR_NULL_INPUT,
                 "hot_pixels_list is NULL");
    espdr_ensure(bad_pixels_list == NULL, CPL_ERROR_NULL_INPUT,
                 "bad_pixels_list is NULL");
    
    espdr_msg_debug("hot_pixels size: %lld",
                    cpl_imagelist_get_size(hot_pixels_list));
    espdr_msg_debug("bad_pixels size: %lld",
                    cpl_imagelist_get_size(bad_pixels_list));
   
    espdr_ensure(cpl_imagelist_get_size(hot_pixels_list) !=
                 cpl_imagelist_get_size(bad_pixels_list),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "hot_pixels list and bad_pixels list length differs");
    for (i = 0; i < cpl_imagelist_get_size(hot_pixels_list); i++) {
        curr_image = cpl_image_add_create(cpl_imagelist_get(hot_pixels_list, i),
                                          cpl_imagelist_get(bad_pixels_list, i));
        
        my_error = cpl_imagelist_set(*pixel_mask_RE,
                                     cpl_image_duplicate(curr_image), i);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_imagelist_set failed: %s",
                     cpl_error_get_message_default(my_error));
        cpl_image_delete(curr_image);
    }
    
    espdr_msg_debug("hot&bad pixels mask created");
    
	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * AMO added (uses less RAM)
 @brief     Compute conad and flux QC keywords
 @param         set                 input frameset, bias removed
 @param         nb_of_sets          number of images with different Texps
 @param         pixels_mask         bad&hot pixels mask
 @param         CCD_geom            CCD geometry parameters structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param         LED_FF_param        LED_FF parameters structure
 @param         gain_windows        windows structure specifying where 
                                    to compute the conversion factor
 @param         keywords_ext        extension FITS header
 @param[out]    keywords            primary FITS header (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_prepare_gain(cpl_frameset **set,
                                  int nb_of_sets,
                                  cpl_imagelist *pixels_mask,
                                  espdr_CCD_geometry *CCD_geom,
                                  espdr_qc_keywords *qc_kws,
                                  espdr_inst_config *inst_config,
                                  espdr_window_coord *gain_windows,
                                  cpl_propertylist **keywords_ext,
                                  cpl_propertylist **keywords_RE) {
    int i, j, k, index;
    double *conad_vector = NULL;
    double *min_flux_vector = NULL;
    double *max_flux_vector = NULL;
    double *mean_flux_vector = NULL;
    double min_flux, max_flux;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    cpl_error_code my_error = CPL_ERROR_NONE;
    int total_output_nb = CCD_geom->total_output_nb;
    int flux_QC = 1;
    
    
    /* Compute conad */
    espdr_msg_debug("Computing the conversion factor");
    conad_vector = (double *)cpl_calloc(total_output_nb, sizeof(double));
    mean_flux_vector = (double *)cpl_calloc(total_output_nb, sizeof(double));

    my_error = espdr_compute_gain(set, nb_of_sets,
                                  pixels_mask, CCD_geom, inst_config,
                                  gain_windows, conad_vector, mean_flux_vector);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_gain failed: %s",
                 cpl_error_get_message_default(my_error));
    
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("conversion factor[%d] = %lf, mean_flux[%d] = %fl", 
			i, conad_vector[i], i, mean_flux_vector[i]);
    }
    
    
    /* Evaluate flux */
    espdr_msg("Evaluating flux");
    min_flux_vector = (double *)cpl_calloc(total_output_nb, sizeof(double));
    max_flux_vector = (double *)cpl_calloc(total_output_nb, sizeof(double));
    
    my_error = espdr_evaluate_flux(set, nb_of_sets,
                                   pixels_mask, CCD_geom,
                                   min_flux_vector, max_flux_vector);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_evaluate_flux failed: %s",
                 cpl_error_get_message_default(my_error));
    
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("min_flux[%d] = %lf", i, min_flux_vector[i]);
    }
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("max_flux[%d] = %lf", i, max_flux_vector[i]);
    }
    
    min_flux = MAX_FLUX;
    max_flux = 0.0;
    for (i = 0; i < total_output_nb; i++) {
        if (min_flux_vector[i] < min_flux) {
            min_flux = min_flux_vector[i];
        }
        if (max_flux_vector[i] > max_flux) {
            max_flux = max_flux_vector[i];
        }
    }
    
    //double nominal_conad = -1.0;
    //char *keyword_to_get = espdr_add_output_index_to_keyword(inst_config->conad_kw_first_part,
    //                                                         inst_config->conad_kw_last_part,
    //                                                         1);
    //espdr_msg("CONAD KW: %s", keyword_to_get);
    //if (inst_config->inst_type == NIR) {
    //    if (cpl_propertylist_has(*keywords_RE, keyword_to_get)) {
    //        nominal_conad = cpl_propertylist_get_double(*keywords_RE,
    //                                                   keyword_to_get);
    //        my_error = cpl_error_get_code();
    //        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
    //                     "Get keyword %s from the propertylist failed: %s",
    //                     keyword_to_get,
    //                     cpl_error_get_message_default(my_error));
    //    }
    //}
    //cpl_free(keyword_to_get);
    //espdr_msg("nominal conad: %f", nominal_conad);
    //double conv_to_put = 1.0;
    /* write flux values in the FITS header for QC */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_gain_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                sprintf(comment, "Conv factor[e-|ADU] for ext %d, out %d, %d", i, j, k);
                //if (nominal_conad == -1.0) {
                //    conv_to_put = conad_vector[index];
                //} else {
                //    conv_to_put = nominal_conad;
                //}
                //espdr_msg("Conversion factor to put in the FITS header: %f", conv_to_put);
                espdr_msg("Conversion factor to put in the FITS header: %f", conad_vector[index]);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    //conv_to_put,
                                                    conad_vector[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword,
                             cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_mean_flux_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                sprintf(comment, "Mean flux for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_int(new_keyword,
                                                    (int)mean_flux_vector[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to th propertylist failed: %s",
                             new_keyword,
                             cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);

                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_min_flux_kw_part,
                                                  inst_config->prefix,
                                                      i, j, k);
                sprintf(comment, "Min flux[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_int(new_keyword,
                                                    (int)min_flux_vector[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to th propertylist failed: %s",
                             new_keyword,
                             cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_max_flux_kw_part,
                                                  inst_config->prefix,
                                                      i, j, k);
                sprintf(comment, "Max flux[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_int(new_keyword,
                                                    (int)max_flux_vector[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to th propertylist failed: %s",
                             new_keyword,
                             cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                index++;
            }
        }
    }
    
    my_error = espdr_keyword_add_double(qc_kws->qc_ledff_min_flux_kw, min_flux,
                                        "Global minimal flux[ADU]",
                                        keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_MIN_FLUX_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_double(qc_kws->qc_ledff_max_flux_kw, max_flux,
                                        "Global maximal flux[ADU]",
                                        keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_MAX_FLUX_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (min_flux < inst_config->flux_min_limit) {
        flux_QC = 0;
        espdr_msg("WARNING: Flux level too low to properly identify non-linear pixels");
    }
    my_error = espdr_keyword_add_int(qc_kws->qc_ledff_flux_check_kw, flux_QC,
                                     "Flux QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LEDFF_FLUX_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* CONAD QC */
    my_error = espdr_gain_QC(CCD_geom, qc_kws, inst_config,
                             keywords_ext, conad_vector, keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_gain_QC failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* free memory */
    cpl_free(conad_vector);
    cpl_free(min_flux_vector);
    cpl_free(max_flux_vector);
    cpl_free(mean_flux_vector);
    
    if(cpl_error_get_code() != CPL_ERROR_NONE){
        /* the following function prints where the error occurred and why */
        espdr_print_rec_status(0);
        return cpl_error_get_code();
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 * AMO added
 @brief    Compute CONAD
 @param         set                 input frameset, bias removed
 @param         nb_of_sets      number of different Texp
 @param         pixels_mask     bad&hot pixels mask
 @param         CCD_geom        CCD geometry parameters structure
 @param         LED_FF_param    LED_FF parameters structure
 @param         gain_windows    windows structure specifying where 
                                to compute the convertion factor
 @param[out]    total_conad_RE   computed conversion factor (returned)
 @return   CPL_ERROR_NONE iff OK
 
 TODO: remove commented-out code
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_compute_gain(cpl_frameset **set,
                                  int nb_of_sets,
                                  cpl_imagelist *pixels_mask,
                                  espdr_CCD_geometry *CCD_geom,
                                  espdr_inst_config *inst_config,
                                  espdr_window_coord *gain_windows,
                                  double *total_conad_RE,
                                  double *mean_flux_RE) {
    
    int i, k, index;
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_image *mean_image, *diff_image;
    cpl_image *extracted_pixel_mask;
    cpl_image *extracted_led_ff1, *extracted_led_ff2;
    cpl_vector *mean_vector = NULL;
    cpl_vector *diff_vector = NULL;
    int total_output_nb = CCD_geom->total_output_nb;
    double *all_mean = NULL;
    double *all_diff = NULL;
    int *pixel_mask_vector = NULL;
    int good_pixels_nb = 0;
    int bad_pixels_nb = 0;
    int reject_zeroMAD = 1; // not used anymore
    /* Value for the MAD in case it is == 0.0 */
    double forced_MAD = 0.1;
    double CoD, sigma, sig_mean, stddev, reject_low, reject_high;
    int cosmics;
    double Flux;
    double conad[nb_of_sets-1];
    int llx, lly, urx, ury;
    double gain_window_mean_flux = 0.0;
    
    cpl_frame* frame = NULL;
    const char* fname = NULL;
    int images_nb = 0;
    int frameset_size = 0;
    /* AMO: added here some info useful to test on provided HARPS example:
     * The number total_output_nb=2*/
    /* The number of sets is instead nb_of_sets=3 */
    /* The size of the frameset is 20 */
    /* The images nb is 10 */
    frameset_size = cpl_frameset_get_size(set[0]);
    images_nb = frameset_size / total_output_nb;
    /* AMO: the following is useful to check results
     espdr_msg("total_output=%d", total_output_nb);
     espdr_msg("number of sets=%d", nb_of_sets);
     espdr_msg("framelist size=%d", frameset_size);
     espdr_msg("images nb=%d", images_nb);
     exit(0);
     *
     */
    for (i = 0; i < total_output_nb; i++) {
        
        llx = gain_windows[i].llx;
        lly = gain_windows[i].lly;
        urx = gain_windows[i].urx;
        ury = gain_windows[i].ury;
        
        extracted_pixel_mask = cpl_image_extract(cpl_imagelist_get(pixels_mask, i),
                                                 llx, lly, urx, ury);
        pixel_mask_vector = cpl_image_get_data_int(extracted_pixel_mask);
        
        gain_window_mean_flux = 0.0;
	
        /* Allocate only the needed memory, to have the correct
           vectors lengths */
        good_pixels_nb = (urx-llx+1)*(ury-lly+1);
        bad_pixels_nb = 0;
        if (pixel_mask_vector != NULL ){
            for (k = 0; k < (urx-llx+1)*(ury-lly+1); k++) {
                if (pixel_mask_vector[k] != 0) {
                    bad_pixels_nb++;
                }
            }
        }
        good_pixels_nb -= bad_pixels_nb;
            
        espdr_msg("output: %d, set: 0, good pxls: %d", i, good_pixels_nb);
            
        if (good_pixels_nb < 5) {
            conad[0] = -1.0;
            gain_window_mean_flux = -1.0;
        } else {
            
            double *mean = (double *)cpl_calloc(good_pixels_nb, sizeof(double));
            double *diff = (double *)cpl_calloc(good_pixels_nb, sizeof(double));
            frame = cpl_frameset_get_position(set[0], i*images_nb);
            // 4th frame
            //frame = cpl_frameset_get_position(set[j], i*images_nb+3);
            fname = cpl_frame_get_filename(frame);
            //espdr_msg("load fname1=%s",fname);
            
            extracted_led_ff1 = cpl_image_load_window(fname, CPL_TYPE_DOUBLE,
                                                      0, 0, llx, lly, urx, ury);
            /* how to deal with proper range llx, lly, urx, ury? (see ref
             * implementation)
             * */
            /* Note that in my frame numbering I have frames of same exposure
             * time next to the other, instead Danuta has next to each other
             * different extensions, which repeats in her imagelist every
             * total_output_nb, thus
             * i-> i*images_nb
             * i+total_output_nb->i*images_nb+1
             */
            frame = cpl_frameset_get_position(set[0], i*images_nb+1);
            // 5th frame
            //frame = cpl_frameset_get_position(set[j], i*images_nb+4);
            fname = cpl_frame_get_filename(frame);
            //espdr_msg("load fname2=%s",fname);
            extracted_led_ff2 = cpl_image_load_window(fname, CPL_TYPE_DOUBLE,
                                                      0, 0, llx, lly, urx, ury);
            
            //sprintf(name1,"led2_ff1_%d_%d.fits",i,j);
            //sprintf(name2,"led2_ff2_%d_%d.fits",i,j);
            //espdr_msg("check fname1=%s",name1);
            //espdr_msg("check fname2=%s",name2);
            /* AMO: the following is to debug
             cpl_image_save(extracted_led_ff1,name1,CPL_TYPE_FLOAT,NULL,CPL_IO_DEFAULT);
             cpl_image_save(extracted_led_ff2,name2,CPL_TYPE_FLOAT,NULL,CPL_IO_DEFAULT);
             */
            mean_image = cpl_image_add_create(extracted_led_ff1,
                                              extracted_led_ff2);
            my_error = cpl_image_divide_scalar(mean_image, 2.0);
            
            diff_image = cpl_image_subtract_create(extracted_led_ff2,
                                                   extracted_led_ff1);
            
            /* Take off the bad & hot pixels from the images */
            all_mean = cpl_image_get_data_double(mean_image);
            all_diff = cpl_image_get_data_double(diff_image);
            index = 0;
            for (k = 0; k < (urx-llx+1)*(ury-lly+1); k++) {
                if (pixel_mask_vector[k] == 0) {
                    mean[index] = all_mean[k];
                    diff[index] = all_diff[k];
                    index++;
                }
            }
            
            /* AMO to debug
             for (k = 0; k < good_pixels_nb; k++) {
             espdr_msg("mean[%d] = %lf", k, mean[k]);
             }
             for (k = 0; k < good_pixels_nb; k++) {
             espdr_msg("diff[%d] = %lf", k, diff[k]);
             }
             */
            
            mean_vector = cpl_vector_wrap(good_pixels_nb, mean);
            my_error = espdr_sig_clip(mean_vector,
                                      inst_config->gain_ksigma,
                                      inst_config->gain_ksigma,
                                      inst_config->gain_sigma_clipping_method,
                                      inst_config->gain_max_iter,
                                      reject_zeroMAD, forced_MAD,
                                      &CoD, &sigma, &cosmics,
                                      &sig_mean, &stddev,
                                      &reject_low, &reject_high);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_sig_clip failed: %s",
                         cpl_error_get_message_default(my_error));
            Flux = sig_mean;
            //espdr_msg("output: %d mean vector mean: CoD = %lf, sigma = %lf, cosmics = %d",
            //                i, CoD, sigma, cosmics);
            //espdr_msg("output: %d mean vector mean: sig_mean = %lf, stddev = %lf",
            //                i, sig_mean, stddev);
            
            gain_window_mean_flux = CoD;
            diff_vector = cpl_vector_wrap(good_pixels_nb, diff);
            my_error = espdr_sig_clip(diff_vector,
                                      inst_config->gain_ksigma,
                                      inst_config->gain_ksigma,
                                      inst_config->gain_sigma_clipping_method,
                                      inst_config->gain_max_iter,
                                      reject_zeroMAD, forced_MAD,
                                      &CoD, &sigma, &cosmics,
                                      &sig_mean, &stddev,
                                      &reject_low, &reject_high);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_sig_clip failed: %s",
                         cpl_error_get_message_default(my_error));
            
            //espdr_msg("output: %d diff vector diff: CoD = %lf, sigma = %lf, cosmics = %d",
            //                i, CoD, sigma, cosmics);
            //espdr_msg("output: %d diff vector diff: sig_mean = %lf, stddev = %lf",
            //                i, sig_mean, stddev);
            
            conad[0] = (2*Flux)/(stddev*stddev);
            
            //espdr_msg("output: %d CONAD[0] = %lf", i, conad[0]);
            //espdr_msg("output: %d Flux of mean = %lf", i, Flux);
            //espdr_msg("output: %d Sigma of diff = %lf", i, sigma);
            
            cpl_image_delete(extracted_led_ff1);
            cpl_image_delete(extracted_led_ff2);
            cpl_image_delete(mean_image);
            cpl_image_delete(diff_image);
            cpl_free(mean);
            cpl_free(diff);
            cpl_vector_unwrap(mean_vector);
            cpl_vector_unwrap(diff_vector);
        } // if (good_pixels_nb < 5)
        
        total_conad_RE[i] = conad[0];
        mean_flux_RE[i] = gain_window_mean_flux;
        
        if (isnan(total_conad_RE[i])) {
            total_conad_RE[i] = DBL_REPLACING_NAN;
        }
        if (isinf(total_conad_RE[i])) {
            total_conad_RE[i] = -1.0;
        }
        if (isnan(mean_flux_RE[i])) {
            mean_flux_RE[i] = DBL_REPLACING_NAN;
        }
        if (isinf(mean_flux_RE[i])) {
            mean_flux_RE[i] = -1.0;
        }
        
        cpl_image_delete(extracted_pixel_mask);
    }
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * AMO added
 @brief     Evaluate flux
 @param         set                 input frameset, bias removed
 @param         nb_of_sets          number of different Texps
 @param         pixels_mask         bad&hot pixels mask
 @param         CCD_geom            CCD geometry parameters structure
 @param[out]    min_flux_RE         minimal flux of the shortest exposure image
 @param[out]    max_flux_RE         maximal flux of the longest exposure image
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_evaluate_flux(cpl_frameset **set,
                                   int nb_of_sets,
                                   cpl_imagelist *pixels_mask,
                                   espdr_CCD_geometry *CCD_geom,
                                   double *min_flux_RE,
                                   double *max_flux_RE) {
    
    int i, j, k, l, index, index_bis;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int total_outputs = CCD_geom->total_output_nb;
    int real_llx, real_lly, real_urx, real_ury, used_real_nx, used_real_ny;
    cpl_image *min_flux_image = NULL;
    cpl_image *max_flux_image = NULL;
    double *all_min = NULL;
    double *all_max = NULL;
    cpl_vector *min_flux_vector = NULL;
    cpl_vector *max_flux_vector = NULL;
    int *pixel_mask_vector = NULL;
    int good_pixels_nb;
    cpl_frame* frame = NULL;
    const char* fname = NULL;
    
    int images_nb = 0;
    int frameset_size = 0;
    
    /* AMO: useful number valid for provided example on HARPS data:
     * The number total_output_nb=2*/
    /* The number of sets is instead nb_of_sets=3 */
    /* The size of the frameset is 20 */
    /* The images nb is 10 */
    
    frameset_size = cpl_frameset_get_size(set[0]);
    images_nb = frameset_size / total_outputs;
    /* AMO to check results
     espdr_msg("total_output=%d", total_outputs);
     espdr_msg("number of sets=%d", nb_of_sets);
     espdr_msg("framelist size=%d", frameset_size);
     espdr_msg("images nb=%d", images_nb);
     */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                
                real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1
                                - CCD_geom->exts[i].outputs[j][k].ppscan_ny
                                - CCD_geom->exts[i].outputs[j][k].poscan_ny;
                
                pixel_mask_vector = cpl_image_get_data_int(cpl_imagelist_get
                                                           (pixels_mask, index));
                
                frame = cpl_frameset_get_position(set[0], index*images_nb);
                fname = cpl_frame_get_filename(frame);
                min_flux_image = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 0);
                
                frame = cpl_frameset_get_position(set[1],
                                                  index*images_nb);
                fname = cpl_frame_get_filename(frame);
                max_flux_image = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 0);
                
                /*
                 min_flux_image = cpl_image_duplicate(
                 cpl_imagelist_get(master_led_ff_list[0], index));
                 
                 max_flux_image = cpl_image_duplicate(
                 cpl_imagelist_get(master_led_ff_list[nb_of_sets-1],
                 index));
                 */
                
                /* Take off the bad & hot pixels from the images */
                all_max = cpl_image_get_data_double(max_flux_image);
                all_min = cpl_image_get_data_double(min_flux_image);
                
                /* Allocate only the needed memory, to have the correct
                 vectors lengths */
                good_pixels_nb = 0;
                for (l = 0; l < used_real_nx * used_real_ny; l++) {
                    if (pixel_mask_vector[l] == 0) {
                        good_pixels_nb++;
                    }
                }

                if (good_pixels_nb == 0) {
                    min_flux_RE[index] = -1.0;
                    max_flux_RE[index] = -1.0;
                } else {
                    double *min_flux = (double *)cpl_calloc(good_pixels_nb,
                                                            sizeof(double));
                    double *max_flux = (double *)cpl_calloc(good_pixels_nb,
                                                            sizeof(double));
                    index_bis = 0;
                    for (l = 0; l < used_real_nx * used_real_ny; l++) {
                        if (pixel_mask_vector[l] == 0) {
                            min_flux[index_bis] = all_min[l];
                            max_flux[index_bis] = all_max[l];
                            index_bis++;
                        }
                    }
                    min_flux_vector = cpl_vector_wrap(index_bis, min_flux);
                    max_flux_vector = cpl_vector_wrap(index_bis, max_flux);
                    //espdr_msg("measure flux at index %d",index);
                    min_flux_RE[index] = cpl_vector_get_median(min_flux_vector);
                    max_flux_RE[index] = cpl_vector_get_max(max_flux_vector);
                    
                    cpl_vector_unwrap(min_flux_vector);
                    cpl_vector_unwrap(max_flux_vector);
                    cpl_free(min_flux);
                    cpl_free(max_flux);
                }
                cpl_image_delete(min_flux_image);
                cpl_image_delete(max_flux_image);
                
                index++;
            }
        }
    }
    
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Flux evaluation failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (CPL_ERROR_NONE);
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Check the gain QC parameters
 @param         CCD_geom        CCD geometry parameters structure
 @param         qc_kws          KWs names
 @param         inst_config     instrument config
 @param         keywords_ext    extension FITS header
 @param         computed_conad  computed conversion factor
 @param[out]    keywords_RE     primary FITS header (returned)
 @return   CPL_ERROR_NONE iff OK
 
 TODO: there is no need to pass keywords_RE as double pointer
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_gain_QC(espdr_CCD_geometry *CCD_geom,
                             espdr_qc_keywords *qc_kws,
                             espdr_inst_config *inst_config,
                             cpl_propertylist **keywords_ext,
                             double *computed_conad,
                             cpl_propertylist **keywords_RE) {
    
    int i, j, k, index;
    cpl_error_code my_error = CPL_ERROR_NONE;
    double nominated_conad = 0.0;
    char *keyword_to_get = NULL;
    int total_output_nb = CCD_geom->total_output_nb;
    int conad_QC[total_output_nb];
    int global_conad_QC = 1;
    double conad_diff = 0.0;
    
    espdr_msg_debug("Starting espdr_gain_QC()");
    
    for (i = 0; i < total_output_nb; i++) {
        conad_QC[i] = 1;
    }
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                keyword_to_get =
                espdr_add_output_index_to_keyword(inst_config->conad_kw_first_part,
                                                  inst_config->conad_kw_last_part,
                                                  j+1);
                
                if (cpl_propertylist_has(keywords_ext[i], keyword_to_get)) {
                    nominated_conad = cpl_propertylist_get_double(keywords_ext[i],
                                                                 keyword_to_get);
                } else {
                    if (cpl_propertylist_has(*keywords_RE, keyword_to_get)) {
                        nominated_conad = cpl_propertylist_get_double(*keywords_RE,
                                                                     keyword_to_get);
                    } else{
                        espdr_msg_warning(ANSI_COLOR_RED"The conad KW: %s not found in the header, nominated conad set to 1.0"ANSI_COLOR_RESET,
                                          keyword_to_get);
                        nominated_conad = 1.0;
                    }
                }
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Get keyword %s from the propertylist failed: %s",
                             keyword_to_get, cpl_error_get_message_default(my_error));
                cpl_free(keyword_to_get);
            
                espdr_msg("Conversion factor read[%d]: %lf", j, nominated_conad);
                espdr_msg("Conversion factor computed[%d]: %lf, gain[%d] computed: %f",
                          index, computed_conad[index], index, 1.0/computed_conad[index]);
            
                conad_diff = nominated_conad - computed_conad[index];
                if (conad_diff < 0.0) conad_diff = 0.0 - conad_diff;
                espdr_msg_debug("conad_diff = %lf", conad_diff);
                if (conad_diff > inst_config->gain_tolerance) {
                    conad_QC[index] = 0;
                }
                index++;
            }
        }
    }
    
    for (i = 0; i < total_output_nb; i++) {
        if (conad_QC[i] == 0) {
            global_conad_QC = 0;
        }
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_ledff_gain_check_kw,
                                     global_conad_QC,
                                     "Conversion factor QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Add keyword QC_LEDFF_GAIN_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg_debug("espdr_gain_QC finished");
    
    return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Check the led_ff QC
 @param         CCD_geom        CCD geometry parameters structure
 @param         qc_kws          KWs names
 @param[out]    keywords_RE     primary FITS header
 @return   CPL_ERROR_NONE iff OK
 
 TODO: there is no need to pass keywords_RE as double pointer
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_led_ff_QC(espdr_CCD_geometry *CCD_geom,
                               espdr_qc_keywords *qc_kws,
                               espdr_inst_config *inst_config,
                               double *max_flux,
                               cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int saturation_QC = 1;
    int bad_pixel_QC = 1;
    int flux_level_QC = 1;
	int conad_QC = 1;
    int global_QC = 1;
    int i, j, k, index;
    char *new_keyword = NULL;

    /* max flux KWs */
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_max_flux_kw,
                                                      inst_config->prefix,
                                                      i, j, k);
                
                my_error = espdr_keyword_add_double(new_keyword,
                                                    max_flux[index],
                                                    "Max flux, raw image [ADU]",
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_MAX_FLUX_KW to the propertylist failed: %s",
                             cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                if (max_flux[index] >= inst_config->satur_limit_adu) {
                    saturation_QC = 0;
                    global_QC = 0;
                }
                
                index++;
            }
        }
    }
    
    /* QC saturation CHECK KWs */
    
    my_error = espdr_keyword_add_int(qc_kws->qc_saturation_check_kw,
                                     saturation_QC,
                                     "Saturation [ADU] QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_SATUR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Check KWs */
    
    bad_pixel_QC = cpl_propertylist_get_int(*keywords_RE,
                                            qc_kws->qc_ledff_badpix_check_kw);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Couldn't get the QC LEDFF BADPIX CHECK KW.");
    
    flux_level_QC = cpl_propertylist_get_int(*keywords_RE,
                                             qc_kws->qc_ledff_flux_check_kw);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Couldn't get the QC LEDFF FLUX CHECK KW.");
    
    conad_QC = cpl_propertylist_get_int(*keywords_RE,
                                       qc_kws->qc_ledff_gain_check_kw);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Couldn't get the QC LEDFF GAIN CHECK KW.");
    
    if (bad_pixel_QC == 0) global_QC = 0;
    if (flux_level_QC == 0) global_QC = 0;
    if (conad_QC == 0) global_QC = 0;
    
    my_error = espdr_keyword_add_int(qc_kws->qc_ledff_check_kw, global_QC,
                                     "Global LED_FF QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LEDFF_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    return(CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Cleanup the frameset structure
 @param     set     frameset structure to be cleaned
 @param     nb      number of framesets in the structure
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_led_ff_cleanup(cpl_frameset** set, const int nb) {
    
    const char* fname = NULL;
    cpl_frame* frame = NULL;
    int size = 0;
    char cmd[COMMENT_LENGTH];
    for (int it = 0; it < nb; it++) {
        size = cpl_frameset_get_size(set[it]);
        for (int i = 0; i < size; i++) {
            frame = cpl_frameset_get_position(set[it], i);
            fname = cpl_frame_get_filename(frame);
            sprintf(cmd, "rm  %s", fname);
            system(cmd);
        }
    }
    
    return (cpl_error_get_code());
}


