/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2014-12-16 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


//#include <espdr_wave_cal.h>
#include <espdr_wave_FP_cal.h>

#define KSIG_EXTRACTION_MIN -2.0
#define KSIG_EXTRACTION_MAX 100.0
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_wave_FP");

    espdr_add_qc_key_stat_pri(keywords, ext_nb, "MAX FLUX", ksuff, CPL_FALSE, CPL_FALSE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE);

    espdr_add_qc_key_wave_FP_ord_ref(keywords, ext_nb, ord_ref, "FP LINES NB", ksuff,
                                     CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE, CPL_FALSE);

    espdr_add_qc_key_stat_ord_pri(keywords, ext_nb, "FP LINES NB", "DUMMY",
                                  CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE,
                                  CPL_FALSE);

    cpl_free(ord_ref);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 @brief QC on FP lines
 @param         lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         inst_config             instrument config
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_FP_lines_qc(espdr_line_param *lines_table,
                                 int lines_nb,
                                 espdr_inst_config *inst_config) {

    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "The input table is NULL");
    int i;
    int test_order = -1;


    for (i = 0; i < lines_nb; i++) {

        if (lines_table[i].order == test_order) {
            espdr_msg("order %d, pxl %lf, ll: %lf, before check qc = %d",
                      lines_table[i].order, lines_table[i].peak_pxl,
                      lines_table[i].wavelength, lines_table[i].qc_flag);
        }

        // Check if the fit produced any nans or infs
        if (isnan(lines_table[i].peak_pxl) || isinf(lines_table[i].peak_pxl) ||
            isnan(lines_table[i].peak_pxl_err) || isinf(lines_table[i].peak_pxl_err) ||
            isnan(lines_table[i].fwhm) || isinf(lines_table[i].fwhm) ||
            isnan(lines_table[i].fwhm_err) || isinf(lines_table[i].fwhm_err) ||
            isnan(lines_table[i].peak_flux) || isinf(lines_table[i].peak_flux) ||
            isnan(lines_table[i].peak_flux_err) || isinf(lines_table[i].peak_flux_err)) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, nans or infs",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }

        //QC_flag_orig = lines_table[i].qc_flag;
        if (lines_table[i].peak_pxl_err > inst_config->line_max_sig_x0) {
            //if (lines_table[i].peak_pxl_err > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, peak pxl error too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_pxl_err);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }
        if ((lines_table[i].fwhm > inst_config->line_max_fwhm) ||
            (lines_table[i].fwhm < 0.0)) {
            //if ((lines_table[i].fwhm > 8) || (lines_table[i].fwhm < 3)) {
            //if (lines_table[i].fwhm > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].fwhm);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }
        if (lines_table[i].peak_flux < inst_config->line_min_flux_el) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, flux too small (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_flux);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }
#if 0
        // The following test is added,
        // because the fit to compute the resolution was failing
        if (lines_table[i].fwhm_err == 0.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm error == 0.0",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }
#endif
    }

    for (i = 0; i < lines_nb; i++) {
        // Lines was not fitted because of close to the border
        // or too small flux difference between max_flux and min_flux
        // But the rest of the group is OK
        if (lines_table[i].qc_flag == 2) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, qc = 2",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
            lines_table[i].wavelength = -1.0;
            lines_table[i].wavelength_err = -1.0;
        }
    }

    return(cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_wave_FP(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

    cpl_frameset *wave_frames_to_reduce = NULL;
    cpl_frame *wave_frame = NULL;
    cpl_frameset *used_frames = NULL;

    int i = 0, j = 0, k = 0, fibre = 0;

    char line_table_filename[FILENAME_LENGTH];
    char new_keyword[KEYWORD_LENGTH];

    espdr_line_param **lines_table = NULL;
    cpl_table **lines_table_FITS = NULL;
    int **lines_nb_per_order = NULL;

    char pro_catg_tag[TAG_LENGTH];

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 12;
    const char* rec_tags[12] = {ESPDR_WAVE_FP_FP_RAW,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_PIXEL_GEOM_A, ESPDR_PRO_CATG_PIXEL_GEOM_B,
        ESPDR_PRO_CATG_PIXEL_SIZE_A, ESPDR_PRO_CATG_PIXEL_SIZE_B,
        ESPDR_ORDERS_MASK
    };
    int is_required[12] = {1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0};

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting wave FP");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong inst_config input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 1) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw wave FP-FP image, tagged as FP_FP\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B");

    espdr_CCD_geometry *CCD_geom       = NULL;
    espdr_inst_config *inst_config     = NULL;
    espdr_qc_keywords *qc_kws          = NULL;
    espdr_OVSC_param *OVSC_param       = NULL;
    espdr_WAVE_FP_param *WAVE_FP_param = NULL;
    cpl_frame* CCD_geom_frame          = NULL;
    cpl_frame* inst_config_frame       = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL, cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    WAVE_FP_param     = espdr_WAVE_FP_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_FP_FP_RAW);
    espdr_ensure(wave_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "wave (FP_FP) frame not found!");
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    wave_frames_to_reduce = cpl_frameset_new();
    cpl_frameset_insert(wave_frames_to_reduce, cpl_frame_duplicate(wave_frame));

    const char *FP_tag = cpl_frame_get_tag(wave_frame);
    espdr_src_type fibre_source[inst_config->fibres_nb];
    char **fibre_WAVE = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_WAVE[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    my_error = espdr_get_fibre_source(FP_tag,
                                      fibre_source,
                                      fibre_WAVE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_fibre_source failed: %s",
                 cpl_error_get_message_default(my_error));

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    cpl_image **pixel_geom_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **pixel_size_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_propertylist *));
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    cpl_table ***orders_coeffs = NULL;
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    my_error = espdr_process_wave_till_extraction(frameset, parameters, recipe_id,
                                                  wave_frames_to_reduce,
                                                  wave_frame,
                                                  used_frames,
                                                  inst_config,
                                                  CCD_geom,
                                                  qc_kws,
                                                  orders_nb_per_fibre,
                                                  fibre_source,
                                                  fibre_WAVE,
                                                  keywords_fibre,
                                                  CCD_corrected_image,
                                                  &orders_coeffs,
                                                  RON,
                                                  flat_corr_spectrum,
                                                  flat_corr_error,
                                                  flat_corr_qual,
                                                  flat_blaze_corr_spectrum,
                                                  flat_blaze_corr_error,
                                                  flat_blaze_corr_qual,
                                                  pixel_geom_image,
                                                  pixel_size_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    /* WAVE LINES TABLE reading */
    lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    for (i = 0; i < inst_config->fibres_nb; i++) {
        lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
        my_error = espdr_lines_table_init(lines_table[i],
                                          MAX_NB_SPECTRAL_LINES);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_lines_table_init of lines_table for fibre %c failed: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
    }

    int lines_nb[inst_config->fibres_nb];
    lines_nb_per_order = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(int *));
    for (i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Searching lines for fibre: %c", fibre_name[i]);
        my_error = espdr_search_lines(flat_blaze_corr_spectrum[i],
                                      flat_blaze_corr_error[i],
                                      flat_blaze_corr_qual[i],
                                      inst_config,
                                      FP,
                                      lines_table[i],
                                      &lines_nb[i]);
        espdr_msg("Lines nb after search: %d", lines_nb[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_FP
        my_error = espdr_save_debug_table_RDB(lines_table[i], lines_nb[i],
                                              "FP_SEARCHED_LINE_TABLE_no_fit",
                                              i, inst_config);
#endif

        espdr_msg("Fitting lines");
        espdr_msg("Orders nb for fibre %c: %d",
                  fibre_name[i], orders_nb_per_fibre[i]);
        lines_nb_per_order[i] = (int *)cpl_calloc(orders_nb_per_fibre[i],
                                                  sizeof(int));
        for (j = 0; j < orders_nb_per_fibre[i]; j++) {
            lines_nb_per_order[i][j] = 0;
        }

        espdr_ensure((flat_blaze_corr_spectrum[i] == NULL) ||
                     (flat_blaze_corr_error[i] == NULL) ||
                     (flat_blaze_corr_qual[i] == NULL),
                     CPL_ERROR_NULL_INPUT,
                     "FLAT & BLAZE corrected spectrum for fibre %c is NULL", fibre_name[i]);

        my_error = espdr_fit_lines(lines_table[i],
                                   lines_nb[i],
                                   flat_blaze_corr_spectrum[i],
                                   flat_blaze_corr_error[i],
                                   flat_blaze_corr_qual[i],
                                   pixel_geom_image[i],
                                   inst_config->fp_peak_flux_threshold,
                                   inst_config->fp_fit_window_size,
                                   inst_config->fp_fit_tolerance_window_size,
                                   inst_config->fp_fit_ngauss_flag,
                                   lines_nb_per_order[i]);

        if (inst_config->fp_fit_ngauss_flag) { // No cleaning if the fit was not performed
            my_error = espdr_FP_lines_qc(lines_table[i],
                                         lines_nb[i],
                                         inst_config);
        }

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_lines failed: %s",
                     cpl_error_get_message_default(my_error));

        espdr_msg("espdr_fit_lines done");

#if SAVE_DEBUG_PRODUCT_WAVE_FP
        my_error = espdr_save_debug_table_RDB(lines_table[i], lines_nb[i],
                                              "FP_SEARCHED_LINE_TABLE_after_fit",
                                              i, inst_config);
#endif

        espdr_msg("Checking FP peaks for fibre %c", fibre_name[i]);
        my_error = espdr_check_delta_FP_peaks(lines_table[i], lines_nb[i],
                                              inst_config->delta_fp_tolerance);
        espdr_ensure(my_error == CPL_ERROR_INCOMPATIBLE_INPUT, my_error,
                     "Missing peaks in the FP image, aborting");
    }

    lines_table_FITS = (cpl_table **)cpl_malloc
                                (inst_config->fibres_nb * sizeof(cpl_table*));
    for (i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Nb of peaks for fibre %c: %d", fibre_name[i], lines_nb[i]);
        lines_table_FITS[i] = cpl_table_new(lines_nb[i]);
        my_error = espdr_FITS_lines_table_create(lines_table_FITS[i]);
        my_error = espdr_FITS_lines_table_fill(lines_table_FITS[i],
                                               lines_table[i], lines_nb[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "FITS table creation for fibre %c failed: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
        espdr_msg("Line table for fibre %c has %lld rows",
                  fibre_name[i],
                  cpl_table_get_nrow(lines_table_FITS[i]));
    }

    /* Add QC KWs to the header */
    for (i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_wave_FP_QC(lines_nb_per_order[i],
                                    orders_nb_per_fibre[i],
                                    inst_config, qc_kws,
                                    keywords_fibre[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_wave_QC failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error),
                     fibre_name[i]);
    }

    // Set the global QC to 0, if on the other fiber it is 0
    int global_QC_A = cpl_propertylist_get_int(keywords_fibre[0], qc_kws->qc_wave_check_kw);
    int global_QC_B = cpl_propertylist_get_int(keywords_fibre[1], qc_kws->qc_wave_check_kw);
    char comment[COMMENT_LENGTH];
    if ((global_QC_A == 0) || (global_QC_B == 0)) {
        if (global_QC_A == 1) {
            sprintf(comment, "Overall QC set to 0, because QC on B = 0");
            my_error = espdr_keyword_add_int(qc_kws->qc_wave_check_kw,
                                             0,
                                             comment,
                                             &keywords_fibre[0]);
        }
        if (global_QC_B == 1) {
            sprintf(comment, "Overall QC set to 0, because QC on A = 0");
            my_error = espdr_keyword_add_int(qc_kws->qc_wave_check_kw,
                                             0,
                                             comment,
                                             &keywords_fibre[1]);
        }
    }

    espdr_msg("Saving products");
    for (i = 0; i < inst_config->fibres_nb; i++) {

        my_error = espdr_save_wave_S2D_products(frameset, parameters, recipe_id,
                                                used_frames, "FP_FP", i,
                                                keywords_fibre[i],
                                                inst_config,
                                                flat_corr_spectrum[i],
                                                flat_corr_error[i],
                                                flat_corr_qual[i],
                                                flat_blaze_corr_spectrum[i],
                                                flat_blaze_corr_error[i],
                                                flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_wave_S2D_products failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error), fibre_name[i]);

        /* Save the PRO.CATG */
        sprintf(new_keyword, "%s_%s_%c",
                ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE, FP_tag, fibre_name[i]);
        espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
        my_error = cpl_propertylist_update_string(keywords_fibre[i],
                                                  PRO_CATG_KW,
                                                  new_keyword);

        /* Save the spectrum lines fits table */
        sprintf(line_table_filename, "%s_%c.fits",
                inst_config->fp_searched_line_table_filename_no_fits,
                fibre_name[i]);
        espdr_msg("FP SEARCHED LINE TABLE filename: %s", line_table_filename);

        if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
            (strcmp(inst_config->instrument, "CORALIE") != 0)) {
            my_error = espdr_add_extra_qc(&(keywords_fibre)[i], CCD_geom->ext_nb);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_add_extra_qc failed: %s ",
                         cpl_error_get_message_default(my_error));
       }

        my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                      NULL, lines_table_FITS[i], NULL,
									  recipe_id, keywords_fibre[i], NULL,
                                      PACKAGE "/" PACKAGE_VERSION,
                                      line_table_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     line_table_filename);

#if SAVE_DEBUG_PRODUCT_WAVE_FP
        my_error = espdr_save_debug_table_RDB(lines_table[i], lines_nb[i],
                                              "FP_SEARCHED_LINE_TABLE_FP_FP",
                                              i, inst_config);
#endif

    }

    espdr_msg("Cleaning the memory");
    espdr_msg("Cleaning keywords lists");
    for (i = 0; i < inst_config->fibres_nb; i++) {
        cpl_propertylist_delete(keywords_fibre[i]);
    }
    cpl_free(keywords_fibre);

    cpl_frameset_delete(used_frames);

    cpl_imagelist_delete(CCD_corrected_image);

    espdr_msg("Cleaning calibration products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_corr_spectrum[i]);
        cpl_image_delete(flat_corr_error[i]);
        cpl_image_delete(flat_corr_qual[i]);
    }

    cpl_free(flat_corr_spectrum);
    cpl_free(flat_corr_error);
    cpl_free(flat_corr_qual);

    for (i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(pixel_geom_image[i]);
        cpl_image_delete(flat_blaze_corr_spectrum[i]);
        cpl_image_delete(flat_blaze_corr_error[i]);
        cpl_image_delete(flat_blaze_corr_qual[i]);
    }

    cpl_free(pixel_geom_image);
    cpl_free(flat_blaze_corr_spectrum);
    cpl_free(flat_blaze_corr_error);
    cpl_free(flat_blaze_corr_qual);

    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);
    for (i = 0; i < inst_config->fibres_nb; i++) {
    	cpl_free(lines_table[i]);
    }
    cpl_free(lines_table);

    for (i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(lines_nb_per_order[i]);
    }

    cpl_free(lines_nb_per_order);

    cpl_free(orders_nb_per_fibre);

    for (i = 0; i < inst_config->fibres_nb; i++) {
        cpl_table_delete(lines_table_FITS[i]);
    }
    cpl_free(lines_table_FITS);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_WAVE_FP_delete(WAVE_FP_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);
    cpl_frameset_delete(wave_frames_to_reduce);
    cpl_free(qc_kws);

    espdr_msg("End of the recipe espdr_wave_FP");

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_FP_create(const char* recipe_id,
                                               cpl_parameterlist *list,
                                               espdr_WAVE_FP_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	double ksig_extraction_min = KSIG_EXTRACTION_MIN;
	double ksig_extraction_max = KSIG_EXTRACTION_MAX;
    
	/* Fill the parameter list */
	cpl_error_code error_got = CPL_ERROR_NONE;
	char comment[COMMENT_LENGTH];
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "fp_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");

    sprintf(comment, "ksigma for extraction, must be between: %.2lf and %.2lf",
            ksig_extraction_min, ksig_extraction_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "fp_extraction_ksigma",
                                                  p->ksig_extraction,
                                                  KSIG_EXTRACTION_MIN,
                                                  KSIG_EXTRACTION_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksig_extraction to the list");
    return (CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_FP_delete(espdr_WAVE_FP_param* p) {
	
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the WAVE recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_FP_get(const char* recipe_id,
                                            cpl_parameterlist* param_list,
                                            espdr_WAVE_FP_param *WAVE_FP_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    WAVE_FP_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                   param_list,
                                                                   "fp_extraction_method");
    
    WAVE_FP_param->ksig_extraction = espdr_parameters_get_double(recipe_id,
                                                                 param_list,
                                                                 "fp_extraction_ksigma");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the WAVE parameters
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_FP_print(espdr_WAVE_FP_param *WAVE_FP_param) {
	
	espdr_msg("\tWAVE FP parameters:");
    espdr_msg("\t\tWAVE_FP extraction method = %s",
              WAVE_FP_param->extraction_method);
    espdr_msg("\t\tWAVE_FP extraction ksigma = %.2f",
              WAVE_FP_param->ksig_extraction);
    
	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init WAVE_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_WAVE_FP_param *espdr_WAVE_FP_param_init(const char *recipe_id,
                                              cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_WAVE_FP_param *WAVE_FP_param =
                    (espdr_WAVE_FP_param *)cpl_malloc(sizeof(espdr_WAVE_FP_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_WAVE_FP_get(recipe_id, parameters, WAVE_FP_param);
    my_error = espdr_parameters_WAVE_FP_print(WAVE_FP_param);
    //if(cpl_error_get_code() != CPL_ERROR_NONE) {
    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return WAVE_FP_param;
    }
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief    Read the wave recipe parameters
 @param         parameters   recipe input parameters
 @param         frameset     recipe input frames
 @param         recipe_id    recipe name
 @param[out]    CCD_geom     CCD geometry structure
 @param[out]    inst_config  instrument config
 @param[out]    OVSC_param   overscan parameter structure
 @param[out]    WAVE_param   WAVE parameter structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_FP_read_parameters(cpl_frameset *frameset,
                                             espdr_CCD_geometry *CCD_geom,
                                             espdr_inst_config *inst_config) {
	
	cpl_frameset *CCD_geom_frames = cpl_frameset_new();
	cpl_frame *CCD_geom_table_frame = NULL;
	const char *CCD_geom_filename=NULL;
    cpl_frameset *inst_config_frames = cpl_frameset_new();
    cpl_frame *inst_config_table_frame = NULL;
    const char *inst_config_filename = NULL;
	cpl_error_code my_error;
	
	/* Extract and read the CCD geometry FITS table */
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_CCD_GEOM,
                                          CCD_geom_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, 
			   "CCD_GEOM parameters frame extraction failed");
	
	espdr_ensure(CCD_geom_frames == NULL, CPL_ERROR_NULL_INPUT, 
			   "No CCD geometry parameters table %s, exiting", 
			   cpl_error_get_message());
	
    /* Extract and read the instrument configuration FITS table */
    my_error = espdr_frame_extract_by_tag(frameset, ESPDR_INST_CONFIG,
                                          inst_config_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Instrument parameters frame extraction failed");
    
    espdr_ensure(inst_config_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "No instrument parameters table %s, exiting",
                 cpl_error_get_message());
    
    /* Read the CCD geometry configuration file provided in SOF */
    CCD_geom_table_frame = cpl_frameset_get_position(CCD_geom_frames, 0);
    CCD_geom_filename = cpl_frame_get_filename(CCD_geom_table_frame);
    my_error = espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting CCD geometry: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Read the instrument configuration file provided in SOF */
    inst_config_table_frame = cpl_frameset_get_position(inst_config_frames, 0);
    inst_config_filename = cpl_frame_get_filename(inst_config_table_frame);
    my_error = espdr_read_inst_parameters(inst_config_filename,
                                          CCD_geom->ext_nb,
                                          inst_config);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting instrument configuration: %s",
                 cpl_error_get_message_default(my_error));
    
	cpl_frameset_delete(CCD_geom_frames);
    cpl_frameset_delete(inst_config_frames);
	
	return cpl_error_get_code();
}
#endif


/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs
 @param         lines_nb_per_order  number of valid lines for each order
 @param         thar_rejected_lines number of rejected lines
 @param         flux ratio          flux ratio
 @param         global_shift        global shift
 @param         chisq_per_order     reduced CHI2 of ll solution per order
 @param         rms_per_order       dispersion of residuals around fit per order
 @param         res_per_order       spectral resolution per order
 @param         lamp_offset         ThAr lamp offset
 @param         orders_nb           number of orders
 @param         fibre_source        calibration source type
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_FP_QC(int *lines_nb_per_order,
                                int orders_nb,
                                espdr_inst_config *inst_config,
                                espdr_qc_keywords *qc_kws,
                                cpl_propertylist *keywords_RE) {
    
    int i, total_lines_nb = 0;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    char comment_order[COMMENT_LENGTH];
    int order_lines_nb_QC = 1;
    int tot_lines_nb_QC = 1;
    int global_QC = 1;
    int previous_global_QC = 1;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int min_lines_per_order = inst_config->wave_sol_min_lines_per_order;
    
    sprintf(comment_order, "Nb of lines per order is OK");
    /* QC KWs per order */
    for (i = 0; i < orders_nb; i++) {
        new_keyword =
        espdr_add_index_to_keyword(qc_kws->qc_wave_order_lines_nb_kw_first,
                                   qc_kws->qc_wave_order_lines_nb_kw_last,
                                   i+1);
        sprintf(comment, "Number of valid lines in order %d", i+1);
        my_error = espdr_keyword_add_int(new_keyword,
                                         lines_nb_per_order[i],
                                         comment,
                                         &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER%d_LINES_NB_KW to the propertylist failed: %s", i+1,
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        if (lines_nb_per_order[i] < min_lines_per_order) {
            sprintf(comment_order, "Not enough lines in order %d", i+1);
            order_lines_nb_QC = 0;
            global_QC = 0;
        }
        
        total_lines_nb += lines_nb_per_order[i];
    }
    
    if (total_lines_nb < inst_config->fp_min_lines_total) {
        tot_lines_nb_QC = 0;
    }
    
    /* Global QC KWs */
    
    sprintf(comment, "Total number of valid lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_nb_kw,
                                     total_lines_nb,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LINES_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "QC on number of valid lines per order");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_order_check_kw,
                                     order_lines_nb_QC,
                                     comment_order,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LINES_ORDER_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "QC on total number of valid lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_tot_check_kw,
                                     tot_lines_nb_QC,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LINES_TOT_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    /* QC CHECK KWs */
    
    previous_global_QC = cpl_propertylist_get_int(keywords_RE, qc_kws->qc_wave_check_kw);
    global_QC = global_QC & previous_global_QC;
    
    sprintf(comment, "Overall WAVE QC");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_check_kw,
                                     global_QC,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    return(cpl_error_get_code());
}


