/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2017-12-04 19:28:49 +0100 (Mon, 04 Dec 2017) $
 * $Revision: 212639 $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_wave_LFC_cal.h>
#include <espdr_wave_FP_cal.h>
#define KSIG_EXTRACTION_MIN -2.0
#define KSIG_EXTRACTION_MAX 20.0


#define WAVE_SOL_POLY_DEG_MIN 1
#define WAVE_SOL_POLY_DEG_MAX 10
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     orders_fit  ...
 @param     orders_nb   number of orders
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int* orders_fit,
                                         const int orders_nb, const int ext_nb) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_wave_LFC");
    espdr_add_qc_key_stat_pri(keywords, ext_nb, "MAX FLUX", ksuff, CPL_FALSE, CPL_FALSE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE);

    int ord_min=200, ord_max=-1;
    for (int i = orders_nb-1; i >= 0; i--) {
        if(orders_fit[i]==1) {
            ord_min = (i < ord_min) ? i : ord_min;
        }
    }

    ord_min +=1; // as orders starts from 1 and not 0 as C array indexes
    for (int i = 0; i < orders_nb; i++) {
        if(orders_fit[i]==1) {
            ord_max = (i > ord_max) ? i : ord_max;
        }
    }

    ord_max +=1; // as orders starts from 1 and not 0 as C array indexes
    int norder = ord_max-ord_min+1;
    if(norder<0) {
       espdr_msg_warning("no orders with QC ORDERi LFC FIT WAVE = 1");
    }
    char key_val[40];
    sprintf(key_val,"ESO QC LFC ORDER MIN");
    cpl_propertylist_append_double(*keywords, key_val, ord_min);
    cpl_propertylist_set_comment(*keywords, key_val,
                                 "Smallest order <o> with QC.ORDER<o>.LFC.FIT.WAVE == 1");

    sprintf(key_val,"ESO QC LFC ORDER MAX");
    cpl_propertylist_append_double(*keywords, key_val, ord_max);
    cpl_propertylist_set_comment(*keywords, key_val,
                                 "Largest order <o> with QC.ORDER<o>.LFC.FIT.WAVE == 1");

    sprintf(key_val,"ESO QC LFC NORDER");
    cpl_propertylist_append_double(*keywords, key_val, norder);
    cpl_propertylist_set_comment(*keywords, key_val,
                                 "Num of orders with QC.ORDER<o>.LFC.FIT.WAVE == 1");

    espdr_add_qc_key_wave_LFC_ord_ref(keywords, orders_fit, ext_nb, ord_ref, "CHI2", ksuff,
                                      CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE, CPL_FALSE);

    cpl_propertylist* r1list = NULL;
    cpl_propertylist* r2list = NULL;
    cpl_propertylist* r3list = NULL;
    cpl_property* p;
    cpl_propertylist* olist = cpl_propertylist_duplicate(*keywords);
    cpl_propertylist_erase_regexp(olist, "QC.ORDER", 1);
    cpl_propertylist_erase_regexp(olist, "^HIERARCH ESO QC EXT", 0);
    char prefix[40];
    if(norder>0) {
        cpl_vector* chi2_v = cpl_vector_new(norder);
        cpl_vector* rms_v = cpl_vector_new(norder);
        double* p_chi2_v = cpl_vector_get_data(chi2_v);
        double* p_rms_v = cpl_vector_get_data(rms_v);
        const char* key_com = "QC.ORDER<o>.";

        for (int i = ord_min, k=0; i <= ord_max; i++, k++) {

            r2list = cpl_propertylist_duplicate(olist);
            r3list = cpl_propertylist_duplicate(olist);

            sprintf(prefix,"QC.ORDER%d.CHI2$",i);
            cpl_propertylist_erase_regexp(r2list, prefix,1);
            //cpl_propertylist_dump(r2list,stdout);
            p = cpl_propertylist_get(r2list,0);
            p_chi2_v[k] = cpl_property_get_double(p);

            sprintf(prefix,"QC.ORDER%d.RMS$",i);
            cpl_propertylist_erase_regexp(r3list, prefix,1);
            p = cpl_propertylist_get(r3list,0);
            p_rms_v[k] = cpl_property_get_double(p);


            cpl_propertylist_delete(r2list);
            cpl_propertylist_delete(r3list);
        }

        double chi2_avg = cpl_vector_get_mean(chi2_v);
        double chi2_rms = 0.0;
        if (cpl_vector_get_size(chi2_v) > 1) {
            chi2_rms = cpl_vector_get_stdev(chi2_v);
        } else {
            chi2_rms = 0.0;
        }

        double rms_avg = cpl_vector_get_mean(rms_v);
        double rms_rms = 0.0;
        if (cpl_vector_get_size(rms_v) > 1) {
            rms_rms = cpl_vector_get_stdev(rms_v);
        } else {
            rms_rms = 0.0;
        }
        char kname[80];
        char kcomm[80];
        sprintf(kname,"ESO QC CHI2 AVG");
        cpl_propertylist_append_double(*keywords,kname,chi2_avg);
        sprintf(kcomm,"Avg of ORDER<o>.CHI2 & QC.ORDER<o>.LFC.FIT.WAVE==1");

        sprintf(kname,"ESO QC CHI2 RMS");
        cpl_propertylist_append_double(*keywords,kname,chi2_rms);
        sprintf(kcomm,"RMS of ORDER<o>.CHI2 & QC.ORDER<o>.LFC.FIT.WAVE==1");

        sprintf(kname,"ESO QC RMS AVG");
        cpl_propertylist_append_double(*keywords,kname,rms_avg);
        sprintf(kcomm,"Avg of ORDER<o>.RMS & QC.ORDER<o>.LFC.FIT.WAVE==1");

        sprintf(kname,"ESO QC RMS RMS");
        cpl_propertylist_append_double(*keywords,kname,rms_rms);
        sprintf(kcomm,"RMS of ORDER<o>.RMS & QC.ORDER<o>.LFC.FIT.WAVE==1");

        cpl_vector_delete(chi2_v);
        cpl_vector_delete(rms_v);
        }

    cpl_free(ord_ref);

    return cpl_error_get_code();
}




/*---------------------------------------------------------------------------*/
/**
 @brief     Save wave_LFC products
 @param
 @return    CPL_ERROR_NONE iff OK

 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_wave_LFC_products(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            const char *recipe_id,
                                            cpl_frameset *used_frames,
                                            const char *WAVE_tag,
                                            int fibre_nr,
                                            cpl_propertylist *keywords,
                                            espdr_inst_config *inst_config,
                                            cpl_image *wave_matrix,
                                            cpl_image *air_wave_matrix,
                                            cpl_image *dll_matrix,
                                            cpl_image *air_dll_matrix,
                                            cpl_table *coeffs_table,
                                            cpl_table *LFC_table) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char filename[FILENAME_LENGTH];
    char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));

    // Saving vacuum WAVE_MATRIX
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_WAVE_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_wave_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("WAVE_MATRIX saved under: %s", filename);

    // Saving air WAVE_MATRIX
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_WAVE_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_air_wave_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("AIR_WAVE_MATRIX saved under: %s", filename);

    // Saving vacuum and air dll matrices
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_DLL_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_dll_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("DLL_MATRIX saved under: %s", filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_DLL_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_air_dll_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("AIR_DLL_MATRIX saved under: %s", filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_WAVE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_wave_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving WAVE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, coeffs_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("WAVE_TABLE saved under: %s", filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_LFC_FITTED_LINE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->lfc_fitted_line_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving LFC FITTED LINES TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, LFC_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

    return (cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_wave_LFC(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

    cpl_frameset *wave_frames_to_reduce = NULL;
    cpl_frame *wave_frame = NULL;
    cpl_frameset **THAR_wave_matrix_frames = NULL;
    cpl_frameset **THAR_dll_matrix_frames = NULL;
    cpl_frame **THAR_wave_matrix_frame = NULL;
    cpl_frame **THAR_dll_matrix_frame = NULL;

    cpl_frameset *used_frames = NULL;

    cpl_image *lfc_wave_matrix_sol = NULL;
    cpl_image *lfc_dll_matrix_sol = NULL;
    cpl_image *lfc_air_wave_matrix_sol = NULL;
    cpl_image *lfc_air_dll_matrix_sol = NULL;
    cpl_image *wave_matrix = NULL;
    cpl_image *wave_matrix_order = NULL;
    cpl_image *dll_matrix = NULL;
    cpl_image *final_wave_matrix = NULL;
    cpl_image *final_air_wave_matrix = NULL;
    cpl_image *final_dll_matrix = NULL;
    cpl_image *final_air_dll_matrix = NULL;

	/* input frames filenames */
    const char **THAR_wave_matrix_filename = NULL;
    const char **THAR_dll_matrix_filename = NULL;

    espdr_line_param **FP_lines_table = NULL;
    espdr_line_param **LFC_lines_table = NULL;
    espdr_line_param **LFC_lines_table_final = NULL;
    cpl_table **LFC_lines_table_FITS = NULL;
    cpl_table **coeffs_table_FITS = NULL;
    char column_name[KEYWORD_LENGTH];

    int size_x, size_y;
    double *wave_matrix_order_data = NULL;
    double **lfc_wave_matrix_sol_data = NULL;
    double **lfc_dll_matrix_sol_data = NULL;
    double **lfc_air_wave_matrix_sol_data = NULL;
    double **lfc_air_dll_matrix_sol_data = NULL;
    int **LFC_lines_nb_per_order = NULL;
    double ***coeffs_per_order = NULL;
    double **rms_per_order = NULL;
    double **chisq_per_order = NULL;
    double *residuals = NULL;
    int **order_fit = NULL;
    double ***wave_matrix_data = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 28;
    const char* rec_tags[28] = {
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_ORDER_PROFILE_A, ESPDR_PRO_CATG_ORDER_PROFILE_B,
        ESPDR_PRO_CATG_FLAT_A, ESPDR_PRO_CATG_FLAT_B,
        ESPDR_PRO_CATG_BLAZE_A, ESPDR_PRO_CATG_BLAZE_B,
        ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_A,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_B, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_FP_A,
        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_A, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_B,
        ESPDR_PRO_CATG_PIXEL_GEOM_A, ESPDR_PRO_CATG_PIXEL_GEOM_B,
        ESPDR_PRO_CATG_PIXEL_SIZE_A, ESPDR_PRO_CATG_PIXEL_SIZE_B,
        ESPDR_WAVE_LFC_FP_RAW, ESPDR_WAVE_FP_LFC_RAW
    };
    int is_required[28] = {1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // THAR wave solutions, can be A or B
        0, 0, 0, 0, // pixel geometry & size maps
        0, 0}; // raw frames, can be LFC_FP or FP_LFC

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting wave LFC");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw wave LFC_FP or FP_LFC image, tagged as LFC_FP or FP_LFC\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n"
                 "wave matrix for fibres A or B, tagged as WAVE_MATRIX_THAR_FP_A or WAVE_MATRIX_FP_THAR_B\n"
                 "dll matrix for fibres A or B, tagged as DLL_MATRIX_THAR_FP_A or DLL_MATRIX_FP_THAR_B\n");

    espdr_CCD_geometry *CCD_geom            = NULL;
    espdr_inst_config *inst_config          = NULL;
    espdr_qc_keywords *qc_kws               = NULL;
    espdr_OVSC_param *OVSC_param            = NULL;
    espdr_WAVE_LFC_param *WAVE_LFC_param    = NULL;
    cpl_frame* CCD_geom_frame               = NULL;
    cpl_frame* inst_config_frame            = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    WAVE_LFC_param    = espdr_WAVE_LFC_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_LFC_FP_RAW);
    if (wave_frame == NULL) {
        wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_FP_LFC_RAW);
        if (wave_frame == NULL) {
            espdr_msg_error("No input raw frame with any of the TAGs: LFC_FP, FP_LFC, exiting");
            return(CPL_ERROR_NULL_INPUT);
        }
    }

    const char *WAVE_tag = cpl_frame_get_tag(wave_frame);
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    wave_frames_to_reduce = cpl_frameset_new();
    cpl_frameset_insert(wave_frames_to_reduce, cpl_frame_duplicate(wave_frame));
    espdr_msg("WAVE_tag: %s", WAVE_tag);

    espdr_src_type fibre_source[inst_config->fibres_nb];
    char **fibre_WAVE = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_WAVE[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    my_error = espdr_get_fibre_source(WAVE_tag, fibre_source, fibre_WAVE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_fibre_source failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("fibre_WAVE: %s & %s", fibre_WAVE[0], fibre_WAVE[1]);

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }


    /* Check for LFC & FP source */
    int LFC_present[inst_config->fibres_nb];
    int FP_present[inst_config->fibres_nb];
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        LFC_present[i] = 0;
        if (fibre_source[i] == LFC) {
            LFC_present[i] = 1;
        }
        FP_present[i] = 0;
        if (fibre_source[i] == FP) {
            FP_present[i] = 1;
        }
    }

    /* We need the previous WAVE MATRIX & previous DLL MATRIX in the SoF */
    char pro_catg_tag[TAG_LENGTH];
    THAR_wave_matrix_frames = (cpl_frameset **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frameset *));
    THAR_dll_matrix_frames = (cpl_frameset **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frameset *));
    THAR_wave_matrix_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    THAR_dll_matrix_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    THAR_wave_matrix_filename = (const char **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    THAR_dll_matrix_filename = (const char **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (LFC_present[i]) {
            // LFC needs previousely computed wave matrix
            if (i == 0) {
                THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A);
                if (THAR_wave_matrix_frame[i] == NULL) {
                    THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_FP_A);
                    if (THAR_wave_matrix_frame[i] == NULL) {
                        THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_A);
                        if (THAR_wave_matrix_frame[i] == NULL) {
                            espdr_msg_error("No THAR_WAVE_MATRIX_A, exiting");
                            exit(0);
                        } else {
                            sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_A);
                        }
                    } else {
                        sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_FP_A);
                    }
                } else {
                    sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A);
                }
                espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT,
                             "THAR WAVE MATRIX A frame extraction failed: %s",
                             cpl_error_get_message_default(cpl_error_get_code()));
            } else {
                THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B);
                if (THAR_wave_matrix_frame[i] == NULL) {
                    THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_B);
                    if (THAR_wave_matrix_frame[i] == NULL) {
                        espdr_msg_error("No THAR_WAVE_MATRIX_B, exiting");
                        exit(0);
                    } else {
                        sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_B);
                    }
                } else {
                    sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B);
                }
                espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT,
                             "THAR WAVE MATRIX B frame extraction failed: %s",
                             cpl_error_get_message_default(cpl_error_get_code()));
            }
            
            //if (i == 0) {
            //    sprintf(pro_catg_tag, "%s_THAR_FP_%c", ESPDR_PRO_CATG_WAVE_MATRIX, fibre_name[i]);
            //} else {
            //    sprintf(pro_catg_tag, "%s_FP_THAR_%c", ESPDR_PRO_CATG_WAVE_MATRIX, fibre_name[i]);
            //}
            //THAR_wave_matrix_frame[i] = espdr_frame_find(frameset, pro_catg_tag);
            //espdr_ensure(THAR_wave_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
            //             "THAR WAVE MATRIX frame extraction failed");

            THAR_wave_matrix_filename[i] = cpl_frame_get_filename(THAR_wave_matrix_frame[i]);
            cpl_propertylist *THAR_keywords = cpl_propertylist_load(THAR_wave_matrix_filename[i], 0);
            const char *THAR_pro_catg = cpl_propertylist_get_string(THAR_keywords, qc_kws->pro_catg_kw);
            //espdr_msg("THAR WAVE PRO CATG:::: %s %s",
            //          qc_kws->pro_catg_kw, THAR_pro_catg);
            espdr_ensure(strcmp(THAR_pro_catg, pro_catg_tag) != 0,
                         CPL_ERROR_INCOMPATIBLE_INPUT,
                         "The input WAVE_MATRIX_%s_%c has wrong PRO.CATG (%s)",
                         pro_catg_tag, fibre_name[i], THAR_pro_catg);
            cpl_propertylist_delete(THAR_keywords);

            my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(THAR_wave_matrix_frame[i]));

            // LFC needs previousely computed dll matrix
            if (i == 0) {
                THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A);
                if (THAR_dll_matrix_frame[i] == NULL) {
                    THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_FP_A);
                    if (THAR_dll_matrix_frame[i] == NULL) {
                        THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_A);
                        if (THAR_dll_matrix_frame[i] == NULL) {
                            espdr_msg_error("No THAR_DLL_MATRIX_A, exiting");
                            exit(0);
                        } else {
                            sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_A);
                        }
                    } else {
                        sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_FP_A);
                    }
                } else {
                    sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A);
                }
                espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT,
                             "THAR DLL MATRIX A frame extraction failed: %s",
                             cpl_error_get_message_default(cpl_error_get_code()));
            } else {
                THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B);
                if (THAR_dll_matrix_frame[i] == NULL) {
                    THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_B);
                    if (THAR_dll_matrix_frame[i] == NULL) {
                        espdr_msg_error("No THAR_DLL_MATRIX_B, exiting");
                        exit(0);
                    } else {
                        sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_B);
                    }
                } else {
                    sprintf(pro_catg_tag, "%s", ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B);
                }
                espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, CPL_ERROR_NULL_INPUT,
                             "THAR DLL MATRIX B frame extraction failed: %s",
                             cpl_error_get_message_default(cpl_error_get_code()));
            }
            
            //if (i == 0) {
            //    sprintf(pro_catg_tag, "%s_THAR_FP_%c", ESPDR_PRO_CATG_DLL_MATRIX, fibre_name[i]);
            //} else {
            //    sprintf(pro_catg_tag, "%s_FP_THAR_%c", ESPDR_PRO_CATG_DLL_MATRIX, fibre_name[i]);
            //}
            //THAR_dll_matrix_frame[i] = espdr_frame_find(frameset, pro_catg_tag);
            //espdr_ensure(THAR_dll_matrix_frame == NULL, CPL_ERROR_NULL_INPUT,
            //             "THAR DLL MATRIX frame extraction failed");

            THAR_dll_matrix_filename[i] = cpl_frame_get_filename(THAR_dll_matrix_frame[i]);
            cpl_propertylist *DLL_keywords = cpl_propertylist_load(THAR_dll_matrix_filename[i], 0);
            THAR_pro_catg = cpl_propertylist_get_string(DLL_keywords, qc_kws->pro_catg_kw);
            //espdr_msg("THAR DLL PRO CATG:::: %s %s",
            //          qc_kws->pro_catg_kw, THAR_pro_catg);
            espdr_ensure(strcmp(THAR_pro_catg, pro_catg_tag) != 0,
                         CPL_ERROR_INCOMPATIBLE_INPUT,
                         "The input DLL_MATRIX_%s_%c has wrong PRO.CATG (%s)",
                         pro_catg_tag, fibre_name[i], THAR_pro_catg);
            cpl_propertylist_delete(DLL_keywords);

            my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(THAR_dll_matrix_frame[i]));
        }
    }

    cpl_image **pixel_geom_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **pixel_size_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_propertylist *));
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    cpl_table ***orders_coeffs = NULL;
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    cpl_frame** REF_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_frame** FP_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_image **static_wave_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **static_dll_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_frame *s2d_blaze_FP_frame = NULL;
    my_error = espdr_process_wave_till_extraction(frameset, parameters, recipe_id,
                                                  wave_frames_to_reduce,
                                                  wave_frame,
                                                  used_frames,
                                                  inst_config,
                                                  CCD_geom,
                                                  qc_kws,
                                                  orders_nb_per_fibre,
                                                  fibre_source,
                                                  fibre_WAVE,
                                                  keywords_fibre,
                                                  CCD_corrected_image,
                                                  &orders_coeffs,
                                                  RON,
                                                  flat_corr_spectrum,
                                                  flat_corr_error,
                                                  flat_corr_qual,
                                                  flat_blaze_corr_spectrum,
                                                  flat_blaze_corr_error,
                                                  flat_blaze_corr_qual,
                                                  pixel_geom_image,
                                                  pixel_size_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    /* LINES TABLE reading */
    espdr_msg("Lines tables & coeffs tables memory allocation");
    LFC_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    LFC_lines_table_final = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    FP_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        LFC_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
        LFC_lines_table_final[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
        FP_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
        my_error = espdr_lines_table_init(FP_lines_table[i], MAX_NB_SPECTRAL_LINES);
        my_error = espdr_lines_table_init(LFC_lines_table[i], MAX_NB_SPECTRAL_LINES);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_lines_table_init of LFC_lines_table for fibre %c failed: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
        my_error = espdr_lines_table_init(LFC_lines_table_final[i],
                                          MAX_NB_SPECTRAL_LINES);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_lines_table_init of LFC_lines_table_final for fibre %c failed: %s",
                     fibre_name[i], cpl_error_get_message_default(my_error));
    }

    int LFC_lines_nb[inst_config->fibres_nb];
    LFC_lines_nb_per_order = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(int *));
    coeffs_per_order = (double ***)cpl_malloc(inst_config->fibres_nb*sizeof(double **));
    coeffs_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table*));
    rms_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    chisq_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    order_fit = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        size_x = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
        LFC_lines_nb_per_order[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        coeffs_per_order[i] = (double **)cpl_malloc(orders_nb_per_fibre[i] * sizeof(double*));
        rms_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
        chisq_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
        order_fit[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
            coeffs_per_order[i][j] = (double *)cpl_calloc(inst_config->wave_sol_poly_deg, sizeof(double));
            for (int k = 0; k < inst_config->wave_sol_poly_deg; k++) {
                coeffs_per_order[i][j][k] = 0.0;
            }
        }

        coeffs_table_FITS[i] = cpl_table_new(orders_nb_per_fibre[i]);
        my_error = cpl_table_new_column(coeffs_table_FITS[i],
                                        COL_NAME_ORDER,
                                        CPL_TYPE_INT);
        for (int j = 0; j < inst_config->wave_sol_poly_deg; j++) {
            sprintf(column_name, "COEFF_%d", j);
            //espdr_msg("column_name: %s", column_name);
            my_error = cpl_table_new_column(coeffs_table_FITS[i], column_name,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed for coeff %d: %s",
                         j, cpl_error_get_message_default(my_error));
        }
    }

    /* Read the pixel geometry image or prepare one if not provided */
    wave_matrix_data = (double ***)cpl_malloc(inst_config->fibres_nb * sizeof(double **));
    lfc_wave_matrix_sol_data = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double));
    lfc_dll_matrix_sol_data = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double));
    lfc_air_wave_matrix_sol_data = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double));
    lfc_air_dll_matrix_sol_data = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double));

    /* Lines search */
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Fibre %c TAG: %s", fibre_name[i], fibre_WAVE[i]);
        if (fibre_source[i] == LFC) {
            LFC_lines_nb[i] = 0;
            espdr_msg("Orders nb for fibre %c: %d", fibre_name[i], orders_nb_per_fibre[i]);
            for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                LFC_lines_nb_per_order[i][j] = 0;
            }

            // LFC
            espdr_ensure((flat_blaze_corr_spectrum[i] == NULL) ||
                         (flat_blaze_corr_error[i] == NULL) ||
                         (flat_blaze_corr_qual[i] == NULL),
                         CPL_ERROR_NULL_INPUT,
                         "FLAT & BLAZE corrected spectrum for fibre %c is NULL", fibre_name[i]);

            espdr_msg("Searching LFC lines for fibre: %c", fibre_name[i]);
            my_error = espdr_search_lines(flat_blaze_corr_spectrum[i],
                                          flat_blaze_corr_error[i],
                                          flat_blaze_corr_qual[i],
                                          inst_config,
                                          LFC,
                                          LFC_lines_table[i],
                                          &LFC_lines_nb[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_search_lines failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("LFC lines nb after search: %d", LFC_lines_nb[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_LFC
            my_error = espdr_save_debug_table_RDB(LFC_lines_table[i], LFC_lines_nb[i],
                                                  "LFC_SEARCHED_LINE_TABLE",
                                                  i, inst_config);
#endif

            espdr_msg("Fitting LFC lines");
            espdr_msg("Orders nb for fibre %c: %d", fibre_name[i], orders_nb_per_fibre[i]);

            my_error = espdr_fit_lines(LFC_lines_table[i],
                                       LFC_lines_nb[i],
                                       flat_blaze_corr_spectrum[i],
                                       flat_blaze_corr_error[i],
                                       flat_blaze_corr_qual[i],
                                       pixel_geom_image[i],
                                       inst_config->fp_peak_flux_threshold,
                                       inst_config->fp_fit_window_size,
                                       inst_config->fp_fit_tolerance_window_size,
                                       inst_config->fp_fit_ngauss_flag,
                                       LFC_lines_nb_per_order[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_lines failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("espdr_fit_lines done");

            if (inst_config->fp_fit_ngauss_flag) { // No cleaning if the fit was not performed
            	my_error = espdr_FP_lines_qc(LFC_lines_table[i],
                                             LFC_lines_nb[i],
                                             inst_config);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_FP_lines_QC failed: %s",
                             cpl_error_get_message_default(my_error));
                espdr_msg("espdr_FP_lines_qc done");
            }

#if SAVE_DEBUG_PRODUCT_WAVE_LFC
            my_error = espdr_save_debug_table_RDB(LFC_lines_table[i], LFC_lines_nb[i],
                                                  "LFC_FITTED_LINE_TABLE",
                                                  i, inst_config);
#endif

            wave_matrix = cpl_image_load(THAR_wave_matrix_filename[i],
                                         CPL_TYPE_DOUBLE, 0, 1);
            my_error = cpl_error_get_code();
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "WAVE_MATRIX (%s) load failed: %s",
                         THAR_wave_matrix_filename[i],
                         cpl_error_get_message_default(my_error));

            size_x = cpl_image_get_size_x(wave_matrix);
            size_y = cpl_image_get_size_y(wave_matrix);
            espdr_msg("wave matrix size: %d x %d", size_x, size_y);
            wave_matrix_data[i] = (double **)cpl_malloc(size_y * sizeof(double *));

            for (int order = 0; order < size_y; order++) {
                wave_matrix_order = cpl_image_extract(wave_matrix, 1, order+1, size_x, order+1);
                wave_matrix_order_data = cpl_image_get_data_double(wave_matrix_order);
                wave_matrix_data[i][order] = (double *)cpl_calloc (cpl_image_get_size_x(wave_matrix),
                                                                   sizeof(double));
                for (int j = 0; j < size_x; j++) {
                    wave_matrix_data[i][order][j] = wave_matrix_order_data[j];
                }
                cpl_image_delete(wave_matrix_order);
            }

            double lfc_rep_rate = cpl_propertylist_get_double(keywords_fibre[i],
                                                              inst_config->rep_rate_kw);
            lfc_rep_rate = lfc_rep_rate / 1000000000.0;
            double lfc_anchor_freq = cpl_propertylist_get_double(keywords_fibre[i],
                                                                 inst_config->anchor_freq_kw);
            lfc_anchor_freq = lfc_anchor_freq / 1000000000000.0;
            espdr_msg("Assigning LFC wavelengths for fibre: %c with rep rate = %e and anchor freq = %e",
                      fibre_name[i], lfc_rep_rate, lfc_anchor_freq);
            my_error = espdr_assign_comb_wavelengths(LFC_lines_table[i],
                                                     wave_matrix_data[i],
                                                     lfc_rep_rate,
                                                     lfc_anchor_freq);
            espdr_msg("espdrp_assign_comb_wavelengths done");

            for (int order = 0; order < size_y; order++) {
                cpl_free(wave_matrix_data[i][order]);
            }
            cpl_free(wave_matrix_data[i]);
            cpl_image_delete(wave_matrix);

            /*
            int line = 0;
            LFC_lines_nb[i] = 0;
            while ((LFC_lines_table[i][line].order > 0) &&
                   (line < MAX_NB_SPECTRAL_LINES)) {
                if (fabs(LFC_lines_table[i][line].Nreal - LFC_lines_table[i][line].N) <
                    inst_config->lfc_spectral_line_qc) {
                    LFC_lines_nb[i]++;
                }
                line++;
            }
            */

            espdr_msg("Cleaning LFC lines table");
            my_error = espdr_clean_LFC_lines_table(LFC_lines_table[i],
                                                   LFC_lines_table_final[i],
                                                   inst_config->lfc_spectral_line_qc,
                                                   fibre_source[i],
                                                   &LFC_lines_nb[i]);
            espdr_msg("LFC lines nb after cleaning: %d", LFC_lines_nb[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_LFC
            my_error = espdr_save_debug_table_RDB(LFC_lines_table[i], LFC_lines_nb[i],
                                                  "LFC_ASSIGNED_LINE_TABLE",
                                                  i, inst_config);
#endif

            espdr_msg("Fitting the LFC wavelength solution for fibre %c", fibre_name[i]);
            espdr_msg("WAVE_MATRIX size: %d x %d", size_x, size_y);
            int xy = size_x * size_y;
            lfc_wave_matrix_sol_data[i] = (double *)cpl_calloc(xy, sizeof(double));
            lfc_dll_matrix_sol_data[i] = (double *)cpl_calloc(xy, sizeof(double));
            lfc_air_wave_matrix_sol_data[i] = (double *)cpl_calloc(xy, sizeof(double));
            lfc_air_dll_matrix_sol_data[i] = (double *)cpl_calloc(xy, sizeof(double));

            my_error = espdr_fit_ll_sol(LFC_lines_table_final[i],
                                        LFC_lines_nb[i],
                                        inst_config,
                                        pixel_geom_image[i],
                                        pixel_size_image[i],
                                        fibre_source[i],
                                        i,
                                        LFC_lines_nb_per_order[i],
                                        rms_per_order[i],
                                        chisq_per_order[i],
                                        order_fit[i],
                                        coeffs_per_order[i],
                                        lfc_wave_matrix_sol_data[i],
                                        lfc_dll_matrix_sol_data[i],
                                        lfc_air_wave_matrix_sol_data[i],
                                        lfc_air_dll_matrix_sol_data[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_ll_sol failed for fibre %c: %s",
                         fibre_name[i],
                         cpl_error_get_message_default(my_error));

            espdr_msg("Computing dlldx for LFC lines for fibre %c", fibre_name[i]);
            my_error = espdr_compute_dlldx(LFC_lines_table_final[i],
                                           LFC_lines_nb[i],
                                           orders_nb_per_fibre[i],
                                           inst_config,
                                           coeffs_per_order[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_dlldx failed: %s",
                         cpl_error_get_message_default(my_error));

            /* Create vacuum and air WAVE_MATRIX images */
            size_x = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
            espdr_msg("size_x = %d", size_x);
            lfc_wave_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                 CPL_TYPE_DOUBLE,
                                                 lfc_wave_matrix_sol_data[i]);
            espdr_msg("WAVE_MATRIX size: %lld x %lld",
                      cpl_image_get_size_x(lfc_wave_matrix_sol),
                      cpl_image_get_size_y(lfc_wave_matrix_sol));

            lfc_air_wave_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                     CPL_TYPE_DOUBLE,
                                                     lfc_air_wave_matrix_sol_data[i]);
            espdr_msg("AIR_WAVE_MATRIX size: %lld x %lld",
                      cpl_image_get_size_x(lfc_air_wave_matrix_sol),
                      cpl_image_get_size_y(lfc_air_wave_matrix_sol));

            /* Create vacuum and air DLL_MATRIX image */
            lfc_dll_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                CPL_TYPE_DOUBLE,
                                                lfc_dll_matrix_sol_data[i]);

            lfc_air_dll_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                    CPL_TYPE_DOUBLE,
                                                    lfc_air_dll_matrix_sol_data[i]);

            /* Create WAVE_TABLE fits */
            for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                my_error = cpl_table_set_int(coeffs_table_FITS[i],
                                             COL_NAME_ORDER, j, j+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_table_set_int failed for order %d: %s",
                             j, cpl_error_get_message_default(my_error));

                for (int k = 0; k < inst_config->wave_sol_poly_deg; k++) {
                    sprintf(column_name, "COEFF_%d", k);
                    my_error = cpl_table_set_double(coeffs_table_FITS[i],
                                                    column_name, j,
                                                    coeffs_per_order[i][j][k]);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_table_set_double failed for coeff %d: %s",
                                 k, cpl_error_get_message_default(my_error));
                }
            }
        } else {
            for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                LFC_lines_nb_per_order[i][j] = -1;
                chisq_per_order[i][j] = -1.0;
                rms_per_order[i][j] = -1.0;
                order_fit[i][j] = -1.0;
            }
        } // end if LFC

    } // for loop on fibres

    espdr_msg("Computations done, now saving products");
    LFC_lines_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table*));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == LFC) {
            LFC_lines_table_FITS[i] = cpl_table_new(LFC_lines_nb[i]);
            my_error = espdr_FITS_lines_table_create(LFC_lines_table_FITS[i]);
            my_error = espdr_FITS_lines_table_fill(LFC_lines_table_FITS[i],
                                                   LFC_lines_table_final[i],
                                                   LFC_lines_nb[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "FITS LFC table creation for fibre %c failed: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
            espdr_msg("LFC line table for fibre %c has %lld rows",
                      fibre_name[i], cpl_table_get_nrow(LFC_lines_table_FITS[i]));
        }
    }

    espdr_msg("Adding QC KWs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_propertylist *wave_keywords = NULL;
        if (fibre_source[i] == LFC) {
            // Merge the LFC wave solution with the input THAR one for non-existing LFC orders
            wave_matrix = cpl_image_load(THAR_wave_matrix_filename[i],
                                         CPL_TYPE_DOUBLE, 0, 1);
            wave_keywords = cpl_propertylist_load(THAR_wave_matrix_filename[i], 0);
            my_error = espdr_merge_LFC_THAR(lfc_wave_matrix_sol,
                                            wave_matrix,
                                            order_fit[i],
                                            &final_wave_matrix,
                                            &final_air_wave_matrix);
            espdr_msg("Final merged THAR-LFC wave matrix size: %lld x %lld",
                      cpl_image_get_size_x(final_wave_matrix),
                      cpl_image_get_size_y(final_wave_matrix));

            dll_matrix = cpl_image_load(THAR_dll_matrix_filename[i],
                                         CPL_TYPE_DOUBLE, 0, 1);
            my_error = espdr_merge_LFC_THAR(lfc_dll_matrix_sol,
                                            dll_matrix,
                                            order_fit[i],
                                            &final_dll_matrix,
                                            &final_air_dll_matrix);
        }

        my_error = espdr_wave_LFC_QC(LFC_lines_nb_per_order[i],
                                     chisq_per_order[i],
                                     rms_per_order[i],
                                     order_fit[i],
                                     orders_nb_per_fibre[i],
                                     i, inst_config, qc_kws,
                                     wave_keywords,
                                     keywords_fibre[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_wave_LFC_QC failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error),
                     fibre_name[i]);

        if (fibre_source[i] == LFC) {
            cpl_propertylist_delete(wave_keywords);
        }
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        // Put the global wave check on the non-LFC fiber as on the LFC fiber
        if (fibre_source[i] != LFC) {
            int wave_check_LFC = cpl_propertylist_get_int(keywords_fibre[(i+1)%2],
                                                           qc_kws->qc_lfc_check_kw);
            my_error = cpl_propertylist_update_int(keywords_fibre[i],
                                                   qc_kws->qc_lfc_check_kw,
                                                   wave_check_LFC);
        }
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {

        my_error = espdr_save_wave_S2D_products(frameset, parameters, recipe_id,
                                                used_frames, WAVE_tag, i,
                                                keywords_fibre[i],
                                                inst_config,
                                                flat_corr_spectrum[i],
                                                flat_corr_error[i],
                                                flat_corr_qual[i],
                                                flat_blaze_corr_spectrum[i],
                                                flat_blaze_corr_error[i],
                                                flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_wave_S2D_products failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error), fibre_name[i]);

        if (fibre_source[i] == LFC) {
            if (strcmp(inst_config->instrument, "HARPN") != 0 ) {
                my_error = espdr_add_extra_qc(&(keywords_fibre)[i],  order_fit[i],
                                              orders_nb_per_fibre[i], CCD_geom->ext_nb);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_add_extra_qc failed: %s ",
                             cpl_error_get_message_default(my_error));
            }

            my_error = espdr_save_wave_LFC_products(frameset, parameters, recipe_id,
                                                    used_frames, WAVE_tag, i,
                                                    keywords_fibre[i],
                                                    inst_config,
                                                    final_wave_matrix,
                                                    final_air_wave_matrix,
                                                    final_dll_matrix,
                                                    final_air_dll_matrix,
                                                    coeffs_table_FITS[i],
                                                    LFC_lines_table_FITS[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_save_wave_THAR_products failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
        }
    }

    espdr_msg("Cleaning the memory");

	// Cleaning memory.
    espdr_msg("Cleaning keywords lists");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_propertylist_delete(keywords_fibre[i]);
    }

	cpl_free(qc_kws);

    espdr_msg("Cleaning calibration products");
	cpl_imagelist_delete(CCD_corrected_image);
    cpl_frameset_delete(wave_frames_to_reduce);
    cpl_frameset_delete(used_frames);

    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);

    espdr_msg("Cleaning products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_corr_spectrum[i]);
        cpl_image_delete(flat_corr_error[i]);
        cpl_image_delete(flat_corr_qual[i]);
    }
    cpl_free(flat_corr_spectrum);
    cpl_free(flat_corr_error);
    cpl_free(flat_corr_qual);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_blaze_corr_spectrum[i]);
        cpl_image_delete(flat_blaze_corr_error[i]);
        cpl_image_delete(flat_blaze_corr_qual[i]);
    }
    cpl_free(flat_blaze_corr_spectrum);
    cpl_free(flat_blaze_corr_error);
    cpl_free(flat_blaze_corr_qual);

    espdr_msg("Cleaning pixel geometry & size pointers");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(pixel_geom_image[i]);
        cpl_image_delete(pixel_size_image[i]);
    }
    cpl_free(pixel_geom_image);
    cpl_free(pixel_size_image);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);
    cpl_free(THAR_wave_matrix_filename);
    cpl_free(THAR_dll_matrix_filename);

    espdr_msg("Cleaning global QC structures");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(LFC_lines_nb_per_order[i]);
        cpl_free(rms_per_order[i]);
        cpl_free(chisq_per_order[i]);
        cpl_free(order_fit[i]);
        for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
            cpl_free(coeffs_per_order[i][j]);
        }
        cpl_free(coeffs_per_order[i]);
    }

    cpl_free(LFC_lines_nb_per_order);
    cpl_free(rms_per_order);
    cpl_free(chisq_per_order);
    cpl_free(order_fit);
    cpl_free(coeffs_per_order);

    cpl_free(orders_nb_per_fibre);

    espdr_msg("Cleaning additional data tables");
    cpl_free(residuals);

    espdr_msg("Cleaning matrices data structures");
    cpl_image_unwrap(lfc_wave_matrix_sol);
    cpl_image_unwrap(lfc_dll_matrix_sol);
    cpl_image_unwrap(lfc_air_wave_matrix_sol);
    cpl_image_unwrap(lfc_air_dll_matrix_sol);
    cpl_free(lfc_wave_matrix_sol_data);
    cpl_free(lfc_dll_matrix_sol_data);
    cpl_free(lfc_air_wave_matrix_sol_data);
    cpl_free(lfc_air_dll_matrix_sol_data);
    cpl_image_delete(final_wave_matrix);
    cpl_image_delete(final_air_wave_matrix);
    cpl_image_delete(final_dll_matrix);
    cpl_image_delete(final_air_dll_matrix);

    espdr_msg("Cleaning lines tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(FP_lines_table[i]);
        cpl_free(LFC_lines_table[i]);
        cpl_free(LFC_lines_table_final[i]);
    }
    cpl_free(FP_lines_table);
    cpl_free(LFC_lines_table);
    cpl_free(LFC_lines_table_final);

    espdr_msg("Cleaning FITS tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        // problem with freeing this part
        cpl_table_delete(coeffs_table_FITS[i]);
        if (fibre_source[i] == LFC) {
            cpl_table_delete(LFC_lines_table_FITS[i]);
        }
    }
    cpl_free(coeffs_table_FITS);
    cpl_free(LFC_lines_table_FITS);

    espdr_msg("Cleaning parameters");
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_WAVE_LFC_delete(WAVE_LFC_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    espdr_msg("End of the recipe espdr_wave_LFC");

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_create(const char* recipe_id,
                                                cpl_parameterlist *list,
                                                espdr_WAVE_LFC_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    cpl_error_code my_error = cpl_error_get_code();
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	double ksig_extraction_min = KSIG_EXTRACTION_MIN;
	double ksig_extraction_max = KSIG_EXTRACTION_MAX;
    
	cpl_error_code error_got = CPL_ERROR_NONE;
	char comment[COMMENT_LENGTH];
    
    /* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "lfc_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");

	sprintf(comment, "ksigma for extraction, must be between: %.2lf and %.2lf",
			ksig_extraction_min, ksig_extraction_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "lfc_extraction_ksigma",
                                                  p->ksig_extraction,
                                                  KSIG_EXTRACTION_MIN,
                                                  KSIG_EXTRACTION_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksig_extraction to the list");

    sprintf(comment, "wavelength solution polynomial degree in wave_THAR, must be between: %d and %d",
            WAVE_SOL_POLY_DEG_MIN, WAVE_SOL_POLY_DEG_MIN);
    my_error = espdr_parameters_new_range_int(recipe_id, list,
                                              "wave_sol_poly_deg",
                                              p->wave_sol_poly_deg,
                                              WAVE_SOL_POLY_DEG_MIN,
                                              WAVE_SOL_POLY_DEG_MAX,
                                              comment);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error adding parameter d_fit_poly_deg to the list");
    
    return (CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_delete(espdr_WAVE_LFC_param* p) {
	
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the WAVE recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_get(const char* recipe_id,
                                             cpl_parameterlist* param_list,
                                             espdr_WAVE_LFC_param *WAVE_LFC_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    WAVE_LFC_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                    param_list,
                                                                    "lfc_extraction_method");
    
    WAVE_LFC_param->ksig_extraction = espdr_parameters_get_double(recipe_id,
                                                                  param_list,
                                                                  "lfc_extraction_ksigma");
    
    WAVE_LFC_param->wave_sol_poly_deg = espdr_parameters_get_int(recipe_id,
                                                                 param_list,
                                                                 "wave_sol_poly_deg");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the WAVE parameters
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_print(espdr_WAVE_LFC_param *WAVE_LFC_param) {
	
	espdr_msg("\tWAVE LFC parameters:");
    espdr_msg("\t\tWAVE_LFC extraction method = %s",
              WAVE_LFC_param->extraction_method);
    espdr_msg("\t\tWAVE_LFC extraction ksigma = %.2f",
              WAVE_LFC_param->ksig_extraction);
    espdr_msg("\t\tWAVE_LFC wavelnegth solution fit polynomial degree = %d",
              WAVE_LFC_param->wave_sol_poly_deg);
    
	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init WAVE_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_WAVE_LFC_param *espdr_WAVE_LFC_param_init(const char *recipe_id,
                                                cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_WAVE_LFC_param *WAVE_LFC_param =
                (espdr_WAVE_LFC_param *)cpl_malloc(sizeof(espdr_WAVE_LFC_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_WAVE_LFC_get(recipe_id, parameters, WAVE_LFC_param);
    my_error = espdr_parameters_WAVE_LFC_print(WAVE_LFC_param);
    if (my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return WAVE_LFC_param;
    }
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief    Read the wave recipe parameters
 @param         parameters   recipe input parameters
 @param         frameset     recipe input frames
 @param         recipe_id    recipe name
 @param[out]    CCD_geom     CCD geometry structure
 @param[out]    inst_config  instrument config
 @param[out]    OVSC_param   overscan parameter structure
 @param[out]    WAVE_param   WAVE parameter structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_LFC_read_parameters(cpl_frameset *frameset,
                                              espdr_CCD_geometry *CCD_geom,
                                              espdr_inst_config *inst_config) {
	
	cpl_frameset *CCD_geom_frames = cpl_frameset_new();
	cpl_frame *CCD_geom_table_frame = NULL;
	const char *CCD_geom_filename=NULL;
    cpl_frameset *inst_config_frames = cpl_frameset_new();
    cpl_frame *inst_config_table_frame = NULL;
    const char *inst_config_filename = NULL;
	cpl_error_code my_error;
	
	/* Extract and read the CCD geometry FITS table */
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_CCD_GEOM,
                                          CCD_geom_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, 
			   "CCD_GEOM parameters frame extraction failed");
	
	espdr_ensure(CCD_geom_frames == NULL, CPL_ERROR_NULL_INPUT, 
			   "No CCD geometry parameters table %s, exiting", 
			   cpl_error_get_message());
	
    /* Extract and read the instrument configuration FITS table */
    my_error = espdr_frame_extract_by_tag(frameset, ESPDR_INST_CONFIG,
                                          inst_config_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Instrument parameters frame extraction failed");
    
    espdr_ensure(inst_config_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "No instrument parameters table %s, exiting",
                 cpl_error_get_message());
    
    /* Read the CCD geometry configuration file provided in SOF */
    CCD_geom_table_frame = cpl_frameset_get_position(CCD_geom_frames, 0);
    CCD_geom_filename = cpl_frame_get_filename(CCD_geom_table_frame);
    my_error = espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting CCD geometry: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Read the instrument configuration file provided in SOF */
    inst_config_table_frame = cpl_frameset_get_position(inst_config_frames, 0);
    inst_config_filename = cpl_frame_get_filename(inst_config_table_frame);
    my_error = espdr_read_inst_parameters(inst_config_filename,
                                          CCD_geom->ext_nb,
                                          inst_config);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting instrument configuration: %s",
                 cpl_error_get_message_default(my_error));
    
	cpl_frameset_delete(CCD_geom_frames);
    cpl_frameset_delete(inst_config_frames);
	
	return cpl_error_get_code();
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief Find spectral lines for FP or LC
 @param         s2d_data        input S2D image for 1 fiber
 @param         s2d_error       input S2D image errors for 1 fiber
 @param         s2d_quality     input S2D image quality flags for 1 fiber
 @param         wave_matrix     static matrix with lines wavelengths
                                (optional, not used yet)
 @param         flux_threshold  threshold on the flux to detect the spectral pic
 @param         noise_threshold threshold on the noise to detect the spectral pic
 @param[out]    lines_table     computed detected lines table
 @param[out]    lines_nb_RE     computer number of detected lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_search_LFC_lines(cpl_image *s2d_data,
                                      cpl_image *s2d_error,
                                      cpl_image *s2d_quality,
                                      cpl_table *wave_matrix,
                                      int CCD_border_cut,
                                      double flux_threshold,
                                      double noise_threshold,
                                      espdr_line_param *lines_table,
                                      int *lines_nb_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i, order, pxl, pxl_min, pxl_max;
    int index, index_p_data;
    int orders_nb, order_length;
    cpl_image *order_image;
    double *order_data = NULL;
    cpl_image *error_image;
    double *error_data = NULL;
    cpl_image *quality_image;
    int *quality_data = NULL;
    double *deriv1 = NULL;
    double *deriv2 = NULL;
    double rms, mean;
    espdr_poly_data *p_data = NULL;
    double *fit = NULL;
    double *coeffs = NULL;
    double *coeffs_err = NULL;
    double max_flux;
    int max_flux_pxl;
    double fitted_pic_dist;
    double chisq = 0.0;
   
    if(wave_matrix == NULL) {
	   /* ASE : Do nothing Just to document this var is not used */
    }

    espdr_ensure(s2d_data == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D image is NULL");
    espdr_ensure(s2d_error == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D error image is NULL");
    espdr_ensure(s2d_quality == NULL, CPL_ERROR_NULL_INPUT,
                 "The input S2D quality image is NULL");
    
    //int size_x = cpl_image_get_size_x(s2d_quality);
    //int size_y = cpl_image_get_size_y(s2d_quality);
    //espdr_msg("s2d_q is of size %d x %d", size_x, size_y);
    
    orders_nb = cpl_image_get_size_y(s2d_data);
    order_length = cpl_image_get_size_x(s2d_data);
    
    p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    p_data->x = (double *) cpl_calloc (5, sizeof (double));
    p_data->y = (double *) cpl_calloc (5, sizeof (double));
    p_data->err = (double *) cpl_calloc (5, sizeof (double));
    p_data->n = 5;
    
    fit = (double *) cpl_calloc (5, sizeof(double));
    coeffs = (double *) cpl_calloc (3, sizeof(double));
    coeffs_err = (double *) cpl_calloc (3, sizeof(double));
    
    deriv1 = (double *)cpl_calloc(order_length, sizeof(double));
    deriv2 = (double *)cpl_calloc(order_length, sizeof(double));
    
    espdr_msg_debug("Start espdrp_search_lines, flux threshold: %lf, noise threshold: %lf",
              flux_threshold, noise_threshold);
    
    index = 0;
    for (order = 0; order < orders_nb; order++) {
        espdr_msg_debug("Searching lines for order %d", order+1);
        order_image = cpl_image_extract(s2d_data, 1, order+1,
                                        order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_data");
        order_data = cpl_image_get_data_double(order_image);
        error_image = cpl_image_extract(s2d_error, 1, order+1,
                                        order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_error");
        error_data = cpl_image_get_data_double(error_image);
        quality_image = cpl_image_extract(s2d_quality, 1, order+1,
                                          order_length, order+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_extract failed for s2d_quality");
        quality_data = cpl_image_get_data_int(quality_image);
        
        pxl_min = 0;
        if (quality_data == NULL) espdr_msg("Q_data is NULL");
        while ((quality_data[pxl_min] == OTHER_BAD_PIXEL) &&
               (pxl_min < order_length)) {
            pxl_min++;
        }
        espdr_ensure(pxl_min == order_length, CPL_ERROR_INCOMPATIBLE_INPUT,
                     "The whole order is not valid, exiting");
        
        pxl_max = order_length-1;
        while ((quality_data[pxl_max] == OTHER_BAD_PIXEL) &&
               (pxl_max >= 0)) {
            pxl_max--;
        }
        espdr_ensure(pxl_max < 0, CPL_ERROR_INCOMPATIBLE_INPUT,
                     "The whole order is not valid, exiting");
        
        for (pxl = 0; pxl < order_length; pxl++) {
            deriv1[pxl] = 0.0;
            deriv2[pxl] = 0.0;
        }
        
        for (pxl = pxl_min+CCD_border_cut-2;
             pxl < pxl_max-CCD_border_cut+2; pxl++) {
            deriv1[pxl] = (order_data[pxl-1]-order_data[pxl+1])/2.0;
        }
        for (pxl = pxl_min+CCD_border_cut-1;
             pxl < pxl_max-CCD_border_cut+1; pxl++) {
            deriv2[pxl] = (deriv1[pxl-1]-deriv1[pxl+1])/2.0;
        }
        
        for (pxl = pxl_min+CCD_border_cut; pxl < pxl_max-CCD_border_cut; pxl++) {
            rms = cpl_image_get_stdev_window(order_image, pxl-1, 1, pxl+3, 1);
            mean = cpl_image_get_mean_window(order_image, pxl-1, 1, pxl+3, 1);
            //espdr_msg("data[%d] = %lf, rms = %lf, mean = %lf",
            //          pxl, order_data[pxl], rms, mean);
            
            if ((deriv1[pxl] * deriv1[pxl+1] < 0.0) && (deriv2[pxl] < 0.0) &&
                ((rms > noise_threshold * error_data[pxl]) ||
                (mean > flux_threshold))) {
                
                max_flux = 0.0;
                for (i = pxl; i < pxl+2; i++) {
                    if (order_data[i-1] < order_data[i]) {
                        max_flux = order_data[i];
                        max_flux_pxl = i;
                    }
                }
                /* fit poly */
                index_p_data = 0;
                for (i = pxl-2; i < pxl+3; i++) {
                    p_data->x[index_p_data] = i;
                    p_data->y[index_p_data] = order_data[i];
                    p_data->err[index_p_data] = error_data[i];
                    index_p_data++;
                }
                
                my_error = espdr_fit_poly(p_data, 3,
                                          fit, coeffs, coeffs_err, &chisq, 0);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_fit_poly failed: %s",
                             cpl_error_get_message_default(my_error));
                
                fitted_pic_dist = -coeffs[1] / (2.0 * coeffs[2]) - max_flux_pxl;
                if ((fitted_pic_dist < 1.5) && (fitted_pic_dist > -1.5)) {
                    lines_table[index].peak_pxl = -coeffs[1] / (2.0 * coeffs[2]) +1;
                } else {
                    lines_table[index].peak_pxl = max_flux_pxl+1;
                }
                lines_table[index].peak_flux = order_data[max_flux_pxl];
                lines_table[index].order = order+1;
                
                index++;
            }
        }
        
        cpl_image_delete(order_image);
        cpl_image_delete(error_image);
        cpl_image_delete(quality_image);
    }
    
    cpl_free(p_data->x);
    cpl_free(p_data->y);
    cpl_free(p_data->err);
    cpl_free(p_data);
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    cpl_free(deriv1);
    cpl_free(deriv2);
    
    *lines_nb_RE = index;
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Assign the LC wavelengths to pixels
 @param[out]    lines_table         detected lines peaks
 @param         wave_matrix         THAR wevelengths matrix, 
                                    optional, not used yet
 @param         repetition_rate     LFC frequency
 @param         anchor_frequency    initial LFC frequency
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_assign_comb_wavelengths(espdr_line_param *lines_table,
                                             double **wave_matrix,
                                             double repetition_rate,
                                             double anchor_frequency) {
    
    int index, order;
    double N, pxl, lambda_peak;
    
    /* To be able to compute the N analytically, we need to have 
     all the wavelengths in vacuum. We use n_air(lambda_air) instead 
     of n_air(lambda_vacuum) to compute the lambda_vacuum, since these 
     two n_air have values close enough for our purpose */
    
    index = 0;
    while ((lines_table[index].order > 0) && (index < MAX_NB_SPECTRAL_LINES)) {
    //while ((lines_table[index].order_nb > 0) && (index < 10)) {
        order = lines_table[index].order - 1;
        pxl = lines_table[index].peak_pxl;
        //espdr_msg("order_nb: %d, pxl: %lf, (int)pxl: %d", order, pxl, (int)pxl);
        if (pxl == (int)pxl) {
            lambda_peak = wave_matrix[order][(int)pxl-1];
        } else {
            lambda_peak = wave_matrix[order][(int)pxl-1] +
                (wave_matrix[order][(int)pxl] - wave_matrix[order][(int)pxl-1]) *
                (pxl - (int)pxl);
        }
        /* take the lambda in vacuum */
        /*
        lambda_peak = lambda_peak * espdr_compute_n_air(lambda_peak,
                                                        N_AIR_COEFF_T,
                                                        N_AIR_COEFF_P);
        
        lines_table[index].lambda_peak = lambda_peak /
                                espdr_compute_n_air(lambda_peak,
                                                    N_AIR_COEFF_T,
                                                    N_AIR_COEFF_P);
        */
        
        N = (10.0/repetition_rate) *
                        (LIGHT_SPEED / lambda_peak - 100.0 * anchor_frequency);
        
        lines_table[index].Nreal = N;
        
        //N = (int)(N + 0.5);
        N = round(N);
        
        lines_table[index].wavelength = LIGHT_SPEED /
                        (anchor_frequency * 100.0 + N * repetition_rate/10.0);
        
        /*
        lines_table[index].wavelength = lines_table[index].wavelength /
                            espdr_compute_n_air(lines_table[index].wavelength,
                                                N_AIR_COEFF_T, N_AIR_COEFF_P);
        */
        
        lines_table[index].N = N;
        
        strcpy(lines_table[index].element_name, "LFC");
        
        index++;
    }
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Fit calibration lines
 @param         lines_table         detected lines peaks
 @param[out]    lines_table_final   cleaned lines table
 @param         spectral_line_qc    tolerance on the lines accuracy
 @param         fibre_source        fibre calibration source (THAR, FP or LFC)
 @param[out]    lines_nb_RE         returned number of clean lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_clean_LFC_lines_table(espdr_line_param *lines_table,
                                           espdr_line_param *lines_table_final,
                                           double spectral_line_qc,
                                           espdr_src_type fibre_source,
                                           int *lines_nb_RE) {
    
    int line, line_index;
    
    line = 0;
    line_index = 0;
    while ((lines_table[line].order > 0) && (line < MAX_NB_SPECTRAL_LINES)) {
        //espdr_msg("line: %d, line_index: %d, order: %d, Nreal: %f, N: %d, diff: %f",
        //          line, line_index, lines_table[line].order,
        //          lines_table[line].Nreal, lines_table[line].N,
        //          fabs(lines_table[line].Nreal - lines_table[line].N));
        if (fabs(lines_table[line].Nreal - lines_table[line].N) < spectral_line_qc) {
            lines_table_final[line_index].order = lines_table[line].order;
            lines_table_final[line_index].peak_pxl = lines_table[line].peak_pxl;
            lines_table_final[line_index].peak_pxl_err =
                                                lines_table[line].peak_pxl_err;
            lines_table_final[line_index].fwhm = lines_table[line].fwhm;
            lines_table_final[line_index].fwhm_err = lines_table[line].fwhm_err;
            lines_table_final[line_index].peak_flux =
                                                lines_table[line].peak_flux;
            lines_table_final[line_index].peak_flux_err =
                                            lines_table[line].peak_flux_err;
            lines_table_final[line_index].qc_flag = lines_table[line].qc_flag;
            lines_table_final[line_index].wavelength =
                                                lines_table[line].wavelength;
            lines_table_final[line_index].lambda_peak =
                                                lines_table[line].lambda_peak;
            lines_table_final[line_index].Nreal = lines_table[line].Nreal;
            lines_table_final[line_index].N = lines_table[line].N;
            lines_table_final[line_index].dispersion =
                                                lines_table[line].dispersion;
            lines_table_final[line_index].grouping = lines_table[line].grouping;
            strcpy(lines_table_final[line_index].element_name,
                   lines_table[line].element_name);
            if (fibre_source == FP) {
                strcpy(lines_table_final[line_index].element_name, "FP");
            }
            if (fibre_source == LFC) {
                strcpy(lines_table_final[line_index].element_name, "LFC");
            }
            line_index++;
        }
        line++;
    }
    
    *lines_nb_RE = line_index;
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief QC on lines
 @param         lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         inst_config             instrument config
 @param[out]    lines_nb_per_order_RE   number of valid lines per order
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_LFC_lines_qc(espdr_line_param *lines_table,
                                  int lines_nb,
                                  espdr_inst_config *inst_config,
                                  int *lines_nb_per_order_RE) {
    
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "The input table is NULL");
    int i, j, k, group, group_QC;
    int test_order = -1;
    
    
    for (i = 0; i < lines_nb; i++) {
        
        if (lines_table[i].order == test_order) {
            espdr_msg("order %d, pxl %lf, ll: %lf, before check qc = %d",
                      lines_table[i].order, lines_table[i].peak_pxl,
                      lines_table[i].wavelength, lines_table[i].qc_flag);
        }

        // Check if the fit produced any nans or infs
        if (isnan(lines_table[i].peak_pxl) || isinf(lines_table[i].peak_pxl) ||
            isnan(lines_table[i].peak_pxl_err) || isinf(lines_table[i].peak_pxl_err) ||
            isnan(lines_table[i].fwhm) || isinf(lines_table[i].fwhm) ||
            isnan(lines_table[i].fwhm_err) || isinf(lines_table[i].fwhm_err) ||
            isnan(lines_table[i].peak_flux) || isinf(lines_table[i].peak_flux) ||
            isnan(lines_table[i].peak_flux_err) || isinf(lines_table[i].peak_flux_err)) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, nans or infs",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        
        //QC_flag_orig = lines_table[i].qc_flag;
        if (lines_table[i].peak_pxl_err > inst_config->line_max_sig_x0) {
        //if (lines_table[i].peak_pxl_err > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, peak pxl error too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_pxl_err);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        if (lines_table[i].fwhm > inst_config->line_max_fwhm) {
        //if ((lines_table[i].fwhm > 8) || (lines_table[i].fwhm < 3)) {
        //if (lines_table[i].fwhm > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].fwhm);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        if (lines_table[i].peak_flux < inst_config->line_min_flux_el) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, flux too small (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_flux);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        // The following test is added,
        // because the fit to compute the resolution was failing
        if (lines_table[i].fwhm_err == 0.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm error == 0.0",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
    }
    
    /* Take off the whole group which has the line with qc_flag == 0 */
    i = 0;
    while (i < lines_nb) {
        j = i;
        group = lines_table[j].grouping;
        group_QC = 1;
        while ((j < lines_nb) && (lines_table[j].grouping == group)) {
            if (lines_table[j].qc_flag == 0) {
                group_QC = 0;
                if (lines_table[i].order == test_order) {
                    espdr_msg("order %d, pxl %lf, ll: %lf, one of the group qc = 0",
                              lines_table[i].order, lines_table[i].peak_pxl,
                              lines_table[i].wavelength);
                }
            }
            j++;
        }
        if (group_QC == 0) {
            for (k = i; k < j; k++) {
                if (lines_table[i].order == test_order) {
                    espdr_msg("order %d, pxl %lf, ll: %lf, group = 0",
                              lines_table[i].order, lines_table[i].peak_pxl,
                              lines_table[i].wavelength);
                }
                lines_table[k].qc_flag = 0;
                lines_table[i].fwhm = -1.0;
                lines_table[i].fwhm_err = -1.0;
                lines_table[i].peak_flux = -1.0;
                lines_table[i].peak_flux_err = -1.0;
                lines_table[i].dispersion = -1.0;
                lines_nb_per_order_RE[lines_table[k].order-1]--;
            }
        }
        i = j;
    }
    
    for (i = 0; i < lines_nb; i++) {
        // Lines was not fitted because of close to the border
        // or too small flux difference between max_flux and min_flux
        // But the rest of the group is OK
        if (lines_table[i].qc_flag == 2) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, qc = 2",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
    }
    
    return(cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief Merge the LFC wave solution with the input THAR one 
        for non-existing LFC orders
 
 @param         lines_nb_per_order  number of valid lines for each order
 @param         thar_rejected_lines number of rejected lines
 @param         flux ratio          flux ratio
 @param         global_shift        global shift
 @param         chisq_per_order     reduced CHI2 of ll solution per order
 @param         rms_per_order       dispersion of residuals around fit per order
 @param         res_per_order       spectral resolution per order
 @param         lamp_offset         ThAr lamp offset
 @param         orders_nb           number of orders
 @param         fibre_source        calibration source type
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_merge_LFC_THAR(cpl_image *LFC_wave_matrix,
                                    cpl_image *THAR_wave_matrix,
                                    int *LFC_fitted_orders,
                                    cpl_image **wave_matrix_RE,
                                    cpl_image **air_wave_matrix_RE) {
    
    int i = 0, j = 0, index = 0;
    //cpl_error_code my_error = CPL_ERROR_NONE;
    double *final_wave_data = NULL;
    double *final_air_wave_data = NULL;
    int size_x, size_y;
    cpl_image *wave_order_image = NULL;
    double *wave_order_data = NULL;
    cpl_image *final_wave_matrix = NULL;
    cpl_image *final_air_wave_matrix = NULL;
    double ll;
    
    size_x = cpl_image_get_size_x(THAR_wave_matrix);
    size_y = cpl_image_get_size_y(THAR_wave_matrix);
    final_wave_data = (double *) cpl_calloc (size_x * size_y, sizeof(double));
    final_air_wave_data = (double *) cpl_calloc (size_x * size_y, sizeof(double));
    
    index = 0;
    for (i = 0; i < size_y; i++) {
        if (LFC_fitted_orders[i] == 1) {
            // take order from LFC wave matrix
            wave_order_image = cpl_image_extract(LFC_wave_matrix,
                                                 1, i+1, size_x, i+1);
            //espdr_msg("Taking LFC order %d, mean: %f",
            //          i+1, cpl_image_get_mean(wave_order_image));
        } else {
            // take order from THAR wave matrix
            wave_order_image = cpl_image_extract(THAR_wave_matrix,
                                                 1, i+1, size_x, i+1);
            //espdr_msg("Taking THAR order %d, mean: %f",
            //          i+1, cpl_image_get_mean(wave_order_image));
        }
        wave_order_data = cpl_image_get_data_double(wave_order_image);
        for (j = 0; j < size_x; j++) {
            ll = wave_order_data[j];
            final_wave_data[index] = ll;
            final_air_wave_data[index] = ll / espdr_compute_n_air(ll,
                                                            N_AIR_COEFF_T,
                                                            N_AIR_COEFF_P);
            index++;
        }
        //espdr_msg("final_wave_data = %f", final_wave_data[index-4600]);
        cpl_image_delete(wave_order_image);
    }
    
    final_wave_matrix = cpl_image_wrap_double(size_x, size_y, final_wave_data);
    *wave_matrix_RE = cpl_image_duplicate(final_wave_matrix);
    
    final_air_wave_matrix = cpl_image_wrap_double(size_x, size_y, final_air_wave_data);
    *air_wave_matrix_RE = cpl_image_duplicate(final_air_wave_matrix);
    
    cpl_image_unwrap(final_wave_matrix);
    cpl_free(final_wave_data);
    cpl_image_unwrap(final_air_wave_matrix);
    cpl_free(final_air_wave_data);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs
 @param         chisq_per_order     reduced CHI2 of ll solution per order
 @param         rms_per_order       dispersion of residuals around fit per order
 @param         res_per_order       spectral resolution per order
 @param         lamp_offset         ThAr lamp offset
 @param         orders_nb           number of orders
 @param         fibre_source        calibration source type
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_LFC_QC(int *lines_nb_per_order,
                                 double *chisq_per_order,
                                 double *rms_per_order,
                                 int *order_fit,
                                 int orders_nb,
                                 int fibre_nr,
                                 espdr_inst_config *inst_config,
                                 espdr_qc_keywords *qc_kws,
                                 cpl_propertylist *wave_keywords,
                                 cpl_propertylist *keywords_RE) {
    
    int i;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int lines_nb_per_order_QC = 1;
    int total_lines_nb_QC = 1;
    int chi2_QC = 1;
    int rms_QC = 1;
    int global_QC = 1;
    int previous_global_QC = 1;
    cpl_error_code my_error = CPL_ERROR_NONE;
    double chi2_to_be_set, rms_to_be_set;
    int total_lfc_lines_nb = 0;
    
    int min_lfc_lines_total = inst_config->lfc_min_lines_total;
    int min_lfc_lines_per_order = inst_config->lfc_sol_min_lines_per_order[fibre_nr];
    double lfc_sol_max_chi2 = inst_config->lfc_sol_max_chi2;
    double lfc_sol_max_rms = inst_config->lfc_sol_max_rms;

    /* Added keywords checks + change to -1.0 */
    for (i = 0; i < orders_nb; i++) {
        if (isnan(chisq_per_order[i]) || isinf(chisq_per_order[i])) {
            chisq_per_order[i] = -1.0;
        }
        if (isnan(rms_per_order[i]) || isinf(rms_per_order[i])) {
            rms_per_order[i] = -1.0;
        }
    }
    
    /* QC KWs per order */
    for (i = 0; i < orders_nb; i++) {
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_lfc_order_lines_nb_kw_first,
                                                 qc_kws->qc_lfc_order_lines_nb_kw_last,
                                                 i+1);
        sprintf(comment, "Number of LFC valid lines in order %d", i+1);
        my_error = espdr_keyword_add_int(new_keyword, lines_nb_per_order[i], comment,
                                         &keywords_RE);
        cpl_free(new_keyword);
        
        if ((order_fit[i] == 1) || (order_fit[i] == -1)) {
            total_lfc_lines_nb = total_lfc_lines_nb + lines_nb_per_order[i];
            if ((lines_nb_per_order[i] >= 0) && (lines_nb_per_order[i] < min_lfc_lines_per_order)) {
                lines_nb_per_order_QC = 0;
                global_QC = 0;
            }
        }
        
        //espdr_msg("Adding CHI2 = %f for order %d", chisq_per_order[i], i+1);
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_order_chisq_kw_first,
                                                 qc_kws->qc_wave_order_chisq_kw_last,
                                                 i+1);
        if (order_fit[i] == 1) {
            chi2_to_be_set = chisq_per_order[i];
            sprintf(comment, "Reduced CHI2 of LFC ll sol for order %d", i+1);
        } else {
            if (wave_keywords != NULL) {
                if (cpl_propertylist_has(wave_keywords, new_keyword)) {
                    chi2_to_be_set = cpl_propertylist_get_double(wave_keywords, new_keyword);
                    sprintf(comment, "Reduced CHI2 of THAR ll sol for order %d", i+1);
                } else {
                    chi2_to_be_set = -1.0;
                    sprintf(comment, "No CHI2 of THAR ll sol for order %d", i+1);
                }
            } else {
                chi2_to_be_set = -1.0;
            }
        }
        
        my_error = espdr_keyword_add_double(new_keyword, chi2_to_be_set, comment, &keywords_RE);
        cpl_free(new_keyword);
        
        if (order_fit[i] == 1) {
            if (chisq_per_order[i] > lfc_sol_max_chi2) {
                chi2_QC = 0;
                global_QC = 0;
            }
        }
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_order_rms_kw_first,
                                                 qc_kws->qc_wave_order_rms_kw_last,
                                                 i+1);
        if (order_fit[i] == 1) {
            rms_to_be_set = rms_per_order[i];
            sprintf(comment, "Disp of resid. around LFC fit for order %d", i+1);
        } else {
            if (wave_keywords != NULL) {
                if (cpl_propertylist_has(wave_keywords, new_keyword)) {
                    rms_to_be_set = cpl_propertylist_get_double(wave_keywords, new_keyword);
                    sprintf(comment, "Disp. of resid. around THAR fit for order %d", i+1);
                } else {
                    rms_to_be_set = -1.0;
                    sprintf(comment, "No THAR fit RMS for order %d", i+1);
                }
            } else {
                rms_to_be_set = -1.0;
            }
        }
        my_error = espdr_keyword_add_double(new_keyword, rms_to_be_set, comment, &keywords_RE);
        cpl_free(new_keyword);
        
        if (order_fit[i] == 1) {
            if (rms_per_order[i] > lfc_sol_max_rms) {
                rms_QC = 0;
                global_QC = 0;
            }
        }
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_lfc_order_fit_wave_kw_first,
                                                 qc_kws->qc_lfc_order_fit_wave_kw_last,
                                                 i+1);
        if (order_fit[i] == -1) {
            sprintf(comment, "Order %d fitted on the other fiber", i+1);
        } else {
            if (order_fit[i] == 1) {
                sprintf(comment, "Order %d fitted with LFC", i+1);
            } else {
                sprintf(comment, "Order %d fitted with THAR", i+1);
            }
        }
        my_error = espdr_keyword_add_int(new_keyword, order_fit[i], comment, &keywords_RE);
        cpl_free(new_keyword);
    }
    
    if (total_lfc_lines_nb < 0) total_lfc_lines_nb = -1;
    /* Global QC KWs */
    sprintf(comment, "Total number of LFC valid lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_lines_nb_kw, total_lfc_lines_nb,
                                     comment, &keywords_RE);
        
    /* QC CHECK KWs */
    sprintf(comment, "QC on min number of LFC lines/order");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_order_lines_min_check_kw,
                                     lines_nb_per_order_QC, comment, &keywords_RE);
    
    if ((total_lfc_lines_nb >= 0) && (total_lfc_lines_nb < min_lfc_lines_total)) {
        total_lines_nb_QC = 0;
        global_QC = 0;
    }
    
    sprintf(comment, "QC on total number of detected LFC lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_lines_tot_check_kw,
                                     total_lines_nb_QC, comment, &keywords_RE);
    
    sprintf(comment, "QC on wavelength solutions (CHI2)");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_chisq_check_kw,
                                     chi2_QC, comment, &keywords_RE);
    
    sprintf(comment, "QC on wavelength solutions (RMS)");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_rms_check_kw,
                                     rms_QC, comment, &keywords_RE);
    
    previous_global_QC = cpl_propertylist_get_int(keywords_RE, qc_kws->qc_lfc_check_kw);
    global_QC = global_QC & previous_global_QC;
    
    sprintf(comment, "Overall LFC QC");
    my_error = espdr_keyword_add_int(qc_kws->qc_lfc_check_kw, global_QC, comment,
                                     &keywords_RE);
    
    return(cpl_error_get_code());
}

