/* $Id: espdr_dfs-test.c,v 1.2 2013-07-26 11:44:44 amodigli Exp $
 *
 * This file is part of the ESPDR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: dsosnows $
 * $Date: 2013-07-26 11:44:44 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>

#include <espdr_msg.h>
#include <espdr_parameters.h>
#include <espdr_bias.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_led_ff  Unit test of espdr_flat
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test for master bias creation (espdr_bias.c)
 */
/*----------------------------------------------------------------------------*/

void
espdr_ccd_geom_init(int ext_nb, int j, espdr_CCD_geometry* CCD_geom)
{
    espdr_msg("Filling up the CCD geom structure");
    //strcpy(CCD_geom->instrument, "TEST");
    sprintf(CCD_geom->instrument, "HARPS");
    CCD_geom->ext_nb = ext_nb;
    CCD_geom->total_output_nb = 1;
    CCD_geom->exts = (espdr_CCD_extension*) cpl_malloc(
                    ext_nb * sizeof(espdr_CCD_extension));
    CCD_geom->exts->CCD_nx = 5;
    CCD_geom->exts->CCD_ny = 5;
    CCD_geom->exts->out_nb_x = 1;
    CCD_geom->exts->out_nb_y = 1;
    //CCD_geom->prescan_x = 0;
    //CCD_geom->prescan_y = 0;
    //CCD_geom->overscan_x = 0;
    //CCD_geom->overscan_y = 0;
    //CCD_geom->real_left = 0;
    //CCD_geom->real_right = 5;
    //CCD_geom->real_lower = 0;
    //CCD_geom->real_upper = 5;
    CCD_geom->exts[0].coll_or = 0;
    CCD_geom->exts[0].corr_type = 0;
    CCD_geom->exts[0].rot_angle = 0;
    CCD_geom->exts[0].ver_flip = 0;
    CCD_geom->exts[0].hor_flip = 0;
    CCD_geom->exts[0].pxl_size = 15;
    int out_nb_x = 1;
    int out_nb_y = 1;
    CCD_geom->exts[0].outputs = (espdr_CCD_output**) cpl_malloc(
                    out_nb_x * sizeof(espdr_CCD_output*));
    for (j = 0; j < out_nb_x; j++) {
        CCD_geom->exts[0].outputs[j] = (espdr_CCD_output*) cpl_malloc(
                        out_nb_y * sizeof(espdr_CCD_output));
    }
    CCD_geom->exts[0].outputs[0][0].real_llx = 1;
    CCD_geom->exts[0].outputs[0][0].real_lly = 1;
    CCD_geom->exts[0].outputs[0][0].real_urx = 5;
    CCD_geom->exts[0].outputs[0][0].real_ury = 5;
    CCD_geom->exts[0].outputs[0][0].oscan_llx = 0;
    CCD_geom->exts[0].outputs[0][0].oscan_lly = 0;
    CCD_geom->exts[0].outputs[0][0].oscan_urx = 0;
    CCD_geom->exts[0].outputs[0][0].oscan_ury = 0;
    CCD_geom->exts[0].outputs[0][0].poscan_llx = 0;
    CCD_geom->exts[0].outputs[0][0].poscan_lly = 0;
    CCD_geom->exts[0].outputs[0][0].poscan_urx = 0;
    CCD_geom->exts[0].outputs[0][0].poscan_ury = 0;
}


static cpl_error_code create_CCD_geom(espdr_CCD_geometry* CCD_geom)
{

    int image_size = 5;
    int ext_nb = 1;
    int out_nb_x = 1;
    int out_nb_y = 1;
    int output_nb=1;
    int i,k;

    /* NOTE: CCD_geom structure should be allocate outside */
    strcpy(CCD_geom->instrument, "TEST");

    CCD_geom->exts = (espdr_CCD_extension *) cpl_malloc
                    (ext_nb * sizeof(espdr_CCD_extension));
    CCD_geom->total_output_nb = output_nb;
    CCD_geom->ext_nb = ext_nb;

    for(k=0;k<ext_nb;k++) {
        CCD_geom->exts[k].CCD_nx = image_size;
        CCD_geom->exts[k].CCD_ny = image_size;

        CCD_geom->exts[k].coll_or = 1;
        CCD_geom->exts[k].corr_type = 0;
        CCD_geom->exts[k].rot_angle = 0;
        CCD_geom->exts[k].ver_flip = 0;
        CCD_geom->exts[k].hor_flip = 0;
        CCD_geom->exts[k].pxl_size = 1; /* dummy for unit test */

        CCD_geom->exts[k].ext_no = ext_nb;
        CCD_geom->exts[k].out_nb_x = out_nb_x;
        CCD_geom->exts[k].out_nb_y = out_nb_y;

        CCD_geom->exts[k].image_nx = image_size;
        CCD_geom->exts[k].image_ny = image_size;

        CCD_geom->exts[k].outputs =
                        (espdr_CCD_output **) cpl_malloc(out_nb_x *
                                        sizeof(espdr_CCD_output *));
        for(i=0;i<out_nb_x;i++) {
            CCD_geom->exts[k].outputs[i] =
                            (espdr_CCD_output *) cpl_malloc(out_nb_y *
                                            sizeof(espdr_CCD_output));
        }
    }

    for(k=0;k<ext_nb;k++) {
        for(i=0;i<out_nb_x;i++) {

            CCD_geom->exts[k].outputs[i]->pscan_llx = 0;
            CCD_geom->exts[k].outputs[i]->pscan_lly = 0;


            CCD_geom->exts[k].outputs[i]->oscan_llx = 0;
            CCD_geom->exts[k].outputs[i]->oscan_lly = 0;


            CCD_geom->exts[k].outputs[i]->real_llx = 0;
            CCD_geom->exts[k].outputs[i]->real_urx = image_size;
            CCD_geom->exts[k].outputs[i]->real_lly = 0;
            CCD_geom->exts[k].outputs[i]->real_ury = image_size;

        }

    }
    return (cpl_error_get_code());

}
cpl_error_code
test_mbias_noise(void)
{
    int nima=5;
    int sx=1000;
    int sy=1000;
    int i=0;
    cpl_imagelist* iml=NULL;
    cpl_image* ima=NULL;
    cpl_image* master_bias=NULL;
    double RON;
    int ncrh;
    double bias_value=100;
    double bias_rms=3;
    double bias_min=bias_value-bias_rms;
    double bias_max=bias_value+bias_rms;

    int out_x=0;
    int out_y=0;
    int ext_no=0;

    iml=cpl_imagelist_new();
    for(i=0;i<nima;i++) {

        ima=cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(ima,bias_min,bias_max);
        cpl_imagelist_set(iml,cpl_image_duplicate(ima),(cpl_size)i);
        cpl_image_delete(ima);
    }

    //cpl_imagelist_save(iml,"cube.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    espdr_CCD_geometry *CCD_geom =
                          (espdr_CCD_geometry *)cpl_malloc(sizeof(espdr_CCD_geometry));
    create_CCD_geom(CCD_geom);
    espdr_inst_config *inst_config=NULL;
    double mean, median, rms;
    espdr_master_bias_one_output(iml,CCD_geom,inst_config,sx,sy,ext_no,out_x,out_y,
                                 &master_bias,&RON,&mean,&ncrh);

    //cpl_image_save(master_bias,"master.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    mean=cpl_image_get_mean(master_bias);
    median=cpl_image_get_median(master_bias);
    rms=cpl_image_get_stdev(master_bias);
    espdr_msg_debug("mean=%g",mean);
    espdr_msg_debug("median=%g",median);
    espdr_msg_debug("rms=%g",rms);

    cpl_test_abs( mean, bias_value, 0.0015);
    cpl_test_abs( median, bias_value, 0.0015);


    cpl_imagelist_delete(iml);
    cpl_image_delete(master_bias);

    return cpl_error_get_code();
}


cpl_error_code
test_mbias_uniform_plus_one_outlier(void)
{
    int nima=10;
    int sx=1000;
    int sy=1000;
    int i=0;
    cpl_imagelist* iml=NULL;
    cpl_image* ima=NULL;
    cpl_image* master_bias=NULL;
    double RON;
    int ncrh;
    double bias_value=100;
    int out_x=0;
    int out_y=0;
    int ext_no=0;

    iml=cpl_imagelist_new();
    for(i=0;i<nima;i++) {

        ima=cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);
        //cpl_image_fill_noise_uniform(ima,bias_min,bias_max);
        cpl_image_add_scalar(ima,bias_value);
        if(i==0) {
           cpl_image_set(ima,sx/2,sy/2,bias_value+bias_value*1000);
        } else {
           cpl_image_set(ima,sx/2,sy/2,bias_value+i);
        }
        cpl_imagelist_set(iml,cpl_image_duplicate(ima),(cpl_size)i);
        cpl_image_delete(ima);
    }

    //cpl_imagelist_save(iml,"cube.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    espdr_CCD_geometry *CCD_geom =
                          (espdr_CCD_geometry *)cpl_malloc(sizeof(espdr_CCD_geometry));
    create_CCD_geom(CCD_geom);
    espdr_inst_config *inst_config=NULL;
    double mean, median, rms;
    espdr_master_bias_one_output(iml,CCD_geom,inst_config,sx,sy,ext_no,out_x,out_y,
                                 &master_bias,&RON,&mean,&ncrh);

    //cpl_image_save(master_bias,"master.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    mean=cpl_image_get_mean(master_bias);
    median=cpl_image_get_median(master_bias);
    rms=cpl_image_get_stdev(master_bias);
    espdr_msg_debug("mean=%g",mean);
    espdr_msg_debug("median=%g",median);
    espdr_msg_debug("rms=%g",rms);

    cpl_test_abs( mean, bias_value, 0.001);
    cpl_test_abs( median, bias_value, 0.001);
    //cpl_test_abs( rms, 0, 0.001);

    cpl_imagelist_delete(iml);
    cpl_image_delete(master_bias);

    return cpl_error_get_code();
   
}

cpl_error_code
test_mbias_uniform(void)
{
    int nima=5;
    int sx=1000;
    int sy=1000;
    int i=0;
    cpl_imagelist* iml=NULL;
    cpl_image* ima=NULL;
    cpl_image* master_bias=NULL;
    double RON;
    int ncrh;
    double bias_value=100;

    int out_x=0;
    int out_y=0;
    int ext_no=0;

    iml=cpl_imagelist_new();
    for(i=0;i<nima;i++) {

        ima=cpl_image_new(sx,sy,CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(ima,bias_value);
        cpl_imagelist_set(iml,cpl_image_duplicate(ima),(cpl_size)i);
        cpl_image_delete(ima);
    }
    //cpl_imagelist_save(iml,"cube.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    espdr_CCD_geometry *CCD_geom =
                        (espdr_CCD_geometry *)cpl_malloc(sizeof(espdr_CCD_geometry));
    create_CCD_geom(CCD_geom);


    espdr_inst_config *inst_config=NULL;
    double mean, median, rms;
    espdr_master_bias_one_output(iml,CCD_geom,inst_config,sx,sy,ext_no,out_x,out_y,
                                 &master_bias,&RON,&mean,&ncrh);

    //cpl_image_save(master_bias,"master.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    mean=cpl_image_get_mean(master_bias);
    median=cpl_image_get_median(master_bias);
    rms=cpl_image_get_stdev(master_bias);
    espdr_msg_debug("mean=%g",mean);
    espdr_msg_debug("median=%g",median);
    espdr_msg_debug("rms=%g",rms);

    cpl_test_abs( mean, bias_value, 0.001);
    cpl_test_abs( median, bias_value, 0.001);
    cpl_test_abs( rms, 0, 0.001);

    cpl_imagelist_delete(iml);
    cpl_image_delete(master_bias);

    return cpl_error_get_code();
    
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests for master bias creation (espdr_bias.c module)
 */
/*----------------------------------------------------------------------------*/

int main(void) {
	
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    //test_mbias_uniform();
    //test_mbias_noise();
    //test_mbias_uniform_plus_one_outlier();

    return cpl_test_end(0);
}

/**@}*/
