/*                                                                           a
 *   This file is part of the ESO X-shooter Pipeline                         *
 *   Copyright (C) 2006 European Southern Observatory                        *
 *                                                                           *
 *   This library is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the Free Software             *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA    *
 *                                                                           */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-06 14:49:56 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

#include <espdr_CCD.h>
#include <espdr_parameters.h>
#include <espdr_utils.h>
#include <espdr_msg.h>
#include <espdr_instrument.h>

/*----------------------------------------------------------------------------
 Functions
 ----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
 @brief   verify if a parameter value has been changed (from command line or
          or crc file by a user)
 @param    p    parameter
 @return   1 if parameter has changes, else 0
 */
/*----------------------------------------------------------------------------*/
int espdr_param_has_changed(cpl_parameter* p) {
    cpl_type type = cpl_parameter_get_type(p);
    int has_changed=0;
    /* handle different data types */
    switch (type) {
            
        case CPL_TYPE_INT: {
            int val = cpl_parameter_get_int(p);
            int def = cpl_parameter_get_default_int(p);
            has_changed = (val != def)  ? 1: 0;
        }
            break;
            
        case CPL_TYPE_FLOAT: {
            float val = cpl_parameter_get_double(p);
            float def = cpl_parameter_get_default_double(p);
            has_changed = (val != def)  ? 1: 0;
        }
            break;
            
        case CPL_TYPE_DOUBLE: {
            double val = cpl_parameter_get_double(p);
            double def = cpl_parameter_get_default_double(p);
            has_changed = (val != def)  ? 1: 0;
        }
            break;
            
        case CPL_TYPE_STRING:{
            const char* val = cpl_parameter_get_string(p);
            const char* def = cpl_parameter_get_default_string(p);
            if ( strcmp(val,def) !=0 ) {
                has_changed = 1;
            }
        }
            break;
            
        default:
            espdr_msg_error("case not found! %d string type: %d",type,CPL_TYPE_STRING);
            break;
            
    }
    return has_changed;
}

/*----------------------------------------------------------------------------*/
/**
 @brief  init a parameter by a FITS table
 @param  tab      input FITS table
 @param  col_name table column name
 @param  col_type table column type
 @param  param  output parameter
 @return cpl error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_inst_config_init_by_tab(const cpl_table* tab,
                                                    const char* col_name,
                                                    cpl_type type,
                                                    void* param) {
    /* check input parameters are proper, else exit */
    cpl_ensure(tab != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    
    if(!cpl_table_has_column(tab,col_name)) {
        espdr_msg_debug("missing input table column name %s!",col_name);
        return cpl_error_get_code();
    }
    
    if(type != cpl_table_get_column_type(tab, col_name)) {
        espdr_msg("wrong input table column type %s!",col_name);
        return cpl_error_get_code();
    }
    
    /* handle different data types */
    switch (type) {
            
        case CPL_TYPE_INT: {
            *((int*)param) = cpl_table_get_int(tab,col_name, 0, NULL);
        }
            break;
            
        case CPL_TYPE_FLOAT: {
            *((float*)param) = cpl_table_get_float(tab,col_name, 0, NULL);
        }
            break;
            
        case CPL_TYPE_DOUBLE: {
            *((double*)param) = cpl_table_get_double(tab,col_name, 0, NULL);
        }
            break;
            
        case CPL_TYPE_STRING:{
            const char * str = cpl_table_get_string(tab,col_name, 0);
            if(str) {
                sprintf((char*)param, "%s", str);
                espdr_msg_debug("copy string %s",str);
                espdr_msg_debug("now par is %s",(char *)param);
            }
            
        }
            break;
            
        default:
            espdr_msg_error("case not found! %d string type: %d",type,CPL_TYPE_STRING);
            break;
            
    }
    espdr_check_error_code("espdr_inst_config_init_by_tab");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief  init an (inst_config) parameter by a recipe configuration parameter
 @param  params  input parameterlist
 @param  rec_id  input parameter name
 @param  type    input parameter type
 @param  param   output parameter
 @return cpl error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
espdr_inst_config_init_from_rec_params(cpl_parameterlist* params,
                                       const char* rec_id,
                                       cpl_type type, const char* param_name,
                                       void* param) {
    
    switch (type) {
            
        case CPL_TYPE_INT: {
            *((int*)param) = espdr_parameters_get_int(rec_id, params,param_name);
            //espdr_msg("now par is %d",*(int*)param);
        }
            break;
            
        case CPL_TYPE_FLOAT: {
            *((float*)param) = espdr_parameters_get_double(rec_id, params,param_name);
            //espdr_msg("now par is %g",*(float*)param);
        }
            break;
            
        case CPL_TYPE_DOUBLE: {
            *((double*)param) = espdr_parameters_get_double(rec_id, params,param_name);
            //espdr_msg("now par is %g",*(double*)param);
        }
            break;
            
        case CPL_TYPE_STRING:{
            const char * str = espdr_parameters_get_string(rec_id, params,param_name);
            
            if(str) {
                sprintf((char*)param, "%s", str);
                espdr_msg_debug("copy string %s",str);
                espdr_msg_debug("now par is %s",(char *)param);
                cpl_free(str);
            }
            //cpl_free(str);
            //espdr_msg("now par is %s",param);
            
        }
            break;
            
        default:
            espdr_msg_error("case not found! %d string type: %d",type,CPL_TYPE_STRING);
            break;
    }
    espdr_check_error_code("espdr_inst_config_init_from_rec_params");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief  init a parameter from a FITS table (mode dependent parameters)
 @param  params   input parameterlist
 @param  rec_id   input recipe name
 @param  tab      input FITS table
 @param  col_name table column name
 @param  type table column type
 @param  param  output parameter
 @return cpl error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
espdr_inst_config_init_mode_dep(cpl_parameterlist* params,const char* rec_id,
                                const cpl_table* tab, const char* col_name,
                                cpl_type type, const char* param_name, void* param)
{
    /* check input parameters are proper, else exit */
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(tab != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    
    cpl_parameter* p;
    char paramname[PARAM_NAME_LENGTH];
    char recipename[PARAM_NAME_LENGTH];
    sprintf(recipename,"espdr.%s",rec_id);
    sprintf(paramname,"%s.%s",recipename,param_name);
    p = cpl_parameterlist_find(params,paramname);
    
    
    espdr_msg("setting mode dependent parameter: %s",paramname);
    espdr_msg_debug("Reading input table column name %s!",col_name);
    /* first initialise with whatever the recipe sets to have the
     * param initialised to something reasonable */
    espdr_inst_config_init_from_rec_params(params,rec_id,type,param_name,param);
    /* Then checks if the parameter was touched from user in which case use
     * the user setting.
     */
    if(cpl_parameter_get_default_flag(p)) {
        /* In the parameter has changed use the parameter value */
        /* handle different data types */
        espdr_msg_warning("parameter %s.%s changed by user",
                          cpl_parameter_get_context(p),param_name);
        /* Note that the following line is not really needed as done earlier */
        espdr_inst_config_init_from_rec_params(params,rec_id,type,param_name,param);
        
    } else {
        /* In the parameter has NOT changed use the table parameter value */
        /* handle different data types. This overwrites the parameter value
         * As consortium asks in case the column value is set */
        espdr_inst_config_init_by_tab(tab, col_name, type, param);
        
    }
    espdr_check_error_code("espdr_inst_config_init_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief allocate memory and initialise the inst_config parameters structure
 @param parameters          parameters list
 @param ext_nb              number of extensions for the instrument
 @param inst_config_frame   frame with instrument config
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/

espdr_inst_config* espdr_inst_config_init(cpl_parameterlist *parameters,
                                          int ext_nb,
                                          cpl_frame* inst_config_frame) {
    
    const char *inst_config_filename = NULL;
    cpl_error_code my_error;
    espdr_inst_config* inst_config = NULL;
    
    /* check input function parameters */
    cpl_ensure(parameters != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(inst_config_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    
    inst_config = (espdr_inst_config *)cpl_malloc( sizeof(espdr_inst_config));
    
    inst_config_filename = cpl_frame_get_filename(inst_config_frame);
    my_error = espdr_read_inst_parameters(inst_config_filename,
                                          ext_nb,
                                          inst_config,
                                          parameters);
    
    espdr_msg("Input parameters for %s:", inst_config->instrument);
    
    /* TODO: AMO: this printing should be optional */
    const char *context = NULL;
    cpl_parameter *p = NULL;
    p = cpl_parameterlist_get_first(parameters);
    context = cpl_parameter_get_context(p);
    
    espdr_parameters_inst_config_print(inst_config, ext_nb, context);
    
    if ( my_error != CPL_ERROR_NONE ) {
        espdr_msg_error("Error during inst_config parameters initialisation");
        espdr_msg_error("%s", cpl_error_get_message());
        espdr_msg_error("%s", cpl_error_get_message_default(cpl_error_get_code()));
        return NULL;
    }
    
    espdr_check_error_code("espdr_inst_config_init");
    return inst_config;
}

/*----------------------------------------------------------------------------*/
/**
 @brief init overscan parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param rec_id            recipe id
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_oscan_mode_dep(const cpl_table *inst_config_read,
                                                  cpl_parameterlist* params,
                                                  const char* rec_id,
                                                  espdr_inst_config* inst_config) {
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_OVSC_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "ovsc_sig_clip_method",
                                    &inst_config->ovsc_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_OVSC_KSIGMA,
                                    CPL_TYPE_DOUBLE, "ovsc_ksigma",
                                    &inst_config->ovsc_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_OVSC_MAX_ITER,
                                    CPL_TYPE_INT, "ovsc_max_iter",
                                    &inst_config->ovsc_max_iter);
    espdr_check_error_code("espdr_params_oscan_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init bias parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_sbias_mode_dep(const cpl_table *inst_config_read,
                                                  cpl_parameterlist* params,
                                                  espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_single_bias";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    espdr_check_error_code("espdr_params_sbias_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init bias parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_bias_mode_dep(const cpl_table *inst_config_read,
                                                 cpl_parameterlist* params,
                                                 espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_mbias";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BIAS_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "bias_sig_clip_method",
                                    &inst_config->bias_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BIAS_KSIGMA,
                                    CPL_TYPE_DOUBLE, "bias_ksigma",
                                    &inst_config->bias_ksigma);
    espdr_check_error_code("espdr_params_bias_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init dark parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_dark_mode_dep(const cpl_table *inst_config_read,
                                                 cpl_parameterlist* params,
                                                 espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_mdark";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_DARK_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "dark_sig_clip_method",
                                    &inst_config->dark_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_DARK_KSIGMA,
                                    CPL_TYPE_DOUBLE, "dark_ksigma",
                                    &inst_config->dark_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_HOTPIXELS_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "hot_pix_sig_clip_method",
                                    &inst_config->hotpixels_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_HOTPIXELS_KSIGMA,
                                    CPL_TYPE_DOUBLE, "hot_pix_ksigma",
                                    &inst_config->hotpixels_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_HOTPIXELS_MAX_ITER,
                                    CPL_TYPE_INT, "hot_pix_max_iter",
                                    &inst_config->hotpixels_max_iter);
    espdr_check_error_code("espdr_params_dark_mode_dep");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief init ledff parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_ledff_mode_dep(const cpl_table *inst_config_read,
                                                  cpl_parameterlist* params,
                                                  espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_led_ff";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_LED_FF_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "led_ff_sig_clip_method",
                                    &inst_config->led_ff_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_LED_FF_KSIGMA,
                                    CPL_TYPE_DOUBLE, "led_ff_ksigma",
                                    &inst_config->led_ff_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BADPIXELS_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "bad_pix_sig_clip_method",
                                    &inst_config->badpixels_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BADPIXELS_KSIGMA,
                                    CPL_TYPE_DOUBLE, "bad_pix_ksigma",
                                    &inst_config->badpixels_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BADPIXELS_MAX_ITER,
                                    CPL_TYPE_INT, "bad_pix_max_iter",
                                    &inst_config->badpixels_max_iter);
    espdr_check_error_code("espdr_params_ledff_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init orderdef parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_orderdef_mode_dep(const cpl_table *inst_config_read,
                                                     cpl_parameterlist* params,
                                                     espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_orderdef";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    espdr_check_error_code("espdr_params_orderdef_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init flat parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_flat_mode_dep(const cpl_table *inst_config_read,
                                                 cpl_parameterlist* params,
                                                 espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_mflat";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_FLAT_SIGMA_CLIPPING_METHOD,
                                    CPL_TYPE_STRING, "flat_sig_clip_method",
                                    &inst_config->flat_sigma_clipping_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_FLAT_KSIGMA,
                                    CPL_TYPE_DOUBLE, "flat_ksigma",
                                    &inst_config->flat_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_TUN,
                                    CPL_TYPE_DOUBLE, "extraction_ksigma",
                                    &inst_config->extraction_ksigma_tun);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BKGR_GRID_SIZE_X,
                                    CPL_TYPE_INT, "bkgr_grid_size_x",
                                    &inst_config->bkgr_grid_size_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_BKGR_GRID_SIZE_Y,
                                    CPL_TYPE_INT, "bkgr_grid_size_y",
                                    &inst_config->bkgr_grid_size_y);
    espdr_check_error_code("espdr_params_flat_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init wave_FP parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_wave_FP_mode_dep(const cpl_table *inst_config_read,
                                                    cpl_parameterlist* params,
                                                    espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_wave_FP";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "fp_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_FP,
                                    CPL_TYPE_DOUBLE, "fp_extraction_ksigma",
                                    &inst_config->extraction_ksigma_fp);
    espdr_check_error_code("espdr_params_wave_FP_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init wave_THAR parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_wave_THAR_mode_dep(const cpl_table *inst_config_read,
                                                      cpl_parameterlist *params,
                                                      espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_wave_THAR";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "thar_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_THAR,
                                    CPL_TYPE_DOUBLE, "thar_extraction_ksigma",
                                    &inst_config->extraction_ksigma_thar);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_D_FIT_SMALL_RES_LIMIT,
                                    CPL_TYPE_DOUBLE, "d_fit_small_res_limit",
                                    &inst_config->d_fit_small_res_limit);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_D_FIT_POLY_DEG,
                                    CPL_TYPE_INT, "d_fit_poly_deg",
                                    &inst_config->d_fit_poly_deg);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_WAVE_SOL_POLY_DEG,
                                    CPL_TYPE_INT, "wave_sol_poly_deg",
                                    &inst_config->wave_sol_poly_deg);
    espdr_check_error_code("espdr_params_wave_THAR_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init wave_THAR parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_wave_TH_drift_mode_dep(const cpl_table *inst_config_read,
                                                          cpl_parameterlist *params,
                                                          espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_wave_TH_drift";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "thar_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_THAR,
                                    CPL_TYPE_DOUBLE, "thar_extraction_ksigma",
                                    &inst_config->extraction_ksigma_thar);
    espdr_check_error_code("espdr_params_wave_TH_drift_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init wave_LFC parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_wave_LFC_mode_dep(const cpl_table *inst_config_read,
                                                     cpl_parameterlist *params,
                                                     espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_wave_LFC";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "lfc_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_LFC,
                                    CPL_TYPE_DOUBLE, "lfc_extraction_ksigma",
                                    &inst_config->extraction_ksigma_lfc);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_WAVE_SOL_POLY_DEG,
                                    CPL_TYPE_INT, "wave_sol_poly_deg",
                                    &inst_config->wave_sol_poly_deg);
    espdr_check_error_code("espdr_params_wave_LFC_mode_dep");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief init contam parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_contam_mode_dep(const cpl_table *inst_config_read,
                                                   cpl_parameterlist *params,
                                                   espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_cal_contam";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_CONTAM_BKGR_GRID_SIZE_X,
                                    CPL_TYPE_INT, "contam_bkgr_grid_size_x",
                                    &inst_config->contam_bkgr_grid_size_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_CONTAM_BKGR_GRID_SIZE_Y,
                                    CPL_TYPE_INT, "contam_bkgr_grid_size_y",
                                    &inst_config->contam_bkgr_grid_size_y);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "contam_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_DARK,
                                    CPL_TYPE_DOUBLE, "contam_extraction_ksigma",
                                    &inst_config->extraction_ksigma_dark);
    espdr_check_error_code("espdr_params_contam_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init eff_ab parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_eff_ab_mode_dep(const cpl_table *inst_config_read,
                                                   cpl_parameterlist *params,
                                                   espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_cal_eff_ab";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EFF_AB_BKGR_GRID_SIZE_X,
                                    CPL_TYPE_INT, "eff_ab_bkgr_grid_size_x",
                                    &inst_config->eff_ab_bkgr_grid_size_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EFF_AB_BKGR_GRID_SIZE_Y,
                                    CPL_TYPE_INT, "eff_ab_bkgr_grid_size_y",
                                    &inst_config->eff_ab_bkgr_grid_size_y);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "eff_ab_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_SKY,
                                    CPL_TYPE_DOUBLE, "eff_ab_extraction_ksigma",
                                    &inst_config->extraction_ksigma_sky);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EFF_AB_POLY_DEG,
                                    CPL_TYPE_INT, "eff_ab_poly_deg",
                                    &inst_config->eff_ab_poly_deg);
    espdr_check_error_code("espdr_params_eff_ab_mode_dep");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief init flux parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_flux_mode_dep(const cpl_table *inst_config_read,
                                                 cpl_parameterlist *params,
                                                 espdr_inst_config* inst_config) {
    const char* rec_id = "espdr_cal_flux";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_SCI_BKGR_GRID_SIZE_X,
                                    CPL_TYPE_INT, "flux_bkgr_grid_size_x",
                                    &inst_config->sci_bkgr_grid_size_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_SCI_BKGR_GRID_SIZE_Y,
                                    CPL_TYPE_INT, "flux_bkgr_grid_size_y",
                                    &inst_config->sci_bkgr_grid_size_y);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_METHOD,
                                    CPL_TYPE_STRING, "flux_extraction_method",
                                    &inst_config->extraction_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_OBJECT,
                                    CPL_TYPE_DOUBLE, "flux_extraction_ksigma",
                                    &inst_config->extraction_ksigma_object);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_FLUX_POLY_DEG,
                                    CPL_TYPE_INT, "flux_poly_deg",
                                    &inst_config->flux_poly_deg);
    espdr_check_error_code("espdr_params_flux_mode_dep");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief init scired parameters (mode dependent)
 @params inst_config_read input FITS table
 @param params            recipe parameters list
 @param inst_config       instrument config ESPRESSO structure
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_params_scired_mode_dep(const cpl_table *inst_config_read,
                                                   cpl_parameterlist *params,
                                                   espdr_inst_config* inst_config) {
    
    const char* rec_id="espdr_sci_red";
    
    espdr_params_oscan_mode_dep(inst_config_read, params, rec_id, inst_config);
    
    inst_config->rv_range = espdr_parameters_get_double(rec_id, params,"rv_range");
    
    // This doesn't work for inst_config parameters, only the user provided ones (?)
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_SCI_BKGR_GRID_SIZE_X,
                                    CPL_TYPE_INT,
                                    "sci_bkgr_grid_size_x",
                                    &inst_config->sci_bkgr_grid_size_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_SCI_BKGR_GRID_SIZE_Y,
                                    CPL_TYPE_INT,
                                    "sci_bkgr_grid_size_y",
                                    &inst_config->sci_bkgr_grid_size_y);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_RV_STEP,
                                    CPL_TYPE_DOUBLE,
                                    "rv_step",
                                    &inst_config->rv_step);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_EXTRACTION_KSIGMA_SKY,
                                    CPL_TYPE_DOUBLE, "ksigma_cosmic",
                                    &inst_config->extraction_ksigma_sky);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_DRIFT_METHOD_FP, CPL_TYPE_STRING,
                                    "drift_method_fp",
                                    &inst_config->drift_method_fp);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_DRIFT_SPACE, CPL_TYPE_STRING,
                                    "drift_space", &inst_config->drift_space);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_DRIFT_KSIGMA, CPL_TYPE_DOUBLE,
                                    "drift_ksigma", &inst_config->drift_ksigma);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_SCI_SKY_CORR_METHOD, CPL_TYPE_STRING,
                                    "sky_corr_method", &inst_config->sci_sky_corr_method);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_SCI_SKY_SUB_SLIDING_BOX_SIZE,
                                    CPL_TYPE_INT,
                                    "sky_sub_sliding_box_size",
                                    &inst_config->sci_sky_sub_sliding_box_size);
    
    espdr_inst_config_init_mode_dep(params, rec_id, inst_config_read,
                                    COL_NAME_LACOSMIC_SKY_SW, CPL_TYPE_INT,
                                    "cosmic_detection_sw", &inst_config->lacosmic_sky_sw);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_SIGMA_LIM, CPL_TYPE_DOUBLE,
                                    "lacosmic.sigma_lim", &inst_config->lacosmic_sigma_lim);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_F_LIM, CPL_TYPE_DOUBLE,
                                    "lacosmic.f_lim", &inst_config->lacosmic_f_lim);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_MAX_ITER, CPL_TYPE_INT,
                                    "lacosmic.max_iter", &inst_config->lacosmic_max_iter);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_POST_FILTER_X, CPL_TYPE_INT,
                                    "lacosmic.post-filter-x",
                                    &inst_config->lacosmic_post_filter_x);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_POST_FILTER_Y, CPL_TYPE_INT,
                                    "lacosmic.post-filter-y",
                                    &inst_config->lacosmic_post_filter_y);
    
    espdr_inst_config_init_mode_dep(params, rec_id,inst_config_read,
                                    COL_NAME_LACOSMIC_POST_FILTER_MODE, CPL_TYPE_STRING,
                                    "lacosmic.post-filter-mode",
                                    &inst_config->lacosmic_post_filter_mode);
    
    espdr_check_error_code("espdr_params_scired_mode_dep");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_general_parameters(cpl_table *inst_config_read,
                                   espdr_inst_config *inst_config) {
    
    // General
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FIBRE_SIZE,
                                  CPL_TYPE_DOUBLE, &inst_config->fibre_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_VLTSH_WAVELENGTH,
                                  CPL_TYPE_DOUBLE, &inst_config->vltsh_wavelength);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_VLTSH_IQ,
                                  CPL_TYPE_DOUBLE, &inst_config->vltsh_iq);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_INST_IQ,
                                  CPL_TYPE_DOUBLE, &inst_config->inst_iq);
    espdr_check_error_code("espdr_read_inst_general_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_ovsc_parameters(cpl_table *inst_config_read,
                                espdr_inst_config *inst_config) {
    
    
    // OVSC
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_OVSC_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->ovsc_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_OVSC_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->ovsc_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_OVSC_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->ovsc_max_iter);
    
    espdr_check_error_code("espdr_read_inst_ovsc_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_bias_parameters(cpl_table *inst_config_read,
                                espdr_inst_config *inst_config) {
    
    // BIAS
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RAW_BIAS_LIMIT_NUMBER,
                                  CPL_TYPE_INT, &inst_config->raw_bias_limit_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_RON_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->bias_ron_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_MEAN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->bias_mean_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->bias_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->bias_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_STAT_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->stat_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_STAT_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->stat_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_STAT_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->stat_max_iter);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_RESIDUALS_MEAN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->bias_residuals_mean_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BIAS_RESIDUALS_STDEV_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->bias_residuals_stdev_limit);
    
    espdr_check_error_code("espdr_read_inst_bias_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_cte_parameters(cpl_table *inst_config_read, const int ext_nb,
                               espdr_inst_config *inst_config) {
    
    // CTE
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CTE_FLAG,
                                  CPL_TYPE_INT, &inst_config->cte_flag);
    
    int i;
    for (i = 0; i < ext_nb; i++) {
        inst_config->cte_alpha[i] = cpl_table_get_double(inst_config_read,
                                                         COL_NAME_CTE_ALPHA,
                                                         i, NULL);
        inst_config->cte_beta[i] = cpl_table_get_double(inst_config_read,
                                                        COL_NAME_CTE_BETA,
                                                        i, NULL);
    }
    
    espdr_check_error_code("espdr_read_inst_cte_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_dark_parameters(cpl_table *inst_config_read,
                                espdr_inst_config *inst_config) {
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RAW_DARK_LIMIT_NUMBER,
                                  CPL_TYPE_INT, &inst_config->raw_dark_limit_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DARK_CURRENT_LIMIT_EL,
                                  CPL_TYPE_DOUBLE, &inst_config->dark_current_limit_el);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_HOT_PIXELS_LIMIT,
                                  CPL_TYPE_INT, &inst_config->hot_pixels_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_COSMICS_LIMIT_PER_PIXEL,
                                  CPL_TYPE_DOUBLE, &inst_config->cosmics_limit_per_pixel);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DARK_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->dark_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DARK_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->dark_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_HOTPIXELS_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->hotpixels_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_HOTPIXELS_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->hotpixels_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_HOTPIXELS_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->hotpixels_max_iter);
    
    espdr_check_error_code("espdr_read_inst_dark_parameters");
    return cpl_error_get_code();
}
static cpl_error_code
espdr_read_inst_ledff_parameters(cpl_table *inst_config_read,
                                 espdr_inst_config *inst_config) {
    
    // LED_FF
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RAW_LED_FF_PER_TEXP_LIMIT_NUMBER,
                                  CPL_TYPE_INT, &inst_config->raw_led_ff_per_texp_limit_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RAW_LED_FF_DIFFERENT_TEXP,
                                  CPL_TYPE_INT, &inst_config->raw_led_ff_different_texp);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BAD_PIXELS_LIMIT,
                                  CPL_TYPE_INT, &inst_config->bad_pixels_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLUX_MIN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flux_min_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLUX_MAX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flux_max_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_GAIN_TOLERANCE,
                                  CPL_TYPE_DOUBLE, &inst_config->gain_tolerance);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DEAD_PIXEL_FLUX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->dead_pixel_flux_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LED_FF_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->led_ff_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LED_FF_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->led_ff_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BADPIXELS_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->badpixels_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BADPIXELS_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->badpixels_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BADPIXELS_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->badpixels_max_iter);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_GAIN_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->gain_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_GAIN_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->gain_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_GAIN_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->gain_max_iter);
    
    espdr_check_error_code("espdr_read_inst_ledff_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_orderdef_parameters(cpl_table *inst_config_read, int ext_nb,
                                    espdr_inst_config *inst_config) {
    
    // ORDERDEF
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SATUR_LIMIT_ADU,
                                  CPL_TYPE_DOUBLE, &inst_config->satur_limit_adu);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_STDEV_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->res_stdev_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_MIN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->res_min_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_MAX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->res_max_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SLICES_NB,
                                  CPL_TYPE_INT, &inst_config->slices_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_INTER_SLICE_DIST,
                                  CPL_TYPE_INT, &inst_config->inter_slice_dist);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SLICE_WIDTH,
                                  CPL_TYPE_INT, &inst_config->slice_width);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ORDER_WIDTH,
                                  CPL_TYPE_INT, &inst_config->order_width);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SLICES_NB_PER_PHYS_ORDER,
                                  CPL_TYPE_INT, &inst_config->slices_nb_per_phys_order);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BACKGROUND_PXLS,
                                  CPL_TYPE_INT, &inst_config->background_pxls);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FIBRES_NB,
                                  CPL_TYPE_INT, &inst_config->fibres_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->res_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->res_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RES_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->res_max_iter);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CLUSTER_DIST_X,
                                  CPL_TYPE_INT, &inst_config->cluster_dist_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CLUSTER_DIST_Y,
                                  CPL_TYPE_INT, &inst_config->cluster_dist_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_MIN_ORDER_Y_DIST,
                                  CPL_TYPE_INT, &inst_config->min_order_y_dist);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_MIN_ORDER_WIDTH,
                                  CPL_TYPE_INT, &inst_config->min_order_width);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ORDER_FIT_STEP,
                                  CPL_TYPE_INT, &inst_config->order_fit_step);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ORDER_FIT_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->order_fit_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ORDER_FLUX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->order_flux_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_IMAGE_COSMICS_PART,
                                  CPL_TYPE_DOUBLE, &inst_config->image_cosmics_part);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_IMAGE_THAR_PART,
                                  CPL_TYPE_DOUBLE, &inst_config->image_thar_part);
    
    int i;
    for (i = 0; i < inst_config->fibres_nb*ext_nb; i++) {
        inst_config->orders_nb[i] = cpl_table_get_int(inst_config_read,
                                                      COL_NAME_ORDERS_NB,
                                                      i, NULL);
        inst_config->order_phys_nb[i] = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_ORDER_PHYS_NB,
                                                          i, NULL);
        inst_config->order_start[i] = cpl_table_get_int(inst_config_read,
                                                        COL_NAME_ORDER_START,
                                                        i, NULL);
        inst_config->order_end[i] = cpl_table_get_int(inst_config_read,
                                                      COL_NAME_ORDER_END,
                                                      i, NULL);
        inst_config->orders_detection_start[i] = cpl_table_get_int(inst_config_read,
                                                                   COL_NAME_ORDERS_DETECTION_START,
                                                                   i, NULL);
    }
    
    for (i = 0; i < inst_config->fibres_nb; i++) {
        inst_config->order_flux_factor[i] = cpl_table_get_double(inst_config_read,
                                                                 COL_NAME_ORDER_FLUX_FACTOR,
                                                                 i, NULL);
        inst_config->min_order_len[i] = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_MIN_ORDER_LEN,
                                                          i, NULL);
        inst_config->min_width_pixels_nb[i] = cpl_table_get_int(inst_config_read,
                                                                COL_NAME_MIN_WIDTH_PIXELS_NB,
                                                                i, NULL);
        inst_config->min_order_x[i] = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_MIN_ORDER_X,
                                                          i, NULL);
        inst_config->max_order_y[i] = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_MAX_ORDER_Y,
                                                          i, NULL);
   }
    
    espdr_check_error_code("espdr_read_inst_orderdef_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_extract_parameters(cpl_table *inst_config_read,
                                   espdr_inst_config *inst_config) {
    
    // Extraction
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_BKGR_CALC_FRACTION,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_bkgr_calc_fraction);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_FRACTION_VALID_PIXELS,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_fraction_valid_pixels);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_METHOD,
                                  CPL_TYPE_STRING, &inst_config->extraction_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_TUN,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_tun);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_THAR,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_thar);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_FP,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_fp);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_LFC,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_lfc);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_OBJECT,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_object);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_SKY,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_sky);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EXTRACTION_KSIGMA_DARK,
                                  CPL_TYPE_DOUBLE, &inst_config->extraction_ksigma_dark);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TOLERANCE_REJECTION,
                                  CPL_TYPE_DOUBLE, &inst_config->tolerance_rejection);
    int i;
    for (i = 0; i < inst_config->fibres_nb; i++) {
        inst_config->extraction_window_size[i] =
        cpl_table_get_int(inst_config_read,
                          COL_NAME_EXTRACTION_WINDOW_SIZE, i, NULL);
    }
    
    espdr_check_error_code("espdr_read_inst_extract_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_flat_parameters(cpl_table *inst_config_read,
                                espdr_inst_config *inst_config) {
    
    // FLAT
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_RAW_FLAT_LIMIT_NUMBER,
                                  CPL_TYPE_INT, &inst_config->raw_flat_limit_nb);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_CURRENT_LIMIT_EL,
                                  CPL_TYPE_DOUBLE, &inst_config->flat_current_limit_el);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_SNR_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flat_snr_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_RMS_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flat_rms_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_BKGR_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flat_bkgr_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BKGR_GRID_SIZE_X,
                                  CPL_TYPE_INT, &inst_config->bkgr_grid_size_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BKGR_GRID_SIZE_Y,
                                  CPL_TYPE_INT, &inst_config->bkgr_grid_size_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_MINLEVEL_FACTOR,
                                  CPL_TYPE_INT, &inst_config->minlevel_factor);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->flat_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLAT_SIGMA_CLIPPING_METHOD,
                                  CPL_TYPE_STRING, &inst_config->flat_sigma_clipping_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_FLAT_CALC_METHOD,
                                  CPL_TYPE_STRING, &inst_config->blaze_flat_calc_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_FLAT_CALC_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->blaze_flat_calc_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_FLAT_CALC_SLIDE_BOX,
                                  CPL_TYPE_INT, &inst_config->blaze_flat_calc_slide_box);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_FLAT_CALC_CLIP,
                                  CPL_TYPE_DOUBLE, &inst_config->blaze_flat_calc_clip);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_PEAKS_FIT_METHOD,
                                  CPL_TYPE_STRING, &inst_config->blaze_peaks_fit_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_PEAKS_FIT_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->blaze_peaks_fit_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_BLAZE_PEAKS_FIT_SLIDE_BOX,
                                  CPL_TYPE_INT, &inst_config->blaze_peaks_fit_slide_box);
    
    espdr_check_error_code("espdr_read_inst_flat_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_wave_gen_parameters(cpl_table *inst_config_read,
                                    espdr_inst_config *inst_config) {
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LINE_MAX_SIG_X0,
                                  CPL_TYPE_DOUBLE, &inst_config->line_max_sig_x0);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LINE_MIN_SIG_X0,
                                  CPL_TYPE_DOUBLE, &inst_config->line_min_sig_x0);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LINE_MAX_FWHM,
                                  CPL_TYPE_DOUBLE, &inst_config->line_max_fwhm);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LINE_MIN_FLUX_EL,
                                  CPL_TYPE_DOUBLE, &inst_config->line_min_flux_el);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_SOL_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->wave_sol_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_SOL_FP_REJECT_ALPHA,
                                  CPL_TYPE_DOUBLE, &inst_config->wave_sol_FP_reject_alpha);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_SOL_FP_REJECT_BETA,
                                  CPL_TYPE_DOUBLE, &inst_config->wave_sol_FP_reject_beta);
    
    int i;
    for (i = 0; i < inst_config->fibres_nb; i++) {
        inst_config->thar_min_flux_ratio[i] = cpl_table_get_double(inst_config_read,
                                                                   COL_NAME_THAR_MIN_FLUX_RATIO,
                                                                   i, NULL);
        inst_config->thar_max_flux_ratio[i] = cpl_table_get_double(inst_config_read,
                                                                   COL_NAME_THAR_MAX_FLUX_RATIO,
                                                                   i, NULL);
    }
    
    espdr_check_error_code("espdr_read_inst_wave_gen_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_wave_fp_parameters(cpl_table *inst_config_read,
                                   espdr_inst_config *inst_config) {
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_FIT_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->fp_fit_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_FIT_TOLERANCE_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->fp_fit_tolerance_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_FIT_NGAUSS_FLAG,
                                  CPL_TYPE_INT, &inst_config->fp_fit_ngauss_flag);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_SEARCH_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->fp_search_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DELTA_FP_TOLERANCE,
                                  CPL_TYPE_DOUBLE, &inst_config->delta_fp_tolerance);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DELTA_FP_PEAK_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->delta_fp_peak_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_CCD_BORDER_CUT,
                                  CPL_TYPE_INT, &inst_config->fp_ccd_border_cut);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_NOISE_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_noise_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_PEAK_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_peak_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_PEAK_RMS_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_peak_rms_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_SPECTRAL_LINE_QC,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_spectral_line_qc);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_MIN_LINES_TOTAL,
                                  CPL_TYPE_INT, &inst_config->fp_min_lines_total);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_ADD_FAKE_PEAKS,
                                  CPL_TYPE_INT, &inst_config->fp_add_fake_peaks);
    
    espdr_check_error_code("espdr_read_inst_wave_fp_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_wave_thar_parameters(cpl_table *inst_config_read,
                                     espdr_inst_config *inst_config) {
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_THAR_FIT_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->thar_thar_fit_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_FP_FIT_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->thar_fp_fit_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_FIT_TOLERANCE_WINDOW_SIZE,
                                  CPL_TYPE_INT, &inst_config->thar_fit_tolerance_window_size);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_CCD_BORDER_CUT,
                                  CPL_TYPE_INT, &inst_config->thar_ccd_border_cut);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_NOISE_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_noise_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_PEAK_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_peak_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_SPECTRAL_LINE_QC,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_spectral_line_qc);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FP_D_ANGSTROM,
                                  CPL_TYPE_DOUBLE, &inst_config->fp_d_angstrom);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_D_FIT_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->d_fit_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_D_FIT_BIG_RES_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->d_fit_big_res_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_D_FIT_SMALL_RES_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->d_fit_small_res_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_D_FIT_KERNEL_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->d_fit_kernel_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_D_FIT_KERNEL_FWHM,
                                  CPL_TYPE_DOUBLE, &inst_config->d_fit_kernel_fwhm);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_SOL_MAX_CHI2,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_sol_max_chi2);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_SOL_MAX_RMS,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_sol_max_rms);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_MIN_RESOL,
                                  CPL_TYPE_DOUBLE, &inst_config->wave_min_resol);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_SOL_MIN_LINES_PER_ORDER,
                                  CPL_TYPE_INT, &inst_config->wave_sol_min_lines_per_order);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_MIN_LINES_TOTAL,
                                  CPL_TYPE_INT, &inst_config->thar_min_lines_total);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_DRIFT_TH_FP_SW,
                                  CPL_TYPE_INT, &inst_config->thar_drift_th_fp_sw);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_DRIFT_FP_TH_SW,
                                  CPL_TYPE_INT, &inst_config->thar_drift_fp_th_sw);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_THAR_DRIFT_CHI2_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->thar_drift_chi2_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_WAVE_ORDERS_ANALYTICAL_LINK_SW,
                                  CPL_TYPE_INT, &inst_config->wave_orders_analytical_link_sw);
    
    
    espdr_check_error_code("espdr_read_inst_wave_thar_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_wave_drift_parameters(cpl_table *inst_config_read,
                                      espdr_inst_config *inst_config) {
    
    // WAVE TH DRIFT
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_LAMP_OFFSET_AR,
                                  CPL_TYPE_DOUBLE, &inst_config->th_lamp_offset_ar);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_LAMP_OFFSET_AR1,
                                  CPL_TYPE_DOUBLE, &inst_config->th_lamp_offset_ar1);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_LAMP_OFFSET_AR2,
                                  CPL_TYPE_DOUBLE, &inst_config->th_lamp_offset_ar2);
    
    int i;
    for (i = 0; i < inst_config->fibres_nb; i++) {
        inst_config->th_drift_min_lines_total[i] = cpl_table_get_int(inst_config_read,
                                                                     COL_NAME_TH_DRIFT_MIN_LINES_TOTAL,
                                                                     i, NULL);
    }
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_MAX_SATUR_LINES,
                                  CPL_TYPE_INT, &inst_config->th_drift_max_satur_lines);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_FIT_DEG_X,
                                  CPL_TYPE_INT, &inst_config->th_drift_fit_deg_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_FIT_DEG_Y,
                                  CPL_TYPE_INT, &inst_config->th_drift_fit_deg_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_FIT_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->th_drift_fit_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_FIT_MAX_MAD,
                                  CPL_TYPE_DOUBLE, &inst_config->th_drift_fit_max_mad);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_TH_DRIFT_MAX_REJECT_LINES_FRACTION,
                                  CPL_TYPE_DOUBLE, &inst_config->th_drift_max_reject_lines_fraction);
    
    
    espdr_check_error_code("espdr_read_inst_wave_drift_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_wave_lfc_parameters(cpl_table *inst_config_read,
                                    espdr_inst_config *inst_config) {
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_ANCHOR_FREQ,
                                  CPL_TYPE_DOUBLE, &inst_config->lfc_anchor_freq);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_REPETITION_RATE,
                                  CPL_TYPE_DOUBLE, &inst_config->lfc_repetition_rate);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_SPECTRAL_LINE_QC,
                                  CPL_TYPE_DOUBLE, &inst_config->lfc_spectral_line_qc);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_SOL_MAX_CHI2,
                                  CPL_TYPE_DOUBLE, &inst_config->lfc_sol_max_chi2);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_SOL_MAX_RMS,
                                  CPL_TYPE_DOUBLE, &inst_config->lfc_sol_max_rms);
    
    int i;
    for (i = 0; i < inst_config->fibres_nb; i++) {
        inst_config->lfc_sol_min_lines_per_order[i] = cpl_table_get_int(inst_config_read,
                                                                        COL_NAME_LFC_SOL_MIN_LINES_PER_ORDER,
                                                                        i, NULL);
    }
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LFC_MIN_LINES_TOTAL,
                                  CPL_TYPE_INT, &inst_config->lfc_min_lines_total);
    
    
    espdr_check_error_code("espdr_read_inst_wave_lfc_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_contam_parameters(cpl_table *inst_config_read,
                                  espdr_inst_config *inst_config) {
    
    // CONTAM
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CONTAM_BKGR_GRID_SIZE_X,
                                  CPL_TYPE_INT, &inst_config->contam_bkgr_grid_size_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CONTAM_BKGR_GRID_SIZE_Y,
                                  CPL_TYPE_INT, &inst_config->contam_bkgr_grid_size_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_CONTAM_FLUX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->contam_flux_limit);
    
    
    espdr_check_error_code("espdr_read_inst_contam_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_effab_parameters(cpl_table *inst_config_read,
                                 espdr_inst_config *inst_config) {
    
    // EFF_AB
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EFF_AB_SNR_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->eff_ab_snr_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EFF_AB_BKGR_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->eff_ab_bkgr_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EFF_AB_BKGR_GRID_SIZE_X,
                                  CPL_TYPE_INT, &inst_config->eff_ab_bkgr_grid_size_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EFF_AB_BKGR_GRID_SIZE_Y,
                                  CPL_TYPE_INT, &inst_config->eff_ab_bkgr_grid_size_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_EFF_AB_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->eff_ab_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_REL_EFF_MIN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->rel_eff_min_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_REL_EFF_MAX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->rel_eff_max_limit);
    
    espdr_check_error_code("espdr_read_inst_effab_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_flux_parameters(cpl_table *inst_config_read,
                                espdr_inst_config *inst_config) {
    
    // FLUX
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLUX_EFF_INTERP_METHOD,
                                  CPL_TYPE_STRING, &inst_config->flux_eff_interp_method);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLUX_POLY_DEG,
                                  CPL_TYPE_INT, &inst_config->flux_poly_deg);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_FLUX_SNR_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->flux_snr_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ABS_EFF_MIN_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->abs_eff_min_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_ABS_EFF_MAX_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->abs_eff_max_limit);
    
    espdr_check_error_code("espdr_read_inst_flux_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_read_inst_science_parameters(cpl_table *inst_config_read,
                                   espdr_inst_config *inst_config) {
    
    const char *read_keyword = NULL;
    
    // SCIENCE
    inst_config->sci_bkgr_grid_size_x = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_SCI_BKGR_GRID_SIZE_X, 0, NULL);
    inst_config->sci_bkgr_grid_size_y = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_SCI_BKGR_GRID_SIZE_Y, 0, NULL);
    inst_config->snr_averaging_window = cpl_table_get_int(inst_config_read,
                                                          COL_NAME_SNR_AVERAGING_WINDOW, 0, NULL);
    inst_config->berv_type_init = cpl_table_get_int(inst_config_read,
                                                    COL_NAME_BERV_TYPE_INIT, 0, NULL);
    inst_config->tel_geolon = cpl_table_get_double(inst_config_read,
                                                   COL_NAME_TEL_GEOLON, 0, NULL);
    inst_config->tel_geolat = cpl_table_get_double(inst_config_read,
                                                   COL_NAME_TEL_GEOLAT, 0, NULL);
    inst_config->tel_geoelev = cpl_table_get_double(inst_config_read,
                                                    COL_NAME_TEL_GEOELEV, 0, NULL);
    inst_config->mask_width = cpl_table_get_double(inst_config_read,
                                                   COL_NAME_MASK_WIDTH_KW, 0, NULL);
    inst_config->log_lambda_ini = cpl_table_get_double(inst_config_read,
                                                       COL_NAME_LOG_LAMBDA_INI_KW, 0, NULL);
    inst_config->delta_log_lambda = cpl_table_get_double(inst_config_read,
                                                         COL_NAME_DELTA_LOG_LAMBDA_KW, 0, NULL);
    inst_config->log_lambda_end = cpl_table_get_double(inst_config_read,
                                                       COL_NAME_LOG_LAMBDA_END_KW, 0, NULL);
    inst_config->rv_range = cpl_table_get_double(inst_config_read,
                                                 COL_NAME_RV_RANGE, 0, NULL);
    inst_config->rv_step = cpl_table_get_double(inst_config_read,
                                                COL_NAME_RV_STEP, 0, NULL);
    inst_config->tel_surface = cpl_table_get_double(inst_config_read,
                                                    COL_NAME_TEL_SURFACE, 0, NULL);
    read_keyword = cpl_table_get_string(inst_config_read,
                                        COL_NAME_DEFAULT_SPEC_TYPE, 0);
    sprintf(inst_config->default_spec_type, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read,
                                        COL_NAME_SCI_SKY_CORR_METHOD, 0);
    sprintf(inst_config->sci_sky_corr_method, "%s", read_keyword);
    inst_config->sci_sky_sub_sliding_box_size = cpl_table_get_int(inst_config_read,
                                            COL_NAME_SCI_SKY_SUB_SLIDING_BOX_SIZE, 0, NULL);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SCI_OH_CORR_LINES_NB_LIMIT,
                                  CPL_TYPE_INT, &inst_config->oh_corr_lines_nb_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SCI_OH_CORR_RATIO_LOWER_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->oh_corr_ratio_lower_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SCI_OH_CORR_RATIO_UPPER_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->oh_corr_ratio_upper_limit);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_SCI_OH_CORR_RATIO_MAD_LIMIT,
                                  CPL_TYPE_DOUBLE, &inst_config->oh_corr_ratio_mad_limit);

    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_SKY_SW,
                                  CPL_TYPE_INT, &inst_config->lacosmic_sky_sw);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_CALIB_SW,
                                  CPL_TYPE_INT, &inst_config->lacosmic_calib_sw);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_POST_FILTER_X,
                                  CPL_TYPE_INT, &inst_config->lacosmic_post_filter_x);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_POST_FILTER_Y,
                                  CPL_TYPE_INT, &inst_config->lacosmic_post_filter_y);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_POST_FILTER_MODE,
                                  CPL_TYPE_STRING, &inst_config->lacosmic_post_filter_mode);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_SIGMA_LIM,
                                  CPL_TYPE_DOUBLE, &inst_config->lacosmic_sigma_lim);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_F_LIM,
                                  CPL_TYPE_DOUBLE, &inst_config->lacosmic_f_lim);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_LACOSMIC_MAX_ITER,
                                  CPL_TYPE_INT, &inst_config->lacosmic_max_iter);
    
    
    espdr_check_error_code("espdr_read_inst_science_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_read_inst_drift_parameters(cpl_table *inst_config_read,
                                 espdr_inst_config *inst_config) {
    
    // DRIFT
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_KSIGMA,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_ksigma);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_METHOD_THAR,
                                  CPL_TYPE_STRING, &inst_config->drift_method_thar);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_METHOD_FP,
                                  CPL_TYPE_STRING, &inst_config->drift_method_fp);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_METHOD_LFC,
                                  CPL_TYPE_STRING, &inst_config->drift_method_lfc);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_SPACE,
                                  CPL_TYPE_STRING, &inst_config->drift_space);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_MAX_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_max_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_MIN_FLUX_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_min_flux_threshold);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_MAX_FLUX_RATIO,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_max_flux_ratio);
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_MIN_FLUX_RATIO,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_min_flux_ratio);
    
    if (strcmp(inst_config->drift_space, "pixel") == 0) {
        inst_config->drift_mean_threshold = cpl_table_get_double(inst_config_read,
                                                                 COL_NAME_DRIFT_MEAN_THRESHOLD,
                                                                 0, NULL);
        inst_config->drift_mean_err_threshold = cpl_table_get_double(inst_config_read,
                                                                     COL_NAME_DRIFT_MEAN_ERR_THRESHOLD,
                                                                     0, NULL);
    } else {
        inst_config->drift_mean_threshold = cpl_table_get_double(inst_config_read,
                                                                 COL_NAME_DRIFT_MEAN_THRESHOLD,
                                                                 1, NULL);
        inst_config->drift_mean_err_threshold = cpl_table_get_double(inst_config_read,
                                                                     COL_NAME_DRIFT_MEAN_ERR_THRESHOLD,
                                                                     1, NULL);
    }
    
    espdr_inst_config_init_by_tab(inst_config_read, COL_NAME_DRIFT_CHI2_THRESHOLD,
                                  CPL_TYPE_DOUBLE, &inst_config->drift_chi2_threshold);
    
    
    espdr_check_error_code("espdr_read_inst_drift_parameters");
    return cpl_error_get_code();
    
}

static cpl_error_code
espdr_define_product_filenames(espdr_inst_config *inst_config) {
    
    // PRODUCTS filenames
    sprintf(inst_config->master_bias_filename, "%s_master_bias.fits", inst_config->instrument);
    sprintf(inst_config->master_overscan_filename, "%s_master_overscan.fits", inst_config->instrument);
    sprintf(inst_config->master_bias_res_filename, "%s_master_bias_res.fits", inst_config->instrument);
    sprintf(inst_config->single_bias_filename, "%s_single_bias.fits", inst_config->instrument);
    sprintf(inst_config->single_bias_res_filename, "%s_single_bias_res.fits", inst_config->instrument);
    sprintf(inst_config->hot_pixels_filename, "%s_hot_pixels.fits", inst_config->instrument);
    sprintf(inst_config->master_dark_filename, "%s_master_dark.fits", inst_config->instrument);
    sprintf(inst_config->bad_pixels_filename, "%s_bad_pixels.fits", inst_config->instrument);
    sprintf(inst_config->master_led_ff_filename, "%s_master_led_ff.fits", inst_config->instrument);
    sprintf(inst_config->orders_map_filename, "%s_ORDER_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->orders_map_filename_no_fits, "%s_ORDER_TABLE", inst_config->instrument);
    sprintf(inst_config->master_flat_filename, "%s_master_flat.fits", inst_config->instrument);
    sprintf(inst_config->flat_filename, "%s_FLAT.fits", inst_config->instrument);
    sprintf(inst_config->flat_filename_no_fits, "%s_FLAT", inst_config->instrument);
    sprintf(inst_config->master_flat_order_profile_filename, "%s_ORDER_PROFILE.fits", inst_config->instrument);
    sprintf(inst_config->master_flat_order_profile_filename_no_fits, "%s_ORDER_PROFILE", inst_config->instrument);
    sprintf(inst_config->flat_blaze_filename, "%s_BLAZE.fits", inst_config->instrument);
    sprintf(inst_config->flat_blaze_filename_no_fits, "%s_BLAZE", inst_config->instrument);
    sprintf(inst_config->master_wave_filename, "%s_cleaned_wave.fits", inst_config->instrument);
    sprintf(inst_config->extr_spectr_filename, "%s_S2D.fits", inst_config->instrument);
    sprintf(inst_config->extr_spectr_filename_no_fits, "%s_S2D", inst_config->instrument);
    sprintf(inst_config->line_table_filename, "%s_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->line_table_filename_no_fits, "%s_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->fp_fitted_line_table_filename, "%s_FP_FITTED_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->fp_fitted_line_table_filename_no_fits, "%s_FP_FITTED_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->fp_searched_line_table_filename, "%s_FP_SEARCHED_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->fp_searched_line_table_filename_no_fits, "%s_FP_SEARCHED_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->lfc_fitted_line_table_filename, "%s_LFC_FITTED_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->lfc_fitted_line_table_filename_no_fits, "%s_LFC_FITTED_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->thar_line_table_filename, "%s_THAR_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->thar_line_table_filename_no_fits, "%s_THAR_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->th_line_table_filename, "%s_TH_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->th_line_table_filename_no_fits, "%s_TH_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->ar_line_table_filename, "%s_AR_LINE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->ar_line_table_filename_no_fits, "%s_AR_LINE_TABLE", inst_config->instrument);
    sprintf(inst_config->line_table_raw_filename, "%s_LINE_TABLE_RAW.fits", inst_config->instrument);
    sprintf(inst_config->line_table_raw_filename_no_fits, "%s_LINE_TABLE_RAW", inst_config->instrument);
    sprintf(inst_config->wave_table_filename, "%s_WAVE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->wave_table_filename_no_fits, "%s_WAVE_TABLE", inst_config->instrument);
    sprintf(inst_config->wave_matrix_filename, "%s_WAVE_MATRIX.fits", inst_config->instrument);
    sprintf(inst_config->wave_matrix_filename_no_fits, "%s_WAVE_MATRIX", inst_config->instrument);
    sprintf(inst_config->dll_matrix_filename_no_fits, "%s_DLL_MATRIX", inst_config->instrument);
    sprintf(inst_config->air_wave_matrix_filename, "%s_AIR_WAVE_MATRIX.fits", inst_config->instrument);
    sprintf(inst_config->air_wave_matrix_filename_no_fits, "%s_AIR_WAVE_MATRIX", inst_config->instrument);
    sprintf(inst_config->air_dll_matrix_filename_no_fits, "%s_AIR_DLL_MATRIX", inst_config->instrument);
    sprintf(inst_config->wave_matrix_drift_filename, "%s_WAVE_MATRIX_DRIFT.fits", inst_config->instrument);
    sprintf(inst_config->wave_matrix_drift_filename_no_fits, "%s_WAVE_MATRIX_DRIFT", inst_config->instrument);
    sprintf(inst_config->dll_matrix_drift_filename_no_fits, "%s_DLL_MATRIX_DRIFT", inst_config->instrument);
    sprintf(inst_config->air_wave_matrix_drift_filename, "%s_AIR_WAVE_MATRIX_DRIFT.fits", inst_config->instrument);
    sprintf(inst_config->air_wave_matrix_drift_filename_no_fits, "%s_AIR_WAVE_MATRIX_DRIFT", inst_config->instrument);
    sprintf(inst_config->air_dll_matrix_drift_filename_no_fits, "%s_AIR_DLL_MATRIX_DRIFT", inst_config->instrument);
    sprintf(inst_config->res_map_filename, "%s_RES_MAP.fits", inst_config->instrument);
    sprintf(inst_config->res_map_filename_no_fits, "%s_RES_MAP", inst_config->instrument);
    sprintf(inst_config->lfc_wave_table_filename, "%s_LFC_WAVE_TABLE.fits", inst_config->instrument);
    sprintf(inst_config->lfc_wave_table_filename_no_fits, "%s_LFC_WAVE_TABLE", inst_config->instrument);
    sprintf(inst_config->lfc_wave_matrix_filename, "%s_LFC_WAVE_MATRIX.fits", inst_config->instrument);
    sprintf(inst_config->lfc_wave_matrix_filename_no_fits, "%s_LFC_WAVE_MATRIX", inst_config->instrument);
    sprintf(inst_config->lfc_dll_matrix_filename_no_fits, "%s_LFC_DLL_MATRIX", inst_config->instrument);
    sprintf(inst_config->lfc_air_wave_matrix_filename, "%s_LFC_AIR_WAVE_MATRIX.fits", inst_config->instrument);
    sprintf(inst_config->lfc_air_wave_matrix_filename_no_fits, "%s_LFC_AIR_WAVE_MATRIX", inst_config->instrument);
    sprintf(inst_config->lfc_air_dll_matrix_filename_no_fits, "%s_LFC_AIR_DLL_MATRIX", inst_config->instrument);
    
    espdr_check_error_code("espdr_define_product_filenames");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_handling_recipe_parameters(cpl_table *inst_config_read,
                                 espdr_inst_config *inst_config,
                                 cpl_parameterlist *parameters) {
    
    // Handling of recipe parameters: they overwrite what set from table
    const char* context = NULL;
    cpl_parameter* p = NULL;
    espdr_recipe_id rec_id;
    p=cpl_parameterlist_get_first(parameters);
    context=cpl_parameter_get_context(p);
    espdr_msg("parameter context = %s",context);
    
    if(strcmp(context,"espdr.espdr_mbias") == 0)     rec_id = ESPDR_R_MBIAS;
    if(strcmp(context,"espdr.espdr_single_bias") == 0)     rec_id = ESPDR_R_SINGLE_BIAS;
    if(strcmp(context,"espdr.espdr_mdark") == 0)     rec_id = ESPDR_R_MDARK;
    if(strcmp(context,"espdr.espdr_led_ff") == 0)    rec_id = ESPDR_R_LED_FF;
    if(strcmp(context,"espdr.espdr_orderdef") == 0)  rec_id = ESPDR_R_ORDERDEF;
    if(strcmp(context,"espdr.espdr_mflat") == 0)     rec_id = ESPDR_R_MFLAT;
    if(strcmp(context,"espdr.espdr_wave_FP") == 0)   rec_id = ESPDR_R_WAVE_FP;
    if(strcmp(context,"espdr.espdr_wave_THAR") == 0) rec_id = ESPDR_R_WAVE_THAR;
    if(strcmp(context,"espdr.espdr_wave_TH_drift") == 0) rec_id = ESPDR_R_WAVE_TH_DRIFT;
    if(strcmp(context,"espdr.espdr_wave_LFC") == 0)  rec_id = ESPDR_R_WAVE_LFC;
    if(strcmp(context,"espdr.espdr_cal_contam") == 0)    rec_id = ESPDR_R_CONTAM;
    if(strcmp(context,"espdr.espdr_cal_eff_ab") == 0)    rec_id = ESPDR_R_EFF_AB;
    if(strcmp(context,"espdr.espdr_cal_flux") == 0)      rec_id = ESPDR_R_FLUX;
    if(strcmp(context,"espdr.espdr_sci_red") == 0)       rec_id = ESPDR_R_SCI_RED;
    if(strcmp(context,"espdr.espdr_wave_LFC_LFC") == 0)  rec_id = ESPDR_R_WAVE_LFC_LFC;
    if(strcmp(context,"espdr.espdr_wave_THAR_THAR") == 0)   rec_id = ESPDR_R_WAVE_THAR_THAR;
    if(strcmp(context,"espdr.espdr_compu_drift") == 0)   rec_id = ESPDR_R_COMPU_DRIFT;
    if(strcmp(context,"espdr.espdr_shifted_extraction") == 0)   rec_id = ESPDR_R_SHIFTED_EXTRACTION;

    switch(rec_id) {
        case ESPDR_R_MBIAS: {
            espdr_params_bias_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_SINGLE_BIAS: {
            espdr_params_sbias_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_MDARK: {
            espdr_params_dark_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_LED_FF: {
            espdr_params_ledff_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_ORDERDEF: {
            espdr_params_orderdef_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_MFLAT: {
            espdr_params_flat_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_WAVE_FP: {
            espdr_params_wave_FP_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_WAVE_THAR: {
            espdr_params_wave_THAR_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_WAVE_TH_DRIFT: {
            espdr_params_wave_TH_drift_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_WAVE_LFC: {
            espdr_params_wave_LFC_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_CONTAM: {
            espdr_params_contam_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_EFF_AB: {
            espdr_params_eff_ab_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_FLUX: {
            espdr_params_flux_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_SCI_RED: {
            espdr_params_scired_mode_dep(inst_config_read, parameters, inst_config);
        }
            break;
        case ESPDR_R_WAVE_LFC_LFC: {
            espdr_msg("context %s not yet supported",context);
            espdr_params_oscan_mode_dep(inst_config_read, parameters, "espdr_wave_LFC_LFC", inst_config);
            //Internal recipe for commissioning
        }
            break;
        case ESPDR_R_WAVE_THAR_THAR: {
            espdr_params_oscan_mode_dep(inst_config_read, parameters, "espdr_wave_THAR_THAR", inst_config);
            espdr_msg("context %s not yet supported",context);
            //Internal recipe for commissioning
        }
            break;
        case ESPDR_R_COMPU_DRIFT: {
            espdr_msg("context %s not yet supported",context);
            //Internal recipe for commissioning
        }
            break;
        case ESPDR_R_SHIFTED_EXTRACTION: {
            espdr_msg("context %s not yet supported",context);
            //Internal recipe for commissioning
        }
            break;
        default: {
            espdr_msg("context %s not yet supported",context);
            //Internal recipe for commissioning
        }
            break;
    }
    
    espdr_check_error_code("espdr_handling_recipe_parameters");
    return cpl_error_get_code();
    
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Read the instrument configuration parameters from a FITS table
 @param     filename    the filename of the FITS table
 @param     ext_nb      extensions number for the instrument
 @param     inst_config the data structure with all instrument parameters
 @param     parameters  recipe user parameters
 @return   CPL_ERROR_NONE iff OK else NULL
 
 TODO: do one really need to distinguish between extensions? Isn't the
 geometry the same?
 --> DSO:   yes, it can happen, that extensions differ.
 */
/*--------------------------------------------------------------------------*/
cpl_error_code espdr_read_inst_parameters(const char *filename,
                                          int ext_nb,
                                          espdr_inst_config *inst_config,
                                          cpl_parameterlist *parameters) {
    
    const char *read_inst = NULL;
    const char *read_prefix = NULL;
    const char *read_keyword = NULL;
    
    cpl_table *inst_config_read = NULL;
    
    espdr_ensure(filename == NULL, CPL_ERROR_NULL_INPUT,
                 "The input filename is NULL");
    
    espdr_msg("Instrument configuration filename: %s", filename);
    
    inst_config_read = cpl_table_load(filename, 1, 0);
    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "Error loading FITS table with instrument parameters");
    
    read_inst = cpl_table_get_string(inst_config_read, COL_NAME_INST_NAME, 0);
    sprintf(inst_config->instrument, "%s", read_inst);
    read_inst = cpl_table_get_string(inst_config_read, COL_NAME_INST_TYPE, 0);
    if (strcmp(read_inst, "VIS") == 0) {
        inst_config->inst_type = VIS;
    } else {
        inst_config->inst_type = NIR;
    }
    read_prefix = cpl_table_get_string(inst_config_read, COL_NAME_PREFIX_KW, 0);
    sprintf(inst_config->prefix, "%s", read_prefix);
    
    espdr_read_inst_general_parameters(inst_config_read, inst_config);
    espdr_read_inst_ovsc_parameters(inst_config_read, inst_config);
    espdr_read_inst_bias_parameters(inst_config_read, inst_config);
    espdr_read_inst_cte_parameters(inst_config_read, ext_nb, inst_config);
    espdr_read_inst_dark_parameters(inst_config_read, inst_config);
    espdr_read_inst_ledff_parameters(inst_config_read, inst_config);
    espdr_read_inst_orderdef_parameters(inst_config_read, ext_nb, inst_config);
    espdr_read_inst_extract_parameters(inst_config_read, inst_config);
    espdr_read_inst_flat_parameters(inst_config_read, inst_config);
    espdr_read_inst_wave_gen_parameters(inst_config_read, inst_config);
    espdr_read_inst_wave_fp_parameters(inst_config_read, inst_config);
    espdr_read_inst_wave_thar_parameters(inst_config_read, inst_config);
    espdr_read_inst_wave_drift_parameters(inst_config_read, inst_config);
    espdr_read_inst_wave_lfc_parameters(inst_config_read, inst_config);
    espdr_read_inst_contam_parameters(inst_config_read, inst_config);
    espdr_read_inst_effab_parameters(inst_config_read, inst_config);
    espdr_read_inst_flux_parameters(inst_config_read, inst_config);
    espdr_read_inst_science_parameters(inst_config_read, inst_config);
    espdr_read_inst_drift_parameters(inst_config_read, inst_config);
    
    // PRODUCTS filenames
    espdr_define_product_filenames(inst_config);
    
    // KEYWORDS
    //espdr_define_keywords_names(inst_config_read, &inst_config);
    
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEXP_KW, 0);
    sprintf(inst_config->Texp_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TMMEAN_KW, 0);
    sprintf(inst_config->tmmean_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read,COL_NAME_GAIN_KW_FIRST_PART, 0);
    sprintf(inst_config->conad_kw_first_part, "%s %s",read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_GAIN_KW_LAST_PART, 0);
    sprintf(inst_config->conad_kw_last_part, " %s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_DET_BINX_KW, 0);
    sprintf(inst_config->det_binx_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_DET_BINY_KW, 0);
    sprintf(inst_config->det_biny_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEL_TARG_ALPHA_KW, 0);
    sprintf(inst_config->tel_alpha_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEL_TARG_DELTA_KW, 0);
    sprintf(inst_config->tel_delta_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEL_TARG_PMA_KW, 0);
    sprintf(inst_config->tel_pma_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEL_TARG_PMD_KW, 0);
    sprintf(inst_config->tel_pmd_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_DATE_OBS_KW, 0);
    sprintf(inst_config->tel_date_obs_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TARG_NAME_KW, 0);
    sprintf(inst_config->targ_name_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_SPEC_TYPE_VAL_KW, 0);
    sprintf(inst_config->spec_type_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_ZEM_KW, 0);
    sprintf(inst_config->zem_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_WAVECAL_SRC_KW, 0);
    sprintf(inst_config->wavecal_src_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_RV_CENTER_KW, 0);
    sprintf(inst_config->rv_center_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_AIRM_START_KW, 0);
    sprintf(inst_config->airm_start_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_AIRM_END_KW, 0);
    sprintf(inst_config->airm_end_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_PRES_START_KW, 0);
    sprintf(inst_config->pres_start_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_PRES_END_KW, 0);
    sprintf(inst_config->pres_end_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_TEMP_KW, 0);
    sprintf(inst_config->temp_kw, "%s", read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_DPR_TYPE_KW, 0);
    sprintf(inst_config->dpr_type_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_REP_RATE_KW, 0);
    sprintf(inst_config->rep_rate_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_ANCHOR_FREQ_KW, 0);
    sprintf(inst_config->anchor_freq_kw, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_SEEING_KW_FIRST_PART, 0);
    sprintf(inst_config->seeing_kw_first_part, "%s %s", read_prefix, read_keyword);
    read_keyword = cpl_table_get_string(inst_config_read, COL_NAME_SEEING_KW_LAST_PART, 0);
    sprintf(inst_config->seeing_kw_last_part, "%s", read_keyword);
    
    espdr_handling_recipe_parameters(inst_config_read, inst_config, parameters);
    
    cpl_table_delete(inst_config_read);
    espdr_check_error_code("espdr_read_inst_parameters");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_general_parameters_inst_config(espdr_inst_config *inst_config) {
    espdr_msg("\t\tGenaral parameters:");
    espdr_msg("\t\t\tFibre size: %.2lf",
              inst_config->fibre_size);
    espdr_msg("\t\t\tEffective VLTSH wavelength: %.2lf",
              inst_config->vltsh_wavelength);
    espdr_msg("\t\t\tVLTSH Image Quality: %.2lf",
              inst_config->vltsh_iq);
    espdr_msg("\t\t\tInstrument Image Quality: %.2lf",
              inst_config->inst_iq);
    
    espdr_msg("\t\tOVSC parameters:");
    espdr_msg("\t\t\tOVSC sigma clipping ksigma: %.2lf",
              inst_config->ovsc_ksigma);
    espdr_msg("\t\t\tOVSC sigma clipping method: %s",
              inst_config->ovsc_sigma_clipping_method);
    espdr_msg("\t\t\tOVSC sigma clipping max iter: %d",
              inst_config->ovsc_max_iter);
    espdr_check_error_code("espdr_print_general_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_bias_parameters_inst_config(espdr_inst_config *inst_config) {
    espdr_msg("\t\tCAL_BIAS parameters:");
    espdr_msg("\t\t\tRaw BIAS limit number: %d",
              inst_config->raw_bias_limit_nb);
    espdr_msg("\t\t\tBIAS RON limit: %.2lf",
              inst_config->bias_ron_limit);
    espdr_msg("\t\t\tBIAS mean limit: %.2lf",
              inst_config->bias_mean_limit);
    espdr_msg("\t\t\tBIAS ksigma: %.2lf",
              inst_config->bias_ksigma);
    espdr_msg("\t\t\tBIAS sigma clipping method: %s",
              inst_config->bias_sigma_clipping_method);
    espdr_msg("\t\t\tSTAT sigma clipping ksigma: %.2lf",
              inst_config->stat_ksigma);
    espdr_msg("\t\t\tSTAT sigma clipping method: %s",
              inst_config->stat_sigma_clipping_method);
    espdr_msg("\t\t\tSTAT sigma clipping max iter: %d",
              inst_config->stat_max_iter);
    espdr_msg("\t\t\tBIAS residuals mean limit: %.2lf",
              inst_config->bias_residuals_mean_limit);
    espdr_msg("\t\t\tBIAS residuals stdev limit: %.2lf",
              inst_config->bias_residuals_stdev_limit);
    espdr_check_error_code("espdr_print_bias_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_single_bias_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_BIAS parameters:");
    espdr_msg("\t\t\tBIAS RON limit: %.2lf",
              inst_config->bias_ron_limit);
    espdr_msg("\t\t\tBIAS mean limit: %.2lf",
              inst_config->bias_mean_limit);
    espdr_msg("\t\t\tSTAT sigma clipping ksigma: %.2lf",
              inst_config->stat_ksigma);
    espdr_msg("\t\t\tSTAT sigma clipping method: %s",
              inst_config->stat_sigma_clipping_method);
    espdr_msg("\t\t\tSTAT sigma clipping max iter: %d",
              inst_config->stat_max_iter);
    espdr_check_error_code("espdr_print_single_bias_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_dark_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_DARK parameters:");
    espdr_msg("\t\t\tRaw DARK limit number: %d",
              inst_config->raw_dark_limit_nb);
    espdr_msg("\t\t\tDARK current limit [e-]: %.2lf",
              inst_config->dark_current_limit_el);
    espdr_msg("\t\t\tHot pixels limit: %d",
              inst_config->hot_pixels_limit);
    espdr_msg("\t\t\tCosmics limit per pixel: %.2lf",
              inst_config->cosmics_limit_per_pixel);
    espdr_msg("\t\t\tDARK sigma clipping ksigma: %.2lf",
              inst_config->dark_ksigma);
    espdr_msg("\t\t\tDARK sigma clipping method: %s",
              inst_config->dark_sigma_clipping_method);
    espdr_msg("\t\t\tHOTPIXELS sigma clipping ksigma: %.2lf",
              inst_config->hotpixels_ksigma);
    espdr_msg("\t\t\tHOTPIXELS sigma clipping method: %s",
              inst_config->hotpixels_sigma_clipping_method);
    espdr_msg("\t\t\tHOTPIXELS sigma clipping max iter: %d",
              inst_config->hotpixels_max_iter);
    espdr_check_error_code("espdr_print_dark_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_ledff_parameters_inst_config(espdr_inst_config *inst_config) {
    espdr_msg("\t\tCAL_LED_FF parameters:");
    espdr_msg("\t\t\tRaw LED_FF per Texp limit number: %d",
              inst_config->raw_led_ff_per_texp_limit_nb);
    espdr_msg("\t\t\tRaw LED_FF different Texp: %d",
              inst_config->raw_led_ff_different_texp);
    espdr_msg("\t\t\tLED_FF sigma clipping ksigma: %.2lf",
              inst_config->led_ff_ksigma);
    espdr_msg("\t\t\tLED_FF sigma clipping method: %s",
              inst_config->led_ff_sigma_clipping_method);
    espdr_msg("\t\t\tBADPIXELS sigma clipping ksigma: %.2lf",
              inst_config->badpixels_ksigma);
    espdr_msg("\t\t\tBADPIXELS sigma clipping method: %s",
              inst_config->badpixels_sigma_clipping_method);
    espdr_msg("\t\t\tBADPIXELS sigma clipping max iter: %d",
              inst_config->badpixels_max_iter);
    espdr_msg("\t\t\tBad pixels limit: %d",
              inst_config->bad_pixels_limit);
    espdr_msg("\t\t\tFlux min limit: %.2lf",
              inst_config->flux_min_limit);
    espdr_msg("\t\t\tFlux max limit: %.2lf",
              inst_config->flux_max_limit);
    espdr_msg("\t\t\tGain tolerance: %.2lf",
              inst_config->gain_tolerance);
    espdr_msg("\t\t\tDead pixel flux limit: %.2lf",
              inst_config->dead_pixel_flux_limit);
    espdr_check_error_code("espdr_print_ledff_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_orderdef_parameters_inst_config(espdr_inst_config *inst_config,
                                            const int ext_nb) {
    espdr_msg("\t\tCAL_ORDERDEF parameters:");
    espdr_msg("\t\t\tSaturation limit [ADU]: %.2lf",
              inst_config->satur_limit_adu);
    espdr_msg("\t\t\tResiduals min limit [pxl]: %.3lf",
              inst_config->res_min_limit);
    espdr_msg("\t\t\tResiduals max limit [pxl]: %.3lf",
              inst_config->res_max_limit);
    espdr_msg("\t\t\tResiduals stdev limit [pxl]: %.3lf",
              inst_config->res_stdev_limit);
    espdr_msg("\t\t\tResiduals ksigma: %.3lf",
              inst_config->res_ksigma);
    espdr_msg("\t\t\tResiduals sigma clipping method: %s",
              inst_config->res_sigma_clipping_method);
    espdr_msg("\t\t\tNumber of slices: %d",
              inst_config->slices_nb);
    espdr_msg("\t\t\tInter-slice distance: %d",
              inst_config->inter_slice_dist);
    espdr_msg("\t\t\tSlice width: %d",
              inst_config->slice_width);
    espdr_msg("\t\t\tOrder width: %d",
              inst_config->order_width);
    espdr_msg("\t\t\tNumber of slices per physical order: %d",
              inst_config->slices_nb_per_phys_order);
    espdr_msg("\t\t\tNumber of background pixels: %d",
              inst_config->background_pxls);
    espdr_msg("\t\t\tNumber of fibres: %d",
              inst_config->fibres_nb);
    espdr_msg("\t\t\tNumber of orders per fibre per extension:");
    int index = 0;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < ext_nb; j++) {
            espdr_msg("\t\t\t\tOn fiber %c, ext %d: %d",
                      fibre_name[i], j,
                      inst_config->orders_nb[index]);
            index++;
        }
    }
    espdr_msg("\t\t\tNumber of the first order per fibre per extension:");
    index = 0;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < ext_nb; j++) {
            espdr_msg("\t\t\t\tOn fiber %c, ext %d: %d",
                      fibre_name[i], j,
                      inst_config->order_phys_nb[index]);
            index++;
        }
    }
    espdr_msg("\t\t\tOrders limits per fibre per extension");
    index = 0;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < ext_nb; j++) {
            espdr_msg("\t\t\t\tOn fiber %c, ext %d: start: %d, end: %d",
                      fibre_name[i], j,
                      inst_config->order_start[index],
                      inst_config->order_end[index]);
            index++;
        }
    }
    espdr_msg("\t\t\tResiduals sigma clipping ksigma: %.2lf",
              inst_config->res_ksigma);
    espdr_msg("\t\t\tResiduals sigma clipping method: %s",
              inst_config->res_sigma_clipping_method);
    espdr_msg("\t\t\tResiduals sigma clipping max iter: %d",
              inst_config->res_max_iter);
    espdr_msg("\t\t\tCluster distance for clumps detection x: %d & y: %d",
              inst_config->cluster_dist_x,
              inst_config->cluster_dist_y);
    espdr_msg("\t\t\tOrder minimum distance from upper and lower CCD borders: %d",
              inst_config->min_order_y_dist);
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("\t\t\tMinimum order length for fibre %c: %d",
                  fibre_name[i], inst_config->min_order_len[i]);
        espdr_msg("\t\t\tMinimum number of pixels where order has a minimal length for fibre %c: %d",
                  fibre_name[i], inst_config->min_width_pixels_nb[i]);
    }
    espdr_msg("\t\t\tMinimum order width: %d",
              inst_config->min_order_width);
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("\t\t\tMinimum order x coordinate for fibre %c: %d",
                  fibre_name[i], inst_config->min_order_x[i]);
        espdr_msg("\t\t\tMaximum order y coordinate for fibre %c: %d",
                  fibre_name[i], inst_config->max_order_y[i]);
    }
    espdr_msg("\t\t\tOrders detection start per fibre per extension");
    index = 0;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < ext_nb; j++) {
            espdr_msg("\t\t\t\tOn fiber %c, ext %d: %d",
                      fibre_name[i], j,
                      inst_config->orders_detection_start[index]);
            index++;
        }
    }
    espdr_msg("\t\t\tStep in orders polynomial fit: %d",
              inst_config->order_fit_step);
    espdr_msg("\t\t\tPolynomial degree in order fit: %d",
              inst_config->order_fit_poly_deg);
    espdr_msg("\t\t\tFlux limit for order detection: %.2lf",
              inst_config->order_flux_limit);
    espdr_msg("\t\t\tFlux factor for order detection per fibre");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("\t\t\t\tOn fiber %c: %.2f",
                  fibre_name[i],
                  inst_config->order_flux_factor[i]);
    }
    espdr_check_error_code("espdr_print_orderdef_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_extraction_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tEXTRACTION parameters:");
    espdr_msg("\t\t\tFraction of order width for background calculation: %.2lf",
              inst_config->extraction_bkgr_calc_fraction);
    espdr_msg("\t\t\tFraction of valid pixels for bad ones interpolation: %.2lf",
              inst_config->extraction_fraction_valid_pixels);
    espdr_msg("\t\t\tExtraction method: %s",
              inst_config->extraction_method);
    espdr_msg("\t\t\tExtraction ksigma TUN: %.2lf",
              inst_config->extraction_ksigma_tun);
    espdr_msg("\t\t\tExtraction ksigma THAR: %.2lf",
              inst_config->extraction_ksigma_thar);
    espdr_msg("\t\t\tExtraction ksigma FP: %.2lf",
              inst_config->extraction_ksigma_fp);
    espdr_msg("\t\t\tExtraction ksigma LFC: %.2lf",
              inst_config->extraction_ksigma_lfc);
    espdr_msg("\t\t\tExtraction ksigma OBJECT: %.2lf",
              inst_config->extraction_ksigma_object);
    espdr_msg("\t\t\tExtraction ksigma SKY: %.2lf",
              inst_config->extraction_ksigma_sky);
    espdr_msg("\t\t\tExtraction ksigma DARK: %.2lf",
              inst_config->extraction_ksigma_dark);
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("\t\t\tExtraction window size for fibre %c: %d",
                  fibre_name[i],
                  inst_config->extraction_window_size[i]);
    }
    espdr_msg("\t\t\tExtraction tolerance rejection: %.2lf",
              inst_config->tolerance_rejection);
    espdr_check_error_code("espdr_print_extraction_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_flat_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_FLAT parameters:");
    espdr_msg("\t\t\tRaw FLAT limit number: %d",
              inst_config->raw_flat_limit_nb);
    espdr_msg("\t\t\tFLAT current limit [e-]: %.2lf",
              inst_config->flat_current_limit_el);
    espdr_msg("\t\t\tFLAT SNR limit: %.2lf",
              inst_config->flat_snr_limit);
    espdr_msg("\t\t\tFLAT RMS limit: %.2lf",
              inst_config->flat_rms_limit);
    espdr_msg("\t\t\tFLAT background / max_flux limit: %.5f",
              inst_config->flat_bkgr_limit);
    espdr_msg("\t\t\tBackground measurement grid size x: %d",
              inst_config->bkgr_grid_size_x);
    espdr_msg("\t\t\tBackground measurement grid size y: %d",
              inst_config->bkgr_grid_size_y);
    espdr_msg("\t\t\tBackground measurement minimun level detection factor: %d",
              inst_config->minlevel_factor);
    espdr_msg("\t\t\tFLAT sigma clipping ksigma: %.2lf",
              inst_config->flat_ksigma);
    espdr_msg("\t\t\tFLAT sigma clipping method: %s",
              inst_config->flat_sigma_clipping_method);
    espdr_msg("\t\t\tFlat and blaze calculation method: %s",
              inst_config->blaze_flat_calc_method);
    espdr_msg("\t\t\tFlat and blaze calculation polynomial degree: %d",
              inst_config->blaze_flat_calc_poly_deg);
    espdr_msg("\t\t\tFlat and blaze calculation sliding box size: %d",
              inst_config->blaze_flat_calc_slide_box);
    espdr_msg("\t\t\tFlat and blaze calculation clip: %.2lf",
              inst_config->blaze_flat_calc_clip);
    espdr_msg("\t\t\tBlaze peaks fit method: %s",
              inst_config->blaze_peaks_fit_method);
    espdr_msg("\t\t\tBlaze peaks polynomial fit degree: %d",
              inst_config->blaze_peaks_fit_poly_deg);
    espdr_msg("\t\t\tBlaze peaks fit sliding box size: %d",
              inst_config->blaze_peaks_fit_slide_box);
    espdr_check_error_code("espdr_print_flat_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_wave_fp_parameters_inst_config(espdr_inst_config *inst_config) {
    
    
    espdr_msg("\t\tCAL_WAVE_FP parameters:");
    espdr_msg("\t\t\tFit window size: %d",
              inst_config->fp_fit_window_size);
    espdr_msg("\t\t\tTolerance fit window size: %d",
              inst_config->fp_fit_tolerance_window_size);
    espdr_msg("\t\t\tFit NGauss flag: %d",
              inst_config->fp_fit_ngauss_flag);
    espdr_msg("\t\t\tSearch window size: %d",
              inst_config->fp_search_window_size);
    espdr_msg("\t\t\tDelta FP tolerance: %.2f",
              inst_config->delta_fp_tolerance);
    espdr_msg("\t\t\tDelta FP peak limit: %.2f",
              inst_config->delta_fp_peak_limit);
    espdr_msg("\t\t\tCCD border cut: %d",
              inst_config->fp_ccd_border_cut);
    espdr_msg("\t\t\tFlux threshold: %.2lf",
              inst_config->fp_flux_threshold);
    espdr_msg("\t\t\tNoise threshold: %.2lf",
              inst_config->fp_noise_threshold);
    espdr_msg("\t\t\tPeak flux threshold: %.2lf",
              inst_config->fp_peak_flux_threshold);
    espdr_msg("\t\t\tPeak rms threshold: %.2lf",
              inst_config->fp_peak_rms_threshold);
    espdr_msg("\t\t\tSpectral line QC: %.2lf",
              inst_config->fp_spectral_line_qc);
    espdr_msg("\t\t\tMin total lines: %d",
              inst_config->fp_min_lines_total);
    espdr_msg("\t\t\tAdd fake FP peaks flag: %d",
              inst_config->fp_add_fake_peaks);
    espdr_check_error_code("espdr_print_wave_fp_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_wave_thar_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_WAVE_THAR parameters:");
    espdr_msg("\t\t\tThar fit window size: %d",
              inst_config->thar_thar_fit_window_size);
    espdr_msg("\t\t\tFp fit window size: %d",
              inst_config->thar_fp_fit_window_size);
    espdr_msg("\t\t\tThar fit tolerance window size: %d",
              inst_config->thar_fit_tolerance_window_size);
    espdr_msg("\t\t\tCCD border cut: %d",
              inst_config->thar_ccd_border_cut);
    espdr_msg("\t\t\tFlux threshold: %.2lf",
              inst_config->thar_flux_threshold);
    espdr_msg("\t\t\tNoise threshold: %.2lf",
              inst_config->thar_noise_threshold);
    espdr_msg("\t\t\tPeak flux threshold: %.2lf",
              inst_config->thar_peak_flux_threshold);
    espdr_msg("\t\t\tTh lines flux ratio for fibre A: %.2f <-> %.2f",
              inst_config->thar_min_flux_ratio[0],
              inst_config->thar_max_flux_ratio[0]);
    espdr_msg("\t\t\tTh lines flux ratio for fibre B: %.2f <-> %.2f",
              inst_config->thar_min_flux_ratio[1],
              inst_config->thar_max_flux_ratio[1]);
    espdr_msg("\t\t\tSpectral line QC: %.2lf",
              inst_config->thar_spectral_line_qc);
    espdr_msg("\t\t\tInitial FP mirrors distance D [A]: %.2f",
              inst_config->fp_d_angstrom);
    espdr_msg("\t\t\tD fit polynomial degree: %d",
              inst_config->d_fit_poly_deg);
    espdr_msg("\t\t\tD first fit (big) residuals limit: %.2f",
              inst_config->d_fit_big_res_limit);
    espdr_msg("\t\t\tD second fit (small) residuals limit: %.2f",
              inst_config->d_fit_small_res_limit);
    espdr_msg("\t\t\tD fit kernel polynomial degree: %d",
              inst_config->d_fit_kernel_poly_deg);
    espdr_msg("\t\t\tD fit kernel fwhm: %.2f",
              inst_config->d_fit_kernel_fwhm);
    espdr_msg("\t\t\tMaximum CHI2 allowed in wavelegth solution for THAR: %.2lf",
              inst_config->thar_sol_max_chi2);
    espdr_msg("\t\t\tMaximum RMS allowed in wavelegth solution for THAR: %.2lf",
              inst_config->thar_sol_max_rms);
    espdr_msg("\t\t\tMinimum mean resolution allowed in wavelegth solution for THAR: %.2lf",
              inst_config->wave_min_resol);
    espdr_msg("\t\t\tMin lines per order in wavelength solution: %d",
              inst_config->wave_sol_min_lines_per_order);
    espdr_msg("\t\t\tMin total lines: %d",
              inst_config->thar_min_lines_total);
    espdr_msg("\t\t\tDrift computation on THAR-FP switch = %d",
              inst_config->thar_drift_th_fp_sw);
    espdr_msg("\t\t\tDrift computation on FP-THAR switch = %d",
              inst_config->thar_drift_fp_th_sw);
    espdr_msg("\t\t\tDrift CHI2 limit = %.2lf",
              inst_config->thar_drift_chi2_limit);
    espdr_msg("\t\t\tAnalytically linking of orders with no overlapping wavelengths switch = %d",
              inst_config->wave_orders_analytical_link_sw);
    espdr_check_error_code("espdr_print_wave_thar_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_wave_th_drift_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_WAVE_TH_drift parameters:");
    espdr_msg("\t\t\tThar fit window size: %d",
              inst_config->thar_thar_fit_window_size);
    espdr_msg("\t\t\tThar fit tolerance window size: %d",
              inst_config->thar_fit_tolerance_window_size);
    espdr_msg("\t\t\tCCD border cut: %d",
              inst_config->thar_ccd_border_cut);
    espdr_msg("\t\t\tFlux threshold: %.2lf",
              inst_config->thar_flux_threshold);
    espdr_msg("\t\t\tNoise threshold: %.2lf",
              inst_config->thar_noise_threshold);
    espdr_msg("\t\t\tPeak flux threshold: %.2lf",
              inst_config->thar_peak_flux_threshold);
    espdr_msg("\t\t\tTh lines flux ratio for fibre A: %.2f <-> %.2f",
              inst_config->thar_min_flux_ratio[0],
              inst_config->thar_max_flux_ratio[0]);
    espdr_msg("\t\t\tTh lines flux ratio for fibre B: %.2f <-> %.2f",
              inst_config->thar_min_flux_ratio[1],
              inst_config->thar_max_flux_ratio[1]);
    espdr_msg("\t\t\tSpectral line QC: %.2lf",
              inst_config->thar_spectral_line_qc);
    espdr_msg("\t\t\tMinimum mean resolution allowed in wavelegth solution for THAR: %.2lf",
              inst_config->wave_min_resol);
    espdr_msg("\t\t\tMin lines per order in wavelength solution: %d",
              inst_config->wave_sol_min_lines_per_order);
    espdr_msg("\t\t\tMin total lines: %d",
              inst_config->thar_min_lines_total);
    espdr_msg("\t\t\tTH lamp offset AR: %.2lf",
              inst_config->th_lamp_offset_ar);
    espdr_msg("\t\t\tTH lamp offset AR1: %.2lf",
              inst_config->th_lamp_offset_ar1);
    espdr_msg("\t\t\tTH lamp offset AR2: %.2lf",
              inst_config->th_lamp_offset_ar2);
    espdr_msg("\t\t\tMin total lines for fibre A: %d and B: %d",
              inst_config->th_drift_min_lines_total[0],
              inst_config->th_drift_min_lines_total[1]);
    espdr_msg("\t\t\tMax saturated lines: %d",
              inst_config->th_drift_max_satur_lines);
    espdr_msg("\t\t\tDrift fit degrees X: %d and Y: %d",
              inst_config->th_drift_fit_deg_x,
              inst_config->th_drift_fit_deg_y);
    espdr_msg("\t\t\tKsigma for residuals rejection in the drift fit: %.2f",
              inst_config->th_drift_fit_ksigma);
    espdr_msg("\t\t\tDrift fit residuals MAD limit: %.3f",
              inst_config->th_drift_fit_max_mad);
    espdr_msg("\t\t\tLimit for fraction of rejected lines in the drift fit: %.3f",
              inst_config->th_drift_max_reject_lines_fraction);
    espdr_check_error_code("espdr_print_wave_th_drift_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_wave_lfc_parameters_inst_config(espdr_inst_config *inst_config) {
    espdr_msg("\t\tCAL_WAVE_LFC parameters:");
    espdr_msg("\t\t\tLFC anchor frequency: %.6f",
              inst_config->lfc_anchor_freq);
    espdr_msg("\t\t\tLFC repetition rate: %.2f",
              inst_config->lfc_repetition_rate);
    espdr_msg("\t\t\tLFC spectral line QC: %.6f",
              inst_config->lfc_spectral_line_qc);
    espdr_msg("\t\t\tMaximum CHI2 allowed in wavelegth solution for LFC: %.2lf",
              inst_config->lfc_sol_max_chi2);
    espdr_msg("\t\t\tMaximum RMS allowed in wavelegth solution for LFC: %.2lf",
              inst_config->lfc_sol_max_rms);
    espdr_msg("\t\t\tMin total lines: %d",
              inst_config->lfc_min_lines_total);
    espdr_msg("\t\t\tMin lines per order in wavelength solution:");
    int index = 0;
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("\t\t\t\tFor fiber %c: %d",
                  fibre_name[i],
                  inst_config->lfc_sol_min_lines_per_order[index]);
        index++;
    }
    espdr_check_error_code("espdr_print_wave_lfc_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_eff_ab_parameters_inst_config(espdr_inst_config *inst_config) {
    espdr_msg("\t\tCAL_EFF_AB parameters:");
    espdr_msg("\t\t\tSNR limit: %.2lf",
              inst_config->eff_ab_snr_limit);
    espdr_msg("\t\t\tBackground / max_flux limit: %.2lf",
              inst_config->eff_ab_bkgr_limit);
    espdr_msg("\t\t\tBackground measurement grid size x: %d",
              inst_config->eff_ab_bkgr_grid_size_x);
    espdr_msg("\t\t\tBackground measurement grid size y: %d",
              inst_config->eff_ab_bkgr_grid_size_y);
    espdr_msg("\t\t\tDegree of polynomial fit: %d",
              inst_config->eff_ab_poly_deg);
    espdr_msg("\t\t\tRelative efficiency minimum limit: %.2lf",
              inst_config->rel_eff_min_limit);
    espdr_msg("\t\t\tRelative efficiency maximum limit: %.2lf",
              inst_config->rel_eff_max_limit);
    espdr_check_error_code("espdr_print_eff_ab_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_flux_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tCAL_FLUX parameters:");
    espdr_msg("\t\t\tEff interpolation method: %s",
              inst_config->flux_eff_interp_method);
    espdr_msg("\t\t\tInterpolation degree: %d",
              inst_config->flux_poly_deg);
    espdr_msg("\t\t\tSNR limit: %.2lf",
              inst_config->flux_snr_limit);
    espdr_msg("\t\t\tAbsolut minimal efficiency minimum limit: %.3lf",
              inst_config->abs_eff_min_limit);
    espdr_msg("\t\t\tAbsolut maximal efficiency minimum limit: %.3lf",
              inst_config->abs_eff_max_limit);
    espdr_check_error_code("espdr_print_flux_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_science_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tSCI_RED parameters:");
    espdr_msg("\t\t\tBackground measurement grid size x: %d",
              inst_config->sci_bkgr_grid_size_x);
    espdr_msg("\t\t\tBackground measurement grid size y: %d",
              inst_config->sci_bkgr_grid_size_y);
    espdr_msg("\t\t\tSNR averaging window: %d",
              inst_config->snr_averaging_window);
    espdr_msg("\t\t\tBERV type init: %d",
              inst_config->berv_type_init);
    espdr_msg("\t\t\tTEL geo longitude: %lf",
              inst_config->tel_geolon);
    espdr_msg("\t\t\tTEL geo latitude: %lf",
              inst_config->tel_geolat);
    espdr_msg("\t\t\tTEL geo elevation: %lf",
              inst_config->tel_geoelev);
    espdr_msg("\t\t\tMask width: %.2lf",
              inst_config->mask_width);
    espdr_msg("\t\t\tLog Lambda init: %.2lf",
              inst_config->log_lambda_ini);
    espdr_msg("\t\t\tDelta Log Lambda: %.2lf",
              inst_config->delta_log_lambda);
    espdr_msg("\t\t\tLog Lambda end: %.2lf",
              inst_config->log_lambda_end);
    espdr_msg("\t\t\tDefault spectral type : %s",
              inst_config->default_spec_type);
    espdr_msg("\t\t\tRadial velocity table range = %.2lf",
              inst_config->rv_range);
    espdr_msg("\t\t\tRadial velocity table step = %.2lf",
              inst_config->rv_step);
    espdr_msg("\t\t\tTelescope surface = %.2lf",
              inst_config->tel_surface);
    espdr_msg("\t\t\tSky correction method = %s",
              inst_config->sci_sky_corr_method);
    if (strcmp(inst_config->sci_sky_corr_method, "smoothed") == 0) {
        espdr_msg("\t\t\tSky subtraction sliding box size = %d",
                  inst_config->sci_sky_sub_sliding_box_size);
    }
    if (strcmp(inst_config->sci_sky_corr_method, "oh-corr") == 0) {
        espdr_msg("\t\t\tOH correction lines nb limit = %d",
                  inst_config->oh_corr_lines_nb_limit);
        espdr_msg("\t\t\tOH correction ratio lower & upper limits = %f & %f",
                  inst_config->oh_corr_ratio_lower_limit,
                  inst_config->oh_corr_ratio_upper_limit);
        espdr_msg("\t\t\tOH correction ratio MAD limit = %f",
                  inst_config->oh_corr_ratio_mad_limit);
    }
    espdr_msg("\t\t\tLA Cosmic switch for OBJ,SKY or OBJ,DARK = %d",
              inst_config->lacosmic_sky_sw);
    espdr_msg("\t\t\tLA Cosmic switch for OBJ,FP or OBJ,THAR or OBJ,LFC = %d",
              inst_config->lacosmic_calib_sw);
    if ((inst_config->lacosmic_sky_sw == 1) || (inst_config->lacosmic_calib_sw == 1)) {
        espdr_msg("\t\t\tLA Cosmic x size of the post filtering kernel = %d",
                  inst_config->lacosmic_post_filter_x);
        espdr_msg("\t\t\tLA Cosmic y size of the post filtering kernel = %d",
                  inst_config->lacosmic_post_filter_y);
        espdr_msg("\t\t\tLA Cosmic post filtering mode = %s",
                  inst_config->lacosmic_post_filter_mode);
        espdr_msg("\t\t\tLA Cosmic Poisson fluctuation threshold = %.2f",
                  inst_config->lacosmic_sigma_lim);
        espdr_msg("\t\t\tLA Cosmic minimum contrast = %.2f",
                  inst_config->lacosmic_f_lim);
        espdr_msg("\t\t\tLA Cosmic maximum number of iterations = %d",
                  inst_config->lacosmic_max_iter);
    }
    espdr_check_error_code("espdr_print_science_parameters_inst_config");
    return cpl_error_get_code();
}

static cpl_error_code
espdr_print_drift_parameters_inst_config(espdr_inst_config *inst_config) {
    
    espdr_msg("\t\tDRIFT parameters:");
    espdr_msg("\t\t\tDrift ksigma: %.2lf",
              inst_config->drift_ksigma);
    espdr_msg("\t\t\tDrift method THAR: %s",
              inst_config->drift_method_thar);
    espdr_msg("\t\t\tDrift method FP: %s",
              inst_config->drift_method_fp);
    espdr_msg("\t\t\tDrift method LFC: %s",
              inst_config->drift_method_lfc);
    espdr_msg("\t\t\tDrift computation space: %s",
              inst_config->drift_space);
    espdr_msg("\t\t\tDrift min flux threshold: %.2lf",
              inst_config->drift_min_flux_threshold);
    espdr_msg("\t\t\tDrift max flux threshold: %.2lf",
              inst_config->drift_max_flux_threshold);
    espdr_msg("\t\t\tDrift min flux ratio: %.2lf",
              inst_config->drift_min_flux_ratio);
    espdr_msg("\t\t\tDrift max flux ratio: %.2lf",
              inst_config->drift_max_flux_ratio);
    espdr_msg("\t\t\tDrift mean threshold: %.5lf",
              inst_config->drift_mean_threshold);
    espdr_msg("\t\t\tDrift mean error threshold: %.5lf",
              inst_config->drift_mean_err_threshold);
    espdr_msg("\t\t\tDrift CHI2 threshold = %.2lf",
              inst_config->drift_chi2_threshold);
    espdr_check_error_code("espdr_print_drift_parameters_inst_config");
    return cpl_error_get_code();
}
/*---------------------------------------------------------------------------*/
/**
 @brief print the inst_config parameters
 @param     inst_config     instrument config structure
 @param     ext_nb          extensiosn number for the instrument
 @return   CPL_ERROR_NONE iff OK else NULL
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_inst_config_print(espdr_inst_config *inst_config,
                                                  int ext_nb,
                                                  const char *context) {
    
    int i;
    
    espdr_msg("\t%s configuration (prefix: %s):",
              inst_config->instrument,
              inst_config->prefix);
    
    espdr_print_general_parameters_inst_config(inst_config);
    
    if (strcmp(context, "espdr.espdr_mbias") == 0) {
        espdr_print_bias_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_single_bias") == 0) {
        espdr_print_single_bias_parameters_inst_config(inst_config);
    }
    
    if (inst_config->cte_flag == 1) {
        espdr_msg("\t\tCTE parameters:");
        espdr_msg("\t\t\tCTE ALPHA & BETA per extension:");
        for (i = 0; i < ext_nb; i++) {
            espdr_msg("\t\t\t\tOn ext %d: %f %f", i,
                      inst_config->cte_alpha[i],
                      inst_config->cte_beta[i]);
        }
    }
    
    if (strcmp(context, "espdr.espdr_mdark") == 0) {
        espdr_print_dark_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_led_ff") == 0) {
        espdr_print_ledff_parameters_inst_config(inst_config);
    }
    
    if ((strcmp(context, "espdr.espdr_orderdef") == 0) ||
        (strcmp(context, "espdr.espdr_mflat") == 0) ||
        (strcmp(context, "espdr.espdr_wave_FP") == 0) ||
        (strcmp(context, "espdr.espdr_wave_THAR") == 0) ||
        (strcmp(context, "espdr.espdr_wave_FP") == 0) ||
        (strcmp(context, "espdr.espdr_wave_LFC") == 0) ||
        (strcmp(context, "espdr.espdr_cal_contam") == 0) ||
        (strcmp(context, "espdr.espdr_cal_eff_ab") == 0) ||
        (strcmp(context, "espdr.espdr_cal_flux") == 0) ||
        (strcmp(context, "espdr.espdr_sci_red") == 0)) {
        espdr_msg("\t\tSaturation parameters:");
        espdr_msg("\t\t\tCosmics part to be rejected in max flux calculation: %.8lf",
                  inst_config->image_cosmics_part);
    }
    
    if (strcmp(context, "espdr.espdr_sci_red") == 0) {
        espdr_msg("\t\t\tSaturated THAR lines part to be rejected in max flux calculation: %.8lf",
                  inst_config->image_thar_part);
    }
    
    if (strcmp(context, "espdr.espdr_orderdef") == 0) {
        espdr_print_orderdef_parameters_inst_config(inst_config, ext_nb);
    }
    
    if ((strcmp(context, "espdr.espdr_mflat") == 0) ||
        (strcmp(context, "espdr.espdr_wave_FP") == 0) ||
        (strcmp(context, "espdr.espdr_wave_THAR") == 0) ||
        (strcmp(context, "espdr.espdr_wave_THAR_THAR") == 0) ||
        (strcmp(context, "espdr.espdr_wave_LFC") == 0) ||
        (strcmp(context, "espdr.espdr_wave_LFC_LFC") == 0) ||
        (strcmp(context, "espdr.espdr_cal_contam") == 0) ||
        (strcmp(context, "espdr.espdr_cal_eff_ab") == 0) ||
        (strcmp(context, "espdr.espdr_cal_flux") == 0) ||
        (strcmp(context, "espdr.espdr_sci_red") == 0)) {
        espdr_print_extraction_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_mflat") == 0) {
        espdr_print_flat_parameters_inst_config(inst_config);
    }
    
    if ((strcmp(context, "espdr.espdr_wave_FP") == 0) ||
        (strcmp(context, "espdr.espdr_wave_THAR") == 0) ||
        (strcmp(context, "espdr.espdr_wave_TH_drift") == 0) ||
        (strcmp(context, "espdr.espdr_wave_LFC") == 0)) {
        espdr_msg("\t\tCAL_WAVE general parameters:");
        espdr_msg("\t\t\tMaximum allowed line position uncertainty: %.2lf",
                  inst_config->line_max_sig_x0);
        espdr_msg("\t\t\tMinimum allowed line position uncertainty: %.3lf",
                  inst_config->line_min_sig_x0);
        espdr_msg("\t\t\tMaximum allowed line FWHM: %.2lf",
                  inst_config->line_max_fwhm);
        espdr_msg("\t\t\tMinimum allowed peak flux: %.2lf",
                  inst_config->line_min_flux_el);
        espdr_msg("\t\t\tPolynome degree for wavelength solution: %d",
                  inst_config->wave_sol_poly_deg);
        espdr_msg("\t\t\tWavelength solution: FP lines rejection coeffs: %.2f & %.2f",
                  inst_config->wave_sol_FP_reject_alpha,
                  inst_config->wave_sol_FP_reject_beta);
    }
    
    if (strcmp(context, "espdr.espdr_wave_FP") == 0) {
        espdr_print_wave_fp_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_wave_THAR") == 0) {
        espdr_print_wave_thar_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_wave_TH_drift") == 0) {
        espdr_print_wave_th_drift_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_wave_LFC") == 0) {
        espdr_print_wave_lfc_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_cal_contam") == 0) {
        espdr_msg("\t\tCAL_CONTAM parameters:");
        espdr_msg("\t\t\tBackground measurement grid size x: %d",
                  inst_config->contam_bkgr_grid_size_x);
        espdr_msg("\t\t\tBackground measurement grid size y: %d",
                  inst_config->contam_bkgr_grid_size_y);
        espdr_msg("\t\t\tS2DA flux limit: %.2lf",
                  inst_config->contam_flux_limit);
    }
    
    if (strcmp(context, "espdr.espdr_cal_eff_ab") == 0) {
        espdr_print_eff_ab_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_cal_flux") == 0) {
        espdr_print_flux_parameters_inst_config(inst_config);
    }
    
    if (strcmp(context, "espdr.espdr_sci_red") == 0) {
        espdr_print_science_parameters_inst_config(inst_config);
    }
    
    if ((strcmp(context, "espdr.espdr_wave_TH_drift") == 0) ||
        (strcmp(context, "espdr.espdr_wave_THAR") == 0) ||
        (strcmp(context, "espdr.espdr_wave_LFC") == 0) ||
        (strcmp(context, "espdr.espdr_sci_red") == 0)) {
        espdr_print_drift_parameters_inst_config(inst_config);
    }
    
    espdr_msg("\t\tUsed header KWs names:");
    espdr_msg("\t\t\tKeyword containing the dpr type : %s",
              inst_config->dpr_type_kw);
    espdr_msg("\t\t\tDetector binning: %s x %s",
              inst_config->det_binx_kw,
              inst_config->det_biny_kw);
    espdr_msg("\t\t\tActual integration time: %s",
              inst_config->Texp_kw);
    if (strcmp(context, "espdr.espdr_led_ff") == 0) {
        espdr_msg("\t\t\tCONAD KW per output: %s <output_nr> %s",
                  inst_config->conad_kw_first_part,
                  inst_config->conad_kw_last_part);
    }
    if ((strcmp(context, "espdr.espdr_cal_flux") == 0) ||
        (strcmp(context, "espdr.espdr_sci_red") == 0)) {
        espdr_msg("\t\t\tNormalized mean exposure time: %s",
                  inst_config->tmmean_kw);
        espdr_msg("\t\t\tTEL alpha: %s",
                  inst_config->tel_alpha_kw);
        espdr_msg("\t\t\tTEL delta: %s",
                  inst_config->tel_delta_kw);
        espdr_msg("\t\t\tTEL proper motion alpha: %s",
                  inst_config->tel_pma_kw);
        espdr_msg("\t\t\tTEL proper motion delta: %s",
                  inst_config->tel_pmd_kw);
        espdr_msg("\t\t\tTEL date obs: %s",
                  inst_config->tel_date_obs_kw);
        espdr_msg("\t\t\tKeyword containing the spectral type: %s",
                  inst_config->spec_type_kw);
        espdr_msg("\t\t\tKeyword containing the zem value (!= 0 for quazars): %s",
                  inst_config->zem_kw);
        espdr_msg("\t\t\tKeyword containing the target name: %s",
                  inst_config->targ_name_kw);
        espdr_msg("\t\t\tKeyword containing the wave calibration source: %s",
                  inst_config->wavecal_src_kw);
        espdr_msg("\t\t\tKeyword containing the radial velocity: %s",
                  inst_config->rv_center_kw);
        espdr_msg("\t\t\tAirmass start: %s, end: %s",
                  inst_config->airm_start_kw, inst_config->airm_end_kw);
        espdr_msg("\t\t\tPressure start: %s, end: %s",
                  inst_config->pres_start_kw, inst_config->pres_end_kw);
        espdr_msg("\t\t\tTemperature: %s",
                  inst_config->temp_kw);
        espdr_msg("\t\t\tSeeing KW: %s <telescope_nr> %s",
                  inst_config->seeing_kw_first_part,
                  inst_config->seeing_kw_last_part);
    }
    if ((strcmp(context, "espdr.espdr_flux") == 0) ||
        (strcmp(context, "espdr.espdr_sci_red") == 0)) {
        espdr_msg("\t\t\tLFC repetition rate: %s",
                  inst_config->rep_rate_kw);
        espdr_msg("\t\t\tLFC anchor frequency: %s",
                  inst_config->anchor_freq_kw);
    }
    espdr_check_error_code("espdr_parameters_inst_config_print");
    return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the inst_config structure
 @param     inst_config     instrument config structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_inst_config_delete(espdr_inst_config *inst_config)
{
    
    cpl_free(inst_config);
    inst_config = NULL;
    espdr_check_error_code("espdr_parameters_inst_config_delete");
    return cpl_error_get_code();
}

