/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2015-01-12 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_extraction.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief Simple extraction of spectrum on all fibers
 @param         input_images            imagelist of images to extract
 @param         hot_bad_pixels_mask     mask of hot & bad pixels
 @param         saturation mask         mask of saturated pixels
 @param         orders_coeffs           coefficients of the polynomial describing
                                        orders position
 @param         inst_config             instrument configuration
 @param         CCD_geom                CCD geometry
 @param         RON                     RON for all teh outputs
 @param         extraction_window_size  size of the extraction window = 2*size+1
 @param[out]    extr_spectr_RE          extracted spectrum
 @param[out]    spectr_error_RE         extracted spectrum error
 @param[out]    spectr_qual_RE          extracted spectrum quality (saturation &
                                        hot/bad pixels)
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_simple_extraction(cpl_imagelist *input_images,
                                       cpl_imagelist *hot_bad_pixels_mask,
                                       cpl_imagelist *saturation_mask,
                                       cpl_mask **crh_mask,
                                       cpl_imagelist *background,
                                       cpl_imagelist *master_dark,
                                       cpl_imagelist *master_dark_pxl_nb,
                                       cpl_table ***orders_coeffs,
                                       espdr_inst_config *inst_config,
                                       espdr_CCD_geometry *CCD_geom,
                                       double *RON,
                                       double *DARK,
                                       double *DARK_RON,
                                       double *GAIN,
                                       double Texp,
                                       double Texp_master_dark,
                                       int *extraction_window_size,
                                       cpl_image **extr_spectr_RE,
                                       cpl_image **spectr_error_RE,
                                       cpl_image **spectr_qual_RE) {
    
    
    int i;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(input_images == NULL, CPL_ERROR_NULL_INPUT,
                 "Input images are NULL");
    espdr_ensure(orders_coeffs == NULL, CPL_ERROR_NULL_INPUT,
                 "Orders coefficients table is NULL");
    
    for (i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Simple extraction on fibre %c", fibre_name[i]);
        my_error = espdr_simple_extraction_one_fibre(input_images,
                                                     hot_bad_pixels_mask,
                                                     saturation_mask,
                                                     crh_mask,
                                                     background,
                                                     master_dark,
                                                     master_dark_pxl_nb,
                                                     orders_coeffs[i],
                                                     i,
                                                     inst_config,
                                                     CCD_geom,
                                                     RON, DARK, DARK_RON, GAIN,
                                                     Texp, Texp_master_dark,
                                                     extraction_window_size[i],
                                                     &extr_spectr_RE[i],
                                                     &spectr_error_RE[i],
                                                     &spectr_qual_RE[i]);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
	"espdr_simple_extraction_one_fibre failed: %s",
	cpl_error_get_message_default(my_error));

        espdr_msg_debug("extr_spectr_RE size: %lld x %lld",
                        cpl_image_get_size_x(extr_spectr_RE[i]),
                        cpl_image_get_size_y(extr_spectr_RE[i]));
    }
    
    espdr_msg("Extraction finished");
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Simple extraction of spectrum on one fiber
 @param         input_images            imagelist of images to extract
 @param         hot_bad_pixels_mask     mask of hot & bad pixels
 @param         saturation mask         mask of saturated pixels
 @param         orders_coeffs           coefficients of the polynomial describing
                                        orders position
 @param         fibre_nr                number of the fiber to extract
 @param         inst_config             instrument configuration
 @param         CCD_geom                CCD geometry
 @param         RON                     RON for all teh outputs
 @param         extraction_window_size  size of the extraction window = 2*size+1
 @param[out]    extr_spectr_RE          extracted spectrum
 @param[out]    spectr_error_RE         extracted spectrum error
 @param[out]    spectr_qual_RE          extracted spectrum quality (saturation &
                                        hot/bad pixels)
 @return   CPL_ERROR_NONE iff OK
 */
// Add CRH mask, DARK in the error
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_simple_extraction_one_fibre(cpl_imagelist *input_images,
                                                 cpl_imagelist *hot_bad_pixels_mask,
                                                 cpl_imagelist *saturation_mask,
                                                 cpl_mask **crh_mask,
                                                 cpl_imagelist *background,
                                                 cpl_imagelist *master_dark,
                                                 cpl_imagelist *master_dark_pxl_nb,
                                                 cpl_table **orders_coeffs,
                                                 int fibre_nr,
                                                 espdr_inst_config *inst_config,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 double *RON,
                                                 double *DARK,
                                                 double *DARK_RON,
                                                 double *GAIN,
                                                 double Texp,
                                                 double Texp_master_dark,
                                                 int extraction_window_size,
                                                 cpl_image **extr_spectr_RE,
                                                 cpl_image **spectr_error_RE,
                                                 cpl_image **spectr_qual_RE) {


    int i, ext, order, coeff, pxl, fibre_ext_index, index;
    double *one_order_coeffs = NULL;
    int coeffs_nb = 0;
    char column_name[KEYWORD_LENGTH];
    double order_pos = 0.0;
    double *extracted_spectrum = NULL;
    cpl_image *extracted_spectrum_image = NULL;
    double *spectrum_error = NULL;
    cpl_image *spectrum_error_image = NULL;
    int *spectrum_quality = NULL;
    cpl_image *spectrum_quality_image = NULL;
    cpl_image *hot_bad_image = NULL;
    int *hot_bad_data = NULL;
    cpl_image *satur_image = NULL;
    int *satur_data = NULL;
    cpl_mask *crh_mask_window = NULL;
    cpl_binary *crh_data = NULL;
    int window_base;
    int max_order_length, min_order_start, max_order_end;
    int identified_number_of_orders, total_number_of_orders;
    int output_index = 0;
    cpl_image *input_extraction_image = NULL;
    double *input_data = NULL;
    double *background_data = NULL;
    cpl_image *background_image = NULL;
    double *master_dark_data = NULL;
    cpl_image *master_dark_image = NULL;
    int *master_dark_pxl_nb_data = NULL;
    cpl_image *master_dark_pxl_nb_image = NULL;
    int img_size_x, img_size_y;
    double RON_EL[CCD_geom->total_output_nb];
    double DARK_EL[CCD_geom->total_output_nb];
    double DARK_RON_EL[CCD_geom->total_output_nb];

    int order_index = 0;
    
    int test_fibre = -1;
    int test_ext = 0;
    int test_order = 26;
    int test_pixel_min = 1882;
    int test_pixel_max = 1886;
    
    espdr_ensure(input_images == NULL, CPL_ERROR_NULL_INPUT,
                 "Input images are NULL");
    espdr_ensure(orders_coeffs == NULL, CPL_ERROR_NULL_INPUT,
                 "Orders coefficients table is NULL");
    
    espdr_msg("Starting simple extraction on fibre %c", fibre_name[fibre_nr]);
    
    for (int i = 0; i < CCD_geom->total_output_nb; i++) {
        RON_EL[i] = RON[i] * GAIN[i];
        DARK_EL[i] = DARK[i] * GAIN[i];
        DARK_RON_EL[i] = DARK_RON[i] * GAIN[i];
    }
    
    max_order_length = 0;
    total_number_of_orders = 0;
    identified_number_of_orders = 0;
    min_order_start = 10000;
    max_order_end = 0;
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        //espdr_msg("ext: %d, fibre ext index: %d", ext, fibre_ext_index);
        
        if (min_order_start > inst_config->order_start[fibre_ext_index]) {
            min_order_start = inst_config->order_start[fibre_ext_index];
        }
        if (max_order_end < inst_config->order_end[fibre_ext_index]) {
            max_order_end = inst_config->order_end[fibre_ext_index];
        }
        
        total_number_of_orders = total_number_of_orders +
        inst_config->orders_nb[fibre_ext_index];
        
        identified_number_of_orders = identified_number_of_orders +
        cpl_table_get_nrow(orders_coeffs[ext]);
    }
    
    max_order_length = max_order_end - min_order_start + 1;
    
    if (identified_number_of_orders < total_number_of_orders) {
        
        espdr_msg(ANSI_COLOR_RED
                  "Warning : nominal number of orders: %d, identified number of orders: %d"
                  ANSI_COLOR_RESET,
                  total_number_of_orders, identified_number_of_orders);
        
        espdr_msg(ANSI_COLOR_RED
                  "Extracting nb of orders: %d, orders length: %d"
                  ANSI_COLOR_RESET,
                  identified_number_of_orders, max_order_length);
        
    } else {
        
        espdr_msg("Nominal number of orders: %d, identified number of orders: %d",
                  total_number_of_orders, identified_number_of_orders);
        espdr_msg("Extracting nb of orders: %d, orders length: %d",
                  identified_number_of_orders, max_order_length);
        
    }
    
    extracted_spectrum = (double *)cpl_calloc(max_order_length *
                                              total_number_of_orders,
                                              sizeof(double));
    spectrum_error = (double *)cpl_calloc(max_order_length *
                                          total_number_of_orders,
                                          sizeof(double));
    spectrum_quality = (int *)cpl_calloc(max_order_length *
                                         total_number_of_orders,
                                         sizeof(int));
    if (background == NULL) {
        background_data = (double *)cpl_calloc
                            ((extraction_window_size*2+1), sizeof(double));
    }
    
    index = 0;
    int order_length = 0;
    int number_of_orders = 0;
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        order_length = inst_config->order_end[fibre_ext_index] -
                        inst_config->order_start[fibre_ext_index] + 1;
        number_of_orders = inst_config->orders_nb[fibre_ext_index];
        coeffs_nb = cpl_table_get_ncol(orders_coeffs[ext]);
        one_order_coeffs = (double *)cpl_calloc(coeffs_nb, sizeof(double));
        img_size_x = cpl_image_get_size_x(cpl_imagelist_get(input_images, ext));
        img_size_y = cpl_image_get_size_y(cpl_imagelist_get(input_images, ext));
        
        double RON_mean = 0.0;
        int i_start;
        if (ext == 0) {
            i_start = 0;
        } else {
            i_start = CCD_geom->exts[ext-1].out_nb_x * CCD_geom->exts[ext-1].out_nb_y;
        }
        for (i = i_start; i < i_start + CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y; i++) {
            RON_mean += RON_EL[i];
        }
        RON_mean = RON_mean / (CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y);
        
        // If some orders are not identified, they are not extracted
        for (order = 0; order < cpl_table_get_nrow(orders_coeffs[ext]); order++) {
            //espdr_msg("Extracting order %d", order);
            
            for (coeff = 0; coeff < coeffs_nb; coeff++) {
                sprintf(column_name, "COEFF_%d", coeff);
                one_order_coeffs[coeff] = cpl_table_get(orders_coeffs[ext],
                                                        column_name, order, NULL);
            }
            
            if (inst_config->order_start[fibre_ext_index] > min_order_start) {
                for (pxl = min_order_start;
                     pxl < inst_config->order_start[fibre_ext_index];
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            
            for (pxl = inst_config->order_start[fibre_ext_index];
                 pxl <= inst_config->order_end[fibre_ext_index]; pxl++) {
                
                //espdr_msg("Extracting pxl %d", pxl);
                order_pos = 0.0;
                for (i = coeffs_nb-1; i > 0; i--) {
                    order_pos = (order_pos + one_order_coeffs[i]) * (double)pxl;
                }
                order_pos = order_pos + one_order_coeffs[0];
                double order_rest = order_pos - (int)order_pos;
                if (order_rest > 0.5) {
                    window_base = (int)order_pos - extraction_window_size + 1;
                } else {
                    window_base = (int)order_pos - extraction_window_size;
                }
                
                if ((window_base > 0) &&
                    (window_base+extraction_window_size*2 <= img_size_y)) {
                    // It's OK, the extraction can be done
                    
                    int pxl_raw = img_size_x * (window_base + extraction_window_size) + pxl;
                    output_index = espdr_get_output_index_for_pixel(ext, pxl_raw, CCD_geom);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "espdr_get_output_index_for_pixel failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    spectrum_quality[index] = 0;
                    
                    //espdr_msg("Extracting hot_bad data");
                    hot_bad_image = cpl_image_extract(cpl_imagelist_get(hot_bad_pixels_mask, ext),
                                                      pxl, window_base, pxl,
                                                      window_base + extraction_window_size*2);
                    if (cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT) {
                        espdr_msg("BAD coord (order %d): %d, %d, %d, %d",
                                  order, pxl, window_base, pxl,
                                  window_base + extraction_window_size*2);
                    }
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    hot_bad_data = cpl_image_get_data_int(hot_bad_image);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_int failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    if (background != NULL) {
                        //espdr_msg("Extracting background");
                        background_image =
                        cpl_image_extract(cpl_imagelist_get(background, ext),
                                          pxl, window_base, pxl,
                                          window_base + extraction_window_size*2);
                        background_data = cpl_image_get_data_double(background_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_image_extract failed for background: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    if (master_dark != NULL) {
                        //espdr_msg("Extracting master_dark");
                        master_dark_image = cpl_image_extract(cpl_imagelist_get(master_dark, ext),
                                                              pxl, window_base, pxl,
                                                              window_base + extraction_window_size*2);
                        master_dark_data = cpl_image_get_data_double(master_dark_image);
                        master_dark_pxl_nb_image = cpl_image_extract(cpl_imagelist_get(master_dark, ext+CCD_geom->ext_nb),
                                                                     pxl, window_base, pxl,
                                                                     window_base + extraction_window_size*2);
                        master_dark_pxl_nb_data = cpl_image_get_data_int(master_dark_pxl_nb_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_image_extract failed for master_dark: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    //espdr_msg("Extracting saturation");
                    satur_image = cpl_image_extract(cpl_imagelist_get(saturation_mask, ext),
                                                    pxl, window_base, pxl,
                                                    window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for saturation_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    satur_data = cpl_image_get_data_int(satur_image);
                    if (cpl_image_get_flux(satur_image) > 0.0) {
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (satur_data[i] != 0) {
                                spectrum_quality[index] = spectrum_quality[index] | satur_data[i];
                                //satur_pixels_nb ++;
                            }
                        }
                    }
                    
                    //espdr_msg("Extracting cosmics");
                    if (crh_mask != NULL) {
                        crh_mask_window = cpl_mask_extract(crh_mask[ext],
                                                           pxl, window_base, pxl,
                                                           window_base + extraction_window_size*2);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_mask_extract failed for crh_mask: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                        
                        crh_data = cpl_mask_get_data(crh_mask_window);
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (crh_data[i] == CPL_BINARY_1) {
                                spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                            }
                        }
                    }
                    
                    //espdr_msg("Extracting image");
                    input_extraction_image = cpl_image_extract(cpl_imagelist_get(input_images, ext),
                                                               pxl, window_base, pxl,
                                                               window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for input_data: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    input_data = cpl_image_get_data_double(input_extraction_image);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_double failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("EXTRACTION DEBUG START -------------------------------->");
                        espdr_msg("ext %d, order: %d, pxl: %d, window_base: %d",
                                  ext, order, pxl, window_base);
                    }
                    
                    /********************/
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("Image read: ext: %d, order: %d, pxl: %d",
                                  ext, order, pxl);
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            espdr_msg("input_data[%d] = %lf\thot_bad[%d] = %d\tsatur[%d] = %d",
                                      i, input_data[i],
                                      i, hot_bad_data[i],
                                      i, satur_data[i]);
                        }
                    }
                    
                    /********************/
                    
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = 0.0;
                    double background_total = 0.0;
                    double dark_total = 0.0;
                    double RON_total2 = 0.0;
                    for (i = 0; i < extraction_window_size*2+1; i++) {
                        if (hot_bad_data[i] == 0) {
                            extracted_spectrum[index] += input_data[i];
                            background_total += background_data[i];
                            if (master_dark == NULL) {
                                dark_total += DARK_EL[output_index] * Texp;
                            } else {
                                dark_total += (master_dark_data[i] * GAIN[output_index] * Texp / Texp_master_dark) / master_dark_pxl_nb_data[i];
                                RON_total2 += (DARK_RON_EL[output_index]*DARK_RON_EL[output_index]) / master_dark_pxl_nb_data[i];
                            }
                            RON_total2 += RON_EL[output_index]*RON_EL[output_index];
                        }
                        
                        spectrum_quality[index] = spectrum_quality[index] | hot_bad_data[i];
                        //espdr_msg("order: %d, pxl: %d, i: %d, hot_bad: %d, qual: %d",
                        //    order+1, pxl, i, hot_bad_data[i], spectrum_quality[index]);
                    }
                    
                    spectrum_error[index] = sqrt(ESPDR_MAX(0, extracted_spectrum[index] + background_total + dark_total) + RON_total2);
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) &&
                        (order == test_order) && (spectrum_quality[index] != 0)) {
                        espdr_msg("----------------> FINAL qual != 0 for ext: %d, order %d, pxl: %d, qual: %d",
                                  ext, order, pxl, spectrum_quality[index]);
                    }
                    /********************/
                    
                    
                    cpl_image_delete(input_extraction_image);
                    cpl_image_delete(hot_bad_image);
                    cpl_image_delete(satur_image);
                    if (background != NULL) {
                        cpl_image_delete(background_image);
                    }
                    cpl_mask_delete(crh_mask_window);
                    
                } else { // if (window_base > 0) && (window_base*2+1 < img_size_y)
                    
                    // Fill the rest of the extracted spectrum with 0.0
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    
                } // if (window_base > 0) && (window_base*2+1 < img_size_y)
                
                index++;
            } // end of the for loop on pixels
            
            if (inst_config->order_end[fibre_ext_index] < max_order_end) {
                for (pxl = inst_config->order_end[fibre_ext_index] + 1;
                     pxl <= max_order_end;
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            //cosmics_nb_RE[order_index] = cosmics_nb;
            order_index++;
            
        } // end of for loop on orders
        cpl_free(one_order_coeffs);
    } // end of for loop on extensions
    
    extracted_spectrum_image = cpl_image_wrap_double(max_order_length,
                                                     total_number_of_orders,
                                                     extracted_spectrum);
    *extr_spectr_RE = cpl_image_duplicate(extracted_spectrum_image);
    cpl_image_unwrap(extracted_spectrum_image);
    cpl_free(extracted_spectrum);
    
    spectrum_error_image = cpl_image_wrap_double(max_order_length,
                                                 total_number_of_orders,
                                                 spectrum_error);
    *spectr_error_RE = cpl_image_duplicate(spectrum_error_image);
    cpl_image_unwrap(spectrum_error_image);
    cpl_free(spectrum_error);
    
    spectrum_quality_image = cpl_image_wrap_int(max_order_length,
                                                total_number_of_orders,
                                                spectrum_quality);
    *spectr_qual_RE = cpl_image_duplicate(spectrum_quality_image);
    cpl_image_unwrap(spectrum_quality_image);
    cpl_free(spectrum_quality);
    
    if (background == NULL) {
        cpl_free(background_data);
    }
    
    espdr_msg_debug("EOF: extr_spectr_RE size: %lld x %lld",
                    cpl_image_get_size_x(*extr_spectr_RE),
                    cpl_image_get_size_y(*extr_spectr_RE));
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Optimal (Horne) extraction of spectrum on all fibers
 @param         input_images        imagelist of images to extract
 @param         hot_bad_pixels_mask mask of hot & bad pixels
 @param         saturation mask     mask of saturated pixels
 @param         background          background image
 @param         order_profile       order profile image
 @param         contam              contamination image
 @param         orders_coeffs       coefficients of the polynomial describing
                                    orders position
 @param         inst_config         instrument configuration
 @param         CCD_geom            CCD geometry
 @param         RON                 RON for all the outputs
 @param         DARK                DARK for all the outputs
 @param         Texp                input frame exposure time in hours
 @param         extraction_window_size  size of the extraction window = 2*size+1
 @param         tolerance_reject    tolerance of residuals rejection
 @param         ksig_extraction     limit of residuals rejection
 @param[out]    cosmics_nb_RE   number of cosmics detected per order of each fibre
 @param[out]    extr_spectr_RE  extracted spectrum
 @param[out]    spectr_error_RE extracted spectrum error
 @param[out]    spectr_qual_RE  extracted spectrum quality (saturation & hot/bad
                                pixels)
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_horne_extraction(cpl_imagelist *input_images,
                                      cpl_imagelist *hot_bad_pixels_mask,
                                      cpl_imagelist *saturation_mask,
                                      cpl_mask **crh_mask,
                                      cpl_imagelist *background,
                                      cpl_imagelist **order_profile,
                                      cpl_imagelist *contam,
                                      cpl_imagelist *master_dark,
                                      cpl_imagelist *master_dark_pxl_nb,
                                      cpl_table ***orders_coeffs,
                                      espdr_inst_config *inst_config,
                                      espdr_CCD_geometry *CCD_geom,
                                      double *RON,
                                      double *DARK,
                                      double *DARK_RON,
                                      double *GAIN,
                                      double Texp,
                                      double Texp_master_dark,
                                      int *extraction_window_size,
                                      double tolerance_reject,
                                      double ksig_extraction,
                                      int **cosmics_nb_RE,
                                      cpl_image **extr_spectr_RE,
                                      cpl_image **spectr_error_RE,
                                      cpl_image **spectr_qual_RE) {
    
    
    int i;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(input_images == NULL, CPL_ERROR_NULL_INPUT,
                 "Input images are NULL");
    espdr_ensure(orders_coeffs == NULL, CPL_ERROR_NULL_INPUT,
                 "Orders coefficients table is NULL");

    for (i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Horne extraction on fibre %c", fibre_name[i]);
        espdr_ensure(order_profile[i] == NULL, CPL_ERROR_NULL_INPUT,
                     "No ORDER_PROFILE for fibre %c", fibre_name[i]);
        my_error = espdr_horne_extraction_one_fibre(input_images,
                                                    hot_bad_pixels_mask,
                                                    saturation_mask,
                                                    crh_mask,
                                                    background,
                                                    order_profile[i],
                                                    contam,
                                                    master_dark,
                                                    master_dark_pxl_nb,
                                                    orders_coeffs[i],
                                                    i,
                                                    inst_config,
                                                    CCD_geom,
                                                    RON, DARK, DARK_RON, GAIN,
                                                    Texp, Texp_master_dark,
                                                    extraction_window_size[i],
                                                    tolerance_reject,
                                                    ksig_extraction,
                                                    cosmics_nb_RE[i],
                                                    &extr_spectr_RE[i],
                                                    &spectr_error_RE[i],
                                                    &spectr_qual_RE[i]);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_horne_extraction_one_fibre failed: %s",
                     cpl_error_get_message_default(my_error));
        
        espdr_msg_debug("extr_spectr_RE size: %lld x %lld",
                        cpl_image_get_size_x(extr_spectr_RE[i]),
                        cpl_image_get_size_y(extr_spectr_RE[i]));
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Optimal (Horne) extraction of spectrum on one fiber
 @param         input_images        imagelist of images to extract
 @param         hot_bad_pixels_mask mask of hot & bad pixels
 @param         saturation mask     mask of saturated pixels
 @param         background          background image
 @param         order_profile       order profile image
 @param         contam              contamination image
 @param         orders_coeffs       coefficients of the polynomial describing
                                    orders position
 @param         fibre_nr            number of the fiber to extract
 @param         inst_config         instrument configuration
 @param         CCD_geom            CCD geometry
 @param         RON                 RON for all the outputs
 @param         DARK                DARK for all the outputs
 @param         Texp                input frame exposure time in hours
 @param         extraction_window_size  size of the extraction window = 2*size+1
 @param         tolerance_reject    tolerance of residuals rejection
 @param         ksig_extraction     limit of residuals rejection
 @param[out]    cosmics_nb_RE   number of cosmics detected per order of each fibre
 @param[out]    extr_spectr_RE  extracted spectrum
 @param[out]    spectr_error_RE extracted spectrum error
 @param[out]    spectr_qual_RE  extracted spectrum quality (saturation & hot/bad
                                pixels)
 @return   CPL_ERROR_NONE iff OK
 
 Comment about cosmics:
 Once we get the cosmics flaggiing from the LA_cosmics algorithm, the flagged
 pixels will be treated the same way as saturated pixels:
 - they will not be included in the bad pixels, so not interpolated or preventing
   the optimal extraction
 - they will be passed to the optimal extraction as masked ones
 What to do to get the extracted flux as close to the one behind the hot/bad pixels,
 as possible?
 - create an artificial profile through a gaussian defined by a FWHM saved in the
   inst_config for each mode
 - use the artificial profile when the spline interpolation is not possible
   (too many hot/bad pixels), but there are still some good pixels
 - of course nothing can be done if all the pixels are hot or bad
 
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_horne_extraction_one_fibre(cpl_imagelist *input_images,
                                                cpl_imagelist *hot_bad_pixels_mask,
                                                cpl_imagelist *saturation_mask,
                                                cpl_mask **crh_mask_whole,
                                                cpl_imagelist *background,
                                                cpl_imagelist *order_profile,
                                                cpl_imagelist *contam,
                                                cpl_imagelist *master_dark,
                                                cpl_imagelist *master_dark_pxl_nb,
                                                cpl_table **orders_coeffs,
                                                int fibre_nr,
                                                espdr_inst_config *inst_config,
                                                espdr_CCD_geometry *CCD_geom,
                                                double *RON,
                                                double *DARK,
                                                double *DARK_RON,
                                                double *GAIN,
                                                double Texp,
                                                double Texp_master_dark,
                                                int extraction_window_size,
                                                double tolerance_reject,
                                                double ksig_extraction,
                                                int *cosmics_nb_RE,
                                                cpl_image **extr_spectr_RE,
                                                cpl_image **spectr_error_RE,
                                                cpl_image **spectr_qual_RE) {
    
    int i, ext, order, coeff, pxl, fibre_ext_index, index;
    double *one_order_coeffs = NULL;
    int coeffs_nb = 0;
    char column_name[KEYWORD_LENGTH];
    double order_pos = 0.0;
    double *extracted_spectrum = NULL;
    cpl_image *extracted_spectrum_image = NULL;
    double *spectrum_error = NULL;
    cpl_image *spectrum_error_image = NULL;
    int *spectrum_quality = NULL;
    cpl_image *spectrum_quality_image = NULL;
    cpl_image *hot_bad_image = NULL;
    int *hot_bad_data = NULL;
    cpl_image *satur_image = NULL;
    int *satur_data = NULL;
    cpl_mask *crh_mask_window = NULL;
    cpl_binary *crh_data = NULL;
    cpl_image *order_profile_image = NULL;
    double *order_profile_data = NULL;
    int order_length, number_of_orders, window_base;
    int max_order_length, min_order_start, max_order_end;
    int identified_number_of_orders, total_number_of_orders;
    int output_index = 0;
    double flux = 0.0;
    cpl_image *input_extraction_image = NULL;
    double *input_data = NULL;
    int *mask = NULL;
    int *corr_mask = NULL;
    int *satur_mask = NULL;
    int *crh_mask = NULL;
    int flux_min_index_lower, flux_min_index_upper;
    double flux_min_lower, flux_min_upper, profile_background;
    int hot_bad_pixels_nb = 0;
    int satur_pixels_nb = 0;
    int crh_pixels_nb = 0;
    double *var = NULL, *res = NULL;
    double *background_data = NULL;
    cpl_image *background_image = NULL;
    double *contam_data = NULL;
    cpl_image *contam_image = NULL;
    double *master_dark_data = NULL;
    cpl_image *master_dark_image = NULL;
    int *master_dark_pxl_nb_data = NULL;
    cpl_image *master_dark_pxl_nb_image = NULL;
    double sum_prof, sumPD, sumPP, res_max, variance_flux = 0.0;
    int resmax_index;
    int img_size_x, img_size_y, cosmics_nb;
    double sumC;
    double sumPC, sumCD, sumCC, k;
    int good_pxls_nb;
    int window_fraction_background_compute = extraction_window_size+1;
    double RON_EL[CCD_geom->total_output_nb];
    double DARK_EL[CCD_geom->total_output_nb];
    double DARK_RON_EL[CCD_geom->total_output_nb];

    int order_index = 0;
    
    int test_fibre = -1;
    int test_ext = 0;
    int test_order = 2;
    int test_pixel_min = 45;
    int test_pixel_max = 47;

    espdr_ensure(input_images == NULL, CPL_ERROR_NULL_INPUT,
                 "Input images are NULL");
    espdr_ensure(orders_coeffs == NULL, CPL_ERROR_NULL_INPUT,
                 "Orders coefficients table is NULL");
    
    espdr_msg("Starting optimal extraction on fibre %c, with ksig = %.2f, window size: %d, Texp: %f, Texp_mdark: %f",
              fibre_name[fibre_nr], ksig_extraction, extraction_window_size, Texp, Texp_master_dark);
    
    for (int i = 0; i < CCD_geom->total_output_nb; i++) {
        RON_EL[i] = RON[i] * GAIN[i];
        DARK_EL[i] = DARK[i] * GAIN[i];
        DARK_RON_EL[i] = DARK_RON[i] * GAIN[i];
        //espdr_msg("RON[%d] = %f, DARK[%d] = %f, DARK_RON[%d] = %f, GAIN[%d] = %f, RON_EL[%d] = %f, DARK_EL[%d] = %f, DARK_RON_EL[%d] = %f",
        //          i, RON[i], i, DARK[i], i, DARK_RON[i], i, GAIN[i], i, RON_EL[i], i, DARK_EL[i], i, DARK_RON_EL[i]);
    }
    
    max_order_length = 0;
    total_number_of_orders = 0;
    identified_number_of_orders = 0;
    min_order_start = 10000;
    max_order_end = 0;
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        //espdr_msg("ext: %d, fibre ext index: %d", ext, fibre_ext_index);
        
        if (min_order_start > inst_config->order_start[fibre_ext_index]) {
            min_order_start = inst_config->order_start[fibre_ext_index];
        }
        if (max_order_end < inst_config->order_end[fibre_ext_index]) {
            max_order_end = inst_config->order_end[fibre_ext_index];
        }
        
        total_number_of_orders = total_number_of_orders +
                                        inst_config->orders_nb[fibre_ext_index];
        
        identified_number_of_orders = identified_number_of_orders +
                                        cpl_table_get_nrow(orders_coeffs[ext]);
    }

    for (i = 0; i < total_number_of_orders; i++) {
        cosmics_nb_RE[i] = 0;
        //espdr_msg("Cosmics for order %d initialized to 0.0", i);
    }
    
    max_order_length = max_order_end - min_order_start + 1;

    if(identified_number_of_orders < total_number_of_orders) {
        
        espdr_msg(ANSI_COLOR_RED
                  "Warning : nominal number of orders: %d, identified number of orders: %d"
                  ANSI_COLOR_RESET,
                  total_number_of_orders, identified_number_of_orders);
        
        espdr_msg(ANSI_COLOR_RED
                  "Extracting nb of orders: %d, orders length: %d"
                  ANSI_COLOR_RESET,
                  identified_number_of_orders, max_order_length);
        
    } else {
        
        espdr_msg("Nominal number of orders: %d, identified number of orders: %d",
                  total_number_of_orders, identified_number_of_orders);
        espdr_msg("Extracting nb of orders: %d, orders length: %d",
                  identified_number_of_orders, max_order_length);
        
    }
    
    extracted_spectrum = (double *)cpl_calloc(max_order_length *
                                              total_number_of_orders,
                                              sizeof(double));
    spectrum_error = (double *)cpl_calloc(max_order_length *
                                          total_number_of_orders,
                                          sizeof(double));
    spectrum_quality = (int *)cpl_calloc(max_order_length *
                                         total_number_of_orders,
                                         sizeof(int));
    mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    corr_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    satur_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    crh_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    var = (double *) cpl_calloc ((extraction_window_size*2+1), sizeof(double));
    res = (double *) cpl_calloc ((extraction_window_size*2+1), sizeof(double));
    if (background == NULL) {
        background_data = (double *)cpl_calloc
                            ((extraction_window_size*2+1), sizeof(double));
    }
    
    index = 0;
    
    cpl_imagelist *geometry_corrected_master_dark = cpl_imagelist_new();
    cpl_imagelist *geometry_corrected_master_dark_pxl_nb = cpl_imagelist_new();
    if (master_dark != NULL) {
        //espdr_msg("Correcting the master_dark geometry");
        cpl_imagelist *merged_master_dark = cpl_imagelist_new();
        espdr_image_merge_2_dim(master_dark, CCD_geom, CPL_TYPE_DOUBLE,
                                           &merged_master_dark);
        espdr_correct_geometry(merged_master_dark, CCD_geom,
                                          geometry_corrected_master_dark);
        cpl_imagelist_delete(merged_master_dark);
        cpl_imagelist *merged_master_dark_pxl_nb = cpl_imagelist_new();
        espdr_image_merge_2_dim(master_dark_pxl_nb, CCD_geom, CPL_TYPE_INT,
                                           &merged_master_dark_pxl_nb);
        espdr_correct_geometry(merged_master_dark_pxl_nb, CCD_geom,
                                          geometry_corrected_master_dark_pxl_nb);
        cpl_imagelist_delete(merged_master_dark_pxl_nb);
    }
    
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        order_length = inst_config->order_end[fibre_ext_index] -
                        inst_config->order_start[fibre_ext_index] + 1;
        number_of_orders = inst_config->orders_nb[fibre_ext_index];
        coeffs_nb = cpl_table_get_ncol(orders_coeffs[ext]);
        one_order_coeffs = (double *)cpl_calloc(coeffs_nb, sizeof(double));
        img_size_x = cpl_image_get_size_x(cpl_imagelist_get(input_images, ext));
        img_size_y = cpl_image_get_size_y(cpl_imagelist_get(input_images, ext));
        
        double RON_mean = 0.0;
        int i_start;
        if (ext == 0) {
            i_start = 0;
        } else {
            i_start = CCD_geom->exts[ext-1].out_nb_x * CCD_geom->exts[ext-1].out_nb_y;
        }
        for (i = i_start; i < i_start + CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y; i++) {
            RON_mean += RON_EL[i];
        }
        RON_mean = RON_mean / (CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y);
        
        // If some orders are not identified, they are not extracted
        for (order = 0; order < cpl_table_get_nrow(orders_coeffs[ext]); order++) {
            //espdr_msg("Extracting order %d", order);

    	    cosmics_nb = 0;

            for (coeff = 0; coeff < coeffs_nb; coeff++) {
                sprintf(column_name, "COEFF_%d", coeff);
                one_order_coeffs[coeff] = cpl_table_get(orders_coeffs[ext],
                                                        column_name, order, NULL);
            }
            
            if (inst_config->order_start[fibre_ext_index] > min_order_start) {
                for (pxl = min_order_start;
                     pxl < inst_config->order_start[fibre_ext_index];
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            for (pxl = inst_config->order_start[fibre_ext_index];
                 pxl <= inst_config->order_end[fibre_ext_index]; pxl++) {
                
                satur_pixels_nb = 0;
                crh_pixels_nb = 0;
                
                //espdr_msg("Extracting pxl %d", pxl);
                order_pos = 0.0;
                for (i = coeffs_nb-1; i > 0; i--) {
                    order_pos = (order_pos + one_order_coeffs[i]) * (double)pxl;
                }
                order_pos = order_pos + one_order_coeffs[0];
                double order_rest = order_pos - (int)order_pos;
                if (order_rest > 0.5) {
                    window_base = (int)order_pos - extraction_window_size + 1;
                } else {
                    window_base = (int)order_pos - extraction_window_size;
                }
                
                
                if ((window_base > 0) && (window_base+extraction_window_size*2 <= img_size_y)) {
                    // It's OK, the extraction can be done
                
                    espdr_ensure(((window_base < 1) || (window_base+extraction_window_size*2 > img_size_y)),
                                 CPL_ERROR_INCOMPATIBLE_INPUT,
                                 "The extraction window is too large (%d), please reduce its size and relaunch the recipe.",
                                 extraction_window_size);
                    
                    int pxl_raw = img_size_x * (window_base + extraction_window_size) + pxl;
                    output_index = espdr_get_output_index_for_pixel(ext, pxl_raw, CCD_geom);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "espdr_get_output_index_for_pixel failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    spectrum_quality[index] = 0;
                    
                    //espdr_msg("Extracting hot_bad data");
                    hot_bad_image = cpl_image_extract(cpl_imagelist_get(hot_bad_pixels_mask, ext),
                                                      pxl, window_base, pxl,
                                                      window_base + extraction_window_size*2);
                    if (cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT) {
                        espdr_msg("BAD coord (order %d): %d, %d, %d, %d",
                                  order, pxl, window_base, pxl, 
                                  window_base + extraction_window_size*2);
                    }
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    hot_bad_data = cpl_image_get_data_int(hot_bad_image);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_int failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    if (background != NULL) {
                        //espdr_msg("Extracting background");
                        background_image =  cpl_image_extract(cpl_imagelist_get(background, ext),
                                                              pxl, window_base, pxl,
                                                              window_base + extraction_window_size*2);
                        background_data = cpl_image_get_data_double(background_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_image_extract failed for background: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    //espdr_msg("Extracting saturation");
                    satur_image = cpl_image_extract(cpl_imagelist_get(saturation_mask, ext),
                                                    pxl, window_base, pxl,
                                                    window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for saturation_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                              
                    satur_data = cpl_image_get_data_int(satur_image);
                    for (i = 0; i < extraction_window_size*2+1; i++) {
                        if (satur_data[i] != 0) {
                            satur_pixels_nb ++;
                            satur_mask[i] = 0;
                        } else {
                            satur_mask[i] = 1;
                        }
                    }

                    //espdr_msg("Extracting cosmics");
                    if (crh_mask_whole != NULL) {
                        crh_mask_window = cpl_mask_extract(crh_mask_whole[ext],
                                                           pxl, window_base, pxl,
                                                           window_base + extraction_window_size*2);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_mask_extract failed for crh_mask: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                        
                        crh_data = cpl_mask_get_data(crh_mask_window);
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (crh_data[i] == CPL_BINARY_1) {
                                crh_pixels_nb ++;
                                crh_mask[i] = 0;
                            } else {
                                crh_mask[i] = 1;
                            }
                        }
                    } else {
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            crh_mask[i] = 1;
                        }
                    }
                    
                    if (master_dark != NULL) {
                        //espdr_msg("Extracting master_dark for master_dark error calculation");
                        master_dark_image =  cpl_image_extract(cpl_imagelist_get(geometry_corrected_master_dark, ext),
                                                               pxl, window_base, pxl,
                                                               window_base + extraction_window_size*2);
                        master_dark_data = cpl_image_get_data_double(master_dark_image);
                        master_dark_pxl_nb_image =  cpl_image_extract(cpl_imagelist_get(geometry_corrected_master_dark_pxl_nb, ext),
                                                                      pxl, window_base, pxl,
                                                                      window_base + extraction_window_size*2);
                        master_dark_pxl_nb_data = cpl_image_get_data_int(master_dark_pxl_nb_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                                     "cpl_image_extract failed for master_dark: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    //espdr_msg("Extracting image");
                    input_extraction_image = cpl_image_extract(cpl_imagelist_get(input_images, ext),
                                                               pxl, window_base, pxl,
                                                               window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for input_data: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    input_data = cpl_image_get_data_double(input_extraction_image);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_double failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("EXTRACTION DEBUG START -------------------------------->");
                        espdr_msg("ext %d, order: %d, pxl: %d, window_base: %d",
                                  ext, order, pxl, window_base);
                    }
                    
                    /********************/
                    
                    espdr_msg_debug("Extracting order profile");
                    order_profile_image =
                    cpl_image_extract(cpl_imagelist_get(order_profile, ext),
                                      pxl, window_base, pxl,
                                      window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for input_data: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    order_profile_data = cpl_image_get_data_double(order_profile_image);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_double failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("Image read: ext: %d, order: %d, pxl: %d",
                                  ext, order, pxl);
                        if (crh_mask_whole != NULL) {
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                espdr_msg("order_profile[%d] = %lf\tinput_data[%d] = %lf\thot_bad[%d] = %d\tsatur[%d] = %d\tcrh[%d] = %d",
                                          i, order_profile_data[i],
                                          i, input_data[i],
                                          i, hot_bad_data[i],
                                          i, satur_data[i],
                                          i, crh_data[i]);
                            }
                        } else {
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                espdr_msg("order_profile[%d] = %lf\tinput_data[%d] = %lf\thot_bad[%d] = %d\tsatur[%d] = %d",
                                          i, order_profile_data[i],
                                          i, input_data[i],
                                          i, hot_bad_data[i],
                                          i, satur_data[i]);
                            }
                        }
                    }
                    
                    
                    
                    /* Compute the background level */
                    espdr_msg_debug("Computing background level");
                    hot_bad_pixels_nb = 0;
                    good_pxls_nb = 0;
                    for (i = 0; i < extraction_window_size*2+1; i++) {
                        if (hot_bad_data[i] > 0) {
                            mask[i] = 0;
                            hot_bad_pixels_nb++;
                        } else {
                            mask[i] = 1;
                        }
                        
                        if ((mask[i] == 1) && (satur_mask[i] == 1) && (crh_mask[i] == 1)) {
                            good_pxls_nb++;
                        }
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("order: %d, pxl: %d, window_base: %d, hot_bad[%d]: %d, hot_bad_nb %d, satur[%d] = %d, crh[%d] = %d, qual: %d",
                                      order, pxl, window_base, i, hot_bad_data[i], hot_bad_pixels_nb,
                                      i, satur_mask[i], i, crh_mask[i], spectrum_quality[index]);
                        }
                    }
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("good_pxls_nb with all masks: %d", good_pxls_nb);
                    }
                    
                    if (good_pxls_nb == 0) {
                        extracted_spectrum[index] = 0.0;
                        spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (hot_bad_data[i] > 0) {
                                spectrum_quality[index] = spectrum_quality[index] | hot_bad_data[i];
                            }
                            if (satur_data[i] > 0) {
                                spectrum_quality[index] = spectrum_quality[index] | satur_data[i];
                            }
                            if (crh_mask_whole != NULL) {
                                if (crh_data[i] > 0) {
                                    spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                }
                            }
                        }
                    } else {
                        // at least one good pixel present in the extraction window
                    
                        window_fraction_background_compute = (int)(extraction_window_size/
                                                                   inst_config->slices_nb)+1;
                        // To avoid dust on the order effect, we take just the 30% border of the order
                        // to find the correct background
                        if (inst_config->slices_nb == 1) {
                            window_fraction_background_compute = (int)(extraction_window_size*
                                                                       inst_config->extraction_bkgr_calc_fraction);
                        }
                        flux_min_lower = DBL_MAX;
                        flux_min_index_lower = -1;
                        for (i = 0; i < window_fraction_background_compute; i++) {
                            if (hot_bad_data[i] == 0) {
                                if (order_profile_data[i] < flux_min_lower) {
                                    flux_min_lower = order_profile_data[i];
                                    flux_min_index_lower = i;
                                }
                            }
                        }
                        
                        flux_min_upper = DBL_MAX;
                        flux_min_index_upper = -1;
                        for (i = extraction_window_size * 2;
                             i > extraction_window_size*2 - window_fraction_background_compute;
                             i--) {
                            if (hot_bad_data[i] == 0) {
                                if (order_profile_data[i] < flux_min_upper) {
                                    flux_min_upper = order_profile_data[i];
                                    flux_min_index_upper = i;
                                }
                            }
                        }
                        
                        if ((flux_min_index_lower == -1) && (flux_min_index_upper == -1)) {
                            profile_background = 0.0;
                        } else {
                            if (flux_min_index_lower == -1) {
                                profile_background = order_profile_data[flux_min_index_upper];
                            } else {
                                if (flux_min_index_upper == -1) {
                                    profile_background = order_profile_data[flux_min_index_lower];
                                } else {
                                    if (order_profile_data[flux_min_index_lower] >
                                        order_profile_data[flux_min_index_upper]) {
                                        profile_background = order_profile_data[flux_min_index_lower];
                                    } else {
                                        profile_background = order_profile_data[flux_min_index_upper];
                                    }
                                }
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("profile_background = %lf", profile_background);
                            espdr_msg("window fraction background compute = %d",
                                      window_fraction_background_compute);
                            espdr_msg("hot bad pixels nb = %d",
                                      hot_bad_pixels_nb);
                            espdr_msg("Quality: %d", spectrum_quality[index]);
                        }
                        
                        /********************/
                        
                        // If the upper index is -1 --> all profile -> 0.0, so we have to put it to the last index in the window
                        if (flux_min_index_upper == -1) {
                            flux_min_index_upper = extraction_window_size*2+1;
                        }
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (mask[i] == 1) {
                                if ((i < flux_min_index_lower) ||
                                    (i > flux_min_index_upper)) {
                                    order_profile_data[i] = 0.0;
                                } else {
                                    if (order_profile_data[i] < profile_background) {
                                        order_profile_data[i] = 0.0;
                                    } else {
                                        order_profile_data[i] -= profile_background;
                                    }
                                }
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("EXTRACTION DEBUG------>");
                            espdr_msg("flux min lower: %lf at %d, flux min upper: %lf at %d",
                                      flux_min_lower, flux_min_index_lower,
                                      flux_min_upper, flux_min_index_upper);
                        }
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("After background level: order: %d, pxl: %d, hot_bad_nb: %d, qual: %d",
                                      order, pxl, hot_bad_pixels_nb, spectrum_quality[index]);
                            if (crh_mask_whole != NULL) {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    espdr_msg("order_profile[%d] = %f\tinput_data[%d] = %f\thot_bad[%d] = %d\tsatur_data[%d] = %d\tsatur_mask[%d] = %d\tcrh_data[%d] = %d",
                                              i, order_profile_data[i],
                                              i, input_data[i],
                                              i, hot_bad_data[i],
                                              i, satur_data[i],
                                              i, satur_mask[i],
                                              i, crh_data[i]);
                                }
                            } else {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    espdr_msg("order_profile[%d] = %f\tinput_data[%d] = %f\thot_bad[%d] = %d\tsatur_data[%d] = %d\tsatur_mask[%d] = %d",
                                              i, order_profile_data[i],
                                              i, input_data[i],
                                              i, hot_bad_data[i],
                                              i, satur_data[i],
                                              i, satur_mask[i]);
                                }
                            }
                        }
                        
                        /********************/
                        
                        /* Extraction */
                        
                        /* Without the contamination image */
                        if (contam == NULL) {
                            sum_prof = 0.0;
                            flux = 0.0;
                            for(i = 0; i < extraction_window_size*2+1; i++) {
                                sum_prof += order_profile_data[i] * mask[i]; // only good & saturated pixels
                                flux += input_data[i] * mask[i]; // only good & saturated pixels
                            }
                            
                            if (sum_prof == 0) {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    extracted_spectrum[index] += input_data[i] * mask[i] * satur_mask[i] * crh_mask[i]; // only good pixels
                                    spectrum_quality[index] = spectrum_quality[index] | CALIBRATION_DEFECT;
                                }
                                double MAXSPEC = ESPDR_MAX(0, extracted_spectrum[index];
                                spectrum_error[index] = sqrt(MAXSPEC + (extraction_window_size*2+1) * RON_EL[output_index] * RON_EL[output_index]));
                            } else {
                                
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    order_profile_data[i] = order_profile_data[i]/sum_prof;
                                    var[i] = flux*order_profile_data[i]+background_data[i];
                                    if (master_dark == NULL) {
                                        var[i] += DARK_EL[output_index] * Texp;
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                    } else {
                                        var[i] += master_dark_data[i]*GAIN[output_index] * Texp / Texp_master_dark;
                                        var[i] += (master_dark_data[i]*GAIN[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                        var[i] += (DARK_RON_EL[output_index]*DARK_RON_EL[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                    }
                                }
                                
                                do {
                                    sumPD = 0.0;
                                    sumPP = 0.0;
                                    res_max = 0.0;
                                    resmax_index = 0;
                                    
                                    /* Compute the flux */
                                    for (i = 0; i < extraction_window_size*2+1; i++) {
                                        sumPD += (order_profile_data[i]/var[i]) * mask[i] * satur_mask[i] * input_data[i] * crh_mask[i];
                                        sumPP += (order_profile_data[i] * order_profile_data[i]/var[i]) * mask[i] * satur_mask[i] * crh_mask[i];
                                    }
                                    flux = sumPD / sumPP;
                                    
                                    if (sumPP == 0.0) {
                                        flux = 0.0;
                                        for (i = 0; i < extraction_window_size*2+1; i++) {
                                            if (hot_bad_data[i] > 0) {
                                                spectrum_quality[index] = spectrum_quality[index] | hot_bad_data[i];
                                            }
                                            if (satur_data[i] > 0) {
                                                spectrum_quality[index] = spectrum_quality[index] | satur_data[i];
                                            }
                                            if (crh_mask_whole != NULL) {
                                                if (crh_data[i] > 0) {
                                                    spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                                }
                                            }
                                        }
                                        good_pxls_nb = 0;
                                    }
                                    
                                    /* Compute the variance and residuals */
                                    for (i = 0; i < extraction_window_size*2+1; i++) {
                                        var[i] = flux * order_profile_data[i] + background_data[i];
                                        if (master_dark == NULL) {
                                            var[i] += DARK_EL[output_index] * Texp;
                                            var[i] = ESPDR_MAX(0,var[i]);
                                            var[i] += RON_EL[output_index] * RON_EL[output_index];
                                        } else {
                                            var[i] += master_dark_data[i]*GAIN[output_index] * Texp / Texp_master_dark;
                                            var[i] += (master_dark_data[i]*GAIN[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                            var[i] = ESPDR_MAX(0,var[i]);
                                            var[i] += RON_EL[output_index] * RON_EL[output_index];
                                            var[i] += (DARK_RON_EL[output_index]*DARK_RON_EL[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                        }

                                        res[i] = (input_data[i] - flux*order_profile_data[i]) * mask[i] * satur_mask[i] * crh_mask[i] /
                                                    sqrt(var[i] + flux * flux * tolerance_reject * tolerance_reject);
                                        if (res[i] > res_max) {
                                            res_max = res[i];
                                            resmax_index = i;
                                        }
                                    }
                                    
                                    /* if ksig_extraction < 0.0 we don't reject cosmics */
                                    if ((res_max>ksig_extraction) && (ksig_extraction>0.0)) {
                                        mask[resmax_index] = 0;
                                        cosmics_nb++;
                                        if (good_pxls_nb > 0) { // if all the pixels are already bad - don't remove
                                            good_pxls_nb--;
                                        }
                                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                                            espdr_msg("EXTRACTION DEBUG------> cosmics sigma clipping");
                                            espdr_msg("ext: %d, order: %d, pxl: %d, res_max = %f",
                                                      ext, order+1, pxl, res_max);
                                        }
                                    }
                                } while ((res_max>ksig_extraction) && (ksig_extraction>0.0) && (good_pxls_nb>0));
                            } // if sum_prof == 0
                            
                        } else {
                            /* With the contamination image */
                            
                            espdr_msg_debug("Extracting contamination image");
                            contam_image =
                            cpl_image_extract(cpl_imagelist_get(contam, ext),
                                              pxl, window_base, pxl,
                                              window_base + extraction_window_size*2);
                            espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                         cpl_error_get_code(),
                                         "cpl_image_extract failed for contam: %s",
                                         cpl_error_get_message_default(cpl_error_get_code()));
                            
                            contam_data = cpl_image_get_data_double(contam_image);
                            espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                         cpl_error_get_code(),
                                         "cpl_image_get_data_double failed: %s",
                                         cpl_error_get_message_default(cpl_error_get_code()));
                            
                            
                            sum_prof = 0.0;
                            flux = 0.0;
                            sumC = 0.0;
                            RON_mean = 0.0;
                            good_pxls_nb = 0;
                            
                            for(i = 0; i < extraction_window_size*2+1; i++) {
                                sum_prof += order_profile_data[i] * mask[i]; // only good & saturated pixels
                                flux += input_data[i] * mask[i]; // only good & saturated pixels
                                sumC += contam_data[i] * mask[i]; // only good pixels
                                RON_mean += RON_EL[output_index] * mask[i] * satur_mask[i] * crh_mask[i];
                                if ((mask[i] == 1) && (satur_mask[i] == 1) && (crh_mask[i] == 1)) {
                                    good_pxls_nb++;
                                }
                            }
                            
                            if (good_pxls_nb == 0) {
                                RON_mean = 0.0;
                            } else {
                                RON_mean = RON_mean / good_pxls_nb;
                            }
                            
                            if (sum_prof == 0) {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    extracted_spectrum[index] += input_data[i] * mask[i] * satur_mask[i] * crh_mask[i]; // only good pixels
                                    spectrum_quality[index] = spectrum_quality[index] | CALIBRATION_DEFECT;
                                }
                                double MAXSPEC = ESPDR_MAX(0, extracted_spectrum[index];
                                spectrum_error[index] = sqrt(MAXSPEC + (extraction_window_size*2+1) * RON_EL[output_index] * RON_EL[output_index]));
                            } else {
                                
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    order_profile_data[i] = order_profile_data[i]/sum_prof;
                                    
                                    if (sumC < 2.0*(2*extraction_window_size+1)*RON_mean) {
                                        var[i] = flux * order_profile_data[i] + background_data[i];
                                    } else {
                                        var[i] = input_data[i] + background_data[i];
                                    }
                                    
                                    if (master_dark == NULL) {
                                        var[i] += DARK_EL[output_index] * Texp;
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                    } else {
                                        var[i] += master_dark_data[i]*GAIN[output_index] * Texp / Texp_master_dark;
                                        var[i] += (master_dark_data[i]*GAIN[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                        var[i] += (DARK_RON_EL[output_index]*DARK_RON_EL[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                    }
                                }
                                
                                do {
                                    sumPD = 0.0;
                                    sumPP = 0.0;
                                    sumPC = 0.0;
                                    sumCD = 0.0;
                                    sumCC = 0.0;
                                    RON_mean = 0.0;
                                    res_max = 0.0;
                                    resmax_index = 0;
                                    
                                    /* Compute the flux */
                                    for (i = 0; i < extraction_window_size*2+1; i++) {
                                        sumPD += (order_profile_data[i]/var[i]) * mask[i] * satur_mask[i] * crh_mask[i] * input_data[i];
                                        sumPP += (order_profile_data[i] *
                                                  order_profile_data[i] / var[i]) * mask[i] * satur_mask[i] * crh_mask[i];
                                        sumPC += (order_profile_data[i] * contam_data[i] /
                                                  var[i]) * mask[i];
                                        sumCD += (contam_data[i] / var[i]) * mask[i] * input_data[i];
                                        sumCC += (contam_data[i]*contam_data[i]/var[i]) * mask[i];
                                        RON_mean += RON_EL[output_index] * mask[i];
                                    }
                                    if (good_pxls_nb == 0) {
                                        RON_mean = 0.0;
                                        sumPC = 0.0;
                                        sumCD = 0.0;
                                        sumCC = 1.0;
                                    } else {
                                        RON_mean = RON_mean / good_pxls_nb;
                                    }
                                    /* If contamination is small we don't count it in */
                                    if (sumC < 2.0*(2*extraction_window_size+1)*RON_mean) {
                                        sumPC = 0.0;
                                        sumCD = 0.0;
                                        sumCC = 1.0;
                                    }
                                    
                                    flux = (sumPC * sumCD - sumPD * sumCC) / (sumPC * sumPC - sumPP * sumCC);
                                    k = (sumPC * sumPD - sumCD * sumPP) / (sumPC * sumPC - sumPP * sumCC);
                                    
                                    /* Compute the variance and residuals */
                                    for (i = 0; i < extraction_window_size*2+1; i++) {
                                        var[i] = flux * order_profile_data[i] + k * contam_data[i] + background_data[i];
                                        
                                        if (master_dark == NULL) {
                                            var[i] += DARK_EL[output_index] * Texp;
                                            var[i] = ESPDR_MAX(0,var[i]);
                                            var[i] += RON_EL[output_index] * RON_EL[output_index];
                                        } else {
                                            var[i] += master_dark_data[i]*GAIN[output_index] * Texp / Texp_master_dark;
                                            var[i] += (master_dark_data[i]*GAIN[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                            var[i] = ESPDR_MAX(0,var[i]);
                                            var[i] += RON_EL[output_index] * RON_EL[output_index];
                                            var[i] += (DARK_RON_EL[output_index]*DARK_RON_EL[output_index] * Texp*Texp / (Texp_master_dark*Texp_master_dark)) / master_dark_pxl_nb_data[i];
                                        }

                                        res[i] = (input_data[i]-flux*order_profile_data[i] - k * contam_data[i]) * mask[i] /
                                                    sqrt(var[i] + flux * flux * tolerance_reject * tolerance_reject);
                                        if (res[i] > res_max) {
                                            res_max = res[i];
                                            resmax_index = i;
                                        }
                                    }
                                    
                                    /* if ksig_extraction < 0.0 we don't reject cosmics */
                                    if ((res_max>ksig_extraction) && (ksig_extraction>0.0)) {
                                        mask[resmax_index] = 0;
                                        cosmics_nb++;
                                        if (good_pxls_nb > 0) { // if all the pixels are already bad - don't remove
                                            good_pxls_nb--;
                                        }
                                    }
                                } while ((res_max>ksig_extraction) && (ksig_extraction>0.0) && (good_pxls_nb > 0));
                                //cosmics_nb_RE[order] = cosmics_nb;
                            } // else of if sum_proof == 0.0
                            cpl_image_delete(contam_image);
                        } // end if (contam == NULL)
                        
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("sum_prof: %f, good_pxls_nb: %d, flux: %f, extr_spec: %f, error: %f, qual: %d",
                                      sum_prof, good_pxls_nb, flux, extracted_spectrum[index],
                                      spectrum_error[index], spectrum_quality[index]);
                        }
                        
                        if (sum_prof != 0.0) {
                            if (good_pxls_nb == 0) {
                                extracted_spectrum[index] = 0.0;
                                spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                                if (sumPP != 0.0) { // if sum_PP == 0.0: vicious case of 1 good pixel on the border, no cosmics
                                    spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                }
                            } else {
                                extracted_spectrum[index] = flux;
                                variance_flux = 0.0;
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    variance_flux += (order_profile_data[i] * order_profile_data[i] / var[i]) *
                                                    mask[i] * satur_mask[i] * crh_mask[i];
                                }
                                
                                spectrum_error[index] = sqrt(1.0 / variance_flux);
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("EXTRACTION DEBUG------> FINAL++++++++++++++++++++++++++++");
                            espdr_msg("ext: %d, order: %d, pxl: %d, flux: %lf, err: %f, quality: %d",
                                      ext, order, pxl, extracted_spectrum[index],
                                      spectrum_error[index], spectrum_quality[index]);
                        }
                        
                        if (isnan(extracted_spectrum[index]) || isinf(extracted_spectrum[index])) {
                            espdr_msg_warning("Extracted pixel %d is NAN or INF, fibre: %d, ext: %d, order: %d, pxl: %d",
                                              index, fibre_nr, ext, order, pxl);
                            return(CPL_ERROR_INCOMPATIBLE_INPUT);
                        }
                        
                        if (isinf(spectrum_error[index]) || isnan(spectrum_error[index])) {
                            espdr_msg_warning("Extracted pixel %d error is INF or NAN, fibre: %d, ext: %d, order: %d, pxl: %d",
                                              index, fibre_nr, ext, order, pxl);
                            return(CPL_ERROR_INCOMPATIBLE_INPUT);
                        }
                        
                        /********************/
                    } // else if (all_hot_bad || all_saturated)
                    
                    cpl_image_delete(input_extraction_image);
                    cpl_image_delete(hot_bad_image);
                    cpl_image_delete(satur_image);
                    if (crh_mask_whole != NULL) {
                        cpl_mask_delete(crh_mask_window);
                    }
                    if (background != NULL) {
                        cpl_image_delete(background_image);
                    }
                    cpl_image_delete(order_profile_image);
                    
                } else { // if (window_base > 0) && (window_base*2+1 < img_size_y)
                    
                    // Fill the rest of the extracted spectrum with 0.0
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    
                } // if (window_base > 0) && (window_base*2+1 < img_size_y)
                
                index++;
            } // end of the for loop on pixels
            
            if (inst_config->order_end[fibre_ext_index] < max_order_end) {
                for (pxl = inst_config->order_end[fibre_ext_index] + 1;
                     pxl <= max_order_end;
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            cosmics_nb_RE[order_index] = cosmics_nb;
            order_index++;
            
        } // end of for loop on orders
        cpl_free(one_order_coeffs);
    } // end of for loop on extensions
    
    //espdr_msg("Constructing extracted spectrum image");
    extracted_spectrum_image = cpl_image_wrap_double(max_order_length,
                                                     total_number_of_orders,
                                                     extracted_spectrum);
    *extr_spectr_RE = cpl_image_duplicate(extracted_spectrum_image);
    cpl_image_unwrap(extracted_spectrum_image);
    cpl_free(extracted_spectrum);
    
    spectrum_error_image = cpl_image_wrap_double(max_order_length,
                                                 total_number_of_orders,
                                                 spectrum_error);
    *spectr_error_RE = cpl_image_duplicate(spectrum_error_image);
    cpl_image_unwrap(spectrum_error_image);
    cpl_free(spectrum_error);
    
    spectrum_quality_image = cpl_image_wrap_int(max_order_length,
                                                total_number_of_orders,
                                                spectrum_quality);
    *spectr_qual_RE = cpl_image_duplicate(spectrum_quality_image);
    cpl_image_unwrap(spectrum_quality_image);
    cpl_free(spectrum_quality);
    
    if (background == NULL) {
        cpl_free(background_data);
    }
    cpl_free(mask);
    cpl_free(corr_mask);
    cpl_free(satur_mask);
    cpl_free(crh_mask);
    cpl_free(var);
    cpl_free(res);
    cpl_imagelist_delete(geometry_corrected_master_dark);
    cpl_imagelist_delete(geometry_corrected_master_dark_pxl_nb);
    
    espdr_msg_debug("EOF: extr_spectr_RE size: %lld x %lld",
                    cpl_image_get_size_x(*extr_spectr_RE),
                    cpl_image_get_size_y(*extr_spectr_RE));
    
    return cpl_error_get_code();
}

