/*                                                                           a
 *   This file is part of the ESO X-shooter Pipeline                         *
 *   Copyright (C) 2006 European Southern Observatory                        *
 *                                                                           *
 *   This library is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the Free Software             *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA    *
 *                                                                           */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ESPDR_INSTRUMENT_H
#define ESPDR_INSTRUMENT_H

/*----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include <string.h>
#include <cpl.h>

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

/* Input filenames lengths */
#define FILENAME_LENGTH 512
#define KEYWORD_LENGTH 64
#define CCD_GEOM_FILENAME_LENGTH	FILENAME_LENGTH
#define RAW_FRAME_FILENAME_LENGTH	FILENAME_LENGTH
#define MAX_NB_OF_FIBRES_AND_EXTS   100
#define MAX_NB_OF_FIBRES            5
#define MAX_NB_OF_EXTS              4
//#define MASK_LUT_LENGTH            35


typedef enum {VIS, NIR} espdr_inst_type;

/* Structure holding the instrument parameters with up to 5 fibres */

typedef struct {
    char instrument[10];
    espdr_inst_type inst_type;
    char prefix[5];
    // filenames
    char master_bias_filename[FILENAME_LENGTH];
    char master_overscan_filename[FILENAME_LENGTH];
    char master_bias_res_filename[FILENAME_LENGTH];
    char single_bias_filename[FILENAME_LENGTH];
    char single_bias_res_filename[FILENAME_LENGTH];
    char hot_pixels_filename[FILENAME_LENGTH];
    char master_dark_filename[FILENAME_LENGTH];
    char master_flat_filename[FILENAME_LENGTH];
    char flat_filename[FILENAME_LENGTH];	
    char flat_filename_no_fits[FILENAME_LENGTH];	
    char master_flat_order_profile_filename[FILENAME_LENGTH];
    char master_flat_order_profile_filename_no_fits[FILENAME_LENGTH];
    char flat_blaze_filename[FILENAME_LENGTH];	
    char flat_blaze_filename_no_fits[FILENAME_LENGTH];	
    char bad_pixels_filename[FILENAME_LENGTH];
    char master_led_ff_filename[FILENAME_LENGTH];
    char orders_map_filename[FILENAME_LENGTH];
    char orders_map_filename_no_fits[FILENAME_LENGTH];
    char master_wave_filename[FILENAME_LENGTH];
    char extr_spectr_filename[FILENAME_LENGTH];
    char extr_spectr_filename_no_fits[FILENAME_LENGTH];
    char line_table_filename[FILENAME_LENGTH];
    char line_table_filename_no_fits[FILENAME_LENGTH];
    char fp_fitted_line_table_filename[FILENAME_LENGTH];
    char fp_fitted_line_table_filename_no_fits[FILENAME_LENGTH];
    char fp_searched_line_table_filename[FILENAME_LENGTH];
    char fp_searched_line_table_filename_no_fits[FILENAME_LENGTH];
    char lfc_fitted_line_table_filename[FILENAME_LENGTH];
    char lfc_fitted_line_table_filename_no_fits[FILENAME_LENGTH];
    char thar_line_table_filename[FILENAME_LENGTH];
    char thar_line_table_filename_no_fits[FILENAME_LENGTH];
    char th_line_table_filename[FILENAME_LENGTH];
    char th_line_table_filename_no_fits[FILENAME_LENGTH];
    char ar_line_table_filename[FILENAME_LENGTH];
    char ar_line_table_filename_no_fits[FILENAME_LENGTH];
    char line_table_raw_filename[FILENAME_LENGTH];
    char line_table_raw_filename_no_fits[FILENAME_LENGTH];
    char wave_table_filename[FILENAME_LENGTH];
    char wave_table_filename_no_fits[FILENAME_LENGTH];
    char wave_matrix_filename[FILENAME_LENGTH];
    char wave_matrix_filename_no_fits[FILENAME_LENGTH];
    char dll_matrix_filename_no_fits[FILENAME_LENGTH];
    char air_wave_matrix_filename[FILENAME_LENGTH];
    char air_wave_matrix_filename_no_fits[FILENAME_LENGTH];
    char air_dll_matrix_filename_no_fits[FILENAME_LENGTH];
    char wave_matrix_drift_filename[FILENAME_LENGTH];
    char wave_matrix_drift_filename_no_fits[FILENAME_LENGTH];
    char dll_matrix_drift_filename_no_fits[FILENAME_LENGTH];
    char air_wave_matrix_drift_filename[FILENAME_LENGTH];
    char air_wave_matrix_drift_filename_no_fits[FILENAME_LENGTH];
    char air_dll_matrix_drift_filename_no_fits[FILENAME_LENGTH];
    char res_map_filename[FILENAME_LENGTH];
    char res_map_filename_no_fits[FILENAME_LENGTH];
    char lfc_wave_table_filename[FILENAME_LENGTH];
    char lfc_wave_table_filename_no_fits[FILENAME_LENGTH];
    char lfc_wave_matrix_filename[FILENAME_LENGTH];
    char lfc_wave_matrix_filename_no_fits[FILENAME_LENGTH];
    char lfc_dll_matrix_filename_no_fits[FILENAME_LENGTH];
    char lfc_air_wave_matrix_filename[FILENAME_LENGTH];
    char lfc_air_wave_matrix_filename_no_fits[FILENAME_LENGTH];
    char lfc_air_dll_matrix_filename_no_fits[FILENAME_LENGTH];
    // keywords
    char Texp_kw[KEYWORD_LENGTH];
    char tmmean_kw[KEYWORD_LENGTH];
    char conad_kw_first_part[KEYWORD_LENGTH];
    char conad_kw_last_part[KEYWORD_LENGTH];
    char det_binx_kw[KEYWORD_LENGTH];
    char det_biny_kw[KEYWORD_LENGTH];
    char tel_alpha_kw[KEYWORD_LENGTH];
    char tel_delta_kw[KEYWORD_LENGTH];
    char tel_pma_kw[KEYWORD_LENGTH];
    char tel_pmd_kw[KEYWORD_LENGTH];
    char tel_date_obs_kw[KEYWORD_LENGTH];
    char targ_name_kw[KEYWORD_LENGTH];
    char spec_type_kw[KEYWORD_LENGTH];
    char zem_kw[KEYWORD_LENGTH];
    char wavecal_src_kw[KEYWORD_LENGTH];
    char rv_center_kw[KEYWORD_LENGTH];
    char airm_start_kw[KEYWORD_LENGTH];
    char airm_end_kw[KEYWORD_LENGTH];
    char pres_start_kw[KEYWORD_LENGTH];
    char pres_end_kw[KEYWORD_LENGTH];
    char temp_kw[KEYWORD_LENGTH];
    char dpr_type_kw[KEYWORD_LENGTH];
    char rep_rate_kw[KEYWORD_LENGTH];
    char anchor_freq_kw[KEYWORD_LENGTH];
    char seeing_kw_first_part[KEYWORD_LENGTH];
    char seeing_kw_last_part[KEYWORD_LENGTH];
    // General
    double fibre_size;
    double vltsh_wavelength;
    double vltsh_iq;
    double inst_iq;
    // BIAS
    int raw_bias_limit_nb;
    double bias_ron_limit;
    double bias_mean_limit;
    double bias_ksigma;
    char bias_sigma_clipping_method[32];
    double ovsc_ksigma;
    char ovsc_sigma_clipping_method[32];
    int ovsc_max_iter;
    double stat_ksigma;
    char stat_sigma_clipping_method[32];
    int stat_max_iter;
    double bias_residuals_mean_limit;
    double bias_residuals_stdev_limit;
    // CTE
    int cte_flag;
    double cte_alpha[MAX_NB_OF_EXTS];
    double cte_beta[MAX_NB_OF_EXTS];
    // DARK
    int raw_dark_limit_nb;
    double dark_current_limit_el;
    int hot_pixels_limit;
    double cosmics_limit_per_pixel;
    double dark_ksigma;
    char dark_sigma_clipping_method[32];
    double hotpixels_ksigma;
    char hotpixels_sigma_clipping_method[32];
    int hotpixels_max_iter;
    // LED_FF
    int raw_led_ff_per_texp_limit_nb;
    int raw_led_ff_different_texp;
    int bad_pixels_limit;
    double flux_min_limit;
    double flux_max_limit;
    double gain_tolerance;
    double dead_pixel_flux_limit;
    double led_ff_ksigma;
    char led_ff_sigma_clipping_method[32];
    double badpixels_ksigma;
    char badpixels_sigma_clipping_method[32];
    int badpixels_max_iter;
    double gain_ksigma;
    char gain_sigma_clipping_method[32];
    int gain_max_iter;
    // ORDERDEF
    double satur_limit_adu;
    double res_stdev_limit;
    double res_min_limit;
    double res_max_limit;
    int slices_nb;
    int inter_slice_dist;
    int slice_width;
    int order_width;
    int slices_nb_per_phys_order;
    int background_pxls;
    int fibres_nb;
    int orders_nb[MAX_NB_OF_FIBRES_AND_EXTS];
    int order_phys_nb[MAX_NB_OF_FIBRES_AND_EXTS];
    int order_start[MAX_NB_OF_FIBRES_AND_EXTS];
    int order_end[MAX_NB_OF_FIBRES_AND_EXTS];
    double res_ksigma;
    char res_sigma_clipping_method[32];
    int res_max_iter;
    int cluster_dist_x;
    int cluster_dist_y;
    int min_order_y_dist;
    int min_order_len[MAX_NB_OF_FIBRES_AND_EXTS];
    int min_order_width;
    int min_width_pixels_nb[MAX_NB_OF_FIBRES_AND_EXTS];
    int min_order_x[MAX_NB_OF_FIBRES_AND_EXTS];
    int max_order_y[MAX_NB_OF_FIBRES_AND_EXTS];
    int orders_detection_start[MAX_NB_OF_FIBRES_AND_EXTS];
    int order_fit_step;
    int order_fit_poly_deg;
    double order_flux_limit;
    double order_flux_factor[MAX_NB_OF_FIBRES_AND_EXTS];
    double image_cosmics_part;
    double image_thar_part;
    // Extraction
    double extraction_bkgr_calc_fraction;
    double extraction_fraction_valid_pixels;
    char extraction_method[32];
    double extraction_ksigma_tun;
    double extraction_ksigma_thar;
    double extraction_ksigma_fp;
    double extraction_ksigma_lfc;
    double extraction_ksigma_object;
    double extraction_ksigma_sky;
    double extraction_ksigma_dark;
    int extraction_window_size[MAX_NB_OF_FIBRES];
    double tolerance_rejection;
    // FLAT
    int raw_flat_limit_nb;
    double flat_current_limit_el;
    double flat_snr_limit;
    double flat_rms_limit;
    double flat_bkgr_limit;
    int bkgr_grid_size_x;
    int bkgr_grid_size_y;
    int minlevel_factor;
    double flat_ksigma;
    char flat_sigma_clipping_method[32];
    char blaze_flat_calc_method[32];
    int blaze_flat_calc_poly_deg;
    int blaze_flat_calc_slide_box;
    double blaze_flat_calc_clip;
    char blaze_peaks_fit_method[32];
    int blaze_peaks_fit_poly_deg;
    int blaze_peaks_fit_slide_box;
    // WAVE general
    double line_max_sig_x0;
    double line_min_sig_x0;
    double line_max_fwhm;
    double line_min_flux_el;
    int wave_sol_poly_deg;
    double wave_sol_FP_reject_alpha;
    double wave_sol_FP_reject_beta;
    double thar_min_flux_ratio[MAX_NB_OF_FIBRES];
    double thar_max_flux_ratio[MAX_NB_OF_FIBRES];
    // WAVE FP
    int fp_fit_window_size;
    int fp_fit_tolerance_window_size;
    int fp_fit_ngauss_flag;
    int fp_search_window_size;
    double delta_fp_tolerance;
    double delta_fp_peak_limit;
    int fp_ccd_border_cut;
    double fp_flux_threshold;
    double fp_noise_threshold;
    double fp_peak_flux_threshold;
    double fp_peak_rms_threshold;
    double fp_spectral_line_qc;
    int fp_min_lines_total;
    int fp_add_fake_peaks;
    // WAVE THAR
    int thar_thar_fit_window_size;
    int thar_fp_fit_window_size;
    int thar_fit_tolerance_window_size;
    int thar_ccd_border_cut;
    double thar_flux_threshold;
    double thar_noise_threshold;
    double thar_peak_flux_threshold;
    double thar_spectral_line_qc;
    double fp_d_angstrom;
    int d_fit_poly_deg;
    double d_fit_big_res_limit;
    double d_fit_small_res_limit;
    int d_fit_kernel_poly_deg;
    double d_fit_kernel_fwhm;
    double thar_sol_max_chi2;
    double thar_sol_max_rms;
    double wave_min_resol;
    int wave_sol_min_lines_per_order;
    int thar_min_lines_total;
    int thar_drift_th_fp_sw;
    int thar_drift_fp_th_sw;
    double thar_drift_chi2_limit;
    int wave_orders_analytical_link_sw;
    // WAVE TH DRIFT
    double th_lamp_offset_ar;
    double th_lamp_offset_ar1;
    double th_lamp_offset_ar2;
    int th_drift_min_lines_total[MAX_NB_OF_FIBRES];
    int th_drift_max_satur_lines;
    int th_drift_fit_deg_x;
    int th_drift_fit_deg_y;
    double th_drift_fit_ksigma;
    double th_drift_fit_max_mad;
    double th_drift_max_reject_lines_fraction;
    // WAVE LFC
    double lfc_anchor_freq;
    double lfc_repetition_rate;
    double lfc_spectral_line_qc;
    double lfc_sol_max_chi2;
    double lfc_sol_max_rms;
    int lfc_sol_min_lines_per_order[MAX_NB_OF_FIBRES];
    int lfc_min_lines_total;
    // CONTAM
    int contam_bkgr_grid_size_x;
    int contam_bkgr_grid_size_y;
    double contam_flux_limit;
    // EFF_AB
    double eff_ab_snr_limit;
    double eff_ab_bkgr_limit;
    int eff_ab_bkgr_grid_size_x;
    int eff_ab_bkgr_grid_size_y;
    int eff_ab_poly_deg;
    double rel_eff_min_limit;
    double rel_eff_max_limit;
    // FLUX
    char flux_eff_interp_method[32];
    int flux_poly_deg;
    double flux_snr_limit;
    double abs_eff_min_limit;
    double abs_eff_max_limit;
    // SCIENCE
    int sci_bkgr_grid_size_x;
    int sci_bkgr_grid_size_y;
    int snr_averaging_window;
    int berv_type_init;
    double tel_geolon;
    double tel_geolat;
    double tel_geoelev;
    double mask_width;
    double log_lambda_ini;
    double delta_log_lambda;
    double log_lambda_end;
    char default_spec_type[5];
    double rv_range;
    double rv_step;
    double tel_surface;	
    char sci_sky_corr_method[32];
    int sci_sky_sub_sliding_box_size;
    int oh_corr_lines_nb_limit;
    double oh_corr_ratio_lower_limit;
    double oh_corr_ratio_upper_limit;
    double oh_corr_ratio_mad_limit;
    int lacosmic_sky_sw;
    int lacosmic_calib_sw;
    int lacosmic_post_filter_x;
    int lacosmic_post_filter_y;
    char lacosmic_post_filter_mode[32];
    double lacosmic_sigma_lim;
    double lacosmic_f_lim;
    int lacosmic_max_iter;
    // DRIFT
    double drift_ksigma;
    char drift_method_thar[64];
    char drift_method_fp[64];
    char drift_method_lfc[64];
    char drift_space[32];
    double drift_max_flux_threshold;
    double drift_min_flux_threshold;
    double drift_max_flux_ratio;
    double drift_min_flux_ratio;
    double drift_mean_threshold;
    double drift_mean_err_threshold;
    double drift_chi2_threshold;
} espdr_inst_config;

//static const char COL_NAME_INST_NAME[] = "INST_NAME";
static const char COL_NAME_INST_TYPE[] = "INST_TYPE";
static const char COL_NAME_PREFIX_KW[] = "PREFIX_KW";

// General KWs
/* Fibre size */
static const char COL_NAME_FIBRE_SIZE[] = "FIBRE_SIZE";
/* Effective wavelength of the VLTSH */
static const char COL_NAME_VLTSH_WAVELENGTH[] = "VLTSH_WAVELENGTH";
/* Image Quality of the VLTSH */
static const char COL_NAME_VLTSH_IQ[] = "VLTSH_IQ";
/* Image Quality of the instrument */
static const char COL_NAME_INST_IQ[] = "INST_IQ";

//BIAS:
/* Lower limit of the number of the input raw bias frames */
//#define RAW_BIAS_LIMIT_NUMBER	5
static const char COL_NAME_RAW_BIAS_LIMIT_NUMBER[] = "RAW_BIAS_LIMIT_NUMBER";
/* Upper limit for MASTER BIAS RON */
//#define BIAS_RON_LIMIT  5.0
static const char COL_NAME_BIAS_RON_LIMIT[] = "BIAS_RON_LIMIT";
/* Upper limit on MASTER BIAS mean */
//#define BIAS_MEAN_LIMIT 300.0
static const char COL_NAME_BIAS_MEAN_LIMIT[] = "BIAS_MEAN_LIMIT";
/* sigma clipping ksigma for the bias stacking */
static const char COL_NAME_BIAS_KSIGMA[] = "BIAS_KSIGMA";
/* sigma clipping method for bias stacking */
static const char COL_NAME_BIAS_SIGMA_CLIPPING_METHOD[] = "BIAS_SIGMA_CLIPPING_METHOD";
/* sigma clipping ksigma for the overscan */
static const char COL_NAME_OVSC_KSIGMA[] = "OVSC_KSIGMA";
/* sigma clipping method for the overscan */
static const char COL_NAME_OVSC_SIGMA_CLIPPING_METHOD[] = "OVSC_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the overscan */
static const char COL_NAME_OVSC_MAX_ITER[] = "OVSC_MAX_ITER";
/* sigma clipping ksigma for the bias test */
static const char COL_NAME_STAT_KSIGMA[] = "STAT_KSIGMA";
/* sigma clipping method for the bias test */
static const char COL_NAME_STAT_SIGMA_CLIPPING_METHOD[] = "STAT_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the bias test */
static const char COL_NAME_STAT_MAX_ITER[] = "STAT_MAX_ITER";
/* Upper limit on mean of residuals */
static const char COL_NAME_BIAS_RESIDUALS_MEAN_LIMIT[] = "BIAS_RESIDUALS_MEAN_LIMIT";
/* tolerance on residuals stdev */
static const char COL_NAME_BIAS_RESIDUALS_STDEV_LIMIT[] = "BIAS_RESIDUALS_STDEV_LIMIT";

//CTE
/* ALPHA and BETA parameters for the CTE correction */
static const char COL_NAME_CTE_FLAG[] = "CTE_FLAG";
static const char COL_NAME_CTE_ALPHA[] = "CTE_ALPHA";
static const char COL_NAME_CTE_BETA[] = "CTE_BETA";

//DARK:
/* Lower limit of the number of the input raw dark frames */
//#define RAW_DARK_LIMIT_NUMBER	5
static const char COL_NAME_RAW_DARK_LIMIT_NUMBER[] = "RAW_DARK_LIMIT_NUMBER";
/* Maximum dark current allowed [e-] */
//#define DARK_CURRENT_LIMIT		5
static const char COL_NAME_DARK_CURRENT_LIMIT_EL[] = "DARK_CURRENT_LIMIT_EL";
/* Maximum number of hot pixels allowed */
//#define HOT_PIXELS_LIMIT		3000
static const char COL_NAME_HOT_PIXELS_LIMIT[] = "HOT_PIXELS_LIMIT";
/* Maximum cosmics per pixel allowed */
//#define COSMICS_LIMIT_PER_PIXEL 0.01
static const char COL_NAME_COSMICS_LIMIT_PER_PIXEL[] = "COSMICS_LIMIT_PER_PIXEL";
/* sigma clipping ksigma for the dark stacking */
static const char COL_NAME_DARK_KSIGMA[] = "DARK_KSIGMA";
/* sigma clipping method for the dark stacking */
static const char COL_NAME_DARK_SIGMA_CLIPPING_METHOD[] = "DARK_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the data stacking */
static const char COL_NAME_DARK_MAX_ITER[] = "DARK_MAX_ITER";
/* sigma clipping ksigma for the hotpixels */
static const char COL_NAME_HOTPIXELS_KSIGMA[] = "HOTPIXELS_KSIGMA";
/* sigma clipping method for the hotpixels */
static const char COL_NAME_HOTPIXELS_SIGMA_CLIPPING_METHOD[] = "HOTPIXELS_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the hotpixels */
static const char COL_NAME_HOTPIXELS_MAX_ITER[] = "HOTPIXELS_MAX_ITER";

//LED_FF:
/* Lower limit of the number of the input led_ff frames with the same Texp */
static const char COL_NAME_RAW_LED_FF_PER_TEXP_LIMIT_NUMBER[] =
                                            "RAW_LED_FF_PER_TEXP_LIMIT_NUMBER";
/* Minimum number of different Texp within input led_ff frames */
static const char COL_NAME_RAW_LED_FF_DIFFERENT_TEXP[] =
                                            "RAW_LED_FF_DIFFERENT_TEXP";
/* Maximum accepted number of bad pixels */
static const char COL_NAME_BAD_PIXELS_LIMIT[] = "BAD_PIXELS_LIMIT";
/* Minimum accepted flux */
static const char COL_NAME_FLUX_MIN_LIMIT[] = "FLUX_MIN_LIMIT";
/* Maximum accepted flux - saturation limit */
static const char COL_NAME_FLUX_MAX_LIMIT[] = "FLUX_MAX_LIMIT";
/* Gain tolerance */
static const char COL_NAME_GAIN_TOLERANCE[] = "GAIN_TOLERANCE";
/* Flux limit in dead pixels */
static const char COL_NAME_DEAD_PIXEL_FLUX_LIMIT[] = "DEAD_PIXEL_FLUX_LIMIT";
/* sigma clipping ksigma for the led_ff stacking */
static const char COL_NAME_LED_FF_KSIGMA[] = "LED_FF_KSIGMA";
/* sigma clipping method for the led_ff stacking */
static const char COL_NAME_LED_FF_SIGMA_CLIPPING_METHOD[] = "LED_FF_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the led_ff stacking */
static const char COL_NAME_LED_FF_MAX_ITER[] = "LED_FF_MAX_ITER";
/* sigma clipping ksigma for the badpixels */
static const char COL_NAME_BADPIXELS_KSIGMA[] = "BADPIXELS_KSIGMA";
/* sigma clipping method for the badpixels */
static const char COL_NAME_BADPIXELS_SIGMA_CLIPPING_METHOD[] = "BADPIXELS_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the badpixels */
static const char COL_NAME_BADPIXELS_MAX_ITER[] = "BADPIXELS_MAX_ITER";
/* sigma clipping ksigma for the gain stacking */
static const char COL_NAME_GAIN_KSIGMA[] = "GAIN_KSIGMA";
/* sigma clipping method for the gain stacking */
static const char COL_NAME_GAIN_SIGMA_CLIPPING_METHOD[] = "GAIN_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the gain stacking */
static const char COL_NAME_GAIN_MAX_ITER[] = "GAIN_MAX_ITER";

//ORDERDEF:
/* Limits for QC header KWs */
static const char COL_NAME_SATUR_LIMIT_ADU[] = "SATUR_LIMIT_ADU";
static const char COL_NAME_RES_STDEV_LIMIT[] = "RES_STDEV_LIMIT";
static const char COL_NAME_RES_MIN_LIMIT[] = "RES_MIN_LIMIT";
static const char COL_NAME_RES_MAX_LIMIT[] = "RES_MAX_LIMIT";
/* Nb of slices treated */
static const char COL_NAME_SLICES_NB[] = "SLICES_NB";
/* Inter-slice distance */
static const char COL_NAME_INTER_SLICE_DIST[] = "INTER_SLICE_DIST";
/* Slice width */
static const char COL_NAME_SLICE_WIDTH[] = "SLICE_WIDTH";
/* Order width */
static const char COL_NAME_ORDER_WIDTH[] = "ORDER_WIDTH";
/* Nb of slices per physical order */
static const char COL_NAME_SLICES_NB_PER_PHYS_ORDER[] = "SLICES_NB_PER_PHYS_ORDER";
/* Nb of background pixels */
static const char COL_NAME_BACKGROUND_PXLS[] = "BACKGROUND_PXLS";
/* Nb of fibers in the instrument */
static const char COL_NAME_FIBRES_NB[] = "FIBRES_NB";
/* Nb of orders per fiber per extension */
//#define ORDERS_NB 70 70
static const char COL_NAME_ORDERS_NB[] = "ORDERS_NB";
/* Nb of first physical order per fiber per extension */
static const char COL_NAME_ORDER_PHYS_NB[] = "ORDER_PHYS_NB";
/* Order limits per fibre per extension */
static const char COL_NAME_ORDER_START[] = "ORDER_START";
static const char COL_NAME_ORDER_END[] = "ORDER_END";
/* sigma clipping ksigma for the residuals */
static const char COL_NAME_RES_KSIGMA[] = "RES_KSIGMA";
/* sigma clipping method for the residuals */
static const char COL_NAME_RES_SIGMA_CLIPPING_METHOD[] = "RES_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the residuals */
static const char COL_NAME_RES_MAX_ITER[] = "RES_MAX_ITER";
/* cluster distance for clumps identification */
static const char COL_NAME_CLUSTER_DIST_X[] = "CLUSTER_DIST_X";
static const char COL_NAME_CLUSTER_DIST_Y[] = "CLUSTER_DIST_Y";
/* distance from CCD upper and lower borders for orders identification */
static const char COL_NAME_MIN_ORDER_Y_DIST[] = "MIN_ORDER_Y_DIST";
/* minimal order length for orders identification */
static const char COL_NAME_MIN_ORDER_LEN[] = "MIN_ORDER_LEN";
/* minimal order width for orders identification */
static const char COL_NAME_MIN_ORDER_WIDTH[] = "MIN_ORDER_WIDTH";
/* number o fpixels in the order with width > min width */
static const char COL_NAME_MIN_WIDTH_PIXELS_NB[] = "MIN_WIDTH_PIXELS_NB";
/* smallest coordinate X of the order allowed */
static const char COL_NAME_MIN_ORDER_X[] = "MIN_ORDER_X";
/* largest coordinate Y of the order allowed */
static const char COL_NAME_MAX_ORDER_Y[] = "MAX_ORDER_Y";
/* Start (in y on CCD cleaned image) of orders detection per fibre per extension */
static const char COL_NAME_ORDERS_DETECTION_START[] = "ORDERS_DETECTION_START";
/* order fit step */
static const char COL_NAME_ORDER_FIT_STEP[] = "ORDER_FIT_STEP";
/* order fit plynomial degree */
static const char COL_NAME_ORDER_FIT_POLY_DEG[] = "ORDER_FIT_POLY_DEG";
/* flux limit for orders detection */
static const char COL_NAME_ORDER_FLUX_LIMIT[] = "ORDER_FLUX_LIMIT";
/* flux factor for orders detection */
static const char COL_NAME_ORDER_FLUX_FACTOR[] = "ORDER_FLUX_FACTOR";
/* part of highest pixels estimated to be cosmics rejected for flux estimation */
static const char COL_NAME_IMAGE_COSMICS_PART[] = "IMAGE_COSMICS_PART";
/* part of highest pixels estimated to be saturated THAR lines rejected for flux estimation */
static const char COL_NAME_IMAGE_THAR_PART[] = "IMAGE_THAR_PART";
/* Peak and fwhm for the blaze gaussian shape of the background map */
static const char COL_NAME_BKGR_MAP_BLAZE_PEAK[] = "BKGR_MAP_BLAZE_PEAK";
static const char COL_NAME_BKGR_MAP_BLAZE_FWHM[] = "BKGR_MAP_BLAZE_FWHM";

// Extraction
/* fraction of the order width for background calculation */
static const char COL_NAME_EXTRACTION_BKGR_CALC_FRACTION[] = "EXTRACTION_BKGR_CALC_FRACTION";
/* min number of valid pixels to interplate the bad ones */
static const char COL_NAME_EXTRACTION_FRACTION_VALID_PIXELS[] = "EXTRACTION_FRACTION_VALID_PIXELS";
/* extraction parameters */
static const char COL_NAME_EXTRACTION_METHOD[] = "EXTRACTION_METHOD";
static const char COL_NAME_EXTRACTION_KSIGMA_TUN[] = "EXTRACTION_KSIGMA_TUN";
static const char COL_NAME_EXTRACTION_KSIGMA_THAR[] = "EXTRACTION_KSIGMA_THAR";
static const char COL_NAME_EXTRACTION_KSIGMA_FP[] = "EXTRACTION_KSIGMA_FP";
static const char COL_NAME_EXTRACTION_KSIGMA_LFC[] = "EXTRACTION_KSIGMA_LFC";
static const char COL_NAME_EXTRACTION_KSIGMA_OBJECT[] = "EXTRACTION_KSIGMA_OBJECT";
static const char COL_NAME_EXTRACTION_KSIGMA_SKY[] = "EXTRACTION_KSIGMA_SKY";
static const char COL_NAME_EXTRACTION_KSIGMA_DARK[] = "EXTRACTION_KSIGMA_DARK";
static const char COL_NAME_TOLERANCE_REJECTION[] = "TOLERANCE_REJECTION";
static const char COL_NAME_EXTRACTION_WINDOW_SIZE[] = "EXTRACTION_WINDOW_SIZE";

// FLAT
/* Lower limit of the number of the input raw flat frames */
static const char COL_NAME_RAW_FLAT_LIMIT_NUMBER[] = "RAW_FLAT_LIMIT_NUMBER";
/* Maximum flat current allowed [e-] */
static const char COL_NAME_FLAT_CURRENT_LIMIT_EL[] = "FLAT_CURRENT_LIMIT_EL";
/* Minimum allowed SNR */
static const char COL_NAME_FLAT_SNR_LIMIT[] = "FLAT_SNR_LIMIT";
/* Maximum allowed RMS */
static const char COL_NAME_FLAT_RMS_LIMIT[] = "FLAT_RMS_LIMIT";
/* Maximum allowed ratio between background amd max flux */
static const char COL_NAME_FLAT_BKGR_LIMIT[] = "FLAT_BKGR_LIMIT";
/* Grid size in X and Y, this is the sub-region size of the stacked flats frames
 * product destinated to measure the inter-order backgroud */
static const char COL_NAME_BKGR_GRID_SIZE_X[] = "BKGR_GRID_SIZE_X";
static const char COL_NAME_BKGR_GRID_SIZE_Y[] = "BKGR_GRID_SIZE_Y";
/* Minimun level flux detection factor, this is destinated to tune the minimun
 * flux detection in the grid previously define */
static const char COL_NAME_MINLEVEL_FACTOR[] = "MINLEVEL_FACTOR";
/* sigma clipping ksigma for the flat stacking */
static const char COL_NAME_FLAT_KSIGMA[] = "FLAT_KSIGMA";
/* sigma clipping method for the flat stacking */
static const char COL_NAME_FLAT_SIGMA_CLIPPING_METHOD[] = "FLAT_SIGMA_CLIPPING_METHOD";
/* maximum iterations of sigma clipping for the flat stacking */
static const char COL_NAME_FLAT_MAX_ITER[] = "FLAT_MAX_ITER";
/* blaze and flat calculation method */
static const char COL_NAME_BLAZE_FLAT_CALC_METHOD[] = "BLAZE_FLAT_CALC_METHOD";
/* blaze and flat calculation polynomial degree */
static const char COL_NAME_BLAZE_FLAT_CALC_POLY_DEG[] = "BLAZE_FLAT_CALC_POLY_DEG";
/* blaze and flat calculation sliding box size */
static const char COL_NAME_BLAZE_FLAT_CALC_SLIDE_BOX[] = "BLAZE_FLAT_CALC_SLIDE_BOX";
/* blaze and flat calculation clip */
static const char COL_NAME_BLAZE_FLAT_CALC_CLIP[] = "BLAZE_FLAT_CALC_CLIP";
/* blaze peaks fit method: polynomial or spline */
static const char COL_NAME_BLAZE_PEAKS_FIT_METHOD[] = "BLAZE_PEAKS_FIT_METHOD";
/* blaze peaks polynomial fit degree */
static const char COL_NAME_BLAZE_PEAKS_FIT_POLY_DEG[] = "BLAZE_PEAKS_FIT_POLY_DEG";
/* blaze peaks sliding box size */
static const char COL_NAME_BLAZE_PEAKS_FIT_SLIDE_BOX[] = "BLAZE_PEAKS_FIT_SLIDE_BOX";

//WAVE general
/* Maximum allowed line position uncertainty */
static const char COL_NAME_LINE_MAX_SIG_X0[] = "LINE_MAX_SIG_X0";
/* Minimum allowed line position uncertainty */
static const char COL_NAME_LINE_MIN_SIG_X0[] = "LINE_MIN_SIG_X0";
/* Maximum allowed line FWHM */
static const char COL_NAME_LINE_MAX_FWHM[] = "LINE_MAX_FWHM";
/* Minimum allowed peak flux */
static const char COL_NAME_LINE_MIN_FLUX_EL[] = "LINE_MIN_FLUX_EL";
/* Polynomial degree for wavelangth solution */
static const char COL_NAME_WAVE_SOL_POLY_DEG[] = "WAVE_SOL_POLY_DEG";
/* Wavelength solution: coeffs in FP lines rejection */
static const char COL_NAME_WAVE_SOL_FP_REJECT_ALPHA[] = "WAVE_SOL_FP_REJECT_ALPHA";
static const char COL_NAME_WAVE_SOL_FP_REJECT_BETA[] = "WAVE_SOL_FP_REJECT_BETA";
/* ThAr lines flux ratio min & max */
static const char COL_NAME_THAR_MIN_FLUX_RATIO[] = "THAR_MIN_FLUX_RATIO";
static const char COL_NAME_THAR_MAX_FLUX_RATIO[] = "THAR_MAX_FLUX_RATIO";

// WAVE FP
/* FP lines fit window size */
static const char COL_NAME_FP_FIT_WINDOW_SIZE[] = "FP_FIT_WINDOW_SIZE";
/* FP lines fit tolerance window size */
static const char COL_NAME_FP_FIT_TOLERANCE_WINDOW_SIZE[] = "FP_FIT_TOLERANCE_WINDOW_SIZE";
/* FP lines fit NGauss flag */
static const char COL_NAME_FP_FIT_NGAUSS_FLAG[] = "FP_FIT_NGAUSS_FLAG";
/* FP lines search window size */
static const char COL_NAME_FP_SEARCH_WINDOW_SIZE[] = "FP_SEARCH_WINDOW_SIZE";
/* Distance between FP peaks tolerance */
static const char COL_NAME_DELTA_FP_TOLERANCE[] = "DELTA_FP_TOLERANCE";
/* Distance between FP peaks limit */
static const char COL_NAME_DELTA_FP_PEAK_LIMIT[] = "DELTA_FP_PEAK_LIMIT";
/* */
static const char COL_NAME_FP_CCD_BORDER_CUT[] = "FP_CCD_BORDER_CUT";
/* */
static const char COL_NAME_FP_FLUX_THRESHOLD[] = "FP_FLUX_THRESHOLD";
/* */
static const char COL_NAME_FP_NOISE_THRESHOLD[] = "FP_NOISE_THRESHOLD";
/* */
static const char COL_NAME_FP_PEAK_FLUX_THRESHOLD[] = "FP_PEAK_FLUX_THRESHOLD";
/* */
static const char COL_NAME_FP_PEAK_RMS_THRESHOLD[] = "FP_PEAK_RMS_THRESHOLD";
/* */
static const char COL_NAME_FP_SPECTRAL_LINE_QC[] = "FP_SPECTRAL_LINE_QC";
/* Minimum total lines number */
static const char COL_NAME_FP_MIN_LINES_TOTAL[] = "FP_MIN_LINES_TOTAL";
/* */
static const char COL_NAME_FP_ADD_FAKE_PEAKS[] = "FP_ADD_FAKE_PEAKS";

// WAVE THAR
/* THAR lines fit window size */
static const char COL_NAME_THAR_THAR_FIT_WINDOW_SIZE[] = "THAR_THAR_FIT_WINDOW_SIZE";
/* FP lines fit window size */
static const char COL_NAME_THAR_FP_FIT_WINDOW_SIZE[] = "THAR_FP_FIT_WINDOW_SIZE";
/* THAR lines fit tolerance window size */
static const char COL_NAME_THAR_FIT_TOLERANCE_WINDOW_SIZE[] = "THAR_FIT_TOLERANCE_WINDOW_SIZE";
/* */
static const char COL_NAME_THAR_CCD_BORDER_CUT[] = "THAR_CCD_BORDER_CUT";
/* */
static const char COL_NAME_THAR_FLUX_THRESHOLD[] = "THAR_FLUX_THRESHOLD";
/* */
static const char COL_NAME_THAR_NOISE_THRESHOLD[] = "THAR_NOISE_THRESHOLD";
/* */
static const char COL_NAME_THAR_PEAK_FLUX_THRESHOLD[] = "THAR_PEAK_FLUX_THRESHOLD";
/* */
static const char COL_NAME_THAR_SPECTRAL_LINE_QC[] = "THAR_SPECTRAL_LINE_QC";
/* FP mirrors distance D in A */
static const char COL_NAME_FP_D_ANGSTROM[] = "FP_D_ANGSTROM";
/* D fit polynomial degree */
static const char COL_NAME_D_FIT_POLY_DEG[] = "D_FIT_POLY_DEG";
/* D fit first (big) residuals limit */
static const char COL_NAME_D_FIT_BIG_RES_LIMIT[] = "D_FIT_BIG_RES_LIMIT";
/* D fit second (small) residuals limit */
static const char COL_NAME_D_FIT_SMALL_RES_LIMIT[] = "D_FIT_SMALL_RES_LIMIT";
/* D fit kernel polynomial degree */
static const char COL_NAME_D_FIT_KERNEL_POLY_DEG[] = "D_FIT_KERNEL_POLY_DEG";
/* D fit kernel fwhm */
static const char COL_NAME_D_FIT_KERNEL_FWHM[] = "D_FIT_KERNEL_FWHM";
/* Maximum allowed CHI2 */
static const char COL_NAME_THAR_SOL_MAX_CHI2[] = "THAR_SOL_MAX_CHI2";
/* Maximum allowed RMS */
static const char COL_NAME_THAR_SOL_MAX_RMS[] = "THAR_SOL_MAX_RMS";
/* Minimum allowed resolution */
static const char COL_NAME_WAVE_MIN_RESOL[] = "WAVE_MIN_RESOL";
/* Minimum lines number per order for wavelength solution */
static const char COL_NAME_WAVE_SOL_MIN_LINES_PER_ORDER[] = "WAVE_SOL_MIN_LINES_PER_ORDER";
/* Minimum total lines number */
static const char COL_NAME_THAR_MIN_LINES_TOTAL[] = "THAR_MIN_LINES_TOTAL";
/* Drift THAR-FP switch */
static const char COL_NAME_THAR_DRIFT_TH_FP_SW[] = "THAR_DRIFT_TH_FP_SW";
/* Drift FP-THAR switch */
static const char COL_NAME_THAR_DRIFT_FP_TH_SW[] = "THAR_DRIFT_FP_TH_SW";
/* Drift CHI2 limit */
static const char COL_NAME_THAR_DRIFT_CHI2_LIMIT[] = "THAR_DRIFT_CHI2_LIMIT";
/* Analytical linking of orders with non overlapping wavelengths switch */
static const char COL_NAME_WAVE_ORDERS_ANALYTICAL_LINK_SW[] = "WAVE_ORDERS_ANALYTICAL_LINK_SW";

// WAVE TH DRIFT
/* TH lamp offset AR */
static const char COL_NAME_TH_LAMP_OFFSET_AR[] = "TH_LAMP_OFFSET_AR";
/* TH lamp offset AR1 */
static const char COL_NAME_TH_LAMP_OFFSET_AR1[] = "TH_LAMP_OFFSET_AR1";
/* TH lamp offset AR2 */
static const char COL_NAME_TH_LAMP_OFFSET_AR2[] = "TH_LAMP_OFFSET_AR2";
/* Minimum total lines number  */
static const char COL_NAME_TH_DRIFT_MIN_LINES_TOTAL[] = "TH_DRIFT_MIN_LINES_TOTAL";
/* Maximum saturated lines number  */
static const char COL_NAME_TH_DRIFT_MAX_SATUR_LINES[] = "TH_DRIFT_MAX_SATUR_LINES";
/* Degree in X & Y of the drift fit */
static const char COL_NAME_TH_DRIFT_FIT_DEG_X[] = "TH_DRIFT_FIT_DEG_X";
static const char COL_NAME_TH_DRIFT_FIT_DEG_Y[] = "TH_DRIFT_FIT_DEG_Y";
/* KSigma in the drift fit residuals rejection  */
static const char COL_NAME_TH_DRIFT_FIT_KSIGMA[] = "TH_DRIFT_FIT_KSIGMA";
/* Maximum MAD of the residuals of the drift fit  */
static const char COL_NAME_TH_DRIFT_FIT_MAX_MAD[] = "TH_DRIFT_FIT_MAX_MAD";
/* Maximum fraction of the lines rejected in the drift fit  */
static const char COL_NAME_TH_DRIFT_MAX_REJECT_LINES_FRACTION[] = "TH_DRIFT_MAX_REJECT_LINES_FRACTION";

// WAVE LFC
/* LFC anchor frequency */
static const char COL_NAME_LFC_ANCHOR_FREQ[] = "LFC_ANCHOR_FREQ";
/* LFC repetition rate */
static const char COL_NAME_LFC_REPETITION_RATE[] = "LFC_REPETITION_RATE";
/* LFC spectral line QC */
static const char COL_NAME_LFC_SPECTRAL_LINE_QC[] = "LFC_SPECTRAL_LINE_QC";
/* Maximum allowed CHI2 */
static const char COL_NAME_LFC_SOL_MAX_CHI2[] = "LFC_SOL_MAX_CHI2";
/* Maximum allowed RMS */
static const char COL_NAME_LFC_SOL_MAX_RMS[] = "LFC_SOL_MAX_RMS";
/* Minimum lines number per order for wavelength solution */
static const char COL_NAME_LFC_SOL_MIN_LINES_PER_ORDER[] = "LFC_SOL_MIN_LINES_PER_ORDER";
/* Minimum total lines number */
static const char COL_NAME_LFC_MIN_LINES_TOTAL[] = "LFC_MIN_LINES_TOTAL";

// CONTAM
/* Grid size in X and Y, this is the sub-region size of the stacked flats frames
 * product destinated to measure the inter-order backgroud */
static const char COL_NAME_CONTAM_BKGR_GRID_SIZE_X[] = "CONTAM_BKGR_GRID_SIZE_X";
static const char COL_NAME_CONTAM_BKGR_GRID_SIZE_Y[] = "CONTAM_BKGR_GRID_SIZE_Y";
/* Flux limit on S2DA */
static const char COL_NAME_CONTAM_FLUX_LIMIT[] = "CONTAM_FLUX_LIMIT";

// EFF_AB
/* Minimum allowed SNR */
static const char COL_NAME_EFF_AB_SNR_LIMIT[] = "EFF_AB_SNR_LIMIT";
/* Maximum allowed ratio between background amd max flux */
static const char COL_NAME_EFF_AB_BKGR_LIMIT[] = "EFF_AB_BKGR_LIMIT";
/* Grid size in X and Y, this is the sub-region size of the stacked flats frames
 * product destinated to measure the inter-order backgroud */
static const char COL_NAME_EFF_AB_BKGR_GRID_SIZE_X[] = "EFF_AB_BKGR_GRID_SIZE_X";
static const char COL_NAME_EFF_AB_BKGR_GRID_SIZE_Y[] = "EFF_AB_BKGR_GRID_SIZE_Y";
/* Polynomial degree of polynomial fit */
static const char COL_NAME_EFF_AB_POLY_DEG[] = "EFF_AB_POLY_DEG";
/* Minimum allowed relative efficiency */
static const char COL_NAME_REL_EFF_MIN_LIMIT[] = "REL_EFF_MIN_LIMIT";
/* Maximum allowed relative efficiency */
static const char COL_NAME_REL_EFF_MAX_LIMIT[] = "REL_EFF_MAX_LIMIT";

// FLUX
static const char COL_NAME_FLUX_EFF_INTERP_METHOD[] = "FLUX_EFF_INTERP_METHOD";
static const char COL_NAME_FLUX_POLY_DEG[] = "FLUX_POLY_DEG";
/* Minimum allowed SNR */
static const char COL_NAME_FLUX_SNR_LIMIT[] = "FLUX_SNR_LIMIT";
/* Minimum allowed minimal flux */
static const char COL_NAME_ABS_EFF_MIN_LIMIT[] = "ABS_EFF_MIN_LIMIT";
/* Minimum allowed maximal flux */
static const char COL_NAME_ABS_EFF_MAX_LIMIT[] = "ABS_EFF_MAX_LIMIT";

//SCI_RED
/* WAVE grid matrix and CCF related keywords */
static const char COL_NAME_MASK_WIDTH_KW[] = "MASK_WIDTH";
static const char COL_NAME_LOG_LAMBDA_INI_KW[] = "LOG_LAMBDA_INI";
static const char COL_NAME_DELTA_LOG_LAMBDA_KW[] = "DELTA_LOG_LAMBDA";
static const char COL_NAME_LOG_LAMBDA_END_KW[] = "LOG_LAMBDA_END";	
/* Other SCIENCE values */
static const char COL_NAME_DEFAULT_SPEC_TYPE[] = "DEFAULT_SPEC_TYPE";
static const char COL_NAME_RV_RANGE[] = "RV_RANGE";	
static const char COL_NAME_RV_STEP[] = "RV_STEP";	
static const char COL_NAME_TEL_SURFACE[] = "TEL_SURFACE";	
static const char COL_NAME_SNR_AVERAGING_WINDOW[] = "SNR_AVERAGING_WINDOW";
static const char COL_NAME_BERV_TYPE_INIT[] = "BERV_TYPE_INIT";
static const char COL_NAME_TEL_GEOLON[] = "TEL_GEOLON";
static const char COL_NAME_TEL_GEOLAT[] = "TEL_GEOLAT";
static const char COL_NAME_TEL_GEOELEV[] = "TEL_GEOELEV";
/* Grid size in X and Y, this is the sub-region size of the stacked flats frames
 * product destinated to measure the inter-order backgroud */
static const char COL_NAME_SCI_BKGR_GRID_SIZE_X[] = "SCI_BKGR_GRID_SIZE_X";
static const char COL_NAME_SCI_BKGR_GRID_SIZE_Y[] = "SCI_BKGR_GRID_SIZE_Y";
/* SKY correction parameters */
static const char COL_NAME_SCI_SKY_CORR_METHOD[] = "SCI_SKY_CORR_METHOD";
static const char COL_NAME_SCI_SKY_SUB_SLIDING_BOX_SIZE[] = "SCI_SKY_SUB_SLIDING_BOX_SIZE";
/* OH correction parameters */
static const char COL_NAME_SCI_OH_CORR_LINES_NB_LIMIT[] = "OH_CORR_LINES_NB_LIMIT";
static const char COL_NAME_SCI_OH_CORR_RATIO_LOWER_LIMIT[] = "OH_CORR_RATIO_LOWER_LIMIT";
static const char COL_NAME_SCI_OH_CORR_RATIO_UPPER_LIMIT[] = "OH_CORR_RATIO_UPPER_LIMIT";
static const char COL_NAME_SCI_OH_CORR_RATIO_MAD_LIMIT[] = "OH_CORR_RATIO_MAD_LIMIT";
/* LA cosmic parameters */
static const char COL_NAME_LACOSMIC_SKY_SW[] = "LACOSMIC_SKY_SW";
static const char COL_NAME_LACOSMIC_CALIB_SW[] = "LACOSMIC_CALIB_SW";
static const char COL_NAME_LACOSMIC_POST_FILTER_X[] = "LACOSMIC_POST_FILTER_X";
static const char COL_NAME_LACOSMIC_POST_FILTER_Y[] = "LACOSMIC_POST_FILTER_Y";
static const char COL_NAME_LACOSMIC_POST_FILTER_MODE[] = "LACOSMIC_POST_FILTER_MODE";
static const char COL_NAME_LACOSMIC_SIGMA_LIM[] = "LACOSMIC_SIGMA_LIM";
static const char COL_NAME_LACOSMIC_F_LIM[] = "LACOSMIC_F_LIM";
static const char COL_NAME_LACOSMIC_MAX_ITER[] = "LACOSMIC_MAX_ITER";

//DRIFT
static const char COL_NAME_DRIFT_KSIGMA[] = "DRIFT_KSIGMA";
static const char COL_NAME_DRIFT_METHOD_THAR[] = "DRIFT_METHOD_THAR";
static const char COL_NAME_DRIFT_METHOD_FP[] = "DRIFT_METHOD_FP";
static const char COL_NAME_DRIFT_METHOD_LFC[] = "DRIFT_METHOD_LFC";
static const char COL_NAME_DRIFT_SPACE[] = "DRIFT_SPACE";
/* Drift max & min flux thresholds */
static const char COL_NAME_DRIFT_MAX_FLUX_THRESHOLD[] = "DRIFT_MAX_FLUX_THRESHOLD";
static const char COL_NAME_DRIFT_MIN_FLUX_THRESHOLD[] = "DRIFT_MIN_FLUX_THRESHOLD";
/* Max & min limits for the flux ratio */
static const char COL_NAME_DRIFT_MAX_FLUX_RATIO[] = "DRIFT_MAX_FLUX_RATIO";
static const char COL_NAME_DRIFT_MIN_FLUX_RATIO[] = "DRIFT_MIN_FLUX_RATIO";
/* Drift mean threshold */
static const char COL_NAME_DRIFT_MEAN_THRESHOLD[] = "DRIFT_MEAN_THRESHOLD";
/* Drift mean error threshold */
static const char COL_NAME_DRIFT_MEAN_ERR_THRESHOLD[] = "DRIFT_MEAN_ERR_THRESHOLD";
/* Drift CHI2 threshold */
static const char COL_NAME_DRIFT_CHI2_THRESHOLD[] = "DRIFT_CHI2_THRESHOLD";

// KWs names
/* Actual integration time*/
static const char COL_NAME_TEXP_KW[] = "TEXP_KW";
/* Normalised mean exposure time KW */
static const char COL_NAME_TMMEAN_KW[] = "TMMEAN_KW";
/* Gain KWs per output */
static const char COL_NAME_GAIN_KW_FIRST_PART[] = "GAIN_KW_FIRST_PART";
static const char COL_NAME_GAIN_KW_LAST_PART[] = "GAIN_KW_LAST_PART";
/* Detector binning KWs */
static const char COL_NAME_DET_BINX_KW[] = "DET_BINX_KW";
static const char COL_NAME_DET_BINY_KW[] = "DET_BINY_KW";
/* BERV related keywords */
static const char COL_NAME_TEL_TARG_ALPHA_KW[] = "TEL_TARG_ALPHA_KW";
static const char COL_NAME_TEL_TARG_DELTA_KW[] = "TEL_TARG_DELTA_KW";
static const char COL_NAME_TEL_TARG_PMA_KW[] = "TEL_TARG_PMA_KW";
static const char COL_NAME_TEL_TARG_PMD_KW[] = "TEL_TARG_PMD_KW";
static const char COL_NAME_DATE_OBS_KW[] = "DATE_OBS_KW";
static const char COL_NAME_TARG_NAME_KW[] = "TARG_NAME_KW";
static const char COL_NAME_SPEC_TYPE_VAL_KW[] = "SPEC_TYPE_VAL_KW";
static const char COL_NAME_ZEM_KW[] = "ZEM_KW";
static const char COL_NAME_WAVECAL_SRC_KW[] = "WAVECAL_SRC_KW";
static const char COL_NAME_RV_CENTER_KW[] = "RV_CENTER_KW";
static const char COL_NAME_AIRM_START_KW[] = "AIRM_START_KW";
static const char COL_NAME_AIRM_END_KW[] = "AIRM_END_KW";
/* Telluric correction related keywords */
static const char COL_NAME_PRES_START_KW[] = "PRES_START_KW";
static const char COL_NAME_PRES_END_KW[] = "PRES_END_KW";
static const char COL_NAME_TEMP_KW[] = "TEMP_KW";
/* DPR.TYPE KW */
static const char COL_NAME_DPR_TYPE_KW[] = "DPR_TYPE_KW";
/* LFC repetition rate */
static const char COL_NAME_REP_RATE_KW[] = "REP_RATE_KW";
/* LFC anchor frequency */
static const char COL_NAME_ANCHOR_FREQ_KW[] = "ANCHOR_FREQ_KW";
/* Telescope Image Quality */
static const char COL_NAME_SEEING_KW_FIRST_PART[] = "SEEING_KW_FIRST_PART";
static const char COL_NAME_SEEING_KW_LAST_PART[] = "SEEING_KW_LAST_PART";

// Fibers names
static const char fibre_name[MAX_NB_OF_FIBRES] = {'A', 'B', 'C', 'D', 'E'};

/*----------------------------------------------------------------------------
 Prototypes
 ----------------------------------------------------------------------------*/
int espdr_param_has_changed(cpl_parameter* p);
espdr_inst_config* espdr_inst_config_init(cpl_parameterlist *parameters,
                                          int ext_nb, 
                                          cpl_frame* inst_config_frame);

cpl_error_code espdr_read_inst_parameters(const char *filename,
                                          int ext_nb,
                                          espdr_inst_config *inst_config,
										  cpl_parameterlist *parameters);

cpl_error_code espdr_parameters_inst_config_print(espdr_inst_config *inst_config,
                                                  int ext_nb,
                                                  const char *context);

cpl_error_code espdr_parameters_inst_config_delete(espdr_inst_config *inst_config);

#endif
