/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_orders.h>

#include <hdrl.h>
#include <sys/time.h>
/* used for measure computation time:
static struct timeval tv1, tv2;
*/

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     kext        property list of the extensions (?)
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_add_extra_qc(cpl_propertylist* keywords,
                                         const int exp_nb,
                                         cpl_propertylist*** kext) {

    const char* ksuff[2] = {"", ""};
    cpl_propertylist_append_int(keywords,"ESO QC PROC VERSION", 2);
    espdr_add_qc_key_stat_ext(keywords, exp_nb, "MAX FLUX", ksuff, CPL_FALSE, CPL_FALSE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE, kext);

    espdr_add_qc_key_stat_ord(keywords, exp_nb, "RES STDEV", CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE, kext);

    int* ord_ref = espdr_get_ord_ref(keywords, "espdr_orderdef");

    espdr_add_qc_key_orderdef_ord_ref(keywords, exp_nb, ord_ref, "RES STDEV",
                                      CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_TRUE,
                                      CPL_FALSE, kext);
    cpl_free(ord_ref);

    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Save orderdef products
 @param
 @return    CPL_ERROR_NONE iff OK

 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_save_orderdef_products(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            const char *recipe_id,
                                            cpl_frameset **used_frames,
                                            cpl_propertylist **keywords,
                                            cpl_propertylist ***keywords_ext,
                                            espdr_inst_config *inst_config,
                                            espdr_CCD_geometry *CCD_geom,
                                            cpl_imagelist **numbered_orders_map,
                                            cpl_table ***coeffs_tables) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);

    for (int t = 0; t < inst_config->fibres_nb; t++) {
        /* Save the PRO.CATG */
        char *new_keyword = (char *)cpl_malloc(KEYWORD_LENGTH * sizeof(char));
        sprintf(new_keyword, "%s_%c", ESPDR_PRO_CATG_ORDERS, fibre_name[t]);
        espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
        my_error = cpl_propertylist_update_string(keywords[t], PRO_CATG_KW, new_keyword);
        if (my_error != CPL_ERROR_NONE) {
            espdr_msg_error("Error adding product category is %s", cpl_error_get_message());
        }
        cpl_free(new_keyword);

        /* Save the orders table */
        espdr_msg("Saving orders maps & polynome coeffs from orders position fitting");
        char orders_filename[FILENAME_LENGTH];
        sprintf(orders_filename, "%s_%c.fits",
                inst_config->orders_map_filename_no_fits, fibre_name[t]);

        if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
            (strcmp(inst_config->instrument, "CORALIE") != 0)) {
            my_error = espdr_add_extra_qc(keywords[t], CCD_geom->ext_nb, &keywords_ext[t]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_add_extra_qc failed: %s ",
                         cpl_error_get_message_default(my_error));
        }

        my_error = espdr_dfs_image_save(frameset_orig,
                                        parameters,
                                        used_frames[t],
                                        recipe_id,
                                        keywords[t], keywords_ext[t],
                                        orders_filename,
                                        numbered_orders_map[t],
                                        CPL_TYPE_INT, CCD_geom);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error), orders_filename);

        /* Save the FITS tables with coeffs */
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            my_error = cpl_table_save(coeffs_tables[t][j], NULL, NULL,
                                      orders_filename, CPL_IO_EXTEND);
        }

        espdr_msg("Saved %s", orders_filename);

        cpl_frame* product_frame;
        char pro_catg[FILENAME_LENGTH];
        sprintf(pro_catg, "%s_%c", ESPDR_PRO_CATG_ORDERS, fibre_name[t]);
        espdr_frame_new(&product_frame, orders_filename,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg);
        cpl_frameset_insert(frameset, product_frame);
    }
    cpl_frameset_delete(frameset_orig);

    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Interpret the command line options and execute the data processing
 @param     parameters  the parameters list
 @param     frameset    the frames list

 In case of failure the cpl_error_code is set.

 */
/*---------------------------------------------------------------------------*/
int espdr_orderdef(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id)
{
    cpl_frameset **used_frames = NULL;

    int *orders_nr = NULL;
    int iml_size = 0;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 7;
    const char* rec_tags[7] = {ESPDR_ORDERDEF_A_RAW, ESPDR_ORDERDEF_B_RAW,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM, ESPDR_ORDERS_MASK};
    int is_required[7] = {1, 1, 1, 0, 1, 1, 0};

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting orderdef");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "order fibre frame for each fibre,\n"
                 "tagged as ORDERDEF_A, ORDERDEF_B, etc\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as INST_CONFIG");

    espdr_CCD_geometry *CCD_geom         = NULL;
    espdr_inst_config *inst_config       = NULL;
    espdr_qc_keywords *qc_kws            = NULL;
    espdr_OVSC_param *OVSC_param         = NULL;
    espdr_ORDERDEF_param *ORDERDEF_param = NULL;
    cpl_frame* CCD_geom_frame            = NULL;
    cpl_frame* inst_config_frame         = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL, cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    ORDERDEF_param    = espdr_ORDERDEF_param_init(recipe_id, parameters);

    used_frames = (cpl_frameset **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frameset *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        used_frames[i] = cpl_frameset_new();
        my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(CCD_geom_frame));
        my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(inst_config_frame));
    }

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s", cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "QC KWs structure is NULL");

    int number_of_fibres = inst_config->fibres_nb;
    cpl_frame **orderdef_frame = (cpl_frame **)cpl_malloc(number_of_fibres * sizeof(cpl_frame *));
    cpl_imagelist **CCD_corrected_orderdef = (cpl_imagelist**)cpl_malloc
                                                (number_of_fibres*sizeof(cpl_imagelist *));
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    cpl_propertylist **keywords = (cpl_propertylist **)cpl_malloc
                                        (number_of_fibres*sizeof(cpl_propertylist *));
    cpl_propertylist ***keywords_ext = (cpl_propertylist ***)cpl_malloc
                                        (number_of_fibres*sizeof(cpl_propertylist **));
    for (int i = 0; i < number_of_fibres; i++) {
        keywords_ext[i] = (cpl_propertylist **)cpl_malloc(CCD_geom->ext_nb*sizeof(cpl_propertylist *));
    }
    double **RON = (double **)cpl_malloc(number_of_fibres*sizeof(double *));
    double **CONAD = (double **)cpl_malloc(number_of_fibres*sizeof(double *));
    for (int i = 0; i < number_of_fibres; i++) {
        RON[i] = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
        CONAD[i] = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    }
    my_error = espdr_orderdef_process_inputs(frameset, used_frames,
                                             CCD_geom, inst_config, qc_kws,
                                             RON, CONAD,
                                             orderdef_frame,
                                             keywords, keywords_ext,
                                             geometry_corrected_pixels_mask,
                                             &geometry_corrected_saturation_mask,
                                             CCD_corrected_orderdef);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_orderdef_process_inputs failed: %s",
                 cpl_error_get_message_default(my_error));
    
    int recipes[10] = {1, 1, 1, 0, 0, 0, 0, 0, 0, 0};
    for (int i = 0; i < number_of_fibres; i++) {
        my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords[i],
                                                        NULL, NULL, NULL, "ORDERDEF", recipes);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_calibrations_intervals failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    int satur_mask_size = cpl_imagelist_get_size(geometry_corrected_saturation_mask);
    int hot_bad_mask_size = cpl_imagelist_get_size(geometry_corrected_pixels_mask);
    espdr_ensure(satur_mask_size != hot_bad_mask_size, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "saturation and hot-bad pixels masks sizes differ: satur: %d, hot_bad: %d",
                 satur_mask_size, hot_bad_mask_size);

    cpl_imagelist *geometry_corrected_total_mask = cpl_imagelist_duplicate(geometry_corrected_saturation_mask);
    my_error = cpl_imagelist_add(geometry_corrected_total_mask, geometry_corrected_pixels_mask);

    cpl_imagelist **numbered_orders_map = (cpl_imagelist **)cpl_malloc(number_of_fibres*sizeof(cpl_imagelist *));
    cpl_imagelist *orders_map = NULL;
    cpl_table ***coeffs_tables = (cpl_table ***)cpl_malloc(number_of_fibres*sizeof(cpl_table **));
    //cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);
    /* Processing one orderdef frame at a time */
    for (int t = 0; t < number_of_fibres; t++) {
        // Compute the image background
        cpl_imagelist *final_master_background_measure_flat = cpl_imagelist_new();
        cpl_imagelist *final_master_background_substracted_flat = cpl_imagelist_new();
        my_error = espdr_measure_background_ff_by_ext(CCD_corrected_orderdef[t],
                                                      CCD_geom,
                                                      inst_config->bkgr_grid_size_x,
                                                      inst_config->bkgr_grid_size_y,
                                                      inst_config->minlevel_factor,
                                                      final_master_background_measure_flat,
                                                      final_master_background_substracted_flat);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_measure_background_ff_by_ext failed");
        cpl_imagelist_delete(final_master_background_substracted_flat);

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        /* Save the background measured fits frame */
        my_error = espdr_save_orderdef_debug_product(frameset, parameters,
                                                     inst_config, CCD_geom, qc_kws,
                                                     keywords[t], keywords_ext[t],
                                                     "background_map_orderdef",
                                                     ESPDR_PRO_CATG_FF_BKG_MAP,
                                                     final_master_background_measure_flat,
                                                     t);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_orderdef_debug_product failed");
#endif

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        my_error = espdr_save_orderdef_bkgr_div(CCD_corrected_orderdef[t],
                                                final_master_background_measure_flat,
                                                keywords[t], inst_config, CCD_geom->ext_nb, t);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_orderdef_bkgr_div failed");
#endif


        /* Orders detection */
        espdr_msg("Identifying orders");
        orders_map = cpl_imagelist_new();
        my_error = espdr_detect_orders(CCD_corrected_orderdef[t],
                                       geometry_corrected_pixels_mask,
                                       final_master_background_measure_flat,
                                       CCD_geom, inst_config, t, orders_map);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_detect_orders failed: %s", cpl_error_get_message_default(my_error));
        espdr_msg("Orders map list has %lld images", cpl_imagelist_get_size(orders_map));
        cpl_imagelist_delete(final_master_background_measure_flat);

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        my_error = espdr_save_orderdef_simple_debug_product(orders_map, "detected_orders_ext",
                                                            keywords[t], inst_config,
                                                            CCD_geom->ext_nb, t);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_orderdef_simple_debug_product failed");
#endif

        /* Orders numbering */
        espdr_msg("Numbering orders");
        numbered_orders_map[t] = cpl_imagelist_new();
        orders_nr = (int *)cpl_calloc(cpl_imagelist_get_size(orders_map), sizeof(int));
        my_error = espdr_number_valid_orders(CCD_corrected_orderdef[t],
                                             orders_map,
                                             inst_config,
                                             inst_config->min_order_len[t],
                                             inst_config->min_order_x[t],
                                             inst_config->max_order_y[t],
                                             orders_nr,
                                             numbered_orders_map[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_number_valid_orders failed: %s",
                     cpl_error_get_message_default(my_error));
        cpl_imagelist_delete(orders_map);

        // Check on orders nb
        int index = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            int order_param_index = t * CCD_geom->ext_nb + j;
            if (orders_nr[index] != inst_config->orders_nb[order_param_index]) {
                espdr_msg_warning(ANSI_COLOR_RED
                                  "For fiber %c and ext %d the number of orders found (%d) differs from the nominal number (%d)"
                                  ANSI_COLOR_RESET,
                                  fibre_name[t], j, orders_nr[index],
                                  inst_config->orders_nb[order_param_index]);
            } else {
                espdr_msg("Nb of orders detected for fibre %c and ext %d: %d - same as nominal",
                          fibre_name[t], j, orders_nr[index]);
            }
            index++;
        }

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        my_error = espdr_save_orderdef_simple_debug_product(numbered_orders_map[t],
                                                            "numbered_orders_ext",
                                                            keywords[t], inst_config,
                                                            CCD_geom->ext_nb, t);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_orderdef_simple_debug_product failed");
#endif

        /* Fitting orders */
        espdr_msg("Fitting orders to find their centre");
        iml_size = cpl_imagelist_get_size(CCD_corrected_orderdef[t]);
        coeffs_tables[t] = (cpl_table **)cpl_malloc(iml_size*sizeof(cpl_table *));
        double **orders_pos = (double **)cpl_malloc(iml_size * sizeof(double *));
        double **orders_res_stdev = (double **)cpl_malloc(iml_size * sizeof(double *));
        double **orders_res_min = (double **)cpl_malloc(iml_size * sizeof(double *));
        double **orders_res_max = (double **)cpl_malloc(iml_size * sizeof(double *));
        for (int j = 0; j < iml_size; j++) {
            coeffs_tables[t][j] = cpl_table_new(orders_nr[j]);
            orders_pos[j] = (double *)cpl_calloc(orders_nr[j], sizeof(double));
            orders_res_stdev[j] = (double *)cpl_calloc(orders_nr[j], sizeof(double));
            orders_res_min[j] = (double *)cpl_calloc(orders_nr[j], sizeof(double));
            orders_res_max[j] = (double *)cpl_calloc(orders_nr[j], sizeof(double));
        }
        for (int j = 0; j < iml_size; j++) {
            my_error = espdr_create_coeffs_table(coeffs_tables[t][j],
                                                 inst_config->order_fit_poly_deg);
        }
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_create_coeffs_table failed: %s",
                     cpl_error_get_message_default(my_error));

        my_error = espdr_fit_orders(CCD_corrected_orderdef[t],
                                    numbered_orders_map[t],
                                    geometry_corrected_total_mask,
                                    t, orders_nr,
                                    RON[t], CONAD[t],
                                    CCD_geom, inst_config,
                                    orders_pos,
                                    coeffs_tables[t],
                                    orders_res_stdev,
                                    orders_res_min,
                                    orders_res_max);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_orders failed: %s",
                     cpl_error_get_message_default(my_error));

#if 0
        /* Check saved coeffs */
        char column_name[15];
        double read_value = 0.0;
        int poly_deg = inst_config->order_fit_poly_deg;
        int k;
        double my_coeffs[poly_deg];
        double position;
        for (int i = 0; i < iml_size; i++) {
            for (int j = 0; j < orders_nr[i]; j++) {
                for (k = 0; k < poly_deg; k++) {
                    sprintf(column_name, "COEFF_%d", k);
                    read_value = cpl_table_get_double(coeffs_tables[t][i], column_name, j, NULL);
                    //espdr_msg("Image %d, order %d, coeff %d: %.16lf",
                    //          i, j, k, read_value);
                    my_coeffs[k] = read_value;
                }
                if (j == 0) {
                    position = 0.0;
                    for (k = poly_deg-1; k > 0; k--) {
                        position = (position + my_coeffs[k]) * 4500.0;
                    }
                    position = position + my_coeffs[0];
                    espdr_msg("ORDER POSITION :::::::::::::: %lf", position);
                }
            }
        }
#endif

#if 0
        /* Check physical order numbers calculation */
        int phys_orders_nb[inst_config->orders_nb[0]+inst_config->orders_nb[1]];
        my_error = espdr_get_order_phys_numbers(CCD_geom, inst_config,
                                                0, phys_orders_nb);
        espdr_msg("Physical orders numbers (slices per phys: %d):",
                  inst_config->slices_nb_per_phys_order);
        for (int i = 0; i < inst_config->orders_nb[0]+inst_config->orders_nb[1]; i++) {
            espdr_msg("Order %d: %d", i+1, phys_orders_nb[i]);
        }
        /* End check physical order numbers */
#endif

        /* Draw fitted orders on the image */
        espdr_msg("Drawing orders on the products image");
        int CCD_size_y;
        for (int i = 0; i < CCD_geom->ext_nb; i++) {
            if ((CCD_geom->exts[i].rot_angle == 90) ||
                (CCD_geom->exts[i].rot_angle == 270)) {
                CCD_size_y = CCD_geom->exts[0].CCD_nx;
            } else {
                CCD_size_y = CCD_geom->exts[0].CCD_ny;
            }
            int order_param_index = t * CCD_geom->ext_nb + i;
            my_error = espdr_draw_orders(cpl_imagelist_get(numbered_orders_map[t], i),
                                         orders_nr[i],
                                         inst_config->order_start[order_param_index],
                                         inst_config->order_end[order_param_index],
                                         inst_config->order_fit_poly_deg,
                                         CCD_size_y,
                                         coeffs_tables[t][i]);

            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_draw_orders failed: %s",
                         cpl_error_get_message_default(my_error));
        }

        /* Save the QC keywords */
        int **orders_phys_nb = (int **)cpl_malloc(iml_size * sizeof(int *));
        for (int j = 0; j < iml_size; j++) {
            orders_phys_nb[j] = (int *)cpl_calloc(orders_nr[j], sizeof(int));
        }

        my_error = espdr_orderdef_QC(CCD_geom, qc_kws, inst_config, t,
                                     orders_nr,
                                     orders_pos,
                                     orders_res_stdev,
                                     orders_res_min,
                                     orders_res_max,
                                     orders_phys_nb,
                                     &keywords[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_orderdef_QC failed: %s",
                     cpl_error_get_message_default(my_error));


        

        /* Free the tables*/
        for (int j = 0; j < iml_size; j++) {
            cpl_free(orders_pos[j]);
            cpl_free(orders_res_stdev[j]);
            cpl_free(orders_res_min[j]);
            cpl_free(orders_res_max[j]);
            cpl_free(orders_phys_nb[j]);
        }
        cpl_free(orders_pos);
        cpl_free(orders_res_stdev);
        cpl_free(orders_res_min);
        cpl_free(orders_res_max);
        cpl_free(orders_phys_nb);
        cpl_free(orders_nr);
        espdr_msg("Orders localized for fibre %c", fibre_name[t]);
    }

    my_error = espdr_save_orderdef_products(frameset, parameters, recipe_id,
                                            used_frames, keywords, keywords_ext,
                                            inst_config, CCD_geom,
                                            numbered_orders_map,
                                            coeffs_tables);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_orderdef_products failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Freeing the memory");
    /* Free the propertylists */
    for (int i = 0; i < number_of_fibres; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_propertylist_delete(keywords_ext[i][j]);
            cpl_table_delete(coeffs_tables[i][j]);
        }
        cpl_propertylist_delete(keywords[i]);
        cpl_free(keywords_ext[i]);
        cpl_imagelist_delete(CCD_corrected_orderdef[i]);
        cpl_imagelist_delete(numbered_orders_map[i]);
        cpl_free(RON[i]);
        cpl_free(CONAD[i]);
        cpl_free(coeffs_tables[i]);
    }
    cpl_free(keywords);
    cpl_free(keywords_ext);
    cpl_free(CCD_corrected_orderdef);
    cpl_free(numbered_orders_map);
    cpl_free(RON);
    cpl_free(CONAD);
    cpl_free(coeffs_tables);

    /* Free the images and frames */
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
    cpl_imagelist_delete(geometry_corrected_total_mask);

    for (int i = 0; i < number_of_fibres; i++) {
        //cpl_frame_delete(orderdef_frame[i]);
        cpl_frameset_delete(used_frames[i]);
    }
    cpl_free(orderdef_frame);
    cpl_free(used_frames);

    /* Free the parameters structures */
    cpl_free(qc_kws);
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_ORDERDEF_delete(ORDERDEF_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for ORDEFDER input parameters
 @param         recipe_id	recipe identifier
 @param         list			list of input parameters
 @param[out]    p			input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_ORDERDEF_create(const char* recipe_id,
                                                cpl_parameterlist *list,
                                                espdr_ORDERDEF_param *p) {

	espdr_ensure(recipe_id == NULL,
			CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");

	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");

	/* Fill the parameter list */
	p = NULL;

	return (CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 * AMO added
 @brief     Init the orderdef parameter structure using the recipe parameters
 @param     recipe_id       recipe identifier
 @param     param_list      parameters list
 @return    ORDERDEF_param iff OK else NULL
 
 TODO: same comment as for previous function regarding the function name.
 */
/*----------------------------------------------------------------------------*/
espdr_ORDERDEF_param* espdr_ORDERDEF_param_init(const char* recipe_id,
                                                cpl_parameterlist* param_list) {
    
    espdr_ORDERDEF_param* ORDERDEF_param = NULL;
    
    cpl_ensure(recipe_id != NULL, CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(param_list != NULL, CPL_ERROR_NULL_INPUT,NULL);
    
    ORDERDEF_param = (espdr_ORDERDEF_param *)cpl_malloc(sizeof(espdr_ORDERDEF_param));
    espdr_parameters_ORDERDEF_get(recipe_id, param_list, ORDERDEF_param);
    
    /* TODO: AMO: this print should be optional or removed after debug phase */
    espdr_parameters_ORDERDEF_print(ORDERDEF_param);
    
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        espdr_msg_error("Error during ORDERDEF parameters initialisation");
        espdr_parameters_ORDERDEF_delete(ORDERDEF_param);
    }
    
    return ORDERDEF_param;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_ORDERDEF_delete(espdr_ORDERDEF_param* p) {

	cpl_free(p);
	p = NULL;
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the orderdef recipe parameters
 @param         recipe_id   recipe ID
 @param         param_list  parameters list
 @param[out]    ORDERDEF_param  ORDERDEF parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_ORDERDEF_get(const char* recipe_id,
		cpl_parameterlist* param_list,
		espdr_ORDERDEF_param *ORDERDEF_param) {

	espdr_ensure(recipe_id == NULL,
			CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");

	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
			"Parameters list is NULL");

	/* Fill the structure */
	ORDERDEF_param = NULL;

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the ORDERDEF parameters
 @param     ORDEFDEF_param  ORDERDEF parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_ORDERDEF_print(espdr_ORDERDEF_param *ORDERDEF_param) {

	espdr_msg("\tORDERDEF parameters: NONE");
	ORDERDEF_param = NULL;

	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_prepare_static_orders_mask(cpl_frame *orders_mask_frame,
                                                espdr_inst_config *inst_config,
                                                double mjd_obs,
                                                char *date,
                                                char *mode) {
    
    char const *tmp_filename = cpl_frame_get_filename(orders_mask_frame);
    cpl_image *tmp_mask_img = cpl_image_load(tmp_filename, CPL_TYPE_INT, 0, 0);
    
    cpl_propertylist *mask_plist = cpl_propertylist_new();
    cpl_propertylist_append_double(mask_plist, "MJD-OBS", mjd_obs);
    cpl_propertylist_append_string(mask_plist, "INSTRUME",
                                              inst_config->instrument);
    cpl_propertylist_append_string(mask_plist, "ESO INS MODE", mode);
    cpl_propertylist_append_int(mask_plist, inst_config->det_binx_kw, 1);
    cpl_propertylist_append_int(mask_plist, inst_config->det_biny_kw, 1);
    cpl_propertylist_append_string(mask_plist, PRO_CATG_KW, ESPDR_ORDERS_MASK);
    
    char new_mask_filename[64];
    sprintf(new_mask_filename, "%s_%s_%s_%s.fits", inst_config->instrument, mode,
            ESPDR_ORDERS_MASK, date);
    
    cpl_image_save(NULL, new_mask_filename, CPL_TYPE_INT, mask_plist, CPL_IO_CREATE);
    cpl_image_save(tmp_mask_img, new_mask_filename, CPL_TYPE_INT, NULL, CPL_IO_EXTEND);
    espdr_msg("ORDERS_MASK saved under: %s", new_mask_filename);
    cpl_propertylist_delete(mask_plist);
    cpl_image_delete(tmp_mask_img);
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_orderdef_process_inputs(cpl_frameset *frameset,
                                             cpl_frameset **used_frames,
                                             espdr_CCD_geometry *CCD_geom,
                                             espdr_inst_config *inst_config,
                                             espdr_qc_keywords *qc_kws,
                                             double **RON,
                                             double **CONAD,
                                             cpl_frame **orderdef_frame,
                                             cpl_propertylist **keywords,
                                             cpl_propertylist ***keywords_ext,
                                             cpl_imagelist *geometry_corrected_pixels_mask,
                                             cpl_imagelist **geometry_corrected_saturation_mask,
                                             cpl_imagelist **CCD_corrected_orderdef) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame = espdr_frame_find(frameset, ESPDR_PRO_CATG_HOT_PIXELS);
    espdr_ensure(hot_pixels_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "HOT PIXELS frame extraction failed");
    
    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_frame_find(frameset, ESPDR_PRO_CATG_MDARK);
    
    /* Extract bad pixels mask frame */
    cpl_frame* bad_pixels_frame = espdr_frame_find(frameset, ESPDR_PRO_CATG_BAD_PIXELS);
    espdr_ensure(bad_pixels_frame == NULL, CPL_ERROR_NONE,
                 "BAD PIXELS frame extraction failed");
    
    /* Extract orders masking frame */
    cpl_frame* orders_mask_frame = espdr_frame_find(frameset, ESPDR_ORDERS_MASK);
    
#if SAVE_ORDERS_MASK
    // Save orders mask in proper format
    my_error = espdr_prepare_static_orders_mask(orders_mask_frame, inst_config,
                                                //59215.0,
                                                //59335.0,
                                                //59670.0,
                                                59884,
                                                //"2021-01-01",
                                                //"2021-05-01",
                                                //"2022-04-01",
                                                "2022-11-01",
                                                "HE");
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_prepare_static_orders_mask failed: %s",
                 cpl_error_get_message_default(my_error));
#endif
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(hot_pixels_frame));
        if (master_dark_frame != NULL) {
            my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(master_dark_frame));
        }
        my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(bad_pixels_frame));
        if (orders_mask_frame != NULL) {
            my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(orders_mask_frame));
        }
    }
    
    /* Extract the raw orderdef frames and put them into orderdef_frames frameset */
    int number_of_fibres = inst_config->fibres_nb;
    for (int i = 0; i < number_of_fibres; i++) {
        char orderdef_tag[TAG_LENGTH];
        sprintf(orderdef_tag, "%s_%c", ESPDR_ORDERDEF_RAW, fibre_name[i]);
        espdr_msg("ORDERDEF TAG: %s", orderdef_tag);
        orderdef_frame[i] = espdr_frame_find(frameset, orderdef_tag);
        espdr_ensure(orderdef_frame[i] == NULL, CPL_ERROR_NULL_INPUT,
                     "orderdef frame (%s) frame not found!", orderdef_tag);
        my_error = cpl_frameset_insert(used_frames[i], cpl_frame_duplicate(orderdef_frame[i]));
    }
    
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT, CCD_geom, 0,
                                          &hot_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("HOT PIXELS MASK size: %lld x %lld",
              cpl_image_get_size_x(cpl_imagelist_get(hot_pixels_list, 0)),
              cpl_image_get_size_y(cpl_imagelist_get(hot_pixels_list, 0)));
    
    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_dark_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &bad_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_raw_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("BAD PIXELS MASK size: %lld x %lld",
              cpl_image_get_size_x(cpl_imagelist_get(bad_pixels_list, 0)),
              cpl_image_get_size_y(cpl_imagelist_get(bad_pixels_list, 0)));
    
    /* Extract all the outputs from the orders masks frame if provided */
    cpl_imagelist *orders_mask_list = NULL;
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame, CPL_TYPE_INT,
                                            CCD_geom->ext_nb, &orders_mask_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot&bad pixels mask");
    
    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        cpl_frameset *raw_orderdef = cpl_frameset_new();
        cpl_frameset_insert(raw_orderdef, cpl_frame_duplicate(orderdef_frame[0]));
        cpl_frameset_insert(raw_orderdef, cpl_frame_duplicate(orderdef_frame[1]));
        my_error = espdr_update_bad_pixel_mask(raw_orderdef, CCD_geom,
                                               cpl_imagelist_get(bad_pixels_list, 0));
        cpl_frameset_delete(raw_orderdef);
    }
    
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list, bad_pixels_list,
                                                &pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_hot_bad_pixels_mask failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    
    //geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    cpl_imagelist_delete(merged_pixels_mask);
    
#if SAVE_DEBUG_PRODUCT_ORDERDEF
    char filename_mask[64];
    sprintf(filename_mask, "%s_final_hot_bad_pixels_mask.fits", inst_config->instrument);
    my_error = cpl_image_save(cpl_imagelist_get(geometry_corrected_pixels_mask, 0),
                              filename_mask, CPL_TYPE_INT, NULL, CPL_IO_CREATE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Final mask image save failed: %s",
                 cpl_error_get_message_default(my_error));
#endif
    
    /* input frames filenames */
    const char *input_filename = NULL;
    const char *input_filename_HP = NULL;
    const char *input_filename_BP = NULL;
    /* keywords contains all the keywords read from the frame header */
    cpl_propertylist *keywords_HP = NULL;
    cpl_propertylist *keywords_BP = NULL;
    
    /* Load primary header keywords from the hot pixels image */
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s", input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s", input_filename_HP);
    }
    keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s", input_filename_BP);
    keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    /* Processing one orderdef frame at a time */
    for (int t = 0; t < number_of_fibres; t++) {
        cpl_frameset *curr_set = cpl_frameset_new();
        cpl_frameset_insert(curr_set, cpl_frame_duplicate(orderdef_frame[t]));
        
        input_filename = cpl_frame_get_filename(orderdef_frame[t]);
        espdr_msg("KEYWORDS input filename: %s", input_filename);
        keywords[t] = cpl_propertylist_load(input_filename, 0);
        espdr_ensure(keywords[t] == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "keywords are NULL");
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            keywords_ext[t][j] = cpl_propertylist_load(input_filename, j+1);
        }
        
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
        // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
        if (cpl_propertylist_has(keywords[t], "ARCFILE") == 0) {
            char *arc_filename = strrchr(input_filename, '/')+1;
            espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
            my_error = cpl_propertylist_update_string(keywords[t], "ARCFILE", arc_filename);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "Updating the ARCFILE KW (%s) failed: %s",
                         arc_filename, cpl_error_get_message_default(my_error));
        }
#endif
        
        espdr_msg("CCD signature correction on ORDERDEF frame (with %d ext)", CCD_geom->ext_nb);
        
        cpl_imagelist *raw_orderdef_imagelist = cpl_imagelist_new();
        cpl_imagelist *real_orderdef_imagelist = cpl_imagelist_new();
        
        // Used to get data from the image
        if (inst_config->inst_type == NIR) {
            my_error = espdr_extract_extensions(orderdef_frame[t], CPL_TYPE_DOUBLE,
                                                CCD_geom->ext_nb, &raw_orderdef_imagelist);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_extract_extensions failed: %s",
                         cpl_error_get_message_default(my_error));
        } else {
            my_error = espdr_extract_raw_outputs(curr_set, CPL_TYPE_DOUBLE, CCD_geom,
                                                 &raw_orderdef_imagelist);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_extract_raw_outputs failed: %s",
                         cpl_error_get_message_default(my_error));
        }
        
        // Used only in MAX FLUX calculation
        my_error = espdr_extract_real_outputs_from_raw(curr_set, CPL_TYPE_DOUBLE,
                                                       CCD_geom, &real_orderdef_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs_from_raw failed: %s",
                     cpl_error_get_message_default(my_error));
        
        double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_orderdef_imagelist),
                                                sizeof(double));
        // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
        if (inst_config->inst_type == NIR) {
            double texp = cpl_propertylist_get_double(keywords[t], inst_config->Texp_kw);
            //espdr_msg("--->>>>>  exposure time: %f", texp);
            my_error = cpl_imagelist_multiply_scalar(real_orderdef_imagelist, texp);
        }
        
        /* Checking for saturation, creating saturation mask */
        
        // Patch for EGGS, between 2005-07-01 (MJD = 53552) - 2005-12-31 (MJD = 53735),
        // where ORDERDEF_B is slightly saturated
        // There is no other way to get the data reduced, so we accept this slightly saturated ORDERDEF_B
        double cosmics_part = inst_config->image_cosmics_part;
        
        if ((strcmp(inst_config->instrument, "HARPS") == 0) && (t == 1)) {
            if (strcmp(cpl_propertylist_get_string(keywords[t], "ESO INS MODE"), "EGGS") == 0) {
                if (cpl_propertylist_has(keywords[t], "MJD-OBS")) {
                    double mjd_obs = cpl_propertylist_get_double(keywords[t], "MJD-OBS");
                    if ((mjd_obs > 53552) && (mjd_obs < 53735)) {
                        cosmics_part = 0.0025;
                        espdr_msg_warning(ANSI_COLOR_RED"Frame slightly saturated can be accepted, cosmics part = %f, instead of %f"ANSI_COLOR_RESET,
                                          cosmics_part, inst_config->image_cosmics_part);
                    }
                }
            }
        }
        
        my_error = espdr_check_saturation(real_orderdef_imagelist,
                                          CCD_geom, inst_config, pixels_mask,
                                          cosmics_part,
                                          max_flux, 1,
                                          geometry_corrected_saturation_mask);
        
        cpl_imagelist_delete(real_orderdef_imagelist);
        espdr_msg("geom corr sat mask right after creation: size: %lld",
                  cpl_imagelist_get_size(*geometry_corrected_saturation_mask));
        my_error = espdr_max_flux_QC(inst_config, CCD_geom, qc_kws,
                                     max_flux, qc_kws->qc_orderdef_check_kw, keywords[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_max_flux_QC failed: %s",
                     cpl_error_get_message_default(my_error));
        cpl_free(max_flux);

        /* Correcting the CCD signature: bias, dark, geometry */
        espdr_msg("Removing the CCD signature");
        double *DARK = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
        double *DARK_RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
        CCD_corrected_orderdef[t] = cpl_imagelist_new();
        my_error = espdr_remove_det_signature(raw_orderdef_imagelist,
                                              orders_mask_list,
                                              NULL, master_dark_list,
                                              keywords[t], keywords_HP, keywords_BP,
                                              qc_kws, inst_config, 0,
                                              CCD_geom, CPL_TYPE_DOUBLE,
                                              RON[t], DARK, DARK_RON, CONAD[t],
                                              CCD_corrected_orderdef[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_remove_detector_signature failed: %s",
                     cpl_error_get_message_default(my_error));
        cpl_imagelist_delete(raw_orderdef_imagelist);
        cpl_free(DARK);

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        char filename_CCD_cor_orders[FILENAME_LENGTH];
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            sprintf(filename_CCD_cor_orders, "%s_CCD_corr_orderdef_ext_%d_%c.fits",
                    inst_config->instrument, j, fibre_name[t]);
            my_error = cpl_image_save(cpl_imagelist_get(CCD_corrected_orderdef[t], j),
                                      filename_CCD_cor_orders, CPL_TYPE_FLOAT,
                                      keywords[t], CPL_IO_CREATE);
            espdr_msg("%s saved", filename_CCD_cor_orders);
        }
#endif
        
        cpl_frameset_delete(curr_set);
    }
    
    cpl_propertylist_delete(keywords_BP);
    cpl_propertylist_delete(keywords_HP);
    cpl_imagelist_delete(hot_pixels_list);
    if (master_dark_list != NULL) {
        cpl_imagelist_delete(master_dark_list);
    }
    cpl_imagelist_delete(bad_pixels_list);
    if (orders_mask_list != NULL) {
        cpl_imagelist_delete(orders_mask_list);
    }
    cpl_imagelist_delete(pixels_mask);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Detect orders
 @param         input_iml       input corrected raw images
 @param         total_mask_iml  list of masks
 @param         ORDERDEF_param  orderdef params
 @param[out]    orders_map_RE   list of images with orders maps
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_detect_orders(const cpl_imagelist *input_iml,
                                   const cpl_imagelist *total_mask_iml,
                                   cpl_imagelist *background_map,
                                   const espdr_CCD_geometry *CCD_geom,
                                   espdr_inst_config *inst_config,
                                   const int fibre_nr,
                                   cpl_imagelist *orders_map_RE) {

    espdr_ensure(input_iml == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL");
    
    int i, index, nx, ny;
    const double *data_table = NULL;
    const cpl_image *curr_image = NULL;
    const int *mask_table = NULL;
    int mask_size;
    const cpl_image *curr_mask = NULL;
    int *orders_data = NULL;
    cpl_image *orders_image = NULL;
    cpl_image *background_img = NULL;
    double *background_data = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    index = 0;
    mask_size = cpl_imagelist_get_size(total_mask_iml);
    
    for (i = 0; i < cpl_imagelist_get_size(input_iml); i++) {
        curr_image = cpl_imagelist_get_const(input_iml, i);
        data_table = cpl_image_get_data_double_const(curr_image);
        nx = cpl_image_get_size_x(curr_image);
        ny = cpl_image_get_size_y(curr_image);
        
        curr_mask = cpl_imagelist_get_const(total_mask_iml, index);
        mask_table = cpl_image_get_data_int_const(curr_mask);
        
        background_img = cpl_imagelist_get(background_map, index);
        background_data = cpl_image_get_data_double(background_img);
        
        espdr_msg("Treating image %d, background: %lf",
                  i, inst_config->order_flux_limit);
        orders_data = (int *)cpl_calloc(nx*ny, sizeof(int));
        my_error = espdr_identify_orders(data_table, mask_table,
                                         background_data,
                                         nx, ny,
                                         fibre_nr, i, CCD_geom, inst_config,
                                         inst_config->order_flux_limit,
                                         inst_config->cluster_dist_x,
                                         inst_config->cluster_dist_y,
                                         orders_data);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_identify_orders failed: %s",
                     cpl_error_get_message_default(my_error));
        
        orders_image = cpl_image_wrap_int(nx, ny, orders_data);
        cpl_imagelist_set(orders_map_RE,
                          cpl_image_duplicate(orders_image), i);
        cpl_image_unwrap(orders_image);
        cpl_free(orders_data);
        index++;
        if (index == mask_size) {
            index = 0;
        }
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Identify orders
 @param         data_table          data with orders
 @param         mask_table          data of masks
 @param         nx                  x size of the data
 @param         ny                  y size of the data
 @param         background_level    background level
 @param         cluster_distance    maximum allowed distance within a cluster
 @param[out]    orders_data         array with identified orders
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_identify_orders(const double *data_table,
                                     const int *mask_table,
                                     double *background_data,
                                     const int nx,
                                     const int ny,
                                     const int fibre_nr,
                                     const int ext_nr,
                                     const espdr_CCD_geometry *CCD_geom,
                                     espdr_inst_config *inst_config,
                                     const double background_level,
                                     const int cluster_distance_x,
                                     const int cluster_distance_y,
                                     int *orders_data) {

    espdr_ensure(data_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input data is NULL");
    
    int i, j, pixel;
    int counter_flag;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int order_param_index = fibre_nr * CCD_geom->ext_nb + ext_nr;
    
    // extract the part of the image within the orders limits and pass this one to identify_clumps
    
    int order_start = inst_config->order_start[order_param_index];
    int order_end = inst_config->order_end[order_param_index];
    int start_orders_detect = inst_config->orders_detection_start[order_param_index];
    
    cpl_image *bkgr_min = cpl_image_wrap_double(nx, ny, background_data);
    double min_bkgr = cpl_image_get_min(bkgr_min);
    if (min_bkgr < 0.0) {
        min_bkgr = -min_bkgr * 1.1;
    } else {
        if (min_bkgr < 1.0) {
            min_bkgr = 1.0;
        }
    }
    cpl_image_unwrap(bkgr_min);
    //espdr_msg("--->>>> min backgr:p %f", min_bkgr);
    
    for (j = 0; j < ny; j++) {
        for (i = 0; i < nx; i++) {
            pixel = j * nx + i;
            
            if ((i >= order_start) && (i <= order_end) && (j >= start_orders_detect)) {
                if (mask_table[pixel] != 0) {
                    orders_data[pixel] = 0;
                } else {
                    if ((data_table[pixel] / (background_data[pixel] + min_bkgr)) > inst_config->order_flux_factor[fibre_nr]) { // ESPRESSO, OK HE A/B, HA A/B
                        orders_data[pixel] = -1;
                    } else {
                        orders_data[pixel] = 0;
                    }
                    
                    if ((i == -1) && (j > 2920) && (j < 2960)) { // i = 1900
                        espdr_msg("too faint: data[%d, %d] = %f\tbackgr[%d, %d] = %f\torders_data[%d, %d] = %d",
                                  i, j, data_table[pixel],
                                  i, j, background_data[pixel],
                                  i, j, orders_data[pixel]);
                    }
                    
                    if ((i == -1) && (j > 2800) && (j < 2990)) { // i = 2105
                        espdr_msg("too strong: data[%d, %d] = %f\tbackgr[%d, %d] = %f\torders_data[%d, %d] = %d",
                                  i, j, data_table[pixel],
                                  i, j, background_data[pixel],
                                  i, j, orders_data[pixel]);
                    }
                }
            } else {
                orders_data[pixel] = 0;
            }
            
        }
    }
    
#if SAVE_DEBUG_PRODUCT_ORDERDEF
    char filename_orders_ident[FILENAME_LENGTH];
    sprintf(filename_orders_ident,
            "%s_orders_ident_ext_%d_%c.fits",
            inst_config->instrument, ext_nr, fibre_name[fibre_nr]);
    my_error = cpl_image_save(cpl_image_wrap_int(nx, ny, orders_data),
                              filename_orders_ident,
                              CPL_TYPE_INT, NULL, CPL_IO_CREATE);
#endif
    
    espdr_msg("Clumps identification");
    
    counter_flag = 0;
    for (i = 0; i < nx*ny; i++) {
        if (orders_data[i] == -1) {
            counter_flag++;
            my_error = espdr_identify_clump(orders_data, nx, ny,
                                            i, counter_flag,
                                            cluster_distance_x,
                                            cluster_distance_y);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_identify_clump failed for clump: %d",
                         counter_flag);
        }
    }
    
    espdr_msg("Clumps identified");
    
    return cpl_error_get_code();
}


/* ----------- AMO changes to speed-up the orderdef ------------- */

typedef struct {

	/*Backing data structure*/
	cpl_array * vec;
	/*Number of valid elements*/
	cpl_size length;

}espdr_stack;

espdr_stack * espdr_stack_new(){
	espdr_stack * to_ret = cpl_calloc(1, sizeof(*to_ret));
	to_ret->vec = cpl_array_new(1, CPL_TYPE_SIZE);
	to_ret->length = 0;
	return to_ret;
}

void espdr_stack_delete(espdr_stack * v){
	if(v == NULL) return;

	cpl_array_delete(v->vec);
	cpl_free(v);
}

cpl_error_code espdr_stack_push(espdr_stack * v, const cpl_size value){

	cpl_ensure_code(v != NULL, CPL_ERROR_NULL_INPUT);

	cpl_size capacity = cpl_array_get_size(v->vec);
	if(capacity == v->length){
		const cpl_error_code err = cpl_array_set_size(v->vec, 2 * capacity);
		if(err) return err;
	}
	const cpl_error_code err = cpl_array_set(v->vec, v->length, value);
	v->length++;
	return err;
}

cpl_boolean espdr_stack_is_empty(const espdr_stack * v){
	if(v == NULL) return CPL_TRUE;
	return v->length <= 0;
}

cpl_size espdr_stack_pop(espdr_stack * v){
	if(v == NULL || espdr_stack_is_empty(v))
		return -1;

	int rej;
	const cpl_size ret = cpl_array_get_cplsize(v->vec, v->length - 1, &rej);
	v->length--;
	if(rej)
		return -1;

	return ret;
}


cpl_error_code mark_pixels_surroundings(const int *orders_data,
		int nx,
		int ny,
		int pixel,
		int cluster_distance_x,
		int cluster_distance_y,
		espdr_stack * new_indexes){

	for (int delta = 1; delta <= cluster_distance_x; delta++) {
		if ((pixel % nx) < (nx - delta)) {
			if (orders_data[pixel+delta] == -1) {
				const cpl_error_code my_error =
						espdr_stack_push(new_indexes, pixel + delta);

				espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
						"espdr_identify_clump failed for pixel: %d",
						pixel + delta);
			}
		}

		if ((pixel % nx) >= delta) {
			if (orders_data[pixel-delta] == -1) {
				const cpl_error_code my_error =
						espdr_stack_push(new_indexes, pixel - delta);
				espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
						"espdr_identify_clump failed for pixel: %d",
						pixel - delta);
			}
		}

	}

	for (int delta = 1; delta <= cluster_distance_y; delta++) {
		if ((pixel < nx * (ny - delta)) &&
				((pixel % nx) < (nx - 1)) &&
				((pixel % nx) >= 1)) {
			if ((orders_data[pixel+delta*nx] == -1) &&
					(orders_data[pixel+delta*nx-1] == -1) &&
					(orders_data[pixel+delta*nx+1] == -1)) {
				const cpl_error_code my_error =
						espdr_stack_push(new_indexes, pixel + nx * delta);
				espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
						"espdr_identify_clump failed for pixel: %d",
						pixel + nx * delta);
			}
		}

		if ((pixel >= (delta * nx)) &&
				((pixel % nx) < (nx - 1)) &&
				((pixel % nx) >= 1)) {
			if ((orders_data[pixel-delta*nx] == -1) &&
					(orders_data[pixel-delta*nx-1] == -1) &&
					(orders_data[pixel-delta*nx+1] == -1)) {
				const cpl_error_code my_error =
						espdr_stack_push(new_indexes, pixel - nx * delta);
				espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
						"espdr_identify_clump failed for pixel: %d",
						pixel - nx * delta);
			}
		}

	}

	return CPL_ERROR_NONE;
}

cpl_error_code mark_and_get_new_indexes(int *orders_data,
		int nx,
		int ny,
		int counter_flag,
		int cluster_distance_x,
		int cluster_distance_y,
		espdr_stack * indexes){

	if(espdr_stack_is_empty(indexes)) return CPL_ERROR_NONE;

	cpl_size pixel = espdr_stack_pop(indexes);

	if(pixel < 0 || pixel >= nx * ny) return CPL_ERROR_NONE;

	orders_data[pixel] = counter_flag;
	return mark_pixels_surroundings(orders_data, nx, ny, pixel, cluster_distance_x, cluster_distance_y, indexes);
}

cpl_error_code espdr_identify_clump(int *orders_data,
		int nx,
		int ny,
		int pixel,
		int counter_flag,
		int cluster_distance_x,
		int cluster_distance_y) {

	espdr_ensure(orders_data == NULL, CPL_ERROR_NULL_INPUT,
			"Orders data is NULL");

	if ((pixel >= nx*ny) || (pixel < 0)) {
		espdr_msg_error("pixel out of range: %d", pixel);
		return cpl_error_get_code();
	}

	//espdr_msg("Entering identify_clumps with recursion depth = %d", depth);

	espdr_stack * indexes = espdr_stack_new();
	espdr_stack_push(indexes, pixel);

	cpl_error_code err = CPL_ERROR_NONE;
	while(!espdr_stack_is_empty(indexes) && !err){
		err = mark_and_get_new_indexes(orders_data, nx, ny,
				counter_flag, cluster_distance_x,
				cluster_distance_y, indexes);
	}

	espdr_stack_delete(indexes);

	if(err) return err;

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Number valid orders
 @param         orders_data_imagelist   orders data
 @param         orders_map_imagelist    images list of idetified orders
 @param         ORDERDEF_param          orderdef params
 @param         inst_config             instrument config
 @param         orders_nr               orders number in teh images
 @param[out]    orders_numbered_RE      list of images with numbered orders
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_number_valid_orders(cpl_imagelist *orders_data_imagelist,
		cpl_imagelist *orders_map_imagelist,
		espdr_inst_config *inst_config,
		int min_order_len,
        int min_order_x,
        int max_order_y,
		int *orders_nr,
		cpl_imagelist *orders_numbered_RE) {

	int i, nx, ny;
	cpl_image *curr_image = NULL;
	cpl_image *curr_map = NULL;
	double *data_table = NULL;
	int *map_table = NULL;
	int *orders_data = NULL;
	cpl_image *orders_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(orders_data_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");

	espdr_ensure(orders_map_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");

	int data_size = cpl_imagelist_get_size(orders_data_imagelist);
	int map_size = cpl_imagelist_get_size(orders_map_imagelist);
	espdr_ensure(data_size != map_size, CPL_ERROR_INCOMPATIBLE_INPUT,
			"Input imagelists have different sizes");


	for (i = 0; i < cpl_imagelist_get_size(orders_data_imagelist); i++) {
		curr_image = cpl_imagelist_get(orders_data_imagelist, i);
		data_table = cpl_image_get_data_double(curr_image);
		nx = cpl_image_get_size_x(curr_image);
		ny = cpl_image_get_size_y(curr_image);

		curr_map = cpl_imagelist_get(orders_map_imagelist, i);
		map_table = cpl_image_get_data_int(curr_map);
		espdr_ensure(nx != cpl_image_get_size_x(curr_map),
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"The orders map size x differes from the data size x");
		espdr_ensure(ny != cpl_image_get_size_y(curr_map),
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"The orders map size y differes from the data size y");

		espdr_msg("Treating image %d", i);
		orders_data = (int *)cpl_calloc(nx*ny, sizeof(int));
		my_error = espdr_number_valid_orders_one_image(data_table,
				map_table,
				nx, ny,
				inst_config,
				min_order_len,
                min_order_x,
                max_order_y,
				&orders_nr[i],
				orders_data);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_number_vaid_orders_one_image failed: %s",
				cpl_error_get_message_default(my_error));
		orders_image = cpl_image_wrap_int(nx, ny, orders_data);
		cpl_imagelist_set(orders_numbered_RE,
				cpl_image_duplicate(orders_image), i);
		cpl_image_unwrap(orders_image);
		cpl_free(orders_data);
	}

	return cpl_error_get_code();


}


/*----------------------------------------------------------------------------*/
/**
 @brief     Number valid orders
 @param         orders_data     orders data
 @param         orders_map      images list of idetified orders
 @param         nx              x size of data
 @param         ny              y size of data
 @param         ORDERDEF_param  orderdef params
 @param         inst_config     instrument config
 @param         orders_nr       orders number in teh images
 @param[out]    orders_RE       list of images with numbered orders
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_number_valid_orders_one_image(double *orders_data,
		int *orders_map,
		int nx, int ny,
		espdr_inst_config *inst_config,
		int min_order_len,
        int min_order_x,
        int max_order_y,
		int *order_nr,
		int *orders_RE) {


	espdr_ensure(orders_data == NULL, CPL_ERROR_NULL_INPUT,
			"Input orders data is NULL");

	int i, j, max_order_nb = 0;
	espdr_clump_limit *clump = NULL;
	int min_y_distance = inst_config->min_order_y_dist;
	int order_width_limit = inst_config->min_order_width;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_msg("Extracted order length: %d", min_order_len);

	for (i = 0; i < nx * ny; i++) {
		if (orders_map[i] > max_order_nb) {
			max_order_nb = orders_map[i];
		}
		orders_RE[i] = -1;
	}
	max_order_nb++; // we start at 0, so they are one more

	espdr_msg("Number of clumps identified: %d", max_order_nb);

	clump = (espdr_clump_limit *)cpl_malloc
			(max_order_nb * sizeof(espdr_clump_limit));
	for (i = 0; i < max_order_nb; i++) {
		clump[i].min_x = 10000;
		clump[i].max_x = 0;
		clump[i].min_y = 10000;
		clump[i].max_y = 0;
		clump[i].width_OK = 0;
        clump[i].pixels_width_OK = -1;
		clump[i].order_nr = -1;
	}


	my_error = espdr_find_clumps_limits(orders_map, nx, ny,
			order_width_limit,
			clump);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"espdr_find_clumps_limits failed: %s",
			cpl_error_get_message_default(my_error));

    //espdr_msg("#################### min_y_dist: %d, min_order_len: %d, min_order_x: %d, max_order_y: %d, ny: %d, min_width_pixels_nb: %d",
	//          min_y_distance, min_order_len, min_order_x, max_order_y, ny, inst_config->min_width_pixels_nb[0]);


	*order_nr = 1;
	for (i = 0; i < max_order_nb; i++) {
		if ((clump[i].max_x - clump[i].min_x) > 500) {
		    //espdr_msg("-------------> order: %d", *order_nr);
		    //espdr_msg("clump[%d] min_y: %d, max_y: %d, min_x:%d, max_x: %d order length: %d --> width %d",
		    //          i, clump[i].min_y, clump[i].max_y, clump[i].min_x, clump[i].max_x,
		    //          clump[i].max_x - clump[i].min_x, clump[i].width_OK);
		}
		if (clump[i].width_OK == 1) {
			if ((clump[i].min_y >= min_y_distance) &&
					(clump[i].max_y < ny - min_y_distance)) {
				if ((clump[i].max_x - clump[i].min_x) > min_order_len) {
                    //if (clump[i].min_x < min_order_x) { // doesn't work for the NIRPS new spectral format (NOV 2022)
                    if ((clump[i].min_y > 1) || (clump[i].min_x < min_order_x)) { // validated on all instruments/modes
                        if (clump[i].max_y < max_order_y) {
                            if (clump[i].pixels_width_OK > inst_config->min_width_pixels_nb[0]) {
                                //espdr_msg("Order %d has length %d, min_x = %d, max_x = %d, min_y = %d, max_y = %d",
                                //          *order_nr, (clump[i].max_x - clump[i].min_x),
                                //          clump[i].min_x, clump[i].max_x, clump[i].min_y, clump[i].max_y);
                                clump[i].order_nr = *order_nr;
                                (*order_nr)++;
                            }
                        }
                    }
				}
			}
		}
	}
	(*order_nr)--;
	espdr_msg("Number of orders identified: %d", *order_nr);
	/*
    for (i = 0; i <= max_order_nb; i++) {
        espdr_msg("clump nr %d has order nr: %d", i, clump[i].order_nr);
        espdr_msg("clump nr %d limits: minx: %d, maxx: %d, miny: %d, maxy: %d",
                  i, clump[i].min_x, clump[i].max_x,
                  clump[i].min_y, clump[i].max_y);
        espdr_msg("clump nr %d widthOK %d, order nr: %d",
                  i, clump[i].width_OK, clump[i].order_nr);
    }
	 */
    
    int *clump_index = (int *)cpl_calloc(*order_nr, sizeof(int));
    for (i = 0; i < max_order_nb; i++) {
        if (clump[i].order_nr > 0) {
            clump_index[clump[i].order_nr-1] = i;
        }
    }
    
    cpl_boolean all_sorted = CPL_TRUE;
    for (i = 1; i < *order_nr; i++) {
        if (clump[clump_index[i-1]].max_y > clump[clump_index[i]].max_y) {
            all_sorted = CPL_FALSE;
        }
    }
    
    if (!all_sorted) {
        espdr_msg("Orders not sorted, will proceed to sort");
    
        int tmp_order_nr = -1;
        for (i = 0; i < *order_nr; i++) {
            for (j = i+1; j < *order_nr; j++) {
                if (clump[clump_index[j]].max_y < clump[clump_index[i]].max_y) {
                    espdr_msg("Wrong order: %d (max_y = %d) should be before %d (max_y = %d) - swapping order number %d with %d",
                              j, clump[clump_index[j]].max_y, i, clump[clump_index[i]].max_y,
                              clump[clump_index[j]].order_nr, clump[clump_index[i]].order_nr);
                    tmp_order_nr = clump[clump_index[i]].order_nr;
                    clump[clump_index[i]].order_nr = clump[clump_index[j]].order_nr;
                    clump[clump_index[j]].order_nr = tmp_order_nr;
                }
            }
        }
        
        //for (i = 0; i < max_order_nb; i++) {
        //    if (clump[i].order_nr > 0) {
        //        espdr_msg("Order %d has length %d, min_x = %d, max_x = %d, min_y = %d, max_y = %d",
        //                  clump[i].order_nr, (clump[i].max_x - clump[i].min_x),
        //                  clump[i].min_x, clump[i].max_x, clump[i].min_y, clump[i].max_y);
        //    }
        //}
    }
    
    //for (i = 0; i < *order_nr; i++) {
    //    espdr_msg("ORDER %d, length: %d, pixels_width_nb: %d",
    //              i+1, clump[clump_index[i]].max_x - clump[clump_index[i]].min_x,
    //              clump[clump_index[i]].pixels_width_OK);
    //}
    
	for (i = 0; i < nx * ny; i++) {
		orders_RE[i] = clump[orders_map[i]].order_nr;
	}

    cpl_free(clump_index);
	cpl_free(clump);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Find clumps limits
 @param         orders_data     clumps vector
 @param         nx              nx image size
 @param         ny              ny image size
 @param[out]    clump_RE        limits table for each clump
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_find_clumps_limits(int *orders_data,
		int nx, int ny,
		int order_width_limit,
		espdr_clump_limit *clump_RE) {

	int i, x, y, pixel, clump_nr;

	espdr_ensure(orders_data == NULL, CPL_ERROR_NULL_INPUT,
			"Input orders data is NULL");

	clump_RE[0].order_nr = 0;

	for (y = 0; y < ny; y++) {
		for (x = 0; x < nx; x++) {
			pixel = y * nx + x;
			clump_nr = orders_data[pixel];
			if (clump_nr > 0) {
				if (clump_RE[clump_nr].min_x > x) {
					clump_RE[clump_nr].min_x = x;
				}
				if (clump_RE[clump_nr].max_x < x) {
					clump_RE[clump_nr].max_x = x;
				}
				if (clump_RE[clump_nr].min_y > y) {
					clump_RE[clump_nr].min_y = y;
				}
				if (clump_RE[clump_nr].max_y < y) {
					clump_RE[clump_nr].max_y = y;
				}

				if (clump_RE[clump_nr].width_OK == 0) {
					if (y < (ny - order_width_limit)) {
						i = 1;
						while ((i < order_width_limit) &&
								(orders_data[(y+i) * nx + x] == clump_nr)) {
							i++;
						}
						if (i == order_width_limit) {
							clump_RE[clump_nr].width_OK = 1;
						}
					}
				}

                if (y < (ny - order_width_limit)) {
                    i = 1;
                    while ((i < order_width_limit) &&
                            (orders_data[(y+i) * nx + x] == clump_nr)) {
                        i++;
                    }
                    if (i == order_width_limit) {
                        clump_RE[clump_nr].pixels_width_OK++;
                    }
                }

			}
		}
	}

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Fit orders
 @param         data_iml            list of input images
 @param         orders_map_iml      list of identified orders images
 @param         total_mask_iml      list of masks
 @param         ORDERDEF_param      orderdef params
 @param         orders_nr           number of orders
 @param         RON                 computed RON
 @param         CONAD               computed conversion factor
 @param         CCD_geom            CCD geometry
 @param         inst_config         instrument config
 @param[out]    order_pos_RE        orders position in the middle
 @param[out]    coeffs_table_RE     fit result coeffs
 @param[out]    order_res_stdev_RE  residuals stdev
 @param[out]    order_res_min_RE    residuals minimum
 @param[out]    order_res_max_RE    residuals maximum
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_orders(cpl_imagelist *data_iml,
		cpl_imagelist *orders_map_iml,
		cpl_imagelist *total_mask_iml,
		int fibre_nr,
		int *orders_nr,
		double *RON,
		double *CONAD,
		espdr_CCD_geometry *CCD_geom,
		espdr_inst_config *inst_config,
		double **orders_pos_RE,
		cpl_table **coeffs_table_RE,
		double **order_res_stdev_RE,
		double **order_res_min_RE,
		double **order_res_max_RE) {

	int i, j, k, index, nx, ny;
	cpl_image *curr_image = NULL;
	cpl_image *curr_map = NULL;
	cpl_image *curr_mask = NULL;
	double *data_table = NULL;
	int *map_table = NULL;
	int *mask_table = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	espdr_bissector **bissector;
    int n_gauss = inst_config->slices_nb;
    int order_fit_step = inst_config->order_fit_step;
	int poly_deg = inst_config->order_fit_poly_deg;
	int order_param_index, order_start, order_end;

	espdr_ensure(data_iml == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(orders_map_iml == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(total_mask_iml == NULL, CPL_ERROR_NULL_INPUT,
			"Mask imagelist is NULL");

	int data_size = cpl_imagelist_get_size(data_iml);
	int map_size = cpl_imagelist_get_size(orders_map_iml);
	espdr_ensure(data_size != map_size, CPL_ERROR_INCOMPATIBLE_INPUT,
			"Input imagelists have different sizes");
	int mask_size = cpl_imagelist_get_size(total_mask_iml);
    
    cpl_table **orders_centroids = (cpl_table **)cpl_malloc(data_size * sizeof(cpl_table *));
    char *filename_centroids = (char *)cpl_malloc(64 * sizeof(char));
    
	index = 0;
	for (i = 0; i < cpl_imagelist_get_size(data_iml); i++) {
        
        orders_centroids[i] = cpl_table_new(10000);

		curr_image = cpl_imagelist_get(data_iml, i);
		data_table = cpl_image_get_data_double(curr_image);
		nx = cpl_image_get_size_x(curr_image);
		ny = cpl_image_get_size_y(curr_image);

		curr_map = cpl_imagelist_get(orders_map_iml, i);
		map_table = cpl_image_get_data_int(curr_map);
		espdr_ensure(nx != cpl_image_get_size_x(curr_map),
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"The orders map size x differs from the data size x");
		espdr_ensure(ny != cpl_image_get_size_y(curr_map),
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"The orders map size y differs from the data size y");

		curr_mask = cpl_imagelist_get(total_mask_iml, index);
		mask_table = cpl_image_get_data_int(curr_mask);
		espdr_ensure(cpl_image_get_size_x(curr_mask) != nx,
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"Mask size X differs from the data size X");
		espdr_ensure(cpl_image_get_size_y(curr_mask) != ny,
				CPL_ERROR_INCOMPATIBLE_INPUT,
				"Mask size Y differs from the data size Y");
		index++;
		if (index == mask_size) {
			index = 0;
		}

		order_param_index = fibre_nr * CCD_geom->ext_nb + i;
		order_start = inst_config->order_start[order_param_index];
		order_end = inst_config->order_end[order_param_index];
		//espdr_msg("fibre: %d, ext: %d, order index: %d, order start: %d, order_end: %d",
		//          fibre_nr, i, order_param_index, order_start, order_end);

		bissector = (espdr_bissector **)cpl_malloc
				(orders_nr[i]*sizeof(espdr_bissector *));
		for (j = 0; j < orders_nr[i]; j++) {
			bissector[j] = (espdr_bissector *)cpl_malloc
					(nx * sizeof(espdr_bissector));
			for (k = 0; k < nx; k++) {
				bissector[j][k].width = 0;
				bissector[j][k].first_y = 0;
				bissector[j][k].last_y = 0;
				bissector[j][k].centroid = 0.0;
				bissector[j][k].sig_centroid = 0.0;
			}
		}

		my_error = espdr_compute_bissector_one_image(data_table, map_table,
				mask_table,
				nx, ny, order_fit_step,
				order_start, order_end,
				orders_nr[i], n_gauss,
				i, RON, CONAD,
				CCD_geom, inst_config,
				orders_pos_RE[i],
				bissector);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_compute_bissector_one_image failed: %s",
				cpl_error_get_message_default(my_error));
        
		/* Fit each order according to the bissector */

        //char column_name[KEYWORD_LENGTH];
		//for (j = 0; j < poly_deg; j++) {
		//	sprintf(column_name, "COEFF_%d", j);
		//	espdr_msg_debug("column_name: %s", column_name);

		//	my_error = cpl_table_new_column(coeffs_table_RE[i], column_name,
		//			CPL_TYPE_DOUBLE);
		//	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		//			"cpl_table_new_column failed for coeff %d: %s",
		//			j, cpl_error_get_message_default(my_error));
		//}

		my_error = espdr_fit_orders_polynomial(bissector, nx,
                                               inst_config->instrument, fibre_nr, i,
                                               order_start, order_end,
                                               orders_nr[i],
                                               order_fit_step, poly_deg,
                                               inst_config->res_ksigma,
                                               &coeffs_table_RE[i],
                                               order_res_stdev_RE[i],
                                               order_res_min_RE[i],
                                               order_res_max_RE[i]);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_fit_orders_polynomial failed: %s",
				cpl_error_get_message_default(my_error));

#if SAVE_DEBUG_PRODUCT_ORDERDEF
        char column_name_centroid[32];
        for (j = 0; j < orders_nr[i]; j++) {
            sprintf(column_name_centroid, "ORDER_%d", j+1);
            my_error = cpl_table_new_column(orders_centroids[i],
                                            column_name_centroid, CPL_TYPE_DOUBLE);
            sprintf(column_name, "COEFF_0");
            double read_value = cpl_table_get_double(coeffs_table_RE[i],
                                              column_name, j, NULL);
            double my_coeff = read_value;
            
            int index_centroid = 0;
            for (k = 0; k < nx; k++) {
                if (bissector[j][k].centroid != 0.0) {
                    sprintf(column_name_centroid, "ORDER_%d", j+1);
                    my_error = cpl_table_set_double(orders_centroids[i],
                                                    column_name_centroid,
                                                    index_centroid,
                                                    bissector[j][k].centroid-my_coeff);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_table_set_double failed for order %d: %s",
                                 j+1, cpl_error_get_message_default(my_error));
                    index_centroid++;
                }
            }
        }
        
        sprintf(filename_centroids, "%s_orders_centroids_%c_%d.fits",
                inst_config->instrument, fibre_name[fibre_nr], i);
        my_error = cpl_table_save(orders_centroids[i], NULL, NULL,
                                  filename_centroids, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_save failed for %s: %s",
                     filename_centroids, cpl_error_get_message_default(my_error));
        
        espdr_msg("%s saved", filename_centroids);
#endif
        
		for (j = 0; j < orders_nr[i]; j++) {
			cpl_free(bissector[j]);
		}
		cpl_free(bissector);
	}
    
    
    cpl_free(filename_centroids);
    for (i = 0; i < cpl_imagelist_get_size(data_iml); i++) {
        cpl_table_delete(orders_centroids[i]);
    }
    cpl_free(orders_centroids);
    
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Find orders bissector
 @param         orders_data     input image data
 @param         orders_map      identified orders image
 @param         mask            mask
 @param         nx              nx image size
 @param         ny              ny image size
 @param         order_fit_step  step, the order is fit with
 @param         orders_nr       number of orders
 @param         n_gauss         number of gaussians = number of slices
 @param         ext_nr          extension nr
 @param         RON             computed RON
 @param         CONAD           computed conversion factor
 @param         CCD_geom        CCD geometry
 @param         inst_config     instrument config
 @param[out]    order_pos_RE    orders position
 @param[out]    bissector_RE    orders maximum
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_bissector_one_image(double *orders_data,
		int *orders_map,
		int *mask,
		int nx, int ny,
		int order_fit_step,
		int order_start,
		int order_end,
		int orders_nr,
		int n_gauss,
		int ext_nr,
		double *RON,
		double *CONAD,
		espdr_CCD_geometry *CCD_geom,
		espdr_inst_config *inst_config,
		double *orders_pos_RE,
		espdr_bissector **bissector_RE) {

	espdr_msg("Compute order centre for one image: slices: %d, gaussians: %d",
			inst_config->slices_nb, n_gauss);

	espdr_ensure(orders_data == NULL, CPL_ERROR_NULL_INPUT,
			"Input orders data is NULL");
	espdr_ensure(orders_map == NULL, CPL_ERROR_NULL_INPUT,
			"Input orders map is NULL");
	espdr_ensure(mask == NULL, CPL_ERROR_NULL_INPUT,
			"Mask is NULL");

	int i, j, k, l, x, y, pixel, output_index;
	cpl_error_code my_error = CPL_ERROR_NONE;
	/* The fit_start is computed to ensure
       that the centroid is computed for nx/2 regarding the order_fit_step */
	int fit_start = (int)(nx/2 - order_start) % order_fit_step + order_start;
	int fit_end = order_end;
	int background_pxls = inst_config->background_pxls;
	int inter_slice_dist = inst_config->inter_slice_dist;
	int slice_width = inst_config->slice_width;

	espdr_ngauss_data *g_data = (espdr_ngauss_data *)cpl_malloc
			(sizeof(espdr_ngauss_data));
	double centroid_ngauss = 0.0;
	double sig_centroid_ngauss = 0.0;
	double flux_ngauss = 0.0;
	double sig_flux_ngauss = 0.0;
	double fwhm_ngauss = 0.0;
	double sig_fwhm_ngauss = 0.0;

	for (x = 0; x < nx; x++) {
		for (y = 0; y < ny; y++) {
			pixel = y * nx + x;
			if (orders_map[pixel] > 0) {
				if (bissector_RE[orders_map[pixel]-1][x].first_y == 0) {
					if ((y - background_pxls) > 0) {
						bissector_RE[orders_map[pixel]-1][x].first_y =
								y - background_pxls;
					} else {
						bissector_RE[orders_map[pixel]-1][x].first_y = 1;
					}
				}
			}
		}
		for (y = ny-1; y >= 0; y--) {
			pixel = y * nx + x;
			if (orders_map[pixel] > 0) {
				if (bissector_RE[orders_map[pixel]-1][x].last_y == 0) {
					if ((y + background_pxls) < (ny - 1)) {
						bissector_RE[orders_map[pixel]-1][x].last_y =
								y + background_pxls;
					} else {
						bissector_RE[orders_map[pixel]-1][x].last_y = ny - 1;
					}
				}

				bissector_RE[orders_map[pixel]-1][x].width =
						bissector_RE[orders_map[pixel]-1][x].last_y -
						bissector_RE[orders_map[pixel]-1][x].first_y + 1;

				//espdr_msg("order: %d, pixel: [%d, %d], first: %d, last %d, width: %d, value: %lf",
				//          orders_map[pixel]-1, x, y,
				//          bissector_RE[orders_map[pixel]-1][x].first_y,
				//          bissector_RE[orders_map[pixel]-1][x].last_y,
				//          bissector_RE[orders_map[pixel]-1][x].width,
				//          orders_data[pixel]);
			}
			if (orders_map[pixel] == 1) {
				espdr_msg_debug("order 0: pixel: %d, width: %d",
						x, bissector_RE[orders_map[pixel]-1][x].width);
			}
		}
	}


	espdr_msg("Finding order centre");
	for (i = 0; i < orders_nr; i++) {
		espdr_msg("Fitting order (NGauss): %d", i);
		for (j = fit_start; j < fit_end; j = j + order_fit_step) {
			l = 0;
			bissector_RE[i][j].centroid = 0.0;
			bissector_RE[i][j].sig_centroid = 0.0;

			int valid_pixels = 0;
			for (k = bissector_RE[i][j].first_y;
					k < bissector_RE[i][j].first_y + bissector_RE[i][j].width;
					k++) {
				pixel = k * nx + j;

				if (mask[pixel] == 0) {
					valid_pixels++;
				}
			}

			if ((i == -1) && (j > 0)) {
				espdr_msg("order %d pxl %d: width: %d, valid pixels %d",
						i, j, bissector_RE[i][j].width, valid_pixels);
			}

			if (valid_pixels > 3*inst_config->slices_nb+2) {
				g_data->x = (double *)cpl_calloc
						(bissector_RE[i][j].width, sizeof (double));
				g_data->y = (double *)cpl_calloc
						(bissector_RE[i][j].width, sizeof (double));
				g_data->err = (double *)cpl_calloc
						(bissector_RE[i][j].width, sizeof (double));
				g_data->fit = (double *)cpl_calloc
						(bissector_RE[i][j].width, sizeof (double));

				for (k = bissector_RE[i][j].first_y;
						k < bissector_RE[i][j].first_y + bissector_RE[i][j].width;
						k++) {
					pixel = k * nx + j;

					if ((i == -1) && (j > 0)) {
						espdr_msg("order %d pxl %d: orders_data: %f, mask: %d",
								i, j, orders_data[pixel], mask[pixel]);
					}

					if (mask[pixel] == 0) {
						output_index =
								espdr_get_output_index_for_pixel(ext_nr,
										pixel,
										CCD_geom);
						g_data->x[l] = k;
						// +1 is to make the difference between the y index starting at 0 and image pixels starting at 1
						//g_data->x[l] = k+1;
						g_data->y[l] = orders_data[pixel];
						int data_pixel;
						if (orders_data[pixel] < 0.0) {
							data_pixel = 0.0;
						} else {
							data_pixel = orders_data[pixel];
						}
						g_data->err[l] = sqrt(data_pixel +
								RON[output_index] * RON[output_index]
														* CONAD[output_index] * CONAD[output_index]);
						l++;
					}
				}

				g_data->n = l;
				g_data->m = n_gauss;
				g_data->delta = 0.0;

				if ((i == -1) && (j > 1500) && (j < 1510)) {
					espdr_msg("Fitting with ngauss: %d points (order %d, pxl %d)", g_data->n, i, j);
					for (k = 0; k < g_data->n; k++) {
						espdr_msg("g_data->x[%d] = %lf, g_data->y[%d] = %lf, g_data->err[%d] = %lf",
								k, g_data->x[k], k, g_data->y[k],
								k, g_data->err[k]);
					}
				}

                if ((i == -1) && (j > 1500) && (j < 1510)) {
					my_error = espdr_fit_Ngauss_equi_dist(g_data,
							inter_slice_dist,
							slice_width,
							&centroid_ngauss,
							&sig_centroid_ngauss,
							&flux_ngauss,
							&sig_flux_ngauss,
							&fwhm_ngauss,
							&sig_fwhm_ngauss,
							1, 1);
				} else {
					my_error = espdr_fit_Ngauss_equi_dist(g_data,
							inter_slice_dist,
							slice_width,
							&centroid_ngauss,
							&sig_centroid_ngauss,
							&flux_ngauss,
							&sig_flux_ngauss,
							&fwhm_ngauss,
							&sig_fwhm_ngauss,
							1, 0);
				}

				espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
						"espdr_fit_Ngauss_equi_dist failed: %s",
						cpl_error_get_message_default(my_error));

				espdr_msg_debug("NGauss fitted");
				// +1 is to get pixels on the image, first pixel has x=1
				bissector_RE[i][j].centroid = centroid_ngauss+1;
				bissector_RE[i][j].sig_centroid = sig_centroid_ngauss;

				// if fit error == 0, smth wrong happened, point sould be excluded from further fit
				if ((isnan(sig_centroid_ngauss)) || (sig_centroid_ngauss == 0.0))  {
					bissector_RE[i][j].sig_centroid = DBL_MAX;
				}
                // if the fwhm is very small - the fit is wrong
                if (fwhm_ngauss < 1.5) {
                    bissector_RE[i][j].sig_centroid = DBL_MAX;
                }
                
                if ((i == -1) && (j > 1500) && (j < 1510)) {
					espdr_msg("order: %d, pixel %d, centroid: %f, sig_centroid: %.16f, fwhm_ngauss: %f",
							i, j, centroid_ngauss, sig_centroid_ngauss, fwhm_ngauss);
				}
				cpl_free(g_data->x);
				cpl_free(g_data->y);
				cpl_free(g_data->fit);
				cpl_free(g_data->err);
			}
		}
		orders_pos_RE[i] = bissector_RE[i][(int)(nx/2)].centroid;
	}

	cpl_free(g_data);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Polynomial orders fit
 @param         bissector           orders maximum
 @param         nx                  nx image size
 @param         orders_nr           number of orders
 @param         order_fit_step      step, the order is fit with
 @param         poly_deg            polynomial degree
 @param         ksigma              ksigma for sigma clipping
 @param[out]    coeffs_table_RE     fit result coeffs
 @param[out]    order_res_stdev_RE  residuals stdev
 @param[out]    order_res_min_RE    residuals minimum
 @param[out]    order_res_max_RE    residuals maximum
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_orders_polynomial(espdr_bissector **bissector,
                                           int nx,
                                           char *instrument,
                                           int fibre_nr,
                                           int det_nr,
                                           int order_start,
                                           int order_end,
                                           int orders_nr,
                                           int order_fit_step,
                                           cpl_size poly_deg,
                                           double ksigma,
                                           cpl_table **coeffs_table_RE,
                                           double *order_res_stdev_RE,
                                           double *order_res_min_RE,
                                           double *order_res_max_RE) {

	int i, j, index;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_msg_debug("Fitting polynomial");
	int vector_length = (int)(nx / order_fit_step) + 1;
	espdr_msg_debug("vector length: %d", vector_length);
	espdr_msg_debug("order fit step: %d", order_fit_step);
	double **orders_fit_coeffs = (double **)cpl_malloc(orders_nr*sizeof(double *));
	for (i = 0; i < orders_nr; i++) {
		orders_fit_coeffs[i] = (double *)cpl_calloc((poly_deg+1), sizeof(double));
	}

	int fit_start = (int)(nx/2 - order_start) % order_fit_step + order_start;
	int fit_end = order_end;
	int order_length = 0;
	int order_length_no_outliers = 0;

	char column_name[KEYWORD_LENGTH] = {};

	double *residuals = NULL;
	double *residuals_no_outliers = NULL;
	int cosmics = 0;
	cpl_vector *residuals_no_outliers_vector = NULL;


	espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc
			(sizeof(espdr_poly_data));
	p_data->x = (double *) cpl_calloc (vector_length, sizeof (double));
	p_data->y = (double *) cpl_calloc (vector_length, sizeof (double));
	p_data->err = (double *) cpl_calloc (vector_length, sizeof (double));
	p_data->n = vector_length;

	espdr_poly_data *p_data_no_outliers = (espdr_poly_data *)cpl_malloc
			(sizeof(espdr_poly_data));

	double *fit = NULL;
	double *coeffs = NULL;
	double *coeffs_err = NULL;
	double chisq = 0.0;

	int error_median_vector_length = 0;
	int error_median_index = 0;
	double *error_median;
	cpl_vector * error_median_vector = NULL;
	double error_median_global = 1.0;

	for (j = fit_start; j < fit_end; j = j + order_fit_step) {
		error_median_vector_length++;
	}
	error_median_vector_length++;

    cpl_table *orders_res = cpl_table_new(10000);
    char *filename_res = (char *)cpl_malloc(64 * sizeof(char));
    
	espdr_msg("Fitting %d orders (polynomial)", orders_nr);
	/* Fit a polynome on an order */
	for (i = 0; i < orders_nr; i++) {
		espdr_msg("Fitting order (polynomial): %d", i);

		for (j = 0; j < vector_length; j++) {
			p_data->x[j] = 0.0;
			p_data->y[j] = 0.0;
			p_data->err[j] = 1.0;
		}

		index = 0;

		error_median = (double *)cpl_calloc(error_median_vector_length, sizeof(double));
		error_median_index = 0;
		for (j = fit_start; j < fit_end; j = j + order_fit_step) {
			error_median[error_median_index] = 1000.0;
			if (bissector[i][j].centroid > 0.0) {
				error_median[error_median_index] = bissector[i][j].sig_centroid;
			}
			//espdr_msg("error_median[%d] = %f",
			//          error_median_index, error_median[error_median_index]);
			error_median_index++;
		}
		error_median_vector = cpl_vector_wrap(error_median_vector_length, error_median);
		error_median_global = cpl_vector_get_median(error_median_vector);
		//espdr_msg("MEDIAN ERROR: %f", error_median_global);

		for (j = fit_start; j < fit_end; j = j + order_fit_step) {
			if (bissector[i][j].centroid > 0.0) {
				p_data->x[index] = (double)j;
				p_data->y[index] = bissector[i][j].centroid;
				p_data->err[index] = bissector[i][j].sig_centroid;
				if ((i == -1) & (index < 100000)) {
				    espdr_msg("bissector %d: centroid = %lf, sig = %lf, width = %d", j,
				              bissector[i][j].centroid,
				              bissector[i][j].sig_centroid,
				              bissector[i][j].width);
				}
				index++;
			}
		}
		order_length = index;
		p_data->n = order_length;

		fit = (double *) cpl_calloc (order_length, sizeof(double));
		coeffs = (double *) cpl_calloc (poly_deg, sizeof(double));
		coeffs_err = (double *) cpl_calloc (poly_deg, sizeof(double));

		my_error = espdr_fit_poly(p_data, poly_deg,
				fit, coeffs, coeffs_err, &chisq, 0);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_fit_poly failed: %s",
				cpl_error_get_message_default(my_error));

		//espdr_msg("Compute first residuals for order: %d", i);
		residuals = (double *)cpl_calloc(order_length, sizeof(double));
		my_error = espdr_compute_residuals(p_data, fit, order_length,
				residuals);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_compute_residuals failed: %s",
				cpl_error_get_message_default(my_error));

		cpl_free(fit);
		cpl_free(coeffs);
		cpl_free(coeffs_err);

		cosmics = 0;
		for (j = 0; j < order_length; j++) {
			//if ((i != -1) && (j < order_length) &&
			//    ((residuals[j]/error_median_global < -ksigma) ||
			//    (residuals[j]/error_median_global > ksigma))) {
			//    espdr_msg("p_data->y[%d] = %lf, fit[%d] = %lf, res[%d] = %f, err[%d] = %lf, ksigma = %lf, res/err = %f",
			//              j, p_data->y[j], j, fit[j], j, residuals[j],
			//              j, p_data->err[j], ksigma, residuals[j]/error_median_global);
			//}
			//if ((residuals[j]/p_data->err[j] < -ksigma) ||
			//    (residuals[j]/p_data->err[j] > ksigma)) {
			if ((residuals[j]/error_median_global < -ksigma) ||
					(residuals[j]/error_median_global > ksigma)) {
				cosmics++;
			}
		}

		espdr_msg("order :%d outliers: %d", i, cosmics);

		order_length_no_outliers = order_length - cosmics;
		p_data_no_outliers->x = (double *)cpl_calloc((order_length_no_outliers),
				sizeof(double));
		p_data_no_outliers->y = (double *)cpl_calloc((order_length_no_outliers),
				sizeof(double));
		p_data_no_outliers->err = (double *)cpl_calloc((order_length_no_outliers),
				sizeof(double));

		index = 0;
		for (j = 0; j < order_length; j++) {
			if ((residuals[j]/error_median_global < ksigma) &&
					(residuals[j]/error_median_global > -ksigma)) {
				p_data_no_outliers->x[index] = p_data->x[j];
				p_data_no_outliers->y[index] = p_data->y[j];
				p_data_no_outliers->err[index] = p_data->err[j];
				index++;
			}
			if (i == -1) {
			    espdr_msg("res[%d] = %lf, err[%d] = %lf, ksigma = %lf",
			              j, residuals[j], j, p_data->err[j], ksigma);
			}
		}
		//espdr_msg("index = %d, order length = %d", index, order_length);
		// if cosmics are 0 and index is 0, fit has given NaNs

		if (cosmics == 0 && index == 0) {
			index = order_length;
		}

		//espdr_ensure(index == 0, CPL_ERROR_ILLEGAL_INPUT,
		//"Impossible to fit order %d, ", i);

		p_data_no_outliers->n = index;

		cpl_free(residuals);

		fit = (double *) cpl_calloc (order_length_no_outliers, sizeof(double));
		coeffs = (double *) cpl_calloc (poly_deg, sizeof(double));
		coeffs_err = (double *) cpl_calloc (poly_deg, sizeof(double));

		my_error = espdr_fit_poly(p_data_no_outliers, poly_deg,
				fit, coeffs, coeffs_err, &chisq, 0);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_fit_poly failed: %s",
				cpl_error_get_message_default(my_error));

		for (j = 0; j < poly_deg; j++) {
			orders_fit_coeffs[i][j] = coeffs[j];
			espdr_msg_debug("Coeff %d = %lf", j, coeffs[j]);

			sprintf(column_name, "COEFF_%d", j);
			my_error = cpl_table_set_double(*coeffs_table_RE,
					column_name, i,
					coeffs[j]);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_table_set_double failed for coeff 0: %s",
					cpl_error_get_message_default(my_error));
		}

		residuals_no_outliers = (double *)cpl_calloc((order_length_no_outliers),
				sizeof(double));
		my_error = espdr_compute_residuals(p_data_no_outliers, fit,
				order_length_no_outliers,
				residuals_no_outliers);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_compute_residuals failed: %s",
				cpl_error_get_message_default(my_error));

		residuals_no_outliers_vector = cpl_vector_wrap(order_length_no_outliers,
				residuals_no_outliers);

		order_res_stdev_RE[i] = cpl_vector_get_stdev(residuals_no_outliers_vector);
		order_res_min_RE[i] = cpl_vector_get_min(residuals_no_outliers_vector);
		order_res_max_RE[i] = cpl_vector_get_max(residuals_no_outliers_vector);

		//espdr_msg("order_res_stdev[%d] = %lf", i, order_res_stdev_RE[i]);
		//espdr_msg("order_res_min[%d] = %lf", i, order_res_min_RE[i]);
		//espdr_msg("order_res_max[%d] = %lf", i, order_res_max_RE[i]);
        
#if SAVE_DEBUG_PRODUCT_ORDERDEF
        char column_name_res[32];
        sprintf(column_name_res, "ORDER_RES_%d", i+1);
        my_error = cpl_table_new_column(orders_res, column_name_res, CPL_TYPE_DOUBLE);
        
        for (j = 0; j < order_length_no_outliers; j++) {
            sprintf(column_name_res, "ORDER_RES_%d", i+1);
            my_error = cpl_table_set_double(orders_res,
                                            column_name_res,
                                            j,
                                            residuals_no_outliers[j]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_set_double failed for order %d: %s",
                         i+1, cpl_error_get_message_default(my_error));
        }
        
        sprintf(filename_res, "%s_orders_residuals_%c_%d.fits",
                instrument, fibre_name[fibre_nr], det_nr);
        my_error = cpl_table_save(orders_res, NULL, NULL,
                                  filename_res, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_save failed for %s: %s",
                     filename_res, cpl_error_get_message_default(my_error));
#endif
        
		cpl_vector_unwrap(error_median_vector);
		cpl_free(error_median);
		cpl_vector_unwrap(residuals_no_outliers_vector);
		cpl_free(residuals_no_outliers);
		cpl_free(p_data_no_outliers->x);
		cpl_free(p_data_no_outliers->y);
		cpl_free(p_data_no_outliers->err);
		cpl_free(fit);
		cpl_free(coeffs);
		cpl_free(coeffs_err);
	}

#if SAVE_DEBUG_PRODUCT_ORDERDEF
    espdr_msg("%s saved", filename_res);
#endif
    
	cpl_free(p_data->x);
	cpl_free(p_data->y);
	cpl_free(p_data->err);
	cpl_free(p_data);
	cpl_free(p_data_no_outliers);
	for (i = 0; i < orders_nr; i++) {
		cpl_free(orders_fit_coeffs[i]);
	}
	cpl_free(orders_fit_coeffs);
    
    cpl_free(filename_res);
    cpl_table_delete(orders_res);
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Create a coeffs cpl_table with columns
 @param[out]    coeffs_table    table to insert columns
 @param         poly_deg        number of columns to insert
 @return	CPL_ERROR_NONE iff OK

 The function merges all the images form the list and saves it into one.
 The order of frames is first column, then row.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_create_coeffs_table(cpl_table *coeffs_table,
		int poly_deg) {


	cpl_error_code my_error;
	int i;
	char column_name[KEYWORD_LENGTH];

	for (i = 0; i < poly_deg; i++) {
		sprintf(column_name, "COEFF_%d", i);
		//espdr_msg("column_name: %s", column_name);

		my_error = cpl_table_new_column(coeffs_table, column_name,
				CPL_TYPE_DOUBLE);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"cpl_table_new_column failed for coeff %d: %s",
				i, cpl_error_get_message_default(my_error));
	}

	return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief		Compute residuals
 @param         p_data          data for the fit
 @param         fit             the fit
 @param         length          data length
 @param[out]    residuals_RE    residuals of the fit
 @return	CPL_ERROR_NONE iff OK

 The function merges all the images form the list and saves it into one.
 The order of frames is first column, then row.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_compute_residuals(espdr_poly_data *p_data,
		double *fit,
		int length,
		double *residuals_RE) {

	int i;
	double sum_res = 0.0;

	for (i = 0; i < length; i++) {
		//residuals_RE[i] = fabs(p_data->y[i] - fit[i]);
		residuals_RE[i] = p_data->y[i] - fit[i];
		sum_res += fabs(residuals_RE[i]);
	}

	espdr_msg_debug("Sum of residuals: %lf", sum_res);

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief		Get the physical number for orders
 @param         CCD_geom        CCD geometry
 @param         inst_config     instrument config
 @param         fibre_nr        number of the fibre
 @param[out]    orders_phys_nb  physical orders numbers
 @return	CPL_ERROR_NONE iff OK

 The function merges all the images form the list and saves it into one.
 The order of frames is first column, then row.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_get_order_phys_numbers(espdr_CCD_geometry *CCD_geom,
		espdr_inst_config *inst_config,
		int fibre_nr,
		int *orders_phys_nb) {

	int i, j, index, first_order, fibre_ext_id, slices_nb;

	index = 0;
	first_order = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		fibre_ext_id = fibre_nr*CCD_geom->ext_nb+i;
		first_order = inst_config->order_phys_nb[fibre_ext_id];
		slices_nb = inst_config->slices_nb_per_phys_order/
				inst_config->slices_nb;
		for (j = 0; j < inst_config->orders_nb[fibre_ext_id]; j++) {
            //if ((strcmp(inst_config->instrument, "NIRPS") == 0) && (j == 44)) {
            //    first_order--;
            //}
            // patch to compute correctly the order physical numbers for NIRPS in May-June and November
            if (strcmp(inst_config->instrument, "NIRPS") == 0) {
                if ((inst_config->orders_nb[fibre_nr] == 70) && (j == 44)) { // May-June frames
                    first_order--;
                }
                if ((inst_config->orders_nb[fibre_nr] == 71) && (j == 43)) { // November frames
                    first_order = first_order - 2;
                }
            }
            
			orders_phys_nb[index] = first_order;
			//espdr_msg("order_phys[%d] = %d", j, first_order);
			if (slices_nb == 1) {
				first_order--;
				slices_nb = inst_config->slices_nb_per_phys_order/
						inst_config->slices_nb;
			} else {
				slices_nb--;
			}
			index++;
		}
	}

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Draw orders on the products
 @param         image   product image
 @param         coeffs  orders coeffs
 @return	CPL_ERROR_NONE iff OK

 The function draws fitted orders on the product image
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_draw_orders(cpl_image *orders_table,
		int orders_nb,
		int order_start,
		int order_end,
		int poly_deg,
		int CCD_size_y,
		cpl_table *coeffs) {

	int i, j, k, position_int;
	double position;
	char column_name[15];
	double my_coeffs[poly_deg];
	double read_value = 0.0;

	//espdr_msg("Polynomial is of degree %d", poly_deg);
	for (i = 0; i < orders_nb; i++) {
		//espdr_msg("Drawing order %d", i);
		for (k = 0; k < poly_deg; k++) {
			sprintf(column_name, "COEFF_%d", k);
			read_value = cpl_table_get_double(coeffs,
					column_name, i, NULL);
			//espdr_msg("order %d, coeff %d: %.16lf",
			//          i, k, read_value);
			my_coeffs[k] = read_value;
		}

		for (j = order_start; j < order_end; j++) {
			position = 0.0;
			for (k = poly_deg-1; k > 0; k--) {
				position = (position + my_coeffs[k]) * (double)j;
			}
			position = position + my_coeffs[0];
			position_int = round(position);
			//espdr_msg("Drawing pxl %d at position %f", j, position);
			if ((position_int > 0) && (position_int < CCD_size_y)) {
				cpl_image_set(orders_table, j, position_int, 300);
			}

		}
	}

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save debug product
 @param
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_save_orderdef_debug_product(cpl_frameset *frameset,
                                                 cpl_parameterlist *parameters,
                                                 espdr_inst_config *inst_config,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 espdr_qc_keywords *qc_kws,
                                                 cpl_propertylist *keywords,
                                                 cpl_propertylist **keywords_ext,
                                                 char *filename,
                                                 char *pro_catg,
                                                 cpl_imagelist *iml_to_save,
                                                 int fiber_nr) {
    
    char full_filename[50];
    sprintf(full_filename, "%s_%s_%c.fits",
            inst_config->instrument, filename, fibre_name[fiber_nr]);
    
    char *new_keyword = (char *)cpl_malloc(KEYWORD_LENGTH * sizeof(char));
    sprintf(new_keyword, "%s_%c", pro_catg, fibre_name[fiber_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    cpl_free(new_keyword);
    cpl_propertylist_append_int(keywords, qc_kws->qc_orderdef_check_kw, 1);
    
    espdr_dfs_image_save(frameset, parameters,
                         frameset, "espdr_orderdef",
                         keywords, keywords_ext,
                         full_filename, iml_to_save,
                         CPL_TYPE_FLOAT, CCD_geom);
    espdr_msg("Saved %s", full_filename);
    
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Save debug product
 @param
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_save_orderdef_bkgr_div(cpl_imagelist *CCD_corr_img,
                                            cpl_imagelist *bkgr_img,
                                            cpl_propertylist *keywords,
                                            espdr_inst_config *inst_config,
                                            int number_of_ext,
                                            int fibre_nr) {
    
    char filename[FILENAME_LENGTH];
    for (int j = 0; j < number_of_ext; j++) {
        sprintf(filename, "%s_div_CCD_corr_bkgr_ext_%d_%c.fits",
                inst_config->instrument, j, fibre_name[fibre_nr]);
        cpl_image *bkgr_ext = cpl_imagelist_get(bkgr_img, j);
        double min_bkgr = cpl_image_get_min(bkgr_ext);
        if (min_bkgr < 0.0) {
            min_bkgr = -min_bkgr * 1.1;
        } else {
            if (min_bkgr < 1.0) {
                min_bkgr = 1.0;
            }
        }
        cpl_image *bkgr_min = cpl_image_add_scalar_create(bkgr_ext, min_bkgr);
        cpl_image *final_img = cpl_image_divide_create(cpl_imagelist_get(CCD_corr_img, j),
                                                       bkgr_min);
        //cpl_image *data_ext = cpl_imagelist_get(CCD_corrected_orderdef, j);
        //cpl_image *data_bkgr_sub = cpl_image_subtract_create(data_ext, bkgr_ext);
        //cpl_image *final_img = cpl_image_divide_create(data_bkgr_sub, bkgr_ext);
        
        espdr_msg("Saving %s...", filename);
        cpl_image_save(final_img, filename, CPL_TYPE_FLOAT, keywords, CPL_IO_CREATE);
        espdr_msg("%s saved", filename);
        cpl_image_delete(bkgr_min);
        cpl_image_delete(final_img);
    }
    
    return cpl_error_get_code();
}
                                 
/*---------------------------------------------------------------------------*/
/**
 @brief     Save debug product
 @param
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_save_orderdef_simple_debug_product(cpl_imagelist *iml_to_save,
                                                        char *filename,
                                                        cpl_propertylist *keywords,
                                                        espdr_inst_config *inst_config,
                                                        int number_of_ext,
                                                        int fibre_nr) {
    
    char full_filename[FILENAME_LENGTH];
    for (int j = 0; j < number_of_ext; j++) {
        sprintf(full_filename, "%s_%s_%d_%c.fits",
                inst_config->instrument, filename, j, fibre_name[fibre_nr]);
        cpl_image_save(cpl_imagelist_get(iml_to_save, j),
                       full_filename, CPL_TYPE_FLOAT,
                       keywords, CPL_IO_CREATE);
        espdr_msg("%s saved", full_filename);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Check the orderdef QC
 @param         CCD_geom        CCD geometry
 @param         qc_kws          KWs names
 @param         inst_config     instrument config
 @param         fibre_nr        fibre number
 @param         orders_nb       detected orders number
 @param         orders_pos      orders positions
 @param         order_res_stdev stedv of orders residuals
 @param         order_res_min   minimum of orders residuals
 @param         order_res_max   maxinum of orders residuals
 @param         orders_phys_nb  physical orders numbers
 @param[out]    keywords_RE     fits header to hold QC
 @return	CPL_ERROR_NONE iff OK

 TODO: keywords_RE does not need to be passed as double pointer parameter
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_orderdef_QC(espdr_CCD_geometry *CCD_geom,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		int fibre_nr,
		int *orders_nb,
		double **orders_pos,
		double **order_res_stdev,
		double **order_res_min,
		double **order_res_max,
		int **orders_phys_nb,
		cpl_propertylist **keywords_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;
	int i, j, index;
	char *new_keyword = NULL;
	int saturation_QC = 1;
	int orders_nb_all_ext = 0;
	int orders_nb_QC = 1;
	char comment[COMMENT_LENGTH];
	int res_stdev_QC = 1;
	int res_min_QC = 1;
	int res_max_QC = 1;
	int global_QC = 1;


    if (cpl_propertylist_has(*keywords_RE, qc_kws->qc_saturation_check_kw)) {
        saturation_QC = cpl_propertylist_get_int(*keywords_RE, qc_kws->qc_saturation_check_kw);
        if (saturation_QC == 0) {
            global_QC = 0;
        }
    }
    
	/* QC saturation CHECK KWs */

	my_error = espdr_keyword_add_int(qc_kws->qc_saturation_check_kw,
			saturation_QC,
			"Saturation [ADU] QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_WAVE_SATUR_CHECK to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));


	for (i = 0; i < CCD_geom->ext_nb; i++) {
		orders_nb_all_ext += orders_nb[i];
	}

	my_error = espdr_keyword_add_int(qc_kws->qc_order_nb_kw,
			orders_nb_all_ext,
			"Nb of orders in the image",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDER_NB_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));

	int orders_nb_ref = 0;
	for (i = (fibre_nr * CCD_geom->ext_nb);
			i < ((fibre_nr+1) * CCD_geom->ext_nb); i++) {
		orders_nb_ref += inst_config->orders_nb[i];
	}
	espdr_msg("Nominal total number of orders in the image: %d", orders_nb_ref);
	espdr_msg("Real total number of orders in the image: %d", orders_nb_all_ext);

	if (orders_nb_all_ext != orders_nb_ref) {
		orders_nb_QC = 0;
	}

	my_error = espdr_keyword_add_int(qc_kws->qc_orderdef_order_check_kw,
			orders_nb_QC,
			"Orders nb QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDERDEF_ORDER_CHECK_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));
    
    int total_order_nb = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        total_order_nb += inst_config->orders_nb[i];
    }
    int phys_order_nb_flat[total_order_nb];

    my_error = espdr_get_order_phys_numbers(CCD_geom, inst_config,
                                            fibre_nr, phys_order_nb_flat);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_order_phys_numbers failed: %s",
                 cpl_error_get_message_default(my_error));
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < orders_nb[i]; j++) {
            orders_phys_nb[i][j] = phys_order_nb_flat[index];
            index++;
        }
    }
    
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		espdr_msg_debug("orders_nb[%d] = %d", i, orders_nb[i]);
		for (j = 0; j < orders_nb[i]; j++) {
			espdr_msg_debug("orders_pos[%d][%d] = %lf", i, j, orders_pos[i][j]);
			new_keyword =
					espdr_add_ext_nr_index_to_keyword(qc_kws->qc_order_pos_kw_first,
							qc_kws->qc_order_pos_kw_last,
							i, j+1);

			sprintf(comment, "Order %d position [ext %d]", j+1, i);
			my_error = espdr_keyword_add_double(new_keyword,
					orders_pos[i][j],
					comment,
					keywords_RE);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Add keyword QC_ORDER_POS_KW to the propertylist failed: %s",
					cpl_error_get_message_default(my_error));

			cpl_free(new_keyword);


			new_keyword =
					espdr_add_ext_nr_index_to_keyword(qc_kws->qc_order_stdev_kw_first,
							qc_kws->qc_order_stdev_kw_last,
							i, j+1);

			sprintf(comment, "Order %d res stdev [ext %d]", j+1, i);

			if(isnan(order_res_stdev[i][j])) order_res_stdev[i][j]=-9999.9;

			my_error = espdr_keyword_add_double(new_keyword,
					order_res_stdev[i][j],
					comment,
					keywords_RE);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Add keyword QC_ORDER_STDEV_KW to the propertylist failed: %s",
					cpl_error_get_message_default(my_error));

			cpl_free(new_keyword);

			if (order_res_stdev[i][j] > inst_config->res_stdev_limit) {
				res_stdev_QC = 0;
			}

			new_keyword =
					espdr_add_ext_nr_index_to_keyword(qc_kws->qc_order_min_kw_first,
							qc_kws->qc_order_min_kw_last,
							i, j+1);

			if(isnan(order_res_min[i][j])) order_res_min[i][j]=-9999.9;

			sprintf(comment, "Order %d res min [ext %d]", j+1, i);
			my_error = espdr_keyword_add_double(new_keyword,
					order_res_min[i][j],
					comment,
					keywords_RE);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Add keyword QC_ORDER_MIN_KW to the propertylist failed: %s",
					cpl_error_get_message_default(my_error));

			cpl_free(new_keyword);

			// Add res_min_limit to inst_config file
			if (order_res_min[i][j] < inst_config->res_min_limit) {
				res_min_QC = 0;
			}

			new_keyword =
					espdr_add_ext_nr_index_to_keyword(qc_kws->qc_order_max_kw_first,
							qc_kws->qc_order_max_kw_last,
							i, j+1);

			if(isnan(order_res_max[i][j])) order_res_max[i][j]=-9999.9;

			sprintf(comment, "Order %d res max [ext %d]", j+1, i);
			my_error = espdr_keyword_add_double(new_keyword,
					order_res_max[i][j],
					comment,
					keywords_RE);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Add keyword QC_ORDER_MAX_KW to the propertylist failed: %s",
					cpl_error_get_message_default(my_error));

			cpl_free(new_keyword);

			// Add res_max_limit to inst_config file
			if (order_res_max[i][j] > inst_config->res_max_limit) {
				res_max_QC = 0;
			}


			new_keyword =
					espdr_add_ext_nr_index_to_keyword(qc_kws->qc_order_phys_nb_kw_first,
							qc_kws->qc_order_phys_nb_kw_last,
							i, j+1);

			sprintf(comment, "Order %d number [ext %d]", j+1, i);
			my_error = espdr_keyword_add_int(new_keyword,
					orders_phys_nb[i][j],
					comment,
					keywords_RE);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Add keyword QC_ORDER_PHYS_NB_KW to the propertylist failed: %s",
					cpl_error_get_message_default(my_error));

			cpl_free(new_keyword);

		}
	}

	my_error = espdr_keyword_add_int(qc_kws->qc_orderdef_stdev_check_kw,
			res_stdev_QC,
			"Orders residuals stdev QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDERDEF_STDEV_CHECK_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));

	my_error = espdr_keyword_add_int(qc_kws->qc_orderdef_min_check_kw,
			res_min_QC,
			"Orders residuals min QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDERDEF_MIN_CHECK_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));

	my_error = espdr_keyword_add_int(qc_kws->qc_orderdef_max_check_kw,
			res_max_QC,
			"Orders residuals max QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDERDEF_MAX_CHECK_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));

	if (orders_nb_QC == 0) {
		global_QC = 0;
	}
	if (saturation_QC == 0) {
		global_QC = 0;
	}
	if (res_stdev_QC == 0) {
		global_QC = 0;
	}
	if (res_min_QC == 0) {
		global_QC = 0;
	}
	if (res_max_QC == 0) {
		global_QC = 0;
	}

	my_error = espdr_keyword_add_int(qc_kws->qc_orderdef_check_kw,
			global_QC,
			"Orderdef global QC",
			keywords_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Add keyword QC_ORDERDEF_CHECK_KW to the propertylist failed: %s",
			cpl_error_get_message_default(my_error));


	return cpl_error_get_code();
}




