/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 14:00:00 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

/* Utility functions */
#include <espdr_utils.h>
#include <espdr_drift.h>

/* DFS functions */
#include <espdr_dfs.h>
#include <espdr_instrument.h>
#include <espdr_keywords.h>

/* DRL functions */

/* Library */
#include <cpl.h>
#include <assert.h>
#include <gsl/gsl_statistics_double.h>	

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

#define RECIPE_ID "espdr_compu_drift"
#define RECIPE_AUTHOR "Ch. Lovis, A. Segovia"
#define RECIPE_CONTACT "https://support.eso.org"

/*-----------------------------------------------------------------------------
 Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list);

/*----------------------------------------------------------------------------
 Private functions prototypes
 ----------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

static int espdr_compu_drift_create(cpl_plugin *);
static int espdr_compu_drift_exec(cpl_plugin *);
static int espdr_compu_drift_destroy(cpl_plugin *);

/*----------------------------------------------------------------------------
 Static variables
 ----------------------------------------------------------------------------*/

static char espdr_compu_drift_description_short[] = 
"Measures instrumental drift on wavelength calibration spectra.";

static char espdr_compu_drift_description[] =
"This recipe computes the instrumental drift on S2D non-deblazed spectra \
Input Frames : \n\
- A static table indicating the detector geometry (Tag = CCD_GEOM)\n\
- A static table to set instrument config. parameters (Tag = INST_CONFIG)\n\
- A wave map (vacuum) for the fibre, the S2D spectra are provided (Format=PRE, Tag = WAVE_MATRIX_<lamp_lamp_fiber>)\n\
- A DLL map (vacuum) for the fibre, the S2D spectra are provided (Format=PRE, Tag = DLL_MATRIX_<lamp_lamp_fiber>)\n\
- A sequence of at least 2 S2D non-deblazed spectra of a wave calibration source (THAR, FP or LFC) (TAG = S2D_BLAZE_<lamp_lamp_fiber>)\n\
Products : \n\
- the FITS table(s) with drift results per detector, per slice (PRO.CATG = DRIFT_<lamp_lamp_fiber_slice>)";

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Build the list of available plugins, for this module.
 @param    list    the plugin list
 @return   0 if everything is ok, -1 otherwise

 Create the recipe instance and make it available to the application using
 the interface. This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list)
{
	cpl_recipe *recipe = NULL;
	cpl_plugin *plugin = NULL;

	recipe = cpl_calloc(1, sizeof(*recipe));

	if (recipe == NULL) {
		return 1;
	}

	plugin = &recipe->interface;
	if (cpl_plugin_init(plugin,
		CPL_PLUGIN_API,	/* Plugin API */
		ESPDR_BINARY_VERSION,	/* Plugin version */
		CPL_PLUGIN_TYPE_RECIPE,	/* Plugin type */
		RECIPE_ID,	/* Plugin name */
		espdr_compu_drift_description_short,	/* Short help */
		espdr_compu_drift_description,	/* Detailed help */
		RECIPE_AUTHOR,	/* Author name */
		RECIPE_CONTACT,	/* Contact address as in xsh,
			           PACKAGE_BUGREPORT in the tpl */
		espdr_get_license(),	/* Copyright */
		espdr_compu_drift_create,
		espdr_compu_drift_exec,
		espdr_compu_drift_destroy)) {
		cpl_msg_error(cpl_func, "Plugin initialiazation failed");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}

	if (cpl_pluginlist_append(list, plugin)) {
		cpl_msg_error(cpl_func, "Error adding plugin to list");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}


    /*return (cpl_error_get_code() != CPL_ERROR_NONE); as in xsh*/
	return 0;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Setup the recipe options
 @param    plugin  the plugin
 @return   0 if everything is ok

 Create the recipe instance and make it available to the application using
 the interface.

 */
/*--------------------------------------------------------------------------*/

static int espdr_compu_drift_create(cpl_plugin * plugin)
{
    cpl_recipe    *recipe;
    cpl_parameter *param;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    /* Fill the parameters list */

/*
 * Default values for all parameters are set to INV_DOUBLE = -9999.0, because no
 * other default has been determined yet.
 */

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.method_thar",
        CPL_TYPE_STRING, "drift measurement method_thar",
        "espdr.espdr_compu_drift", "global");
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "method_thar");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.method_fp",
        CPL_TYPE_STRING, "drift measurement method_fp",
        "espdr.espdr_compu_drift", "global");
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "method_fp");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.method_lfc",
        CPL_TYPE_STRING, "drift measurement method_lfc",
        "espdr.espdr_compu_drift", "global");
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "method_lfc");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.space",
        CPL_TYPE_STRING, "drift measurement space",
        "espdr.espdr_compu_drift", "pixel");
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "space");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    //param = cpl_parameter_new_value("espdr.espdr_compu_drift.window_size",
        //CPL_TYPE_DOUBLE, "window size for drift measurements of individual lines",
        //"espdr.espdr_compu_drift", 1.);
    //cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "window_size");
    //cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    //cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.ksig",
        CPL_TYPE_DOUBLE, "Calcium index alpha",
        "espdr.espdr_compu_drift", -1.0);
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "ksig");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.max_flux_threshold",
        CPL_TYPE_DOUBLE, "Maximun flux threshold",
        "espdr.espdr_compu_drift", 0.0);
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "max_flux_threshold");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    param = cpl_parameter_new_value("espdr.espdr_compu_drift.min_flux_threshold",
        CPL_TYPE_DOUBLE, "Minimun flux threshold",
        "espdr.espdr_compu_drift", 0.0);
    cpl_parameter_set_alias(param, CPL_PARAMETER_MODE_CLI, "min_flux_threshold");
    cpl_parameter_disable(param, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, param);

    return 0;
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Execute the plugin instance given by the interface
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_compu_drift_exec(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	int recipe_status;
	cpl_errorstate initial_errorstate = cpl_errorstate_get();

	/* Return immediately if an error code is already set */
	if (cpl_error_get_code() != CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
		cpl_func, __LINE__, cpl_error_get_where());
		return (int)cpl_error_get_code();
	}

	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

	/* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE,
	CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

	/* Get the recipe */
	recipe = (cpl_recipe *) plugin;

	/* Verify parameter and frame lists */
	espdr_ensure(recipe->parameters == NULL,
	CPL_ERROR_NULL_INPUT,
	"Recipe invoked with NULL parameter list");

	espdr_ensure(recipe->frames == NULL, CPL_ERROR_NULL_INPUT,
	"Recipe invoked with NULL frame set");

    /* Set the time display for logs */
    cpl_msg_set_time_on();
    
	/* Invoke the recipe */
	recipe_status = espdr_compu_drift(recipe->parameters, recipe->frames,
			RECIPE_ID);

	/* Ensure DFS-compliance of the products */
	if (cpl_dfs_update_product_header(recipe->frames)) {
		if (!recipe_status) recipe_status = (int)cpl_error_get_code();
	}

	if (!cpl_errorstate_is_equal(initial_errorstate)) {
		/* Dump the error history since recipe execution start.
		At this point the recipe cannot recover from the error */
		cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
	}

	/*printf("recipe status: %d\n", recipe_status);*/
    
    /* Cancel the time display for logs */
    cpl_msg_set_time_off();
    
	return recipe_status;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Destroy what has been created by the 'create' function
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_compu_drift_destroy(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;

	cpl_error_reset();

	/* Verify the plugin existance */
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

	/* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE,
	CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

	/* Get the recipe */
	recipe = (cpl_recipe *) plugin;

	cpl_parameterlist_delete(recipe->parameters);

	return (cpl_error_get_code());
}


