/*
 * espdr_bias.c
 *
 *  Created on: Mar 29, 2012
 *      Author: Danuta Sosnowska
 */

/*
 * $Author: dsosnows $
 * $Date: 2013-10-23 14:48:13 $
 * $Revision: 1.16 $
 * $Name: not supported by cvs2svn $
 */


/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/
#include <espdr_bias.h>
#include <espdr_hdrl_func.h>
/*----------------------------------------------------------------------------
                              Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Overscan coputation
 @param
 @return   error code
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_single_bias_compute_oscan(cpl_imagelist* master_bias_imagelist,
                                                      cpl_imagelist* MB_res_iml,
                                                      cpl_imagelist* overscan_imagelist,
                                                      espdr_CCD_geometry* CCD_geom,
                                                      espdr_inst_config* inst_config,
                                                      espdr_qc_keywords *qc_kws,
                                                      cpl_propertylist* keywords,
                                                      const int treated_frames_nb,
                                                      const int total_output_nb,
                                                      const int use_hdrl,
                                                      double* RON_vector_OV) {

	cpl_error_code my_error;
	espdr_msg("Preparing overscan computation");

    my_error = espdr_overscan_prepare(master_bias_imagelist, CCD_geom,
                                      qc_kws, inst_config,
                                      treated_frames_nb,
                                      &overscan_imagelist, &keywords,
                                      RON_vector_OV,use_hdrl);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Overscan computation failed: %s",
                 cpl_error_get_message_default(my_error));

	int i, j, k;
	for (i = 0; i < total_output_nb; i++) {
		espdr_msg_debug("OVSC RON[%d] = %lf", i, RON_vector_OV[i]);
	}

	cpl_image *MB_res_img = NULL;

	/* Remove the overscan from master bias */
	espdr_msg("Removing overscan form the master bias");
	int index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                my_error = espdr_remove_overscan(cpl_imagelist_get
                                                 (master_bias_imagelist, index),
                                                 i, j, k, CCD_geom,
                                                 cpl_imagelist_get
                                                 (overscan_imagelist, index),
                                                 &MB_res_img);

                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Removing OVSC from MB failed: %s",
                             cpl_error_get_message_default(my_error));

                cpl_imagelist_set(MB_res_iml,
                                  cpl_image_duplicate(MB_res_img),
                                  index);

				cpl_image_delete(MB_res_img);

				index++;
			}
		}
	}

	return (cpl_error_get_code());
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Save mbias products
 @param
 @return   error code
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_single_bias_save_products(espdr_inst_config* inst_config,
                                                      espdr_CCD_geometry* CCD_geom,
													  const char* recipe_id,
                                                      cpl_frameset* frameset,
                                                      cpl_parameterlist* parameters,
                                                      cpl_frameset* used_frames,
                                                      cpl_propertylist* keywords,
                                                      cpl_propertylist** keywords_ext,
                                                      cpl_imagelist* final_master_bias,
                                                      cpl_imagelist* final_MB_res) {

	cpl_error_code my_error;

	/* Save the calculated master bias with the DRS keywords */
	char* filename = (char*) cpl_malloc(FILENAME_LENGTH * sizeof(char));
    strcpy(filename, inst_config->single_bias_filename);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_SBIAS);

	my_error = espdr_dfs_image_save(frameset, parameters, used_frames, recipe_id,
                                    keywords, keywords_ext, filename,
                                    final_master_bias, CPL_TYPE_FLOAT, CCD_geom);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

	/* Save the master bias with removed overscan with the DRS keywords */
    strcpy(filename, inst_config->single_bias_res_filename);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_SBIAS_RES);

	my_error = espdr_dfs_image_save(frameset, parameters, used_frames, recipe_id,
                                    keywords, keywords_ext, filename,
                                    final_MB_res, CPL_TYPE_FLOAT, CCD_geom);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

	cpl_free((void *)filename);

	return (cpl_error_get_code());
}

/**@}*/
/*---------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    parameters     the parameters list
  @param    frameset   the frames list

  In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/
int espdr_single_bias(cpl_parameterlist *parameters,
                             cpl_frameset *frameset,
							 const char* recipe_id) {

    const int rec_ntags = 2;
    const char* rec_tags[2] = {ESPDR_BIAS_RAW, ESPDR_CCD_GEOM};
    int is_required[2] = {1, 1};

	cpl_error_code my_error = CPL_ERROR_NONE;

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting single bias");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
            cpl_error_get_code(), "Wrong input tag!");
    /* Identify the RAW and CALIB frames in the input frameset */
	espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw bias image, tagged as BIAS\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG");

    cpl_frame* CCD_geom_frame = espdr_frame_find(frameset,
                                                 ESPDR_CCD_GEOM);
    espdr_CCD_geometry *CCD_geom = espdr_CCD_geom_init(parameters,
                                                       CCD_geom_frame);
    cpl_frame* inst_config_frame = espdr_get_inst_config(frameset);
    espdr_inst_config *inst_config = espdr_inst_config_init(parameters,
                                                            CCD_geom->ext_nb,
                                                            inst_config_frame);
    espdr_OVSC_param *OVSC_param  = espdr_parameters_OVSC_init(recipe_id, parameters);

	/* Extract the raw bias frames and put them into bias_frames frameset */
	cpl_frameset *bias_frames = cpl_frameset_new();
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_BIAS_RAW,
                                          bias_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "BIAS frames extraction failed.");
    cpl_frame *raw_bias_frame = cpl_frameset_get_position(bias_frames, 0);

    /* For one input raw bias frame only the overscan will be computed */
    if (cpl_frameset_get_size(bias_frames) < 1) {
        espdr_msg_warning("No raw BIAS frame, exiting.");
        cpl_frameset_delete(bias_frames);
        espdr_parameters_OVSC_delete(OVSC_param);
        espdr_parameters_CCD_geometry_delete(CCD_geom);
        espdr_parameters_inst_config_delete(inst_config);
        return(CPL_ERROR_ILLEGAL_INPUT);
    }

    cpl_frameset *used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(raw_bias_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = (espdr_qc_keywords *)
                                        cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    int total_output_nb = CCD_geom->total_output_nb;
    double res_mean_limit = inst_config->bias_residuals_mean_limit;
    double res_stdev_tolerance = inst_config->bias_residuals_stdev_limit;
	espdr_msg("\tres_mean_limit = %.2lf", res_mean_limit);
	espdr_msg("\tres_stdev_tolerance = %.2lf", res_stdev_tolerance);

	/* keywords contains all the keywords read from
	 the first input frame header */
	cpl_propertylist *keywords = NULL;

	/* keywords_ext contains keywords of all the extension headers
	 of the first input frame */
	cpl_propertylist **keywords_ext =
		(cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb*sizeof(cpl_propertylist*));

	const char *input_filename = cpl_frame_get_filename(raw_bias_frame);

    /* Load keywords from the first image (primary header and extensions headers) */
    if (strcmp(inst_config->instrument, "HARPN") == 0) {
        keywords = cpl_propertylist_load_regexp(input_filename, 0, "TNG DET WIN1 UIT1", 1);
        if (cpl_propertylist_has(keywords, "TNG DET READ REG")) {
            my_error = cpl_propertylist_insert_after_double(keywords, "TNG DET READ REG",
                                                            "TNG DET WIN1 UIT1", 0.0);
        } else {
            my_error = cpl_propertylist_append_double(keywords, "TNG DET WIN1 UIT1", 0.0);
        }
    } else {
        keywords = cpl_propertylist_load(input_filename, 0);
    }
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        keywords_ext[i] = cpl_propertylist_load(input_filename, i+1);
    }

    int use_hdrl = 0;
    espdr_msg("Preparing single bias computation");
    cpl_imagelist *master_bias_imagelist = cpl_imagelist_new();
    my_error = espdr_single_bias_prepare(bias_frames, CCD_geom,
                                         qc_kws, inst_config, 1,
                                         &master_bias_imagelist,
                                         &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Raw bias frame reading failed: %s",
                 cpl_error_get_message_default(my_error));

	cpl_frameset_delete(bias_frames);

	/* Calculate the overscan of the master bias images and get its RON
	 and cosmics rejected */

	double *RON_vector_OV = (double *)cpl_calloc(total_output_nb,
                                                 sizeof(double));
	cpl_imagelist *overscan_imagelist = cpl_imagelist_new();
	cpl_imagelist *MB_res_iml = cpl_imagelist_new();

    my_error = espdr_single_bias_compute_oscan(master_bias_imagelist,
                                               MB_res_iml,
                                               overscan_imagelist,
                                               CCD_geom, inst_config,
                                               qc_kws, keywords,
                                               1,
                                               total_output_nb,
                                               use_hdrl,
                                               RON_vector_OV);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_mbias_compute_correct_oscan failed: %s",
                 cpl_error_get_message_default(my_error));

    my_error = espdr_bias_QC(master_bias_imagelist, overscan_imagelist,
                             CCD_geom, qc_kws, inst_config, RON_vector_OV,
                             res_mean_limit, res_stdev_tolerance,
                             &keywords);

    cpl_imagelist_delete(overscan_imagelist);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "bias QC failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Add QC KWs expressed in e- */
    my_error = espdr_add_QC_e(keywords_ext, CCD_geom, inst_config, qc_kws,
                              &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_add_QC_e failed: %s",
                 cpl_error_get_message_default(my_error));
    cpl_free(qc_kws);

	/* Merge all master bias output images into one extension,
	 all the merged extensions are in the final_master_bias */

	cpl_imagelist *final_master_bias = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(master_bias_imagelist,
                                       CCD_geom,
                                       CPL_TYPE_DOUBLE,
                                       &final_master_bias);
	cpl_imagelist_delete(master_bias_imagelist);

    cpl_imagelist *final_MB_res = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(MB_res_iml,
                                       CCD_geom,
                                       CPL_TYPE_DOUBLE,
                                       &final_MB_res);
    cpl_imagelist_delete(MB_res_iml);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Save the calculated master bias with the DRS keywords */
    my_error = espdr_single_bias_save_products(inst_config, CCD_geom, recipe_id,
                                               frameset, parameters,
                                               used_frames,
                                               keywords, keywords_ext,
                                               final_master_bias,
                                               final_MB_res);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_mbias_save_products failed: %s ",
                 cpl_error_get_message_default(my_error));


    cpl_imagelist_delete(final_master_bias);
    cpl_imagelist_delete(final_MB_res);

    cpl_frameset_delete(used_frames);

	/* Destruction of allocated data structures */
	cpl_propertylist_delete(keywords);
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);

    //cpl_free(inst_config);
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);
    cpl_free(RON_vector_OV);


    return (0);
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Overscan coputation
 @param
 @return   error code
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_mbias_compute_oscan(cpl_imagelist* master_bias_imagelist,
                                                cpl_imagelist* MB_res_iml,
                                                cpl_imagelist* overscan_imagelist,
                                                espdr_CCD_geometry* CCD_geom,
                                                espdr_inst_config* inst_config,
                                                espdr_qc_keywords *qc_kws,
                                                cpl_propertylist* keywords,
                                                const int treated_frames_nb,
                                                const int total_output_nb,
                                                const int use_hdrl,
                                                double* RON_vector_OV) {

	cpl_error_code my_error;
	espdr_msg("Preparing overscan computation");

    my_error = espdr_overscan_prepare(master_bias_imagelist, CCD_geom,
                                      qc_kws, inst_config,
                                      treated_frames_nb,
                                      &overscan_imagelist, &keywords,
                                      RON_vector_OV,use_hdrl);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Overscan computation failed: %s",
                 cpl_error_get_message_default(my_error));

	int i, j, k;
	for (i = 0; i < total_output_nb; i++) {
		espdr_msg_debug("OVSC RON[%d] = %lf", i, RON_vector_OV[i]);
	}

	cpl_image *MB_res_img = NULL;

	/* Remove the overscan from master bias */
	espdr_msg("Removing overscan form the master bias");
	int index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                my_error = espdr_remove_overscan(cpl_imagelist_get
                                                 (master_bias_imagelist, index),
                                                 i, j, k, CCD_geom,
                                                 cpl_imagelist_get
                                                 (overscan_imagelist, index),
                                                 &MB_res_img);

                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Removing OVSC from MB failed: %s",
                             cpl_error_get_message_default(my_error));

                cpl_imagelist_set(MB_res_iml,
                                  cpl_image_duplicate(MB_res_img),
                                  index);

				cpl_image_delete(MB_res_img);

				index++;
			}
		}
	}

	return (cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Save mbias products
 @param
 @return   error code
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_mbias_save_products(espdr_inst_config* inst_config,
                                                espdr_CCD_geometry* CCD_geom,
												const char* recipe_id,
                                                cpl_frameset* frameset,
                                                cpl_parameterlist* parameters,
                                                cpl_frameset* used_frames,
                                                cpl_propertylist* keywords,
                                                cpl_propertylist** keywords_ext,
                                                cpl_imagelist* final_master_bias,
                                                cpl_imagelist* final_MB_res) {

	cpl_error_code my_error;

	/* Save the calculated master bias with the DRS keywords */
	char* filename = (char*) cpl_malloc(FILENAME_LENGTH * sizeof(char));
    strcpy(filename, inst_config->master_bias_filename);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_MBIAS);

	my_error = espdr_dfs_image_save(frameset, parameters, used_frames, recipe_id,
                                    keywords, keywords_ext, filename,
                                    final_master_bias, CPL_TYPE_FLOAT, CCD_geom);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

	/* Save the master bias with removed overscan with the DRS keywords */
    strcpy(filename, inst_config->master_bias_res_filename);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_MBIAS_RES);

	my_error = espdr_dfs_image_save(frameset, parameters, used_frames, recipe_id,
                                    keywords, keywords_ext, filename,
                                    final_MB_res, CPL_TYPE_FLOAT,
                                    CCD_geom);
	cpl_free((void *)filename);

	return (cpl_error_get_code());
}




/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     kext        property list of the extensions (?)
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist* keywords,
                                         const int ext_nb,
                                         cpl_propertylist*** kext) {

    const char* ksuff[2] = {"", ""};
    cpl_propertylist_append_int(keywords,"ESO QC PROC VERSION", 2);
    espdr_add_qc_key_stat_ext(keywords, ext_nb, "BIAS MEAN", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);

    espdr_add_qc_key_stat_ext(keywords,ext_nb, "BIAS RON", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);

    espdr_add_qc_key_stat_ext(keywords,ext_nb, "OVSC RON", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);

    espdr_add_qc_key_stat_ext(keywords,ext_nb, "BIAS STRUCTX", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "BIAS STRUCTY", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);

    espdr_add_qc_key_stat_ext(keywords,ext_nb, "RES MEAN", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);
    espdr_add_qc_key_stat_ext(keywords,ext_nb, "RES STDEV", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);
    espdr_copy_qc_ext_key_to_ext(keywords, kext, ext_nb);
    espdr_check_error_code("espdr_add_extra_qc");
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    parameters     the parameters list
  @param    frameset   the frames list
  @param    recipe_id  the recipe name
  In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/
int espdr_mbias(cpl_parameterlist *parameters,
                       cpl_frameset *frameset,
					   const char* recipe_id) {

    const int rec_ntags = 2;
    const char* rec_tags[2] = {ESPDR_BIAS_RAW, ESPDR_CCD_GEOM};
    int is_required[2] = {1, 1};

	cpl_error_code my_error = CPL_ERROR_NONE;

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting master bias");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
            cpl_error_get_code(), "Wrong input tag!");
    /* Identify the RAW and CALIB frames in the input frameset */
	espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw bias images, tagged as BIAS\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG");

    cpl_frame* CCD_geom_frame = espdr_frame_find(frameset,
                                                 ESPDR_CCD_GEOM);
    espdr_CCD_geometry *CCD_geom = espdr_CCD_geom_init(parameters,
                                                       CCD_geom_frame);
    cpl_frame* inst_config_frame = espdr_get_inst_config(frameset);
    espdr_inst_config *inst_config = espdr_inst_config_init(parameters,
                                                            CCD_geom->ext_nb,
                                                            inst_config_frame);
    espdr_OVSC_param *OVSC_param  = espdr_parameters_OVSC_init(recipe_id, parameters);
    espdr_BIAS_param *BIAS_param  = espdr_BIAS_param_init(recipe_id, parameters);

	/* Extract the raw bias frames and put them into bias_frames frameset */
	cpl_frameset *bias_frames = cpl_frameset_new();
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_BIAS_RAW,
                                          bias_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "BIAS frames extraction failed.");

	int bias_frameset_size = cpl_frameset_get_size(bias_frames);
	espdr_msg("The input frameset contains %d BIAS frames",
              bias_frameset_size);

    if (bias_frameset_size < inst_config->raw_bias_limit_nb) {
        espdr_msg_warning("The number of input raw BIAS frames is not enough (%d), should be minimum %d, exiting.",
                          bias_frameset_size, inst_config->raw_bias_limit_nb);
        cpl_frameset_delete(bias_frames);
        espdr_parameters_OVSC_delete(OVSC_param);
        espdr_parameters_BIAS_delete(BIAS_param);
        espdr_parameters_CCD_geometry_delete(CCD_geom);
        espdr_parameters_inst_config_delete(inst_config);
        return(CPL_ERROR_ILLEGAL_INPUT);
    }

	cpl_frameset *used_frames = cpl_frameset_duplicate(bias_frames);

    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = (espdr_qc_keywords *)
                                        cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    int total_output_nb = CCD_geom->total_output_nb;
    double res_mean_limit = inst_config->bias_residuals_mean_limit;
    double res_stdev_tolerance = inst_config->bias_residuals_stdev_limit;
	espdr_msg("\tres_mean_limit = %.2lf", res_mean_limit);
	espdr_msg("\tres_stdev_tolerance = %.2lf", res_stdev_tolerance);

    int treated_frames_nb = 0;
    treated_frames_nb = (bias_frameset_size < inst_config->raw_bias_limit_nb) ? 1 : bias_frameset_size;
    espdr_msg("Treated number of frames is %d", treated_frames_nb);

	/* keywords contains all the keywords read from
	 the first input frame header */
	cpl_propertylist *keywords = NULL;

	/* keywords_ext contains keywords of all the extension headers
	 of the first input frame */
	cpl_propertylist **keywords_ext =
		(cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb*sizeof(cpl_propertylist*));

	const char *input_filename =
                    cpl_frame_get_filename(cpl_frameset_get_position(bias_frames,0));

    /* Load keywords from the first image (primary header and extensions headers) */
    keywords = cpl_propertylist_load(input_filename, 0);
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        keywords_ext[i] = cpl_propertylist_load(input_filename, i+1);
    }

    int use_hdrl = 0;
    espdr_msg("Preparing master bias computation");
    cpl_imagelist *master_bias_imagelist = cpl_imagelist_new();
    my_error = espdr_master_bias_prepare(bias_frames,
                                         CCD_geom, qc_kws, inst_config,
                                         &master_bias_imagelist,
                                         &keywords,
                                         use_hdrl);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Master Bias computation failed: %s",
                 cpl_error_get_message_default(my_error));

	cpl_frameset_delete(bias_frames);

	/* Calculate the overscan of the master bias images and get its RON
	 and cosmics rejected */

	double *RON_vector_OV = (double *)cpl_calloc(total_output_nb,
                                                 sizeof(double));
	cpl_imagelist *overscan_imagelist = cpl_imagelist_new();
	cpl_imagelist *MB_res_iml = cpl_imagelist_new();

    my_error = espdr_mbias_compute_oscan(master_bias_imagelist,
                                         MB_res_iml,
                                         overscan_imagelist,
                                         CCD_geom, inst_config,
                                         qc_kws, keywords,
                                         treated_frames_nb,
                                         total_output_nb,
                                         use_hdrl,
                                         RON_vector_OV);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_mbias_compute_correct_oscan failed: %s",
                 cpl_error_get_message_default(my_error));

    my_error = espdr_bias_QC(master_bias_imagelist, overscan_imagelist,
                             CCD_geom, qc_kws, inst_config, RON_vector_OV,
                             res_mean_limit, res_stdev_tolerance,
                             &keywords);

    cpl_imagelist_delete(overscan_imagelist);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "bias QC failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Add QC KWs expressed in e- */
    my_error = espdr_add_QC_e(keywords_ext, CCD_geom, inst_config, qc_kws,
                              &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_add_QC_e failed: %s",
                 cpl_error_get_message_default(my_error));
    cpl_free(qc_kws);

	/* Merge all master bias output images into one extension,
	 all the merged extensions are in the final_master_bias */

	cpl_imagelist *final_master_bias = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(master_bias_imagelist,
                                       CCD_geom,
                                       CPL_TYPE_DOUBLE,
                                       &final_master_bias);
	cpl_imagelist_delete(master_bias_imagelist);

    cpl_imagelist *final_MB_res = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(MB_res_iml,
                                       CCD_geom,
                                       CPL_TYPE_DOUBLE,
                                       &final_MB_res);
    cpl_imagelist_delete(MB_res_iml);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Save the calculated master bias with the DRS keywords */
    if (strcmp(inst_config->instrument, "HARPN") != 0 ) {
        my_error = espdr_add_extra_qc(keywords, CCD_geom->ext_nb,
                                      &keywords_ext);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
    }

    my_error = espdr_mbias_save_products(inst_config, CCD_geom, recipe_id,
                                         frameset, parameters,
                                         used_frames,
                                         keywords, keywords_ext,
                                         final_master_bias,
                                         final_MB_res);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_mbias_save_products failed: %s ",
                 cpl_error_get_message_default(my_error));


    cpl_imagelist_delete(final_master_bias);
    cpl_imagelist_delete(final_MB_res);

    cpl_frameset_delete(used_frames);

	/* Destruction of allocated data structures */
	cpl_propertylist_delete(keywords);
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);

    //cpl_free(inst_config);
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_BIAS_delete(BIAS_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);
    cpl_free(RON_vector_OV);


    return (0);
}




/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for BIAS input parameters
 @param    recipe_id	recipe identifier
 @param    param_list	list of input parameters
 @param    p			input BIAS parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_BIAS_create(const char* recipe_id,
                                            cpl_parameterlist *param_list,
                                            espdr_BIAS_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL,
                 CPL_ERROR_NULL_INPUT, "The parameters list is NULL");
	
	/* Fill the parameter list */
    //p = NULL;
	cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    double ksigma_min = KSIGMA_MIN;
    double ksigma_max = KSIGMA_MAX;

	/* check parameters */
	espdr_ensure(param_list == NULL,
                 CPL_ERROR_NULL_INPUT, "The parameters list NULL");

	/* Fill the parameter list */
	error_got = espdr_parameters_new_string(recipe_id, param_list,
			"bias_sig_clip_method",
			p->bias_sigma_clipping_method,
			"method for sigma clipping in MBIAS, can be: mean or median");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter bias_sigma_clipping_method to the list");

	sprintf(comment,
			"ksigma for sigma clipping in MBIAS, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);
	error_got = espdr_parameters_new_range_double(recipe_id, param_list,
			"bias_ksigma", p->bias_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter bias_ksigma to the list");

	return(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for BIAS input parameters
 @param    p    input BIAS parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_BIAS_delete(espdr_BIAS_param* p) {
    
    cpl_free((void *)p->bias_sigma_clipping_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the mbias recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     BIAS_param  BIAS parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_BIAS_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_BIAS_param *BIAS_param) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
    /* check parameters */
    espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
                 "Parameters list is NULL");
    
    /* Fill the structure */
    BIAS_param->bias_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                         param_list,
                                                                         "bias_sig_clip_method");
    
    BIAS_param->bias_ksigma = espdr_parameters_get_double(recipe_id,
                                                          param_list,
                                                          "bias_ksigma");
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the BIAS parameters
 @param     BIAS_param  BIAS parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_BIAS_print(espdr_BIAS_param *BIAS_param) {
    
    espdr_msg("\tBIAS parameters: ");
    espdr_msg("\t\tBIAS stacking sigma clipping method = %s",
              BIAS_param->bias_sigma_clipping_method);
    espdr_msg("\t\tBIAS stacking ksigma = %.2f",
              BIAS_param->bias_ksigma);
    
    return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init BIAS_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_BIAS_param * espdr_BIAS_param_init(const char *recipe_id,
                                         cpl_parameterlist *parameters) {
    
    espdr_BIAS_param *BIAS_param =
                    (espdr_BIAS_param *)cpl_malloc(sizeof(espdr_BIAS_param));
    
    /* Read the cal_contam parameters */
    espdr_parameters_BIAS_get(recipe_id, parameters, BIAS_param);
    espdr_parameters_BIAS_print(BIAS_param);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return BIAS_param;
    }
}

/*----------------------------------------------------------------------------*/
/* AMO */
/**
 @brief Prepare the computation the master bias on a set of raw bias frames
 @param         bias_set        input frameset
 @param         CCD_geom        CCD geometry structure
 @param         BIAS_param      bias parameter structure
 @param         stats_param     stat parameter structure
 @param         qc_kws          KWs names
 @param         inst_config     instrument config
 @param[out]    master_bias_imagelist_RE   output master bias image list
 @param[out]    keywords_RE                output relevant FITS keywords
 @return    CPL_ERROR_NONE iff OK
 
 TODO: this function is too complex. One may factor it out
 1) master bias computation
 2) generation of FITS keywords for
 a) RON,
 b) RON_CHECK,
 c) BIAS_MEAN
 3) Not clear why need to pass double pointer for keywords_RE
 4) raw image list here is the full-size image list.
 This why one need to have the CCD_param.
 More efficient in term of RAM, easy of code, (and probably keeping same
 performance) would be to pass the relevant raw_bias image list already
 chopped to the needed area. Then one could easily unit test this function.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_master_bias_prepare(const cpl_frameset *bias_set,
                                         const espdr_CCD_geometry *CCD_geom,
                                         espdr_qc_keywords *qc_kws,
                                         espdr_inst_config *inst_config,
                                         cpl_imagelist **master_bias_imagelist_RE,
                                         cpl_propertylist **keywords_RE,
                                         const int use_hdrl) {
    
    /* indexes for loops*/
    int i, j, k, index;
    /* error code */
    cpl_error_code my_error = CPL_ERROR_NONE;
    int QC_RON = 1;
    int QC_mean = 1;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int total_output_nb = CCD_geom->total_output_nb;
    
    espdr_ensure(bias_set == NULL,
                 CPL_ERROR_NULL_INPUT,"Input frameset is NULL");
    
    double *RON_vector_MB = (double *)cpl_calloc(total_output_nb, sizeof(double));
    double *master_bias_mean = (double *)cpl_calloc(total_output_nb, sizeof(double));
    int total_cosmics_nb = 0;
    
    my_error = espdr_master_bias(bias_set, CCD_geom, inst_config,
                                 master_bias_imagelist_RE, master_bias_mean,
                                 RON_vector_MB, &total_cosmics_nb,use_hdrl);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error in master bias computation");
    
    /* Add IMAGE_COSMICS keyword */
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_outliers_kw,
                                     total_cosmics_nb,
                                     "Total cosmics number in the image",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_outliers_kw,
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("IMAGE COSMICS NB is: %d",
              cpl_propertylist_get_int(*keywords_RE,
                                       qc_kws->qc_bias_outliers_kw));
    
    /* Add MBIAS_RON keyword, one for each output and each extension */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
        espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_kw_part,
                                              inst_config->prefix,
                                              i, j, k);
                sprintf(comment, "RON[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RON_vector_MB[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                index++;
            }
        }
    }
    
    /* Add QC_BIAS_RON_CHECK keyword, QC_BIAS_RON_CHECK is 1 (OK) if none
     of the RONs exceeds the maximal value (only if MasterBias created) */
    QC_RON = 1;
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg_debug("RON MB[%d] = %lf", i, RON_vector_MB[i]);
        if (RON_vector_MB[i] > inst_config->bias_ron_limit) {
            QC_RON = 0;
        }
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_ron_check_kw, QC_RON,
                                     "QC on RON",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_ron_check_kw,
                 cpl_error_get_message_default(my_error));
    
    /* Add IMAGE_MEAN keyword */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_mean_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment, "Mean[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    master_bias_mean[index],
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                index++;
            }
        }
    }
    
    /* Add QC_BIAS_MEAN_CHECK keyword, QC_BIAS_MEAN_CHECK is 1 (OK) if none
     of the means exceeds the maximal value (only if MasterBias created) */
    QC_mean = 1;
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("MASTER BIAS mean[%d] = %lf", i, master_bias_mean[i]);
        if (master_bias_mean[i] > inst_config->bias_mean_limit) {
            QC_mean = 0;
        }
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_mean_check_kw, QC_mean,
                                     "QC on mean",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_mean_check_kw,
                 cpl_error_get_message_default(my_error));
    
    cpl_free(RON_vector_MB);
    cpl_free(master_bias_mean);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/* AMO */
/**
 @brief         Compute the master bias on a set of raw bias frames
 @param         bias_set    the input frameset of bias raw frames
 @param         CCD_geom    CCD geometry parameters structure
 @param         BIAS_param  BIAS parameters structure
 @param         stats_param stat parameter structure
 @param[out]    master_bias_imagelist_RE    computed master bias (returned)
 @param[out]    master_bias_mean    computed mean (returned)
 @param[out]    RON_RE              computed RON (returned)
 @param[out]    total_cosmics_RE    computed cosmics number (returned)
 @param         use_hdrl            HDRL use flag
 @return        CPL_ERROR_NONE iff OK
 
 TODO: Is it really needed/efficient to pass as input the input image with
 full size one could for example use cpl_image_load_window to load the proper
 detector chip from each input frame and then process just that. This would
 drastically reduce RAM consumption.
 Alternative solution is to define a memory buffer (more flexible as one may
 change the buffer size at will, but require to define and handle that properly)
 
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_master_bias(const cpl_frameset *bias_set,
                                 const espdr_CCD_geometry *CCD_geom,
                                 const espdr_inst_config *inst_config,
                                 cpl_imagelist **master_bias_imagelist_RE,
                                 double *master_bias_mean,
                                 double *RON_RE,
                                 int *total_cosmics_RE,
                                 const int use_hdrl) {
    
    /* Loop  indices */
    int i, j, k, l, index;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    cpl_image *curr_image = NULL;
    cpl_imagelist *images_stack = NULL;
    cpl_image *master_bias_image = NULL;
    int imagelist_place = 0;
    int cosmics_nb = 0;
    int total_output_nb = CCD_geom->total_output_nb;
    int nx = 0, ny = 0;
    int ix,iy;
    int llx,lly,urx,ury;
    cpl_frame* frame=NULL;
    const char* fname=NULL;
    int bias_max_iter = cpl_frameset_get_size(bias_set) - 1;
    
    espdr_ensure(bias_set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frameset is NULL");
    
    *total_cosmics_RE = 0;
    for (j = 0; j < CCD_geom->ext_nb; j++) {
        
        ix = 0;
        for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {
            
            llx = ix+1;
            urx = ix+CCD_geom->exts[j].outputs[k][0].raw_nx;
            iy=0;
            for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {
                index = 0;
                images_stack = cpl_imagelist_new();
                //lly = iy+1;
                lly = iy+1+CCD_geom->exts[j].outputs[k][l].ppscan_ny;
                ury = iy+CCD_geom->exts[j].outputs[k][l].raw_ny
                        -CCD_geom->exts[j].outputs[k][l].poscan_ny;

                int nset = cpl_frameset_get_size(bias_set);
                cpl_frameset_iterator* iter =
                                        cpl_frameset_iterator_new(bias_set);
                frame = cpl_frameset_iterator_get(iter);
                
                espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
                          llx, lly, urx, ury);
                
                for (int iter_frame = 0; iter_frame < nset; iter_frame++) {
                    
                    fname = cpl_frame_get_filename(frame);
                    curr_image = cpl_image_load_window(fname,
                                                       CPL_TYPE_DOUBLE, 0,
                                                       j+1,llx, lly, urx, ury);
                    
                    espdr_ensure(curr_image == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                                 "Raw bias image got from the list is NULL");
                    
                    /* Caution, the imagelist can only contain the images
                     of the same size */
                    
                    my_error = cpl_imagelist_set(images_stack,
                                                 cpl_image_duplicate(curr_image),
                                                 index);
                    cpl_image_delete(curr_image);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_imagelist_set failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    index++;
                    /* images_stack imagelist contains the same CCD part
                     of all input BIAS frames */
                    cpl_frameset_iterator_advance(iter, 1);
                    frame = cpl_frameset_iterator_get(iter);
                }
                cpl_frameset_iterator_delete(iter);
                /* Apply the master BIAS on these images */
                /* Calculte the master bias with its RON and number
                 of cosmics rejected */
                nx = CCD_geom->exts[j].outputs[k][l].raw_nx;
                ny = CCD_geom->exts[j].outputs[k][l].raw_ny
                        -CCD_geom->exts[j].outputs[k][l].ppscan_ny
                        -CCD_geom->exts[j].outputs[k][l].poscan_ny;
                espdr_msg("Computing master bias for ext %d, output [%d, %d]",
                          j, k, l);

                if(use_hdrl) {
                    // HDRL based computation
                    my_error = espdr_hdrl_master_port(images_stack,
                                    inst_config->bias_ksigma,
                                    inst_config->bias_ksigma,
                                    bias_max_iter,
                                    &master_bias_image,
                                    &RON_RE[imagelist_place],
                                    &cosmics_nb);
                } else {
                    /* Not activated, else results would change
                	if (strcmp(inst_config->bias_sigma_clipping_method, "median") == 0) {

                		my_error = espdr_master_frame_method_median(images_stack,
                				CCD_geom, inst_config,
								nx, ny, j, k, l,
								&master_bias_image,
								&RON_RE[imagelist_place],
								&master_bias_mean[imagelist_place],
								&cosmics_nb);

                	} else {
                		my_error = espdr_master_frame_method_mean(images_stack,
                				CCD_geom, inst_config,
								nx, ny, j, k, l,
								&master_bias_image,
								&RON_RE[imagelist_place],
								&master_bias_mean[imagelist_place],
								&cosmics_nb);

                	}
                    */

                    my_error = espdr_master_bias_one_output(images_stack,
                                    CCD_geom, inst_config,
                                    nx, ny, j, k, l,
                                    &master_bias_image,
                                    &RON_RE[imagelist_place],
                                    &master_bias_mean[imagelist_place],
                                    &cosmics_nb);
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_master_bias_one_output failed: %s",
                                 cpl_error_get_message_default(my_error));
                }
                
                espdr_msg_debug("RON[%d] = %lf",
                                imagelist_place, RON_RE[imagelist_place]);
                espdr_msg_debug("cosmics_nb[%d] = %d",
                                imagelist_place, cosmics_nb);
                
                cpl_imagelist_delete(images_stack);
                
                *total_cosmics_RE = *total_cosmics_RE + cosmics_nb;
                
                /* Save the result in an imagelist */
                my_error = cpl_imagelist_set(*master_bias_imagelist_RE,
                                             cpl_image_duplicate(master_bias_image),
                                             imagelist_place);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelis_set failed: %s",
                             cpl_error_get_message_default(my_error));
                
                cpl_image_delete(master_bias_image);
                
                imagelist_place++;
                iy = ury+CCD_geom->exts[j].outputs[k][l].poscan_ny;
            }
            ix = urx;
        }
    }
    
    espdr_msg("RON: ");
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("\t%lf ", RON_RE[i]);
    }
    espdr_msg("total cosmics nb = %d", *total_cosmics_RE);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the master bias for one output
 @param         raw_bias_imagelist  imagelist of one output of raw bias images
 @param         nx              x size of the output
 @param         ny              y size of the output
 @param         BIAS_param      BIAS parameters structure
 @param         stats_param     stat parameter structure
 @param[out]    master_bias_image_RE    computed one output master bias (returned)
 @param[out]    RON_RE          computed one output RON (returned)
 @param[out]    mean_RE         computed one output mean (returned)
 @param[out]    totalCosmics_RE computed cosmics number (returned)
 @return    CPL_ERROR_NONE iff OK
 
 TODO: AMO: This function computes the RON on each raw frame and on the master
 On the other side the computation of the RON on each frame is
 not used. Either we should add that information as QC or drop
 the storing of that information on the corresponding RON_image
 vector.
 
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_master_bias_one_output(const cpl_imagelist *raw_bias_imagelist,
                                            const espdr_CCD_geometry *CCD_geom,
                                            const espdr_inst_config *inst_config,
                                            const int nx,
                                            const int ny,
                                            const int ext_no,
                                            const int out_x,
                                            const int out_y,
                                            cpl_image **master_bias_image_RE,
                                            double *RON_RE,
                                            double *mean_RE,
                                            int *totalCosmics_RE) {
    
    /* Loop indices */
    int i = 0;
    
    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* MAD forced in case MAD == 0.0 */
    double forced_MAD = 1.0;
    
    /* error code */
    cpl_error_code my_error;
    
    espdr_msg_debug("Starting MasterBias with kSigma = %.2lf, sig_clip_method = %s",
                    inst_config->bias_ksigma,
                    inst_config->bias_sigma_clipping_method);
    espdr_ensure(raw_bias_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "Input bias list is NULL");
    
    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(raw_bias_imagelist);
    
    espdr_msg_debug("On %d frames", frames_nb);
    
    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get_const(raw_bias_imagelist, i) == NULL,
                     CPL_ERROR_NULL_INPUT,
                     "Input bias image %d in the raw bias imagelist is NULL", i);
    }
    
    /* center of data: mean or MAD */
    double CoD = 0.0;
    
    /* number of cosmics rejected */
    int cosmics = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;
    
    /* mean of the vector */
    double mean = 0.0;
    
    /* standard deviation of the vector */
    double stddev = 0.0;
    
    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;
    
    /* Total size of the image */
    int image_size = nx * ny;
    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(image_size, sizeof(double));
    
    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */

    if (strcmp(inst_config->bias_sigma_clipping_method, "median") == 0) {
    	espdr_get_master_and_cosmics_parallel_median(raw_bias_imagelist,frames_nb, nx, ny,
    			inst_config->bias_ksigma, inst_config->bias_ksigma,
				master_bias_image_RE, totalCosmics_RE);
    } else {
    	espdr_get_master_and_cosmics_parallel_mean(raw_bias_imagelist,frames_nb, nx, ny,
    			inst_config->bias_ksigma, inst_config->bias_ksigma,
				master_bias_image_RE, totalCosmics_RE);
    }


    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);
    
    // RON is computed only on the real part of the image
    
    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
    
    cpl_image *real_master_bias = cpl_image_extract(*master_bias_image_RE,
                                                    real_llx, real_lly,
                                                    real_urx, real_ury);
                                                    
    double *real_master_bias_data = cpl_image_get_data_double(real_master_bias);
    
    int real_master_bias_size = (real_urx - real_llx + 1) * (real_ury - real_lly + 1);
    cpl_vector *master_bias_vector_wrapped = cpl_vector_wrap(real_master_bias_size,
                                                             real_master_bias_data);

    my_error = espdr_sig_clip(master_bias_vector_wrapped,
                              inst_config->stat_ksigma,
                              inst_config->stat_ksigma,
                              inst_config->stat_sigma_clipping_method,
                              inst_config->stat_max_iter,
                              reject_zeroMAD, forced_MAD,
                              &CoD, &sigma, &cosmics,
                              &mean, &stddev,
                              &reject_low, &reject_high);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_sig_clip failed");
    cpl_vector_unwrap(master_bias_vector_wrapped);
    
    *RON_RE = stddev * sqrt(frames_nb);
    *mean_RE = mean;
    espdr_msg_debug("RON for 1 output = %lf, mean for 1 output = %lf, nb eliminated: %d",
                    *RON_RE, *mean_RE, cosmics);
    
    /*espdr_msg("total cosmics: %d", cosmics);*/
    
    cpl_image_delete(real_master_bias);
    cpl_free(RON_image);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/* AMO */
/**
 @brief     Prepare the computation of RON and mean on a single frame
 @param         bias_set input raw bias frame set
 @param         CCD_geom            CCD geometry parameters structure
 @param         BIAS_param          BIAS parameters structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param         reject_zeroMAD      flag to reject sigma == 0 for MAD
 @param[out]    bias_imagelist_RE   computed single bias (returned)
 @param[out]    keywords_RE         primary FITS header (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_single_bias_prepare(const cpl_frameset *bias_set,
                                         const espdr_CCD_geometry *CCD_geom,
                                         espdr_qc_keywords *qc_kws,
                                         espdr_inst_config *inst_config,
                                         const int reject_zeroMAD,
                                         cpl_imagelist **bias_imagelist_RE,
                                         cpl_propertylist **keywords_RE) {
    
    int i, j, k, index;
    cpl_error_code my_error = CPL_ERROR_NONE;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int total_output_nb = CCD_geom->total_output_nb;
    const char *fname = NULL;
    cpl_frame *frame = NULL;
    
    espdr_ensure(bias_set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frameset is NULL");
    
    double *BIAS_RON = (double *)cpl_calloc(total_output_nb, sizeof(double));
    
    double *BIAS_mean = (double *)cpl_calloc(total_output_nb, sizeof(double));
    
    my_error = espdr_RON_compute(bias_set, CCD_geom,
                                 inst_config->bias_ksigma,
                                 inst_config->bias_ksigma,
                                 inst_config->bias_sigma_clipping_method,
                                 1000, reject_zeroMAD, 1.0,
                                 BIAS_RON, BIAS_mean);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_RON_compute failed");
    
    
    /* AMO: the setting of keywords has the same index loop as in RON
     * computation==> can be moved there
     */
    /* Add MBIAS_RON keyword, one for each output and each extension */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment, "RON for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword, BIAS_RON[index],
                                                    comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                index++;
            }
        }
    }
    
    /* Add QC_BIAS_RON_CHECK keyword, QC_BIAS_RON_CHECK is 1 (OK) if none
     of the RONs exceeds the maximal value (only if MasterBias created) */
    int QC_RON = 1;
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("RON bias[%d] = %lf", i, BIAS_RON[i]);
        if (BIAS_RON[i] > inst_config->bias_ron_limit) {
            QC_RON = 0;
        }
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_ron_check_kw, QC_RON,
                                     "QC on RON", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_ron_check_kw,
                 cpl_error_get_message_default(my_error));
    
    /* Add IMAGE_MEAN keyword */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_mean_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment, "Mean for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword, BIAS_mean[index],
                                                    comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                index++;
            }
        }
    }
    
    /* Add QC_BIAS_MEAN_CHECK keyword, QC_BIAS_MEAN_CHECK is 1 (OK) if none
     of the means exceeds the maximal value (only if MasterBias created) */
    int QC_mean = 1;
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg("MASTER BIAS mean[%d] = %lf", i, BIAS_mean[i]);
        if (BIAS_mean[i] > inst_config->bias_mean_limit) {
            QC_mean = 0;
        }
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_mean_check_kw, QC_mean,
                                     "QC on mean", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_mean_check_kw,
                 cpl_error_get_message_default(my_error));
    
    cpl_free(BIAS_RON);
    cpl_free(BIAS_mean);
    
    index = 0;
    int ix, iy, llx, urx, lly, ury,l;
    cpl_image* curr_image;
    for (j = 0; j < CCD_geom->ext_nb; j++) {
        ix = 0;
        for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {

            llx = ix+1;
            urx = ix+CCD_geom->exts[j].outputs[k][0].raw_nx;
            iy=0;
            for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {

                //lly = iy+1;
                lly = iy+1+CCD_geom->exts[j].outputs[k][l].ppscan_ny;
                ury = iy+CCD_geom->exts[j].outputs[k][l].raw_ny
                        -CCD_geom->exts[j].outputs[k][l].poscan_ny;

                int nset = cpl_frameset_get_size(bias_set);
                cpl_frameset_iterator* iter =
                                        cpl_frameset_iterator_new(bias_set);
                frame = cpl_frameset_iterator_get(iter);

                for (int iter_frame = 0; iter_frame < nset; iter_frame++) {

                    fname = cpl_frame_get_filename(frame);
                    curr_image = cpl_image_load_window(fname,
                                                       CPL_TYPE_DOUBLE, 0, j+1,llx, lly, urx, ury);

                    espdr_ensure(curr_image == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                                 "Raw bias image got from the list is NULL");

                    /* Caution, the imagelist can only contain the images
                     of the same size */
                    my_error = cpl_imagelist_set(*bias_imagelist_RE,
                                                 cpl_image_duplicate(curr_image),
                                                 index);
                    cpl_image_delete(curr_image);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_imagelist_set failed: %s",
                                 cpl_error_get_message_default(my_error));

                    index++;
                    /* images_stack imagelist contains the same CCD part
                     of all input BIAS frames */
                    cpl_frameset_iterator_advance(iter, 1);
                    frame = cpl_frameset_iterator_get(iter);

                }
                cpl_frameset_iterator_delete(iter);

                iy = ury+CCD_geom->exts[j].outputs[k][l].poscan_ny;
            }
            ix = urx;
        }
    }

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Verify the bias quality and save the QC keywords
 @param         master_bias_imagelist   input master bias
 @param         overscan_imagelist      input overscan image
 @param         CCD_geom                CCD geometry parameters structure
 @param         stats_param             stat parameters structure
 @param         qc_kws                  KWs names
 @param         inst_config             instrument config
 @param         RON_vector_OV           input overscan RON
 @param         res_mean_limit          limit for mean of residuals
 @param         res_stdev_tolerance     limit for stdev of residuals
 @param[out]    keywords_RE             primary FITS header (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_bias_QC(const cpl_imagelist *master_bias_imagelist,
                             const cpl_imagelist *overscan_imagelist,
                             const espdr_CCD_geometry *CCD_geom,
                             espdr_qc_keywords *qc_kws,
                             espdr_inst_config *inst_config,
                             double *RON_vector_OV,
                             const double res_mean_limit,
                             const double res_stdev_tolerance,
                             cpl_propertylist **keywords_RE) {
    
    
	/* indexes for loops*/
	int i, j, k, index;
	/* ext_nr * output_nx * output_ny */
	int total_output_nb = CCD_geom->total_output_nb;
	int RES_TEST_flag = 1;
    int QC_RON = 1;
    int QC_master = 0;
	char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
	/* error code */
	cpl_error_code my_error = CPL_ERROR_NONE;

	/* residuals_mean contains the mean of residuals for each image */
	double *residuals_mean = (double*)cpl_calloc(total_output_nb, sizeof(double));
    
	/* residuals_stddev contains the standard deviation of residuals
	 for each image */
	double *residuals_stddev = (double*)cpl_calloc(total_output_nb, sizeof(double));
    
	/* KSTEST_extension contains the result the kstest for each image */
	int *RES_TEST_extension = (int*)cpl_calloc(total_output_nb, sizeof(int));
    
    /* RES_structx & RES_structy contain computed master bias structure */
    double *RES_structx = (double*)cpl_calloc(total_output_nb, sizeof(double));
    double *RES_structy = (double*)cpl_calloc(total_output_nb, sizeof(double));
    
	/* Evaluate the quality of the master bias versus the overscan and get
	 the residuals mean and standard deviation and the result of the kstest */
    int output_index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                my_error = espdr_bias_evaluation(cpl_imagelist_get_const
                                                 (master_bias_imagelist,
                                                  output_index),
                                                 cpl_imagelist_get_const
                                                 (overscan_imagelist,
                                                  output_index),
                                                 i, j, k,
                                                 CCD_geom,
                                                 inst_config,
                                                 RON_vector_OV[output_index],
                                                 res_mean_limit,
                                                 res_stdev_tolerance,
                                                 &residuals_mean[output_index],
                                                 &residuals_stddev[output_index],
                                                 &RES_TEST_extension[output_index]);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Bias evaluation failed: %s",
                             cpl_error_get_message_default(my_error));
                
                /* Structure computation added by Andrea Modigliani */
                espdr_qc_structure(cpl_imagelist_get_const(master_bias_imagelist,
                                                     output_index),
                                   &RES_structx[output_index], &RES_structy[output_index]);
                espdr_msg("structure: %g %g", RES_structx[output_index], RES_structy[output_index]);
                
                output_index++;
            }
        }
	}
	
	/* Add RESIDUALS_MEAN, RESIDUALS_STDDEV & RESIDUALS_TEST keywords */
    index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
				new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_mean_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment, "Residuals mean[ADU] for ext %d, out %d, %d",
                        i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    residuals_mean[index],
                                                    comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
				cpl_free(new_keyword);
                
				new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_stdev_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment,
                        "Residuals stddev[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    residuals_stddev[index],
                                                    comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
				cpl_free(new_keyword);
                
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_test_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment,
                        "Residuals test result for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_int(new_keyword,
                                                 RES_TEST_extension[index],
                                                 comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                
				cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_structx_kw_part,
				                                                  inst_config->prefix,
				                                                  i, j, k);
                sprintf(comment,"Structure for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RES_structx[index],
                                                    comment, keywords_RE);
                cpl_free(new_keyword);
                
                sprintf(comment,"Structure for ext %d, out %d, %d", i, j, k);
                
                new_keyword = 
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_structy_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RES_structy[index],
                                                    comment, keywords_RE);
                cpl_free(new_keyword);
                
				index++;
			}
		}
	}
	
	
	/* Add the KSTEST_RESULT keyword, if any of the KSTEST is failed (=0)
	 the general one is also failed (=0) */
	for (i = 0; i < total_output_nb; i++) {
        if (RES_TEST_extension[i] == 0) {
            RES_TEST_flag = 0;
        }
	}
    
    my_error = espdr_keyword_add_int(qc_kws->qc_res_test_kw, RES_TEST_flag,
                                     "Residuals test global result",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_res_test_kw,
                 cpl_error_get_message_default(my_error));
    
	/* Add the RES_TEST keyword, if RES_TEST is passed and the RON
	 is within the limits, the QC is passed */
    QC_RON = cpl_propertylist_get_int(*keywords_RE,
                                      qc_kws->qc_bias_ron_check_kw);
	QC_master = RES_TEST_flag & QC_RON;
    
    my_error = espdr_keyword_add_int(qc_kws->qc_bias_check_kw, QC_master,
                                     "Global mbias QC check", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_bias_check_kw,
                 cpl_error_get_message_default(my_error));

	cpl_free(residuals_mean);
	cpl_free(residuals_stddev);
	cpl_free(RES_TEST_extension);
    cpl_free(RES_structx);
    cpl_free(RES_structy);
	
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Add QC keywords in e-
 @param         keywords_ext            extension FITS header (returned)
 @param[out]    keywords_RE             primary FITS header (returned)
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_add_QC_e(cpl_propertylist **keywords_ext,
                              const espdr_CCD_geometry *CCD_geom,
                              espdr_inst_config *inst_config,
                              espdr_qc_keywords *qc_kws,
                              cpl_propertylist **keywords_RE) {
    
    int i, j, k;
    double nominal_conad = 0.0;
    char *conad_KW = (char *) malloc (KEYWORD_LENGTH * sizeof(char));
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    double KW_value;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                espdr_msg("Treating output %d,%d (%d)", j, k,
                          CCD_geom->exts[i].outputs[j][k].out_phys_no);
                sprintf(conad_KW, "%s%d%s",
                        inst_config->conad_kw_first_part,
                        CCD_geom->exts[i].outputs[j][k].out_phys_no,
                        inst_config->conad_kw_last_part);
                nominal_conad = cpl_propertylist_get_double(keywords_ext[i],
                                                            conad_KW);
                espdr_msg("Nominal CONAD = %lf", nominal_conad);
                
                // Adding BIAS RON
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = cpl_propertylist_get_double(*keywords_RE,
                                                       new_keyword);
                espdr_msg("BIAS RON [ADU] = %lf", KW_value);
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_e_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = KW_value * nominal_conad;
                espdr_msg("BIAS RON [e-] = %lf", KW_value);
                sprintf(comment, "RON[e-] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    KW_value,
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                // Add BIAS MEAN
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_mean_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = cpl_propertylist_get_double(*keywords_RE,
                                                       new_keyword);
                espdr_msg("BIAS mean [ADU] = %lf", KW_value);
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_mean_e_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = KW_value * nominal_conad;
                espdr_msg("BIAS mean [e-] = %lf", KW_value);
                sprintf(comment, "Mean[e-] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    KW_value,
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                // Add OVSC RON
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_ovsc_ron_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = cpl_propertylist_get_double(*keywords_RE,
                                                       new_keyword);
                espdr_msg("OVSC RON [ADU] = %lf", KW_value);
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_ovsc_ron_e_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = KW_value * nominal_conad;
                espdr_msg("OVSC RON [e-] = %lf", KW_value);
                sprintf(comment, "OVSC RON[e-] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    KW_value,
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                // Add RES MEAN
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_mean_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = cpl_propertylist_get_double(*keywords_RE,
                                                       new_keyword);
                espdr_msg("Residuals mean [ADU] = %lf", KW_value);
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_mean_e_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = KW_value * nominal_conad;
                espdr_msg("Residuals mean [e-] = %lf", KW_value);
                sprintf(comment, "Res mean[e-] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    KW_value,
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
                
                // Add RES STDEV
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_stdev_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = cpl_propertylist_get_double(*keywords_RE,
                                                       new_keyword);
                espdr_msg("Residuals stdev [ADU] = %lf", KW_value);
                cpl_free(new_keyword);
                
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_res_stdev_e_kw_part,
                                                      inst_config->prefix,
                                                      i, j, k);
                KW_value = KW_value * nominal_conad;
                espdr_msg("Residuals stdev [e-] = %lf", KW_value);
                sprintf(comment, "Res stdev[e-] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    KW_value,
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);
            }
        }
    }
    
    cpl_free(conad_KW);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Evaluate the quality of the master bias calculation by comparing
            it with the overscan. To test the quality the Kolmogorov-Smirnoff
            test is used.
 @param         master_bias_image       input master bias
 @param         overscan_image          input overscan image
 @param         ext_no                  extension number
 @param         out_x                   x position output port
 @param         out_y                   y position output port
 @param         CCD_geom                CCD geometry parameters structure
 @param         stats_param             stat parameters structure
 @param         RON                     input RON
 @param         res_mean_limit          limit for mean of residuals
 @param         res_stdev_tolerance     limit for stdev of residuals
 @param[out]    residuals_mean_RE       mean of the residuals (returned)
 @param[out]    residuals_stddev_RE     stddev of the residuals (returned)
 @param[out]    QC_flag_RE              QC flag (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_bias_evaluation(const cpl_image *master_bias_image,
                                     const cpl_image *overscan_image,
                                     const int ext_no,
                                     const int out_x,
                                     const int out_y,
                                     const espdr_CCD_geometry *CCD_geom,
                                     const espdr_inst_config *inst_config,
                                     const double RON,
                                     const double res_mean_limit,
                                     const double res_stdev_tolerance,
                                     double *residuals_mean_RE,
                                     double *residuals_stddev_RE,
                                     int *QC_flag_RE) {

	int nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
	int ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
    int used_real_nx = real_urx - real_llx + 1;
    int used_real_ny = real_ury - real_lly + 1;
	int collumn_orientation = CCD_geom->exts[ext_no].coll_or;
	int correction_type = CCD_geom->exts[ext_no].corr_type;
    int pscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_urx;
    int pscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ury;
    int absolut_llx = 0, absolut_lly = 0, absolut_urx = 0, absolut_ury = 0;
	   
    cpl_image *residuals;
    double *residuals_data;
	/* mean and standard deviation of residuals */
	double residuals_mean;
	double residuals_stddev;
    /* parameters to protect from MAD = 0.0 */
    int reject_zeroMAD = 1;
    double forced_MAD = 1.0;
    /* results of sigma_clipping*/
    double CoD = 0.0;
    double sigma = 0.0;
    int cosmics = 0;
    double reject_low = 0.0;
    double reject_high = 0.0;
	/* quality control flag resulting from the KSTest */
	int QC_flag;
	/* possible error code */
	cpl_error_code my_error = CPL_ERROR_NONE;
	
	espdr_msg_debug("Starting espdr_bias_evaluation with");
	espdr_msg_debug("\treal_llx = %d, real_lly = %d, real_urx = %d, real_ury = %d",
					real_llx, real_lly, real_urx, real_ury);
	espdr_msg_debug("\tnx = %d, ny = %d", nx, ny);
	espdr_msg_debug("\tres_mean_limit = %lf, res_stdev_tolerance = %lf",
					res_mean_limit, res_stdev_tolerance);
	
	espdr_ensure(master_bias_image == NULL, CPL_ERROR_NULL_INPUT, 
			   "Input master bias image is NULL");
	
	espdr_ensure(overscan_image == NULL, CPL_ERROR_NULL_INPUT, 
			   "Input overscan image is NULL");
	
	/* the direction is set by collumn_orientation and correction_type */
	if (((collumn_orientation == 0) && (correction_type == 1)) ||
		((collumn_orientation == 1) && (correction_type == 0))) {
        absolut_llx = pscan_urx + real_llx;
        absolut_lly = real_lly;
        absolut_urx = pscan_urx + real_urx;
        absolut_ury = real_ury;
	} else {
		if (((collumn_orientation == 1) && (correction_type == 1)) ||
			((collumn_orientation == 0) && (correction_type == 0))) {
            absolut_llx = real_llx;
            absolut_lly = pscan_ury + real_lly;
            absolut_urx = real_urx;
            absolut_ury = pscan_ury + real_ury;
		} else {
            espdr_ensure(0, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Wrong combination of collumn_orientation and correction_type, exiting");
		}
	}
    
    /* Compute the residuals */
    residuals = cpl_image_extract(master_bias_image,
                                  absolut_llx, absolut_lly,
                                  absolut_urx, absolut_ury);
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_extract failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_image_subtract(residuals, overscan_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_subtract failed: %s",
                 cpl_error_get_message_default(my_error));
    

    residuals_data = cpl_image_get_data_double(residuals);
    cpl_vector *residuals_vector = cpl_vector_wrap(used_real_nx*used_real_ny,
                                                   residuals_data);
    /*
    	my_error = espdr_sig_clip(residuals_vector,
    			inst_config->stat_ksigma,
				inst_config->stat_ksigma,
				inst_config->stat_sigma_clipping_method,
				inst_config->stat_max_iter,
				reject_zeroMAD, forced_MAD,
				&CoD, &sigma, &cosmics,
				&residuals_mean, &residuals_stddev,
				&reject_low, &reject_high);
				*/

    if (strcmp(inst_config->ovsc_sigma_clipping_method, "mean") == 0) {

    	my_error = espdr_sig_clip_method_mean(residuals_vector,
    			inst_config->stat_ksigma,
				inst_config->stat_ksigma,
				inst_config->stat_max_iter,
				reject_zeroMAD, forced_MAD,
				&CoD, &sigma, &cosmics,
				&residuals_mean, &residuals_stddev,
				&reject_low, &reject_high);
     
    } else {
       	my_error = espdr_sig_clip_method_median(residuals_vector,
        			inst_config->stat_ksigma,
    				inst_config->stat_ksigma,
    				inst_config->stat_max_iter,
    				reject_zeroMAD, forced_MAD,
    				&CoD, &sigma, &cosmics,
    				&residuals_mean, &residuals_stddev,
    				&reject_low, &reject_high);

    }

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_sig_clip failed");
    cpl_vector_unwrap(residuals_vector);
    espdr_msg("For %d [%d, %d] residuals mean = %.2lf, residuals stddev = %.2lf, RON = %.2f",
              ext_no, out_x, out_y, residuals_mean, residuals_stddev, RON);
    espdr_msg("Number of bad pixels rejected: %d", cosmics);
    

    
    QC_flag = 1;
    if (fabs(residuals_mean) > res_mean_limit) {
        //espdr_msg("---------->>>>>>>>> res mean too high: %.2f limit: %.2f",
        //          residuals_mean, res_mean_limit);
        QC_flag = 0;
    }
    
    if (fabs(residuals_stddev - RON) > res_stdev_tolerance) {
        //espdr_msg("---------->>>>>>>>> res stdev too high: %.2f RON: %.2f diff: %.2f limit: %.2f",
        //          residuals_stddev, RON, fabs(residuals_stddev - RON), res_stdev_tolerance);
        QC_flag = 0;
    }
    
	/* copy the output variables */
	*residuals_mean_RE = residuals_mean;
	*residuals_stddev_RE = residuals_stddev;
	*QC_flag_RE = QC_flag;
	
	cpl_image_delete(residuals);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Correct the bias effect in raw input frames
 @param         input_image         image to be bias corrected
 @param         master_bias_image   master bias image
 @param         keywords            FITS header
 @param         qc_keywords         KWs names
 @param         ext_no              extension number
 @param         out_x               x position output port
 @param         out_y               y position output port
 @param         CCD_geom            CCD geometry parameters structure
 @param         OVSC_param          overscan parameters structure
 @param         inst_config         instrument config
 @param[out]    RON_RE              computed RON
 @param[out]    input_corrected_RE  images corrected for BIAS (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_bias_one_output(const cpl_image *input_image,
                                             const cpl_image *master_bias_res_image,
                                             //cpl_propertylist *keywords,
                                             //espdr_qc_keywords *qc_kws,
                                             const int ext_no,
                                             const int out_x,
                                             const int out_y,
                                             const espdr_CCD_geometry *CCD_geom,
                                             espdr_inst_config *inst_config,
                                             const int remove_bias_res,
                                             double *RON_RE,
                                             cpl_image **input_corrected_RE) {
	
	const cpl_image *curr_image = NULL;
	cpl_image *overscan_image = NULL;
	cpl_image *corrected_image = NULL;
    cpl_image *res_corrected_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	double RON = 0.0;
	int cosmics_nb;
	
	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL");
    
    espdr_msg("BIAS correction method is OVERSCAN");
    
    curr_image = input_image;
    my_error = espdr_overscan(curr_image,
                              ext_no, out_x, out_y,
                              CCD_geom, inst_config,
                              &overscan_image,
                              &RON,
                              &cosmics_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_overscan failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_remove_overscan(curr_image,
                                     ext_no, out_x, out_y, CCD_geom,
                                     overscan_image,
                                     &corrected_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_overscan failed: %s",
                 cpl_error_get_message_default(my_error));
    //cpl_image_delete(curr_image);
    
    if (remove_bias_res) {
        espdr_msg("and master bias residuals will be removed");
        
        espdr_ensure(master_bias_res_image == NULL, CPL_ERROR_NULL_INPUT,
                     "Input master bias resilduals image is NULL");
        
        //char *new_keyword = NULL;
        //new_keyword = espdr_add_ext_output_index_to_keyword(
        //                                      qc_kws->qc_out_bias_ron_kw_part,
        //                                      inst_config->prefix,
        //                                      ext_no,out_x,out_y);
        //RON = cpl_propertylist_get_double(keywords, new_keyword);
        //espdr_msg_debug("espdr_correct_bias: RON = %lf",RON);
        //cpl_free(new_keyword);
        
        my_error = espdr_remove_bias_one_output(corrected_image,
                                                ext_no, out_x, out_y,
                                                CCD_geom,
                                                master_bias_res_image,
                                                &res_corrected_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_remove_bias_one_output failed: %s",
                     cpl_error_get_message_default(my_error));
        
        *input_corrected_RE = cpl_image_duplicate(res_corrected_image);
        cpl_image_delete(res_corrected_image);
        
    } else {
        espdr_msg("and master bias residuals will NOT be removed");
        
        *input_corrected_RE = cpl_image_duplicate(corrected_image);
    }
    
    cpl_image_delete(overscan_image);
    cpl_image_delete(corrected_image);
    
	*RON_RE = RON;

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Correct the bias effect in raw input frames
 @param         input_imagelist         list of images to be bias corrected
 @param         master_bias_imagelist   list of master biases to correct BIAS frames
 @param         keywords                FITS header
 @param         qc_keywords             KWs names
 @param         inst_config             instrument config
 @param         CCD_geom                CCD geometry parameters structure
 @param         OVSC_param              overscan parameters structure
 @param[out]    RON                     computed RON
 @param[out]    input_corrected_RE      images corrected for BIAS (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_bias(const cpl_imagelist *input_imagelist,
                                  const cpl_imagelist *master_bias_imagelist,
                                  espdr_inst_config *inst_config,
                                  const espdr_CCD_geometry *CCD_geom,
                                  const int remove_bias_res,
                                  double *RON,
                                  cpl_imagelist *input_corrected_RE) {
	
	const cpl_image *curr_image = NULL, *curr_bias = NULL;
    cpl_image *overscan_image = NULL;
	cpl_image *corrected_image = NULL;
    cpl_image *res_corrected_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	int imagelist_size = 0;
    int images_nb = 0;
	int i, j, k, l, index, index_MB;
	int cosmics_nb;
    char *new_keyword = NULL;
	
    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL");
    
    imagelist_size = cpl_imagelist_get_size(input_imagelist);
	espdr_msg_debug("Starting espdr_correct_bias(), input imagelist size: %d",
                    imagelist_size);
    images_nb = imagelist_size / CCD_geom->total_output_nb;
    espdr_ensure(imagelist_size != CCD_geom->total_output_nb*images_nb,
                 CPL_ERROR_ILLEGAL_INPUT,
                 "The number of images in the input imagelist incompatible with the total nb of outputs expected");
    
    espdr_msg("BIAS correction method is OVERSCAN");
    if (remove_bias_res) {
        espdr_msg("and MASTER_BIAS residuals will be removed");
    } else {
        espdr_msg("and master bias residuals will NOT be removed");
    }
    
    index = 0;
    for (i = 0; i < images_nb; i++) {
        index_MB = 0;
        for (j = 0; j < CCD_geom->ext_nb; j++) {
            for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {
                for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {
                    curr_image = cpl_imagelist_get_const(input_imagelist, index);
                    
                    my_error = espdr_overscan(curr_image,
                                              j, k, l,
                                              CCD_geom, inst_config,
                                              &overscan_image,
                                              &RON[index],
                                              &cosmics_nb);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_overscan failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    //espdr_msg("oscan output: [%d %d %d] RON = %lf crh = %d, overscan_mean = %f",
                    //          j, k, l, RON[index], cosmics_nb,
                    //          cpl_image_get_mean(overscan_image));
                    
                    espdr_msg_debug("espdr_correct_bias: RON[%d] = %lf",
                                    index, RON[index]);
                    
                    my_error = espdr_remove_overscan(curr_image,
                                                     j, k, l, CCD_geom,
                                                     overscan_image,
                                                     &corrected_image);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_remove_overscan failed: %s",
                                 cpl_error_get_message_default(my_error));
                    //cpl_image_delete(curr_image);
                    
                    if (remove_bias_res) {
                        espdr_ensure(master_bias_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                                     "Master bias imagelist is NULL");
                        
                        curr_bias = cpl_imagelist_get_const(master_bias_imagelist,
                                                            index_MB);
                        my_error = espdr_remove_bias(corrected_image, j, k, l, CCD_geom,
                                                     curr_bias, &res_corrected_image);
                        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                     "espdr_remove_bias failed: %s",
                                     cpl_error_get_message_default(my_error));
                    } else {
                        res_corrected_image = cpl_image_duplicate(corrected_image);
                    }
                    
                    index_MB++;
                    
                    cpl_free(new_keyword);
                    
                    my_error = cpl_imagelist_set(input_corrected_RE,
                                                 cpl_image_duplicate(res_corrected_image),
                                                 index);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_imagelist_set failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    cpl_image_delete(overscan_image);
                    cpl_image_delete(corrected_image);
                    cpl_image_delete(res_corrected_image);
                    
                    index++;
                }
            }
        }
    }
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 AMO added
 @brief     Correct the bias effect in raw input frames
 @param         raw_bias_set        list of frames to be bias corrected
 @param         master_bias_set     list of master biases to correct BIAS frames
 @param         CCD_geom            CCD geometry parameters structure
 @param         OVSC_param          overscan parameters structure
 @param[out]    input_corrected_set frames corrected for BIAS (returned)
 @param         use_hdrl            HDRL use flag
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_bias_frames(cpl_frameset *raw_bias_set,
                                         cpl_frameset *master_bias_set,
                                         espdr_CCD_geometry *CCD_geom,
                                         espdr_inst_config *inst_config,
                                         int remove_bias_res,
                                         cpl_frameset** input_corrected_set,
                                         const int use_hdrl) {
	
    cpl_image *curr_image = NULL;
    cpl_image *overscan_image = NULL;
    cpl_image *curr_bias = NULL;
    cpl_image *corrected_image = NULL;
    cpl_image *res_corrected_image = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int imagelist_size = 0;
    int images_nb = 0;
    int j, k, l, index, index_MB;
    double RON_OV;
    int cosmics_nb;
	
    /* AMO added */
    cpl_frame* raw_frame = NULL;
    cpl_frame* mst_frame = NULL;
    const char* raw_name = NULL;
    const char* mst_name = NULL;
    int ix = 0, iy = 0;
    int llx, lly, urx, ury;
    int ix_MB = 0, iy_MB = 0;
    int llx_MB, lly_MB, urx_MB, ury_MB;
    char tmp_name[FILENAME_LENGTH];
    cpl_propertylist* plist=NULL;
    
    espdr_ensure(raw_bias_set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frameset is NULL");
    images_nb = cpl_frameset_get_size(raw_bias_set);
    
    imagelist_size = images_nb*CCD_geom->total_output_nb;
    espdr_msg_debug("Starting espdr_correct_bias(), input imagelist size: %d",
                    imagelist_size);
    
    espdr_msg("BIAS correction method is OVERSCAN");
    if (remove_bias_res) {
        espdr_msg("and master bias residuals will be removed");
    }
    
    *input_corrected_set = cpl_frameset_new();
    
    index = 0;
    int nset = cpl_frameset_get_size(raw_bias_set);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(raw_bias_set);
    raw_frame = cpl_frameset_iterator_get(iter);
    
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        
        //raw_frame = cpl_frameset_get_frame(raw_bias_set, index);
        raw_name = cpl_frame_get_filename(raw_frame);
        plist = cpl_propertylist_load(raw_name,0);
        //espdr_msg("fname = %s", raw_name);
        
        index_MB = 0;
        for (j = 0; j < CCD_geom->ext_nb; j++) {
            
            ix = 0;
            ix_MB = 0;
            
            //espdr_msg("Process j = %d", j);
            for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {
                
                llx = ix+1;
                urx = ix+CCD_geom->exts[j].outputs[k][0].raw_nx;
                iy=0;
                
                llx_MB = ix_MB + 1;
                urx_MB = ix_MB + CCD_geom->exts[j].outputs[k][0].real_nx;
                iy_MB = 0;
                
                //espdr_msg("Process k = %d", k);
                for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {
                    
                    lly = iy+1+CCD_geom->exts[j].outputs[k][l].ppscan_ny;
                    ury = iy+CCD_geom->exts[j].outputs[k][l].raw_ny
                                -CCD_geom->exts[j].outputs[k][l].poscan_ny;
                    
                    lly_MB = iy_MB + 1;
                    ury_MB = iy_MB + CCD_geom->exts[j].outputs[k][l].real_ny;
                    
                    //espdr_msg("Process l = %d", l);
                    //espdr_msg("Process index = %d", index);
                    
                    /* AMO here we should read each output port */
                    curr_image = cpl_image_load_window(raw_name,
                                                       CPL_TYPE_DOUBLE,
                                                       0, j+1,
                                                       llx, lly, urx, ury);
                    
                    //espdr_msg("ext: %d out_x: %d out_y: %d", j, k, l);
                    //curr_image = cpl_imagelist_get(input_imagelist, index);
                    
                    if(use_hdrl) {
                        my_error = espdr_hdrl_overscan(curr_image,
                                                       j, k, l,
                                                       CCD_geom, inst_config,
                                                       &overscan_image,
                                                       &RON_OV,
                                                       &cosmics_nb);
                    } else {
                        /* AMO: here we could use HDRL */
                        my_error = espdr_overscan(curr_image,
                                                  j, k, l,
                                                  CCD_geom, inst_config,
                                                  &overscan_image,
                                                  &RON_OV,
                                                  &cosmics_nb);
                    }
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_overscan failed: %s",
                                 cpl_error_get_message_default(my_error));
                    /* AMO added to check results
                     sprintf(tmp_name,"raw_dark2_ext_%d_port_x%d_y%d_index_%d.fits",j,k,l,index);
                     cpl_image_save(curr_image, tmp_name, CPL_TYPE_DOUBLE,NULL, CPL_IO_DEFAULT);
                     sprintf(tmp_name,"mst_bias2_ext_%d_port_x%d_y%d_index_%d.fits",j,k,l,index);
                     cpl_image_save(overscan_image, tmp_name, CPL_TYPE_DOUBLE,NULL, CPL_IO_DEFAULT);
                     */
                    /* AMO: here we could use HDRL */
                    my_error = espdr_remove_overscan(curr_image,
                                                     j, k, l, CCD_geom,
                                                     overscan_image,
                                                     &corrected_image);
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_remove_overscan failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    if (remove_bias_res) {
                        mst_frame = cpl_frameset_get_position(master_bias_set, 0);
                        mst_name = cpl_frame_get_filename(mst_frame);
                        
                        //espdr_msg("MASTER_BIAS coord: llx = %d, lly = %d, urx = %d, ury = %d",
                        //          llx_MB, lly_MB, urx_MB, ury_MB);
                        
                        curr_bias = cpl_image_load_window(mst_name,
                                                          CPL_TYPE_DOUBLE,
                                                          0, j+1,
                                                          llx_MB, lly_MB,
                                                          urx_MB, ury_MB);
                        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                     "cpl_image_load_window on MB failed: %s",
                                     cpl_error_get_message_default(my_error));

                        my_error = espdr_remove_bias(corrected_image, j, k, l, CCD_geom,
                                                     curr_bias, &res_corrected_image);
                        cpl_image_delete(curr_bias);
                        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                     "espdr_remove_bias failed: %s",
                                     cpl_error_get_message_default(my_error));
                    } else {
                        res_corrected_image = cpl_image_duplicate(corrected_image);
                    }
                    
                    
                    /* AMO: here we should rather save on disk and add to a
                     * frameset the filename of the saved image */
                    sprintf(tmp_name,
                            "cor_dark_ext_%d_port_x%d_y%d_index_%d.fits",
                            j, k, l, index);
                    
                    cpl_frame* frame_cor = NULL;
                    frame_cor = cpl_frame_new();
                    cpl_frame_set_filename(frame_cor, tmp_name);
                    cpl_frame_set_tag(frame_cor, "DUMMY");
                    
                    cpl_frameset_insert(*input_corrected_set,
                                        cpl_frame_duplicate(frame_cor));
                    
                    cpl_image_save(res_corrected_image, tmp_name, CPL_TYPE_DOUBLE,
                                   plist, CPL_IO_DEFAULT);
                    
                    cpl_frame_delete(frame_cor);
                    
                    cpl_image_delete(overscan_image);
                    cpl_image_delete(corrected_image);
                    cpl_image_delete(res_corrected_image);
                    cpl_image_delete(curr_image);
                    
                    index++;
                    
                    iy = ury+CCD_geom->exts[j].outputs[k][l].poscan_ny;
                    iy_MB = ury_MB;
                    
                } /* end loop over y read-out */
                
                ix = urx;
                ix_MB = urx_MB;
                
            } /* end loop over x-read-out */
            
        } /* end loop over extensions */
        
        cpl_propertylist_delete(plist);
        cpl_frameset_iterator_advance(iter, 1);
        raw_frame = cpl_frameset_iterator_get(iter);
        
    } /* end loop over frames */
    cpl_frameset_iterator_delete(iter);
    
    return cpl_error_get_code();
    
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Remove the bias effect from (one output) raw input frames with master bias
 @param         input_image         image to be corrected for BIAS
 @param         ext_no              extension number
 @param         out_x               x position output port
 @param         out_y               y position output port
 @param         CCD_geom            CCD geometry parameters structure
 @param         master_bias_image   master bias frame to correct BIAS with
 @param[out]    corrected_image_RE  image corrected for BIAS (returned)
 @return    CPL_ERROR_NONE iff OK
 AMO note: decided to first extract the same region from master and raw image
 and then subtract one from the other (instead than first subtraction the full
 images and then extracting the relevant region) to speed-up.
 --> DSO    OK, that's fine.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_remove_bias_one_output(const cpl_image *input_image,
                                            const int ext_no,
                                            const int out_x,
                                            const int out_y,
                                            const espdr_CCD_geometry *CCD_geom,
                                            const cpl_image *master_bias_image,
                                            cpl_image **corrected_image_RE) {
	
    cpl_image *tmp_master_image = NULL;
	
	cpl_error_code prestate = cpl_error_get_code();
	espdr_msg_debug("Error prestate: %d", prestate);
	
	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
                 "Input image is NULL");
	espdr_ensure(master_bias_image == NULL, CPL_ERROR_NULL_INPUT, 
                 "Master bias image os NULL");
	
	int input_image_size_x = cpl_image_get_size_x(input_image);
	int input_image_size_y = cpl_image_get_size_y(input_image);
	int mbias_image_size_x = cpl_image_get_size_x(master_bias_image);
	int mbias_image_size_y = cpl_image_get_size_y(master_bias_image);
	
	//espdr_msg("REMOVE BIAS: input size x: %d, y: %d",
    //          input_image_size_x, input_image_size_y);
	//espdr_msg("REMOVE BIAS: mbias res size x: %d, y: %d",
    //          mbias_image_size_x, mbias_image_size_y);
	
	espdr_ensure(((input_image_size_x != mbias_image_size_x) || 
				(input_image_size_y != mbias_image_size_y)), 
			   CPL_ERROR_INCOMPATIBLE_INPUT, 
			   "Input and master bias images sizes differ");
	
    /* Identify the real pixels region of the detector output */
    
	int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
	int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
	int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
	int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
	
	tmp_master_image = cpl_image_extract(master_bias_image,
                                         real_llx, real_lly,
                                         real_urx, real_ury);
    
	*corrected_image_RE = cpl_image_extract(input_image,
                                            real_llx, real_lly,
                                            real_urx, real_ury);
    
	cpl_image_subtract(*corrected_image_RE, tmp_master_image);
    
    	cpl_image_delete(tmp_master_image);
    
	espdr_ensure(*corrected_image_RE == NULL, CPL_ERROR_NULL_INPUT,
			   "Corrected image is NULL");
	
	espdr_msg_debug("Leaving espdr_remove_bias_one_output with error code: %d",
				  cpl_error_get_code());
	
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Remove the bias effect from raw input frames with master bias
 @param         input_image     image to be corrected for BIAS
 @param         ext_no          extension number
 @param         out_x           x position output port
 @param         out_y           y position output port
 @param         CCD_geom        CCD geometry parameters structure
 @param         master_bias     master bias frame to correct BIAS with
 @param[out]    corrected_image_RE  image corrected for BIAS (returned)
 @return    CPL_ERROR_NONE iff OK
 AMO note: decided to first extract the same region from master and raw image
 and then subtract one from the other (instead than first subtraction the full
 images and then extracting the relevant region) to speed-up.
 --> DSO    OK, that's fine.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_remove_bias(const cpl_image *input_image,
                                 const int ext_no,
                                 const int out_x,
                                 const int out_y,
                                 const espdr_CCD_geometry *CCD_geom,
                                 const cpl_image *master_bias_image,
                                 cpl_image **corrected_image_RE) {
	
    cpl_image *tmp_master_image = NULL;
	
	cpl_error_code prestate = cpl_error_get_code();
	espdr_msg_debug("Error prestate: %d", prestate);
	
	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT, "Input image is NULL");
	espdr_ensure(master_bias_image == NULL, CPL_ERROR_NULL_INPUT, 
			   "Master bias image is NULL");
	
	int input_image_size_x = cpl_image_get_size_x(input_image);
	int input_image_size_y = cpl_image_get_size_y(input_image);
	int mbias_image_size_x = cpl_image_get_size_x(master_bias_image);
	int mbias_image_size_y = cpl_image_get_size_y(master_bias_image);
	
	//espdr_msg("REMOVE BIAS: input size x: %d, y: %d",
    //          input_image_size_x, input_image_size_y);
	//espdr_msg("REMOVE BIAS: mbias size x: %d, y: %d",
    //          mbias_image_size_x, mbias_image_size_y);
	
	espdr_ensure(((input_image_size_x != mbias_image_size_x) || 
				(input_image_size_y != mbias_image_size_y)), 
			   CPL_ERROR_INCOMPATIBLE_INPUT, 
			   "Input and master bias images sizes differ");
	
    /* Identify the real pixels region of the detector output */
    
	int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
	int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
	int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
	int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
	
    //espdr_msg("Extract coord: [%d, %d] <-> [%d, %d]",
    //          real_llx, real_lly, real_urx, real_ury);
    
	tmp_master_image = cpl_image_extract(master_bias_image,
                                         real_llx, real_lly,
                                         real_urx, real_ury);
    
	*corrected_image_RE = cpl_image_extract(input_image,
                                            real_llx, real_lly,
                                            real_urx, real_ury);
    
	cpl_image_subtract(*corrected_image_RE, tmp_master_image);
    
    cpl_image_delete(tmp_master_image);
    
	espdr_ensure(*corrected_image_RE == NULL, CPL_ERROR_NULL_INPUT,
			   "Corrected image is NULL");
	
	espdr_msg_debug("Leaving espdr_remove_bias with error code: %d",
				  cpl_error_get_code());
	
	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*  The following functions are added by Andrea Modigliani
    to detect the structure present in the bias frames
 */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Reject outlier pixels
 @param    image       image with pixels
 @param    min         reject pixels below this value
 @param    max         reject pixels abouve this value
 */
/*---------------------------------------------------------------------------*/
void reject_lo_hi(cpl_image *image, double min, double max) {
    
    cpl_mask *mask_lo = NULL;
    cpl_mask *mask_hi = NULL;
    
    mask_lo = cpl_mask_threshold_image_create(image, -DBL_MAX, min);
    mask_hi = cpl_mask_threshold_image_create(image, max, DBL_MAX);
    //assure_mem( mask_lo );
    //assure_mem( mask_hi );
    
    cpl_mask_or(mask_lo, mask_hi);
    
    cpl_image_reject_from_mask(image, mask_lo);
    
    /* free memory */
    cpl_mask_delete(mask_lo);
    cpl_mask_delete(mask_hi);
    return;
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Compute the number of good pixels
 @param    image     input image
 @return   number of good pixels
 */
/*---------------------------------------------------------------------------*/
int espdr_count_good(const cpl_image *image) {
    return (cpl_image_get_size_x(image) * cpl_image_get_size_y(image) -
            cpl_image_count_rejected(image));
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Compute structure along rows
 @param    tima     input image
 @return   computed structure
 */
/*---------------------------------------------------------------------------*/
double espdr_get_qc_structure_row_region(cpl_image *tima) {
    
    cpl_image *avg_row = NULL;
    double min = 0;
    double max = 0;
    double struct_row = 0;
    
    avg_row = cpl_image_collapse_create(tima, 0);
    cpl_image_divide_scalar(avg_row, cpl_image_get_size_y(tima));
    
    /* restricts statistics to +/- 2 ADU around mean */
    min = cpl_image_get_mean(avg_row) - 2;
    max = cpl_image_get_mean(avg_row) + 2;
    
    /* replace with MIDAS
     stat/ima avg_row + exc={min},{max};
     */
    reject_lo_hi(avg_row, min, max);
    if (espdr_count_good(avg_row) >= 2) {
        struct_row = cpl_image_get_stdev(avg_row);
    } else {
        struct_row = -1;
        espdr_msg_warning("Only %d good pixels in image. Setting QC parameter to -1",
                          espdr_count_good(avg_row));
    }
    
    /* free memory */
    cpl_image_delete(avg_row);
    
    return struct_row;
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Compute structure along columns
 @param    tima     input image
 @return   computed structure
 */
/*---------------------------------------------------------------------------*/
double espdr_get_qc_structure_col_region(cpl_image *tima) {
    
    cpl_image *avg_col = NULL;
    double min = 0;
    double max = 0;
    double struct_col = 0;
    
    avg_col = cpl_image_collapse_create(tima, 1);
    cpl_image_divide_scalar(avg_col, cpl_image_get_size_x(tima));
    
    /* restricts statistics to +/- 2 ADU around mean */
    min = cpl_image_get_mean(avg_col) - 2;
    max = cpl_image_get_mean(avg_col) + 2;
    
    /* replace with MIDAS
     stat/ima avg_col + exc={min},{max};
     */
    reject_lo_hi(avg_col, min, max);
    if (espdr_count_good(avg_col) >= 2) {
        struct_col = cpl_image_get_stdev(avg_col);
    } else {
        struct_col = -1;
        espdr_msg_warning("Only %d good pixels in image. Setting QC parameter to -1",
                          espdr_count_good(avg_col));
    }
    
    /* free memory */
    cpl_image_delete(avg_col);
    
    return struct_col;
}

/*---------------------------------------------------------------------------*/
/**
 * Calculates and sets QC parameters: Median, Average, Stdev and Slope, for
 * primary image AND qual.
 *
 * @param master Master Bias Frame (PRE format)
 * @param llx lower left X
 * @param lly lower left Y
 * @param urx upper right X
 * @param ury upper right Y
 * @param ref_x reference X
 * @param ref_y reference Y
 * @param reg_id region id
 * @param dlevel debug level
 * @return void
 */
/*---------------------------------------------------------------------------*/
void espdr_qc_structure_region(const cpl_image *master,
                               const int llx,
                               const int lly,
                               const int urx,
                               const int ury,
                               const int ref_x,
                               const int ref_y,
                               const int reg_id,
                               double *structx,
                               double *structy) {
    /* UVES algorithm */
    cpl_image *xima = NULL;
    cpl_image *yima = NULL;
    double struct_row = 0;
    double struct_col = 0;
    
    //double mean = cpl_image_get_mean_window(master, llx, lly, urx, ury);
    double median = cpl_image_get_mean_window(master, llx, lly, urx, ury);
    double stdev = cpl_image_get_stdev_window(master, llx, lly, urx, ury);
    
    
    if (reg_id == 1) {
        xima = cpl_image_extract(master, llx, lly, urx, ref_y);
    } else {
        xima = cpl_image_extract(master, llx, ref_y, urx, ury);
    }
    
    if (0) {
        /*
         replace/ima {mbia} {tmpfrm} 300,>=300.;
         */
        cpl_image_threshold(xima, -DBL_MAX, median + 3 * stdev,
                            -DBL_MAX, median + 3 * stdev);
    }
    
    if (reg_id == 1) {
        yima = cpl_image_extract(master, llx, lly, ref_x, ury);
    } else {
        yima = cpl_image_extract(master, ref_x, lly, urx, ury);
    }
    
    if (0) {
        /*
         replace/ima {mbia} {tmpfrm} 300,>=300.;
         */
        cpl_image_threshold(yima, -DBL_MAX, median + 3 * stdev,
                            -DBL_MAX, median + 3 * stdev);
    }
    
    struct_row = espdr_get_qc_structure_row_region(xima);
    *structx = struct_row;
    
    struct_col = espdr_get_qc_structure_col_region(yima);
    *structy = struct_col;
    //espdr_msg("Structure: %g %g",*structx,*structy);
    
    /* free memory */
    cpl_image_delete(xima);
    cpl_image_delete(yima);
    
    return;
}

/*---------------------------------------------------------------------------*/
/**
 * Calculates and sets QC parameters: Median, Average, Stdev and Slope, for
 * primary image AND qual.
 *
 * @param master Master Bias Frame (PRE format)
 * @param parameters Recipe input parameters
 */
/*---------------------------------------------------------------------------*/
void espdr_qc_structure(const cpl_image *master,
                        double *structx,
                        double *structy) {
    
    //cpl_parameter *p = NULL;
    int ref_x = -1;
    int ref_y = -1;
    
    int ref_llx = 0;
    int ref_lly = 0;
    int ref_urx = 0;
    int ref_ury = 0;
    
    int sx = 0;
    int sy = 0;
    //int dlevel = 0;
    
    sx = cpl_image_get_size_x(master);
    sy = cpl_image_get_size_y(master);
    
    /* TODO: define/use proper parameter
     p = cpl_parameterlist_find(parameters,"xsh.xsh_mbias.struct_refx");
     ref_x = cpl_parameter_get_int(p);
     
     p = cpl_parameterlist_find(parameters,"xsh.xsh_mbias.struct_refy");
     ref_y = cpl_parameter_get_int(p);
     */
    
    ref_llx = 1;
    ref_urx = sx;
    ref_lly = 1;
    ref_ury = sy;
    
    if (ref_x == -1) {
        ref_x = sx / 2;
    }
    
    if (ref_y == -1) {
        ref_y=sy/2;
    }
    
    ref_x = (ref_x>0) ? ref_x : 1;
    ref_y = (ref_y>0) ? ref_y : 1;
    ref_x = (ref_x<sx) ? ref_x : sx;
    ref_y = (ref_y<sy) ? ref_y : sy;
    espdr_qc_structure_region(master,
                              ref_llx, ref_lly, ref_urx, ref_ury,
                              ref_x, ref_y, 1, structx, structy);
    //espdr_qc_structure_region(master,ref_llx,ref_lly,ref_urx,ref_ury,ref_x,ref_y,2);
    
    return;
}

/* End of Andrea Modigliani code */

