/*
 * espdr_ccf.c
 *
 *  Created on: Mar 4, 2016
 *      Author: Alex Segovia
 *	
 *  Exactly copied from espdr_christophe.c	
 */

/*
 * $Author: asegovia $
 * $Date: 2016-03-04 14:48:13 $
 * $Revision: 1.16 $
 * $Name: not supported by cvs2svn $
 */


/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/
#include <espdr_ccf.h>

/*----------------------------------------------------------------------------
                              Functions code
 ----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the Cross-Correlation Function
 @param     ll                S2D wavelength matrix in barycentric frame
 @param     dll               S2D matrix of pixel sizes in wavelength units
 @param     flux              Input S2D spectrum (with blaze)
 @param     error             Input S2D error array
 @param     quality           Input S2D quality map
 @param     RV_table          Array of RV values on which to sample CCF (km/s)
 @param     mask              Line list for cross-correlation (wavelengths and 
			      contrasts)
 @param     mask_width        Width of mask holes in km/s
 @param     berv              Barycentric correction in km/s
 @param     bervmax           Maximum barycentric correction over the 
			      year (km/s)
 @param     CCF_flux_RE       CCF (returned)
 @param     CCF_error_RE      CCF error array (returned)
 @param     CCF_quality_RE    CCF quality map (returned)
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_CCF(cpl_image *ll,
                                 cpl_image *dll,
                                 cpl_image *flux,
                                 cpl_image *error,
                                 cpl_image *blaze,
                                 cpl_image *quality,
                                 cpl_array *RV_table,
                                 cpl_table *mask,
                                 double mask_width,
                                 double berv,
                                 double bervmax,
                                 cpl_image **CCF_flux_RE,
                                 cpl_image **CCF_error_RE,
                                 cpl_image **CCF_quality_RE ) {
    
    int i, j, k, order, pir, imin, imax, index, index1, index2, index3, index_center;
    double c = 299792.458;
    double llmin, llmax, llcenter, llstart, llstop;
    //double Sc, Nc;
    double contrast, w;
    double *ccf_flux, *ccf_error;
    int *ccf_quality;
    int nx_s2d = cpl_image_get_size_x(flux);
    int ny_s2d = cpl_image_get_size_y(flux);
    int n_mask = cpl_table_get_nrow(mask);
    int nx_ccf = cpl_array_get_size(RV_table);

    cpl_image *ll2, *dll2;
    
    ccf_flux = cpl_image_get_data_double(*CCF_flux_RE);
    ccf_error = cpl_image_get_data_double(*CCF_error_RE);
    ccf_quality = cpl_image_get_data_int(*CCF_quality_RE);

    espdr_msg_debug("Starting compute_CCF");
    espdr_ensure(flux == NULL, CPL_ERROR_NULL_INPUT, "flux input image is NULL");
    espdr_ensure(ll == NULL, CPL_ERROR_NULL_INPUT, "ll input image is NULL");
    espdr_ensure(dll == NULL, CPL_ERROR_NULL_INPUT, "dll input image is NULL");
    espdr_ensure(error == NULL, CPL_ERROR_NULL_INPUT, "error input image is NULL");
    espdr_ensure(blaze == NULL, CPL_ERROR_NULL_INPUT, "blaze input image is NULL");
    espdr_ensure(quality == NULL, CPL_ERROR_NULL_INPUT, "quality input image is NULL");
    espdr_ensure(RV_table == NULL, CPL_ERROR_NULL_INPUT, "RV_table input array is NULL");
    espdr_ensure(mask == NULL, CPL_ERROR_NULL_INPUT, "input mask table is NULL");
    espdr_ensure(n_mask == 0, CPL_ERROR_INCOMPATIBLE_INPUT, "input mask has no data");
    
    /*
    espdr_msg("ASE DEBUG : ll  size_x=%lld size_y=%lld",
              cpl_image_get_size_x(ll),
              cpl_image_get_size_y(ll));
    espdr_msg("ASE DEBUG : dll  size_x=%lld size_y=%lld",
              cpl_image_get_size_x(dll),
              cpl_image_get_size_y(dll));
    espdr_msg("ASE DEBUG : flux  size_x=%lld size_y=%lld",
              cpl_image_get_size_x(flux),
              cpl_image_get_size_y(flux));
    espdr_msg("ASE DEBUG : error size_x=%lld size_y=%lld",
              cpl_image_get_size_x(error),
              cpl_image_get_size_y(error));
    espdr_msg("ASE DEBUG : blaze size_x=%lld size_y=%lld",
              cpl_image_get_size_x(blaze),
              cpl_image_get_size_y(blaze));
    espdr_msg("ASE DEBUG : qual  size_x=%lld size_y=%lld",
              cpl_image_get_size_x(quality),
              cpl_image_get_size_y(quality));
    */
    
    dll2 = cpl_image_divide_scalar_create(dll,2.);
    ll2 = cpl_image_subtract_create(ll,dll2);   
   
    for (order = 1; order <= ny_s2d; order++) {
        
        imin = 1; imax = nx_s2d;
        
        while ((imin < nx_s2d) && (cpl_image_get(quality,imin,order,&pir) != 0)) imin++;
        while ((imax > 1) && (cpl_image_get(quality,imax,order,&pir) != 0)) imax--;
        
        if (imin >= imax) continue;
        
        llmin = cpl_image_get(ll,imin+1,order,&pir)/(1.+berv/c)*
                            (1.+bervmax/c)/(1.+cpl_array_get(RV_table,0,NULL)/c);
        llmax = cpl_image_get(ll,imax-1,order,&pir)/(1.+berv/c)*
                            (1.-bervmax/c)/(1.+cpl_array_get(RV_table,nx_ccf-1,NULL)/c);

        //Sc = fmax(cpl_image_get_median_window(flux,imin,order,imax,order),0.);
        
        imin = 0; imax = n_mask-1;
        
        if (!cpl_table_has_column(mask, "lambda")) {
            espdr_msg_warning("The input mask does not have the column 'lambda', exiting");
            return (CPL_ERROR_INCOMPATIBLE_INPUT);
        }
        
        if (!cpl_table_has_column(mask, "contrast")) {
            espdr_msg_warning("The input mask does not have the column 'contrast', exiting");
            return (CPL_ERROR_INCOMPATIBLE_INPUT);
        }
        
        while ((imin < n_mask) &&
              (cpl_table_get(mask,"lambda",imin,NULL) < (llmin+0.5*mask_width/c*llmin))) {
            imin++;
        }
        
        while ((imax >= 0) &&
              (cpl_table_get(mask,"lambda",imax,NULL) > (llmax-0.5*mask_width/c*llmax))) {
            imax--;
        }
        
        for (i = imin; i <= imax; i++) {

            /*
             llstart = cpl_table_get(mask,"lambda",i,NULL)*(1.+cpl_array_get(RV_table,0,NULL)/c);
             llstop = cpl_table_get(mask,"lambda",i,NULL)*(1.+cpl_array_get(RV_table,nx_ccf-1,NULL)/c);
             index1 = 1;
             while (cpl_image_get(ll,index1,order,&pir) < llstart) index1++;
             index2 = index1;
             while (cpl_image_get(ll,index2,order,&pir) < llstop) index2++;
             Nc = cpl_image_get_median_window(error,index1,order,index2,order);
             */
            
            llcenter = cpl_table_get(mask,"lambda",i,NULL)*
            (1.+cpl_array_get(RV_table,nx_ccf/2,NULL)/c);
            
            index_center = 1;
            
            while (cpl_image_get(ll,index_center,order,&pir) < llcenter) index_center++;
            
            contrast = cpl_table_get(mask,"contrast",i,NULL);
            
            /*if (Nc > 0.) {
             //w = Sc/Nc/Nc*contrast*contrast;
             w = 1.0/Nc/Nc*contrast*contrast;
             } else {
             w = 0.;
             }*/
            
            w = contrast*contrast;
            
            for (j = 0; j < nx_ccf; j++) {

               llcenter = cpl_table_get(mask,"lambda",i,NULL)*
                                        (1.+cpl_array_get(RV_table,j,NULL)/c);
               llstart = llcenter - 0.5 * mask_width/c * llcenter;
               llstop = llcenter + 0.5 * mask_width/c  *llcenter;
               index1 = 1;
               while (cpl_image_get(ll2,index1,order,&pir) < llstart) index1++;
                
               index2 = index1;
               while (cpl_image_get(ll2,index2,order,&pir) < llcenter) index2++;
                
               index3 = index2;
               while (cpl_image_get(ll2,index3,order,&pir) < llstop) index3++;
                
               k = nx_ccf*(order-1)+j;
                
               for (index = index1; index < index3; index++) {
                   ccf_flux[k] += w*cpl_image_get(flux,index,order,&pir)/
                                    cpl_image_get(blaze,index,order,&pir)*
                                    cpl_image_get(blaze,index_center,order,&pir);
               }
                    
               ccf_flux[k] += w * cpl_image_get(flux,index1-1,order,&pir)*
                                (cpl_image_get(ll2,index1,order,&pir)-llstart)/
                                cpl_image_get(dll,index1-1,order,&pir)/
                                cpl_image_get(blaze,index1-1,order,&pir)*
                                cpl_image_get(blaze,index_center,order,&pir);
                
               ccf_flux[k] -= w * cpl_image_get(flux,index3-1,order,&pir)*
                                (cpl_image_get(ll2,index3,order,&pir)-llstop)/
                                cpl_image_get(dll,index3-1,order,&pir)/
                                cpl_image_get(blaze,index3-1,order,&pir)*
                                cpl_image_get(blaze,index_center,order,&pir);
                
               ccf_error[k] += w * w * cpl_image_get(error,index2-1,order,&pir)*
                                cpl_image_get(error,index2-1,order,&pir);
                
               //ccf_quality[k] += cpl_image_get(quality,index2-1,order,&pir);
                ccf_quality[k] = ccf_quality[k] | (int)(cpl_image_get(quality,index2-1,order,&pir));
            }
        }
    }

    for (j = 0; j < nx_ccf; j++) {
        for (order = 0; order < ny_s2d; order++) {
            ccf_flux[nx_ccf*ny_s2d+j] += ccf_flux[nx_ccf*order+j];
            ccf_error[nx_ccf*ny_s2d+j] += ccf_error[nx_ccf*order+j];
            ccf_quality[nx_ccf*ny_s2d+j] += ccf_quality[nx_ccf*order+j];
        }
    }

    cpl_image_power(*CCF_error_RE,0.5);
    
    cpl_image_delete(ll2);
    cpl_image_delete(dll2);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_ccf(cpl_array *RV_table,
                             int fibres_processed,
                             cpl_image *CCF_flux,
                             cpl_image *CCF_error,
                             cpl_image *CCF_quality,
                             double *asym_RE,
                             double *asym_err_RE,
                             double *bispan_RE,
                             double *bispan_err_RE,
                             cpl_table **residuals_RE,
                             espdr_ngauss_result *g_res_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    cpl_image *CCF_last_order_flux;
    cpl_image *CCF_last_order_error;
    cpl_image *CCF_last_order_quality;
    
    double *CCF_last_order_data_flux;
    double *CCF_last_order_data_error;
    double *CCF_slope;
    int *CCF_last_order_data_quality;
    
    double *x0;
    double CCF_weight_sum=0.0;
    
    int CCF_x,CCF_y;
    
    int RV_length = cpl_array_get_size(RV_table);
    size_t min_index;
    
    espdr_ngauss_data *g_data = NULL;
    
    g_data = (espdr_ngauss_data *)cpl_malloc(sizeof(espdr_ngauss_data));
    g_data->n = RV_length;
    g_data->m = 1;
    g_data->delta = 0.0;
    g_data->x = (double *)cpl_calloc (RV_length , sizeof(double));
    g_data->y = (double *)cpl_calloc (RV_length , sizeof(double));
    g_data->err = (double *)cpl_calloc (RV_length , sizeof(double));
    g_data->fit = (double *)cpl_calloc (RV_length , sizeof(double));
    
    x0 = (double *)cpl_malloc(sizeof(double));
    
    double sigma = 0.;
    double *model = (double *)cpl_calloc (RV_length , sizeof(double));
    
    *residuals_RE = cpl_table_new(RV_length);
    
    my_error = cpl_table_new_column(*residuals_RE,
                                    "rv",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed: %s",
                 cpl_error_get_message_default(my_error));
    my_error = cpl_table_set_column_unit(*residuals_RE,
                                         "rv",
                                         "[km/s]");
    
    my_error = cpl_table_new_column(*residuals_RE,
                                    "residuals",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed: %s",
                 cpl_error_get_message_default(my_error));
    my_error = cpl_table_set_column_unit(*residuals_RE,
                                         "residuals",
                                         "[km/s]");
    
    my_error = cpl_table_new_column(*residuals_RE,
                                    "error",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed: %s",
                 cpl_error_get_message_default(my_error));
    my_error = cpl_table_set_column_unit(*residuals_RE,
                                         "error",
                                         "[km/s]");
    
    my_error = cpl_table_new_column(*residuals_RE,
                                    "quality",
                                    CPL_TYPE_INT);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed: %s",
                 cpl_error_get_message_default(my_error));
    my_error = cpl_table_set_column_unit(*residuals_RE,
                                         "quality",
                                         "");
        
    g_res_RE->k = (double *)cpl_malloc (sizeof(double));
    g_res_RE->sig_k = (double *)cpl_malloc (sizeof(double));
    g_res_RE->x0 = (double *)cpl_malloc (sizeof(double));
    g_res_RE->sig_x0 = (double *)cpl_malloc (sizeof(double));
    g_res_RE->fwhm = (double *)cpl_malloc (sizeof(double));
    g_res_RE->sig_fwhm = (double *)cpl_malloc (sizeof(double));

    //espdr_msg("Fitting the CCF...");
    
    if (CCF_flux != NULL && CCF_error != NULL) {
        
        CCF_x = cpl_image_get_size_x(CCF_flux);
        CCF_y = cpl_image_get_size_y(CCF_flux);
        
        CCF_last_order_flux = cpl_image_extract(CCF_flux,
                                                1,CCF_y,
                                                CCF_x,CCF_y);
        
        CCF_last_order_data_flux =
        cpl_image_get_data_double(CCF_last_order_flux);
        
        CCF_last_order_error = cpl_image_extract(CCF_error,
                                                 1,CCF_y,
                                                 CCF_x,CCF_y);
        
        CCF_last_order_data_error =
        cpl_image_get_data_double(CCF_last_order_error);
        
        CCF_last_order_quality = cpl_image_extract(CCF_quality,
                                                   1,CCF_y,
                                                   CCF_x,CCF_y);
        
        CCF_last_order_data_quality =
        cpl_image_get_data_int(CCF_last_order_quality);
        
        for(int i = 0; i < RV_length; i++) {
            g_data->x[i]    = cpl_array_get(RV_table,i,NULL);
            g_data->y[i]    = CCF_last_order_data_flux[i];
            //g_data->err[i]  = CCF_last_order_data_error[i];
            g_data->err[i]  = 1.0;       // Fitting CCF with uniform weights
        }
        
        min_index = gsl_stats_min_index(g_data->y,1,RV_length);
        
        x0[0] = g_data->x[(int)min_index];
        
        my_error = espdr_fit_Ngauss(g_data, x0, 10.0, g_res_RE, 0, 0);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_Ngauss failed for fibre %s",
                     cpl_error_get_message_default(my_error));
        
        g_res_RE->fwhm[0] = fabs(g_res_RE->fwhm[0]);    // Forcing positivity of FWHM
        
        /* CCF slope calculation */
        
        CCF_slope = (double *)cpl_calloc (RV_length , sizeof(double));
        
        for(int i = 1; i < RV_length-1; i++) {
            CCF_slope[i] =
            (CCF_last_order_data_flux[i+1]-CCF_last_order_data_flux[i-1])/
            (cpl_array_get(RV_table,i+1,NULL)-cpl_array_get(RV_table,i-1,NULL));
        }
        
        /* CCF error calculation */
        
        for(int i = 1; i < RV_length-1; i++) {
            CCF_weight_sum += pow(CCF_slope[i]/CCF_last_order_data_error[i],2.0);
        }
        
        cpl_free(CCF_slope);
        
        g_res_RE->sig_x0[0] = 1.0/sqrt(CCF_weight_sum);
        g_res_RE->sig_fwhm[0] = 1.0/sqrt(CCF_weight_sum)*2.0;
        if (g_res_RE->fwhm[0] != 0.) {
            g_res_RE->sig_k[0] = fabs(1.0/sqrt(CCF_weight_sum)*2.0/g_res_RE->fwhm[0]*g_res_RE->k[0]);
        } else {
            g_res_RE->sig_k[0] = 0.;
        }
        
        /* model and residuals table calculation */
                
        if ( g_res_RE->fwhm[0] != 0. && !isnan(g_res_RE->fwhm[0]) && !isinf(g_res_RE->fwhm[0]) ) {
            sigma = g_res_RE->fwhm[0]/2./sqrt(2.*log(2.));
            for (int i = 0; i < RV_length; i++)
                model[i] = g_res_RE->c + g_res_RE->k[0]*exp(-1.*(g_data->x[i]-g_res_RE->x0[0])*
                                                            (g_data->x[i]-g_res_RE->x0[0])/2./sigma/sigma);
        } else {
            for (int i = 0; i < RV_length; i++)
                model[i] = 0.;
        }
        
        for (int i = 0; i < RV_length; i++) {
            cpl_table_set(*residuals_RE,
                          "rv",i,g_data->x[i]);
            
            cpl_table_set(*residuals_RE,
                          "residuals",i,g_data->y[i]-model[i]);
            
            cpl_table_set(*residuals_RE,
                          "error",i,CCF_last_order_data_error[i]);
            
            cpl_table_set(*residuals_RE,
                          "quality",i,CCF_last_order_data_quality[i]);
        }
        
        espdr_msg("CCF fitted, RV = %.10f", g_res_RE->x0[0]);
        
        /* Compute the CCF flux asymmetry */
        
        espdr_msg("Computing the flux assymetry...");
        
        int il = 0, im = 0, ir = 0;
        double dx = (g_data->x[1] - g_data->x[0])/2.;
        while ((il < (RV_length-1)) && (g_data->x[il] < (g_res_RE->x0[0] - 1.5*g_res_RE->fwhm[0] + dx))) il++;
        while ((im < (RV_length-1)) && (g_data->x[im] < (g_res_RE->x0[0] + dx))) im++;
        while ((ir < (RV_length-1)) && (g_data->x[ir] < (g_res_RE->x0[0] + 1.5*g_res_RE->fwhm[0] + dx))) ir++;
        
        double suml = 0., sumr = 0., sumlerr = 0., sumrerr = 0.;
        if ( (il > 0) && (im > il) && (ir > im) && (ir < (RV_length-1)) ) {
            
            for (int i = il; i < (im-1); i++) {
                suml += g_data->y[i]-model[i];
                sumlerr += CCF_last_order_data_error[i]*CCF_last_order_data_error[i];
            }
            suml += (g_data->y[il-1]-model[il-1])*(g_data->x[il-1] + dx - g_res_RE->x0[0] + 1.5*g_res_RE->fwhm[0])/2./dx;
            suml += (g_data->y[im-1]-model[im-1])*(g_res_RE->x0[0] - g_data->x[im-1] + dx)/2./dx;
            
            for (int i = im; i < (ir-1); i++) {
                sumr += g_data->y[i]-model[i];
                sumrerr += CCF_last_order_data_error[i]*CCF_last_order_data_error[i];
            }
            sumr += (g_data->y[im-1]-model[im-1])*(g_data->x[im-1] + dx - g_res_RE->x0[0])/2./dx;
            sumr += (g_data->y[ir-1]-model[ir-1])*(g_res_RE->x0[0] + 1.5*g_res_RE->fwhm[0] - g_data->x[ir-1] + dx)/2./dx;
        }
        
        *asym_RE = (sumr - suml) * 2. * dx / g_res_RE->c;
        *asym_err_RE = sqrt(sumlerr + sumrerr) * 2. * dx / g_res_RE->c;
        
        espdr_msg("Flux assymetry computed: %.10f (err: %.10f)",
                  *asym_RE, *asym_err_RE);
        
        /* Compute the CCF bisector span */
        
        espdr_msg("Computing the bissector span...");
        
        espdr_ngauss_result *g_res_bis = (espdr_ngauss_result *)cpl_malloc (sizeof(espdr_ngauss_result));
        g_res_bis->k = (double *)cpl_malloc (sizeof(double));
        g_res_bis->sig_k = (double *)cpl_malloc (sizeof(double));
        g_res_bis->x0 = (double *)cpl_malloc (sizeof(double));
        g_res_bis->sig_x0 = (double *)cpl_malloc (sizeof(double));
        g_res_bis->fwhm = (double *)cpl_malloc (sizeof(double));
        g_res_bis->sig_fwhm = (double *)cpl_malloc (sizeof(double));
        
        // Higher errors are set on the bottom of the ccf fit, to fit only the top part
        for (int j = 0; j < RV_length; j++) {
            double arg_err = (model[j] - g_res_RE->c)/g_res_RE->k[0];
            double sigma_err = 0.4 / 2.3548;
            double gauss_err = 1.0 + 50.0 * exp(-(arg_err-1.0)*(arg_err-1.0)/2.0/sigma_err/sigma_err);
            //espdr_msg("RV[%d] = %f\tmodel[%d] = %f\t(model-c)/k0 = %.10f\tgauss: %f",
            //          j, cpl_array_get(RV_table,j,NULL), j, model[j], arg_err, gauss_err);
            
            g_data->err[j] = gauss_err;
        }
        
        my_error = espdr_fit_Ngauss(g_data, x0, 10.0, g_res_bis, 0, 0);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_Ngauss failed for fibre %s",
                     cpl_error_get_message_default(my_error));

        double RV_top = g_res_bis->x0[0];
        
        // Higher errors are set on the top of the ccf fit, to fit only the bottom part
        // The continuum (< 0.01 of contrast) is forced to 1.0
        for (int j = 0; j < RV_length; j++) {
            double arg_err = (model[j] - g_res_RE->c)/g_res_RE->k[0];
            double sigma_err = 0.25 / 2.3548;
            double gauss_err = 1.0 + 50.0 * exp(-(arg_err-0.25)*(arg_err-0.25)/2.0/sigma_err/sigma_err);
            
            g_data->err[j] = gauss_err;
            
            if (arg_err < 0.01) {
                g_data->err[j] = 1.0;
            }
            //espdr_msg("RV[%d] = %f\tmodel[%d] = %f\t(model-c)/k0 = %.10f\tgauss: %f\tg_data_err = %.10f",
            //          j, cpl_array_get(RV_table,j,NULL), j, model[j], arg_err, gauss_err, g_data->err[j]);
        }
        
        my_error = espdr_fit_Ngauss(g_data, x0, 10.0, g_res_bis, 0, 0);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_Ngauss failed for fibre %s",
                     cpl_error_get_message_default(my_error));
        
        double RV_bottom = g_res_bis->x0[0];
        
        if (!isnan(RV_top) && !isinf(RV_top) && !isnan(RV_bottom) && !isinf(RV_bottom)) {
            *bispan_RE = RV_top - RV_bottom;
            *bispan_err_RE = 2.0 * g_res_RE->sig_x0[0];
        } else {
            *bispan_RE = 0.;
            *bispan_err_RE = 0.;
        }
        
        espdr_msg("Bissector span computed: %.10f (err: %.10f)",
                  *bispan_RE, *bispan_err_RE);
        
        cpl_free(g_res_bis->k);
        cpl_free(g_res_bis->sig_k);
        cpl_free(g_res_bis->x0);
        cpl_free(g_res_bis->sig_x0);
        cpl_free(g_res_bis->fwhm);
        cpl_free(g_res_bis->sig_fwhm);
        cpl_free(g_res_bis);
        
    }
    
    cpl_free(x0);
    cpl_free(model);
    cpl_free(g_data->x);
    cpl_free(g_data->y);
    cpl_free(g_data->err);
    cpl_free(g_data->fit);
    cpl_free(g_data);
    
    return cpl_error_get_code();
}

