/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-10-23 14:47:28 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#include <espdr_overscan.h>
#include <espdr_hdrl_func.h>

#if defined (_OPENMP) && !defined( __APPLE__)
  #define HDRL_OMP(x) _Pragma (#x)
#else
  #define HDRL_OMP(x)
#endif
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for OVSC input parameters
 @param         recipe_id	recipe identifier
 @param         param_list	list of input parameters
 @param[out]    p			input OVSC parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_OVSC_create(const char* recipe_id,
										  cpl_parameterlist *param_list,
										  espdr_OVSC_param *p) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	
	/* Fill the parameter list */
    //p = NULL;
	cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    double ksigma_min = KSIGMA_MIN;
    double ksigma_max = KSIGMA_MAX;
    int sig_clip_max_iter_min = SIGMA_CLIP_MAX_ITER_MIN;
    int sig_clip_max_iter_max = SIGMA_CLIP_MAX_ITER_MAX;

    /* check parameters */
    espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");

    /* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, param_list,
    		"ovsc_sig_clip_method",
			p->ovsc_sigma_clipping_method,
			"method for sigma clipping in OVSC, can be mean or median");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
    		"Error adding parameter ovsc_sigma_clipping_method to the list");

    sprintf(comment,
    		"ksigma for sigma clipping in OVSC, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);
    error_got = espdr_parameters_new_range_double(recipe_id, param_list,
    		"ovsc_ksigma", p->ovsc_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
    		"Error adding parameter ovsc_ksigma to the list");

    sprintf(comment,
    		"maximal number of iterations in OVSC, must be between: %d and %d",
			sig_clip_max_iter_min, sig_clip_max_iter_max);
    error_got = espdr_parameters_new_range_int(recipe_id, param_list,
    		"ovsc_max_iter", p->ovsc_max_iter,
			SIGMA_CLIP_MAX_ITER_MIN,
			SIGMA_CLIP_MAX_ITER_MAX,
			comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
    		"Error adding parameter ovsc_max_iter to the list");

	return(CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 * AMO: added
 @brief     Allocate memory and init the overscan parameters structure
 @param     recipe_id   recipe ID
 @param     param_list  list of parameters
 @return    OVSC parameters structure if OKK or NULL
 */
/*----------------------------------------------------------------------------*/

espdr_OVSC_param* espdr_parameters_OVSC_init(const char* recipe_id,
                                             cpl_parameterlist* param_list) {
    
    if (recipe_id == NULL) return NULL;
    
    espdr_OVSC_param *OVSC_param =
    (espdr_OVSC_param *)cpl_malloc(sizeof(espdr_OVSC_param));
    espdr_parameters_OVSC_get(recipe_id, param_list, OVSC_param);
    
    /* TODO: AMO this print should be optional */
    espdr_parameters_OVSC_print(OVSC_param);
    
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        espdr_msg_error("Error during OVSC parameters initialisation");
        espdr_parameters_OVSC_delete(OVSC_param);
        return NULL;
    }
    return OVSC_param;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for OVSC input parameters
 @param    p	input OVSC parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_OVSC_delete(espdr_OVSC_param* p) {
	
    cpl_free((void *)p->ovsc_sigma_clipping_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Read the overscan parameters
 @param     recipe ID
 @param     list of parameters
 @param     OVSC parameters data structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_OVSC_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_OVSC_param *OVSC_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
                 "Parameters list is NULL");
	
	/* Fill the structure */
    OVSC_param->ovsc_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                         param_list,
                                                                         "ovsc_sig_clip_method");
    
    OVSC_param->ovsc_ksigma = espdr_parameters_get_double(recipe_id,
                                                          param_list,
                                                          "ovsc_ksigma");
    
    OVSC_param->ovsc_max_iter = espdr_parameters_get_int(recipe_id,
                                                         param_list,
                                                         "ovsc_max_iter");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief print the OVSC parameters
 @param OVSC_param OVSC parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_OVSC_print(espdr_OVSC_param *OVSC_param) {
	
	espdr_msg("\tOVSC parameters: ");
    espdr_msg("\t\tOVSC sigma clipping method = %s",
              OVSC_param->ovsc_sigma_clipping_method);
    espdr_msg("\t\tOVSC ksigma = %.2f",
              OVSC_param->ovsc_ksigma);
    espdr_msg("\t\tOVSC max iter = %d",
              OVSC_param->ovsc_max_iter);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Prepare to compute the overscan and create QC keywords
 @param     master_bias_imagelist   input image, outputs in the imagelist
 @param     CCD_geom                CCD_geometry structure
 @param     OVSC_param              overscan parameters stucture
 @param     qc_kws                  KWs names
 @param     inst_config             instrument config
 @param     treat_frames_nb         number of treated frames
 @param[out] overscan_imagelist_RE  computed overscan image
 @param[out] keywords_RE            primary FITS headers
 @param[out] RON_vector_OV_RE       computed RON
 @param     use_hdrl                HDRL use flag
 @return   CPL_ERROR_NONE iff OK

 TODO: define specific functions to handle the creation/initialisation/delete
 for CCD geometry and OVSC (and any other kind of) parameters.
 */
/*----------------------------------------------------------------------------*/
/* AMO added to (eventually) use HDRL */
cpl_error_code espdr_overscan_prepare(const cpl_imagelist *master_bias_imagelist,
                                      const espdr_CCD_geometry *CCD_geom,
                                      espdr_qc_keywords *qc_kws,
                                      espdr_inst_config *inst_config,
                                      const int treated_frames_nb,
                                      cpl_imagelist **overscan_imagelist_RE,
                                      cpl_propertylist **keywords_RE,
                                      double *RON_vector_OV_RE,
                                      const int use_hdrl) {

    int i, j, k, index;
    /* ext_nr * output_nx * output_ny */
    int total_cosmics_nb = 0;
    int cosmics_nb = 0;
    cpl_image *overscan_image = NULL;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    cpl_error_code my_error = CPL_ERROR_NONE;

    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {

                if(use_hdrl) {
                    /* use HDRL based overscan determination */
                    my_error = espdr_hdrl_overscan(cpl_imagelist_get_const
                                                   (master_bias_imagelist,index),
                                                   i, j, k,
                                                   CCD_geom, inst_config,
                                                   &overscan_image,
                                                   &RON_vector_OV_RE[index],
                                                   &cosmics_nb);
                } else {
                    my_error = espdr_overscan(cpl_imagelist_get_const
                                              (master_bias_imagelist, index),
                                              i, j, k,
                                              CCD_geom, inst_config,
                                              &overscan_image,
                                              &RON_vector_OV_RE[index],
                                              &cosmics_nb);
                }
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                               "overscan failed: %s",
                               cpl_error_get_message_default(my_error));

                espdr_msg("RON overscan[%d]: %lf",
                          index, RON_vector_OV_RE[index]);
                espdr_msg_debug("cosmics nb = %d", cosmics_nb);

                /* Calculate the mean of each OVERSCAN image */

                espdr_ensure(overscan_image == NULL, CPL_ERROR_NULL_INPUT,
                             "Overscan image is NULL");

                double real_mean = 0.0;

                real_mean = cpl_image_get_mean(overscan_image);
                espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                             cpl_error_get_code(),
                             "cpl_image_get_mean() failed: %s",
                             cpl_error_get_message());
                espdr_msg("Mean of the overscan image is %lf", real_mean);

                total_cosmics_nb = total_cosmics_nb + cosmics_nb;

                /* Save the result in an imagelist */
                my_error = cpl_imagelist_set(*overscan_imagelist_RE,
                                             cpl_image_duplicate(overscan_image),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed: %s",
                             cpl_error_get_message_default(my_error));

                cpl_image_delete(overscan_image);

                index++;
            }
        }
    }

    espdr_msg("Adding OVSC_RON keyword");
    /* Add OVSC_RON keyword */
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_ovsc_ron_kw_part,
                                                  inst_config->prefix,
                                                  i, j, k);
                sprintf(comment, "OVSC RON[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RON_vector_OV_RE[index] *
                                                    sqrt(treated_frames_nb),
                                                    comment,
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));

                cpl_free(new_keyword);
                
                index++;
            }
        }
    }

    espdr_msg("Adding OVSC_COSMICS keyword");
    /* Add OVSC_COSMICS keyword */
    espdr_msg_debug("total cosmics nb in overscan is %d", total_cosmics_nb);

    my_error = espdr_keyword_add_int(qc_kws->qc_ovsc_outliers_kw,
                                     total_cosmics_nb,
                                     "Total nb of cosmics in OVSC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword %s to the propertylist failed: %s",
                 qc_kws->qc_ovsc_outliers_kw,
                 cpl_error_get_message_default(my_error));

    espdr_msg("OVSC COSMICS NB is: %d",
              cpl_propertylist_get_int(*keywords_RE,
                                       qc_kws->qc_ovsc_outliers_kw));

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Compute the horizontal overscan
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_get_oscan_horizontal(const int data_size_y,
                                                 const int data_size_x,
                                                 const int reject_zeroMAD,
                                                 const double forced_MAD,
                                                 const int used_real_nx,
                                                 double *RON_median,
                                                 const cpl_image *pre_overscan,
                                                 const espdr_inst_config *inst_config,
                                                 double *overscan_vector,
                                                 double *std_dev_data,
                                                 int *total_cosmics_RE) {

	double reject_low;
	double reject_high;
	double stddev;
	double mean;
	double line_overscan;
	int ovsc_point = 0;
	cpl_image* pre_overscan_line;
	cpl_vector* data_vector_wrapped;
	int line;
	int i;

	for (line = 0; line < data_size_y; line++) {
		pre_overscan_line = cpl_image_extract(pre_overscan, 1, line + 1,
                                              data_size_x, line + 1);

		double *data_vector = cpl_image_get_data_double(pre_overscan_line);
		data_vector_wrapped = cpl_vector_wrap(data_size_x,data_vector);

		if (strcmp(inst_config->ovsc_sigma_clipping_method, "mean") == 0) {
			espdr_sig_clip_method_mean(data_vector_wrapped,
					inst_config->ovsc_ksigma, inst_config->ovsc_ksigma,
					data_size_x / 2.0, reject_zeroMAD, forced_MAD, &line_overscan,
					&std_dev_data[line], total_cosmics_RE, &mean, &stddev,
					&reject_low, &reject_high);
		} else {
			espdr_sig_clip_method_median(data_vector_wrapped,
					inst_config->ovsc_ksigma, inst_config->ovsc_ksigma,
					data_size_x / 2.0, reject_zeroMAD, forced_MAD, &line_overscan,
					&std_dev_data[line], total_cosmics_RE, &mean, &stddev,
					&reject_low, &reject_high);
		}

        /*
		espdr_sig_clip(data_vector_wrapped,
				inst_config->ovsc_ksigma, inst_config->ovsc_ksigma,
				inst_config->ovsc_sigma_clipping_method, data_size_x / 2.0,
				reject_zeroMAD, forced_MAD, &line_overscan,
				&std_dev_data[line], total_cosmics_RE, &mean, &stddev,
				&reject_low, &reject_high);
				*/

		/*
			 espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			 "espdr_sig_clip failed");
		 */
		cpl_vector_unwrap(data_vector_wrapped);
		/* for the whole line of real pixels, the overscan value
			 equals the pre-overscan mean */
		for (i = 0; i < used_real_nx; i++) {
			overscan_vector[ovsc_point] = line_overscan;
			ovsc_point++;
		}
		cpl_image_delete(pre_overscan_line);
	}

	/* RON contains the mean of all the (prescan,overscan) standard deviations.
	 There is one RON per output */
	cpl_vector *std_dev_vector = cpl_vector_wrap(data_size_y, std_dev_data);
	*RON_median = cpl_vector_get_median(std_dev_vector);
	cpl_vector_unwrap(std_dev_vector);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Compute the horizontal overscan
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code espdr_get_oscan_vertical(const int data_size_x,
                                               const int data_size_y,
                                               const int reject_zeroMAD,
                                               const double forced_MAD,
                                               const int used_real_ny,
                                               double* RON_median,
                                               const cpl_image* pre_overscan,
                                               const espdr_inst_config* inst_config,
                                               double* overscan_vector,
                                               double* std_dev_data,
                                               int* total_cosmics_RE) {

	int ovsc_point = 0;
	double line_overscan;
	double mean;
	double stddev;
	double reject_low;
	double reject_high;
	cpl_image* pre_overscan_line;
	for (int line = 0; line < data_size_x; line++) {
        pre_overscan_line = cpl_image_extract(pre_overscan, line + 1, 1,
                                              line + 1, data_size_y);
        
		double* data_vector = cpl_image_get_data_double(pre_overscan_line);
        cpl_vector* data_vector_wrapped = cpl_vector_wrap(data_size_y,
                                                          data_vector);
        cpl_error_code my_error = espdr_sig_clip(data_vector_wrapped,
                                                 inst_config->ovsc_ksigma,
                                                 inst_config->ovsc_ksigma,
                                                 inst_config->ovsc_sigma_clipping_method,
                                                 data_size_y / 2.0,
                                                 reject_zeroMAD, forced_MAD,
                                                 &line_overscan,
                                                 &std_dev_data[line],
                                                 total_cosmics_RE, &mean, &stddev,
                                                 &reject_low, &reject_high);
		
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
        
		cpl_vector_unwrap(data_vector_wrapped);
		/* for the whole line of real pixels, the overscan value
			 equals the pre-overscan mean */
		for (int i = 0; i < used_real_ny; i++) {
			overscan_vector[ovsc_point] = line_overscan;
			ovsc_point++;
		}

		cpl_image_delete(pre_overscan_line);
	}
	/* RON contains the mean of all the (prescan,overscan) standard deviations.
	 There is one RON per output */
	cpl_vector* std_dev_vector = cpl_vector_wrap(data_size_y, std_dev_data);
	*RON_median = cpl_vector_get_median(std_dev_vector);
	cpl_vector_unwrap(std_dev_vector);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the overscan
 @param         master_bias_image       one output input image
 @param         ext_no                  extension number
 @param         out_x                   x position output port
 @param         out_y                   y position output port
 @param         CCD_geom                CCD_geometry structure
 @param         OVSC_param              overscan parameters stucture
 @param[out]    overscan_image_RE       computed overscan image (returned)
 @param[out]    RON_RE                  computed RON (returned)
 @param[out]    total_cosmics_RE        computed cosmics number (returned)
 @return    CPL_ERROR_NONE iff OK
 TODO: This function is way too complex. Should be simplified (re-factored)
 1) debug statements on CCD_geom should be moved to a sub-function.
 Can one use the function that print the whole CCD structure content
 (eventually generalizing that if needed)?
 2) the code that is part of horizontal case seems to be similar to the one
 for vertical case one may factor that out eventually generalizing it
 3) It is not clear why a result is obtained with a cpl_image_wrap_double and
 not a cpl_image_extract.
 4) It is not clear why uses a vector to store image data?
 Cannot you use statistic on images or wrap a double buffer to the image data area?
 5) The part on QC computation can be factored out.
 6) the total number of cosmic seems not to be a relevant parameter for overscan
 7) As the overscan is computed on a given single image, why should you need to
 specify the ext_no? That indeed is just used to extract the CCD geom values
 from the proper extension. Do we need to have different CCD geom values for
 different extensions? If that is related just to the fact that the CCD geom table
 is now defined as a table with different raws I think that is wrong, because
 one should rather have just a single raw and different FITS CCD geom tables for
 different instruments.
 --> DSO to AMO: it is not sure that all the extensions have the same structure,
                that's why the ext_no is relevant
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_overscan(const cpl_image *input_image,
                              const int ext_no,
                              const int out_x,
                              const int out_y,
                              const espdr_CCD_geometry *CCD_geom,
                              const espdr_inst_config *inst_config,
                              cpl_image **overscan_image_RE,
                              double *RON_RE,
                              int *total_cosmics_RE) {
	
    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
	int nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
	int ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
	

    int used_real_nx = real_urx - real_llx + 1;
    int used_real_ny = real_ury - real_lly + 1;
	int collumn_orientation = CCD_geom->exts[ext_no].coll_or;
	int correction_type = CCD_geom->exts[ext_no].corr_type;
	
	int pscan_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_llx;
    int pscan_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_lly;
    int pscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_urx;
    int pscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ury;
	int pscan_nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_nx;
	int pscan_ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ny;
	
    int oscan_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_llx;
    int oscan_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_lly;
    int oscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_urx;
    int oscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ury;
    int oscan_nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_nx;
    int oscan_ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ny;
	
	
    espdr_msg_debug("Starting espdr_overscan with");
    espdr_msg_debug("\tREAL llx = %d, lly = %d, urx = %d, ury = %d",
                       real_llx, real_lly, real_urx, real_ury);

    espdr_msg_debug("\tREAL nx = %d, ny = %d, coll = %d, corr = %d",
                    nx,ny,
                    CCD_geom->exts[ext_no].coll_or,
                    CCD_geom->exts[ext_no].corr_type);
    espdr_msg_debug("\tPSCAN llx = %d, lly = %d, urx = %d, ury = %d",
                    pscan_llx, pscan_lly, pscan_urx, pscan_ury);

    espdr_msg_debug("\tPSCAN nx = %d, ny = %d",
                    pscan_nx,pscan_ny);

    espdr_msg_debug("\tOSCAN llx = %d, lly = %d, urx = %d, ury = %d",
                    oscan_llx, oscan_lly, oscan_urx, oscan_ury);
    espdr_msg_debug("\tOSCAN nx = %d, ny = %d",
                    oscan_nx, oscan_ny);
    
    int horizontal;
    int data_size_x;
    int data_size_y;
    double *std_dev_data = NULL;
	/* the direction is set by collumn_orientation and correction_type */
	if (((collumn_orientation == 0) && (correction_type == 1)) || 
		((collumn_orientation == 1) && (correction_type == 0))) {
        horizontal = 1;
		/* average value calculated along the horizontal direction */
		/* data_size = length of the prescan + overscan */
		data_size_x = (pscan_urx - pscan_llx + 1) + (oscan_urx - oscan_llx + 1);
        data_size_y = used_real_ny;
		/* std_dev_data contains the standard deviation of 
		 the (prescan,overscan) line */
        /* vector length is the number of (prescan,overscan) lines */
		std_dev_data = (double *) cpl_calloc(data_size_y, sizeof(double));
		//espdr_msg("horizzontal 1");
		//espdr_msg("data_size_y=%d",data_size_y);
	} else {
		if (((collumn_orientation == 1) && (correction_type == 1)) || 
			((collumn_orientation == 0) && (correction_type == 0))) {
            horizontal = 0;
			/* average value calculated along the vertical direction */
			/* data_size = length of the prescan + overscan */
            data_size_x = used_real_nx;
            data_size_y = (pscan_ury - pscan_lly + 1) + (oscan_ury - oscan_lly + 1);
			/* std_dev_data contains the standard deviation of
			 the (prescan,overscan) column */
			std_dev_data = (double *) cpl_calloc(data_size_x, sizeof(double));
			//espdr_msg("horizzontal 0");
			//espdr_msg("data_size_x=%d",data_size_x);
		} else {
            espdr_ensure(0, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Wrong combination of collumn_orientation and correction_type, exiting");
		}
	}
    
    /* Merge the pre and overscan images into one */
	cpl_image *prescan, *overscan, *pre_overscan;
    pre_overscan = cpl_image_new(data_size_x, data_size_y, CPL_TYPE_DOUBLE);
    
    
    prescan = cpl_image_extract(input_image, pscan_llx, pscan_lly,
                                pscan_urx, pscan_ury);
    espdr_ensure(prescan == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "Prescan image resulting from extract is NULL");

    if (horizontal) {
        overscan = cpl_image_extract(input_image,
                                     pscan_urx + nx + oscan_llx, oscan_lly,
                                     pscan_urx + nx + oscan_urx, oscan_ury);
    } else {
        overscan = cpl_image_extract(input_image,
                                     oscan_llx, pscan_ury + ny + oscan_lly,
                                     oscan_urx, pscan_ury + ny + oscan_ury);
    }

    espdr_ensure(overscan == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "Overscan image resulting from extract is NULL");
    
    /*
    espdr_msg("whole pre-overscan data_size_x = %d, data_size_y = %d",
              data_size_x, data_size_y);
    espdr_msg("pscan: %d %d %d %d", pscan_llx, pscan_lly, pscan_urx, pscan_ury);
    */
    cpl_error_code my_error = CPL_ERROR_NONE;

    my_error = cpl_image_copy(pre_overscan, prescan, 1, 1);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_copy with prescan failed: %s",
                 cpl_error_get_message_default(my_error));

    if (horizontal) {
        my_error = cpl_image_copy(pre_overscan, overscan,
                                  (pscan_urx-pscan_llx+2), 1);
    } else {
        my_error = cpl_image_copy(pre_overscan, overscan,
                                  1, (pscan_ury-pscan_lly+2));
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_copy with overscan failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Overscan computation */
	/* overscan_vector is the whole real pixels vector */
	double* overscan_vector = (double *)cpl_calloc(used_real_nx*used_real_ny, sizeof(double));
	double RON_median;
	double forced_MAD = 0.1;
	int reject_zeroMAD = 1;
    //espdr_msg("ok0");
    if (horizontal) {
        my_error = espdr_get_oscan_horizontal(data_size_y, data_size_x,
                                              reject_zeroMAD, forced_MAD,
                                              used_real_nx, &RON_median,
                                              pre_overscan, inst_config,
                                              overscan_vector, std_dev_data,
                                              total_cosmics_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_get_oscan_horizontal() has failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_get_oscan_vertical(data_size_x, data_size_y,
                                            reject_zeroMAD, forced_MAD,
                                            used_real_ny, &RON_median,
                                            pre_overscan, inst_config,
                                            overscan_vector, std_dev_data,
                                            total_cosmics_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_get_oscan_vertical() has failed: %s",
                     cpl_error_get_message_default(my_error));
    }
	*overscan_image_RE = cpl_image_wrap_double(used_real_nx, used_real_ny,
                                               overscan_vector);
	my_error = cpl_error_get_code();
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_wrap_double failed");
	/* copy the output variables */
    //espdr_msg("RON = %.12f", RON_median);
    espdr_ensure(RON_median == 0.0,
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "RON = 0.0, probably a detector problem, exiting");
    
	*RON_RE = RON_median;
	//espdr_msg("OVERSCAN RON_RE = %lf", *RON_RE);
	/*cpl_free(image_vector);*/
	//cpl_free(data_vector);
	cpl_free(std_dev_data);
	/* Can't free because it points to the master overscan image */
	/*cpl_free(overscan_vector);*/
	//cpl_free(overscan_vector);
    
    cpl_image_delete(prescan);
    cpl_image_delete(overscan);
    //cpl_image_delete(pre_overscan_line);
    cpl_image_delete(pre_overscan);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Remove the bias effect from raw input frames through overscan
 @param         input_image         image to be corrected
 @param         ext_no              extension number
 @param         out_x               x position output port
 @param         out_y               y position output port
 @param         CCD_geom            CCD geometry parameters structure
 @param         overscan_image      overscan image to correct with
 @param[out]    corrected_image_RE  corrected image (returned)
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_remove_overscan(const cpl_image *input_image,
                                     const int ext_no,
                                     const int out_x,
                                     const int out_y,
                                     const espdr_CCD_geometry *CCD_geom,
                                     cpl_image *overscan_image,
                                     cpl_image **corrected_image_RE) {
	
	cpl_error_code my_error = CPL_ERROR_NONE;
    
	int nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
	int ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
    int real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    int real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    int real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    int real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
    

	int collumn_orientation = CCD_geom->exts[ext_no].coll_or;
	int correction_type = CCD_geom->exts[ext_no].corr_type;

    int pscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_urx;
    int pscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ury;

    int absolut_llx = 0; 
    int absolut_lly = 0; 
    int	absolut_urx = 0; 
    int absolut_ury = 0;
	
	espdr_msg_debug("nx = %d, ny = %d", nx, ny);
	
	/* the direction is set by collumn_orientation and correction_type */
	if (((collumn_orientation == 0) && (correction_type == 1)) ||
		((collumn_orientation == 1) && (correction_type == 0))) {
        absolut_llx = pscan_urx + real_llx;
        absolut_lly = real_lly;
        absolut_urx = pscan_urx + real_urx;
        absolut_ury = real_ury;
        espdr_msg_debug("llx = %d, lly = %d, urx = %d, ury = %d",
                        absolut_llx, absolut_lly, absolut_urx, absolut_ury);
	} else {
		if (((collumn_orientation == 1) && (correction_type == 1)) ||
			((collumn_orientation == 0) && (correction_type == 0))) {
            absolut_llx = real_llx;
            absolut_lly = pscan_ury + real_lly;
            absolut_urx = real_urx;
            absolut_ury = pscan_ury + real_ury;
            espdr_msg_debug("llx = %d, lly = %d, urx = %d, ury = %d",
                            absolut_llx, absolut_lly, absolut_urx, absolut_ury);
        } else {
            espdr_ensure(0, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Wrong combination of collumn_orientation and correction_type, exiting");
        }
	}
    
    
	*corrected_image_RE = cpl_image_extract(input_image,
                                            absolut_llx, absolut_lly,
											absolut_urx, absolut_ury);
	my_error = cpl_error_get_code();
	espdr_ensure(my_error != CPL_ERROR_NONE,
                 my_error, "cpl_image_extract failed");
	
	cpl_image_subtract(*corrected_image_RE, overscan_image);
	my_error = cpl_error_get_code();
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_subtract failed");
	
	return cpl_error_get_code();
}



