/* $Id: espdr_pfits.c,v 1.1.1.1 2013-06-04 13:04:08 amodigli Exp $
 *
 * This file is part of the ESPDR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: amodigli $
 * $Date: 2013-06-04 13:04:08 $
 * $Revision: 1.1.1.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>

#include "espdr_pfits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_pfits     FITS header protected access
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/**
 * @brief Convert a frame type to a string
 * @param ft  Frame type to convert
 * @return A textual representation of @em  ft.
 */
/*----------------------------------------------------------------*/
static const char *
espdr_tostring_cpl_frame_type (cpl_frame_type ft)
{
  switch (ft) {
  case CPL_FRAME_TYPE_NONE:
    return "NONE";
    break;
  case CPL_FRAME_TYPE_IMAGE:
    return "IMAGE";
    break;
  case CPL_FRAME_TYPE_MATRIX:
    return "MATRIX";
    break;
  case CPL_FRAME_TYPE_TABLE:
    return "TABLE";
    break;
  default:
    return "unrecognized frame type";
  }
}
/*----------------------------------------------------------------*/
/**
 * @brief Convert a CPL type to a string
 * @param t  Type to convert
 * @return A textual representation of @em  t.
 */
/*----------------------------------------------------------------*/
const char *
espdr_tostring_cpl_type (cpl_type t)
{

  /* Note that CPL_TYPE_STRING is shorthand
     for CPL_TYPE_CHAR | CPL_TYPE_FLAG_ARRAY . */

  if (!(t & CPL_TYPE_FLAG_ARRAY))
    switch (t & (~CPL_TYPE_FLAG_ARRAY)) {
    case CPL_TYPE_CHAR:
      return "char";
      break;
    case CPL_TYPE_UCHAR:
      return "uchar";
      break;
    case CPL_TYPE_BOOL:
      return "boolean";
      break;
    case CPL_TYPE_INT:
      return "int";
      break;
    case CPL_TYPE_UINT:
      return "uint";
      break;
    case CPL_TYPE_LONG:
      return "long";
      break;
    case CPL_TYPE_ULONG:
      return "ulong";
      break;
    case CPL_TYPE_FLOAT:
      return "float";
      break;
    case CPL_TYPE_DOUBLE:
      return "double";
      break;
    case CPL_TYPE_POINTER:
      return "pointer";
      break;
/* not in CPL3.0: case CPL_TYPE_COMPLEX:    return "complex"; break; */
    case CPL_TYPE_INVALID:
      return "invalid";
      break;
    default:
      return "unrecognized type";
    }
  else
    switch (t & (~CPL_TYPE_FLAG_ARRAY)) {
    case CPL_TYPE_CHAR:
      return "string (char array)";
      break;
    case CPL_TYPE_UCHAR:
      return "uchar array";
      break;
    case CPL_TYPE_BOOL:
      return "boolean array";
      break;
    case CPL_TYPE_INT:
      return "int array";
      break;
    case CPL_TYPE_UINT:
      return "uint array";
      break;
    case CPL_TYPE_LONG:
      return "long array";
      break;
    case CPL_TYPE_ULONG:
      return "ulong array";
      break;
    case CPL_TYPE_FLOAT:
      return "float array";
      break;
    case CPL_TYPE_DOUBLE:
      return "double array";
      break;
    case CPL_TYPE_POINTER:
      return "pointer array";
      break;
/* not in CPL3.0: case CPL_TYPE_COMPLEX:    return "complex array"; break; */
    case CPL_TYPE_INVALID:
      return "invalid (array)";
      break;
    default:
      return "unrecognized type";
    }
}

/*---------------------------------------------------------------------------*/
/**
   @brief    Read a property value from a property list
   @param    plist       Propertylist to read
   @param    keyword     Name of property to read
   @param    keywordtype Type of keyword
   @param    result      The value read

   @return   CPL_ERROR_NONE iff OK

   This function wraps  @c cpl_propertylist_get_int(),
   @c cpl_propertylist_get_bool(), @c cpl_propertylist_get_double() and
   @c cpl_propertylist_get_string().
   It checks existence and type of the requested keyword before reading
   and describes what went wrong if the property could not be read.

   @note The result is written to the variable pointed to by the
   parameter @em result. Because this is a void pointer,
   it is the responsibility of the caller to make sure that the type
    of this pointer variable corresponds to the requested @em keywordtype.
   E.g. if @em keywordtype is CPL_TYPE_BOOL, then @em result must be an bool
   pointer (bool *). If @em keywordtype isCPL_TYPE_STRING, then @em result
   must be a char **, and so on.

*/
/*---------------------------------------------------------------------------*/
static cpl_error_code
espdr_get_property_value (const cpl_propertylist * plist,
			const char *keyword,
                        cpl_type keywordtype,
			void *result)
{
  cpl_type t;

  /* Check input */
  cpl_error_ensure (plist != NULL, CPL_ERROR_NULL_INPUT, return CPL_ERROR_NULL_INPUT,
      "Null property list");
  cpl_error_ensure (keyword != NULL, CPL_ERROR_NULL_INPUT,  return CPL_ERROR_NULL_INPUT,
      "Null keyword");
  /* Check for existence... */
  cpl_error_ensure (cpl_propertylist_has (plist, keyword), CPL_ERROR_DATA_NOT_FOUND,
     return CPL_ERROR_NULL_INPUT,
	  "Keyword %s does not exist", keyword);
  /* ...and type of keyword */
  cpl_error_ensure (t = cpl_propertylist_get_type (plist, keyword),
	     CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
		 "Could not read type of keyword '%s'", keyword);
  cpl_error_ensure (t == keywordtype, CPL_ERROR_TYPE_MISMATCH,
		    return CPL_ERROR_TYPE_MISMATCH,
	  "Keyword '%s' has wrong type (%s). %s expected",
	  keyword, espdr_tostring_cpl_type (t),
	  espdr_tostring_cpl_type (keywordtype));
  /* Read the keyword */
  switch (keywordtype) {
  case CPL_TYPE_INT:
    cpl_error_ensure (*((int *) result) =
	             cpl_propertylist_get_int (plist, keyword),
		      CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
			  "Could not get (integer) value of %s", keyword);
    break;
  case CPL_TYPE_BOOL:
    cpl_error_ensure (*((cpl_boolean *) result) =
	             cpl_propertylist_get_bool (plist, keyword),
		      CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
	       "Could not get (boolean) value of %s", keyword);
    break;
  case CPL_TYPE_DOUBLE:
    cpl_error_ensure (*((double *) result) =
	       cpl_propertylist_get_double (plist, keyword),
	       CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
	       "Could not get (double) value of %s", keyword);
    break;
  case CPL_TYPE_STRING:
    cpl_error_ensure (*((const char **) result) =
	       cpl_propertylist_get_string (plist, keyword),
	       CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
	       "Could not get (string) value of %s", keyword);
    break;
  default:
    cpl_error_ensure (CPL_FALSE, CPL_ERROR_INVALID_TYPE,
		      return CPL_ERROR_INVALID_TYPE, "Unknown type");
  }


  return cpl_error_get_code ();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the arcfile   
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * espdr_pfits_get_arcfile(const cpl_propertylist * plist)
{
    const char * value = cpl_propertylist_get_string(plist, "ARCFILE");

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the DIT value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double espdr_pfits_get_dit(const cpl_propertylist * plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, "ESO DET DIT");

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find out the modified julian observation date
  @param    plist       Header to read from
  @return   The requested value, or undefined on error
 */
/*----------------------------------------------------------------------------*/
double espdr_pfits_get_mjdobs(const cpl_propertylist * plist)
{
    double returnvalue = 0;

    returnvalue = cpl_propertylist_get_double(plist,"MJD-OBS");
    /*
    cpl_error_ensure(espdr_get_property_value(plist, "MJD-OBS",
                                     CPL_TYPE_DOUBLE, &returnvalue),
		     CPL_ERROR_DATA_NOT_FOUND, return 0,
       "Error reading keyword '%s'", "MJD-OBS");
       */


    return returnvalue;
}

/**@}*/
