/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-06-12 11:18:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_bepop.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 @brief  computes the background ilumination for science frames
 @param  input_image image under treatment
 @param  ccdgain CCD gain value
 @param  ccdsigdet CCD RON value
 @param  image_background_measured_RE resultant background image
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_BERV(
                        double ra,
                        double dec,
                        double pm_ra,
                        double pm_dec,
                        int year,
                        int month,
                        int day,
                        int hour,
                        int minutes,
                        double seconds,
                        double TimeToMidExposure,
                        double longitude,
                        double latitude,
                        double altitude,
                        int typeInit,
                        double *BERV_RE,
                        double *VTOT_RE,
                        double *BJD_RE,
                        double *BERVMAX_RE,
                        double *SED_RE) {

	typeParamBEPOP * ptrBEPOP = NULL;
	typeOutBEPOP resBEPOP;

	/* typeInit value 
	 * typeInit (in): model type (1, 2 or 3) 
	 *
	 * typeInit = 1 : the most precise model (EOPs taken into account, complete CIP
	 * nutations)
	 * typeInit = 2 : faster (simplified nutations), same precision, but not "up to
	 * date"
	 * typeInit = 3 : less precise (simplified nutations, no EOPs taken into
	 * account)
	 */
    /*
     Mode 3 ignores the Earth orientation parameters, at the cost of a few cm/s in accuracy on BERV. It is the only mode that allows for an input date that is more recent than the latest available EOPs. Thus, it is the only usable mode for live execution of the code on just-acquired data. Leap seconds are not affected by the choice of mode. They are managed by the ERFA library, which must be periodically updated.
     */
	
	if((typeInit > 3) || (typeInit < 1)) { 
	espdr_msg("ERROR: wrong typeInit value. Should be between 1->3");
	espdr_msg("typeInit = 1 : the most precise model (EOPs taken into"
	"account, complete CIP nutations)");
	espdr_msg("typeInit = 2 : faster (simplified nutations), same"
	"precision, but not updated");
	espdr_msg("typeInit = 3 : less precise (simplified nutations, no EOPs"
	"taken into account)");
	exit(EXIT_FAILURE);
	}

	ptrBEPOP=BEPOP_initDefault(typeInit,ptrBEPOP);

	/* Prints parameters */
	espdr_msg(" "); 	
	espdr_msg("BERV calculation parameters:");
	espdr_msg("RA                =%f",ra);
	espdr_msg("DEC               =%f",dec);
	espdr_msg("Proper motion RA  =%f",pm_ra);
	espdr_msg("Proper motion DEC =%f",pm_dec);
	espdr_msg("Year              =%d",year);
	espdr_msg("Month             =%d",month);
	espdr_msg("Day               =%d",day);
	espdr_msg("Hour              =%d",hour);
	espdr_msg("Minutes           =%d",minutes);
	espdr_msg("Seconds           =%f",seconds);
	espdr_msg("TimeToMidExposure =%f",TimeToMidExposure);
	espdr_msg("Longitude         =%f",longitude);
	espdr_msg("Latitude          =%f",latitude);
	espdr_msg("Altitude          =%f",altitude);
	espdr_msg("typeInit          =%d",typeInit);

	/* Computes the quantities used for the radial velocities data reductions */
	BEPOP_compute(ra,dec,pm_ra,pm_dec,                 /* source coordinates */
		 year,month,day,hour,minutes,seconds, /* date and time */
		 TimeToMidExposure,longitude,latitude,altitude,
		 ptrBEPOP,                            /* model parameters */
		 &resBEPOP);                          /* results */

	/* Prints the results */
	espdr_msg(" "); 	
	espdr_msg("BERV calculation results:");
	espdr_msg("Barycentric Earth Radial Velocity (in km/s)            : "
	ANSI_COLOR_GREEN"%10.6f"ANSI_COLOR_RESET,resBEPOP.BERV);
	espdr_msg("maximum value of BERVMX (in km/s)                      : "
	ANSI_COLOR_GREEN"%6.2f"ANSI_COLOR_RESET,resBEPOP.BERVMX);
	espdr_msg("Barycentric Time of Light Arrival, in julian day (TDB) : "
	ANSI_COLOR_GREEN"%20.12f"ANSI_COLOR_RESET,resBEPOP.BTLA);
	espdr_msg("Barycentric Observer's Velocity (in km/s)              : "
	ANSI_COLOR_GREEN"%10.6f"ANSI_COLOR_RESET,resBEPOP.BOV);
	espdr_msg("Sun-Earth distance (in km)                             : "
	ANSI_COLOR_GREEN"%16.6f"ANSI_COLOR_RESET,resBEPOP.SED);
	espdr_msg(" "); 	

	*BERV_RE = resBEPOP.BERV;	
	*BERVMAX_RE = resBEPOP.BERVMX;	
	*BJD_RE = resBEPOP.BTLA;	
	*VTOT_RE = resBEPOP.BOV;	
    *SED_RE = resBEPOP.SED;
    
	ptrBEPOP=BEPOP_close(ptrBEPOP);	

	return cpl_error_get_code();

}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_BERV_factor(double berv,
                                         double sed,
                                         double vtot,
                                         double *berv_factor_RE) {
    
    double LIGHT_SPEED_SQ = LIGHT_SPEED * LIGHT_SPEED;
    
    //espdr_msg("BERV: %e", berv);
    //double berv_factor_simple = (1. + 1.55e-8) * (1. + berv / 299792.458);
    //espdr_msg("berv_factor (simple): %.15f", berv_factor_simple);
    //espdr_msg("berv_factor (simple) [m/s]: %f", (berv_factor_simple - 1.0) * LIGHT_SPEED);

    sed = sed * 1000.0; // km -> m
    vtot = vtot * 1000.0; // km -> m
    double phi = GRAVITATIONAL_CONSTANT * (M_SUN/sed + M_EARTH/R_EARTH);
    double berv_factor = (1.0/(1.0 - phi/LIGHT_SPEED_SQ - vtot*vtot/2.0/LIGHT_SPEED_SQ))*
                        (1.0 + berv/(LIGHT_SPEED/1000.0));
    //espdr_msg("berv_factor (complete): %.15f", berv_factor);
    //espdr_msg("berv_factor (complete) [m/s]: %f", (berv_factor - 1.0) * LIGHT_SPEED);
    
    *berv_factor_RE = berv_factor;
    
    return cpl_error_get_code();
}

