/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.8 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ESPDR_BIAS_H
#define ESPDR_BIAS_H

/*---------------------------------------------------------------------------*/
/**
 * bias
 *
 * bias functions header
 */


/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/

/* Utility fonctions */
#include <math.h>

/* DRL functions*/
#include <espdr_overscan.h>
#include <espdr_sigma_clipping.h>
#include <espdr_dfs.h>
#include <espdr_utils.h>
#include <espdr_msg.h>
#include <espdr_instrument.h>
#include <espdr_parameters.h>
#include <espdr_CCD.h>
#include <espdr_keywords.h>

/* Library */
#include <cpl.h>
#include <assert.h>


/*----------------------------------------------------------------------------
								Constants
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

enum espdr_bias_corr_method {
    ESPDR_OSCAN_METHOD,
    ESPDR_MBIAS_METHOD,
};


/*----------------------------------------------------------------------------
 Functions prototypes
 ----------------------------------------------------------------------------*/
int espdr_single_bias(cpl_parameterlist *parameters, cpl_frameset *frameset,
							 const char* recipe_id);
int espdr_mbias(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id);
cpl_error_code espdr_parameters_BIAS_create(const char* recipe_id,
										  cpl_parameterlist *list,
										  espdr_BIAS_param *p);

cpl_error_code espdr_parameters_BIAS_delete(espdr_BIAS_param* p);

cpl_error_code espdr_parameters_BIAS_get(const char* recipe_id, 
									   cpl_parameterlist* list,
									   espdr_BIAS_param *BIAS_param);
	
cpl_error_code espdr_parameters_BIAS_print(espdr_BIAS_param *BIAS_param);

cpl_error_code espdr_parameters_stats_create(const char* recipe_id,
                                             cpl_parameterlist *list,
                                             espdr_stats_param *p);

cpl_error_code espdr_parameters_stats_delete(espdr_stats_param* p);

cpl_error_code espdr_parameters_stats_get(const char* recipe_id,
                                          cpl_parameterlist* list,
                                          espdr_stats_param *stats_param);

cpl_error_code espdr_parameters_stats_print(espdr_stats_param *stats_param);

/* AMO added */
espdr_BIAS_param *espdr_BIAS_param_init(const char *recipe_id,
                                        cpl_parameterlist *parameters);
espdr_stats_param *espdr_stats_param_init(const char *recipe_id,
                                          cpl_parameterlist *parameters);
/*************/

cpl_error_code espdr_master_bias_read_parameters(cpl_frameset *frameset,
                                                 espdr_CCD_geometry *CCD_geom,
                                                 espdr_inst_config *inst_config);

//const char* espdr_bias_get_correction_method(espdr_OVSC_param *OVSC_param,
//                                             espdr_qc_keywords *qc_kws,
//                                             cpl_propertylist *keywords);

cpl_error_code espdr_master_bias_prepare(const cpl_frameset *bias_set,
                                         const espdr_CCD_geometry *CCD_geom,
                                         espdr_qc_keywords *qc_kws,
                                         espdr_inst_config *inst_config,
                                         cpl_imagelist **master_bias_imagelist_RE,
                                         cpl_propertylist **keywords_RE,
                                         const int use_hdrl);

cpl_error_code espdr_master_bias(const cpl_frameset *bias_set,
                                 const espdr_CCD_geometry *CCD_geom,
                                 const espdr_inst_config *inst_config,
                                 cpl_imagelist **master_bias_imagelist_RE,
                                 double *master_bias_mean,
                                 double *RON_RE,
                                 int *total_cosmics_RE,
                                 const int use_hdrl);

cpl_error_code espdr_master_bias_one_output(const cpl_imagelist *raw_bias_imagelist,
                                            const espdr_CCD_geometry *CCD_geom,
                                            const espdr_inst_config *inst_config,
                                            const int nx,
                                            const int ny,
                                            const int ext_no,
                                            const int out_x,
                                            const int out_y,
                                            cpl_image **masterBiasTable_RE,
                                            double *RON_RE,
                                            double *mean_RE,
                                            int *totalCosmics_RE);

cpl_error_code espdr_single_bias_prepare(const cpl_frameset *bias_set,
                                         const espdr_CCD_geometry *CCD_geom,
                                         espdr_qc_keywords *qc_kws,
                                         espdr_inst_config *inst_config,
                                         const int reject_zeroMAD,
                                         cpl_imagelist **bias_imagelist_RE,
                                         cpl_propertylist **keywords_RE);

cpl_error_code espdr_bias_QC(const cpl_imagelist *master_bias_imagelist,
                             const cpl_imagelist *overscan_imagelist,
                             const espdr_CCD_geometry *CCD_geom,
                             espdr_qc_keywords *qc_kws,
                             espdr_inst_config *inst_config,
                             double *RON_vector_OV,
                             const double res_mean_limit,
                             const double res_stdev_tolerance,
                             cpl_propertylist **keywords_RE);

cpl_error_code espdr_add_QC_e(cpl_propertylist **keywords_ext,
                              const espdr_CCD_geometry *CCD_geom,
                              espdr_inst_config *inst_config,
                              espdr_qc_keywords *qc_kws,
                              cpl_propertylist **keywords_RE);

cpl_error_code espdr_bias_evaluation (const cpl_image *masterBiasTable,
                                      const cpl_image *masterOverscanTable,
                                      const int ext_no,
                                      const int out_x,
                                      const int out_y,
                                      const espdr_CCD_geometry *CCD_geom,
                                      const espdr_inst_config *inst_config,
                                      const double RON,
                                      const double res_mean_limit,
                                      const double res_stdev_tolerance,
                                      double *residuals_mean_RE,
                                      double *residuals_stddev_RE,
                                      int *QC_flag_RE);

cpl_error_code espdr_correct_bias_one_output(const cpl_image *input_image,
                                             const cpl_image *master_bias_image,
                                             //cpl_propertylist *keywords,
                                             //espdr_qc_keywords *qc_kws,
                                             const int ext_no,
                                             const int out_x,
                                             const int out_y,
                                             const espdr_CCD_geometry *CCD_geom,
                                             espdr_inst_config *inst_config,
                                             const int remove_bias_res,
                                             double *RON_RE,
                                             cpl_image **input_corrected_RE);

cpl_error_code espdr_correct_bias(const cpl_imagelist *input_imagelist,
                                  const cpl_imagelist *master_bias_imagelist,
                                  espdr_inst_config *inst_config,
                                  const espdr_CCD_geometry *CCD_geom,
                                  const int remove_bias_res,
                                  double *RON,
                                  cpl_imagelist *input_corrected_RE);

cpl_error_code espdr_correct_bias_frames(cpl_frameset *raw_bias_set,
                                         cpl_frameset *master_bias_set,
                                         espdr_CCD_geometry *CCD_geom,
                                         espdr_inst_config *inst_config,
                                         int remove_bias_res,
                                         cpl_frameset** input_corrected_set,
                                         const int use_hdrl);

cpl_error_code espdr_remove_bias_one_output(const cpl_image *input_image,
                                            const int ext_no,
                                            const int out_x,
                                            const int out_y,
                                            const espdr_CCD_geometry *CCD_geom,
                                            const cpl_image *master_bias_image,
                                            cpl_image **corrected_image_RE);

cpl_error_code espdr_remove_bias(const cpl_image *input_image,
                                 const int ext_no,
                                 const int out_x,
                                 const int out_y,
                                 const espdr_CCD_geometry *CCD_geom,
                                 const cpl_image *master_bias_image,
                                 cpl_image **corrected_image_RE);
	
/*  The following functions are added by Andrea Modigliani
    to detect the structure present in the bias frames
 */
void espdr_qc_structure(const cpl_image *master,
                               double *structx,
                               double *structy);

void espdr_qc_structure_region(const cpl_image *master,
                               const int llx,
                               const int lly,
                               const int urx,
                               const int ury,
                               const int ref_x,
                               const int ref_y,
                               const int reg_id,
                               double *structx,
                               double *structy);

double espdr_get_qc_structure_col_region(cpl_image *tima);

double espdr_get_qc_structure_row_region(cpl_image *tima);

int espdr_count_good(const cpl_image *image);

void reject_lo_hi(cpl_image *image, double min, double max);


#endif /* espdr_BIAS_H */
